"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.printBlock = exports.print = exports.printObject = void 0;
const ast_1 = require("./ast");
const TAB = '  ';
function printIf(node, indent = '') {
    if (node.inline) {
        return `#if( ${printExpr(node.predicate, '')} ) ${printExpr(node.expr, '')} #end`;
    }
    return `${indent}#if( ${printExpr(node.predicate, '')} )\n${printExpr(node.expr, indent + TAB)}\n${indent}#end`;
}
function printIfElse(node, indent = '') {
    if (node.inline) {
        return `#if( ${printExpr(node.predicate)} ) ` + `${printExpr(node.ifExpr)} ` + `#else ` + `${printExpr(node.elseExpr)} ` + `#end`;
    }
    return (`${indent}#if( ${printExpr(node.predicate)} )\n` +
        `${printExpr(node.ifExpr, indent + TAB)}\n` +
        `${indent}#else\n` +
        `${printExpr(node.elseExpr, indent + TAB)}\n` +
        `${indent}#end`);
}
function printAnd(node, indent = '') {
    return indent + node.expressions.map((e) => printExpr(e)).join(' && ');
}
function printOr(node, indent = '') {
    return indent + node.expressions.map((e) => printExpr(e)).join(' || ');
}
function printParens(node, indent = '') {
    return `${indent}(${printExpr(node.expr)})`;
}
function printEquals(node, indent = '') {
    return `${indent}${printExpr(node.leftExpr)} == ${printExpr(node.rightExpr)}`;
}
function printNotEquals(node, indent = '') {
    return `${indent}${printExpr(node.leftExpr)} != ${printExpr(node.rightExpr)}`;
}
function printForEach(node, indent = '') {
    return (`${indent}#foreach( ${printExpr(node.key)} in ${printExpr(node.collection)} )\n` +
        node.expressions.map((e) => printExpr(e, indent + TAB)).join('\n') +
        `\n${indent}#end`);
}
function printString(node) {
    return `"${node.value}"`;
}
function printBool(node) {
    return `${node.value}`;
}
function printRaw(node, indent = '') {
    return `${indent}${node.value}`;
}
function printQuotes(node) {
    return `"${printExpr(node.expr)}"`;
}
function printInt(node) {
    return `${node.value}`;
}
function printFloat(node) {
    return `${node.value}`;
}
function printNull(node) {
    return `null`;
}
function printReference(node) {
    return `\$${node.value}`;
}
function printQuietReference(node, indent = '') {
    const val = typeof node.value === 'string' ? node.value : printExpr(node.value);
    return `${indent}$util.qr(${val})`;
}
function printObject(node, indent = '') {
    const attributes = node.attributes.map((attr, i) => {
        return `${indent}${TAB}"${attr[0]}": ${printExpr(attr[1], indent + TAB)}${i < node.attributes.length - 1 ? ',' : ''}`;
    });
    const divider = attributes.length > 0 ? `\n${indent}` : '';
    return `{${divider}${attributes.join(divider)}${divider}}`;
}
exports.printObject = printObject;
function printList(node, indent = '') {
    const values = node.expressions.map((e) => printExpr(e, '')).join(', ');
    return `${indent}[${values}]`;
}
function printSet(node, indent = '') {
    return `${indent}#set( ${printReference(node.key)} = ${printExpr(node.value, '')} )`;
}
function printComment(node, indent = '') {
    return `${indent}## ${node.text} **`;
}
function printCompoundExpression(node, indent = '') {
    if (node.recurseIndent) {
        return node.expressions.map((node) => printExpr(node, indent)).join(node.joiner);
    }
    return indent + node.expressions.map((node) => printExpr(node)).join(node.joiner);
}
function printToJson(node, indent = '') {
    return `${indent}$util.toJson(${printExpr(node.expr, '')})`;
}
function printIsNullOrEmpty(node, indent = '') {
    return `${indent}$util.isNullOrEmpty(${printExpr(node.expr, '')})`;
}
function printNot(node, indent = '') {
    return `${indent}!${printExpr(node.expr, '')}`;
}
function printNewLine(node) {
    return '\n';
}
function printReturn(node, indent = '') {
    var suffix = '';
    if (node.value !== undefined) {
        suffix = printParens(ast_1.parens(node.value));
    }
    return `${indent}#return` + suffix;
}
function printExpr(expr, indent = '') {
    if (!expr) {
        return '';
    }
    switch (expr.kind) {
        case 'If':
            return printIf(expr, indent);
        case 'IfElse':
            return printIfElse(expr, indent);
        case 'And':
            return printAnd(expr, indent);
        case 'Or':
            return printOr(expr, indent);
        case 'Parens':
            return printParens(expr, indent);
        case 'Equals':
            return printEquals(expr, indent);
        case 'NotEquals':
            return printNotEquals(expr, indent);
        case 'ForEach':
            return printForEach(expr, indent);
        case 'String':
            return printString(expr);
        case 'Raw':
            return printRaw(expr, indent);
        case 'Quotes':
            return printQuotes(expr);
        case 'Float':
            return printFloat(expr);
        case 'Int':
            return printInt(expr);
        case 'Boolean':
            return printBool(expr);
        case 'Null':
            return printNull(expr);
        case 'Reference':
            return printReference(expr);
        case 'QuietReference':
            return printQuietReference(expr, indent);
        case 'Object':
            return printObject(expr, indent);
        case 'List':
            return printList(expr, indent);
        case 'Set':
            return printSet(expr, indent);
        case 'Comment':
            return printComment(expr, indent);
        case 'CompoundExpression':
            return printCompoundExpression(expr, indent);
        case 'Util.ToJson':
            return printToJson(expr, indent);
        case 'Util.isNullOrEmpty':
            return printIsNullOrEmpty(expr, indent);
        case 'Not':
            return printNot(expr, indent);
        case 'NewLine':
            return printNewLine(expr);
        case 'Return':
            return printReturn(expr, indent);
        default:
            return '';
    }
}
function print(expr) {
    return printExpr(expr);
}
exports.print = print;
function printBlock(name) {
    return (expr) => {
        const wrappedExpr = ast_1.compoundExpression([ast_1.comment(`[Start] ${name}.`), expr, ast_1.comment(`[End] ${name}.`)]);
        return printExpr(wrappedExpr);
    };
}
exports.printBlock = printBlock;
//# sourceMappingURL=print.js.map