from datetime import datetime
from enum import Enum
from typing import ClassVar, Optional

from pydantic import BaseModel


class EnergyUsageUnit(Enum):
    """
    Energy usage unit.
    """

    WATT = "watts"
    MILLIWATT = "milliwatts"


class UsageType(Enum):
    """
    Usage type.
    """

    HOST = "host"
    CPU = "cpu"
    MEMORY = "memory"
    GPU = "gpu"


class EnergyUsage(BaseModel):
    """
    Energy report in watts.
    """

    host_energy_usage: float = 0.0
    cpu_energy_usage: Optional[float] = None
    memory_energy_usage: Optional[float] = None
    gpu_energy_usage: Optional[float] = None
    unit: EnergyUsageUnit = EnergyUsageUnit.WATT

    def get_energy_usage_on_type(self, usage_type: UsageType) -> Optional[float]:
        """
        Get the energy usage based on the type.

        :param: usage_type: the type of energy to return
        :return: the energy of the type
        """
        if usage_type == UsageType.CPU:
            return self.cpu_energy_usage
        elif usage_type == UsageType.GPU:
            return self.gpu_energy_usage
        elif usage_type == UsageType.HOST:
            return self.host_energy_usage
        elif usage_type == UsageType.MEMORY:
            return self.memory_energy_usage
        return None

    def convert_unit(self, unit: EnergyUsageUnit) -> None:
        """
        Convert the EnergyUsage with the right energy usage type

        :param: unit: the energy usage unit for the conversion
        """
        if self.unit != unit:
            if unit == EnergyUsageUnit.WATT and self.unit == EnergyUsageUnit.MILLIWATT:
                self.host_energy_usage = self.host_energy_usage * 1000
                self.cpu_energy_usage = (
                    self.cpu_energy_usage * 1000
                    if self.cpu_energy_usage is not None
                    else None
                )
                self.memory_energy_usage = (
                    self.memory_energy_usage * 1000
                    if self.memory_energy_usage is not None
                    else None
                )
                self.gpu_energy_usage = (
                    self.gpu_energy_usage / 1000
                    if self.gpu_energy_usage is not None
                    else None
                )
                self.unit = EnergyUsageUnit.MILLIWATT
            elif (
                unit == EnergyUsageUnit.MILLIWATT and self.unit == EnergyUsageUnit.WATT
            ):
                self.host_energy_usage = self.host_energy_usage * 1000

                self.cpu_energy_usage = (
                    self.cpu_energy_usage * 1000
                    if self.cpu_energy_usage is not None
                    else None
                )
                self.memory_energy_usage = (
                    self.memory_energy_usage * 1000
                    if self.memory_energy_usage is not None
                    else None
                )
                self.gpu_energy_usage = (
                    self.gpu_energy_usage * 1000
                    if self.gpu_energy_usage is not None
                    else None
                )
                self.unit = EnergyUsageUnit.MILLIWATT


class Power(BaseModel):
    """Power utility"""

    MICROJOULES_TO_WATT_FACTOR: ClassVar[int] = 1000000
    WH_TO_KWH_FACTOR: ClassVar[int] = 1000
    SECONDS_TO_HOURS_FACTOR: ClassVar[int] = 3600

    @staticmethod
    def watts_to_watt_hours(
        watts: float, previous_energy_measurement_time: Optional[datetime] = None
    ) -> float:
        """
        Convert current watts to watt-hours W/h using the previous energy measurement.

        :param watts: the wattage in W
        :param previous_energy_measurement_time: the previous measurement time
        :return: watt-hours W/h
        """
        now = datetime.now()
        if previous_energy_measurement_time:
            time_difference_in_seconds = (
                now - previous_energy_measurement_time
            ).total_seconds()
        else:
            time_difference_in_seconds = 1
        return watts * (time_difference_in_seconds / Power.SECONDS_TO_HOURS_FACTOR)

    @staticmethod
    def co2g_from_watts_hour(watts_hour: float, co2g_per_kwh: float) -> float:
        """
        Calculate the CO2g generated using watt hours and the CO2g/kwh.

        :return: the CO2g generated by the energy consumption
        """
        return (watts_hour / Power.WH_TO_KWH_FACTOR) * co2g_per_kwh

    @staticmethod
    def watts_from_microjoules(
        uj: float,
    ) -> float:
        """
        Get watts from microjoules.

        :param: uj: energy in microjoules
        :return: watts
        """
        return uj / Power.MICROJOULES_TO_WATT_FACTOR
