# Copyright (C) 2020-2021 Sebastian Blauth
#
# This file is part of CASHOCS.
#
# CASHOCS is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CASHOCS is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with CASHOCS.  If not, see <https://www.gnu.org/licenses/>.

"""Mesh generation and import tools.

This module consists of tools for for the fast generation
or import of meshes into fenics. The :py:func:`import_mesh <cashocs.geometry.import_mesh>` function
is used to import (converted) GMSH mesh files, and the :py:func:`regular_mesh <cashocs.geometry.regular_mesh>`
and :py:func:`regular_box_mesh <cashocs.geometry.regular_box_mesh>` commands create 2D and 3D box meshes
which are great for testing.
"""

import configparser
import json
import os
import subprocess
import sys
import tempfile
import time
from collections import Counter

import fenics
import numpy as np
from petsc4py import PETSc
from ufl import Jacobian, JacobianInverse
from ufl.measure import Measure

from ._exceptions import CashocsException, ConfigError, InputError
from ._loggers import debug, info, warning
from .utils import (
    _assemble_petsc_system,
    _parse_remesh,
    _setup_petsc_options,
    _solve_linear_problem,
    create_dirichlet_bcs,
    write_out_mesh,
)


def import_mesh(input_arg):
    """Imports a mesh file for use with CASHOCS / FEniCS.

    This function imports a mesh file that was generated by GMSH and converted to
    .xdmf with the command line function :ref:`cashocs-convert <cashocs_convert>`.
    If there are Physical quantities specified in the GMSH file, these are imported
    to the subdomains and boundaries output of this function and can also be directly
    accessed via the measures, e.g., with ``dx(1)``, ``ds(1)``, etc.

    Parameters
    ----------
    input_arg : str or configparser.ConfigParser
            This is either a string, in which case it corresponds to the location
            of the mesh file in .xdmf file format, or a config file that
            has this path stored in its settings, under the section Mesh, as
            parameter ``mesh_file``.

    Returns
    -------
    mesh : dolfin.cpp.mesh.Mesh.Mesh
            The imported (computational) mesh.
    subdomains : dolfin.cpp.mesh.MeshFunctionSizet.MeshFunctionSizet
            A :py:class:`fenics.MeshFunction` object containing the subdomains,
            i.e., the Physical regions marked in the GMSH
            file.
    boundaries : dolfin.cpp.mesh.MeshFunctionSizet.MeshFunctionSizet
            A MeshFunction object containing the boundaries,
            i.e., the Physical regions marked in the GMSH
            file. Can, e.g., be used to set up boundary
            conditions.
    dx : ufl.measure.Measure
            The volume measure of the mesh corresponding to
            the subdomains (i.e. GMSH Physical region indices).
    ds : ufl.measure.Measure
            The surface measure of the mesh corresponding to
            the boundaries (i.e. GMSH Physical region indices).
    dS : ufl.measure.Measure
            The interior facet measure of the mesh corresponding
            to boundaries (i.e. GMSH Physical region indices).

    Notes
    -----
    In case the boundaries in the Gmsh .msh file are not only marked with numbers (as pyhsical
    groups), but also with names (i.e. strings), these strings can be used with the integration
    measures ``dx`` and ``ds`` returned by this method. E.g., if one specified the
    following in a 2D Gmsh .geo file ::

        Physical Surface("domain", 1) = {i,j,k};

    where i,j,k are representative for some integers, then this can be used in the measure
    ``dx`` (as we are 2D) as follows. The command ::

        dx(1)

    is completely equivalent to ::

       dx("domain")

    and both can be used interchangeably.
    """

    start_time = time.time()
    info("Importing mesh.")

    cashocs_remesh_flag, temp_dir = _parse_remesh()

    # Check for the file format
    if isinstance(input_arg, str):
        mesh_file = input_arg
        mesh_attribute = "str"
    elif isinstance(input_arg, configparser.ConfigParser):
        mesh_attribute = "config"
        ### overloading for remeshing
        if not input_arg.getboolean("Mesh", "remesh", fallback=False):
            mesh_file = input_arg.get("Mesh", "mesh_file")
        else:
            if not cashocs_remesh_flag:
                mesh_file = input_arg.get("Mesh", "mesh_file")
            else:
                with open(f"{temp_dir}/temp_dict.json", "r") as file:
                    temp_dict = json.load(file)
                mesh_file = temp_dict["mesh_file"]

    else:
        raise InputError(
            "cashocs.geometry.import_mesh",
            "input_arg",
            "Not a valid argument for import_mesh. Has to be either a path to a mesh file (str) or a config.",
        )

    if mesh_file[-5:] == ".xdmf":
        file_string = mesh_file[:-5]
    else:
        raise InputError(
            "cashocs.geometry.import_mesh",
            "input_arg",
            "Not a suitable mesh file format. Has to end in .xdmf.",
        )

    mesh = fenics.Mesh()
    xdmf_file = fenics.XDMFFile(mesh.mpi_comm(), mesh_file)
    xdmf_file.read(mesh)
    xdmf_file.close()

    subdomains_mvc = fenics.MeshValueCollection(
        "size_t", mesh, mesh.geometric_dimension()
    )
    boundaries_mvc = fenics.MeshValueCollection(
        "size_t", mesh, mesh.geometric_dimension() - 1
    )

    if os.path.isfile(f"{file_string}_subdomains.xdmf"):
        xdmf_subdomains = fenics.XDMFFile(
            mesh.mpi_comm(), f"{file_string}_subdomains.xdmf"
        )
        xdmf_subdomains.read(subdomains_mvc, "subdomains")
        xdmf_subdomains.close()
    if os.path.isfile(f"{file_string}_boundaries.xdmf"):
        xdmf_boundaries = fenics.XDMFFile(
            mesh.mpi_comm(), f"{file_string}_boundaries.xdmf"
        )
        xdmf_boundaries.read(boundaries_mvc, "boundaries")
        xdmf_boundaries.close()

    physical_groups = None
    if os.path.isfile(f"{file_string}_physical_groups.json"):
        with open(f"{file_string}_physical_groups.json") as file:
            physical_groups = json.load(file)

    subdomains = fenics.MeshFunction("size_t", mesh, subdomains_mvc)
    boundaries = fenics.MeshFunction("size_t", mesh, boundaries_mvc)

    dx = _NamedMeasure(
        "dx", domain=mesh, subdomain_data=subdomains, physical_groups=physical_groups
    )
    ds = _NamedMeasure(
        "ds", domain=mesh, subdomain_data=boundaries, physical_groups=physical_groups
    )
    dS = _NamedMeasure(
        "dS", domain=mesh, subdomain_data=boundaries, physical_groups=physical_groups
    )

    end_time = time.time()
    info(f"Done importing mesh. Elapsed time: {end_time - start_time:.2f} s")
    info(
        f"Mesh contains {mesh.num_vertices():,} vertices and {mesh.num_cells():,} cells of type {mesh.ufl_cell().cellname()}.\n"
    )

    # Add an attribute to the mesh to show with what procedure it was generated
    mesh._cashocs_generator = mesh_attribute
    # Add the physical groups to the mesh in case they are present
    if physical_groups is not None:
        mesh._physical_groups = physical_groups

    # Check the mesh quality of the imported mesh in case a config file is passed
    if isinstance(input_arg, configparser.ConfigParser):
        mesh_quality_tol_lower = input_arg.getfloat(
            "MeshQuality", "tol_lower", fallback=0.0
        )
        mesh_quality_tol_upper = input_arg.getfloat(
            "MeshQuality", "tol_upper", fallback=1e-15
        )

        if not mesh_quality_tol_lower < mesh_quality_tol_upper:
            raise ConfigError(
                "MeshQuality",
                "tol_lower",
                "The lower remeshing tolerance has to be strictly smaller than the upper remeshing tolerance",
            )

        if mesh_quality_tol_lower > 0.9 * mesh_quality_tol_upper:
            warning(
                "You are using a lower remesh tolerance (tol_lower) close to the upper one (tol_upper). This may slow down the optimization considerably."
            )

        mesh_quality_measure = input_arg.get(
            "MeshQuality", "measure", fallback="skewness"
        )
        if not mesh_quality_measure in [
            "skewness",
            "maximum_angle",
            "radius_ratios",
            "condition_number",
        ]:
            raise ConfigError(
                "MeshQuality",
                "measure",
                "Has to be one of 'skewness', 'maximum_angle', 'condition_number', or 'radius_ratios'.",
            )

        mesh_quality_type = input_arg.get("MeshQuality", "type", fallback="min")
        if not mesh_quality_type in ["min", "minimum", "avg", "average"]:
            raise ConfigError(
                "MeshQuality",
                "type",
                "Has to be one of 'min', 'minimum', 'avg', or 'average'.",
            )

        if mesh_quality_type in ["min", "minimum"]:
            if mesh_quality_measure == "skewness":
                current_mesh_quality = MeshQuality.min_skewness(mesh)
            elif mesh_quality_measure == "maximum_angle":
                current_mesh_quality = MeshQuality.min_maximum_angle(mesh)
            elif mesh_quality_measure == "radius_ratios":
                current_mesh_quality = MeshQuality.min_radius_ratios(mesh)
            elif mesh_quality_measure == "condition_number":
                current_mesh_quality = MeshQuality.min_condition_number(mesh)

        else:
            if mesh_quality_measure == "skewness":
                current_mesh_quality = MeshQuality.avg_skewness(mesh)
            elif mesh_quality_measure == "maximum_angle":
                current_mesh_quality = MeshQuality.avg_maximum_angle(mesh)
            elif mesh_quality_measure == "radius_ratios":
                current_mesh_quality = MeshQuality.avg_radius_ratios(mesh)
            elif mesh_quality_measure == "condition_number":
                current_mesh_quality = MeshQuality.avg_condition_number(mesh)

        if not cashocs_remesh_flag:
            if current_mesh_quality < mesh_quality_tol_lower:
                raise InputError(
                    "cashocs.geometry.import_mesh",
                    "input_arg",
                    "The quality of the mesh file you have specified is not sufficient for evaluating the cost functional.\n"
                    + f"It currently is {current_mesh_quality:.3e} but has to be at least {mesh_quality_tol_lower:.3e}.",
                )

            if current_mesh_quality < mesh_quality_tol_upper:
                raise InputError(
                    "cashocs.geometry.import_mesh",
                    "input_arg",
                    "The quality of the mesh file you have specified is not sufficient for computing the shape gradient.\n "
                    + f"It currently is {current_mesh_quality:.3e} but has to be at least {mesh_quality_tol_lower:.3e}.",
                )

        else:
            if current_mesh_quality < mesh_quality_tol_lower:
                raise InputError(
                    "cashocs.geometry.import_mesh",
                    "input_arg",
                    "Remeshing failed.\n"
                    "The quality of the mesh file generated through remeshing is not sufficient for evaluating the cost functional.\n"
                    + f"It currently is {current_mesh_quality:.3e} but has to be at least {mesh_quality_tol_lower:.3e}.",
                )

            if current_mesh_quality < mesh_quality_tol_upper:
                raise InputError(
                    "cashocs.geometry.import_mesh",
                    "input_arg",
                    "Remeshing failed.\n"
                    "The quality of the mesh file generated through remeshing is not sufficient for computing the shape gradient.\n "
                    + f"It currently is {current_mesh_quality:.3e} but has to be at least {mesh_quality_tol_upper:.3e}.",
                )

    return mesh, subdomains, boundaries, dx, ds, dS


def regular_mesh(n=10, L_x=1.0, L_y=1.0, L_z=None, diagonal="right"):
    r"""Creates a mesh corresponding to a rectangle or cube.

	This function creates a uniform mesh of either a rectangle
	or a cube, starting at the origin and having length specified
	in ``L_x``, ``L_y``, and ``L_z``. The resulting mesh uses ``n`` elements along the
	shortest direction and accordingly many along the longer ones.
	The resulting domain is

	.. math::
		\begin{alignedat}{2}
		&[0, L_x] \times [0, L_y] \quad &&\text{ in } 2D, \\
		&[0, L_x] \times [0, L_y] \times [0, L_z] \quad &&\text{ in } 3D.
		\end{alignedat}

	The boundary markers are ordered as follows:

	  - 1 corresponds to :math:`x=0`.

	  - 2 corresponds to :math:`x=L_x`.

	  - 3 corresponds to :math:`y=0`.

	  - 4 corresponds to :math:`y=L_y`.

	  - 5 corresponds to :math:`z=0` (only in 3D).

	  - 6 corresponds to :math:`z=L_z` (only in 3D).

	Parameters
	----------
	n : int
		Number of elements in the shortest coordinate direction.
	L_x : float
		Length in x-direction.
	L_y : float
		Length in y-direction.
	L_z : float or None, optional
		Length in z-direction, if this is ``None``, then the geometry
		will be two-dimensional (default is ``None``).
	diagonal : str, optional
	    This defines the type of diagonal used to create the box mesh in 2D. This can be
	    one of ``"right"``, ``"left"``, ``"left/right"``, ``"right/left"`` or
	    ``"crossed"``.

	Returns
	-------
	mesh : dolfin.cpp.mesh.Mesh.Mesh
		The computational mesh.
	subdomains : dolfin.cpp.mesh.MeshFunctionSizet.MeshFunctionSizet
		A :py:class:`fenics.MeshFunction` object containing the subdomains.
	boundaries : dolfin.cpp.mesh.MeshFunctionSizet.MeshFunctionSizet
		A MeshFunction object containing the boundaries.
	dx : ufl.measure.Measure
		The volume measure of the mesh corresponding to subdomains.
	ds : ufl.measure.Measure
		The surface measure of the mesh corresponding to boundaries.
	dS : ufl.measure.Measure
		The interior facet measure of the mesh corresponding to boundaries.
	"""
    if not n > 0:
        raise InputError(
            "cashocs.geometry.regular_mesh", "n", "This needs to be positive."
        )
    if not L_x > 0.0:
        raise InputError(
            "cashocs.geometry.regular_mesh", "L_x", "L_x needs to be positive"
        )
    if not L_y > 0.0:
        raise InputError(
            "cashocs.geometry.regular_mesh", "L_y", "L_y needs to be positive"
        )
    if not (L_z is None or L_z > 0.0):
        raise InputError(
            "cashocs.geometry.regular_mesh",
            "L_z",
            "L_z needs to be positive or None (for 2D mesh)",
        )

    n = int(n)

    if L_z is None:
        sizes = [L_x, L_y]
        dim = 2
    else:
        sizes = [L_x, L_y, L_z]
        dim = 3

    size_min = np.min(sizes)
    num_points = [int(np.round(length / size_min * n)) for length in sizes]

    if L_z is None:
        mesh = fenics.RectangleMesh(
            fenics.Point(0, 0),
            fenics.Point(sizes),
            num_points[0],
            num_points[1],
            diagonal=diagonal,
        )
    else:
        mesh = fenics.BoxMesh(
            fenics.Point(0, 0, 0),
            fenics.Point(sizes),
            num_points[0],
            num_points[1],
            num_points[2],
        )

    subdomains = fenics.MeshFunction("size_t", mesh, dim=dim)
    boundaries = fenics.MeshFunction("size_t", mesh, dim=dim - 1)

    x_min = fenics.CompiledSubDomain(
        "on_boundary && near(x[0], 0, tol)", tol=fenics.DOLFIN_EPS
    )
    x_max = fenics.CompiledSubDomain(
        "on_boundary && near(x[0], length, tol)", tol=fenics.DOLFIN_EPS, length=sizes[0]
    )
    x_min.mark(boundaries, 1)
    x_max.mark(boundaries, 2)

    y_min = fenics.CompiledSubDomain(
        "on_boundary && near(x[1], 0, tol)", tol=fenics.DOLFIN_EPS
    )
    y_max = fenics.CompiledSubDomain(
        "on_boundary && near(x[1], length, tol)", tol=fenics.DOLFIN_EPS, length=sizes[1]
    )
    y_min.mark(boundaries, 3)
    y_max.mark(boundaries, 4)

    if L_z is not None:
        z_min = fenics.CompiledSubDomain(
            "on_boundary && near(x[2], 0, tol)", tol=fenics.DOLFIN_EPS
        )
        z_max = fenics.CompiledSubDomain(
            "on_boundary && near(x[2], length, tol)",
            tol=fenics.DOLFIN_EPS,
            length=sizes[2],
        )
        z_min.mark(boundaries, 5)
        z_max.mark(boundaries, 6)

    dx = _NamedMeasure("dx", mesh, subdomain_data=subdomains)
    ds = _NamedMeasure("ds", mesh, subdomain_data=boundaries)
    dS = _NamedMeasure("dS", mesh)

    return mesh, subdomains, boundaries, dx, ds, dS


def regular_box_mesh(
    n=10, S_x=0.0, S_y=0.0, S_z=None, E_x=1.0, E_y=1.0, E_z=None, diagonal="right"
):
    r"""Creates a mesh corresponding to a rectangle or cube.

	This function creates a uniform mesh of either a rectangle
	or a cube, with specified start (``S_``) and end points (``E_``).
	The resulting mesh uses ``n`` elements along the shortest direction
	and accordingly many along the longer ones. The resulting domain is

	.. math::
		\begin{alignedat}{2}
			&[S_x, E_x] \times [S_y, E_y] \quad &&\text{ in } 2D, \\
			&[S_x, E_x] \times [S_y, E_y] \times [S_z, E_z] \quad &&\text{ in } 3D.
		\end{alignedat}

	The boundary markers are ordered as follows:

	  - 1 corresponds to :math:`x=S_x`.

	  - 2 corresponds to :math:`x=E_x`.

	  - 3 corresponds to :math:`y=S_y`.

	  - 4 corresponds to :math:`y=E_y`.

	  - 5 corresponds to :math:`z=S_z` (only in 3D).

	  - 6 corresponds to :math:`z=E_z` (only in 3D).

	Parameters
	----------
	n : int
		Number of elements in the shortest coordinate direction.
	S_x : float
		Start of the x-interval.
	S_y : float
		Start of the y-interval.
	S_z : float or None, optional
		Start of the z-interval, mesh is 2D if this is ``None``
		(default is ``None``).
	E_x : float
		End of the x-interval.
	E_y : float
		End of the y-interval.
	E_z : float or None, optional
		End of the z-interval, mesh is 2D if this is ``None``
		(default is ``None``).
	diagonal : str, optional
	    This defines the type of diagonal used to create the box mesh in 2D. This can be
	    one of ``"right"``, ``"left"``, ``"left/right"``, ``"right/left"`` or
	    ``"crossed"``.

	Returns
	-------
	mesh : dolfin.cpp.mesh.Mesh.Mesh
		The computational mesh.
	subdomains : dolfin.cpp.mesh.MeshFunctionSizet.MeshFunctionSizet
		A MeshFunction object containing the subdomains.
	boundaries : dolfin.cpp.mesh.MeshFunctionSizet.MeshFunctionSizet
		A MeshFunction object containing the boundaries.
	dx : ufl.measure.Measure
		The volume measure of the mesh corresponding to subdomains.
	ds : ufl.measure.Measure
		The surface measure of the mesh corresponding to boundaries.
	dS : ufl.measure.Measure
		The interior facet measure of the mesh corresponding to boundaries.
	"""

    n = int(n)

    if not n > 0:
        raise InputError(
            "cashocs.geometry.regular_box_mesh", "n", "This needs to be positive."
        )

    if not S_x < E_x:
        raise InputError(
            "cashocs.geometry.regular_box_mesh",
            "S_x",
            "Incorrect input for the x-coordinate. S_x has to be smaller than E_x.",
        )
    if not S_y < E_y:
        raise InputError(
            "cashocs.geometry.regular_box_mesh",
            "S_y",
            "Incorrect input for the y-coordinate. S_y has to be smaller than E_y.",
        )
    if not ((S_z is None and E_z is None) or (S_z < E_z)):
        raise InputError(
            "cashocs.geometry.regular_box_mesh",
            "S_z",
            "Incorrect input for the z-coordinate. S_z has to be smaller than E_z, or only one of them is specified.",
        )

    if S_z is None:
        lx = E_x - S_x
        ly = E_y - S_y
        sizes = [lx, ly]
        dim = 2
    else:
        lx = E_x - S_x
        ly = E_y - S_y
        lz = E_z - S_z
        sizes = [lx, ly, lz]
        dim = 3

    size_min = np.min(sizes)
    num_points = [int(np.round(length / size_min * n)) for length in sizes]

    if S_z is None:
        mesh = fenics.RectangleMesh(
            fenics.Point(S_x, S_y),
            fenics.Point(E_x, E_y),
            num_points[0],
            num_points[1],
            diagonal=diagonal,
        )
    else:
        mesh = fenics.BoxMesh(
            fenics.Point(S_x, S_y, S_z),
            fenics.Point(E_x, E_y, E_z),
            num_points[0],
            num_points[1],
            num_points[2],
        )

    subdomains = fenics.MeshFunction("size_t", mesh, dim=dim)
    boundaries = fenics.MeshFunction("size_t", mesh, dim=dim - 1)

    x_min = fenics.CompiledSubDomain(
        "on_boundary && near(x[0], sx, tol)", tol=fenics.DOLFIN_EPS, sx=S_x
    )
    x_max = fenics.CompiledSubDomain(
        "on_boundary && near(x[0], ex, tol)", tol=fenics.DOLFIN_EPS, ex=E_x
    )
    x_min.mark(boundaries, 1)
    x_max.mark(boundaries, 2)

    y_min = fenics.CompiledSubDomain(
        "on_boundary && near(x[1], sy, tol)", tol=fenics.DOLFIN_EPS, sy=S_y
    )
    y_max = fenics.CompiledSubDomain(
        "on_boundary && near(x[1], ey, tol)", tol=fenics.DOLFIN_EPS, ey=E_y
    )
    y_min.mark(boundaries, 3)
    y_max.mark(boundaries, 4)

    if S_z is not None:
        z_min = fenics.CompiledSubDomain(
            "on_boundary && near(x[2], sz, tol)", tol=fenics.DOLFIN_EPS, sz=S_z
        )
        z_max = fenics.CompiledSubDomain(
            "on_boundary && near(x[2], ez, tol)", tol=fenics.DOLFIN_EPS, ez=E_z
        )
        z_min.mark(boundaries, 5)
        z_max.mark(boundaries, 6)

    dx = _NamedMeasure("dx", mesh, subdomain_data=subdomains)
    ds = _NamedMeasure("ds", mesh, subdomain_data=boundaries)
    dS = _NamedMeasure("dS", mesh)

    return mesh, subdomains, boundaries, dx, ds, dS


def compute_mesh_quality(mesh, type="min", measure="skewness"):
    """This computes the mesh quality of a given mesh.

    Parameters
    ----------
    mesh : dolfin.cpp.mesh.Mesh.Mesh
        The mesh whose quality shall be computed
    type : {'min', 'minimum', 'avg', 'average'}, optional
        The type of measurement for the mesh quality, either minimum quality or average
        quality over all mesh cells, default is 'min'
    measure : {'skewness', 'maximum_angle', 'radius_ratios', 'condition_number'}, optional
        The type of quality measure which is used to compute the quality measure, default
        is 'skewness'

    Returns
    -------
     : float
        The quality of the mesh, in the interval :math:`[0,1]`, where 0 is the worst, and
        1 the best possible quality.
    """

    if type in ["min", "minimum"]:
        if measure == "skewness":
            quality = MeshQuality.min_skewness(mesh)
        elif measure == "maximum_angle":
            quality = MeshQuality.min_maximum_angle(mesh)
        elif measure == "radius_ratios":
            quality = MeshQuality.min_radius_ratios(mesh)
        elif measure == "condition_number":
            quality = MeshQuality.min_condition_number(mesh)
        else:
            raise InputError(
                "cashocs.geometry.compute_mesh_quality",
                "measure",
                "The parameter `measure` has to be either 'min', 'minimum' or 'avg', 'average'",
            )

    elif type in ["avg", "average"]:
        if measure == "skewness":
            quality = MeshQuality.avg_skewness(mesh)
        elif measure == "maximum_angle":
            quality = MeshQuality.avg_maximum_angle(mesh)
        elif measure == "radius_ratios":
            quality = MeshQuality.avg_radius_ratios(mesh)
        elif measure == "condition_number":
            quality = MeshQuality.avg_condition_number(mesh)
        else:
            raise InputError(
                "cashocs.geometry.compute_mesh_quality",
                "measure",
                "The parameter `measure` has to be either 'min', 'minimum' or 'avg', 'average'",
            )
    else:
        raise InputError(
            "cashocs.geometry.compute_mesh_quality",
            "type",
            "The parameter `type` has to be one of 'skewness', 'maximum_angle', 'radius_ratios', or 'condition_number'",
        )

    return quality


def compute_boundary_distance(
    mesh, boundaries=None, boundary_idcs=None, tol=1e-1, max_iter=10
):
    """Computes (an approximation of) the distance to the boundary.

    The function iteratively solves the Eikonal equation to compute the distance to the
    boundary.

    The user can specify which boundaries are considered for the distance computation
    by specifying the parameters `boundaries` and `boundary_idcs`. Default is to
    consider all boundaries.

    Parameters
    ----------
    mesh : dolfin.cpp.mesh.Mesh.Mesh
        The dolfin mesh object, representing the computational domain
    boundaries : dolfin.cpp.mesh.MeshFunctionSizet.MeshFunctionSizet, optional
        A meshfunction for the boundaries, which is needed in case specific boundaries
        are targeted for the distance computation (while others are ignored), default
        is `None` (all boundaries are used)
    boundary_idcs : list[int], optional
        A list of indices which indicate, which parts of the boundaries should be used
        for the distance computation, default is `None` (all boundaries are used).
    tol : float, optional
        A tolerance for the iterative solution of the eikonal equation. Default is 1e-1.
    max_iter : int, optional
        Number of iterations for the iterative solution of the eikonal equation. Default
        is 10.

    Returns
    -------
    u_curr : dolfin.function.function.Function
        A fenics function representing an approximation of the distance to the boundary.

    """

    V = fenics.FunctionSpace(mesh, "CG", 1)
    dx = _NamedMeasure("dx", mesh)

    ksp = PETSc.KSP().create()
    ksp_options = [
        ["ksp_type", "cg"],
        ["pc_type", "hypre"],
        ["pc_hypre_type", "boomeramg"],
        ["pc_hypre_boomeramg_strong_threshold", 0.7],
        ["ksp_rtol", 1e-20],
        ["ksp_atol", 1e-50],
        ["ksp_max_it", 1000],
    ]
    _setup_petsc_options([ksp], [ksp_options])

    u = fenics.TrialFunction(V)
    v = fenics.TestFunction(V)

    u_curr = fenics.Function(V)
    u_prev = fenics.Function(V)
    norm_u_prev = fenics.sqrt(fenics.dot(fenics.grad(u_prev), fenics.grad(u_prev)))

    if (boundaries is not None) and (boundary_idcs is not None):
        if len(boundary_idcs) > 0:
            bcs = create_dirichlet_bcs(
                V, fenics.Constant(0.0), boundaries, boundary_idcs
            )
        else:
            bcs = fenics.DirichletBC(
                V, fenics.Constant(0.0), fenics.CompiledSubDomain("on_boundary")
            )
    else:
        bcs = fenics.DirichletBC(
            V, fenics.Constant(0.0), fenics.CompiledSubDomain("on_boundary")
        )

    a = fenics.dot(fenics.grad(u), fenics.grad(v)) * dx
    L = fenics.Constant(1.0) * v * dx

    A, b = _assemble_petsc_system(a, L, bcs)
    _solve_linear_problem(ksp, A, b, u_curr.vector().vec())

    L = fenics.dot(fenics.grad(u_prev) / norm_u_prev, fenics.grad(v)) * dx

    F_res = (
        pow(
            fenics.sqrt(fenics.dot(fenics.grad(u_curr), fenics.grad(u_curr)))
            - fenics.Constant(1.0),
            2,
        )
        * dx
    )

    res_0 = np.sqrt(fenics.assemble(F_res))

    for i in range(max_iter):
        u_prev.vector().vec().aypx(0.0, u_curr.vector().vec())
        A, b = _assemble_petsc_system(a, L, bcs)
        _solve_linear_problem(ksp, A, b, u_curr.vector().vec())
        res = np.sqrt(fenics.assemble(F_res))

        if res <= res_0 * tol:
            break

    return u_curr


def generate_measure(idx, measure):
    """Generates a measure based on indices.

    Generates a :py:class:`fenics.MeasureSum` or :py:class:`_EmptyMeasure <cashocs.geometry._EmptyMeasure>`
    object corresponding to ``measure`` and the subdomains / boundaries specified in idx. This
    is a convenient shortcut to writing ``dx(1) + dx(2) + dx(3)``
    in case many measures are involved.

    Parameters
    ----------
    idx : list[int]
            A list of indices for the boundary / volume markers that
            define the (new) measure.
    measure : ufl.measure.Measure
            The corresponding UFL measure.

    Returns
    -------
    ufl.measure.Measure or cashocs.geometry._EmptyMeasure
            The corresponding sum of the measures or an empty measure.

    Examples
    --------
    Here, we create a wrapper for the surface measure on the top and bottom of
    the unit square::

        from fenics import *
        import cashocs
        mesh, _, boundaries, dx, ds, _ = cashocs.regular_mesh(25)
        top_bottom_measure = cashocs.geometry.generate_measure([3,4], ds)
        assemble(1*top_bottom_measure)
    """

    if len(idx) == 0:
        out_measure = _EmptyMeasure(measure)

    else:
        out_measure = measure(idx[0])

        for i in idx[1:]:
            out_measure += measure(i)

    return out_measure


class _NamedMeasure(Measure):
    """A named integration measure, which can use names for subdomains defined in a gmsh
    .msh file.

    """

    def __init__(
        self,
        integral_type,
        domain=None,
        subdomain_id="everywhere",
        metadata=None,
        subdomain_data=None,
        physical_groups=None,
    ):
        super().__init__(
            integral_type,
            domain=domain,
            subdomain_id=subdomain_id,
            metadata=metadata,
            subdomain_data=subdomain_data,
        )
        self.physical_groups = physical_groups

    def __call__(
        self,
        subdomain_id=None,
        metadata=None,
        domain=None,
        subdomain_data=None,
        degree=None,
        scheme=None,
        rule=None,
    ):

        if isinstance(subdomain_id, int):
            return super().__call__(
                subdomain_id=subdomain_id,
                metadata=metadata,
                domain=domain,
                subdomain_data=subdomain_data,
                degree=degree,
                scheme=scheme,
                rule=rule,
            )

        elif isinstance(subdomain_id, str):
            try:
                if (
                    subdomain_id in self.physical_groups["dx"].keys()
                    and self._integral_type == "cell"
                ):
                    integer_id = self.physical_groups["dx"][subdomain_id]
                elif subdomain_id in self.physical_groups[
                    "ds"
                ].keys() and self._integral_type in [
                    "exterior_facet",
                    "interior_facet",
                ]:
                    integer_id = self.physical_groups["ds"][subdomain_id]
                else:
                    raise InputError("cashocs.geometry.NamedMeasure", "subdomain_id")
            except:
                raise InputError("cashocs.geometry.NamedMeasure", "subdomain_id")

            return super().__call__(
                subdomain_id=integer_id,
                metadata=metadata,
                domain=domain,
                subdomain_data=subdomain_data,
                degree=degree,
                scheme=scheme,
                rule=rule,
            )

        elif isinstance(subdomain_id, (list, tuple)):
            return generate_measure(subdomain_id, self)


class _EmptyMeasure(Measure):
    """Implements an empty measure (e.g. of a null set).

    This is used for automatic measure generation, e.g., if
    the fixed boundary is empty for a shape optimization problem,
    and is used to avoid case distinctions.

    Examples
    --------
    The code ::

        dm = _EmptyMeasure(dx)
        u*dm

    is equivalent to ::

        Constant(0)*u*dm

    so that ``fenics.assemble(u*dm)`` generates zeros.
    """

    def __init__(self, measure):
        """Initializes self.

        Parameters
        ----------
        measure : ufl.measure.Measure
                The underlying UFL measure.
        """

        super().__init__(measure.integral_type())

        self.measure = measure

    def __rmul__(self, other):
        """Multiplies the empty measure to the right.

        Parameters
        ----------
        other : ufl.core.expr.Expr
                A UFL expression to be integrated over an empty measure.

        Returns
        -------
        ufl.form.Form
                The resulting UFL form.
        """

        return fenics.Constant(0) * other * self.measure


class _MeshHandler:
    """Handles the mesh for shape optimization problems.

    This class implements all mesh related things for the shape optimization,
     such as transformations and remeshing. Also includes mesh quality control
     checks.
    """

    def __init__(self, shape_optimization_problem):
        """Initializes the MeshHandler object.

        Parameters
        ----------
        shape_optimization_problem : cashocs._shape_optimization.shape_optimization_problem.ShapeOptimizationProblem
            The corresponding shape optimization problem.
        """

        self.form_handler = shape_optimization_problem.form_handler
        # Namespacing
        self.mesh = self.form_handler.mesh
        self.deformation_handler = DeformationHandler(self.mesh)
        self.dx = self.form_handler.dx
        self.bbtree = self.mesh.bounding_box_tree()
        self.config = self.form_handler.config

        # setup from config
        self.volume_change = float(
            self.config.get("MeshQuality", "volume_change", fallback="inf")
        )
        self.angle_change = float(
            self.config.get("MeshQuality", "angle_change", fallback="inf")
        )

        self.mesh_quality_tol_lower = self.config.getfloat(
            "MeshQuality", "tol_lower", fallback=0.0
        )
        self.mesh_quality_tol_upper = self.config.getfloat(
            "MeshQuality", "tol_upper", fallback=1e-15
        )
        if not self.mesh_quality_tol_lower < self.mesh_quality_tol_upper:
            raise ConfigError(
                "MeshQuality",
                "tol_lower",
                "The lower remeshing tolerance has to be strictly smaller than the upper remeshing tolerance",
            )

        if self.mesh_quality_tol_lower > 0.9 * self.mesh_quality_tol_upper:
            warning(
                "You are using a lower remesh tolerance (tol_lower) close to the upper one (tol_upper). This may slow down the optimization considerably."
            )

        self.mesh_quality_measure = self.config.get(
            "MeshQuality", "measure", fallback="skewness"
        )
        if not self.mesh_quality_measure in [
            "skewness",
            "maximum_angle",
            "radius_ratios",
            "condition_number",
        ]:
            raise ConfigError(
                "MeshQuality",
                "measure",
                "Has to be one of 'skewness', 'maximum_angle', 'condition_number', or 'radius_ratios'.",
            )

        self.mesh_quality_type = self.config.get("MeshQuality", "type", fallback="min")
        if not self.mesh_quality_type in ["min", "minimum", "avg", "average"]:
            raise ConfigError(
                "MeshQuality",
                "type",
                "Has to be one of 'min', 'minimum', 'avg', or 'average'.",
            )

        self.current_mesh_quality = 1.0
        self.current_mesh_quality = compute_mesh_quality(
            self.mesh, self.mesh_quality_type, self.mesh_quality_measure
        )

        self.__setup_decrease_computation()
        self.__setup_a_priori()

        # Remeshing initializations
        self.do_remesh = self.config.getboolean("Mesh", "remesh", fallback=False)
        self.save_optimized_mesh = self.config.getboolean(
            "Output", "save_mesh", fallback=False
        )

        if self.do_remesh or self.save_optimized_mesh:
            try:
                self.mesh_directory = os.path.dirname(
                    os.path.realpath(self.config.get("Mesh", "gmsh_file"))
                )
            except configparser.Error:
                if self.do_remesh:
                    raise ConfigError(
                        "Mesh",
                        "gmsh_file",
                        "Remeshing is only available with gmsh meshes. Please specify gmsh_file.",
                    )
                elif self.save_optimized_mesh:
                    raise ConfigError(
                        "Mesh",
                        "save_mesh",
                        "The config option OptimizationRoutine.save_mesh is only available for gmsh meshes. \n"
                        "If you already use a gmsh mesh, please specify gmsh_file.",
                    )

        if self.do_remesh:
            self.temp_dict = shape_optimization_problem.temp_dict
            self.gmsh_file = self.temp_dict["gmsh_file"]
            self.remesh_counter = self.temp_dict.get("remesh_counter", 0)

            if not self.gmsh_file[-4:] == ".msh":
                raise ConfigError(
                    "Mesh", "gmsh_file", "Not a valid gmsh file. Has to end in .msh"
                )

            if not self.form_handler.has_cashocs_remesh_flag:
                self.remesh_directory = tempfile.mkdtemp(
                    prefix="cashocs_remesh_", dir=self.mesh_directory
                )
            else:
                self.remesh_directory = self.temp_dict["remesh_directory"]
            if not os.path.isdir(os.path.realpath(self.remesh_directory)):
                os.mkdir(self.remesh_directory)
            self.remesh_geo_file = f"{self.remesh_directory}/remesh.geo"

        elif self.save_optimized_mesh:
            self.gmsh_file = self.config.get("Mesh", "gmsh_file")

        # create a copy of the initial mesh file
        if self.do_remesh and self.remesh_counter == 0:
            self.gmsh_file_init = (
                f"{self.remesh_directory}/mesh_{self.remesh_counter:d}.msh"
            )
            subprocess.run(["cp", self.gmsh_file, self.gmsh_file_init], check=True)
            self.gmsh_file = self.gmsh_file_init

    def move_mesh(self, transformation):
        r"""Transforms the mesh by perturbation of identity.

        Moves the mesh according to the deformation given by

        .. math:: \text{id} + \mathcal{V}(x),

        where :math:`\mathcal{V}` is the transformation. This
        represents the perturbation of identity.

        Parameters
        ----------
        transformation : dolfin.function.function.Function
                The transformation for the mesh, a vector CG1 Function.
        """

        if not (
            transformation.ufl_element().family() == "Lagrange"
            and transformation.ufl_element().degree() == 1
        ):
            raise CashocsException("Not a valid mesh transformation")

        if not self.__test_a_priori(transformation):
            debug("Mesh transformation rejected due to a priori check.")
            return False
        else:
            success_flag = self.deformation_handler.move_mesh(
                transformation, validated_a_priori=True
            )
            self.current_mesh_quality = compute_mesh_quality(
                self.mesh, self.mesh_quality_type, self.mesh_quality_measure
            )
            return success_flag

    def revert_transformation(self):
        """Reverts the previous mesh transformation.

        This is used when the mesh quality for the resulting deformed mesh
        is not sufficient, or when the solution algorithm terminates, e.g., due
        to lack of sufficient decrease in the Armijo rule

        Returns
        -------
        None
        """

        self.deformation_handler.revert_transformation()

    def __setup_decrease_computation(self):
        """Initializes attributes and solver for the frobenius norm check.

        Returns
        -------
        None
        """

        if not self.angle_change > 0:
            raise ConfigError(
                "MeshQuality", "angle_change", "This parameter has to be positive."
            )

        self.options_frobenius = [
            ["ksp_type", "preonly"],
            ["pc_type", "jacobi"],
            ["pc_jacobi_type", "diagonal"],
            ["ksp_rtol", 1e-16],
            ["ksp_atol", 1e-20],
            ["ksp_max_it", 1000],
        ]
        self.ksp_frobenius = PETSc.KSP().create()
        _setup_petsc_options([self.ksp_frobenius], [self.options_frobenius])

        self.trial_dg0 = fenics.TrialFunction(self.form_handler.DG0)
        self.test_dg0 = fenics.TestFunction(self.form_handler.DG0)

        if not (self.angle_change == float("inf")):
            self.search_direction_container = fenics.Function(
                self.form_handler.deformation_space
            )

            self.a_frobenius = self.trial_dg0 * self.test_dg0 * self.dx
            self.L_frobenius = (
                fenics.sqrt(
                    fenics.inner(
                        fenics.grad(self.search_direction_container),
                        fenics.grad(self.search_direction_container),
                    )
                )
                * self.test_dg0
                * self.dx
            )

    def compute_decreases(self, search_direction, stepsize):
        """Estimates the number of Armijo decreases for a certain mesh quality.

        Gives a better estimation of the stepsize. The output is
        the number of Armijo decreases we have to do in order to
        get a transformation that satisfies norm(transformation)_fro <= tol,
        where transformation = stepsize*search_direction and tol is specified in
        the config file under "angle_change". Due to the linearity
        of the norm this has to be done only once, all smaller stepsizes are
        feasible w.r.t. this criterion as well.

        Parameters
        ----------
        search_direction : dolfin.function.function.Function
                The search direction in the optimization routine / descent algorithm.
        stepsize : float
                The stepsize in the descent algorithm.

        Returns
        -------
        int
                A guess for the number of "Armijo halvings" to get a better stepsize
        """

        if self.angle_change == float("inf"):
            return 0

        else:
            self.search_direction_container.vector().vec().aypx(
                0.0, search_direction.vector().vec()
            )
            A, b = _assemble_petsc_system(self.a_frobenius, self.L_frobenius)
            x = _solve_linear_problem(
                self.ksp_frobenius, A, b, ksp_options=self.options_frobenius
            )

            frobenius_norm = np.max(x[:])
            beta_armijo = self.config.getfloat(
                "OptimizationRoutine", "beta_armijo", fallback=2
            )

            return int(
                np.maximum(
                    np.ceil(
                        np.log(self.angle_change / stepsize / frobenius_norm)
                        / np.log(1 / beta_armijo)
                    ),
                    0.0,
                )
            )

    def __setup_a_priori(self):
        """Sets up the attributes and petsc solver for the a priori quality check.

        Returns
        -------
        None
        """

        self.options_prior = [
            ["ksp_type", "preonly"],
            ["pc_type", "jacobi"],
            ["pc_jacobi_type", "diagonal"],
            ["ksp_rtol", 1e-16],
            ["ksp_atol", 1e-20],
            ["ksp_max_it", 1000],
        ]
        self.ksp_prior = PETSc.KSP().create()
        _setup_petsc_options([self.ksp_prior], [self.options_prior])

        self.transformation_container = fenics.Function(
            self.form_handler.deformation_space
        )
        dim = self.mesh.geometric_dimension()

        if not self.volume_change > 1:
            raise ConfigError(
                "MeshQuality",
                "volume_change",
                "This parameter has to be larger than 1.",
            )

        self.a_prior = self.trial_dg0 * self.test_dg0 * self.dx
        self.L_prior = (
            fenics.det(
                fenics.Identity(dim) + fenics.grad(self.transformation_container)
            )
            * self.test_dg0
            * self.dx
        )

    def __test_a_priori(self, transformation):
        r"""Check the quality of the transformation before the actual mesh is moved.

        Checks the quality of the transformation. The criterion is that

        .. math:: \det(I + D \texttt{transformation})

        should neither be too large nor too small in order to achieve the best
        transformations.

        Parameters
        ----------
        transformation : dolfin.function.function.Function
                The transformation for the mesh.

        Returns
        -------
        bool
                A boolean that indicates whether the desired transformation is feasible
        """

        self.transformation_container.vector().vec().aypx(
            0.0, transformation.vector().vec()
        )
        A, b = _assemble_petsc_system(self.a_prior, self.L_prior)
        x = _solve_linear_problem(self.ksp_prior, A, b, ksp_options=self.options_prior)

        min_det = np.min(x[:])
        max_det = np.max(x[:])

        return (min_det >= 1 / self.volume_change) and (max_det <= self.volume_change)

    def __generate_remesh_geo(self, input_mesh_file):
        """Generates a .geo file used for remeshing.

        The .geo file is generated via the original .geo file for the
        initial geometry, so that mesh size fields are correctly given
        for the remeshing.

        Parameters
        ----------
        input_mesh_file : str
                Path to the mesh file used for generating the new .geo file

        Returns
        -------
        None
        """

        with open(self.remesh_geo_file, "w") as file:
            temp_name = os.path.split(input_mesh_file)[1]

            file.write(f"Merge '{temp_name}';\n")
            file.write("CreateGeometry;\n")
            file.write("\n")

            geo_file = self.temp_dict["geo_file"]
            with open(geo_file, "r") as f:
                for line in f:
                    if line[0].islower():
                        # if line[:2] == 'lc':
                        file.write(line)
                    if line[:5] == "Field":
                        file.write(line)
                    if line[:16] == "Background Field":
                        file.write(line)
                    if line[:19] == "BoundaryLayer Field":
                        file.write(line)
                    if line[:5] == "Mesh.":
                        file.write(line)

    def __remove_gmsh_parametrizations(self, mesh_file):
        """Removes the parametrizations section from a Gmsh file.

        This is needed in case several remeshing iterations have to be
        executed.

        Parameters
        ----------
        mesh_file : str
                Path to the Gmsh file.

        Returns
        -------
        None
        """

        if not mesh_file[-4:] == ".msh":
            raise InputError(
                "cashocs.geometry.__remove_gmsh_parametrizations",
                "mesh_file",
                "Format for mesh_file is wrong, has to end in .msh",
            )

        temp_location = f"{mesh_file[:-4]}_temp.msh"

        with open(mesh_file, "r") as in_file, open(temp_location, "w") as temp_file:

            parametrizations_section = False

            for line in in_file:

                if line == "$Parametrizations\n":
                    parametrizations_section = True

                if not parametrizations_section:
                    temp_file.write(line)
                else:
                    pass

                if line == "$EndParametrizations\n":
                    parametrizations_section = False

        subprocess.run(["mv", temp_location, mesh_file], check=True)

    def clean_previous_gmsh_files(self):
        """Removes the gmsh files from the previous remeshing iterations to save disk space

        Returns
        -------
        None
        """

        gmsh_file = f"{self.remesh_directory}/mesh_{self.remesh_counter - 1:d}.msh"
        if os.path.isfile(gmsh_file):
            subprocess.run(["rm", gmsh_file], check=True)

        gmsh_pre_remesh_file = (
            f"{self.remesh_directory}/mesh_{self.remesh_counter-1:d}_pre_remesh.msh"
        )
        if os.path.isfile(gmsh_pre_remesh_file):
            subprocess.run(["rm", gmsh_pre_remesh_file], check=True)

        mesh_h5_file = f"{self.remesh_directory}/mesh_{self.remesh_counter-1:d}.h5"
        if os.path.isfile(mesh_h5_file):
            subprocess.run(["rm", mesh_h5_file], check=True)

        mesh_xdmf_file = f"{self.remesh_directory}/mesh_{self.remesh_counter-1:d}.xdmf"
        if os.path.isfile(mesh_xdmf_file):
            subprocess.run(["rm", mesh_xdmf_file], check=True)

        boundaries_h5_file = (
            f"{self.remesh_directory}/mesh_{self.remesh_counter-1:d}_boundaries.h5"
        )
        if os.path.isfile(boundaries_h5_file):
            subprocess.run(["rm", boundaries_h5_file], check=True)

        boundaries_xdmf_file = (
            f"{self.remesh_directory}/mesh_{self.remesh_counter-1:d}_boundaries.xdmf"
        )
        if os.path.isfile(boundaries_xdmf_file):
            subprocess.run(["rm", boundaries_xdmf_file], check=True)

        subdomains_h5_file = (
            f"{self.remesh_directory}/mesh_{self.remesh_counter-1:d}_subdomains.h5"
        )
        if os.path.isfile(subdomains_h5_file):
            subprocess.run(["rm", subdomains_h5_file], check=True)

        subdomains_xdmf_file = (
            f"{self.remesh_directory}/mesh_{self.remesh_counter-1:d}_subdomains.xdmf"
        )
        if os.path.isfile(subdomains_xdmf_file):
            subprocess.run(["rm", subdomains_xdmf_file], check=True)

    def remesh(self, solver):
        """Remeshes the current geometry with GMSH.

        Performs a remeshing of the geometry, and then restarts
        the optimization problem with the new mesh.

        Returns
        -------
        None
        """

        if self.do_remesh:
            self.remesh_counter += 1
            self.temp_file = (
                f"{self.remesh_directory}/mesh_{self.remesh_counter:d}_pre_remesh.msh"
            )
            write_out_mesh(self.mesh, self.gmsh_file, self.temp_file)
            self.__generate_remesh_geo(self.temp_file)

            # save the output dict (without the last entries since they are "remeshed")
            self.temp_dict["output_dict"] = {}
            self.temp_dict["output_dict"][
                "state_solves"
            ] = solver.state_problem.number_of_solves
            self.temp_dict["output_dict"][
                "adjoint_solves"
            ] = solver.adjoint_problem.number_of_solves
            self.temp_dict["output_dict"]["iterations"] = solver.iteration + 1

            self.temp_dict["output_dict"]["cost_function_value"] = solver.output_dict[
                "cost_function_value"
            ][:]
            self.temp_dict["output_dict"]["gradient_norm"] = solver.output_dict[
                "gradient_norm"
            ][:]
            self.temp_dict["output_dict"]["stepsize"] = solver.output_dict["stepsize"][
                :
            ]
            self.temp_dict["output_dict"]["MeshQuality"] = solver.output_dict[
                "MeshQuality"
            ][:]

            dim = self.mesh.geometric_dimension()

            self.new_gmsh_file = (
                f"{self.remesh_directory}/mesh_{self.remesh_counter:d}.msh"
            )

            gmsh_cmd_list = [
                "gmsh",
                self.remesh_geo_file,
                f"-{int(dim):d}",
                "-o",
                self.new_gmsh_file,
            ]
            if not self.config.getboolean("Mesh", "show_gmsh_output", fallback=False):
                subprocess.run(
                    gmsh_cmd_list,
                    check=True,
                    stdout=subprocess.DEVNULL,
                )
            else:
                subprocess.run(gmsh_cmd_list, check=True)

            self.__remove_gmsh_parametrizations(self.new_gmsh_file)

            self.temp_dict["remesh_counter"] = self.remesh_counter
            self.temp_dict["remesh_directory"] = self.remesh_directory
            self.temp_dict["result_dir"] = solver.result_dir

            self.new_xdmf_file = (
                f"{self.remesh_directory}/mesh_{self.remesh_counter:d}.xdmf"
            )

            subprocess.run(
                ["cashocs-convert", self.new_gmsh_file, self.new_xdmf_file],
                check=True,
            )

            self.clean_previous_gmsh_files()

            self.temp_dict["mesh_file"] = self.new_xdmf_file
            self.temp_dict["gmsh_file"] = self.new_gmsh_file

            self.temp_dict["OptimizationRoutine"]["iteration_counter"] = (
                solver.iteration + 1
            )
            self.temp_dict["OptimizationRoutine"][
                "gradient_norm_initial"
            ] = solver.gradient_norm_initial

            self.temp_dir = self.temp_dict["temp_dir"]

            with open(f"{self.temp_dir}/temp_dict.json", "w") as file:
                json.dump(self.temp_dict, file)

            def filter_sys_argv():
                """Filters the command line arguments for the cashocs remesh flag

                Returns
                -------
                 : list[str]
                    The filtered list of command line arguments
                """
                arg_list = sys.argv.copy()
                idx_cashocs_remesh_flag = [
                    i for i, s in enumerate(arg_list) if s == "--cashocs_remesh"
                ]
                if len(idx_cashocs_remesh_flag) > 1:
                    raise InputError(
                        "Command line options",
                        "--cashocs_remesh",
                        "The --cashocs_remesh flag should only be present once.",
                    )
                elif len(idx_cashocs_remesh_flag) == 1:
                    arg_list.pop(idx_cashocs_remesh_flag[0])

                idx_temp_dir = [i for i, s in enumerate(arg_list) if s == self.temp_dir]
                if len(idx_temp_dir) > 1:
                    raise InputError(
                        "Command line options",
                        "--temp_dir",
                        "The --temp_dir flag should only be present once.",
                    )
                elif len(idx_temp_dir) == 1:
                    arg_list.pop(idx_temp_dir[0])

                idx_temp_dir_flag = [
                    i for i, s in enumerate(arg_list) if s == "--temp_dir"
                ]
                if len(idx_temp_dir) > 1:
                    raise InputError(
                        "Command line options",
                        "--temp_dir",
                        "The --temp_dir flag should only be present once.",
                    )
                elif len(idx_temp_dir_flag) == 1:
                    arg_list.pop(idx_temp_dir_flag[0])

                return arg_list

            if not self.form_handler.has_cashocs_remesh_flag:
                os.execv(
                    sys.executable,
                    [sys.executable]
                    + filter_sys_argv()
                    + ["--cashocs_remesh"]
                    + ["--temp_dir"]
                    + [self.temp_dir],
                )
            else:
                os.execv(
                    sys.executable,
                    [sys.executable]
                    + filter_sys_argv()
                    + ["--cashocs_remesh"]
                    + ["--temp_dir"]
                    + [self.temp_dir],
                )


class DeformationHandler:
    """A class, which implements mesh deformations.

    The deformations can be due to a deformation vector field or a (piecewise) update of
    the mesh coordinates.

    """

    def __init__(self, mesh):
        """

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
            The fenics mesh which is to be deformed
        """
        self.mesh = mesh
        self.dx = _NamedMeasure("dx", self.mesh)
        self.old_coordinates = self.mesh.coordinates().copy()
        self.shape_coordinates = self.old_coordinates.shape
        self.VCG = fenics.VectorFunctionSpace(mesh, "CG", 1)
        self.DG0 = fenics.FunctionSpace(mesh, "DG", 0)
        self.bbtree = self.mesh.bounding_box_tree()
        self.__setup_a_priori()
        self.v2d = fenics.vertex_to_dof_map(self.VCG).reshape(
            (-1, self.mesh.geometry().dim())
        )
        self.d2v = fenics.dof_to_vertex_map(self.VCG)

        cells = self.mesh.cells()
        flat_cells = cells.flatten().tolist()
        self.cell_counter = Counter(flat_cells)
        self.occurrences = np.array(
            [self.cell_counter[i] for i in range(self.mesh.num_vertices())]
        )
        self.coordinates = self.mesh.coordinates()

    def __setup_a_priori(self):
        """Sets up the attributes and petsc solver for the a priori quality check.

        Returns
        -------
        None
        """

        self.options_prior = [
            ["ksp_type", "preonly"],
            ["pc_type", "jacobi"],
            ["pc_jacobi_type", "diagonal"],
            ["ksp_rtol", 1e-16],
            ["ksp_atol", 1e-20],
            ["ksp_max_it", 1000],
        ]
        self.ksp_prior = PETSc.KSP().create()
        _setup_petsc_options([self.ksp_prior], [self.options_prior])

        self.transformation_container = fenics.Function(self.VCG)
        dim = self.mesh.geometric_dimension()

        self.a_prior = (
            fenics.TrialFunction(self.DG0) * fenics.TestFunction(self.DG0) * self.dx
        )
        self.L_prior = (
            fenics.det(
                fenics.Identity(dim) + fenics.grad(self.transformation_container)
            )
            * fenics.TestFunction(self.DG0)
            * self.dx
        )

    def __test_a_priori(self, transformation):
        r"""Check the quality of the transformation before the actual mesh is moved.

        Checks the quality of the transformation. The criterion is that

        .. math:: \det(I + D \texttt{transformation})

        should neither be too large nor too small in order to achieve the best
        transformations.

        Parameters
        ----------
        transformation : dolfin.function.function.Function
            The transformation for the mesh.

        Returns
        -------
        bool
            A boolean that indicates whether the desired transformation is feasible
        """

        self.transformation_container.vector().vec().aypx(
            0.0, transformation.vector().vec()
        )
        A, b = _assemble_petsc_system(self.a_prior, self.L_prior)
        x = _solve_linear_problem(self.ksp_prior, A, b, ksp_options=self.options_prior)
        min_det = np.min(x[:])

        return min_det > 0

    def __test_a_posteriori(self):
        """Checks the quality of the transformation after the actual mesh is moved.

        Checks whether the mesh is a valid finite element mesh
        after it has been moved, i.e., if there are no overlapping
        or self intersecting elements.

        Returns
        -------
        bool
            True if the test is successful, False otherwise

        Notes
        -----
        fenics itself does not check whether the used mesh is a valid finite
        element mesh, so this check has to be done manually.
        """

        self_intersections = False
        collisions = CollisionCounter.compute_collisions(self.mesh)
        if not (collisions == self.occurrences).all():
            self_intersections = True

        if self_intersections:
            self.revert_transformation()
            debug("Mesh transformation rejected due to a posteriori check.")
            return False
        else:
            return True

    def revert_transformation(self):
        """Reverts the previous mesh transformation.

        This is used when the mesh quality for the resulting deformed mesh
        is not sufficient, or when the solution algorithm terminates, e.g., due
        to lack of sufficient decrease in the Armijo rule

        Returns
        -------
        None
        """

        self.mesh.coordinates()[:, :] = self.old_coordinates
        del self.old_coordinates
        self.bbtree.build(self.mesh)

    def move_mesh(self, transformation, validated_a_priori=False):
        r"""Transforms the mesh by perturbation of identity.

        Moves the mesh according to the deformation given by

        .. math:: \text{id} + \mathcal{V}(x),

        where :math:`\mathcal{V}` is the transformation. This
        represents the perturbation of identity.

        Parameters
        ----------
        transformation : dolfin.function.function.Function or np.ndarray
            The transformation for the mesh, a vector CG1 Function.
        validated_a_priori : bool
            A boolean flag, which indicates whether an a-priori check has
            already been performed before moving the mesh. Default is
            ``False``
        """
        if isinstance(transformation, np.ndarray):
            if not transformation.shape == self.coordinates.shape:
                raise CashocsException("Not a valid dimension for the transformation")
            else:
                coordinate_transformation = transformation
        else:
            coordinate_transformation = self.dof_to_coordinate(transformation)

        if not validated_a_priori:
            if isinstance(transformation, np.ndarray):
                dof_transformation = self.coordinate_to_dof(transformation)
            else:
                dof_transformation = transformation
            if not self.__test_a_priori(dof_transformation):
                debug(
                    "Mesh transformation rejected due to a priori check. \nReason: Transformation would result in inverted mesh elements."
                )
                return False
            else:
                self.old_coordinates = self.mesh.coordinates().copy()
                self.coordinates += coordinate_transformation
                # fenics.ALE.move(self.mesh, transformation)
                self.bbtree.build(self.mesh)

                return self.__test_a_posteriori()
        else:
            self.old_coordinates = self.mesh.coordinates().copy()
            self.coordinates += coordinate_transformation
            # fenics.ALE.move(self.mesh, transformation)
            self.bbtree.build(self.mesh)

            return self.__test_a_posteriori()

    def move_mesh_ale(self, transformation, validated_a_priori=False):
        r"""Transforms the mesh by perturbation of identity.

        Moves the mesh according to the deformation given by

        .. math:: \text{id} + \mathcal{V}(x),

        where :math:`\mathcal{V}` is the transformation. This
        represents the perturbation of identity.

        Parameters
        ----------
        transformation : dolfin.function.function.Function or np.ndarray
            The transformation for the mesh, a vector CG1 Function.
        validated_a_priori : bool
            A boolean flag, which indicates whether an a-priori check has
            already been performed before moving the mesh. Default is
            ``False``
        """

        if isinstance(transformation, np.ndarray):
            transformation = self.coordinate_to_dof(transformation)

        if not (
            transformation.ufl_element().family() == "Lagrange"
            and transformation.ufl_element().degree() == 1
        ):
            raise CashocsException("Not a valid mesh transformation")

        if not validated_a_priori:
            if not self.__test_a_priori(transformation):
                debug(
                    "Mesh transformation rejected due to a priori check. \nReason: Transformation would result in inverted mesh elements."
                )
                return False
            else:
                self.old_coordinates = self.mesh.coordinates().copy()
                fenics.ALE.move(self.mesh, transformation)
                self.bbtree.build(self.mesh)

                return self.__test_a_posteriori()
        else:
            self.old_coordinates = self.mesh.coordinates().copy()
            fenics.ALE.move(self.mesh, transformation)
            self.bbtree.build(self.mesh)

            return self.__test_a_posteriori()

    def coordinate_to_dof(self, coordinate_deformation):
        """Converts a coordinate deformation to a deformation vector field (dof based)

        Parameters
        ----------
        coordinate_deformation : np.ndarray
            The deformation for the mesh coordinates.

        Returns
        -------
        dof_deformation : dolfin.function.function.Function
            The deformation vector field.

        """

        if not (coordinate_deformation.shape == self.shape_coordinates):
            raise InputError(
                "cashocs.geometry.DeformationHandler.coordinate_to_dof",
                "coordinate_deformation",
                "Shape of coordinate deformation has to be the same as self.mesh.coordinates().shape",
            )

        dof_vector = coordinate_deformation.reshape(-1)[self.d2v]
        dof_deformation = fenics.Function(self.VCG)
        dof_deformation.vector()[:] = dof_vector

        return dof_deformation

    def dof_to_coordinate(self, dof_deformation):
        """Converts a deformation vector field (dof-based) to a coordinate based deformation.

        Parameters
        ----------
        dof_deformation : dolfin.function.function.Function
            The deformation vector field.

        Returns
        -------
        coordinate_deformation : np.ndarray
            The array which can be used to deform the mesh coordinates.

        """

        if not (
            dof_deformation.ufl_element().family() == "Lagrange"
            and dof_deformation.ufl_element().degree() == 1
        ):
            raise InputError(
                "cashocs.geometry.DeformationHandler.dof_to_coordinate",
                "dof_deformation",
                "dof_deformation has to be a piecewise linear Lagrange vector field.",
            )

        coordinate_deformation = dof_deformation.vector().vec()[self.v2d]

        return coordinate_deformation

    def assign_coordinates(self, coordinates):
        """Assigns coordinates to self.mesh.

        Parameters
        ----------
        coordinates : np.ndarray
            Array of mesh coordinates, which you want to assign.

        Returns
        -------
         : bool
            ``True`` if the assignment was possible, ``False`` if not

        """

        if not self.mesh.geometric_dimension() == coordinates.shape[1]:
            raise InputError(
                "DeformationHandler.assign_coordinates",
                "coordinates",
                "The dimension of coordinates is wrong.",
            )
        if not self.mesh.num_vertices() == coordinates.shape[0]:
            raise InputError(
                "DeformationHandler.assign_coordinates",
                "coordinates",
                "The number of vertices is wrong.",
            )
        self.old_coordinates = self.mesh.coordinates().copy()
        self.mesh.coordinates()[:, :] = coordinates[:, :]
        self.bbtree.build(self.mesh)

        return self.__test_a_posteriori()


class CollisionCounter:
    _cpp_code = """
    #include <pybind11/pybind11.h>
    #include <pybind11/eigen.h>
    #include <pybind11/stl.h>
    namespace py = pybind11;
    
    #include <dolfin/mesh/Mesh.h>
    #include <dolfin/mesh/Vertex.h>
    #include <dolfin/geometry/BoundingBoxTree.h>
    #include <dolfin/geometry/Point.h>
    
    using namespace dolfin;
    
    Eigen::VectorXi
    compute_collisions(std::shared_ptr<const Mesh> mesh)
    {
      int num_vertices;
      std::vector<unsigned int> colliding_cells;
      
      num_vertices = mesh->num_vertices();
      Eigen::VectorXi collisions(num_vertices);

      int i = 0;
      for (VertexIterator v(*mesh); !v.end(); ++v)
      {
        colliding_cells = mesh->bounding_box_tree()->compute_entity_collisions(v->point());
        collisions[i] = colliding_cells.size();
        
        ++i;
      }
      return collisions;
    }
    
    PYBIND11_MODULE(SIGNATURE, m)
    {
      m.def("compute_collisions", &compute_collisions);
    }
    """
    _cpp_object = fenics.compile_cpp_code(_cpp_code)

    def __init__(self):
        pass

    @classmethod
    def compute_collisions(cls, mesh):
        return cls._cpp_object.compute_collisions(mesh)


class MeshQuality:
    r"""A class used to compute the quality of a mesh.

    This class implements either a skewness quality measure, one based
    on the maximum angle of the elements, or one based on the radius ratios.
    All quality measures have values in :math:`[0,1]`, where 1 corresponds
    to the reference (optimal) element, and 0 corresponds to degenerate elements.

    Examples
    --------
    This class can be directly used, without any instantiation, as shown here ::

        import cashocs

        mesh, _, _, _, _, _ = cashocs.regular_mesh(10)

        min_skew = cashocs.MeshQuality.min_skewness(mesh)
        avg_skew = cashocs.MeshQuality.avg_skewness(mesh)

        min_angle = cashocs.MeshQuality.min_maximum_angle(mesh)
        avg_angle = cashocs.MeshQuality.avg_maximum_angle(mesh)

        min_rad = cashocs.MeshQuality.min_radius_ratios(mesh)
        avg_rad = cashocs.MeshQuality.avg_radius_ratios(mesh)

        min_cond = cashocs.MeshQuality.min_condition_number(mesh)
        avg_cond = cashocs.MeshQuality.avg_condition_number(mesh)

    This works analogously for any mesh compatible with FEniCS.
    """

    _cpp_code_mesh_quality = """
			#include <pybind11/pybind11.h>
			#include <pybind11/eigen.h>
			namespace py = pybind11;

			#include <dolfin/mesh/Mesh.h>
			#include <dolfin/mesh/Vertex.h>
			#include <dolfin/mesh/MeshFunction.h>
			#include <dolfin/mesh/Cell.h>
			#include <dolfin/mesh/Vertex.h>

			using namespace dolfin;


			void angles_triangle(const Cell& cell, std::vector<double>& angs)
			{
			  const Mesh& mesh = cell.mesh();
			  angs.resize(3);
			  const std::size_t i0 = cell.entities(0)[0];
			  const std::size_t i1 = cell.entities(0)[1];
			  const std::size_t i2 = cell.entities(0)[2];

			  const Point p0 = Vertex(mesh, i0).point();
			  const Point p1 = Vertex(mesh, i1).point();
			  const Point p2 = Vertex(mesh, i2).point();
			  Point e0 = p1 - p0;
			  Point e1 = p2 - p0;
			  Point e2 = p2 - p1;

			  e0 /= e0.norm();
			  e1 /= e1.norm();
			  e2 /= e2.norm();

			  angs[0] = acos(e0.dot(e1));
			  angs[1] = acos(e0.dot(e2));
			  angs[2] = acos(e1.dot(e2));
			}



			void dihedral_angles(const Cell& cell, std::vector<double>& angs)
			{
			  const Mesh& mesh = cell.mesh();
			  angs.resize(6);

			  const std::size_t i0 = cell.entities(0)[0];
			  const std::size_t i1 = cell.entities(0)[1];
			  const std::size_t i2 = cell.entities(0)[2];
			  const std::size_t i3 = cell.entities(0)[3];

			  const Point p0 = Vertex(mesh, i0).point();
			  const Point p1 = Vertex(mesh, i1).point();
			  const Point p2 = Vertex(mesh, i2).point();
			  const Point p3 = Vertex(mesh, i3).point();

			  const Point e0 = p1 - p0;
			  const Point e1 = p2 - p0;
			  const Point e2 = p3 - p0;
			  const Point e3 = p2 - p1;
			  const Point e4 = p3 - p1;

			  Point n0 = e0.cross(e1);
			  Point n1 = e0.cross(e2);
			  Point n2 = e1.cross(e2);
			  Point n3 = e3.cross(e4);

			  n0 /= n0.norm();
			  n1 /= n1.norm();
			  n2 /= n2.norm();
			  n3 /= n3.norm();

			  angs[0] = acos(n0.dot(n1));
			  angs[1] = acos(-n0.dot(n2));
			  angs[2] = acos(n1.dot(n2));
			  angs[3] = acos(n0.dot(n3));
			  angs[4] = acos(n1.dot(-n3));
			  angs[5] = acos(n2.dot(n3));
			}



			dolfin::MeshFunction<double>
			skewness(std::shared_ptr<const Mesh> mesh)
			{
			  MeshFunction<double> cf(mesh, mesh->topology().dim(), 0.0);

			  double opt_angle;
			  std::vector<double> angs;
			  std::vector<double> quals;

			  for (CellIterator cell(*mesh); !cell.end(); ++cell)
			  {
				if (cell->dim() == 2)
				{
				  quals.resize(3);
				  angles_triangle(*cell, angs);
				  opt_angle = DOLFIN_PI / 3.0;
				}
				else if (cell->dim() == 3)
				{
				  quals.resize(6);
				  dihedral_angles(*cell, angs);
				  opt_angle = acos(1.0/3.0);
				}
				else
				{
				  dolfin_error("cashocs_quality.cpp", "skewness", "Not a valid dimension for the mesh.");
				}

				for (unsigned int i = 0; i < angs.size(); ++i)
				{
				  quals[i] = 1 - std::max((angs[i] - opt_angle) / (DOLFIN_PI - opt_angle), (opt_angle - angs[i]) / opt_angle);
				}
				cf[*cell] = *std::min_element(quals.begin(), quals.end());
			  }
			  return cf;
			}



			dolfin::MeshFunction<double>
			maximum_angle(std::shared_ptr<const Mesh> mesh)
			{
			  MeshFunction<double> cf(mesh, mesh->topology().dim(), 0.0);

			  double opt_angle;
			  std::vector<double> angs;
			  std::vector<double> quals;

			  for (CellIterator cell(*mesh); !cell.end(); ++cell)
			  {
				if (cell->dim() == 2)
				{
				  quals.resize(3);
				  angles_triangle(*cell, angs);
				  opt_angle = DOLFIN_PI / 3.0;
				}
				else if (cell->dim() == 3)
				{
				  quals.resize(6);
				  dihedral_angles(*cell, angs);
				  opt_angle = acos(1.0/3.0);
				}
				else
				{
				  dolfin_error("cashocs_quality.cpp", "maximum_angle", "Not a valid dimension for the mesh.");
				}

				for (unsigned int i = 0; i < angs.size(); ++i)
				{
				  quals[i] = 1 - std::max((angs[i] - opt_angle) / (DOLFIN_PI - opt_angle), 0.0);
				}
				cf[*cell] = *std::min_element(quals.begin(), quals.end());
			  }
			  return cf;
			}

			PYBIND11_MODULE(SIGNATURE, m)
			{
			  m.def("skewness", &skewness);
			  m.def("maximum_angle", &maximum_angle);
			}

		"""
    _quality_object = fenics.compile_cpp_code(_cpp_code_mesh_quality)

    def __init__(self):
        pass

    @classmethod
    def min_skewness(cls, mesh):
        r"""Computes the minimal skewness of the mesh.

        This measure the relative distance of a triangle's angles or
        a tetrahedron's dihedral angles to the corresponding optimal
        angle. The optimal angle is defined as the angle an equilateral,
        and thus equiangular, element has. The skewness lies in
        :math:`[0,1]`, where 1 corresponds to the case of an optimal
        (equilateral) element, and 0 corresponds to a degenerate
        element. The skewness corresponding to some (dihedral) angle
        :math:`\alpha` is defined as

        .. math:: 1 - \max \left( \frac{\alpha - \alpha^*}{\pi - \alpha*} , \frac{\alpha^* - \alpha}{\alpha^* - 0} \right),

        where :math:`\alpha^*` is the corresponding angle of the reference
        element. To compute the quality measure, the minimum of this expression
        over all elements and all of their (dihedral) angles is computed.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh whose quality shall be computed.

        Returns
        -------
        float
                The skewness of the mesh.
        """

        return np.min(cls._quality_object.skewness(mesh).array())

    @classmethod
    def avg_skewness(cls, mesh):
        r"""Computes the average skewness of the mesh.

        This measure the relative distance of a triangle's angles or
        a tetrahedron's dihedral angles to the corresponding optimal
        angle. The optimal angle is defined as the angle an equilateral,
        and thus equiangular, element has. The skewness lies in
        :math:`[0,1]`, where 1 corresponds to the case of an optimal
        (equilateral) element, and 0 corresponds to a degenerate
        element. The skewness corresponding to some (dihedral) angle
        :math:`\alpha` is defined as

        .. math:: 1 - \max \left( \frac{\alpha - \alpha^*}{\pi - \alpha*} , \frac{\alpha^* - \alpha}{\alpha^* - 0} \right),

        where :math:`\alpha^*` is the corresponding angle of the reference
        element. To compute the quality measure, the average of this expression
        over all elements and all of their (dihedral) angles is computed.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh, whose quality shall be computed.

        Returns
        -------
        flat
                The average skewness of the mesh.
        """

        return np.average(cls._quality_object.skewness(mesh).array())

    @classmethod
    def min_maximum_angle(cls, mesh):
        r"""Computes the minimal quality measure based on the largest angle.

        This measures the relative distance of a triangle's angles or a
        tetrahedron's dihedral angles to the corresponding optimal
        angle. The optimal angle is defined as the angle an equilateral
        (and thus equiangular) element has. This is defined as

        .. math:: 1 - \max\left( \frac{\alpha - \alpha^*}{\pi - \alpha^*} , 0 \right),

        where :math:`\alpha` is the corresponding (dihedral) angle of the element
        and :math:`\alpha^*` is the corresponding (dihedral) angle of the reference
        element. To compute the quality measure, the minimum of this expression
        over all elements and all of their (dihedral) angles is computed.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh, whose quality shall be computed.

        Returns
        -------
        float
                The minimum value of the maximum angle quality measure.
        """

        return np.min(cls._quality_object.maximum_angle(mesh).array())

    @classmethod
    def avg_maximum_angle(cls, mesh):
        r"""Computes the average quality measure based on the largest angle.

        This measures the relative distance of a triangle's angles or a
        tetrahedron's dihedral angles to the corresponding optimal
        angle. The optimal angle is defined as the angle an equilateral
        (and thus equiangular) element has. This is defined as

        .. math:: 1 - \max\left( \frac{\alpha - \alpha^*}{\pi - \alpha^*} , 0 \right),

        where :math:`\alpha` is the corresponding (dihedral) angle of the element
        and :math:`\alpha^*` is the corresponding (dihedral) angle of the reference
        element. To compute the quality measure, the average of this expression
        over all elements and all of their (dihedral) angles is computed.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh, whose quality shall be computed.

        Returns
        -------
        float
                The average quality, based on the maximum angle measure.
        """

        return np.average(cls._quality_object.maximum_angle(mesh).array())

    @staticmethod
    def min_radius_ratios(mesh):
        r"""Computes the minimal radius ratio of the mesh.

        This measures the ratio of the element's inradius to it's circumradius,
        normalized by the geometric dimension. This is computed via

        .. math:: d \frac{r}{R},

        where :math:`d` is the spatial dimension, :math:`r` is the inradius, and :math:`R` is
        the circumradius. To compute the (global) quality measure, the minimum
        of this expression over all elements is returned.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh, whose quality shall be computed.

        Returns
        -------
        float
                The minimal radius ratio of the mesh.
        """

        return np.min(fenics.MeshQuality.radius_ratios(mesh).array())

    @staticmethod
    def avg_radius_ratios(mesh):
        r"""Computes the average radius ratio of the mesh.

        This measures the ratio of the element's inradius to it's circumradius,
        normalized by the geometric dimension. This is computed via

        .. math:: d \frac{r}{R},

        where :math:`d` is the spatial dimension, :math:`r` is the inradius, and :math:`R` is
        the circumradius. To compute the (global) quality measure, the average
        of this expression over all elements is returned.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh, whose quality shall be computed.

        Returns
        -------
        float
                The average radius ratio of the mesh.
        """

        return np.average(fenics.MeshQuality.radius_ratios(mesh).array())

    @staticmethod
    def min_condition_number(mesh):
        r"""Computes minimal mesh quality based on the condition number of the reference mapping.

        This quality criterion uses the condition number (in the Frobenius norm) of the
        (linear) mapping from the elements of the mesh to the reference element. Computes
        the minimum of the condition number over all elements.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh, whose quality shall be computed.

        Returns
        -------
        float
                The minimal condition number quality measure.
        """

        DG0 = fenics.FunctionSpace(mesh, "DG", 0)
        jac = Jacobian(mesh)
        inv = JacobianInverse(mesh)

        options = [
            ["ksp_type", "preonly"],
            ["pc_type", "jacobi"],
            ["pc_jacobi_type", "diagonal"],
            ["ksp_rtol", 1e-16],
            ["ksp_atol", 1e-20],
            ["ksp_max_it", 1000],
        ]
        ksp = PETSc.KSP().create()
        _setup_petsc_options([ksp], [options])

        dx = _NamedMeasure("dx", mesh)
        a = fenics.TrialFunction(DG0) * fenics.TestFunction(DG0) * dx
        L = (
            fenics.sqrt(fenics.inner(jac, jac))
            * fenics.sqrt(fenics.inner(inv, inv))
            * fenics.TestFunction(DG0)
            * dx
        )

        cond = fenics.Function(DG0)

        A, b = _assemble_petsc_system(a, L)
        _solve_linear_problem(ksp, A, b, cond.vector().vec(), options)
        cond.vector().apply("")
        cond.vector().vec().reciprocal()
        cond.vector().vec().scale(np.sqrt(mesh.geometric_dimension()))

        return cond.vector().vec().min()[1]

    @staticmethod
    def avg_condition_number(mesh):
        """Computes average mesh quality based on the condition number of the reference mapping.

        This quality criterion uses the condition number (in the Frobenius norm) of the
        (linear) mapping from the elements of the mesh to the reference element. Computes
        the average of the condition number over all elements.

        Parameters
        ----------
        mesh : dolfin.cpp.mesh.Mesh.Mesh
                The mesh, whose quality shall be computed.

        Returns
        -------
        float
                The average mesh quality based on the condition number.
        """

        DG0 = fenics.FunctionSpace(mesh, "DG", 0)
        jac = Jacobian(mesh)
        inv = JacobianInverse(mesh)

        options = [
            ["ksp_type", "preonly"],
            ["pc_type", "jacobi"],
            ["pc_jacobi_type", "diagonal"],
            ["ksp_rtol", 1e-16],
            ["ksp_atol", 1e-20],
            ["ksp_max_it", 1000],
        ]
        ksp = PETSc.KSP().create()
        _setup_petsc_options([ksp], [options])

        dx = _NamedMeasure("dx", mesh)
        a = fenics.TrialFunction(DG0) * fenics.TestFunction(DG0) * dx
        L = (
            fenics.sqrt(fenics.inner(jac, jac))
            * fenics.sqrt(fenics.inner(inv, inv))
            * fenics.TestFunction(DG0)
            * dx
        )

        cond = fenics.Function(DG0)

        A, b = _assemble_petsc_system(a, L)
        _solve_linear_problem(ksp, A, b, cond.vector().vec(), options)
        cond.vector().apply("")

        cond.vector().vec().reciprocal()
        cond.vector().vec().scale(np.sqrt(mesh.geometric_dimension()))

        return cond.vector().vec().sum() / cond.vector().vec().getSize()
