from ..value_set import ValueSet


class AnaphylacticReactionToCommonBakersYeast(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  an anaphylactic reaction to baker's yeast.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an anaphylactic reaction to baker's yeast.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Anaphylactic Reaction to Common Baker's Yeast"
    OID = "2.16.840.1.113883.3.464.1003.199.12.1032"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "34015007",  # Bakers' asthma (disorder)
        "419447004",  # Allergic reaction caused by flour dust (disorder)
    }


class AnaphylacticReactionToDtapVaccine(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  an anaphylactic reaction to DTaP (diphtheria, tetanus, and acellular pertussis) vaccine.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an anaphylactic reaction to DTaP vaccine.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Anaphylactic Reaction to DTaP Vaccine"
    OID = "2.16.840.1.113883.3.464.1003.199.12.1031"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "219084006",  # Adverse reaction to vaccine product containing Clostridium tetani antigen (disorder)
        "293108006",  # Adverse reaction to vaccine product containing Clostridium tetani and Corynebacterium diphtheriae antigens (disorder)
        "428281000124107",  # Anaphylaxis due to diphtheria and tetanus vaccine (disorder)
        "428291000124105",  # Anaphylaxis due to tetanus, diphtheria and acellular pertussis vaccine (disorder)
    }


class DisordersOfTheImmuneSystem(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of conditions affecting the immune system.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of disorders of the immune system including disorders of the immunoglobulins and the complement system.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Disorders of the Immune System"
    OID = "2.16.840.1.113883.3.464.1003.120.12.1001"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "D800",  # Hereditary hypogammaglobulinemia
        "D801",  # Nonfamilial hypogammaglobulinemia
        "D802",  # Selective deficiency of immunoglobulin A [IgA]
        "D803",  # Selective deficiency of immunoglobulin G [IgG] subclasses
        "D804",  # Selective deficiency of immunoglobulin M [IgM]
        "D805",  # Immunodeficiency with increased immunoglobulin M [IgM]
        "D806",  # Antibody deficiency with near-normal immunoglobulins or with hyperimmunoglobulinemia
        "D807",  # Transient hypogammaglobulinemia of infancy
        "D808",  # Other immunodeficiencies with predominantly antibody defects
        "D809",  # Immunodeficiency with predominantly antibody defects, unspecified
        "D810",  # Severe combined immunodeficiency [SCID] with reticular dysgenesis
        "D811",  # Severe combined immunodeficiency [SCID] with low T- and B-cell numbers
        "D812",  # Severe combined immunodeficiency [SCID] with low or normal B-cell numbers
        "D814",  # Nezelof's syndrome
        "D816",  # Major histocompatibility complex class I deficiency
        "D817",  # Major histocompatibility complex class II deficiency
        "D8189",  # Other combined immunodeficiencies
        "D819",  # Combined immunodeficiency, unspecified
        "D820",  # Wiskott-Aldrich syndrome
        "D821",  # Di George's syndrome
        "D822",  # Immunodeficiency with short-limbed stature
        "D823",  # Immunodeficiency following hereditary defective response to Epstein-Barr virus
        "D824",  # Hyperimmunoglobulin E [IgE] syndrome
        "D828",  # Immunodeficiency associated with other specified major defects
        "D829",  # Immunodeficiency associated with major defect, unspecified
        "D830",  # Common variable immunodeficiency with predominant abnormalities of B-cell numbers and function
        "D831",  # Common variable immunodeficiency with predominant immunoregulatory T-cell disorders
        "D832",  # Common variable immunodeficiency with autoantibodies to B- or T-cells
        "D838",  # Other common variable immunodeficiencies
        "D839",  # Common variable immunodeficiency, unspecified
        "D840",  # Lymphocyte function antigen-1 [LFA-1] defect
        "D841",  # Defects in the complement system
        "D848",  # Other specified immunodeficiencies
        "D8481",  # Immunodeficiency due to conditions classified elsewhere
        "D84821",  # Immunodeficiency due to drugs
        "D84822",  # Immunodeficiency due to external causes
        "D8489",  # Other immunodeficiencies
        "D849",  # Immunodeficiency, unspecified
        "D893",  # Immune reconstitution syndrome
        "D89810",  # Acute graft-versus-host disease
        "D89811",  # Chronic graft-versus-host disease
        "D89812",  # Acute on chronic graft-versus-host disease
        "D89813",  # Graft-versus-host disease, unspecified
        "D8982",  # Autoimmune lymphoproliferative syndrome [ALPS]
        "D89831",  # Cytokine release syndrome, grade 1
        "D89832",  # Cytokine release syndrome, grade 2
        "D89833",  # Cytokine release syndrome, grade 3
        "D89834",  # Cytokine release syndrome, grade 4
        "D89835",  # Cytokine release syndrome, grade 5
        "D89839",  # Cytokine release syndrome, grade unspecified
        "D8989",  # Other specified disorders involving the immune mechanism, not elsewhere classified
        "D899",  # Disorder involving the immune mechanism, unspecified
    }
    SNOMEDCT = {
        "18827005",  # Complement abnormality (disorder)
        "23238000",  # Common variable agammaglobulinemia (disorder)
        "24419001",  # Disorder of complement (disorder)
        "24743004",  # Complement deficiency disease (disorder)
        "36070007",  # Wiskott-Aldrich syndrome (disorder)
        "39674000",  # Familial C3B inhibitor deficiency syndrome (disorder)
        "55602000",  # Nezelof's syndrome (disorder)
        "81166004",  # Properdin deficiency disease (disorder)
        "82286005",  # Hyperimmunoglobulin M syndrome (disorder)
        "82966003",  # Hereditary angioedema (disorder)
        "88714009",  # Transient hypogammaglobulinemia of infancy (disorder)
        "190979003",  # Selective immunoglobulin A deficiency (disorder)
        "190980000",  # Selective immunoglobulin M deficiency (disorder)
        "190981001",  # Selective immunoglobulin G deficiency (disorder)
        "190996002",  # Severe combined immunodeficiency with reticular dysgenesis (disorder)
        "190997006",  # Severe combined immunodeficiency with low T- and B-cell numbers (disorder)
        "190998001",  # Severe combined immunodeficiency with low or normal B-cell numbers (disorder)
        "191001007",  # Major histocompatibility complex class I deficiency (disorder)
        "191002000",  # Major histocompatibility complex class II deficiency (disorder)
        "191011000",  # Common variable immunodeficiency with predominant abnormalities of B-cell numbers and functions (disorder)
        "191012007",  # Common variable immunodeficiency with predominant immunoregulatory T-cell disorders (disorder)
        "191013002",  # Common variable immunodeficiency with autoantibodies to B- or T-cells (disorder)
        "191018006",  # Lymphocyte function antigen-1 defect (disorder)
        "234416002",  # X-linked hypogammaglobulinemia (disorder)
        "234532001",  # Immunodeficiency disorder (disorder)
        "234562007",  # Anti-protein antibody deficiency (disorder)
        "234593008",  # Classical complement pathway abnormality (disorder)
        "234594002",  # Complement 1q deficiency (disorder)
        "234595001",  # Complement 1q beta chain deficiency (disorder)
        "234596000",  # Complement 1q dysfunction (disorder)
        "234597009",  # Complement 1r deficiency (disorder)
        "234598004",  # Complement 1s deficiency (disorder)
        "234599007",  # Complement 2 deficiency (disorder)
        "234600005",  # Complement 4 deficiency (disorder)
        "234601009",  # Complement 4A deficiency (disorder)
        "234602002",  # Complement 4B deficiency (disorder)
        "234604001",  # Alternative pathway deficiency (disorder)
        "234605000",  # Factor B deficiency (disorder)
        "234607008",  # Factor D deficiency (disorder)
        "234608003",  # Terminal component deficiency (disorder)
        "234609006",  # Complement 5 deficiency (disorder)
        "234611002",  # Complement 6 deficiency (disorder)
        "234612009",  # Complement 7 deficiency (disorder)
        "234613004",  # Combined complement 6 and 7 deficiencies (disorder)
        "234614005",  # Complement 8 beta chain deficiency (disorder)
        "234615006",  # Complement 8 beta chain dysfunction (disorder)
        "234616007",  # Complement 8 alpha-gamma deficiency (disorder)
        "234617003",  # Complement 9 deficiency (disorder)
        "234618008",  # Complement regulatory factor defect (disorder)
        "234619000",  # Hereditary C1 esterase inhibitor deficiency - deficient factor (disorder)
        "234620006",  # Hereditary C1 esterase inhibitor deficiency - dysfunctional factor (disorder)
        "234621005",  # Factor I deficiency (disorder)
        "234622003",  # Factor H deficiency (disorder)
        "234623008",  # Complement 4 binding protein deficiency (disorder)
        "234624002",  # Decay accelerating factor deficiency (disorder)
        "234625001",  # Homologous restriction factor deficiency (disorder)
        "234626000",  # Complement 5a inhibitor deficiency (disorder)
        "234627009",  # Anaphylotoxin inactivator deficiency (disorder)
        "234628004",  # Complement receptor deficiency (disorder)
        "234629007",  # Complement receptor 1 deficiency (disorder)
        "234630002",  # Complement receptor 3 deficiency (disorder)
        "234631003",  # Immunodeficiency with major anomalies (disorder)
        "234646005",  # Graft-versus-host disease (disorder)
        "263661007",  # Complement 5 dysfunction (disorder)
        "402355000",  # Acute graft-versus-host disease (disorder)
        "402356004",  # Chronic graft-versus-host disease (disorder)
        "402357008",  # Lichenoid chronic graft-versus-host disease (disorder)
        "402358003",  # Sclerodermoid chronic graft-versus-host disease (disorder)
        "402360001",  # Materno-fetal graft-versus-host disease (disorder)
        "403835002",  # X-linked hyper-immunoglobulin M syndrome (disorder)
        "403836001",  # Autosomal recessive hyperimmunoglobulin M syndrome (disorder)
        "403837005",  # Wiskott-Aldrich autosomal dominant variant syndrome (disorder)
        "414029004",  # Disorder of immune function (disorder)
        "426202004",  # Immune reconstitution syndrome (disorder)
        "427167008",  # Hereditary angioedema with normal C1 esterase inhibitor activity (disorder)
        "442459007",  # Combined immunodeficiency disease (disorder)
        "444547006",  # Graft versus host disease of skin (disorder)
        "449187006",  # Kappa light chain deficiency (disorder)
        "702444009",  # Autoimmune lymphoproliferative syndrome (disorder)
        "719156006",  # X-linked intellectual disability with hypogammaglobulinemia and progressive neurological deterioration syndrome (disorder)
        "720853005",  # Severe combined immunodeficiency, microcephaly, growth retardation, sensitivity to ionizing radiation syndrome (disorder)
        "722067005",  # Severe combined immunodeficiency with hypereosinophilia (disorder)
        "763623001",  # Severe combined immunodeficiency due to cytidine 5-prime triphosphate synthetase 1 deficiency (disorder)
        "765188009",  # Severe combined immunodeficiency due to complete recombination-activating gene 1 and/or recombination-activating gene 2 deficiency (disorder)
        "771078002",  # Immunodeficiency due to mannan binding lectin serine peptidase 2 deficiency (disorder)
        "771443008",  # Complement component 3 deficiency (disorder)
        "778027003",  # Primary CD59 deficiency (disorder)
        "783007005",  # Recurrent Neisseria infection due to factor D deficiency (disorder)
        "783248004",  # Hyperimmunoglobulin M syndrome with susceptibility to opportunistic infection (disorder)
        "783249007",  # Hyperimmunoglobulin M syndrome without susceptibility to opportunistic infection (disorder)
        "783617001",  # Severe combined immunodeficiency due to lymphocyte-specific protein-tyrosine kinase deficiency (disorder)
        "783621008",  # Immunodeficiency with factor I anomaly (disorder)
        "789777007",  # Short-limb skeletal dysplasia with severe combined immunodeficiency (disorder)
        "846606007",  # Graft versus host disease of liver (disorder)
        "121121000119106",  # Acute graft-versus-host reaction following bone marrow transplant (disorder)
        "121131000119109",  # Chronic graft versus host disease after transplantation of bone marrow (disorder)
        "128631000119109",  # Chronic graft-versus-host disease following kidney transplant (disorder)
    }


class EncephalopathyDueToChildhoodVaccination(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  encephalopathy due to childhood vaccination.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of encephalopathy due to childhood vaccination.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Encephalopathy due to Childhood Vaccination"
    OID = "2.16.840.1.113883.3.464.1003.114.12.1007"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "G0432",  # Postimmunization acute necrotizing hemorrhagic encephalopathy
    }
    SNOMEDCT = {
        "192704009",  # Post-immunization encephalitis (disorder)
        "192705005",  # Post-bacillus Calmette-Guerin vaccination encephalitis (disorder)
        "192706006",  # Post typhoid vaccination encephalitis (disorder)
        "192707002",  # Post paratyphoid vaccination encephalitis (disorder)
        "192708007",  # Post cholera vaccination encephalitis (disorder)
        "192709004",  # Post plague vaccination encephalitis (disorder)
        "192710009",  # Post tetanus vaccination encephalitis (disorder)
        "192711008",  # Post diphtheria vaccination encephalitis (disorder)
        "192712001",  # Post pertussis vaccination encephalitis (disorder)
        "192713006",  # Post smallpox vaccination encephalitis (disorder)
        "192714000",  # Post rabies vaccination encephalitis (disorder)
        "192715004",  # Post typhus vaccination encephalitis (disorder)
        "192716003",  # Post yellow fever vaccination encephalitis (disorder)
        "192717007",  # Post measles vaccination encephalitis (disorder)
        "192718002",  # Post polio vaccination encephalitis (disorder)
        "192719005",  # Post mumps vaccination encephalitis (disorder)
        "192720004",  # Post rubella vaccination encephalitis (disorder)
        "192721000",  # Post influenza vaccination encephalitis (disorder)
        "192722007",  # Post hepatitis A vaccination encephalitis (disorder)
        "192723002",  # Post hepatitis B vaccination encephalitis (disorder)
        "192724008",  # Post mixed vaccination encephalitis (disorder)
    }


class Hiv(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of human immunodeficiency virus (HIV) disease.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of HIV and acquired immunodeficiency syndrome.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS153v10, CMS117v10, CMS646v2
    """

    VALUE_SET_NAME = "HIV"
    OID = "2.16.840.1.113883.3.464.1003.120.12.1003"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "B20",  # Human immunodeficiency virus [HIV] disease
        "B9735",  # Human immunodeficiency virus, type 2 [HIV 2] as the cause of diseases classified elsewhere
        "Z21",  # Asymptomatic human immunodeficiency virus [HIV] infection status
    }
    ICD9CM = {
        "042",  # Human immunodeficiency virus [HIV] disease
        "07953",  # Human immunodeficiency virus, type 2 [HIV-2]
        "V08",  # Asymptomatic human immunodeficiency virus [HIV] infection status
    }
    SNOMEDCT = {
        "40780007",  # Human immunodeficiency virus I infection (disorder)
        "48794007",  # Human immunodeficiency virus infection with infectious mononucleosis-like syndrome (disorder)
        "52079000",  # Congenital human immunodeficiency virus infection (disorder)
        "62479008",  # Acquired immune deficiency syndrome (disorder)
        "79019005",  # Human immunodeficiency virus II infection (disorder)
        "86406008",  # Human immunodeficiency virus infection (disorder)
        "87117006",  # Human immunodeficiency virus infection with acute lymphadenitis (disorder)
        "91923005",  # Acquired immunodeficiency syndrome virus infection associated with pregnancy (disorder)
        "91947003",  # Asymptomatic human immunodeficiency virus infection (disorder)
        "91948008",  # Asymptomatic human immunodeficiency virus infection in pregnancy (disorder)
        "111880001",  # Acute human immunodeficiency virus infection (disorder)
        "165816005",  # Human immunodeficiency virus positive (finding)
        "186706006",  # Human immunodeficiency virus infection constitutional disease (disorder)
        "186707002",  # Human immunodeficiency virus infection with neurological disease (disorder)
        "186708007",  # Human immunodeficiency virus infection with secondary clinical infectious disease (disorder)
        "230180003",  # Human immunodeficiency virus leukoencephalopathy (disorder)
        "230202002",  # Vacuolar myelopathy (disorder)
        "230598008",  # Neuropathy caused by human immunodeficiency virus (disorder)
        "235009000",  # Human immunodeficiency virus-associated periodontitis (disorder)
        "235726002",  # Human immunodeficiency virus enteropathy (disorder)
        "236406007",  # Acquired immune deficiency syndrome-related nephropathy (disorder)
        "240103002",  # Human immunodeficiency virus myopathy (disorder)
        "276665006",  # Congenital acquired immune deficiency syndrome (disorder)
        "276666007",  # Congenital human immunodeficiency virus positive status syndrome (disorder)
        "315019000",  # Human immunodeficiency virus infection with aseptic meningitis (disorder)
        "397763006",  # Human immunodeficiency virus encephalopathy (disorder)
        "398329009",  # Human immunodeficiency virus encephalitis (disorder)
        "402901009",  # Oral hairy leukoplakia associated with human immunodeficiency virus disease (disorder)
        "405631006",  # Pediatric human immunodeficiency virus infection (disorder)
        "406109008",  # Human immunodeficiency virus enzyme-linked immunosorbent assay test positive (finding)
        "414604009",  # Leukoplakia of tongue associated with human immunodeficiency virus disease (disorder)
        "416729007",  # Neutropenia associated with acquired immunodeficiency syndrome (disorder)
        "420244003",  # Encephalitis associated with acquired immunodeficiency syndrome (disorder)
        "420281004",  # Skin rash associated with acquired immunodeficiency syndrome (disorder)
        "420302007",  # Reticulosarcoma associated with acquired immunodeficiency syndrome (disorder)
        "420308006",  # Retinal vascular changes associated with acquired immunodeficiency syndrome (disorder)
        "420321004",  # Intestinal malabsorption associated with acquired immunodeficiency syndrome (disorder)
        "420384005",  # Hematopoietic system disease associated with acquired immunodeficiency syndrome (disorder)
        "420395004",  # Acute endocarditis associated with acquired immunodeficiency syndrome (disorder)
        "420403001",  # Pneumocystosis associated with acquired immunodeficiency syndrome (disorder)
        "420452002",  # Myelopathy associated with acquired immunodeficiency syndrome (disorder)
        "420524008",  # Kaposi's sarcoma associated with acquired immunodeficiency syndrome (disorder)
        "420543008",  # Anemia associated with acquired immunodeficiency syndrome (disorder)
        "420544002",  # Bacterial pneumonia associated with acquired immunodeficiency syndrome (disorder)
        "420549007",  # Salivary gland disease associated with acquired immunodeficiency syndrome (disorder)
        "420554003",  # Progressive multifocal leukoencephalopathy associated with acquired immunodeficiency syndrome (disorder)
        "420614009",  # Organic dementia associated with acquired immunodeficiency syndrome (disorder)
        "420658009",  # Radiculitis associated with acquired immunodeficiency syndrome (disorder)
        "420687005",  # Ill-defined intestinal infection associated with acquired immunodeficiency syndrome (disorder)
        "420691000",  # Nutritional deficiency associated with acquired immunodeficiency syndrome (disorder)
        "420718004",  # Central nervous system demyelinating disease associated with acquired immunodeficiency syndrome (disorder)
        "420721002",  # Acquired immunodeficiency syndrome-associated disorder (disorder)
        "420764009",  # Salmonella infection associated with acquired immunodeficiency syndrome (disorder)
        "420774007",  # Organic brain syndrome associated with acquired immunodeficiency syndrome (disorder)
        "420787001",  # Pneumococcal pneumonia associated with acquired immunodeficiency syndrome (disorder)
        "420801006",  # Malaise associated with acquired immunodeficiency syndrome (disorder)
        "420818005",  # Mycobacteriosis associated with acquired immunodeficiency syndrome (disorder)
        "420877009",  # Dermatomycosis associated with acquired immunodeficiency syndrome (disorder)
        "420900006",  # Fatigue associated with acquired immunodeficiency syndrome (disorder)
        "420938005",  # Subacute endocarditis associated with acquired immunodeficiency syndrome (disorder)
        "420945005",  # Histoplasmosis associated with acquired immunodeficiency syndrome (disorder)
        "421020000",  # Microsporidiosis associated with acquired immunodeficiency syndrome (disorder)
        "421023003",  # Presenile dementia associated with acquired immunodeficiency syndrome (disorder)
        "421047005",  # Candidiasis of lung associated with acquired immunodeficiency syndrome (disorder)
        "421077004",  # Disseminated candidiasis associated with acquired immunodeficiency syndrome (disorder)
        "421102007",  # Aplastic anemia associated with acquired immunodeficiency syndrome (disorder)
        "421230000",  # Hepatomegaly associated with acquired immunodeficiency syndrome (disorder)
        "421272004",  # Subacute myocarditis associated with acquired immunodeficiency syndrome (disorder)
        "421283008",  # Primary lymphoma of brain associated with acquired immunodeficiency syndrome (disorder)
        "421312009",  # Agranulocytosis associated with acquired immunodeficiency syndrome (disorder)
        "421315006",  # Myelitis associated with acquired immunodeficiency syndrome (disorder)
        "421394009",  # Skin disorder associated with acquired immunodeficiency syndrome (disorder)
        "421403008",  # Cryptococcosis associated with acquired immunodeficiency syndrome (disorder)
        "421415007",  # Subacute adenoviral encephalitis associated with acquired immunodeficiency syndrome (disorder)
        "421431004",  # Nocardiosis associated with acquired immunodeficiency syndrome (disorder)
        "421454008",  # Infectious gastroenteritis associated with acquired immunodeficiency syndrome (disorder)
        "421460008",  # Retinopathy associated with acquired immunodeficiency syndrome (disorder)
        "421508002",  # Viral pneumonia associated with acquired immunodeficiency syndrome (disorder)
        "421510000",  # Isosporiasis associated with acquired immunodeficiency syndrome (disorder)
        "421529006",  # Dementia associated with acquired immunodeficiency syndrome (disorder)
        "421571007",  # Tuberculosis associated with acquired immunodeficiency syndrome (disorder)
        "421597001",  # Polyneuropathy associated with acquired immunodeficiency syndrome (disorder)
        "421660003",  # Failure to thrive in infant associated with acquired immunodeficiency syndrome (disorder)
        "421666009",  # Toxoplasmosis associated with acquired immunodeficiency syndrome (disorder)
        "421671002",  # Pneumonia associated with acquired immunodeficiency syndrome (disorder)
        "421695000",  # Abnormal weight loss associated with acquired immunodeficiency syndrome (finding)
        "421706001",  # Blindness associated with acquired immunodeficiency syndrome (disorder)
        "421708000",  # Infective arthritis associated with acquired immunodeficiency syndrome (disorder)
        "421710003",  # Candidiasis of mouth associated with acquired immunodeficiency syndrome (disorder)
        "421766003",  # Thrombocytopenia associated with acquired immunodeficiency syndrome (disorder)
        "421827003",  # Encephalopathy associated with acquired immunodeficiency syndrome (disorder)
        "421851008",  # Acquired hemolytic anemia associated with acquired immunodeficiency syndrome (disorder)
        "421874007",  # Respiratory disorder associated with acquired immunodeficiency syndrome (disorder)
        "421883002",  # Strongyloidiasis associated with acquired immunodeficiency syndrome (disorder)
        "421929001",  # Myocarditis associated with acquired immunodeficiency syndrome (disorder)
        "421983003",  # Noninfectious gastroenteritis associated with acquired immunodeficiency syndrome (disorder)
        "421998001",  # Central nervous disorder associated with acquired immunodeficiency syndrome (disorder)
        "422003001",  # Cachexia associated with acquired immunodeficiency syndrome (finding)
        "422012004",  # Neuritis associated with acquired immunodeficiency syndrome (disorder)
        "422074008",  # Coccidiosis associated with acquired immunodeficiency syndrome (disorder)
        "422089004",  # Encephalomyelitis associated with acquired immunodeficiency syndrome (disorder)
        "422127002",  # Herpes zoster associated with acquired immunodeficiency syndrome (disorder)
        "422136003",  # Neuralgia associated with acquired immunodeficiency syndrome (disorder)
        "422177004",  # Dyspnea associated with acquired immunodeficiency syndrome (disorder)
        "422189002",  # Low vision associated with acquired immunodeficiency syndrome (disorder)
        "422194002",  # Hyperhidrosis associated with acquired immunodeficiency syndrome (disorder)
        "422282000",  # Malignant neoplasm associated with acquired immunodeficiency syndrome (disorder)
        "422337001",  # Coccidioidomycosis associated with acquired immunodeficiency syndrome (disorder)
        "432218001",  # History of asymptomatic human immunodeficiency virus infection (situation)
        "442537007",  # Non-Hodgkin lymphoma associated with Human immunodeficiency virus infection (disorder)
        "442662004",  # Copies of human immunodeficiency virus ribonucleic acid above reference range (finding)
        "445945000",  # Infectious disease associated with acquired immune deficiency syndrome (disorder)
        "697904001",  # Pulmonary arterial hypertension associated with human immunodeficiency virus infection (disorder)
        "697965002",  # Cholangitis associated with acquired immunodeficiency syndrome (disorder)
        "699433000",  # Human immunodeficiency virus carrier (finding)
        "700053002",  # Sepsis associated with acquired immunodeficiency syndrome (disorder)
        "713260006",  # Subacute adenoviral encephalitis co-occurrent with human immunodeficiency virus infection (disorder)
        "713275003",  # Splenomegaly co-occurrent with human immunodeficiency virus infection (disorder)
        "713278001",  # Neuralgia co-occurrent with human immunodeficiency virus infection (disorder)
        "713297001",  # Candidiasis of esophagus co-occurrent with human immunodeficiency virus infection (disorder)
        "713298006",  # Heart disease co-occurrent with human immunodeficiency virus infection (disorder)
        "713299003",  # Disorder of eye proper co-occurrent with human immunodeficiency virus infection (disorder)
        "713300006",  # Disorder of gastrointestinal tract co-occurrent with human immunodeficiency virus infection (disorder)
        "713316008",  # Eruption of skin co-occurrent with human immunodeficiency virus infection (disorder)
        "713318009",  # Myocarditis co-occurrent with human immunodeficiency virus infection (disorder)
        "713320007",  # Radiculitis co-occurrent with human immunodeficiency virus infection (disorder)
        "713325002",  # Primary cerebral lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        "713339002",  # Infection caused by Strongyloides co-occurrent with human immunodeficiency virus infection (disorder)
        "713340000",  # Disorder of skin co-occurrent with human immunodeficiency virus infection (disorder)
        "713341001",  # Myelitis co-occurrent with human immunodeficiency virus infection (disorder)
        "713342008",  # Infection caused by Salmonella co-occurrent with human immunodeficiency virus infection (disorder)
        "713349004",  # Anemia co-occurrent with human immunodeficiency virus infection (disorder)
        "713444005",  # Hemophagocytic syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        "713445006",  # Disseminated infection caused by Strongyloides co-occurrent with human immunodeficiency virus infection (disorder)
        "713446007",  # Chronic infection caused by herpes simplex virus co-occurrent with human immunodeficiency virus infection (disorder)
        "713483007",  # Reticulosarcoma co-occurrent with human immunodeficiency virus infection (disorder)
        "713484001",  # Disorder of respiratory system co-occurrent with human immunodeficiency virus infection (disorder)
        "713487008",  # Progressive multifocal leukoencephalopathy co-occurrent with human immunodeficiency virus infection (disorder)
        "713488003",  # Presenile dementia co-occurrent with human immunodeficiency virus infection (disorder)
        "713489006",  # Polyneuropathy co-occurrent with human immunodeficiency virus infection (disorder)
        "713490002",  # Infection caused by Pneumocystis co-occurrent with human immunodeficiency virus infection (disorder)
        "713491003",  # Organic brain syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        "713497004",  # Candidiasis of mouth co-occurrent with human immunodeficiency virus infection (disorder)
        "713503007",  # Disorder of spinal cord co-occurrent with human immunodeficiency virus infection (disorder)
        "713504001",  # Disorder of kidney co-occurrent with human immunodeficiency virus infection (disorder)
        "713505000",  # Gastrointestinal malabsorption syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        "713506004",  # Neuritis co-occurrent with human immunodeficiency virus infection (disorder)
        "713507008",  # Lymphadenopathy co-occurrent with human immunodeficiency virus infection (disorder)
        "713508003",  # Aplastic anemia co-occurrent with human immunodeficiency virus infection (disorder)
        "713510001",  # Enlargement of liver co-occurrent with human immunodeficiency virus infection (disorder)
        "713511002",  # Acute endocarditis co-occurrent with human immunodeficiency virus infection (disorder)
        "713523008",  # Cardiomyopathy co-occurrent with human immunodeficiency virus infection (disorder)
        "713526000",  # Recurrent bacterial pneumonia co-occurrent with human immunodeficiency virus infection (disorder)
        "713527009",  # Disorder of peripheral nervous system co-occurrent with human immunodeficiency virus infection (disorder)
        "713530002",  # Agranulocytosis co-occurrent with human immunodeficiency virus infection (disorder)
        "713531003",  # Visual impairment co-occurrent with human immunodeficiency virus infection (disorder)
        "713532005",  # Infective arthritis co-occurrent with human immunodeficiency virus infection (disorder)
        "713533000",  # Acquired hemolytic anemia co-occurrent with human immunodeficiency virus infection (disorder)
        "713543002",  # Demyelinating disease of central nervous system co-occurrent with human immunodeficiency virus infection (disorder)
        "713544008",  # Bacterial pneumonia co-occurrent with human immunodeficiency virus infection (disorder)
        "713545009",  # Infection caused by Nocardia co-occurrent with human immunodeficiency virus infection (disorder)
        "713546005",  # Isosporiasis co-occurrent with human immunodeficiency virus infection (disorder)
        "713570009",  # Infectious gastroenteritis co-occurrent with human immunodeficiency virus infection (disorder)
        "713571008",  # Disorder of central nervous system co-occurrent with human immunodeficiency virus infection (disorder)
        "713572001",  # Malignant neoplastic disease co-occurrent with human immunodeficiency virus infection (disorder)
        "713695001",  # Nephrotic syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        "713696000",  # Renal failure syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        "713718006",  # Diffuse non-Hodgkin immunoblastic lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        "713722001",  # Infection caused by Cytomegalovirus co-occurrent with human immunodeficiency virus infection (disorder)
        "713729005",  # Infection caused by Coccidia co-occurrent with human immunodeficiency virus infection (disorder)
        "713730000",  # Infection caused by herpes simplex virus co-occurrent with human immunodeficiency virus infection (disorder)
        "713731001",  # Pyrexia of unknown origin co-occurrent with human immunodeficiency virus infection (disorder)
        "713732008",  # Infection caused by Aspergillus co-occurrent with human immunodeficiency virus infection (disorder)
        "713733003",  # Infection caused by herpes zoster virus co-occurrent with human immunodeficiency virus infection (disorder)
        "713734009",  # Infection caused by Dermatophyte co-occurrent with human immunodeficiency virus infection (disorder)
        "713742005",  # Human immunodeficiency virus antibody positive (finding)
        "713844000",  # Dementia co-occurrent with human immunodeficiency virus infection (disorder)
        "713845004",  # Infection caused by Cryptosporidium co-occurrent with human immunodeficiency virus infection (disorder)
        "713880000",  # Opportunistic mycosis co-occurrent with human immunodeficiency virus infection (disorder)
        "713881001",  # Infection caused by Microsporidia co-occurrent with human immunodeficiency virus infection (disorder)
        "713887002",  # Focal segmental glomerulosclerosis co-occurrent with human immunodeficiency virus infection (disorder)
        "713897006",  # Burkitt lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        "713964006",  # Multidermatomal infection caused by Herpes zoster co-occurrent with human immunodeficiency virus infection (disorder)
        "713967004",  # Disseminated atypical infection caused by Mycobacterium co-occurrent with human immunodeficiency virus infection (disorder)
        "714083007",  # Recurrent salmonella sepsis co-occurrent with human immunodeficiency virus infection (disorder)
        "714464009",  # Immune reconstitution inflammatory syndrome caused by human immunodeficiency virus infection (disorder)
        "719522009",  # Infection of upper respiratory tract caused by Candida co-occurrent with human immunodeficiency virus infection (disorder)
        "721166000",  # Human immunodeficiency virus complicating pregnancy childbirth and the puerperium (disorder)
        "722557007",  # Parkinsonism due to human immunodeficiency virus infection (disorder)
        "733834006",  # Invasive carcinoma of uterine cervix co-occurrent with human immunodeficiency virus infection (disorder)
        "733835007",  # Extrapulmonary tuberculosis co-occurrent with human immunodeficiency virus infection (disorder)
        "735521001",  # Human immunodeficiency virus World Health Organization 2007 stage 1 co-occurrent with tuberculosis (disorder)
        "735522008",  # Human immunodeficiency virus World Health Organization 2007 stage 1 co-occurrent with malaria (disorder)
        "735523003",  # Human immunodeficiency virus World Health Organization 2007 stage 2 co-occurrent with tuberculosis (disorder)
        "735524009",  # Human immunodeficiency virus World Health Organization 2007 stage 2 co-occurrent with malaria (disorder)
        "735525005",  # Human immunodeficiency virus World Health Organization 2007 stage 3 co-occurrent with tuberculosis (disorder)
        "735526006",  # Human immunodeficiency virus World Health Organization 2007 stage 3 co-occurrent with malaria (disorder)
        "735527002",  # Human immunodeficiency virus World Health Organization 2007 stage 4 co-occurrent with tuberculosis (disorder)
        "735528007",  # Human immunodeficiency virus World Health Organization 2007 stage 4 co-occurrent with malaria (disorder)
        "771119002",  # Infection caused by Coccidia co-occurrent with acquired immunodeficiency syndrome (disorder)
        "771126002",  # Infection caused by Toxoplasma gondii co-occurrent with acquired immunodeficiency syndrome (disorder)
        "771127006",  # Infection caused by Isospora co-occurrent with acquired immunodeficiency syndrome (disorder)
        "838377003",  # Chronic hepatitis C co-occurrent with human immunodeficiency virus infection (disorder)
        "840442003",  # Encephalitis caused by human immunodeficiency virus type 2 (disorder)
        "840498003",  # Encephalitis caused by human immunodeficiency virus type 1 (disorder)
        "81000119104",  # Symptomatic human immunodeficiency virus infection (disorder)
        "72621000119104",  # Human immunodeficiency virus (HIV) II infection category B1 (disorder)
        "72631000119101",  # Human immunodeficiency virus (HIV) II infection category B2 (disorder)
        "76981000119106",  # Human immunodeficiency virus (HIV) infection category B1 (disorder)
        "76991000119109",  # Human immunodeficiency virus (HIV) infection category B2 (disorder)
        "80191000119101",  # Symptomatic human immunodeficiency virus I infection (disorder)
        "90681000119107",  # Asymptomatic human immunodeficiency virus A1 infection (disorder)
        "90691000119105",  # Asymptomatic human immunodeficiency virus A2 infection (disorder)
        "10746341000119109",  # Acquired immune deficiency syndrome complicating childbirth (disorder)
        "10755671000119100",  # Human immunodeficiency virus in mother complicating childbirth (disorder)
        "15928141000119107",  # Cognitive impairment co-occurrent and due to human immunodeficiency virus infection (disorder)
    }


class HepatitisA(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of Hepatitis A infection.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of Hepatitis A.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS347v5, CMS117v10
    """

    VALUE_SET_NAME = "Hepatitis A"
    OID = "2.16.840.1.113883.3.464.1003.110.12.1024"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "B150",  # Hepatitis A with hepatic coma
        "B159",  # Hepatitis A without hepatic coma
    }
    SNOMEDCT = {
        "16060001",  # Hepatic coma due to viral hepatitis A (disorder)
        "18917003",  # Acute fulminating type A viral hepatitis (disorder)
        "25102003",  # Acute type A viral hepatitis (disorder)
        "40468003",  # Viral hepatitis, type A (disorder)
        "43634002",  # Relapsing type A viral hepatitis (disorder)
        "79031007",  # Anicteric type A viral hepatitis (disorder)
        "111879004",  # Viral hepatitis A without hepatic coma (disorder)
        "165997004",  # Hepatitis A test positive (finding)
        "206373002",  # Congenital hepatitis A infection (disorder)
        "278971009",  # Hepatitis A immune (finding)
        "310875001",  # Hepatitis A - current infection (finding)
        "424758008",  # Viral hepatitis A without hepatic coma, without hepatitis delta (disorder)
        "428030001",  # History of hepatitis A (situation)
        "105801000119103",  # Immunity to hepatitis A by positive serology (finding)
    }


class HepatitisB(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for acute or chronic Hepatitis B infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of acute or chronic Hepatitis B.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS347v5, CMS117v10
    """

    VALUE_SET_NAME = "Hepatitis B"
    OID = "2.16.840.1.113883.3.67.1.101.1.269"
    DEFINITION_VERSION = "20170825"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "B160",  # Acute hepatitis B with delta-agent with hepatic coma
        "B161",  # Acute hepatitis B with delta-agent without hepatic coma
        "B162",  # Acute hepatitis B without delta-agent with hepatic coma
        "B169",  # Acute hepatitis B without delta-agent and without hepatic coma
        "B170",  # Acute delta-(super) infection of hepatitis B carrier
        "B180",  # Chronic viral hepatitis B with delta-agent
        "B181",  # Chronic viral hepatitis B without delta-agent
        "B1910",  # Unspecified viral hepatitis B without hepatic coma
        "B1911",  # Unspecified viral hepatitis B with hepatic coma
    }
    SNOMEDCT = {
        "1116000",  # Chronic aggressive type B viral hepatitis (disorder)
        "13265006",  # Acute fulminating type B viral hepatitis (disorder)
        "26206000",  # Hepatic coma due to viral hepatitis B (disorder)
        "38662009",  # Chronic persistent type B viral hepatitis (disorder)
        "50167007",  # Chronic active type B viral hepatitis (disorder)
        "53425008",  # Anicteric type B viral hepatitis (disorder)
        "60498001",  # Congenital viral hepatitis B infection (disorder)
        "61977001",  # Chronic type B viral hepatitis (disorder)
        "66071002",  # Viral hepatitis type B (disorder)
        "76795007",  # Acute type B viral hepatitis (disorder)
        "111891008",  # Viral hepatitis B without hepatic coma (disorder)
        "165806002",  # Hepatitis B surface antigen positive (finding)
        "186624004",  # Hepatic coma due to acute hepatitis B with delta agent (disorder)
        "186626002",  # Acute hepatitis B with delta-agent (coinfection) without hepatic coma (disorder)
        "186639003",  # Chronic viral hepatitis B without delta-agent (disorder)
        "235864009",  # Acute hepatitis B with hepatitis D (disorder)
        "235865005",  # Hepatitis D superinfection of hepatitis B carrier (disorder)
        "235869004",  # Chronic viral hepatitis B with hepatitis D (disorder)
        "235871004",  # Hepatitis B carrier (finding)
        "271511000",  # Hepatitis B immune (finding)
        "313234004",  # Hepatitis B antibody present (finding)
        "406117000",  # Hepatitis B surface antigen enzyme-linked immunosorbent assay positive (finding)
        "424099008",  # Hepatic coma due to acute hepatitis B (disorder)
        "424340000",  # Hepatic coma due to chronic hepatitis B (disorder)
        "442134007",  # Hepatitis B associated with Human immunodeficiency virus infection (disorder)
        "442374005",  # Hepatitis B and hepatitis C (disorder)
        "446698005",  # Reactivation of hepatitis B viral hepatitis (disorder)
        "838380002",  # Chronic hepatitis B co-occurrent with hepatitis C and hepatitis D (disorder)
        "153091000119109",  # Hepatic coma due to chronic hepatitis B with delta agent (disorder)
    }


class Intussusception(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of intussusception.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of intussusception.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Intussusception"
    OID = "2.16.840.1.113883.3.464.1003.199.12.1056"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "K561",  # Intussusception
    }
    SNOMEDCT = {
        "11578004",  # Intussusception of rectum (disorder)
        "17186003",  # Ileocolic intussusception (disorder)
        "24610009",  # Sigmoidorectal intussusception (disorder)
        "26316005",  # Internal complete rectal prolapse with intussusception of rectosigmoid (disorder)
        "27673007",  # Intussusception of colon (disorder)
        "38397000",  # Gastroesophageal intussusception (disorder)
        "49723003",  # Intussusception of intestine (disorder)
        "64976006",  # Cecocolic intussusception (disorder)
        "91446004",  # Intussusception of appendix (disorder)
        "111361008",  # Intussusception of cecum (disorder)
        "197046003",  # Gastroduodenal intussusception (disorder)
        "197047007",  # Jejunogastric intussusception (disorder)
        "197049005",  # Ileocecal intussusception (disorder)
        "197050005",  # Chronic intussusception (disorder)
        "197051009",  # Compound intussusception (disorder)
        "197052002",  # Multiple intussusception (disorder)
        "197053007",  # Retrograde intussusception (disorder)
        "197054001",  # Secondary intussusception (disorder)
        "313326008",  # Internal intussusception of rectum (disorder)
        "313327004",  # Inflammatory cloacogenic polyp (disorder)
        "405558008",  # Ileal intussusception (disorder)
        "721651002",  # Intussusception of small intestine (disorder)
        "721676001",  # Intussusception of large intestine (disorder)
    }


class MalignantNeoplasmOfLymphaticAndHematopoieticTissue(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of lymphoma and other cancers of the lymph and hematologic systems.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of lymphoma, leukemia and other malignant cancers of the lymphatic and hematologic systems.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Malignant Neoplasm of Lymphatic and Hematopoietic Tissue"
    OID = "2.16.840.1.113883.3.464.1003.108.12.1009"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "C8100",  # Nodular lymphocyte predominant Hodgkin lymphoma, unspecified site
        "C8101",  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of head, face, and neck
        "C8102",  # Nodular lymphocyte predominant Hodgkin lymphoma, intrathoracic lymph nodes
        "C8103",  # Nodular lymphocyte predominant Hodgkin lymphoma, intra-abdominal lymph nodes
        "C8104",  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of axilla and upper limb
        "C8105",  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        "C8106",  # Nodular lymphocyte predominant Hodgkin lymphoma, intrapelvic lymph nodes
        "C8107",  # Nodular lymphocyte predominant Hodgkin lymphoma, spleen
        "C8108",  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of multiple sites
        "C8109",  # Nodular lymphocyte predominant Hodgkin lymphoma, extranodal and solid organ sites
        "C8110",  # Nodular sclerosis Hodgkin lymphoma, unspecified site
        "C8111",  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of head, face, and neck
        "C8112",  # Nodular sclerosis Hodgkin lymphoma, intrathoracic lymph nodes
        "C8113",  # Nodular sclerosis Hodgkin lymphoma, intra-abdominal lymph nodes
        "C8114",  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of axilla and upper limb
        "C8115",  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        "C8116",  # Nodular sclerosis Hodgkin lymphoma, intrapelvic lymph nodes
        "C8117",  # Nodular sclerosis Hodgkin lymphoma, spleen
        "C8118",  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of multiple sites
        "C8119",  # Nodular sclerosis Hodgkin lymphoma, extranodal and solid organ sites
        "C8120",  # Mixed cellularity Hodgkin lymphoma, unspecified site
        "C8121",  # Mixed cellularity Hodgkin lymphoma, lymph nodes of head, face, and neck
        "C8122",  # Mixed cellularity Hodgkin lymphoma, intrathoracic lymph nodes
        "C8123",  # Mixed cellularity Hodgkin lymphoma, intra-abdominal lymph nodes
        "C8124",  # Mixed cellularity Hodgkin lymphoma, lymph nodes of axilla and upper limb
        "C8125",  # Mixed cellularity Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        "C8126",  # Mixed cellularity Hodgkin lymphoma, intrapelvic lymph nodes
        "C8127",  # Mixed cellularity Hodgkin lymphoma, spleen
        "C8128",  # Mixed cellularity Hodgkin lymphoma, lymph nodes of multiple sites
        "C8129",  # Mixed cellularity Hodgkin lymphoma, extranodal and solid organ sites
        "C8130",  # Lymphocyte depleted Hodgkin lymphoma, unspecified site
        "C8131",  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of head, face, and neck
        "C8132",  # Lymphocyte depleted Hodgkin lymphoma, intrathoracic lymph nodes
        "C8133",  # Lymphocyte depleted Hodgkin lymphoma, intra-abdominal lymph nodes
        "C8134",  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of axilla and upper limb
        "C8135",  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        "C8136",  # Lymphocyte depleted Hodgkin lymphoma, intrapelvic lymph nodes
        "C8137",  # Lymphocyte depleted Hodgkin lymphoma, spleen
        "C8138",  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of multiple sites
        "C8139",  # Lymphocyte depleted Hodgkin lymphoma, extranodal and solid organ sites
        "C8140",  # Lymphocyte-rich Hodgkin lymphoma, unspecified site
        "C8141",  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of head, face, and neck
        "C8142",  # Lymphocyte-rich Hodgkin lymphoma, intrathoracic lymph nodes
        "C8143",  # Lymphocyte-rich Hodgkin lymphoma, intra-abdominal lymph nodes
        "C8144",  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of axilla and upper limb
        "C8145",  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        "C8146",  # Lymphocyte-rich Hodgkin lymphoma, intrapelvic lymph nodes
        "C8147",  # Lymphocyte-rich Hodgkin lymphoma, spleen
        "C8148",  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of multiple sites
        "C8149",  # Lymphocyte-rich Hodgkin lymphoma, extranodal and solid organ sites
        "C8170",  # Other Hodgkin lymphoma, unspecified site
        "C8171",  # Other Hodgkin lymphoma, lymph nodes of head, face, and neck
        "C8172",  # Other Hodgkin lymphoma, intrathoracic lymph nodes
        "C8173",  # Other Hodgkin lymphoma, intra-abdominal lymph nodes
        "C8174",  # Other Hodgkin lymphoma, lymph nodes of axilla and upper limb
        "C8175",  # Other Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        "C8176",  # Other Hodgkin lymphoma, intrapelvic lymph nodes
        "C8177",  # Other Hodgkin lymphoma, spleen
        "C8178",  # Other Hodgkin lymphoma, lymph nodes of multiple sites
        "C8179",  # Other Hodgkin lymphoma, extranodal and solid organ sites
        "C8190",  # Hodgkin lymphoma, unspecified, unspecified site
        "C8191",  # Hodgkin lymphoma, unspecified, lymph nodes of head, face, and neck
        "C8192",  # Hodgkin lymphoma, unspecified, intrathoracic lymph nodes
        "C8193",  # Hodgkin lymphoma, unspecified, intra-abdominal lymph nodes
        "C8194",  # Hodgkin lymphoma, unspecified, lymph nodes of axilla and upper limb
        "C8195",  # Hodgkin lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        "C8196",  # Hodgkin lymphoma, unspecified, intrapelvic lymph nodes
        "C8197",  # Hodgkin lymphoma, unspecified, spleen
        "C8198",  # Hodgkin lymphoma, unspecified, lymph nodes of multiple sites
        "C8199",  # Hodgkin lymphoma, unspecified, extranodal and solid organ sites
        "C8200",  # Follicular lymphoma grade I, unspecified site
        "C8201",  # Follicular lymphoma grade I, lymph nodes of head, face, and neck
        "C8202",  # Follicular lymphoma grade I, intrathoracic lymph nodes
        "C8203",  # Follicular lymphoma grade I, intra-abdominal lymph nodes
        "C8204",  # Follicular lymphoma grade I, lymph nodes of axilla and upper limb
        "C8205",  # Follicular lymphoma grade I, lymph nodes of inguinal region and lower limb
        "C8206",  # Follicular lymphoma grade I, intrapelvic lymph nodes
        "C8207",  # Follicular lymphoma grade I, spleen
        "C8208",  # Follicular lymphoma grade I, lymph nodes of multiple sites
        "C8209",  # Follicular lymphoma grade I, extranodal and solid organ sites
        "C8210",  # Follicular lymphoma grade II, unspecified site
        "C8211",  # Follicular lymphoma grade II, lymph nodes of head, face, and neck
        "C8212",  # Follicular lymphoma grade II, intrathoracic lymph nodes
        "C8213",  # Follicular lymphoma grade II, intra-abdominal lymph nodes
        "C8214",  # Follicular lymphoma grade II, lymph nodes of axilla and upper limb
        "C8215",  # Follicular lymphoma grade II, lymph nodes of inguinal region and lower limb
        "C8216",  # Follicular lymphoma grade II, intrapelvic lymph nodes
        "C8217",  # Follicular lymphoma grade II, spleen
        "C8218",  # Follicular lymphoma grade II, lymph nodes of multiple sites
        "C8219",  # Follicular lymphoma grade II, extranodal and solid organ sites
        "C8220",  # Follicular lymphoma grade III, unspecified, unspecified site
        "C8221",  # Follicular lymphoma grade III, unspecified, lymph nodes of head, face, and neck
        "C8222",  # Follicular lymphoma grade III, unspecified, intrathoracic lymph nodes
        "C8223",  # Follicular lymphoma grade III, unspecified, intra-abdominal lymph nodes
        "C8224",  # Follicular lymphoma grade III, unspecified, lymph nodes of axilla and upper limb
        "C8225",  # Follicular lymphoma grade III, unspecified, lymph nodes of inguinal region and lower limb
        "C8226",  # Follicular lymphoma grade III, unspecified, intrapelvic lymph nodes
        "C8227",  # Follicular lymphoma grade III, unspecified, spleen
        "C8228",  # Follicular lymphoma grade III, unspecified, lymph nodes of multiple sites
        "C8229",  # Follicular lymphoma grade III, unspecified, extranodal and solid organ sites
        "C8230",  # Follicular lymphoma grade IIIa, unspecified site
        "C8231",  # Follicular lymphoma grade IIIa, lymph nodes of head, face, and neck
        "C8232",  # Follicular lymphoma grade IIIa, intrathoracic lymph nodes
        "C8233",  # Follicular lymphoma grade IIIa, intra-abdominal lymph nodes
        "C8234",  # Follicular lymphoma grade IIIa, lymph nodes of axilla and upper limb
        "C8235",  # Follicular lymphoma grade IIIa, lymph nodes of inguinal region and lower limb
        "C8236",  # Follicular lymphoma grade IIIa, intrapelvic lymph nodes
        "C8237",  # Follicular lymphoma grade IIIa, spleen
        "C8238",  # Follicular lymphoma grade IIIa, lymph nodes of multiple sites
        "C8239",  # Follicular lymphoma grade IIIa, extranodal and solid organ sites
        "C8240",  # Follicular lymphoma grade IIIb, unspecified site
        "C8241",  # Follicular lymphoma grade IIIb, lymph nodes of head, face, and neck
        "C8242",  # Follicular lymphoma grade IIIb, intrathoracic lymph nodes
        "C8243",  # Follicular lymphoma grade IIIb, intra-abdominal lymph nodes
        "C8244",  # Follicular lymphoma grade IIIb, lymph nodes of axilla and upper limb
        "C8245",  # Follicular lymphoma grade IIIb, lymph nodes of inguinal region and lower limb
        "C8246",  # Follicular lymphoma grade IIIb, intrapelvic lymph nodes
        "C8247",  # Follicular lymphoma grade IIIb, spleen
        "C8248",  # Follicular lymphoma grade IIIb, lymph nodes of multiple sites
        "C8249",  # Follicular lymphoma grade IIIb, extranodal and solid organ sites
        "C8250",  # Diffuse follicle center lymphoma, unspecified site
        "C8251",  # Diffuse follicle center lymphoma, lymph nodes of head, face, and neck
        "C8252",  # Diffuse follicle center lymphoma, intrathoracic lymph nodes
        "C8253",  # Diffuse follicle center lymphoma, intra-abdominal lymph nodes
        "C8254",  # Diffuse follicle center lymphoma, lymph nodes of axilla and upper limb
        "C8255",  # Diffuse follicle center lymphoma, lymph nodes of inguinal region and lower limb
        "C8256",  # Diffuse follicle center lymphoma, intrapelvic lymph nodes
        "C8257",  # Diffuse follicle center lymphoma, spleen
        "C8258",  # Diffuse follicle center lymphoma, lymph nodes of multiple sites
        "C8259",  # Diffuse follicle center lymphoma, extranodal and solid organ sites
        "C8260",  # Cutaneous follicle center lymphoma, unspecified site
        "C8261",  # Cutaneous follicle center lymphoma, lymph nodes of head, face, and neck
        "C8262",  # Cutaneous follicle center lymphoma, intrathoracic lymph nodes
        "C8263",  # Cutaneous follicle center lymphoma, intra-abdominal lymph nodes
        "C8264",  # Cutaneous follicle center lymphoma, lymph nodes of axilla and upper limb
        "C8265",  # Cutaneous follicle center lymphoma, lymph nodes of inguinal region and lower limb
        "C8266",  # Cutaneous follicle center lymphoma, intrapelvic lymph nodes
        "C8267",  # Cutaneous follicle center lymphoma, spleen
        "C8268",  # Cutaneous follicle center lymphoma, lymph nodes of multiple sites
        "C8269",  # Cutaneous follicle center lymphoma, extranodal and solid organ sites
        "C8280",  # Other types of follicular lymphoma, unspecified site
        "C8281",  # Other types of follicular lymphoma, lymph nodes of head, face, and neck
        "C8282",  # Other types of follicular lymphoma, intrathoracic lymph nodes
        "C8283",  # Other types of follicular lymphoma, intra-abdominal lymph nodes
        "C8284",  # Other types of follicular lymphoma, lymph nodes of axilla and upper limb
        "C8285",  # Other types of follicular lymphoma, lymph nodes of inguinal region and lower limb
        "C8286",  # Other types of follicular lymphoma, intrapelvic lymph nodes
        "C8287",  # Other types of follicular lymphoma, spleen
        "C8288",  # Other types of follicular lymphoma, lymph nodes of multiple sites
        "C8289",  # Other types of follicular lymphoma, extranodal and solid organ sites
        "C8290",  # Follicular lymphoma, unspecified, unspecified site
        "C8291",  # Follicular lymphoma, unspecified, lymph nodes of head, face, and neck
        "C8292",  # Follicular lymphoma, unspecified, intrathoracic lymph nodes
        "C8293",  # Follicular lymphoma, unspecified, intra-abdominal lymph nodes
        "C8294",  # Follicular lymphoma, unspecified, lymph nodes of axilla and upper limb
        "C8295",  # Follicular lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        "C8296",  # Follicular lymphoma, unspecified, intrapelvic lymph nodes
        "C8297",  # Follicular lymphoma, unspecified, spleen
        "C8298",  # Follicular lymphoma, unspecified, lymph nodes of multiple sites
        "C8299",  # Follicular lymphoma, unspecified, extranodal and solid organ sites
        "C8300",  # Small cell B-cell lymphoma, unspecified site
        "C8301",  # Small cell B-cell lymphoma, lymph nodes of head, face, and neck
        "C8302",  # Small cell B-cell lymphoma, intrathoracic lymph nodes
        "C8303",  # Small cell B-cell lymphoma, intra-abdominal lymph nodes
        "C8304",  # Small cell B-cell lymphoma, lymph nodes of axilla and upper limb
        "C8305",  # Small cell B-cell lymphoma, lymph nodes of inguinal region and lower limb
        "C8306",  # Small cell B-cell lymphoma, intrapelvic lymph nodes
        "C8307",  # Small cell B-cell lymphoma, spleen
        "C8308",  # Small cell B-cell lymphoma, lymph nodes of multiple sites
        "C8309",  # Small cell B-cell lymphoma, extranodal and solid organ sites
        "C8310",  # Mantle cell lymphoma, unspecified site
        "C8311",  # Mantle cell lymphoma, lymph nodes of head, face, and neck
        "C8312",  # Mantle cell lymphoma, intrathoracic lymph nodes
        "C8313",  # Mantle cell lymphoma, intra-abdominal lymph nodes
        "C8314",  # Mantle cell lymphoma, lymph nodes of axilla and upper limb
        "C8315",  # Mantle cell lymphoma, lymph nodes of inguinal region and lower limb
        "C8316",  # Mantle cell lymphoma, intrapelvic lymph nodes
        "C8317",  # Mantle cell lymphoma, spleen
        "C8318",  # Mantle cell lymphoma, lymph nodes of multiple sites
        "C8319",  # Mantle cell lymphoma, extranodal and solid organ sites
        "C8330",  # Diffuse large B-cell lymphoma, unspecified site
        "C8331",  # Diffuse large B-cell lymphoma, lymph nodes of head, face, and neck
        "C8332",  # Diffuse large B-cell lymphoma, intrathoracic lymph nodes
        "C8333",  # Diffuse large B-cell lymphoma, intra-abdominal lymph nodes
        "C8334",  # Diffuse large B-cell lymphoma, lymph nodes of axilla and upper limb
        "C8335",  # Diffuse large B-cell lymphoma, lymph nodes of inguinal region and lower limb
        "C8336",  # Diffuse large B-cell lymphoma, intrapelvic lymph nodes
        "C8337",  # Diffuse large B-cell lymphoma, spleen
        "C8338",  # Diffuse large B-cell lymphoma, lymph nodes of multiple sites
        "C8339",  # Diffuse large B-cell lymphoma, extranodal and solid organ sites
        "C8350",  # Lymphoblastic (diffuse) lymphoma, unspecified site
        "C8351",  # Lymphoblastic (diffuse) lymphoma, lymph nodes of head, face, and neck
        "C8352",  # Lymphoblastic (diffuse) lymphoma, intrathoracic lymph nodes
        "C8353",  # Lymphoblastic (diffuse) lymphoma, intra-abdominal lymph nodes
        "C8354",  # Lymphoblastic (diffuse) lymphoma, lymph nodes of axilla and upper limb
        "C8355",  # Lymphoblastic (diffuse) lymphoma, lymph nodes of inguinal region and lower limb
        "C8356",  # Lymphoblastic (diffuse) lymphoma, intrapelvic lymph nodes
        "C8357",  # Lymphoblastic (diffuse) lymphoma, spleen
        "C8358",  # Lymphoblastic (diffuse) lymphoma, lymph nodes of multiple sites
        "C8359",  # Lymphoblastic (diffuse) lymphoma, extranodal and solid organ sites
        "C8370",  # Burkitt lymphoma, unspecified site
        "C8371",  # Burkitt lymphoma, lymph nodes of head, face, and neck
        "C8372",  # Burkitt lymphoma, intrathoracic lymph nodes
        "C8373",  # Burkitt lymphoma, intra-abdominal lymph nodes
        "C8374",  # Burkitt lymphoma, lymph nodes of axilla and upper limb
        "C8375",  # Burkitt lymphoma, lymph nodes of inguinal region and lower limb
        "C8376",  # Burkitt lymphoma, intrapelvic lymph nodes
        "C8377",  # Burkitt lymphoma, spleen
        "C8378",  # Burkitt lymphoma, lymph nodes of multiple sites
        "C8379",  # Burkitt lymphoma, extranodal and solid organ sites
        "C8380",  # Other non-follicular lymphoma, unspecified site
        "C8381",  # Other non-follicular lymphoma, lymph nodes of head, face, and neck
        "C8382",  # Other non-follicular lymphoma, intrathoracic lymph nodes
        "C8383",  # Other non-follicular lymphoma, intra-abdominal lymph nodes
        "C8384",  # Other non-follicular lymphoma, lymph nodes of axilla and upper limb
        "C8385",  # Other non-follicular lymphoma, lymph nodes of inguinal region and lower limb
        "C8386",  # Other non-follicular lymphoma, intrapelvic lymph nodes
        "C8387",  # Other non-follicular lymphoma, spleen
        "C8388",  # Other non-follicular lymphoma, lymph nodes of multiple sites
        "C8389",  # Other non-follicular lymphoma, extranodal and solid organ sites
        "C8390",  # Non-follicular (diffuse) lymphoma, unspecified, unspecified site
        "C8391",  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of head, face, and neck
        "C8392",  # Non-follicular (diffuse) lymphoma, unspecified, intrathoracic lymph nodes
        "C8393",  # Non-follicular (diffuse) lymphoma, unspecified, intra-abdominal lymph nodes
        "C8394",  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of axilla and upper limb
        "C8395",  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        "C8396",  # Non-follicular (diffuse) lymphoma, unspecified, intrapelvic lymph nodes
        "C8397",  # Non-follicular (diffuse) lymphoma, unspecified, spleen
        "C8398",  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of multiple sites
        "C8399",  # Non-follicular (diffuse) lymphoma, unspecified, extranodal and solid organ sites
        "C8400",  # Mycosis fungoides, unspecified site
        "C8401",  # Mycosis fungoides, lymph nodes of head, face, and neck
        "C8402",  # Mycosis fungoides, intrathoracic lymph nodes
        "C8403",  # Mycosis fungoides, intra-abdominal lymph nodes
        "C8404",  # Mycosis fungoides, lymph nodes of axilla and upper limb
        "C8405",  # Mycosis fungoides, lymph nodes of inguinal region and lower limb
        "C8406",  # Mycosis fungoides, intrapelvic lymph nodes
        "C8407",  # Mycosis fungoides, spleen
        "C8408",  # Mycosis fungoides, lymph nodes of multiple sites
        "C8409",  # Mycosis fungoides, extranodal and solid organ sites
        "C8410",  # Sézary disease, unspecified site
        "C8411",  # Sézary disease, lymph nodes of head, face, and neck
        "C8412",  # Sézary disease, intrathoracic lymph nodes
        "C8413",  # Sézary disease, intra-abdominal lymph nodes
        "C8414",  # Sézary disease, lymph nodes of axilla and upper limb
        "C8415",  # Sézary disease, lymph nodes of inguinal region and lower limb
        "C8416",  # Sézary disease, intrapelvic lymph nodes
        "C8417",  # Sézary disease, spleen
        "C8418",  # Sézary disease, lymph nodes of multiple sites
        "C8419",  # Sézary disease, extranodal and solid organ sites
        "C8440",  # Peripheral T-cell lymphoma, not classified, unspecified site
        "C8441",  # Peripheral T-cell lymphoma, not classified, lymph nodes of head, face, and neck
        "C8442",  # Peripheral T-cell lymphoma, not classified, intrathoracic lymph nodes
        "C8443",  # Peripheral T-cell lymphoma, not classified, intra-abdominal lymph nodes
        "C8444",  # Peripheral T-cell lymphoma, not classified, lymph nodes of axilla and upper limb
        "C8445",  # Peripheral T-cell lymphoma, not classified, lymph nodes of inguinal region and lower limb
        "C8446",  # Peripheral T-cell lymphoma, not classified, intrapelvic lymph nodes
        "C8447",  # Peripheral T-cell lymphoma, not classified, spleen
        "C8448",  # Peripheral T-cell lymphoma, not classified, lymph nodes of multiple sites
        "C8449",  # Peripheral T-cell lymphoma, not classified, extranodal and solid organ sites
        "C8460",  # Anaplastic large cell lymphoma, ALK-positive, unspecified site
        "C8461",  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of head, face, and neck
        "C8462",  # Anaplastic large cell lymphoma, ALK-positive, intrathoracic lymph nodes
        "C8463",  # Anaplastic large cell lymphoma, ALK-positive, intra-abdominal lymph nodes
        "C8464",  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of axilla and upper limb
        "C8465",  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of inguinal region and lower limb
        "C8466",  # Anaplastic large cell lymphoma, ALK-positive, intrapelvic lymph nodes
        "C8467",  # Anaplastic large cell lymphoma, ALK-positive, spleen
        "C8468",  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of multiple sites
        "C8469",  # Anaplastic large cell lymphoma, ALK-positive, extranodal and solid organ sites
        "C8470",  # Anaplastic large cell lymphoma, ALK-negative, unspecified site
        "C8471",  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of head, face, and neck
        "C8472",  # Anaplastic large cell lymphoma, ALK-negative, intrathoracic lymph nodes
        "C8473",  # Anaplastic large cell lymphoma, ALK-negative, intra-abdominal lymph nodes
        "C8474",  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of axilla and upper limb
        "C8475",  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of inguinal region and lower limb
        "C8476",  # Anaplastic large cell lymphoma, ALK-negative, intrapelvic lymph nodes
        "C8477",  # Anaplastic large cell lymphoma, ALK-negative, spleen
        "C8478",  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of multiple sites
        "C8479",  # Anaplastic large cell lymphoma, ALK-negative, extranodal and solid organ sites
        "C8490",  # Mature T/NK-cell lymphomas, unspecified, unspecified site
        "C8491",  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of head, face, and neck
        "C8492",  # Mature T/NK-cell lymphomas, unspecified, intrathoracic lymph nodes
        "C8493",  # Mature T/NK-cell lymphomas, unspecified, intra-abdominal lymph nodes
        "C8494",  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of axilla and upper limb
        "C8495",  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of inguinal region and lower limb
        "C8496",  # Mature T/NK-cell lymphomas, unspecified, intrapelvic lymph nodes
        "C8497",  # Mature T/NK-cell lymphomas, unspecified, spleen
        "C8498",  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of multiple sites
        "C8499",  # Mature T/NK-cell lymphomas, unspecified, extranodal and solid organ sites
        "C84A0",  # Cutaneous T-cell lymphoma, unspecified, unspecified site
        "C84A1",  # Cutaneous T-cell lymphoma, unspecified lymph nodes of head, face, and neck
        "C84A2",  # Cutaneous T-cell lymphoma, unspecified, intrathoracic lymph nodes
        "C84A3",  # Cutaneous T-cell lymphoma, unspecified, intra-abdominal lymph nodes
        "C84A4",  # Cutaneous T-cell lymphoma, unspecified, lymph nodes of axilla and upper limb
        "C84A5",  # Cutaneous T-cell lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        "C84A6",  # Cutaneous T-cell lymphoma, unspecified, intrapelvic lymph nodes
        "C84A7",  # Cutaneous T-cell lymphoma, unspecified, spleen
        "C84A8",  # Cutaneous T-cell lymphoma, unspecified, lymph nodes of multiple sites
        "C84A9",  # Cutaneous T-cell lymphoma, unspecified, extranodal and solid organ sites
        "C84Z0",  # Other mature T/NK-cell lymphomas, unspecified site
        "C84Z1",  # Other mature T/NK-cell lymphomas, lymph nodes of head, face, and neck
        "C84Z2",  # Other mature T/NK-cell lymphomas, intrathoracic lymph nodes
        "C84Z3",  # Other mature T/NK-cell lymphomas, intra-abdominal lymph nodes
        "C84Z4",  # Other mature T/NK-cell lymphomas, lymph nodes of axilla and upper limb
        "C84Z5",  # Other mature T/NK-cell lymphomas, lymph nodes of inguinal region and lower limb
        "C84Z6",  # Other mature T/NK-cell lymphomas, intrapelvic lymph nodes
        "C84Z7",  # Other mature T/NK-cell lymphomas, spleen
        "C84Z8",  # Other mature T/NK-cell lymphomas, lymph nodes of multiple sites
        "C84Z9",  # Other mature T/NK-cell lymphomas, extranodal and solid organ sites
        "C8510",  # Unspecified B-cell lymphoma, unspecified site
        "C8511",  # Unspecified B-cell lymphoma, lymph nodes of head, face, and neck
        "C8512",  # Unspecified B-cell lymphoma, intrathoracic lymph nodes
        "C8513",  # Unspecified B-cell lymphoma, intra-abdominal lymph nodes
        "C8514",  # Unspecified B-cell lymphoma, lymph nodes of axilla and upper limb
        "C8515",  # Unspecified B-cell lymphoma, lymph nodes of inguinal region and lower limb
        "C8516",  # Unspecified B-cell lymphoma, intrapelvic lymph nodes
        "C8517",  # Unspecified B-cell lymphoma, spleen
        "C8518",  # Unspecified B-cell lymphoma, lymph nodes of multiple sites
        "C8519",  # Unspecified B-cell lymphoma, extranodal and solid organ sites
        "C8520",  # Mediastinal (thymic) large B-cell lymphoma, unspecified site
        "C8521",  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of head, face, and neck
        "C8522",  # Mediastinal (thymic) large B-cell lymphoma, intrathoracic lymph nodes
        "C8523",  # Mediastinal (thymic) large B-cell lymphoma, intra-abdominal lymph nodes
        "C8524",  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of axilla and upper limb
        "C8525",  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of inguinal region and lower limb
        "C8526",  # Mediastinal (thymic) large B-cell lymphoma, intrapelvic lymph nodes
        "C8527",  # Mediastinal (thymic) large B-cell lymphoma, spleen
        "C8528",  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of multiple sites
        "C8529",  # Mediastinal (thymic) large B-cell lymphoma, extranodal and solid organ sites
        "C8580",  # Other specified types of non-Hodgkin lymphoma, unspecified site
        "C8581",  # Other specified types of non-Hodgkin lymphoma, lymph nodes of head, face, and neck
        "C8582",  # Other specified types of non-Hodgkin lymphoma, intrathoracic lymph nodes
        "C8583",  # Other specified types of non-Hodgkin lymphoma, intra-abdominal lymph nodes
        "C8584",  # Other specified types of non-Hodgkin lymphoma, lymph nodes of axilla and upper limb
        "C8585",  # Other specified types of non-Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        "C8586",  # Other specified types of non-Hodgkin lymphoma, intrapelvic lymph nodes
        "C8587",  # Other specified types of non-Hodgkin lymphoma, spleen
        "C8588",  # Other specified types of non-Hodgkin lymphoma, lymph nodes of multiple sites
        "C8589",  # Other specified types of non-Hodgkin lymphoma, extranodal and solid organ sites
        "C8590",  # Non-Hodgkin lymphoma, unspecified, unspecified site
        "C8591",  # Non-Hodgkin lymphoma, unspecified, lymph nodes of head, face, and neck
        "C8592",  # Non-Hodgkin lymphoma, unspecified, intrathoracic lymph nodes
        "C8593",  # Non-Hodgkin lymphoma, unspecified, intra-abdominal lymph nodes
        "C8594",  # Non-Hodgkin lymphoma, unspecified, lymph nodes of axilla and upper limb
        "C8595",  # Non-Hodgkin lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        "C8596",  # Non-Hodgkin lymphoma, unspecified, intrapelvic lymph nodes
        "C8597",  # Non-Hodgkin lymphoma, unspecified, spleen
        "C8598",  # Non-Hodgkin lymphoma, unspecified, lymph nodes of multiple sites
        "C8599",  # Non-Hodgkin lymphoma, unspecified, extranodal and solid organ sites
        "C860",  # Extranodal NK/T-cell lymphoma, nasal type
        "C861",  # Hepatosplenic T-cell lymphoma
        "C862",  # Enteropathy-type (intestinal) T-cell lymphoma
        "C863",  # Subcutaneous panniculitis-like T-cell lymphoma
        "C864",  # Blastic NK-cell lymphoma
        "C865",  # Angioimmunoblastic T-cell lymphoma
        "C866",  # Primary cutaneous CD30-positive T-cell proliferations
        "C882",  # Heavy chain disease
        "C883",  # Immunoproliferative small intestinal disease
        "C884",  # Extranodal marginal zone B-cell lymphoma of mucosa-associated lymphoid tissue [MALT-lymphoma]
        "C888",  # Other malignant immunoproliferative diseases
        "C889",  # Malignant immunoproliferative disease, unspecified
        "C9000",  # Multiple myeloma not having achieved remission
        "C9001",  # Multiple myeloma in remission
        "C9002",  # Multiple myeloma in relapse
        "C9010",  # Plasma cell leukemia not having achieved remission
        "C9011",  # Plasma cell leukemia in remission
        "C9012",  # Plasma cell leukemia in relapse
        "C9020",  # Extramedullary plasmacytoma not having achieved remission
        "C9021",  # Extramedullary plasmacytoma in remission
        "C9022",  # Extramedullary plasmacytoma in relapse
        "C9030",  # Solitary plasmacytoma not having achieved remission
        "C9031",  # Solitary plasmacytoma in remission
        "C9032",  # Solitary plasmacytoma in relapse
        "C9100",  # Acute lymphoblastic leukemia not having achieved remission
        "C9101",  # Acute lymphoblastic leukemia, in remission
        "C9102",  # Acute lymphoblastic leukemia, in relapse
        "C9110",  # Chronic lymphocytic leukemia of B-cell type not having achieved remission
        "C9111",  # Chronic lymphocytic leukemia of B-cell type in remission
        "C9112",  # Chronic lymphocytic leukemia of B-cell type in relapse
        "C9130",  # Prolymphocytic leukemia of B-cell type not having achieved remission
        "C9131",  # Prolymphocytic leukemia of B-cell type, in remission
        "C9132",  # Prolymphocytic leukemia of B-cell type, in relapse
        "C9140",  # Hairy cell leukemia not having achieved remission
        "C9141",  # Hairy cell leukemia, in remission
        "C9142",  # Hairy cell leukemia, in relapse
        "C9150",  # Adult T-cell lymphoma/leukemia (HTLV-1-associated) not having achieved remission
        "C9151",  # Adult T-cell lymphoma/leukemia (HTLV-1-associated), in remission
        "C9152",  # Adult T-cell lymphoma/leukemia (HTLV-1-associated), in relapse
        "C9160",  # Prolymphocytic leukemia of T-cell type not having achieved remission
        "C9161",  # Prolymphocytic leukemia of T-cell type, in remission
        "C9162",  # Prolymphocytic leukemia of T-cell type, in relapse
        "C9190",  # Lymphoid leukemia, unspecified not having achieved remission
        "C9191",  # Lymphoid leukemia, unspecified, in remission
        "C9192",  # Lymphoid leukemia, unspecified, in relapse
        "C91A0",  # Mature B-cell leukemia Burkitt-type not having achieved remission
        "C91A1",  # Mature B-cell leukemia Burkitt-type, in remission
        "C91A2",  # Mature B-cell leukemia Burkitt-type, in relapse
        "C91Z0",  # Other lymphoid leukemia not having achieved remission
        "C91Z1",  # Other lymphoid leukemia, in remission
        "C91Z2",  # Other lymphoid leukemia, in relapse
        "C9200",  # Acute myeloblastic leukemia, not having achieved remission
        "C9201",  # Acute myeloblastic leukemia, in remission
        "C9202",  # Acute myeloblastic leukemia, in relapse
        "C9210",  # Chronic myeloid leukemia, BCR/ABL-positive, not having achieved remission
        "C9211",  # Chronic myeloid leukemia, BCR/ABL-positive, in remission
        "C9212",  # Chronic myeloid leukemia, BCR/ABL-positive, in relapse
        "C9220",  # Atypical chronic myeloid leukemia, BCR/ABL-negative, not having achieved remission
        "C9221",  # Atypical chronic myeloid leukemia, BCR/ABL-negative, in remission
        "C9222",  # Atypical chronic myeloid leukemia, BCR/ABL-negative, in relapse
        "C9230",  # Myeloid sarcoma, not having achieved remission
        "C9231",  # Myeloid sarcoma, in remission
        "C9232",  # Myeloid sarcoma, in relapse
        "C9240",  # Acute promyelocytic leukemia, not having achieved remission
        "C9241",  # Acute promyelocytic leukemia, in remission
        "C9242",  # Acute promyelocytic leukemia, in relapse
        "C9250",  # Acute myelomonocytic leukemia, not having achieved remission
        "C9251",  # Acute myelomonocytic leukemia, in remission
        "C9252",  # Acute myelomonocytic leukemia, in relapse
        "C9260",  # Acute myeloid leukemia with 11q23-abnormality not having achieved remission
        "C9261",  # Acute myeloid leukemia with 11q23-abnormality in remission
        "C9262",  # Acute myeloid leukemia with 11q23-abnormality in relapse
        "C9290",  # Myeloid leukemia, unspecified, not having achieved remission
        "C9291",  # Myeloid leukemia, unspecified in remission
        "C9292",  # Myeloid leukemia, unspecified in relapse
        "C92A0",  # Acute myeloid leukemia with multilineage dysplasia, not having achieved remission
        "C92A1",  # Acute myeloid leukemia with multilineage dysplasia, in remission
        "C92A2",  # Acute myeloid leukemia with multilineage dysplasia, in relapse
        "C92Z0",  # Other myeloid leukemia not having achieved remission
        "C92Z1",  # Other myeloid leukemia, in remission
        "C92Z2",  # Other myeloid leukemia, in relapse
        "C9300",  # Acute monoblastic/monocytic leukemia, not having achieved remission
        "C9301",  # Acute monoblastic/monocytic leukemia, in remission
        "C9302",  # Acute monoblastic/monocytic leukemia, in relapse
        "C9310",  # Chronic myelomonocytic leukemia not having achieved remission
        "C9311",  # Chronic myelomonocytic leukemia, in remission
        "C9312",  # Chronic myelomonocytic leukemia, in relapse
        "C9330",  # Juvenile myelomonocytic leukemia, not having achieved remission
        "C9331",  # Juvenile myelomonocytic leukemia, in remission
        "C9332",  # Juvenile myelomonocytic leukemia, in relapse
        "C9390",  # Monocytic leukemia, unspecified, not having achieved remission
        "C9391",  # Monocytic leukemia, unspecified in remission
        "C9392",  # Monocytic leukemia, unspecified in relapse
        "C93Z0",  # Other monocytic leukemia, not having achieved remission
        "C93Z1",  # Other monocytic leukemia, in remission
        "C93Z2",  # Other monocytic leukemia, in relapse
        "C9400",  # Acute erythroid leukemia, not having achieved remission
        "C9401",  # Acute erythroid leukemia, in remission
        "C9402",  # Acute erythroid leukemia, in relapse
        "C9420",  # Acute megakaryoblastic leukemia not having achieved remission
        "C9421",  # Acute megakaryoblastic leukemia, in remission
        "C9422",  # Acute megakaryoblastic leukemia, in relapse
        "C9430",  # Mast cell leukemia not having achieved remission
        "C9431",  # Mast cell leukemia, in remission
        "C9432",  # Mast cell leukemia, in relapse
        "C9480",  # Other specified leukemias not having achieved remission
        "C9481",  # Other specified leukemias, in remission
        "C9482",  # Other specified leukemias, in relapse
        "C9500",  # Acute leukemia of unspecified cell type not having achieved remission
        "C9501",  # Acute leukemia of unspecified cell type, in remission
        "C9502",  # Acute leukemia of unspecified cell type, in relapse
        "C9510",  # Chronic leukemia of unspecified cell type not having achieved remission
        "C9511",  # Chronic leukemia of unspecified cell type, in remission
        "C9512",  # Chronic leukemia of unspecified cell type, in relapse
        "C9590",  # Leukemia, unspecified not having achieved remission
        "C9591",  # Leukemia, unspecified, in remission
        "C9592",  # Leukemia, unspecified, in relapse
        "C960",  # Multifocal and multisystemic (disseminated) Langerhans-cell histiocytosis
        "C962",  # Malignant mast cell neoplasm
        "C9620",  # Malignant mast cell neoplasm, unspecified
        "C9621",  # Aggressive systemic mastocytosis
        "C9622",  # Mast cell sarcoma
        "C9629",  # Other malignant mast cell neoplasm
        "C964",  # Sarcoma of dendritic cells (accessory cells)
        "C969",  # Malignant neoplasm of lymphoid, hematopoietic and related tissue, unspecified
        "C96A",  # Histiocytic sarcoma
        "C96Z",  # Other specified malignant neoplasms of lymphoid, hematopoietic and related tissue
    }
    SNOMEDCT = {
        "13048006",  # Orbital lymphoma (disorder)
        "31047003",  # Lymphomatoid papulosis (disorder)
        "58961005",  # Lethal midline granuloma (disorder)
        "91854005",  # Acute leukemia in remission (disorder)
        "91855006",  # Acute leukemia, disease (disorder)
        "91856007",  # Acute lymphoid leukemia in remission (disorder)
        "91857003",  # Acute lymphoid leukemia, disease (disorder)
        "91858008",  # Acute monocytic leukemia in remission (disorder)
        "91860005",  # Acute myeloid leukemia in remission (disorder)
        "91861009",  # Acute myeloid leukemia, disease (disorder)
        "92516002",  # Burkitt's tumor of extranodal AND/OR solid organ site (disorder)
        "92811003",  # Chronic leukemia in remission (disorder)
        "92812005",  # Chronic leukemia, disease (disorder)
        "92813000",  # Chronic lymphoid leukemia in remission (disorder)
        "92814006",  # Chronic lymphoid leukemia, disease (disorder)
        "92817004",  # Chronic myeloid leukemia in remission (disorder)
        "92818009",  # Chronic myeloid leukemia, disease (disorder)
        "93133006",  # Letterer-Siwe disease of intra-abdominal lymph nodes (disorder)
        "93134000",  # Letterer-Siwe disease of intrapelvic lymph nodes (disorder)
        "93135004",  # Letterer-Siwe disease of intrathoracic lymph nodes (disorder)
        "93136003",  # Letterer-Siwe disease of lymph nodes of axilla AND/OR upper limb (disorder)
        "93137007",  # Letterer-Siwe disease of lymph nodes of head, face AND/OR neck (disorder)
        "93138002",  # Letterer-Siwe disease of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93139005",  # Letterer-Siwe disease of lymph nodes of multiple sites (disorder)
        "93140007",  # Letterer-Siwe disease of spleen (disorder)
        "93141006",  # Letterer-Siwe disease of extranodal AND/OR solid organ site (disorder)
        "93142004",  # Leukemia in remission (disorder)
        "93143009",  # Leukemia, disease (disorder)
        "93144003",  # Leukemic reticuloendotheliosis of intra-abdominal lymph nodes (disorder)
        "93145002",  # Leukemic reticuloendotheliosis of intrapelvic lymph nodes (disorder)
        "93146001",  # Leukemic reticuloendotheliosis of intrathoracic lymph nodes (disorder)
        "93150008",  # Leukemic reticuloendotheliosis of lymph nodes of multiple sites (disorder)
        "93151007",  # Hairy cell leukemia of spleen (disorder)
        "93152000",  # Leukemic reticuloendotheliosis of extranodal AND/OR solid organ site (disorder)
        "93169003",  # Lymphoid leukemia in remission (disorder)
        "93182006",  # Malignant histiocytosis of intra-abdominal lymph nodes (disorder)
        "93183001",  # Malignant histiocytosis of intrapelvic lymph nodes (disorder)
        "93184007",  # Malignant histiocytosis of intrathoracic lymph nodes (disorder)
        "93185008",  # Malignant histiocytosis of lymph nodes of axilla AND/OR upper limb (disorder)
        "93186009",  # Malignant histiocytosis of lymph nodes of head, face AND/OR neck (disorder)
        "93187000",  # Malignant histiocytosis of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93188005",  # Malignant histiocytosis of lymph nodes of multiple sites (disorder)
        "93189002",  # Malignant histiocytosis of spleen (disorder)
        "93190006",  # Malignant histiocytosis of extranodal AND/OR solid organ site (disorder)
        "93191005",  # Malignant lymphoma of intra-abdominal lymph nodes (disorder)
        "93192003",  # Malignant lymphoma of intrapelvic lymph nodes (disorder)
        "93193008",  # Malignant lymphoma of intrathoracic lymph nodes (disorder)
        "93194002",  # Malignant lymphoma of lymph nodes of axilla AND/OR upper limb (disorder)
        "93195001",  # Malignant lymphoma of lymph nodes of head, face AND/OR neck (disorder)
        "93196000",  # Malignant lymphoma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93197009",  # Malignant lymphoma of lymph nodes of multiple sites (disorder)
        "93198004",  # Malignant lymphoma of spleen (disorder)
        "93199007",  # Malignant lymphoma of extranodal AND/OR solid organ site (disorder)
        "93200005",  # Malignant mast cell tumor of intra-abdominal lymph nodes (disorder)
        "93201009",  # Malignant mast cell tumor of intrapelvic lymph nodes (disorder)
        "93202002",  # Malignant mast cell tumor of intrathoracic lymph nodes (disorder)
        "93203007",  # Malignant mast cell tumor of lymph nodes of axilla AND/OR upper limb (disorder)
        "93204001",  # Malignant mast cell tumor of lymph nodes of head, face AND/OR neck (disorder)
        "93205000",  # Malignant mast cell tumor of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93206004",  # Malignant mast cell tumor of lymph nodes of multiple sites (disorder)
        "93207008",  # Malignant mast cell tumor of spleen (disorder)
        "93208003",  # Malignant mast cell tumor of extranodal AND/OR solid organ site (disorder)
        "93451002",  # Erythroleukemia, FAB M6 (disorder)
        "93487009",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of axilla AND/OR upper limb (disorder)
        "93488004",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of head, face AND/OR neck (disorder)
        "93489007",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93492006",  # Hodgkin's disease, lymphocytic depletion of extranodal AND/OR solid organ site (disorder)
        "93493001",  # Hodgkin's disease, lymphocytic-histiocytic predominance of intra-abdominal lymph nodes (disorder)
        "93494007",  # Hodgkin's disease, lymphocytic-histiocytic predominance of intrapelvic lymph nodes (disorder)
        "93495008",  # Hodgkin's disease, lymphocytic-histiocytic predominance of intrathoracic lymph nodes (disorder)
        "93496009",  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of axilla AND/OR upper limb (disorder)
        "93497000",  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of head, face AND/OR neck (disorder)
        "93498005",  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93500006",  # Hodgkin's disease, lymphocytic-histiocytic predominance of spleen (disorder)
        "93501005",  # Hodgkin's disease, lymphocytic-histiocytic predominance of extranodal AND/OR solid organ site (disorder)
        "93505001",  # Hodgkin's disease, mixed cellularity of lymph nodes of axilla AND/OR upper limb (disorder)
        "93506000",  # Hodgkin's disease, mixed cellularity of lymph nodes of head, face AND/OR neck (disorder)
        "93507009",  # Hodgkin's disease, mixed cellularity of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93509007",  # Hodgkin's disease, mixed cellularity of spleen (disorder)
        "93510002",  # Hodgkin's disease, mixed cellularity of extranodal AND/OR solid organ site (disorder)
        "93514006",  # Hodgkin's disease, nodular sclerosis of lymph nodes of axilla AND/OR upper limb (disorder)
        "93515007",  # Hodgkin's disease, nodular sclerosis of lymph nodes of head, face AND/OR neck (disorder)
        "93516008",  # Hodgkin's disease, nodular sclerosis of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93518009",  # Hodgkin's disease, nodular sclerosis of spleen (disorder)
        "93519001",  # Hodgkin's disease, nodular sclerosis of extranodal AND/OR solid organ site (disorder)
        "93520007",  # Hodgkin's disease of intra-abdominal lymph nodes (disorder)
        "93521006",  # Hodgkin's disease of intrapelvic lymph nodes (disorder)
        "93522004",  # Hodgkin's disease of intrathoracic lymph nodes (disorder)
        "93523009",  # Hodgkin's disease of lymph nodes of axilla AND/OR upper limb (disorder)
        "93524003",  # Hodgkin's disease of lymph nodes of head, face AND/OR neck (disorder)
        "93525002",  # Hodgkin's disease of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93526001",  # Hodgkin's disease of lymph nodes of multiple sites (disorder)
        "93527005",  # Hodgkin's disease of spleen (disorder)
        "93528000",  # Hodgkin's disease of extranodal AND/OR solid organ site (disorder)
        "93530003",  # Hodgkin's granuloma of intrapelvic lymph nodes (disorder)
        "93531004",  # Hodgkin's granuloma of intrathoracic lymph nodes (disorder)
        "93532006",  # Hodgkin's granuloma of lymph nodes of axilla AND/OR upper limb (disorder)
        "93533001",  # Hodgkin's granuloma of lymph nodes of head, face AND/OR neck (disorder)
        "93534007",  # Hodgkin's granuloma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93536009",  # Hodgkin's granuloma of spleen (disorder)
        "93537000",  # Hodgkin's granuloma of extranodal AND/OR solid organ site (disorder)
        "93541001",  # Hodgkin's paragranuloma of lymph nodes of axilla AND/OR upper limb (disorder)
        "93542008",  # Hodgkin's paragranuloma of lymph nodes of head, face AND/OR neck (disorder)
        "93543003",  # Hodgkin's paragranuloma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93546006",  # Hodgkin's paragranuloma of extranodal AND/OR solid organ site (disorder)
        "93547002",  # Hodgkin's sarcoma of intra-abdominal lymph nodes (disorder)
        "93548007",  # Hodgkin's sarcoma of intrapelvic lymph nodes (disorder)
        "93549004",  # Hodgkin's sarcoma of intrathoracic lymph nodes (disorder)
        "93550004",  # Hodgkin's sarcoma of lymph nodes of axilla AND/OR upper limb (disorder)
        "93551000",  # Hodgkin's sarcoma of lymph nodes of head, face AND/OR neck (disorder)
        "93552007",  # Hodgkin's sarcoma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "93554008",  # Hodgkin's sarcoma of spleen (disorder)
        "93555009",  # Hodgkin's sarcoma of extranodal AND/OR solid organ site (disorder)
        "94071006",  # Primary malignant neoplasm of spleen (disorder)
        "94148006",  # Megakaryocytic leukemia in remission (disorder)
        "94686001",  # Mixed cell type lymphosarcoma of intra-abdominal lymph nodes (disorder)
        "94687005",  # Mixed cell type lymphosarcoma of intrapelvic lymph nodes (disorder)
        "94688000",  # Mixed cell type lymphosarcoma of intrathoracic lymph nodes (disorder)
        "94690004",  # Mixed cell type lymphosarcoma of lymph nodes of head, face, and neck (disorder)
        "94707004",  # Mycosis fungoides of intra-abdominal lymph nodes (disorder)
        "94708009",  # Mycosis fungoides of intrapelvic lymph nodes (disorder)
        "94709001",  # Mycosis fungoides of intrathoracic lymph nodes (disorder)
        "94710006",  # Mycosis fungoides of lymph nodes of axilla AND/OR upper limb (disorder)
        "94711005",  # Mycosis fungoides of lymph nodes of head, face AND/OR neck (disorder)
        "94712003",  # Mycosis fungoides of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "94714002",  # Mycosis fungoides of spleen (disorder)
        "94715001",  # Mycosis fungoides of extranodal AND/OR solid organ site (disorder)
        "94716000",  # Myeloid leukemia in remission (disorder)
        "94718004",  # Myeloid sarcoma in remission (disorder)
        "94719007",  # Myeloid sarcoma, disease (disorder)
        "95186006",  # Nodular lymphoma of intra-abdominal lymph nodes (disorder)
        "95187002",  # Nodular lymphoma of intrapelvic lymph nodes (disorder)
        "95188007",  # Nodular lymphoma of intrathoracic lymph nodes (disorder)
        "95192000",  # Nodular lymphoma of lymph nodes of multiple sites (disorder)
        "95193005",  # Nodular lymphoma of spleen (disorder)
        "95194004",  # Nodular lymphoma of extranodal AND/OR solid organ site (disorder)
        "95209008",  # Plasma cell leukemia in remission (disorder)
        "95210003",  # Plasma cell leukemia, disease (disorder)
        "95224004",  # Reticulosarcoma of intra-abdominal lymph nodes (disorder)
        "95225003",  # Reticulosarcoma of intrapelvic lymph nodes (disorder)
        "95226002",  # Reticulosarcoma of intrathoracic lymph nodes (disorder)
        "95230004",  # Reticulosarcoma of lymph nodes of multiple sites (disorder)
        "95231000",  # Reticulosarcoma of spleen (disorder)
        "95260009",  # Sézary's disease of lymph nodes of head, face AND/OR neck (disorder)
        "95261008",  # Sézary's disease of lymph nodes of inguinal region AND/OR lower limb (disorder)
        "95263006",  # Sézary's disease of spleen (disorder)
        "95264000",  # Sézary's disease of extranodal AND/OR solid organ site (disorder)
        "109962001",  # Diffuse non-Hodgkin's lymphoma (disorder)
        "109964000",  # Diffuse non-Hodgkin's lymphoma, undifferentiated (disorder)
        "109965004",  # Diffuse non-Hodgkin's lymphoma, lymphoblastic (disorder)
        "109966003",  # Diffuse non-Hodgkin's lymphoma, immunoblastic (disorder)
        "109967007",  # Diffuse non-Hodgkin's lymphoma, small cleaved cell (disorder)
        "109968002",  # Diffuse non-Hodgkin's lymphoma, small cell (disorder)
        "109969005",  # Diffuse non-Hodgkin's lymphoma, large cell (disorder)
        "109970006",  # Follicular non-Hodgkin's lymphoma, small cleaved cell (disorder)
        "109971005",  # Follicular non-Hodgkin's lymphoma, mixed small cleaved cell and large cell (disorder)
        "109972003",  # Follicular non-Hodgkin's lymphoma, large cell (disorder)
        "109975001",  # T-zone lymphoma (disorder)
        "109976000",  # Lymphoepithelioid lymphoma (disorder)
        "109977009",  # Peripheral T-cell lymphoma (disorder)
        "109978004",  # T-cell lymphoma (disorder)
        "109979007",  # B-cell lymphoma (disorder)
        "109980005",  # Malignant immunoproliferative disease (disorder)
        "109988003",  # Histiocytic sarcoma (disorder)
        "109994006",  # Essential thrombocythemia (disorder)
        "109995007",  # Myelodysplastic syndrome (disorder)
        "109996008",  # Myelodysplastic syndrome: Refractory anemia, without ringed sideroblasts, without excess blasts (disorder)
        "109998009",  # Refractory anemia with ringed sideroblasts (disorder)
        "110000005",  # Refractory anemia with excess blasts in transformation (disorder)
        "110002002",  # Mast cell leukemia (disorder)
        "110004001",  # Acute promyelocytic leukemia, FAB M3 (disorder)
        "110005000",  # Acute myelomonocytic leukemia, FAB M4 (disorder)
        "110006004",  # Prolymphocytic leukemia (disorder)
        "110007008",  # Adult T-cell leukemia/lymphoma (disorder)
        "118599009",  # Hodgkin's disease (disorder)
        "118600007",  # Malignant lymphoma (disorder)
        "118601006",  # Non-Hodgkin's lymphoma (disorder)
        "118602004",  # Hodgkin's granuloma (disorder)
        "118605002",  # Hodgkin lymphoma, nodular lymphocyte predominance (disorder)
        "118606001",  # Hodgkin's sarcoma (disorder)
        "118607005",  # Hodgkin lymphoma, lymphocyte-rich (disorder)
        "118608000",  # Hodgkin's disease, nodular sclerosis (disorder)
        "118609008",  # Hodgkin's disease, mixed cellularity (disorder)
        "118610003",  # Hodgkin's disease, lymphocytic depletion (disorder)
        "118611004",  # Sézary's disease (disorder)
        "118612006",  # Malignant histiocytosis (disorder)
        "118613001",  # Hairy cell leukemia (disorder)
        "118614007",  # Langerhans cell histiocytosis, disseminated (disorder)
        "118615008",  # Malignant mast cell tumor (disorder)
        "118617000",  # Burkitt's lymphoma (disorder)
        "118618005",  # Mycosis fungoides (disorder)
        "127070008",  # Malignant histiocytic disorder (disorder)
        "127220001",  # Malignant lymphoma of lymph nodes (disorder)
        "127225006",  # Chronic myelomonocytic leukemia (disorder)
        "128874001",  # Cutaneous CD30+ lymphoproliferative disorder (disorder)
        "128875000",  # Primary cutaneous CD30 antigen positive large T-cell lymphoma (disorder)
        "187822008",  # Fibrosarcoma of spleen (disorder)
        "188487008",  # Lymphosarcoma and reticulosarcoma (disorder)
        "188489006",  # Reticulosarcoma of lymph nodes of head, face and neck (disorder)
        "188492005",  # Reticulosarcoma of lymph nodes of axilla and upper limb (disorder)
        "188493000",  # Reticulosarcoma of lymph nodes of inguinal region and lower limb (disorder)
        "188498009",  # Lymphosarcoma (disorder)
        "188500005",  # Lymphosarcoma of lymph nodes of head, face and neck (disorder)
        "188501009",  # Lymphosarcoma of intrathoracic lymph nodes (disorder)
        "188502002",  # Lymphosarcoma of intra-abdominal lymph nodes (disorder)
        "188503007",  # Lymphosarcoma of lymph nodes of axilla and upper limb (disorder)
        "188504001",  # Lymphosarcoma of lymph nodes of inguinal region and lower limb (disorder)
        "188505000",  # Lymphosarcoma of intrapelvic lymph nodes (disorder)
        "188506004",  # Lymphosarcoma of spleen (disorder)
        "188507008",  # Lymphosarcoma of lymph nodes of multiple sites (disorder)
        "188511002",  # Burkitt's lymphoma of intrathoracic lymph nodes (disorder)
        "188512009",  # Burkitt's lymphoma of intra-abdominal lymph nodes (disorder)
        "188513004",  # Burkitt's lymphoma of lymph nodes of axilla and upper limb (disorder)
        "188514005",  # Burkitt's lymphoma of lymph nodes of inguinal region and lower limb (disorder)
        "188515006",  # Burkitt's lymphoma of intrapelvic lymph nodes (disorder)
        "188516007",  # Burkitt's lymphoma of spleen (disorder)
        "188517003",  # Burkitt's lymphoma of lymph nodes of multiple sites (disorder)
        "188524002",  # Hodgkin's paragranuloma of intrathoracic lymph nodes (disorder)
        "188529007",  # Hodgkin's paragranuloma of intrapelvic lymph nodes (disorder)
        "188531003",  # Hodgkin's paragranuloma of lymph nodes of multiple sites (disorder)
        "188536008",  # Hodgkin's granuloma of intra-abdominal lymph nodes (disorder)
        "188537004",  # Hodgkin's granuloma of lymph nodes of axilla and upper limb (disorder)
        "188538009",  # Hodgkin's granuloma of lymph nodes of inguinal region and lower limb (disorder)
        "188541000",  # Hodgkin's granuloma of lymph nodes of multiple sites (disorder)
        "188547001",  # Hodgkin's sarcoma of lymph nodes of axilla and upper limb (disorder)
        "188548006",  # Hodgkin's sarcoma of lymph nodes of inguinal region and lower limb (disorder)
        "188551004",  # Hodgkin's sarcoma of lymph nodes of multiple sites (disorder)
        "188554007",  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of head, face and neck (disorder)
        "188558005",  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of axilla and upper limb (disorder)
        "188559002",  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of inguinal region and lower limb (disorder)
        "188562004",  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of multiple sites (disorder)
        "188565002",  # Hodgkin's disease, nodular sclerosis of lymph nodes of head, face and neck (disorder)
        "188566001",  # Hodgkin's disease, nodular sclerosis of intrathoracic lymph nodes (disorder)
        "188567005",  # Hodgkin's disease, nodular sclerosis of intra-abdominal lymph nodes (disorder)
        "188568000",  # Hodgkin's disease, nodular sclerosis of lymph nodes of axilla and upper limb (disorder)
        "188569008",  # Hodgkin's disease, nodular sclerosis of lymph nodes of inguinal region and lower limb (disorder)
        "188570009",  # Hodgkin's disease, nodular sclerosis of intrapelvic lymph nodes (disorder)
        "188572001",  # Hodgkin's disease, nodular sclerosis of lymph nodes of multiple sites (disorder)
        "188575004",  # Hodgkin's disease, mixed cellularity of lymph nodes of head, face and neck (disorder)
        "188576003",  # Hodgkin's disease, mixed cellularity of intrathoracic lymph nodes (disorder)
        "188577007",  # Hodgkin's disease, mixed cellularity of intra-abdominal lymph nodes (disorder)
        "188578002",  # Hodgkin's disease, mixed cellularity of lymph nodes of axilla and upper limb (disorder)
        "188579005",  # Hodgkin's disease, mixed cellularity of lymph nodes of inguinal region and lower limb (disorder)
        "188580008",  # Hodgkin's disease, mixed cellularity of intrapelvic lymph nodes (disorder)
        "188582000",  # Hodgkin's disease, mixed cellularity of lymph nodes of multiple sites (disorder)
        "188585003",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of head, face and neck (disorder)
        "188586002",  # Hodgkin's disease, lymphocytic depletion of intrathoracic lymph nodes (disorder)
        "188587006",  # Hodgkin's disease, lymphocytic depletion of intra-abdominal lymph nodes (disorder)
        "188589009",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of axilla and upper limb (disorder)
        "188590000",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of inguinal region and lower limb (disorder)
        "188591001",  # Hodgkin's disease, lymphocytic depletion of intrapelvic lymph nodes (disorder)
        "188592008",  # Hodgkin's disease, lymphocytic depletion of spleen (disorder)
        "188593003",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of multiple sites (disorder)
        "188609000",  # Nodular lymphoma of lymph nodes of head, face and neck (disorder)
        "188612002",  # Nodular lymphoma of lymph nodes of axilla and upper limb (disorder)
        "188613007",  # Nodular lymphoma of lymph nodes of inguinal region and lower limb (disorder)
        "188627002",  # Mycosis fungoides of lymph nodes of multiple sites (disorder)
        "188631008",  # Sézary's disease of intrathoracic lymph nodes (disorder)
        "188632001",  # Sézary's disease of intra-abdominal lymph nodes (disorder)
        "188633006",  # Sézary's disease of lymph nodes of axilla and upper limb (disorder)
        "188634000",  # Sézary's disease of lymph nodes of inguinal region and lower limb (disorder)
        "188635004",  # Sézary's disease of intrapelvic lymph nodes (disorder)
        "188637007",  # Sézary's disease of lymph nodes of multiple sites (disorder)
        "188641006",  # Malignant histiocytosis of lymph nodes of axilla and upper limb (disorder)
        "188642004",  # Malignant histiocytosis of lymph nodes of inguinal region and lower limb (disorder)
        "188645002",  # Leukemic reticuloendotheliosis of lymph nodes of head, face and neck (disorder)
        "188648000",  # Leukemic reticuloendotheliosis of lymph nodes of axilla and upper limb (disorder)
        "188649008",  # Leukemic reticuloendotheliosis of lymph nodes of inguinal region and lower limb (disorder)
        "188660004",  # Malignant mast cell tumors (disorder)
        "188662007",  # Mast cell malignancy of lymph nodes of head, face and neck (disorder)
        "188663002",  # Mast cell malignancy of intrathoracic lymph nodes (disorder)
        "188664008",  # Mast cell malignancy of intra-abdominal lymph nodes (disorder)
        "188665009",  # Mast cell malignancy of lymph nodes of axilla and upper limb (disorder)
        "188666005",  # Mast cell malignancy of lymph nodes of inguinal region and lower limb (disorder)
        "188667001",  # Mast cell malignancy of intrapelvic lymph nodes (disorder)
        "188668006",  # Mast cell malignancy of spleen (disorder)
        "188669003",  # Mast cell malignancy of lymph nodes of multiple sites (disorder)
        "188672005",  # Follicular non-Hodgkin's mixed small cleaved and large cell lymphoma (disorder)
        "188674006",  # Diffuse malignant lymphoma - small non-cleaved cell (disorder)
        "188675007",  # Malignant lymphoma - small cleaved cell (disorder)
        "188676008",  # Malignant lymphoma - mixed small and large cell (disorder)
        "188679001",  # Diffuse non-Hodgkin's lymphoma undifferentiated (diffuse) (disorder)
        "188718006",  # Malignant plasma cell neoplasm, extramedullary plasmacytoma (disorder)
        "188725004",  # Lymphoid leukemia (disorder)
        "188726003",  # Subacute lymphoid leukemia (disorder)
        "188728002",  # Aleukemic lymphoid leukemia (disorder)
        "188729005",  # Adult T-cell leukemia (disorder)
        "188732008",  # Myeloid leukemia (disorder)
        "188733003",  # Chronic eosinophilic leukemia (disorder)
        "188734009",  # Chronic neutrophilic leukemia (disorder)
        "188736006",  # Subacute myeloid leukemia (disorder)
        "188737002",  # Chloroma (disorder)
        "188738007",  # Granulocytic sarcoma (disorder)
        "188741003",  # Aleukemic myeloid leukemia (disorder)
        "188744006",  # Monocytic leukemia (disorder)
        "188745007",  # Chronic monocytic leukemia (disorder)
        "188746008",  # Subacute monocytic leukemia (disorder)
        "188748009",  # Aleukemic monocytic leukemia (disorder)
        "188754005",  # Megakaryocytic leukemia (disorder)
        "188768003",  # Myelomonocytic leukemia (disorder)
        "188770007",  # Subacute myelomonocytic leukemia (disorder)
        "189509003",  # Refractory anemia without sideroblasts, so stated (disorder)
        "190030009",  # Compound leukemias (disorder)
        "232075002",  # Lymphoma of retina (disorder)
        "236512004",  # Leukemic infiltrate of kidney (disorder)
        "236513009",  # Lymphoma of kidney (disorder)
        "240531002",  # African Burkitt's lymphoma (disorder)
        "254792006",  # Proliferating angioendotheliomatosis (disorder)
        "255101006",  # Sézary disease of skin (disorder)
        "255102004",  # Angioendotheliomatosis (disorder)
        "255191003",  # Localized malignant reticulohistiocytoma (disorder)
        "269475001",  # Malignant tumor of lymphoid, hemopoietic AND/OR related tissue (disorder)
        "269476000",  # Nodular lymphoma (disorder)
        "274905008",  # Malignant lymphoma - lymphocytic, intermediate differentiation (disorder)
        "276811008",  # Gastric lymphoma (disorder)
        "276815004",  # Lymphoma of intestine (disorder)
        "276836002",  # Primary cerebral lymphoma (disorder)
        "277473004",  # B-cell chronic lymphocytic leukemia (disorder)
        "277474005",  # B-cell chronic lymphocytic leukemia variant (disorder)
        "277545003",  # T-cell chronic lymphocytic leukemia (disorder)
        "277549009",  # Chronic lymphocytic prolymphocytic leukemia syndrome (disorder)
        "277550009",  # Richter's syndrome (disorder)
        "277551008",  # Splenic lymphoma with villous lymphocytes (disorder)
        "277567002",  # T-cell prolymphocytic leukemia (disorder)
        "277568007",  # Hairy cell leukemia variant (disorder)
        "277569004",  # Large granular lymphocytic leukemia (disorder)
        "277570003",  # Lymphoma with spill (disorder)
        "277571004",  # B-cell acute lymphoblastic leukemia (disorder)
        "277572006",  # Precursor B-cell acute lymphoblastic leukemia (disorder)
        "277573001",  # Common acute lymphoblastic leukemia (disorder)
        "277574007",  # Null cell acute lymphoblastic leukemia (disorder)
        "277575008",  # T-cell acute lymphoblastic leukemia (disorder)
        "277587001",  # Juvenile chronic myeloid leukemia (disorder)
        "277589003",  # Atypical chronic myeloid leukemia (disorder)
        "277597005",  # Myelodysplastic syndrome with isolated del(5q) (disorder)
        "277601005",  # Acute monoblastic leukemia (disorder)
        "277602003",  # Acute megakaryoblastic leukemia (disorder)
        "277604002",  # Acute eosinophilic leukemia (disorder)
        "277609007",  # Hodgkin's disease, lymphocytic predominance - diffuse (disorder)
        "277610002",  # Hodgkin's disease, nodular sclerosis - lymphocytic predominance (disorder)
        "277611003",  # Hodgkin's disease, nodular sclerosis - mixed cellularity (disorder)
        "277612005",  # Hodgkin's disease, nodular sclerosis - lymphocytic depletion (disorder)
        "277613000",  # Cutaneous/peripheral T-cell lymphoma (disorder)
        "277614006",  # Prethymic and thymic T-cell lymphoma/leukemia (disorder)
        "277615007",  # Low grade B-cell lymphoma (disorder)
        "277616008",  # Diffuse low grade B-cell lymphoma (disorder)
        "277617004",  # High grade B-cell lymphoma (disorder)
        "277618009",  # Follicular low grade B-cell lymphoma (disorder)
        "277619001",  # B-cell prolymphocytic leukemia (disorder)
        "277622004",  # Mucosa-associated lymphoma (disorder)
        "277623009",  # Monocytoid B-cell lymphoma (disorder)
        "277624003",  # Follicular malignant lymphoma - mixed cell type (disorder)
        "277625002",  # Follicular malignant lymphoma - small cleaved cell (disorder)
        "277626001",  # Diffuse high grade B-cell lymphoma (disorder)
        "277627005",  # Nodular high grade B-cell lymphoma (disorder)
        "277628000",  # Diffuse malignant lymphoma - large cleaved cell (disorder)
        "277629008",  # Diffuse malignant lymphoma - large non-cleaved cell (disorder)
        "277632006",  # Diffuse malignant lymphoma - centroblastic polymorphic (disorder)
        "277637000",  # Large cell anaplastic lymphoma (disorder)
        "277642008",  # Low grade T-cell lymphoma (disorder)
        "277643003",  # High grade T-cell lymphoma (disorder)
        "277651000",  # Peripheral T-cell lymphoma - pleomorphic small cell (disorder)
        "277653002",  # Peripheral T-cell lymphoma - pleomorphic medium and large cell (disorder)
        "277654008",  # Enteropathy-associated T-cell lymphoma (disorder)
        "277664004",  # Malignant lymphoma of testis (disorder)
        "278051002",  # Malignant lymphoma of thyroid gland (disorder)
        "278052009",  # Malignant lymphoma of breast (disorder)
        "278189009",  # Hypergranular promyelocytic leukemia (disorder)
        "278453007",  # Acute biphenotypic leukemia (disorder)
        "285769009",  # Acute promyelocytic leukemia - hypogranular variant (disorder)
        "285776004",  # Intermediate grade B-cell lymphoma (disorder)
        "285839005",  # Acute myelomonocytic leukemia - eosinophilic variant (disorder)
        "302841002",  # Malignant lymphoma - small lymphocytic (disorder)
        "302842009",  # Diffuse malignant lymphoma - centroblastic (disorder)
        "302845006",  # Nodular malignant lymphoma, lymphocytic - well differentiated (disorder)
        "302848008",  # Nodular malignant lymphoma, lymphocytic - intermediate differentiation (disorder)
        "302855005",  # Subacute leukemia (disorder)
        "302856006",  # Aleukemic leukemia (disorder)
        "303017006",  # Malignant lymphoma, convoluted cell type (disorder)
        "303055001",  # Malignant lymphoma, follicular center cell (disorder)
        "303056000",  # Malignant lymphoma, follicular center cell, cleaved (disorder)
        "303057009",  # Malignant lymphoma, follicular center cell, non-cleaved (disorder)
        "307340003",  # Monosomy 7 syndrome (disorder)
        "307341004",  # Atypical hairy cell leukemia (disorder)
        "307592006",  # Basophilic leukemia (disorder)
        "307617006",  # Neutrophilic leukemia (disorder)
        "307622006",  # Prolymphocytic lymphosarcoma (disorder)
        "307623001",  # Malignant lymphoma - lymphoplasmacytic (disorder)
        "307624007",  # Diffuse malignant lymphoma - centroblastic-centrocytic (disorder)
        "307625008",  # Malignant lymphoma - centrocytic (disorder)
        "307633009",  # Hodgkin's disease, lymphocytic depletion, diffuse fibrosis (disorder)
        "307634003",  # Hodgkin's disease, lymphocytic depletion, reticular type (disorder)
        "307635002",  # Hodgkin's disease, nodular sclerosis - cellular phase (disorder)
        "307636001",  # Malignant lymphoma, mixed lymphocytic-histiocytic, nodular (disorder)
        "307637005",  # Malignant lymphoma, centroblastic-centrocytic, follicular (disorder)
        "307646004",  # Malignant lymphoma, lymphocytic, poorly differentiated, nodular (disorder)
        "307647008",  # Malignant lymphoma, centroblastic type, follicular (disorder)
        "307649006",  # Microglioma (disorder)
        "307650006",  # Histiocytic medullary reticulosis (disorder)
        "308121000",  # Follicular non-Hodgkin's lymphoma (disorder)
        "359631009",  # Acute myeloid leukemia, minimal differentiation, FAB M0 (disorder)
        "359640008",  # Acute myeloid leukemia without maturation, FAB M1 (disorder)
        "359648001",  # Acute myeloid leukemia with maturation, FAB M2 (disorder)
        "371012000",  # Acute lymphoblastic leukemia, transitional pre-B-cell (disorder)
        "371134001",  # Malignant lymphoma, large cell, polymorphous, immunoblastic (disorder)
        "373168002",  # Reticulosarcoma (disorder)
        "397008008",  # Aggressive lymphadenopathic mastocytosis with eosinophilia (disorder)
        "397009000",  # Mast cell malignancy (disorder)
        "397011009",  # Mast cell malignancy of lymph nodes (disorder)
        "397015000",  # Systemic mastocytosis with associated clonal hematological non-mast cell lineage disease (disorder)
        "398623004",  # Refractory anemia with excess blasts (disorder)
        "400001003",  # Primary cutaneous lymphoma (disorder)
        "400122007",  # Primary cutaneous T-cell lymphoma (disorder)
        "402880009",  # Primary cutaneous large T-cell lymphoma (disorder)
        "402881008",  # Primary cutaneous B-cell lymphoma (disorder)
        "402882001",  # Hodgkin's disease affecting skin (disorder)
        "404103007",  # Lymphomatoid papulosis type A (CD-30 positive type) (disorder)
        "404104001",  # Lymphomatoid papulosis type B - mycosis fungoides-like (disorder)
        "404105000",  # Lymphomatoid papulosis type C (anaplastic large-cell lymphoma-like) (disorder)
        "404106004",  # Lymphomatoid papulosis with Hodgkin's disease (disorder)
        "404107008",  # Patch/plaque stage mycosis fungoides (disorder)
        "404108003",  # Poikilodermatous mycosis fungoides (disorder)
        "404109006",  # Follicular mucinosis type mycosis fungoides (disorder)
        "404110001",  # Hypomelanotic mycosis fungoides (disorder)
        "404111002",  # Lymphomatoid papulosis-associated mycosis fungoides (disorder)
        "404112009",  # Granulomatous mycosis fungoides (disorder)
        "404113004",  # Tumor stage mycosis fungoides (disorder)
        "404114005",  # Erythrodermic mycosis fungoides (disorder)
        "404115006",  # Bullous mycosis fungoides (disorder)
        "404116007",  # Mycosis fungoides with systemic infiltration (disorder)
        "404117003",  # Spongiotic mycosis fungoides (disorder)
        "404118008",  # Syringotropic mycosis fungoides (disorder)
        "404119000",  # Pagetoid reticulosis (disorder)
        "404120006",  # Localized pagetoid reticulosis (disorder)
        "404121005",  # Generalized pagetoid reticulosis (disorder)
        "404122003",  # Leukemic infiltration of skin (chronic T-cell lymphocytic leukemia) (disorder)
        "404123008",  # Leukemic infiltration of skin (T-cell prolymphocytic leukemia) (disorder)
        "404124002",  # Leukemic infiltration of skin (T-cell lymphoblastic leukemia) (disorder)
        "404126000",  # CD-30 positive pleomorphic large T-cell cutaneous lymphoma (disorder)
        "404127009",  # CD-30 positive T-immunoblastic cutaneous lymphoma (disorder)
        "404128004",  # CD-30 negative cutaneous T-cell lymphoma (disorder)
        "404129007",  # CD-30 negative anaplastic large T-cell cutaneous lymphoma (disorder)
        "404130002",  # CD-30 negative pleomorphic large T-cell cutaneous lymphoma (disorder)
        "404131003",  # CD-30 negative T-immunoblastic cutaneous lymphoma (disorder)
        "404133000",  # Subcutaneous panniculitic cutaneous T-cell lymphoma (disorder)
        "404134006",  # Anaplastic large T-cell systemic malignant lymphoma (disorder)
        "404135007",  # Angiocentric natural killer/T-cell malignant lymphoma involving skin (disorder)
        "404136008",  # Aggressive natural killer-cell leukemia involving skin (disorder)
        "404137004",  # Precursor B-cell lymphoblastic lymphoma involving skin (disorder)
        "404138009",  # Small lymphocytic B-cell lymphoma involving skin (disorder)
        "404139001",  # Leukemic infiltration of skin in hairy-cell leukemia (disorder)
        "404140004",  # Primary cutaneous marginal zone B-cell lymphoma (disorder)
        "404141000",  # Primary cutaneous immunocytoma (disorder)
        "404142007",  # Primary cutaneous plasmacytoma (disorder)
        "404143002",  # Primary cutaneous follicular center B-cell lymphoma (disorder)
        "404144008",  # Primary cutaneous diffuse large cell B-cell lymphoma (disorder)
        "404145009",  # Primary cutaneous anaplastic large cell B-cell lymphoma (disorder)
        "404147001",  # Follicular center B-cell lymphoma (nodal/systemic with skin involvement) (disorder)
        "404148006",  # Diffuse large B-cell lymphoma (nodal/systemic with skin involvement) (disorder)
        "404149003",  # Lymphoplasmacytic B-cell lymphoma, nodal/systemic with skin involvement (disorder)
        "404150003",  # Mantle cell B-cell lymphoma (nodal/systemic with skin involvement) (disorder)
        "404151004",  # Leukemic infiltration of skin in myeloid leukemia (disorder)
        "404152006",  # Leukemic infiltration of skin in acute myeloid leukemia (disorder)
        "404153001",  # Leukemic infiltration of skin in chronic myeloid leukemia (disorder)
        "404154007",  # Leukemic infiltration of skin in monocytic leukemia (disorder)
        "404155008",  # Granulocytic sarcoma affecting skin (disorder)
        "404157000",  # Specific skin infiltration in Hodgkin's disease (disorder)
        "404169008",  # Malignant histiocytosis involving skin (disorder)
        "404172001",  # Mast cell leukemia affecting skin (disorder)
        "413389003",  # Accelerated phase chronic myeloid leukemia (disorder)
        "413441006",  # Acute monocytic leukemia, FAB M5b (disorder)
        "413442004",  # Acute monocytic/monoblastic leukemia (disorder)
        "413537009",  # Angioimmunoblastic T-cell lymphoma (disorder)
        "413656006",  # Blastic phase chronic myeloid leukemia (disorder)
        "413842007",  # Chronic myeloid leukemia in lymphoid blast crisis (disorder)
        "413843002",  # Chronic myeloid leukemia in myeloid blast crisis (disorder)
        "413847001",  # Chronic phase chronic myeloid leukemia (disorder)
        "414166008",  # Extranodal natural killer/T-cell lymphoma, nasal type (disorder)
        "414780005",  # Mucosa-associated lymphoid tissue lymphoma of orbit (disorder)
        "414785000",  # Multiple solitary plasmacytomas (disorder)
        "415112005",  # Plasmacytoma (disorder)
        "415283002",  # Refractory anemia with excess blasts-1 (disorder)
        "415284008",  # Refractory anemia with excess blasts-2 (disorder)
        "415285009",  # Refractory cytopenia with multilineage dysplasia (disorder)
        "415286005",  # Refractory cytopenia with multilineage dysplasia and ringed sideroblasts (disorder)
        "415287001",  # Relapsing chronic myeloid leukemia (disorder)
        "420302007",  # Reticulosarcoma associated with acquired immunodeficiency syndrome (disorder)
        "420519005",  # Malignant lymphoma of the eye region (disorder)
        "420788006",  # Intraocular non-Hodgkin malignant lymphoma (disorder)
        "421246008",  # Precursor T-cell lymphoblastic lymphoma (disorder)
        "421283008",  # Primary lymphoma of brain associated with acquired immunodeficiency syndrome (disorder)
        "422853008",  # Lymphoma of retroperitoneal space (disorder)
        "425688002",  # Philadelphia chromosome-positive acute lymphoblastic leukemia (disorder)
        "425749006",  # Subacute myeloid leukemia in remission (disorder)
        "425869007",  # Acute promyelocytic leukemia, FAB M3, in remission (disorder)
        "425941003",  # Precursor B-cell acute lymphoblastic leukemia in remission (disorder)
        "426071002",  # Hodgkin's disease in remission (disorder)
        "426124006",  # Acute myeloid leukemia with maturation, FAB M2, in remission (disorder)
        "426217000",  # Aleukemic leukemia in remission (disorder)
        "426248008",  # Aleukemic lymphoid leukemia in remission (disorder)
        "426336007",  # Solitary osseous myeloma (disorder)
        "426370008",  # Subacute lymphoid leukemia in remission (disorder)
        "426642002",  # Erythroleukemia, FAB M6 in remission (disorder)
        "426885008",  # Hodgkin's disease, lymphocytic depletion of lymph nodes of head (disorder)
        "427056005",  # Subacute leukemia in remission (disorder)
        "427141003",  # Malignant lymphoma in remission (disorder)
        "427642009",  # T-cell acute lymphoblastic leukemia in remission (disorder)
        "427658007",  # Acute myelomonocytic leukemia, FAB M4, in remission (disorder)
        "430338009",  # Smoldering chronic lymphocytic leukemia (disorder)
        "441559006",  # Mantle cell lymphoma of spleen (disorder)
        "441962003",  # Large cell lymphoma of intrapelvic lymph nodes (disorder)
        "442537007",  # Non-Hodgkin lymphoma associated with Human immunodeficiency virus infection (disorder)
        "443487006",  # Mantle cell lymphoma (disorder)
        "444597005",  # Extranodal marginal zone lymphoma of mucosa-associated lymphoid tissue of stomach (disorder)
        "444910004",  # Primary mediastinal (thymic) large B-cell lymphoma (disorder)
        "444911000",  # Acute myeloid leukemia with t(9:11)(p22;q23); MLLT3-MLL (disorder)
        "445227008",  # Juvenile myelomonocytic leukemia (disorder)
        "445269007",  # Extranodal marginal zone B-cell lymphoma of mucosa-associated lymphoid tissue (disorder)
        "445406001",  # Hepatosplenic T-cell lymphoma (disorder)
        "445448008",  # Acute myeloid leukemia with myelodysplasia-related changes (disorder)
        "447100004",  # Marginal zone lymphoma (disorder)
        "447656001",  # Lymphoma of pylorus of stomach (disorder)
        "447658000",  # Lymphoma of fundus of stomach (disorder)
        "447766003",  # Lymphoma of pyloric antrum of stomach (disorder)
        "447805007",  # Lymphoma of greater curvature of stomach (disorder)
        "447806008",  # Lymphoma of cardia of stomach (disorder)
        "447989004",  # Non-Hodgkin's lymphoma of extranodal site (disorder)
        "448212009",  # Anaplastic lymphoma kinase negative anaplastic large cell lymphoma (disorder)
        "448213004",  # Diffuse non-Hodgkin's lymphoma of prostate (disorder)
        "448217003",  # Follicular non-Hodgkin's lymphoma of prostate (disorder)
        "448220006",  # Non-Hodgkin's lymphoma of bone (disorder)
        "448231003",  # Follicular non-Hodgkin's lymphoma of nose (disorder)
        "448254007",  # Non-Hodgkin's lymphoma of central nervous system (disorder)
        "448269008",  # Lymphoma of lesser curvature of stomach (disorder)
        "448317000",  # Follicular non-Hodgkin's lymphoma of soft tissue (disorder)
        "448319002",  # Diffuse non-Hodgkin's lymphoma of nasopharynx (disorder)
        "448354009",  # Non-Hodgkin's lymphoma of intestine (disorder)
        "448371005",  # Non-Hodgkin's lymphoma of nasopharynx (disorder)
        "448372003",  # Non-Hodgkin's lymphoma of lung (disorder)
        "448376000",  # Non-Hodgkin's lymphoma of ovary (disorder)
        "448384001",  # Non-Hodgkin's lymphoma of nose (disorder)
        "448386004",  # Non-Hodgkin's lymphoma of oral cavity (disorder)
        "448387008",  # Non-Hodgkin's lymphoma of testis (disorder)
        "448447004",  # Non-Hodgkin's lymphoma of skin (disorder)
        "448465000",  # Diffuse non-Hodgkin's lymphoma of testis (disorder)
        "448468003",  # Diffuse non-Hodgkin's lymphoma of oral cavity (disorder)
        "448553002",  # Lymphoma of pelvis (disorder)
        "448555009",  # Lymphoma of body of stomach (disorder)
        "448560008",  # Diffuse non-Hodgkin's lymphoma of extranodal site (disorder)
        "448561007",  # Follicular non-Hodgkin's lymphoma of extranodal site (disorder)
        "448607004",  # Diffuse non-Hodgkin's lymphoma of uterine cervix (disorder)
        "448609001",  # Diffuse non-Hodgkin's lymphoma of ovary (disorder)
        "448663003",  # Diffuse non-Hodgkin's lymphoma of stomach (disorder)
        "448666006",  # Follicular non-Hodgkin's lymphoma of bone (disorder)
        "448672006",  # Follicular non-Hodgkin's lymphoma of lung (disorder)
        "448709005",  # Non-Hodgkin's lymphoma of stomach (disorder)
        "448738008",  # Non-Hodgkin's lymphoma of soft tissue (disorder)
        "448774004",  # Non-Hodgkin's lymphoma of uterine cervix (disorder)
        "448865007",  # Follicular non-Hodgkin's lymphoma of skin (disorder)
        "448867004",  # Diffuse non-Hodgkin's lymphoma of lung (disorder)
        "448995000",  # Follicular non-Hodgkin's lymphoma of central nervous system (disorder)
        "449053004",  # Lymphoma of lower esophagus (disorder)
        "449058008",  # Follicular non-Hodgkin's lymphoma of tonsil (disorder)
        "449059000",  # Follicular non-Hodgkin's lymphoma of uterine cervix (disorder)
        "449063007",  # Follicular non-Hodgkin's lymphoma of oral cavity (disorder)
        "449065000",  # Diffuse non-Hodgkin's lymphoma of nose (disorder)
        "449072004",  # Lymphoma of gastrointestinal tract (disorder)
        "449074003",  # Lymphoma of small intestine (disorder)
        "449075002",  # Lymphoma of cardioesophageal junction (disorder)
        "449108003",  # Philadelphia chromosome positive chronic myelogenous leukemia (disorder)
        "449173006",  # Diffuse non-Hodgkin's lymphoma of tonsil (disorder)
        "449176003",  # Diffuse non-Hodgkin's lymphoma of intestine (disorder)
        "449177007",  # Diffuse non-Hodgkin's lymphoma of bone (disorder)
        "449216004",  # Diffuse non-Hodgkin's lymphoma of soft tissue (disorder)
        "449217008",  # Diffuse non-Hodgkin's lymphoma of skin (disorder)
        "449218003",  # Lymphoma of sigmoid colon (disorder)
        "449219006",  # Follicular non-Hodgkin's lymphoma of nasopharynx (disorder)
        "449220000",  # Diffuse follicle center lymphoma (disorder)
        "449221001",  # Diffuse non-Hodgkin's lymphoma of central nervous system (disorder)
        "449222008",  # Follicular non-Hodgkin's lymphoma of stomach (disorder)
        "449292003",  # Non-Hodgkin's lymphoma of tonsil (disorder)
        "449307001",  # Follicular non-Hodgkin's lymphoma of ovary (disorder)
        "449318001",  # Non-Hodgkin's lymphoma of prostate (disorder)
        "449386007",  # Philadelphia chromosome negative chronic myelogenous leukemia (disorder)
        "449418000",  # Follicular non-Hodgkin's lymphoma of testis (disorder)
        "449419008",  # Follicular non-Hodgkin's lymphoma of intestine (disorder)
        "698646006",  # Acute monoblastic leukemia in remission (disorder)
        "699657009",  # Hepatosplenic gamma-delta cell lymphoma (disorder)
        "699818003",  # T-cell large granular lymphocytic leukemia (disorder)
        "702446006",  # Core binding factor acute myeloid leukemia (disorder)
        "702476004",  # Therapy-related myelodysplastic syndrome (disorder)
        "702785000",  # Large cell anaplastic lymphoma T cell and Null cell type (disorder)
        "702786004",  # Follicular non-Hodgkin's lymphoma diffuse follicle center sub-type grade 1 (disorder)
        "702977001",  # Follicular non-Hodgkin's lymphoma diffuse follicle center cell sub-type grade 2 (disorder)
        "703387000",  # Acute myeloid leukemia with normal karyotype (disorder)
        "703626001",  # Anaplastic large cell lymphoma, T/Null cell, primary systemic type (disorder)
        "705061009",  # Childhood myelodysplastic syndrome (disorder)
        "709471005",  # Periodontitis co-occurrent with leukemia (disorder)
        "713325002",  # Primary cerebral lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        "713483007",  # Reticulosarcoma co-occurrent with human immunodeficiency virus infection (disorder)
        "713516007",  # Primary effusion lymphoma (disorder)
        "713718006",  # Diffuse non-Hodgkin immunoblastic lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        "713897006",  # Burkitt lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        "714251006",  # Philadelphia chromosome-negative precursor B-cell acute lymphoblastic leukemia (disorder)
        "714463003",  # Primary effusion lymphoma co-occurrent with infection caused by Human herpesvirus 8 (disorder)
        "715664005",  # Interdigitating dendritic cell sarcoma (disorder)
        "715950008",  # Anaplastic lymphoma kinase positive large B-cell lymphoma (disorder)
        "716788007",  # Epstein-Barr virus positive diffuse large B-cell lymphoma of elderly (disorder)
        "718200007",  # Primary pulmonary lymphoma (disorder)
        "721302006",  # Refractory anemia with ringed sideroblasts associated with marked thrombocytosis (disorder)
        "721303001",  # Refractory neutropenia (disorder)
        "721304007",  # Refractory thrombocytopenia (disorder)
        "721305008",  # Acute myeloid leukemia due to recurrent genetic abnormality (disorder)
        "721306009",  # Therapy related acute myeloid leukemia and myelodysplastic syndrome (disorder)
        "721308005",  # Acute leukemia of ambiguous lineage (disorder)
        "721310007",  # Aggressive natural killer-cell leukemia (disorder)
        "721313009",  # Indeterminate dendritic cell neoplasm (disorder)
        "721314003",  # Fibroblastic reticular cell neoplasm (disorder)
        "721555001",  # Follicular lymphoma of small intestine (disorder)
        "721762007",  # Adult T-cell leukemia/lymphoma of skin (disorder)
        "722795004",  # Meningeal leukemia (disorder)
        "722953004",  # B-cell lymphoma unclassifiable with features intermediate between Burkitt lymphoma and diffuse large B-cell lymphoma (disorder)
        "722954005",  # B-cell lymphoma unclassifiable with features intermediate between classical Hodgkin lymphoma and diffuse large B-cell lymphoma (disorder)
        "723889003",  # B lymphoblastic leukemia lymphoma with t(9:22) (q34;q11.2); BCR-ABL 1 (disorder)
        "724644005",  # Myeloid leukemia co-occurrent with Down syndrome (disorder)
        "724645006",  # T-cell histiocyte rich large B-cell lymphoma (disorder)
        "724647003",  # Diffuse large B-cell lymphoma co-occurrent with chronic inflammation caused by Epstein-Barr virus (disorder)
        "724648008",  # Plasmablastic lymphoma (disorder)
        "724649000",  # Langerhans cell sarcoma (disorder)
        "724650000",  # Primary follicular dendritic cell sarcoma (disorder)
        "725390002",  # Acute myeloid leukemia with t(8;16)(p11;p13) translocation (disorder)
        "725437002",  # Chronic lymphocytic leukemia genetic mutation variant (disorder)
        "726721002",  # Nodal marginal zone B-cell lymphoma (disorder)
        "733598001",  # Acute myeloid leukemia with t(6;9)(p23;q34) translocation (disorder)
        "733627006",  # Primary cutaneous gamma-delta-positive T-cell lymphoma (disorder)
        "734066005",  # Diffuse large B-cell lymphoma of central nervous system (disorder)
        "734522002",  # Acute myeloid leukemia with FMS-like tyrosine kinase-3 mutation (disorder)
        "735332000",  # Primary cutaneous diffuse large cell B-cell lymphoma of lower extremity (disorder)
        "736322001",  # Pediatric follicular lymphoma (disorder)
        "738527001",  # Myeloid and/or lymphoid neoplasm associated with platelet derived growth factor receptor alpha rearrangement (disorder)
        "762315004",  # Therapy related acute myeloid leukemia due to and following administration of antineoplastic agent (disorder)
        "762690000",  # Classical Hodgkin lymphoma (disorder)
        "763309005",  # Acute myeloid leukemia with nucleophosmin 1 somatic mutation (disorder)
        "763477007",  # Primary lymphoma of conjunctiva (disorder)
        "763666008",  # Splenic marginal zone B-cell lymphoma (disorder)
        "763719001",  # Hydroa vacciniforme-like lymphoma (disorder)
        "763796007",  # Megakaryoblastic acute myeloid leukemia with t(1;22)(p13;q13) (disorder)
        "763884007",  # Splenic diffuse red pulp small B-cell lymphoma (disorder)
        "764855007",  # Acute myeloid leukemia with CCAAT/enhancer binding protein alpha somatic mutation (disorder)
        "764940002",  # Inherited acute myeloid leukemia (disorder)
        "765136002",  # Primary cutaneous CD8 positive aggressive epidermotropic cytotoxic T-cell lymphoma (disorder)
        "765328000",  # Classic mycosis fungoides (disorder)
        "766045006",  # Acute myeloid leukemia and myelodysplastic syndrome related to alkylating agent (disorder)
        "766046007",  # Acute myeloid leukemia and myelodysplastic syndrome related to topoisomerase type 2 inhibitor (disorder)
        "766048008",  # Acute myeloid leukemia and myelodysplastic syndrome related to radiation (disorder)
        "766935007",  # Primary bone lymphoma (disorder)
        "770402000",  # Aleukemic mast cell leukemia (disorder)
        "773537001",  # Differentiation syndrome due to and following chemotherapy co-occurrent with acute promyelocytic leukemia (disorder)
        "773995001",  # Primary cutaneous anaplastic large cell lymphoma (disorder)
        "780841002",  # Classic mast cell leukemia (disorder)
        "780844005",  # Acute myeloid leukemia with inv(3)(q21q26.2) or t(3;3)(q21;q26.2); RPN1-EVI1 (disorder)
        "783541009",  # Breast implant-associated anaplastic large-cell lymphoma (disorder)
        "783565007",  # Indolent T-cell lymphoproliferative disorder of gastrointestinal tract (disorder)
        "783615009",  # Erythropoietic uroporphyria associated with myeloid malignancy (disorder)
        "788874003",  # B-cell prolymphocytic leukemia in remission (disorder)
        "788972003",  # Juvenile myelomonocytic leukemia in remission (disorder)
        "830057003",  # Relapsing classical Hodgkin lymphoma (disorder)
        "836486002",  # Lymphomatous infiltrate of kidney (disorder)
        "838340006",  # B lymphoblastic leukemia lymphoma with t(5;14)(q31;q32); IL3-IGH (disorder)
        "838341005",  # B lymphoblastic leukemia lymphoma with t(v;11q23); MLL rearranged (disorder)
        "838342003",  # B lymphoblastic leukemia lymphoma with t(12;21) (p13;q22); TEL/AML1 (ETV6-RUNX1) (disorder)
        "838343008",  # B lymphoblastic leukemia lymphoma with t(1;19)(Q23;P13.3); E2A-PBX1 (TCF3/PBX1) (disorder)
        "838344002",  # B lymphoblastic leukemia lymphoma with hypodiploidy (disorder)
        "838346000",  # B lymphoblastic leukemia lymphoma with hyperdiploidy (disorder)
        "838355002",  # Acute myeloid leukemia with inv(16)(p13.1q22) or t(16;16)(p13.1;q22) CBFB-MYH11 (disorder)
        "840423002",  # Diffuse large B-cell lymphoma of small intestine (disorder)
        "840424008",  # Diffuse large B-cell lymphoma of stomach (disorder)
        "866098005",  # Large B-cell lymphoma arising in HHV8-associated multicentric Castleman disease (disorder)
        "12281000132104",  # Relapsing acute myeloid leukemia (disorder)
        "12291000132102",  # Refractory acute myeloid leukemia (disorder)
        "12301000132103",  # Acute lymphoid leukemia relapse (disorder)
        "12311000132101",  # Refractory acute lymphoid leukemia (disorder)
        "61291000119103",  # Disorder of central nervous system co-occurrent and due to acute lymphoid leukemia in remission (disorder)
        "61301000119102",  # Disorder of central nervous system co-occurrent and due to acute lymphoid leukemia (disorder)
        "61311000119104",  # Acute myeloid leukemia, minimal differentiation, FAB M0 in remission (disorder)
        "61321000119106",  # Acute myeloid leukemia without maturation, FAB M1 in remission (disorder)
        "68261000119102",  # Diffuse non-Hodgkin's lymphoma Lugano stage I (disorder)
        "68271000119108",  # Diffuse non-Hodgkin's lymphoma Lugano stage II (disorder)
        "68291000119109",  # Diffuse non-Hodgkin's lymphoma Lugano stage IV (disorder)
        "84811000119107",  # Follicular dendritic sarcoma of intraabdominal lymph node (disorder)
        "84831000119102",  # Follicular dendritic sarcoma of lymph nodes of multiple sites (disorder)
        "116691000119101",  # Marginal zone lymphoma of spleen (disorder)
        "116711000119103",  # Marginal zone lymphoma of inguinal lymph node (disorder)
        "116741000119104",  # Marginal zone lymphoma of thoracic lymph node (disorder)
        "116811000119106",  # Non-Hodgkin lymphoma of central nervous system metastatic to lymph node of lower limb (disorder)
        "116821000119104",  # Non-Hodgkin lymphoma of central nervous system metastatic to lymph node of upper limb (disorder)
        "116841000119105",  # Marginal zone lymphoma of lymph nodes of multiple sites (disorder)
        "116871000119103",  # Mantle cell lymphoma of lymph nodes of multiple sites (disorder)
        "117061000119101",  # Peripheral T-cell lymphoma of spleen (disorder)
        "117091000119108",  # Peripheral T-cell lymphoma of axillary lymph node (disorder)
        "117111000119100",  # Peripheral T-cell lymphoma of thoracic lymph node (disorder)
        "117211000119105",  # Peripheral T-cell lymphoma of lymph nodes of multiple sites (disorder)
        "122881000119107",  # Chronic monocytic leukemia in relapse (disorder)
        "122901000119109",  # Myeloid leukemia in relapse (disorder)
        "122951000119108",  # Lymphoid leukemia in relapse (disorder)
        "122961000119105",  # Chronic lymphoid leukemia in relapse (disorder)
        "122981000119101",  # Plasma cell leukemia in relapse (disorder)
        "123781000119107",  # Chronic leukemia in relapse (disorder)
        "127961000119108",  # Aleukemic myeloid leukemia in relapse (disorder)
        "133751000119102",  # Lymphoma of colon (disorder)
        "350951000119101",  # B-cell lymphoma of intra-abdominal lymph nodes (disorder)
        "351211000119104",  # B-cell lymphoma of lymph nodes of multiple sites (disorder)
        "352251000119109",  # Small lymphocytic B-cell lymphoma of lymph nodes of multiple sites (disorder)
        "352411000119109",  # Small lymphocytic B-cell lymphoma of intra-abdominal lymph nodes (disorder)
        "352791000119108",  # Non-Hodgkin's lymphoma of lymph nodes of multiple sites (disorder)
        "354851000119101",  # Follicular non-Hodgkin's lymphoma of lymph nodes of multiple sites (disorder)
        "450521000124109",  # Marginal zone lymphoma of axillary lymph node (disorder)
        "456911000124108",  # Diffuse non-Hodgkin's lymphoma Lugano stage III (disorder)
        "461311000124107",  # Refractory Hodgkin lymphoma (disorder)
        "461351000124108",  # Breast implant-associated anaplastic large cell lymphoma (disorder)
    }


class Measles(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  measles infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of measles infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Measles"
    OID = "2.16.840.1.113883.3.464.1003.110.12.1053"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "B050",  # Measles complicated by encephalitis
        "B051",  # Measles complicated by meningitis
        "B052",  # Measles complicated by pneumonia
        "B053",  # Measles complicated by otitis media
        "B054",  # Measles with intestinal complications
        "B0581",  # Measles keratitis and keratoconjunctivitis
        "B0589",  # Other measles complications
        "B059",  # Measles without complication
    }
    SNOMEDCT = {
        "14189004",  # Measles (disorder)
        "28463004",  # Measles keratitis (disorder)
        "38921001",  # Measles with complication (disorder)
        "60013002",  # Measles keratoconjunctivitis (disorder)
        "74918002",  # Measles myocarditis (disorder)
        "111873003",  # Measles without complication (disorder)
        "161419000",  # History of measles (situation)
        "186561002",  # Measles complicated by meningitis (disorder)
        "186562009",  # Measles with intestinal complications (disorder)
        "195900001",  # Pneumonia co-occurrent and due to measles (disorder)
        "240483006",  # Atypical measles (disorder)
        "240484000",  # Modified measles (disorder)
        "359686005",  # Van Bogaert's sclerosing leukoencephalitis (disorder)
        "371111005",  # Measles immune (finding)
        "406592004",  # Measles of the central nervous system (disorder)
        "105841000119101",  # Immunity to measles by positive serology (finding)
    }


class Mumps(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  mumps infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of mumps infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Mumps"
    OID = "2.16.840.1.113883.3.464.1003.110.12.1032"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "B260",  # Mumps orchitis
        "B261",  # Mumps meningitis
        "B262",  # Mumps encephalitis
        "B263",  # Mumps pancreatitis
        "B2681",  # Mumps hepatitis
        "B2682",  # Mumps myocarditis
        "B2683",  # Mumps nephritis
        "B2684",  # Mumps polyneuropathy
        "B2685",  # Mumps arthritis
        "B2689",  # Other mumps complications
        "B269",  # Mumps without complication
    }
    SNOMEDCT = {
        "10665004",  # Mumps pancreatitis (disorder)
        "17121006",  # Mumps nephritis (disorder)
        "31524007",  # Mumps polyneuropathy (disorder)
        "31646008",  # Mumps encephalitis (disorder)
        "36989005",  # Mumps (disorder)
        "40099009",  # Meningoencephalitis caused by mumps (disorder)
        "44201003",  # Mumps meningitis (disorder)
        "63462008",  # Mumps myocarditis (disorder)
        "72071001",  # Mumps thyroiditis (disorder)
        "74717002",  # Mumps with complication (disorder)
        "75548002",  # Mumps oophoritis (disorder)
        "78580004",  # Mumps orchitis (disorder)
        "89231008",  # Mumps hepatitis (disorder)
        "89764009",  # Mumps arthritis (disorder)
        "111870000",  # Mumps without complication (disorder)
        "161420006",  # History of mumps (situation)
        "235123001",  # Mumps acute sialadenitis (disorder)
        "236771002",  # Mumps epididymo-orchitis (disorder)
        "237443002",  # Mastitis of mumps (disorder)
        "240526004",  # Mumps parotitis (disorder)
        "240527008",  # Mumps submandibular sialadenitis (disorder)
        "240529006",  # Mumps sublingual sialadenitis (disorder)
        "371112003",  # Mumps immune (finding)
        "105821000119107",  # Immunity to mumps by positive serology (finding)
    }


class Rubella(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  rubella infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of rubella infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Rubella"
    OID = "2.16.840.1.113883.3.464.1003.110.12.1037"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "B0600",  # Rubella with neurological complication, unspecified
        "B0601",  # Rubella encephalitis
        "B0602",  # Rubella meningitis
        "B0609",  # Other neurological complications of rubella
        "B0681",  # Rubella pneumonia
        "B0682",  # Rubella arthritis
        "B0689",  # Other rubella complications
        "B069",  # Rubella without complication
    }
    SNOMEDCT = {
        "10082001",  # Progressive rubella panencephalitis (disorder)
        "13225007",  # Rubella meningoencephalitis (disorder)
        "19431000",  # Arthritis co-occurrent and due to rubella (disorder)
        "36653000",  # Rubella (disorder)
        "51490003",  # Disease caused by Rubivirus (disorder)
        "64190005",  # Rubella myocarditis (disorder)
        "79303006",  # Expanded rubella syndrome (disorder)
        "128191000",  # Encephalomyelitis co-occurrent and due to rubella (disorder)
        "161421005",  # History of rubella (situation)
        "165792000",  # Rubella antibody present - immune (finding)
        "186567003",  # Rubella with neurological complication (disorder)
        "186570004",  # Rubella deafness (disorder)
        "192689006",  # Rubella encephalitis (disorder)
        "231985001",  # Rubella retinopathy (disorder)
        "232312000",  # Endocochlear rubella (disorder)
        "240485004",  # Hemorrhagic rubella (disorder)
        "253227001",  # Rubella cataract (disorder)
        "406112006",  # Rubella enzyme-linked immunosorbent assay test result, greater than 10iu/ml rubella specific immunoglobulin G detected (finding)
        "406113001",  # Rubella enzyme-linked immunosorbent assay test result, greater than 15iu/ml rubella specific immunoglobulin G detected (finding)
        "1092361000119109",  # Rubella pneumonia (disorder)
        "10759761000119100",  # Rubella in mother complicating childbirth (disorder)
    }


class SevereCombinedImmunodeficiency(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  severe combined immunodeficiency including both genetic and other causes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a severe combined immunodeficiency including genetic conditions and other causes.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Severe Combined Immunodeficiency"
    OID = "2.16.840.1.113883.3.464.1003.120.12.1007"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "D810",  # Severe combined immunodeficiency [SCID] with reticular dysgenesis
        "D811",  # Severe combined immunodeficiency [SCID] with low T- and B-cell numbers
        "D812",  # Severe combined immunodeficiency [SCID] with low or normal B-cell numbers
        "D819",  # Combined immunodeficiency, unspecified
    }
    SNOMEDCT = {
        "3439009",  # Severe combined immunodeficiency due to absent peripheral T cell maturation (disorder)
        "22406001",  # Severe combined immunodeficiency due to absent lymphoid stem cells (disorder)
        "31323000",  # Severe combined immunodeficiency disease (disorder)
        "36980009",  # Severe combined immunodeficiency due to absent adenosine deaminase (disorder)
        "44940001",  # Adenosine deaminase deficiency (disorder)
        "45390000",  # Severe combined immunodeficiency due to absent interleukin-2 production (disorder)
        "49555001",  # Severe combined immunodeficiency due to absent T cell receptor (disorder)
        "55602000",  # Nezelof's syndrome (disorder)
        "71904008",  # Severe combined immunodeficiency due to absent class II human leukocyte antigens (disorder)
        "111584000",  # Reticular dysgenesis (disorder)
        "111587007",  # Severe combined immunodeficiency due to absent interleukin-2 receptor (disorder)
        "190993005",  # Autosomal recessive severe combined immunodeficiency (disorder)
        "190996002",  # Severe combined immunodeficiency with reticular dysgenesis (disorder)
        "190997006",  # Severe combined immunodeficiency with low T- and B-cell numbers (disorder)
        "190998001",  # Severe combined immunodeficiency with low or normal B-cell numbers (disorder)
        "191001007",  # Major histocompatibility complex class I deficiency (disorder)
        "191002000",  # Major histocompatibility complex class II deficiency (disorder)
        "234570002",  # Severe combined immunodeficiency with maternofetal engraftment (disorder)
        "234571003",  # Warts, hypogammaglobulinemia, infections, and myelokathexis (disorder)
        "350353007",  # De Vaal's syndrome (disorder)
        "351287008",  # Reticular dysgenesis with congenital aleukocytosis (disorder)
        "362993009",  # Autosomal recessive severe combined immunodeficiency disease (disorder)
        "715982006",  # Severe combined immunodeficiency due to deoxyribonucleic acid cross-link repair protein 1c deficiency (disorder)
        "716378008",  # Combined immunodeficiency due to Zeta-chain associated protein kinase 70 deficiency (disorder)
        "716871006",  # Severe combined immunodeficiency due to deoxyribonucleic acid dependent protein kinase catalytic subunit deficiency (disorder)
        "718107000",  # Severe combined immunodeficiency T-cell negative B-cell positive due to janus kinase-3 deficiency (disorder)
        "720345008",  # Severe T-cell immunodeficiency, congenital alopecia, nail dystrophy syndrome (disorder)
        "720853005",  # Severe combined immunodeficiency, microcephaly, growth retardation, sensitivity to ionizing radiation syndrome (disorder)
        "720986005",  # Anhidrotic ectodermal dysplasia, immunodeficiency, osteopetrosis, lymphedema syndrome (disorder)
        "721977007",  # Lung fibrosis, immunodeficiency, 46,XX gonadal dysgenesis syndrome (disorder)
        "722067005",  # Severe combined immunodeficiency with hypereosinophilia (disorder)
        "724177005",  # Ligase 4 syndrome (disorder)
        "724361001",  # Hepatic veno-occlusive disease with immunodeficiency syndrome (disorder)
        "725135004",  # Combined immunodeficiency due to CD3gamma deficiency (disorder)
        "725136003",  # Immunodeficiency by defective expression of human leukocyte antigen class 1 (disorder)
        "725290000",  # Combined immunodeficiency due to partial recombination-activating gene 1 deficiency (disorder)
    }


class VaricellaZoster(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  varicella zoster infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of varicella zoster infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Varicella Zoster"
    OID = "2.16.840.1.113883.3.464.1003.110.12.1039"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "B010",  # Varicella meningitis
        "B0111",  # Varicella encephalitis and encephalomyelitis
        "B0112",  # Varicella myelitis
        "B012",  # Varicella pneumonia
        "B0181",  # Varicella keratitis
        "B0189",  # Other varicella complications
        "B019",  # Varicella without complication
        "B020",  # Zoster encephalitis
        "B021",  # Zoster meningitis
        "B0221",  # Postherpetic geniculate ganglionitis
        "B0222",  # Postherpetic trigeminal neuralgia
        "B0223",  # Postherpetic polyneuropathy
        "B0224",  # Postherpetic myelitis
        "B0229",  # Other postherpetic nervous system involvement
        "B0230",  # Zoster ocular disease, unspecified
        "B0231",  # Zoster conjunctivitis
        "B0232",  # Zoster iridocyclitis
        "B0233",  # Zoster keratitis
        "B0234",  # Zoster scleritis
        "B0239",  # Other herpes zoster eye disease
        "B027",  # Disseminated zoster
        "B028",  # Zoster with other complications
        "B029",  # Zoster without complications
    }
    SNOMEDCT = {
        "4740000",  # Herpes zoster (disorder)
        "10698009",  # Herpes zoster iridocyclitis (disorder)
        "21954000",  # Herpes zoster auricularis (disorder)
        "23737006",  # Chickenpox with complication (disorder)
        "24059009",  # Acute cerebellar ataxia caused by varicella (disorder)
        "36292003",  # Otitis externa caused by Herpes zoster (disorder)
        "38907003",  # Varicella (disorder)
        "42448002",  # Herpes zoster keratoconjunctivitis (disorder)
        "49183009",  # Herpes zoster with complication (disorder)
        "55560002",  # Disseminated herpes zoster (disorder)
        "87513003",  # Herpes zoster ophthalmicus (disorder)
        "111859007",  # Herpes zoster without complication (disorder)
        "111861003",  # Herpes zoster with nervous system complication (disorder)
        "161423008",  # History of chickenpox (situation)
        "186524006",  # Herpes zoster with ophthalmic complication (disorder)
        "195911009",  # Chickenpox pneumonia (disorder)
        "230176008",  # Herpes zoster encephalitis (disorder)
        "230198004",  # Varicella transverse myelitis (disorder)
        "230262004",  # Amyotrophy caused by herpes zoster (disorder)
        "230536009",  # Acute trigeminal herpes zoster (disorder)
        "232400003",  # Acute herpes zoster pharyngitis (disorder)
        "235059009",  # Herpes zoster infection of oral mucosa (disorder)
        "240468001",  # Neurological varicella (disorder)
        "240470005",  # Thoracic herpes zoster infection (disorder)
        "240471009",  # Lumbar herpes zoster infection (disorder)
        "240472002",  # Maxillary herpes zoster infection (disorder)
        "240473007",  # Mandibular herpes zoster infection (disorder)
        "240474001",  # Multidermatomal herpes zoster infection (disorder)
        "309465005",  # Varicella-zoster virus infection (disorder)
        "371113008",  # Varicella immune (finding)
        "397573005",  # Herpes zoster keratitis (disorder)
        "400020001",  # Herpes zoster dermatitis (disorder)
        "402897003",  # Herpes zoster involving cervical dermatome (disorder)
        "402898008",  # Herpes zoster involving sacral dermatome (disorder)
        "402899000",  # Anetoderma following varicella (disorder)
        "410500004",  # Hutchinson's sign (disorder)
        "410509003",  # Herpes zoster conjunctivitis (disorder)
        "421029004",  # Herpes zoster retinitis (disorder)
        "422127002",  # Herpes zoster associated with acquired immunodeficiency syndrome (disorder)
        "422446008",  # Herpes zoster subepithelial infiltrates (disorder)
        "422471006",  # Herpes zoster acute retinal necrosis (disorder)
        "422666006",  # Herpes zoster corneal epitheliopathy (disorder)
        "423333008",  # Exanthem caused by chicken pox (disorder)
        "423628002",  # Acute herpes zoster neuropathy (disorder)
        "424353002",  # Corneal haze caused by herpes zoster (disorder)
        "424435009",  # Herpes zoster pseudodendrites (disorder)
        "424801004",  # Herpes zoster corneal endotheliitis (disorder)
        "424941009",  # Herpes zoster radiculitis (disorder)
        "425356002",  # Herpes zoster retinal pigment epithelial detachment (disorder)
        "426570007",  # Herpes zoster interstitial keratitis (disorder)
        "428633000",  # Exanthem caused by herpes zoster (disorder)
        "713250002",  # Gingival disease caused by varicella-zoster virus (disorder)
        "713733003",  # Infection caused by herpes zoster virus co-occurrent with human immunodeficiency virus infection (disorder)
        "713964006",  # Multidermatomal infection caused by Herpes zoster co-occurrent with human immunodeficiency virus infection (disorder)
        "715223009",  # Fetal varicella syndrome (disorder)
        "723109003",  # Herpes zoster of skin and/or mucous membranes without complications (disorder)
        "838357005",  # Acute glossopharyngeal neuropathy caused by Herpes zoster (disorder)
        "12551000132107",  # Cranial neuropathy due to Herpes zoster (disorder)
        "12561000132105",  # Bacterial infection co-occurrent and due to herpes zoster (disorder)
        "12571000132104",  # Pneumonitis due to Herpes zoster (disorder)
        "98541000119101",  # Herpes zoster myelitis (disorder)
        "331071000119101",  # Herpes zoster iritis (disorder)
        "681221000119108",  # History of herpes zoster (situation)
        "15678761000119105",  # Iridocyclitis of right eye due to herpes zoster (disorder)
        "15678801000119102",  # Iridocyclitis of left eye due to herpes zoster (disorder)
        "15680201000119106",  # Keratoconjunctivitis of right eye due to herpes zoster (disorder)
        "15680241000119108",  # Keratoconjunctivitis of left eye due to herpes zoster (disorder)
        "15685081000119102",  # Corneal epitheliopathy of right eye due to herpes zoster (disorder)
        "15685121000119100",  # Corneal epitheliopathy of left eye due to herpes zoster (disorder)
        "15685201000119100",  # Interstitial keratitis of left eye caused by Herpes zoster virus (disorder)
        "15685281000119108",  # Interstitial keratitis of right eye caused by Herpes zoster virus (disorder)
        "15936581000119108",  # Otitis externa of right external auditory canal caused by human herpesvirus 3 (disorder)
        "15936621000119108",  # Otitis externa of left external auditory canal caused by human herpesvirus 3 (disorder)
        "15989271000119107",  # Conjunctivitis of right eye caused by herpes zoster virus (disorder)
        "15989311000119107",  # Conjunctivitis of bilateral eyes caused by herpes zoster virus (disorder)
        "15989351000119108",  # Conjunctivitis of left eye caused by herpes zoster virus (disorder)
        "15991711000119108",  # Iritis of right eye caused by herpes zoster virus (disorder)
        "15991791000119104",  # Iritis of left eye caused by herpes zoster virus (disorder)
    }


class Diabetes(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  diabetes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of type I or type II diabetes.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of gestational diabetes or steroid-induced diabetes.

    ** Used in:** CMS347v5, CMS134v10, CMS131v10, CMS122v10
    """

    VALUE_SET_NAME = "Diabetes"
    OID = "2.16.840.1.113883.3.464.1003.103.12.1001"
    DEFINITION_VERSION = "20190315"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "E1010",  # Type 1 diabetes mellitus with ketoacidosis without coma
        "E1011",  # Type 1 diabetes mellitus with ketoacidosis with coma
        "E1021",  # Type 1 diabetes mellitus with diabetic nephropathy
        "E1022",  # Type 1 diabetes mellitus with diabetic chronic kidney disease
        "E1029",  # Type 1 diabetes mellitus with other diabetic kidney complication
        "E10311",  # Type 1 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E10319",  # Type 1 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        "E10321",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema
        "E103211",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E103212",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E103213",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103219",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E10329",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema
        "E103291",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E103292",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E103293",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E103299",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E10331",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema
        "E103311",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E103312",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E103313",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103319",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E10339",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema
        "E103391",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E103392",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E103393",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E103399",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E10341",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema
        "E103411",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E103412",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E103413",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103419",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E10349",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema
        "E103491",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E103492",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E103493",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E103499",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E10351",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema
        "E103511",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E103512",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E103513",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E103519",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, unspecified eye
        "E103521",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E103522",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E103523",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E103529",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, unspecified eye
        "E103531",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E103532",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E103533",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E103539",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, unspecified eye
        "E103541",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E103542",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E103543",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E103549",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, unspecified eye
        "E103551",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E103552",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E103553",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E103559",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, unspecified eye
        "E10359",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema
        "E103591",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        "E103592",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        "E103593",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        "E103599",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, unspecified eye
        "E1036",  # Type 1 diabetes mellitus with diabetic cataract
        "E1037X1",  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        "E1037X2",  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        "E1037X3",  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        "E1037X9",  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, unspecified eye
        "E1039",  # Type 1 diabetes mellitus with other diabetic ophthalmic complication
        "E1040",  # Type 1 diabetes mellitus with diabetic neuropathy, unspecified
        "E1041",  # Type 1 diabetes mellitus with diabetic mononeuropathy
        "E1042",  # Type 1 diabetes mellitus with diabetic polyneuropathy
        "E1043",  # Type 1 diabetes mellitus with diabetic autonomic (poly)neuropathy
        "E1044",  # Type 1 diabetes mellitus with diabetic amyotrophy
        "E1049",  # Type 1 diabetes mellitus with other diabetic neurological complication
        "E1051",  # Type 1 diabetes mellitus with diabetic peripheral angiopathy without gangrene
        "E1052",  # Type 1 diabetes mellitus with diabetic peripheral angiopathy with gangrene
        "E1059",  # Type 1 diabetes mellitus with other circulatory complications
        "E10610",  # Type 1 diabetes mellitus with diabetic neuropathic arthropathy
        "E10618",  # Type 1 diabetes mellitus with other diabetic arthropathy
        "E10620",  # Type 1 diabetes mellitus with diabetic dermatitis
        "E10621",  # Type 1 diabetes mellitus with foot ulcer
        "E10622",  # Type 1 diabetes mellitus with other skin ulcer
        "E10628",  # Type 1 diabetes mellitus with other skin complications
        "E10630",  # Type 1 diabetes mellitus with periodontal disease
        "E10638",  # Type 1 diabetes mellitus with other oral complications
        "E10641",  # Type 1 diabetes mellitus with hypoglycemia with coma
        "E10649",  # Type 1 diabetes mellitus with hypoglycemia without coma
        "E1065",  # Type 1 diabetes mellitus with hyperglycemia
        "E1069",  # Type 1 diabetes mellitus with other specified complication
        "E108",  # Type 1 diabetes mellitus with unspecified complications
        "E109",  # Type 1 diabetes mellitus without complications
        "E1100",  # Type 2 diabetes mellitus with hyperosmolarity without nonketotic hyperglycemic-hyperosmolar coma (NKHHC)
        "E1101",  # Type 2 diabetes mellitus with hyperosmolarity with coma
        "E1110",  # Type 2 diabetes mellitus with ketoacidosis without coma
        "E1111",  # Type 2 diabetes mellitus with ketoacidosis with coma
        "E1121",  # Type 2 diabetes mellitus with diabetic nephropathy
        "E1122",  # Type 2 diabetes mellitus with diabetic chronic kidney disease
        "E1129",  # Type 2 diabetes mellitus with other diabetic kidney complication
        "E11311",  # Type 2 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E11319",  # Type 2 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        "E11321",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema
        "E113211",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E113212",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E113213",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113219",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E11329",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema
        "E113291",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E113292",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E113293",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E113299",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E11331",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema
        "E113311",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E113312",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E113313",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113319",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E11339",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema
        "E113391",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E113392",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E113393",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E113399",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E11341",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema
        "E113411",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E113412",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E113413",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113419",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E11349",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema
        "E113491",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E113492",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E113493",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E113499",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E11351",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema
        "E113511",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E113512",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E113513",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E113519",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, unspecified eye
        "E113521",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E113522",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E113523",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E113529",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, unspecified eye
        "E113531",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E113532",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E113533",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E113539",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, unspecified eye
        "E113541",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E113542",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E113543",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E113549",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, unspecified eye
        "E113551",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E113552",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E113553",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E113559",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, unspecified eye
        "E11359",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema
        "E113591",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        "E113592",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        "E113593",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        "E113599",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, unspecified eye
        "E1136",  # Type 2 diabetes mellitus with diabetic cataract
        "E1137X1",  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        "E1137X2",  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        "E1137X3",  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        "E1137X9",  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, unspecified eye
        "E1139",  # Type 2 diabetes mellitus with other diabetic ophthalmic complication
        "E1140",  # Type 2 diabetes mellitus with diabetic neuropathy, unspecified
        "E1141",  # Type 2 diabetes mellitus with diabetic mononeuropathy
        "E1142",  # Type 2 diabetes mellitus with diabetic polyneuropathy
        "E1143",  # Type 2 diabetes mellitus with diabetic autonomic (poly)neuropathy
        "E1144",  # Type 2 diabetes mellitus with diabetic amyotrophy
        "E1149",  # Type 2 diabetes mellitus with other diabetic neurological complication
        "E1151",  # Type 2 diabetes mellitus with diabetic peripheral angiopathy without gangrene
        "E1152",  # Type 2 diabetes mellitus with diabetic peripheral angiopathy with gangrene
        "E1159",  # Type 2 diabetes mellitus with other circulatory complications
        "E11610",  # Type 2 diabetes mellitus with diabetic neuropathic arthropathy
        "E11618",  # Type 2 diabetes mellitus with other diabetic arthropathy
        "E11620",  # Type 2 diabetes mellitus with diabetic dermatitis
        "E11621",  # Type 2 diabetes mellitus with foot ulcer
        "E11622",  # Type 2 diabetes mellitus with other skin ulcer
        "E11628",  # Type 2 diabetes mellitus with other skin complications
        "E11630",  # Type 2 diabetes mellitus with periodontal disease
        "E11638",  # Type 2 diabetes mellitus with other oral complications
        "E11641",  # Type 2 diabetes mellitus with hypoglycemia with coma
        "E11649",  # Type 2 diabetes mellitus with hypoglycemia without coma
        "E1165",  # Type 2 diabetes mellitus with hyperglycemia
        "E1169",  # Type 2 diabetes mellitus with other specified complication
        "E118",  # Type 2 diabetes mellitus with unspecified complications
        "E119",  # Type 2 diabetes mellitus without complications
        "E1300",  # Other specified diabetes mellitus with hyperosmolarity without nonketotic hyperglycemic-hyperosmolar coma (NKHHC)
        "E1301",  # Other specified diabetes mellitus with hyperosmolarity with coma
        "E1310",  # Other specified diabetes mellitus with ketoacidosis without coma
        "E1311",  # Other specified diabetes mellitus with ketoacidosis with coma
        "E1321",  # Other specified diabetes mellitus with diabetic nephropathy
        "E1322",  # Other specified diabetes mellitus with diabetic chronic kidney disease
        "E1329",  # Other specified diabetes mellitus with other diabetic kidney complication
        "E13311",  # Other specified diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E13319",  # Other specified diabetes mellitus with unspecified diabetic retinopathy without macular edema
        "E13321",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema
        "E133211",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E133212",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E133213",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133219",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E13329",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema
        "E133291",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E133292",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E133293",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E133299",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E13331",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema
        "E133311",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E133312",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E133313",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133319",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E13339",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema
        "E133391",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E133392",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E133393",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E133399",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E13341",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema
        "E133411",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E133412",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E133413",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133419",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, unspecified eye
        "E13349",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema
        "E133491",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E133492",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E133493",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E133499",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, unspecified eye
        "E13351",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema
        "E133511",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E133512",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E133513",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E133519",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, unspecified eye
        "E133521",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E133522",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E133523",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E133529",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, unspecified eye
        "E133531",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E133532",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E133533",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E133539",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, unspecified eye
        "E133541",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E133542",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E133543",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E133549",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, unspecified eye
        "E133551",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E133552",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E133553",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E133559",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, unspecified eye
        "E13359",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema
        "E133591",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        "E133592",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        "E133593",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        "E133599",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, unspecified eye
        "E1336",  # Other specified diabetes mellitus with diabetic cataract
        "E1337X1",  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        "E1337X2",  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        "E1337X3",  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        "E1337X9",  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, unspecified eye
        "E1339",  # Other specified diabetes mellitus with other diabetic ophthalmic complication
        "E1340",  # Other specified diabetes mellitus with diabetic neuropathy, unspecified
        "E1341",  # Other specified diabetes mellitus with diabetic mononeuropathy
        "E1342",  # Other specified diabetes mellitus with diabetic polyneuropathy
        "E1343",  # Other specified diabetes mellitus with diabetic autonomic (poly)neuropathy
        "E1344",  # Other specified diabetes mellitus with diabetic amyotrophy
        "E1349",  # Other specified diabetes mellitus with other diabetic neurological complication
        "E1351",  # Other specified diabetes mellitus with diabetic peripheral angiopathy without gangrene
        "E1352",  # Other specified diabetes mellitus with diabetic peripheral angiopathy with gangrene
        "E1359",  # Other specified diabetes mellitus with other circulatory complications
        "E13610",  # Other specified diabetes mellitus with diabetic neuropathic arthropathy
        "E13618",  # Other specified diabetes mellitus with other diabetic arthropathy
        "E13620",  # Other specified diabetes mellitus with diabetic dermatitis
        "E13621",  # Other specified diabetes mellitus with foot ulcer
        "E13622",  # Other specified diabetes mellitus with other skin ulcer
        "E13628",  # Other specified diabetes mellitus with other skin complications
        "E13630",  # Other specified diabetes mellitus with periodontal disease
        "E13638",  # Other specified diabetes mellitus with other oral complications
        "E13641",  # Other specified diabetes mellitus with hypoglycemia with coma
        "E13649",  # Other specified diabetes mellitus with hypoglycemia without coma
        "E1365",  # Other specified diabetes mellitus with hyperglycemia
        "E1369",  # Other specified diabetes mellitus with other specified complication
        "E138",  # Other specified diabetes mellitus with unspecified complications
        "E139",  # Other specified diabetes mellitus without complications
        "O24011",  # Pre-existing type 1 diabetes mellitus, in pregnancy, first trimester
        "O24012",  # Pre-existing type 1 diabetes mellitus, in pregnancy, second trimester
        "O24013",  # Pre-existing type 1 diabetes mellitus, in pregnancy, third trimester
        "O24019",  # Pre-existing type 1 diabetes mellitus, in pregnancy, unspecified trimester
        "O2402",  # Pre-existing type 1 diabetes mellitus, in childbirth
        "O2403",  # Pre-existing type 1 diabetes mellitus, in the puerperium
        "O24111",  # Pre-existing type 2 diabetes mellitus, in pregnancy, first trimester
        "O24112",  # Pre-existing type 2 diabetes mellitus, in pregnancy, second trimester
        "O24113",  # Pre-existing type 2 diabetes mellitus, in pregnancy, third trimester
        "O24119",  # Pre-existing type 2 diabetes mellitus, in pregnancy, unspecified trimester
        "O2412",  # Pre-existing type 2 diabetes mellitus, in childbirth
        "O2413",  # Pre-existing type 2 diabetes mellitus, in the puerperium
        "O24311",  # Unspecified pre-existing diabetes mellitus in pregnancy, first trimester
        "O24312",  # Unspecified pre-existing diabetes mellitus in pregnancy, second trimester
        "O24313",  # Unspecified pre-existing diabetes mellitus in pregnancy, third trimester
        "O24319",  # Unspecified pre-existing diabetes mellitus in pregnancy, unspecified trimester
        "O2432",  # Unspecified pre-existing diabetes mellitus in childbirth
        "O2433",  # Unspecified pre-existing diabetes mellitus in the puerperium
        "O24811",  # Other pre-existing diabetes mellitus in pregnancy, first trimester
        "O24812",  # Other pre-existing diabetes mellitus in pregnancy, second trimester
        "O24813",  # Other pre-existing diabetes mellitus in pregnancy, third trimester
        "O24819",  # Other pre-existing diabetes mellitus in pregnancy, unspecified trimester
        "O2482",  # Other pre-existing diabetes mellitus in childbirth
        "O2483",  # Other pre-existing diabetes mellitus in the puerperium
    }
    ICD9CM = {
        "25000",  # Diabetes mellitus without mention of complication, type II or unspecified type, not stated as uncontrolled
        "25001",  # Diabetes mellitus without mention of complication, type I [juvenile type], not stated as uncontrolled
        "25002",  # Diabetes mellitus without mention of complication, type II or unspecified type, uncontrolled
        "25003",  # Diabetes mellitus without mention of complication, type I [juvenile type], uncontrolled
        "25010",  # Diabetes with ketoacidosis, type II or unspecified type, not stated as uncontrolled
        "25011",  # Diabetes with ketoacidosis, type I [juvenile type], not stated as uncontrolled
        "25012",  # Diabetes with ketoacidosis, type II or unspecified type, uncontrolled
        "25013",  # Diabetes with ketoacidosis, type I [juvenile type], uncontrolled
        "25020",  # Diabetes with hyperosmolarity, type II or unspecified type, not stated as uncontrolled
        "25021",  # Diabetes with hyperosmolarity, type I [juvenile type], not stated as uncontrolled
        "25022",  # Diabetes with hyperosmolarity, type II or unspecified type, uncontrolled
        "25023",  # Diabetes with hyperosmolarity, type I [juvenile type], uncontrolled
        "25030",  # Diabetes with other coma, type II or unspecified type, not stated as uncontrolled
        "25031",  # Diabetes with other coma, type I [juvenile type], not stated as uncontrolled
        "25032",  # Diabetes with other coma, type II or unspecified type, uncontrolled
        "25033",  # Diabetes with other coma, type I [juvenile type], uncontrolled
        "25040",  # Diabetes with renal manifestations, type II or unspecified type, not stated as uncontrolled
        "25041",  # Diabetes with renal manifestations, type I [juvenile type], not stated as uncontrolled
        "25042",  # Diabetes with renal manifestations, type II or unspecified type, uncontrolled
        "25043",  # Diabetes with renal manifestations, type I [juvenile type], uncontrolled
        "25050",  # Diabetes with ophthalmic manifestations, type II or unspecified type, not stated as uncontrolled
        "25051",  # Diabetes with ophthalmic manifestations, type I [juvenile type], not stated as uncontrolled
        "25052",  # Diabetes with ophthalmic manifestations, type II or unspecified type, uncontrolled
        "25053",  # Diabetes with ophthalmic manifestations, type I [juvenile type], uncontrolled
        "25060",  # Diabetes with neurological manifestations, type II or unspecified type, not stated as uncontrolled
        "25061",  # Diabetes with neurological manifestations, type I [juvenile type], not stated as uncontrolled
        "25062",  # Diabetes with neurological manifestations, type II or unspecified type, uncontrolled
        "25063",  # Diabetes with neurological manifestations, type I [juvenile type], uncontrolled
        "25070",  # Diabetes with peripheral circulatory disorders, type II or unspecified type, not stated as uncontrolled
        "25071",  # Diabetes with peripheral circulatory disorders, type I [juvenile type], not stated as uncontrolled
        "25072",  # Diabetes with peripheral circulatory disorders, type II or unspecified type, uncontrolled
        "25073",  # Diabetes with peripheral circulatory disorders, type I [juvenile type], uncontrolled
        "25080",  # Diabetes with other specified manifestations, type II or unspecified type, not stated as uncontrolled
        "25081",  # Diabetes with other specified manifestations, type I [juvenile type], not stated as uncontrolled
        "25082",  # Diabetes with other specified manifestations, type II or unspecified type, uncontrolled
        "25083",  # Diabetes with other specified manifestations, type I [juvenile type], uncontrolled
        "25090",  # Diabetes with unspecified complication, type II or unspecified type, not stated as uncontrolled
        "25091",  # Diabetes with unspecified complication, type I [juvenile type], not stated as uncontrolled
        "25092",  # Diabetes with unspecified complication, type II or unspecified type, uncontrolled
        "25093",  # Diabetes with unspecified complication, type I [juvenile type], uncontrolled
        "3572",  # Polyneuropathy in diabetes
        "36201",  # Background diabetic retinopathy
        "36202",  # Proliferative diabetic retinopathy
        "36203",  # Nonproliferative diabetic retinopathy NOS
        "36204",  # Mild nonproliferative diabetic retinopathy
        "36205",  # Moderate nonproliferative diabetic retinopathy
        "36206",  # Severe nonproliferative diabetic retinopathy
        "36207",  # Diabetic macular edema
        "36641",  # Diabetic cataract
        "64800",  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, unspecified as to episode of care or not applicable
        "64801",  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, delivered, with or without mention of antepartum condition
        "64802",  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, delivered, with mention of postpartum complication
        "64803",  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, antepartum condition or complication
        "64804",  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, postpartum condition or complication
    }
    SNOMEDCT = {
        "9859006",  # Acanthosis nigricans due to type 2 diabetes mellitus (disorder)
        "23045005",  # Insulin dependent diabetes mellitus type IA (disorder)
        "28032008",  # Insulin dependent diabetes mellitus type IB (disorder)
        "44054006",  # Diabetes mellitus type 2 (disorder)
        "46635009",  # Diabetes mellitus type 1 (disorder)
        "81531005",  # Diabetes mellitus type 2 in obese (disorder)
        "190330002",  # Hyperosmolar coma due to type 1 diabetes mellitus (disorder)
        "190331003",  # Hyperosmolar coma due to type 2 diabetes mellitus (disorder)
        "190368000",  # Type I diabetes mellitus with ulcer (disorder)
        "190369008",  # Type I diabetes mellitus with gangrene (disorder)
        "190372001",  # Type I diabetes mellitus maturity onset (disorder)
        "190389009",  # Type II diabetes mellitus with ulcer (disorder)
        "199229001",  # Pre-existing type 1 diabetes mellitus (disorder)
        "199230006",  # Pre-existing type 2 diabetes mellitus (disorder)
        "237599002",  # Insulin treated type 2 diabetes mellitus (disorder)
        "237604008",  # Maturity onset diabetes of the young, type 2 (disorder)
        "237618001",  # Insulin-dependent diabetes mellitus secretory diarrhea syndrome (disorder)
        "313435000",  # Type I diabetes mellitus without complication (disorder)
        "313436004",  # Type II diabetes mellitus without complication (disorder)
        "314771006",  # Type I diabetes mellitus with hypoglycemic coma (disorder)
        "314893005",  # Arthropathy due to type 1 diabetes mellitus (disorder)
        "314902007",  # Peripheral angiopathy due to type 2 diabetes mellitus (disorder)
        "314903002",  # Arthropathy due to type 2 diabetes mellitus (disorder)
        "314904008",  # Type II diabetes mellitus with neuropathic arthropathy (disorder)
        "359642000",  # Diabetes mellitus type 2 in nonobese (disorder)
        "420279001",  # Renal disorder due to type 2 diabetes mellitus (disorder)
        "420436000",  # Mononeuropathy due to type 2 diabetes mellitus (disorder)
        "420486006",  # Exudative maculopathy due to type 1 diabetes mellitus (disorder)
        "420789003",  # Retinopathy due to type 1 diabetes mellitus (disorder)
        "420918009",  # Mononeuropathy due to type 1 diabetes mellitus (disorder)
        "421075007",  # Ketoacidotic coma due to type 1 diabetes mellitus (disorder)
        "421326000",  # Disorder of nervous system due to type 2 diabetes mellitus (disorder)
        "421365002",  # Peripheral circulatory disorder due to type 1 diabetes mellitus (disorder)
        "421437000",  # Hypoglycemic coma due to type 1 diabetes mellitus (disorder)
        "421468001",  # Disorder of nervous system due to type 1 diabetes mellitus (disorder)
        "421779007",  # Exudative maculopathy due to type 2 diabetes mellitus (disorder)
        "421847006",  # Ketoacidotic coma due to type 2 diabetes mellitus (disorder)
        "421893009",  # Renal disorder due to type 1 diabetes mellitus (disorder)
        "422034002",  # Retinopathy due to type 2 diabetes mellitus (disorder)
        "422099009",  # Disorder of eye due to type 2 diabetes mellitus (disorder)
        "422166005",  # Peripheral circulatory disorder due to type 2 diabetes mellitus (disorder)
        "427027005",  # Lumbosacral radiculoplexus neuropathy due to type 2 diabetes mellitus (disorder)
        "427571000",  # Lumbosacral radiculoplexus neuropathy due to type 1 diabetes mellitus (disorder)
        "428007007",  # Erectile dysfunction due to type 2 diabetes mellitus (disorder)
        "609562003",  # Maturity onset diabetes of the young, type 1 (disorder)
        "609564002",  # Pre-existing type 1 diabetes mellitus in pregnancy (disorder)
        "609566000",  # Pregnancy and type 1 diabetes mellitus (disorder)
        "609567009",  # Pre-existing type 2 diabetes mellitus in pregnancy (disorder)
        "712882000",  # Autonomic neuropathy due to type 1 diabetes mellitus (disorder)
        "712883005",  # Autonomic neuropathy due to type 2 diabetes mellitus (disorder)
        "713702000",  # Gastroparesis due to type 1 diabetes mellitus (disorder)
        "713703005",  # Gastroparesis due to type 2 diabetes mellitus (disorder)
        "713705003",  # Polyneuropathy due to type 1 diabetes mellitus (disorder)
        "713706002",  # Polyneuropathy due to type 2 diabetes mellitus (disorder)
        "719216001",  # Hypoglycemic coma due to type 2 diabetes mellitus (disorder)
        "739681000",  # Disorder of eye due to type 1 diabetes mellitus (disorder)
        "770098001",  # Cranial nerve palsy due to type 1 diabetes mellitus (disorder)
        "691000119103",  # Erectile dysfunction due to type 1 diabetes mellitus (disorder)
        "1481000119100",  # Diabetes mellitus type 2 without retinopathy (disorder)
        "1501000119109",  # Proliferative retinopathy due to type 2 diabetes mellitus (disorder)
        "1511000119107",  # Peripheral neuropathy due to type 2 diabetes mellitus (disorder)
        "1551000119108",  # Nonproliferative retinopathy due to type 2 diabetes mellitus (disorder)
        "28331000119107",  # Retinal edema due to type 2 diabetes mellitus (disorder)
        "31211000119101",  # Peripheral angiopathy due to type 1 diabetes mellitus (disorder)
        "31321000119102",  # Diabetes mellitus type 1 without retinopathy (disorder)
        "41911000119107",  # Glaucoma due to type 2 diabetes mellitus (disorder)
        "60951000119105",  # Blindness due to type 2 diabetes mellitus (disorder)
        "60961000119107",  # Nonproliferative diabetic retinopathy due to type 1 diabetes mellitus (disorder)
        "60971000119101",  # Proliferative retinopathy due to type 1 diabetes mellitus (disorder)
        "60991000119100",  # Blindness due to type 1 diabetes mellitus (disorder)
        "71441000119104",  # Nephrotic syndrome due to type 2 diabetes mellitus (disorder)
        "71721000119101",  # Nephrotic syndrome due to type 1 diabetes mellitus (disorder)
        "71791000119104",  # Peripheral neuropathy due to type 1 diabetes mellitus (disorder)
        "82541000119100",  # Traction detachment of retina due to type 2 diabetes mellitus (disorder)
        "82551000119103",  # Rubeosis iridis due to type 2 diabetes mellitus (disorder)
        "82571000119107",  # Traction detachment of retina due to type 1 diabetes mellitus (disorder)
        "82581000119105",  # Rubeosis iridis due to type 1 diabetes mellitus (disorder)
        "87921000119104",  # Cranial nerve palsy due to type 2 diabetes mellitus (disorder)
        "97331000119101",  # Macular edema and retinopathy due to type 2 diabetes mellitus (disorder)
        "102781000119107",  # Sensory neuropathy due to type 1 diabetes mellitus (disorder)
        "104941000119109",  # Ischemia of retina due to type 1 diabetes mellitus (disorder)
        "104961000119108",  # Ischemia of retina due to type 2 diabetes mellitus (disorder)
        "109171000119104",  # Retinal edema due to type 1 diabetes mellitus (disorder)
        "110181000119105",  # Peripheral sensory neuropathy due to type 2 diabetes mellitus (disorder)
        "138881000119106",  # Mild nonproliferative retinopathy due to type 1 diabetes mellitus (disorder)
        "138891000119109",  # Moderate nonproliferative retinopathy due to type 1 diabetes mellitus (disorder)
        "138901000119108",  # Severe nonproliferative retinopathy due to diabetes mellitus type 1 (disorder)
        "138911000119106",  # Mild nonproliferative retinopathy due to type 2 diabetes mellitus (disorder)
        "138921000119104",  # Moderate nonproliferative retinopathy due to type 2 diabetes mellitus (disorder)
        "138941000119105",  # Severe nonproliferative retinopathy due to diabetes mellitus type 2 (disorder)
        "368101000119109",  # Periodontal disease co-occurrent and due to diabetes mellitus type 2 (disorder)
        "368521000119107",  # Disorder of nerve co-occurrent and due to type 1 diabetes mellitus (disorder)
        "368581000119106",  # Neuropathy due to type 2 diabetes mellitus (disorder)
    }


class FrailtyDiagnosis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  potential indicators of frailty.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent that a diagnosis of potential frailty when coupled with older age: dependence on durable medical equipment for long-term use, limitation of activities of daily living due to disability, need for assistance with care, pressure ulcers, and falls.

    **Exclusion Criteria:** Exclude concepts that indicate potential frailty when associated with pregnancy, fetal, juvenile, childhood or neonatal conditions; codes that indicate temporary need of durable medical equipment; codes that indicate accidental falls caused by another person, being pushed/shoved/collided into, falls on ice due to ice skates, falls due to impact against a baby buggy/shopping cart/wheelbarrow.

    ** Used in:** CMS134v10, CMS165v10, CMS131v10, CMS122v10, CMS125v10, CMS130v10
    """

    VALUE_SET_NAME = "Frailty Diagnosis"
    OID = "2.16.840.1.113883.3.464.1003.113.12.1074"
    DEFINITION_VERSION = "20190315"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "L89001",  # Pressure ulcer of unspecified elbow, stage 1
        "L89002",  # Pressure ulcer of unspecified elbow, stage 2
        "L89003",  # Pressure ulcer of unspecified elbow, stage 3
        "L89004",  # Pressure ulcer of unspecified elbow, stage 4
        "L89006",  # Pressure-induced deep tissue damage of unspecified elbow
        "L89009",  # Pressure ulcer of unspecified elbow, unspecified stage
        "L89010",  # Pressure ulcer of right elbow, unstageable
        "L89011",  # Pressure ulcer of right elbow, stage 1
        "L89012",  # Pressure ulcer of right elbow, stage 2
        "L89013",  # Pressure ulcer of right elbow, stage 3
        "L89014",  # Pressure ulcer of right elbow, stage 4
        "L89016",  # Pressure-induced deep tissue damage of right elbow
        "L89019",  # Pressure ulcer of right elbow, unspecified stage
        "L89020",  # Pressure ulcer of left elbow, unstageable
        "L89021",  # Pressure ulcer of left elbow, stage 1
        "L89022",  # Pressure ulcer of left elbow, stage 2
        "L89023",  # Pressure ulcer of left elbow, stage 3
        "L89024",  # Pressure ulcer of left elbow, stage 4
        "L89026",  # Pressure-induced deep tissue damage of left elbow
        "L89029",  # Pressure ulcer of left elbow, unspecified stage
        "L89100",  # Pressure ulcer of unspecified part of back, unstageable
        "L89101",  # Pressure ulcer of unspecified part of back, stage 1
        "L89102",  # Pressure ulcer of unspecified part of back, stage 2
        "L89103",  # Pressure ulcer of unspecified part of back, stage 3
        "L89104",  # Pressure ulcer of unspecified part of back, stage 4
        "L89106",  # Pressure-induced deep tissue damage of unspecified part of back
        "L89109",  # Pressure ulcer of unspecified part of back, unspecified stage
        "L89110",  # Pressure ulcer of right upper back, unstageable
        "L89111",  # Pressure ulcer of right upper back, stage 1
        "L89112",  # Pressure ulcer of right upper back, stage 2
        "L89113",  # Pressure ulcer of right upper back, stage 3
        "L89114",  # Pressure ulcer of right upper back, stage 4
        "L89116",  # Pressure-induced deep tissue damage of right upper back
        "L89119",  # Pressure ulcer of right upper back, unspecified stage
        "L89120",  # Pressure ulcer of left upper back, unstageable
        "L89121",  # Pressure ulcer of left upper back, stage 1
        "L89122",  # Pressure ulcer of left upper back, stage 2
        "L89123",  # Pressure ulcer of left upper back, stage 3
        "L89124",  # Pressure ulcer of left upper back, stage 4
        "L89126",  # Pressure-induced deep tissue damage of left upper back
        "L89129",  # Pressure ulcer of left upper back, unspecified stage
        "L89130",  # Pressure ulcer of right lower back, unstageable
        "L89131",  # Pressure ulcer of right lower back, stage 1
        "L89132",  # Pressure ulcer of right lower back, stage 2
        "L89133",  # Pressure ulcer of right lower back, stage 3
        "L89134",  # Pressure ulcer of right lower back, stage 4
        "L89136",  # Pressure-induced deep tissue damage of right lower back
        "L89139",  # Pressure ulcer of right lower back, unspecified stage
        "L89140",  # Pressure ulcer of left lower back, unstageable
        "L89141",  # Pressure ulcer of left lower back, stage 1
        "L89142",  # Pressure ulcer of left lower back, stage 2
        "L89143",  # Pressure ulcer of left lower back, stage 3
        "L89144",  # Pressure ulcer of left lower back, stage 4
        "L89146",  # Pressure-induced deep tissue damage of left lower back
        "L89149",  # Pressure ulcer of left lower back, unspecified stage
        "L89150",  # Pressure ulcer of sacral region, unstageable
        "L89151",  # Pressure ulcer of sacral region, stage 1
        "L89152",  # Pressure ulcer of sacral region, stage 2
        "L89153",  # Pressure ulcer of sacral region, stage 3
        "L89154",  # Pressure ulcer of sacral region, stage 4
        "L89156",  # Pressure-induced deep tissue damage of sacral region
        "L89159",  # Pressure ulcer of sacral region, unspecified stage
        "L89200",  # Pressure ulcer of unspecified hip, unstageable
        "L89201",  # Pressure ulcer of unspecified hip, stage 1
        "L89202",  # Pressure ulcer of unspecified hip, stage 2
        "L89203",  # Pressure ulcer of unspecified hip, stage 3
        "L89204",  # Pressure ulcer of unspecified hip, stage 4
        "L89206",  # Pressure-induced deep tissue damage of unspecified hip
        "L89209",  # Pressure ulcer of unspecified hip, unspecified stage
        "L89210",  # Pressure ulcer of right hip, unstageable
        "L89211",  # Pressure ulcer of right hip, stage 1
        "L89212",  # Pressure ulcer of right hip, stage 2
        "L89213",  # Pressure ulcer of right hip, stage 3
        "L89214",  # Pressure ulcer of right hip, stage 4
        "L89216",  # Pressure-induced deep tissue damage of right hip
        "L89219",  # Pressure ulcer of right hip, unspecified stage
        "L89220",  # Pressure ulcer of left hip, unstageable
        "L89221",  # Pressure ulcer of left hip, stage 1
        "L89222",  # Pressure ulcer of left hip, stage 2
        "L89223",  # Pressure ulcer of left hip, stage 3
        "L89224",  # Pressure ulcer of left hip, stage 4
        "L89226",  # Pressure-induced deep tissue damage of left hip
        "L89229",  # Pressure ulcer of left hip, unspecified stage
        "L89300",  # Pressure ulcer of unspecified buttock, unstageable
        "L89301",  # Pressure ulcer of unspecified buttock, stage 1
        "L89302",  # Pressure ulcer of unspecified buttock, stage 2
        "L89303",  # Pressure ulcer of unspecified buttock, stage 3
        "L89304",  # Pressure ulcer of unspecified buttock, stage 4
        "L89306",  # Pressure-induced deep tissue damage of unspecified buttock
        "L89309",  # Pressure ulcer of unspecified buttock, unspecified stage
        "L89310",  # Pressure ulcer of right buttock, unstageable
        "L89311",  # Pressure ulcer of right buttock, stage 1
        "L89312",  # Pressure ulcer of right buttock, stage 2
        "L89313",  # Pressure ulcer of right buttock, stage 3
        "L89314",  # Pressure ulcer of right buttock, stage 4
        "L89316",  # Pressure-induced deep tissue damage of right buttock
        "L89319",  # Pressure ulcer of right buttock, unspecified stage
        "L89320",  # Pressure ulcer of left buttock, unstageable
        "L89321",  # Pressure ulcer of left buttock, stage 1
        "L89322",  # Pressure ulcer of left buttock, stage 2
        "L89323",  # Pressure ulcer of left buttock, stage 3
        "L89324",  # Pressure ulcer of left buttock, stage 4
        "L89326",  # Pressure-induced deep tissue damage of left buttock
        "L89329",  # Pressure ulcer of left buttock, unspecified stage
        "L8940",  # Pressure ulcer of contiguous site of back, buttock and hip, unspecified stage
        "L8941",  # Pressure ulcer of contiguous site of back, buttock and hip, stage 1
        "L8942",  # Pressure ulcer of contiguous site of back, buttock and hip, stage 2
        "L8943",  # Pressure ulcer of contiguous site of back, buttock and hip, stage 3
        "L8944",  # Pressure ulcer of contiguous site of back, buttock and hip, stage 4
        "L8945",  # Pressure ulcer of contiguous site of back, buttock and hip, unstageable
        "L8946",  # Pressure-induced deep tissue damage of contiguous site of back, buttock and hip
        "L89500",  # Pressure ulcer of unspecified ankle, unstageable
        "L89501",  # Pressure ulcer of unspecified ankle, stage 1
        "L89502",  # Pressure ulcer of unspecified ankle, stage 2
        "L89503",  # Pressure ulcer of unspecified ankle, stage 3
        "L89504",  # Pressure ulcer of unspecified ankle, stage 4
        "L89506",  # Pressure-induced deep tissue damage of unspecified ankle
        "L89509",  # Pressure ulcer of unspecified ankle, unspecified stage
        "L89510",  # Pressure ulcer of right ankle, unstageable
        "L89511",  # Pressure ulcer of right ankle, stage 1
        "L89512",  # Pressure ulcer of right ankle, stage 2
        "L89513",  # Pressure ulcer of right ankle, stage 3
        "L89514",  # Pressure ulcer of right ankle, stage 4
        "L89516",  # Pressure-induced deep tissue damage of right ankle
        "L89519",  # Pressure ulcer of right ankle, unspecified stage
        "L89520",  # Pressure ulcer of left ankle, unstageable
        "L89521",  # Pressure ulcer of left ankle, stage 1
        "L89522",  # Pressure ulcer of left ankle, stage 2
        "L89523",  # Pressure ulcer of left ankle, stage 3
        "L89524",  # Pressure ulcer of left ankle, stage 4
        "L89526",  # Pressure-induced deep tissue damage of left ankle
        "L89529",  # Pressure ulcer of left ankle, unspecified stage
        "L89600",  # Pressure ulcer of unspecified heel, unstageable
        "L89601",  # Pressure ulcer of unspecified heel, stage 1
        "L89602",  # Pressure ulcer of unspecified heel, stage 2
        "L89603",  # Pressure ulcer of unspecified heel, stage 3
        "L89604",  # Pressure ulcer of unspecified heel, stage 4
        "L89606",  # Pressure-induced deep tissue damage of unspecified heel
        "L89609",  # Pressure ulcer of unspecified heel, unspecified stage
        "L89610",  # Pressure ulcer of right heel, unstageable
        "L89611",  # Pressure ulcer of right heel, stage 1
        "L89612",  # Pressure ulcer of right heel, stage 2
        "L89613",  # Pressure ulcer of right heel, stage 3
        "L89614",  # Pressure ulcer of right heel, stage 4
        "L89616",  # Pressure-induced deep tissue damage of right heel
        "L89619",  # Pressure ulcer of right heel, unspecified stage
        "L89620",  # Pressure ulcer of left heel, unstageable
        "L89621",  # Pressure ulcer of left heel, stage 1
        "L89622",  # Pressure ulcer of left heel, stage 2
        "L89623",  # Pressure ulcer of left heel, stage 3
        "L89624",  # Pressure ulcer of left heel, stage 4
        "L89626",  # Pressure-induced deep tissue damage of left heel
        "L89629",  # Pressure ulcer of left heel, unspecified stage
        "L89810",  # Pressure ulcer of head, unstageable
        "L89811",  # Pressure ulcer of head, stage 1
        "L89812",  # Pressure ulcer of head, stage 2
        "L89813",  # Pressure ulcer of head, stage 3
        "L89814",  # Pressure ulcer of head, stage 4
        "L89816",  # Pressure-induced deep tissue damage of head
        "L89819",  # Pressure ulcer of head, unspecified stage
        "L89890",  # Pressure ulcer of other site, unstageable
        "L89891",  # Pressure ulcer of other site, stage 1
        "L89892",  # Pressure ulcer of other site, stage 2
        "L89893",  # Pressure ulcer of other site, stage 3
        "L89894",  # Pressure ulcer of other site, stage 4
        "L89896",  # Pressure-induced deep tissue damage of other site
        "L89899",  # Pressure ulcer of other site, unspecified stage
        "L8990",  # Pressure ulcer of unspecified site, unspecified stage
        "L8991",  # Pressure ulcer of unspecified site, stage 1
        "L8992",  # Pressure ulcer of unspecified site, stage 2
        "L8993",  # Pressure ulcer of unspecified site, stage 3
        "L8994",  # Pressure ulcer of unspecified site, stage 4
        "L8995",  # Pressure ulcer of unspecified site, unstageable
        "L8996",  # Pressure-induced deep tissue damage of unspecified site
        "M6250",  # Muscle wasting and atrophy, not elsewhere classified, unspecified site
        "M6281",  # Muscle weakness (generalized)
        "M6284",  # Sarcopenia
        "W010XXA",  # Fall on same level from slipping, tripping and stumbling without subsequent striking against object, initial encounter
        "W010XXD",  # Fall on same level from slipping, tripping and stumbling without subsequent striking against object, subsequent encounter
        "W010XXS",  # Fall on same level from slipping, tripping and stumbling without subsequent striking against object, sequela
        "W0110XA",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified object, initial encounter
        "W0110XD",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified object, subsequent encounter
        "W0110XS",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified object, sequela
        "W01110A",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against sharp glass, initial encounter
        "W01110D",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against sharp glass, subsequent encounter
        "W01110S",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against sharp glass, sequela
        "W01111A",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against power tool or machine, initial encounter
        "W01111D",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against power tool or machine, subsequent encounter
        "W01111S",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against power tool or machine, sequela
        "W01118A",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other sharp object, initial encounter
        "W01118D",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other sharp object, subsequent encounter
        "W01118S",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other sharp object, sequela
        "W01119A",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified sharp object, initial encounter
        "W01119D",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified sharp object, subsequent encounter
        "W01119S",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified sharp object, sequela
        "W01190A",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against furniture, initial encounter
        "W01190D",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against furniture, subsequent encounter
        "W01190S",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against furniture, sequela
        "W01198A",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other object, initial encounter
        "W01198D",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other object, subsequent encounter
        "W01198S",  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other object, sequela
        "W06XXXA",  # Fall from bed, initial encounter
        "W06XXXD",  # Fall from bed, subsequent encounter
        "W06XXXS",  # Fall from bed, sequela
        "W07XXXA",  # Fall from chair, initial encounter
        "W07XXXD",  # Fall from chair, subsequent encounter
        "W07XXXS",  # Fall from chair, sequela
        "W08XXXA",  # Fall from other furniture, initial encounter
        "W08XXXD",  # Fall from other furniture, subsequent encounter
        "W08XXXS",  # Fall from other furniture, sequela
        "W100XXA",  # Fall (on)(from) escalator, initial encounter
        "W100XXD",  # Fall (on)(from) escalator, subsequent encounter
        "W100XXS",  # Fall (on)(from) escalator, sequela
        "W101XXA",  # Fall (on)(from) sidewalk curb, initial encounter
        "W101XXD",  # Fall (on)(from) sidewalk curb, subsequent encounter
        "W101XXS",  # Fall (on)(from) sidewalk curb, sequela
        "W102XXA",  # Fall (on)(from) incline, initial encounter
        "W102XXD",  # Fall (on)(from) incline, subsequent encounter
        "W102XXS",  # Fall (on)(from) incline, sequela
        "W108XXA",  # Fall (on) (from) other stairs and steps, initial encounter
        "W108XXD",  # Fall (on) (from) other stairs and steps, subsequent encounter
        "W108XXS",  # Fall (on) (from) other stairs and steps, sequela
        "W109XXA",  # Fall (on) (from) unspecified stairs and steps, initial encounter
        "W109XXD",  # Fall (on) (from) unspecified stairs and steps, subsequent encounter
        "W109XXS",  # Fall (on) (from) unspecified stairs and steps, sequela
        "W1800XA",  # Striking against unspecified object with subsequent fall, initial encounter
        "W1800XD",  # Striking against unspecified object with subsequent fall, subsequent encounter
        "W1800XS",  # Striking against unspecified object with subsequent fall, sequela
        "W1802XA",  # Striking against glass with subsequent fall, initial encounter
        "W1802XD",  # Striking against glass with subsequent fall, subsequent encounter
        "W1802XS",  # Striking against glass with subsequent fall, sequela
        "W1809XA",  # Striking against other object with subsequent fall, initial encounter
        "W1809XD",  # Striking against other object with subsequent fall, subsequent encounter
        "W1809XS",  # Striking against other object with subsequent fall, sequela
        "W1811XA",  # Fall from or off toilet without subsequent striking against object, initial encounter
        "W1811XD",  # Fall from or off toilet without subsequent striking against object, subsequent encounter
        "W1811XS",  # Fall from or off toilet without subsequent striking against object, sequela
        "W1812XA",  # Fall from or off toilet with subsequent striking against object, initial encounter
        "W1812XD",  # Fall from or off toilet with subsequent striking against object, subsequent encounter
        "W1812XS",  # Fall from or off toilet with subsequent striking against object, sequela
        "W182XXA",  # Fall in (into) shower or empty bathtub, initial encounter
        "W182XXD",  # Fall in (into) shower or empty bathtub, subsequent encounter
        "W182XXS",  # Fall in (into) shower or empty bathtub, sequela
        "W1830XA",  # Fall on same level, unspecified, initial encounter
        "W1830XD",  # Fall on same level, unspecified, subsequent encounter
        "W1830XS",  # Fall on same level, unspecified, sequela
        "W1831XA",  # Fall on same level due to stepping on an object, initial encounter
        "W1831XD",  # Fall on same level due to stepping on an object, subsequent encounter
        "W1831XS",  # Fall on same level due to stepping on an object, sequela
        "W1839XA",  # Other fall on same level, initial encounter
        "W1839XD",  # Other fall on same level, subsequent encounter
        "W1839XS",  # Other fall on same level, sequela
        "W19XXXA",  # Unspecified fall, initial encounter
        "W19XXXD",  # Unspecified fall, subsequent encounter
        "W19XXXS",  # Unspecified fall, sequela
        "Y92199",  # Unspecified place in other specified residential institution as the place of occurrence of the external cause
        "Z593",  # Problems related to living in residential institution
        "Z736",  # Limitation of activities due to disability
        "Z7401",  # Bed confinement status
        "Z7409",  # Other reduced mobility
        "Z741",  # Need for assistance with personal care
        "Z742",  # Need for assistance at home and no other household member able to render care
        "Z743",  # Need for continuous supervision
        "Z748",  # Other problems related to care provider dependency
        "Z749",  # Problem related to care provider dependency, unspecified
        "Z9181",  # History of falling
        "Z9911",  # Dependence on respirator [ventilator] status
        "Z993",  # Dependence on wheelchair
        "Z9981",  # Dependence on supplemental oxygen
        "Z9989",  # Dependence on other enabling machines and devices
    }
    SNOMEDCT = {
        "16728003",  # Inherited spastic paresis (disorder)
        "17886000",  # Fall from wheelchair (event)
        "20902002",  # Fall from bed (event)
        "33036003",  # Fall on same level (event)
        "40104005",  # Fall in home (event)
        "44188002",  # Fall in shower (event)
        "52702003",  # Chronic fatigue syndrome (disorder)
        "56307009",  # Fall from table (event)
        "67223001",  # Fall on escalator (event)
        "74541001",  # Fall from bench (event)
        "83468000",  # Fall from chair (event)
        "90619006",  # Fall in bathtub (event)
        "129588001",  # Adult failure to thrive syndrome (disorder)
        "162845004",  # On examination - failure to thrive (disorder)
        "214436006",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting (event)
        "214437002",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, driver of motor vehicle injured (event)
        "214438007",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, passenger of motor vehicle injured (event)
        "214439004",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, motor cyclist injured (event)
        "214441003",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, occupant of tram injured (event)
        "214442005",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, rider of animal or occupant of animal-drawn vehicle injured (event)
        "214443000",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, pedal cyclist injured (event)
        "214444006",  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, pedestrian injured (event)
        "217082002",  # Accidental fall (event)
        "217083007",  # Fall on or from stairs or steps (event)
        "217084001",  # Fall on or from escalator (event)
        "217086004",  # Fall from escalator (event)
        "217088003",  # Fall on or from stairs (event)
        "217090002",  # Fall from stairs (event)
        "217092005",  # Fall on or from steps (event)
        "217093000",  # Fall on steps (event)
        "217094006",  # Fall from steps (event)
        "217142006",  # Fall from chair or bed (event)
        "217154006",  # Fall on same level from slipping, tripping or stumbling (event)
        "217155007",  # Fall on same level from slipping (event)
        "217156008",  # Fall on same level from tripping (event)
        "217157004",  # Fall on same level from stumbling (event)
        "217158009",  # Fall on moving sidewalk (event)
        "217173005",  # Fall from bump against object (event)
        "225558004",  # Pressure ulcer of back (disorder)
        "225562005",  # Pressure ulcer of sacral region (disorder)
        "225563000",  # Pressure ulcer of buttock (disorder)
        "242109009",  # Fall down stairs of motor bus while boarding or alighting (event)
        "242389003",  # Fall due to wet surface (event)
        "242390007",  # Fall due to polished surface (event)
        "242391006",  # Fall due to discarded object (event)
        "242392004",  # Fall in bath or shower (event)
        "242395002",  # Fall due to trip on loose carpet (event)
        "242396001",  # Fall due to uneven surface indoors (event)
        "242413007",  # Fall from furniture (event)
        "242414001",  # Fall from stool (event)
        "242419006",  # Fall from toilet seat (event)
        "269699007",  # Fall on same level from impact against object (event)
        "274918000",  # Fall on same level due to nature of surface (event)
        "414188008",  # Fall down stairs (event)
        "414189000",  # Fall down steps (event)
        "414190009",  # Fall on stairs (event)
        "427849003",  # Fall on hard surface (event)
        "428484005",  # Organic failure to thrive (disorder)
        "429621003",  # Fall on concrete (event)
        "699214007",  # Pressure ulcer of hip (disorder)
        "699215008",  # Pressure ulcer of lower back (disorder)
        "699216009",  # Pressure ulcer of coccygeal region (disorder)
        "699218005",  # Pressure ulcer of upper back (disorder)
        "715504003",  # Spastic paraparesis co-occurrent with deafness (disorder)
        "763829004",  # Oculopharyngodistal myopathy (disorder)
        "823018004",  # Muscle paresis due to and following neuromuscular blockade (disorder)
        "92341000119107",  # Weakness of extremities as sequela of stroke (disorder)
        "138371000119104",  # Paresis as late effect of poliomyelitis (disorder)
        "459821000124104",  # McArdle sign (finding)
        "8960001000004106",  # Pressure ulcer of skin of right ischial tuberosity region (disorder)
        "10637031000119106",  # Pressure ulcer of buttock stage 1 (disorder)
        "10637071000119109",  # Pressure ulcer of buttock stage 2 (disorder)
        "10637111000119102",  # Pressure ulcer of buttock stage 3 (disorder)
        "10637151000119101",  # Pressure ulcer of buttock stage 4 (disorder)
    }


class CongenitalOrAcquiredAbsenceOfCervix(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent diagnoses related to absence of a cervix.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an absence of cervix with or without remaining uterus, congenital or acquired absence of cervix, radical or total hysterectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a partial hysterectomy, absence of uterus with remaining cervical stump.

    ** Used in:** CMS124v10
    """

    VALUE_SET_NAME = "Congenital or Acquired Absence of Cervix"
    OID = "2.16.840.1.113883.3.464.1003.111.12.1016"
    DEFINITION_VERSION = "20210224"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Q515",  # Agenesis and aplasia of cervix
        "Z90710",  # Acquired absence of both cervix and uterus
        "Z90712",  # Acquired absence of cervix with remaining uterus
    }
    ICD9CM = {
        "6185",  # Prolapse of vaginal vault after hysterectomy
        "75243",  # Cervical agenesis
        "V8801",  # Acquired absence of both cervix and uterus
        "V8803",  # Acquired absence of cervix with remaining uterus
    }
    SNOMEDCT = {
        "37687000",  # Congenital absence of cervix (disorder)
        "248911005",  # Uterine cervix absent (finding)
        "428078001",  # History of total hysterectomy (situation)
        "429290001",  # History of radical hysterectomy (situation)
        "429763009",  # History of total hysterectomy with bilateral salpingo-oophorectomy (situation)
        "723171001",  # Acquired absence of cervix and uterus (disorder)
        "10738891000119107",  # History of total hysterectomy without abnormal cervical Papanicolaou smear (situation)
    }


class HistoryOfBilateralMastectomy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a history of bilateral mastectomy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a history of a bilateral mastectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a history of a unilateral or unspecified mastectomy.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = "History of bilateral mastectomy"
    OID = "2.16.840.1.113883.3.464.1003.198.12.1068"
    DEFINITION_VERSION = "20171219"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Z9013",  # Acquired absence of bilateral breasts and nipples
    }
    SNOMEDCT = {
        "428529004",  # History of bilateral mastectomy (situation)
        "136071000119101",  # History of bilateral prophylactic mastectomy (situation)
    }


class StatusPostLeftMastectomy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a history of a left mastectomy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a left mastectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a right, bilateral, or unspecified mastectomy.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = "Status Post Left Mastectomy"
    OID = "2.16.840.1.113883.3.464.1003.198.12.1069"
    DEFINITION_VERSION = "20171216"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Z9012",  # Acquired absence of left breast and nipple
    }
    SNOMEDCT = {
        "429009003",  # History of left mastectomy (situation)
        "137671000119105",  # History of prophylactic mastectomy of left breast (situation)
    }


class StatusPostRightMastectomy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a history of a right mastectomy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a right mastectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a left, bilateral, or unspecified mastectomy.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = "Status Post Right Mastectomy"
    OID = "2.16.840.1.113883.3.464.1003.198.12.1070"
    DEFINITION_VERSION = "20171216"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Z9011",  # Acquired absence of right breast and nipple
    }
    SNOMEDCT = {
        "429242008",  # History of right mastectomy (situation)
        "137681000119108",  # History of prophylactic mastectomy of right breast (situation)
    }


class UnilateralMastectomy_UnspecifiedLaterality(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a unilateral mastectomy where the side was not specified.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a unilateral mastectomy where the side was unspecified.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a bilateral mastectomy or a unilateral mastectomy where the side is specified.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = "Unilateral Mastectomy, Unspecified Laterality"
    OID = "2.16.840.1.113883.3.464.1003.198.12.1071"
    DEFINITION_VERSION = "20171216"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Z9010",  # Acquired absence of unspecified breast and nipple
    }
    ICD9CM = {
        "V4571",  # Acquired absence of breast and nipple
    }
    SNOMEDCT = {
        "8115005",  # Unilateral radical mastectomy (situation)
        "22964006",  # Unilateral simple mastectomy (situation)
        "72269009",  # Unilateral modified radical mastectomy (situation)
        "73359007",  # Unilateral mastectomy (situation)
        "88764002",  # Unilateral extended simple mastectomy (situation)
        "248802009",  # Absence of breast (finding)
    }


class MajorDepression(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  major depression.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of major depression including mild, moderate, and severe depression, maternal depression, and depression with and without psychotic features.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of depression in remission.

    ** Used in:** CMS128v10
    """

    VALUE_SET_NAME = "Major Depression"
    OID = "2.16.840.1.113883.3.464.1003.105.12.1007"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "F320",  # Major depressive disorder, single episode, mild
        "F321",  # Major depressive disorder, single episode, moderate
        "F322",  # Major depressive disorder, single episode, severe without psychotic features
        "F323",  # Major depressive disorder, single episode, severe with psychotic features
        "F324",  # Major depressive disorder, single episode, in partial remission
        "F329",  # Major depressive disorder, single episode, unspecified
        "F330",  # Major depressive disorder, recurrent, mild
        "F331",  # Major depressive disorder, recurrent, moderate
        "F332",  # Major depressive disorder, recurrent severe without psychotic features
        "F333",  # Major depressive disorder, recurrent, severe with psychotic symptoms
        "F3341",  # Major depressive disorder, recurrent, in partial remission
        "F339",  # Major depressive disorder, recurrent, unspecified
    }
    SNOMEDCT = {
        "832007",  # Moderate major depression (disorder)
        "2618002",  # Chronic recurrent major depressive disorder (disorder)
        "14183003",  # Chronic major depressive disorder, single episode (disorder)
        "15193003",  # Severe recurrent major depression with psychotic features, mood-incongruent (disorder)
        "15639000",  # Moderate major depression, single episode (disorder)
        "18818009",  # Moderate recurrent major depression (disorder)
        "20250007",  # Severe major depression, single episode, with psychotic features, mood-incongruent (disorder)
        "25922000",  # Major depressive disorder, single episode with postpartum onset (disorder)
        "28475009",  # Severe recurrent major depression with psychotic features (disorder)
        "33078009",  # Severe recurrent major depression with psychotic features, mood-congruent (disorder)
        "33736005",  # Severe major depression with psychotic features, mood-congruent (disorder)
        "36474008",  # Severe recurrent major depression without psychotic features (disorder)
        "36923009",  # Major depression, single episode (disorder)
        "38694004",  # Recurrent major depressive disorder with atypical features (disorder)
        "39809009",  # Recurrent major depressive disorder with catatonic features (disorder)
        "40379007",  # Mild recurrent major depression (disorder)
        "42925002",  # Major depressive disorder, single episode with atypical features (disorder)
        "60099002",  # Severe major depression with psychotic features, mood-incongruent (disorder)
        "63778009",  # Major depressive disorder, single episode with melancholic features (disorder)
        "66344007",  # Recurrent major depression (disorder)
        "69392006",  # Major depressive disorder, single episode with catatonic features (disorder)
        "71336009",  # Recurrent major depressive disorder with postpartum onset (disorder)
        "73867007",  # Severe major depression with psychotic features (disorder)
        "75084000",  # Severe major depression without psychotic features (disorder)
        "76441001",  # Severe major depression, single episode, without psychotic features (disorder)
        "77911002",  # Severe major depression, single episode, with psychotic features, mood-congruent (disorder)
        "79298009",  # Mild major depression, single episode (disorder)
        "87512008",  # Mild major depression (disorder)
        "191604000",  # Single major depressive episode, severe, with psychosis (disorder)
        "191610000",  # Recurrent major depressive episodes, mild (disorder)
        "191611001",  # Recurrent major depressive episodes, moderate (disorder)
        "191613003",  # Recurrent major depressive episodes, severe, with psychosis (disorder)
        "268621008",  # Recurrent major depressive episodes (disorder)
        "319768000",  # Recurrent major depressive disorder with melancholic features (disorder)
        "320751009",  # Major depression, melancholic type (disorder)
        "370143000",  # Major depressive disorder (disorder)
        "430852001",  # Severe major depression, single episode, with psychotic features (disorder)
        "450714000",  # Severe major depression (disorder)
        "719592004",  # Moderately severe major depression (disorder)
        "720451004",  # Minimal recurrent major depression (disorder)
        "720452006",  # Moderately severe recurrent major depression (disorder)
        "720453001",  # Moderately severe major depression single episode (disorder)
        "720454007",  # Minimal major depression single episode (disorder)
        "720455008",  # Minimal major depression (disorder)
        "726772006",  # Major depression with psychotic features (disorder)
        "251000119105",  # Severe major depression, single episode (disorder)
        "281000119103",  # Severe recurrent major depression (disorder)
        "10811121000119102",  # Major depressive disorder in mother complicating childbirth (disorder)
        "10811161000119107",  # Major depressive disorder in mother complicating pregnancy (disorder)
        "16264621000119109",  # Recurrent mild major depressive disorder co-occurrent with anxiety (disorder)
        "16264901000119109",  # Recurrent moderate major depressive disorder co-occurrent with anxiety (disorder)
        "16265951000119109",  # Mild major depressive disorder co-occurrent with anxiety single episode (disorder)
        "16266831000119100",  # Moderate major depressive disorder co-occurrent with anxiety single episode (disorder)
        "16266991000119108",  # Severe major depressive disorder co-occurrent with anxiety single episode (disorder)
    }


class PainRelatedToProstateCancer(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent types of pain or conditions that could cause pain that could be associated with prostate cancer.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent  types of pain or conditions that could cause pain that could be associated with prostate cancer.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS129v11
    """

    VALUE_SET_NAME = "Pain Related to Prostate Cancer"
    OID = "2.16.840.1.113883.3.526.3.451"
    DEFINITION_VERSION = "20200306"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "G893",  # Neoplasm related pain (acute) (chronic)
        "M4327",  # Fusion of spine, lumbosacral region
        "M4328",  # Fusion of spine, sacral and sacrococcygeal region
        "M532X7",  # Spinal instabilities, lumbosacral region
        "M532X8",  # Spinal instabilities, sacral and sacrococcygeal region
        "M533",  # Sacrococcygeal disorders, not elsewhere classified
        "M5386",  # Other specified dorsopathies, lumbar region
        "M5387",  # Other specified dorsopathies, lumbosacral region
        "M5388",  # Other specified dorsopathies, sacral and sacrococcygeal region
        "M546",  # Pain in thoracic spine
        "M5489",  # Other dorsalgia
        "M549",  # Dorsalgia, unspecified
        "M859",  # Disorder of bone density and structure, unspecified
        "M899",  # Disorder of bone, unspecified
        "M949",  # Disorder of cartilage, unspecified
        "R0782",  # Intercostal pain
        "R0789",  # Other chest pain
        "R079",  # Chest pain, unspecified
        "R100",  # Acute abdomen
        "R109",  # Unspecified abdominal pain
    }
    SNOMEDCT = {
        "3200003",  # Sacrocoxalgia (finding)
        "12584003",  # Bone pain (finding)
        "20793008",  # Scapulalgia (finding)
        "34789001",  # Pain in the coccyx (finding)
        "36729000",  # Prostatic pain (finding)
        "48926009",  # Pain in spine (finding)
        "89638008",  # Xiphodynia (finding)
        "161977000",  # Costal margin chest pain (finding)
        "203508001",  # Bony pelvic pain (finding)
        "203509009",  # Clavicle pain (finding)
        "267981009",  # Pain in thoracic spine (finding)
        "282743009",  # Malignant bone pain (finding)
        "297217002",  # Rib pain (finding)
        "298731003",  # Pain of sternum (finding)
        "298740004",  # Rib tender (finding)
        "301773003",  # Pain in femur (finding)
        "426135001",  # Chronic prostatitis - chronic pelvic pain syndrome (disorder)
    }


class ProstateCancer(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for prostate cancer.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of prostate cancer.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS129v11, CMS645v5
    """

    VALUE_SET_NAME = "Prostate Cancer"
    OID = "2.16.840.1.113883.3.526.3.319"
    DEFINITION_VERSION = "20200306"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "C61",  # Malignant neoplasm of prostate
    }
    SNOMEDCT = {
        "93974005",  # Primary malignant neoplasm of prostate (disorder)
        "94503003",  # Secondary malignant neoplasm of prostate (disorder)
        "254900004",  # Carcinoma of prostate (disorder)
        "278060005",  # Endometrioid carcinoma of prostate (disorder)
        "314969001",  # Local recurrence of malignant tumor of prostate (disorder)
        "369485004",  # Malignant tumor involving prostate by direct extension from bladder (disorder)
        "369486003",  # Malignant tumor involving prostate by separate metastasis from bladder (disorder)
        "396198006",  # Small cell carcinoma of prostate (disorder)
        "399068003",  # Malignant tumor of prostate (disorder)
        "399490008",  # Adenocarcinoma of prostate (disorder)
        "399590005",  # Squamous cell carcinoma of prostate (disorder)
        "427492003",  # Hormone refractory prostate cancer (disorder)
    }


class MalignantNeoplasmOfColon(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  malignant neoplasms of the colon.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of malignant neoplasms of the colon, rectum and anus.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS130v10
    """

    VALUE_SET_NAME = "Malignant Neoplasm of Colon"
    OID = "2.16.840.1.113883.3.464.1003.108.12.1001"
    DEFINITION_VERSION = "20171222"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "C180",  # Malignant neoplasm of cecum
        "C181",  # Malignant neoplasm of appendix
        "C182",  # Malignant neoplasm of ascending colon
        "C183",  # Malignant neoplasm of hepatic flexure
        "C184",  # Malignant neoplasm of transverse colon
        "C185",  # Malignant neoplasm of splenic flexure
        "C186",  # Malignant neoplasm of descending colon
        "C187",  # Malignant neoplasm of sigmoid colon
        "C188",  # Malignant neoplasm of overlapping sites of colon
        "C189",  # Malignant neoplasm of colon, unspecified
        "C19",  # Malignant neoplasm of rectosigmoid junction
        "C20",  # Malignant neoplasm of rectum
        "C212",  # Malignant neoplasm of cloacogenic zone
        "C218",  # Malignant neoplasm of overlapping sites of rectum, anus and anal canal
        "C785",  # Secondary malignant neoplasm of large intestine and rectum
    }
    ICD9CM = {
        "1530",  # Malignant neoplasm of hepatic flexure
        "1531",  # Malignant neoplasm of transverse colon
        "1532",  # Malignant neoplasm of descending colon
        "1533",  # Malignant neoplasm of sigmoid colon
        "1534",  # Malignant neoplasm of cecum
        "1535",  # Malignant neoplasm of appendix vermiformis
        "1536",  # Malignant neoplasm of ascending colon
        "1537",  # Malignant neoplasm of splenic flexure
        "1538",  # Malignant neoplasm of other specified sites of large intestine
        "1539",  # Malignant neoplasm of colon, unspecified site
        "1540",  # Malignant neoplasm of rectosigmoid junction
        "1541",  # Malignant neoplasm of rectum
        "1975",  # Secondary malignant neoplasm of large intestine and rectum
    }
    SNOMEDCT = {
        "93761005",  # Primary malignant neoplasm of colon (disorder)
        "93771007",  # Primary malignant neoplasm of descending colon (disorder)
        "93826009",  # Primary malignant neoplasm of hepatic flexure of colon (disorder)
        "93980002",  # Primary malignant neoplasm of rectosigmoid junction (disorder)
        "94006002",  # Primary malignant neoplasm of sigmoid colon (disorder)
        "94072004",  # Primary malignant neoplasm of splenic flexure of colon (disorder)
        "94105000",  # Primary malignant neoplasm of transverse colon (disorder)
        "94179005",  # Secondary malignant neoplasm of ascending colon (disorder)
        "94260004",  # Secondary malignant neoplasm of colon (disorder)
        "94271003",  # Secondary malignant neoplasm of descending colon (disorder)
        "94328005",  # Secondary malignant neoplasm of hepatic flexure of colon (disorder)
        "94509004",  # Secondary malignant neoplasm of rectosigmoid junction (disorder)
        "94538001",  # Secondary malignant neoplasm of sigmoid colon (disorder)
        "94604000",  # Secondary malignant neoplasm of splenic flexure of colon (disorder)
        "94643001",  # Secondary malignant neoplasm of transverse colon (disorder)
        "109838007",  # Overlapping malignant neoplasm of colon (disorder)
        "187757001",  # Malignant neoplasm, overlapping lesion of colon (disorder)
        "269533000",  # Carcinoma of colon (disorder)
        "269544008",  # Carcinoma of the rectosigmoid junction (disorder)
        "285312008",  # Carcinoma of sigmoid colon (disorder)
        "285611007",  # Metastasis to colon of unknown primary (disorder)
        "301756000",  # Adenocarcinoma of sigmoid colon (disorder)
        "312111009",  # Carcinoma of ascending colon (disorder)
        "312112002",  # Carcinoma of transverse colon (disorder)
        "312113007",  # Carcinoma of descending colon (disorder)
        "312114001",  # Carcinoma of hepatic flexure (disorder)
        "312115000",  # Carcinoma of splenic flexure (disorder)
        "314965007",  # Local recurrence of malignant tumor of colon (disorder)
        "315058005",  # Hereditary nonpolyposis colon cancer (disorder)
        "363406005",  # Malignant neoplasm of colon (disorder)
        "363407001",  # Malignant tumor of hepatic flexure (disorder)
        "363408006",  # Malignant tumor of transverse colon (disorder)
        "363409003",  # Malignant tumor of descending colon (disorder)
        "363410008",  # Malignant tumor of sigmoid colon (disorder)
        "363412000",  # Malignant tumor of ascending colon (disorder)
        "363413005",  # Malignant tumor of splenic flexure (disorder)
        "363414004",  # Malignant tumor of rectosigmoid junction (disorder)
        "363510005",  # Malignant tumor of large intestine (disorder)
        "425178004",  # Adenocarcinoma of rectosigmoid junction (disorder)
        "449218003",  # Lymphoma of sigmoid colon (disorder)
        "716654007",  # Non-polyposis Turcot syndrome (disorder)
        "721695008",  # Primary adenocarcinoma of ascending colon and right flexure (disorder)
        "721696009",  # Primary adenocarcinoma of transverse colon (disorder)
        "721699002",  # Primary adenocarcinoma of descending colon and splenic flexure (disorder)
        "737058005",  # Microsatellite instability-high colorectal cancer (disorder)
        "766981007",  # Squamous cell carcinoma of colon (disorder)
        "1701000119104",  # Primary adenocarcinoma of colon (disorder)
        "96281000119107",  # Overlapping malignant neoplasm of colon and rectum (disorder)
        "96981000119102",  # Malignant neoplasm of rectosigmoid junction metastatic to brain (disorder)
        "130381000119103",  # Primary malignant neuroendocrine neoplasm of ascending colon (disorder)
        "133751000119102",  # Lymphoma of colon (disorder)
        "184881000119106",  # Primary adenocarcinoma of rectosigmoid junction (disorder)
        "681601000119101",  # Primary adenocarcinoma of ascending colon (disorder)
    }


class DiabeticRetinopathy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of diabetic retinopathy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of diabetic retinopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS142v10, CMS133v10, CMS131v10
    """

    VALUE_SET_NAME = "Diabetic Retinopathy"
    OID = "2.16.840.1.113883.3.526.3.327"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "E08311",  # Diabetes mellitus due to underlying condition with unspecified diabetic retinopathy with macular edema
        "E08319",  # Diabetes mellitus due to underlying condition with unspecified diabetic retinopathy without macular edema
        "E083211",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E083212",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E083213",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E083291",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E083292",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E083293",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E083311",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E083312",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E083313",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E083391",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E083392",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E083393",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E083411",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E083412",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E083413",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E083491",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E083492",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E083493",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E083511",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, right eye
        "E083512",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, left eye
        "E083513",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, bilateral
        "E083521",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E083522",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E083523",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E083531",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E083532",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E083533",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E083541",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E083542",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E083543",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E083551",  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, right eye
        "E083552",  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, left eye
        "E083553",  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, bilateral
        "E083591",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy without macular edema, right eye
        "E083592",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy without macular edema, left eye
        "E083593",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy without macular edema, bilateral
        "E09311",  # Drug or chemical induced diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E09319",  # Drug or chemical induced diabetes mellitus with unspecified diabetic retinopathy without macular edema
        "E093211",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E093212",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E093213",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E093291",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E093292",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E093293",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E093311",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E093312",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E093313",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E093391",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E093392",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E093393",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E093411",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E093412",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E093413",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E093491",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E093492",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E093493",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E093511",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E093512",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E093513",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E093521",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E093522",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E093523",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E093531",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E093532",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E093533",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E093541",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E093542",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E093543",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E093551",  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E093552",  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E093553",  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E093591",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        "E093592",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        "E093593",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        "E10311",  # Type 1 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E10319",  # Type 1 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        "E103211",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E103212",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E103213",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103291",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E103292",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E103293",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E103311",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E103312",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E103313",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103391",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E103392",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E103393",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E103411",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E103412",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E103413",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103491",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E103492",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E103493",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E103511",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E103512",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E103513",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E103521",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E103522",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E103523",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E103531",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E103532",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E103533",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E103541",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E103542",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E103543",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E103551",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E103552",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E103553",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E103591",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        "E103592",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        "E103593",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        "E11311",  # Type 2 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E11319",  # Type 2 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        "E113211",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E113212",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E113213",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113291",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E113292",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E113293",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E113311",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E113312",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E113313",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113391",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E113392",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E113393",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E113411",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E113412",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E113413",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113491",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E113492",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E113493",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E113511",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E113512",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E113513",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E113521",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E113522",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E113523",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E113531",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E113532",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E113533",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E113541",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E113542",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E113543",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E113551",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E113552",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E113553",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E113591",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        "E113592",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        "E113593",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        "E13311",  # Other specified diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E13319",  # Other specified diabetes mellitus with unspecified diabetic retinopathy without macular edema
        "E133211",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E133212",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E133213",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133291",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        "E133292",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        "E133293",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        "E133311",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E133312",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E133313",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133391",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        "E133392",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        "E133393",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        "E133411",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E133412",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E133413",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133491",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        "E133492",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        "E133493",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        "E133511",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E133512",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E133513",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E133521",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E133522",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E133523",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E133531",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E133532",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E133533",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E133541",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E133542",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E133543",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E133551",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E133552",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E133553",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E133591",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        "E133592",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        "E133593",  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
    }
    SNOMEDCT = {
        "4855003",  # Retinopathy due to diabetes mellitus (disorder)
        "25412000",  # Microaneurysm of retinal artery due to diabetes mellitus (disorder)
        "59276001",  # Proliferative retinopathy due to diabetes mellitus (disorder)
        "193349004",  # Preproliferative retinopathy due to diabetes mellitus (disorder)
        "193350004",  # Advanced maculopathy due to diabetes mellitus (disorder)
        "232020009",  # Disorder of macula due to diabetes mellitus (disorder)
        "232021008",  # Proliferative retinopathy with optic disc neovascularization due to diabetes mellitus (disorder)
        "232022001",  # Proliferative retinopathy with neovascularization elsewhere than the optic disc due to diabetes mellitus (disorder)
        "232023006",  # Traction detachment of retina due to diabetes mellitus (disorder)
        "311782002",  # Advanced retinal disease due to diabetes mellitus (disorder)
        "312903003",  # Mild nonproliferative retinopathy due to diabetes mellitus (disorder)
        "312904009",  # Moderate nonproliferative retinopathy due to diabetes mellitus (disorder)
        "312905005",  # Severe nonproliferative retinopathy due to diabetes mellitus (disorder)
        "312906006",  # Non-high-risk proliferative retinopathy due to diabetes mellitus (disorder)
        "312907002",  # High risk proliferative retinopathy due to diabetes mellitus (disorder)
        "312908007",  # Quiescent proliferative retinopathy due to diabetes mellitus (disorder)
        "312909004",  # Proliferative retinopathy with iris neovascularization due to diabetes mellitus (disorder)
        "312912001",  # Macular edema due to diabetes mellitus (disorder)
        "314010006",  # Diffuse exudative maculopathy due to diabetes mellitus (disorder)
        "314011005",  # Focal exudative maculopathy due to diabetes mellitus (disorder)
        "314014002",  # Ischemic maculopathy due to diabetes mellitus (disorder)
        "314015001",  # Mixed maculopathy due to diabetes mellitus (disorder)
        "390834004",  # Nonproliferative retinopathy due to diabetes mellitus (disorder)
        "399862001",  # High risk proliferative retinopathy without macular edema due to diabetes mellitus (disorder)
        "399863006",  # Very severe nonproliferative retinopathy without macular edema due to diabetes mellitus (disorder)
        "399864000",  # Macular edema not clinically significant due to diabetes mellitus (disorder)
        "399865004",  # Very severe proliferative retinopathy due to diabetes mellitus (disorder)
        "399866003",  # Venous beading of retina due to diabetes mellitus (disorder)
        "399868002",  # Intraretinal microvascular anomaly due to diabetes mellitus (disorder)
        "399869005",  # High risk proliferative retinopathy not amenable to photocoagulation due to diabetes mellitus (disorder)
        "399870006",  # Non-high-risk proliferative retinopathy with no macular edema due to diabetes mellitus (disorder)
        "399871005",  # Visually threatening retinopathy due to diabetes mellitus (disorder)
        "399872003",  # Severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        "399873008",  # Severe nonproliferative retinopathy without macular edema due to diabetes mellitus (disorder)
        "399874002",  # High risk proliferative retinopathy with clinically significant macula edema due to diabetes mellitus (disorder)
        "399875001",  # Non-high-risk proliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        "399876000",  # Very severe nonproliferative retinopathy due to diabetes mellitus (disorder)
        "399877009",  # Very severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        "420486006",  # Exudative maculopathy due to type 1 diabetes mellitus (disorder)
        "420789003",  # Retinopathy due to type 1 diabetes mellitus (disorder)
        "421779007",  # Exudative maculopathy due to type 2 diabetes mellitus (disorder)
        "422034002",  # Retinopathy due to type 2 diabetes mellitus (disorder)
        "870420005",  # Severe nonproliferative retinopathy with venous beading of retina due to diabetes mellitus (disorder)
    }


class AcuteAndSubacuteIridocyclitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of acute and subacute iridocyclitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of acute and subacute iridocyclitis, including primary and secondary iridocyclitis, and hypopyon.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Acute and Subacute Iridocyclitis"
    OID = "2.16.840.1.113883.3.526.3.1241"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H2000",  # Unspecified acute and subacute iridocyclitis
        "H20011",  # Primary iridocyclitis, right eye
        "H20012",  # Primary iridocyclitis, left eye
        "H20013",  # Primary iridocyclitis, bilateral
        "H20021",  # Recurrent acute iridocyclitis, right eye
        "H20022",  # Recurrent acute iridocyclitis, left eye
        "H20023",  # Recurrent acute iridocyclitis, bilateral
        "H20031",  # Secondary infectious iridocyclitis, right eye
        "H20032",  # Secondary infectious iridocyclitis, left eye
        "H20033",  # Secondary infectious iridocyclitis, bilateral
        "H20041",  # Secondary noninfectious iridocyclitis, right eye
        "H20042",  # Secondary noninfectious iridocyclitis, left eye
        "H20043",  # Secondary noninfectious iridocyclitis, bilateral
        "H20051",  # Hypopyon, right eye
        "H20052",  # Hypopyon, left eye
        "H20053",  # Hypopyon, bilateral
    }
    SNOMEDCT = {
        "4927003",  # Acute anterior uveitis (disorder)
        "6869001",  # Recurrent iridocyclitis (disorder)
        "9091006",  # Gonococcal iridocyclitis (disorder)
        "10698009",  # Herpes zoster iridocyclitis (disorder)
        "12630008",  # Primary iridocyclitis (disorder)
        "13608004",  # Herpes simplex iridocyclitis (disorder)
        "14427009",  # Subacute iritis (disorder)
        "20555001",  # Herpes simplex iritis (disorder)
        "29050005",  # Acute iritis (disorder)
        "29538005",  # Glaucomatocyclitic crisis (disorder)
        "30080002",  # Secondary syphilitic iridocyclitis (disorder)
        "43193009",  # Gouty iritis (disorder)
        "46916006",  # Noninfectious secondary iridocyclitis (disorder)
        "64015001",  # Subacute anterior uveitis (disorder)
        "65074000",  # Iritis (disorder)
        "70461003",  # Lens-induced iridocyclitis (disorder)
        "77971008",  # Iridocyclitis (disorder)
        "89114005",  # Infectious secondary iridocyclitis (disorder)
        "193487008",  # Secondary infected iridocyclitis (disorder)
        "193488003",  # Secondary non-infected iridocyclitis (disorder)
        "193489006",  # Iritis due to diabetes mellitus (disorder)
        "231947004",  # Anterior uveitis idiopathic (disorder)
        "267618008",  # Acute and subacute iridocyclitis (disorder)
        "267619000",  # Non-infectious anterior uveitis (disorder)
        "410480004",  # Iritis in Behcet's syndrome (disorder)
        "410482007",  # Iritis in psoriatic arthritis (disorder)
        "410692006",  # Anterior uveitis (disorder)
        "413851004",  # Cidofovir-induced anterior uveitis (disorder)
        "414600000",  # Lepromatous anterior uveitis (disorder)
        "416666007",  # Anterior uveitis in juvenile idiopathic arthritis (disorder)
        "417020006",  # Traumatic iritis (disorder)
        "420485005",  # Herpetic iridocyclitis (disorder)
        "733317009",  # Primary anterior uveitis (disorder)
        "35881000119106",  # Noninfectious secondary iritis (disorder)
        "220101000119108",  # Infectious secondary iritis (disorder)
        "331071000119101",  # Herpes zoster iritis (disorder)
        "332921000119107",  # Primary iridocyclitis of right eye (disorder)
        "336531000119103",  # Acute iritis of right eye (disorder)
        "336541000119107",  # Acute anterior uveitis of right eye (disorder)
        "336921000119106",  # Lens induced iritis (disorder)
        "338531000119102",  # Primary iridocyclitis of left eye (disorder)
        "342141000119102",  # Acute iritis of left eye (disorder)
        "342151000119100",  # Acute anterior uveitis of left eye (disorder)
        "342381000119109",  # Gonococcal iritis (disorder)
        "344301000119109",  # Primary iridocyclitis of bilateral eyes (disorder)
        "347301000119103",  # Acute iritis of bilateral eyes (disorder)
        "347311000119100",  # Acute anterior uveitis of bilateral eyes (disorder)
        "352941000119102",  # Sarcoid iridocyclitis (disorder)
        "675361000119103",  # Subacute iridocyclitis of bilateral eyes (disorder)
        "675371000119109",  # Subacute iridocyclitis of left eye (disorder)
        "675381000119107",  # Subacute iridocyclitis of right eye (disorder)
        "15638451000119104",  # Iritis of right eye (disorder)
        "15638491000119109",  # Iritis of left eye (disorder)
        "15638531000119109",  # Iritis of bilateral eyes (disorder)
        "15679361000119103",  # Iridocyclitis of left eye (disorder)
        "15679401000119107",  # Iridocyclitis of right eye (disorder)
        "15679441000119109",  # Iridocyclitis of bilateral eyes (disorder)
    }


class Amblyopia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of amblyopia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of amblyopia, including suspected amblyopia.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Amblyopia"
    OID = "2.16.840.1.113883.3.526.3.1448"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H53001",  # Unspecified amblyopia, right eye
        "H53002",  # Unspecified amblyopia, left eye
        "H53003",  # Unspecified amblyopia, bilateral
        "H53011",  # Deprivation amblyopia, right eye
        "H53012",  # Deprivation amblyopia, left eye
        "H53013",  # Deprivation amblyopia, bilateral
        "H53021",  # Refractive amblyopia, right eye
        "H53022",  # Refractive amblyopia, left eye
        "H53023",  # Refractive amblyopia, bilateral
        "H53031",  # Strabismic amblyopia, right eye
        "H53032",  # Strabismic amblyopia, left eye
        "H53033",  # Strabismic amblyopia, bilateral
        "H53041",  # Amblyopia suspect, right eye
        "H53042",  # Amblyopia suspect, left eye
        "H53043",  # Amblyopia suspect, bilateral
    }
    SNOMEDCT = {
        "965003",  # Toxic amblyopia (disorder)
        "35600002",  # Strabismic amblyopia (disorder)
        "70701004",  # Alcoholic amblyopia (disorder)
        "75834006",  # Strachan's syndrome (disorder)
        "90927000",  # Refractive amblyopia (disorder)
        "193638002",  # Stimulus deprivation amblyopia (disorder)
        "232147001",  # Anisometropic amblyopia (disorder)
        "361120000",  # Nutritional amblyopia (disorder)
        "387742006",  # Amblyopia (disorder)
        "387743001",  # Functional amblyopia (disorder)
        "396184003",  # Isometropic amblyopia (disorder)
        "410486005",  # Occlusion amblyopia (disorder)
        "331961000119108",  # Strabismic amblyopia of right eye (disorder)
        "332701000119101",  # Refractive amblyopia of right eye (disorder)
        "335461000119102",  # Deprivation amblyopia of right eye (disorder)
        "336451000119108",  # Amblyopia of right eye (disorder)
        "337581000119100",  # Strabismic amblyopia of left eye (disorder)
        "338311000119102",  # Refractive amblyopia of left eye (disorder)
        "341071000119100",  # Deprivation amblyopia of left eye (disorder)
        "342061000119102",  # Amblyopia of left eye (disorder)
        "343371000119101",  # Strabismic amblyopia of bilateral eyes (disorder)
        "344081000119104",  # Refractive amblyopia of bilateral eyes (disorder)
        "346331000119107",  # Deprivation amblyopia of bilateral eyes (disorder)
        "347241000119107",  # Amblyopia of bilateral eyes (disorder)
    }


class BurnConfinedToEyeAndAdnexa(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of a burn of the eye and/or adnexa.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of burns of various parts of the eye, or eye corrosion.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Burn Confined to Eye and Adnexa"
    OID = "2.16.840.1.113883.3.526.3.1409"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "T2601XA",  # Burn of right eyelid and periocular area, initial encounter
        "T2602XA",  # Burn of left eyelid and periocular area, initial encounter
        "T2611XA",  # Burn of cornea and conjunctival sac, right eye, initial encounter
        "T2612XA",  # Burn of cornea and conjunctival sac, left eye, initial encounter
        "T2621XA",  # Burn with resulting rupture and destruction of right eyeball, initial encounter
        "T2622XA",  # Burn with resulting rupture and destruction of left eyeball, initial encounter
        "T2631XA",  # Burns of other specified parts of right eye and adnexa, initial encounter
        "T2632XA",  # Burns of other specified parts of left eye and adnexa, initial encounter
        "T2641XA",  # Burn of right eye and adnexa, part unspecified, initial encounter
        "T2642XA",  # Burn of left eye and adnexa, part unspecified, initial encounter
        "T2651XA",  # Corrosion of right eyelid and periocular area, initial encounter
        "T2652XA",  # Corrosion of left eyelid and periocular area, initial encounter
        "T2661XA",  # Corrosion of cornea and conjunctival sac, right eye, initial encounter
        "T2662XA",  # Corrosion of cornea and conjunctival sac, left eye, initial encounter
        "T2671XA",  # Corrosion with resulting rupture and destruction of right eyeball, initial encounter
        "T2672XA",  # Corrosion with resulting rupture and destruction of left eyeball, initial encounter
        "T2681XA",  # Corrosions of other specified parts of right eye and adnexa, initial encounter
        "T2682XA",  # Corrosions of other specified parts of left eye and adnexa, initial encounter
        "T2691XA",  # Corrosion of right eye and adnexa, part unspecified, initial encounter
        "T2692XA",  # Corrosion of left eye and adnexa, part unspecified, initial encounter
    }
    SNOMEDCT = {
        "38916004",  # Alkaline chemical burn of cornea AND/OR conjunctival sac (disorder)
        "63542002",  # Acid chemical burn of cornea AND/OR conjunctival sac (disorder)
        "110170006",  # Burn of periorbital region (disorder)
        "110171005",  # Epidermal burn of periorbital region (disorder)
        "110172003",  # Partial thickness burn of periorbital region (disorder)
        "110173008",  # Full thickness burn of periorbital region (disorder)
        "211659009",  # Chemical burn of eyelid region (disorder)
        "211661000",  # Alkaline chemical burn of cornea and conjunctival sac (disorder)
        "211662007",  # Acid chemical burn of cornea and conjunctival sac (disorder)
        "231819004",  # Thermal burn of eyelid (disorder)
        "231844007",  # Lacrimal punctum burn (disorder)
        "231867009",  # Thermal injury to conjunctiva (disorder)
        "231869007",  # Chemical burn injury to conjunctiva (disorder)
        "231884003",  # Burn of sclera (disorder)
        "231943000",  # Thermal and radiation injury to the cornea (disorder)
        "231945007",  # Chemical injury to cornea (disorder)
        "232028002",  # Laser coagulation burn to retina (disorder)
        "269232009",  # Burn confined to eye and adnexa (disorder)
        "274204004",  # Burn of cornea (disorder)
        "274205003",  # Burn of eye region (disorder)
        "284537006",  # Eyelid burn (disorder)
        "284541005",  # Burn of conjunctiva (disorder)
        "287132002",  # Epidermal burn of eyelid (disorder)
        "287133007",  # Burn to eyelid - blister (disorder)
        "287134001",  # Full thickness burn of eyelid (disorder)
        "287137008",  # Burn to cornea - blister (disorder)
        "287138003",  # Burn to cornea - full thickness (disorder)
        "312646002",  # Burn of skin of eye region (disorder)
        "314531003",  # Burn of anterior eyeball segment (disorder)
        "418591009",  # Corneal wound burn (disorder)
        "423838005",  # Scleral surgical wound burn (disorder)
        "445832003",  # Chemical burn of eyelid (disorder)
        "445850005",  # Deep third degree burn of eye region (disorder)
        "446234004",  # Alkaline chemical burn of conjunctival sac (disorder)
        "446556000",  # Acid chemical burn of cornea (disorder)
        "446856005",  # Burn of conjuctival sac (disorder)
        "447094002",  # Alkaline chemical burn of cornea (disorder)
        "447095001",  # Acid chemical burn of conjunctival sac (disorder)
        "447333000",  # Burn of ocular adnexa (disorder)
    }


class CataractSecondaryToOcularDisorders(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of cataract that is a secondary diagnosis to other ocular conditions.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of cataract secondary to degenerative or inflammatory ocular disorders, cataract with neovascularization, or subcapsular glaucomatous flecks.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Cataract Secondary to Ocular Disorders"
    OID = "2.16.840.1.113883.3.526.3.1410"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H26211",  # Cataract with neovascularization, right eye
        "H26212",  # Cataract with neovascularization, left eye
        "H26213",  # Cataract with neovascularization, bilateral
        "H26221",  # Cataract secondary to ocular disorders (degenerative) (inflammatory), right eye
        "H26222",  # Cataract secondary to ocular disorders (degenerative) (inflammatory), left eye
        "H26223",  # Cataract secondary to ocular disorders (degenerative) (inflammatory), bilateral
    }
    SNOMEDCT = {
        "67733005",  # Subcapsular glaucomatous flecks (disorder)
        "76309006",  # Cataract with neovascularization (disorder)
        "193600001",  # Cataract secondary to ocular disease (disorder)
        "193602009",  # Cataract due to inflammatory disorder (disorder)
        "15738161000119104",  # Cataract of right eye due to ocular disease (disorder)
        "15738201000119109",  # Cataract of left eye due ocular disease (disorder)
        "15738241000119106",  # Cataract of left eye due to inflammatory disorder (disorder)
        "15738281000119101",  # Cataract of right eye due to inflammatory disorder (disorder)
        "15738321000119106",  # Cataract of bilateral eyes due to inflammatory disorder (disorder)
    }


class Cataract_Congenital(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of congenital cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of congenital cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Cataract, Congenital"
    OID = "2.16.840.1.113883.3.526.3.1412"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Q120",  # Congenital cataract
    }
    SNOMEDCT = {
        "21590003",  # Congenital zonular cataract (disorder)
        "28550007",  # Congenital capsular cataract (disorder)
        "29590001",  # Congenital total cataract (disorder)
        "66499004",  # Congenital cortical cataract (disorder)
        "76562003",  # Congenital subcapsular cataract (disorder)
        "79410001",  # Congenital cataract (disorder)
        "204128001",  # Congenital lamellar cataract (disorder)
        "204138006",  # Congenital blue dot cataract (disorder)
        "204139003",  # Congenital membranous cataract (disorder)
        "253223002",  # Congenital polar cataract (disorder)
        "253224008",  # Congenital anterior polar cataract (disorder)
        "253225009",  # Congenital posterior polar cataract (disorder)
        "253226005",  # Congenital sutural cataract (disorder)
        "253227001",  # Rubella cataract (disorder)
        "335841000119103",  # Congenital nuclear cataract of right eye (disorder)
        "341451000119100",  # Congenital nuclear cataract of left eye (disorder)
        "346711000119101",  # Congenital combined form cataract of bilateral eyes (disorder)
        "346731000119106",  # Congenital anterior subcapsular polar cataract of bilateral eyes (disorder)
        "349281000119109",  # Congenital zonular cataract of bilateral eyes (disorder)
        "679911000119104",  # Congenital cortical cataract of left eye (disorder)
        "679931000119109",  # Congenital cortical cataract of right eye (disorder)
        "681041000119106",  # Congenital cortical cataract of bilateral eyes (disorder)
    }


class Cataract_MatureOrHypermature(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of mature or hypermature cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of mature or hypermature cataract, including senile or intumescent cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Cataract, Mature or Hypermature"
    OID = "2.16.840.1.113883.3.526.3.1413"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H269",  # Unspecified cataract
    }
    SNOMEDCT = {
        "849000",  # Mature cataract (disorder)
        "8656007",  # Total traumatic cataract (disorder)
        "95724003",  # Intumescent cataract (disorder)
        "193590000",  # Total, mature senile cataract (disorder)
        "264443002",  # Morgagnian cataract (disorder)
        "267626000",  # Hypermature cataract (disorder)
        "217791000119107",  # Hypermature senile cataract (disorder)
        "347461000119107",  # Hypermature senile cataract of right eye (disorder)
        "347521000119103",  # Hypermature senile cataract of left eye (disorder)
        "347581000119104",  # Hypermature senile cataract of bilateral eyes (disorder)
    }


class Cataract_PosteriorPolar(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of posterior polar cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of posterior polar cataract, including capsular or subcapsular cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Cataract, Posterior Polar"
    OID = "2.16.840.1.113883.3.526.3.1414"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H25041",  # Posterior subcapsular polar age-related cataract, right eye
        "H25042",  # Posterior subcapsular polar age-related cataract, left eye
        "H25043",  # Posterior subcapsular polar age-related cataract, bilateral
    }
    SNOMEDCT = {
        "5318001",  # Posterior subcapsular polar senile cataract (disorder)
        "34533008",  # Posterior subcapsular polar cataract (disorder)
        "253225009",  # Congenital posterior polar cataract (disorder)
        "315353005",  # Posterior subcapsular cataract (disorder)
        "335831000119107",  # Congenital posterior subcapsular polar cataract of right eye (disorder)
        "341441000119102",  # Congenital posterior subcapsular polar cataract of left eye (disorder)
        "342821000119103",  # Congenital posterior subcapsular polar cataract (disorder)
        "346691000119104",  # Congenital posterior subcapsular polar cataract of bilateral eyes (disorder)
        "347561000119108",  # Posterior subcapsular polar senile cataract of bilateral eyes (disorder)
        "1078791000119109",  # Posterior subcapsular polar senile cataract of left eye (disorder)
        "1078801000119105",  # Posterior subcapsular polar senile cataract of right eye (disorder)
        "15639401000119105",  # Cataract of posterior subcapsule of right eye (disorder)
        "15639441000119107",  # Cataract of posterior subcapsule of left eye (disorder)
        "15639481000119102",  # Cataract of posterior subcapsule of bilateral eyes (disorder)
        "15737681000119103",  # Anterior subcapsular polar cataract of bilateral eyes (disorder)
        "15737881000119104",  # Posterior subcapsular polar cataract of bilateral eyes (disorder)
        "15737921000119106",  # Posterior subcapsular polar cataract of left eye (disorder)
        "15737961000119101",  # Posterior subcapsular polar cataract of right eye (disorder)
        "16477021000119103",  # Anterior subcapsular polar senile cataract of bilateral eyes (disorder)
    }


class CentralCornealUlcer(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of central corneal ulcer.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of central corneal ulcer or keratitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Central Corneal Ulcer"
    OID = "2.16.840.1.113883.3.526.3.1428"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H16011",  # Central corneal ulcer, right eye
        "H16012",  # Central corneal ulcer, left eye
        "H16013",  # Central corneal ulcer, bilateral
    }
    SNOMEDCT = {
        "7426009",  # Central corneal ulcer (disorder)
        "42513006",  # Punctate keratitis (disorder)
        "95732006",  # Necrotizing keratitis (disorder)
        "95733001",  # Eosinophilic keratitis (disorder)
        "231901007",  # Neuropathic corneal ulcer (disorder)
        "397567009",  # Keratoconjunctivitis caused by molluscum contagiosum (disorder)
        "397570008",  # Herpes simplex stromal keratitis (disorder)
        "332801000119108",  # Punctate keratitis of right eye (disorder)
        "338411000119106",  # Punctate keratitis of left eye (disorder)
        "344181000119103",  # Punctate keratitis of bilateral eyes (disorder)
    }


class CertainTypesOfIridocyclitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of various types of iridocyclitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of lens-induced iridocyclitis, Fuchs' heterochromic cyclitis, Vogt-Koyanagi syndrome, and other types of iridocyclitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Certain Types of Iridocyclitis"
    OID = "2.16.840.1.113883.3.526.3.1415"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H2021",  # Lens-induced iridocyclitis, right eye
        "H2022",  # Lens-induced iridocyclitis, left eye
        "H2023",  # Lens-induced iridocyclitis, bilateral
        "H20811",  # Fuchs' heterochromic cyclitis, right eye
        "H20812",  # Fuchs' heterochromic cyclitis, left eye
        "H20813",  # Fuchs' heterochromic cyclitis, bilateral
        "H20821",  # Vogt-Koyanagi syndrome, right eye
        "H20822",  # Vogt-Koyanagi syndrome, left eye
        "H20823",  # Vogt-Koyanagi syndrome, bilateral
        "H209",  # Unspecified iridocyclitis
    }
    SNOMEDCT = {
        "6869001",  # Recurrent iridocyclitis (disorder)
        "9091006",  # Gonococcal iridocyclitis (disorder)
        "10698009",  # Herpes zoster iridocyclitis (disorder)
        "11226001",  # Fuchs' heterochromic cyclitis (disorder)
        "12630008",  # Primary iridocyclitis (disorder)
        "13608004",  # Herpes simplex iridocyclitis (disorder)
        "23798003",  # Translucency of iris (disorder)
        "29538005",  # Glaucomatocyclitic crisis (disorder)
        "30080002",  # Secondary syphilitic iridocyclitis (disorder)
        "46916006",  # Noninfectious secondary iridocyclitis (disorder)
        "70461003",  # Lens-induced iridocyclitis (disorder)
        "77971008",  # Iridocyclitis (disorder)
        "89114005",  # Infectious secondary iridocyclitis (disorder)
        "193487008",  # Secondary infected iridocyclitis (disorder)
        "193488003",  # Secondary non-infected iridocyclitis (disorder)
        "267619000",  # Non-infectious anterior uveitis (disorder)
        "420485005",  # Herpetic iridocyclitis (disorder)
        "15678641000119101",  # Iridocyclitis of right eye caused by human herpes simplex virus (disorder)
        "15678681000119106",  # Iridocyclitis of left eye caused by human herpes simplex virus (disorder)
        "15678761000119105",  # Iridocyclitis of right eye due to herpes zoster (disorder)
        "15678801000119102",  # Iridocyclitis of left eye due to herpes zoster (disorder)
    }


class ChoroidalDegenerations(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of choroidal degenerations.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of degenerations of the choroid, including choroid atrophy and dystrophy, angioid streaks, and choroideremia.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Choroidal Degenerations"
    OID = "2.16.840.1.113883.3.526.3.1450"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H3533",  # Angioid streaks of macula
    }
    SNOMEDCT = {
        "19871000",  # Total generalized choroidal dystrophy (disorder)
        "24950005",  # Partial generalized choroidal dystrophy (disorder)
        "38513001",  # Senile atrophy of choroid (disorder)
        "41250005",  # Diffuse choroidal sclerosis (disorder)
        "59753003",  # Total circumpapillary dystrophy of choroid (disorder)
        "74469006",  # Hereditary choroidal dystrophy (disorder)
        "75241009",  # Choroideremia (disorder)
        "86103006",  # Angioid streaks of choroid (disorder)
        "92820007",  # Circumpapillary dystrophy of choroid (disorder)
        "193463006",  # Diffuse secondary choroid atrophy (disorder)
        "193466003",  # Partial circumpapillary choroid dystrophy (disorder)
        "231996009",  # Central areolar choroidal sclerosis (disorder)
        "231997000",  # Peripapillary choroidal sclerosis (disorder)
        "312927001",  # Bietti's crystalline retinopathy (disorder)
        "406446000",  # Choroidal degeneration (disorder)
    }


class ChoroidalDetachment(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of choroidal detachment.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis choroidal detachment, or effusion.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Choroidal Detachment"
    OID = "2.16.840.1.113883.3.526.3.1451"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H31401",  # Unspecified choroidal detachment, right eye
        "H31402",  # Unspecified choroidal detachment, left eye
        "H31403",  # Unspecified choroidal detachment, bilateral
        "H31411",  # Hemorrhagic choroidal detachment, right eye
        "H31412",  # Hemorrhagic choroidal detachment, left eye
        "H31413",  # Hemorrhagic choroidal detachment, bilateral
        "H31421",  # Serous choroidal detachment, right eye
        "H31422",  # Serous choroidal detachment, left eye
        "H31423",  # Serous choroidal detachment, bilateral
    }
    SNOMEDCT = {
        "19031009",  # Hemorrhagic choroidal detachment (disorder)
        "27871007",  # Choroidal detachment (disorder)
        "78584008",  # Serous choroidal detachment (disorder)
        "232000008",  # Choroidal effusion (disorder)
        "247198004",  # Annular choroidal detachment (disorder)
        "278805002",  # Uveal effusion syndrome (disorder)
        "778049009",  # Idiopathic uveal effusion syndrome (disorder)
        "332111000119102",  # Serous choroidal detachment of right eye (disorder)
        "333881000119109",  # Minor opacity of right cornea (disorder)
        "334901000119109",  # Hemorrhagic detachment of right choroid (disorder)
        "336051000119104",  # Rupture of choroid of right eye (disorder)
        "336061000119102",  # Hemorrhage of choroid of right eye (disorder)
        "336071000119108",  # Choroidal detachment of right eye (disorder)
        "337721000119107",  # Serous choroidal detachment of left eye (disorder)
        "339481000119109",  # Minor opacity of left cornea (disorder)
        "340511000119109",  # Hemorrhagic detachment of left choroid (disorder)
        "341661000119101",  # Rupture of choroid of left eye (disorder)
        "341671000119107",  # Hemorrhage of choroid of left eye (disorder)
        "341681000119105",  # Choroidal detachment of left eye (disorder)
        "343491000119109",  # Serous choroidal detachment of bilateral eyes (disorder)
        "344611000119101",  # Peripheral opacity of bilateral corneas (disorder)
    }


class ChoroidalHemorrhageAndRupture(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of choroidal hemorrhage and/or rupture.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of choroidal hemorrhage and rupture.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Choroidal Hemorrhage and Rupture"
    OID = "2.16.840.1.113883.3.526.3.1452"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H31301",  # Unspecified choroidal hemorrhage, right eye
        "H31302",  # Unspecified choroidal hemorrhage, left eye
        "H31303",  # Unspecified choroidal hemorrhage, bilateral
        "H31311",  # Expulsive choroidal hemorrhage, right eye
        "H31312",  # Expulsive choroidal hemorrhage, left eye
        "H31313",  # Expulsive choroidal hemorrhage, bilateral
        "H31321",  # Choroidal rupture, right eye
        "H31322",  # Choroidal rupture, left eye
        "H31323",  # Choroidal rupture, bilateral
    }
    SNOMEDCT = {
        "122003",  # Choroidal hemorrhage (disorder)
        "2532009",  # Choroidal rupture (disorder)
        "11623000",  # Expulsive choroidal hemorrhage (disorder)
        "193474002",  # Choroidal hemorrhage and rupture (disorder)
        "419596007",  # Suprachoroidal hemorrhage (disorder)
        "336051000119104",  # Rupture of choroid of right eye (disorder)
        "336061000119102",  # Hemorrhage of choroid of right eye (disorder)
        "341661000119101",  # Rupture of choroid of left eye (disorder)
        "341671000119107",  # Hemorrhage of choroid of left eye (disorder)
    }


class ChronicIridocyclitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of chronic iridocyclitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of chronic or recurrent iridocyclitis or tuberculous iridocyclitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Chronic Iridocyclitis"
    OID = "2.16.840.1.113883.3.526.3.1416"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "A1854",  # Tuberculous iridocyclitis
        "H2011",  # Chronic iridocyclitis, right eye
        "H2012",  # Chronic iridocyclitis, left eye
        "H2013",  # Chronic iridocyclitis, bilateral
        "H209",  # Unspecified iridocyclitis
    }
    SNOMEDCT = {
        "6869001",  # Recurrent iridocyclitis (disorder)
        "186266008",  # Tuberculous chronic iridocyclitis (disorder)
        "193493000",  # Chronic iridocyclitis due to another disorder (disorder)
        "416449002",  # Chronic iridocyclitis in young girls (disorder)
    }


class CloudyCornea(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of cloudy cornea.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of opacity of the cornea, adherent or corneal leukoma, and haziness of the cornea.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Cloudy Cornea"
    OID = "2.16.840.1.113883.3.526.3.1417"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H1701",  # Adherent leukoma, right eye
        "H1702",  # Adherent leukoma, left eye
        "H1703",  # Adherent leukoma, bilateral
        "H1711",  # Central corneal opacity, right eye
        "H1712",  # Central corneal opacity, left eye
        "H1713",  # Central corneal opacity, bilateral
        "H17811",  # Minor opacity of cornea, right eye
        "H17812",  # Minor opacity of cornea, left eye
        "H17813",  # Minor opacity of cornea, bilateral
        "H17821",  # Peripheral opacity of cornea, right eye
        "H17822",  # Peripheral opacity of cornea, left eye
        "H17823",  # Peripheral opacity of cornea, bilateral
    }
    SNOMEDCT = {
        "22135003",  # Corneal leukoma interfering with central vision (disorder)
        "30337003",  # Adherent leukoma (disorder)
        "54359000",  # Congenital corneal opacity not interfering with vision (disorder)
        "55713007",  # Peripheral opacity of cornea (disorder)
        "56530002",  # Central opacity of cornea (disorder)
        "64634000",  # Corneal opacity (disorder)
        "66104007",  # Minor opacity of cornea (disorder)
        "72494008",  # Corneal nebula (disorder)
        "204149000",  # Congenital corneal opacity without visual deficit (disorder)
        "413921009",  # Corneal subepithelial haze (disorder)
        "424353002",  # Corneal haze caused by herpes zoster (disorder)
        "425961009",  # Corneal haze caused by herpes simplex (disorder)
        "333881000119109",  # Minor opacity of right cornea (disorder)
        "336501000119105",  # Adherent leukoma of right eye (disorder)
        "339481000119109",  # Minor opacity of left cornea (disorder)
        "342111000119101",  # Adherent leukoma of left eye (disorder)
        "344611000119101",  # Peripheral opacity of bilateral corneas (disorder)
        "347271000119100",  # Adherent leukoma of bilateral eyes (disorder)
        "1099031000119105",  # Opacity of cornea of bilateral eyes (disorder)
        "1099041000119101",  # Opacity of cornea of left eye (disorder)
        "1099051000119104",  # Opacity of cornea of right eye (disorder)
    }


class CornealEdema(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of corneal edema.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of corneal edema, including keratopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Corneal Edema"
    OID = "2.16.840.1.113883.3.526.3.1418"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H1811",  # Bullous keratopathy, right eye
        "H1812",  # Bullous keratopathy, left eye
        "H1813",  # Bullous keratopathy, bilateral
        "H1820",  # Unspecified corneal edema
        "H18221",  # Idiopathic corneal edema, right eye
        "H18222",  # Idiopathic corneal edema, left eye
        "H18223",  # Idiopathic corneal edema, bilateral
        "H18231",  # Secondary corneal edema, right eye
        "H18232",  # Secondary corneal edema, left eye
        "H18233",  # Secondary corneal edema, bilateral
        "H18421",  # Band keratopathy, right eye
        "H18422",  # Band keratopathy, left eye
        "H18423",  # Band keratopathy, bilateral
        "H1843",  # Other calcerous corneal degeneration
    }
    SNOMEDCT = {
        "1794009",  # Idiopathic corneal edema (disorder)
        "27194006",  # Corneal edema (disorder)
        "27536004",  # Secondary corneal edema (disorder)
        "49362009",  # Corneal edema caused by wearing of contact lenses (disorder)
        "57207003",  # Bullous keratopathy (disorder)
        "129627002",  # Keratitis bullosa (disorder)
        "231921008",  # Phakic corneal edema (disorder)
        "231922001",  # Pseudophakic corneal edema (disorder)
        "231923006",  # Aphakic corneal edema (disorder)
        "248503005",  # Fine corneal edema (disorder)
        "373428006",  # Corneal epithelial edema (finding)
        "373429003",  # Bullous corneal epithelial edema (finding)
        "373430008",  # Edema of corneal stroma (disorder)
        "336241000119105",  # Bullous keratopathy of right eye (disorder)
        "341851000119102",  # Bullous keratopathy of left eye (disorder)
        "347061000119104",  # Bullous keratopathy of bilateral eyes (disorder)
        "678991000119103",  # Pseudophakic bullous keratopathy of left eye (disorder)
        "679011000119100",  # Pseudophakic bullous keratopathy of right eye (disorder)
        "15736081000119108",  # Microcystic edema of cornea of left eye (disorder)
        "15736161000119100",  # Microcystic edema of cornea of right eye (disorder)
        "15736201000119105",  # Edema of cornea of left eye (disorder)
        "15736241000119107",  # Edema of cornea of bilateral eyes (disorder)
        "15736281000119102",  # Edema of cornea of right eye (disorder)
    }


class CornealOpacityAndOtherDisordersOfCornea(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of corneal opacity and other corneal disorders.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of opacity of the cornea, adherent or corneal leukoma, scarring of the cornea, and corneal haze.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Corneal Opacity and Other Disorders of Cornea"
    OID = "2.16.840.1.113883.3.526.3.1419"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H1701",  # Adherent leukoma, right eye
        "H1702",  # Adherent leukoma, left eye
        "H1703",  # Adherent leukoma, bilateral
        "H1711",  # Central corneal opacity, right eye
        "H1712",  # Central corneal opacity, left eye
        "H1713",  # Central corneal opacity, bilateral
        "H1789",  # Other corneal scars and opacities
        "H179",  # Unspecified corneal scar and opacity
    }
    SNOMEDCT = {
        "13282008",  # Congenital corneal opacity interfering with vision (disorder)
        "16350000",  # Corneal macula interfering with central vision (disorder)
        "22135003",  # Corneal leukoma interfering with central vision (disorder)
        "30337003",  # Adherent leukoma (disorder)
        "54359000",  # Congenital corneal opacity not interfering with vision (disorder)
        "55713007",  # Peripheral opacity of cornea (disorder)
        "56530002",  # Central opacity of cornea (disorder)
        "64634000",  # Corneal opacity (disorder)
        "66104007",  # Minor opacity of cornea (disorder)
        "72494008",  # Corneal nebula (disorder)
        "95479005",  # Congenital sclerocornea (disorder)
        "95726001",  # Corneal scar (disorder)
        "193795008",  # Corneal scars and opacities (disorder)
        "204148008",  # Congenital corneal opacity with visual deficit (disorder)
        "204149000",  # Congenital corneal opacity without visual deficit (disorder)
        "268160006",  # Congenital corneal opacity (disorder)
        "314005007",  # Visually significant corneal scar (disorder)
        "370505008",  # Corneal facet (disorder)
        "413921009",  # Corneal subepithelial haze (disorder)
        "419496009",  # Congenital corneal keloid (disorder)
        "422947009",  # Corneal subepithelial scar (finding)
        "424353002",  # Corneal haze caused by herpes zoster (disorder)
        "425961009",  # Corneal haze caused by herpes simplex (disorder)
        "336501000119105",  # Adherent leukoma of right eye (disorder)
        "342111000119101",  # Adherent leukoma of left eye (disorder)
        "347271000119100",  # Adherent leukoma of bilateral eyes (disorder)
        "1099031000119105",  # Opacity of cornea of bilateral eyes (disorder)
        "1099041000119101",  # Opacity of cornea of left eye (disorder)
        "1099051000119104",  # Opacity of cornea of right eye (disorder)
        "15999271000119108",  # Scar of cornea of right eye (disorder)
        "15999311000119108",  # Scar of cornea of bilateral eyes (disorder)
        "15999351000119109",  # Scar of cornea of left eye (disorder)
    }


class DegenerationOfMaculaAndPosteriorPole(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of degeneration of macula and posterior pole.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of macula and/or posterior pole degenerations, including age-related degenerations, macular cyst, maculopathy, macular puckering, macular hole, atrophy or dystrophy of macula.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Degeneration of Macula and Posterior Pole"
    OID = "2.16.840.1.113883.3.526.3.1453"
    DEFINITION_VERSION = "20210210"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H3530",  # Unspecified macular degeneration
        "H353110",  # Nonexudative age-related macular degeneration, right eye, stage unspecified
        "H353111",  # Nonexudative age-related macular degeneration, right eye, early dry stage
        "H353112",  # Nonexudative age-related macular degeneration, right eye, intermediate dry stage
        "H353113",  # Nonexudative age-related macular degeneration, right eye, advanced atrophic without subfoveal involvement
        "H353114",  # Nonexudative age-related macular degeneration, right eye, advanced atrophic with subfoveal involvement
        "H353120",  # Nonexudative age-related macular degeneration, left eye, stage unspecified
        "H353121",  # Nonexudative age-related macular degeneration, left eye, early dry stage
        "H353122",  # Nonexudative age-related macular degeneration, left eye, intermediate dry stage
        "H353123",  # Nonexudative age-related macular degeneration, left eye, advanced atrophic without subfoveal involvement
        "H353124",  # Nonexudative age-related macular degeneration, left eye, advanced atrophic with subfoveal involvement
        "H353130",  # Nonexudative age-related macular degeneration, bilateral, stage unspecified
        "H353131",  # Nonexudative age-related macular degeneration, bilateral, early dry stage
        "H353132",  # Nonexudative age-related macular degeneration, bilateral, intermediate dry stage
        "H353133",  # Nonexudative age-related macular degeneration, bilateral, advanced atrophic without subfoveal involvement
        "H353134",  # Nonexudative age-related macular degeneration, bilateral, advanced atrophic with subfoveal involvement
        "H353210",  # Exudative age-related macular degeneration, right eye, stage unspecified
        "H353211",  # Exudative age-related macular degeneration, right eye, with active choroidal neovascularization
        "H353212",  # Exudative age-related macular degeneration, right eye, with inactive choroidal neovascularization
        "H353213",  # Exudative age-related macular degeneration, right eye, with inactive scar
        "H353220",  # Exudative age-related macular degeneration, left eye, stage unspecified
        "H353221",  # Exudative age-related macular degeneration, left eye, with active choroidal neovascularization
        "H353222",  # Exudative age-related macular degeneration, left eye, with inactive choroidal neovascularization
        "H353223",  # Exudative age-related macular degeneration, left eye, with inactive scar
        "H353230",  # Exudative age-related macular degeneration, bilateral, stage unspecified
        "H353231",  # Exudative age-related macular degeneration, bilateral, with active choroidal neovascularization
        "H353232",  # Exudative age-related macular degeneration, bilateral, with inactive choroidal neovascularization
        "H353233",  # Exudative age-related macular degeneration, bilateral, with inactive scar
        "H35341",  # Macular cyst, hole, or pseudohole, right eye
        "H35342",  # Macular cyst, hole, or pseudohole, left eye
        "H35343",  # Macular cyst, hole, or pseudohole, bilateral
        "H35351",  # Cystoid macular degeneration, right eye
        "H35352",  # Cystoid macular degeneration, left eye
        "H35353",  # Cystoid macular degeneration, bilateral
        "H35361",  # Drusen (degenerative) of macula, right eye
        "H35362",  # Drusen (degenerative) of macula, left eye
        "H35363",  # Drusen (degenerative) of macula, bilateral
        "H35371",  # Puckering of macula, right eye
        "H35372",  # Puckering of macula, left eye
        "H35373",  # Puckering of macula, bilateral
        "H35381",  # Toxic maculopathy, right eye
        "H35382",  # Toxic maculopathy, left eye
        "H35383",  # Toxic maculopathy, bilateral
    }
    SNOMEDCT = {
        "14046000",  # Cystoid macular retinal degeneration (disorder)
        "18410006",  # Scarred macula (disorder)
        "37231002",  # Macular retinal edema (disorder)
        "44115007",  # Toxic maculopathy (disorder)
        "47673003",  # Fundus flavimaculatus (disorder)
        "70099003",  # Stargardt's disease (disorder)
        "71047006",  # Familial pseudoinflammatory macular degeneration (disorder)
        "90036004",  # Vitelliform dystrophy (disorder)
        "193350004",  # Advanced maculopathy due to diabetes mellitus (disorder)
        "193387007",  # Cystoid macular edema (disorder)
        "204179007",  # Congenital macular changes (disorder)
        "232006002",  # Macular hole (disorder)
        "232013002",  # Macular retinoschisis (disorder)
        "232020009",  # Disorder of macula due to diabetes mellitus (disorder)
        "232049001",  # Adult vitelliform macular dystrophy (disorder)
        "232050001",  # Bull's eye macular dystrophy (disorder)
        "232051002",  # Pattern dystrophy of macula (disorder)
        "232071006",  # Occult neovascularization of macula (disorder)
        "232072004",  # Forster-Fuchs' spot (disorder)
        "247146001",  # Macular pigment deposit (disorder)
        "247147005",  # Macular diffuse atrophy (disorder)
        "247149008",  # Macular choroidal sclerosis (disorder)
        "247158001",  # Macular pigment epithelial detachment (disorder)
        "247159009",  # Macular pigment epithelial rip (disorder)
        "247172005",  # Retinal macular break (disorder)
        "247200005",  # Choroidal atrophy (finding)
        "264633009",  # Drusen stage macular degeneration (disorder)
        "267611002",  # Degeneration of macula and posterior pole (disorder)
        "267612009",  # Degeneration of macula due to cyst, hole or pseudohole (disorder)
        "267718000",  # Age-related macular degeneration (disorder)
        "276436007",  # Hereditary macular dystrophy (disorder)
        "312701008",  # Impending macular hole (disorder)
        "312702001",  # Full thickness macular hole stage II (disorder)
        "312703006",  # Full thickness macular hole stage III (disorder)
        "312704000",  # Full thickness macular hole stage IV (disorder)
        "312898002",  # Myopic macular degeneration (disorder)
        "312899005",  # Fibrovascular macular scar (disorder)
        "312901001",  # Vitreomacular traction syndrome (disorder)
        "312902008",  # Macular pseudohole (disorder)
        "312911008",  # Clinically significant macular edema (disorder)
        "312912001",  # Macular edema due to diabetes mellitus (disorder)
        "312913006",  # Ischemic maculopathy (disorder)
        "312920004",  # Postoperative cystoid macular edema (disorder)
        "312921000",  # Autosomal dominant cystoid macular edema (disorder)
        "312922007",  # Uveitis related cystoid macular edema (disorder)
        "312925009",  # North Carolina macular dystrophy (disorder)
        "312999006",  # Disorder of macula of retina (disorder)
        "313002004",  # Acute macular neuroretinitis (disorder)
        "314010006",  # Diffuse exudative maculopathy due to diabetes mellitus (disorder)
        "314011005",  # Focal exudative maculopathy due to diabetes mellitus (disorder)
        "314014002",  # Ischemic maculopathy due to diabetes mellitus (disorder)
        "314015001",  # Mixed maculopathy due to diabetes mellitus (disorder)
        "314427006",  # Drusen plus pigment change stage macular degeneration (disorder)
        "371098002",  # Post-radiation maculopathy (disorder)
        "399864000",  # Macular edema not clinically significant due to diabetes mellitus (disorder)
        "399872003",  # Severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        "399874002",  # High risk proliferative retinopathy with clinically significant macula edema due to diabetes mellitus (disorder)
        "399875001",  # Non-high-risk proliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        "399877009",  # Very severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        "414173003",  # Exudative age-related macular degeneration (disorder)
        "414875008",  # Nonexudative age-related macular degeneration (disorder)
        "418918009",  # Macular infarction (disorder)
        "420030000",  # Serous macular detachment (disorder)
        "420486006",  # Exudative maculopathy due to type 1 diabetes mellitus (disorder)
        "421779007",  # Exudative maculopathy due to type 2 diabetes mellitus (disorder)
        "422338006",  # Degenerative disorder of macula (disorder)
        "423084001",  # Macular exudate (disorder)
        "425622004",  # Scar of macula following trauma (disorder)
        "429449002",  # Congenital hypoplasia of fovea centralis (disorder)
        "432789001",  # Noncystoid edema of macula of retina (disorder)
        "443437004",  # Full thickness hole of macula lutea (disorder)
        "789220007",  # Atrophy of macula lutea (disorder)
        "335531000119109",  # Cystoid macular degeneration of right retina (disorder)
        "341141000119106",  # Cystoid macular degeneration of left retina (disorder)
        "346401000119108",  # Cystoid macular degeneration of bilateral retinas (disorder)
    }


class DegenerativeDisordersOfGlobe(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of degenerative disorders of globe.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of degenerative disorders of globe, including degenerative myopia, chalcosis, siderosis, and other degenerative diagnoses.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Degenerative Disorders of Globe"
    OID = "2.16.840.1.113883.3.526.3.1454"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4421",  # Degenerative myopia, right eye
        "H4422",  # Degenerative myopia, left eye
        "H4423",  # Degenerative myopia, bilateral
        "H442A1",  # Degenerative myopia with choroidal neovascularization, right eye
        "H442A2",  # Degenerative myopia with choroidal neovascularization, left eye
        "H442A3",  # Degenerative myopia with choroidal neovascularization, bilateral eye
        "H442B1",  # Degenerative myopia with macular hole, right eye
        "H442B2",  # Degenerative myopia with macular hole, left eye
        "H442B3",  # Degenerative myopia with macular hole, bilateral eye
        "H442C1",  # Degenerative myopia with retinal detachment, right eye
        "H442C2",  # Degenerative myopia with retinal detachment, left eye
        "H442C3",  # Degenerative myopia with retinal detachment, bilateral eye
        "H442D1",  # Degenerative myopia with foveoschisis, right eye
        "H442D2",  # Degenerative myopia with foveoschisis, left eye
        "H442D3",  # Degenerative myopia with foveoschisis, bilateral eye
        "H442E1",  # Degenerative myopia with other maculopathy, right eye
        "H442E2",  # Degenerative myopia with other maculopathy, left eye
        "H442E3",  # Degenerative myopia with other maculopathy, bilateral eye
        "H44311",  # Chalcosis, right eye
        "H44312",  # Chalcosis, left eye
        "H44313",  # Chalcosis, bilateral
        "H44321",  # Siderosis of eye, right eye
        "H44322",  # Siderosis of eye, left eye
        "H44323",  # Siderosis of eye, bilateral
        "H44391",  # Other degenerative disorders of globe, right eye
        "H44392",  # Other degenerative disorders of globe, left eye
        "H44393",  # Other degenerative disorders of globe, bilateral
    }
    SNOMEDCT = {
        "264008",  # Blind hypertensive eye (disorder)
        "3577000",  # Retinal lattice degeneration (disorder)
        "14046000",  # Cystoid macular retinal degeneration (disorder)
        "25277000",  # Siderosis of eye (disorder)
        "35545005",  # Secondary pigmentary retinal degeneration (disorder)
        "37075008",  # Microcystoid retinal degeneration (disorder)
        "40787005",  # Conjunctival degeneration (disorder)
        "54184008",  # Senile reticular retinal degeneration (disorder)
        "60189009",  # Vitreous degeneration (disorder)
        "61536007",  # Peripheral retinal degeneration (disorder)
        "62585004",  # Degenerative disorder of eye (disorder)
        "69134001",  # Paving stone retinal degeneration (disorder)
        "71047006",  # Familial pseudoinflammatory macular degeneration (disorder)
        "95695004",  # Degeneration of retina (disorder)
        "95696003",  # Glaucomatous retinal degeneration (disorder)
        "95697007",  # Generalized retinal degeneration (disorder)
        "95699005",  # Multifocal retinal degeneration (disorder)
        "95700006",  # Sudden acquired retinal degeneration (disorder)
        "111521006",  # Corneal degeneration (disorder)
        "231918006",  # Chalcosis of cornea (disorder)
        "231919003",  # Siderosis of cornea (disorder)
        "232001007",  # Snail-track retinal degeneration (disorder)
        "232002000",  # Snowflake retinal degeneration (disorder)
        "246877005",  # Bulbar conjunctival degeneration (disorder)
        "247177004",  # Chorioretinal degeneration (disorder)
        "247182006",  # Vitreoretinal degeneration (disorder)
        "247183001",  # Peripheral snowflake retinal degeneration (disorder)
        "264633009",  # Drusen stage macular degeneration (disorder)
        "267611002",  # Degeneration of macula and posterior pole (disorder)
        "267612009",  # Degeneration of macula due to cyst, hole or pseudohole (disorder)
        "267718000",  # Age-related macular degeneration (disorder)
        "314427006",  # Drusen plus pigment change stage macular degeneration (disorder)
        "414173003",  # Exudative age-related macular degeneration (disorder)
        "414875008",  # Nonexudative age-related macular degeneration (disorder)
        "417939004",  # Amyloid corneal degeneration (disorder)
        "418871005",  # Spheroidal degeneration of conjunctiva (disorder)
        "428212003",  # Photoreceptor degeneration (disorder)
    }


class DiabeticMacularEdema(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of diabetic macular edema.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of diabetic macular edema or diabetic retinopathy with macular edema.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Diabetic Macular Edema"
    OID = "2.16.840.1.113883.3.526.3.1455"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "E08311",  # Diabetes mellitus due to underlying condition with unspecified diabetic retinopathy with macular edema
        "E083211",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E083212",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E083213",  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E083311",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E083312",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E083313",  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E083411",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E083412",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E083413",  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E083511",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, right eye
        "E083512",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, left eye
        "E083513",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, bilateral
        "E083521",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E083522",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E083523",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E083531",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E083532",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E083533",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E083541",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E083542",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E083543",  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E083551",  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, right eye
        "E083552",  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, left eye
        "E083553",  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, bilateral
        "E0837X1",  # Diabetes mellitus due to underlying condition with diabetic macular edema, resolved following treatment, right eye
        "E0837X2",  # Diabetes mellitus due to underlying condition with diabetic macular edema, resolved following treatment, left eye
        "E0837X3",  # Diabetes mellitus due to underlying condition with diabetic macular edema, resolved following treatment, bilateral
        "E09311",  # Drug or chemical induced diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E093211",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E093212",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E093213",  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E093311",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E093312",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E093313",  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E093411",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E093412",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E093413",  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E093511",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E093512",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E093513",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E093521",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E093522",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E093523",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E093531",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E093532",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E093533",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E093541",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E093542",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E093543",  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E093551",  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E093552",  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E093553",  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E0937X1",  # Drug or chemical induced diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        "E0937X2",  # Drug or chemical induced diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        "E0937X3",  # Drug or chemical induced diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        "E10311",  # Type 1 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E103211",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E103212",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E103213",  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103311",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E103312",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E103313",  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103411",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E103412",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E103413",  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E103511",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E103512",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E103513",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E103521",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E103522",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E103523",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E103531",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E103532",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E103533",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E103541",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E103542",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E103543",  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E103551",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E103552",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E103553",  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E1037X1",  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        "E1037X2",  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        "E1037X3",  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        "E11311",  # Type 2 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E113211",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E113212",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E113213",  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113311",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E113312",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E113313",  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113411",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E113412",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E113413",  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E113511",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E113512",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E113513",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E113521",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E113522",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E113523",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E113531",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E113532",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E113533",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E113541",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E113542",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E113543",  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E113551",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E113552",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E113553",  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E1137X1",  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        "E1137X2",  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        "E1137X3",  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        "E13311",  # Other specified diabetes mellitus with unspecified diabetic retinopathy with macular edema
        "E133211",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        "E133212",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        "E133213",  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133311",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        "E133312",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        "E133313",  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133411",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        "E133412",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        "E133413",  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        "E133511",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        "E133512",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        "E133513",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        "E133521",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        "E133522",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        "E133523",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        "E133531",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        "E133532",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        "E133533",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        "E133541",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        "E133542",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        "E133543",  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        "E133551",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        "E133552",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        "E133553",  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        "E1337X1",  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        "E1337X2",  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        "E1337X3",  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
    }
    SNOMEDCT = {
        "312912001",  # Macular edema due to diabetes mellitus (disorder)
        "399864000",  # Macular edema not clinically significant due to diabetes mellitus (disorder)
        "769219006",  # Macular edema due to type 1 diabetes mellitus (disorder)
        "769220000",  # Macular edema due to type 2 diabetes mellitus (disorder)
    }


class DisordersOfOpticChiasm(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of disorders of optic chiasm.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of optic chiasm disorders or injuries or optic glioma.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Disorders of Optic Chiasm"
    OID = "2.16.840.1.113883.3.526.3.1457"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4741",  # Disorders of optic chiasm in (due to) inflammatory disorders
        "H4742",  # Disorders of optic chiasm in (due to) neoplasm
        "H4743",  # Disorders of optic chiasm in (due to) vascular disorders
        "H4749",  # Disorders of optic chiasm in (due to) other disorders
    }
    SNOMEDCT = {
        "64246009",  # Disorder of optic chiasm associated with non-pituitary neoplasm (disorder)
        "69820004",  # Injury of optic chiasm (disorder)
        "70476006",  # Optic chiasm disorder (disorder)
        "89654006",  # Disorder of optic chiasm associated with vascular disorder (disorder)
        "194057002",  # Disorder of optic chiasm due to pituitary disorder (disorder)
        "194058007",  # Disorder of optic chiasm due to non-pituitary neoplasm (disorder)
        "194059004",  # Disorder of optic chiasm due to vascular disorder (disorder)
        "194060009",  # Inflammatory disorder of optic chiasm (disorder)
        "230517004",  # Compression of optic chiasm (disorder)
        "302902003",  # Infarction of optic chiasm (disorder)
        "404653000",  # Chiasmal glioma (disorder)
        "404664002",  # Malignant optic glioma (disorder)
        "431769004",  # Disorder of optic chiasm due to neoplasm (disorder)
        "703429003",  # Malignant optic glioma of adulthood (disorder)
    }


class DisordersOfVisualCortex(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of disorders of visual cortex.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a disorder of visual cortex, including inflammatory disorders, neoplasm, and cortical blindness.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Disorders of Visual Cortex"
    OID = "2.16.840.1.113883.3.526.3.1458"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H47611",  # Cortical blindness, right side of brain
        "H47612",  # Cortical blindness, left side of brain
        "H47621",  # Disorders of visual cortex in (due to) inflammatory disorders, right side of brain
        "H47622",  # Disorders of visual cortex in (due to) inflammatory disorders, left side of brain
        "H47631",  # Disorders of visual cortex in (due to) neoplasm, right side of brain
        "H47632",  # Disorders of visual cortex in (due to) neoplasm, left side of brain
        "H47641",  # Disorders of visual cortex in (due to) vascular disorders, right side of brain
        "H47642",  # Disorders of visual cortex in (due to) vascular disorders, left side of brain
    }
    SNOMEDCT = {
        "68574006",  # Cortical blindness (disorder)
        "128329001",  # Disorder of visual cortex (disorder)
        "194068002",  # Visual cortex disorder due to neoplasm (disorder)
        "194069005",  # Visual cortex disorder due to vascular disorder (disorder)
        "194070006",  # Inflammatory disorder of visual cortex (disorder)
        "230526001",  # Compression of visual cortex (disorder)
        "302904002",  # Infarction of visual cortex (disorder)
        "732251003",  # Cortical blindness, intellectual disability, polydactyly syndrome (disorder)
        "342741000119103",  # Cortical blindness of right side of brain (disorder)
        "342751000119101",  # Cortical blindness of left side of brain (disorder)
    }


class DisseminatedChorioretinitisAndDisseminatedRetinochoroiditis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of disseminated chorioretinitis and/or retinochoroiditis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of disseminated chorioretinal inflammation, disseminated retinochoroiditis, and placoid pigment epitheliopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Disseminated Chorioretinitis and Disseminated Retinochoroiditis"
    OID = "2.16.840.1.113883.3.526.3.1459"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "A1853",  # Tuberculous chorioretinitis
        "H30101",  # Unspecified disseminated chorioretinal inflammation, right eye
        "H30102",  # Unspecified disseminated chorioretinal inflammation, left eye
        "H30103",  # Unspecified disseminated chorioretinal inflammation, bilateral
        "H30111",  # Disseminated chorioretinal inflammation of posterior pole, right eye
        "H30112",  # Disseminated chorioretinal inflammation of posterior pole, left eye
        "H30113",  # Disseminated chorioretinal inflammation of posterior pole, bilateral
        "H30121",  # Disseminated chorioretinal inflammation, peripheral right eye
        "H30122",  # Disseminated chorioretinal inflammation, peripheral, left eye
        "H30123",  # Disseminated chorioretinal inflammation, peripheral, bilateral
        "H30131",  # Disseminated chorioretinal inflammation, generalized, right eye
        "H30132",  # Disseminated chorioretinal inflammation, generalized, left eye
        "H30133",  # Disseminated chorioretinal inflammation, generalized, bilateral
        "H30141",  # Acute posterior multifocal placoid pigment epitheliopathy, right eye
        "H30142",  # Acute posterior multifocal placoid pigment epitheliopathy, left eye
        "H30143",  # Acute posterior multifocal placoid pigment epitheliopathy, bilateral
    }
    SNOMEDCT = {
        "14894002",  # Generalized disseminated choroiditis AND chorioretinitis (disorder)
        "48225000",  # Tuberculous disseminated chorioretinitis (disorder)
        "51579006",  # Disseminated choroiditis AND chorioretinitis, posterior pole (disorder)
        "61517002",  # Metastatic disseminated retinitis AND retinochoroiditis (disorder)
        "69811008",  # Peripheral disseminated choroiditis AND chorioretinitis (disorder)
        "77939001",  # Syphilitic disseminated retinochoroiditis (disorder)
        "78769001",  # Disseminated chorioretinitis (disorder)
        "193442009",  # Disseminated posterior pole chorioretinitis (disorder)
        "193443004",  # Disseminated peripheral chorioretinitis (disorder)
        "193444005",  # General disseminated chorioretinitis (disorder)
        "371099005",  # Generalized disseminated choroiditis AND chorioretinitis with acute necrosis (disorder)
    }


class FocalChorioretinitisAndFocalRetinochoroiditis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of focal chorioretinitis and/or retinochoroiditis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of focal chorioretinitis, choroiditis, retinochoroiditis and retinitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Focal Chorioretinitis and Focal Retinochoroiditis"
    OID = "2.16.840.1.113883.3.526.3.1460"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H30001",  # Unspecified focal chorioretinal inflammation, right eye
        "H30002",  # Unspecified focal chorioretinal inflammation, left eye
        "H30003",  # Unspecified focal chorioretinal inflammation, bilateral
        "H30011",  # Focal chorioretinal inflammation, juxtapapillary, right eye
        "H30012",  # Focal chorioretinal inflammation, juxtapapillary, left eye
        "H30013",  # Focal chorioretinal inflammation, juxtapapillary, bilateral
        "H30021",  # Focal chorioretinal inflammation of posterior pole, right eye
        "H30022",  # Focal chorioretinal inflammation of posterior pole, left eye
        "H30023",  # Focal chorioretinal inflammation of posterior pole, bilateral
        "H30031",  # Focal chorioretinal inflammation, peripheral, right eye
        "H30032",  # Focal chorioretinal inflammation, peripheral, left eye
        "H30033",  # Focal chorioretinal inflammation, peripheral, bilateral
        "H30041",  # Focal chorioretinal inflammation, macular or paramacular, right eye
        "H30042",  # Focal chorioretinal inflammation, macular or paramacular, left eye
        "H30043",  # Focal chorioretinal inflammation, macular or paramacular, bilateral
    }
    SNOMEDCT = {
        "6313001",  # Juxtapapillary focal choroiditis AND chorioretinitis (disorder)
        "15847003",  # Focal chorioretinitis (disorder)
        "36535003",  # Peripheral focal retinitis AND retinochoroiditis (disorder)
        "42031001",  # Macular focal retinitis AND retinochoroiditis (disorder)
        "44674000",  # Focal choroiditis AND chorioretinitis of other posterior pole (disorder)
        "46497009",  # Juxtapapillary focal retinitis AND retinochoroiditis (disorder)
        "56787009",  # Peripheral focal choroiditis AND chorioretinitis (disorder)
        "88290000",  # Focal chorioretinitis caused by acquired toxoplasmosis (disorder)
        "193434008",  # Focal juxtapapillary choroiditis (disorder)
        "193436005",  # Juxtapapillary focal chorioretinitis (disorder)
        "193438006",  # Peripheral focal retinochoroiditis (disorder)
    }


class Glaucoma(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for glaucoma.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** This value set may use a model element related to Diagnosis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Glaucoma"
    OID = "2.16.840.1.113883.3.526.3.1423"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4010X0",  # Unspecified open-angle glaucoma, stage unspecified
        "H4010X1",  # Unspecified open-angle glaucoma, mild stage
        "H4010X2",  # Unspecified open-angle glaucoma, moderate stage
        "H4010X3",  # Unspecified open-angle glaucoma, severe stage
        "H4010X4",  # Unspecified open-angle glaucoma, indeterminate stage
        "H401110",  # Primary open-angle glaucoma, right eye, stage unspecified
        "H401111",  # Primary open-angle glaucoma, right eye, mild stage
        "H401112",  # Primary open-angle glaucoma, right eye, moderate stage
        "H401113",  # Primary open-angle glaucoma, right eye, severe stage
        "H401114",  # Primary open-angle glaucoma, right eye, indeterminate stage
        "H401120",  # Primary open-angle glaucoma, left eye, stage unspecified
        "H401121",  # Primary open-angle glaucoma, left eye, mild stage
        "H401122",  # Primary open-angle glaucoma, left eye, moderate stage
        "H401123",  # Primary open-angle glaucoma, left eye, severe stage
        "H401124",  # Primary open-angle glaucoma, left eye, indeterminate stage
        "H401130",  # Primary open-angle glaucoma, bilateral, stage unspecified
        "H401131",  # Primary open-angle glaucoma, bilateral, mild stage
        "H401132",  # Primary open-angle glaucoma, bilateral, moderate stage
        "H401133",  # Primary open-angle glaucoma, bilateral, severe stage
        "H401134",  # Primary open-angle glaucoma, bilateral, indeterminate stage
        "H401210",  # Low-tension glaucoma, right eye, stage unspecified
        "H401211",  # Low-tension glaucoma, right eye, mild stage
        "H401212",  # Low-tension glaucoma, right eye, moderate stage
        "H401213",  # Low-tension glaucoma, right eye, severe stage
        "H401214",  # Low-tension glaucoma, right eye, indeterminate stage
        "H401220",  # Low-tension glaucoma, left eye, stage unspecified
        "H401221",  # Low-tension glaucoma, left eye, mild stage
        "H401222",  # Low-tension glaucoma, left eye, moderate stage
        "H401223",  # Low-tension glaucoma, left eye, severe stage
        "H401224",  # Low-tension glaucoma, left eye, indeterminate stage
        "H401230",  # Low-tension glaucoma, bilateral, stage unspecified
        "H401231",  # Low-tension glaucoma, bilateral, mild stage
        "H401232",  # Low-tension glaucoma, bilateral, moderate stage
        "H401233",  # Low-tension glaucoma, bilateral, severe stage
        "H401234",  # Low-tension glaucoma, bilateral, indeterminate stage
        "H401310",  # Pigmentary glaucoma, right eye, stage unspecified
        "H401311",  # Pigmentary glaucoma, right eye, mild stage
        "H401312",  # Pigmentary glaucoma, right eye, moderate stage
        "H401313",  # Pigmentary glaucoma, right eye, severe stage
        "H401314",  # Pigmentary glaucoma, right eye, indeterminate stage
        "H401320",  # Pigmentary glaucoma, left eye, stage unspecified
        "H401321",  # Pigmentary glaucoma, left eye, mild stage
        "H401322",  # Pigmentary glaucoma, left eye, moderate stage
        "H401323",  # Pigmentary glaucoma, left eye, severe stage
        "H401324",  # Pigmentary glaucoma, left eye, indeterminate stage
        "H401330",  # Pigmentary glaucoma, bilateral, stage unspecified
        "H401331",  # Pigmentary glaucoma, bilateral, mild stage
        "H401332",  # Pigmentary glaucoma, bilateral, moderate stage
        "H401333",  # Pigmentary glaucoma, bilateral, severe stage
        "H401334",  # Pigmentary glaucoma, bilateral, indeterminate stage
        "H401410",  # Capsular glaucoma with pseudoexfoliation of lens, right eye, stage unspecified
        "H401411",  # Capsular glaucoma with pseudoexfoliation of lens, right eye, mild stage
        "H401412",  # Capsular glaucoma with pseudoexfoliation of lens, right eye, moderate stage
        "H401413",  # Capsular glaucoma with pseudoexfoliation of lens, right eye, severe stage
        "H401414",  # Capsular glaucoma with pseudoexfoliation of lens, right eye, indeterminate stage
        "H401420",  # Capsular glaucoma with pseudoexfoliation of lens, left eye, stage unspecified
        "H401421",  # Capsular glaucoma with pseudoexfoliation of lens, left eye, mild stage
        "H401422",  # Capsular glaucoma with pseudoexfoliation of lens, left eye, moderate stage
        "H401423",  # Capsular glaucoma with pseudoexfoliation of lens, left eye, severe stage
        "H401424",  # Capsular glaucoma with pseudoexfoliation of lens, left eye, indeterminate stage
        "H401430",  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, stage unspecified
        "H401431",  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, mild stage
        "H401432",  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, moderate stage
        "H401433",  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, severe stage
        "H401434",  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, indeterminate stage
        "H40151",  # Residual stage of open-angle glaucoma, right eye
        "H40152",  # Residual stage of open-angle glaucoma, left eye
        "H40153",  # Residual stage of open-angle glaucoma, bilateral
        "H4020X0",  # Unspecified primary angle-closure glaucoma, stage unspecified
        "H4020X1",  # Unspecified primary angle-closure glaucoma, mild stage
        "H4020X2",  # Unspecified primary angle-closure glaucoma, moderate stage
        "H4020X3",  # Unspecified primary angle-closure glaucoma, severe stage
        "H4020X4",  # Unspecified primary angle-closure glaucoma, indeterminate stage
        "H40211",  # Acute angle-closure glaucoma, right eye
        "H40212",  # Acute angle-closure glaucoma, left eye
        "H40213",  # Acute angle-closure glaucoma, bilateral
        "H402210",  # Chronic angle-closure glaucoma, right eye, stage unspecified
        "H402211",  # Chronic angle-closure glaucoma, right eye, mild stage
        "H402212",  # Chronic angle-closure glaucoma, right eye, moderate stage
        "H402213",  # Chronic angle-closure glaucoma, right eye, severe stage
        "H402214",  # Chronic angle-closure glaucoma, right eye, indeterminate stage
        "H402220",  # Chronic angle-closure glaucoma, left eye, stage unspecified
        "H402221",  # Chronic angle-closure glaucoma, left eye, mild stage
        "H402222",  # Chronic angle-closure glaucoma, left eye, moderate stage
        "H402223",  # Chronic angle-closure glaucoma, left eye, severe stage
        "H402224",  # Chronic angle-closure glaucoma, left eye, indeterminate stage
        "H402230",  # Chronic angle-closure glaucoma, bilateral, stage unspecified
        "H402231",  # Chronic angle-closure glaucoma, bilateral, mild stage
        "H402232",  # Chronic angle-closure glaucoma, bilateral, moderate stage
        "H402233",  # Chronic angle-closure glaucoma, bilateral, severe stage
        "H402234",  # Chronic angle-closure glaucoma, bilateral, indeterminate stage
        "H40231",  # Intermittent angle-closure glaucoma, right eye
        "H40232",  # Intermittent angle-closure glaucoma, left eye
        "H40233",  # Intermittent angle-closure glaucoma, bilateral
        "H40241",  # Residual stage of angle-closure glaucoma, right eye
        "H40242",  # Residual stage of angle-closure glaucoma, left eye
        "H40243",  # Residual stage of angle-closure glaucoma, bilateral
        "H4031X0",  # Glaucoma secondary to eye trauma, right eye, stage unspecified
        "H4031X1",  # Glaucoma secondary to eye trauma, right eye, mild stage
        "H4031X2",  # Glaucoma secondary to eye trauma, right eye, moderate stage
        "H4031X3",  # Glaucoma secondary to eye trauma, right eye, severe stage
        "H4031X4",  # Glaucoma secondary to eye trauma, right eye, indeterminate stage
        "H4032X0",  # Glaucoma secondary to eye trauma, left eye, stage unspecified
        "H4032X1",  # Glaucoma secondary to eye trauma, left eye, mild stage
        "H4032X2",  # Glaucoma secondary to eye trauma, left eye, moderate stage
        "H4032X3",  # Glaucoma secondary to eye trauma, left eye, severe stage
        "H4032X4",  # Glaucoma secondary to eye trauma, left eye, indeterminate stage
        "H4033X0",  # Glaucoma secondary to eye trauma, bilateral, stage unspecified
        "H4033X1",  # Glaucoma secondary to eye trauma, bilateral, mild stage
        "H4033X2",  # Glaucoma secondary to eye trauma, bilateral, moderate stage
        "H4033X3",  # Glaucoma secondary to eye trauma, bilateral, severe stage
        "H4033X4",  # Glaucoma secondary to eye trauma, bilateral, indeterminate stage
        "H4041X0",  # Glaucoma secondary to eye inflammation, right eye, stage unspecified
        "H4041X1",  # Glaucoma secondary to eye inflammation, right eye, mild stage
        "H4041X2",  # Glaucoma secondary to eye inflammation, right eye, moderate stage
        "H4041X3",  # Glaucoma secondary to eye inflammation, right eye, severe stage
        "H4041X4",  # Glaucoma secondary to eye inflammation, right eye, indeterminate stage
        "H4042X0",  # Glaucoma secondary to eye inflammation, left eye, stage unspecified
        "H4042X1",  # Glaucoma secondary to eye inflammation, left eye, mild stage
        "H4042X2",  # Glaucoma secondary to eye inflammation, left eye, moderate stage
        "H4042X3",  # Glaucoma secondary to eye inflammation, left eye, severe stage
        "H4042X4",  # Glaucoma secondary to eye inflammation, left eye, indeterminate stage
        "H4043X0",  # Glaucoma secondary to eye inflammation, bilateral, stage unspecified
        "H4043X1",  # Glaucoma secondary to eye inflammation, bilateral, mild stage
        "H4043X2",  # Glaucoma secondary to eye inflammation, bilateral, moderate stage
        "H4043X3",  # Glaucoma secondary to eye inflammation, bilateral, severe stage
        "H4043X4",  # Glaucoma secondary to eye inflammation, bilateral, indeterminate stage
        "H4051X0",  # Glaucoma secondary to other eye disorders, right eye, stage unspecified
        "H4051X1",  # Glaucoma secondary to other eye disorders, right eye, mild stage
        "H4051X2",  # Glaucoma secondary to other eye disorders, right eye, moderate stage
        "H4051X3",  # Glaucoma secondary to other eye disorders, right eye, severe stage
        "H4051X4",  # Glaucoma secondary to other eye disorders, right eye, indeterminate stage
        "H4052X0",  # Glaucoma secondary to other eye disorders, left eye, stage unspecified
        "H4052X1",  # Glaucoma secondary to other eye disorders, left eye, mild stage
        "H4052X2",  # Glaucoma secondary to other eye disorders, left eye, moderate stage
        "H4052X3",  # Glaucoma secondary to other eye disorders, left eye, severe stage
        "H4052X4",  # Glaucoma secondary to other eye disorders, left eye, indeterminate stage
        "H4053X0",  # Glaucoma secondary to other eye disorders, bilateral, stage unspecified
        "H4053X1",  # Glaucoma secondary to other eye disorders, bilateral, mild stage
        "H4053X2",  # Glaucoma secondary to other eye disorders, bilateral, moderate stage
        "H4053X3",  # Glaucoma secondary to other eye disorders, bilateral, severe stage
        "H4053X4",  # Glaucoma secondary to other eye disorders, bilateral, indeterminate stage
        "H4061X0",  # Glaucoma secondary to drugs, right eye, stage unspecified
        "H4061X1",  # Glaucoma secondary to drugs, right eye, mild stage
        "H4061X2",  # Glaucoma secondary to drugs, right eye, moderate stage
        "H4061X3",  # Glaucoma secondary to drugs, right eye, severe stage
        "H4061X4",  # Glaucoma secondary to drugs, right eye, indeterminate stage
        "H4062X0",  # Glaucoma secondary to drugs, left eye, stage unspecified
        "H4062X1",  # Glaucoma secondary to drugs, left eye, mild stage
        "H4062X2",  # Glaucoma secondary to drugs, left eye, moderate stage
        "H4062X3",  # Glaucoma secondary to drugs, left eye, severe stage
        "H4062X4",  # Glaucoma secondary to drugs, left eye, indeterminate stage
        "H4063X0",  # Glaucoma secondary to drugs, bilateral, stage unspecified
        "H4063X1",  # Glaucoma secondary to drugs, bilateral, mild stage
        "H4063X2",  # Glaucoma secondary to drugs, bilateral, moderate stage
        "H4063X3",  # Glaucoma secondary to drugs, bilateral, severe stage
        "H4063X4",  # Glaucoma secondary to drugs, bilateral, indeterminate stage
        "H40811",  # Glaucoma with increased episcleral venous pressure, right eye
        "H40812",  # Glaucoma with increased episcleral venous pressure, left eye
        "H40813",  # Glaucoma with increased episcleral venous pressure, bilateral
        "H40821",  # Hypersecretion glaucoma, right eye
        "H40822",  # Hypersecretion glaucoma, left eye
        "H40823",  # Hypersecretion glaucoma, bilateral
        "H40831",  # Aqueous misdirection, right eye
        "H40832",  # Aqueous misdirection, left eye
        "H40833",  # Aqueous misdirection, bilateral
        "H4089",  # Other specified glaucoma
        "Q150",  # Congenital glaucoma
    }
    SNOMEDCT = {
        "1654001",  # Corticosteroid-induced glaucoma (disorder)
        "15374009",  # Secondary glaucoma due to aphakia (disorder)
        "19144002",  # Absolute glaucoma (disorder)
        "19309007",  # Glaucoma associated with vascular disorder (disorder)
        "21571006",  # Secondary angle-closure glaucoma (disorder)
        "21928008",  # Secondary open-angle glaucoma (disorder)
        "23986001",  # Glaucoma (disorder)
        "27735002",  # Glaucoma associated with tumors AND/OR cysts (disorder)
        "29369005",  # Hypersecretion glaucoma (disorder)
        "30041005",  # Acute angle-closure glaucoma (disorder)
        "32893002",  # Phacolytic glaucoma (disorder)
        "33647009",  # Anatomical narrow angle glaucoma (disorder)
        "34623005",  # Glaucoma with increased episcleral venous pressure (disorder)
        "37155002",  # Glaucoma associated with ocular inflammation (disorder)
        "45623002",  # Glaucoma associated with anterior segment anomaly (disorder)
        "46168003",  # Pigmentary glaucoma (disorder)
        "50485007",  # Low tension glaucoma (disorder)
        "53667005",  # Glaucoma associated with systemic syndromes (disorder)
        "65460003",  # Intermittent angle-closure glaucoma (disorder)
        "66725002",  # Glaucoma due to perforating injury (disorder)
        "66747002",  # Glaucoma associated with ocular disorder (disorder)
        "68241007",  # Glaucoma associated with ocular trauma (disorder)
        "77075001",  # Primary open angle glaucoma (disorder)
        "84333006",  # Glaucoma associated with lens disorder (disorder)
        "84494001",  # Open-angle glaucoma (disorder)
        "89215000",  # Postoperative angle-closure glaucoma (disorder)
        "92829008",  # Glaucoma due to combination of mechanisms (disorder)
        "95213001",  # Primary glaucoma due to combination of mechanisms (disorder)
        "95250000",  # Secondary glaucoma due to combination mechanisms (disorder)
        "95717004",  # Secondary glaucoma (disorder)
        "111513000",  # Advanced open-angle glaucoma (disorder)
        "111514006",  # Pseudoexfoliation glaucoma (disorder)
        "193548006",  # Steroid-induced glaucoma glaucomatous stage (disorder)
        "193552006",  # Glaucoma due to chamber angle anomaly (disorder)
        "193553001",  # Glaucoma due to iris anomaly (disorder)
        "193555008",  # Glaucoma due to systemic syndrome (disorder)
        "193561006",  # Secondary angle-closure glaucoma with pupil block (disorder)
        "193562004",  # Glaucoma due to ocular vascular disorder (disorder)
        "204113001",  # Congenital glaucoma (disorder)
        "206248004",  # Traumatic glaucoma due to birth trauma (disorder)
        "232081005",  # Iatrogenic glaucoma (disorder)
        "232082003",  # Iatrogenic angle-closure glaucoma (disorder)
        "232083008",  # Glaucoma and corneal anomaly (disorder)
        "232086000",  # Neovascular glaucoma (disorder)
        "232087009",  # Glaucoma with intraocular hemorrhage (disorder)
        "232088004",  # Ghost cell glaucoma (disorder)
        "232090003",  # Glaucoma following surgery (disorder)
        "267625001",  # Glaucoma due to ocular tumor or cyst (disorder)
        "275477002",  # Glaucoma due to ocular cyst (disorder)
        "314017009",  # Acute-on-chronic glaucoma (disorder)
        "314033007",  # Uveitic glaucoma (disorder)
        "314784002",  # Secondary angle-closure glaucoma - synechial (disorder)
        "370504007",  # Aqueous humor misdirect (disorder)
        "392030001",  # Hemolytic glaucoma (disorder)
        "392288006",  # Primary angle-closure glaucoma (disorder)
        "392291006",  # Angle-closure glaucoma (disorder)
        "392300000",  # Phacomorphic glaucoma (disorder)
        "392352004",  # Angle recession glaucoma (disorder)
        "404634005",  # Glaucoma associated with iridocorneal endothelial syndrome (disorder)
        "404648005",  # Lens particle glaucoma (disorder)
        "415176004",  # Primary congenital glaucoma (disorder)
        "444863008",  # Anatomical narrow angle glaucoma with borderline intraocular pressure (disorder)
        "698840003",  # Neovascular glaucoma due to hyphema (disorder)
        "713457002",  # Neovascular glaucoma due to diabetes mellitus (disorder)
        "715144004",  # Glaucoma caused by silicone oil (disorder)
        "716166002",  # Microcornea with glaucoma and absent frontal sinus syndrome (disorder)
        "721898008",  # Corticosteroid-induced glaucoma of bilateral eyes (disorder)
        "722321001",  # Corticosteroid-induced glaucoma of left eye (disorder)
        "722329004",  # Corticosteroid-induced glaucoma of right eye (disorder)
        "787051000",  # Open-angle glaucoma of left eye (disorder)
        "787052007",  # Open-angle glaucoma of right eye (disorder)
        "24151000119106",  # Steroid-induced open angle glaucoma (disorder)
        "41911000119107",  # Glaucoma due to type 2 diabetes mellitus (disorder)
        "60981000119103",  # Glaucoma due to diabetes mellitus type 1 (disorder)
        "336611000119109",  # Acute angle-closure glaucoma of right eye (disorder)
        "336631000119104",  # Absolute glaucoma right eye (disorder)
        "342221000119104",  # Acute angle-closure glaucoma of left eye (disorder)
        "342241000119105",  # Absolute glaucoma left eye (disorder)
        "347381000119106",  # Bilateral acute angle-closure glaucoma (disorder)
        "347401000119106",  # Bilateral absolute glaucoma (disorder)
        "12239301000119102",  # Bilateral open-angle glaucoma (disorder)
        "12239421000119101",  # Bilateral glaucoma (disorder)
        "12239461000119106",  # Glaucoma of left eye (disorder)
        "12239501000119106",  # Glaucoma of right eye (disorder)
        "15633281000119103",  # Bilateral primary open angle glaucoma (disorder)
        "15633321000119108",  # Primary open angle glaucoma of left eye (disorder)
        "15640441000119104",  # Primary open angle glaucoma of right eye (disorder)
        "15673001000119103",  # Congenital glaucoma of bilateral eyes (disorder)
        "15673081000119106",  # Congenital glaucoma of left eye (disorder)
        "15679801000119105",  # Uveitis-glaucoma-hyphema syndrome of bilateral eyes (disorder)
        "15736441000119108",  # Primary angle-closure glaucoma of bilateral eyes (disorder)
        "15736481000119103",  # Primary angle-closure glaucoma of right eye (disorder)
        "15736521000119103",  # Primary angle-closure glaucoma of left eye (disorder)
        "15736561000119108",  # Neovascular glaucoma of bilateral eyes (disorder)
        "15736601000119108",  # Neovascular glaucoma of right eye (disorder)
        "15736641000119105",  # Neovascular glaucoma of left eye (disorder)
        "15736681000119100",  # Narrow-angle glaucoma of right eye (disorder)
        "15736721000119106",  # Angle-closure glaucoma of bilateral eyes (disorder)
        "15736761000119101",  # Narrow angle glaucoma of left eye (disorder)
        "15738841000119105",  # Glaucoma of bilateral eyes due to combination of mechanisms (disorder)
        "15738881000119100",  # Glaucoma of left eye due to combination of mechanisms (disorder)
        "15738921000119107",  # Glaucoma of right eye due to combination of mechanisms (disorder)
        "15739041000119106",  # Glaucoma of bilateral eyes due to iris anomaly (disorder)
        "15739081000119101",  # Glaucoma of bilateral eyes due to anterior segment anomaly (disorder)
        "15739121000119104",  # Glaucoma of right eye due to anterior segment anomaly (disorder)
        "15739161000119109",  # Glaucoma of left eye due to anterior segment anomaly (disorder)
        "15739201000119104",  # Glaucoma of left eye due to ocular vascular disorder (disorder)
        "15739241000119102",  # Glaucoma of right eye due to ocular vascular disorder (disorder)
        "15739281000119107",  # Glaucoma of bilateral eyes due to ocular vascular disorder (disorder)
        "15739321000119102",  # Glaucoma of right eye due to chamber angle anomaly (disorder)
        "15739361000119107",  # Glaucoma of bilateral eyes due to chamber angle anomaly (disorder)
        "15739401000119103",  # Glaucoma of left eye due to chamber angle anomaly (disorder)
        "15739441000119101",  # Glaucoma of right eye due to trauma (disorder)
        "15739481000119106",  # Glaucoma of left eye due to trauma (disorder)
        "15739561000119101",  # Glaucoma of right eye due to lens disorder (disorder)
        "15739641000119104",  # Glaucoma of left eye due to lens disorder (disorder)
        "15739681000119109",  # Glaucoma of right eye due to systemic disorder (disorder)
        "15739721000119103",  # Glaucoma of left eye due to systemic disorder (disorder)
        "15739761000119108",  # Glaucoma of bilateral eyes due to systemic disorder (disorder)
        "15993671000119108",  # Open-angle glaucoma of right eye caused by steroid (disorder)
        "15993711000119107",  # Open-angle glaucoma of left eye caused by steroid (disorder)
        "15993751000119108",  # Open-angle glaucoma of bilateral eyes caused by steroid (disorder)
        "15996831000119101",  # Glaucoma caused by Onchocerca volvulus (disorder)
    }


class GlaucomaAssociatedWithCongenitalAnomalies_Dystrophies_AndSystemicSyndromes(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of glaucoma associated with congenital anomalies, dystrophies, and systemic syndromes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of glaucoma that may be due to congenital anomalies, dystrophies, or other systemic syndromes, including trauma, or a secondary diagnosis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = (
        "Glaucoma Associated with Congenital Anomalies, Dystrophies, and Systemic Syndromes"
    )
    OID = "2.16.840.1.113883.3.526.3.1461"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4031X0",  # Glaucoma secondary to eye trauma, right eye, stage unspecified
        "H4031X1",  # Glaucoma secondary to eye trauma, right eye, mild stage
        "H4031X2",  # Glaucoma secondary to eye trauma, right eye, moderate stage
        "H4031X3",  # Glaucoma secondary to eye trauma, right eye, severe stage
        "H4031X4",  # Glaucoma secondary to eye trauma, right eye, indeterminate stage
        "H4032X0",  # Glaucoma secondary to eye trauma, left eye, stage unspecified
        "H4032X1",  # Glaucoma secondary to eye trauma, left eye, mild stage
        "H4032X2",  # Glaucoma secondary to eye trauma, left eye, moderate stage
        "H4032X3",  # Glaucoma secondary to eye trauma, left eye, severe stage
        "H4032X4",  # Glaucoma secondary to eye trauma, left eye, indeterminate stage
        "H4033X0",  # Glaucoma secondary to eye trauma, bilateral, stage unspecified
        "H4033X1",  # Glaucoma secondary to eye trauma, bilateral, mild stage
        "H4033X2",  # Glaucoma secondary to eye trauma, bilateral, moderate stage
        "H4033X3",  # Glaucoma secondary to eye trauma, bilateral, severe stage
        "H4033X4",  # Glaucoma secondary to eye trauma, bilateral, indeterminate stage
        "H4041X0",  # Glaucoma secondary to eye inflammation, right eye, stage unspecified
        "H4041X1",  # Glaucoma secondary to eye inflammation, right eye, mild stage
        "H4041X2",  # Glaucoma secondary to eye inflammation, right eye, moderate stage
        "H4041X3",  # Glaucoma secondary to eye inflammation, right eye, severe stage
        "H4041X4",  # Glaucoma secondary to eye inflammation, right eye, indeterminate stage
        "H4042X0",  # Glaucoma secondary to eye inflammation, left eye, stage unspecified
        "H4042X1",  # Glaucoma secondary to eye inflammation, left eye, mild stage
        "H4042X2",  # Glaucoma secondary to eye inflammation, left eye, moderate stage
        "H4042X3",  # Glaucoma secondary to eye inflammation, left eye, severe stage
        "H4042X4",  # Glaucoma secondary to eye inflammation, left eye, indeterminate stage
        "H4043X0",  # Glaucoma secondary to eye inflammation, bilateral, stage unspecified
        "H4043X1",  # Glaucoma secondary to eye inflammation, bilateral, mild stage
        "H4043X2",  # Glaucoma secondary to eye inflammation, bilateral, moderate stage
        "H4043X3",  # Glaucoma secondary to eye inflammation, bilateral, severe stage
        "H4043X4",  # Glaucoma secondary to eye inflammation, bilateral, indeterminate stage
        "H4051X0",  # Glaucoma secondary to other eye disorders, right eye, stage unspecified
        "H4051X1",  # Glaucoma secondary to other eye disorders, right eye, mild stage
        "H4051X2",  # Glaucoma secondary to other eye disorders, right eye, moderate stage
        "H4051X3",  # Glaucoma secondary to other eye disorders, right eye, severe stage
        "H4051X4",  # Glaucoma secondary to other eye disorders, right eye, indeterminate stage
        "H4052X0",  # Glaucoma secondary to other eye disorders, left eye, stage unspecified
        "H4052X1",  # Glaucoma secondary to other eye disorders, left eye, mild stage
        "H4052X2",  # Glaucoma secondary to other eye disorders, left eye, moderate stage
        "H4052X3",  # Glaucoma secondary to other eye disorders, left eye, severe stage
        "H4052X4",  # Glaucoma secondary to other eye disorders, left eye, indeterminate stage
        "H4053X0",  # Glaucoma secondary to other eye disorders, bilateral, stage unspecified
        "H4053X1",  # Glaucoma secondary to other eye disorders, bilateral, mild stage
        "H4053X2",  # Glaucoma secondary to other eye disorders, bilateral, moderate stage
        "H4053X3",  # Glaucoma secondary to other eye disorders, bilateral, severe stage
        "H4053X4",  # Glaucoma secondary to other eye disorders, bilateral, indeterminate stage
        "H40811",  # Glaucoma with increased episcleral venous pressure, right eye
        "H40812",  # Glaucoma with increased episcleral venous pressure, left eye
        "H40813",  # Glaucoma with increased episcleral venous pressure, bilateral
        "H40821",  # Hypersecretion glaucoma, right eye
        "H40822",  # Hypersecretion glaucoma, left eye
        "H40823",  # Hypersecretion glaucoma, bilateral
        "H40831",  # Aqueous misdirection, right eye
        "H40832",  # Aqueous misdirection, left eye
        "H40833",  # Aqueous misdirection, bilateral
        "H4089",  # Other specified glaucoma
        "H409",  # Unspecified glaucoma
        "H42",  # Glaucoma in diseases classified elsewhere
    }
    SNOMEDCT = {
        "19309007",  # Glaucoma associated with vascular disorder (disorder)
        "29538005",  # Glaucomatocyclitic crisis (disorder)
        "32893002",  # Phacolytic glaucoma (disorder)
        "37155002",  # Glaucoma associated with ocular inflammation (disorder)
        "45623002",  # Glaucoma associated with anterior segment anomaly (disorder)
        "46168003",  # Pigmentary glaucoma (disorder)
        "66725002",  # Glaucoma due to perforating injury (disorder)
        "66747002",  # Glaucoma associated with ocular disorder (disorder)
        "68241007",  # Glaucoma associated with ocular trauma (disorder)
        "84333006",  # Glaucoma associated with lens disorder (disorder)
        "95717004",  # Secondary glaucoma (disorder)
        "193552006",  # Glaucoma due to chamber angle anomaly (disorder)
        "193553001",  # Glaucoma due to iris anomaly (disorder)
        "193555008",  # Glaucoma due to systemic syndrome (disorder)
        "193556009",  # Glaucoma in endocrine, nutritional and metabolic diseases (disorder)
        "193561006",  # Secondary angle-closure glaucoma with pupil block (disorder)
        "206248004",  # Traumatic glaucoma due to birth trauma (disorder)
        "232083008",  # Glaucoma and corneal anomaly (disorder)
        "232086000",  # Neovascular glaucoma (disorder)
        "314033007",  # Uveitic glaucoma (disorder)
        "370504007",  # Aqueous humor misdirect (disorder)
        "392300000",  # Phacomorphic glaucoma (disorder)
        "392352004",  # Angle recession glaucoma (disorder)
        "404634005",  # Glaucoma associated with iridocorneal endothelial syndrome (disorder)
        "404648005",  # Lens particle glaucoma (disorder)
        "418435001",  # Macular corneal dystrophy Type II (disorder)
        "333151000119108",  # Phacolytic glaucoma of right eye (disorder)
        "338761000119103",  # Phacolytic glaucoma of left eye (disorder)
        "15673041000119101",  # Congenital glaucoma of right eye (disorder)
        "15679801000119105",  # Uveitis-glaucoma-hyphema syndrome of bilateral eyes (disorder)
        "15736561000119108",  # Neovascular glaucoma of bilateral eyes (disorder)
        "15736601000119108",  # Neovascular glaucoma of right eye (disorder)
        "15736641000119105",  # Neovascular glaucoma of left eye (disorder)
        "15739041000119106",  # Glaucoma of bilateral eyes due to iris anomaly (disorder)
        "15739081000119101",  # Glaucoma of bilateral eyes due to anterior segment anomaly (disorder)
        "15739121000119104",  # Glaucoma of right eye due to anterior segment anomaly (disorder)
        "15739161000119109",  # Glaucoma of left eye due to anterior segment anomaly (disorder)
        "15739321000119102",  # Glaucoma of right eye due to chamber angle anomaly (disorder)
        "15739361000119107",  # Glaucoma of bilateral eyes due to chamber angle anomaly (disorder)
        "15739401000119103",  # Glaucoma of left eye due to chamber angle anomaly (disorder)
        "15739441000119101",  # Glaucoma of right eye due to trauma (disorder)
        "15739481000119106",  # Glaucoma of left eye due to trauma (disorder)
        "15739561000119101",  # Glaucoma of right eye due to lens disorder (disorder)
        "15739681000119109",  # Glaucoma of right eye due to systemic disorder (disorder)
        "15739721000119103",  # Glaucoma of left eye due to systemic disorder (disorder)
        "15739761000119108",  # Glaucoma of bilateral eyes due to systemic disorder (disorder)
    }


class HereditaryChoroidalDystrophies(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of hereditary choroidal dystrophies.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hereditary choroidal dystrophies, including choroideremia and atrophy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Hereditary Choroidal Dystrophies"
    OID = "2.16.840.1.113883.3.526.3.1462"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H3120",  # Hereditary choroidal dystrophy, unspecified
        "H3121",  # Choroideremia
        "H3122",  # Choroidal dystrophy (central areolar) (generalized) (peripapillary)
        "H3123",  # Gyrate atrophy, choroid
        "H3129",  # Other hereditary choroidal dystrophy
    }
    SNOMEDCT = {
        "19871000",  # Total generalized choroidal dystrophy (disorder)
        "24950005",  # Partial generalized choroidal dystrophy (disorder)
        "41250005",  # Diffuse choroidal sclerosis (disorder)
        "59753003",  # Total circumpapillary dystrophy of choroid (disorder)
        "74469006",  # Hereditary choroidal dystrophy (disorder)
        "75241009",  # Choroideremia (disorder)
        "92820007",  # Circumpapillary dystrophy of choroid (disorder)
        "193466003",  # Partial circumpapillary choroid dystrophy (disorder)
        "231996009",  # Central areolar choroidal sclerosis (disorder)
        "231997000",  # Peripapillary choroidal sclerosis (disorder)
        "247200005",  # Choroidal atrophy (finding)
        "312918002",  # Choroidal dystrophy (disorder)
        "314467007",  # Gyrate atrophy (disorder)
        "715984007",  # Cerebellar ataxia with hypogonadism and choroidal dystrophy syndrome (disorder)
        "717761005",  # Choroideremia with deafness and obesity syndrome (disorder)
        "720850008",  # Choroidal atrophy and alopecia syndrome (disorder)
        "733110004",  # Van den Bosch syndrome (disorder)
    }


class HereditaryCornealDystrophies(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of hereditary corneal dystrophies.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hereditary corneal dystrophies, including congenital, epithelial/juvenile, granular, lattice or macular corneal dystrophies.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Hereditary Corneal Dystrophies"
    OID = "2.16.840.1.113883.3.526.3.1424"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H18511",  # Endothelial corneal dystrophy, right eye
        "H18512",  # Endothelial corneal dystrophy, left eye
        "H18513",  # Endothelial corneal dystrophy, bilateral
        "H18521",  # Epithelial (juvenile) corneal dystrophy, right eye
        "H18522",  # Epithelial (juvenile) corneal dystrophy, left eye
        "H18523",  # Epithelial (juvenile) corneal dystrophy, bilateral
        "H18531",  # Granular corneal dystrophy, right eye
        "H18532",  # Granular corneal dystrophy, left eye
        "H18533",  # Granular corneal dystrophy, bilateral
        "H18541",  # Lattice corneal dystrophy, right eye
        "H18542",  # Lattice corneal dystrophy, left eye
        "H18543",  # Lattice corneal dystrophy, bilateral
        "H18551",  # Macular corneal dystrophy, right eye
        "H18552",  # Macular corneal dystrophy, left eye
        "H18553",  # Macular corneal dystrophy, bilateral
        "H18591",  # Other hereditary corneal dystrophies, right eye
        "H18592",  # Other hereditary corneal dystrophies, left eye
        "H18593",  # Other hereditary corneal dystrophies, bilateral
    }
    SNOMEDCT = {
        "1674008",  # Juvenile epithelial corneal dystrophy (disorder)
        "5587004",  # Corneal dystrophy (disorder)
        "29504002",  # Posterior polymorphous corneal dystrophy (disorder)
        "39662004",  # Crystalline corneal dystrophy (disorder)
        "45283008",  # Granular corneal dystrophy (disorder)
        "60258001",  # Macular corneal dystrophy (disorder)
        "75663006",  # Ring-like corneal dystrophy (disorder)
        "77797009",  # Hereditary corneal dystrophy (disorder)
        "95488001",  # Congenital macular corneal dystrophy (disorder)
        "123670005",  # Adiposa cornea dystrophy (disorder)
        "129623003",  # Iridocorneal endothelial syndrome (disorder)
        "193837009",  # Combined corneal dystrophy (disorder)
        "193839007",  # Fuchs' corneal dystrophy (disorder)
        "231930000",  # Reis-Bucklers' corneal dystrophy (disorder)
        "231931001",  # Stromal corneal dystrophy (disorder)
        "231933003",  # Lattice corneal dystrophy, isolated form (disorder)
        "231934009",  # Pre-descemet's corneal dystrophy (disorder)
        "361199007",  # Lattice corneal dystrophy (disorder)
        "373426005",  # Epithelial basement membrane dystrophy (disorder)
        "397568004",  # Granular corneal dystrophy type II (disorder)
        "404633004",  # Cogan-Reese syndrome (disorder)
        "416633008",  # Congenital hereditary endothelial dystrophy type 1 (disorder)
        "416960004",  # Corneal endothelial dystrophy (disorder)
        "417065002",  # Thiel-Behnke corneal dystrophy (disorder)
        "417183007",  # Fleck corneal dystrophy (disorder)
        "417395001",  # Congenital hereditary endothelial dystrophy type 2 (disorder)
        "417651000",  # Congenital hereditary endothelial dystrophy (disorder)
        "418054005",  # Macular corneal dystrophy Type I (disorder)
        "418435001",  # Macular corneal dystrophy Type II (disorder)
        "419039007",  # Granular corneal dystrophy type I (disorder)
        "419197009",  # Lattice corneal dystrophy Type I (disorder)
        "419395007",  # Schnyder crystalline cornea dystrophy (disorder)
        "419900000",  # Gelatinous droplike corneal dystrophy (disorder)
        "430888006",  # Dystrophy of anterior cornea (disorder)
    }


class HereditaryRetinalDystrophies(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of hereditary retinal dystrophies.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hereditary retinal or vitreoretinal dystrophies, including Usher syndrome.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Hereditary Retinal Dystrophies"
    OID = "2.16.840.1.113883.3.526.3.1463"
    DEFINITION_VERSION = "20210219"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H3550",  # Unspecified hereditary retinal dystrophy
        "H3551",  # Vitreoretinal dystrophy
        "H3552",  # Pigmentary retinal dystrophy
        "H3553",  # Other dystrophies primarily involving the sensory retina
        "H3554",  # Dystrophies primarily involving the retinal pigment epithelium
        "H36",  # Retinal disorders in diseases classified elsewhere
    }
    SNOMEDCT = {
        "15059000",  # Retinal dystrophy in systemic lipidosis (disorder)
        "28835009",  # Retinitis pigmentosa (disorder)
        "41799005",  # Hereditary retinal dystrophy (disorder)
        "57838006",  # Retinitis pigmentosa-deafness syndrome (disorder)
        "79556007",  # Vitreoretinal dystrophy (disorder)
        "80328002",  # Progressive cone-rod dystrophy (disorder)
        "192788009",  # Retinal dystrophy in cerebroretinal lipidosis (disorder)
        "193400007",  # Hereditary retinal dystrophies in lipidoses (disorder)
        "193409008",  # Hyaline retinal dystrophy (disorder)
        "193415008",  # Tapetoretinal dystrophy (disorder)
        "232052009",  # Autosomal dominant retinitis pigmentosa (disorder)
        "232053004",  # Autosomal recessive retinitis pigmentosa (disorder)
        "232056007",  # Progressive rod dystrophy (disorder)
        "232057003",  # Usher syndrome type 1 (disorder)
        "232058008",  # Usher syndrome type 2 (disorder)
        "267613004",  # Progressive cone dystrophy (without rod involvement) (disorder)
        "312917007",  # Cone dystrophy (disorder)
        "313003009",  # Rod dystrophy (disorder)
        "764939004",  # Fundus albipunctatus (disorder)
    }


class HypotonyOfEye(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of hypotony of eye.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hypotony of eye.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Hypotony of Eye"
    OID = "2.16.840.1.113883.3.526.3.1426"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4440",  # Unspecified hypotony of eye
        "H44411",  # Flat anterior chamber hypotony of right eye
        "H44412",  # Flat anterior chamber hypotony of left eye
        "H44413",  # Flat anterior chamber hypotony of eye, bilateral
        "H44421",  # Hypotony of right eye due to ocular fistula
        "H44422",  # Hypotony of left eye due to ocular fistula
        "H44423",  # Hypotony of eye due to ocular fistula, bilateral
        "H44431",  # Hypotony of eye due to other ocular disorders, right eye
        "H44432",  # Hypotony of eye due to other ocular disorders, left eye
        "H44433",  # Hypotony of eye due to other ocular disorders, bilateral
        "H44441",  # Primary hypotony of right eye
        "H44442",  # Primary hypotony of left eye
        "H44443",  # Primary hypotony of eye, bilateral
    }
    SNOMEDCT = {
        "2251002",  # Primary hypotony of eye (disorder)
        "19721008",  # Hypotony of eye (disorder)
        "53885001",  # Hypotony of eye co-occurrent and due to ocular fistula (disorder)
        "95255005",  # Secondary hypotony of eye (disorder)
        "424497004",  # Optic disc edema due to hypotony (disorder)
        "700280001",  # Flat anterior chamber of eye with hypotony (disorder)
    }


class InjuryToOpticNerveAndPathways(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of injury to optic nerve and optic pathways.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of injuries to optic nerve, optic chiasm, optic tract, optic pathways and optic visual cortex.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Injury to Optic Nerve and Pathways"
    OID = "2.16.840.1.113883.3.526.3.1427"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "S04011A",  # Injury of optic nerve, right eye, initial encounter
        "S04012A",  # Injury of optic nerve, left eye, initial encounter
        "S0402XA",  # Injury of optic chiasm, initial encounter
        "S04031A",  # Injury of optic tract and pathways, right side, initial encounter
        "S04032A",  # Injury of optic tract and pathways, left side, initial encounter
        "S04041A",  # Injury of visual cortex, right side, initial encounter
        "S04042A",  # Injury of visual cortex, left side, initial encounter
    }
    SNOMEDCT = {
        "48233004",  # Traumatic optic nerve injury (disorder)
        "69820004",  # Injury of optic chiasm (disorder)
        "80499003",  # Traumatic injury of visual pathways (disorder)
        "212141007",  # Optic nerve and pathway injury (disorder)
        "212142000",  # Visual cortex injury (disorder)
        "230513000",  # Avulsion of optic nerve (disorder)
        "230514006",  # Radiation damage to optic nerve (disorder)
        "230515007",  # Traumatic optic neuropathy (disorder)
        "230519001",  # Injury of optic tract (disorder)
        "735649003",  # Injury of bilateral optic tracts (disorder)
        "736321008",  # Injury of both visual cortices (disorder)
        "762362006",  # Traumatic injury of left optic tract (disorder)
        "762363001",  # Traumatic injury of right optic tract (disorder)
        "762436001",  # Injury of left visual cortex (disorder)
        "762617003",  # Injury of right visual cortex (disorder)
    }


class MacularScarOfPosteriorPolar(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of macular scar of posterior polar.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of macular or posterior pole scarring.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Macular Scar of Posterior Polar"
    OID = "2.16.840.1.113883.3.526.3.1559"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H31011",  # Macula scars of posterior pole (postinflammatory) (post-traumatic), right eye
        "H31012",  # Macula scars of posterior pole (postinflammatory) (post-traumatic), left eye
        "H31013",  # Macula scars of posterior pole (postinflammatory) (post-traumatic), bilateral
    }
    SNOMEDCT = {
        "18410006",  # Scarred macula (disorder)
        "21939006",  # Scar of posterior pole of eye (disorder)
        "267611002",  # Degeneration of macula and posterior pole (disorder)
        "425622004",  # Scar of macula following trauma (disorder)
        "737273002",  # Degeneration of posterior pole of eye (disorder)
        "774010000",  # Scarred macula of left eye (disorder)
        "344341000119106",  # Postinflammatory chorioretinal scar of bilateral eyes (disorder)
        "15631531000119100",  # Scarred macula of bilateral eyes (disorder)
        "15631611000119106",  # Scarred macula of right eye (disorder)
        "15710081000119102",  # Scar of macula of right eye following trauma (disorder)
        "15710121000119100",  # Scar of macula of bilateral eyes following trauma (disorder)
        "15710161000119105",  # Scar of macula of left eye following trauma (disorder)
    }


class MorgagnianCataract(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of morgagnian cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a morgagnian cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Morgagnian Cataract"
    OID = "2.16.840.1.113883.3.526.3.1558"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H2521",  # Age-related cataract, morgagnian type, right eye
        "H2522",  # Age-related cataract, morgagnian type, left eye
        "H2523",  # Age-related cataract, morgagnian type, bilateral
    }
    SNOMEDCT = {
        "264443002",  # Morgagnian cataract (disorder)
    }


class NystagmusAndOtherIrregularEyeMovements(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for nystagmus and other irregular eye movements.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of nystagmus.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Nystagmus and Other Irregular Eye Movements"
    OID = "2.16.840.1.113883.3.526.3.1465"
    DEFINITION_VERSION = "20210308"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H5500",  # Unspecified nystagmus
        "H5501",  # Congenital nystagmus
        "H5502",  # Latent nystagmus
        "H5503",  # Visual deprivation nystagmus
        "H5504",  # Dissociated nystagmus
        "H5509",  # Other forms of nystagmus
        "H5581",  # Deficient saccadic eye movements
        "H5582",  # Deficient smooth pursuit eye movements
        "H5589",  # Other irregular eye movements
    }
    SNOMEDCT = {
        "563001",  # Nystagmus (disorder)
        "3327002",  # Jerk nystagmus (disorder)
        "8134005",  # Periodic alternating nystagmus (disorder)
        "9074007",  # Retraction nystagmus (disorder)
        "9520006",  # Dissociated nystagmus (disorder)
        "29356006",  # End-position nystagmus (disorder)
        "35743001",  # Pendular nystagmus (disorder)
        "39737009",  # Symptomatic nystagmus (disorder)
        "40782004",  # Hennebert's syndrome (disorder)
        "44526006",  # Rotary nystagmus (disorder)
        "45339001",  # Visual deprivation nystagmus (disorder)
        "46888001",  # Nystagmus associated with disorder of the vestibular system (disorder)
        "64635004",  # Congenital nystagmus (disorder)
        "68610001",  # Latent nystagmus (disorder)
        "81756001",  # Horizontal nystagmus (disorder)
        "83655001",  # Rebound nystagmus (disorder)
        "95777008",  # Temporary nystagmus (disorder)
        "95778003",  # Permanent nystagmus (disorder)
        "95779006",  # Unidirectional nystagmus (disorder)
        "95780009",  # Multidirectional nystagmus (disorder)
        "95781008",  # Conjugate nystagmus (disorder)
        "95782001",  # See-saw nystagmus (disorder)
        "95783006",  # Rotational nystagmus (disorder)
        "95784000",  # Convergence nystagmus (disorder)
        "95785004",  # Positional nystagmus (disorder)
        "95786003",  # Fatigable positional nystagmus (disorder)
        "95787007",  # Non-fatigable positional nystagmus (disorder)
        "95788002",  # Positional nystagmus, variable eye direction (disorder)
        "95790001",  # Fixation nystagmus (disorder)
        "111533001",  # Vertical nystagmus (disorder)
        "232124002",  # Voluntary nystagmus (disorder)
        "246761002",  # Uniocular nystagmus (disorder)
        "246762009",  # Gaze paretic nystagmus (disorder)
        "278468008",  # Pressure-induced nystagmus (disorder)
        "278469000",  # Sound intensity-induced nystagmus (disorder)
        "285765003",  # Manifest-latent nystagmus (disorder)
        "307670000",  # Congenital nystagmus with sensory abnormality (disorder)
        "307671001",  # Congenital nystagmus without sensory abnormality (disorder)
        "307672008",  # Oculopalatal myoclonus (disorder)
        "307673003",  # Peripheral vestibular nystagmus (disorder)
        "307674009",  # Benign paroxysmal positional vertigo nystagmus (disorder)
        "307675005",  # Central vestibular nystagmus (disorder)
        "307676006",  # Downbeat nystagmus (disorder)
        "307677002",  # Upbeat nystagmus (disorder)
        "307678007",  # Torsional central vestibular nystagmus (disorder)
        "307679004",  # Hemi-seesaw nystagmus (disorder)
        "307680001",  # Abducting nystagmus (disorder)
        "307700006",  # Ataxic nystagmus (disorder)
        "314935002",  # Acquired nystagmus (disorder)
        "359687001",  # Acquired pendular nystagmus (disorder)
        "373590007",  # Convergence retraction nystagmus (disorder)
        "400948003",  # Spasmus nutans (disorder)
        "400949006",  # Divergence retraction nystagmus (disorder)
        "404676002",  # Bruns nystagmus (disorder)
        "404677006",  # Elliptical nystagmus (disorder)
        "414489005",  # Infantile nystagmus syndrome (disorder)
        "418472007",  # Heartbeat nystagmus (disorder)
        "733334001",  # Perverted nystagmus (disorder)
        "733335000",  # Eyelid nystagmus (disorder)
        "762438000",  # Physiological nystagmus (disorder)
        "762439008",  # Torsional nystagmus (disorder)
        "6881000119109",  # Sensory nystagmus (disorder)
        "431951000124101",  # Idiopathic nystagmus (disorder)
    }


class OpenWoundOfEyeball(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of open wound of the eyeball.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of choroidal rupture, traumatic enucleation of eye, and avulsion, contusion, laceration, penetration and/or rupture of eye.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Open Wound of Eyeball"
    OID = "2.16.840.1.113883.3.526.3.1430"
    DEFINITION_VERSION = "20210406"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "S0511XA",  # Contusion of eyeball and orbital tissues, right eye, initial encounter
        "S0512XA",  # Contusion of eyeball and orbital tissues, left eye, initial encounter
        "S0521XA",  # Ocular laceration and rupture with prolapse or loss of intraocular tissue, right eye, initial encounter
        "S0522XA",  # Ocular laceration and rupture with prolapse or loss of intraocular tissue, left eye, initial encounter
        "S0531XA",  # Ocular laceration without prolapse or loss of intraocular tissue, right eye, initial encounter
        "S0532XA",  # Ocular laceration without prolapse or loss of intraocular tissue, left eye, initial encounter
        "S0551XA",  # Penetrating wound with foreign body of right eyeball, initial encounter
        "S0552XA",  # Penetrating wound with foreign body of left eyeball, initial encounter
        "S0561XA",  # Penetrating wound without foreign body of right eyeball, initial encounter
        "S0562XA",  # Penetrating wound without foreign body of left eyeball, initial encounter
        "S0571XA",  # Avulsion of right eye, initial encounter
        "S0572XA",  # Avulsion of left eye, initial encounter
        "S058X1A",  # Other injuries of right eye and orbit, initial encounter
        "S058X2A",  # Other injuries of left eye and orbit, initial encounter
    }
    SNOMEDCT = {
        "1869002",  # Rupture of sphincter of pupil (disorder)
        "2532009",  # Choroidal rupture (disorder)
        "3018008",  # Penetration of eyeball with magnetic foreign body (disorder)
        "12193006",  # Avulsion of eye (disorder)
        "21549001",  # Superficial laceration of cornea (disorder)
        "23293006",  # Rupture of eye with partial loss of intraocular tissue (disorder)
        "37911007",  # Ocular laceration with prolapse AND/OR exposure of intraocular tissue (disorder)
        "41658006",  # Traumatic enucleation of eye (disorder)
        "42312000",  # Rupture in Descemet's membrane (disorder)
        "42697001",  # Contusion of orbital tissues (disorder)
        "44199000",  # Penetration of eyeball with nonmagnetic foreign body (disorder)
        "85100000",  # Open wound of eyeball (disorder)
        "95718009",  # Rupture of lens (disorder)
        "95725002",  # Corneal laceration (disorder)
        "110262009",  # Contusion of periorbital region (disorder)
        "125687005",  # Contusion of ocular adnexa (disorder)
        "210285009",  # Ocular laceration with intraocular prolapse (disorder)
        "211483007",  # Contusion of globe of eye (disorder)
        "231791008",  # Rupture of globe (disorder)
        "231795004",  # Laceration of globe of eye (disorder)
        "231952009",  # Tears of iris stroma (disorder)
        "315296002",  # Penetrating wound of eye (disorder)
        "367423000",  # Contusion of eye (disorder)
        "416179002",  # Contusion of eye AND ocular adnexa (disorder)
        "416786002",  # Penetrating injury of lens (disorder)
        "416931008",  # Corneoscleral laceration (disorder)
        "417654008",  # Contusion of ocular adnexa and periocular tissues (disorder)
        "419743001",  # Ruptured Descemet membrane due to birth trauma (disorder)
        "722613000",  # Penetrating wound of eyeball without foreign body (disorder)
        "735657000",  # Penetrating injury of both eyeballs (disorder)
        "870312007",  # Penetrating wound of eyeball with foreign body (disorder)
        "10942751000119100",  # Commotio retinae of right retina (disorder)
        "10942791000119105",  # Commotio retinae of left retina (disorder)
        "12403101000119108",  # Penetrating wound of right eye caused by foreign body (disorder)
    }


class OpticAtrophy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for optic atrophy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of optic atrophy, which may include atrophy of the optic disc.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Optic Atrophy"
    OID = "2.16.840.1.113883.3.526.3.1466"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4720",  # Unspecified optic atrophy
        "H47211",  # Primary optic atrophy, right eye
        "H47212",  # Primary optic atrophy, left eye
        "H47213",  # Primary optic atrophy, bilateral
        "H4722",  # Hereditary optic atrophy
        "H47231",  # Glaucomatous optic atrophy, right eye
        "H47232",  # Glaucomatous optic atrophy, left eye
        "H47233",  # Glaucomatous optic atrophy, bilateral
        "H47291",  # Other optic atrophy, right eye
        "H47292",  # Other optic atrophy, left eye
        "H47293",  # Other optic atrophy, bilateral
    }
    SNOMEDCT = {
        "1207009",  # Glaucomatous atrophy of optic disc (disorder)
        "2065009",  # Dominant hereditary optic atrophy (disorder)
        "19448008",  # Optic atrophy associated with retinal dystrophy (disorder)
        "21098003",  # Primary optic atrophy (disorder)
        "22386003",  # Syphilitic optic atrophy (disorder)
        "24003004",  # Postinflammatory optic atrophy (disorder)
        "26360005",  # Hereditary optic atrophy (disorder)
        "58610003",  # Leber's optic atrophy (disorder)
        "70694009",  # Diabetes mellitus AND insipidus with optic atrophy AND deafness (disorder)
        "76976005",  # Optic atrophy (disorder)
        "95774001",  # Atrophy of optic disc (disorder)
        "95775000",  # Retrobulbar optic nerve atrophy (disorder)
        "111527005",  # Partial optic atrophy (disorder)
        "123615007",  # Secondary optic atrophy (disorder)
        "128203003",  # Hereditary motor and sensory neuropathy with optic atrophy (disorder)
        "194043004",  # Optic atrophy secondary to retinal disease (disorder)
        "230511003",  # Optic atrophy secondary to papilledema (disorder)
        "239028001",  # Odontotrichomelic syndrome (disorder)
        "247219000",  # Atrophy of sector of optic disc (disorder)
        "312944002",  # Compressive optic atrophy (disorder)
        "359683002",  # Complete optic atrophy (disorder)
        "715374003",  # Autosomal dominant optic atrophy plus syndrome (disorder)
        "717336005",  # Autosomal dominant optic atrophy classic form (disorder)
        "717975006",  # Autosomal dominant optic atrophy and peripheral neuropathy syndrome (disorder)
        "719517009",  # Autosomal dominant optic atrophy and cataract (disorder)
        "838307002",  # Childhood-onset autosomal dominant optic atrophy (disorder)
        "838345001",  # Autosomal recessive optic atrophy type 6 (disorder)
        "870319003",  # Optic atrophy due to late syphilis (disorder)
        "348051000119102",  # Atrophy of right optic nerve following inflammation (disorder)
        "348501000119108",  # Atrophy of left optic nerve following inflammation (disorder)
        "680991000119108",  # Atrophy of bilateral optic nerves following inflammation (disorder)
    }


class OpticNeuritis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of optic neuritis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of optic neuritis, neuropathy, neuroretinitis, and papillitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Optic Neuritis"
    OID = "2.16.840.1.113883.3.526.3.1467"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4601",  # Optic papillitis, right eye
        "H4602",  # Optic papillitis, left eye
        "H4603",  # Optic papillitis, bilateral
        "H4611",  # Retrobulbar neuritis, right eye
        "H4612",  # Retrobulbar neuritis, left eye
        "H4613",  # Retrobulbar neuritis, bilateral
        "H462",  # Nutritional optic neuropathy
        "H463",  # Toxic optic neuropathy
        "H468",  # Other optic neuritis
        "H469",  # Unspecified optic neuritis
    }
    SNOMEDCT = {
        "25044007",  # Neuromyelitis optica (disorder)
        "26125006",  # Toxic optic neuropathy (disorder)
        "36276008",  # Syphilitic retrobulbar neuritis (disorder)
        "51604006",  # Acute retrobulbar neuritis (disorder)
        "66760008",  # Optic neuritis (disorder)
        "73221001",  # Optic papillitis (disorder)
        "73431005",  # Meningococcal optic neuritis (disorder)
        "82108004",  # Nutritional optic neuropathy (disorder)
        "230507009",  # Retrobulbar neuritis (disorder)
        "314537004",  # Optic papillopathy due to diabetes mellitus (disorder)
        "314538009",  # Tobacco related optic neuropathy (disorder)
        "314539001",  # Alcohol related optic neuropathy (disorder)
        "314540004",  # Drug induced optic neuropathy (disorder)
        "314787009",  # Optic neuroretinitis (disorder)
        "410471004",  # Neuroretinitis (disorder)
        "410472006",  # Bartonella henselae neuroretinitis (disorder)
        "416913007",  # Toxoplasma neuroretinitis (disorder)
        "420920007",  # Optic neuropathy due to vitamin B12 deficiency (disorder)
        "2691000124105",  # Recurrent optic neuritis (disorder)
        "432521000124106",  # Bilateral optic neuritis (disorder)
        "15631011000119102",  # Neuritis of right optic nerve (disorder)
        "15631051000119101",  # Neuritis of left optic nerve (disorder)
    }


class OtherBackgroundRetinopathyAndRetinalVascularChanges(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of other background retinopathy and retinal vascular changes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of various retinal changes and retinopathies, including exudative retinopathy, retinal neovascularization, arteriolitis, arteritis, endarteritis, phlebitis, and vasculitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Other Background Retinopathy and Retinal Vascular Changes"
    OID = "2.16.840.1.113883.3.526.3.1469"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H35021",  # Exudative retinopathy, right eye
        "H35022",  # Exudative retinopathy, left eye
        "H35023",  # Exudative retinopathy, bilateral
        "H35051",  # Retinal neovascularization, unspecified, right eye
        "H35052",  # Retinal neovascularization, unspecified, left eye
        "H35053",  # Retinal neovascularization, unspecified, bilateral
        "H35061",  # Retinal vasculitis, right eye
        "H35062",  # Retinal vasculitis, left eye
        "H35063",  # Retinal vasculitis, bilateral
    }
    SNOMEDCT = {
        "7089008",  # Retinal phlebitis (disorder)
        "11255009",  # Retinal arteritis (disorder)
        "25506007",  # Exudative retinopathy (disorder)
        "54122009",  # Eales' disease (disorder)
        "61267008",  # Retinal neovascularization (disorder)
        "71719003",  # Thrombophlebitis of retinal vein (disorder)
        "75971007",  # Choroidal retinal neovascularization (disorder)
        "77628002",  # Retinal vasculitis (disorder)
        "86191000",  # Retinal endarteritis (disorder)
        "247099009",  # Optic disc neovascularization (disorder)
        "247100001",  # Peripheral retinal neovascularization (disorder)
        "314265001",  # Classic choroidal neovascular membrane (disorder)
        "414297000",  # Frosted branch angiitis (disorder)
        "416703007",  # Retinal vasculitis due to polyarteritis nodosa (disorder)
        "417019000",  # Retinal arteriolitis (disorder)
        "417303004",  # Retinal vasculitis due to systemic lupus erythematosus (disorder)
        "417544004",  # Segmental retinal periarteritis (disorder)
        "677651000119102",  # Neovascularization of choroid of left eye (disorder)
        "677681000119109",  # Neovascularization of choroid of right eye (disorder)
    }


class OtherDisordersOfOpticNerve(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for other disorders of optic nerve.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of ischemic optic neuropathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Other Disorders of Optic Nerve"
    OID = "2.16.840.1.113883.3.526.3.1471"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H47011",  # Ischemic optic neuropathy, right eye
        "H47012",  # Ischemic optic neuropathy, left eye
        "H47013",  # Ischemic optic neuropathy, bilateral
    }
    SNOMEDCT = {
        "14357004",  # Ischemic optic neuropathy (disorder)
        "230508004",  # Arteritic ischemic optic neuropathy (disorder)
        "230509007",  # Non-arteritic ischemic optic neuropathy (disorder)
        "404659001",  # Anterior ischemic optic neuropathy (disorder)
        "415147002",  # Posterior ischemic optic neuropathy (disorder)
        "733506009",  # Arteritic anterior ischemic optic neuropathy (disorder)
        "838324008",  # Autoimmune optic neuropathy (disorder)
        "838363001",  # Arteritic anterior ischemic optic neuropathy due to giant cell arteritis (disorder)
        "860808008",  # Idiopathic non-arteritic ischemic optic neuropathy (disorder)
        "860884007",  # Idiopathic non-arteritic posterior ischemic optic neuropathy (disorder)
        "334651000119101",  # Ischemic optic neuropathy of right eye (disorder)
        "340251000119108",  # Ischemic optic neuropathy of left eye (disorder)
        "345611000119102",  # Ischemic optic neuropathy of bilateral eyes (disorder)
        "15731601000119100",  # Anterior ischemic optic neuropathy of left eye (disorder)
        "15731641000119103",  # Anterior ischemic optic neuropathy of right eye (disorder)
    }


class OtherEndophthalmitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of other endophthalmitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of ophthalmia nodosa, parasitic cyst or endophthalmitis, panuveitis, sympathetic uveitis, or other endophthalmitis diagnoses.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Other Endophthalmitis"
    OID = "2.16.840.1.113883.3.526.3.1473"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H16241",  # Ophthalmia nodosa, right eye
        "H16242",  # Ophthalmia nodosa, left eye
        "H16243",  # Ophthalmia nodosa, bilateral
        "H21331",  # Parasitic cyst of iris, ciliary body or anterior chamber, right eye
        "H21332",  # Parasitic cyst of iris, ciliary body or anterior chamber, left eye
        "H21333",  # Parasitic cyst of iris, ciliary body or anterior chamber, bilateral
        "H33121",  # Parasitic cyst of retina, right eye
        "H33122",  # Parasitic cyst of retina, left eye
        "H33123",  # Parasitic cyst of retina, bilateral
        "H44111",  # Panuveitis, right eye
        "H44112",  # Panuveitis, left eye
        "H44113",  # Panuveitis, bilateral
        "H44121",  # Parasitic endophthalmitis, unspecified, right eye
        "H44122",  # Parasitic endophthalmitis, unspecified, left eye
        "H44123",  # Parasitic endophthalmitis, unspecified, bilateral
        "H44131",  # Sympathetic uveitis, right eye
        "H44132",  # Sympathetic uveitis, left eye
        "H44133",  # Sympathetic uveitis, bilateral
        "H4419",  # Other endophthalmitis
    }
    SNOMEDCT = {
        "12371008",  # Ophthalmia nodosa (disorder)
        "57100005",  # Parasitic endophthalmitis (disorder)
        "75315001",  # Sympathetic uveitis (disorder)
        "75614007",  # Panuveitis (disorder)
        "231992006",  # Onchocerca chorioretinitis (disorder)
        "240842000",  # Ocular onchocerciasis (disorder)
        "415737009",  # Toxocara endophthalmitis (disorder)
        "766933000",  # Idiopathic panuveitis (disorder)
        "333391000119101",  # Panuveitis of right eye (disorder)
        "339001000119107",  # Panuveitis of left eye (disorder)
        "344751000119109",  # Panuveitis of bilateral eyes (disorder)
    }


class OtherProliferativeRetinopathy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of other proliferative retinopathy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of premature retinopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Other Proliferative Retinopathy"
    OID = "2.16.840.1.113883.3.526.3.1480"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H35101",  # Retinopathy of prematurity, unspecified, right eye
        "H35102",  # Retinopathy of prematurity, unspecified, left eye
        "H35103",  # Retinopathy of prematurity, unspecified, bilateral
        "H35111",  # Retinopathy of prematurity, stage 0, right eye
        "H35112",  # Retinopathy of prematurity, stage 0, left eye
        "H35113",  # Retinopathy of prematurity, stage 0, bilateral
        "H35121",  # Retinopathy of prematurity, stage 1, right eye
        "H35122",  # Retinopathy of prematurity, stage 1, left eye
        "H35123",  # Retinopathy of prematurity, stage 1, bilateral
        "H35131",  # Retinopathy of prematurity, stage 2, right eye
        "H35132",  # Retinopathy of prematurity, stage 2, left eye
        "H35133",  # Retinopathy of prematurity, stage 2, bilateral
        "H35141",  # Retinopathy of prematurity, stage 3, right eye
        "H35142",  # Retinopathy of prematurity, stage 3, left eye
        "H35143",  # Retinopathy of prematurity, stage 3, bilateral
        "H35151",  # Retinopathy of prematurity, stage 4, right eye
        "H35152",  # Retinopathy of prematurity, stage 4, left eye
        "H35153",  # Retinopathy of prematurity, stage 4, bilateral
        "H35161",  # Retinopathy of prematurity, stage 5, right eye
        "H35162",  # Retinopathy of prematurity, stage 5, left eye
        "H35163",  # Retinopathy of prematurity, stage 5, bilateral
        "H35171",  # Retrolental fibroplasia, right eye
        "H35172",  # Retrolental fibroplasia, left eye
        "H35173",  # Retrolental fibroplasia, bilateral
    }
    SNOMEDCT = {
        "408847006",  # Retinopathy of prematurity stage 1 - demarcation line (disorder)
        "408848001",  # Retinopathy of prematurity stage 2 - intraretinal ridge (disorder)
        "408849009",  # Retinopathy of prematurity stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "408850009",  # Retinopathy of prematurity stage 4 - subtotal retinal detachment (disorder)
        "408851008",  # Retinopathy of prematurity stage 5 - total retinal detachment (disorder)
        "415297005",  # Retinopathy of prematurity (disorder)
        "332371000119107",  # Retinopathy of prematurity of right eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "332381000119105",  # Retinopathy of prematurity of right eye stage 2 - intraretinal ridge (disorder)
        "332391000119108",  # Retinopathy of prematurity of right eye stage 1 - demarcation line (disorder)
        "332411000119108",  # Retinopathy of prematurity of right eye (disorder)
        "337981000119108",  # Retinopathy of prematurity of left eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "337991000119106",  # Retinopathy of prematurity of left eye stage 2 - intraretinal ridge (disorder)
        "338001000119102",  # Retinopathy of prematurity of left eye stage 1 - demarcation line (disorder)
        "338021000119106",  # Retinopathy of prematurity of left eye (disorder)
        "343751000119106",  # Retinopathy of prematurity of bilateral eyes stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "343761000119108",  # Retinopathy of prematurity of bilateral eyes stage 2 - intraretinal ridge (disorder)
        "343771000119102",  # Retinopathy of prematurity of bilateral eyes stage 1 - demarcation line (disorder)
        "343791000119101",  # Retinopathy of prematurity of bilateral eyes (disorder)
    }


class OtherAndUnspecifiedFormsOfChorioretinitisAndRetinochoroiditis(ValueSet):
    """
    **Clinical Focus:** Includes concepts that represent a diagnosis of posterior cyclitis, chorioretinal inflammation, other chorioretinitis or retinitis or retinochoroiditis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes only relevant concepts associated with a diagnosis of posterior cyclitis, chorioretinal inflammation, other chorioretinitis or retinitis or retinochoroiditis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Other and Unspecified Forms of Chorioretinitis and Retinochoroiditis"
    OID = "2.16.840.1.113883.3.526.3.1468"
    DEFINITION_VERSION = "20210219"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H3021",  # Posterior cyclitis, right eye
        "H3022",  # Posterior cyclitis, left eye
        "H3023",  # Posterior cyclitis, bilateral
        "H30811",  # Harada's disease, right eye
        "H30812",  # Harada's disease, left eye
        "H30813",  # Harada's disease, bilateral
        "H30891",  # Other chorioretinal inflammations, right eye
        "H30892",  # Other chorioretinal inflammations, left eye
        "H30893",  # Other chorioretinal inflammations, bilateral
        "H3091",  # Unspecified chorioretinal inflammation, right eye
        "H3092",  # Unspecified chorioretinal inflammation, left eye
        "H3093",  # Unspecified chorioretinal inflammation, bilateral
    }
    SNOMEDCT = {
        "6313001",  # Juxtapapillary focal choroiditis AND chorioretinitis (disorder)
        "9941009",  # Congenital syphilitic choroiditis (disorder)
        "14894002",  # Generalized disseminated choroiditis AND chorioretinitis (disorder)
        "15847003",  # Focal chorioretinitis (disorder)
        "22455005",  # Cytomegaloviral retinitis (disorder)
        "36535003",  # Peripheral focal retinitis AND retinochoroiditis (disorder)
        "42031001",  # Macular focal retinitis AND retinochoroiditis (disorder)
        "44674000",  # Focal choroiditis AND chorioretinitis of other posterior pole (disorder)
        "46497009",  # Juxtapapillary focal retinitis AND retinochoroiditis (disorder)
        "46627006",  # Chorioretinitis (disorder)
        "48225000",  # Tuberculous disseminated chorioretinitis (disorder)
        "51579006",  # Disseminated choroiditis AND chorioretinitis, posterior pole (disorder)
        "51930008",  # Focal retinitis AND retinochoroiditis of other posterior pole (disorder)
        "56787009",  # Peripheral focal choroiditis AND chorioretinitis (disorder)
        "61517002",  # Metastatic disseminated retinitis AND retinochoroiditis (disorder)
        "61922005",  # Disseminated retinitis (disorder)
        "66281009",  # Secondary syphilitic chorioretinitis (disorder)
        "69811008",  # Peripheral disseminated choroiditis AND chorioretinitis (disorder)
        "77628002",  # Retinal vasculitis (disorder)
        "77939001",  # Syphilitic disseminated retinochoroiditis (disorder)
        "78769001",  # Disseminated chorioretinitis (disorder)
        "88290000",  # Focal chorioretinitis caused by acquired toxoplasmosis (disorder)
        "95682009",  # Granulomatous chorioretinitis (disorder)
        "95685006",  # Granulomatous retinitis (disorder)
        "186263000",  # Tuberculous chorioretinitis (disorder)
        "187039009",  # Histoplasma capsulatum with retinitis (disorder)
        "187194004",  # Toxoplasmosis chorioretinitis (disorder)
        "193434008",  # Focal juxtapapillary choroiditis (disorder)
        "193436005",  # Juxtapapillary focal chorioretinitis (disorder)
        "193438006",  # Peripheral focal retinochoroiditis (disorder)
        "193442009",  # Disseminated posterior pole chorioretinitis (disorder)
        "193443004",  # Disseminated peripheral chorioretinitis (disorder)
        "193444005",  # General disseminated chorioretinitis (disorder)
        "193497004",  # Vogt-Koyanagi-Harada disease (disorder)
        "231981005",  # Birdshot chorioretinitis (disorder)
        "231984002",  # Viral retinitis (disorder)
        "231988004",  # Bacterial chorioretinitis (disorder)
        "231990003",  # Candida retinitis (disorder)
        "231992006",  # Onchocerca chorioretinitis (disorder)
        "231994007",  # Toxocara chorioretinitis (disorder)
        "312491004",  # Serpiginous choroiditis (disorder)
        "312930008",  # Sarcoid chorioretinitis (disorder)
        "312934004",  # Syphilitic chorioretinitis (disorder)
        "312955002",  # Tertiary syphilitic chorioretinitis (disorder)
        "314031009",  # Acute toxoplasmosis chorioretinitis (disorder)
        "314032002",  # Inactive toxoplasmosis chorioretinitis (disorder)
        "314428001",  # Posterior cyclitis (disorder)
        "359668007",  # Focal retinitis (disorder)
        "359698000",  # Parasitic chorioretinitis (disorder)
        "359707007",  # Fungal chorioretinitis (disorder)
        "371099005",  # Generalized disseminated choroiditis AND chorioretinitis with acute necrosis (disorder)
        "373667006",  # Posterior pole focal chorioretinitis (disorder)
        "399463004",  # Retinitis (disorder)
        "410470003",  # Syphilitic retinitis (disorder)
        "414783007",  # Multifocal choroiditis (disorder)
        "415218004",  # Punctate outer retinal toxoplasmosis (disorder)
        "416589006",  # Toxoplasma retinitis (disorder)
        "416877003",  # Cryptococcal chorioretinitis (disorder)
        "417405005",  # Chorioretinitis with coccidioidmycosis (disorder)
        "417547006",  # Cryptococcal retinitis (disorder)
        "421029004",  # Herpes zoster retinitis (disorder)
        "441854007",  # Reactivation of toxoplasmosis chorioretinitis (disorder)
        "866099002",  # Leber idiopathic stellate neuroretinitis (disorder)
        "866133000",  # Infectious neuroretinitis (disorder)
        "346251000119106",  # Disseminated chorioretinitis of bilateral eyes (disorder)
        "15677801000119103",  # Chorioretinitis of right eye caused by secondary syphilis (disorder)
        "15677881000119106",  # Chorioretinitis of left eye caused by secondary syphilis (disorder)
        "15685401000119104",  # Retinitis of bilateral eyes caused by Toxoplasma gondii (disorder)
        "15685441000119102",  # Retinitis of bilateral eyes caused by Treponema pallidum (disorder)
        "15685801000119101",  # Retinitis of right eye (disorder)
        "15685841000119104",  # Retinitis of left eye (disorder)
        "15685881000119109",  # Retinitis of bilateral eyes (disorder)
        "15992631000119101",  # Neuroretinitis of right eye (disorder)
        "15992671000119103",  # Neuroretinitis of left eye (disorder)
        "15992711000119104",  # Neuroretinitis of bilateral eyes (disorder)
        "15993391000119102",  # Retinitis of bilateral eyes caused by secondary syphilis (disorder)
    }


class PathologicMyopia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for pathologic myopia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of degenerative disorders of eye, Forster-Fuchs' spot, malignant or severe myopia, and myopic macular degeneration.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Pathologic Myopia"
    OID = "2.16.840.1.113883.3.526.3.1432"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4421",  # Degenerative myopia, right eye
        "H4422",  # Degenerative myopia, left eye
        "H4423",  # Degenerative myopia, bilateral
        "H442A1",  # Degenerative myopia with choroidal neovascularization, right eye
        "H442A2",  # Degenerative myopia with choroidal neovascularization, left eye
        "H442A3",  # Degenerative myopia with choroidal neovascularization, bilateral eye
        "H442B1",  # Degenerative myopia with macular hole, right eye
        "H442B2",  # Degenerative myopia with macular hole, left eye
        "H442B3",  # Degenerative myopia with macular hole, bilateral eye
        "H442C1",  # Degenerative myopia with retinal detachment, right eye
        "H442C2",  # Degenerative myopia with retinal detachment, left eye
        "H442C3",  # Degenerative myopia with retinal detachment, bilateral eye
        "H442D1",  # Degenerative myopia with foveoschisis, right eye
        "H442D2",  # Degenerative myopia with foveoschisis, left eye
        "H442D3",  # Degenerative myopia with foveoschisis, bilateral eye
        "H442E1",  # Degenerative myopia with other maculopathy, right eye
        "H442E2",  # Degenerative myopia with other maculopathy, left eye
        "H442E3",  # Degenerative myopia with other maculopathy, bilateral eye
        "H4430",  # Unspecified degenerative disorder of globe
    }
    SNOMEDCT = {
        "3712000",  # Degenerated eye (disorder)
        "32022003",  # Degenerative progressive high myopia (disorder)
        "34187009",  # Severe myopia (disorder)
        "62585004",  # Degenerative disorder of eye (disorder)
        "193276002",  # Malignant myopia (disorder)
        "232072004",  # Forster-Fuchs' spot (disorder)
        "232135007",  # Index myopia (disorder)
        "312898002",  # Myopic macular degeneration (disorder)
        "846621001",  # Degenerative myopia of left eye (disorder)
        "335471000119108",  # Degenerative progressive high myopia of right eye (disorder)
        "346481000119100",  # Neovascularization of bilateral corneas (disorder)
        "16320791000119106",  # Myopic macular degeneration of bilateral eyes (disorder)
        "16837461000119106",  # Neovascularization of choroid of right eye due to degenerative myopia (disorder)
        "16837521000119102",  # Neovascularization of choroid of left eye due to degenerative myopia (disorder)
    }


class PosteriorLenticonus(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for posterior lenticonus.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of posterior lenticonus, including spherophakia.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Posterior Lenticonus"
    OID = "2.16.840.1.113883.3.526.3.1433"
    DEFINITION_VERSION = "20210314"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Q122",  # Coloboma of lens
        "Q124",  # Spherophakia
        "Q128",  # Other congenital lens malformations
    }
    SNOMEDCT = {
        "33521009",  # Congenital anomaly of lens shape (disorder)
        "95482000",  # Posterior lenticonus (disorder)
        "95483005",  # Posterior lentiglobus (disorder)
        "204134008",  # Coloboma of lens (disorder)
        "416671000",  # Microspherophakia (disorder)
        "418653000",  # Secondary lens coloboma (disorder)
        "419544009",  # Primary lens coloboma (disorder)
        "773690008",  # Microcornea, posterior megalolenticonus, persistent fetal vasculature, coloboma syndrome (disorder)
        "840458000",  # Coloboma of bilateral lenses (disorder)
        "15987871000119100",  # Congenital coloboma of right lens (disorder)
        "15987911000119102",  # Congenital coloboma of left lens (disorder)
    }


class PriorPenetratingKeratoplasty(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of penetrating keratoplasty.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of keratoconus.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Prior Penetrating Keratoplasty"
    OID = "2.16.840.1.113883.3.526.3.1475"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H18601",  # Keratoconus, unspecified, right eye
        "H18602",  # Keratoconus, unspecified, left eye
        "H18603",  # Keratoconus, unspecified, bilateral
        "H18611",  # Keratoconus, stable, right eye
        "H18612",  # Keratoconus, stable, left eye
        "H18613",  # Keratoconus, stable, bilateral
        "H18621",  # Keratoconus, unstable, right eye
        "H18622",  # Keratoconus, unstable, left eye
        "H18623",  # Keratoconus, unstable, bilateral
    }
    SNOMEDCT = {
        "63216003",  # Congenital keratoconus (disorder)
        "65636009",  # Keratoconus (disorder)
        "75340008",  # Congenital keratoconus posticus circumscriptus (disorder)
        "111523009",  # Acute hydrops keratoconus (disorder)
        "193844000",  # Keratoconus, stable condition (disorder)
        "314758008",  # Posterior keratoconus (disorder)
        "700281002",  # Unstable keratoconus (disorder)
        "334551000119102",  # Unstable keratoconus of right eye (disorder)
        "334561000119100",  # Stable keratoconus of right eye (disorder)
        "334571000119106",  # Keratoconus of right cornea (disorder)
        "340151000119101",  # Unstable keratoconus of left eye (disorder)
        "340161000119104",  # Stable keratoconus of left eye (disorder)
        "340171000119105",  # Keratoconus of left cornea (disorder)
        "345511000119103",  # Unstable keratoconus of bilateral eyes (disorder)
        "345521000119105",  # Stable keratoconus of bilateral eyes (disorder)
        "345531000119108",  # Keratoconus of bilateral corneas (disorder)
        "348371000119109",  # Acute hydrops keratoconus of right eye (disorder)
        "348821000119101",  # Acute hydrops keratoconus of left eye (disorder)
        "349321000119104",  # Acute hydrops keratoconus of bilateral eyes (disorder)
    }


class PurulentEndophthalmitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for purulent endophthalmitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of purulent endophthalmitis, panophthalmitis, vitreous abscess, and other types of endophthalmitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Purulent Endophthalmitis"
    OID = "2.16.840.1.113883.3.526.3.1477"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H44001",  # Unspecified purulent endophthalmitis, right eye
        "H44002",  # Unspecified purulent endophthalmitis, left eye
        "H44003",  # Unspecified purulent endophthalmitis, bilateral
        "H44011",  # Panophthalmitis (acute), right eye
        "H44012",  # Panophthalmitis (acute), left eye
        "H44013",  # Panophthalmitis (acute), bilateral
        "H44021",  # Vitreous abscess (chronic), right eye
        "H44022",  # Vitreous abscess (chronic), left eye
        "H44023",  # Vitreous abscess (chronic), bilateral
    }
    SNOMEDCT = {
        "1493002",  # Acute endophthalmitis (disorder)
        "1833005",  # Phacoanaphylactic endophthalmitis (disorder)
        "1847009",  # Endophthalmitis (disorder)
        "13978000",  # Chronic endophthalmitis (disorder)
        "41720003",  # Purulent endophthalmitis (disorder)
        "314544008",  # Postoperative endophthalmitis (disorder)
        "314545009",  # Acute postoperative endophthalmitis (disorder)
        "314546005",  # Chronic postoperative endophthalmitis (disorder)
        "314547001",  # Post-traumatic endophthalmitis (disorder)
        "314548006",  # Endogenous endophthalmitis (disorder)
        "410495002",  # Postoperative endophthalmitis caused by Propionibacterium acnes (disorder)
        "410496001",  # Postoperative sterile endophthalmitis (disorder)
        "410497005",  # Postoperative infectious endophthalmitis (disorder)
        "413583003",  # Aspergillus endophthalmitis (disorder)
        "413626002",  # Bacterial endophthalmitis (disorder)
        "413665004",  # Bleb associated endophthalmitis (disorder)
        "413746000",  # Candida endophthalmitis (disorder)
        "414303007",  # Fungal endophthalmitis (disorder)
        "414491002",  # Infectious endophthalmitis (disorder)
        "415182001",  # Propionibacterium acnes endophthalmitis (disorder)
        "415584007",  # Sterile endophthalmitis (disorder)
        "415737009",  # Toxocara endophthalmitis (disorder)
        "416296007",  # Cryptococcal endophthalmitis (disorder)
        "348381000119107",  # Acute endophthalmitis of right eye (disorder)
        "348831000119103",  # Acute endophthalmitis of left eye (disorder)
        "15678441000119103",  # Chronic endophthalmitis of left eye (disorder)
        "15678481000119108",  # Chronic endophthalmitis of right eye (disorder)
    }


class RetinalDetachmentWithRetinalDefect(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for retinal detachment with a retinal defect.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of various types of retinal detachments.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Retinal Detachment with Retinal Defect"
    OID = "2.16.840.1.113883.3.526.3.1478"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H33001",  # Unspecified retinal detachment with retinal break, right eye
        "H33002",  # Unspecified retinal detachment with retinal break, left eye
        "H33003",  # Unspecified retinal detachment with retinal break, bilateral
        "H33011",  # Retinal detachment with single break, right eye
        "H33012",  # Retinal detachment with single break, left eye
        "H33013",  # Retinal detachment with single break, bilateral
        "H33021",  # Retinal detachment with multiple breaks, right eye
        "H33022",  # Retinal detachment with multiple breaks, left eye
        "H33023",  # Retinal detachment with multiple breaks, bilateral
        "H33031",  # Retinal detachment with giant retinal tear, right eye
        "H33032",  # Retinal detachment with giant retinal tear, left eye
        "H33033",  # Retinal detachment with giant retinal tear, bilateral
        "H33041",  # Retinal detachment with retinal dialysis, right eye
        "H33042",  # Retinal detachment with retinal dialysis, left eye
        "H33043",  # Retinal detachment with retinal dialysis, bilateral
        "H33051",  # Total retinal detachment, right eye
        "H33052",  # Total retinal detachment, left eye
        "H33053",  # Total retinal detachment, bilateral
        "H338",  # Other retinal detachments
    }
    SNOMEDCT = {
        "3598000",  # Partial recent retinal detachment with single defect (disorder)
        "4178006",  # Partial recent retinal detachment with multiple defects (disorder)
        "7219007",  # Traction detachment with vitreoretinal organization (disorder)
        "19620000",  # Rhegmatogenous retinal detachment (disorder)
        "34711008",  # Traction detachment of retina (disorder)
        "38579007",  # Partial recent retinal detachment with retinal dialysis (disorder)
        "38599001",  # Serous retinal detachment (disorder)
        "42059000",  # Retinal detachment (disorder)
        "56202001",  # Retinal detachment with retinal defect (disorder)
        "193319001",  # New partial retinal detachment with single defect (disorder)
        "193320007",  # New partial retinal detachment with multiple defects (disorder)
        "193322004",  # New partial retinal detachment with retinal dialysis (disorder)
        "193323009",  # Recent total retinal detachment (disorder)
        "193324003",  # Recent subtotal retinal detachment (disorder)
        "193325002",  # Old partial retinal detachment (disorder)
        "193326001",  # Old total retinal detachment (disorder)
        "193327005",  # Old subtotal retinal detachment (disorder)
        "232008001",  # Rhegmatogenous retinal detachment - macula on (disorder)
        "232009009",  # Rhegmatogenous retinal detachment - macula off (disorder)
        "232010004",  # Traction retinal detachment involving macula (disorder)
        "232011000",  # Traction retinal detachment sparing macula (disorder)
        "232023006",  # Traction detachment of retina due to diabetes mellitus (disorder)
        "247165009",  # Retinal detachment - subretinal fluid (disorder)
        "314494006",  # Pseudophakic retinal detachment (disorder)
        "331831000119109",  # Total detachment of right retina (disorder)
        "337451000119106",  # Total detachment of left retina (disorder)
    }


class RetinalVascularOcclusion(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of retinal vascular occlusion.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of retinal vein or artery occlusion.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Retinal Vascular Occlusion"
    OID = "2.16.840.1.113883.3.526.3.1479"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H3411",  # Central retinal artery occlusion, right eye
        "H3412",  # Central retinal artery occlusion, left eye
        "H3413",  # Central retinal artery occlusion, bilateral
        "H34231",  # Retinal artery branch occlusion, right eye
        "H34232",  # Retinal artery branch occlusion, left eye
        "H34233",  # Retinal artery branch occlusion, bilateral
        "H348110",  # Central retinal vein occlusion, right eye, with macular edema
        "H348111",  # Central retinal vein occlusion, right eye, with retinal neovascularization
        "H348112",  # Central retinal vein occlusion, right eye, stable
        "H348120",  # Central retinal vein occlusion, left eye, with macular edema
        "H348121",  # Central retinal vein occlusion, left eye, with retinal neovascularization
        "H348122",  # Central retinal vein occlusion, left eye, stable
        "H348130",  # Central retinal vein occlusion, bilateral, with macular edema
        "H348131",  # Central retinal vein occlusion, bilateral, with retinal neovascularization
        "H348132",  # Central retinal vein occlusion, bilateral, stable
        "H348310",  # Tributary (branch) retinal vein occlusion, right eye, with macular edema
        "H348311",  # Tributary (branch) retinal vein occlusion, right eye, with retinal neovascularization
        "H348312",  # Tributary (branch) retinal vein occlusion, right eye, stable
        "H348320",  # Tributary (branch) retinal vein occlusion, left eye, with macular edema
        "H348321",  # Tributary (branch) retinal vein occlusion, left eye, with retinal neovascularization
        "H348322",  # Tributary (branch) retinal vein occlusion, left eye, stable
        "H348330",  # Tributary (branch) retinal vein occlusion, bilateral, with macular edema
        "H348331",  # Tributary (branch) retinal vein occlusion, bilateral, with retinal neovascularization
        "H348332",  # Tributary (branch) retinal vein occlusion, bilateral, stable
    }
    SNOMEDCT = {
        "776009",  # Partial occlusion of retinal artery (disorder)
        "24596005",  # Venous retinal branch occlusion (disorder)
        "38742007",  # Central retinal artery occlusion (disorder)
        "50821009",  # Arterial retinal branch occlusion (disorder)
        "65593009",  # Partial occlusion of retinal vein (disorder)
        "68478007",  # Central retinal vein occlusion (disorder)
        "73757007",  # Retinal vascular occlusion (disorder)
        "87224000",  # Transient arterial retinal occlusion (disorder)
        "232035005",  # Retinal artery occlusion (disorder)
        "232037002",  # Branch macular artery occlusion (disorder)
        "232038007",  # Central retinal vein occlusion with neovascularization (disorder)
        "232039004",  # Central retinal vein occlusion with macular edema (disorder)
        "232040002",  # Central retinal vein occlusion - juvenile (disorder)
        "232041003",  # Central retinal vein occlusion - juvenile with neovascularization (disorder)
        "232042005",  # Central retinal vein occlusion - juvenile with macular edema (disorder)
        "232043000",  # Hemispheric retinal vein occlusion (disorder)
        "232044006",  # Hemispheric retinal vein occlusion with neovascularization (disorder)
        "232045007",  # Hemispheric retinal vein occlusion with macular edema (disorder)
        "232046008",  # Branch retinal vein occlusion with neovascularization (disorder)
        "232048009",  # Branch retinal vein occlusion with macular edema (disorder)
        "247121001",  # Macular branch retinal vein occlusion (disorder)
        "312997008",  # Central retinal vein occlusion - ischemic (disorder)
        "312998003",  # Central retinal vein occlusion - non-ischemic (disorder)
        "314000002",  # Branch retinal vein occlusion with no neovascularization (disorder)
        "333281000119108",  # Partial occlusion of right retinal artery (disorder)
        "336191000119105",  # Occlusion of right central retinal artery (disorder)
        "337421000119103",  # Transient occlusion of left retinal artery (disorder)
        "338891000119100",  # Partial occlusion of left retinal artery (disorder)
        "341801000119101",  # Occlusion of left central retinal artery (disorder)
        "677771000119102",  # Occlusion of bilateral branch retinal arteries (disorder)
        "677781000119104",  # Occlusion of left branch retinal artery (disorder)
        "677791000119101",  # Occlusion of right branch retinal artery (disorder)
        "677821000119109",  # Partial occlusion of left retinal vein (disorder)
        "677831000119107",  # Partial occlusion of right retinal vein (disorder)
        "680941000119100",  # Partial occlusion of bilateral retinal veins (disorder)
        "15997911000119103",  # Vascular occlusion of retina of left eye (disorder)
        "15997951000119102",  # Vascular occlusion of retina of right eye (disorder)
        "16279401000119108",  # Occlusion of right cilioretinal artery (disorder)
        "16279441000119105",  # Occlusion of left cilioretinal artery (disorder)
        "16711111000119100",  # Retinal neovascularization due to occlusion of bilateral central retinal veins (disorder)
        "16711631000119103",  # Retinal neovascularization due to occlusion of branch of retinal vein of bilateral eyes (disorder)
    }


class RetrolentalFibroplasias(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for retrolental fibroplasias.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of retrolental fibroplasias, including retinopathy of prematurity.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Retrolental Fibroplasias"
    OID = "2.16.840.1.113883.3.526.3.1438"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H35171",  # Retrolental fibroplasia, right eye
        "H35172",  # Retrolental fibroplasia, left eye
        "H35173",  # Retrolental fibroplasia, bilateral
    }
    SNOMEDCT = {
        "408847006",  # Retinopathy of prematurity stage 1 - demarcation line (disorder)
        "408848001",  # Retinopathy of prematurity stage 2 - intraretinal ridge (disorder)
        "408849009",  # Retinopathy of prematurity stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "408850009",  # Retinopathy of prematurity stage 4 - subtotal retinal detachment (disorder)
        "408851008",  # Retinopathy of prematurity stage 5 - total retinal detachment (disorder)
        "415297005",  # Retinopathy of prematurity (disorder)
        "332371000119107",  # Retinopathy of prematurity of right eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "332381000119105",  # Retinopathy of prematurity of right eye stage 2 - intraretinal ridge (disorder)
        "332391000119108",  # Retinopathy of prematurity of right eye stage 1 - demarcation line (disorder)
        "332411000119108",  # Retinopathy of prematurity of right eye (disorder)
        "337981000119108",  # Retinopathy of prematurity of left eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "337991000119106",  # Retinopathy of prematurity of left eye stage 2 - intraretinal ridge (disorder)
        "338001000119102",  # Retinopathy of prematurity of left eye stage 1 - demarcation line (disorder)
        "338021000119106",  # Retinopathy of prematurity of left eye (disorder)
        "343751000119106",  # Retinopathy of prematurity of bilateral eyes stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        "343761000119108",  # Retinopathy of prematurity of bilateral eyes stage 2 - intraretinal ridge (disorder)
        "343771000119102",  # Retinopathy of prematurity of bilateral eyes stage 1 - demarcation line (disorder)
        "343791000119101",  # Retinopathy of prematurity of bilateral eyes (disorder)
    }


class Scleritis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for scleritis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of specific types of scleritis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Scleritis"
    OID = "2.16.840.1.113762.1.4.1226.1"
    DEFINITION_VERSION = "20210308"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H15021",  # Brawny scleritis, right eye
        "H15022",  # Brawny scleritis, left eye
        "H15023",  # Brawny scleritis, bilateral
        "H15031",  # Posterior scleritis, right eye
        "H15032",  # Posterior scleritis, left eye
        "H15033",  # Posterior scleritis, bilateral
        "H15041",  # Scleritis with corneal involvement, right eye
        "H15042",  # Scleritis with corneal involvement, left eye
        "H15043",  # Scleritis with corneal involvement, bilateral
        "H15051",  # Scleromalacia perforans, right eye
        "H15052",  # Scleromalacia perforans, left eye
        "H15053",  # Scleromalacia perforans, bilateral
        "H15091",  # Other scleritis, right eye
        "H15092",  # Other scleritis, left eye
        "H15093",  # Other scleritis, bilateral
    }
    SNOMEDCT = {
        "26664005",  # Scleromalacia perforans (disorder)
        "27886001",  # Sclerosing keratitis (disorder)
        "42574005",  # Scleritis with corneal involvement (disorder)
        "50675000",  # Scleral abscess (disorder)
        "59165007",  # Rheumatoid scleritis (disorder)
        "78370002",  # Scleritis (disorder)
        "91612009",  # Brawny scleritis (disorder)
        "95195003",  # Nodular scleritis (disorder)
        "95680001",  # Sclerouveitis (disorder)
        "95795006",  # Superficial scleritis (disorder)
        "95796007",  # Deep scleritis (disorder)
        "231876002",  # Non-pyogenic granulomatous scleritis (disorder)
        "267660007",  # Posterior scleritis (disorder)
        "410578007",  # Infective scleritis (disorder)
        "416879000",  # Scleritis due to granulomatosis with polyangiitis (disorder)
        "417290008",  # Sclerokeratitis (disorder)
        "332981000119106",  # Posterior scleritis of right eye (disorder)
        "338591000119103",  # Posterior scleritis of left eye (disorder)
        "344361000119105",  # Posterior scleritis of bilateral eyes (disorder)
    }


class SeparationOfRetinalLayers(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for separation of the layers of the retina.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of serous chorioretinopathy and types of retinal detachment.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Separation of Retinal Layers"
    OID = "2.16.840.1.113883.3.526.3.1482"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H35711",  # Central serous chorioretinopathy, right eye
        "H35712",  # Central serous chorioretinopathy, left eye
        "H35713",  # Central serous chorioretinopathy, bilateral
        "H35721",  # Serous detachment of retinal pigment epithelium, right eye
        "H35722",  # Serous detachment of retinal pigment epithelium, left eye
        "H35723",  # Serous detachment of retinal pigment epithelium, bilateral
        "H35731",  # Hemorrhagic detachment of retinal pigment epithelium, right eye
        "H35732",  # Hemorrhagic detachment of retinal pigment epithelium, left eye
        "H35733",  # Hemorrhagic detachment of retinal pigment epithelium, bilateral
    }
    SNOMEDCT = {
        "3598000",  # Partial recent retinal detachment with single defect (disorder)
        "4178006",  # Partial recent retinal detachment with multiple defects (disorder)
        "7219007",  # Traction detachment with vitreoretinal organization (disorder)
        "19620000",  # Rhegmatogenous retinal detachment (disorder)
        "34711008",  # Traction detachment of retina (disorder)
        "38579007",  # Partial recent retinal detachment with retinal dialysis (disorder)
        "38599001",  # Serous retinal detachment (disorder)
        "42059000",  # Retinal detachment (disorder)
        "51987004",  # Retinal detachment without retinal defect (disorder)
        "56202001",  # Retinal detachment with retinal defect (disorder)
        "193319001",  # New partial retinal detachment with single defect (disorder)
        "193320007",  # New partial retinal detachment with multiple defects (disorder)
        "193322004",  # New partial retinal detachment with retinal dialysis (disorder)
        "193323009",  # Recent total retinal detachment (disorder)
        "193324003",  # Recent subtotal retinal detachment (disorder)
        "193325002",  # Old partial retinal detachment (disorder)
        "193326001",  # Old total retinal detachment (disorder)
        "193327005",  # Old subtotal retinal detachment (disorder)
        "232004004",  # Giant retinal tear (disorder)
        "232008001",  # Rhegmatogenous retinal detachment - macula on (disorder)
        "232009009",  # Rhegmatogenous retinal detachment - macula off (disorder)
        "232010004",  # Traction retinal detachment involving macula (disorder)
        "232011000",  # Traction retinal detachment sparing macula (disorder)
        "232012007",  # Combined traction and rhegmatogenous retinal detachment (disorder)
        "232015009",  # Solid retinal detachment (disorder)
        "232023006",  # Traction detachment of retina due to diabetes mellitus (disorder)
        "232034009",  # Central serous retinopathy with pit of optic disc (disorder)
        "247165009",  # Retinal detachment - subretinal fluid (disorder)
        "312923002",  # Chronic central serous chorioretinopathy (disorder)
        "312924008",  # Inactive central serous chorioretinopathy (disorder)
        "312947009",  # Acute central serous chorioretinopathy (disorder)
        "312956001",  # Central serous chorioretinopathy (disorder)
        "312957005",  # Variant central serous chorioretinopathy (disorder)
        "314006008",  # Acute central serous retinopathy with subretinal fluid (disorder)
        "314007004",  # Chronic central serous retinopathy with diffuse retinal pigment epithelial detachment (disorder)
        "314008009",  # Inactive central serous retinopathy with focal retinal pigment epithelial detachment (disorder)
        "314009001",  # Central serous retinopathy with small retinal pigment epithelial detachment (disorder)
        "334891000119105",  # Hemorrhagic detachment of right retinal pigment epithelium (disorder)
        "340501000119106",  # Hemorrhagic detachment of left retinal pigment epithelium (disorder)
        "345811000119101",  # Hemorrhagic detachment of bilateral retinal pigment epithelium (disorder)
    }


class TraumaticCataract(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for traumatic cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of traumatic cataract, including concussion cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Traumatic Cataract"
    OID = "2.16.840.1.113883.3.526.3.1443"
    DEFINITION_VERSION = "20210219"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H26101",  # Unspecified traumatic cataract, right eye
        "H26102",  # Unspecified traumatic cataract, left eye
        "H26103",  # Unspecified traumatic cataract, bilateral
        "H26111",  # Localized traumatic opacities, right eye
        "H26112",  # Localized traumatic opacities, left eye
        "H26113",  # Localized traumatic opacities, bilateral
        "H26121",  # Partially resolved traumatic cataract, right eye
        "H26122",  # Partially resolved traumatic cataract, left eye
        "H26123",  # Partially resolved traumatic cataract, bilateral
        "H26131",  # Total traumatic cataract, right eye
        "H26132",  # Total traumatic cataract, left eye
        "H26133",  # Total traumatic cataract, bilateral
    }
    SNOMEDCT = {
        "8656007",  # Total traumatic cataract (disorder)
        "34361001",  # Traumatic cataract (disorder)
        "193598007",  # Partial resolved traumatic cataract (disorder)
        "231967008",  # Concussion cataract (disorder)
        "231968003",  # Cataract following rupture of capsule (disorder)
        "866114005",  # Localized cataract opacities due to and following traumatic injury (disorder)
    }


class Uveitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts  for diagnoses of uveitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of uveitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Uveitis"
    OID = "2.16.840.1.113883.3.526.3.1444"
    DEFINITION_VERSION = "20210219"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H44111",  # Panuveitis, right eye
        "H44112",  # Panuveitis, left eye
        "H44113",  # Panuveitis, bilateral
        "H44131",  # Sympathetic uveitis, right eye
        "H44132",  # Sympathetic uveitis, left eye
        "H44133",  # Sympathetic uveitis, bilateral
    }
    SNOMEDCT = {
        "4927003",  # Acute anterior uveitis (disorder)
        "16553002",  # Choroiditis (disorder)
        "33382000",  # Panophthalmitis (disorder)
        "43363007",  # Posterior uveitis (disorder)
        "64015001",  # Subacute anterior uveitis (disorder)
        "75315001",  # Sympathetic uveitis (disorder)
        "75614007",  # Panuveitis (disorder)
        "95706000",  # Phacoantigenic uveitis (disorder)
        "95707009",  # Phacotoxic uveitis (disorder)
        "128473001",  # Uveitis (disorder)
        "186854007",  # Uveitis due to secondary syphilis (disorder)
        "231947004",  # Anterior uveitis idiopathic (disorder)
        "314029000",  # Tuberculous uveitis (disorder)
        "314429009",  # Intermediate uveitis (disorder)
        "398155003",  # Chronic anterior uveitis (disorder)
        "410473001",  # Lyme uveitis (disorder)
        "410476009",  # Uveitis due to leptospirosis (disorder)
        "410477000",  # Infective uveitis (disorder)
        "410494003",  # Lens-induced uveitis (disorder)
        "410501000",  # Drug-induced uveitis (disorder)
        "410692006",  # Anterior uveitis (disorder)
        "413716003",  # Brucellosis uveitis (disorder)
        "413851004",  # Cidofovir-induced anterior uveitis (disorder)
        "414600000",  # Lepromatous anterior uveitis (disorder)
        "414689001",  # Metipranolol-induced anterior uveitis (disorder)
        "414996002",  # Pamidronic acid-induced uveitis (disorder)
        "415344002",  # Rifabutin-induced anterior uveitis (disorder)
        "415359008",  # Sarcoid uveitis (disorder)
        "416666007",  # Anterior uveitis in juvenile idiopathic arthritis (disorder)
        "417321001",  # Human T-cell Lymphoma Virus Type -1 associated uveitis (disorder)
        "418839003",  # Tubulointerstitial nephritis with uveitis syndrome (disorder)
        "428676003",  # Uveitis caused by ranibizumab (disorder)
        "444248002",  # Chronic inflammation of uveal tract (disorder)
        "733317009",  # Primary anterior uveitis (disorder)
        "766933000",  # Idiopathic panuveitis (disorder)
        "870199008",  # Non-infectious uveitis of posterior segment of eye (disorder)
        "870273007",  # Posterior uveitis due to late syphilis (disorder)
        "870329005",  # Panuveitis caused by virus (disorder)
        "870331001",  # Panuveitis caused by Treponema pallidum (disorder)
        "870332008",  # Panuveitis caused by Herpesvirus (disorder)
        "870333003",  # Panuveitis caused by Mycobacterium tuberculosis (disorder)
        "870335005",  # Panuveitis caused by Borrelia burgdorferi (disorder)
        "336541000119107",  # Acute anterior uveitis of right eye (disorder)
        "342151000119100",  # Acute anterior uveitis of left eye (disorder)
        "347311000119100",  # Acute anterior uveitis of bilateral eyes (disorder)
    }


class VascularDisordersOfIrisAndCiliaryBody(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of vascular disorders of iris and ciliary body.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis for a vascular disorder of the iris and/or ciliary body, including rubeosis iridis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Vascular Disorders of Iris and Ciliary Body"
    OID = "2.16.840.1.113883.3.526.3.1445"
    DEFINITION_VERSION = "20210214"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H211X1",  # Other vascular disorders of iris and ciliary body, right eye
        "H211X2",  # Other vascular disorders of iris and ciliary body, left eye
        "H211X3",  # Other vascular disorders of iris and ciliary body, bilateral
    }
    SNOMEDCT = {
        "51995000",  # Rubeosis iridis (disorder)
        "193501009",  # Iris and ciliary body vascular disorder (disorder)
        "678191000119101",  # Rubeosis iridis of bilateral eyes (disorder)
        "678201000119103",  # Rubeosis iridis of left eye (disorder)
        "678211000119100",  # Rubeosis iridis of right eye (disorder)
    }


class VisualFieldDefects(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for visual field defects.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of visual field defects, including scotoma and decreased field of vision.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = "Visual Field Defects"
    OID = "2.16.840.1.113883.3.526.3.1446"
    DEFINITION_VERSION = "20210220"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H53411",  # Scotoma involving central area, right eye
        "H53412",  # Scotoma involving central area, left eye
        "H53413",  # Scotoma involving central area, bilateral
        "H53421",  # Scotoma of blind spot area, right eye
        "H53422",  # Scotoma of blind spot area, left eye
        "H53423",  # Scotoma of blind spot area, bilateral
        "H53431",  # Sector or arcuate defects, right eye
        "H53432",  # Sector or arcuate defects, left eye
        "H53433",  # Sector or arcuate defects, bilateral
        "H53451",  # Other localized visual field defect, right eye
        "H53452",  # Other localized visual field defect, left eye
        "H53453",  # Other localized visual field defect, bilateral
        "H53461",  # Homonymous bilateral field defects, right side
        "H53462",  # Homonymous bilateral field defects, left side
        "H53481",  # Generalized contraction of visual field, right eye
        "H53482",  # Generalized contraction of visual field, left eye
        "H53483",  # Generalized contraction of visual field, bilateral
    }
    SNOMEDCT = {
        "1151008",  # Visual field constriction (finding)
        "2228002",  # Scintillating scotoma (finding)
        "4806007",  # Bjerrum scotoma (finding)
        "12184005",  # Visual field defect (finding)
        "15462009",  # Arcuate scotoma (finding)
        "19900004",  # Decreased peripheral vision (finding)
        "21273008",  # Relative scotoma (finding)
        "23388006",  # Visual field scotoma (finding)
        "24370009",  # Seidel scotoma (finding)
        "33014001",  # Centrocecal scotoma (finding)
        "33048000",  # Peripheral visual field defect (finding)
        "33970004",  # Enlarged blind spot (finding)
        "38950008",  # Central scotoma (finding)
        "49073008",  # Absolute scotoma (finding)
        "59695000",  # Bilateral visual field constriction (finding)
        "64418005",  # Paracentral scotoma (finding)
        "65133002",  # Physiologic scotoma (finding)
        "129625005",  # Ring scotoma (finding)
        "129626006",  # Quadrantic scotoma (finding)
        "193670002",  # Blind spot scotoma (finding)
        "193671003",  # Paracecal scotoma (finding)
        "193672005",  # Angioscotoma (finding)
        "246656009",  # Loss of part of visual field (finding)
        "246658005",  # Seeing blob in vision (finding)
        "246659002",  # Peripheral scotoma (finding)
        "246660007",  # Peripheral ring scotoma (finding)
        "246661006",  # Local peripheral scotoma (finding)
        "246662004",  # Altitudinal scotoma (finding)
        "267628004",  # Generalized visual field constriction (finding)
        "345121001",  # Heteronymous bilateral visual field defects (finding)
        "370961005",  # Glaucomatous visual field defect (finding)
        "404650002",  # Pericentral scotoma (finding)
        "416626008",  # Functional visual field loss (finding)
        "420990001",  # Tunnel visual field constriction (finding)
        "421561005",  # Concentric visual field constriction (finding)
        "445923001",  # Sector visual field defect (finding)
        "446020008",  # Arcuate visual field defect (finding)
        "336171000119109",  # Central scotoma of right eye (finding)
        "341781000119100",  # Central scotoma of left eye (finding)
        "346991000119107",  # Central scotoma of bilateral eyes (finding)
        "16059151000119108",  # Scotoma of left eye (finding)
        "16059191000119103",  # Scotoma of bilateral eyes (finding)
    }


class DiabeticNephropathy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  diabetic nephropathy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis for diabetic nephropathy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = "Diabetic Nephropathy"
    OID = "2.16.840.1.113883.3.464.1003.109.12.1004"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "E0821",  # Diabetes mellitus due to underlying condition with diabetic nephropathy
        "E0822",  # Diabetes mellitus due to underlying condition with diabetic chronic kidney disease
        "E0829",  # Diabetes mellitus due to underlying condition with other diabetic kidney complication
        "E0921",  # Drug or chemical induced diabetes mellitus with diabetic nephropathy
        "E0922",  # Drug or chemical induced diabetes mellitus with diabetic chronic kidney disease
        "E0929",  # Drug or chemical induced diabetes mellitus with other diabetic kidney complication
        "E1021",  # Type 1 diabetes mellitus with diabetic nephropathy
        "E1022",  # Type 1 diabetes mellitus with diabetic chronic kidney disease
        "E1029",  # Type 1 diabetes mellitus with other diabetic kidney complication
        "E1121",  # Type 2 diabetes mellitus with diabetic nephropathy
        "E1122",  # Type 2 diabetes mellitus with diabetic chronic kidney disease
        "E1129",  # Type 2 diabetes mellitus with other diabetic kidney complication
        "E1321",  # Other specified diabetes mellitus with diabetic nephropathy
        "E1322",  # Other specified diabetes mellitus with diabetic chronic kidney disease
        "E1329",  # Other specified diabetes mellitus with other diabetic kidney complication
    }
    ICD9CM = {
        "25041",  # Diabetes with renal manifestations, type I [juvenile type], not stated as uncontrolled
        "25042",  # Diabetes with renal manifestations, type II or unspecified type, uncontrolled
        "25043",  # Diabetes with renal manifestations, type I [juvenile type], uncontrolled
    }
    SNOMEDCT = {
        "38046004",  # Diffuse glomerulosclerosis of kidney due to diabetes mellitus (disorder)
        "54181000",  # Diabetes-nephrosis syndrome (disorder)
        "127013003",  # Disorder of kidney due to diabetes mellitus (disorder)
        "401110002",  # Type 1 diabetes mellitus with persistent microalbuminuria (disorder)
        "420715001",  # Persistent microalbuminuria due to type 2 diabetes mellitus (disorder)
        "421986006",  # Persistent proteinuria due to type 2 diabetes mellitus (disorder)
        "445170001",  # Macroalbuminuric nephropathy due to diabetes mellitus (disorder)
        "792926007",  # Armanni-Ebstein kidney due to diabetes mellitus (disorder)
        "71701000119105",  # Hypertension in chronic kidney disease due to type 1 diabetes mellitus (disorder)
        "90721000119101",  # Chronic kidney disease stage 1 due to type 1 diabetes mellitus (disorder)
        "90731000119103",  # Chronic kidney disease stage 2 due to type 1 diabetes mellitus (disorder)
        "90741000119107",  # Chronic kidney disease stage 3 due to type 1 diabetes mellitus (disorder)
        "90751000119109",  # Chronic kidney disease stage 4 due to type 1 diabetes mellitus (disorder)
        "90761000119106",  # Chronic kidney disease stage 5 due to type 1 diabetes mellitus (disorder)
        "90771000119100",  # End stage renal disease on dialysis due to type 1 diabetes mellitus (disorder)
        "96441000119101",  # Chronic kidney disease due to type 1 diabetes mellitus (disorder)
    }


class GlomerulonephritisAndNephroticSyndrome(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  glomerulonephritis and nephrotic syndrome.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of glomerulonephritis and nephrotic syndrome.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = "Glomerulonephritis and Nephrotic Syndrome"
    OID = "2.16.840.1.113883.3.464.1003.109.12.1018"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "N000",  # Acute nephritic syndrome with minor glomerular abnormality
        "N001",  # Acute nephritic syndrome with focal and segmental glomerular lesions
        "N002",  # Acute nephritic syndrome with diffuse membranous glomerulonephritis
        "N003",  # Acute nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        "N004",  # Acute nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        "N005",  # Acute nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        "N006",  # Acute nephritic syndrome with dense deposit disease
        "N007",  # Acute nephritic syndrome with diffuse crescentic glomerulonephritis
        "N008",  # Acute nephritic syndrome with other morphologic changes
        "N009",  # Acute nephritic syndrome with unspecified morphologic changes
        "N00A",  # Acute nephritic syndrome with C3 glomerulonephritis
        "N010",  # Rapidly progressive nephritic syndrome with minor glomerular abnormality
        "N011",  # Rapidly progressive nephritic syndrome with focal and segmental glomerular lesions
        "N012",  # Rapidly progressive nephritic syndrome with diffuse membranous glomerulonephritis
        "N013",  # Rapidly progressive nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        "N014",  # Rapidly progressive nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        "N015",  # Rapidly progressive nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        "N016",  # Rapidly progressive nephritic syndrome with dense deposit disease
        "N017",  # Rapidly progressive nephritic syndrome with diffuse crescentic glomerulonephritis
        "N018",  # Rapidly progressive nephritic syndrome with other morphologic changes
        "N019",  # Rapidly progressive nephritic syndrome with unspecified morphologic changes
        "N01A",  # Rapidly progressive nephritic syndrome with C3 glomerulonephritis
        "N020",  # Recurrent and persistent hematuria with minor glomerular abnormality
        "N021",  # Recurrent and persistent hematuria with focal and segmental glomerular lesions
        "N022",  # Recurrent and persistent hematuria with diffuse membranous glomerulonephritis
        "N023",  # Recurrent and persistent hematuria with diffuse mesangial proliferative glomerulonephritis
        "N024",  # Recurrent and persistent hematuria with diffuse endocapillary proliferative glomerulonephritis
        "N025",  # Recurrent and persistent hematuria with diffuse mesangiocapillary glomerulonephritis
        "N026",  # Recurrent and persistent hematuria with dense deposit disease
        "N027",  # Recurrent and persistent hematuria with diffuse crescentic glomerulonephritis
        "N028",  # Recurrent and persistent hematuria with other morphologic changes
        "N029",  # Recurrent and persistent hematuria with unspecified morphologic changes
        "N02A",  # Recurrent and persistent hematuria with C3 glomerulonephritis
        "N030",  # Chronic nephritic syndrome with minor glomerular abnormality
        "N031",  # Chronic nephritic syndrome with focal and segmental glomerular lesions
        "N032",  # Chronic nephritic syndrome with diffuse membranous glomerulonephritis
        "N033",  # Chronic nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        "N034",  # Chronic nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        "N035",  # Chronic nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        "N036",  # Chronic nephritic syndrome with dense deposit disease
        "N037",  # Chronic nephritic syndrome with diffuse crescentic glomerulonephritis
        "N038",  # Chronic nephritic syndrome with other morphologic changes
        "N039",  # Chronic nephritic syndrome with unspecified morphologic changes
        "N03A",  # Chronic nephritic syndrome with C3 glomerulonephritis
        "N040",  # Nephrotic syndrome with minor glomerular abnormality
        "N041",  # Nephrotic syndrome with focal and segmental glomerular lesions
        "N042",  # Nephrotic syndrome with diffuse membranous glomerulonephritis
        "N043",  # Nephrotic syndrome with diffuse mesangial proliferative glomerulonephritis
        "N044",  # Nephrotic syndrome with diffuse endocapillary proliferative glomerulonephritis
        "N045",  # Nephrotic syndrome with diffuse mesangiocapillary glomerulonephritis
        "N046",  # Nephrotic syndrome with dense deposit disease
        "N047",  # Nephrotic syndrome with diffuse crescentic glomerulonephritis
        "N048",  # Nephrotic syndrome with other morphologic changes
        "N049",  # Nephrotic syndrome with unspecified morphologic changes
        "N04A",  # Nephrotic syndrome with C3 glomerulonephritis
        "N050",  # Unspecified nephritic syndrome with minor glomerular abnormality
        "N051",  # Unspecified nephritic syndrome with focal and segmental glomerular lesions
        "N052",  # Unspecified nephritic syndrome with diffuse membranous glomerulonephritis
        "N053",  # Unspecified nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        "N054",  # Unspecified nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        "N055",  # Unspecified nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        "N056",  # Unspecified nephritic syndrome with dense deposit disease
        "N057",  # Unspecified nephritic syndrome with diffuse crescentic glomerulonephritis
        "N058",  # Unspecified nephritic syndrome with other morphologic changes
        "N059",  # Unspecified nephritic syndrome with unspecified morphologic changes
        "N05A",  # Unspecified nephritic syndrome with C3 glomerulonephritis
        "N060",  # Isolated proteinuria with minor glomerular abnormality
        "N061",  # Isolated proteinuria with focal and segmental glomerular lesions
        "N062",  # Isolated proteinuria with diffuse membranous glomerulonephritis
        "N063",  # Isolated proteinuria with diffuse mesangial proliferative glomerulonephritis
        "N064",  # Isolated proteinuria with diffuse endocapillary proliferative glomerulonephritis
        "N065",  # Isolated proteinuria with diffuse mesangiocapillary glomerulonephritis
        "N066",  # Isolated proteinuria with dense deposit disease
        "N067",  # Isolated proteinuria with diffuse crescentic glomerulonephritis
        "N068",  # Isolated proteinuria with other morphologic lesion
        "N069",  # Isolated proteinuria with unspecified morphologic lesion
        "N06A",  # Isolated proteinuria with C3 glomerulonephritis
        "N070",  # Hereditary nephropathy, not elsewhere classified with minor glomerular abnormality
        "N071",  # Hereditary nephropathy, not elsewhere classified with focal and segmental glomerular lesions
        "N072",  # Hereditary nephropathy, not elsewhere classified with diffuse membranous glomerulonephritis
        "N073",  # Hereditary nephropathy, not elsewhere classified with diffuse mesangial proliferative glomerulonephritis
        "N074",  # Hereditary nephropathy, not elsewhere classified with diffuse endocapillary proliferative glomerulonephritis
        "N075",  # Hereditary nephropathy, not elsewhere classified with diffuse mesangiocapillary glomerulonephritis
        "N076",  # Hereditary nephropathy, not elsewhere classified with dense deposit disease
        "N077",  # Hereditary nephropathy, not elsewhere classified with diffuse crescentic glomerulonephritis
        "N078",  # Hereditary nephropathy, not elsewhere classified with other morphologic lesions
        "N079",  # Hereditary nephropathy, not elsewhere classified with unspecified morphologic lesions
        "N07A",  # Hereditary nephropathy, not elsewhere classified with C3 glomerulonephritis
        "N08",  # Glomerular disorders in diseases classified elsewhere
        "N140",  # Analgesic nephropathy
        "N141",  # Nephropathy induced by other drugs, medicaments and biological substances
        "N142",  # Nephropathy induced by unspecified drug, medicament or biological substance
        "N143",  # Nephropathy induced by heavy metals
        "N144",  # Toxic nephropathy, not elsewhere classified
        "N250",  # Renal osteodystrophy
        "N251",  # Nephrogenic diabetes insipidus
        "N2581",  # Secondary hyperparathyroidism of renal origin
        "N2589",  # Other disorders resulting from impaired renal tubular function
        "N259",  # Disorder resulting from impaired renal tubular function, unspecified
        "N261",  # Atrophy of kidney (terminal)
        "N262",  # Page kidney
        "N269",  # Renal sclerosis, unspecified
        "Q600",  # Renal agenesis, unilateral
        "Q601",  # Renal agenesis, bilateral
        "Q602",  # Renal agenesis, unspecified
        "Q603",  # Renal hypoplasia, unilateral
        "Q604",  # Renal hypoplasia, bilateral
        "Q605",  # Renal hypoplasia, unspecified
        "Q606",  # Potter's syndrome
        "Q6100",  # Congenital renal cyst, unspecified
        "Q6101",  # Congenital single renal cyst
        "Q6102",  # Congenital multiple renal cysts
        "Q6111",  # Cystic dilatation of collecting ducts
        "Q6119",  # Other polycystic kidney, infantile type
        "Q612",  # Polycystic kidney, adult type
        "Q613",  # Polycystic kidney, unspecified
        "Q614",  # Renal dysplasia
        "Q615",  # Medullary cystic kidney
        "Q618",  # Other cystic kidney diseases
        "Q619",  # Cystic kidney disease, unspecified
    }
    ICD9CM = {
        "5804",  # Acute glomerulonephritis with lesion of rapidly progressive glomerulonephritis
        "58081",  # Acute glomerulonephritis in diseases classified elsewhere
        "58089",  # Acute glomerulonephritis with other specified pathological lesion in kidney
        "5809",  # Acute glomerulonephritis with unspecified pathological lesion in kidney
        "5810",  # Nephrotic syndrome with lesion of proliferative glomerulonephritis
        "5811",  # Nephrotic syndrome with lesion of membranous glomerulonephritis
        "5812",  # Nephrotic syndrome with lesion of membranoproliferative glomerulonephritis
        "5813",  # Nephrotic syndrome with lesion of minimal change glomerulonephritis
        "58181",  # Nephrotic syndrome in diseases classified elsewhere
        "58189",  # Nephrotic syndrome with other specified pathological lesion in kidney
        "5819",  # Nephrotic syndrome with unspecified pathological lesion in kidney
        "5820",  # Chronic glomerulonephritis with lesion of proliferative glomerulonephritis
        "5821",  # Chronic glomerulonephritis with lesion of membranous glomerulonephritis
        "5822",  # Chronic glomerulonephritis with lesion of membranoproliferative glomerulonephritis
        "5824",  # Chronic glomerulonephritis with lesion of rapidly progressive glomerulonephritis
        "58281",  # Chronic glomerulonephritis in diseases classified elsewhere
        "58289",  # Chronic glomerulonephritis with other specified pathological lesion in kidney
        "5829",  # Chronic glomerulonephritis with unspecified pathological lesion in kidney
        "5830",  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of proliferative glomerulonephritis
        "5831",  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of membranous glomerulonephritis
        "5832",  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of membranoproliferative glomerulonephritis
        "5834",  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of rapidly progressive glomerulonephritis
        "5836",  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of renal cortical necrosis
        "5837",  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of renal medullary necrosis
        "58381",  # Nephritis and nephropathy, not specified as acute or chronic, in diseases classified elsewhere
        "58389",  # Nephritis and nephropathy, not specified as acute or chronic, with other specified pathological lesion in kidney
        "5839",  # Nephritis and nephropathy, not specified as acute or chronic, with unspecified pathological lesion in kidney
        "5880",  # Renal osteodystrophy
        "5881",  # Nephrogenic diabetes insipidus
        "58881",  # Secondary hyperparathyroidism (of renal origin)
        "58889",  # Other specified disorders resulting from impaired renal function
        "5889",  # Unspecified disorder resulting from impaired renal function
    }
    SNOMEDCT = {
        "1426004",  # Necrotizing glomerulonephritis (disorder)
        "1776003",  # Renal tubular acidosis (disorder)
        "3704008",  # Diffuse endocapillary proliferative glomerulonephritis (disorder)
        "7703001",  # Renal tubular defect (disorder)
        "7724006",  # Nephritic syndrome (disorder)
        "8199003",  # Focal embolic nephritis syndrome (disorder)
        "11480007",  # Idiopathic granulomatous interstitial nephropathy (disorder)
        "13335004",  # Sclerosing glomerulonephritis (disorder)
        "19351000",  # Acute glomerulonephritis (disorder)
        "20917003",  # Chronic glomerulonephritis (disorder)
        "22702000",  # Glomerulitis (disorder)
        "24790002",  # Proximal renal tubular acidosis (disorder)
        "25821008",  # Focal glomerular sclerosis (disorder)
        "26121002",  # Balkan nephropathy (disorder)
        "33763006",  # Hypercalcemic nephropathy (disorder)
        "35546006",  # Mesangial proliferative glomerulonephritis (disorder)
        "36171008",  # Glomerulonephritis (disorder)
        "37061001",  # Granulomatous sarcoid nephropathy (disorder)
        "37085009",  # Shunt nephritis (disorder)
        "40233000",  # Nephrotic-nephritic syndrome (disorder)
        "44785005",  # Minimal change disease (disorder)
        "45927004",  # Immunologic mediated tubulointerstitial nephritis (disorder)
        "48796009",  # Congenital nephrotic syndrome (disorder)
        "52254009",  # Nephrotic syndrome (disorder)
        "52845002",  # Nephritis (disorder)
        "53556002",  # Cis-platinum nephropathy (disorder)
        "54879000",  # Hypokalemic nephropathy (disorder)
        "55652009",  # Idiopathic crescentic glomerulonephritis, type III (disorder)
        "58574008",  # Acute nephropathy (disorder)
        "59400006",  # Analgesic nephropathy (disorder)
        "59479006",  # Mesangiocapillary glomerulonephritis, type II (disorder)
        "60926001",  # Chronic tubulointerstitial nephritis (disorder)
        "61165007",  # Hereditary nephrogenic diabetes insipidus (disorder)
        "61503006",  # Acute nephritis (disorder)
        "62240004",  # Benign arteriolar nephrosclerosis (disorder)
        "62332007",  # Infantile nephropathic cystinosis (disorder)
        "62853008",  # Anti-glomerular basement membrane tubulointerstitial nephritis (disorder)
        "64168005",  # Idiopathic crescentic glomerulonephritis, type I (disorder)
        "64212008",  # Diffuse crescentic glomerulonephritis (disorder)
        "68544003",  # Acute post-streptococcal glomerulonephritis (disorder)
        "68779003",  # Primary immunoglobulin A nephropathy (disorder)
        "73305009",  # Fibrillary glomerulonephritis (disorder)
        "75888001",  # Mesangiocapillary glomerulonephritis, type I (disorder)
        "77182004",  # Membranous glomerulonephritis (disorder)
        "78815005",  # Hereditary tubulointerstitial disorder (disorder)
        "79385002",  # Lowe syndrome (disorder)
        "80321008",  # Mesangiocapillary glomerulonephritis (disorder)
        "81363003",  # Malignant arteriolar nephrosclerosis (disorder)
        "81475007",  # Acquired nephrogenic diabetes insipidus (disorder)
        "82112005",  # Idiopathic chronic tubulointerstitial nephritis (disorder)
        "83866005",  # Focal AND segmental proliferative glomerulonephritis (disorder)
        "83923004",  # Familial interstitial nephritis (disorder)
        "90493000",  # Arteriolar nephrosclerosis (disorder)
        "110996009",  # Armanni-Ebstein kidney (disorder)
        "111395007",  # Nephrogenic diabetes insipidus (disorder)
        "118951003",  # Drug-induced nephrogenic diabetes insipidus (disorder)
        "123609007",  # Subacute glomerulonephritis (disorder)
        "123610002",  # Healed glomerulonephritis (disorder)
        "123752003",  # Immune-complex glomerulonephritis (disorder)
        "194909006",  # Acute pericarditis co-occurrent and due to uremia (disorder)
        "197579006",  # Acute proliferative glomerulonephritis (disorder)
        "197580009",  # Acute nephritis with lesions of necrotizing glomerulitis (disorder)
        "197582001",  # Acute glomerulonephritis associated with another disorder (disorder)
        "197584000",  # Acute focal nephritis (disorder)
        "197585004",  # Acute diffuse nephritis (disorder)
        "197589005",  # Nephrotic syndrome with proliferative glomerulonephritis (disorder)
        "197590001",  # Nephrotic syndrome with membranous glomerulonephritis (disorder)
        "197591002",  # Nephrotic syndrome with membranoproliferative glomerulonephritis (disorder)
        "197593004",  # Nephrotic syndrome, minor glomerular abnormality (disorder)
        "197594005",  # Nephrotic syndrome, focal and segmental glomerular lesions (disorder)
        "197595006",  # Nephrotic syndrome, diffuse membranous glomerulonephritis (disorder)
        "197596007",  # Nephrotic syndrome, diffuse mesangial proliferative glomerulonephritis (disorder)
        "197597003",  # Nephrotic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        "197598008",  # Nephrotic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        "197599000",  # Nephrotic syndrome, dense deposit disease (disorder)
        "197600002",  # Nephrotic syndrome, diffuse crescentic glomerulonephritis (disorder)
        "197601003",  # Finnish congenital nephrotic syndrome (disorder)
        "197603000",  # Nephrotic syndrome associated with another disorder (disorder)
        "197605007",  # Nephrotic syndrome due to diabetes mellitus (disorder)
        "197607004",  # Nephrotic syndrome in polyarteritis nodosa (disorder)
        "197613008",  # Chronic mesangial proliferative glomerulonephritis (disorder)
        "197616000",  # Chronic glomerulonephritis associated with another disorder (disorder)
        "197617009",  # Chronic exudative glomerulonephritis (disorder)
        "197618004",  # Chronic focal glomerulonephritis (disorder)
        "197619007",  # Chronic diffuse glomerulonephritis (disorder)
        "197626007",  # Focal membranoproliferative glomerulonephritis (disorder)
        "197629000",  # Anaphylactoid glomerulonephritis (disorder)
        "197661001",  # Glomerulosclerosis (disorder)
        "197663003",  # Impaired renal function disorder (disorder)
        "197664009",  # Phosphate-losing tubular disorders (disorder)
        "197671004",  # Renal function impairment with growth failure (disorder)
        "197679002",  # Glomerular disease (disorder)
        "197681000",  # Acute nephritic syndrome, minor glomerular abnormality (disorder)
        "197682007",  # Acute nephritic syndrome, focal and segmental glomerular lesions (disorder)
        "197683002",  # Acute nephritic syndrome, diffuse membranous glomerulonephritis (disorder)
        "197684008",  # Acute nephritic syndrome, diffuse mesangial proliferative glomerulonephritis (disorder)
        "197685009",  # Acute nephritic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        "197686005",  # Acute nephritic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        "197687001",  # Acute nephritic syndrome, dense deposit disease (disorder)
        "197688006",  # Acute nephritic syndrome, diffuse crescentic glomerulonephritis (disorder)
        "197689003",  # Rapidly progressive nephritic syndrome (disorder)
        "197690007",  # Rapidly progressive nephritic syndrome, minor glomerular abnormality (disorder)
        "197691006",  # Rapidly progressive nephritic syndrome, focal and segmental glomerular lesions (disorder)
        "197692004",  # Rapidly progressive nephritic syndrome, diffuse membranous glomerulonephritis (disorder)
        "197693009",  # Rapidly progressive nephritic syndrome, diffuse mesangial proliferative glomerulonephritis (disorder)
        "197694003",  # Rapidly progressive nephritic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        "197695002",  # Rapidly progressive nephritic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        "197696001",  # Rapidly progressive nephritic syndrome, dense deposit disease (disorder)
        "197697005",  # Rapidly progressive nephritic syndrome, diffuse crescentic glomerulonephritis (disorder)
        "197707007",  # Chronic nephritic syndrome (disorder)
        "197708002",  # Chronic nephritic syndrome, minor glomerular abnormality (disorder)
        "197709005",  # Chronic nephritic syndrome, focal and segmental glomerular lesions (disorder)
        "197712008",  # Chronic nephritic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        "197713003",  # Chronic nephritic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        "197714009",  # Chronic nephritic syndrome, dense deposit disease (disorder)
        "197715005",  # Chronic nephritic syndrome, diffuse crescentic glomerulonephritis (disorder)
        "197716006",  # Isolated proteinuria with specified morphological lesion (finding)
        "197717002",  # Isolated proteinuria with specified morphological lesion, minor glomerular abnormality (finding)
        "197719004",  # Isolated proteinuria with specified morphological lesion, focal and segmental glomerular lesions (finding)
        "197720005",  # Isolated proteinuria with specified morphological lesion, diffuse membranous glomerulonephritis (finding)
        "197721009",  # Isolated proteinuria with specified morphological lesion, diffuse mesangial proliferative glomerulonephritis (finding)
        "197722002",  # Isolated proteinuria with specified morphological lesion, diffuse endocapillary proliferative glomerulonephritis (finding)
        "197723007",  # Isolated proteinuria with specified morphological lesion, diffuse mesangiocapillary glomerulonephritis (finding)
        "197724001",  # Isolated proteinuria with specified morphological lesion, dense deposit disease (finding)
        "197725000",  # Isolated proteinuria with specified morphological lesion, diffuse concentric glomerulonephritis (finding)
        "197738008",  # Glomerular disorders in neoplastic diseases (disorder)
        "197739000",  # Glomerular disorders in blood diseases and disorders involving the immune mechanism (disorder)
        "197742006",  # Renal tubulo-interstitial disorders in neoplastic diseases (disorder)
        "197743001",  # Renal tubulo-interstitial disorders in blood diseases and disorders involving the immune mechanism (disorder)
        "197744007",  # Renal tubulo-interstitial disorders in metabolic diseases (disorder)
        "197745008",  # Renal tubulo-interstitial disorders in systemic connective tissue disorders (disorder)
        "197747000",  # Renal tubulo-interstitial disorders in transplant rejection (disorder)
        "197769007",  # Acute pyelonephritis with medullary necrosis (disorder)
        "234485006",  # Epstein syndrome (disorder)
        "236374007",  # Chronic infective interstitial nephritis (disorder)
        "236380004",  # Steroid-sensitive nephrotic syndrome (disorder)
        "236381000",  # Steroid-resistant nephrotic syndrome (disorder)
        "236382007",  # Steroid-dependent nephrotic syndrome (disorder)
        "236383002",  # Familial mesangial sclerosis (disorder)
        "236384008",  # Congenital nephrotic syndrome with focal glomerulosclerosis (disorder)
        "236385009",  # Drash syndrome (disorder)
        "236392004",  # Rapidly progressive glomerulonephritis (disorder)
        "236393009",  # Endocapillary glomerulonephritis (disorder)
        "236394003",  # Idiopathic endocapillary glomerulonephritis (disorder)
        "236395002",  # Post-infectious glomerulonephritis (disorder)
        "236397005",  # Post-infectious glomerulonephritis - Garland variety (disorder)
        "236398000",  # Crescentic glomerulonephritis (disorder)
        "236399008",  # Steroid-sensitive minimal change glomerulonephritis (disorder)
        "236400001",  # Steroid-resistant minimal change glomerulonephritis (disorder)
        "236401002",  # Steroid-dependent minimal change glomerulonephritis (disorder)
        "236402009",  # Glomerulopathy due to ischemia (disorder)
        "236403004",  # Focal segmental glomerulosclerosis (disorder)
        "236404005",  # Classical focal segmental glomerulosclerosis (disorder)
        "236405006",  # Hyperfiltration focal segmental glomerulosclerosis (disorder)
        "236407003",  # Immunoglobulin A nephropathy (disorder)
        "236409000",  # Mesangiocapillary glomerulonephritis type III (disorder)
        "236410005",  # Mesangiocapillary glomerulonephritis type IV (disorder)
        "236411009",  # Immunoglobulin M nephropathy (disorder)
        "236412002",  # C1q nephropathy (disorder)
        "236413007",  # Membranous glomerulonephritis - stage I (disorder)
        "236414001",  # Membranous glomerulonephritis - stage II (disorder)
        "236415000",  # Membranous glomerulonephritis - stage III (disorder)
        "236416004",  # Membranous glomerulonephritis - stage IV (disorder)
        "236417008",  # Membranous glomerulonephritis stage V (disorder)
        "236418003",  # Thin basement membrane disease (disorder)
        "236419006",  # Progressive hereditary glomerulonephritis without deafness (disorder)
        "236422008",  # Fechtner syndrome (disorder)
        "236452000",  # Chronic drug-induced tubulointerstitial nephritis (disorder)
        "236453005",  # Chronic tubulointerstitial nephritis caused by heavy metal (disorder)
        "236454004",  # Chronic lead nephropathy (disorder)
        "236455003",  # Saturnine nephropathy (disorder)
        "236456002",  # Chronic cadmium nephropathy (disorder)
        "236457006",  # Chronic mercury nephropathy (disorder)
        "236458001",  # Granulomatous tubulointerstitial nephritis (disorder)
        "236460004",  # Familial proximal renal tubular acidosis (disorder)
        "236461000",  # Distal renal tubular acidosis (disorder)
        "236463002",  # Hyperkalemic renal tubular acidosis (disorder)
        "236466005",  # Congenital Fanconi syndrome (disorder)
        "236479001",  # Magnesium-losing nephropathy (disorder)
        "236480003",  # Inherited magnesium-losing nephropathy (disorder)
        "236499007",  # Microalbuminuric nephropathy due to diabetes mellitus (disorder)
        "236500003",  # Proteinuric nephropathy due to diabetes mellitus (disorder)
        "236503001",  # Acute scleroderma renal crisis (disorder)
        "236506009",  # Goodpasture's disease (disorder)
        "236508005",  # Malignancy-associated glomerulonephritis (disorder)
        "236524006",  # Acute radiation nephritis (disorder)
        "236530006",  # Pulmonic stenosis and congenital nephrosis (disorder)
        "236532003",  # Renal tubular acidosis with progressive nerve deafness (disorder)
        "236535001",  # Glomerulopathy with giant fibrillar deposits (disorder)
        "236587002",  # Transplant glomerulopathy (disorder)
        "236588007",  # Transplant glomerulopathy - early form (disorder)
        "236589004",  # Transplant glomerulopathy - late form (disorder)
        "236590008",  # Cytomegalovirus-induced glomerulonephritis (disorder)
        "239932005",  # Primary pauci-immune necrotizing and crescentic glomerulonephritis (disorder)
        "253864004",  # Familial hypoplastic, glomerulocystic kidney (disorder)
        "266549004",  # Nephrotic syndrome with minimal change glomerulonephritis (disorder)
        "270494003",  # Acute papillary necrosis (disorder)
        "276585000",  # Congenital nephritis (disorder)
        "282364005",  # Immunoglobulin A nephropathy associated with liver disease (disorder)
        "307755009",  # Renal tubulo-interstitial disorder in systemic lupus erythematosus (disorder)
        "309426007",  # Glomerulopathy due to diabetes mellitus (disorder)
        "311366001",  # Kimmelstiel-Wilson syndrome (disorder)
        "359694003",  # Idiopathic crescentic glomerulonephritis, type II (disorder)
        "363234001",  # Nephrotic syndrome secondary to systemic disease (disorder)
        "370494002",  # Protein-losing nephropathy (disorder)
        "399094007",  # Familial hematuria (disorder)
        "399190000",  # Non-progressive hereditary glomerulonephritis (disorder)
        "399340005",  # Hereditary nephritis (disorder)
        "418839003",  # Tubulointerstitial nephritis with uveitis syndrome (disorder)
        "420514000",  # Persistent proteinuria due to type 1 diabetes mellitus (disorder)
        "420715001",  # Persistent microalbuminuria due to type 2 diabetes mellitus (disorder)
        "421305000",  # Persistent microalbuminuria due to type 1 diabetes mellitus (disorder)
        "421986006",  # Persistent proteinuria due to type 2 diabetes mellitus (disorder)
        "424188009",  # Idiopathic interstitial tubulonephritis (disorder)
        "425384007",  # Sarcoidosis with glomerulonephritis (disorder)
        "428255004",  # Tubulointerstitial nephritis (disorder)
        "439990003",  # Drug-induced tubulointerstitial nephritis (disorder)
        "440018001",  # Heavy-metal-induced tubulointerstitial nephritis (disorder)
        "441815006",  # Proliferative glomerulonephritis (disorder)
        "444977005",  # Autosomal dominant focal segmental glomerulosclerosis (disorder)
        "445170001",  # Macroalbuminuric nephropathy due to diabetes mellitus (disorder)
        "445258009",  # Idiopathic rapidly progressive glomerulonephritis (disorder)
        "445388002",  # Autosomal recessive focal segmental glomerulosclerosis (disorder)
        "445404003",  # Familial immunoglobulin A nephropathy (disorder)
        "446923008",  # Lipoprotein glomerulopathy (disorder)
        "700107006",  # Bartter syndrome antenatal type 1 (disorder)
        "700111000",  # Bartter syndrome type 3 (disorder)
        "700112007",  # Bartter syndrome type 4 (disorder)
        "702718005",  # Acute on chronic tubulointerstitial nephritis (disorder)
        "707742001",  # Bartter syndrome (disorder)
        "707755000",  # Familial hypokalemic and hypomagnesemic tubulopathy (disorder)
        "707756004",  # Gitelman syndrome (disorder)
        "717760006",  # Multi-drug resistant nephrotic syndrome (disorder)
        "717766000",  # Alport syndrome autosomal dominant (disorder)
        "717767009",  # Alport syndrome autosomal recessive (disorder)
        "717768004",  # Alport syndrome X-linked (disorder)
        "717791000",  # Bartter syndrome type 4a (disorder)
        "719839000",  # Tubular renal disease with cardiomyopathy syndrome (disorder)
        "720982007",  # Alport syndrome, intellectual disability, midface hypoplasia, elliptocytosis syndrome (disorder)
        "721175003",  # Idiopathic glomerular disease (disorder)
        "721207002",  # Seizure, sensorineural deafness, ataxia, intellectual disability, electrolyte imbalance syndrome (disorder)
        "721297008",  # Galloway Mowat syndrome (disorder)
        "721840000",  # Hyperuricemia, anemia, renal failure syndrome (disorder)
        "722086002",  # Membranous glomerulonephritis due to malignant neoplastic disease (disorder)
        "722118005",  # Congenital nephrotic syndrome due to congenital infection (disorder)
        "722119002",  # Idiopathic membranous glomerulonephritis (disorder)
        "722120008",  # Membranous glomerulonephritis caused by drug (disorder)
        "722139003",  # Focal segmental glomerulosclerosis caused by lithium (disorder)
        "722147003",  # Focal segmental glomerulosclerosis due to sickle cell disease (disorder)
        "722168002",  # Membranous glomerulonephritis co-occurrent with infectious disease (disorder)
        "722294004",  # Autosomal dominant intermediate Charcot-Marie-Tooth disease type E (disorder)
        "722369003",  # Congenital nephrotic syndrome due to diffuse mesangial sclerosis (disorder)
        "722381004",  # Congenital cataract, nephropathy, encephalopathy syndrome (disorder)
        "722433005",  # Dyschondrosteosis and nephritis syndrome (disorder)
        "722468005",  # Distal renal tubular acidosis co-occurrent with sensorineural deafness (disorder)
        "722758004",  # Complement component 3 glomerulopathy (disorder)
        "722759007",  # Glomerulopathy with fibronectin deposits 2 (disorder)
        "722760002",  # Dense deposit disease (disorder)
        "722761003",  # Complement component 3 glomerulonephritis (disorder)
        "722948009",  # Glomerular disorder due to non-neuropathic heredofamilial amyloidosis (disorder)
        "723074006",  # Renal papillary necrosis due to diabetes mellitus (disorder)
        "724099000",  # Acquired monoclonal immunoglobulin light chain-associated Fanconi syndrome (disorder)
        "725592009",  # Congenital membranous nephropathy due to maternal anti-neutral endopeptidase alloimmunization (disorder)
        "726082003",  # Immunotactoid glomerulonephritis (disorder)
        "726541005",  # Acute kidney injury due to trauma (disorder)
        "733453005",  # Congenital nephrotic syndrome, interstitial lung disease, epidermolysis bullosa syndrome (disorder)
        "766249007",  # Deafness, nephritis, anorectal malformation syndrome (disorder)
        "770414008",  # Alport syndrome (disorder)
        "771447009",  # Laminin subunit beta 2 related infantile-onset nephrotic syndrome (disorder)
        "773647007",  # Nephrotic syndrome, deafness, pretibial epidermolysis bullosa syndrome (disorder)
        "776416004",  # Hyperuricemia, pulmonary hypertension, renal failure, alkalosis syndrome (disorder)
        "782738008",  # Karyomegalic interstitial nephritis (disorder)
        "782771007",  # Mitochondrial deoxyribonucleic acid depletion syndrome hepatocerebrorenal form (disorder)
        "783157004",  # Leigh syndrome with nephrotic syndrome (disorder)
        "783614008",  # Familial steroid-resistant nephrotic syndrome with sensorineural deafness (disorder)
        "783620009",  # Dominant hypophosphatemia with nephrolithiasis and/or osteoporosis (disorder)
        "783787000",  # Retinal vasculopathy with cerebral leukoencephalopathy and systemic manifestations (disorder)
        "788613004",  # Anti-glomerular basement membrane disease (disorder)
        "792926007",  # Armanni-Ebstein kidney due to diabetes mellitus (disorder)
        "818952002",  # Fibronectin glomerulopathy (disorder)
        "829971006",  # Non-amyloid monoclonal immunoglobulin deposition disease (disorder)
        "838353009",  # Acute pyelonephritis caused by bacterium (disorder)
        "860892003",  # Glomerular disease due to infectious disease (disorder)
        "860893008",  # Glomerular disease due to parasitic disease (disorder)
        "12491000132101",  # Isolated proteinuria (finding)
        "12511000132108",  # Persistent proteinuria (finding)
        "15911000119103",  # Primary tubulointerstitial nephritis (disorder)
        "18521000119106",  # Microalbuminuria due to type 1 diabetes mellitus (disorder)
        "28191000119109",  # Chronic nephritic syndrome with membranous glomerulonephritis (disorder)
        "42231000009109",  # Bacterial nephritis (disorder)
        "50411000119108",  # Secondary tubulointerstitial nephritis (disorder)
        "71441000119104",  # Nephrotic syndrome due to type 2 diabetes mellitus (disorder)
        "71721000119101",  # Nephrotic syndrome due to type 1 diabetes mellitus (disorder)
        "90781000119102",  # Microalbuminuria due to type 2 diabetes mellitus (disorder)
        "90971000119103",  # Glomerulonephritis co-occurrent and due to vasculitis (disorder)
        "101711000119105",  # Glomerulonephritis co-occurrent and due to antineutrophil cytoplasmic antibody positive vasculitis (disorder)
        "157141000119108",  # Proteinuria due to type 2 diabetes mellitus (disorder)
        "243421000119104",  # Proteinuria due to type 1 diabetes mellitus (disorder)
        "320681000119102",  # Tubulointerstitial nephropathy due to Sjogren syndrome (disorder)
        "368901000119106",  # Chronic nephritic syndrome co-occurrent and due to membranoproliferative glomerulonephritis type III (disorder)
        "368921000119102",  # Nephritic syndrome co-occurrent and due to membranoproliferative glomerulonephritis type III (disorder)
    }


class HypertensiveChronicKidneyDisease(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of hypertensive chronic kidney disease.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis for hypertensive chronic kidney disease.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = "Hypertensive Chronic Kidney Disease"
    OID = "2.16.840.1.113883.3.464.1003.109.12.1017"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I120",  # Hypertensive chronic kidney disease with stage 5 chronic kidney disease or end stage renal disease
        "I129",  # Hypertensive chronic kidney disease with stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        "I130",  # Hypertensive heart and chronic kidney disease with heart failure and stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        "I1310",  # Hypertensive heart and chronic kidney disease without heart failure, with stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        "I1311",  # Hypertensive heart and chronic kidney disease without heart failure, with stage 5 chronic kidney disease, or end stage renal disease
        "I132",  # Hypertensive heart and chronic kidney disease with heart failure and with stage 5 chronic kidney disease, or end stage renal disease
        "I150",  # Renovascular hypertension
        "I151",  # Hypertension secondary to other renal disorders
    }
    ICD9CM = {
        "40301",  # Hypertensive chronic kidney disease, malignant, with chronic kidney disease stage V or end stage renal disease
        "40310",  # Hypertensive chronic kidney disease, benign, with chronic kidney disease stage I through stage IV, or unspecified
        "40311",  # Hypertensive chronic kidney disease, benign, with chronic kidney disease stage V or end stage renal disease
        "40390",  # Hypertensive chronic kidney disease, unspecified, with chronic kidney disease stage I through stage IV, or unspecified
        "40391",  # Hypertensive chronic kidney disease, unspecified, with chronic kidney disease stage V or end stage renal disease
        "40400",  # Hypertensive heart and chronic kidney disease, malignant, without heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        "40401",  # Hypertensive heart and chronic kidney disease, malignant, with heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        "40402",  # Hypertensive heart and chronic kidney disease, malignant, without heart failure and with chronic kidney disease stage V or end stage renal disease
        "40403",  # Hypertensive heart and chronic kidney disease, malignant, with heart failure and with chronic kidney disease stage V or end stage renal disease
        "40410",  # Hypertensive heart and chronic kidney disease, benign, without heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        "40411",  # Hypertensive heart and chronic kidney disease, benign, with heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        "40412",  # Hypertensive heart and chronic kidney disease, benign, without heart failure and with chronic kidney disease stage V or end stage renal disease
        "40413",  # Hypertensive heart and chronic kidney disease, benign, with heart failure and chronic kidney disease stage V or end stage renal disease
        "40490",  # Hypertensive heart and chronic kidney disease, unspecified, without heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        "40491",  # Hypertensive heart and chronic kidney disease, unspecified, with heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        "40492",  # Hypertensive heart and chronic kidney disease, unspecified, without heart failure and with chronic kidney disease stage V or end stage renal disease
        "40493",  # Hypertensive heart and chronic kidney disease, unspecified, with heart failure and chronic kidney disease stage V or end stage renal disease
    }
    SNOMEDCT = {
        "193003",  # Benign hypertensive renal disease (disorder)
        "14973001",  # Renal sclerosis with hypertension (disorder)
        "19769006",  # High-renin essential hypertension (disorder)
        "23130000",  # Paroxysmal hypertension (disorder)
        "28119000",  # Renal hypertension (disorder)
        "32916005",  # Nephrosclerosis (disorder)
        "38481006",  # Hypertensive renal disease (disorder)
        "39018007",  # Renal arterial hypertension (disorder)
        "45281005",  # Atherosclerosis of renal artery (disorder)
        "49220004",  # Hypertensive renal failure (disorder)
        "57684003",  # Parenchymal renal hypertension (disorder)
        "62240004",  # Benign arteriolar nephrosclerosis (disorder)
        "65443008",  # Malignant hypertensive renal disease (disorder)
        "66052004",  # Benign hypertensive heart AND renal disease (disorder)
        "66610008",  # Malignant hypertensive heart AND renal disease (disorder)
        "73410007",  # Benign secondary renovascular hypertension (disorder)
        "78544004",  # Chronic hypertensive uremia (disorder)
        "81363003",  # Malignant arteriolar nephrosclerosis (disorder)
        "86234004",  # Hypertensive heart AND renal disease (disorder)
        "90493000",  # Arteriolar nephrosclerosis (disorder)
        "111438007",  # Hypertension secondary to renal disease in obstetric context (disorder)
        "123799005",  # Renovascular hypertension (disorder)
        "123800009",  # Goldblatt hypertension (disorder)
        "194774006",  # Hypertensive renal disease with renal failure (disorder)
        "194783001",  # Secondary malignant renovascular hypertension (disorder)
        "302910002",  # Atherosclerotic renal artery stenosis (disorder)
        "397748008",  # Hypertension with albuminuria (disorder)
        "427889009",  # Hypertension associated with transplantation (disorder)
        "428575007",  # Hypertension secondary to kidney transplant (disorder)
        "473392002",  # Hypertensive nephrosclerosis (disorder)
        "285831000119108",  # Malignant hypertensive chronic kidney disease (disorder)
        "285841000119104",  # Malignant hypertensive end stage renal disease (disorder)
        "286371000119107",  # Malignant hypertensive end stage renal disease on dialysis (disorder)
        "431431000124100",  # Arteriosclerosis of renal artery (disorder)
    }


class KidneyFailure(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of kidney failure.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of kidney failure and kidney disease.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = "Kidney Failure"
    OID = "2.16.840.1.113883.3.464.1003.109.12.1028"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "N170",  # Acute kidney failure with tubular necrosis
        "N171",  # Acute kidney failure with acute cortical necrosis
        "N172",  # Acute kidney failure with medullary necrosis
        "N178",  # Other acute kidney failure
        "N179",  # Acute kidney failure, unspecified
        "N181",  # Chronic kidney disease, stage 1
        "N182",  # Chronic kidney disease, stage 2 (mild)
        "N183",  # Chronic kidney disease, stage 3 (moderate)
        "N1830",  # Chronic kidney disease, stage 3 unspecified
        "N1831",  # Chronic kidney disease, stage 3a
        "N1832",  # Chronic kidney disease, stage 3b
        "N184",  # Chronic kidney disease, stage 4 (severe)
        "N185",  # Chronic kidney disease, stage 5
        "N186",  # End stage renal disease
        "N189",  # Chronic kidney disease, unspecified
        "N19",  # Unspecified kidney failure
    }
    ICD9CM = {
        "5845",  # Acute kidney failure with lesion of tubular necrosis
        "5846",  # Acute kidney failure with lesion of renal cortical necrosis
        "5847",  # Acute kidney failure with lesion of renal medullary [papillary] necrosis
        "5848",  # Acute kidney failure with other specified pathological lesion in kidney
        "5849",  # Acute kidney failure, unspecified
        "5851",  # Chronic kidney disease, Stage I
        "5852",  # Chronic kidney disease, Stage II (mild)
        "5853",  # Chronic kidney disease, Stage III (moderate)
        "5854",  # Chronic kidney disease, Stage IV (severe)
        "5855",  # Chronic kidney disease, Stage V
        "5856",  # End stage renal disease
        "5859",  # Chronic kidney disease, unspecified
        "586",  # Renal failure, unspecified
        "587",  # Renal sclerosis, unspecified
    }
    SNOMEDCT = {
        "14669001",  # Acute renal failure syndrome (disorder)
        "46177005",  # End-stage renal disease (disorder)
        "236434000",  # End stage renal failure untreated by renal replacement therapy (disorder)
        "236435004",  # End stage renal failure on dialysis (disorder)
        "236436003",  # End stage renal failure with renal transplant (disorder)
        "429224003",  # Acute renal failure due to acute cortical necrosis (disorder)
        "431855005",  # Chronic kidney disease stage 1 (disorder)
        "431857002",  # Chronic kidney disease stage 4 (disorder)
        "433144002",  # Chronic kidney disease stage 3 (disorder)
        "433146000",  # Chronic kidney disease stage 5 (disorder)
        "700378005",  # Chronic kidney disease stage 3A (disorder)
        "700379002",  # Chronic kidney disease stage 3B (disorder)
        "722098007",  # Chronic kidney disease following donor nephrectomy (disorder)
        "722149000",  # Chronic kidney disease following excision of renal neoplasm (disorder)
        "722150000",  # Chronic kidney disease due to systemic infection (disorder)
        "722467000",  # Chronic kidney disease due to traumatic loss of kidney (disorder)
        "723190009",  # Chronic renal insufficiency (disorder)
        "721000119107",  # Chronic kidney disease stage 4 due to type 2 diabetes mellitus (disorder)
        "731000119105",  # Chronic kidney disease stage 3 due to type 2 diabetes mellitus (disorder)
        "90741000119107",  # Chronic kidney disease stage 3 due to type 1 diabetes mellitus (disorder)
        "90751000119109",  # Chronic kidney disease stage 4 due to type 1 diabetes mellitus (disorder)
        "96721000119103",  # Hypertensive heart AND chronic kidney disease stage 4 (disorder)
        "96741000119109",  # Hypertensive heart AND chronic kidney disease stage 2 (disorder)
        "129151000119102",  # Chronic kidney disease stage 4 due to hypertension (disorder)
        "129171000119106",  # Chronic kidney disease stage 3 due to hypertension (disorder)
        "129181000119109",  # Chronic kidney disease stage 2 due to hypertension (disorder)
        "140111000119107",  # Hypertension in chronic kidney disease stage 4 due to type 2 diabetes mellitus (disorder)
        "140131000119102",  # Hypertension in chronic kidney disease stage 2 due to type 2 diabetes mellitus (disorder)
        "145681000119101",  # Acute renal failure due to tubular necrosis (disorder)
        "284981000119102",  # Chronic kidney disease stage 2 due to benign hypertension (disorder)
        "285001000119105",  # Chronic kidney disease stage 4 due to benign hypertension (disorder)
        "285861000119100",  # Malignant hypertensive chronic kidney disease stage 2 (disorder)
        "285881000119109",  # Malignant hypertensive chronic kidney disease stage 4 (disorder)
    }


class Proteinuria(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  proteinuria.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of proteinuria.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = "Proteinuria"
    OID = "2.16.840.1.113883.3.526.3.1003"
    DEFINITION_VERSION = "20200307"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "R801",  # Persistent proteinuria, unspecified
        "R808",  # Other proteinuria
        "R809",  # Proteinuria, unspecified
    }
    SNOMEDCT = {
        "12178007",  # Nephrogenous proteinuria (finding)
        "29738008",  # Proteinuria (finding)
        "34993002",  # Mixed proteinuria (finding)
        "167582007",  # Urine microalbumin positive (finding)
        "263808002",  # Microproteinuria (finding)
        "264867001",  # Nephrotic range proteinuria (finding)
        "274769005",  # Albuminuria (finding)
        "312975006",  # Microalbuminuria (finding)
    }


class AllergyToAceInhibitorOrArb(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnosis of allergy to an angiotensin-converting enzyme (ACE) inhibitor and angiotensin-receptor blocker (ARB).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of allergy to an angiotensin-converting enzyme (ACE) inhibitor and angiotensin-receptor blocker (ARB).

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10
    """

    VALUE_SET_NAME = "Allergy to ACE Inhibitor or ARB"
    OID = "2.16.840.1.113883.3.526.3.1211"
    DEFINITION_VERSION = "20170908"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "609534005",  # Non-allergic hypersensitivity to losartan (finding)
        "609537003",  # Non-allergic hypersensitivity to angiotensin-converting enzyme inhibitor (finding)
        "609538008",  # Non-allergic hypersensitivity to captopril (finding)
        "609539000",  # Non-allergic hypersensitivity to cilazapril (finding)
        "609540003",  # Non-allergic hypersensitivity to enalapril (finding)
        "609541004",  # Non-allergic hypersensitivity to fosinopril (finding)
        "609542006",  # Non-allergic hypersensitivity to lisinopril (finding)
        "609543001",  # Non-allergic hypersensitivity to perindopril (finding)
        "609544007",  # Non-allergic hypersensitivity to quinapril (finding)
        "609545008",  # Non-allergic hypersensitivity to ramipril (finding)
        "609546009",  # Non-allergic hypersensitivity to trandolapril (finding)
        "609549002",  # Non-allergic hypersensitivity to angiotensin II receptor antagonist (finding)
        "609550002",  # History of angiotensin converting enzyme inhibitor pseudoallergy (situation)
    }


class HeartFailure(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of heart failure.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of heart failure.

    **Exclusion Criteria:** Excludes concepts that represent chronic and acute right-sided heart failure, right heart failure, right ventricular failure, cardiac insufficiency due to prosthesis and following cardiac surgery, or pulmonale.

    ** Used in:** CMS135v10, CMS144v10, CMS90v11
    """

    VALUE_SET_NAME = "Heart Failure"
    OID = "2.16.840.1.113883.3.526.3.376"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I110",  # Hypertensive heart disease with heart failure
        "I130",  # Hypertensive heart and chronic kidney disease with heart failure and stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        "I132",  # Hypertensive heart and chronic kidney disease with heart failure and with stage 5 chronic kidney disease, or end stage renal disease
        "I501",  # Left ventricular failure, unspecified
        "I5020",  # Unspecified systolic (congestive) heart failure
        "I5021",  # Acute systolic (congestive) heart failure
        "I5022",  # Chronic systolic (congestive) heart failure
        "I5023",  # Acute on chronic systolic (congestive) heart failure
        "I5030",  # Unspecified diastolic (congestive) heart failure
        "I5031",  # Acute diastolic (congestive) heart failure
        "I5032",  # Chronic diastolic (congestive) heart failure
        "I5033",  # Acute on chronic diastolic (congestive) heart failure
        "I5040",  # Unspecified combined systolic (congestive) and diastolic (congestive) heart failure
        "I5041",  # Acute combined systolic (congestive) and diastolic (congestive) heart failure
        "I5042",  # Chronic combined systolic (congestive) and diastolic (congestive) heart failure
        "I5043",  # Acute on chronic combined systolic (congestive) and diastolic (congestive) heart failure
        "I50814",  # Right heart failure due to left heart failure
        "I5082",  # Biventricular heart failure
        "I5083",  # High output heart failure
        "I5084",  # End stage heart failure
        "I5089",  # Other heart failure
        "I509",  # Heart failure, unspecified
    }
    SNOMEDCT = {
        "364006",  # Acute left-sided heart failure (disorder)
        "5148006",  # Hypertensive heart disease with congestive heart failure (disorder)
        "5375005",  # Chronic left-sided congestive heart failure (disorder)
        "10091002",  # High output heart failure (disorder)
        "10633002",  # Acute congestive heart failure (disorder)
        "25544003",  # Low output heart failure (disorder)
        "42343007",  # Congestive heart failure (disorder)
        "43736008",  # Rheumatic left ventricular failure (disorder)
        "44088000",  # Low cardiac output syndrome (disorder)
        "44313006",  # Right heart failure secondary to left heart failure (disorder)
        "46113002",  # Hypertensive heart failure (disorder)
        "48447003",  # Chronic heart failure (disorder)
        "56675007",  # Acute heart failure (disorder)
        "74960003",  # Acute left-sided congestive heart failure (disorder)
        "82523003",  # Congestive rheumatic heart failure (disorder)
        "83105008",  # Malignant hypertensive heart disease with congestive heart failure (disorder)
        "84114007",  # Heart failure (disorder)
        "85232009",  # Left heart failure (disorder)
        "88805009",  # Chronic congestive heart failure (disorder)
        "90727007",  # Pleural effusion due to congestive heart failure (disorder)
        "92506005",  # Biventricular congestive heart failure (disorder)
        "111283005",  # Chronic left-sided heart failure (disorder)
        "194767001",  # Benign hypertensive heart disease with congestive cardiac failure (disorder)
        "194779001",  # Hypertensive heart and renal disease with (congestive) heart failure (disorder)
        "194781004",  # Hypertensive heart and renal disease with both (congestive) heart failure and renal failure (disorder)
        "195111005",  # Decompensated cardiac failure (disorder)
        "195112003",  # Compensated cardiac failure (disorder)
        "195114002",  # Acute left ventricular failure (disorder)
        "206586007",  # Congenital cardiac failure (disorder)
        "233924009",  # Heart failure as a complication of care (disorder)
        "314206003",  # Refractory heart failure (disorder)
        "410431009",  # Cardiorespiratory failure (disorder)
        "417996009",  # Systolic heart failure (disorder)
        "418304008",  # Diastolic heart failure (disorder)
        "424404003",  # Decompensated chronic heart failure (disorder)
        "426263006",  # Congestive heart failure due to left ventricular systolic dysfunction (disorder)
        "426611007",  # Congestive heart failure due to valvular disease (disorder)
        "441481004",  # Chronic systolic heart failure (disorder)
        "441530006",  # Chronic diastolic heart failure (disorder)
        "443253003",  # Acute on chronic systolic heart failure (disorder)
        "443254009",  # Acute systolic heart failure (disorder)
        "443343001",  # Acute diastolic heart failure (disorder)
        "443344007",  # Acute on chronic diastolic heart failure (disorder)
        "471880001",  # Heart failure due to end stage congenital heart disease (disorder)
        "698594003",  # Symptomatic congestive heart failure (disorder)
        "703272007",  # Heart failure with reduced ejection fraction (disorder)
        "703273002",  # Heart failure with reduced ejection fraction due to coronary artery disease (disorder)
        "703274008",  # Heart failure with reduced ejection fraction due to myocarditis (disorder)
        "703275009",  # Heart failure with reduced ejection fraction due to cardiomyopathy (disorder)
        "703276005",  # Heart failure with reduced ejection fraction due to heart valve disease (disorder)
        "717840005",  # Congestive heart failure stage B (disorder)
        "871617000",  # Low output heart failure due to and following Fontan operation (disorder)
        "15781000119107",  # Hypertensive heart AND chronic kidney disease with congestive heart failure (disorder)
        "23341000119109",  # Congestive heart failure with right heart failure (disorder)
        "67431000119105",  # Congestive heart failure stage D (disorder)
        "67441000119101",  # Congestive heart failure stage C (disorder)
        "72481000119103",  # Congestive heart failure as early postoperative complication (disorder)
        "101281000119107",  # Congestive heart failure due to cardiomyopathy (disorder)
        "120851000119104",  # Systolic heart failure stage D (disorder)
        "120861000119102",  # Systolic heart failure stage C (disorder)
        "120871000119108",  # Systolic heart failure stage B (disorder)
        "120881000119106",  # Diastolic heart failure stage D (disorder)
        "120891000119109",  # Diastolic heart failure stage C (disorder)
        "120901000119108",  # Diastolic heart failure stage B (disorder)
        "153931000119109",  # Acute combined systolic and diastolic heart failure (disorder)
        "153941000119100",  # Chronic combined systolic and diastolic heart failure (disorder)
        "153951000119103",  # Acute on chronic combined systolic and diastolic heart failure (disorder)
        "15629541000119106",  # Congestive heart failure stage C due to ischemic cardiomyopathy (disorder)
        "15629591000119103",  # Congestive heart failure stage B due to ischemic cardiomyopathy (disorder)
        "15629641000119107",  # Systolic heart failure stage B due to ischemic cardiomyopathy (disorder)
        "15629741000119102",  # Systolic heart failure stage C due to ischemic cardiomyopathy (disorder)
        "15964701000119109",  # Acute cor pulmonale co-occurrent and due to saddle embolus of pulmonary artery (disorder)
    }


class IntoleranceToAceInhibitorOrArb(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses of intolerance to angiotensin-converting enzyme (ACE) inhibitor or angiotensin II receptor blockers (ARB).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of intolerance to ACE inhibitor or ARB.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10
    """

    VALUE_SET_NAME = "Intolerance to ACE Inhibitor or ARB"
    OID = "2.16.840.1.113883.3.526.3.1212"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "293500009",  # Angiotensin-converting-enzyme inhibitor adverse reaction (disorder)
        "293501008",  # Captopril adverse reaction (disorder)
        "293502001",  # Lisinopril adverse reaction (disorder)
        "293503006",  # Quinapril adverse reaction (disorder)
        "293504000",  # Ramipril adverse reaction (disorder)
        "293505004",  # Cilazapril adverse reaction (disorder)
        "293506003",  # Trandolapril adverse reaction (disorder)
        "293507007",  # Enalapril adverse reaction (disorder)
        "293508002",  # Fosinopril adverse reaction (disorder)
        "293509005",  # Perindopril adverse reaction (disorder)
        "293513003",  # Adverse reaction caused by losartan (disorder)
        "371627004",  # Angiotensin converting enzyme inhibitor-aggravated angioedema (disorder)
        "407564000",  # Angiotensin converting enzyme inhibitor not tolerated (situation)
        "407565004",  # Angiotensin II receptor antagonist not tolerated (situation)
        "407590002",  # Angiotensin II receptor antagonist adverse reaction (disorder)
        "422593004",  # Acute renal failure caused by angiotensin-converting-enzyme inhibitor (disorder)
    }


class ModerateOrSevereLvsd(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of moderate or severe left ventricular systolic dysfunction (LVSD).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of moderate or severe LVSD.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10, CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Moderate or Severe LVSD"
    OID = "2.16.840.1.113883.3.526.3.1090"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "981000124106",  # Moderate left ventricular systolic dysfunction (disorder)
        "991000124109",  # Severe left ventricular systolic dysfunction (disorder)
    }


class Pregnancy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of pregnancy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of pregnancy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10, CMS165v10, CMS155v10
    """

    VALUE_SET_NAME = "Pregnancy"
    OID = "2.16.840.1.113883.3.526.3.378"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "O0900",  # Supervision of pregnancy with history of infertility, unspecified trimester
        "O0901",  # Supervision of pregnancy with history of infertility, first trimester
        "O0902",  # Supervision of pregnancy with history of infertility, second trimester
        "O0903",  # Supervision of pregnancy with history of infertility, third trimester
        "O0910",  # Supervision of pregnancy with history of ectopic pregnancy, unspecified trimester
        "O0911",  # Supervision of pregnancy with history of ectopic pregnancy, first trimester
        "O0912",  # Supervision of pregnancy with history of ectopic pregnancy, second trimester
        "O0913",  # Supervision of pregnancy with history of ectopic pregnancy, third trimester
        "O09211",  # Supervision of pregnancy with history of pre-term labor, first trimester
        "O09212",  # Supervision of pregnancy with history of pre-term labor, second trimester
        "O09213",  # Supervision of pregnancy with history of pre-term labor, third trimester
        "O09219",  # Supervision of pregnancy with history of pre-term labor, unspecified trimester
        "O09291",  # Supervision of pregnancy with other poor reproductive or obstetric history, first trimester
        "O09292",  # Supervision of pregnancy with other poor reproductive or obstetric history, second trimester
        "O09293",  # Supervision of pregnancy with other poor reproductive or obstetric history, third trimester
        "O09299",  # Supervision of pregnancy with other poor reproductive or obstetric history, unspecified trimester
        "O0930",  # Supervision of pregnancy with insufficient antenatal care, unspecified trimester
        "O0931",  # Supervision of pregnancy with insufficient antenatal care, first trimester
        "O0932",  # Supervision of pregnancy with insufficient antenatal care, second trimester
        "O0933",  # Supervision of pregnancy with insufficient antenatal care, third trimester
        "O0940",  # Supervision of pregnancy with grand multiparity, unspecified trimester
        "O0941",  # Supervision of pregnancy with grand multiparity, first trimester
        "O0942",  # Supervision of pregnancy with grand multiparity, second trimester
        "O0943",  # Supervision of pregnancy with grand multiparity, third trimester
        "O09511",  # Supervision of elderly primigravida, first trimester
        "O09512",  # Supervision of elderly primigravida, second trimester
        "O09513",  # Supervision of elderly primigravida, third trimester
        "O09519",  # Supervision of elderly primigravida, unspecified trimester
        "O09521",  # Supervision of elderly multigravida, first trimester
        "O09522",  # Supervision of elderly multigravida, second trimester
        "O09523",  # Supervision of elderly multigravida, third trimester
        "O09529",  # Supervision of elderly multigravida, unspecified trimester
        "O09611",  # Supervision of young primigravida, first trimester
        "O09612",  # Supervision of young primigravida, second trimester
        "O09613",  # Supervision of young primigravida, third trimester
        "O09619",  # Supervision of young primigravida, unspecified trimester
        "O09621",  # Supervision of young multigravida, first trimester
        "O09622",  # Supervision of young multigravida, second trimester
        "O09623",  # Supervision of young multigravida, third trimester
        "O09629",  # Supervision of young multigravida, unspecified trimester
        "O0970",  # Supervision of high risk pregnancy due to social problems, unspecified trimester
        "O0971",  # Supervision of high risk pregnancy due to social problems, first trimester
        "O0972",  # Supervision of high risk pregnancy due to social problems, second trimester
        "O0973",  # Supervision of high risk pregnancy due to social problems, third trimester
        "O09811",  # Supervision of pregnancy resulting from assisted reproductive technology, first trimester
        "O09812",  # Supervision of pregnancy resulting from assisted reproductive technology, second trimester
        "O09813",  # Supervision of pregnancy resulting from assisted reproductive technology, third trimester
        "O09819",  # Supervision of pregnancy resulting from assisted reproductive technology, unspecified trimester
        "O09821",  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, first trimester
        "O09822",  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, second trimester
        "O09823",  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, third trimester
        "O09829",  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, unspecified trimester
        "O09891",  # Supervision of other high risk pregnancies, first trimester
        "O09892",  # Supervision of other high risk pregnancies, second trimester
        "O09893",  # Supervision of other high risk pregnancies, third trimester
        "O09899",  # Supervision of other high risk pregnancies, unspecified trimester
        "O0990",  # Supervision of high risk pregnancy, unspecified, unspecified trimester
        "O0991",  # Supervision of high risk pregnancy, unspecified, first trimester
        "O0992",  # Supervision of high risk pregnancy, unspecified, second trimester
        "O0993",  # Supervision of high risk pregnancy, unspecified, third trimester
        "O09A0",  # Supervision of pregnancy with history of molar pregnancy, unspecified trimester
        "O09A1",  # Supervision of pregnancy with history of molar pregnancy, first trimester
        "O09A2",  # Supervision of pregnancy with history of molar pregnancy, second trimester
        "O09A3",  # Supervision of pregnancy with history of molar pregnancy, third trimester
        "O10011",  # Pre-existing essential hypertension complicating pregnancy, first trimester
        "O10012",  # Pre-existing essential hypertension complicating pregnancy, second trimester
        "O10013",  # Pre-existing essential hypertension complicating pregnancy, third trimester
        "O10019",  # Pre-existing essential hypertension complicating pregnancy, unspecified trimester
        "O10111",  # Pre-existing hypertensive heart disease complicating pregnancy, first trimester
        "O10112",  # Pre-existing hypertensive heart disease complicating pregnancy, second trimester
        "O10113",  # Pre-existing hypertensive heart disease complicating pregnancy, third trimester
        "O10119",  # Pre-existing hypertensive heart disease complicating pregnancy, unspecified trimester
        "O10211",  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, first trimester
        "O10212",  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, second trimester
        "O10213",  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, third trimester
        "O10219",  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, unspecified trimester
        "O10311",  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, first trimester
        "O10312",  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, second trimester
        "O10313",  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, third trimester
        "O10319",  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, unspecified trimester
        "O10411",  # Pre-existing secondary hypertension complicating pregnancy, first trimester
        "O10412",  # Pre-existing secondary hypertension complicating pregnancy, second trimester
        "O10413",  # Pre-existing secondary hypertension complicating pregnancy, third trimester
        "O10419",  # Pre-existing secondary hypertension complicating pregnancy, unspecified trimester
        "O10911",  # Unspecified pre-existing hypertension complicating pregnancy, first trimester
        "O10912",  # Unspecified pre-existing hypertension complicating pregnancy, second trimester
        "O10913",  # Unspecified pre-existing hypertension complicating pregnancy, third trimester
        "O10919",  # Unspecified pre-existing hypertension complicating pregnancy, unspecified trimester
        "O111",  # Pre-existing hypertension with pre-eclampsia, first trimester
        "O112",  # Pre-existing hypertension with pre-eclampsia, second trimester
        "O113",  # Pre-existing hypertension with pre-eclampsia, third trimester
        "O119",  # Pre-existing hypertension with pre-eclampsia, unspecified trimester
        "O1200",  # Gestational edema, unspecified trimester
        "O1201",  # Gestational edema, first trimester
        "O1202",  # Gestational edema, second trimester
        "O1203",  # Gestational edema, third trimester
        "O1210",  # Gestational proteinuria, unspecified trimester
        "O1211",  # Gestational proteinuria, first trimester
        "O1212",  # Gestational proteinuria, second trimester
        "O1213",  # Gestational proteinuria, third trimester
        "O1220",  # Gestational edema with proteinuria, unspecified trimester
        "O1221",  # Gestational edema with proteinuria, first trimester
        "O1222",  # Gestational edema with proteinuria, second trimester
        "O1223",  # Gestational edema with proteinuria, third trimester
        "O131",  # Gestational [pregnancy-induced] hypertension without significant proteinuria, first trimester
        "O132",  # Gestational [pregnancy-induced] hypertension without significant proteinuria, second trimester
        "O133",  # Gestational [pregnancy-induced] hypertension without significant proteinuria, third trimester
        "O139",  # Gestational [pregnancy-induced] hypertension without significant proteinuria, unspecified trimester
        "O1400",  # Mild to moderate pre-eclampsia, unspecified trimester
        "O1402",  # Mild to moderate pre-eclampsia, second trimester
        "O1403",  # Mild to moderate pre-eclampsia, third trimester
        "O1410",  # Severe pre-eclampsia, unspecified trimester
        "O1412",  # Severe pre-eclampsia, second trimester
        "O1413",  # Severe pre-eclampsia, third trimester
        "O1420",  # HELLP syndrome (HELLP), unspecified trimester
        "O1422",  # HELLP syndrome (HELLP), second trimester
        "O1423",  # HELLP syndrome (HELLP), third trimester
        "O1490",  # Unspecified pre-eclampsia, unspecified trimester
        "O1492",  # Unspecified pre-eclampsia, second trimester
        "O1493",  # Unspecified pre-eclampsia, third trimester
        "O1500",  # Eclampsia complicating pregnancy, unspecified trimester
        "O1502",  # Eclampsia complicating pregnancy, second trimester
        "O1503",  # Eclampsia complicating pregnancy, third trimester
        "O161",  # Unspecified maternal hypertension, first trimester
        "O162",  # Unspecified maternal hypertension, second trimester
        "O163",  # Unspecified maternal hypertension, third trimester
        "O169",  # Unspecified maternal hypertension, unspecified trimester
        "O200",  # Threatened abortion
        "O208",  # Other hemorrhage in early pregnancy
        "O209",  # Hemorrhage in early pregnancy, unspecified
        "O210",  # Mild hyperemesis gravidarum
        "O211",  # Hyperemesis gravidarum with metabolic disturbance
        "O212",  # Late vomiting of pregnancy
        "O218",  # Other vomiting complicating pregnancy
        "O219",  # Vomiting of pregnancy, unspecified
        "O2200",  # Varicose veins of lower extremity in pregnancy, unspecified trimester
        "O2201",  # Varicose veins of lower extremity in pregnancy, first trimester
        "O2202",  # Varicose veins of lower extremity in pregnancy, second trimester
        "O2203",  # Varicose veins of lower extremity in pregnancy, third trimester
        "O2210",  # Genital varices in pregnancy, unspecified trimester
        "O2211",  # Genital varices in pregnancy, first trimester
        "O2212",  # Genital varices in pregnancy, second trimester
        "O2213",  # Genital varices in pregnancy, third trimester
        "O2220",  # Superficial thrombophlebitis in pregnancy, unspecified trimester
        "O2221",  # Superficial thrombophlebitis in pregnancy, first trimester
        "O2222",  # Superficial thrombophlebitis in pregnancy, second trimester
        "O2223",  # Superficial thrombophlebitis in pregnancy, third trimester
        "O2230",  # Deep phlebothrombosis in pregnancy, unspecified trimester
        "O2231",  # Deep phlebothrombosis in pregnancy, first trimester
        "O2232",  # Deep phlebothrombosis in pregnancy, second trimester
        "O2233",  # Deep phlebothrombosis in pregnancy, third trimester
        "O2240",  # Hemorrhoids in pregnancy, unspecified trimester
        "O2241",  # Hemorrhoids in pregnancy, first trimester
        "O2242",  # Hemorrhoids in pregnancy, second trimester
        "O2243",  # Hemorrhoids in pregnancy, third trimester
        "O2250",  # Cerebral venous thrombosis in pregnancy, unspecified trimester
        "O2251",  # Cerebral venous thrombosis in pregnancy, first trimester
        "O2252",  # Cerebral venous thrombosis in pregnancy, second trimester
        "O2253",  # Cerebral venous thrombosis in pregnancy, third trimester
        "O228X1",  # Other venous complications in pregnancy, first trimester
        "O228X2",  # Other venous complications in pregnancy, second trimester
        "O228X3",  # Other venous complications in pregnancy, third trimester
        "O228X9",  # Other venous complications in pregnancy, unspecified trimester
        "O2290",  # Venous complication in pregnancy, unspecified, unspecified trimester
        "O2291",  # Venous complication in pregnancy, unspecified, first trimester
        "O2292",  # Venous complication in pregnancy, unspecified, second trimester
        "O2293",  # Venous complication in pregnancy, unspecified, third trimester
        "O2300",  # Infections of kidney in pregnancy, unspecified trimester
        "O2301",  # Infections of kidney in pregnancy, first trimester
        "O2302",  # Infections of kidney in pregnancy, second trimester
        "O2303",  # Infections of kidney in pregnancy, third trimester
        "O2310",  # Infections of bladder in pregnancy, unspecified trimester
        "O2311",  # Infections of bladder in pregnancy, first trimester
        "O2312",  # Infections of bladder in pregnancy, second trimester
        "O2313",  # Infections of bladder in pregnancy, third trimester
        "O2320",  # Infections of urethra in pregnancy, unspecified trimester
        "O2321",  # Infections of urethra in pregnancy, first trimester
        "O2322",  # Infections of urethra in pregnancy, second trimester
        "O2323",  # Infections of urethra in pregnancy, third trimester
        "O2330",  # Infections of other parts of urinary tract in pregnancy, unspecified trimester
        "O2331",  # Infections of other parts of urinary tract in pregnancy, first trimester
        "O2332",  # Infections of other parts of urinary tract in pregnancy, second trimester
        "O2333",  # Infections of other parts of urinary tract in pregnancy, third trimester
        "O2340",  # Unspecified infection of urinary tract in pregnancy, unspecified trimester
        "O2341",  # Unspecified infection of urinary tract in pregnancy, first trimester
        "O2342",  # Unspecified infection of urinary tract in pregnancy, second trimester
        "O2343",  # Unspecified infection of urinary tract in pregnancy, third trimester
        "O23511",  # Infections of cervix in pregnancy, first trimester
        "O23512",  # Infections of cervix in pregnancy, second trimester
        "O23513",  # Infections of cervix in pregnancy, third trimester
        "O23519",  # Infections of cervix in pregnancy, unspecified trimester
        "O23521",  # Salpingo-oophoritis in pregnancy, first trimester
        "O23522",  # Salpingo-oophoritis in pregnancy, second trimester
        "O23523",  # Salpingo-oophoritis in pregnancy, third trimester
        "O23529",  # Salpingo-oophoritis in pregnancy, unspecified trimester
        "O23591",  # Infection of other part of genital tract in pregnancy, first trimester
        "O23592",  # Infection of other part of genital tract in pregnancy, second trimester
        "O23593",  # Infection of other part of genital tract in pregnancy, third trimester
        "O23599",  # Infection of other part of genital tract in pregnancy, unspecified trimester
        "O2390",  # Unspecified genitourinary tract infection in pregnancy, unspecified trimester
        "O2391",  # Unspecified genitourinary tract infection in pregnancy, first trimester
        "O2392",  # Unspecified genitourinary tract infection in pregnancy, second trimester
        "O2393",  # Unspecified genitourinary tract infection in pregnancy, third trimester
        "O24011",  # Pre-existing type 1 diabetes mellitus, in pregnancy, first trimester
        "O24012",  # Pre-existing type 1 diabetes mellitus, in pregnancy, second trimester
        "O24013",  # Pre-existing type 1 diabetes mellitus, in pregnancy, third trimester
        "O24019",  # Pre-existing type 1 diabetes mellitus, in pregnancy, unspecified trimester
        "O24111",  # Pre-existing type 2 diabetes mellitus, in pregnancy, first trimester
        "O24112",  # Pre-existing type 2 diabetes mellitus, in pregnancy, second trimester
        "O24113",  # Pre-existing type 2 diabetes mellitus, in pregnancy, third trimester
        "O24119",  # Pre-existing type 2 diabetes mellitus, in pregnancy, unspecified trimester
        "O24311",  # Unspecified pre-existing diabetes mellitus in pregnancy, first trimester
        "O24312",  # Unspecified pre-existing diabetes mellitus in pregnancy, second trimester
        "O24313",  # Unspecified pre-existing diabetes mellitus in pregnancy, third trimester
        "O24319",  # Unspecified pre-existing diabetes mellitus in pregnancy, unspecified trimester
        "O24410",  # Gestational diabetes mellitus in pregnancy, diet controlled
        "O24414",  # Gestational diabetes mellitus in pregnancy, insulin controlled
        "O24415",  # Gestational diabetes mellitus in pregnancy, controlled by oral hypoglycemic drugs
        "O24419",  # Gestational diabetes mellitus in pregnancy, unspecified control
        "O24811",  # Other pre-existing diabetes mellitus in pregnancy, first trimester
        "O24812",  # Other pre-existing diabetes mellitus in pregnancy, second trimester
        "O24813",  # Other pre-existing diabetes mellitus in pregnancy, third trimester
        "O24819",  # Other pre-existing diabetes mellitus in pregnancy, unspecified trimester
        "O24911",  # Unspecified diabetes mellitus in pregnancy, first trimester
        "O24912",  # Unspecified diabetes mellitus in pregnancy, second trimester
        "O24913",  # Unspecified diabetes mellitus in pregnancy, third trimester
        "O24919",  # Unspecified diabetes mellitus in pregnancy, unspecified trimester
        "O2510",  # Malnutrition in pregnancy, unspecified trimester
        "O2511",  # Malnutrition in pregnancy, first trimester
        "O2512",  # Malnutrition in pregnancy, second trimester
        "O2513",  # Malnutrition in pregnancy, third trimester
        "O2600",  # Excessive weight gain in pregnancy, unspecified trimester
        "O2601",  # Excessive weight gain in pregnancy, first trimester
        "O2602",  # Excessive weight gain in pregnancy, second trimester
        "O2603",  # Excessive weight gain in pregnancy, third trimester
        "O2610",  # Low weight gain in pregnancy, unspecified trimester
        "O2611",  # Low weight gain in pregnancy, first trimester
        "O2612",  # Low weight gain in pregnancy, second trimester
        "O2613",  # Low weight gain in pregnancy, third trimester
        "O2620",  # Pregnancy care for patient with recurrent pregnancy loss, unspecified trimester
        "O2621",  # Pregnancy care for patient with recurrent pregnancy loss, first trimester
        "O2622",  # Pregnancy care for patient with recurrent pregnancy loss, second trimester
        "O2623",  # Pregnancy care for patient with recurrent pregnancy loss, third trimester
        "O2630",  # Retained intrauterine contraceptive device in pregnancy, unspecified trimester
        "O2631",  # Retained intrauterine contraceptive device in pregnancy, first trimester
        "O2632",  # Retained intrauterine contraceptive device in pregnancy, second trimester
        "O2633",  # Retained intrauterine contraceptive device in pregnancy, third trimester
        "O2640",  # Herpes gestationis, unspecified trimester
        "O2641",  # Herpes gestationis, first trimester
        "O2642",  # Herpes gestationis, second trimester
        "O2643",  # Herpes gestationis, third trimester
        "O2650",  # Maternal hypotension syndrome, unspecified trimester
        "O2651",  # Maternal hypotension syndrome, first trimester
        "O2652",  # Maternal hypotension syndrome, second trimester
        "O2653",  # Maternal hypotension syndrome, third trimester
        "O26611",  # Liver and biliary tract disorders in pregnancy, first trimester
        "O26612",  # Liver and biliary tract disorders in pregnancy, second trimester
        "O26613",  # Liver and biliary tract disorders in pregnancy, third trimester
        "O26619",  # Liver and biliary tract disorders in pregnancy, unspecified trimester
        "O26711",  # Subluxation of symphysis (pubis) in pregnancy, first trimester
        "O26712",  # Subluxation of symphysis (pubis) in pregnancy, second trimester
        "O26713",  # Subluxation of symphysis (pubis) in pregnancy, third trimester
        "O26719",  # Subluxation of symphysis (pubis) in pregnancy, unspecified trimester
        "O26811",  # Pregnancy related exhaustion and fatigue, first trimester
        "O26812",  # Pregnancy related exhaustion and fatigue, second trimester
        "O26813",  # Pregnancy related exhaustion and fatigue, third trimester
        "O26819",  # Pregnancy related exhaustion and fatigue, unspecified trimester
        "O26821",  # Pregnancy related peripheral neuritis, first trimester
        "O26822",  # Pregnancy related peripheral neuritis, second trimester
        "O26823",  # Pregnancy related peripheral neuritis, third trimester
        "O26829",  # Pregnancy related peripheral neuritis, unspecified trimester
        "O26831",  # Pregnancy related renal disease, first trimester
        "O26832",  # Pregnancy related renal disease, second trimester
        "O26833",  # Pregnancy related renal disease, third trimester
        "O26839",  # Pregnancy related renal disease, unspecified trimester
        "O26841",  # Uterine size-date discrepancy, first trimester
        "O26842",  # Uterine size-date discrepancy, second trimester
        "O26843",  # Uterine size-date discrepancy, third trimester
        "O26849",  # Uterine size-date discrepancy, unspecified trimester
        "O26851",  # Spotting complicating pregnancy, first trimester
        "O26852",  # Spotting complicating pregnancy, second trimester
        "O26853",  # Spotting complicating pregnancy, third trimester
        "O26859",  # Spotting complicating pregnancy, unspecified trimester
        "O2686",  # Pruritic urticarial papules and plaques of pregnancy (PUPPP)
        "O26872",  # Cervical shortening, second trimester
        "O26873",  # Cervical shortening, third trimester
        "O26879",  # Cervical shortening, unspecified trimester
        "O26891",  # Other specified pregnancy related conditions, first trimester
        "O26892",  # Other specified pregnancy related conditions, second trimester
        "O26893",  # Other specified pregnancy related conditions, third trimester
        "O26899",  # Other specified pregnancy related conditions, unspecified trimester
        "O2690",  # Pregnancy related conditions, unspecified, unspecified trimester
        "O2691",  # Pregnancy related conditions, unspecified, first trimester
        "O2692",  # Pregnancy related conditions, unspecified, second trimester
        "O2693",  # Pregnancy related conditions, unspecified, third trimester
        "O280",  # Abnormal hematological finding on antenatal screening of mother
        "O281",  # Abnormal biochemical finding on antenatal screening of mother
        "O282",  # Abnormal cytological finding on antenatal screening of mother
        "O283",  # Abnormal ultrasonic finding on antenatal screening of mother
        "O284",  # Abnormal radiological finding on antenatal screening of mother
        "O285",  # Abnormal chromosomal and genetic finding on antenatal screening of mother
        "O288",  # Other abnormal findings on antenatal screening of mother
        "O289",  # Unspecified abnormal findings on antenatal screening of mother
        "O29011",  # Aspiration pneumonitis due to anesthesia during pregnancy, first trimester
        "O29012",  # Aspiration pneumonitis due to anesthesia during pregnancy, second trimester
        "O29013",  # Aspiration pneumonitis due to anesthesia during pregnancy, third trimester
        "O29019",  # Aspiration pneumonitis due to anesthesia during pregnancy, unspecified trimester
        "O29021",  # Pressure collapse of lung due to anesthesia during pregnancy, first trimester
        "O29022",  # Pressure collapse of lung due to anesthesia during pregnancy, second trimester
        "O29023",  # Pressure collapse of lung due to anesthesia during pregnancy, third trimester
        "O29029",  # Pressure collapse of lung due to anesthesia during pregnancy, unspecified trimester
        "O29091",  # Other pulmonary complications of anesthesia during pregnancy, first trimester
        "O29092",  # Other pulmonary complications of anesthesia during pregnancy, second trimester
        "O29093",  # Other pulmonary complications of anesthesia during pregnancy, third trimester
        "O29099",  # Other pulmonary complications of anesthesia during pregnancy, unspecified trimester
        "O29111",  # Cardiac arrest due to anesthesia during pregnancy, first trimester
        "O29112",  # Cardiac arrest due to anesthesia during pregnancy, second trimester
        "O29113",  # Cardiac arrest due to anesthesia during pregnancy, third trimester
        "O29119",  # Cardiac arrest due to anesthesia during pregnancy, unspecified trimester
        "O29121",  # Cardiac failure due to anesthesia during pregnancy, first trimester
        "O29122",  # Cardiac failure due to anesthesia during pregnancy, second trimester
        "O29123",  # Cardiac failure due to anesthesia during pregnancy, third trimester
        "O29129",  # Cardiac failure due to anesthesia during pregnancy, unspecified trimester
        "O29191",  # Other cardiac complications of anesthesia during pregnancy, first trimester
        "O29192",  # Other cardiac complications of anesthesia during pregnancy, second trimester
        "O29193",  # Other cardiac complications of anesthesia during pregnancy, third trimester
        "O29199",  # Other cardiac complications of anesthesia during pregnancy, unspecified trimester
        "O29211",  # Cerebral anoxia due to anesthesia during pregnancy, first trimester
        "O29212",  # Cerebral anoxia due to anesthesia during pregnancy, second trimester
        "O29213",  # Cerebral anoxia due to anesthesia during pregnancy, third trimester
        "O29219",  # Cerebral anoxia due to anesthesia during pregnancy, unspecified trimester
        "O29291",  # Other central nervous system complications of anesthesia during pregnancy, first trimester
        "O29292",  # Other central nervous system complications of anesthesia during pregnancy, second trimester
        "O29293",  # Other central nervous system complications of anesthesia during pregnancy, third trimester
        "O29299",  # Other central nervous system complications of anesthesia during pregnancy, unspecified trimester
        "O293X1",  # Toxic reaction to local anesthesia during pregnancy, first trimester
        "O293X2",  # Toxic reaction to local anesthesia during pregnancy, second trimester
        "O293X3",  # Toxic reaction to local anesthesia during pregnancy, third trimester
        "O293X9",  # Toxic reaction to local anesthesia during pregnancy, unspecified trimester
        "O2940",  # Spinal and epidural anesthesia induced headache during pregnancy, unspecified trimester
        "O2941",  # Spinal and epidural anesthesia induced headache during pregnancy, first trimester
        "O2942",  # Spinal and epidural anesthesia induced headache during pregnancy, second trimester
        "O2943",  # Spinal and epidural anesthesia induced headache during pregnancy, third trimester
        "O295X1",  # Other complications of spinal and epidural anesthesia during pregnancy, first trimester
        "O295X2",  # Other complications of spinal and epidural anesthesia during pregnancy, second trimester
        "O295X3",  # Other complications of spinal and epidural anesthesia during pregnancy, third trimester
        "O295X9",  # Other complications of spinal and epidural anesthesia during pregnancy, unspecified trimester
        "O2960",  # Failed or difficult intubation for anesthesia during pregnancy, unspecified trimester
        "O2961",  # Failed or difficult intubation for anesthesia during pregnancy, first trimester
        "O2962",  # Failed or difficult intubation for anesthesia during pregnancy, second trimester
        "O2963",  # Failed or difficult intubation for anesthesia during pregnancy, third trimester
        "O298X1",  # Other complications of anesthesia during pregnancy, first trimester
        "O298X2",  # Other complications of anesthesia during pregnancy, second trimester
        "O298X3",  # Other complications of anesthesia during pregnancy, third trimester
        "O298X9",  # Other complications of anesthesia during pregnancy, unspecified trimester
        "O2990",  # Unspecified complication of anesthesia during pregnancy, unspecified trimester
        "O2991",  # Unspecified complication of anesthesia during pregnancy, first trimester
        "O2992",  # Unspecified complication of anesthesia during pregnancy, second trimester
        "O2993",  # Unspecified complication of anesthesia during pregnancy, third trimester
        "O30001",  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        "O30002",  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        "O30003",  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        "O30009",  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        "O30011",  # Twin pregnancy, monochorionic/monoamniotic, first trimester
        "O30012",  # Twin pregnancy, monochorionic/monoamniotic, second trimester
        "O30013",  # Twin pregnancy, monochorionic/monoamniotic, third trimester
        "O30019",  # Twin pregnancy, monochorionic/monoamniotic, unspecified trimester
        "O30021",  # Conjoined twin pregnancy, first trimester
        "O30022",  # Conjoined twin pregnancy, second trimester
        "O30023",  # Conjoined twin pregnancy, third trimester
        "O30029",  # Conjoined twin pregnancy, unspecified trimester
        "O30031",  # Twin pregnancy, monochorionic/diamniotic, first trimester
        "O30032",  # Twin pregnancy, monochorionic/diamniotic, second trimester
        "O30033",  # Twin pregnancy, monochorionic/diamniotic, third trimester
        "O30039",  # Twin pregnancy, monochorionic/diamniotic, unspecified trimester
        "O30041",  # Twin pregnancy, dichorionic/diamniotic, first trimester
        "O30042",  # Twin pregnancy, dichorionic/diamniotic, second trimester
        "O30043",  # Twin pregnancy, dichorionic/diamniotic, third trimester
        "O30049",  # Twin pregnancy, dichorionic/diamniotic, unspecified trimester
        "O30091",  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, first trimester
        "O30092",  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, second trimester
        "O30093",  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, third trimester
        "O30099",  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        "O30101",  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        "O30102",  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        "O30103",  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        "O30109",  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        "O30111",  # Triplet pregnancy with two or more monochorionic fetuses, first trimester
        "O30112",  # Triplet pregnancy with two or more monochorionic fetuses, second trimester
        "O30113",  # Triplet pregnancy with two or more monochorionic fetuses, third trimester
        "O30119",  # Triplet pregnancy with two or more monochorionic fetuses, unspecified trimester
        "O30121",  # Triplet pregnancy with two or more monoamniotic fetuses, first trimester
        "O30122",  # Triplet pregnancy with two or more monoamniotic fetuses, second trimester
        "O30123",  # Triplet pregnancy with two or more monoamniotic fetuses, third trimester
        "O30129",  # Triplet pregnancy with two or more monoamniotic fetuses, unspecified trimester
        "O30131",  # Triplet pregnancy, trichorionic/triamniotic, first trimester
        "O30132",  # Triplet pregnancy, trichorionic/triamniotic, second trimester
        "O30133",  # Triplet pregnancy, trichorionic/triamniotic, third trimester
        "O30139",  # Triplet pregnancy, trichorionic/triamniotic, unspecified trimester
        "O30191",  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, first trimester
        "O30192",  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, second trimester
        "O30193",  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, third trimester
        "O30199",  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        "O30201",  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        "O30202",  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        "O30203",  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        "O30209",  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        "O30211",  # Quadruplet pregnancy with two or more monochorionic fetuses, first trimester
        "O30212",  # Quadruplet pregnancy with two or more monochorionic fetuses, second trimester
        "O30213",  # Quadruplet pregnancy with two or more monochorionic fetuses, third trimester
        "O30219",  # Quadruplet pregnancy with two or more monochorionic fetuses, unspecified trimester
        "O30221",  # Quadruplet pregnancy with two or more monoamniotic fetuses, first trimester
        "O30222",  # Quadruplet pregnancy with two or more monoamniotic fetuses, second trimester
        "O30223",  # Quadruplet pregnancy with two or more monoamniotic fetuses, third trimester
        "O30229",  # Quadruplet pregnancy with two or more monoamniotic fetuses, unspecified trimester
        "O30231",  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, first trimester
        "O30232",  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, second trimester
        "O30233",  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, third trimester
        "O30239",  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, unspecified trimester
        "O30291",  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, first trimester
        "O30292",  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, second trimester
        "O30293",  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, third trimester
        "O30299",  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        "O30801",  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        "O30802",  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        "O30803",  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        "O30809",  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        "O30811",  # Other specified multiple gestation with two or more monochorionic fetuses, first trimester
        "O30812",  # Other specified multiple gestation with two or more monochorionic fetuses, second trimester
        "O30813",  # Other specified multiple gestation with two or more monochorionic fetuses, third trimester
        "O30819",  # Other specified multiple gestation with two or more monochorionic fetuses, unspecified trimester
        "O30821",  # Other specified multiple gestation with two or more monoamniotic fetuses, first trimester
        "O30822",  # Other specified multiple gestation with two or more monoamniotic fetuses, second trimester
        "O30823",  # Other specified multiple gestation with two or more monoamniotic fetuses, third trimester
        "O30829",  # Other specified multiple gestation with two or more monoamniotic fetuses, unspecified trimester
        "O30831",  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, first trimester
        "O30832",  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, second trimester
        "O30833",  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, third trimester
        "O30839",  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, unspecified trimester
        "O30891",  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, first trimester
        "O30892",  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, second trimester
        "O30893",  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, third trimester
        "O30899",  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        "O3090",  # Multiple gestation, unspecified, unspecified trimester
        "O3091",  # Multiple gestation, unspecified, first trimester
        "O3092",  # Multiple gestation, unspecified, second trimester
        "O3093",  # Multiple gestation, unspecified, third trimester
        "O3100X0",  # Papyraceous fetus, unspecified trimester, not applicable or unspecified
        "O3100X1",  # Papyraceous fetus, unspecified trimester, fetus 1
        "O3100X2",  # Papyraceous fetus, unspecified trimester, fetus 2
        "O3100X3",  # Papyraceous fetus, unspecified trimester, fetus 3
        "O3100X4",  # Papyraceous fetus, unspecified trimester, fetus 4
        "O3100X5",  # Papyraceous fetus, unspecified trimester, fetus 5
        "O3100X9",  # Papyraceous fetus, unspecified trimester, other fetus
        "O3101X0",  # Papyraceous fetus, first trimester, not applicable or unspecified
        "O3101X1",  # Papyraceous fetus, first trimester, fetus 1
        "O3101X2",  # Papyraceous fetus, first trimester, fetus 2
        "O3101X3",  # Papyraceous fetus, first trimester, fetus 3
        "O3101X4",  # Papyraceous fetus, first trimester, fetus 4
        "O3101X5",  # Papyraceous fetus, first trimester, fetus 5
        "O3101X9",  # Papyraceous fetus, first trimester, other fetus
        "O3102X0",  # Papyraceous fetus, second trimester, not applicable or unspecified
        "O3102X1",  # Papyraceous fetus, second trimester, fetus 1
        "O3102X2",  # Papyraceous fetus, second trimester, fetus 2
        "O3102X3",  # Papyraceous fetus, second trimester, fetus 3
        "O3102X4",  # Papyraceous fetus, second trimester, fetus 4
        "O3102X5",  # Papyraceous fetus, second trimester, fetus 5
        "O3102X9",  # Papyraceous fetus, second trimester, other fetus
        "O3103X0",  # Papyraceous fetus, third trimester, not applicable or unspecified
        "O3103X1",  # Papyraceous fetus, third trimester, fetus 1
        "O3103X2",  # Papyraceous fetus, third trimester, fetus 2
        "O3103X3",  # Papyraceous fetus, third trimester, fetus 3
        "O3103X4",  # Papyraceous fetus, third trimester, fetus 4
        "O3103X5",  # Papyraceous fetus, third trimester, fetus 5
        "O3103X9",  # Papyraceous fetus, third trimester, other fetus
        "O3110X0",  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, not applicable or unspecified
        "O3110X1",  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 1
        "O3110X2",  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 2
        "O3110X3",  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 3
        "O3110X4",  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 4
        "O3110X5",  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 5
        "O3110X9",  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, other fetus
        "O3111X0",  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, not applicable or unspecified
        "O3111X1",  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 1
        "O3111X2",  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 2
        "O3111X3",  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 3
        "O3111X4",  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 4
        "O3111X5",  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 5
        "O3111X9",  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, other fetus
        "O3112X0",  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, not applicable or unspecified
        "O3112X1",  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 1
        "O3112X2",  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 2
        "O3112X3",  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 3
        "O3112X4",  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 4
        "O3112X5",  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 5
        "O3112X9",  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, other fetus
        "O3113X0",  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, not applicable or unspecified
        "O3113X1",  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 1
        "O3113X2",  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 2
        "O3113X3",  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 3
        "O3113X4",  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 4
        "O3113X5",  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 5
        "O3113X9",  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, other fetus
        "O3120X0",  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, not applicable or unspecified
        "O3120X1",  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 1
        "O3120X2",  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 2
        "O3120X3",  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 3
        "O3120X4",  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 4
        "O3120X5",  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 5
        "O3120X9",  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, other fetus
        "O3121X0",  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, not applicable or unspecified
        "O3121X1",  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 1
        "O3121X2",  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 2
        "O3121X3",  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 3
        "O3121X4",  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 4
        "O3121X5",  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 5
        "O3121X9",  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, other fetus
        "O3122X0",  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, not applicable or unspecified
        "O3122X1",  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 1
        "O3122X2",  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 2
        "O3122X3",  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 3
        "O3122X4",  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 4
        "O3122X5",  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 5
        "O3122X9",  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, other fetus
        "O3123X0",  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, not applicable or unspecified
        "O3123X1",  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 1
        "O3123X2",  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 2
        "O3123X3",  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 3
        "O3123X4",  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 4
        "O3123X5",  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 5
        "O3123X9",  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, other fetus
        "O3130X0",  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, not applicable or unspecified
        "O3130X1",  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 1
        "O3130X2",  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 2
        "O3130X3",  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 3
        "O3130X4",  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 4
        "O3130X5",  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 5
        "O3130X9",  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, other fetus
        "O3131X0",  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, not applicable or unspecified
        "O3131X1",  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 1
        "O3131X2",  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 2
        "O3131X3",  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 3
        "O3131X4",  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 4
        "O3131X5",  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 5
        "O3131X9",  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, other fetus
        "O3132X0",  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, not applicable or unspecified
        "O3132X1",  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 1
        "O3132X2",  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 2
        "O3132X3",  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 3
        "O3132X4",  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 4
        "O3132X5",  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 5
        "O3132X9",  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, other fetus
        "O3133X0",  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, not applicable or unspecified
        "O3133X1",  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 1
        "O3133X2",  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 2
        "O3133X3",  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 3
        "O3133X4",  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 4
        "O3133X5",  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 5
        "O3133X9",  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, other fetus
        "O318X10",  # Other complications specific to multiple gestation, first trimester, not applicable or unspecified
        "O318X11",  # Other complications specific to multiple gestation, first trimester, fetus 1
        "O318X12",  # Other complications specific to multiple gestation, first trimester, fetus 2
        "O318X13",  # Other complications specific to multiple gestation, first trimester, fetus 3
        "O318X14",  # Other complications specific to multiple gestation, first trimester, fetus 4
        "O318X15",  # Other complications specific to multiple gestation, first trimester, fetus 5
        "O318X19",  # Other complications specific to multiple gestation, first trimester, other fetus
        "O318X20",  # Other complications specific to multiple gestation, second trimester, not applicable or unspecified
        "O318X21",  # Other complications specific to multiple gestation, second trimester, fetus 1
        "O318X22",  # Other complications specific to multiple gestation, second trimester, fetus 2
        "O318X23",  # Other complications specific to multiple gestation, second trimester, fetus 3
        "O318X24",  # Other complications specific to multiple gestation, second trimester, fetus 4
        "O318X25",  # Other complications specific to multiple gestation, second trimester, fetus 5
        "O318X29",  # Other complications specific to multiple gestation, second trimester, other fetus
        "O318X30",  # Other complications specific to multiple gestation, third trimester, not applicable or unspecified
        "O318X31",  # Other complications specific to multiple gestation, third trimester, fetus 1
        "O318X32",  # Other complications specific to multiple gestation, third trimester, fetus 2
        "O318X33",  # Other complications specific to multiple gestation, third trimester, fetus 3
        "O318X34",  # Other complications specific to multiple gestation, third trimester, fetus 4
        "O318X35",  # Other complications specific to multiple gestation, third trimester, fetus 5
        "O318X39",  # Other complications specific to multiple gestation, third trimester, other fetus
        "O318X90",  # Other complications specific to multiple gestation, unspecified trimester, not applicable or unspecified
        "O318X91",  # Other complications specific to multiple gestation, unspecified trimester, fetus 1
        "O318X92",  # Other complications specific to multiple gestation, unspecified trimester, fetus 2
        "O318X93",  # Other complications specific to multiple gestation, unspecified trimester, fetus 3
        "O318X94",  # Other complications specific to multiple gestation, unspecified trimester, fetus 4
        "O318X95",  # Other complications specific to multiple gestation, unspecified trimester, fetus 5
        "O318X99",  # Other complications specific to multiple gestation, unspecified trimester, other fetus
        "O320XX0",  # Maternal care for unstable lie, not applicable or unspecified
        "O320XX1",  # Maternal care for unstable lie, fetus 1
        "O320XX2",  # Maternal care for unstable lie, fetus 2
        "O320XX3",  # Maternal care for unstable lie, fetus 3
        "O320XX4",  # Maternal care for unstable lie, fetus 4
        "O320XX5",  # Maternal care for unstable lie, fetus 5
        "O320XX9",  # Maternal care for unstable lie, other fetus
        "O321XX0",  # Maternal care for breech presentation, not applicable or unspecified
        "O321XX1",  # Maternal care for breech presentation, fetus 1
        "O321XX2",  # Maternal care for breech presentation, fetus 2
        "O321XX3",  # Maternal care for breech presentation, fetus 3
        "O321XX4",  # Maternal care for breech presentation, fetus 4
        "O321XX5",  # Maternal care for breech presentation, fetus 5
        "O321XX9",  # Maternal care for breech presentation, other fetus
        "O322XX0",  # Maternal care for transverse and oblique lie, not applicable or unspecified
        "O322XX1",  # Maternal care for transverse and oblique lie, fetus 1
        "O322XX2",  # Maternal care for transverse and oblique lie, fetus 2
        "O322XX3",  # Maternal care for transverse and oblique lie, fetus 3
        "O322XX4",  # Maternal care for transverse and oblique lie, fetus 4
        "O322XX5",  # Maternal care for transverse and oblique lie, fetus 5
        "O322XX9",  # Maternal care for transverse and oblique lie, other fetus
        "O323XX0",  # Maternal care for face, brow and chin presentation, not applicable or unspecified
        "O323XX1",  # Maternal care for face, brow and chin presentation, fetus 1
        "O323XX2",  # Maternal care for face, brow and chin presentation, fetus 2
        "O323XX3",  # Maternal care for face, brow and chin presentation, fetus 3
        "O323XX4",  # Maternal care for face, brow and chin presentation, fetus 4
        "O323XX5",  # Maternal care for face, brow and chin presentation, fetus 5
        "O323XX9",  # Maternal care for face, brow and chin presentation, other fetus
        "O324XX0",  # Maternal care for high head at term, not applicable or unspecified
        "O324XX1",  # Maternal care for high head at term, fetus 1
        "O324XX2",  # Maternal care for high head at term, fetus 2
        "O324XX3",  # Maternal care for high head at term, fetus 3
        "O324XX4",  # Maternal care for high head at term, fetus 4
        "O324XX5",  # Maternal care for high head at term, fetus 5
        "O324XX9",  # Maternal care for high head at term, other fetus
        "O326XX0",  # Maternal care for compound presentation, not applicable or unspecified
        "O326XX1",  # Maternal care for compound presentation, fetus 1
        "O326XX2",  # Maternal care for compound presentation, fetus 2
        "O326XX3",  # Maternal care for compound presentation, fetus 3
        "O326XX4",  # Maternal care for compound presentation, fetus 4
        "O326XX5",  # Maternal care for compound presentation, fetus 5
        "O326XX9",  # Maternal care for compound presentation, other fetus
        "O328XX0",  # Maternal care for other malpresentation of fetus, not applicable or unspecified
        "O328XX1",  # Maternal care for other malpresentation of fetus, fetus 1
        "O328XX2",  # Maternal care for other malpresentation of fetus, fetus 2
        "O328XX3",  # Maternal care for other malpresentation of fetus, fetus 3
        "O328XX4",  # Maternal care for other malpresentation of fetus, fetus 4
        "O328XX5",  # Maternal care for other malpresentation of fetus, fetus 5
        "O328XX9",  # Maternal care for other malpresentation of fetus, other fetus
        "O329XX0",  # Maternal care for malpresentation of fetus, unspecified, not applicable or unspecified
        "O329XX1",  # Maternal care for malpresentation of fetus, unspecified, fetus 1
        "O329XX2",  # Maternal care for malpresentation of fetus, unspecified, fetus 2
        "O329XX3",  # Maternal care for malpresentation of fetus, unspecified, fetus 3
        "O329XX4",  # Maternal care for malpresentation of fetus, unspecified, fetus 4
        "O329XX5",  # Maternal care for malpresentation of fetus, unspecified, fetus 5
        "O329XX9",  # Maternal care for malpresentation of fetus, unspecified, other fetus
        "O330",  # Maternal care for disproportion due to deformity of maternal pelvic bones
        "O331",  # Maternal care for disproportion due to generally contracted pelvis
        "O332",  # Maternal care for disproportion due to inlet contraction of pelvis
        "O333XX0",  # Maternal care for disproportion due to outlet contraction of pelvis, not applicable or unspecified
        "O333XX1",  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 1
        "O333XX2",  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 2
        "O333XX3",  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 3
        "O333XX4",  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 4
        "O333XX5",  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 5
        "O333XX9",  # Maternal care for disproportion due to outlet contraction of pelvis, other fetus
        "O334XX0",  # Maternal care for disproportion of mixed maternal and fetal origin, not applicable or unspecified
        "O334XX1",  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 1
        "O334XX2",  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 2
        "O334XX3",  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 3
        "O334XX4",  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 4
        "O334XX5",  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 5
        "O334XX9",  # Maternal care for disproportion of mixed maternal and fetal origin, other fetus
        "O335XX0",  # Maternal care for disproportion due to unusually large fetus, not applicable or unspecified
        "O335XX1",  # Maternal care for disproportion due to unusually large fetus, fetus 1
        "O335XX2",  # Maternal care for disproportion due to unusually large fetus, fetus 2
        "O335XX3",  # Maternal care for disproportion due to unusually large fetus, fetus 3
        "O335XX4",  # Maternal care for disproportion due to unusually large fetus, fetus 4
        "O335XX5",  # Maternal care for disproportion due to unusually large fetus, fetus 5
        "O335XX9",  # Maternal care for disproportion due to unusually large fetus, other fetus
        "O336XX0",  # Maternal care for disproportion due to hydrocephalic fetus, not applicable or unspecified
        "O336XX1",  # Maternal care for disproportion due to hydrocephalic fetus, fetus 1
        "O336XX2",  # Maternal care for disproportion due to hydrocephalic fetus, fetus 2
        "O336XX3",  # Maternal care for disproportion due to hydrocephalic fetus, fetus 3
        "O336XX4",  # Maternal care for disproportion due to hydrocephalic fetus, fetus 4
        "O336XX5",  # Maternal care for disproportion due to hydrocephalic fetus, fetus 5
        "O336XX9",  # Maternal care for disproportion due to hydrocephalic fetus, other fetus
        "O337XX0",  # Maternal care for disproportion due to other fetal deformities, not applicable or unspecified
        "O337XX1",  # Maternal care for disproportion due to other fetal deformities, fetus 1
        "O337XX2",  # Maternal care for disproportion due to other fetal deformities, fetus 2
        "O337XX3",  # Maternal care for disproportion due to other fetal deformities, fetus 3
        "O337XX4",  # Maternal care for disproportion due to other fetal deformities, fetus 4
        "O337XX5",  # Maternal care for disproportion due to other fetal deformities, fetus 5
        "O337XX9",  # Maternal care for disproportion due to other fetal deformities, other fetus
        "O338",  # Maternal care for disproportion of other origin
        "O339",  # Maternal care for disproportion, unspecified
        "O3400",  # Maternal care for unspecified congenital malformation of uterus, unspecified trimester
        "O3401",  # Maternal care for unspecified congenital malformation of uterus, first trimester
        "O3402",  # Maternal care for unspecified congenital malformation of uterus, second trimester
        "O3403",  # Maternal care for unspecified congenital malformation of uterus, third trimester
        "O3410",  # Maternal care for benign tumor of corpus uteri, unspecified trimester
        "O3411",  # Maternal care for benign tumor of corpus uteri, first trimester
        "O3412",  # Maternal care for benign tumor of corpus uteri, second trimester
        "O3413",  # Maternal care for benign tumor of corpus uteri, third trimester
        "O34211",  # Maternal care for low transverse scar from previous cesarean delivery
        "O34212",  # Maternal care for vertical scar from previous cesarean delivery
        "O34219",  # Maternal care for unspecified type scar from previous cesarean delivery
        "O3429",  # Maternal care due to uterine scar from other previous surgery
        "O3430",  # Maternal care for cervical incompetence, unspecified trimester
        "O3431",  # Maternal care for cervical incompetence, first trimester
        "O3432",  # Maternal care for cervical incompetence, second trimester
        "O3433",  # Maternal care for cervical incompetence, third trimester
        "O3440",  # Maternal care for other abnormalities of cervix, unspecified trimester
        "O3441",  # Maternal care for other abnormalities of cervix, first trimester
        "O3442",  # Maternal care for other abnormalities of cervix, second trimester
        "O3443",  # Maternal care for other abnormalities of cervix, third trimester
        "O34511",  # Maternal care for incarceration of gravid uterus, first trimester
        "O34512",  # Maternal care for incarceration of gravid uterus, second trimester
        "O34513",  # Maternal care for incarceration of gravid uterus, third trimester
        "O34519",  # Maternal care for incarceration of gravid uterus, unspecified trimester
        "O34521",  # Maternal care for prolapse of gravid uterus, first trimester
        "O34522",  # Maternal care for prolapse of gravid uterus, second trimester
        "O34523",  # Maternal care for prolapse of gravid uterus, third trimester
        "O34529",  # Maternal care for prolapse of gravid uterus, unspecified trimester
        "O34531",  # Maternal care for retroversion of gravid uterus, first trimester
        "O34532",  # Maternal care for retroversion of gravid uterus, second trimester
        "O34533",  # Maternal care for retroversion of gravid uterus, third trimester
        "O34539",  # Maternal care for retroversion of gravid uterus, unspecified trimester
        "O34591",  # Maternal care for other abnormalities of gravid uterus, first trimester
        "O34592",  # Maternal care for other abnormalities of gravid uterus, second trimester
        "O34593",  # Maternal care for other abnormalities of gravid uterus, third trimester
        "O34599",  # Maternal care for other abnormalities of gravid uterus, unspecified trimester
        "O3460",  # Maternal care for abnormality of vagina, unspecified trimester
        "O3461",  # Maternal care for abnormality of vagina, first trimester
        "O3462",  # Maternal care for abnormality of vagina, second trimester
        "O3463",  # Maternal care for abnormality of vagina, third trimester
        "O3470",  # Maternal care for abnormality of vulva and perineum, unspecified trimester
        "O3471",  # Maternal care for abnormality of vulva and perineum, first trimester
        "O3472",  # Maternal care for abnormality of vulva and perineum, second trimester
        "O3473",  # Maternal care for abnormality of vulva and perineum, third trimester
        "O3480",  # Maternal care for other abnormalities of pelvic organs, unspecified trimester
        "O3481",  # Maternal care for other abnormalities of pelvic organs, first trimester
        "O3482",  # Maternal care for other abnormalities of pelvic organs, second trimester
        "O3483",  # Maternal care for other abnormalities of pelvic organs, third trimester
        "O3490",  # Maternal care for abnormality of pelvic organ, unspecified, unspecified trimester
        "O3491",  # Maternal care for abnormality of pelvic organ, unspecified, first trimester
        "O3492",  # Maternal care for abnormality of pelvic organ, unspecified, second trimester
        "O3493",  # Maternal care for abnormality of pelvic organ, unspecified, third trimester
        "O350XX0",  # Maternal care for (suspected) central nervous system malformation in fetus, not applicable or unspecified
        "O350XX1",  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 1
        "O350XX2",  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 2
        "O350XX3",  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 3
        "O350XX4",  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 4
        "O350XX5",  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 5
        "O350XX9",  # Maternal care for (suspected) central nervous system malformation in fetus, other fetus
        "O351XX0",  # Maternal care for (suspected) chromosomal abnormality in fetus, not applicable or unspecified
        "O351XX1",  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 1
        "O351XX2",  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 2
        "O351XX3",  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 3
        "O351XX4",  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 4
        "O351XX5",  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 5
        "O351XX9",  # Maternal care for (suspected) chromosomal abnormality in fetus, other fetus
        "O352XX0",  # Maternal care for (suspected) hereditary disease in fetus, not applicable or unspecified
        "O352XX1",  # Maternal care for (suspected) hereditary disease in fetus, fetus 1
        "O352XX2",  # Maternal care for (suspected) hereditary disease in fetus, fetus 2
        "O352XX3",  # Maternal care for (suspected) hereditary disease in fetus, fetus 3
        "O352XX4",  # Maternal care for (suspected) hereditary disease in fetus, fetus 4
        "O352XX5",  # Maternal care for (suspected) hereditary disease in fetus, fetus 5
        "O352XX9",  # Maternal care for (suspected) hereditary disease in fetus, other fetus
        "O353XX0",  # Maternal care for (suspected) damage to fetus from viral disease in mother, not applicable or unspecified
        "O353XX1",  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 1
        "O353XX2",  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 2
        "O353XX3",  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 3
        "O353XX4",  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 4
        "O353XX5",  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 5
        "O353XX9",  # Maternal care for (suspected) damage to fetus from viral disease in mother, other fetus
        "O354XX0",  # Maternal care for (suspected) damage to fetus from alcohol, not applicable or unspecified
        "O354XX1",  # Maternal care for (suspected) damage to fetus from alcohol, fetus 1
        "O354XX2",  # Maternal care for (suspected) damage to fetus from alcohol, fetus 2
        "O354XX3",  # Maternal care for (suspected) damage to fetus from alcohol, fetus 3
        "O354XX4",  # Maternal care for (suspected) damage to fetus from alcohol, fetus 4
        "O354XX5",  # Maternal care for (suspected) damage to fetus from alcohol, fetus 5
        "O354XX9",  # Maternal care for (suspected) damage to fetus from alcohol, other fetus
        "O355XX0",  # Maternal care for (suspected) damage to fetus by drugs, not applicable or unspecified
        "O355XX1",  # Maternal care for (suspected) damage to fetus by drugs, fetus 1
        "O355XX2",  # Maternal care for (suspected) damage to fetus by drugs, fetus 2
        "O355XX3",  # Maternal care for (suspected) damage to fetus by drugs, fetus 3
        "O355XX4",  # Maternal care for (suspected) damage to fetus by drugs, fetus 4
        "O355XX5",  # Maternal care for (suspected) damage to fetus by drugs, fetus 5
        "O355XX9",  # Maternal care for (suspected) damage to fetus by drugs, other fetus
        "O356XX0",  # Maternal care for (suspected) damage to fetus by radiation, not applicable or unspecified
        "O356XX1",  # Maternal care for (suspected) damage to fetus by radiation, fetus 1
        "O356XX2",  # Maternal care for (suspected) damage to fetus by radiation, fetus 2
        "O356XX3",  # Maternal care for (suspected) damage to fetus by radiation, fetus 3
        "O356XX4",  # Maternal care for (suspected) damage to fetus by radiation, fetus 4
        "O356XX5",  # Maternal care for (suspected) damage to fetus by radiation, fetus 5
        "O356XX9",  # Maternal care for (suspected) damage to fetus by radiation, other fetus
        "O357XX0",  # Maternal care for (suspected) damage to fetus by other medical procedures, not applicable or unspecified
        "O357XX1",  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 1
        "O357XX2",  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 2
        "O357XX3",  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 3
        "O357XX4",  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 4
        "O357XX5",  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 5
        "O357XX9",  # Maternal care for (suspected) damage to fetus by other medical procedures, other fetus
        "O358XX0",  # Maternal care for other (suspected) fetal abnormality and damage, not applicable or unspecified
        "O358XX1",  # Maternal care for other (suspected) fetal abnormality and damage, fetus 1
        "O358XX2",  # Maternal care for other (suspected) fetal abnormality and damage, fetus 2
        "O358XX3",  # Maternal care for other (suspected) fetal abnormality and damage, fetus 3
        "O358XX4",  # Maternal care for other (suspected) fetal abnormality and damage, fetus 4
        "O358XX5",  # Maternal care for other (suspected) fetal abnormality and damage, fetus 5
        "O358XX9",  # Maternal care for other (suspected) fetal abnormality and damage, other fetus
        "O359XX0",  # Maternal care for (suspected) fetal abnormality and damage, unspecified, not applicable or unspecified
        "O359XX1",  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 1
        "O359XX2",  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 2
        "O359XX3",  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 3
        "O359XX4",  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 4
        "O359XX5",  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 5
        "O359XX9",  # Maternal care for (suspected) fetal abnormality and damage, unspecified, other fetus
        "O360110",  # Maternal care for anti-D [Rh] antibodies, first trimester, not applicable or unspecified
        "O360111",  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 1
        "O360112",  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 2
        "O360113",  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 3
        "O360114",  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 4
        "O360115",  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 5
        "O360119",  # Maternal care for anti-D [Rh] antibodies, first trimester, other fetus
        "O360120",  # Maternal care for anti-D [Rh] antibodies, second trimester, not applicable or unspecified
        "O360121",  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 1
        "O360122",  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 2
        "O360123",  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 3
        "O360124",  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 4
        "O360125",  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 5
        "O360129",  # Maternal care for anti-D [Rh] antibodies, second trimester, other fetus
        "O360130",  # Maternal care for anti-D [Rh] antibodies, third trimester, not applicable or unspecified
        "O360131",  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 1
        "O360132",  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 2
        "O360133",  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 3
        "O360134",  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 4
        "O360135",  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 5
        "O360139",  # Maternal care for anti-D [Rh] antibodies, third trimester, other fetus
        "O360190",  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, not applicable or unspecified
        "O360191",  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 1
        "O360192",  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 2
        "O360193",  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 3
        "O360194",  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 4
        "O360195",  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 5
        "O360199",  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, other fetus
        "O360910",  # Maternal care for other rhesus isoimmunization, first trimester, not applicable or unspecified
        "O360911",  # Maternal care for other rhesus isoimmunization, first trimester, fetus 1
        "O360912",  # Maternal care for other rhesus isoimmunization, first trimester, fetus 2
        "O360913",  # Maternal care for other rhesus isoimmunization, first trimester, fetus 3
        "O360914",  # Maternal care for other rhesus isoimmunization, first trimester, fetus 4
        "O360915",  # Maternal care for other rhesus isoimmunization, first trimester, fetus 5
        "O360919",  # Maternal care for other rhesus isoimmunization, first trimester, other fetus
        "O360920",  # Maternal care for other rhesus isoimmunization, second trimester, not applicable or unspecified
        "O360921",  # Maternal care for other rhesus isoimmunization, second trimester, fetus 1
        "O360922",  # Maternal care for other rhesus isoimmunization, second trimester, fetus 2
        "O360923",  # Maternal care for other rhesus isoimmunization, second trimester, fetus 3
        "O360924",  # Maternal care for other rhesus isoimmunization, second trimester, fetus 4
        "O360925",  # Maternal care for other rhesus isoimmunization, second trimester, fetus 5
        "O360929",  # Maternal care for other rhesus isoimmunization, second trimester, other fetus
        "O360930",  # Maternal care for other rhesus isoimmunization, third trimester, not applicable or unspecified
        "O360931",  # Maternal care for other rhesus isoimmunization, third trimester, fetus 1
        "O360932",  # Maternal care for other rhesus isoimmunization, third trimester, fetus 2
        "O360933",  # Maternal care for other rhesus isoimmunization, third trimester, fetus 3
        "O360934",  # Maternal care for other rhesus isoimmunization, third trimester, fetus 4
        "O360935",  # Maternal care for other rhesus isoimmunization, third trimester, fetus 5
        "O360939",  # Maternal care for other rhesus isoimmunization, third trimester, other fetus
        "O360990",  # Maternal care for other rhesus isoimmunization, unspecified trimester, not applicable or unspecified
        "O360991",  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 1
        "O360992",  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 2
        "O360993",  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 3
        "O360994",  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 4
        "O360995",  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 5
        "O360999",  # Maternal care for other rhesus isoimmunization, unspecified trimester, other fetus
        "O361110",  # Maternal care for Anti-A sensitization, first trimester, not applicable or unspecified
        "O361111",  # Maternal care for Anti-A sensitization, first trimester, fetus 1
        "O361112",  # Maternal care for Anti-A sensitization, first trimester, fetus 2
        "O361113",  # Maternal care for Anti-A sensitization, first trimester, fetus 3
        "O361114",  # Maternal care for Anti-A sensitization, first trimester, fetus 4
        "O361115",  # Maternal care for Anti-A sensitization, first trimester, fetus 5
        "O361119",  # Maternal care for Anti-A sensitization, first trimester, other fetus
        "O361120",  # Maternal care for Anti-A sensitization, second trimester, not applicable or unspecified
        "O361121",  # Maternal care for Anti-A sensitization, second trimester, fetus 1
        "O361122",  # Maternal care for Anti-A sensitization, second trimester, fetus 2
        "O361123",  # Maternal care for Anti-A sensitization, second trimester, fetus 3
        "O361124",  # Maternal care for Anti-A sensitization, second trimester, fetus 4
        "O361125",  # Maternal care for Anti-A sensitization, second trimester, fetus 5
        "O361129",  # Maternal care for Anti-A sensitization, second trimester, other fetus
        "O361130",  # Maternal care for Anti-A sensitization, third trimester, not applicable or unspecified
        "O361131",  # Maternal care for Anti-A sensitization, third trimester, fetus 1
        "O361132",  # Maternal care for Anti-A sensitization, third trimester, fetus 2
        "O361133",  # Maternal care for Anti-A sensitization, third trimester, fetus 3
        "O361134",  # Maternal care for Anti-A sensitization, third trimester, fetus 4
        "O361135",  # Maternal care for Anti-A sensitization, third trimester, fetus 5
        "O361139",  # Maternal care for Anti-A sensitization, third trimester, other fetus
        "O361190",  # Maternal care for Anti-A sensitization, unspecified trimester, not applicable or unspecified
        "O361191",  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 1
        "O361192",  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 2
        "O361193",  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 3
        "O361194",  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 4
        "O361195",  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 5
        "O361199",  # Maternal care for Anti-A sensitization, unspecified trimester, other fetus
        "O361910",  # Maternal care for other isoimmunization, first trimester, not applicable or unspecified
        "O361911",  # Maternal care for other isoimmunization, first trimester, fetus 1
        "O361912",  # Maternal care for other isoimmunization, first trimester, fetus 2
        "O361913",  # Maternal care for other isoimmunization, first trimester, fetus 3
        "O361914",  # Maternal care for other isoimmunization, first trimester, fetus 4
        "O361915",  # Maternal care for other isoimmunization, first trimester, fetus 5
        "O361919",  # Maternal care for other isoimmunization, first trimester, other fetus
        "O361920",  # Maternal care for other isoimmunization, second trimester, not applicable or unspecified
        "O361921",  # Maternal care for other isoimmunization, second trimester, fetus 1
        "O361922",  # Maternal care for other isoimmunization, second trimester, fetus 2
        "O361923",  # Maternal care for other isoimmunization, second trimester, fetus 3
        "O361924",  # Maternal care for other isoimmunization, second trimester, fetus 4
        "O361925",  # Maternal care for other isoimmunization, second trimester, fetus 5
        "O361929",  # Maternal care for other isoimmunization, second trimester, other fetus
        "O361930",  # Maternal care for other isoimmunization, third trimester, not applicable or unspecified
        "O361931",  # Maternal care for other isoimmunization, third trimester, fetus 1
        "O361932",  # Maternal care for other isoimmunization, third trimester, fetus 2
        "O361933",  # Maternal care for other isoimmunization, third trimester, fetus 3
        "O361934",  # Maternal care for other isoimmunization, third trimester, fetus 4
        "O361935",  # Maternal care for other isoimmunization, third trimester, fetus 5
        "O361939",  # Maternal care for other isoimmunization, third trimester, other fetus
        "O361990",  # Maternal care for other isoimmunization, unspecified trimester, not applicable or unspecified
        "O361991",  # Maternal care for other isoimmunization, unspecified trimester, fetus 1
        "O361992",  # Maternal care for other isoimmunization, unspecified trimester, fetus 2
        "O361993",  # Maternal care for other isoimmunization, unspecified trimester, fetus 3
        "O361994",  # Maternal care for other isoimmunization, unspecified trimester, fetus 4
        "O361995",  # Maternal care for other isoimmunization, unspecified trimester, fetus 5
        "O361999",  # Maternal care for other isoimmunization, unspecified trimester, other fetus
        "O3620X0",  # Maternal care for hydrops fetalis, unspecified trimester, not applicable or unspecified
        "O3620X1",  # Maternal care for hydrops fetalis, unspecified trimester, fetus 1
        "O3620X2",  # Maternal care for hydrops fetalis, unspecified trimester, fetus 2
        "O3620X3",  # Maternal care for hydrops fetalis, unspecified trimester, fetus 3
        "O3620X4",  # Maternal care for hydrops fetalis, unspecified trimester, fetus 4
        "O3620X5",  # Maternal care for hydrops fetalis, unspecified trimester, fetus 5
        "O3620X9",  # Maternal care for hydrops fetalis, unspecified trimester, other fetus
        "O3621X0",  # Maternal care for hydrops fetalis, first trimester, not applicable or unspecified
        "O3621X1",  # Maternal care for hydrops fetalis, first trimester, fetus 1
        "O3621X2",  # Maternal care for hydrops fetalis, first trimester, fetus 2
        "O3621X3",  # Maternal care for hydrops fetalis, first trimester, fetus 3
        "O3621X4",  # Maternal care for hydrops fetalis, first trimester, fetus 4
        "O3621X5",  # Maternal care for hydrops fetalis, first trimester, fetus 5
        "O3621X9",  # Maternal care for hydrops fetalis, first trimester, other fetus
        "O3622X0",  # Maternal care for hydrops fetalis, second trimester, not applicable or unspecified
        "O3622X1",  # Maternal care for hydrops fetalis, second trimester, fetus 1
        "O3622X2",  # Maternal care for hydrops fetalis, second trimester, fetus 2
        "O3622X3",  # Maternal care for hydrops fetalis, second trimester, fetus 3
        "O3622X4",  # Maternal care for hydrops fetalis, second trimester, fetus 4
        "O3622X5",  # Maternal care for hydrops fetalis, second trimester, fetus 5
        "O3622X9",  # Maternal care for hydrops fetalis, second trimester, other fetus
        "O3623X0",  # Maternal care for hydrops fetalis, third trimester, not applicable or unspecified
        "O3623X1",  # Maternal care for hydrops fetalis, third trimester, fetus 1
        "O3623X2",  # Maternal care for hydrops fetalis, third trimester, fetus 2
        "O3623X3",  # Maternal care for hydrops fetalis, third trimester, fetus 3
        "O3623X4",  # Maternal care for hydrops fetalis, third trimester, fetus 4
        "O3623X5",  # Maternal care for hydrops fetalis, third trimester, fetus 5
        "O3623X9",  # Maternal care for hydrops fetalis, third trimester, other fetus
        "O364XX0",  # Maternal care for intrauterine death, not applicable or unspecified
        "O364XX1",  # Maternal care for intrauterine death, fetus 1
        "O364XX2",  # Maternal care for intrauterine death, fetus 2
        "O364XX3",  # Maternal care for intrauterine death, fetus 3
        "O364XX4",  # Maternal care for intrauterine death, fetus 4
        "O364XX5",  # Maternal care for intrauterine death, fetus 5
        "O364XX9",  # Maternal care for intrauterine death, other fetus
        "O365110",  # Maternal care for known or suspected placental insufficiency, first trimester, not applicable or unspecified
        "O365111",  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 1
        "O365112",  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 2
        "O365113",  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 3
        "O365114",  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 4
        "O365115",  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 5
        "O365119",  # Maternal care for known or suspected placental insufficiency, first trimester, other fetus
        "O365120",  # Maternal care for known or suspected placental insufficiency, second trimester, not applicable or unspecified
        "O365121",  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 1
        "O365122",  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 2
        "O365123",  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 3
        "O365124",  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 4
        "O365125",  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 5
        "O365129",  # Maternal care for known or suspected placental insufficiency, second trimester, other fetus
        "O365130",  # Maternal care for known or suspected placental insufficiency, third trimester, not applicable or unspecified
        "O365131",  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 1
        "O365132",  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 2
        "O365133",  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 3
        "O365134",  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 4
        "O365135",  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 5
        "O365139",  # Maternal care for known or suspected placental insufficiency, third trimester, other fetus
        "O365190",  # Maternal care for known or suspected placental insufficiency, unspecified trimester, not applicable or unspecified
        "O365191",  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 1
        "O365192",  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 2
        "O365193",  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 3
        "O365194",  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 4
        "O365195",  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 5
        "O365199",  # Maternal care for known or suspected placental insufficiency, unspecified trimester, other fetus
        "O365910",  # Maternal care for other known or suspected poor fetal growth, first trimester, not applicable or unspecified
        "O365911",  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 1
        "O365912",  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 2
        "O365913",  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 3
        "O365914",  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 4
        "O365915",  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 5
        "O365919",  # Maternal care for other known or suspected poor fetal growth, first trimester, other fetus
        "O365920",  # Maternal care for other known or suspected poor fetal growth, second trimester, not applicable or unspecified
        "O365921",  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 1
        "O365922",  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 2
        "O365923",  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 3
        "O365924",  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 4
        "O365925",  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 5
        "O365929",  # Maternal care for other known or suspected poor fetal growth, second trimester, other fetus
        "O365930",  # Maternal care for other known or suspected poor fetal growth, third trimester, not applicable or unspecified
        "O365931",  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 1
        "O365932",  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 2
        "O365933",  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 3
        "O365934",  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 4
        "O365935",  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 5
        "O365939",  # Maternal care for other known or suspected poor fetal growth, third trimester, other fetus
        "O365990",  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, not applicable or unspecified
        "O365991",  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 1
        "O365992",  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 2
        "O365993",  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 3
        "O365994",  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 4
        "O365995",  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 5
        "O365999",  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, other fetus
        "O3660X0",  # Maternal care for excessive fetal growth, unspecified trimester, not applicable or unspecified
        "O3660X1",  # Maternal care for excessive fetal growth, unspecified trimester, fetus 1
        "O3660X2",  # Maternal care for excessive fetal growth, unspecified trimester, fetus 2
        "O3660X3",  # Maternal care for excessive fetal growth, unspecified trimester, fetus 3
        "O3660X4",  # Maternal care for excessive fetal growth, unspecified trimester, fetus 4
        "O3660X5",  # Maternal care for excessive fetal growth, unspecified trimester, fetus 5
        "O3660X9",  # Maternal care for excessive fetal growth, unspecified trimester, other fetus
        "O3661X0",  # Maternal care for excessive fetal growth, first trimester, not applicable or unspecified
        "O3661X1",  # Maternal care for excessive fetal growth, first trimester, fetus 1
        "O3661X2",  # Maternal care for excessive fetal growth, first trimester, fetus 2
        "O3661X3",  # Maternal care for excessive fetal growth, first trimester, fetus 3
        "O3661X4",  # Maternal care for excessive fetal growth, first trimester, fetus 4
        "O3661X5",  # Maternal care for excessive fetal growth, first trimester, fetus 5
        "O3661X9",  # Maternal care for excessive fetal growth, first trimester, other fetus
        "O3662X0",  # Maternal care for excessive fetal growth, second trimester, not applicable or unspecified
        "O3662X1",  # Maternal care for excessive fetal growth, second trimester, fetus 1
        "O3662X2",  # Maternal care for excessive fetal growth, second trimester, fetus 2
        "O3662X3",  # Maternal care for excessive fetal growth, second trimester, fetus 3
        "O3662X4",  # Maternal care for excessive fetal growth, second trimester, fetus 4
        "O3662X5",  # Maternal care for excessive fetal growth, second trimester, fetus 5
        "O3662X9",  # Maternal care for excessive fetal growth, second trimester, other fetus
        "O3663X0",  # Maternal care for excessive fetal growth, third trimester, not applicable or unspecified
        "O3663X1",  # Maternal care for excessive fetal growth, third trimester, fetus 1
        "O3663X2",  # Maternal care for excessive fetal growth, third trimester, fetus 2
        "O3663X3",  # Maternal care for excessive fetal growth, third trimester, fetus 3
        "O3663X4",  # Maternal care for excessive fetal growth, third trimester, fetus 4
        "O3663X5",  # Maternal care for excessive fetal growth, third trimester, fetus 5
        "O3663X9",  # Maternal care for excessive fetal growth, third trimester, other fetus
        "O3670X0",  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, not applicable or unspecified
        "O3670X1",  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 1
        "O3670X2",  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 2
        "O3670X3",  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 3
        "O3670X4",  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 4
        "O3670X5",  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 5
        "O3670X9",  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, other fetus
        "O3671X0",  # Maternal care for viable fetus in abdominal pregnancy, first trimester, not applicable or unspecified
        "O3671X1",  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 1
        "O3671X2",  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 2
        "O3671X3",  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 3
        "O3671X4",  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 4
        "O3671X5",  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 5
        "O3671X9",  # Maternal care for viable fetus in abdominal pregnancy, first trimester, other fetus
        "O3672X0",  # Maternal care for viable fetus in abdominal pregnancy, second trimester, not applicable or unspecified
        "O3672X1",  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 1
        "O3672X2",  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 2
        "O3672X3",  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 3
        "O3672X4",  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 4
        "O3672X5",  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 5
        "O3672X9",  # Maternal care for viable fetus in abdominal pregnancy, second trimester, other fetus
        "O3673X0",  # Maternal care for viable fetus in abdominal pregnancy, third trimester, not applicable or unspecified
        "O3673X1",  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 1
        "O3673X2",  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 2
        "O3673X3",  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 3
        "O3673X4",  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 4
        "O3673X5",  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 5
        "O3673X9",  # Maternal care for viable fetus in abdominal pregnancy, third trimester, other fetus
        "O368120",  # Decreased fetal movements, second trimester, not applicable or unspecified
        "O368121",  # Decreased fetal movements, second trimester, fetus 1
        "O368122",  # Decreased fetal movements, second trimester, fetus 2
        "O368123",  # Decreased fetal movements, second trimester, fetus 3
        "O368124",  # Decreased fetal movements, second trimester, fetus 4
        "O368125",  # Decreased fetal movements, second trimester, fetus 5
        "O368129",  # Decreased fetal movements, second trimester, other fetus
        "O368130",  # Decreased fetal movements, third trimester, not applicable or unspecified
        "O368131",  # Decreased fetal movements, third trimester, fetus 1
        "O368132",  # Decreased fetal movements, third trimester, fetus 2
        "O368133",  # Decreased fetal movements, third trimester, fetus 3
        "O368134",  # Decreased fetal movements, third trimester, fetus 4
        "O368135",  # Decreased fetal movements, third trimester, fetus 5
        "O368139",  # Decreased fetal movements, third trimester, other fetus
        "O368190",  # Decreased fetal movements, unspecified trimester, not applicable or unspecified
        "O368191",  # Decreased fetal movements, unspecified trimester, fetus 1
        "O368192",  # Decreased fetal movements, unspecified trimester, fetus 2
        "O368193",  # Decreased fetal movements, unspecified trimester, fetus 3
        "O368194",  # Decreased fetal movements, unspecified trimester, fetus 4
        "O368195",  # Decreased fetal movements, unspecified trimester, fetus 5
        "O368199",  # Decreased fetal movements, unspecified trimester, other fetus
        "O368210",  # Fetal anemia and thrombocytopenia, first trimester, not applicable or unspecified
        "O368211",  # Fetal anemia and thrombocytopenia, first trimester, fetus 1
        "O368212",  # Fetal anemia and thrombocytopenia, first trimester, fetus 2
        "O368213",  # Fetal anemia and thrombocytopenia, first trimester, fetus 3
        "O368214",  # Fetal anemia and thrombocytopenia, first trimester, fetus 4
        "O368215",  # Fetal anemia and thrombocytopenia, first trimester, fetus 5
        "O368219",  # Fetal anemia and thrombocytopenia, first trimester, other fetus
        "O368220",  # Fetal anemia and thrombocytopenia, second trimester, not applicable or unspecified
        "O368221",  # Fetal anemia and thrombocytopenia, second trimester, fetus 1
        "O368222",  # Fetal anemia and thrombocytopenia, second trimester, fetus 2
        "O368223",  # Fetal anemia and thrombocytopenia, second trimester, fetus 3
        "O368224",  # Fetal anemia and thrombocytopenia, second trimester, fetus 4
        "O368225",  # Fetal anemia and thrombocytopenia, second trimester, fetus 5
        "O368229",  # Fetal anemia and thrombocytopenia, second trimester, other fetus
        "O368230",  # Fetal anemia and thrombocytopenia, third trimester, not applicable or unspecified
        "O368231",  # Fetal anemia and thrombocytopenia, third trimester, fetus 1
        "O368232",  # Fetal anemia and thrombocytopenia, third trimester, fetus 2
        "O368233",  # Fetal anemia and thrombocytopenia, third trimester, fetus 3
        "O368234",  # Fetal anemia and thrombocytopenia, third trimester, fetus 4
        "O368235",  # Fetal anemia and thrombocytopenia, third trimester, fetus 5
        "O368239",  # Fetal anemia and thrombocytopenia, third trimester, other fetus
        "O368290",  # Fetal anemia and thrombocytopenia, unspecified trimester, not applicable or unspecified
        "O368291",  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 1
        "O368292",  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 2
        "O368293",  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 3
        "O368294",  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 4
        "O368295",  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 5
        "O368299",  # Fetal anemia and thrombocytopenia, unspecified trimester, other fetus
        "O368310",  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, not applicable or unspecified
        "O368311",  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 1
        "O368312",  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 2
        "O368313",  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 3
        "O368314",  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 4
        "O368315",  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 5
        "O368319",  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, other fetus
        "O368320",  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, not applicable or unspecified
        "O368321",  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 1
        "O368322",  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 2
        "O368323",  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 3
        "O368324",  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 4
        "O368325",  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 5
        "O368329",  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, other fetus
        "O368330",  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, not applicable or unspecified
        "O368331",  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 1
        "O368332",  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 2
        "O368333",  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 3
        "O368334",  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 4
        "O368335",  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 5
        "O368339",  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, other fetus
        "O368390",  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, not applicable or unspecified
        "O368391",  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 1
        "O368392",  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 2
        "O368393",  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 3
        "O368394",  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 4
        "O368395",  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 5
        "O368399",  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, other fetus
        "O368910",  # Maternal care for other specified fetal problems, first trimester, not applicable or unspecified
        "O368911",  # Maternal care for other specified fetal problems, first trimester, fetus 1
        "O368912",  # Maternal care for other specified fetal problems, first trimester, fetus 2
        "O368913",  # Maternal care for other specified fetal problems, first trimester, fetus 3
        "O368914",  # Maternal care for other specified fetal problems, first trimester, fetus 4
        "O368915",  # Maternal care for other specified fetal problems, first trimester, fetus 5
        "O368919",  # Maternal care for other specified fetal problems, first trimester, other fetus
        "O368920",  # Maternal care for other specified fetal problems, second trimester, not applicable or unspecified
        "O368921",  # Maternal care for other specified fetal problems, second trimester, fetus 1
        "O368922",  # Maternal care for other specified fetal problems, second trimester, fetus 2
        "O368923",  # Maternal care for other specified fetal problems, second trimester, fetus 3
        "O368924",  # Maternal care for other specified fetal problems, second trimester, fetus 4
        "O368925",  # Maternal care for other specified fetal problems, second trimester, fetus 5
        "O368929",  # Maternal care for other specified fetal problems, second trimester, other fetus
        "O368930",  # Maternal care for other specified fetal problems, third trimester, not applicable or unspecified
        "O368931",  # Maternal care for other specified fetal problems, third trimester, fetus 1
        "O368932",  # Maternal care for other specified fetal problems, third trimester, fetus 2
        "O368933",  # Maternal care for other specified fetal problems, third trimester, fetus 3
        "O368934",  # Maternal care for other specified fetal problems, third trimester, fetus 4
        "O368935",  # Maternal care for other specified fetal problems, third trimester, fetus 5
        "O368939",  # Maternal care for other specified fetal problems, third trimester, other fetus
        "O368990",  # Maternal care for other specified fetal problems, unspecified trimester, not applicable or unspecified
        "O368991",  # Maternal care for other specified fetal problems, unspecified trimester, fetus 1
        "O368992",  # Maternal care for other specified fetal problems, unspecified trimester, fetus 2
        "O368993",  # Maternal care for other specified fetal problems, unspecified trimester, fetus 3
        "O368994",  # Maternal care for other specified fetal problems, unspecified trimester, fetus 4
        "O368995",  # Maternal care for other specified fetal problems, unspecified trimester, fetus 5
        "O368999",  # Maternal care for other specified fetal problems, unspecified trimester, other fetus
        "O3690X0",  # Maternal care for fetal problem, unspecified, unspecified trimester, not applicable or unspecified
        "O3690X1",  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 1
        "O3690X2",  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 2
        "O3690X3",  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 3
        "O3690X4",  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 4
        "O3690X5",  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 5
        "O3690X9",  # Maternal care for fetal problem, unspecified, unspecified trimester, other fetus
        "O3691X0",  # Maternal care for fetal problem, unspecified, first trimester, not applicable or unspecified
        "O3691X1",  # Maternal care for fetal problem, unspecified, first trimester, fetus 1
        "O3691X2",  # Maternal care for fetal problem, unspecified, first trimester, fetus 2
        "O3691X3",  # Maternal care for fetal problem, unspecified, first trimester, fetus 3
        "O3691X4",  # Maternal care for fetal problem, unspecified, first trimester, fetus 4
        "O3691X5",  # Maternal care for fetal problem, unspecified, first trimester, fetus 5
        "O3691X9",  # Maternal care for fetal problem, unspecified, first trimester, other fetus
        "O3692X0",  # Maternal care for fetal problem, unspecified, second trimester, not applicable or unspecified
        "O3692X1",  # Maternal care for fetal problem, unspecified, second trimester, fetus 1
        "O3692X2",  # Maternal care for fetal problem, unspecified, second trimester, fetus 2
        "O3692X3",  # Maternal care for fetal problem, unspecified, second trimester, fetus 3
        "O3692X4",  # Maternal care for fetal problem, unspecified, second trimester, fetus 4
        "O3692X5",  # Maternal care for fetal problem, unspecified, second trimester, fetus 5
        "O3692X9",  # Maternal care for fetal problem, unspecified, second trimester, other fetus
        "O3693X0",  # Maternal care for fetal problem, unspecified, third trimester, not applicable or unspecified
        "O3693X1",  # Maternal care for fetal problem, unspecified, third trimester, fetus 1
        "O3693X2",  # Maternal care for fetal problem, unspecified, third trimester, fetus 2
        "O3693X3",  # Maternal care for fetal problem, unspecified, third trimester, fetus 3
        "O3693X4",  # Maternal care for fetal problem, unspecified, third trimester, fetus 4
        "O3693X5",  # Maternal care for fetal problem, unspecified, third trimester, fetus 5
        "O3693X9",  # Maternal care for fetal problem, unspecified, third trimester, other fetus
        "O401XX0",  # Polyhydramnios, first trimester, not applicable or unspecified
        "O401XX1",  # Polyhydramnios, first trimester, fetus 1
        "O401XX2",  # Polyhydramnios, first trimester, fetus 2
        "O401XX3",  # Polyhydramnios, first trimester, fetus 3
        "O401XX4",  # Polyhydramnios, first trimester, fetus 4
        "O401XX5",  # Polyhydramnios, first trimester, fetus 5
        "O401XX9",  # Polyhydramnios, first trimester, other fetus
        "O402XX0",  # Polyhydramnios, second trimester, not applicable or unspecified
        "O402XX1",  # Polyhydramnios, second trimester, fetus 1
        "O402XX2",  # Polyhydramnios, second trimester, fetus 2
        "O402XX3",  # Polyhydramnios, second trimester, fetus 3
        "O402XX4",  # Polyhydramnios, second trimester, fetus 4
        "O402XX5",  # Polyhydramnios, second trimester, fetus 5
        "O402XX9",  # Polyhydramnios, second trimester, other fetus
        "O403XX0",  # Polyhydramnios, third trimester, not applicable or unspecified
        "O403XX1",  # Polyhydramnios, third trimester, fetus 1
        "O403XX2",  # Polyhydramnios, third trimester, fetus 2
        "O403XX3",  # Polyhydramnios, third trimester, fetus 3
        "O403XX4",  # Polyhydramnios, third trimester, fetus 4
        "O403XX5",  # Polyhydramnios, third trimester, fetus 5
        "O403XX9",  # Polyhydramnios, third trimester, other fetus
        "O409XX0",  # Polyhydramnios, unspecified trimester, not applicable or unspecified
        "O409XX1",  # Polyhydramnios, unspecified trimester, fetus 1
        "O409XX2",  # Polyhydramnios, unspecified trimester, fetus 2
        "O409XX3",  # Polyhydramnios, unspecified trimester, fetus 3
        "O409XX4",  # Polyhydramnios, unspecified trimester, fetus 4
        "O409XX5",  # Polyhydramnios, unspecified trimester, fetus 5
        "O409XX9",  # Polyhydramnios, unspecified trimester, other fetus
        "O4100X0",  # Oligohydramnios, unspecified trimester, not applicable or unspecified
        "O4100X1",  # Oligohydramnios, unspecified trimester, fetus 1
        "O4100X2",  # Oligohydramnios, unspecified trimester, fetus 2
        "O4100X3",  # Oligohydramnios, unspecified trimester, fetus 3
        "O4100X4",  # Oligohydramnios, unspecified trimester, fetus 4
        "O4100X5",  # Oligohydramnios, unspecified trimester, fetus 5
        "O4100X9",  # Oligohydramnios, unspecified trimester, other fetus
        "O4101X0",  # Oligohydramnios, first trimester, not applicable or unspecified
        "O4101X1",  # Oligohydramnios, first trimester, fetus 1
        "O4101X2",  # Oligohydramnios, first trimester, fetus 2
        "O4101X3",  # Oligohydramnios, first trimester, fetus 3
        "O4101X4",  # Oligohydramnios, first trimester, fetus 4
        "O4101X5",  # Oligohydramnios, first trimester, fetus 5
        "O4101X9",  # Oligohydramnios, first trimester, other fetus
        "O4102X0",  # Oligohydramnios, second trimester, not applicable or unspecified
        "O4102X1",  # Oligohydramnios, second trimester, fetus 1
        "O4102X2",  # Oligohydramnios, second trimester, fetus 2
        "O4102X3",  # Oligohydramnios, second trimester, fetus 3
        "O4102X4",  # Oligohydramnios, second trimester, fetus 4
        "O4102X5",  # Oligohydramnios, second trimester, fetus 5
        "O4102X9",  # Oligohydramnios, second trimester, other fetus
        "O4103X0",  # Oligohydramnios, third trimester, not applicable or unspecified
        "O4103X1",  # Oligohydramnios, third trimester, fetus 1
        "O4103X2",  # Oligohydramnios, third trimester, fetus 2
        "O4103X3",  # Oligohydramnios, third trimester, fetus 3
        "O4103X4",  # Oligohydramnios, third trimester, fetus 4
        "O4103X5",  # Oligohydramnios, third trimester, fetus 5
        "O4103X9",  # Oligohydramnios, third trimester, other fetus
        "O411010",  # Infection of amniotic sac and membranes, unspecified, first trimester, not applicable or unspecified
        "O411011",  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 1
        "O411012",  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 2
        "O411013",  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 3
        "O411014",  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 4
        "O411015",  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 5
        "O411019",  # Infection of amniotic sac and membranes, unspecified, first trimester, other fetus
        "O411020",  # Infection of amniotic sac and membranes, unspecified, second trimester, not applicable or unspecified
        "O411021",  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 1
        "O411022",  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 2
        "O411023",  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 3
        "O411024",  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 4
        "O411025",  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 5
        "O411029",  # Infection of amniotic sac and membranes, unspecified, second trimester, other fetus
        "O411030",  # Infection of amniotic sac and membranes, unspecified, third trimester, not applicable or unspecified
        "O411031",  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 1
        "O411032",  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 2
        "O411033",  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 3
        "O411034",  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 4
        "O411035",  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 5
        "O411039",  # Infection of amniotic sac and membranes, unspecified, third trimester, other fetus
        "O411090",  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, not applicable or unspecified
        "O411091",  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 1
        "O411092",  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 2
        "O411093",  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 3
        "O411094",  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 4
        "O411095",  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 5
        "O411099",  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, other fetus
        "O411210",  # Chorioamnionitis, first trimester, not applicable or unspecified
        "O411211",  # Chorioamnionitis, first trimester, fetus 1
        "O411212",  # Chorioamnionitis, first trimester, fetus 2
        "O411213",  # Chorioamnionitis, first trimester, fetus 3
        "O411214",  # Chorioamnionitis, first trimester, fetus 4
        "O411215",  # Chorioamnionitis, first trimester, fetus 5
        "O411219",  # Chorioamnionitis, first trimester, other fetus
        "O411220",  # Chorioamnionitis, second trimester, not applicable or unspecified
        "O411221",  # Chorioamnionitis, second trimester, fetus 1
        "O411222",  # Chorioamnionitis, second trimester, fetus 2
        "O411223",  # Chorioamnionitis, second trimester, fetus 3
        "O411224",  # Chorioamnionitis, second trimester, fetus 4
        "O411225",  # Chorioamnionitis, second trimester, fetus 5
        "O411229",  # Chorioamnionitis, second trimester, other fetus
        "O411230",  # Chorioamnionitis, third trimester, not applicable or unspecified
        "O411231",  # Chorioamnionitis, third trimester, fetus 1
        "O411232",  # Chorioamnionitis, third trimester, fetus 2
        "O411233",  # Chorioamnionitis, third trimester, fetus 3
        "O411234",  # Chorioamnionitis, third trimester, fetus 4
        "O411235",  # Chorioamnionitis, third trimester, fetus 5
        "O411239",  # Chorioamnionitis, third trimester, other fetus
        "O411290",  # Chorioamnionitis, unspecified trimester, not applicable or unspecified
        "O411291",  # Chorioamnionitis, unspecified trimester, fetus 1
        "O411292",  # Chorioamnionitis, unspecified trimester, fetus 2
        "O411293",  # Chorioamnionitis, unspecified trimester, fetus 3
        "O411294",  # Chorioamnionitis, unspecified trimester, fetus 4
        "O411295",  # Chorioamnionitis, unspecified trimester, fetus 5
        "O411299",  # Chorioamnionitis, unspecified trimester, other fetus
        "O411410",  # Placentitis, first trimester, not applicable or unspecified
        "O411411",  # Placentitis, first trimester, fetus 1
        "O411412",  # Placentitis, first trimester, fetus 2
        "O411413",  # Placentitis, first trimester, fetus 3
        "O411414",  # Placentitis, first trimester, fetus 4
        "O411415",  # Placentitis, first trimester, fetus 5
        "O411419",  # Placentitis, first trimester, other fetus
        "O411420",  # Placentitis, second trimester, not applicable or unspecified
        "O411421",  # Placentitis, second trimester, fetus 1
        "O411422",  # Placentitis, second trimester, fetus 2
        "O411423",  # Placentitis, second trimester, fetus 3
        "O411424",  # Placentitis, second trimester, fetus 4
        "O411425",  # Placentitis, second trimester, fetus 5
        "O411429",  # Placentitis, second trimester, other fetus
        "O411430",  # Placentitis, third trimester, not applicable or unspecified
        "O411431",  # Placentitis, third trimester, fetus 1
        "O411432",  # Placentitis, third trimester, fetus 2
        "O411433",  # Placentitis, third trimester, fetus 3
        "O411434",  # Placentitis, third trimester, fetus 4
        "O411435",  # Placentitis, third trimester, fetus 5
        "O411439",  # Placentitis, third trimester, other fetus
        "O411490",  # Placentitis, unspecified trimester, not applicable or unspecified
        "O411491",  # Placentitis, unspecified trimester, fetus 1
        "O411492",  # Placentitis, unspecified trimester, fetus 2
        "O411493",  # Placentitis, unspecified trimester, fetus 3
        "O411494",  # Placentitis, unspecified trimester, fetus 4
        "O411495",  # Placentitis, unspecified trimester, fetus 5
        "O411499",  # Placentitis, unspecified trimester, other fetus
        "O418X10",  # Other specified disorders of amniotic fluid and membranes, first trimester, not applicable or unspecified
        "O418X11",  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 1
        "O418X12",  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 2
        "O418X13",  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 3
        "O418X14",  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 4
        "O418X15",  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 5
        "O418X19",  # Other specified disorders of amniotic fluid and membranes, first trimester, other fetus
        "O418X20",  # Other specified disorders of amniotic fluid and membranes, second trimester, not applicable or unspecified
        "O418X21",  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 1
        "O418X22",  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 2
        "O418X23",  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 3
        "O418X24",  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 4
        "O418X25",  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 5
        "O418X29",  # Other specified disorders of amniotic fluid and membranes, second trimester, other fetus
        "O418X30",  # Other specified disorders of amniotic fluid and membranes, third trimester, not applicable or unspecified
        "O418X31",  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 1
        "O418X32",  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 2
        "O418X33",  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 3
        "O418X34",  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 4
        "O418X35",  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 5
        "O418X39",  # Other specified disorders of amniotic fluid and membranes, third trimester, other fetus
        "O418X90",  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, not applicable or unspecified
        "O418X91",  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 1
        "O418X92",  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 2
        "O418X93",  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 3
        "O418X94",  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 4
        "O418X95",  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 5
        "O418X99",  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, other fetus
        "O4190X0",  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, not applicable or unspecified
        "O4190X1",  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 1
        "O4190X2",  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 2
        "O4190X3",  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 3
        "O4190X4",  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 4
        "O4190X5",  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 5
        "O4190X9",  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, other fetus
        "O4191X0",  # Disorder of amniotic fluid and membranes, unspecified, first trimester, not applicable or unspecified
        "O4191X1",  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 1
        "O4191X2",  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 2
        "O4191X3",  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 3
        "O4191X4",  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 4
        "O4191X5",  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 5
        "O4191X9",  # Disorder of amniotic fluid and membranes, unspecified, first trimester, other fetus
        "O4192X0",  # Disorder of amniotic fluid and membranes, unspecified, second trimester, not applicable or unspecified
        "O4192X1",  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 1
        "O4192X2",  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 2
        "O4192X3",  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 3
        "O4192X4",  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 4
        "O4192X5",  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 5
        "O4192X9",  # Disorder of amniotic fluid and membranes, unspecified, second trimester, other fetus
        "O4193X0",  # Disorder of amniotic fluid and membranes, unspecified, third trimester, not applicable or unspecified
        "O4193X1",  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 1
        "O4193X2",  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 2
        "O4193X3",  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 3
        "O4193X4",  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 4
        "O4193X5",  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 5
        "O4193X9",  # Disorder of amniotic fluid and membranes, unspecified, third trimester, other fetus
        "O4200",  # Premature rupture of membranes, onset of labor within 24 hours of rupture, unspecified weeks of gestation
        "O42011",  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, first trimester
        "O42012",  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, second trimester
        "O42013",  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, third trimester
        "O42019",  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, unspecified trimester
        "O4202",  # Full-term premature rupture of membranes, onset of labor within 24 hours of rupture
        "O4210",  # Premature rupture of membranes, onset of labor more than 24 hours following rupture, unspecified weeks of gestation
        "O42111",  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, first trimester
        "O42112",  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, second trimester
        "O42113",  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, third trimester
        "O42119",  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, unspecified trimester
        "O4212",  # Full-term premature rupture of membranes, onset of labor more than 24 hours following rupture
        "O4290",  # Premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, unspecified weeks of gestation
        "O42911",  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, first trimester
        "O42912",  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, second trimester
        "O42913",  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, third trimester
        "O42919",  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, unspecified trimester
        "O4292",  # Full-term premature rupture of membranes, unspecified as to length of time between rupture and onset of labor
        "O43011",  # Fetomaternal placental transfusion syndrome, first trimester
        "O43012",  # Fetomaternal placental transfusion syndrome, second trimester
        "O43013",  # Fetomaternal placental transfusion syndrome, third trimester
        "O43019",  # Fetomaternal placental transfusion syndrome, unspecified trimester
        "O43021",  # Fetus-to-fetus placental transfusion syndrome, first trimester
        "O43022",  # Fetus-to-fetus placental transfusion syndrome, second trimester
        "O43023",  # Fetus-to-fetus placental transfusion syndrome, third trimester
        "O43029",  # Fetus-to-fetus placental transfusion syndrome, unspecified trimester
        "O43101",  # Malformation of placenta, unspecified, first trimester
        "O43102",  # Malformation of placenta, unspecified, second trimester
        "O43103",  # Malformation of placenta, unspecified, third trimester
        "O43109",  # Malformation of placenta, unspecified, unspecified trimester
        "O43111",  # Circumvallate placenta, first trimester
        "O43112",  # Circumvallate placenta, second trimester
        "O43113",  # Circumvallate placenta, third trimester
        "O43119",  # Circumvallate placenta, unspecified trimester
        "O43121",  # Velamentous insertion of umbilical cord, first trimester
        "O43122",  # Velamentous insertion of umbilical cord, second trimester
        "O43123",  # Velamentous insertion of umbilical cord, third trimester
        "O43129",  # Velamentous insertion of umbilical cord, unspecified trimester
        "O43191",  # Other malformation of placenta, first trimester
        "O43192",  # Other malformation of placenta, second trimester
        "O43193",  # Other malformation of placenta, third trimester
        "O43199",  # Other malformation of placenta, unspecified trimester
        "O43211",  # Placenta accreta, first trimester
        "O43212",  # Placenta accreta, second trimester
        "O43213",  # Placenta accreta, third trimester
        "O43219",  # Placenta accreta, unspecified trimester
        "O43221",  # Placenta increta, first trimester
        "O43222",  # Placenta increta, second trimester
        "O43223",  # Placenta increta, third trimester
        "O43229",  # Placenta increta, unspecified trimester
        "O43231",  # Placenta percreta, first trimester
        "O43232",  # Placenta percreta, second trimester
        "O43233",  # Placenta percreta, third trimester
        "O43239",  # Placenta percreta, unspecified trimester
        "O43811",  # Placental infarction, first trimester
        "O43812",  # Placental infarction, second trimester
        "O43813",  # Placental infarction, third trimester
        "O43819",  # Placental infarction, unspecified trimester
        "O43891",  # Other placental disorders, first trimester
        "O43892",  # Other placental disorders, second trimester
        "O43893",  # Other placental disorders, third trimester
        "O43899",  # Other placental disorders, unspecified trimester
        "O4390",  # Unspecified placental disorder, unspecified trimester
        "O4391",  # Unspecified placental disorder, first trimester
        "O4392",  # Unspecified placental disorder, second trimester
        "O4393",  # Unspecified placental disorder, third trimester
        "O4400",  # Complete placenta previa NOS or without hemorrhage, unspecified trimester
        "O4401",  # Complete placenta previa NOS or without hemorrhage, first trimester
        "O4402",  # Complete placenta previa NOS or without hemorrhage, second trimester
        "O4403",  # Complete placenta previa NOS or without hemorrhage, third trimester
        "O4410",  # Complete placenta previa with hemorrhage, unspecified trimester
        "O4411",  # Complete placenta previa with hemorrhage, first trimester
        "O4412",  # Complete placenta previa with hemorrhage, second trimester
        "O4413",  # Complete placenta previa with hemorrhage, third trimester
        "O4420",  # Partial placenta previa NOS or without hemorrhage, unspecified trimester
        "O4421",  # Partial placenta previa NOS or without hemorrhage, first trimester
        "O4422",  # Partial placenta previa NOS or without hemorrhage, second trimester
        "O4423",  # Partial placenta previa NOS or without hemorrhage, third trimester
        "O4430",  # Partial placenta previa with hemorrhage, unspecified trimester
        "O4431",  # Partial placenta previa with hemorrhage, first trimester
        "O4432",  # Partial placenta previa with hemorrhage, second trimester
        "O4433",  # Partial placenta previa with hemorrhage, third trimester
        "O4440",  # Low lying placenta NOS or without hemorrhage, unspecified trimester
        "O4441",  # Low lying placenta NOS or without hemorrhage, first trimester
        "O4442",  # Low lying placenta NOS or without hemorrhage, second trimester
        "O4443",  # Low lying placenta NOS or without hemorrhage, third trimester
        "O4450",  # Low lying placenta with hemorrhage, unspecified trimester
        "O4451",  # Low lying placenta with hemorrhage, first trimester
        "O4452",  # Low lying placenta with hemorrhage, second trimester
        "O4453",  # Low lying placenta with hemorrhage, third trimester
        "O45001",  # Premature separation of placenta with coagulation defect, unspecified, first trimester
        "O45002",  # Premature separation of placenta with coagulation defect, unspecified, second trimester
        "O45003",  # Premature separation of placenta with coagulation defect, unspecified, third trimester
        "O45009",  # Premature separation of placenta with coagulation defect, unspecified, unspecified trimester
        "O45011",  # Premature separation of placenta with afibrinogenemia, first trimester
        "O45012",  # Premature separation of placenta with afibrinogenemia, second trimester
        "O45013",  # Premature separation of placenta with afibrinogenemia, third trimester
        "O45019",  # Premature separation of placenta with afibrinogenemia, unspecified trimester
        "O45021",  # Premature separation of placenta with disseminated intravascular coagulation, first trimester
        "O45022",  # Premature separation of placenta with disseminated intravascular coagulation, second trimester
        "O45023",  # Premature separation of placenta with disseminated intravascular coagulation, third trimester
        "O45029",  # Premature separation of placenta with disseminated intravascular coagulation, unspecified trimester
        "O45091",  # Premature separation of placenta with other coagulation defect, first trimester
        "O45092",  # Premature separation of placenta with other coagulation defect, second trimester
        "O45093",  # Premature separation of placenta with other coagulation defect, third trimester
        "O45099",  # Premature separation of placenta with other coagulation defect, unspecified trimester
        "O458X1",  # Other premature separation of placenta, first trimester
        "O458X2",  # Other premature separation of placenta, second trimester
        "O458X3",  # Other premature separation of placenta, third trimester
        "O458X9",  # Other premature separation of placenta, unspecified trimester
        "O4590",  # Premature separation of placenta, unspecified, unspecified trimester
        "O4591",  # Premature separation of placenta, unspecified, first trimester
        "O4592",  # Premature separation of placenta, unspecified, second trimester
        "O4593",  # Premature separation of placenta, unspecified, third trimester
        "O46001",  # Antepartum hemorrhage with coagulation defect, unspecified, first trimester
        "O46002",  # Antepartum hemorrhage with coagulation defect, unspecified, second trimester
        "O46003",  # Antepartum hemorrhage with coagulation defect, unspecified, third trimester
        "O46009",  # Antepartum hemorrhage with coagulation defect, unspecified, unspecified trimester
        "O46011",  # Antepartum hemorrhage with afibrinogenemia, first trimester
        "O46012",  # Antepartum hemorrhage with afibrinogenemia, second trimester
        "O46013",  # Antepartum hemorrhage with afibrinogenemia, third trimester
        "O46019",  # Antepartum hemorrhage with afibrinogenemia, unspecified trimester
        "O46021",  # Antepartum hemorrhage with disseminated intravascular coagulation, first trimester
        "O46022",  # Antepartum hemorrhage with disseminated intravascular coagulation, second trimester
        "O46023",  # Antepartum hemorrhage with disseminated intravascular coagulation, third trimester
        "O46029",  # Antepartum hemorrhage with disseminated intravascular coagulation, unspecified trimester
        "O46091",  # Antepartum hemorrhage with other coagulation defect, first trimester
        "O46092",  # Antepartum hemorrhage with other coagulation defect, second trimester
        "O46093",  # Antepartum hemorrhage with other coagulation defect, third trimester
        "O46099",  # Antepartum hemorrhage with other coagulation defect, unspecified trimester
        "O468X1",  # Other antepartum hemorrhage, first trimester
        "O468X2",  # Other antepartum hemorrhage, second trimester
        "O468X3",  # Other antepartum hemorrhage, third trimester
        "O468X9",  # Other antepartum hemorrhage, unspecified trimester
        "O4690",  # Antepartum hemorrhage, unspecified, unspecified trimester
        "O4691",  # Antepartum hemorrhage, unspecified, first trimester
        "O4692",  # Antepartum hemorrhage, unspecified, second trimester
        "O4693",  # Antepartum hemorrhage, unspecified, third trimester
        "O4700",  # False labor before 37 completed weeks of gestation, unspecified trimester
        "O4702",  # False labor before 37 completed weeks of gestation, second trimester
        "O4703",  # False labor before 37 completed weeks of gestation, third trimester
        "O471",  # False labor at or after 37 completed weeks of gestation
        "O479",  # False labor, unspecified
        "O480",  # Post-term pregnancy
        "O481",  # Prolonged pregnancy
        "O6000",  # Preterm labor without delivery, unspecified trimester
        "O6002",  # Preterm labor without delivery, second trimester
        "O6003",  # Preterm labor without delivery, third trimester
        "O7100",  # Rupture of uterus before onset of labor, unspecified trimester
        "O7102",  # Rupture of uterus before onset of labor, second trimester
        "O7103",  # Rupture of uterus before onset of labor, third trimester
        "O88011",  # Air embolism in pregnancy, first trimester
        "O88012",  # Air embolism in pregnancy, second trimester
        "O88013",  # Air embolism in pregnancy, third trimester
        "O88019",  # Air embolism in pregnancy, unspecified trimester
        "O88111",  # Amniotic fluid embolism in pregnancy, first trimester
        "O88112",  # Amniotic fluid embolism in pregnancy, second trimester
        "O88113",  # Amniotic fluid embolism in pregnancy, third trimester
        "O88119",  # Amniotic fluid embolism in pregnancy, unspecified trimester
        "O88211",  # Thromboembolism in pregnancy, first trimester
        "O88212",  # Thromboembolism in pregnancy, second trimester
        "O88213",  # Thromboembolism in pregnancy, third trimester
        "O88219",  # Thromboembolism in pregnancy, unspecified trimester
        "O88311",  # Pyemic and septic embolism in pregnancy, first trimester
        "O88312",  # Pyemic and septic embolism in pregnancy, second trimester
        "O88313",  # Pyemic and septic embolism in pregnancy, third trimester
        "O88319",  # Pyemic and septic embolism in pregnancy, unspecified trimester
        "O88811",  # Other embolism in pregnancy, first trimester
        "O88812",  # Other embolism in pregnancy, second trimester
        "O88813",  # Other embolism in pregnancy, third trimester
        "O88819",  # Other embolism in pregnancy, unspecified trimester
        "O903",  # Peripartum cardiomyopathy
        "O91011",  # Infection of nipple associated with pregnancy, first trimester
        "O91012",  # Infection of nipple associated with pregnancy, second trimester
        "O91013",  # Infection of nipple associated with pregnancy, third trimester
        "O91019",  # Infection of nipple associated with pregnancy, unspecified trimester
        "O91111",  # Abscess of breast associated with pregnancy, first trimester
        "O91112",  # Abscess of breast associated with pregnancy, second trimester
        "O91113",  # Abscess of breast associated with pregnancy, third trimester
        "O91119",  # Abscess of breast associated with pregnancy, unspecified trimester
        "O91211",  # Nonpurulent mastitis associated with pregnancy, first trimester
        "O91212",  # Nonpurulent mastitis associated with pregnancy, second trimester
        "O91213",  # Nonpurulent mastitis associated with pregnancy, third trimester
        "O91219",  # Nonpurulent mastitis associated with pregnancy, unspecified trimester
        "O92011",  # Retracted nipple associated with pregnancy, first trimester
        "O92012",  # Retracted nipple associated with pregnancy, second trimester
        "O92013",  # Retracted nipple associated with pregnancy, third trimester
        "O92019",  # Retracted nipple associated with pregnancy, unspecified trimester
        "O92111",  # Cracked nipple associated with pregnancy, first trimester
        "O92112",  # Cracked nipple associated with pregnancy, second trimester
        "O92113",  # Cracked nipple associated with pregnancy, third trimester
        "O92119",  # Cracked nipple associated with pregnancy, unspecified trimester
        "O9220",  # Unspecified disorder of breast associated with pregnancy and the puerperium
        "O9229",  # Other disorders of breast associated with pregnancy and the puerperium
        "O98011",  # Tuberculosis complicating pregnancy, first trimester
        "O98012",  # Tuberculosis complicating pregnancy, second trimester
        "O98013",  # Tuberculosis complicating pregnancy, third trimester
        "O98019",  # Tuberculosis complicating pregnancy, unspecified trimester
        "O98111",  # Syphilis complicating pregnancy, first trimester
        "O98112",  # Syphilis complicating pregnancy, second trimester
        "O98113",  # Syphilis complicating pregnancy, third trimester
        "O98119",  # Syphilis complicating pregnancy, unspecified trimester
        "O98211",  # Gonorrhea complicating pregnancy, first trimester
        "O98212",  # Gonorrhea complicating pregnancy, second trimester
        "O98213",  # Gonorrhea complicating pregnancy, third trimester
        "O98219",  # Gonorrhea complicating pregnancy, unspecified trimester
        "O98311",  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, first trimester
        "O98312",  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, second trimester
        "O98313",  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, third trimester
        "O98319",  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, unspecified trimester
        "O98411",  # Viral hepatitis complicating pregnancy, first trimester
        "O98412",  # Viral hepatitis complicating pregnancy, second trimester
        "O98413",  # Viral hepatitis complicating pregnancy, third trimester
        "O98419",  # Viral hepatitis complicating pregnancy, unspecified trimester
        "O98511",  # Other viral diseases complicating pregnancy, first trimester
        "O98512",  # Other viral diseases complicating pregnancy, second trimester
        "O98513",  # Other viral diseases complicating pregnancy, third trimester
        "O98519",  # Other viral diseases complicating pregnancy, unspecified trimester
        "O98611",  # Protozoal diseases complicating pregnancy, first trimester
        "O98612",  # Protozoal diseases complicating pregnancy, second trimester
        "O98613",  # Protozoal diseases complicating pregnancy, third trimester
        "O98619",  # Protozoal diseases complicating pregnancy, unspecified trimester
        "O98711",  # Human immunodeficiency virus [HIV] disease complicating pregnancy, first trimester
        "O98712",  # Human immunodeficiency virus [HIV] disease complicating pregnancy, second trimester
        "O98713",  # Human immunodeficiency virus [HIV] disease complicating pregnancy, third trimester
        "O98719",  # Human immunodeficiency virus [HIV] disease complicating pregnancy, unspecified trimester
        "O98811",  # Other maternal infectious and parasitic diseases complicating pregnancy, first trimester
        "O98812",  # Other maternal infectious and parasitic diseases complicating pregnancy, second trimester
        "O98813",  # Other maternal infectious and parasitic diseases complicating pregnancy, third trimester
        "O98819",  # Other maternal infectious and parasitic diseases complicating pregnancy, unspecified trimester
        "O98911",  # Unspecified maternal infectious and parasitic disease complicating pregnancy, first trimester
        "O98912",  # Unspecified maternal infectious and parasitic disease complicating pregnancy, second trimester
        "O98913",  # Unspecified maternal infectious and parasitic disease complicating pregnancy, third trimester
        "O98919",  # Unspecified maternal infectious and parasitic disease complicating pregnancy, unspecified trimester
        "O99011",  # Anemia complicating pregnancy, first trimester
        "O99012",  # Anemia complicating pregnancy, second trimester
        "O99013",  # Anemia complicating pregnancy, third trimester
        "O99019",  # Anemia complicating pregnancy, unspecified trimester
        "O99111",  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, first trimester
        "O99112",  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, second trimester
        "O99113",  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, third trimester
        "O99119",  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, unspecified trimester
        "O99210",  # Obesity complicating pregnancy, unspecified trimester
        "O99211",  # Obesity complicating pregnancy, first trimester
        "O99212",  # Obesity complicating pregnancy, second trimester
        "O99213",  # Obesity complicating pregnancy, third trimester
        "O99280",  # Endocrine, nutritional and metabolic diseases complicating pregnancy, unspecified trimester
        "O99281",  # Endocrine, nutritional and metabolic diseases complicating pregnancy, first trimester
        "O99282",  # Endocrine, nutritional and metabolic diseases complicating pregnancy, second trimester
        "O99283",  # Endocrine, nutritional and metabolic diseases complicating pregnancy, third trimester
        "O99310",  # Alcohol use complicating pregnancy, unspecified trimester
        "O99311",  # Alcohol use complicating pregnancy, first trimester
        "O99312",  # Alcohol use complicating pregnancy, second trimester
        "O99313",  # Alcohol use complicating pregnancy, third trimester
        "O99320",  # Drug use complicating pregnancy, unspecified trimester
        "O99321",  # Drug use complicating pregnancy, first trimester
        "O99322",  # Drug use complicating pregnancy, second trimester
        "O99323",  # Drug use complicating pregnancy, third trimester
        "O99330",  # Smoking (tobacco) complicating pregnancy, unspecified trimester
        "O99331",  # Smoking (tobacco) complicating pregnancy, first trimester
        "O99332",  # Smoking (tobacco) complicating pregnancy, second trimester
        "O99333",  # Smoking (tobacco) complicating pregnancy, third trimester
        "O99340",  # Other mental disorders complicating pregnancy, unspecified trimester
        "O99341",  # Other mental disorders complicating pregnancy, first trimester
        "O99342",  # Other mental disorders complicating pregnancy, second trimester
        "O99343",  # Other mental disorders complicating pregnancy, third trimester
        "O99350",  # Diseases of the nervous system complicating pregnancy, unspecified trimester
        "O99351",  # Diseases of the nervous system complicating pregnancy, first trimester
        "O99352",  # Diseases of the nervous system complicating pregnancy, second trimester
        "O99353",  # Diseases of the nervous system complicating pregnancy, third trimester
        "O99411",  # Diseases of the circulatory system complicating pregnancy, first trimester
        "O99412",  # Diseases of the circulatory system complicating pregnancy, second trimester
        "O99413",  # Diseases of the circulatory system complicating pregnancy, third trimester
        "O99419",  # Diseases of the circulatory system complicating pregnancy, unspecified trimester
        "O99511",  # Diseases of the respiratory system complicating pregnancy, first trimester
        "O99512",  # Diseases of the respiratory system complicating pregnancy, second trimester
        "O99513",  # Diseases of the respiratory system complicating pregnancy, third trimester
        "O99519",  # Diseases of the respiratory system complicating pregnancy, unspecified trimester
        "O99611",  # Diseases of the digestive system complicating pregnancy, first trimester
        "O99612",  # Diseases of the digestive system complicating pregnancy, second trimester
        "O99613",  # Diseases of the digestive system complicating pregnancy, third trimester
        "O99619",  # Diseases of the digestive system complicating pregnancy, unspecified trimester
        "O99711",  # Diseases of the skin and subcutaneous tissue complicating pregnancy, first trimester
        "O99712",  # Diseases of the skin and subcutaneous tissue complicating pregnancy, second trimester
        "O99713",  # Diseases of the skin and subcutaneous tissue complicating pregnancy, third trimester
        "O99719",  # Diseases of the skin and subcutaneous tissue complicating pregnancy, unspecified trimester
        "O99810",  # Abnormal glucose complicating pregnancy
        "O99820",  # Streptococcus B carrier state complicating pregnancy
        "O99830",  # Other infection carrier state complicating pregnancy
        "O99840",  # Bariatric surgery status complicating pregnancy, unspecified trimester
        "O99841",  # Bariatric surgery status complicating pregnancy, first trimester
        "O99842",  # Bariatric surgery status complicating pregnancy, second trimester
        "O99843",  # Bariatric surgery status complicating pregnancy, third trimester
        "O9A111",  # Malignant neoplasm complicating pregnancy, first trimester
        "O9A112",  # Malignant neoplasm complicating pregnancy, second trimester
        "O9A113",  # Malignant neoplasm complicating pregnancy, third trimester
        "O9A119",  # Malignant neoplasm complicating pregnancy, unspecified trimester
        "O9A211",  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, first trimester
        "O9A212",  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, second trimester
        "O9A213",  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, third trimester
        "O9A219",  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, unspecified trimester
        "O9A311",  # Physical abuse complicating pregnancy, first trimester
        "O9A312",  # Physical abuse complicating pregnancy, second trimester
        "O9A313",  # Physical abuse complicating pregnancy, third trimester
        "O9A319",  # Physical abuse complicating pregnancy, unspecified trimester
        "O9A411",  # Sexual abuse complicating pregnancy, first trimester
        "O9A412",  # Sexual abuse complicating pregnancy, second trimester
        "O9A413",  # Sexual abuse complicating pregnancy, third trimester
        "O9A419",  # Sexual abuse complicating pregnancy, unspecified trimester
        "O9A511",  # Psychological abuse complicating pregnancy, first trimester
        "O9A512",  # Psychological abuse complicating pregnancy, second trimester
        "O9A513",  # Psychological abuse complicating pregnancy, third trimester
        "O9A519",  # Psychological abuse complicating pregnancy, unspecified trimester
        "Z331",  # Pregnant state, incidental
        "Z333",  # Pregnant state, gestational carrier
        "Z3400",  # Encounter for supervision of normal first pregnancy, unspecified trimester
        "Z3401",  # Encounter for supervision of normal first pregnancy, first trimester
        "Z3402",  # Encounter for supervision of normal first pregnancy, second trimester
        "Z3403",  # Encounter for supervision of normal first pregnancy, third trimester
        "Z3480",  # Encounter for supervision of other normal pregnancy, unspecified trimester
        "Z3481",  # Encounter for supervision of other normal pregnancy, first trimester
        "Z3482",  # Encounter for supervision of other normal pregnancy, second trimester
        "Z3483",  # Encounter for supervision of other normal pregnancy, third trimester
        "Z3490",  # Encounter for supervision of normal pregnancy, unspecified, unspecified trimester
        "Z3491",  # Encounter for supervision of normal pregnancy, unspecified, first trimester
        "Z3492",  # Encounter for supervision of normal pregnancy, unspecified, second trimester
        "Z3493",  # Encounter for supervision of normal pregnancy, unspecified, third trimester
        "Z3A01",  # Less than 8 weeks gestation of pregnancy
        "Z3A08",  # 8 weeks gestation of pregnancy
        "Z3A09",  # 9 weeks gestation of pregnancy
        "Z3A10",  # 10 weeks gestation of pregnancy
        "Z3A11",  # 11 weeks gestation of pregnancy
        "Z3A12",  # 12 weeks gestation of pregnancy
        "Z3A13",  # 13 weeks gestation of pregnancy
        "Z3A14",  # 14 weeks gestation of pregnancy
        "Z3A15",  # 15 weeks gestation of pregnancy
        "Z3A16",  # 16 weeks gestation of pregnancy
        "Z3A17",  # 17 weeks gestation of pregnancy
        "Z3A18",  # 18 weeks gestation of pregnancy
        "Z3A19",  # 19 weeks gestation of pregnancy
        "Z3A20",  # 20 weeks gestation of pregnancy
        "Z3A21",  # 21 weeks gestation of pregnancy
        "Z3A22",  # 22 weeks gestation of pregnancy
        "Z3A23",  # 23 weeks gestation of pregnancy
        "Z3A24",  # 24 weeks gestation of pregnancy
        "Z3A25",  # 25 weeks gestation of pregnancy
        "Z3A26",  # 26 weeks gestation of pregnancy
        "Z3A27",  # 27 weeks gestation of pregnancy
        "Z3A28",  # 28 weeks gestation of pregnancy
        "Z3A29",  # 29 weeks gestation of pregnancy
        "Z3A30",  # 30 weeks gestation of pregnancy
        "Z3A31",  # 31 weeks gestation of pregnancy
        "Z3A32",  # 32 weeks gestation of pregnancy
        "Z3A33",  # 33 weeks gestation of pregnancy
        "Z3A34",  # 34 weeks gestation of pregnancy
        "Z3A35",  # 35 weeks gestation of pregnancy
        "Z3A36",  # 36 weeks gestation of pregnancy
        "Z3A37",  # 37 weeks gestation of pregnancy
        "Z3A38",  # 38 weeks gestation of pregnancy
        "Z3A39",  # 39 weeks gestation of pregnancy
        "Z3A40",  # 40 weeks gestation of pregnancy
        "Z3A41",  # 41 weeks gestation of pregnancy
        "Z3A42",  # 42 weeks gestation of pregnancy
        "Z3A49",  # Greater than 42 weeks gestation of pregnancy
    }
    SNOMEDCT = {
        "9279009",  # Extra-amniotic pregnancy (finding)
        "11082009",  # Abnormal pregnancy (finding)
        "14080002",  # Illegitimate pregnancy, life event (finding)
        "14418008",  # Precocious pregnancy (finding)
        "16356006",  # Multiple pregnancy (disorder)
        "38720006",  # Septuplet pregnancy (disorder)
        "41587001",  # Third trimester pregnancy (finding)
        "41991004",  # Angiectasis pregnancy (disorder)
        "43990006",  # Sextuplet pregnancy (disorder)
        "45307008",  # Extrachorial pregnancy (finding)
        "47200007",  # High risk pregnancy (finding)
        "57630001",  # First trimester pregnancy (finding)
        "58532003",  # Unwanted pregnancy (finding)
        "59466002",  # Second trimester pregnancy (finding)
        "60810003",  # Quadruplet pregnancy (disorder)
        "64254006",  # Triplet pregnancy (disorder)
        "65147003",  # Twin pregnancy (disorder)
        "65727000",  # Intrauterine pregnancy (finding)
        "72892002",  # Normal pregnancy (finding)
        "77386006",  # Pregnant (finding)
        "80997009",  # Quintuplet pregnancy (disorder)
        "83074005",  # Unplanned pregnancy (finding)
        "87527008",  # Term pregnancy (finding)
        "90968009",  # Prolonged pregnancy (disorder)
        "102872000",  # Pregnancy on oral contraceptive (finding)
        "102875003",  # Surrogate pregnancy (finding)
        "127364007",  # Primigravida (finding)
        "169561007",  # Pregnant - blood test confirms (finding)
        "169562000",  # Pregnant - vaginal examination confirms (finding)
        "169563005",  # Pregnant - on history (finding)
        "169564004",  # Pregnant - on abdominal palpation (finding)
        "169565003",  # Pregnant - planned (finding)
        "169566002",  # Pregnancy unplanned but wanted (finding)
        "169567006",  # Pregnancy unplanned and unwanted (finding)
        "169568001",  # Unplanned pregnancy unknown if child is wanted (finding)
        "199064003",  # Post-term pregnancy - not delivered (disorder)
        "199306007",  # Continuing pregnancy after abortion of one fetus or more (disorder)
        "237233002",  # Concealed pregnancy (finding)
        "237238006",  # Pregnancy with uncertain dates (finding)
        "237239003",  # Low risk pregnancy (finding)
        "237240001",  # Teenage pregnancy (finding)
        "237241002",  # Viable pregnancy (finding)
        "237244005",  # Single pregnancy (finding)
        "248985009",  # Presentation of pregnancy (finding)
        "276367008",  # Wanted pregnancy (finding)
        "281307002",  # Uncertain viability of pregnancy (finding)
        "314204000",  # Early stage of pregnancy (finding)
        "429187001",  # Continuing pregnancy after intrauterine death of twin fetus (disorder)
        "442478007",  # Multiple pregnancy involving intrauterine pregnancy and tubal pregnancy (disorder)
        "444661007",  # High risk pregnancy due to history of preterm labor (finding)
        "459167000",  # Monochorionic twin pregnancy (disorder)
        "459169002",  # Monochorionic diamniotic twin pregnancy with similar amniotic fluid volumes (disorder)
        "459170001",  # Monochorionic diamniotic twin pregnancy with dissimilar amniotic fluid volumes (disorder)
        "472321009",  # Continuing pregnancy after intrauterine death of one twin with intrauterine retention of dead twin (disorder)
        "713575004",  # Dizygotic twin pregnancy (disorder)
        "713576003",  # Monozygotic twin pregnancy (disorder)
        "10231000132102",  # In-vitro fertilization pregnancy (finding)
        "22281000119101",  # Post-term pregnancy of 40 to 42 weeks (disorder)
        "35381000119101",  # Quadruplet pregnancy with loss of one or more fetuses (disorder)
        "36801000119105",  # Continuing triplet pregnancy after spontaneous abortion of one or more fetuses (disorder)
        "134781000119106",  # High risk pregnancy due to recurrent miscarriage (finding)
        "457811000124103",  # Normal pregnancy in primigravida (finding)
        "457821000124106",  # Normal pregnancy in multigravida (finding)
    }


class RenalFailureDueToAceInhibitor(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for renal failure due to angiotensin converting enzyme (ACE) inhibitors.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of renal failure due to ACE inhibitor.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10
    """

    VALUE_SET_NAME = "Renal Failure Due to ACE Inhibitor"
    OID = "2.16.840.1.113883.3.526.3.1151"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "236429004",  # Acute drug-induced renal failure (disorder)
        "422593004",  # Acute renal failure caused by angiotensin-converting-enzyme inhibitor (disorder)
    }


class Narcolepsy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  narcolepsy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of narcolepsy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS136v11
    """

    VALUE_SET_NAME = "Narcolepsy"
    OID = "2.16.840.1.113883.3.464.1003.114.12.1011"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "G47411",  # Narcolepsy with cataplexy
        "G47419",  # Narcolepsy without cataplexy
        "G47421",  # Narcolepsy in conditions classified elsewhere with cataplexy
        "G47429",  # Narcolepsy in conditions classified elsewhere without cataplexy
    }
    ICD9CM = {
        "34700",  # Narcolepsy, without cataplexy
        "34701",  # Narcolepsy, with cataplexy
        "34710",  # Narcolepsy in conditions classified elsewhere, without cataplexy
        "34711",  # Narcolepsy in conditions classified elsewhere, with cataplexy
    }
    SNOMEDCT = {
        "60380001",  # Narcolepsy (disorder)
        "193042000",  # Cataplexy and narcolepsy (disorder)
        "427426006",  # Sleep attack (disorder)
        "735676003",  # Narcolepsy type 1 (disorder)
        "91521000119104",  # Narcolepsy without cataplexy (disorder)
        "434241000124107",  # Secondary narcolepsy (disorder)
        "434251000124109",  # Secondary narcolepsy without cataplexy (disorder)
        "434261000124106",  # Secondary narcolepsy with cataplexy (disorder)
    }


class AlcoholAndDrugDependence(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  alcohol and drug dependence.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of alcohol and drug abuse and dependence.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of alcohol and drug use.

    ** Used in:** CMS137v10
    """

    VALUE_SET_NAME = "Alcohol and Drug Dependence"
    OID = "2.16.840.1.113883.3.464.1003.106.12.1001"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "F1010",  # Alcohol abuse, uncomplicated
        "F10120",  # Alcohol abuse with intoxication, uncomplicated
        "F10121",  # Alcohol abuse with intoxication delirium
        "F10129",  # Alcohol abuse with intoxication, unspecified
        "F10130",  # Alcohol abuse with withdrawal, uncomplicated
        "F10131",  # Alcohol abuse with withdrawal delirium
        "F10132",  # Alcohol abuse with withdrawal with perceptual disturbance
        "F10139",  # Alcohol abuse with withdrawal, unspecified
        "F1014",  # Alcohol abuse with alcohol-induced mood disorder
        "F10150",  # Alcohol abuse with alcohol-induced psychotic disorder with delusions
        "F10151",  # Alcohol abuse with alcohol-induced psychotic disorder with hallucinations
        "F10159",  # Alcohol abuse with alcohol-induced psychotic disorder, unspecified
        "F10180",  # Alcohol abuse with alcohol-induced anxiety disorder
        "F10181",  # Alcohol abuse with alcohol-induced sexual dysfunction
        "F10182",  # Alcohol abuse with alcohol-induced sleep disorder
        "F10188",  # Alcohol abuse with other alcohol-induced disorder
        "F1019",  # Alcohol abuse with unspecified alcohol-induced disorder
        "F1020",  # Alcohol dependence, uncomplicated
        "F10220",  # Alcohol dependence with intoxication, uncomplicated
        "F10221",  # Alcohol dependence with intoxication delirium
        "F10229",  # Alcohol dependence with intoxication, unspecified
        "F10230",  # Alcohol dependence with withdrawal, uncomplicated
        "F10231",  # Alcohol dependence with withdrawal delirium
        "F10232",  # Alcohol dependence with withdrawal with perceptual disturbance
        "F10239",  # Alcohol dependence with withdrawal, unspecified
        "F1024",  # Alcohol dependence with alcohol-induced mood disorder
        "F10250",  # Alcohol dependence with alcohol-induced psychotic disorder with delusions
        "F10251",  # Alcohol dependence with alcohol-induced psychotic disorder with hallucinations
        "F10259",  # Alcohol dependence with alcohol-induced psychotic disorder, unspecified
        "F1026",  # Alcohol dependence with alcohol-induced persisting amnestic disorder
        "F1027",  # Alcohol dependence with alcohol-induced persisting dementia
        "F10280",  # Alcohol dependence with alcohol-induced anxiety disorder
        "F10281",  # Alcohol dependence with alcohol-induced sexual dysfunction
        "F10282",  # Alcohol dependence with alcohol-induced sleep disorder
        "F10288",  # Alcohol dependence with other alcohol-induced disorder
        "F1029",  # Alcohol dependence with unspecified alcohol-induced disorder
        "F1110",  # Opioid abuse, uncomplicated
        "F11120",  # Opioid abuse with intoxication, uncomplicated
        "F11121",  # Opioid abuse with intoxication delirium
        "F11122",  # Opioid abuse with intoxication with perceptual disturbance
        "F11129",  # Opioid abuse with intoxication, unspecified
        "F1113",  # Opioid abuse with withdrawal
        "F1114",  # Opioid abuse with opioid-induced mood disorder
        "F11150",  # Opioid abuse with opioid-induced psychotic disorder with delusions
        "F11151",  # Opioid abuse with opioid-induced psychotic disorder with hallucinations
        "F11159",  # Opioid abuse with opioid-induced psychotic disorder, unspecified
        "F11181",  # Opioid abuse with opioid-induced sexual dysfunction
        "F11182",  # Opioid abuse with opioid-induced sleep disorder
        "F11188",  # Opioid abuse with other opioid-induced disorder
        "F1119",  # Opioid abuse with unspecified opioid-induced disorder
        "F1120",  # Opioid dependence, uncomplicated
        "F11220",  # Opioid dependence with intoxication, uncomplicated
        "F11221",  # Opioid dependence with intoxication delirium
        "F11222",  # Opioid dependence with intoxication with perceptual disturbance
        "F11229",  # Opioid dependence with intoxication, unspecified
        "F1123",  # Opioid dependence with withdrawal
        "F1124",  # Opioid dependence with opioid-induced mood disorder
        "F11250",  # Opioid dependence with opioid-induced psychotic disorder with delusions
        "F11251",  # Opioid dependence with opioid-induced psychotic disorder with hallucinations
        "F11259",  # Opioid dependence with opioid-induced psychotic disorder, unspecified
        "F11281",  # Opioid dependence with opioid-induced sexual dysfunction
        "F11282",  # Opioid dependence with opioid-induced sleep disorder
        "F11288",  # Opioid dependence with other opioid-induced disorder
        "F1129",  # Opioid dependence with unspecified opioid-induced disorder
        "F1210",  # Cannabis abuse, uncomplicated
        "F12120",  # Cannabis abuse with intoxication, uncomplicated
        "F12121",  # Cannabis abuse with intoxication delirium
        "F12122",  # Cannabis abuse with intoxication with perceptual disturbance
        "F12129",  # Cannabis abuse with intoxication, unspecified
        "F1213",  # Cannabis abuse with withdrawal
        "F12150",  # Cannabis abuse with psychotic disorder with delusions
        "F12151",  # Cannabis abuse with psychotic disorder with hallucinations
        "F12159",  # Cannabis abuse with psychotic disorder, unspecified
        "F12180",  # Cannabis abuse with cannabis-induced anxiety disorder
        "F12188",  # Cannabis abuse with other cannabis-induced disorder
        "F1219",  # Cannabis abuse with unspecified cannabis-induced disorder
        "F1220",  # Cannabis dependence, uncomplicated
        "F12220",  # Cannabis dependence with intoxication, uncomplicated
        "F12221",  # Cannabis dependence with intoxication delirium
        "F12222",  # Cannabis dependence with intoxication with perceptual disturbance
        "F12229",  # Cannabis dependence with intoxication, unspecified
        "F1223",  # Cannabis dependence with withdrawal
        "F12250",  # Cannabis dependence with psychotic disorder with delusions
        "F12251",  # Cannabis dependence with psychotic disorder with hallucinations
        "F12259",  # Cannabis dependence with psychotic disorder, unspecified
        "F12280",  # Cannabis dependence with cannabis-induced anxiety disorder
        "F12288",  # Cannabis dependence with other cannabis-induced disorder
        "F1229",  # Cannabis dependence with unspecified cannabis-induced disorder
        "F1310",  # Sedative, hypnotic or anxiolytic abuse, uncomplicated
        "F13120",  # Sedative, hypnotic or anxiolytic abuse with intoxication, uncomplicated
        "F13121",  # Sedative, hypnotic or anxiolytic abuse with intoxication delirium
        "F13129",  # Sedative, hypnotic or anxiolytic abuse with intoxication, unspecified
        "F13130",  # Sedative, hypnotic or anxiolytic abuse with withdrawal, uncomplicated
        "F13131",  # Sedative, hypnotic or anxiolytic abuse with withdrawal delirium
        "F13132",  # Sedative, hypnotic or anxiolytic abuse with withdrawal with perceptual disturbance
        "F13139",  # Sedative, hypnotic or anxiolytic abuse with withdrawal, unspecified
        "F1314",  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced mood disorder
        "F13150",  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced psychotic disorder with delusions
        "F13151",  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced psychotic disorder with hallucinations
        "F13159",  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced psychotic disorder, unspecified
        "F13180",  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced anxiety disorder
        "F13181",  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced sexual dysfunction
        "F13182",  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced sleep disorder
        "F13188",  # Sedative, hypnotic or anxiolytic abuse with other sedative, hypnotic or anxiolytic-induced disorder
        "F1319",  # Sedative, hypnotic or anxiolytic abuse with unspecified sedative, hypnotic or anxiolytic-induced disorder
        "F1320",  # Sedative, hypnotic or anxiolytic dependence, uncomplicated
        "F13220",  # Sedative, hypnotic or anxiolytic dependence with intoxication, uncomplicated
        "F13221",  # Sedative, hypnotic or anxiolytic dependence with intoxication delirium
        "F13229",  # Sedative, hypnotic or anxiolytic dependence with intoxication, unspecified
        "F13230",  # Sedative, hypnotic or anxiolytic dependence with withdrawal, uncomplicated
        "F13231",  # Sedative, hypnotic or anxiolytic dependence with withdrawal delirium
        "F13232",  # Sedative, hypnotic or anxiolytic dependence with withdrawal with perceptual disturbance
        "F13239",  # Sedative, hypnotic or anxiolytic dependence with withdrawal, unspecified
        "F1324",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced mood disorder
        "F13250",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced psychotic disorder with delusions
        "F13251",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced psychotic disorder with hallucinations
        "F13259",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced psychotic disorder, unspecified
        "F1326",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced persisting amnestic disorder
        "F1327",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced persisting dementia
        "F13280",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced anxiety disorder
        "F13281",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced sexual dysfunction
        "F13282",  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced sleep disorder
        "F13288",  # Sedative, hypnotic or anxiolytic dependence with other sedative, hypnotic or anxiolytic-induced disorder
        "F1329",  # Sedative, hypnotic or anxiolytic dependence with unspecified sedative, hypnotic or anxiolytic-induced disorder
        "F1410",  # Cocaine abuse, uncomplicated
        "F14120",  # Cocaine abuse with intoxication, uncomplicated
        "F14121",  # Cocaine abuse with intoxication with delirium
        "F14122",  # Cocaine abuse with intoxication with perceptual disturbance
        "F14129",  # Cocaine abuse with intoxication, unspecified
        "F1413",  # Cocaine abuse, unspecified with withdrawal
        "F1414",  # Cocaine abuse with cocaine-induced mood disorder
        "F14150",  # Cocaine abuse with cocaine-induced psychotic disorder with delusions
        "F14151",  # Cocaine abuse with cocaine-induced psychotic disorder with hallucinations
        "F14159",  # Cocaine abuse with cocaine-induced psychotic disorder, unspecified
        "F14180",  # Cocaine abuse with cocaine-induced anxiety disorder
        "F14181",  # Cocaine abuse with cocaine-induced sexual dysfunction
        "F14182",  # Cocaine abuse with cocaine-induced sleep disorder
        "F14188",  # Cocaine abuse with other cocaine-induced disorder
        "F1419",  # Cocaine abuse with unspecified cocaine-induced disorder
        "F1420",  # Cocaine dependence, uncomplicated
        "F14220",  # Cocaine dependence with intoxication, uncomplicated
        "F14221",  # Cocaine dependence with intoxication delirium
        "F14222",  # Cocaine dependence with intoxication with perceptual disturbance
        "F14229",  # Cocaine dependence with intoxication, unspecified
        "F1423",  # Cocaine dependence with withdrawal
        "F1424",  # Cocaine dependence with cocaine-induced mood disorder
        "F14250",  # Cocaine dependence with cocaine-induced psychotic disorder with delusions
        "F14251",  # Cocaine dependence with cocaine-induced psychotic disorder with hallucinations
        "F14259",  # Cocaine dependence with cocaine-induced psychotic disorder, unspecified
        "F14280",  # Cocaine dependence with cocaine-induced anxiety disorder
        "F14281",  # Cocaine dependence with cocaine-induced sexual dysfunction
        "F14282",  # Cocaine dependence with cocaine-induced sleep disorder
        "F14288",  # Cocaine dependence with other cocaine-induced disorder
        "F1429",  # Cocaine dependence with unspecified cocaine-induced disorder
        "F1510",  # Other stimulant abuse, uncomplicated
        "F15120",  # Other stimulant abuse with intoxication, uncomplicated
        "F15121",  # Other stimulant abuse with intoxication delirium
        "F15122",  # Other stimulant abuse with intoxication with perceptual disturbance
        "F15129",  # Other stimulant abuse with intoxication, unspecified
        "F1513",  # Other stimulant abuse with withdrawal
        "F1514",  # Other stimulant abuse with stimulant-induced mood disorder
        "F15150",  # Other stimulant abuse with stimulant-induced psychotic disorder with delusions
        "F15151",  # Other stimulant abuse with stimulant-induced psychotic disorder with hallucinations
        "F15159",  # Other stimulant abuse with stimulant-induced psychotic disorder, unspecified
        "F15180",  # Other stimulant abuse with stimulant-induced anxiety disorder
        "F15181",  # Other stimulant abuse with stimulant-induced sexual dysfunction
        "F15182",  # Other stimulant abuse with stimulant-induced sleep disorder
        "F15188",  # Other stimulant abuse with other stimulant-induced disorder
        "F1519",  # Other stimulant abuse with unspecified stimulant-induced disorder
        "F1520",  # Other stimulant dependence, uncomplicated
        "F15220",  # Other stimulant dependence with intoxication, uncomplicated
        "F15221",  # Other stimulant dependence with intoxication delirium
        "F15222",  # Other stimulant dependence with intoxication with perceptual disturbance
        "F15229",  # Other stimulant dependence with intoxication, unspecified
        "F1523",  # Other stimulant dependence with withdrawal
        "F1524",  # Other stimulant dependence with stimulant-induced mood disorder
        "F15250",  # Other stimulant dependence with stimulant-induced psychotic disorder with delusions
        "F15251",  # Other stimulant dependence with stimulant-induced psychotic disorder with hallucinations
        "F15259",  # Other stimulant dependence with stimulant-induced psychotic disorder, unspecified
        "F15280",  # Other stimulant dependence with stimulant-induced anxiety disorder
        "F15281",  # Other stimulant dependence with stimulant-induced sexual dysfunction
        "F15282",  # Other stimulant dependence with stimulant-induced sleep disorder
        "F15288",  # Other stimulant dependence with other stimulant-induced disorder
        "F1529",  # Other stimulant dependence with unspecified stimulant-induced disorder
        "F1610",  # Hallucinogen abuse, uncomplicated
        "F16120",  # Hallucinogen abuse with intoxication, uncomplicated
        "F16121",  # Hallucinogen abuse with intoxication with delirium
        "F16122",  # Hallucinogen abuse with intoxication with perceptual disturbance
        "F16129",  # Hallucinogen abuse with intoxication, unspecified
        "F1614",  # Hallucinogen abuse with hallucinogen-induced mood disorder
        "F16150",  # Hallucinogen abuse with hallucinogen-induced psychotic disorder with delusions
        "F16151",  # Hallucinogen abuse with hallucinogen-induced psychotic disorder with hallucinations
        "F16159",  # Hallucinogen abuse with hallucinogen-induced psychotic disorder, unspecified
        "F16180",  # Hallucinogen abuse with hallucinogen-induced anxiety disorder
        "F16183",  # Hallucinogen abuse with hallucinogen persisting perception disorder (flashbacks)
        "F16188",  # Hallucinogen abuse with other hallucinogen-induced disorder
        "F1619",  # Hallucinogen abuse with unspecified hallucinogen-induced disorder
        "F1620",  # Hallucinogen dependence, uncomplicated
        "F16220",  # Hallucinogen dependence with intoxication, uncomplicated
        "F16221",  # Hallucinogen dependence with intoxication with delirium
        "F16229",  # Hallucinogen dependence with intoxication, unspecified
        "F1624",  # Hallucinogen dependence with hallucinogen-induced mood disorder
        "F16250",  # Hallucinogen dependence with hallucinogen-induced psychotic disorder with delusions
        "F16251",  # Hallucinogen dependence with hallucinogen-induced psychotic disorder with hallucinations
        "F16259",  # Hallucinogen dependence with hallucinogen-induced psychotic disorder, unspecified
        "F16280",  # Hallucinogen dependence with hallucinogen-induced anxiety disorder
        "F16283",  # Hallucinogen dependence with hallucinogen persisting perception disorder (flashbacks)
        "F16288",  # Hallucinogen dependence with other hallucinogen-induced disorder
        "F1629",  # Hallucinogen dependence with unspecified hallucinogen-induced disorder
        "F1810",  # Inhalant abuse, uncomplicated
        "F18120",  # Inhalant abuse with intoxication, uncomplicated
        "F18121",  # Inhalant abuse with intoxication delirium
        "F18129",  # Inhalant abuse with intoxication, unspecified
        "F1814",  # Inhalant abuse with inhalant-induced mood disorder
        "F18150",  # Inhalant abuse with inhalant-induced psychotic disorder with delusions
        "F18151",  # Inhalant abuse with inhalant-induced psychotic disorder with hallucinations
        "F18159",  # Inhalant abuse with inhalant-induced psychotic disorder, unspecified
        "F1817",  # Inhalant abuse with inhalant-induced dementia
        "F18180",  # Inhalant abuse with inhalant-induced anxiety disorder
        "F18188",  # Inhalant abuse with other inhalant-induced disorder
        "F1819",  # Inhalant abuse with unspecified inhalant-induced disorder
        "F1820",  # Inhalant dependence, uncomplicated
        "F18220",  # Inhalant dependence with intoxication, uncomplicated
        "F18221",  # Inhalant dependence with intoxication delirium
        "F18229",  # Inhalant dependence with intoxication, unspecified
        "F1824",  # Inhalant dependence with inhalant-induced mood disorder
        "F18250",  # Inhalant dependence with inhalant-induced psychotic disorder with delusions
        "F18251",  # Inhalant dependence with inhalant-induced psychotic disorder with hallucinations
        "F18259",  # Inhalant dependence with inhalant-induced psychotic disorder, unspecified
        "F1827",  # Inhalant dependence with inhalant-induced dementia
        "F18280",  # Inhalant dependence with inhalant-induced anxiety disorder
        "F18288",  # Inhalant dependence with other inhalant-induced disorder
        "F1829",  # Inhalant dependence with unspecified inhalant-induced disorder
        "F1910",  # Other psychoactive substance abuse, uncomplicated
        "F19120",  # Other psychoactive substance abuse with intoxication, uncomplicated
        "F19121",  # Other psychoactive substance abuse with intoxication delirium
        "F19122",  # Other psychoactive substance abuse with intoxication with perceptual disturbances
        "F19129",  # Other psychoactive substance abuse with intoxication, unspecified
        "F19130",  # Other psychoactive substance abuse with withdrawal, uncomplicated
        "F19131",  # Other psychoactive substance abuse with withdrawal delirium
        "F19132",  # Other psychoactive substance abuse with withdrawal with perceptual disturbance
        "F19139",  # Other psychoactive substance abuse with withdrawal, unspecified
        "F1914",  # Other psychoactive substance abuse with psychoactive substance-induced mood disorder
        "F19150",  # Other psychoactive substance abuse with psychoactive substance-induced psychotic disorder with delusions
        "F19151",  # Other psychoactive substance abuse with psychoactive substance-induced psychotic disorder with hallucinations
        "F19159",  # Other psychoactive substance abuse with psychoactive substance-induced psychotic disorder, unspecified
        "F1916",  # Other psychoactive substance abuse with psychoactive substance-induced persisting amnestic disorder
        "F1917",  # Other psychoactive substance abuse with psychoactive substance-induced persisting dementia
        "F19180",  # Other psychoactive substance abuse with psychoactive substance-induced anxiety disorder
        "F19181",  # Other psychoactive substance abuse with psychoactive substance-induced sexual dysfunction
        "F19182",  # Other psychoactive substance abuse with psychoactive substance-induced sleep disorder
        "F19188",  # Other psychoactive substance abuse with other psychoactive substance-induced disorder
        "F1919",  # Other psychoactive substance abuse with unspecified psychoactive substance-induced disorder
        "F1920",  # Other psychoactive substance dependence, uncomplicated
        "F19220",  # Other psychoactive substance dependence with intoxication, uncomplicated
        "F19221",  # Other psychoactive substance dependence with intoxication delirium
        "F19222",  # Other psychoactive substance dependence with intoxication with perceptual disturbance
        "F19229",  # Other psychoactive substance dependence with intoxication, unspecified
        "F19230",  # Other psychoactive substance dependence with withdrawal, uncomplicated
        "F19231",  # Other psychoactive substance dependence with withdrawal delirium
        "F19232",  # Other psychoactive substance dependence with withdrawal with perceptual disturbance
        "F19239",  # Other psychoactive substance dependence with withdrawal, unspecified
        "F1924",  # Other psychoactive substance dependence with psychoactive substance-induced mood disorder
        "F19250",  # Other psychoactive substance dependence with psychoactive substance-induced psychotic disorder with delusions
        "F19251",  # Other psychoactive substance dependence with psychoactive substance-induced psychotic disorder with hallucinations
        "F19259",  # Other psychoactive substance dependence with psychoactive substance-induced psychotic disorder, unspecified
        "F1926",  # Other psychoactive substance dependence with psychoactive substance-induced persisting amnestic disorder
        "F1927",  # Other psychoactive substance dependence with psychoactive substance-induced persisting dementia
        "F19280",  # Other psychoactive substance dependence with psychoactive substance-induced anxiety disorder
        "F19281",  # Other psychoactive substance dependence with psychoactive substance-induced sexual dysfunction
        "F19282",  # Other psychoactive substance dependence with psychoactive substance-induced sleep disorder
        "F19288",  # Other psychoactive substance dependence with other psychoactive substance-induced disorder
        "F1929",  # Other psychoactive substance dependence with unspecified psychoactive substance-induced disorder
    }
    SNOMEDCT = {
        "281004",  # Dementia associated with alcoholism (disorder)
        "1383008",  # Hallucinogen mood disorder (disorder)
        "1686006",  # Sedative, hypnotic AND/OR anxiolytic-induced anxiety disorder (disorder)
        "1973000",  # Sedative, hypnotic AND/OR anxiolytic-induced psychotic disorder with delusions (disorder)
        "2403008",  # Psychoactive substance dependence (disorder)
        "5002000",  # Inhalant dependence (disorder)
        "5444000",  # Sedative, hypnotic AND/OR anxiolytic intoxication delirium (disorder)
        "5602001",  # Opioid abuse (disorder)
        "6348008",  # Cocaine-induced psychotic disorder with hallucinations (disorder)
        "7052005",  # Alcohol hallucinosis (disorder)
        "7071007",  # Phencyclidine abuse (disorder)
        "7200002",  # Alcoholism (disorder)
        "8635005",  # Alcohol withdrawal delirium (disorder)
        "8837000",  # Amphetamine delirium (disorder)
        "10028000",  # Uncomplicated sedative, hypnotic AND/OR anxiolytic withdrawal (disorder)
        "10327003",  # Cocaine-induced mood disorder (disorder)
        "11387009",  # Psychoactive substance-induced organic mental disorder (disorder)
        "14784000",  # Opioid-induced organic mental disorder (disorder)
        "15167005",  # Alcohol abuse (disorder)
        "15277004",  # Hallucinogen-induced anxiety disorder (disorder)
        "18653004",  # Alcohol intoxication delirium (disorder)
        "18689007",  # Inhalant intoxication delirium (disorder)
        "19445006",  # Opioid-induced psychotic disorder with hallucinations (disorder)
        "20385005",  # Opioid-induced psychotic disorder with delusions (disorder)
        "20876004",  # Inhalant-induced anxiety disorder (disorder)
        "21647008",  # Amphetamine dependence (disorder)
        "22574000",  # Cocaine-induced sleep disorder (disorder)
        "25753007",  # Amphetamine-induced sleep disorder (disorder)
        "26714005",  # Cannabis-induced psychotic disorder with hallucinations (disorder)
        "27956007",  # Cocaine intoxication (disorder)
        "28864000",  # Sedative, hypnotic AND/OR anxiolytic-induced mood disorder (disorder)
        "29212009",  # Alcohol-induced organic mental disorder (disorder)
        "29733004",  # Opioid-induced mood disorder (disorder)
        "30491001",  # Cocaine delusional disorder (disorder)
        "31956009",  # Cocaine dependence (disorder)
        "32009006",  # Hallucinogen hallucinosis (disorder)
        "32358001",  # Amphetamine delusional disorder (disorder)
        "32875003",  # Inhalant-induced persisting dementia (disorder)
        "34938008",  # Alcohol-induced anxiety disorder (disorder)
        "37344009",  # Cannabis abuse (disorder)
        "38247002",  # Hallucinogen dependence (disorder)
        "39807006",  # Cannabis intoxication delirium (disorder)
        "39951001",  # Cannabis-induced anxiety disorder (disorder)
        "40571009",  # Hallucinogen intoxication delirium (disorder)
        "41083005",  # Alcohol-induced sleep disorder (disorder)
        "42344001",  # Alcohol-induced psychosis (disorder)
        "43497001",  # Amphetamine-induced mood disorder (disorder)
        "46975003",  # Cocaine-induced organic mental disorder (disorder)
        "47664006",  # Sedative, hypnotic AND/OR anxiolytic-induced psychotic disorder with hallucinations (disorder)
        "50320000",  # Hallucinogen intoxication (disorder)
        "50933003",  # Hallucinogen delusional disorder (disorder)
        "51443000",  # Amphetamine-induced psychotic disorder with hallucinations (disorder)
        "51493001",  # Cocaine-induced anxiety disorder (disorder)
        "52866005",  # Opioid intoxication delirium (disorder)
        "53050002",  # Hallucinogen-induced organic mental disorder (disorder)
        "53936005",  # Alcohol-induced mood disorder (disorder)
        "56194001",  # Caffeine-induced sleep disorder (disorder)
        "57588009",  # Sedative, hypnotic AND/OR anxiolytic-induced sleep disorder (disorder)
        "59651006",  # Sedative, hypnotic AND/OR anxiolytic-induced persisting dementia (disorder)
        "61104008",  # Inhalant-induced organic mental disorder (disorder)
        "61144001",  # Alcohol-induced psychotic disorder with delusions (disorder)
        "63649001",  # Cannabis delusional disorder (disorder)
        "63983005",  # Inhalant-induced psychotic disorder with hallucinations (disorder)
        "64386003",  # Sedative abuse (disorder)
        "66590003",  # Alcohol dependence (disorder)
        "70328006",  # Cocaine delirium (disorder)
        "70340006",  # Inhalant abuse (disorder)
        "70701004",  # Alcoholic amblyopia (disorder)
        "70932007",  # Amphetamine-induced sexual dysfunction (disorder)
        "71328000",  # Opioid-induced sexual dysfunction (disorder)
        "73097000",  # Alcohol amnestic disorder (disorder)
        "74851005",  # Hallucinogen abuse (disorder)
        "74934004",  # Psychoactive substance-induced withdrawal syndrome (disorder)
        "75122001",  # Inhalant-induced psychotic disorder with delusions (disorder)
        "75544000",  # Opioid dependence (disorder)
        "77355000",  # Cannabis-induced organic mental disorder (disorder)
        "78267003",  # Cocaine abuse (disorder)
        "78358001",  # Amphetamine withdrawal (disorder)
        "80868005",  # Cocaine withdrawal (disorder)
        "82339009",  # Amphetamine-induced anxiety disorder (disorder)
        "83367009",  # Amphetamine-induced organic mental disorder (disorder)
        "84758004",  # Amphetamine abuse (disorder)
        "85005007",  # Cannabis dependence (disorder)
        "85561006",  # Uncomplicated alcohol withdrawal (disorder)
        "86325007",  # Non megaloblastic anemia due to alcoholism (disorder)
        "87132004",  # Opioid withdrawal (disorder)
        "87810006",  # Megaloblastic anemia caused by alcoholism (disorder)
        "88926005",  # Opioid-induced sleep disorder (disorder)
        "89451009",  # Inhalant-induced mood disorder (disorder)
        "91388009",  # Psychoactive substance abuse (disorder)
        "95635002",  # Caffeine withdrawal (disorder)
        "95661003",  # Ergotamine withdrawal (disorder)
        "191475009",  # Chronic alcoholic brain syndrome (disorder)
        "191476005",  # Alcohol withdrawal hallucinosis (disorder)
        "191478006",  # Alcoholic paranoia (disorder)
        "191480000",  # Alcohol withdrawal syndrome (disorder)
        "191802004",  # Acute alcoholic intoxication in alcoholism (disorder)
        "191804003",  # Continuous acute alcoholic intoxication in alcoholism (disorder)
        "191805002",  # Episodic acute alcoholic intoxication in alcoholism (disorder)
        "191811004",  # Continuous chronic alcoholism (disorder)
        "191812006",  # Episodic chronic alcoholism (disorder)
        "191819002",  # Continuous opioid dependence (disorder)
        "191820008",  # Episodic opioid dependence (disorder)
        "191825003",  # Hypnotic or anxiolytic dependence, continuous (disorder)
        "191826002",  # Hypnotic or anxiolytic dependence, episodic (disorder)
        "191831000",  # Cocaine dependence, continuous (disorder)
        "191832007",  # Cocaine dependence, episodic (disorder)
        "191837001",  # Cannabis dependence, continuous (disorder)
        "191838006",  # Cannabis dependence, episodic (disorder)
        "191843004",  # Amphetamine or psychostimulant dependence, continuous (disorder)
        "191844005",  # Amphetamine or psychostimulant dependence, episodic (disorder)
        "191849000",  # Hallucinogen dependence, continuous (disorder)
        "191850000",  # Hallucinogen dependence, episodic (disorder)
        "191853003",  # Glue sniffing dependence (disorder)
        "191855005",  # Glue sniffing dependence, continuous (disorder)
        "191856006",  # Glue sniffing dependence, episodic (disorder)
        "191865004",  # Combined opioid with non-opioid drug dependence (disorder)
        "191867007",  # Combined opioid with non-opioid drug dependence, continuous (disorder)
        "191868002",  # Combined opioid with non-opioid drug dependence, episodic (disorder)
        "191882002",  # Nondependent alcohol abuse, continuous (disorder)
        "191883007",  # Nondependent alcohol abuse, episodic (disorder)
        "191891003",  # Nondependent cannabis abuse (disorder)
        "191893000",  # Nondependent cannabis abuse, continuous (disorder)
        "191894006",  # Nondependent cannabis abuse, episodic (disorder)
        "191899001",  # Nondependent hallucinogen abuse, continuous (disorder)
        "191900006",  # Nondependent hallucinogen abuse, episodic (disorder)
        "191905001",  # Nondependent hypnotic or anxiolytic abuse, continuous (disorder)
        "191906000",  # Nondependent hypnotic or anxiolytic abuse, episodic (disorder)
        "191909007",  # Nondependent opioid abuse (disorder)
        "191912005",  # Nondependent opioid abuse, continuous (disorder)
        "191913000",  # Nondependent opioid abuse, episodic (disorder)
        "191916008",  # Nondependent cocaine abuse (disorder)
        "191918009",  # Nondependent cocaine abuse, continuous (disorder)
        "191919001",  # Nondependent cocaine abuse, episodic (disorder)
        "191924003",  # Nondependent amphetamine or psychostimulant abuse, continuous (disorder)
        "191925002",  # Nondependent amphetamine or psychostimulant abuse, episodic (disorder)
        "230443000",  # Narcotic withdrawal epilepsy (disorder)
        "231461004",  # Hypnotic or anxiolytic abuse (disorder)
        "231462006",  # Barbiturate abuse (disorder)
        "231468005",  # Lysergic acid diethylamide dependence (disorder)
        "231469002",  # Mescaline dependence (disorder)
        "231470001",  # Anxiolytic dependence (disorder)
        "231472009",  # Barbiturate dependence (disorder)
        "231473004",  # Benzodiazepine dependence (disorder)
        "231474005",  # Diazepam dependence (disorder)
        "231477003",  # Heroin dependence (disorder)
        "231478008",  # Methadone dependence (disorder)
        "231479000",  # Morphine dependence (disorder)
        "231480002",  # Opium dependence (disorder)
        "268640002",  # Hypnotic or anxiolytic dependence (disorder)
        "268645007",  # Nondependent alcohol abuse (disorder)
        "268646008",  # Nondependent hallucinogen abuse (disorder)
        "268647004",  # Nondependent hypnotic or anxiolytic abuse (disorder)
        "268648009",  # Nondependent amphetamine or other psychostimulant abuse (disorder)
        "275471001",  # Psychostimulant dependence (disorder)
        "284591009",  # Persistent alcohol abuse (disorder)
        "301643003",  # Sedative, hypnotic AND/OR anxiolytic-induced persisting amnestic disorder (disorder)
        "308742005",  # Alcohol withdrawal-induced convulsion (disorder)
        "312098001",  # Sedative, hypnotic AND/OR anxiolytic-related disorder (disorder)
        "312936002",  # Anxiolytic-induced organic mental disorder (disorder)
        "361151007",  # Sedative withdrawal delirium (disorder)
        "416714005",  # Episodic flashbacks (disorder)
        "417143004",  # Short duration flashbacks (disorder)
        "425339005",  # Barbiturate withdrawal (disorder)
        "425885002",  # Continuous phencyclidine abuse (disorder)
        "426001001",  # Fentanyl dependence (disorder)
        "426095000",  # Continuous inhalant abuse (disorder)
        "426873000",  # Methamphetamine dependence (disorder)
        "427205009",  # Amphetamine abuse, continuous (disorder)
        "427229002",  # Episodic inhalant abuse (disorder)
        "427327003",  # Sedative dependence (disorder)
        "428219007",  # Caffeine dependence (disorder)
        "428370001",  # Psychostimulant withdrawal (disorder)
        "429692000",  # Amphetamine abuse, episodic (disorder)
        "441527004",  # Stimulant abuse (disorder)
        "442406005",  # Stimulant dependence (disorder)
        "699449003",  # Methamphetamine abuse (disorder)
        "703845008",  # Buprenorphine dependence (disorder)
        "703848005",  # Cannabis withdrawal (disorder)
        "703849002",  # Benzodiazepine withdrawal (disorder)
        "703850002",  # Benzodiazepine withdrawal delirium (disorder)
        "713583005",  # Mild alcohol dependence (disorder)
        "713862009",  # Severe alcohol dependence (disorder)
        "714829008",  # Moderate alcohol dependence (disorder)
        "723926008",  # Perceptual disturbances and seizures co-occurrent and due to alcohol withdrawal (disorder)
        "723927004",  # Psychotic disorder caused by alcohol with schizophreniform symptoms (disorder)
        "723928009",  # Mood disorder with depressive symptoms caused by alcohol (disorder)
        "723929001",  # Mood disorder with manic symptoms caused by alcohol (disorder)
        "723930006",  # Mood disorder with mixed manic and depressive symptoms caused by alcohol (disorder)
        "723933008",  # Cannabis dependence with current use (disorder)
        "723936000",  # Psychotic disorder caused by cannabis (disorder)
        "723937009",  # Sleep disorder caused by cannabis (disorder)
        "724653003",  # Opioid dependence with current use (disorder)
        "724655005",  # Psychotic disorder caused by opioid (disorder)
        "724656006",  # Hypnotic dependence (disorder)
        "724657002",  # Sedative dependence with current use (disorder)
        "724658007",  # Hypnotic dependence with current use (disorder)
        "724659004",  # Anxiolytic dependence with current use (disorder)
        "724660009",  # Hypnotic withdrawal (disorder)
        "724661008",  # Anxiolytic withdrawal (disorder)
        "724662001",  # Sedative withdrawal without complication (disorder)
        "724663006",  # Hypnotic withdrawal without complication (disorder)
        "724664000",  # Anxiolytic withdrawal without complication (disorder)
        "724665004",  # Perceptual disturbances co-occurrent and due to sedative withdrawal (disorder)
        "724666003",  # Perceptual disturbances co-occurrent and due to hypnotic withdrawal (disorder)
        "724667007",  # Perceptual disturbances co-occurrent and due to anxiolytic withdrawal (disorder)
        "724668002",  # Seizure co-occurrent and due to sedative withdrawal (disorder)
        "724669005",  # Seizure co-occurrent and due to hypnotic withdrawal (disorder)
        "724670006",  # Seizure co-occurrent and due to anxiolytic withdrawal (disorder)
        "724671005",  # Perceptual disturbances and seizures co-occurrent and due to sedative withdrawal (disorder)
        "724672003",  # Perceptual disturbances and seizures co-occurrent and due to hypnotic withdrawal (disorder)
        "724675001",  # Psychotic disorder caused by anxiolytic (disorder)
        "724688003",  # Cocaine dependence with current use (disorder)
        "724689006",  # Psychotic disorder caused by cocaine (disorder)
        "724690002",  # Mood disorder with depressive symptoms caused by cocaine (disorder)
        "724691003",  # Mood disorder with manic symptoms caused by cocaine (disorder)
        "724692005",  # Mood disorder with mixed depressive and manic symptoms caused by cocaine (disorder)
        "724693000",  # Obsessive compulsive disorder caused by cocaine (disorder)
        "724695007",  # Hallucinogen dependence with current use (disorder)
        "724696008",  # Psychotic disorder caused by hallucinogen (disorder)
        "724700000",  # Volatile inhalant dependence with current use (disorder)
        "724701001",  # Volatile inhalant withdrawal (disorder)
        "724702008",  # Psychotic disorder caused by volatile inhalant (disorder)
        "724704009",  # Methylenedioxymethamphetamine dependence with current use (disorder)
        "724706006",  # Psychotic disorder caused by methylenedioxymethamphetamine (disorder)
        "724712001",  # Harmful use of dissociative drug (disorder)
        "724714000",  # Dependence caused by dissociative substance (disorder)
        "724715004",  # Dependence caused by ketamine (disorder)
        "724720004",  # Mood disorder caused by dissociative drug (disorder)
        "724724008",  # Psychoactive substance dependence with current use (disorder)
        "724725009",  # Psychoactive substance withdrawal without complication (disorder)
        "724726005",  # Perceptual disturbances co-occurrent and due to psychoactive substance withdrawal (disorder)
        "724727001",  # Seizure co-occurrent and due to psychoactive substance withdrawal (disorder)
        "724728006",  # Perceptual disturbances and seizures co-occurrent and due to psychoactive substance withdrawal (disorder)
        "724732000",  # Sleep disorder caused by psychoactive substance (disorder)
        "735750005",  # Psychotic disorder with schizophreniform symptoms caused by cocaine (disorder)
        "737338002",  # Synthetic cannabinoid withdrawal (disorder)
        "762320004",  # Mood disorder with manic symptoms caused by opioid (disorder)
        "762321000",  # Mood disorder with depressive symptoms caused by opioid (disorder)
        "762322007",  # Mood disorder with mixed depressive and manic symptoms caused by opioid (disorder)
        "762324008",  # Delirium caused by stimulant (disorder)
        "762325009",  # Psychotic disorder caused by stimulant (disorder)
        "762326005",  # Psychotic disorder with hallucinations caused by stimulant (disorder)
        "762327001",  # Psychotic disorder with delusions caused by stimulant (disorder)
        "762333005",  # Sexual dysfunction caused by stimulant (disorder)
        "762334004",  # Sleep disorder caused by stimulant (disorder)
        "762335003",  # Mood disorder with manic symptoms caused by hallucinogen (disorder)
        "762336002",  # Mood disorder with depressive symptoms caused by hallucinogen (disorder)
        "762337006",  # Mood disorder with mixed depressive and manic symptoms caused by hallucinogen (disorder)
        "762338001",  # Mood disorder with manic symptoms caused by volatile inhalant (disorder)
        "762339009",  # Mood disorder with depressive symptoms caused by volatile inhalant (disorder)
        "762340006",  # Mood disorder with mixed depressive and manic symptoms caused by volatile inhalant (disorder)
        "762344002",  # Mood disorder with manic symptoms caused by dissociative drug (disorder)
        "762345001",  # Mood disorder with depressive symptoms caused by dissociative drug (disorder)
        "762346000",  # Mood disorder with mixed depressive and manic symptoms caused by dissociative drug (disorder)
        "762517008",  # Sexual dysfunction caused by synthetic cathinone (disorder)
        "762672001",  # Synthetic cathinone withdrawal (disorder)
        "785277001",  # Harmful pattern of use of amfetamine and amfetamine derivative (disorder)
        "788863007",  # Amnestic disorder caused by psychoactive substance (disorder)
        "788904003",  # Disorder caused by dissociative drug (disorder)
        "788905002",  # Disorder caused by stimulant (disorder)
        "788955005",  # Impulse control disorder caused by cocaine (disorder)
        "788956006",  # Impulse control disorder caused by psychoactive substance (disorder)
        "788957002",  # Impulse control disorder caused by stimulant (disorder)
        "788983007",  # Mood disorder caused by cannabis (disorder)
        "789119007",  # Mood disorder caused by synthetic cannabinoid (disorder)
        "817962007",  # Chlordiazepoxide dependence (disorder)
        "838527002",  # Amfetamine and amfetamine derivative drug dependence (disorder)
        "838528007",  # Mood disorder with manic symptoms caused by amfetamine and amfetamine derivative (disorder)
        "838529004",  # Mood disorder with mixed depressive and manic symptoms caused by amfetamine and amfetamine derivative (disorder)
        "838530009",  # Mood disorder with depressive symptoms caused by amfetamine and amfetamine derivative (disorder)
        "1081000119105",  # Opioid dependence, on agonist therapy (disorder)
        "23601000119102",  # Sedative withdrawal (disorder)
        "34111000119108",  # Nondependent amphetamine abuse (disorder)
        "86391000119101",  # Inhalant dependence, continuous (disorder)
        "86401000119104",  # Inhalant dependence, episodic (disorder)
        "97571000119109",  # Thrombocytopenia co-occurrent and due to alcoholism (disorder)
        "125851000119106",  # Continuous sedative abuse (disorder)
        "135301000119103",  # Parasomnia caused by alcohol (disorder)
        "135311000119100",  # Insomnia caused by alcohol (disorder)
        "135321000119107",  # Hypersomnia caused by alcohol (disorder)
        "144981000119109",  # Nondependent intraveous amphetamine abuse (disorder)
        "145101000119102",  # Intravenous cocaine abuse (disorder)
        "145121000119106",  # Intravenous nondependent opioid abuse (disorder)
        "145841000119107",  # Episodic phencyclidine abuse (disorder)
        "154211000119108",  # Chronic pancreatitis due to chronic alcoholism (disorder)
        "288021000119107",  # Disorder due to alcohol abuse (disorder)
        "288031000119105",  # Alcohol induced disorder co-occurrent and due to alcohol dependence (disorder)
        "288461000119105",  # Psychosis caused by inhalant (disorder)
        "288851000119106",  # Opioid-induced mood disorder due to opioid abuse (disorder)
        "288861000119108",  # Opioid-induced mood disorder due to opioid dependence (disorder)
        "10741871000119101",  # Alcohol dependence in pregnancy (disorder)
        "10755041000119100",  # Alcohol dependence in childbirth (disorder)
        "11047881000119101",  # Cannabis hyperemesis syndrome co-occurrent and due to cannabis abuse (disorder)
        "11048011000119103",  # Cannabis hyperemesis syndrome co-occurrent and due to cannabis dependence (disorder)
        "12398651000119100",  # Sleep disorder caused by methamphetamine (disorder)
    }


class LimitedLifeExpectancy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  limited life expectancy, including poor prognosis or terminal illness.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis that indicates a poor prognosis of a patient’s health status such as terminal or pre-terminal illness.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS138v10
    """

    VALUE_SET_NAME = "Limited Life Expectancy"
    OID = "2.16.840.1.113883.3.526.3.1259"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "27143004",  # Patient status determination, pre-terminal (finding)
        "111947009",  # Functional disorder present, condition terminal (finding)
        "162607003",  # Terminal illness - early stage (finding)
        "162608008",  # Terminal illness - late stage (finding)
        "170969009",  # Prognosis bad (finding)
        "300936002",  # Terminal illness (finding)
    }


class PrimaryOpenAngleGlaucoma(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of primary open-angle glaucoma (POAG).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of primary open-angle glaucoma (POAG) and open-angle glaucoma.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS143v10
    """

    VALUE_SET_NAME = "Primary Open-Angle Glaucoma"
    OID = "2.16.840.1.113883.3.526.3.326"
    DEFINITION_VERSION = "20210209"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "H4010X0",  # Unspecified open-angle glaucoma, stage unspecified
        "H4010X1",  # Unspecified open-angle glaucoma, mild stage
        "H4010X2",  # Unspecified open-angle glaucoma, moderate stage
        "H4010X3",  # Unspecified open-angle glaucoma, severe stage
        "H4010X4",  # Unspecified open-angle glaucoma, indeterminate stage
        "H401110",  # Primary open-angle glaucoma, right eye, stage unspecified
        "H401111",  # Primary open-angle glaucoma, right eye, mild stage
        "H401112",  # Primary open-angle glaucoma, right eye, moderate stage
        "H401113",  # Primary open-angle glaucoma, right eye, severe stage
        "H401114",  # Primary open-angle glaucoma, right eye, indeterminate stage
        "H401120",  # Primary open-angle glaucoma, left eye, stage unspecified
        "H401121",  # Primary open-angle glaucoma, left eye, mild stage
        "H401122",  # Primary open-angle glaucoma, left eye, moderate stage
        "H401123",  # Primary open-angle glaucoma, left eye, severe stage
        "H401124",  # Primary open-angle glaucoma, left eye, indeterminate stage
        "H401130",  # Primary open-angle glaucoma, bilateral, stage unspecified
        "H401131",  # Primary open-angle glaucoma, bilateral, mild stage
        "H401132",  # Primary open-angle glaucoma, bilateral, moderate stage
        "H401133",  # Primary open-angle glaucoma, bilateral, severe stage
        "H401134",  # Primary open-angle glaucoma, bilateral, indeterminate stage
        "H401210",  # Low-tension glaucoma, right eye, stage unspecified
        "H401211",  # Low-tension glaucoma, right eye, mild stage
        "H401212",  # Low-tension glaucoma, right eye, moderate stage
        "H401213",  # Low-tension glaucoma, right eye, severe stage
        "H401214",  # Low-tension glaucoma, right eye, indeterminate stage
        "H401220",  # Low-tension glaucoma, left eye, stage unspecified
        "H401221",  # Low-tension glaucoma, left eye, mild stage
        "H401222",  # Low-tension glaucoma, left eye, moderate stage
        "H401223",  # Low-tension glaucoma, left eye, severe stage
        "H401224",  # Low-tension glaucoma, left eye, indeterminate stage
        "H401230",  # Low-tension glaucoma, bilateral, stage unspecified
        "H401231",  # Low-tension glaucoma, bilateral, mild stage
        "H401232",  # Low-tension glaucoma, bilateral, moderate stage
        "H401233",  # Low-tension glaucoma, bilateral, severe stage
        "H401234",  # Low-tension glaucoma, bilateral, indeterminate stage
        "H40151",  # Residual stage of open-angle glaucoma, right eye
        "H40152",  # Residual stage of open-angle glaucoma, left eye
        "H40153",  # Residual stage of open-angle glaucoma, bilateral
    }
    SNOMEDCT = {
        "50485007",  # Low tension glaucoma (disorder)
        "77075001",  # Primary open angle glaucoma (disorder)
        "84494001",  # Open-angle glaucoma (disorder)
        "111513000",  # Advanced open-angle glaucoma (disorder)
        "193533000",  # Open-angle glaucoma - borderline (disorder)
        "15633281000119103",  # Bilateral primary open angle glaucoma (disorder)
        "15633321000119108",  # Primary open angle glaucoma of left eye (disorder)
        "15640441000119104",  # Primary open angle glaucoma of right eye (disorder)
    }


class AllergyToBetaBlockerTherapy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of an allergy to beta blocker therapy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an allergy to beta blocker therapy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Allergy to Beta Blocker Therapy"
    OID = "2.16.840.1.113883.3.526.3.1177"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "293962009",  # Allergy to beta adrenergic receptor antagonist (finding)
        "293963004",  # Allergy to beta-1 adrenergic receptor antagonist (finding)
        "293964005",  # Allergy to acebutolol (finding)
        "293965006",  # Allergy to atenolol (finding)
        "293966007",  # Allergy to betaxolol (finding)
        "293967003",  # Allergy to bisoprolol (finding)
        "293968008",  # Allergy to celiprolol (finding)
        "293969000",  # Allergy to esmolol (finding)
        "293970004",  # Allergy to metoprolol (finding)
        "293972007",  # Allergy to nadolol (finding)
        "293973002",  # Allergy to pindolol (finding)
        "293974008",  # Allergy to carvedilol (finding)
        "293975009",  # Allergy to metipranolol (finding)
        "293976005",  # Allergy to carteolol (finding)
        "293977001",  # Allergy to labetalol (finding)
        "293978006",  # Allergy to levobunolol (finding)
        "293979003",  # Allergy to oxprenolol (finding)
        "293980000",  # Allergy to penbutolol (finding)
        "293981001",  # Allergy to practolol (finding)
        "293982008",  # Allergy to propranolol (finding)
        "293983003",  # Allergy to sotalol (finding)
        "293984009",  # Allergy to timolol (finding)
    }


class Arrhythmia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of arrhythmia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of arrhythmia.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Arrhythmia"
    OID = "2.16.840.1.113883.3.526.3.366"
    DEFINITION_VERSION = "20200306"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I498",  # Other specified cardiac arrhythmias
        "I499",  # Cardiac arrhythmia, unspecified
    }
    SNOMEDCT = {
        "184004",  # Withdrawal arrhythmia (disorder)
        "10164001",  # Parasystole (disorder)
        "10626002",  # Multifocal premature ventricular complexes (disorder)
        "11157007",  # Ventricular bigeminy (disorder)
        "11849007",  # Atrioventricular junctional rhythm (disorder)
        "13640000",  # Fusion beats (disorder)
        "17338001",  # Ventricular premature beats (disorder)
        "17366009",  # Atrial arrhythmia (disorder)
        "17869006",  # Anomalous atrioventricular excitation (disorder)
        "26950008",  # Chronic ectopic atrial tachycardia (disorder)
        "27337007",  # Unifocal premature ventricular complexes (disorder)
        "29320008",  # Ectopic rhythm (disorder)
        "33413000",  # Ectopic beats (disorder)
        "36083008",  # Sick sinus syndrome (disorder)
        "38274001",  # Interpolated ventricular premature complexes (disorder)
        "39260000",  # Nonparoxysmal atrioventricular nodal tachycardia (disorder)
        "39357005",  # Paroxysmal atrial tachycardia with block (disorder)
        "40593004",  # Fibrillation (disorder)
        "44808001",  # Conduction disorder of the heart (disorder)
        "47830009",  # Junctional escape beats (disorder)
        "49982000",  # Multifocal atrial tachycardia (disorder)
        "55475008",  # Lown-Ganong-Levine syndrome (disorder)
        "59272004",  # Ventricular parasystole (disorder)
        "60423000",  # Sinus node dysfunction (disorder)
        "61277005",  # Accelerated idioventricular rhythm (disorder)
        "63232000",  # Multifocal premature beats (disorder)
        "63593006",  # Supraventricular premature beats (disorder)
        "69730002",  # Idiojunctional tachycardia (disorder)
        "71908006",  # Ventricular fibrillation (disorder)
        "72654001",  # Supraventricular arrhythmia (disorder)
        "74390002",  # Wolff-Parkinson-White pattern (disorder)
        "74615001",  # Tachycardia-bradycardia (disorder)
        "75532003",  # Ventricular escape beat (disorder)
        "81681009",  # Junctional premature beats (disorder)
        "81898007",  # Ventricular escape rhythm (disorder)
        "88412007",  # Atrio-ventricular node arrhythmia (disorder)
        "195060002",  # Ventricular pre-excitation (disorder)
        "195069001",  # Paroxysmal atrial tachycardia (disorder)
        "195071001",  # Paroxysmal junctional tachycardia (disorder)
        "195072008",  # Paroxysmal nodal tachycardia (disorder)
        "195083004",  # Ventricular fibrillation and flutter (disorder)
        "233891009",  # Sinoatrial node tachycardia (disorder)
        "233892002",  # Ectopic atrial tachycardia (disorder)
        "233893007",  # Re-entrant atrial tachycardia (disorder)
        "233894001",  # Incessant atrial tachycardia (disorder)
        "233895000",  # Ectopic atrioventricular node tachycardia (disorder)
        "233904005",  # Permanent junctional reciprocating tachycardia (disorder)
        "233915000",  # Paroxysmal familial ventricular fibrillation (disorder)
        "233922008",  # Concealed accessory pathway (disorder)
        "233923003",  # Unidirectional retrograde accessory pathway (disorder)
        "234172002",  # Electromechanical dissociation (disorder)
        "251161003",  # Slow ventricular response (disorder)
        "251162005",  # Atrio-ventricular-junctional (nodal) bradycardia (disorder)
        "251163000",  # Atrio-ventricular junctional (nodal) arrest (disorder)
        "251164006",  # Junctional premature complex (disorder)
        "251165007",  # Atrioventricular junctional (nodal) tachycardia (disorder)
        "251166008",  # Atrioventricular nodal re-entry tachycardia (disorder)
        "251167004",  # Aberrant premature complexes (disorder)
        "251168009",  # Supraventricular bigeminy (disorder)
        "251170000",  # Blocked premature atrial contraction (disorder)
        "251172008",  # Run of atrial premature complexes (disorder)
        "251173003",  # Atrial bigeminy (disorder)
        "251174009",  # Atrial trigeminy (disorder)
        "251175005",  # Ventricular premature complex (disorder)
        "251176006",  # Multiple premature ventricular complexes (disorder)
        "251177002",  # Run of ventricular premature complexes (disorder)
        "251178007",  # Ventricular interpolated complexes (disorder)
        "251179004",  # Multiple ventricular interpolated complexes (disorder)
        "251180001",  # Ventricular trigeminy (disorder)
        "251181002",  # Ventricular quadrigeminy (disorder)
        "251182009",  # Paired ventricular premature complexes (disorder)
        "251186007",  # Ventricular escape complex (disorder)
        "251187003",  # Atrial escape complex (disorder)
        "251188008",  # Atrial parasystole (disorder)
        "276796006",  # Atrial tachycardia (disorder)
        "284470004",  # Premature atrial contraction (disorder)
        "287057009",  # Atrial premature complex (disorder)
        "309809007",  # Electromechanical dissociation with successful resuscitation (disorder)
        "406461004",  # Ectopic atrial beats (disorder)
        "418341009",  # Atrioventricular conduction disorder (disorder)
        "418818005",  # Brugada syndrome (disorder)
        "419752005",  # Sinoatrial nodal reentrant tachycardia (disorder)
        "421869004",  # Bradyarrhythmia (disorder)
        "422348008",  # Andersen Tawil syndrome (disorder)
        "429243003",  # Sustained ventricular fibrillation (disorder)
    }


class Asthma(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for asthma.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of asthma.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Asthma"
    OID = "2.16.840.1.113883.3.526.3.362"
    DEFINITION_VERSION = "20200306"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "J4520",  # Mild intermittent asthma, uncomplicated
        "J4521",  # Mild intermittent asthma with (acute) exacerbation
        "J4522",  # Mild intermittent asthma with status asthmaticus
        "J4530",  # Mild persistent asthma, uncomplicated
        "J4531",  # Mild persistent asthma with (acute) exacerbation
        "J4532",  # Mild persistent asthma with status asthmaticus
        "J4540",  # Moderate persistent asthma, uncomplicated
        "J4541",  # Moderate persistent asthma with (acute) exacerbation
        "J4542",  # Moderate persistent asthma with status asthmaticus
        "J4550",  # Severe persistent asthma, uncomplicated
        "J4551",  # Severe persistent asthma with (acute) exacerbation
        "J4552",  # Severe persistent asthma with status asthmaticus
        "J45901",  # Unspecified asthma with (acute) exacerbation
        "J45902",  # Unspecified asthma with status asthmaticus
        "J45909",  # Unspecified asthma, uncomplicated
        "J45990",  # Exercise induced bronchospasm
        "J45991",  # Cough variant asthma
        "J45998",  # Other asthma
    }
    SNOMEDCT = {
        "11641008",  # Millers' asthma (disorder)
        "12428000",  # Intrinsic asthma without status asthmaticus (disorder)
        "13151001",  # Flax-dressers' disease (disorder)
        "31387002",  # Exercise-induced asthma (disorder)
        "55570000",  # Asthma without status asthmaticus (disorder)
        "56968009",  # Asthma caused by wood dust (disorder)
        "59786004",  # Weavers' cough (disorder)
        "63088003",  # Allergic asthma without status asthmaticus (disorder)
        "85761009",  # Byssinosis (disorder)
        "92807009",  # Chemical-induced asthma (disorder)
        "93432008",  # Drug-induced asthma (disorder)
        "195949008",  # Chronic asthmatic bronchitis (disorder)
        "195967001",  # Asthma (disorder)
        "195977004",  # Mixed asthma (disorder)
        "225057002",  # Brittle asthma (disorder)
        "233672007",  # Byssinosis grade 3 (disorder)
        "233678006",  # Childhood asthma (disorder)
        "233679003",  # Late onset asthma (disorder)
        "233683003",  # Hay fever with asthma (disorder)
        "233688007",  # Sulfite-induced asthma (disorder)
        "266361008",  # Non-allergic asthma (disorder)
        "281239006",  # Exacerbation of asthma (disorder)
        "304527002",  # Acute asthma (disorder)
        "370218001",  # Mild asthma (disorder)
        "370219009",  # Moderate asthma (disorder)
        "370220003",  # Occasional asthma (disorder)
        "370221004",  # Severe asthma (disorder)
        "389145006",  # Allergic asthma (disorder)
        "405944004",  # Asthmatic bronchitis (disorder)
        "407674008",  # Aspirin-induced asthma (disorder)
        "409663006",  # Cough variant asthma (disorder)
        "423889005",  # Non-immunoglobulin E mediated allergic asthma (disorder)
        "424199006",  # Substance induced asthma (disorder)
        "424643009",  # Immunoglobulin E-mediated allergic asthma (disorder)
        "425969006",  # Exacerbation of intermittent asthma (disorder)
        "426656000",  # Severe persistent asthma (disorder)
        "426979002",  # Mild persistent asthma (disorder)
        "427295004",  # Moderate persistent asthma (disorder)
        "427603009",  # Intermittent asthma (disorder)
        "427679007",  # Mild intermittent asthma (disorder)
        "442025000",  # Acute exacerbation of chronic asthmatic bronchitis (disorder)
        "707445000",  # Exacerbation of mild persistent asthma (disorder)
        "707446004",  # Exacerbation of moderate persistent asthma (disorder)
        "707447008",  # Exacerbation of severe persistent asthma (disorder)
        "708038006",  # Acute exacerbation of asthma (disorder)
        "708090002",  # Acute severe exacerbation of asthma (disorder)
        "708093000",  # Acute exacerbation of immunoglobulin E-mediated allergic asthma (disorder)
        "708094006",  # Acute exacerbation of intrinsic asthma (disorder)
        "708095007",  # Acute severe exacerbation of immunoglobin E-mediated allergic asthma (disorder)
        "708096008",  # Acute severe exacerbation of intrinsic asthma (disorder)
        "782520007",  # Exacerbation of allergic asthma due to infection (disorder)
    }


class AtrioventricularBlock(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of atrioventricular block.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of atrioventricular block.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Atrioventricular Block"
    OID = "2.16.840.1.113883.3.526.3.367"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I440",  # Atrioventricular block, first degree
        "I441",  # Atrioventricular block, second degree
        "I442",  # Atrioventricular block, complete
        "I4430",  # Unspecified atrioventricular block
        "I4439",  # Other atrioventricular block
    }
    SNOMEDCT = {
        "27885002",  # Complete atrioventricular block (disorder)
        "28189009",  # Mobitz type II atrioventricular block (disorder)
        "54016002",  # Mobitz type I incomplete atrioventricular block (disorder)
        "195042002",  # Second degree atrioventricular block (disorder)
        "204383001",  # Congenital complete atrioventricular heart block (disorder)
        "233917008",  # Atrioventricular block (disorder)
        "233918003",  # Postoperative complete heart block (disorder)
        "251114004",  # Intermittent second degree atrioventricular block (disorder)
        "283645003",  # Lev's syndrome (disorder)
        "733125004",  # Acquired complete atrioventricular block (disorder)
        "102451000119107",  # Complete atrioventricular block as complication of atrioventricular nodal ablation (disorder)
        "284941000119107",  # High degree second degree atrioventricular block (disorder)
    }


class Bradycardia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of bradycardia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of bradycardia.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Bradycardia"
    OID = "2.16.840.1.113883.3.526.3.412"
    DEFINITION_VERSION = "20200306"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I495",  # Sick sinus syndrome
        "I498",  # Other specified cardiac arrhythmias
        "R001",  # Bradycardia, unspecified
    }
    SNOMEDCT = {
        "29894000",  # Vagal autonomic bradycardia (disorder)
        "44602002",  # Persistent sinus bradycardia (disorder)
        "49044005",  # Severe sinus bradycardia (disorder)
        "49710005",  # Sinus bradycardia (disorder)
        "251162005",  # Atrio-ventricular-junctional (nodal) bradycardia (disorder)
        "397841007",  # Drug-induced bradycardia (disorder)
    }


class CardiacPacerInSitu(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of cardiac pacer in situ.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of cardiac pacer in situ.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Cardiac Pacer in Situ"
    OID = "2.16.840.1.113883.3.526.3.368"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "Z950",  # Presence of cardiac pacemaker
        "Z95810",  # Presence of automatic (implantable) cardiac defibrillator
    }
    SNOMEDCT = {
        "441509002",  # Cardiac pacemaker in situ (finding)
    }


class Hypotension(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of hypotension.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hypotension.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Hypotension"
    OID = "2.16.840.1.113883.3.526.3.370"
    DEFINITION_VERSION = "20200306"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I950",  # Idiopathic hypotension
        "I951",  # Orthostatic hypotension
        "I952",  # Hypotension due to drugs
        "I953",  # Hypotension of hemodialysis
        "I9581",  # Postprocedural hypotension
        "I9589",  # Other hypotension
        "I959",  # Hypotension, unspecified
    }
    SNOMEDCT = {
        "45007003",  # Low blood pressure (disorder)
        "61933008",  # Hyperadrenergic postural hypotension (disorder)
        "70247006",  # Hypoadrenergic postural hypotension (disorder)
        "75181005",  # Chronic orthostatic hypotension (disorder)
        "77545000",  # Chronic hypotension (disorder)
        "88887003",  # Maternal hypotension syndrome (disorder)
        "195506001",  # Idiopathic hypotension (disorder)
        "200113008",  # Maternal hypotension syndrome with antenatal problem (disorder)
        "200114002",  # Maternal hypotension syndrome with postnatal problem (disorder)
        "230664009",  # Sympathotonic orthostatic hypotension (disorder)
        "234171009",  # Drug-induced hypotension (disorder)
        "271870002",  # Low blood pressure reading (disorder)
        "286963007",  # Chronic hypotension - idiopathic (disorder)
        "371073003",  # Postural orthostatic tachycardia syndrome (disorder)
        "408667000",  # Hemodialysis-associated hypotension (disorder)
        "408668005",  # Iatrogenic hypotension (disorder)
        "429561008",  # Exertional hypotension (disorder)
    }


class IntoleranceToBetaBlockerTherapy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of an intolerance to beta blocker therapy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an intolerance to beta blocker therapy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = "Intolerance to Beta Blocker Therapy"
    OID = "2.16.840.1.113883.3.526.3.1178"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    SNOMEDCT = {
        "292419005",  # Beta-adrenoceptor blocking drug adverse reaction (disorder)
        "292420004",  # Cardioselective beta-blocker adverse reaction (disorder)
        "292421000",  # Acebutolol adverse reaction (disorder)
        "292424008",  # Betaxolol adverse reaction (disorder)
        "292425009",  # Bisoprolol adverse reaction (disorder)
        "292426005",  # Celiprolol adverse reaction (disorder)
        "292427001",  # Esmolol adverse reaction (disorder)
        "292428006",  # Metoprolol adverse reaction (disorder)
        "292430008",  # Nadolol adverse reaction (disorder)
        "292431007",  # Pindolol adverse reaction (disorder)
        "292432000",  # Carvedilol adverse reaction (disorder)
        "292433005",  # Metipranolol adverse reaction (disorder)
        "292434004",  # Carteolol adverse reaction (disorder)
        "292435003",  # Labetalol adverse reaction (disorder)
        "292436002",  # Levobunolol adverse reaction (disorder)
        "292437006",  # Oxprenolol adverse reaction (disorder)
        "292438001",  # Penbutolol adverse reaction (disorder)
        "292439009",  # Practolol adverse reaction (disorder)
        "292440006",  # Propranolol adverse reaction (disorder)
        "292441005",  # Sotalol adverse reaction (disorder)
        "292442003",  # Timolol adverse reaction (disorder)
        "418370000",  # Atenolol adverse reaction (disorder)
    }


class CoronaryArteryDiseaseNoMi(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of coronary artery disease without myocardial infarction.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of coronary artery disease without myocardial infarction.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS145v10
    """

    VALUE_SET_NAME = "Coronary Artery Disease No MI"
    OID = "2.16.840.1.113883.3.526.3.369"
    DEFINITION_VERSION = "20200306"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I200",  # Unstable angina
        "I201",  # Angina pectoris with documented spasm
        "I208",  # Other forms of angina pectoris
        "I209",  # Angina pectoris, unspecified
        "I240",  # Acute coronary thrombosis not resulting in myocardial infarction
        "I248",  # Other forms of acute ischemic heart disease
        "I249",  # Acute ischemic heart disease, unspecified
        "I2510",  # Atherosclerotic heart disease of native coronary artery without angina pectoris
        "I25110",  # Atherosclerotic heart disease of native coronary artery with unstable angina pectoris
        "I25111",  # Atherosclerotic heart disease of native coronary artery with angina pectoris with documented spasm
        "I25118",  # Atherosclerotic heart disease of native coronary artery with other forms of angina pectoris
        "I25119",  # Atherosclerotic heart disease of native coronary artery with unspecified angina pectoris
        "I255",  # Ischemic cardiomyopathy
        "I256",  # Silent myocardial ischemia
        "I25700",  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with unstable angina pectoris
        "I25701",  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with angina pectoris with documented spasm
        "I25708",  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with other forms of angina pectoris
        "I25709",  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with unspecified angina pectoris
        "I25710",  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with unstable angina pectoris
        "I25711",  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with angina pectoris with documented spasm
        "I25718",  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with other forms of angina pectoris
        "I25719",  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with unspecified angina pectoris
        "I25720",  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with unstable angina pectoris
        "I25721",  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with angina pectoris with documented spasm
        "I25728",  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with other forms of angina pectoris
        "I25729",  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with unspecified angina pectoris
        "I25730",  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with unstable angina pectoris
        "I25731",  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with angina pectoris with documented spasm
        "I25738",  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with other forms of angina pectoris
        "I25739",  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with unspecified angina pectoris
        "I25750",  # Atherosclerosis of native coronary artery of transplanted heart with unstable angina
        "I25751",  # Atherosclerosis of native coronary artery of transplanted heart with angina pectoris with documented spasm
        "I25758",  # Atherosclerosis of native coronary artery of transplanted heart with other forms of angina pectoris
        "I25759",  # Atherosclerosis of native coronary artery of transplanted heart with unspecified angina pectoris
        "I25760",  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with unstable angina
        "I25761",  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with angina pectoris with documented spasm
        "I25768",  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with other forms of angina pectoris
        "I25769",  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with unspecified angina pectoris
        "I25790",  # Atherosclerosis of other coronary artery bypass graft(s) with unstable angina pectoris
        "I25791",  # Atherosclerosis of other coronary artery bypass graft(s) with angina pectoris with documented spasm
        "I25798",  # Atherosclerosis of other coronary artery bypass graft(s) with other forms of angina pectoris
        "I25799",  # Atherosclerosis of other coronary artery bypass graft(s) with unspecified angina pectoris
        "I25810",  # Atherosclerosis of coronary artery bypass graft(s) without angina pectoris
        "I25811",  # Atherosclerosis of native coronary artery of transplanted heart without angina pectoris
        "I25812",  # Atherosclerosis of bypass graft of coronary artery of transplanted heart without angina pectoris
        "I2582",  # Chronic total occlusion of coronary artery
        "I2583",  # Coronary atherosclerosis due to lipid rich plaque
        "I2589",  # Other forms of chronic ischemic heart disease
        "I259",  # Chronic ischemic heart disease, unspecified
        "Z951",  # Presence of aortocoronary bypass graft
        "Z955",  # Presence of coronary angioplasty implant and graft
        "Z9861",  # Coronary angioplasty status
    }
    SNOMEDCT = {
        "4557003",  # Preinfarction syndrome (disorder)
        "10365005",  # Right main coronary artery thrombosis (disorder)
        "19057007",  # Status anginosus (disorder)
        "21470009",  # Syncope anginosa (disorder)
        "28248000",  # Left anterior descending coronary artery thrombosis (disorder)
        "29899005",  # Coronary artery embolism (disorder)
        "35928006",  # Nocturnal angina (disorder)
        "41334000",  # Angina, class II (disorder)
        "50570003",  # Aneurysm of coronary vessels (disorder)
        "53741008",  # Coronary arteriosclerosis (disorder)
        "59021001",  # Angina decubitus (disorder)
        "61490001",  # Angina, class I (disorder)
        "62207008",  # Syphilitic ostial coronary disease (disorder)
        "63739005",  # Coronary occlusion (disorder)
        "67682002",  # Coronary artery atheroma (disorder)
        "74218008",  # Coronary artery arising from main pulmonary artery (disorder)
        "75398000",  # Anomalous origin of coronary artery (disorder)
        "85284003",  # Angina, class III (disorder)
        "87343002",  # Prinzmetal angina (disorder)
        "89323001",  # Angina, class IV (disorder)
        "92517006",  # Calcific coronary arteriosclerosis (disorder)
        "123641001",  # Left coronary artery occlusion (disorder)
        "123642008",  # Right coronary artery occlusion (disorder)
        "194823009",  # Acute coronary insufficiency (disorder)
        "194828000",  # Angina (disorder)
        "194842008",  # Single coronary vessel disease (disorder)
        "194843003",  # Double coronary vessel disease (disorder)
        "233817007",  # Triple vessel disease of the heart (disorder)
        "233819005",  # Stable angina (disorder)
        "233821000",  # New onset angina (disorder)
        "233970002",  # Coronary artery stenosis (disorder)
        "300995000",  # Exercise-induced angina (disorder)
        "314116003",  # Post infarct angina (disorder)
        "315025001",  # Refractory angina (disorder)
        "315348000",  # Asymptomatic coronary heart disease (disorder)
        "371803003",  # Multi vessel coronary artery disease (disorder)
        "371804009",  # Left main coronary artery disease (disorder)
        "371805005",  # Significant coronary bypass graft disease (disorder)
        "371806006",  # Progressive angina (disorder)
        "371807002",  # Atypical angina (disorder)
        "371808007",  # Recurrent angina status post percutaneous transluminal coronary angioplasty (disorder)
        "371809004",  # Recurrent angina status post coronary stent placement (disorder)
        "371810009",  # Recurrent angina status post coronary artery bypass graft (disorder)
        "371811008",  # Recurrent angina status post rotational atherectomy (disorder)
        "371812001",  # Recurrent angina status post directional coronary atherectomy (disorder)
        "398274000",  # Coronary artery thrombosis (disorder)
        "408546009",  # Coronary artery bypass graft occlusion (disorder)
        "420006002",  # Obliterative coronary artery disease (disorder)
        "421327009",  # Coronary artery stent thrombosis (disorder)
        "427919004",  # Coronary arteriosclerosis caused by radiation (disorder)
        "429245005",  # Recurrent coronary arteriosclerosis after percutaneous transluminal coronary angioplasty (disorder)
        "429559004",  # Typical angina (disorder)
        "443502000",  # Atherosclerosis of coronary artery (disorder)
    }


class MyocardialInfarction(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of myocardial infarction.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of myocardial infarction.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS347v5, CMS145v10
    """

    VALUE_SET_NAME = "Myocardial Infarction"
    OID = "2.16.840.1.113883.3.526.3.403"
    DEFINITION_VERSION = "20210218"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "I2101",  # ST elevation (STEMI) myocardial infarction involving left main coronary artery
        "I2102",  # ST elevation (STEMI) myocardial infarction involving left anterior descending coronary artery
        "I2109",  # ST elevation (STEMI) myocardial infarction involving other coronary artery of anterior wall
        "I2111",  # ST elevation (STEMI) myocardial infarction involving right coronary artery
        "I2119",  # ST elevation (STEMI) myocardial infarction involving other coronary artery of inferior wall
        "I2121",  # ST elevation (STEMI) myocardial infarction involving left circumflex coronary artery
        "I2129",  # ST elevation (STEMI) myocardial infarction involving other sites
        "I213",  # ST elevation (STEMI) myocardial infarction of unspecified site
        "I214",  # Non-ST elevation (NSTEMI) myocardial infarction
        "I219",  # Acute myocardial infarction, unspecified
        "I21A1",  # Myocardial infarction type 2
        "I21A9",  # Other myocardial infarction type
        "I220",  # Subsequent ST elevation (STEMI) myocardial infarction of anterior wall
        "I221",  # Subsequent ST elevation (STEMI) myocardial infarction of inferior wall
        "I222",  # Subsequent non-ST elevation (NSTEMI) myocardial infarction
        "I228",  # Subsequent ST elevation (STEMI) myocardial infarction of other sites
        "I229",  # Subsequent ST elevation (STEMI) myocardial infarction of unspecified site
    }
    ICD9CM = {
        "41000",  # Acute myocardial infarction of anterolateral wall, episode of care unspecified
        "41001",  # Acute myocardial infarction of anterolateral wall, initial episode of care
        "41002",  # Acute myocardial infarction of anterolateral wall, subsequent episode of care
        "41010",  # Acute myocardial infarction of other anterior wall, episode of care unspecified
        "41011",  # Acute myocardial infarction of other anterior wall, initial episode of care
        "41012",  # Acute myocardial infarction of other anterior wall, subsequent episode of care
        "41020",  # Acute myocardial infarction of inferolateral wall, episode of care unspecified
        "41021",  # Acute myocardial infarction of inferolateral wall, initial episode of care
        "41022",  # Acute myocardial infarction of inferolateral wall, subsequent episode of care
        "41030",  # Acute myocardial infarction of inferoposterior wall, episode of care unspecified
        "41031",  # Acute myocardial infarction of inferoposterior wall, initial episode of care
        "41032",  # Acute myocardial infarction of inferoposterior wall, subsequent episode of care
        "41040",  # Acute myocardial infarction of other inferior wall, episode of care unspecified
        "41041",  # Acute myocardial infarction of other inferior wall, initial episode of care
        "41042",  # Acute myocardial infarction of other inferior wall, subsequent episode of care
        "41050",  # Acute myocardial infarction of other lateral wall, episode of care unspecified
        "41051",  # Acute myocardial infarction of other lateral wall, initial episode of care
        "41052",  # Acute myocardial infarction of other lateral wall, subsequent episode of care
        "41060",  # True posterior wall infarction, episode of care unspecified
        "41061",  # True posterior wall infarction, initial episode of care
        "41062",  # True posterior wall infarction, subsequent episode of care
        "41070",  # Subendocardial infarction, episode of care unspecified
        "41071",  # Subendocardial infarction, initial episode of care
        "41072",  # Subendocardial infarction, subsequent episode of care
        "41080",  # Acute myocardial infarction of other specified sites, episode of care unspecified
        "41081",  # Acute myocardial infarction of other specified sites, initial episode of care
        "41082",  # Acute myocardial infarction of other specified sites, subsequent episode of care
        "41090",  # Acute myocardial infarction of unspecified site, episode of care unspecified
        "41091",  # Acute myocardial infarction of unspecified site, initial episode of care
        "41092",  # Acute myocardial infarction of unspecified site, subsequent episode of care
    }
    SNOMEDCT = {
        "10273003",  # Acute infarction of papillary muscle (disorder)
        "15990001",  # Acute myocardial infarction of posterolateral wall (disorder)
        "22298006",  # Myocardial infarction (disorder)
        "30277009",  # Rupture of ventricle due to acute myocardial infarction (disorder)
        "32574007",  # Past myocardial infarction diagnosed on electrocardiogram AND/OR other special investigation, but currently presenting no symptoms (disorder)
        "42531007",  # Microinfarct of heart (disorder)
        "52035003",  # Acute anteroapical myocardial infarction (disorder)
        "54329005",  # Acute myocardial infarction of anterior wall (disorder)
        "57054005",  # Acute myocardial infarction (disorder)
        "58612006",  # Acute myocardial infarction of lateral wall (disorder)
        "62695002",  # Acute anteroseptal myocardial infarction (disorder)
        "65547006",  # Acute myocardial infarction of inferolateral wall (disorder)
        "70211005",  # Acute myocardial infarction of anterolateral wall (disorder)
        "70422006",  # Acute subendocardial infarction (disorder)
        "73795002",  # Acute myocardial infarction of inferior wall (disorder)
        "79009004",  # Acute myocardial infarction of septum (disorder)
        "129574000",  # Postoperative myocardial infarction (disorder)
        "194802003",  # True posterior myocardial infarction (disorder)
        "194809007",  # Acute atrial infarction (disorder)
        "194856005",  # Subsequent myocardial infarction (disorder)
        "233835003",  # Acute widespread myocardial infarction (disorder)
        "233838001",  # Acute posterior myocardial infarction (disorder)
        "233843008",  # Silent myocardial infarction (disorder)
        "304914007",  # Acute Q wave myocardial infarction (disorder)
        "307140009",  # Acute non-Q wave infarction (disorder)
        "314207007",  # Non-Q wave myocardial infarction (disorder)
        "371068009",  # Myocardial infarction with complication (disorder)
        "394710008",  # First myocardial infarction (disorder)
        "401303003",  # Acute ST segment elevation myocardial infarction (disorder)
        "401314000",  # Acute non-ST segment elevation myocardial infarction (disorder)
        "418044006",  # Myocardial infarction in recovery phase (disorder)
        "428196007",  # Mixed myocardial ischemia and infarction (disorder)
        "428752002",  # Recent myocardial infarction (situation)
    }


class AcutePharyngitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  bacterial or viral acute pharyngitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of either bacterial or viral acute pharyngitis.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS146v10
    """

    VALUE_SET_NAME = "Acute Pharyngitis"
    OID = "2.16.840.1.113883.3.464.1003.102.12.1011"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "J020",  # Streptococcal pharyngitis
        "J028",  # Acute pharyngitis due to other specified organisms
        "J029",  # Acute pharyngitis, unspecified
    }
    SNOMEDCT = {
        "1532007",  # Viral pharyngitis (disorder)
        "40766000",  # Enteroviral lymphonodular pharyngitis (disorder)
        "43878008",  # Streptococcal sore throat (disorder)
        "58031004",  # Suppurative pharyngitis (disorder)
        "195655000",  # Acute gangrenous pharyngitis (disorder)
        "195656004",  # Acute phlegmonous pharyngitis (disorder)
        "195657008",  # Acute ulcerative pharyngitis (disorder)
        "195658003",  # Acute bacterial pharyngitis (disorder)
        "195659006",  # Acute pneumococcal pharyngitis (disorder)
        "195660001",  # Acute staphylococcal pharyngitis (disorder)
        "195662009",  # Acute viral pharyngitis (disorder)
        "232399005",  # Acute herpes simplex pharyngitis (disorder)
        "232400003",  # Acute herpes zoster pharyngitis (disorder)
        "363746003",  # Acute pharyngitis (disorder)
    }


class AcuteTonsillitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  acute tonsillitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of either bacterial or viral acute tonsillitis.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS146v10
    """

    VALUE_SET_NAME = "Acute Tonsillitis"
    OID = "2.16.840.1.113883.3.464.1003.102.12.1012"
    DEFINITION_VERSION = "20170504"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "J0300",  # Acute streptococcal tonsillitis, unspecified
        "J0301",  # Acute recurrent streptococcal tonsillitis
        "J0380",  # Acute tonsillitis due to other specified organisms
        "J0381",  # Acute recurrent tonsillitis due to other specified organisms
        "J0390",  # Acute tonsillitis, unspecified
        "J0391",  # Acute recurrent tonsillitis, unspecified
    }
    SNOMEDCT = {
        "17741008",  # Acute tonsillitis (disorder)
        "195666007",  # Acute erythematous tonsillitis (disorder)
        "195668008",  # Acute ulcerative tonsillitis (disorder)
        "195669000",  # Acute catarrhal tonsillitis (disorder)
        "195670004",  # Acute gangrenous tonsillitis (disorder)
        "195671000",  # Acute bacterial tonsillitis (disorder)
        "195672007",  # Acute pneumococcal tonsillitis (disorder)
        "195673002",  # Acute staphylococcal tonsillitis (disorder)
        "195676005",  # Acute viral tonsillitis (disorder)
        "195677001",  # Recurrent acute tonsillitis (disorder)
        "302911003",  # Acute lingual tonsillitis (disorder)
        "10629231000119109",  # Recurrent acute streptococcal tonsillitis (disorder)
        "10629271000119107",  # Acute tonsillitis caused by Streptococcus (disorder)
    }


class ComorbidConditionsForRespiratoryConditions(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of immunocompromising and chronic respiratory conditions, which increase susceptibility of contracting an acute bacterial infection.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of immunocompromising and chronic respiratory conditions, which increase susceptibility of contracting an acute bacterial infection.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis that is not associated with increased susceptibility of contracting an acute bacterial infection.

    ** Used in:** CMS154v10, CMS146v10
    """

    VALUE_SET_NAME = "Comorbid Conditions for Respiratory Conditions"
    OID = "2.16.840.1.113883.3.464.1003.102.12.1025"
    DEFINITION_VERSION = "20200310"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "A150",  # Tuberculosis of lung
        "A154",  # Tuberculosis of intrathoracic lymph nodes
        "A155",  # Tuberculosis of larynx, trachea and bronchus
        "A156",  # Tuberculous pleurisy
        "A157",  # Primary respiratory tuberculosis
        "A158",  # Other respiratory tuberculosis
        "A159",  # Respiratory tuberculosis unspecified
        "A170",  # Tuberculous meningitis
        "A171",  # Meningeal tuberculoma
        "A1781",  # Tuberculoma of brain and spinal cord
        "A1782",  # Tuberculous meningoencephalitis
        "A1783",  # Tuberculous neuritis
        "A1789",  # Other tuberculosis of nervous system
        "A179",  # Tuberculosis of nervous system, unspecified
        "A1801",  # Tuberculosis of spine
        "A1802",  # Tuberculous arthritis of other joints
        "A1803",  # Tuberculosis of other bones
        "A1809",  # Other musculoskeletal tuberculosis
        "A1810",  # Tuberculosis of genitourinary system, unspecified
        "A1811",  # Tuberculosis of kidney and ureter
        "A1812",  # Tuberculosis of bladder
        "A1813",  # Tuberculosis of other urinary organs
        "A1814",  # Tuberculosis of prostate
        "A1815",  # Tuberculosis of other male genital organs
        "A1816",  # Tuberculosis of cervix
        "A1817",  # Tuberculous female pelvic inflammatory disease
        "A1818",  # Tuberculosis of other female genital organs
        "A182",  # Tuberculous peripheral lymphadenopathy
        "A1831",  # Tuberculous peritonitis
        "A1832",  # Tuberculous enteritis
        "A1839",  # Retroperitoneal tuberculosis
        "A184",  # Tuberculosis of skin and subcutaneous tissue
        "A1850",  # Tuberculosis of eye, unspecified
        "A1851",  # Tuberculous episcleritis
        "A1852",  # Tuberculous keratitis
        "A1853",  # Tuberculous chorioretinitis
        "A1854",  # Tuberculous iridocyclitis
        "A1859",  # Other tuberculosis of eye
        "A186",  # Tuberculosis of (inner) (middle) ear
        "A187",  # Tuberculosis of adrenal glands
        "A1881",  # Tuberculosis of thyroid gland
        "A1882",  # Tuberculosis of other endocrine glands
        "A1883",  # Tuberculosis of digestive tract organs, not elsewhere classified
        "A1884",  # Tuberculosis of heart
        "A1885",  # Tuberculosis of spleen
        "A1889",  # Tuberculosis of other sites
        "A190",  # Acute miliary tuberculosis of a single specified site
        "A191",  # Acute miliary tuberculosis of multiple sites
        "A192",  # Acute miliary tuberculosis, unspecified
        "A198",  # Other miliary tuberculosis
        "A199",  # Miliary tuberculosis, unspecified
        "B4481",  # Allergic bronchopulmonary aspergillosis
        "D5701",  # Hb-SS disease with acute chest syndrome
        "D5703",  # Hb-SS disease with cerebral vascular involvement
        "D5709",  # Hb-SS disease with crisis with other specified complication
        "D57211",  # Sickle-cell/Hb-C disease with acute chest syndrome
        "D57213",  # Sickle-cell/Hb-C disease with cerebral vascular involvement
        "D57218",  # Sickle-cell/Hb-C disease with crisis with other specified complication
        "D57411",  # Sickle-cell thalassemia, unspecified, with acute chest syndrome
        "D57413",  # Sickle-cell thalassemia, unspecified, with cerebral vascular involvement
        "D57418",  # Sickle-cell thalassemia, unspecified, with crisis with other specified complication
        "D57811",  # Other sickle-cell disorders with acute chest syndrome
        "D57813",  # Other sickle-cell disorders with cerebral vascular involvement
        "D57818",  # Other sickle-cell disorders with crisis with other specified complication
        "D61810",  # Antineoplastic chemotherapy induced pancytopenia
        "D61811",  # Other drug-induced pancytopenia
        "D61818",  # Other pancytopenia
        "D700",  # Congenital agranulocytosis
        "D701",  # Agranulocytosis secondary to cancer chemotherapy
        "D702",  # Other drug-induced agranulocytosis
        "D703",  # Neutropenia due to infection
        "D704",  # Cyclic neutropenia
        "D708",  # Other neutropenia
        "D709",  # Neutropenia, unspecified
        "D71",  # Functional disorders of polymorphonuclear neutrophils
        "D720",  # Genetic anomalies of leukocytes
        "D7581",  # Myelofibrosis
        "D761",  # Hemophagocytic lymphohistiocytosis
        "D762",  # Hemophagocytic syndrome, infection-associated
        "D763",  # Other histiocytosis syndromes
        "D860",  # Sarcoidosis of lung
        "D861",  # Sarcoidosis of lymph nodes
        "D862",  # Sarcoidosis of lung with sarcoidosis of lymph nodes
        "E840",  # Cystic fibrosis with pulmonary manifestations
        "E8411",  # Meconium ileus in cystic fibrosis
        "E8419",  # Cystic fibrosis with other intestinal manifestations
        "E848",  # Cystic fibrosis with other manifestations
        "E849",  # Cystic fibrosis, unspecified
        "J22",  # Unspecified acute lower respiratory infection
        "J410",  # Simple chronic bronchitis
        "J411",  # Mucopurulent chronic bronchitis
        "J418",  # Mixed simple and mucopurulent chronic bronchitis
        "J42",  # Unspecified chronic bronchitis
        "J470",  # Bronchiectasis with acute lower respiratory infection
        "J471",  # Bronchiectasis with (acute) exacerbation
        "J479",  # Bronchiectasis, uncomplicated
        "J60",  # Coalworker's pneumoconiosis
        "J61",  # Pneumoconiosis due to asbestos and other mineral fibers
        "J620",  # Pneumoconiosis due to talc dust
        "J628",  # Pneumoconiosis due to other dust containing silica
        "J630",  # Aluminosis (of lung)
        "J631",  # Bauxite fibrosis (of lung)
        "J632",  # Berylliosis
        "J633",  # Graphite fibrosis (of lung)
        "J634",  # Siderosis
        "J635",  # Stannosis
        "J636",  # Pneumoconiosis due to other specified inorganic dusts
        "J64",  # Unspecified pneumoconiosis
        "J65",  # Pneumoconiosis associated with tuberculosis
        "J660",  # Byssinosis
        "J661",  # Flax-dressers' disease
        "J662",  # Cannabinosis
        "J668",  # Airway disease due to other specific organic dusts
        "J670",  # Farmer's lung
        "J671",  # Bagassosis
        "J672",  # Bird fancier's lung
        "J673",  # Suberosis
        "J674",  # Maltworker's lung
        "J675",  # Mushroom-worker's lung
        "J676",  # Maple-bark-stripper's lung
        "J677",  # Air conditioner and humidifier lung
        "J678",  # Hypersensitivity pneumonitis due to other organic dusts
        "J679",  # Hypersensitivity pneumonitis due to unspecified organic dust
        "J680",  # Bronchitis and pneumonitis due to chemicals, gases, fumes and vapors
        "J681",  # Pulmonary edema due to chemicals, gases, fumes and vapors
        "J682",  # Upper respiratory inflammation due to chemicals, gases, fumes and vapors, not elsewhere classified
        "J683",  # Other acute and subacute respiratory conditions due to chemicals, gases, fumes and vapors
        "J689",  # Unspecified respiratory condition due to chemicals, gases, fumes and vapors
        "J690",  # Pneumonitis due to inhalation of food and vomit
        "J691",  # Pneumonitis due to inhalation of oils and essences
        "J698",  # Pneumonitis due to inhalation of other solids and liquids
        "J700",  # Acute pulmonary manifestations due to radiation
        "J701",  # Chronic and other pulmonary manifestations due to radiation
        "J702",  # Acute drug-induced interstitial lung disorders
        "J703",  # Chronic drug-induced interstitial lung disorders
        "J704",  # Drug-induced interstitial lung disorders, unspecified
        "J705",  # Respiratory conditions due to smoke inhalation
        "J708",  # Respiratory conditions due to other specified external agents
        "J709",  # Respiratory conditions due to unspecified external agent
        "J80",  # Acute respiratory distress syndrome
        "J810",  # Acute pulmonary edema
        "J811",  # Chronic pulmonary edema
        "J82",  # Pulmonary eosinophilia, not elsewhere classified
        "J8281",  # Chronic eosinophilic pneumonia
        "J8282",  # Acute eosinophilic pneumonia
        "J8283",  # Eosinophilic asthma
        "J8289",  # Other pulmonary eosinophilia, not elsewhere classified
        "J8401",  # Alveolar proteinosis
        "J8402",  # Pulmonary alveolar microlithiasis
        "J8403",  # Idiopathic pulmonary hemosiderosis
        "J8409",  # Other alveolar and parieto-alveolar conditions
        "J8410",  # Pulmonary fibrosis, unspecified
        "J84111",  # Idiopathic interstitial pneumonia, not otherwise specified
        "J84112",  # Idiopathic pulmonary fibrosis
        "J84113",  # Idiopathic non-specific interstitial pneumonitis
        "J84114",  # Acute interstitial pneumonitis
        "J84115",  # Respiratory bronchiolitis interstitial lung disease
        "J84116",  # Cryptogenic organizing pneumonia
        "J84117",  # Desquamative interstitial pneumonia
        "J8417",  # Other interstitial pulmonary diseases with fibrosis in diseases classified elsewhere
        "J84170",  # Interstitial lung disease with progressive fibrotic phenotype in diseases classified elsewhere
        "J84178",  # Other interstitial pulmonary diseases with fibrosis in diseases classified elsewhere
        "J842",  # Lymphoid interstitial pneumonia
        "J8481",  # Lymphangioleiomyomatosis
        "J8482",  # Adult pulmonary Langerhans cell histiocytosis
        "J8483",  # Surfactant mutations of the lung
        "J84841",  # Neuroendocrine cell hyperplasia of infancy
        "J84842",  # Pulmonary interstitial glycogenosis
        "J84843",  # Alveolar capillary dysplasia with vein misalignment
        "J84848",  # Other interstitial lung diseases of childhood
        "J8489",  # Other specified interstitial pulmonary diseases
        "J849",  # Interstitial pulmonary disease, unspecified
        "J850",  # Gangrene and necrosis of lung
        "J851",  # Abscess of lung with pneumonia
        "J852",  # Abscess of lung without pneumonia
        "J853",  # Abscess of mediastinum
        "J860",  # Pyothorax with fistula
        "J869",  # Pyothorax without fistula
        "J90",  # Pleural effusion, not elsewhere classified
        "J910",  # Malignant pleural effusion
        "J918",  # Pleural effusion in other conditions classified elsewhere
        "J920",  # Pleural plaque with presence of asbestos
        "J929",  # Pleural plaque without asbestos
        "J930",  # Spontaneous tension pneumothorax
        "J9311",  # Primary spontaneous pneumothorax
        "J9312",  # Secondary spontaneous pneumothorax
        "J9381",  # Chronic pneumothorax
        "J9382",  # Other air leak
        "J9383",  # Other pneumothorax
        "J939",  # Pneumothorax, unspecified
        "J940",  # Chylous effusion
        "J941",  # Fibrothorax
        "J942",  # Hemothorax
        "J948",  # Other specified pleural conditions
        "J949",  # Pleural condition, unspecified
        "J9500",  # Unspecified tracheostomy complication
        "J9501",  # Hemorrhage from tracheostomy stoma
        "J9502",  # Infection of tracheostomy stoma
        "J9503",  # Malfunction of tracheostomy stoma
        "J9504",  # Tracheo-esophageal fistula following tracheostomy
        "J9509",  # Other tracheostomy complication
        "J951",  # Acute pulmonary insufficiency following thoracic surgery
        "J952",  # Acute pulmonary insufficiency following nonthoracic surgery
        "J953",  # Chronic pulmonary insufficiency following surgery
        "J954",  # Chemical pneumonitis due to anesthesia
        "J955",  # Postprocedural subglottic stenosis
        "J9561",  # Intraoperative hemorrhage and hematoma of a respiratory system organ or structure complicating a respiratory system procedure
        "J9562",  # Intraoperative hemorrhage and hematoma of a respiratory system organ or structure complicating other procedure
        "J9571",  # Accidental puncture and laceration of a respiratory system organ or structure during a respiratory system procedure
        "J9572",  # Accidental puncture and laceration of a respiratory system organ or structure during other procedure
        "J95811",  # Postprocedural pneumothorax
        "J95812",  # Postprocedural air leak
        "J95821",  # Acute postprocedural respiratory failure
        "J95822",  # Acute and chronic postprocedural respiratory failure
        "J95830",  # Postprocedural hemorrhage of a respiratory system organ or structure following a respiratory system procedure
        "J95831",  # Postprocedural hemorrhage of a respiratory system organ or structure following other procedure
        "J9584",  # Transfusion-related acute lung injury (TRALI)
        "J95850",  # Mechanical complication of respirator
        "J95851",  # Ventilator associated pneumonia
        "J95859",  # Other complication of respirator [ventilator]
        "J95860",  # Postprocedural hematoma of a respiratory system organ or structure following a respiratory system procedure
        "J95861",  # Postprocedural hematoma of a respiratory system organ or structure following other procedure
        "J95862",  # Postprocedural seroma of a respiratory system organ or structure following a respiratory system procedure
        "J95863",  # Postprocedural seroma of a respiratory system organ or structure following other procedure
        "J9588",  # Other intraoperative complications of respiratory system, not elsewhere classified
        "J9589",  # Other postprocedural complications and disorders of respiratory system, not elsewhere classified
        "J9600",  # Acute respiratory failure, unspecified whether with hypoxia or hypercapnia
        "J9601",  # Acute respiratory failure with hypoxia
        "J9602",  # Acute respiratory failure with hypercapnia
        "J9610",  # Chronic respiratory failure, unspecified whether with hypoxia or hypercapnia
        "J9611",  # Chronic respiratory failure with hypoxia
        "J9612",  # Chronic respiratory failure with hypercapnia
        "J9620",  # Acute and chronic respiratory failure, unspecified whether with hypoxia or hypercapnia
        "J9621",  # Acute and chronic respiratory failure with hypoxia
        "J9622",  # Acute and chronic respiratory failure with hypercapnia
        "J9690",  # Respiratory failure, unspecified, unspecified whether with hypoxia or hypercapnia
        "J9691",  # Respiratory failure, unspecified with hypoxia
        "J9692",  # Respiratory failure, unspecified with hypercapnia
        "J9851",  # Mediastinitis
        "J986",  # Disorders of diaphragm
        "J99",  # Respiratory disorders in diseases classified elsewhere
        "M0510",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified site
        "M05111",  # Rheumatoid lung disease with rheumatoid arthritis of right shoulder
        "M05112",  # Rheumatoid lung disease with rheumatoid arthritis of left shoulder
        "M05119",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified shoulder
        "M05121",  # Rheumatoid lung disease with rheumatoid arthritis of right elbow
        "M05122",  # Rheumatoid lung disease with rheumatoid arthritis of left elbow
        "M05129",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified elbow
        "M05131",  # Rheumatoid lung disease with rheumatoid arthritis of right wrist
        "M05132",  # Rheumatoid lung disease with rheumatoid arthritis of left wrist
        "M05139",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified wrist
        "M05141",  # Rheumatoid lung disease with rheumatoid arthritis of right hand
        "M05142",  # Rheumatoid lung disease with rheumatoid arthritis of left hand
        "M05149",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified hand
        "M05151",  # Rheumatoid lung disease with rheumatoid arthritis of right hip
        "M05152",  # Rheumatoid lung disease with rheumatoid arthritis of left hip
        "M05159",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified hip
        "M05161",  # Rheumatoid lung disease with rheumatoid arthritis of right knee
        "M05162",  # Rheumatoid lung disease with rheumatoid arthritis of left knee
        "M05169",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified knee
        "M05171",  # Rheumatoid lung disease with rheumatoid arthritis of right ankle and foot
        "M05172",  # Rheumatoid lung disease with rheumatoid arthritis of left ankle and foot
        "M05179",  # Rheumatoid lung disease with rheumatoid arthritis of unspecified ankle and foot
        "M0519",  # Rheumatoid lung disease with rheumatoid arthritis of multiple sites
        "M301",  # Polyarteritis with lung involvement [Churg-Strauss]
        "M3213",  # Lung involvement in systemic lupus erythematosus
        "M3301",  # Juvenile dermatomyositis with respiratory involvement
        "M3311",  # Other dermatomyositis with respiratory involvement
        "M3321",  # Polymyositis with respiratory involvement
        "M3391",  # Dermatopolymyositis, unspecified with respiratory involvement
        "M3481",  # Systemic sclerosis with lung involvement
        "M3502",  # Sicca syndrome with lung involvement
        "O98011",  # Tuberculosis complicating pregnancy, first trimester
        "O98012",  # Tuberculosis complicating pregnancy, second trimester
        "O98013",  # Tuberculosis complicating pregnancy, third trimester
        "O98019",  # Tuberculosis complicating pregnancy, unspecified trimester
        "O9802",  # Tuberculosis complicating childbirth
        "O9803",  # Tuberculosis complicating the puerperium
        "P270",  # Wilson-Mikity syndrome
        "P271",  # Bronchopulmonary dysplasia originating in the perinatal period
        "P278",  # Other chronic respiratory diseases originating in the perinatal period
        "P279",  # Unspecified chronic respiratory disease originating in the perinatal period
        "Q2545",  # Double aortic arch
        "Q2547",  # Right aortic arch
        "Q2548",  # Anomalous origin of subclavian artery
        "Q300",  # Choanal atresia
        "Q301",  # Agenesis and underdevelopment of nose
        "Q302",  # Fissured, notched and cleft nose
        "Q303",  # Congenital perforated nasal septum
        "Q308",  # Other congenital malformations of nose
        "Q309",  # Congenital malformation of nose, unspecified
        "Q310",  # Web of larynx
        "Q311",  # Congenital subglottic stenosis
        "Q312",  # Laryngeal hypoplasia
        "Q313",  # Laryngocele
        "Q315",  # Congenital laryngomalacia
        "Q318",  # Other congenital malformations of larynx
        "Q319",  # Congenital malformation of larynx, unspecified
        "Q320",  # Congenital tracheomalacia
        "Q321",  # Other congenital malformations of trachea
        "Q322",  # Congenital bronchomalacia
        "Q323",  # Congenital stenosis of bronchus
        "Q324",  # Other congenital malformations of bronchus
        "Q330",  # Congenital cystic lung
        "Q331",  # Accessory lobe of lung
        "Q332",  # Sequestration of lung
        "Q333",  # Agenesis of lung
        "Q334",  # Congenital bronchiectasis
        "Q335",  # Ectopic tissue in lung
        "Q336",  # Congenital hypoplasia and dysplasia of lung
        "Q338",  # Other congenital malformations of lung
        "Q339",  # Congenital malformation of lung, unspecified
        "Q340",  # Anomaly of pleura
        "Q341",  # Congenital cyst of mediastinum
        "Q348",  # Other specified congenital malformations of respiratory system
        "Q349",  # Congenital malformation of respiratory system, unspecified
        "Q390",  # Atresia of esophagus without fistula
        "Q391",  # Atresia of esophagus with tracheo-esophageal fistula
        "Q392",  # Congenital tracheo-esophageal fistula without atresia
        "Q393",  # Congenital stenosis and stricture of esophagus
        "Q394",  # Esophageal web
        "Q8901",  # Asplenia (congenital)
        "Q893",  # Situs inversus
    }
    SNOMEDCT = {
        "599006",  # Chronic pneumothorax (disorder)
        "1259003",  # Schistosis (disorder)
        "4225003",  # Tuberculosis of nose (disorder)
        "4445009",  # Tuberculosis of genitourinary system (disorder)
        "4501007",  # Tuberculous ascites (disorder)
        "7343008",  # Liparitosis (disorder)
        "7444001",  # Tuberculosis of rectum (disorder)
        "8250007",  # Tuberculosis of subcutaneous cellular tissue (disorder)
        "8832006",  # Tuberculosis orificialis of anus (disorder)
        "10041001",  # Tracheal stenosis following tracheostomy (disorder)
        "10528009",  # Lupus vulgaris (disorder)
        "10706006",  # Tuberculosis of central nervous system (disorder)
        "10893003",  # Tuberculous adenitis (disorder)
        "11676005",  # Tuberculous leptomeningitis (disorder)
        "11689004",  # Linear atelectasis (disorder)
        "12137005",  # Tuberculous enteritis (disorder)
        "12677003",  # Tuberculosis of brain (disorder)
        "12985005",  # Tuberculosis of seminal vesicle (disorder)
        "13570003",  # Tuberculous tenosynovitis (disorder)
        "14188007",  # Tuberculosis of hip (disorder)
        "14527007",  # Tuberculous empyema (disorder)
        "15202009",  # Tuberculoma (disorder)
        "15284007",  # Tuberculosis of esophagus (disorder)
        "16409009",  # Tuberculosis of retroperitoneal lymph nodes (disorder)
        "16537000",  # Compression atelectasis (disorder)
        "16632002",  # Hemopneumothorax (disorder)
        "17136009",  # Tuberculosis of knee (disorder)
        "17653001",  # Tuberculosis of bones and/or joints (disorder)
        "20372007",  # Tuberculosis of prostate (disorder)
        "21504004",  # Pressure collapse of lung after anesthesia AND/OR sedation in labor AND/OR delivery (disorder)
        "22607003",  # Asbestosis (disorder)
        "22990009",  # Chronic tuberculosis (disorder)
        "23022004",  # Tuberculous bronchiectasis (disorder)
        "24235005",  # Tuberculous hydrothorax (disorder)
        "24810006",  # Tuberculosis orificialis of mouth (disorder)
        "25042006",  # Tracheostomy sepsis (disorder)
        "25629007",  # Acute tuberculosis (disorder)
        "25738004",  # Tuberculous synovitis (disorder)
        "25831001",  # Tuberculosis of nasal septum (disorder)
        "26935004",  # Tuberculosis of thyroid gland (disorder)
        "28085001",  # Empyema with bronchopleural fistula (disorder)
        "28295001",  # Middle lobe syndrome (disorder)
        "28399005",  # Tuberculosis of spleen (disorder)
        "29422001",  # Coal workers' pneumoconiosis (disorder)
        "29731002",  # Tuberculous pneumothorax (disorder)
        "30437004",  # Empyema with bronchocutaneous fistula (disorder)
        "31112008",  # Tuberculous meningoencephalitis (disorder)
        "31892009",  # Hemothorax (disorder)
        "32139003",  # Mixed dust pneumoconiosis (disorder)
        "32268008",  # Tuberculosis of bladder (disorder)
        "33072005",  # Traumatic hemothorax with open wound into thorax (disorder)
        "33325001",  # Compensatory emphysema (disorder)
        "33548005",  # Anthracosilicosis (disorder)
        "33631007",  # Pyopneumothorax (disorder)
        "33893009",  # Tuberculosis verrucosa cutis (disorder)
        "34004002",  # Siderosilicosis (disorder)
        "34286007",  # Empyema with hepatopleural fistula (disorder)
        "34320001",  # Tuberculous myelitis (disorder)
        "35585004",  # Tuberculous endometritis (disorder)
        "35786001",  # Tuberculoma of meninges (disorder)
        "35984006",  # Tuberculosis of vertebral column (disorder)
        "36696005",  # Kaolinosis (disorder)
        "37133005",  # Tuberculous pyelitis (disorder)
        "37711000",  # Cadmium pneumonitis (disorder)
        "37981002",  # Allergic bronchopulmonary aspergillosis (disorder)
        "38115001",  # Tuberculosis of spinal meninges (disorder)
        "38279006",  # Tuberculosis of bone (disorder)
        "38420005",  # Tuberculosis of large intestine (disorder)
        "39306006",  # Tuberculosis of anus (disorder)
        "40218008",  # Carbon electrode makers' pneumoconiosis (disorder)
        "40258005",  # Tuberculosis of testis (disorder)
        "40640008",  # Massive fibrosis of lung co-occurrent and due to silicosis (disorder)
        "40779009",  # Discoid atelectasis (disorder)
        "42434002",  # Pneumohemothorax due to trauma (disorder)
        "42458003",  # Hemothorax due to trauma (disorder)
        "42522007",  # Abscess of mediastinum (disorder)
        "42908004",  # Primary atelectasis, in perinatal period (disorder)
        "44182001",  # Tuberculous necrosis of bone (disorder)
        "44323002",  # Tuberculosis of kidney (disorder)
        "44547005",  # Chalicosis (disorder)
        "44572005",  # Tuberculous peritonitis (disorder)
        "44599007",  # Traumatic hemothorax without open wound into thorax (disorder)
        "46621007",  # Atelectasis (disorder)
        "47292005",  # Tuberculosis of myocardium (disorder)
        "47515009",  # Simple silicosis (disorder)
        "47604008",  # Miliary tuberculosis (disorder)
        "48225000",  # Tuberculous disseminated chorioretinitis (disorder)
        "48475001",  # Disorder of diaphragm (disorder)
        "49107007",  # Tuberculosis of eye (disorder)
        "49558004",  # Tuberculous salpingitis (disorder)
        "49807009",  # Tuberculous dactylitis (disorder)
        "49840000",  # Complicated silicosis (disorder)
        "50076003",  # Baritosis (disorder)
        "50589003",  # Silver polishers' lung disease (disorder)
        "50694005",  # Empyema with mediastinal fistula (disorder)
        "51014003",  # Reinfection tuberculosis (disorder)
        "51068008",  # Adult bronchiectasis (disorder)
        "51277007",  # Stannosis (disorder)
        "52721006",  # Tuberculosis of nasal sinus (disorder)
        "52987001",  # Tuberculosis of endocardium (disorder)
        "54084005",  # Cervical tuberculous lymphadenitis (disorder)
        "54959009",  # Perinatal atelectasis (disorder)
        "55419007",  # Tuberculosis of glottis (disorder)
        "55448001",  # Tracheoesophageal fistula following tracheostomy (disorder)
        "56498009",  # Tuberculosis of nasopharynx (disorder)
        "56717001",  # Tuberculosis (disorder)
        "58374007",  # Acute diffuse tuberculosis (disorder)
        "58437007",  # Tuberculosis of meninges (disorder)
        "58525007",  # Obstructive atelectasis (disorder)
        "58554001",  # Empyema of pleura (disorder)
        "58691003",  # Antimony pneumoconiosis (disorder)
        "58764007",  # Tuberculous osteomyelitis (disorder)
        "59113005",  # Perinatal secondary atelectasis (disorder)
        "59190009",  # Traumatic pneumohemothorax without open wound into thorax (disorder)
        "60136008",  # Tuberculosis of intestines (disorder)
        "61448007",  # Focal atelectasis (disorder)
        "62371005",  # Pulmonary siderosis (disorder)
        "62494007",  # Chronic miliary tuberculosis (disorder)
        "62819009",  # Tuberculous cervicitis (disorder)
        "63309002",  # Primary tuberculosis (disorder)
        "63808004",  # Tuberculous chancre (disorder)
        "64631008",  # Fullers' earth disease (disorder)
        "66151009",  # Perinatal complete pulmonary collapse (disorder)
        "66291003",  # Tuberculous phlyctenular keratoconjunctivitis (disorder)
        "66986005",  # Tuberculosis of skin (disorder)
        "67256000",  # Tuberculosis of pericardium (disorder)
        "68033004",  # Tracheostomy complication (disorder)
        "70341005",  # Tuberculous laryngitis (disorder)
        "71375000",  # Tuberculosis of mastoid process (disorder)
        "71943007",  # Chronic granulomatous tuberculosis (disorder)
        "72270005",  # Collagenous pneumoconiosis (disorder)
        "72807002",  # Tuberculous polyserositis (disorder)
        "73452002",  # Abscess of lung (disorder)
        "73477001",  # Traumatic pneumohemothorax with open wound into thorax (disorder)
        "74181004",  # Tuberculosis of female genital organs (disorder)
        "74387008",  # Tuberculosis of hilar lymph nodes (disorder)
        "74610006",  # Tuberculous erythema nodosum (disorder)
        "76157009",  # Bituminosis (disorder)
        "76630008",  # Fibrinopurulent pleurisy (disorder)
        "77038006",  # Tuberculosis of peripheral lymph nodes (disorder)
        "77690003",  # Interstitial emphysema of lung (disorder)
        "78436002",  # Tuberculosis of intrathoracic lymph nodes (disorder)
        "78887005",  # Seropurulent pleurisy (disorder)
        "79158003",  # Tuberculous interstitial keratitis (disorder)
        "80003002",  # Tuberculous pneumonia (disorder)
        "80010008",  # Isolated bronchial tuberculosis (disorder)
        "80602006",  # Nodular tuberculosis of lung (disorder)
        "81133003",  # Tuberculosis of bones of lower extremity (disorder)
        "81359005",  # Tuberculosis of ureter (disorder)
        "81554001",  # Tuberculosis of lung with involvement of bronchus (disorder)
        "82495004",  # Tuberculosis of mediastinal lymph nodes (disorder)
        "82796002",  # Scrofuloderma (disorder)
        "83652003",  # Tuberculosis of epididymis (disorder)
        "83784000",  # Tuberculosis of mediastinum (disorder)
        "84194006",  # Tuberculous oophoritis (disorder)
        "84288007",  # Tuberculous episcleritis (disorder)
        "85692003",  # Tuberculosis of tracheobronchial lymph nodes (disorder)
        "85940005",  # Tuberculous otitis media (disorder)
        "86263001",  # Cobaltosis (disorder)
        "87909002",  # Hard metal pneumoconiosis (disorder)
        "88356006",  # Primary tuberculous complex (disorder)
        "88687001",  # Manganese pneumonitis (disorder)
        "90117007",  # Tuberculous fibrosis of lung (disorder)
        "90302003",  # Tuberculosis of cerebral meninges (disorder)
        "90623003",  # Aluminosis of lung (disorder)
        "91036007",  # Complete atelectasis (disorder)
        "91483004",  # Tuberculous encephalitis (disorder)
        "111466004",  # Perinatal partial atelectasis (disorder)
        "111825008",  # Tuberculosis of conjunctiva (disorder)
        "111830007",  # Tuberculosis of urinary organs (disorder)
        "111832004",  # Primary progressive tuberculosis (disorder)
        "123593009",  # Complete atelectasis of left lung (disorder)
        "123594003",  # Complete atelectasis of right lung (disorder)
        "123712000",  # Acute atelectasis (disorder)
        "123713005",  # Chronic atelectasis (disorder)
        "154283005",  # Pulmonary tuberculosis (disorder)
        "182159002",  # Disseminated tuberculosis (disorder)
        "185113005",  # Tuberculosis of adrenal glands (disorder)
        "186172004",  # Tuberculous pleurisy in primary progressive tuberculosis (disorder)
        "186175002",  # Infiltrative lung tuberculosis (disorder)
        "186177005",  # Tuberculosis of lung with cavitation (disorder)
        "186178000",  # Tuberculosis of bronchus (disorder)
        "186182003",  # Tuberculosis of pleura (disorder)
        "186192006",  # Respiratory tuberculosis, bacteriologically and histologically confirmed (disorder)
        "186193001",  # Tuberculosis of lung, confirmed by sputum microscopy with or without culture (disorder)
        "186194007",  # Tuberculosis of lung, confirmed by culture only (disorder)
        "186195008",  # Tuberculosis of lung, confirmed histologically (disorder)
        "186199002",  # Tuberculosis of larynx, trachea and bronchus, confirmed bacteriologically and histologically (disorder)
        "186200004",  # Tuberculous pleurisy, confirmed bacteriologically and histologically (disorder)
        "186201000",  # Primary respiratory tuberculosis, confirmed bacteriologically and histologically (disorder)
        "186202007",  # Respiratory tuberculosis, not confirmed bacteriologically or histologically (disorder)
        "186203002",  # Tuberculosis of lung, bacteriologically and histologically negative (disorder)
        "186204008",  # Tuberculosis of lung, bacteriological and histological examination not done (disorder)
        "186217006",  # Tuberculous abscess of brain (disorder)
        "186225008",  # Tuberculosis of intestines, peritoneum and mesenteric glands (disorder)
        "186231006",  # Tuberculosis limb bones - Tuberculous dactylitis (disorder)
        "186259007",  # Scrofulous tuberculous abscess (disorder)
        "186263000",  # Tuberculous chorioretinitis (disorder)
        "186266008",  # Tuberculous chronic iridocyclitis (disorder)
        "186267004",  # Tuberculous keratoconjunctivitis (disorder)
        "186269001",  # Tuberculosis of ear (disorder)
        "186270000",  # Tuberculous Addison's disease (disorder)
        "186272008",  # Tuberculosis of stomach (disorder)
        "186273003",  # Tuberculosis of liver (disorder)
        "186276006",  # Acute miliary tuberculosis (disorder)
        "186278007",  # Acute miliary tuberculosis of multiple sites (disorder)
        "190710003",  # Tuberculous arthritis (disorder)
        "192745001",  # Tuberculous intracranial abscess (disorder)
        "194908003",  # Acute tuberculous pericarditis (disorder)
        "194949003",  # Acute myocarditis - tuberculous (disorder)
        "195953005",  # Mixed simple and mucopurulent chronic bronchitis (disorder)
        "195985008",  # Post-infective bronchiectasis (disorder)
        "196017002",  # Pneumoconiosis associated with tuberculosis (disorder)
        "196067009",  # Loculated empyema (disorder)
        "196102003",  # Spontaneous tension pneumothorax (disorder)
        "196107009",  # Abscess of lung and mediastinum (disorder)
        "196112005",  # Abscess of lung with pneumonia (disorder)
        "196133001",  # Lung disease with systemic sclerosis (disorder)
        "196136009",  # Lung disease co-occurrent with polymyositis (disorder)
        "196138005",  # Lung disease with systemic lupus erythematosus (disorder)
        "196143003",  # Postoperative atelectasis (disorder)
        "196188002",  # Acute pulmonary insufficiency following thoracic surgery (disorder)
        "196189005",  # Acute pulmonary insufficiency following nonthoracic surgery (disorder)
        "196190001",  # Chronic pulmonary insufficiency following surgery (disorder)
        "196191002",  # Postprocedural subglottic stenosis (disorder)
        "198241002",  # Female tuberculous pelvic inflammatory disease (disorder)
        "199179007",  # Maternal tuberculosis in the puerperium - baby delivered during current episode of care (disorder)
        "199180005",  # Maternal tuberculosis during pregnancy - baby not yet delivered (disorder)
        "199181009",  # Maternal tuberculosis in the puerperium - baby delivered during previous episode of care (disorder)
        "202994006",  # Tuberculous infection of tendon sheath (disorder)
        "203263006",  # Tuberculosis of cervical spine (disorder)
        "203264000",  # Tuberculosis of thoracic spine (disorder)
        "203265004",  # Tuberculosis of lumbar spine (disorder)
        "203267007",  # Tuberculosis of limb bones (disorder)
        "203269005",  # Tuberculosis of the upper arm bone (disorder)
        "203270006",  # Tuberculosis of the forearm bone (disorder)
        "203271005",  # Tuberculosis of the pelvic and/or thigh bones (disorder)
        "203272003",  # Tuberculosis of the lower leg bone (disorder)
        "203274002",  # Tuberculosis of multiple limb bones (disorder)
        "203278004",  # Tuberculosis of the bones of the shoulder region (disorder)
        "203279007",  # Tuberculosis of the bones of the hand (disorder)
        "203280005",  # Tuberculosis of the bones of the ankle and/or foot (disorder)
        "203282002",  # Tuberculosis of the bones of multiple sites (disorder)
        "213580002",  # Caseating tuberculoid granuloma (disorder)
        "233628009",  # Acquired bronchiectasis (disorder)
        "233629001",  # Idiopathic bronchiectasis (disorder)
        "233631005",  # Toxin-induced bronchiectasis (disorder)
        "233634002",  # Post-lung transplantation bronchiectasis (disorder)
        "233649005",  # Tuberculous chylothorax (disorder)
        "233658003",  # Pleural plaque (disorder)
        "233659006",  # Asbestos-induced pleural plaque (disorder)
        "233749003",  # Complicated pneumoconiosis (disorder)
        "233751004",  # Metal pneumoconiosis (disorder)
        "233754007",  # Cerium pneumoconiosis (disorder)
        "233755008",  # Nickel pneumoconiosis (disorder)
        "233756009",  # Thorium pneumoconiosis (disorder)
        "233757000",  # Zirconium pneumoconiosis (disorder)
        "233758005",  # Mica pneumoconiosis (disorder)
        "233759002",  # Mixed mineral dust pneumoconiosis (disorder)
        "233760007",  # Acute silicosis (disorder)
        "233761006",  # Subacute silicosis (disorder)
        "233762004",  # Chronic silicosis (disorder)
        "233763009",  # Silicotuberculosis (disorder)
        "233794009",  # Tuberculous bronchopleural fistula (disorder)
        "234090004",  # Tuberculous mesenteric adenitis (disorder)
        "235067001",  # Oral tuberculosis (disorder)
        "235694001",  # Tuberculous duodenitis (disorder)
        "236602004",  # Tuberculous stricture of ureter (disorder)
        "236684001",  # Tuberculous urethritis (disorder)
        "236770001",  # Tuberculous epididymo-orchitis (disorder)
        "237445009",  # Tuberculosis of breast (disorder)
        "238430004",  # Cutaneous complication of bacillus Calmette-Guerin immunization (disorder)
        "238431000",  # Bacillus Calmette-Guerin granuloma (disorder)
        "238432007",  # Bacillus Calmette-Guerin ulcer (disorder)
        "240376003",  # Tuberculosis of gastrointestinal tract (disorder)
        "240379005",  # Tuberculosis of male genital organs (disorder)
        "240380008",  # Acute tuberculous ulcer (disorder)
        "240381007",  # Acute miliary cutaneous tuberculosis (disorder)
        "240382000",  # Cutaneous tuberculous gumma (disorder)
        "266355005",  # Bullous emphysema with collapse (disorder)
        "266366003",  # Lung disease due to external agents (disorder)
        "267317007",  # Tuberculosis in pregnancy (disorder)
        "271423008",  # Tuberculosis of skin and subcutaneous tissue (disorder)
        "271503005",  # Pleural empyema with fistula (disorder)
        "271504004",  # Pleural empyema with no fistula (disorder)
        "281623008",  # Genital tuberculosis (disorder)
        "297188009",  # Clotted hemothorax (disorder)
        "301836007",  # Tuberculosis of colon (disorder)
        "302131003",  # Tuberculosis of heart (disorder)
        "307202004",  # Hemorrhagic pleural effusion (disorder)
        "314029000",  # Tuberculous uveitis (disorder)
        "326542006",  # Traumatic atelectasis (disorder)
        "398640008",  # Rheumatoid pneumoconiosis (disorder)
        "402973006",  # Bacillus Calmette-Guerin-induced lupus vulgaris (disorder)
        "402974000",  # Bacillus Calmette-Guerin-induced scrofuloderma (disorder)
        "402975004",  # Bacillus Calmette-Guerin-induced tuberculide (disorder)
        "405576001",  # Nontraumatic hemothorax (disorder)
        "413556004",  # Antibiotic resistant tuberculosis (disorder)
        "413852006",  # Ciprofloxacin resistant tuberculosis (disorder)
        "414146004",  # Ethambutol resistant tuberculosis (disorder)
        "414149006",  # Ethionamide resistant tuberculosis (disorder)
        "414546009",  # Isoniazid resistant tuberculosis (disorder)
        "415222009",  # Pyrazinamide resistant tuberculosis (disorder)
        "415345001",  # Rifampicin resistant tuberculosis (disorder)
        "415622003",  # Streptomycin resistant tuberculosis (disorder)
        "416265003",  # Tuberculoma of brain (disorder)
        "416484003",  # Sickle cell-hemoglobin E disease with crisis (disorder)
        "416826005",  # Sickle cell-thalassemia disease with crisis (disorder)
        "416903004",  # Tuberculoma of spinal cord (disorder)
        "417279003",  # Hemoglobin S sickling disorder with crisis (disorder)
        "417425009",  # Hemoglobin SS disease with crisis (disorder)
        "417484006",  # Tuberculous abscess of spinal cord (disorder)
        "417517009",  # Sickle cell-hemoglobin C disease with crisis (disorder)
        "417748003",  # Sickle cell-hemoglobin D disease with crisis (disorder)
        "418750009",  # On examination - atelectasis (disorder)
        "421571007",  # Tuberculosis associated with acquired immunodeficiency syndrome (disorder)
        "422481005",  # Extreme drug resistant tuberculosis (disorder)
        "423092005",  # Multidrug resistant tuberculosis (disorder)
        "423709000",  # Drug resistant tuberculosis (disorder)
        "423997002",  # Tuberculosis, extrapulmonary (disorder)
        "427099000",  # Active tuberculosis (disorder)
        "428782008",  # Postoperative hemothorax (disorder)
        "440206000",  # Hemoglobin SS disease with vasoocclusive crisis (disorder)
        "441770001",  # Tuberculosis of external body orifice (disorder)
        "443378001",  # Lady Windermere syndrome (disorder)
        "444108000",  # Acute sickle cell splenic sequestration crisis (disorder)
        "446123007",  # Tuberculous abscess (disorder)
        "446543007",  # Tuberculous abscess of lung (disorder)
        "446753005",  # Tuberculosis of oropharynx (disorder)
        "446926000",  # Tuberculous abscess of sacrum (disorder)
        "446946005",  # Reinfection pulmonary tuberculosis (disorder)
        "446986002",  # Tuberculous pleural effusion (disorder)
        "447006007",  # Relapse pulmonary tuberculosis (disorder)
        "447012002",  # Tuberculosis of spinal cord (disorder)
        "447253004",  # Tuberculous arachnoiditis (disorder)
        "447330002",  # Tuberculosis of abdomen (disorder)
        "447332005",  # Tuberculous abscess of epidural space (disorder)
        "447371008",  # Tuberculosis of ileum (disorder)
        "447399004",  # Tuberculous abscess of neck (disorder)
        "448370006",  # Round atelectasis (disorder)
        "449925002",  # Tuberculosis of bone of upper limb (disorder)
        "473426009",  # Primary tuberculous complex confirmed by culture (disorder)
        "473427000",  # Primary tuberculous complex confirmed histologically (disorder)
        "473428005",  # Primary tuberculous complex confirmed by microscopic examination (disorder)
        "473429002",  # Tuberculoma of spinal cord confirmed (disorder)
        "473430007",  # Primary tuberculous complex confirmed (disorder)
        "473431006",  # Primary tuberculous complex with bacteriological or histological examination results unknown (disorder)
        "699326004",  # Accidental puncture of pleura (disorder)
        "700272008",  # Tuberculosis of respiratory system (disorder)
        "700273003",  # Isolated tracheobronchial tuberculosis (disorder)
        "707149004",  # Mycobacterial lymphadenitis (disorder)
        "708025003",  # Pyopneumothorax following infection by Coccidioides (disorder)
        "710106005",  # Extensively drug resistant tuberculosis (disorder)
        "714394004",  # Lupus vulgaris of nose (disorder)
        "717697005",  # Tuberculous ulceration of vulva (disorder)
        "724502006",  # Respiratory disorder co-occurrent and due to microscopic polyangiitis (disorder)
        "724541008",  # Acute miliary tuberculosis of single site (disorder)
        "733497009",  # Chronic suppuration of bronchus (disorder)
        "733835007",  # Extrapulmonary tuberculosis co-occurrent with human immunodeficiency virus infection (disorder)
        "735521001",  # Human immunodeficiency virus World Health Organization 2007 stage 1 co-occurrent with tuberculosis (disorder)
        "735523003",  # Human immunodeficiency virus World Health Organization 2007 stage 2 co-occurrent with tuberculosis (disorder)
        "735525005",  # Human immunodeficiency virus World Health Organization 2007 stage 3 co-occurrent with tuberculosis (disorder)
        "735527002",  # Human immunodeficiency virus World Health Organization 2007 stage 4 co-occurrent with tuberculosis (disorder)
        "770674007",  # Tubercular lesion of lung (disorder)
        "782761005",  # Subacute invasive pulmonary aspergillosis (disorder)
        "789056000",  # Tuberculosis of inner ear (disorder)
        "789057009",  # Tuberculosis of middle ear (disorder)
        "860803004",  # Hydrocephalus due to tuberculosis of brain (disorder)
        "129211000119108",  # Isoniazid resistant tuberculosis of lung (disorder)
        "311671000119100",  # Juvenile dermatomyositis co-occurrent with respiratory involvement (disorder)
        "328561000119107",  # Primary spontaneous pneumothorax (disorder)
        "328571000119101",  # Secondary spontaneous pneumothorax (disorder)
    }


class CompetingConditionsForRespiratoryConditions(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  competing conditions that may warrant treatment with an antibiotic medication for respiratory conditions.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of acute bacterial infections that may warrant treatment with an antibiotic medication.

    **Exclusion Criteria:** Excludes concepts that represent an acute diagnosis that does not warrant treatment with an antibiotic medication.

    ** Used in:** CMS154v10, CMS146v10
    """

    VALUE_SET_NAME = "Competing Conditions for Respiratory Conditions"
    OID = "2.16.840.1.113883.3.464.1003.102.12.1017"
    DEFINITION_VERSION = "20210220"
    EXPANSION_VERSION = "eCQM Update 2021-05-06"

    ICD10CM = {
        "A000",  # Cholera due to Vibrio cholerae 01, biovar cholerae
        "A001",  # Cholera due to Vibrio cholerae 01, biovar eltor
        "A009",  # Cholera, unspecified
        "A0100",  # Typhoid fever, unspecified
        "A0101",  # Typhoid meningitis
        "A0102",  # Typhoid fever with heart involvement
        "A0103",  # Typhoid pneumonia
        "A0104",  # Typhoid arthritis
        "A0105",  # Typhoid osteomyelitis
        "A0109",  # Typhoid fever with other complications
        "A011",  # Paratyphoid fever A
        "A012",  # Paratyphoid fever B
        "A013",  # Paratyphoid fever C
        "A014",  # Paratyphoid fever, unspecified
        "A020",  # Salmonella enteritis
        "A021",  # Salmonella sepsis
        "A0220",  # Localized salmonella infection, unspecified
        "A0221",  # Salmonella meningitis
        "A0222",  # Salmonella pneumonia
        "A0223",  # Salmonella arthritis
        "A0224",  # Salmonella osteomyelitis
        "A0225",  # Salmonella pyelonephritis
        "A0229",  # Salmonella with other localized infection
        "A028",  # Other specified salmonella infections
        "A029",  # Salmonella infection, unspecified
        "A030",  # Shigellosis due to Shigella dysenteriae
        "A031",  # Shigellosis due to Shigella flexneri
        "A032",  # Shigellosis due to Shigella boydii
        "A033",  # Shigellosis due to Shigella sonnei
        "A038",  # Other shigellosis
        "A039",  # Shigellosis, unspecified
        "A040",  # Enteropathogenic Escherichia coli infection
        "A041",  # Enterotoxigenic Escherichia coli infection
        "A042",  # Enteroinvasive Escherichia coli infection
        "A043",  # Enterohemorrhagic Escherichia coli infection
        "A044",  # Other intestinal Escherichia coli infections
        "A045",  # Campylobacter enteritis
        "A046",  # Enteritis due to Yersinia enterocolitica
        "A047",  # Enterocolitis due to Clostridium difficile
        "A0471",  # Enterocolitis due to Clostridium difficile, recurrent
        "A0472",  # Enterocolitis due to Clostridium difficile, not specified as recurrent
        "A048",  # Other specified bacterial intestinal infections
        "A049",  # Bacterial intestinal infection, unspecified
        "A050",  # Foodborne staphylococcal intoxication
        "A051",  # Botulism food poisoning
        "A052",  # Foodborne Clostridium perfringens [Clostridium welchii] intoxication
        "A053",  # Foodborne Vibrio parahaemolyticus intoxication
        "A054",  # Foodborne Bacillus cereus intoxication
        "A055",  # Foodborne Vibrio vulnificus intoxication
        "A058",  # Other specified bacterial foodborne intoxications
        "A059",  # Bacterial foodborne intoxication, unspecified
        "A060",  # Acute amebic dysentery
        "A061",  # Chronic intestinal amebiasis
        "A062",  # Amebic nondysenteric colitis
        "A063",  # Ameboma of intestine
        "A064",  # Amebic liver abscess
        "A065",  # Amebic lung abscess
        "A066",  # Amebic brain abscess
        "A067",  # Cutaneous amebiasis
        "A0681",  # Amebic cystitis
        "A0682",  # Other amebic genitourinary infections
        "A0689",  # Other amebic infections
        "A069",  # Amebiasis, unspecified
        "A070",  # Balantidiasis
        "A071",  # Giardiasis [lambliasis]
        "A072",  # Cryptosporidiosis
        "A073",  # Isosporiasis
        "A074",  # Cyclosporiasis
        "A078",  # Other specified protozoal intestinal diseases
        "A079",  # Protozoal intestinal disease, unspecified
        "A080",  # Rotaviral enteritis
        "A0811",  # Acute gastroenteropathy due to Norwalk agent
        "A0819",  # Acute gastroenteropathy due to other small round viruses
        "A082",  # Adenoviral enteritis
        "A0831",  # Calicivirus enteritis
        "A0832",  # Astrovirus enteritis
        "A0839",  # Other viral enteritis
        "A084",  # Viral intestinal infection, unspecified
        "A088",  # Other specified intestinal infections
        "A09",  # Infectious gastroenteritis and colitis, unspecified
        "A3700",  # Whooping cough due to Bordetella pertussis without pneumonia
        "A3701",  # Whooping cough due to Bordetella pertussis with pneumonia
        "A3710",  # Whooping cough due to Bordetella parapertussis without pneumonia
        "A3711",  # Whooping cough due to Bordetella parapertussis with pneumonia
        "A3780",  # Whooping cough due to other Bordetella species without pneumonia
        "A3781",  # Whooping cough due to other Bordetella species with pneumonia
        "A3790",  # Whooping cough, unspecified species without pneumonia
        "A3791",  # Whooping cough, unspecified species with pneumonia
        "A440",  # Systemic bartonellosis
        "A441",  # Cutaneous and mucocutaneous bartonellosis
        "A448",  # Other forms of bartonellosis
        "A449",  # Bartonellosis, unspecified
        "A499",  # Bacterial infection, unspecified
        "A5001",  # Early congenital syphilitic oculopathy
        "A5002",  # Early congenital syphilitic osteochondropathy
        "A5003",  # Early congenital syphilitic pharyngitis
        "A5004",  # Early congenital syphilitic pneumonia
        "A5005",  # Early congenital syphilitic rhinitis
        "A5006",  # Early cutaneous congenital syphilis
        "A5007",  # Early mucocutaneous congenital syphilis
        "A5008",  # Early visceral congenital syphilis
        "A5009",  # Other early congenital syphilis, symptomatic
        "A501",  # Early congenital syphilis, latent
        "A502",  # Early congenital syphilis, unspecified
        "A5030",  # Late congenital syphilitic oculopathy, unspecified
        "A5031",  # Late congenital syphilitic interstitial keratitis
        "A5032",  # Late congenital syphilitic chorioretinitis
        "A5039",  # Other late congenital syphilitic oculopathy
        "A5040",  # Late congenital neurosyphilis, unspecified
        "A5041",  # Late congenital syphilitic meningitis
        "A5042",  # Late congenital syphilitic encephalitis
        "A5043",  # Late congenital syphilitic polyneuropathy
        "A5044",  # Late congenital syphilitic optic nerve atrophy
        "A5045",  # Juvenile general paresis
        "A5049",  # Other late congenital neurosyphilis
        "A5051",  # Clutton's joints
        "A5052",  # Hutchinson's teeth
        "A5053",  # Hutchinson's triad
        "A5054",  # Late congenital cardiovascular syphilis
        "A5055",  # Late congenital syphilitic arthropathy
        "A5056",  # Late congenital syphilitic osteochondropathy
        "A5057",  # Syphilitic saddle nose
        "A5059",  # Other late congenital syphilis, symptomatic
        "A506",  # Late congenital syphilis, latent
        "A507",  # Late congenital syphilis, unspecified
        "A509",  # Congenital syphilis, unspecified
        "A510",  # Primary genital syphilis
        "A511",  # Primary anal syphilis
        "A512",  # Primary syphilis of other sites
        "A5131",  # Condyloma latum
        "A5132",  # Syphilitic alopecia
        "A5139",  # Other secondary syphilis of skin
        "A5141",  # Secondary syphilitic meningitis
        "A5142",  # Secondary syphilitic female pelvic disease
        "A5143",  # Secondary syphilitic oculopathy
        "A5144",  # Secondary syphilitic nephritis
        "A5145",  # Secondary syphilitic hepatitis
        "A5146",  # Secondary syphilitic osteopathy
        "A5149",  # Other secondary syphilitic conditions
        "A515",  # Early syphilis, latent
        "A519",  # Early syphilis, unspecified
        "A5200",  # Cardiovascular syphilis, unspecified
        "A5201",  # Syphilitic aneurysm of aorta
        "A5202",  # Syphilitic aortitis
        "A5203",  # Syphilitic endocarditis
        "A5204",  # Syphilitic cerebral arteritis
        "A5205",  # Other cerebrovascular syphilis
        "A5206",  # Other syphilitic heart involvement
        "A5209",  # Other cardiovascular syphilis
        "A5210",  # Symptomatic neurosyphilis, unspecified
        "A5211",  # Tabes dorsalis
        "A5212",  # Other cerebrospinal syphilis
        "A5213",  # Late syphilitic meningitis
        "A5214",  # Late syphilitic encephalitis
        "A5215",  # Late syphilitic neuropathy
        "A5216",  # Charcôt's arthropathy (tabetic)
        "A5217",  # General paresis
        "A5219",  # Other symptomatic neurosyphilis
        "A522",  # Asymptomatic neurosyphilis
        "A523",  # Neurosyphilis, unspecified
        "A5271",  # Late syphilitic oculopathy
        "A5272",  # Syphilis of lung and bronchus
        "A5273",  # Symptomatic late syphilis of other respiratory organs
        "A5274",  # Syphilis of liver and other viscera
        "A5275",  # Syphilis of kidney and ureter
        "A5276",  # Other genitourinary symptomatic late syphilis
        "A5277",  # Syphilis of bone and joint
        "A5278",  # Syphilis of other musculoskeletal tissue
        "A5279",  # Other symptomatic late syphilis
        "A528",  # Late syphilis, latent
        "A529",  # Late syphilis, unspecified
        "A530",  # Latent syphilis, unspecified as early or late
        "A539",  # Syphilis, unspecified
        "A5400",  # Gonococcal infection of lower genitourinary tract, unspecified
        "A5401",  # Gonococcal cystitis and urethritis, unspecified
        "A5402",  # Gonococcal vulvovaginitis, unspecified
        "A5403",  # Gonococcal cervicitis, unspecified
        "A5409",  # Other gonococcal infection of lower genitourinary tract
        "A541",  # Gonococcal infection of lower genitourinary tract with periurethral and accessory gland abscess
        "A5421",  # Gonococcal infection of kidney and ureter
        "A5422",  # Gonococcal prostatitis
        "A5423",  # Gonococcal infection of other male genital organs
        "A5424",  # Gonococcal female pelvic inflammatory disease
        "A5429",  # Other gonococcal genitourinary infections
        "A5430",  # Gonococcal infection of eye, unspecified
        "A5431",  # Gonococcal conjunctivitis
        "A5432",  # Gonococcal iridocyclitis
        "A5433",  # Gonococcal keratitis
        "A5439",  # Other gonococcal eye infection
        "A5440",  # Gonococcal infection of musculoskeletal system, unspecified
        "A5441",  # Gonococcal spondylopathy
        "A5442",  # Gonococcal arthritis
        "A5443",  # Gonococcal osteomyelitis
        "A5449",  # Gonococcal infection of other musculoskeletal tissue
        "A545",  # Gonococcal pharyngitis
        "A546",  # Gonococcal infection of anus and rectum
        "A5481",  # Gonococcal meningitis
        "A5482",  # Gonococcal brain abscess
        "A5483",  # Gonococcal heart infection
        "A5484",  # Gonococcal pneumonia
        "A5485",  # Gonococcal peritonitis
        "A5486",  # Gonococcal sepsis
        "A5489",  # Other gonococcal infections
        "A549",  # Gonococcal infection, unspecified
        "A55",  # Chlamydial lymphogranuloma (venereum)
        "A5600",  # Chlamydial infection of lower genitourinary tract, unspecified
        "A5601",  # Chlamydial cystitis and urethritis
        "A5602",  # Chlamydial vulvovaginitis
        "A5609",  # Other chlamydial infection of lower genitourinary tract
        "A5611",  # Chlamydial female pelvic inflammatory disease
        "A5619",  # Other chlamydial genitourinary infection
        "A562",  # Chlamydial infection of genitourinary tract, unspecified
        "A563",  # Chlamydial infection of anus and rectum
        "A564",  # Chlamydial infection of pharynx
        "A568",  # Sexually transmitted chlamydial infection of other sites
        "A57",  # Chancroid
        "A58",  # Granuloma inguinale
        "A5900",  # Urogenital trichomoniasis, unspecified
        "A5901",  # Trichomonal vulvovaginitis
        "A5902",  # Trichomonal prostatitis
        "A5903",  # Trichomonal cystitis and urethritis
        "A5909",  # Other urogenital trichomoniasis
        "A598",  # Trichomoniasis of other sites
        "A599",  # Trichomoniasis, unspecified
        "A630",  # Anogenital (venereal) warts
        "A638",  # Other specified predominantly sexually transmitted diseases
        "A64",  # Unspecified sexually transmitted disease
        "A690",  # Necrotizing ulcerative stomatitis
        "A691",  # Other Vincent's infections
        "A6920",  # Lyme disease, unspecified
        "A6921",  # Meningitis due to Lyme disease
        "A6922",  # Other neurologic disorders in Lyme disease
        "A6923",  # Arthritis due to Lyme disease
        "A6929",  # Other conditions associated with Lyme disease
        "A698",  # Other specified spirochetal infections
        "A699",  # Spirochetal infection, unspecified
        "B600",  # Babesiosis
        "B6000",  # Babesiosis, unspecified
        "B6001",  # Babesiosis due to Babesia microti
        "B6002",  # Babesiosis due to Babesia duncani
        "B6003",  # Babesiosis due to Babesia divergens
        "B6009",  # Other babesiosis
        "B6010",  # Acanthamebiasis, unspecified
        "B6011",  # Meningoencephalitis due to Acanthamoeba (culbertsoni)
        "B6012",  # Conjunctivitis due to Acanthamoeba
        "B6013",  # Keratoconjunctivitis due to Acanthamoeba
        "B6019",  # Other acanthamebic disease
        "B602",  # Naegleriasis
        "B608",  # Other specified protozoal diseases
        "B64",  # Unspecified protozoal disease
        "B781",  # Cutaneous strongyloidiasis
        "B9689",  # Other specified bacterial agents as the cause of diseases classified elsewhere
        "E832",  # Disorders of zinc metabolism
        "H66001",  # Acute suppurative otitis media without spontaneous rupture of ear drum, right ear
        "H66002",  # Acute suppurative otitis media without spontaneous rupture of ear drum, left ear
        "H66003",  # Acute suppurative otitis media without spontaneous rupture of ear drum, bilateral
        "H66004",  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, right ear
        "H66005",  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, left ear
        "H66006",  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, bilateral
        "H66007",  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, unspecified ear
        "H66009",  # Acute suppurative otitis media without spontaneous rupture of ear drum, unspecified ear
        "H66011",  # Acute suppurative otitis media with spontaneous rupture of ear drum, right ear
        "H66012",  # Acute suppurative otitis media with spontaneous rupture of ear drum, left ear
        "H66013",  # Acute suppurative otitis media with spontaneous rupture of ear drum, bilateral
        "H66014",  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, right ear
        "H66015",  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, left ear
        "H66016",  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, bilateral
        "H66017",  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, unspecified ear
        "H66019",  # Acute suppurative otitis media with spontaneous rupture of ear drum, unspecified ear
        "H6610",  # Chronic tubotympanic suppurative otitis media, unspecified
        "H6611",  # Chronic tubotympanic suppurative otitis media, right ear
        "H6612",  # Chronic tubotympanic suppurative otitis media, left ear
        "H6613",  # Chronic tubotympanic suppurative otitis media, bilateral
        "H6620",  # Chronic atticoantral suppurative otitis media, unspecified ear
        "H6621",  # Chronic atticoantral suppurative otitis media, right ear
        "H6622",  # Chronic atticoantral suppurative otitis media, left ear
        "H6623",  # Chronic atticoantral suppurative otitis media, bilateral
        "H663X1",  # Other chronic suppurative otitis media, right ear
        "H663X2",  # Other chronic suppurative otitis media, left ear
        "H663X3",  # Other chronic suppurative otitis media, bilateral
        "H663X9",  # Other chronic suppurative otitis media, unspecified ear
        "H6640",  # Suppurative otitis media, unspecified, unspecified ear
        "H6641",  # Suppurative otitis media, unspecified, right ear
        "H6642",  # Suppurative otitis media, unspecified, left ear
        "H6643",  # Suppurative otitis media, unspecified, bilateral
        "H6690",  # Otitis media, unspecified, unspecified ear
        "H6691",  # Otitis media, unspecified, right ear
        "H6692",  # Otitis media, unspecified, left ear
        "H6693",  # Otitis media, unspecified, bilateral
        "H671",  # Otitis media in diseases classified elsewhere, right ear
        "H672",  # Otitis media in diseases classified elsewhere, left ear
        "H673",  # Otitis media in diseases classified elsewhere, bilateral
        "H679",  # Otitis media in diseases classified elsewhere, unspecified ear
        "H70001",  # Acute mastoiditis without complications, right ear
        "H70002",  # Acute mastoiditis without complications, left ear
        "H70003",  # Acute mastoiditis without complications, bilateral
        "H70009",  # Acute mastoiditis without complications, unspecified ear
        "H70011",  # Subperiosteal abscess of mastoid, right ear
        "H70012",  # Subperiosteal abscess of mastoid, left ear
        "H70013",  # Subperiosteal abscess of mastoid, bilateral
        "H70019",  # Subperiosteal abscess of mastoid, unspecified ear
        "H70091",  # Acute mastoiditis with other complications, right ear
        "H70092",  # Acute mastoiditis with other complications, left ear
        "H70093",  # Acute mastoiditis with other complications, bilateral
        "H70099",  # Acute mastoiditis with other complications, unspecified ear
        "H7010",  # Chronic mastoiditis, unspecified ear
        "H7011",  # Chronic mastoiditis, right ear
        "H7012",  # Chronic mastoiditis, left ear
        "H7013",  # Chronic mastoiditis, bilateral
        "H70201",  # Unspecified petrositis, right ear
        "H70202",  # Unspecified petrositis, left ear
        "H70203",  # Unspecified petrositis, bilateral
        "H70209",  # Unspecified petrositis, unspecified ear
        "H70211",  # Acute petrositis, right ear
        "H70212",  # Acute petrositis, left ear
        "H70213",  # Acute petrositis, bilateral
        "H70219",  # Acute petrositis, unspecified ear
        "H70221",  # Chronic petrositis, right ear
        "H70222",  # Chronic petrositis, left ear
        "H70223",  # Chronic petrositis, bilateral
        "H70229",  # Chronic petrositis, unspecified ear
        "H70811",  # Postauricular fistula, right ear
        "H70812",  # Postauricular fistula, left ear
        "H70813",  # Postauricular fistula, bilateral
        "H70819",  # Postauricular fistula, unspecified ear
        "H70891",  # Other mastoiditis and related conditions, right ear
        "H70892",  # Other mastoiditis and related conditions, left ear
        "H70893",  # Other mastoiditis and related conditions, bilateral
        "H70899",  # Other mastoiditis and related conditions, unspecified ear
        "H7090",  # Unspecified mastoiditis, unspecified ear
        "H7091",  # Unspecified mastoiditis, right ear
        "H7092",  # Unspecified mastoiditis, left ear
        "H7093",  # Unspecified mastoiditis, bilateral
        "H9500",  # Recurrent cholesteatoma of postmastoidectomy cavity, unspecified ear
        "H9501",  # Recurrent cholesteatoma of postmastoidectomy cavity, right ear
        "H9502",  # Recurrent cholesteatoma of postmastoidectomy cavity, left ear
        "H9503",  # Recurrent cholesteatoma of postmastoidectomy cavity, bilateral ears
        "H95111",  # Chronic inflammation of postmastoidectomy cavity, right ear
        "H95112",  # Chronic inflammation of postmastoidectomy cavity, left ear
        "H95113",  # Chronic inflammation of postmastoidectomy cavity, bilateral ears
        "H95119",  # Chronic inflammation of postmastoidectomy cavity, unspecified ear
        "H95121",  # Granulation of postmastoidectomy cavity, right ear
        "H95122",  # Granulation of postmastoidectomy cavity, left ear
        "H95123",  # Granulation of postmastoidectomy cavity, bilateral ears
        "H95129",  # Granulation of postmastoidectomy cavity, unspecified ear
        "H95131",  # Mucosal cyst of postmastoidectomy cavity, right ear
        "H95132",  # Mucosal cyst of postmastoidectomy cavity, left ear
        "H95133",  # Mucosal cyst of postmastoidectomy cavity, bilateral ears
        "H95139",  # Mucosal cyst of postmastoidectomy cavity, unspecified ear
        "H95191",  # Other disorders following mastoidectomy, right ear
        "H95192",  # Other disorders following mastoidectomy, left ear
        "H95193",  # Other disorders following mastoidectomy, bilateral ears
        "H95199",  # Other disorders following mastoidectomy, unspecified ear
        "H9521",  # Intraoperative hemorrhage and hematoma of ear and mastoid process complicating a procedure on the ear and mastoid process
        "H9522",  # Intraoperative hemorrhage and hematoma of ear and mastoid process complicating other procedure
        "H9531",  # Accidental puncture and laceration of the ear and mastoid process during a procedure on the ear and mastoid process
        "H9532",  # Accidental puncture and laceration of the ear and mastoid process during other procedure
        "H9541",  # Postprocedural hemorrhage of ear and mastoid process following a procedure on the ear and mastoid process
        "H9542",  # Postprocedural hemorrhage of ear and mastoid process following other procedure
        "H9551",  # Postprocedural hematoma of ear and mastoid process following a procedure on the ear and mastoid process
        "H9552",  # Postprocedural hematoma of ear and mastoid process following other procedure
        "H9553",  # Postprocedural seroma of ear and mastoid process following a procedure on the ear and mastoid process
        "H9554",  # Postprocedural seroma of ear and mastoid process following other procedure
        "H95811",  # Postprocedural stenosis of right external ear canal
        "H95812",  # Postprocedural stenosis of left external ear canal
        "H95813",  # Postprocedural stenosis of external ear canal, bilateral
        "H95819",  # Postprocedural stenosis of unspecified external ear canal
        "H9588",  # Other intraoperative complications and disorders of the ear and mastoid process, not elsewhere classified
        "H9589",  # Other postprocedural complications and disorders of the ear and mastoid process, not elsewhere classified
        "J0100",  # Acute maxillary sinusitis, unspecified
        "J0101",  # Acute recurrent maxillary sinusitis
        "J0110",  # Acute frontal sinusitis, unspecified
        "J0111",  # Acute recurrent frontal sinusitis
        "J0120",  # Acute ethmoidal sinusitis, unspecified
        "J0121",  # Acute recurrent ethmoidal sinusitis
        "J0130",  # Acute sphenoidal sinusitis, unspecified
        "J0131",  # Acute recurrent sphenoidal sinusitis
        "J0140",  # Acute pansinusitis, unspecified
        "J0141",  # Acute recurrent pansinusitis
        "J0180",  # Other acute sinusitis
        "J0181",  # Other acute recurrent sinusitis
        "J0190",  # Acute sinusitis, unspecified
        "J0191",  # Acute recurrent sinusitis, unspecified
        "J0410",  # Acute tracheitis without obstruction
        "J0411",  # Acute tracheitis with obstruction
        "J042",  # Acute laryngotracheitis
        "J050",  # Acute obstructive laryngitis [croup]
        "J0510",  # Acute epiglottitis without obstruction
        "J0511",  # Acute epiglottitis with obstruction
        "J13",  # Pneumonia due to Streptococcus pneumoniae
        "J14",  # Pneumonia due to Hemophilus influenzae
        "J150",  # Pneumonia due to Klebsiella pneumoniae
        "J151",  # Pneumonia due to Pseudomonas
        "J1520",  # Pneumonia due to staphylococcus, unspecified
        "J15211",  # Pneumonia due to Methicillin susceptible Staphylococcus aureus
        "J15212",  # Pneumonia due to Methicillin resistant Staphylococcus aureus
        "J1529",  # Pneumonia due to other staphylococcus
        "J153",  # Pneumonia due to streptococcus, group B
        "J154",  # Pneumonia due to other streptococci
        "J155",  # Pneumonia due to Escherichia coli
        "J156",  # Pneumonia due to other Gram-negative bacteria
        "J157",  # Pneumonia due to Mycoplasma pneumoniae
        "J158",  # Pneumonia due to other specified bacteria
        "J159",  # Unspecified bacterial pneumonia
        "J160",  # Chlamydial pneumonia
        "J168",  # Pneumonia due to other specified infectious organisms
        "J17",  # Pneumonia in diseases classified elsewhere
        "J180",  # Bronchopneumonia, unspecified organism
        "J181",  # Lobar pneumonia, unspecified organism
        "J188",  # Other pneumonia, unspecified organism
        "J189",  # Pneumonia, unspecified organism
        "J200",  # Acute bronchitis due to Mycoplasma pneumoniae
        "J201",  # Acute bronchitis due to Hemophilus influenzae
        "J202",  # Acute bronchitis due to streptococcus
        "J320",  # Chronic maxillary sinusitis
        "J321",  # Chronic frontal sinusitis
        "J322",  # Chronic ethmoidal sinusitis
        "J323",  # Chronic sphenoidal sinusitis
        "J324",  # Chronic pansinusitis
        "J328",  # Other chronic sinusitis
        "J329",  # Chronic sinusitis, unspecified
        "J3501",  # Chronic tonsillitis
        "J3502",  # Chronic adenoiditis
        "J3503",  # Chronic tonsillitis and adenoiditis
        "J351",  # Hypertrophy of tonsils
        "J352",  # Hypertrophy of adenoids
        "J353",  # Hypertrophy of tonsils with hypertrophy of adenoids
        "J358",  # Other chronic diseases of tonsils and adenoids
        "J359",  # Chronic disease of tonsils and adenoids, unspecified
        "J387",  # Other diseases of larynx
        "J390",  # Retropharyngeal and parapharyngeal abscess
        "J391",  # Other abscess of pharynx
        "J392",  # Other diseases of pharynx
        "J398",  # Other specified diseases of upper respiratory tract
        "J399",  # Disease of upper respiratory tract, unspecified
        "K122",  # Cellulitis and abscess of mouth
        "L0100",  # Impetigo, unspecified
        "L0101",  # Non-bullous impetigo
        "L0102",  # Bockhart's impetigo
        "L0103",  # Bullous impetigo
        "L0109",  # Other impetigo
        "L011",  # Impetiginization of other dermatoses
        "L03011",  # Cellulitis of right finger
        "L03012",  # Cellulitis of left finger
        "L03019",  # Cellulitis of unspecified finger
        "L03021",  # Acute lymphangitis of right finger
        "L03022",  # Acute lymphangitis of left finger
        "L03029",  # Acute lymphangitis of unspecified finger
        "L03031",  # Cellulitis of right toe
        "L03032",  # Cellulitis of left toe
        "L03039",  # Cellulitis of unspecified toe
        "L03041",  # Acute lymphangitis of right toe
        "L03042",  # Acute lymphangitis of left toe
        "L03049",  # Acute lymphangitis of unspecified toe
        "L03111",  # Cellulitis of right axilla
        "L03112",  # Cellulitis of left axilla
        "L03113",  # Cellulitis of right upper limb
        "L03114",  # Cellulitis of left upper limb
        "L03115",  # Cellulitis of right lower limb
        "L03116",  # Cellulitis of left lower limb
        "L03119",  # Cellulitis of unspecified part of limb
        "L03121",  # Acute lymphangitis of right axilla
        "L03122",  # Acute lymphangitis of left axilla
        "L03123",  # Acute lymphangitis of right upper limb
        "L03124",  # Acute lymphangitis of left upper limb
        "L03125",  # Acute lymphangitis of right lower limb
        "L03126",  # Acute lymphangitis of left lower limb
        "L03129",  # Acute lymphangitis of unspecified part of limb
        "L03211",  # Cellulitis of face
        "L03212",  # Acute lymphangitis of face
        "L03213",  # Periorbital cellulitis
        "L03221",  # Cellulitis of neck
        "L03222",  # Acute lymphangitis of neck
        "L03311",  # Cellulitis of abdominal wall
        "L03312",  # Cellulitis of back [any part except buttock]
        "L03313",  # Cellulitis of chest wall
        "L03314",  # Cellulitis of groin
        "L03315",  # Cellulitis of perineum
        "L03316",  # Cellulitis of umbilicus
        "L03317",  # Cellulitis of buttock
        "L03319",  # Cellulitis of trunk, unspecified
        "L03321",  # Acute lymphangitis of abdominal wall
        "L03322",  # Acute lymphangitis of back [any part except buttock]
        "L03323",  # Acute lymphangitis of chest wall
        "L03324",  # Acute lymphangitis of groin
        "L03325",  # Acute lymphangitis of perineum
        "L03326",  # Acute lymphangitis of umbilicus
        "L03327",  # Acute lymphangitis of buttock
        "L03329",  # Acute lymphangitis of trunk, unspecified
        "L03811",  # Cellulitis of head [any part, except face]
        "L03818",  # Cellulitis of other sites
        "L03891",  # Acute lymphangitis of head [any part, except face]
        "L03898",  # Acute lymphangitis of other sites
        "L0390",  # Cellulitis, unspecified
        "L0391",  # Acute lymphangitis, unspecified
        "L040",  # Acute lymphadenitis of face, head and neck
        "L041",  # Acute lymphadenitis of trunk
        "L042",  # Acute lymphadenitis of upper limb
        "L043",  # Acute lymphadenitis of lower limb
        "L048",  # Acute lymphadenitis of other sites
        "L049",  # Acute lymphadenitis, unspecified
        "L081",  # Erythrasma
        "L0881",  # Pyoderma vegetans
        "L0882",  # Omphalitis not of newborn
        "L0889",  # Other specified local infections of the skin and subcutaneous tissue
        "L089",  # Local infection of the skin and subcutaneous tissue, unspecified
        "L88",  # Pyoderma gangrenosum
        "L928",  # Other granulomatous disorders of the skin and subcutaneous tissue
        "L980",  # Pyogenic granuloma
        "L983",  # Eosinophilic cellulitis [Wells]
        "M4620",  # Osteomyelitis of vertebra, site unspecified
        "M4621",  # Osteomyelitis of vertebra, occipito-atlanto-axial region
        "M4622",  # Osteomyelitis of vertebra, cervical region
        "M4623",  # Osteomyelitis of vertebra, cervicothoracic region
        "M4624",  # Osteomyelitis of vertebra, thoracic region
        "M4625",  # Osteomyelitis of vertebra, thoracolumbar region
        "M4626",  # Osteomyelitis of vertebra, lumbar region
        "M4627",  # Osteomyelitis of vertebra, lumbosacral region
        "M4628",  # Osteomyelitis of vertebra, sacral and sacrococcygeal region
        "M4630",  # Infection of intervertebral disc (pyogenic), site unspecified
        "M4631",  # Infection of intervertebral disc (pyogenic), occipito-atlanto-axial region
        "M4632",  # Infection of intervertebral disc (pyogenic), cervical region
        "M4633",  # Infection of intervertebral disc (pyogenic), cervicothoracic region
        "M4634",  # Infection of intervertebral disc (pyogenic), thoracic region
        "M4635",  # Infection of intervertebral disc (pyogenic), thoracolumbar region
        "M4636",  # Infection of intervertebral disc (pyogenic), lumbar region
        "M4637",  # Infection of intervertebral disc (pyogenic), lumbosacral region
        "M4638",  # Infection of intervertebral disc (pyogenic), sacral and sacrococcygeal region
        "M4639",  # Infection of intervertebral disc (pyogenic), multiple sites in spine
        "M8900",  # Algoneurodystrophy, unspecified site
        "M89011",  # Algoneurodystrophy, right shoulder
        "M89012",  # Algoneurodystrophy, left shoulder
        "M89019",  # Algoneurodystrophy, unspecified shoulder
        "M89021",  # Algoneurodystrophy, right upper arm
        "M89022",  # Algoneurodystrophy, left upper arm
        "M89029",  # Algoneurodystrophy, unspecified upper arm
        "M89031",  # Algoneurodystrophy, right forearm
        "M89032",  # Algoneurodystrophy, left forearm
        "M89039",  # Algoneurodystrophy, unspecified forearm
        "M89041",  # Algoneurodystrophy, right hand
        "M89042",  # Algoneurodystrophy, left hand
        "M89049",  # Algoneurodystrophy, unspecified hand
        "M89051",  # Algoneurodystrophy, right thigh
        "M89052",  # Algoneurodystrophy, left thigh
        "M89059",  # Algoneurodystrophy, unspecified thigh
        "M89061",  # Algoneurodystrophy, right lower leg
        "M89062",  # Algoneurodystrophy, left lower leg
        "M89069",  # Algoneurodystrophy, unspecified lower leg
        "M89071",  # Algoneurodystrophy, right ankle and foot
        "M89072",  # Algoneurodystrophy, left ankle and foot
        "M89079",  # Algoneurodystrophy, unspecified ankle and foot
        "M8908",  # Algoneurodystrophy, other site
        "M8909",  # Algoneurodystrophy, multiple sites
        "M89121",  # Complete physeal arrest, right proximal humerus
        "M89122",  # Complete physeal arrest, left proximal humerus
        "M89123",  # Partial physeal arrest, right proximal humerus
        "M89124",  # Partial physeal arrest, left proximal humerus
        "M89125",  # Complete physeal arrest, right distal humerus
        "M89126",  # Complete physeal arrest, left distal humerus
        "M89127",  # Partial physeal arrest, right distal humerus
        "M89128",  # Partial physeal arrest, left distal humerus
        "M89129",  # Physeal arrest, humerus, unspecified
        "M89131",  # Complete physeal arrest, right distal radius
        "M89132",  # Complete physeal arrest, left distal radius
        "M89133",  # Partial physeal arrest, right distal radius
        "M89134",  # Partial physeal arrest, left distal radius
        "M89138",  # Other physeal arrest of forearm
        "M89139",  # Physeal arrest, forearm, unspecified
        "M89151",  # Complete physeal arrest, right proximal femur
        "M89152",  # Complete physeal arrest, left proximal femur
        "M89153",  # Partial physeal arrest, right proximal femur
        "M89154",  # Partial physeal arrest, left proximal femur
        "M89155",  # Complete physeal arrest, right distal femur
        "M89156",  # Complete physeal arrest, left distal femur
        "M89157",  # Partial physeal arrest, right distal femur
        "M89158",  # Partial physeal arrest, left distal femur
        "M89159",  # Physeal arrest, femur, unspecified
        "M89160",  # Complete physeal arrest, right proximal tibia
        "M89161",  # Complete physeal arrest, left proximal tibia
        "M89162",  # Partial physeal arrest, right proximal tibia
        "M89163",  # Partial physeal arrest, left proximal tibia
        "M89164",  # Complete physeal arrest, right distal tibia
        "M89165",  # Complete physeal arrest, left distal tibia
        "M89166",  # Partial physeal arrest, right distal tibia
        "M89167",  # Partial physeal arrest, left distal tibia
        "M89168",  # Other physeal arrest of lower leg
        "M89169",  # Physeal arrest, lower leg, unspecified
        "M8918",  # Physeal arrest, other site
        "M8920",  # Other disorders of bone development and growth, unspecified site
        "M89211",  # Other disorders of bone development and growth, right shoulder
        "M89212",  # Other disorders of bone development and growth, left shoulder
        "M89219",  # Other disorders of bone development and growth, unspecified shoulder
        "M89221",  # Other disorders of bone development and growth, right humerus
        "M89222",  # Other disorders of bone development and growth, left humerus
        "M89229",  # Other disorders of bone development and growth, unspecified humerus
        "M89231",  # Other disorders of bone development and growth, right ulna
        "M89232",  # Other disorders of bone development and growth, left ulna
        "M89233",  # Other disorders of bone development and growth, right radius
        "M89234",  # Other disorders of bone development and growth, left radius
        "M89239",  # Other disorders of bone development and growth, unspecified ulna and radius
        "M89241",  # Other disorders of bone development and growth, right hand
        "M89242",  # Other disorders of bone development and growth, left hand
        "M89249",  # Other disorders of bone development and growth, unspecified hand
        "M89251",  # Other disorders of bone development and growth, right femur
        "M89252",  # Other disorders of bone development and growth, left femur
        "M89259",  # Other disorders of bone development and growth, unspecified femur
        "M89261",  # Other disorders of bone development and growth, right tibia
        "M89262",  # Other disorders of bone development and growth, left tibia
        "M89263",  # Other disorders of bone development and growth, right fibula
        "M89264",  # Other disorders of bone development and growth, left fibula
        "M89269",  # Other disorders of bone development and growth, unspecified lower leg
        "M89271",  # Other disorders of bone development and growth, right ankle and foot
        "M89272",  # Other disorders of bone development and growth, left ankle and foot
        "M89279",  # Other disorders of bone development and growth, unspecified ankle and foot
        "M8928",  # Other disorders of bone development and growth, other site
        "M8929",  # Other disorders of bone development and growth, multiple sites
        "M8940",  # Other hypertrophic osteoarthropathy, unspecified site
        "M89411",  # Other hypertrophic osteoarthropathy, right shoulder
        "M89412",  # Other hypertrophic osteoarthropathy, left shoulder
        "M89419",  # Other hypertrophic osteoarthropathy, unspecified shoulder
        "M89421",  # Other hypertrophic osteoarthropathy, right upper arm
        "M89422",  # Other hypertrophic osteoarthropathy, left upper arm
        "M89429",  # Other hypertrophic osteoarthropathy, unspecified upper arm
        "M89431",  # Other hypertrophic osteoarthropathy, right forearm
        "M89432",  # Other hypertrophic osteoarthropathy, left forearm
        "M89439",  # Other hypertrophic osteoarthropathy, unspecified forearm
        "M89441",  # Other hypertrophic osteoarthropathy, right hand
        "M89442",  # Other hypertrophic osteoarthropathy, left hand
        "M89449",  # Other hypertrophic osteoarthropathy, unspecified hand
        "M89451",  # Other hypertrophic osteoarthropathy, right thigh
        "M89452",  # Other hypertrophic osteoarthropathy, left thigh
        "M89459",  # Other hypertrophic osteoarthropathy, unspecified thigh
        "M89461",  # Other hypertrophic osteoarthropathy, right lower leg
        "M89462",  # Other hypertrophic osteoarthropathy, left lower leg
        "M89469",  # Other hypertrophic osteoarthropathy, unspecified lower leg
        "M89471",  # Other hypertrophic osteoarthropathy, right ankle and foot
        "M89472",  # Other hypertrophic osteoarthropathy, left ankle and foot
        "M89479",  # Other hypertrophic osteoarthropathy, unspecified ankle and foot
        "M8948",  # Other hypertrophic osteoarthropathy, other site
        "M8949",  # Other hypertrophic osteoarthropathy, multiple sites
        "M8950",  # Osteolysis, unspecified site
        "M89511",  # Osteolysis, right shoulder
        "M89512",  # Osteolysis, left shoulder
        "M89519",  # Osteolysis, unspecified shoulder
        "M89521",  # Osteolysis, right upper arm
        "M89522",  # Osteolysis, left upper arm
        "M89529",  # Osteolysis, unspecified upper arm
        "M89531",  # Osteolysis, right forearm
        "M89532",  # Osteolysis, left forearm
        "M89539",  # Osteolysis, unspecified forearm
        "M89541",  # Osteolysis, right hand
        "M89542",  # Osteolysis, left hand
        "M89549",  # Osteolysis, unspecified hand
        "M89551",  # Osteolysis, right thigh
        "M89552",  # Osteolysis, left thigh
        "M89559",  # Osteolysis, unspecified thigh
        "M89561",  # Osteolysis, right lower leg
        "M89562",  # Osteolysis, left lower leg
        "M89569",  # Osteolysis, unspecified lower leg
        "M89571",  # Osteolysis, right ankle and foot
        "M89572",  # Osteolysis, left ankle and foot
        "M89579",  # Osteolysis, unspecified ankle and foot
        "M8958",  # Osteolysis, other site
        "M8959",  # Osteolysis, multiple sites
        "M8960",  # Osteopathy after poliomyelitis, unspecified site
        "M89611",  # Osteopathy after poliomyelitis, right shoulder
        "M89612",  # Osteopathy after poliomyelitis, left shoulder
        "M89619",  # Osteopathy after poliomyelitis, unspecified shoulder
        "M89621",  # Osteopathy after poliomyelitis, right upper arm
        "M89622",  # Osteopathy after poliomyelitis, left upper arm
        "M89629",  # Osteopathy after poliomyelitis, unspecified upper arm
        "M89631",  # Osteopathy after poliomyelitis, right forearm
        "M89632",  # Osteopathy after poliomyelitis, left forearm
        "M89639",  # Osteopathy after poliomyelitis, unspecified forearm
        "M89641",  # Osteopathy after poliomyelitis, right hand
        "M89642",  # Osteopathy after poliomyelitis, left hand
        "M89649",  # Osteopathy after poliomyelitis, unspecified hand
        "M89651",  # Osteopathy after poliomyelitis, right thigh
        "M89652",  # Osteopathy after poliomyelitis, left thigh
        "M89659",  # Osteopathy after poliomyelitis, unspecified thigh
        "M89661",  # Osteopathy after poliomyelitis, right lower leg
        "M89662",  # Osteopathy after poliomyelitis, left lower leg
        "M89669",  # Osteopathy after poliomyelitis, unspecified lower leg
        "M89671",  # Osteopathy after poliomyelitis, right ankle and foot
        "M89672",  # Osteopathy after poliomyelitis, left ankle and foot
        "M89679",  # Osteopathy after poliomyelitis, unspecified ankle and foot
        "M8968",  # Osteopathy after poliomyelitis, other site
        "M8969",  # Osteopathy after poliomyelitis, multiple sites
        "M8970",  # Major osseous defect, unspecified site
        "M89711",  # Major osseous defect, right shoulder region
        "M89712",  # Major osseous defect, left shoulder region
        "M89719",  # Major osseous defect, unspecified shoulder region
        "M89721",  # Major osseous defect, right humerus
        "M89722",  # Major osseous defect, left humerus
        "M89729",  # Major osseous defect, unspecified humerus
        "M89731",  # Major osseous defect, right forearm
        "M89732",  # Major osseous defect, left forearm
        "M89739",  # Major osseous defect, unspecified forearm
        "M89741",  # Major osseous defect, right hand
        "M89742",  # Major osseous defect, left hand
        "M89749",  # Major osseous defect, unspecified hand
        "M89751",  # Major osseous defect, right pelvic region and thigh
        "M89752",  # Major osseous defect, left pelvic region and thigh
        "M89759",  # Major osseous defect, unspecified pelvic region and thigh
        "M89761",  # Major osseous defect, right lower leg
        "M89762",  # Major osseous defect, left lower leg
        "M89769",  # Major osseous defect, unspecified lower leg
        "M89771",  # Major osseous defect, right ankle and foot
        "M89772",  # Major osseous defect, left ankle and foot
        "M89779",  # Major osseous defect, unspecified ankle and foot
        "M8978",  # Major osseous defect, other site
        "M8979",  # Major osseous defect, multiple sites
        "M898X0",  # Other specified disorders of bone, multiple sites
        "M898X1",  # Other specified disorders of bone, shoulder
        "M898X2",  # Other specified disorders of bone, upper arm
        "M898X3",  # Other specified disorders of bone, forearm
        "M898X4",  # Other specified disorders of bone, hand
        "M898X5",  # Other specified disorders of bone, thigh
        "M898X6",  # Other specified disorders of bone, lower leg
        "M898X7",  # Other specified disorders of bone, ankle and foot
        "M898X8",  # Other specified disorders of bone, other site
        "M898X9",  # Other specified disorders of bone, unspecified site
        "M899",  # Disorder of bone, unspecified
        "M9080",  # Osteopathy in diseases classified elsewhere, unspecified site
        "M90811",  # Osteopathy in diseases classified elsewhere, right shoulder
        "M90812",  # Osteopathy in diseases classified elsewhere, left shoulder
        "M90819",  # Osteopathy in diseases classified elsewhere, unspecified shoulder
        "M90821",  # Osteopathy in diseases classified elsewhere, right upper arm
        "M90822",  # Osteopathy in diseases classified elsewhere, left upper arm
        "M90829",  # Osteopathy in diseases classified elsewhere, unspecified upper arm
        "M90831",  # Osteopathy in diseases classified elsewhere, right forearm
        "M90832",  # Osteopathy in diseases classified elsewhere, left forearm
        "M90839",  # Osteopathy in diseases classified elsewhere, unspecified forearm
        "M90841",  # Osteopathy in diseases classified elsewhere, right hand
        "M90842",  # Osteopathy in diseases classified elsewhere, left hand
        "M90849",  # Osteopathy in diseases classified elsewhere, unspecified hand
        "M90851",  # Osteopathy in diseases classified elsewhere, right thigh
        "M90852",  # Osteopathy in diseases classified elsewhere, left thigh
        "M90859",  # Osteopathy in diseases classified elsewhere, unspecified thigh
        "M90861",  # Osteopathy in diseases classified elsewhere, right lower leg
        "M90862",  # Osteopathy in diseases classified elsewhere, left lower leg
        "M90869",  # Osteopathy in diseases classified elsewhere, unspecified lower leg
        "M90871",  # Osteopathy in diseases classified elsewhere, right ankle and foot
        "M90872",  # Osteopathy in diseases classified elsewhere, left ankle and foot
        "M90879",  # Osteopathy in diseases classified elsewhere, unspecified ankle and foot
        "M9088",  # Osteopathy in diseases classified elsewhere, other site
        "M9089",  # Osteopathy in diseases classified elsewhere, multiple sites
        "N10",  # Acute pyelonephritis
        "N110",  # Nonobstructive reflux-associated chronic pyelonephritis
        "N111",  # Chronic obstructive pyelonephritis
        "N118",  # Other chronic tubulo-interstitial nephritis
        "N119",  # Chronic tubulo-interstitial nephritis, unspecified
        "N12",  # Tubulo-interstitial nephritis, not specified as acute or chronic
        "N130",  # Hydronephrosis with ureteropelvic junction obstruction
        "N136",  # Pyonephrosis
        "N151",  # Renal and perinephric abscess
        "N16",  # Renal tubulo-interstitial disorders in diseases classified elsewhere
        "N2884",  # Pyelitis cystica
        "N2885",  # Pyeloureteritis cystica
        "N2886",  # Ureteritis cystica
        "N3000",  # Acute cystitis without hematuria
        "N3001",  # Acute cystitis with hematuria
        "N3010",  # Interstitial cystitis (chronic) without hematuria
        "N3011",  # Interstitial cystitis (chronic) with hematuria
        "N3020",  # Other chronic cystitis without hematuria
        "N3021",  # Other chronic cystitis with hematuria
        "N3030",  # Trigonitis without hematuria
        "N3031",  # Trigonitis with hematuria
        "N3040",  # Irradiation cystitis without hematuria
        "N3041",  # Irradiation cystitis with hematuria
        "N3080",  # Other cystitis without hematuria
        "N3081",  # Other cystitis with hematuria
        "N3090",  # Cystitis, unspecified without hematuria
        "N3091",  # Cystitis, unspecified with hematuria
        "N390",  # Urinary tract infection, site not specified
        "N410",  # Acute prostatitis
        "N411",  # Chronic prostatitis
        "N412",  # Abscess of prostate
        "N413",  # Prostatocystitis
        "N414",  # Granulomatous prostatitis
        "N418",  # Other inflammatory diseases of prostate
        "N419",  # Inflammatory disease of prostate, unspecified
        "N7001",  # Acute salpingitis
        "N7002",  # Acute oophoritis
        "N7003",  # Acute salpingitis and oophoritis
        "N7011",  # Chronic salpingitis
        "N7012",  # Chronic oophoritis
        "N7013",  # Chronic salpingitis and oophoritis
        "N7091",  # Salpingitis, unspecified
        "N7092",  # Oophoritis, unspecified
        "N7093",  # Salpingitis and oophoritis, unspecified
        "N710",  # Acute inflammatory disease of uterus
        "N711",  # Chronic inflammatory disease of uterus
        "N719",  # Inflammatory disease of uterus, unspecified
        "N72",  # Inflammatory disease of cervix uteri
        "N730",  # Acute parametritis and pelvic cellulitis
        "N731",  # Chronic parametritis and pelvic cellulitis
        "N732",  # Unspecified parametritis and pelvic cellulitis
        "N733",  # Female acute pelvic peritonitis
        "N734",  # Female chronic pelvic peritonitis
        "N735",  # Female pelvic peritonitis, unspecified
        "N736",  # Female pelvic peritoneal adhesions (postinfective)
        "N738",  # Other specified female pelvic inflammatory diseases
        "N739",  # Female pelvic inflammatory disease, unspecified
        "N74",  # Female pelvic inflammatory disorders in diseases classified elsewhere
        "N750",  # Cyst of Bartholin's gland
        "N751",  # Abscess of Bartholin's gland
        "N758",  # Other diseases of Bartholin's gland
        "N759",  # Disease of Bartholin's gland, unspecified
        "N760",  # Acute vaginitis
        "N761",  # Subacute and chronic vaginitis
        "N762",  # Acute vulvitis
        "N763",  # Subacute and chronic vulvitis
        "N764",  # Abscess of vulva
        "N765",  # Ulceration of vagina
        "N766",  # Ulceration of vulva
        "N7681",  # Mucositis (ulcerative) of vagina and vulva
        "N7689",  # Other specified inflammation of vagina and vulva
        "N770",  # Ulceration of vulva in diseases classified elsewhere
        "N771",  # Vaginitis, vulvitis and vulvovaginitis in diseases classified elsewhere
        "Z202",  # Contact with and (suspected) exposure to infections with a predominantly sexual mode of transmission
        "Z224",  # Carrier of infections with a predominantly sexual mode of transmission
    }
    SNOMEDCT = {
        "151004",  # Gonococcal meningitis (disorder)
        "181007",  # Hemorrhagic bronchopneumonia (disorder)
        "192008",  # Congenital syphilitic hepatomegaly (disorder)
        "714002",  # Abrasion and/or friction burn of toe with infection (disorder)
        "801006",  # Nonvenomous insect bite of foot with infection (disorder)
        "827006",  # Late congenital syphilis, latent (positive serology - cerebrospinal fluid, 2 years OR more) (disorder)
        "1085006",  # Candidiasis of vulva (disorder)
        "1102005",  # Intraerythrocytic parasitosis caused by Nuttallia (disorder)
        "1107004",  # Early latent syphilis, positive serology, negative cerebrospinal fluid, with relapse after treatment (disorder)
        "1282001",  # Perichondritis of larynx (disorder)
        "1515008",  # Gorham's disease (disorder)
        "2129002",  # Edema of pharynx (disorder)
        "2390000",  # Acute gonococcal vulvovaginitis (disorder)
        "2435008",  # Ascariasis (disorder)
        "2513003",  # Tinea capitis caused by Trichophyton (disorder)
        "2523007",  # Salmonella pneumonia (disorder)
        "2622007",  # Infected ulcer of skin (disorder)
        "3261002",  # Migratory osteolysis (disorder)
        "3464007",  # Infection caused by Oesophagostomum dentatum (disorder)
        "3487004",  # Candidiasis of lung (disorder)
        "3589003",  # Syphilitic pericarditis (disorder)
        "3752003",  # Trichuriasis (disorder)
        "4016003",  # Empyema of mastoid (disorder)
        "4082005",  # Syphilitic myocarditis (disorder)
        "4324001",  # Subacute cystitis (disorder)
        "4340003",  # Acrodermatitis atrophicans chronica (disorder)
        "4359001",  # Early congenital syphilis (less than 2 years) (disorder)
        "4483005",  # Syphilitic punched out ulcer (disorder)
        "4499004",  # Disseminated anergic leishmaniasis (disorder)
        "4669001",  # Pinta, primary lesion (disorder)
        "4834000",  # Typhoid fever (disorder)
        "5085001",  # Gonococcemia (disorder)
        "5093001",  # Cystitis with actinomycosis (disorder)
        "5168000",  # Sporotrichotic chancre (disorder)
        "5441008",  # Tinea capitis (disorder)
        "5490008",  # Tinea barbae caused by Trichophyton verrucosum (disorder)
        "5765005",  # Infection caused by Leishmania tropica minor (disorder)
        "6020002",  # Tinea pedis (disorder)
        "6267005",  # Congenital syphilitic meningitis (disorder)
        "6365004",  # Cutaneous actinomycosis (disorder)
        "7063008",  # Gangrenous pneumonia (disorder)
        "7138005",  # Caries of mastoid (disorder)
        "7448003",  # Acute pyonephrosis with renal medullary necrosis (disorder)
        "7493006",  # Infection caused by Plagiorchis (disorder)
        "7548000",  # Rheumatic pneumonia (disorder)
        "7877005",  # Infection caused by Diphyllobothrium latum (disorder)
        "8098009",  # Sexually transmitted infectious disease (disorder)
        "8549006",  # Desquamative interstitial pneumonia (disorder)
        "8555001",  # Syphilis of lung (disorder)
        "8748000",  # Abrasion and/or friction burn of hip with infection (disorder)
        "8776008",  # Amebic cystitis (disorder)
        "9091006",  # Gonococcal iridocyclitis (disorder)
        "9241004",  # Gonococcal heart disease (disorder)
        "9480005",  # Viral papular dermatitis (disorder)
        "9713002",  # Prostatitis (disorder)
        "9723006",  # Hyperphosphatasemia with bone disease (disorder)
        "9802006",  # Abrasion and/or friction burn of buttock with infection (disorder)
        "9866007",  # Infection caused by Trichuris vulpis (disorder)
        "9889004",  # Tinea barbae caused by Trichophyton violaceum (disorder)
        "9941009",  # Congenital syphilitic choroiditis (disorder)
        "10345003",  # Primary syphilis of tonsils (disorder)
        "10662001",  # Nonvenomous insect bite of face with infection (disorder)
        "10671005",  # Blister of scapular region with infection (disorder)
        "10679007",  # Infection caused by Giardia lamblia (disorder)
        "10920005",  # Abrasion and/or friction burn of lip with infection (disorder)
        "10995009",  # Abrasion and/or friction burn of scapular region with infection (disorder)
        "11251000",  # Irradiation cystitis (disorder)
        "11258006",  # Tinea barbae caused by Trichophyton mentagrophytes (disorder)
        "11338007",  # Syphilitic episcleritis (disorder)
        "11625007",  # Torus mandibularis (disorder)
        "11840006",  # Traveler's diarrhea (disorder)
        "11906007",  # Chronic gonococcal vulvovaginitis (disorder)
        "12008008",  # Palmar papilloma of yaws (disorder)
        "12232008",  # Syphilitic aneurysm of aorta (disorder)
        "12373006",  # Chronic gonococcal bartholinitis (disorder)
        "12435008",  # Abrasion and/or friction burn of anus with infection (disorder)
        "12463005",  # Infectious gastroenteritis (disorder)
        "12602008",  # Non-hereditary acrodermatitis enteropathica (disorder)
        "12804006",  # Tinea cruris caused by Trichophyton rubrum (disorder)
        "13087006",  # Necrosis of larynx (disorder)
        "13095005",  # Primary symptomatic early syphilis (disorder)
        "13125003",  # Chronic myometritis (disorder)
        "13177009",  # Cellulitis of nasopharynx (disorder)
        "13285005",  # Cystitis cystica (disorder)
        "13310005",  # Taboparesis (disorder)
        "13338002",  # Verruca palmaris (disorder)
        "13464007",  # Infestation caused by Cheyletiella parasitivorax (disorder)
        "13471002",  # Infection caused by Trichuris ovis (disorder)
        "13731006",  # Secondary symptomatic early syphilis (disorder)
        "14096004",  # Nonvenomous insect bite of scapular region with infection (disorder)
        "14386001",  # Indeterminate leprosy (disorder)
        "14560005",  # Dermal mycosis (disorder)
        "14948001",  # Acute suppurative otitis media without spontaneous rupture of ear drum (disorder)
        "15033003",  # Peritonsillar abscess (disorder)
        "15034009",  # Nonvenomous insect bite of breast with infection (disorder)
        "15207003",  # Zoonotic form of cutaneous leishmaniasis (disorder)
        "15470004",  # Toxic diffuse goiter with acropachy (disorder)
        "15628003",  # Gonorrhea (disorder)
        "15634005",  # Palmar hyperkeratosis of yaws (disorder)
        "15845006",  # Buruli ulcer (disorder)
        "16036000",  # Acute empyema of frontal sinus (disorder)
        "16070004",  # Syphilitic cirrhosis (disorder)
        "16358007",  # Chronic disease of tonsils AND/OR adenoids (disorder)
        "16429005",  # Cutaneous larva migrans caused by Ancylostoma braziliense (disorder)
        "16442005",  # Leontiasis ossium (disorder)
        "16681000",  # Candidal otitis externa (disorder)
        "17039000",  # Blister of abdominal wall with infection (disorder)
        "17198008",  # Nonvenomous insect bite of ear with infection (disorder)
        "17250009",  # Blister of interscapular region with infection (disorder)
        "17273001",  # Mucocutaneous cryptococcosis (disorder)
        "17305005",  # Acute gonorrhea of genitourinary tract (disorder)
        "17384006",  # Nonvenomous insect bite of toe with infection (disorder)
        "17723000",  # Infection caused by Naegleria (disorder)
        "17794004",  # Recurrent cutaneous leishmaniasis (disorder)
        "18081009",  # Enteric campylobacteriosis (disorder)
        "18099001",  # Retropharyngeal abscess (disorder)
        "18229003",  # Suppurative colitis (disorder)
        "18849004",  # Infection caused by Trichostrongylus axei (disorder)
        "18900005",  # Periporitis (disorder)
        "18901009",  # Cutaneous diphtheria (disorder)
        "19108007",  # Nonvenomous insect bite of trunk with infection (disorder)
        "19138001",  # Epidermodysplasia verruciformis (disorder)
        "19206003",  # Syphilitic acoustic neuritis (disorder)
        "19213003",  # Infectious diarrheal disease (disorder)
        "19290004",  # Clutton's joints (disorder)
        "19362000",  # Cutaneous larva migrans (disorder)
        "19824006",  # Infection of skin and/or subcutaneous tissue (disorder)
        "19905009",  # Chronic prostatitis (disorder)
        "20325007",  # Tinea barbae caused by Microsporum (disorder)
        "20547008",  # Ill-defined intestinal infection (disorder)
        "20735004",  # Syphilitic aortitis (disorder)
        "20860008",  # Infestation of intestine caused by fly larvae (disorder)
        "20943002",  # Acute gonococcal cervicitis (disorder)
        "21061004",  # Babesiosis (disorder)
        "21523006",  # Syphilitic gumma of central nervous system (disorder)
        "21560005",  # Lepromatous leprosy (disorder)
        "21955004",  # Trichomonal cervicitis (disorder)
        "22064009",  # Pinta (disorder)
        "22277006",  # Blister of groin with infection (disorder)
        "22386003",  # Syphilitic optic atrophy (disorder)
        "22500005",  # Infection by Oesophagostomum (disorder)
        "22557000",  # Bullous cystitis (disorder)
        "22754005",  # Staphylococcal pneumonia (disorder)
        "22905009",  # Infection caused by Heterophyes heterophyes (disorder)
        "22922006",  # Infection caused by Physaloptera (disorder)
        "23166004",  # Cellulitis of pharynx (disorder)
        "23191004",  # Early yaws (disorder)
        "23196009",  # Frambeside of early yaws (disorder)
        "23502006",  # Lyme disease (disorder)
        "23550005",  # Syphilis of bursa (disorder)
        "23754003",  # Calculous pyelonephritis (disorder)
        "23874000",  # Chronic amebiasis (disorder)
        "23949002",  # Capillaria philippinensis infection (disorder)
        "23975003",  # Chronic gonococcal seminal vesiculitis (disorder)
        "24347001",  # Cellulitis of vocal cords (disorder)
        "24620004",  # Hemolytic anemia caused by babesiosis (disorder)
        "24752008",  # Infantile cortical hyperostosis (disorder)
        "24789006",  # Viral gastroenteritis caused by Norwalk-like agent (disorder)
        "24868007",  # Acute gonococcal cystitis (disorder)
        "25055007",  # Ingrowing nail with infection (disorder)
        "25188002",  # Pediculosis corporis (disorder)
        "25626000",  # Cutaneous infectious disease caused by Mycobacteria (disorder)
        "25726008",  # Nonvenomous insect bite of groin with infection (disorder)
        "25956006",  # Tinea pedis caused by Trichophyton (disorder)
        "25994006",  # Bone island (disorder)
        "26039008",  # Neurosyphilis (disorder)
        "26067004",  # Madura foot caused by Streptomyces (disorder)
        "26135000",  # Syphilitic encephalitis (disorder)
        "26151008",  # Graves' disease with acropachy AND with thyrotoxic crisis (disorder)
        "26249004",  # Intestinal infection caused by Class Cestoda and/or Class Trematoda and/or Phylum Nemata (disorder)
        "26535009",  # Carbuncle of vulva (disorder)
        "26652002",  # Abrasion and/or friction burn of scalp with infection (disorder)
        "26826005",  # Amebic appendicitis (disorder)
        "27023008",  # Infective dermatitis of eyelid (disorder)
        "27075004",  # Venereal disease in mother complicating pregnancy, childbirth AND/OR puerperium (disorder)
        "27278006",  # Acute empyema of sphenoidal sinus (disorder)
        "27460003",  # Primary syphilis of breast (disorder)
        "27648007",  # Congenital syphilitic periostitis (disorder)
        "27681008",  # Chronic gonorrhea (disorder)
        "27836007",  # Pertussis (disorder)
        "27858009",  # Clostridial gastroenteritis (disorder)
        "28198007",  # Primary syphilis of lip (disorder)
        "28438004",  # Gonococcal conjunctivitis neonatorum (disorder)
        "28572009",  # Chronic gonorrhea of genitourinary tract (disorder)
        "28709001",  # Cellulitis of larynx (disorder)
        "28867007",  # Granuloma inguinale (disorder)
        "28951000",  # Infection caused by Trichuris suis (disorder)
        "29604006",  # Infection caused by Trichostrongylus colubriformis (disorder)
        "29864006",  # Acute gonococcal urethritis (disorder)
        "29979000",  # Infection caused by Enteromonas hominis (disorder)
        "30168008",  # Acute gonococcal epididymo-orchitis (disorder)
        "30242009",  # Scarlet fever (disorder)
        "30283007",  # Initial frambesial ulcer (disorder)
        "30358009",  # Acquired stenosis of external ear canal secondary to surgery (disorder)
        "30493003",  # Intestinal infection caused by Proteus mirabilis (disorder)
        "30595008",  # Acute necrotizing cutaneous leishmaniasis (disorder)
        "30716008",  # Infection caused by Oesophagostomum columbianum (disorder)
        "30818003",  # Nonvenomous insect bite of perineum with infection (disorder)
        "31015008",  # Primary anal syphilis (disorder)
        "31137003",  # Early latent syphilis, positive serology, negative cerebrospinal fluid, less than 2 years after infection (disorder)
        "31201001",  # Knight's disease (disorder)
        "31487001",  # Disseminated idiopathic skeletal hyperostosis (disorder)
        "31604004",  # Blister of flank with infection (disorder)
        "31772005",  # Cicatricial skin lesions of pinta (disorder)
        "31999004",  # Chronic gonococcal endometritis (disorder)
        "32204007",  # Pulmonary actinobacillosis (disorder)
        "32286006",  # Pneumonia in Q fever (disorder)
        "32298001",  # Intestinal infection caused by Trichomonas vaginalis (disorder)
        "32527003",  # Staphylococcal enterocolitis (disorder)
        "32580004",  # Enterovirus enteritis (disorder)
        "32735002",  # Congenital syphilitic encephalitis (disorder)
        "33042004",  # Parasitic skin infestation (disorder)
        "33090006",  # Yabapox (disorder)
        "33155001",  # Nonvenomous insect bite of flank with infection (disorder)
        "33261009",  # Abscess of tonsil (disorder)
        "33655002",  # Chronic cystitis (disorder)
        "33666009",  # Black piedra (disorder)
        "33710003",  # Trichostrongyliasis (disorder)
        "33924006",  # Abscess of larynx (disorder)
        "33934002",  # Hemorrhagic otitis externa (disorder)
        "33937009",  # Lyme arthritis (disorder)
        "34020007",  # Pneumonia caused by Streptococcus (disorder)
        "34240005",  # Infection caused by Diphyllobothrium pacificum (disorder)
        "34242002",  # Syphilis in mother complicating pregnancy, childbirth AND/OR puerperium (disorder)
        "34730008",  # Primate erythroparvovirus 1 infection (disorder)
        "34786008",  # Candidal proctitis (disorder)
        "35037009",  # Primary atypical interstitial pneumonia (disorder)
        "35057008",  # Nonvenomous insect bite of penis with infection (disorder)
        "35089004",  # Urogenital infection caused by Trichomonas vaginalis (disorder)
        "35168006",  # Acute empyema of ethmoidal sinus (disorder)
        "35339003",  # Primary pneumonic plague (disorder)
        "35526001",  # Chronic gonococcal epididymo-orchitis (disorder)
        "35586003",  # White piedra (disorder)
        "35742006",  # Congenital syphilis (disorder)
        "35876006",  # Gonococcal infection of eye (disorder)
        "35923002",  # Chronic maxillary sinusitis (disorder)
        "35974005",  # Cryptococcal gastroenteritis (disorder)
        "36001007",  # Capillaria hepatica infection (disorder)
        "36038007",  # Goundou of yaws (disorder)
        "36139001",  # Chronic ulcerating interstitial cystitis (disorder)
        "36188001",  # Shigellosis (disorder)
        "36276008",  # Syphilitic retrobulbar neuritis (disorder)
        "36292003",  # Otitis externa caused by Herpes zoster (disorder)
        "36334003",  # Hymenolepis diminuta infection (disorder)
        "36529003",  # Intestinal infection caused by Morganella morganii (disorder)
        "36689008",  # Acute pyelonephritis (disorder)
        "36789003",  # Acute infective gastroenteritis (disorder)
        "36866003",  # Mucocutaneous-lymphangitic paracoccidioidomycosis (disorder)
        "37160003",  # Pintid (disorder)
        "37426002",  # Ulcerative laryngitis (disorder)
        "37430004",  # Syphilis of synovium (disorder)
        "37702000",  # Hereditary acrodermatitis enteropathica (disorder)
        "37722001",  # Ulceroglandular tularemia (disorder)
        "37754005",  # Asymptomatic neurosyphilis (disorder)
        "37776001",  # Abrasion and/or friction burn of perineum with infection (disorder)
        "37779008",  # Pyeloureteritis cystica (disorder)
        "37832003",  # Infection caused by Metagonimus yokogawai (disorder)
        "37887002",  # Nonvenomous insect bite of neck with infection (disorder)
        "38189003",  # Nonvenomous insect bite of ankle with infection (disorder)
        "38302000",  # Infection caused by Echinostoma lindoense (disorder)
        "38523005",  # Syphilitic parkinsonism (disorder)
        "38573008",  # Diffuse cutaneous leishmaniasis (disorder)
        "38699009",  # Pneumonia caused by Histoplasma capsulatum (disorder)
        "38700005",  # Madura foot caused by Actinomadura pelletieri (disorder)
        "38822007",  # Cystitis (disorder)
        "38961000",  # Chronic sphenoidal sinusitis (disorder)
        "38976008",  # Pneumonic plague (disorder)
        "39084003",  # Tinea cruris caused by Epidermophyton floccosum (disorder)
        "39085002",  # Secondary syphilis of mucous membrane (disorder)
        "39172002",  # Pneumonia caused by Proteus mirabilis (disorder)
        "39341005",  # Infectious colitis (disorder)
        "39439008",  # Wet form of cutaneous leishmaniasis (disorder)
        "39524009",  # Cercarial dermatitis of freshwater mammalian type (disorder)
        "39629007",  # Pyogenic granuloma of skin (disorder)
        "39747007",  # Enteritis necroticans (disorder)
        "39963006",  # Toddler diarrhea (disorder)
        "40055000",  # Chronic sinusitis (disorder)
        "40070004",  # Infection caused by Molluscum contagiosum (disorder)
        "40149008",  # Gonococcal keratitis (disorder)
        "40635000",  # Cystitis caused by Pseudomonas (disorder)
        "40786001",  # Aspiration pneumonia caused by regurgitated gastric secretions (disorder)
        "41080008",  # Cutaneous yaws (less than 5 years) (disorder)
        "41116009",  # Spirochetal infection (disorder)
        "41235001",  # Dry form of cutaneous leishmaniasis (disorder)
        "41381004",  # Pneumonia caused by Pseudomonas (disorder)
        "41545003",  # Whipple's disease (disorder)
        "41804000",  # Blister of axilla with infection (disorder)
        "41958008",  # Nonvenomous insect bite of back with infection (disorder)
        "42004004",  # Aspiration pneumonia caused by inhalation of vomitus (disorder)
        "42032008",  # Nonvenomous insect bite of abdominal wall with infection (disorder)
        "42089007",  # Nonvenomous insect bite of anus with infection (disorder)
        "42402006",  # Kartagener syndrome (disorder)
        "42640003",  # Suppurative appendicitis (disorder)
        "42680007",  # Endogenous lipoid pneumonia (disorder)
        "42746002",  # Gonorrhea of rectum (disorder)
        "42770003",  # Syphilis of tricuspid valve (disorder)
        "42849000",  # Trypanosomal chancre (disorder)
        "43017006",  # Infection caused by Gastrodiscoides hominis (disorder)
        "43021004",  # Periungual wart (disorder)
        "43114002",  # Chronic trigonitis (disorder)
        "43373009",  # Tinea barbae caused by Microsporum canis (disorder)
        "43581009",  # Tinea pedis caused by Epidermophyton (disorder)
        "43591003",  # Exanthematous infectious disease (disorder)
        "44274007",  # Lymphoid interstitial pneumonia (disorder)
        "44352004",  # Nonvenomous insect bite of hand with infection (disorder)
        "44412000",  # Chronic gonococcal urethritis (disorder)
        "44464006",  # Postpartum AND/OR puerperal erysipelas (disorder)
        "44549008",  # Aspiration pneumonia resulting from a procedure (disorder)
        "44568006",  # Syphilitic interstitial keratitis (disorder)
        "44653001",  # Erysipelas (disorder)
        "44697002",  # Melorheostosis (disorder)
        "44743006",  # Gonococcal infection of joint (disorder)
        "44917000",  # Hymenolepiasis (disorder)
        "45058001",  # Syphilis of pulmonary valve (disorder)
        "45072005",  # Acne necrotica miliaris (disorder)
        "45312009",  # Pneumonia in typhoid fever (disorder)
        "45377007",  # Acute gonococcal salpingitis (disorder)
        "45556008",  # Pulmonary tularemia (disorder)
        "45581006",  # Abrasion and/or friction burn of breast with infection (disorder)
        "45629003",  # Abscess of nasopharynx (disorder)
        "45666009",  # Blister of ear with infection (disorder)
        "45816000",  # Pyelonephritis (disorder)
        "45915002",  # Nonvenomous insect bite of upper arm with infection (disorder)
        "46080009",  # Abrasion and/or friction burn of abdominal wall with infection (disorder)
        "46235002",  # Early latent congenital syphilis, positive serology, negative spinal fluid (disorder)
        "46672003",  # Helicobacter-associated proctitis (disorder)
        "46699001",  # Bursitis caused by Neisseria gonorrhoeae (disorder)
        "46727001",  # Disorder of zinc metabolism (disorder)
        "46752004",  # Torus palatinus (disorder)
        "46799006",  # Dysenteric diarrhea (disorder)
        "46970008",  # Pneumonia caused by Mycoplasma pneumoniae (disorder)
        "46999003",  # Tinea barbae caused by Trichophyton (disorder)
        "47027001",  # Nonvenomous insect bite of scrotum with infection (disorder)
        "47277009",  # Izumi fever (disorder)
        "47306003",  # Lobomycosis (disorder)
        "47452006",  # Variola major (disorder)
        "47807003",  # Cutaneous habronemiasis (disorder)
        "47941007",  # Purulent enteritis (disorder)
        "48278001",  # Diphtheritic cystitis (disorder)
        "48631008",  # Pyonephrosis (disorder)
        "48971001",  # Tinea manus (disorder)
        "48982009",  # Bannwarth syndrome (disorder)
        "49442000",  # Initial lesion of yaws (disorder)
        "49454000",  # Abrasion and/or friction burn of flank with infection (disorder)
        "49524007",  # Abrasion and/or friction burn of groin with infection (disorder)
        "49649001",  # Infection caused by Acanthamoeba (disorder)
        "49882001",  # Viral exanthem (disorder)
        "49883006",  # Candidiasis of skin (disorder)
        "49894005",  # Trichomycosis axillaris (disorder)
        "49923008",  # Syphilis of tendon (disorder)
        "50124004",  # Nonvenomous insect bite of buttock with infection (disorder)
        "50242001",  # Tinea barbae caused by Microsporum gypseum (disorder)
        "50293005",  # Abrasion and/or friction burn of penis with infection (disorder)
        "50414000",  # Late cutaneous leishmaniasis (disorder)
        "50528008",  # Syphilis of bone (disorder)
        "50804000",  # Catarrhal pneumonia (disorder)
        "50970007",  # Acute gonorrhea of upper genitourinary tract (disorder)
        "51152000",  # Abscess of vocal cords (disorder)
        "51212009",  # Pitted keratolysis (disorder)
        "51254007",  # Paratyphoid C fever (disorder)
        "51423006",  # Modified smallpox (disorder)
        "51530003",  # Pneumonia caused by Escherichia coli (disorder)
        "51613008",  # Infection caused by Babesia divergens (disorder)
        "51817006",  # Infection caused by Acanthamoeba castellani (disorder)
        "51882009",  # Nonvenomous insect bite of interscapular region with infection (disorder)
        "51928006",  # General paresis - neurosyphilis (disorder)
        "51960003",  # Secondary syphilis of pharynx (disorder)
        "52123000",  # Emphysematous cystitis (disorder)
        "52281009",  # Meningoencephalitis caused by Naegleria (disorder)
        "52404001",  # Mastoiditis (disorder)
        "52414005",  # Secondary syphilis of vulva (disorder)
        "52705001",  # Nonvenomous insect bite of finger with infection (disorder)
        "52914002",  # Anal candidiasis (disorder)
        "52918004",  # Infection caused by Echinochasmus (disorder)
        "52979002",  # Capillaria infection (disorder)
        "53084003",  # Bacterial pneumonia (disorder)
        "53243005",  # Hyperchromic lesions of pinta (disorder)
        "53316003",  # Otomycosis (disorder)
        "53529004",  # Chronic gonococcal salpingitis (disorder)
        "53664003",  # Gonococcal spondylitis (disorder)
        "53734002",  # Hyperkeratosis of yaws (disorder)
        "53991003",  # Cystitis with bilharziasis (disorder)
        "54026009",  # Cystitis with Echinococcus infestation (disorder)
        "54043007",  # Cercarial dermatitis of freshwater avian type (disorder)
        "54069001",  # Congenital syphilitic mucous patches (disorder)
        "54266002",  # Infection caused by Fasciolopsis buski (disorder)
        "54274001",  # Primary syphilis of fingers (disorder)
        "54275000",  # Chronic metritis (disorder)
        "54385001",  # Exanthema subitum (disorder)
        "54825009",  # Acute gonorrhea of lower genitourinary tract (disorder)
        "55023005",  # Amebic nondysenteric colitis (disorder)
        "55184003",  # Infectious enteritis (disorder)
        "55260003",  # Calcaneal spur (finding)
        "55679008",  # Peribronchial pneumonia (disorder)
        "55768006",  # Syphilitic leukoderma (disorder)
        "55799007",  # Tinea capitis caused by Microsporum (disorder)
        "56140006",  # Progressive vaccina (disorder)
        "56200009",  # Infection caused by Trichuris globulosa (disorder)
        "56892000",  # Nonvenomous insect bite of chest wall with infection (disorder)
        "56978007",  # Generalized vaccinia (disorder)
        "57468008",  # Salpingo-oophoritis following molar AND/OR ectopic pregnancy (disorder)
        "57540006",  # Intestinal nematode infection (disorder)
        "57702005",  # Unresolved pneumonia (disorder)
        "57725006",  # Balantidiasis (disorder)
        "58056005",  # Syphilis of mitral valve (disorder)
        "58213005",  # Bacillary angiomatosis (disorder)
        "58227000",  # Secondary syphilis of viscera (disorder)
        "58392004",  # Congenital syphilitic osteochondritis (disorder)
        "58695007",  # Chromoblastomycosis caused by Fonsecaea compacta (disorder)
        "58763001",  # Acute empyema of nasal sinus (disorder)
        "59121004",  # Infection caused by Leishmania aethiopica (disorder)
        "59158004",  # Infection caused by Oesophagostomum stephanostomum (disorder)
        "59233003",  # Secondary syphilis of skin (disorder)
        "59307008",  # Syphilitic alopecia (disorder)
        "59443003",  # Nonvenomous insect bite of thigh with infection (disorder)
        "59475000",  # Pneumonia in pertussis (disorder)
        "59530001",  # Syphilis of kidney (disorder)
        "59707005",  # Abscess of pharynx (disorder)
        "59721007",  # Congenital syphilitic pemphigus (disorder)
        "59763006",  # Hyperphosphatasemia tarda (disorder)
        "59934002",  # Secondary syphilitic adenopathy (disorder)
        "60100005",  # Cyst of pharynx (disorder)
        "60130002",  # Chronic frontal sinusitis (disorder)
        "60298004",  # Nonvenomous insect bite of forearm with infection (disorder)
        "60335002",  # Gonococcal keratosis (disorder)
        "60416001",  # Edema of nasopharynx (disorder)
        "60485005",  # Pleurobronchopneumonia (disorder)
        "60506006",  # Blister of scalp with infection (disorder)
        "60528006",  # Secondary syphilis of mouth (disorder)
        "60570001",  # Infection caused by Trichuris trichiura (disorder)
        "60600009",  # Disorder of the larynx (disorder)
        "60867007",  # Klebsiella cystitis (disorder)
        "60893000",  # Chronic gonococcal prostatitis (disorder)
        "61048000",  # Gonococcal endocarditis (disorder)
        "61099007",  # Rubeola scarlatinosa (disorder)
        "61251002",  # Plantar hyperkeratosis of yaws (disorder)
        "61341000",  # Abrasion and/or friction burn of thigh with infection (disorder)
        "61383005",  # Abrasion and/or friction burn of ear with infection (disorder)
        "61500009",  # Granulomatous prostatitis (disorder)
        "61612001",  # Syphilitic aortic incompetence (disorder)
        "61624008",  # Chronic recurrent streptococcal erysipelas (disorder)
        "61884008",  # Achromobacter pneumonia (disorder)
        "62078006",  # Infection caused by Hypodereum (disorder)
        "62199007",  # Acute trigonitis (disorder)
        "62207008",  # Syphilitic ostial coronary disease (disorder)
        "62558006",  # Dyschromic skin lesions of pinta (disorder)
        "62861003",  # Secondary syphilis, relapse (untreated) (disorder)
        "63302006",  # Chronic pyelonephritis (disorder)
        "63440008",  # Verruca plantaris (disorder)
        "63605003",  # Infection caused by Naegleria gruberi (disorder)
        "63650001",  # Cholera (disorder)
        "63717008",  # Exostosis of orbit (disorder)
        "63751007",  # Secondary syphilis of bone (disorder)
        "63757006",  # Osler's node (disorder)
        "63848007",  # Abrasion and/or friction burn of wrist with infection (disorder)
        "64102008",  # Syphilitic gumma (disorder)
        "64479007",  # Pneumonia caused by Klebsiella pneumoniae (disorder)
        "64650008",  # Cutaneous nocardiosis (disorder)
        "64667001",  # Interstitial pneumonia (disorder)
        "64703005",  # Terminal bronchopneumonia (disorder)
        "64831005",  # Abrasion and/or friction burn of shoulder with infection (disorder)
        "64956000",  # Follicular cystitis (disorder)
        "65049003",  # Acute gonococcal seminal vesiculitis (disorder)
        "65141002",  # Foreign body pneumonia (disorder)
        "65295003",  # Acute gonococcal endometritis (disorder)
        "65332001",  # Diverticulitis of bladder (disorder)
        "65696007",  # Coccidioidal granuloma (disorder)
        "65984002",  # Salpingitis follicularis (disorder)
        "66160001",  # Cryptosporidial gastroenteritis (disorder)
        "66281009",  # Secondary syphilitic chorioretinitis (disorder)
        "66371007",  # Abrasion and/or friction burn of upper arm with infection (disorder)
        "66379009",  # Bacterial overgrowth syndrome (disorder)
        "66429007",  # Unresolved lobar pneumonia (disorder)
        "66578000",  # Abrasion and/or friction burn of cheek with infection (disorder)
        "66887000",  # Late syphilis, latent (positive serology, negative cephalospinal fluid 2 years after) (disorder)
        "67077008",  # Abrasion and/or friction burn of axilla with infection (disorder)
        "67125004",  # Latent syphilis with positive serology (disorder)
        "67167008",  # Urethrotrigonitis (disorder)
        "67391006",  # Syphilitic endocarditis (disorder)
        "67525007",  # Secondary pneumonic plague (disorder)
        "67624004",  # Abscess of Bartholin's gland (disorder)
        "67685000",  # Prostatocystitis (disorder)
        "67862004",  # Nonvenomous insect bite of elbow with infection (disorder)
        "67896006",  # Post-kala-azar dermal leishmaniasis (disorder)
        "67915005",  # Intestinal trichomoniasis (disorder)
        "67924001",  # Smallpox (disorder)
        "67974004",  # Nonvenomous insect bite of nose with infection (disorder)
        "68202005",  # Intermediate lesion of pinta (disorder)
        "68226007",  # Acute cystitis (disorder)
        "68389001",  # Mucopurulent vaginitis (disorder)
        "68409003",  # Organized pneumonia (disorder)
        "68421004",  # Epiphyseal arrest (disorder)
        "68566005",  # Urinary tract infectious disease (disorder)
        "68764005",  # Juvenile taboparesis (disorder)
        "68863007",  # Acute syphilitic meningitis (disorder)
        "69013005",  # Cutaneous draschiasis (disorder)
        "69163003",  # Taenia saginata infection (disorder)
        "69588003",  # Erythema chronica migrans (disorder)
        "69595007",  # Secondary syphilitic periostitis (disorder)
        "69810009",  # Acute hemorrhagic colitis caused by Escherichia coli (disorder)
        "70036007",  # Haemophilus influenzae pneumonia (disorder)
        "70143003",  # Tuberculoid leprosy (disorder)
        "70413005",  # Dermatophilosis caused by Dermatophilus congolensis (disorder)
        "70880006",  # Adenoviral enteritis (disorder)
        "71011005",  # Infestation caused by Phthirus pubis (disorder)
        "71085009",  # Paratyphoid B fever (disorder)
        "71104009",  # Achromic AND hyperchromic skin lesions of pinta (disorder)
        "71186008",  # Croup (disorder)
        "71418005",  # Cyst of nasopharynx (disorder)
        "71450000",  # Contact ulcer of vocal folds (disorder)
        "71572001",  # Infection caused by Enteromonas (disorder)
        "71577007",  # Deformity of eyelid due to infective dermatitis of eyelid (disorder)
        "71590000",  # Trichomonal prostatitis (disorder)
        "71926009",  # Infective pneumonia acquired prenatally (disorder)
        "72074009",  # Infection caused by Trichostrongylus orientalis (disorder)
        "72083004",  # Late syphilis (disorder)
        "72222004",  # Papillary cystitis (disorder)
        "72225002",  # Urethral stricture due to gonococcal infection (disorder)
        "72294005",  # Alastrim (disorder)
        "72463008",  # Chancriform syndrome (disorder)
        "72512006",  # Abrasion and/or friction burn of nose with infection (disorder)
        "72605008",  # Candidal vulvovaginitis (disorder)
        "72656004",  # Granulomatous pneumonia (disorder)
        "72815004",  # Cystitis glandularis (disorder)
        "72854003",  # Aspiration pneumonia due to near drowning (disorder)
        "72934000",  # Candidiasis of vagina (disorder)
        "73237007",  # Chronic ethmoidal sinusitis (disorder)
        "73281004",  # Chronic zinc deficiency (disorder)
        "73375005",  # Ulcer of larynx (disorder)
        "73532000",  # Cutaneous larva migrans caused by Ancylostoma caninum (disorder)
        "73594001",  # Pinta, late lesion (disorder)
        "73908004",  # Infection caused by Babesia microti (disorder)
        "74372003",  # Gonorrhea of pharynx (disorder)
        "74445007",  # Trigonitis (disorder)
        "74578003",  # Pyoderma gangrenosum (disorder)
        "74621002",  # Epidemic vomiting syndrome (disorder)
        "74724001",  # Smallpox without rash (disorder)
        "75073007",  # Malakoplakia of bladder (disorder)
        "75169006",  # Infection caused by Oesophagostomum apiostomum (disorder)
        "75299005",  # Spastic spinal syphilitic paralysis (disorder)
        "75375008",  # Bacterial enteritis (disorder)
        "75426006",  # Aspiration pneumonia caused by regurgitated food (disorder)
        "75498004",  # Acute bacterial sinusitis (disorder)
        "75530006",  # Streptococcal cervicitis (disorder)
        "75656006",  # Ulcerating cutaneous leishmaniasis (disorder)
        "75996005",  # Tinea pedis caused by Epidermophyton floccosum (disorder)
        "76069003",  # Disorder of bone (disorder)
        "76090006",  # Pittsburgh pneumonia (disorder)
        "76122006",  # Hyperkeratosis of pinta (disorder)
        "76272004",  # Syphilis (disorder)
        "76490001",  # Otomycosis externa caused by Fusarium (disorder)
        "76584003",  # Nonvenomous insect bite of hip with infection (disorder)
        "76623002",  # Paratyphoid A fever (disorder)
        "76653009",  # Acute empyema of maxillary sinus (disorder)
        "76802005",  # Chronic gonococcal cervicitis (disorder)
        "76813008",  # Pyogenic granuloma of oral mucosa (disorder)
        "77028001",  # Secondary syphilis of anus (disorder)
        "77412003",  # Hymenolepis nana infection (disorder)
        "77470003",  # Enteroviral exanthematous fever (disorder)
        "77746001",  # Leontiasis (disorder)
        "77798004",  # Systemic chromomycosis (disorder)
        "77863005",  # Lyme carditis (disorder)
        "77874001",  # Nonvenomous insect bite of shoulder with infection (disorder)
        "77939001",  # Syphilitic disseminated retinochoroiditis (disorder)
        "78128001",  # Helicobacter-associated colitis (disorder)
        "78223000",  # Coccidioidomycosis with erythema nodosum (disorder)
        "78360004",  # Abrasion and/or friction burn of ankle with infection (disorder)
        "78420004",  # Viral enteritis (disorder)
        "78487004",  # Granulomatous dermatophytosis (disorder)
        "78895009",  # Congenital pneumonia (disorder)
        "79077003",  # Abrasion and/or friction burn of back with infection (disorder)
        "79340003",  # Abrasion and/or friction burn of interscapular region with infection (disorder)
        "79411002",  # Acute prostatitis (disorder)
        "79739002",  # Infestation caused by Cheyletiella yasguri (disorder)
        "79949009",  # Primary cutaneous coccidioidomycosis (disorder)
        "79974007",  # Cat scratch disease (disorder)
        "80388004",  # Chronic gonorrhea of upper genitourinary tract (disorder)
        "80604007",  # Acute gonococcal bartholinitis (disorder)
        "80645004",  # Chronic mastoiditis (disorder)
        "80770009",  # Secondary syphilis of liver (disorder)
        "80890005",  # Cutaneous sporotrichosis (disorder)
        "81000006",  # Pediculosis capitis (disorder)
        "81020007",  # Cholera caused by Vibrio cholerae El Tor (disorder)
        "81125004",  # Acute hemorrhagic cystitis (disorder)
        "81339006",  # Secondary syphilis of tonsil (disorder)
        "81598001",  # Trichomonal vulvovaginitis (disorder)
        "82054006",  # Hyperostosis interna frontalis (finding)
        "82074004",  # Achromic skin lesions of pinta (disorder)
        "82214002",  # Trench fever (disorder)
        "82323002",  # Late congenital syphilis (2 years OR more) (disorder)
        "82355002",  # Syphilitic aortic stenosis (disorder)
        "82597009",  # Tinea capitis caused by Trichophyton tonsurans (disorder)
        "82930004",  # Intestinal infection caused by Pseudomonas (disorder)
        "82959004",  # Dementia paralytica juvenilis (disorder)
        "83052004",  # Cutaneous screw-worm myiasis (disorder)
        "83070001",  # Friction blister with infection (disorder)
        "83448006",  # Plantar papilloma of yaws (disorder)
        "83587009",  # Nonvenomous insect bite of cheek with infection (disorder)
        "83608006",  # Aspiration pneumonia caused by inhalation of milk (disorder)
        "83883001",  # Cardiovascular syphilis (disorder)
        "83937008",  # Infection caused by Babesia bovis (disorder)
        "84070009",  # Nonvenomous insect bite of axilla with infection (disorder)
        "84204007",  # Abrasion and/or friction burn of scrotum with infection (disorder)
        "84278006",  # Abrasion and/or friction burn of neck with infection (disorder)
        "84753008",  # Pneumonia in systemic mycosis (disorder)
        "84889008",  # Parapharyngeal abscess (disorder)
        "84980006",  # Cutaneous anthrax (disorder)
        "85495007",  # Acute pyonephrosis without renal medullary necrosis (disorder)
        "85857008",  # Secondary syphilis, relapse (treated) (disorder)
        "85904008",  # Paratyphoid fever (disorder)
        "86028001",  # Syphilis of liver (disorder)
        "86279000",  # Acute suppurative otitis media with spontaneous rupture of ear drum (disorder)
        "86333008",  # Subcutaneous phycomycosis caused by Basidiobolus (disorder)
        "86443005",  # Hutchinson's teeth (disorder)
        "86497005",  # Flat-type smallpox (disorder)
        "86615009",  # Epidemic diarrhea (disorder)
        "86981007",  # Infective otitis externa (disorder)
        "87282003",  # Intestinal parasitism (disorder)
        "87318008",  # Congenital syphilis with gumma (disorder)
        "87603003",  # Lymphocutaneous sporotrichosis (disorder)
        "87628006",  # Bacterial infectious disease (disorder)
        "87696004",  # Hemorrhagic cystitis (disorder)
        "88411000",  # Blister of anus with infection (disorder)
        "88558006",  # Cercarial dermatitis of sea water avian type (disorder)
        "88813005",  # Chronic gonococcal cystitis (disorder)
        "88850006",  # Chronic pansinusitis (disorder)
        "88943008",  # Syphilis of muscle (disorder)
        "88981003",  # Pyometra (disorder)
        "89478006",  # Parasitic melanoderma (disorder)
        "89933001",  # Protozoal intestinal disease (disorder)
        "90145001",  # Frontal bossing (disorder)
        "90271007",  # Intestinal infection caused by Arizona group (disorder)
        "90428001",  # Gonococcal pericarditis (disorder)
        "90679001",  # Mixed intestinal infection caused by Class Cestoda and/or Class Trematoda and/or Phylum Nemata (disorder)
        "90886006",  # Infestation of rectum caused by fly larvae (disorder)
        "91060006",  # Nonvenomous insect bite of wrist with infection (disorder)
        "91374005",  # Carbuncle of skin and/or subcutaneous tissue (disorder)
        "91554004",  # Condyloma latum (disorder)
        "95333004",  # Eosinophilic pustular folliculitis (disorder)
        "95340003",  # Nail bed infection (disorder)
        "95374004",  # Application site infection (disorder)
        "95381006",  # Injection site infection (disorder)
        "95547004",  # Ruptured suppurative appendicitis (disorder)
        "95881004",  # Mycetoma of foot (disorder)
        "95889002",  # Mycoplasmal pyelonephritis (disorder)
        "95896000",  # Protozoan infection (disorder)
        "102453009",  # Peritonsillar cellulitis (disorder)
        "105628008",  # Human typhoid AND/OR paratyphoid fever (disorder)
        "105645003",  # Disease caused by Acanthamoebidae (disorder)
        "105656003",  # Disease caused by Enteromonadidae (disorder)
        "105977003",  # Non-infectious pneumonia (disorder)
        "108365000",  # Infection of skin (disorder)
        "109362006",  # Cellulitis of parapharyngeal space (disorder)
        "109436001",  # Moon's molar teeth (disorder)
        "109802000",  # Abscess of uvula of palate (disorder)
        "109909008",  # Acute suppurative mastoiditis (disorder)
        "109910003",  # Chronic suppurative mastoiditis (disorder)
        "111003001",  # Sporotrichotic gumma (disorder)
        "111347003",  # Exostosis of jaw (disorder)
        "111409009",  # Ulcerative cystitis (disorder)
        "111538005",  # Acute mastoiditis with complication (disorder)
        "111705004",  # Blister of trunk with infection (disorder)
        "111707007",  # Abrasion and/or friction burn of chest wall with infection (disorder)
        "111711001",  # Blister of shoulder with infection (disorder)
        "111712008",  # Abrasion and/or friction burn of elbow with infection (disorder)
        "111796005",  # Madura foot caused by Streptomyces somaliensis (disorder)
        "111806005",  # Acute gonococcal prostatitis (disorder)
        "111807001",  # Gonococcal endophthalmia (disorder)
        "111839008",  # Intestinal infection caused by Escherichia coli (disorder)
        "111843007",  # Viral gastroenteritis (disorder)
        "111852003",  # Vaccinia (disorder)
        "111898002",  # Chronic mycotic otitis externa (disorder)
        "111899005",  # Cutaneous aspergillosis (disorder)
        "111900000",  # Pneumonia in aspergillosis (disorder)
        "111901001",  # Bronchocentric granulomatosis (disorder)
        "111911008",  # Chronic intestinal amebiasis without abscess (disorder)
        "111912001",  # Acute amebic dysentery without abscess (disorder)
        "111921000",  # Trichobilharzia cercarial dermatitis (disorder)
        "111923002",  # Gastrodiscoidiasis (disorder)
        "111939009",  # Dysentery (disorder)
        "120639003",  # Hantavirus pulmonary syndrome (disorder)
        "123587001",  # Acute bronchopneumonia (disorder)
        "123588006",  # Confluent bronchopneumonia with abscess formation (disorder)
        "123589003",  # Necrotizing bronchopneumonia (disorder)
        "123590007",  # Focal pneumonia (disorder)
        "123591006",  # Confluent pneumonia (disorder)
        "123611003",  # Subacute pyelonephritis (disorder)
        "123754002",  # Focal pyelonephritis (disorder)
        "123755001",  # Diffuse pyelonephritis (disorder)
        "123812008",  # Zinc excess (disorder)
        "128936008",  # Bacterial infection of skin (disorder)
        "128937004",  # Viral infection of skin (disorder)
        "128938009",  # Parasitic infection of skin (disorder)
        "128939001",  # Infection of subcutaneous tissue (disorder)
        "129451001",  # Respiratory bronchiolitis associated interstitial lung disease (disorder)
        "129452008",  # Nonspecific interstitial pneumonia (disorder)
        "129668006",  # Ulcer caused by Treponema vincentii (disorder)
        "185367005",  # Microsporosis nigra (disorder)
        "186090001",  # Typhoid and paratyphoid fevers (disorder)
        "186105003",  # Shigella boydii (group C) (disorder)
        "186116005",  # Acute amebic dysentery (disorder)
        "186150001",  # Enteritis caused by rotavirus (disorder)
        "186289000",  # Tinea nigra (disorder)
        "186357007",  # Streptococcal sore throat with scarlatina (disorder)
        "186394003",  # Actinomycosis of cecum (disorder)
        "186397005",  # Madura foot - actinomycotic (disorder)
        "186431008",  # Clostridioides difficile infection (disorder)
        "186525007",  # Herpes zoster with dermatitis of eyelid (disorder)
        "186535001",  # Eczema herpeticum (disorder)
        "186668002",  # Gammaherpesviral mononucleosis (disorder)
        "186807008",  # Asian desert cutaneous leishmaniasis (disorder)
        "186825001",  # Systemic bartonellosis (disorder)
        "186826000",  # Cutaneous and mucocutaneous bartonellosis (disorder)
        "186833000",  # Early congenital syphilis - latent (disorder)
        "186842007",  # Late congenital syphilitic oculopathy (disorder)
        "186846005",  # Early symptomatic syphilis (disorder)
        "186847001",  # Primary genital syphilis (disorder)
        "186850003",  # Secondary syphilis of skin and mucous membrane (disorder)
        "186854007",  # Uveitis due to secondary syphilis (disorder)
        "186861006",  # Secondary syphilis relapse (disorder)
        "186863009",  # Acute secondary syphilitic meningitis (disorder)
        "186867005",  # Latent early syphilis (disorder)
        "186868000",  # Serological relapse after treatment of latent early syphilis (disorder)
        "186875004",  # Syphilitic endocarditis of mitral valve (disorder)
        "186877007",  # Syphilitic endocarditis of tricuspid valve (disorder)
        "186878002",  # Syphilitic endocarditis of pulmonary valve (disorder)
        "186893003",  # Rupture of syphilitic cerebral aneurysm (disorder)
        "186903006",  # Late latent syphilis (disorder)
        "186915005",  # Chronic gonorrhea lower genitourinary tract (disorder)
        "186931002",  # Gonococcal anal infection (disorder)
        "186939000",  # Gonococcal peritonitis (disorder)
        "186946009",  # Lymphogranuloma venereum (disorder)
        "186975003",  # Mixed pinta lesions (disorder)
        "187052004",  # Pulmonary African histoplasmosis (disorder)
        "187065008",  # Cutaneous blastomycosis (disorder)
        "187079000",  # Chromoblastomycosis (disorder)
        "187081003",  # Subcutaneous pheohyphomycotic abscess and cyst (disorder)
        "187089001",  # Actinomycotic mycetoma (disorder)
        "187115002",  # Cutaneous schistosomiasis (disorder)
        "187171000",  # Ascariasis with intestinal complications (disorder)
        "187174008",  # Intestinal strongyloidiasis (disorder)
        "187187002",  # Intestinal angiostrongyliasis (disorder)
        "188463006",  # Chlamydial pelvic inflammatory disease (disorder)
        "191727003",  # Post measles pneumonia (disorder)
        "192647003",  # Secondary syphilitic meningitis (disorder)
        "193786000",  # Keratitis caused by syphilis (disorder)
        "193922006",  # Parasitic eyelid infestation (disorder)
        "194202008",  # Infective otitis externa due to erysipelas (disorder)
        "194203003",  # Infective otitis externa due to impetigo (disorder)
        "194208007",  # Chronic otitis externa due to moniliasis (disorder)
        "194281003",  # Acute suppurative otitis media (disorder)
        "194282005",  # Acute suppurative otitis media due to another disease (disorder)
        "194907008",  # Acute syphilitic pericarditis (disorder)
        "194910001",  # Acute gonococcal pericarditis (disorder)
        "194947001",  # Acute myocarditis - syphilitic (disorder)
        "195720006",  # Acute streptococcal bronchitis (disorder)
        "195721005",  # Acute haemophilus influenzae bronchitis (disorder)
        "195756009",  # Woakes' ethmoiditis (disorder)
        "195798007",  # Chronic adenotonsillitis (disorder)
        "195834004",  # Pharynx or nasopharynx edema (disorder)
        "195866009",  # Laryngeal pachydermia (disorder)
        "195878008",  # Pneumonia and influenza (disorder)
        "195886008",  # Group B streptococcal pneumonia (disorder)
        "195888009",  # Proteus pneumonia (disorder)
        "195889001",  # Legionella pneumonia (disorder)
        "195896004",  # Pneumonia caused by pleuropneumonia-like organism (disorder)
        "195902009",  # Anthrax pneumonia (disorder)
        "195904005",  # Pneumonia with coccidioidomycosis (disorder)
        "195908008",  # Actinomycotic pneumonia (disorder)
        "195909000",  # Nocardial pneumonia (disorder)
        "195920000",  # Influenza with pneumonia, influenza virus identified (disorder)
        "197143009",  # Megacolon in Chagas' disease (disorder)
        "197305002",  # Syphilitic portal cirrhosis (disorder)
        "197347003",  # Hepatitis in late syphilis (disorder)
        "197348008",  # Hepatitis in secondary syphilis (disorder)
        "197757004",  # Late syphilis of kidney (disorder)
        "197760006",  # Chronic pyelonephritis without medullary necrosis (disorder)
        "197761005",  # Chronic pyelonephritis with medullary necrosis (disorder)
        "197763008",  # Chronic pyonephrosis (disorder)
        "197764002",  # Non-obstructive reflux-associated chronic pyelonephritis (disorder)
        "197768004",  # Acute pyelonephritis without medullary necrosis (disorder)
        "197769007",  # Acute pyelonephritis with medullary necrosis (disorder)
        "197770008",  # Acute pyonephrosis (disorder)
        "197782004",  # Pyelonephritis associated with another disorder (disorder)
        "197834003",  # Chronic interstitial cystitis (disorder)
        "197837005",  # Submucous cystitis (disorder)
        "197845000",  # Cystitis associated with another disorder (disorder)
        "197848003",  # Gonococcal cystitis (disorder)
        "197849006",  # Cystitis caused by Candida (disorder)
        "197850006",  # Trichomonal cystitis (disorder)
        "197853008",  # Recurrent cystitis (disorder)
        "197963001",  # Prostatitis associated with another disorder (disorder)
        "197964007",  # Actinomycotic prostatitis (disorder)
        "197965008",  # Blastomycotic prostatitis (disorder)
        "197966009",  # Syphilitic prostatitis (disorder)
        "197967000",  # Gonococcal prostatitis (disorder)
        "197968005",  # Prostatitis caused by Candida (disorder)
        "198130006",  # Female pelvic inflammatory disease (disorder)
        "198141008",  # Chronic perioophoritis (disorder)
        "198159006",  # Chronic parametritis and pelvic cellulitis (disorder)
        "198175009",  # Female syphilitic pelvic inflammatory disease (disorder)
        "198186006",  # Subacute pyometra (disorder)
        "198189004",  # Chronic pyometra (disorder)
        "198235009",  # Carbuncle of vagina (disorder)
        "198242009",  # Female gonococcal pelvic inflammatory disease (disorder)
        "199154009",  # Maternal syphilis during pregnancy, childbirth and the puerperium (disorder)
        "199156006",  # Maternal syphilis during pregnancy - baby delivered (disorder)
        "199157002",  # Maternal syphilis in the puerperium - baby delivered during current episode of care (disorder)
        "199158007",  # Maternal syphilis during pregnancy - baby not yet delivered (disorder)
        "199159004",  # Maternal syphilis in the puerperium - baby delivered during previous episode of care (disorder)
        "199161008",  # Maternal gonorrhea during pregnancy, childbirth and the puerperium (disorder)
        "199163006",  # Maternal gonorrhea during pregnancy - baby delivered (disorder)
        "199164000",  # Maternal gonorrhea in the puerperium - baby delivered during current episode of care (disorder)
        "199165004",  # Maternal gonorrhea during pregnancy - baby not yet delivered (disorder)
        "199166003",  # Maternal gonorrhea in the puerperium - baby delivered during previous episode of care (disorder)
        "199205008",  # Infections of bladder in pregnancy (disorder)
        "200635001",  # Finger web space infection (disorder)
        "200649005",  # Cellulitis and abscess of submandibular region (disorder)
        "200721005",  # Pyogenic progressive granuloma (disorder)
        "200722003",  # Pyogenic granuloma (disorder)
        "201060008",  # Disorder of upper respiratory system (disorder)
        "201517005",  # Seronegative arthritis secondary to Whipple's disease (disorder)
        "202756004",  # Infection of intervertebral disc - pyogenic (disorder)
        "202883008",  # Posterior calcaneal exostosis (disorder)
        "202933002",  # Bursitis caused by Treponema pallidum (disorder)
        "203157008",  # Acute osteomyelitis of cervical spine (disorder)
        "203158003",  # Acute osteomyelitis of thoracic spine (disorder)
        "203159006",  # Acute osteomyelitis of lumbar spine (disorder)
        "203193005",  # Chronic osteomyelitis of cervical spine (disorder)
        "203194004",  # Chronic osteomyelitis of thoracic spine (disorder)
        "203195003",  # Chronic osteomyelitis of lumbar spine (disorder)
        "203198001",  # Brodie's abscess of cervical spine (disorder)
        "203199009",  # Brodie's abscess of thoracic spine (disorder)
        "203200007",  # Brodie's abscess of lumbar spine (disorder)
        "203241002",  # Osteomyelitis of vertebra (disorder)
        "203284001",  # Osteopathy from poliomyelitis (disorder)
        "203286004",  # Poliomyelitis osteopathy of the shoulder region (disorder)
        "203287008",  # Poliomyelitis osteopathy of the upper arm (disorder)
        "203288003",  # Poliomyelitis osteopathy of the forearm (disorder)
        "203289006",  # Poliomyelitis osteopathy of the hand (disorder)
        "203290002",  # Poliomyelitis osteopathy of the pelvic region and/or thigh (disorder)
        "203291003",  # Poliomyelitis osteopathy of the lower leg (disorder)
        "203292005",  # Poliomyelitis osteopathy of the ankle and/or foot (disorder)
        "203294006",  # Poliomyelitis osteopathy of multiple sites (disorder)
        "203296008",  # Subacute osteomyelitis of cervical spine (disorder)
        "203297004",  # Subacute osteomyelitis of thoracic spine (disorder)
        "203298009",  # Subacute osteomyelitis of lumbar spine (disorder)
        "203357004",  # Hypertrophic osteoarthropathy (disorder)
        "203514008",  # Hypertrophy of bone (disorder)
        "203515009",  # Compensatory hypertrophy of bone (disorder)
        "203516005",  # Idiopathic hypertrophy of bone (disorder)
        "203517001",  # Complete epiphyseal arrest (disorder)
        "203518006",  # Partial epiphyseal arrest (disorder)
        "203522001",  # Osteolysis (disorder)
        "204919003",  # Erythematous pinta plaque (disorder)
        "206283000",  # Congenital staphylococcal pneumonia (disorder)
        "206284006",  # Congenital group A hemolytic streptococcal pneumonia (disorder)
        "206285007",  # Congenital group B hemolytic streptococcal pneumonia (disorder)
        "206286008",  # Congenital Escherichia coli pneumonia (disorder)
        "206287004",  # Congenital pseudomonal pneumonia (disorder)
        "206289001",  # Congenital chlamydial pneumonia (disorder)
        "206356004",  # Neonatal candidiasis of perineum (disorder)
        "206358003",  # Neonatal candidiasis of intestine (disorder)
        "206445003",  # Malarial pigment deposition (disorder)
        "206554002",  # Neonatal skin infection (disorder)
        "206598002",  # Relapsing fever of Southwest United States AND/OR Mexico (disorder)
        "209516003",  # Relapsing fever of North Africa (disorder)
        "210992005",  # Abrasion of face, infected (disorder)
        "210993000",  # Abrasion of neck, infected (disorder)
        "210994006",  # Abrasion of scalp, infected (disorder)
        "211012003",  # Insect bite, nonvenomous, of head, infected (disorder)
        "211053006",  # Abrasion and/or friction burn of trunk, infected (disorder)
        "211054000",  # Abrasion of interscapular region, infected (disorder)
        "211055004",  # Abrasion of chest wall, infected (disorder)
        "211056003",  # Abrasion of breast, infected (disorder)
        "211057007",  # Abrasion of abdominal wall, infected (disorder)
        "211058002",  # Abrasion of back, infected (disorder)
        "211059005",  # Abrasion of buttock, infected (disorder)
        "211060000",  