"""
Implements the base class for the tables.
"""
import os
from abc import ABCMeta, abstractmethod

import docutils.core
import matplotlib as mpl
import matplotlib.pyplot as plt
import numpy as np
import pandas as pd

from fitbenchmarking.utils.misc import get_js

FORMAT_DESCRIPTION = \
    {'abs': 'Absolute values are displayed in the table.',
     'rel': 'Relative values are displayed in the table.',
     'both': 'Absolute and relative values are displayed in '
             'the table in the format ``abs (rel)``'}


class Table:
    """
    Base class for the FitBenchmarking HTML and text output tables.

    When inheriting from this, it may be useful to override the following
    functions as required:

    - get_value
    - display_str
    - get_error_str
    - get_link_str
    """
    __metaclass__ = ABCMeta

    def __init__(self, results, best_results, options, group_dir,
                 pp_locations, table_name):
        """
        Initialise the class.

        :param results: Results grouped by row and category (for colouring)
        :type results:
            dict[str, dict[str, list[utils.fitbm_result.FittingResult]]]
        :param best_results: The best results from each row/category
        :type best_results:
            dict[str, dict[str, utils.fitbm_result.FittingResult]]
        :param options: Options used in fitting
        :type options: utils.options.Options
        :param group_dir: path to the directory where group results should be
                          stored
        :type group_dir: str
        :param pp_locations: tuple containing the locations of the
                             performance profiles (acc then runtime)
        :type pp_locations: tuple(str,str)
        :param table_name: Name of the table
        :type table_name: str
        """
        # Flatten to reduce the necessity on having problems as rows.
        self.results = results
        self.best_results = best_results
        self.options = options
        self.group_dir = group_dir
        self.pp_locations = pp_locations
        self.table_name = table_name
        self.name = None

        self.output_string_type = {"abs": '{:.4g}',
                                   "rel": '{:.4g}',
                                   "both": '{0:.4g} ({1:.4g})'}
        self.colour_template = 'background-color: {0}'

        self.has_pp = False
        self.pp_location = ''
        self._table_title = None
        self._file_path = None

        self.cbar_title = "No colour bar description given"
        self.cbar_left_label = "Best (1)"
        self.cbar_right_label = "Worst (>{})".format(self.options.colour_ulim)

        # Set up results as needed
        self.sorted_results = {}
        self.create_results_dict()

    @abstractmethod
    def get_value(self, result):
        """
        Gets the main value to be reported in the tables for a given result

        If more than one value is returned please note that the first value
        will be used in the default colour handling.

        :param result: The result to generate the values for.
        :type result: FittingResult

        :return: The value to convert to a string for the tables
        :rtype: tuple(float)
        """
        raise NotImplementedError

    def display_str(self, value):
        """
        Converts a value generated by
        :meth:`~fitbenchmarking.results_processing.base_table.Table.get_value()`
        into a string respresentation to be used in the tables.
        Base class implementation takes
        the relative and absolute values and uses ``self.output_string_type``
        as a template for the string format. This can be overridden to
        adequately display the results.

        :param value: Relative and absolute values
        :type value: tuple

        :return: string representation of the value for display in the table.
        :rtype: str
        """
        rel_value, abs_value = value
        comp_mode = self.options.comparison_mode
        result_template = self.output_string_type[self.options.comparison_mode]
        if comp_mode == "abs":
            return result_template.format(abs_value)
        if comp_mode == "rel":
            return result_template.format(rel_value)
        # comp_mode == "both"
        return result_template.format(abs_value, rel_value)

    def get_link_str(self, result):
        """
        Get the link as a string for the result.
        This can be overridden if tables require different links.

        :param result: The result to get the link for
        :type result: FittingResult

        :return: The link to go to when the cell is selected
        :rtype: string
        """
        return os.path.relpath(path=result.fitting_report_link,
                               start=self.group_dir)

    @staticmethod
    def get_error_str(result, error_template='[{}]'):
        """
        Get the error string for a result based on error_template
        This can be overridden if tables require different error formatting.

        :param result: The result to get the error string for
        :type result: FittingResult

        :return: A string representation of the error
        :rtype: str
        """
        error_code = result.error_flag
        if error_code == 0:
            return ''

        return error_template.format(error_code)

    def create_results_dict(self):
        """
        Generate a dictionary of results lists with rows and columns as the key
        and list elements respectively.
        This is used to create HTML and txt tables.
        This is stored in self.sorted_results
        """
        self.sorted_results = {k: [r for cat in row.values() for r in cat]
                               for k, row in self.results.items()}

    def get_str_dict(self, html=False):
        """
        Create a dictionary with the table values as strings for display.

        :return: The dictionary of strings for the table
        :rtype: dict[list[str]]
        """
        str_dict = {}
        for k, results in self.sorted_results.items():
            str_dict[k] = [self.get_str_result(r, html)
                           for r in results]
        return str_dict

    def get_colour_df(self, like_df=None):
        """
        Generate a dataframe of colours to add to the html rendering.

        If like_df is passed this will use the column and row indexes of that
        dataframe.

        :param like_df: The dataframe to copy headings from. Defaults to None.
        :type like_df: pandas.DataFrame

        :return: A dataframe with colourings as strings
        :rtype: pandas.DataFrame
        """
        col_dict = {}
        for k, results in self.sorted_results.items():
            col_dict[k] = self.get_colours_for_row(results)

        table = pd.DataFrame.from_dict(col_dict, orient='index')

        if like_df is None:
            row = next(iter(self.sorted_results.values()))
            minimizers_list = [(r.software, r.minimizer) for r in row]
            table.columns = pd.MultiIndex.from_tuples(minimizers_list)
        else:
            table.columns = like_df.columns
            table.index = like_df.index
        return table

    def get_str_result(self, result, html=False):
        """
        Given a single result, generate the string to display in this table.
        The html flag can be used to switch between a plain text and html
        format.

        This is intended to be easily extensible by overriding the following
        functions:

        - get_value
        - display_str
        - get_error_str
        - get_link_str

        If you find yourself overriding this, please consider if changes could
        be made to allow future tables to benefit.

        :param result: The result to generate a string for
        :type result: fitbenchmarking.utils.ftibm_result.FittingResult
        :param html: Flag to control whether to generate a html string or plain
                     text. Defaults to False.
        :type html: bool

        :return: The string representation.
        :rtype: str
        """
        if html:
            val = self.get_value(result)
            val_str = self.display_str(val)
            val_str += self.get_error_str(result,
                                          error_template="<sup>{}</sup>")
            val_str = f'<a href="{self.get_link_str(result)}">{val_str}</a>'
        else:
            val_str = self.display_str(self.get_value(result))
            val_str += self.get_error_str(result, error_template='[{}]')
        return val_str

    def get_colours_for_row(self, results):
        """
        Get the colours as strings for the given results in the table.
        The base class implementation, for example,
        uses the first value from self.get_value and
        ``colour_map``, ``colour_ulim`` and ``cmap_range`` within
        :class:`~fitbenchmarking.utils.options.Options`.

        :param result: Results to get the colours for.
        :type result: list[fitbenchmarking.utils.fitbm_result.FittingResult]

        :return: The colour to use for each cell in the list
        :rtype: list[str]
        """
        values = [self.get_value(r)[0] for r in results]

        cmap_name = self.options.colour_map
        cmap = plt.get_cmap(cmap_name)
        cmap_ulim = self.options.colour_ulim
        cmap_range = self.options.cmap_range
        log_ulim = np.log10(cmap_ulim)  # colour map used with log spacing

        col_strs = ["background-colour: #ffffff" for _ in results]

        colours = self.vals_to_colour(values, cmap, cmap_range, log_ulim)
        for i, c in enumerate(colours):
            try:
                col_strs[i] = self.colour_template.format(c)
            except IndexError:
                col_strs[i] = self.colour_template.format(*c)

        return col_strs

    def create_pandas_data_frame(self, html=False):
        """
        Creates a pandas data frame of results

        :param html: Whether to make the dataframe for html or plain text
        :type html: bool. defaults to False

        :return: DataFrame with string representations of results
        :rtype: pandas.DataFrame
        """
        str_results = self.get_str_dict(html)
        row = next(iter(self.sorted_results.values()))
        minimizers_list = [(r.software, r.modified_minimizer_name(False))
                           for r in row]
        columns = pd.MultiIndex.from_tuples(minimizers_list)
        table = pd.DataFrame.from_dict(str_results,
                                       orient='index',
                                       columns=columns)
        return table

    def to_html(self):
        """
        Generate a html version of the table.

        :return: HTML table output
        :rtype: str
        """
        table = self.create_pandas_data_frame(html=True)

        # Format the table headers
        cost_func_template = '<a class="cost_function_header" ' \
                             'href=https://fitbenchmarking.readthedocs.io/' \
                             'en/latest/users/options/fitting_option.html' \
                             '#cost-function-cost-func-type ' \
                             'target="_blank">{0}</a>'
        software_template = '<a class="software_header" ' \
                            'href="https://fitbenchmarking.readthedocs.io/' \
                            'en/latest/users/options/minimizer_option.html' \
                            '#{0}" target="_blank">{0}</a>'
        minimizer_template = '<a class="minimizer_header" col={0} ' \
                             'title="{1}"' \
                             'href="https://fitbenchmarking.readthedocs.io/' \
                             'en/latest/users/options/minimizer_option.html' \
                             '#{2}" target="_blank">{3}</a>'

        row = next(iter(self.sorted_results.values()))
        minimizers_list = [
            (cost_func_template.format(result.costfun_tag),
             software_template.format(result.software.replace('_', '-')),
             minimizer_template.format(i, result.algorithm_type,
                                       result.software.replace('_', '-'),
                                       result.modified_minimizer_name()))
            for i, result in enumerate(row)]
        columns = pd.MultiIndex.from_tuples(minimizers_list)
        table.columns = columns

        # Format the row labels
        index = []
        for b, i in zip(self.best_results.values(), table.index):
            b = next(iter(b.values()))
            rel_path = os.path.relpath(
                path=b.problem_summary_page_link,
                start=self.group_dir)
            index.append('<a class="problem_header" href="{0}">{1}</a>'
                         .format(rel_path, i))
        table.index = index

        # Get columns where cost function changes
        column_dividers = [table.columns[0]]
        for column in table.columns[1:]:
            if column[0] != column_dividers[-1][0]:
                column_dividers.append(column)
        column_dividers = column_dividers[1:]

        # Set the cell colours and increase bars between cost functions
        table_style = table.style\
            .apply(lambda df: self.get_colour_df(like_df=df), axis=None)\
            .set_table_styles(table_styles={
                k: [{'selector': 'td',
                     'props': [('border-left-width', '3px')]},
                    {'selector': 'th',
                     'props': [('border-left-width', '3px')]}]
                for k in column_dividers})

        return table_style.to_html(table_uuid='table')

    def to_txt(self):
        """
        Generate a plain text version of the table

        :return: Plain text table output
        :rtype: str
        """
        table = self.create_pandas_data_frame(html=False)
        return table.to_string()

    def get_description(self, html_description):
        """
        Generates table description from class docstrings and converts them
        into html

        :param html_description: Dictionary containing table descriptions
        :type html_description: dict

        :return: Dictionary containing table descriptions
        :rtype: dict
        """
        FORMAT_DESCRIPTION[self.name] = self.__doc__
        for name in [self.name, self.options.comparison_mode]:
            descrip = FORMAT_DESCRIPTION[name]
            descrip = descrip.replace(':ref:', '')
            js = get_js(self.options, self.group_dir)
            docsettings = {
                'math_output': 'MathJax '+js['mathjax']
            }
            description_page = docutils.core.publish_parts(
                descrip,
                writer_name='html',
                settings_overrides=docsettings)
            html_description[name] = description_page['body']
            html_description[name] = html_description[name].replace(
                '<blockquote>\n', '')
        return html_description

    @property
    def table_title(self):
        """
        Getter function for table name if self._table_title is None

        :return: name of table
        :rtype: str
        """
        if self._table_title is None:
            self._table_title = "FitBenchmarking: {0} table".format(self.name)
        return self._table_title

    @table_title.setter
    def table_title(self, value):
        """
        Setting function to set the name of the table

        :param value: name of table
        :type value: str
        """
        self._table_title = value

    @property
    def file_path(self):
        """
        Getter function for the path to the table

        :return: path to table
        :rtype: str
        """
        if self._file_path is None:
            self._file_path = os.path.join(self.group_dir, self.table_name)
        return self._file_path

    @file_path.setter
    def file_path(self, value):
        """
        Setting function to set the path to the table

        :param value: path to table
        :type value: str
        """
        self._file_path = value

    @staticmethod
    def vals_to_colour(vals, cmap, cmap_range, log_ulim):
        """
        Converts an array of values to a list of hexadecimal colour
        strings using logarithmic sampling from a matplotlib colourmap
        according to relative value.

        :param vals: values in the range [0, 1] to convert to colour strings
        :type vals: list[float]
        :param cmap: matplotlib colourmap
        :type cmap: matplotlib colourmap object
        :param cmap_range: values in range [0, 1] for colourmap cropping
        :type cmap_range: list[float], 2 elements
        :param log_ulim: log10 of worst shading cutoff value
        :type log_ulim: float

        :return: colours as hex strings for each input value
        :rtype: list[str]
        """
        log_vals = np.log10(vals)
        log_llim = min(log_vals)
        norm_vals = (log_vals - log_llim) /\
            (log_ulim - log_llim)
        norm_vals[norm_vals > 1] = 1  # applying upper cutoff
        norm_vals[np.isnan(norm_vals)] = 1  # deal with nans
        # trimming colour map according to default/user input
        norm_vals = cmap_range[0] + \
            norm_vals*(cmap_range[1] - cmap_range[0])
        rgba = cmap(norm_vals)
        hex_strs = [mpl.colors.rgb2hex(colour) for colour in rgba]

        return hex_strs

    def save_colourbar(self, fig_dir, n_divs=100, sz_in=(3, 0.8)) -> str:
        """
        Generates a png of a labelled colourbar using matplotlib.

        :param fig_dir: path to figures directory
        :type fig_dir: str
        :param n_divs: number of divisions of shading in colourbar
        :type n_divs: int
        :param sz_in: dimensions of png in inches [width, height]
        :type sz_in: list[float] - 2 elements

        :return: The relative path to the colourbar image.
        :rtype: str
        """
        fig_path = os.path.join(fig_dir, "{0}_cbar.png".format(self.name))

        figh = 0.77
        fig, ax = plt.subplots(nrows=1, figsize=(6.4, figh))
        fig.subplots_adjust(top=1 - 0.35 / figh, bottom=0.15 / figh,
                            left=0.3, right=0.7, hspace=1)

        cmap_range = self.options.cmap_range
        gradient = np.linspace(cmap_range[0], cmap_range[1], n_divs)
        gradient = np.vstack((gradient, gradient))

        ax.imshow(gradient, aspect='auto',
                  cmap=plt.get_cmap(self.options.colour_map), vmin=0, vmax=1)

        ax.text(-0.02, 0.5, self.cbar_left_label,
                va='center', ha='right', fontsize=6,
                transform=ax.transAxes)
        ax.text(1.02, 0.5, self.cbar_right_label,
                va='center', ha='left', fontsize=6,
                transform=ax.transAxes)
        ax.set_title(self.cbar_title, fontsize=6)
        ax.set_axis_off()
        fig.set_size_inches(sz_in[0], sz_in[1])

        plt.savefig(fig_path, dpi=150)

        return os.path.relpath(fig_path, self.group_dir)

    def problem_dropdown_html(self) -> str:
        """
        Generates the HTML for a dropdown checklist of problem sets.

        :return: HTML for a dropdown checklist of problem sets.
        :rtype: str
        """
        items = [f'        <li><label class="noselect"><input '
                 f'type="checkbox" checked=true '
                 f'onclick="toggle_problem(\'{problem_name}\')"/> '
                 f'{problem_name}</label></li>'
                 for problem_name in self.sorted_results.keys()]

        return self._dropdown_html("problem_dropdown", "Select Problems",
                                   items)

    def minimizer_dropdown_html(self) -> str:
        """
        Generates the HTML for a dropdown checklist of minimizers.

        :return: HTML for a dropdown checklist of minimizers.
        :rtype: str
        """
        minimizers = [(result.software.replace('_', '-'), result.minimizer)
                      for result in next(iter(self.sorted_results.values()))]
        # Remove duplicates
        minimizers = list(dict.fromkeys(minimizers))

        items = [f'        <li><label class="noselect"><input '
                 f'type="checkbox" checked=true '
                 f'onclick="toggle_minimizer(\'{software}\', '
                 f'\'{minimizer}\')"/> {minimizer}</label></li>'
                 for software, minimizer in minimizers]

        return self._dropdown_html("minimizer_dropdown",
                                   "Select Minimizers", items)

    @staticmethod
    def _dropdown_html(list_id: str, selector_text: str,
                       checklist: list) -> str:
        """
        Generates the HTML for a dropdown checklist. The list of items
        must be provided to this function.

        :param list_id: The ID to give the dropdown button.
        :type list_id: str
        :param selector_text: The text to display on the dropdown button.
        :type selector_text: str
        :param checklist: A list of HTML checkboxes to include in the
        dropdown.
        :type checklist: list
        :return: HTML for a dropdown checklist.
        :rtype: str
        """
        checklist_str = "\n".join(checklist)
        html = f'<div id="{list_id}" class="dropdown-check-list" ' \
               f'tabindex="100">\n' \
               f'    <span class="anchor" onclick="show_dropdown' \
               f'(\'{list_id}\')">{selector_text}</span>\n' \
               '    <ul class="items">\n' \
               f'{checklist_str}\n' \
               '    </ul>\n' \
               '</div>'
        return html
