"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderUnless = exports.helperRemovalPolicy = exports.renderCredentials = exports.defaultDeletionProtection = exports.engineDescription = exports.setupS3ImportExport = exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("../database-secret");
const props_1 = require("../props");
/**
 * The default set of characters we exclude from generated passwords for database users.
 * It's a combination of characters that have a tendency to cause problems in shell scripts,
 * some engine-specific characters (for example, Oracle doesn't like '@' in its passwords),
 * and some that trip up other services, like DMS.
 *
 * This constant is private to the RDS module.
 */
exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = " %+~`#$&*()|[]{}:;<>?!'/@\"\\";
/**
 * Validates the S3 import/export props and returns the import/export roles, if any.
 * If `combineRoles` is true, will reuse the import role for export (or vice versa) if possible.
 *
 * Notably, `combineRoles` is (by default) set to true for instances, but false for clusters.
 * This is because the `combineRoles` functionality is most applicable to instances and didn't exist
 * for the initial clusters implementation. To maintain backwards compatibility, it is set to false for clusters.
 */
function setupS3ImportExport(scope, props, combineRoles) {
    let s3ImportRole = props.s3ImportRole;
    let s3ExportRole = props.s3ExportRole;
    if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
        if (props.s3ImportRole) {
            throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
        }
        s3ImportRole = (combineRoles && s3ExportRole) ? s3ExportRole : new iam.Role(scope, 'S3ImportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ImportBuckets) {
            bucket.grantRead(s3ImportRole);
        }
    }
    if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
        if (props.s3ExportRole) {
            throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
        }
        s3ExportRole = (combineRoles && s3ImportRole) ? s3ImportRole : new iam.Role(scope, 'S3ExportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ExportBuckets) {
            bucket.grantReadWrite(s3ExportRole);
        }
    }
    return { s3ImportRole, s3ExportRole };
}
exports.setupS3ImportExport = setupS3ImportExport;
function engineDescription(engine) {
    var _a;
    return engine.engineType + (((_a = engine.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) ? `-${engine.engineVersion.fullVersion}` : '');
}
exports.engineDescription = engineDescription;
/**
 * By default, deletion protection is disabled.
 * Enable if explicitly provided or if the RemovalPolicy has been set to RETAIN
 */
function defaultDeletionProtection(deletionProtection, removalPolicy) {
    return deletionProtection !== null && deletionProtection !== void 0 ? deletionProtection : (removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
}
exports.defaultDeletionProtection = defaultDeletionProtection;
/**
 * Renders the credentials for an instance or cluster
 */
function renderCredentials(scope, engine, credentials) {
    var _a;
    let renderedCredentials = credentials !== null && credentials !== void 0 ? credentials : props_1.Credentials.fromUsername((_a = engine.defaultUsername) !== null && _a !== void 0 ? _a : 'admin'); // For backwards compatibilty
    if (!renderedCredentials.secret && !renderedCredentials.password) {
        renderedCredentials = props_1.Credentials.fromSecret(new database_secret_1.DatabaseSecret(scope, 'Secret', {
            username: renderedCredentials.username,
            encryptionKey: renderedCredentials.encryptionKey,
            excludeCharacters: renderedCredentials.excludeCharacters,
            // if username must be referenced as a string we can safely replace the
            // secret when customization options are changed without risking a replacement
            replaceOnPasswordCriteriaChanges: credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString,
        }), 
        // pass username if it must be referenced as a string
        (credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString) ? renderedCredentials.username : undefined);
    }
    return renderedCredentials;
}
exports.renderCredentials = renderCredentials;
/**
 * The RemovalPolicy that should be applied to a "helper" resource, if the base resource has the given removal policy
 *
 * - For Clusters, this determines the RemovalPolicy for Instances/SubnetGroups.
 * - For Instances, this determines the RemovalPolicy for SubnetGroups.
 *
 * If the basePolicy is:
 *
 *  DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
 *  RETAIN              -> RETAIN  (anything else will lose data or fail to deploy)
 *  (undefined)         -> DESTROY (base policy is assumed to be SNAPSHOT)
 */
function helperRemovalPolicy(basePolicy) {
    return basePolicy === core_1.RemovalPolicy.RETAIN
        ? core_1.RemovalPolicy.RETAIN
        : core_1.RemovalPolicy.DESTROY;
}
exports.helperRemovalPolicy = helperRemovalPolicy;
/**
 * Return a given value unless it's the same as another value
 */
function renderUnless(value, suppressValue) {
    return value === suppressValue ? undefined : value;
}
exports.renderUnless = renderUnless;
//# sourceMappingURL=data:application/json;base64,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