"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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