###############################################

# X3D Package for Python x3d.py

# generator:  X3duomToX3dPythonPackage.xslt
# X3DUOM:     X3dUnifiedObjectModel-4.0.xml
# Python X3D: https://www.web3d.org/x3d/stylesheets/python/python.html
"""
The x3d.py Python X3D Package supports programmers with Python interfaces and objects for standards-based X3D programming, all as open source.

This work is part of the X3D Python Scene Access Interface Library (X3DPSAIL).
"""

_DEBUG = True       # options True False

###############################################

# Utility Functions

def metaDiagnostics(self, headElement=None):
    """
    Utility function to return any meta info, hint, warning, error, TODO values in this model.
    """
    if  headElement is None:
        headElement = self
    if  isinstance(headElement, X3D):
        headElement = headElement.head
    # print('type(headElement)=' + str(type(headElement)), flush=True) # diagnostic
    if  isinstance(headElement, head):
        result = "meta "
        for each in headElement.children:
            if isinstance(each, meta) and each.name in ('info', 'hint', 'warning', 'error', 'TODO'):
                result += each.name.strip() + ': ' + each.content.strip()
                if  result.endswith('.') or result.endswith(','):
                    result += ' '
                else:
                    result += ', '
        if  result.strip() != "meta":
            return result.rstrip(', ').strip()
    return ''

###############################################

# Field Validation Functions

# Type-specific functions to check for valid values, throw exception if illegal value is found

def fixBoolean(value, default=None):
    """
    Utility function to convert boolean to corresponding Python value.
    """
    # if _DEBUG: print('fixBoolean(value=' + str(value) + ', default=' + str(default) + ')', flush=True)
    if  value is None:
        return default
    if isinstance(value, list) and len(value) == 1:
        # if _DEBUG: print('fixBoolean downcasting by resetting singleton list value=' + str(value) + ' as value=' + str(value[0]), flush=True)
        value = value[0]
    elif isinstance(value, SFBool):
        return value.value # dereference
    elif isinstance(value, MFBool) and len(value) == 1:
        return value.value[0] # dereference
    if value in ('true', 'True'):
        return True
    if value in ('false', 'False'):
        return False
    if isinstance(value, bool):
        return value
    if isinstance(value, list):
        index = 0
        result = value
        for each in value:
#           print('each=' + str(each), flush=True) # debug
            if  each in ('true', 'True'):
                result[index] = True
            elif each in ('false', 'False'):
                result[index] = False
            while isinstance(each, list) and len(each) == 1:
                # if _DEBUG: print('fixBoolean downcasting by extracting singleton list value[' + str(index) + ']=' + str(each) + ' as each[0]=' + str(each[0]), flush=True)
                result[index] = each[0]
            if not isinstance(result[index], bool):
                # print(flush=True)
                raise X3DTypeError('fixBoolean(value=' + str(value) + ') MFBool value[' + str(index) + ']=' + str(each) + ', result[' + str(index) + ']=' + result[index] + ' with type=' + str(type(value)) + ' is not a valid Python bool expression')
            index += 1
        # if _DEBUG: print('...fixBoolean result=' + str(result), flush=True)
        return result
    # print(flush=True)
    raise X3DTypeError('fixBoolean(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python bool')

def isPositive(value):
    """
    Utility function to confirm positive value(s) greater than or equal to zero.
    """
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element <= 0:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each <= 0:
                return False
        return True
    if isinstance(value, (int, float)):
        return value > 0

def assertPositive(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isPositive(value).
    """
    # if _DEBUG: print('* debug: assertPositive(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isPositive(value), fieldName + '=' + str(value) + ' fails assertPositive requirements: value(s) greater than or equal to zero'

def isNonNegative(value):
    """
    Utility function to confirm nonnegative value(s) greater than or equal to zero.
    """
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element < 0:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        # if _DEBUG: print('isNonNegative: ', value, flush=True);
        for each in value:
            if each < 0:
                return False
        return True
    if isinstance(value, (int, float)):
        return value >= 0

def assertNonNegative(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isNonNegative(value).
    """
    # if _DEBUG: print('* debug: assertNonNegative(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isNonNegative(value), fieldName + '=' + str(value) + ' fails assertNonNegative requirements: value(s) greater than or equal to zero'

def isZeroToOne(value):
    """
    Utility function to confirm value(s) in range [0..1]
    """
    # if True or _DEBUG: print('* debug: isZeroToOne(' + str(value) + ')', flush=True);
    if isinstance(value, (list, tuple)):
        for each in value:
            if isinstance(each, (list, tuple)):
                for another in each:
                    if not 0 <= another <= 1:
                        return False
            elif not 0 <= each <= 1:
                return False
        return True
    if isinstance(value, (int, float)):
        return 0 <= value <= 1

def assertZeroToOne(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isZeroToOne(value)
    """
    # if _DEBUG: print('* debug: assertZeroToOne(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isZeroToOne(value), fieldName + '=' + str(value) + ' fails assertZeroToOne requirements: value(s) in range [0..1]'

def isLessThanEquals(value, maximum):
    """
    Utility function to confirm value(s) less than or equal to maximum.
    """
    # if True or _DEBUG: print('* debug: isLessThanEquals(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element > maximum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each > maximum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value <= maximum

def assertLessThanEquals(fieldName, value, maximum):
    """
    Utility function to raise X3DTypeError if not isLessThanEquals(value)
    """
    # if _DEBUG: print('* debug: assertLessThanEquals(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isLessThanEquals(value, maximum), fieldName + '=' + str(value) + ' fails assertLessThanEquals maximum=' + str(maximum)

def isLessThan(value, maximum):
    """
    Utility function to confirm value(s) less than maximum.
    """
    # if True or _DEBUG: print('* debug: isLessThan(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element >= maximum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each >= maximum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value < maximum

def assertLessThan(fieldName, value, maximum):
    """
    Utility function to raise X3DTypeError if not isLessThan(value)
    """
    # if _DEBUG: print('* debug: assertLessThan(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isLessThan(value, maximum), fieldName + '=' + str(value) + ' fails assertLessThan maximum=' + str(maximum)

######
def isGreaterThanEquals(value, minimum):
    """
    Utility function to confirm value(s) less than or equal to minimum.
    """
    # if True or _DEBUG: print('* debug: isGreaterThanEquals(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element < minimum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each < minimum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value >= minimum

def assertGreaterThanEquals(fieldName, value, minimum):
    """
    Utility function to raise X3DTypeError if not isGreaterThanEquals(value)
    """
    # if _DEBUG: print('* debug: assertGreaterThanEquals(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isGreaterThanEquals(value, minimum), fieldName + '=' + str(value) + ' fails assertGreaterThanEquals minimum=' + str(minimum)

def isGreaterThan(value, minimum):
    """
    Utility function to confirm value(s) less than minimum.
    """
    # if True or _DEBUG: print('* debug: isGreaterThan(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element <= minimum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each <= minimum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value > minimum

def assertGreaterThan(fieldName, value, minimum):
    """
    Utility function to raise X3DTypeError if not isGreaterThan(value)
    """
    # if _DEBUG: print('* debug: assertGreaterThan(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isGreaterThan(value, minimum), fieldName + '=' + str(value) + ' fails assertGreaterThan minimum=' + str(minimum)

def isBoundingBox(value):
    """
    Utility function to confirm legal X3D bounding box value of (-1 -1 -1) or nonnegative triple.
    """
    if value is None:
        return None
    # if True or _DEBUG: print('* debug: isBoundingBox(' + str(value) + ')', 'isinstance(value, tuple)=' + str(isinstance(value, tuple)), 'len(value)=' + str(len(value)), flush=True);
    if isinstance(value, (list, tuple)):
        if len(value) != 3:
            return False
        if value[0] == -1 and value[1] == -1 and value[2] == -1:
            return True
        return isNonNegative(value) # legal bounding box tuple

def assertBoundingBox(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isBoundingBox(value)
    """
    # if True or _DEBUG: print('* debug: assertBoundingBox(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isBoundingBox(value), fieldName + '=' + str(value) + ' fails assertBoundingBox requirements: must be (-1, -1, -1) or non-negative 3-tuple'

def isValidSFBool(value):
    """
    Utility function to determine type validity of a SFBool value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFBool) and not isinstance(value, MFBool):
            # if _DEBUG: print('SFBool type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFBool)=' + str(isinstance(value, SFBool)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFBool):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFBool) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, bool):
        return False
    return True

def assertValidSFBool(value):
    """
    Utility function to assert type validity of a SFBool value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFBool) and not isinstance(value, MFBool):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFBool')
    if isinstance(value, SFBool):
        value = value.value # dereference value from this base type
    if isinstance(value, MFBool) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, bool):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid bool value (True or False) for SFBool')
    if not isValidSFBool(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python bool value (True or False) for SFBool')
    return True

def isValidMFBool(value):
    """
    Utility function to determine type validity of a MFBool value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFBool) and not isinstance(value, MFBool):
            # if _DEBUG: print('MFBool type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFBool)=' + str(isinstance(value, MFBool)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFBool):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFBool):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFBool):
            each = each.value # dereference
        if not isinstance(each, bool):
            return False
    return True

def assertValidMFBool(value):
    """
    Utility function to assert type validity of a MFBool value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFBool) and not isinstance(value, MFBool):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFBool')
    if isinstance(value, MFBool):
        value = value.value # dereference value from this base type
    if isinstance(value, SFBool) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFBool')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFBool):
            each = each.value # dereference
        if not isinstance(each, bool):
            # print(flush=True)
            raise X3DTypeError('MFBool list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid bool')
    if not isValidMFBool(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFBool')
    return True

def isValidSFColor(value):
    """
    Utility function to determine type validity of a SFColor value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColor) and not isinstance(value, MFColor):
            # if _DEBUG: print('SFColor type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFColor)=' + str(isinstance(value, SFColor)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFColor):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFColor) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColor):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
        if (each < 0) or (each > 1):
            return False
    if tupleCount != 3:
        return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidSFColor(value):
    """
    Utility function to assert type validity of a SFColor value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColor) and not isinstance(value, MFColor):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFColor')
    if isinstance(value, SFColor):
        value = value.value # dereference value from this base type
    if isinstance(value, MFColor) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColor')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColor):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFColor list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
        if (each < 0) or (each > 1):
            # print(flush=True)
            raise X3DTypeError('SFColor' + str(value)[:100] + ' has value ' + str(each) + ' with type=' + str(type(value)) + ' is  out of range [0..1] and is not a valid SFColor')
    if tupleCount != 3:
        # print(flush=True)
        raise X3DTypeError('SFColor ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    assertZeroToOne('SFColor', value)
    if not isValidSFColor(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColor')
    return True

def isValidMFColor(value):
    """
    Utility function to determine type validity of a MFColor value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColor) and not isinstance(value, MFColor):
            # if _DEBUG: print('MFColor type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFColor)=' + str(isinstance(value, MFColor)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFColor):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFColor):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFColor.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFColor tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColor.TUPLE_SIZE=' + str(MFColor.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
            if (element < 0) or (element > 1):
                return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidMFColor(value):
    """
    Utility function to assert type validity of a MFColor value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColor) and not isinstance(value, MFColor):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFColor')
    if isinstance(value, MFColor):
        value = value.value # dereference value from this base type
    if isinstance(value, SFColor) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColor')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFColor.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFColor tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColor.TUPLE_SIZE=' + str(MFColor.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFColor)):
#                # print(flush=True)
#                raise X3DTypeError('MFColor element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFColor element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
                    if (element < 0) or (element > 1):
                        # print(flush=True)
                        raise X3DTypeError('MFColor' + str(value)[:100] + ' has value ' + str(element) + ' with type=' + str(type(value)) + ' out of range [0..1] and is not a valid MFColor')
    assertZeroToOne('MFColor', value)
    if not isValidMFColor(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColor')
    return True

def isValidSFColorRGBA(value):
    """
    Utility function to determine type validity of a SFColorRGBA value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
            # if _DEBUG: print('SFColorRGBA type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFColorRGBA)=' + str(isinstance(value, SFColorRGBA)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFColorRGBA):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFColorRGBA) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColorRGBA):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
        if (each < 0) or (each > 1):
            return False
    if tupleCount != 4:
        return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidSFColorRGBA(value):
    """
    Utility function to assert type validity of a SFColorRGBA value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFColorRGBA')
    if isinstance(value, SFColorRGBA):
        value = value.value # dereference value from this base type
    if isinstance(value, MFColorRGBA) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColorRGBA')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColorRGBA):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFColorRGBA list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
        if (each < 0) or (each > 1):
            # print(flush=True)
            raise X3DTypeError('SFColorRGBA' + str(value)[:100] + ' has value ' + str(each) + ' with type=' + str(type(value)) + ' is  out of range [0..1] and is not a valid SFColorRGBA')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFColorRGBA ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    assertZeroToOne('SFColorRGBA', value)
    if not isValidSFColorRGBA(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColorRGBA')
    return True

def isValidMFColorRGBA(value):
    """
    Utility function to determine type validity of a MFColorRGBA value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
            # if _DEBUG: print('MFColorRGBA type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFColorRGBA)=' + str(isinstance(value, MFColorRGBA)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFColorRGBA):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFColorRGBA):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFColorRGBA.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFColorRGBA tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColorRGBA.TUPLE_SIZE=' + str(MFColorRGBA.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
            if (element < 0) or (element > 1):
                return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidMFColorRGBA(value):
    """
    Utility function to assert type validity of a MFColorRGBA value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFColorRGBA')
    if isinstance(value, MFColorRGBA):
        value = value.value # dereference value from this base type
    if isinstance(value, SFColorRGBA) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColorRGBA')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFColorRGBA.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFColorRGBA tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColorRGBA.TUPLE_SIZE=' + str(MFColorRGBA.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFColorRGBA)):
#                # print(flush=True)
#                raise X3DTypeError('MFColorRGBA element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFColorRGBA element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
                    if (element < 0) or (element > 1):
                        # print(flush=True)
                        raise X3DTypeError('MFColorRGBA' + str(value)[:100] + ' has value ' + str(element) + ' with type=' + str(type(value)) + ' out of range [0..1] and is not a valid MFColorRGBA')
    assertZeroToOne('MFColorRGBA', value)
    if not isValidMFColorRGBA(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColorRGBA')
    return True

def isValidSFDouble(value):
    """
    Utility function to determine type validity of a SFDouble value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
            # if _DEBUG: print('SFDouble type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFDouble)=' + str(isinstance(value, SFDouble)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFDouble):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFDouble) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, float) and not isinstance(value, int):
        return False
    return True

def assertValidSFDouble(value):
    """
    Utility function to assert type validity of a SFDouble value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFDouble')
    if isinstance(value, SFDouble):
        value = value.value # dereference value from this base type
    if isinstance(value, MFDouble) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, float) and not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFDouble')
    if not isValidSFDouble(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFDouble')
    return True

def isValidMFDouble(value):
    """
    Utility function to determine type validity of a MFDouble value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
            # if _DEBUG: print('MFDouble type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFDouble)=' + str(isinstance(value, MFDouble)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFDouble):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFDouble):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFDouble):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    return True

def assertValidMFDouble(value):
    """
    Utility function to assert type validity of a MFDouble value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFDouble')
    if isinstance(value, MFDouble):
        value = value.value # dereference value from this base type
    if isinstance(value, SFDouble) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFDouble')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFDouble):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFDouble list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if not isValidMFDouble(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFDouble')
    return True

def isValidSFFloat(value):
    """
    Utility function to determine type validity of a SFFloat value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
            # if _DEBUG: print('SFFloat type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFFloat)=' + str(isinstance(value, SFFloat)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFFloat):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFFloat) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, float) and not isinstance(value, int):
        return False
    return True

def assertValidSFFloat(value):
    """
    Utility function to assert type validity of a SFFloat value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFFloat')
    if isinstance(value, SFFloat):
        value = value.value # dereference value from this base type
    if isinstance(value, MFFloat) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, float) and not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFFloat')
    if not isValidSFFloat(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFFloat')
    return True

def isValidMFFloat(value):
    """
    Utility function to determine type validity of a MFFloat value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
            # if _DEBUG: print('MFFloat type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFFloat)=' + str(isinstance(value, MFFloat)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFFloat):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFFloat):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFFloat):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    return True

def assertValidMFFloat(value):
    """
    Utility function to assert type validity of a MFFloat value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFFloat')
    if isinstance(value, MFFloat):
        value = value.value # dereference value from this base type
    if isinstance(value, SFFloat) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFFloat')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFFloat):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFFloat list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if not isValidMFFloat(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFFloat')
    return True

def isValidSFImage(value):
    """
    Utility function to determine type validity of a SFImage value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFImage) and not isinstance(value, MFImage):
            # if _DEBUG: print('SFImage type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFImage)=' + str(isinstance(value, SFImage)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFImage):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFImage) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            return False
    if len(value) > 0: # SFImage list length checks
        if 0 < len(value) < 3:
            return False # SFImage list must start with width, height and number of components (0..4)
        width = value[0]
        height = value[1]
        numberComponents = value[2]
        if  len(value) != (width * height) + 3: # assumes each value in array has all component values in single integer
            print('SFImage array length of ' + str(len(value)) + ' does not equal (width=' + str(width)+ ' * height=' + str(height)+ ') + 3) = ' + str(width * height * numberComponents + 3) + ' (numberComponents=' + numberComponents + ')', flush=True)
            return False # SFImage has invalid list length
    return True

def assertValidSFImage(value):
    """
    Utility function to assert type validity of a SFImage value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFImage) and not isinstance(value, MFImage):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFImage')
    if isinstance(value, SFImage):
        value = value.value # dereference value from this base type
    if isinstance(value, MFImage) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for SFImage')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFImage list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    if not isValidSFImage(value):
        # print(flush=True)
        diagnostic = ''
        if len(value) > 0: # SFImage list length checks
            if 0 < len(value) < 3:
                diagnostic = 'SFImage list must start with width, height and number of components (0..4)'
            else:
                width = value[0]
                height = value[1]
                numberComponents = value[2]
                diagnostic = ' array length of ' + str(len(value)) + ' does not equal (width=' + str(width)+ ' * height=' + str(height)+ ') + 3) = ' + str(width * height + 3)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for SFImage' + diagnostic)
    return True

def isValidMFImage(value):
    """
    Utility function to determine type validity of a MFImage value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFImage) and not isinstance(value, MFImage):
            # if _DEBUG: print('MFImage type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFImage)=' + str(isinstance(value, MFImage)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFImage):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFImage):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            return False
    return True

def assertValidMFImage(value):
    """
    Utility function to assert type validity of a MFImage value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFImage) and not isinstance(value, MFImage):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFImage')
    if isinstance(value, MFImage):
        value = value.value # dereference value from this base type
    if isinstance(value, SFImage) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFImage')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFImage list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    if not isValidMFImage(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFImage')
    return True

def isValidSFInt32(value):
    """
    Utility function to determine type validity of a SFInt32 value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
            # if _DEBUG: print('SFInt32 type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFInt32)=' + str(isinstance(value, SFInt32)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFInt32):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFInt32) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, int):
        return False
    return True

def assertValidSFInt32(value):
    """
    Utility function to assert type validity of a SFInt32 value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFInt32')
    if isinstance(value, SFInt32):
        value = value.value # dereference value from this base type
    if isinstance(value, MFInt32) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid int value for SFInt32')
    if not isValidSFInt32(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python int value for SFInt32')
    return True

def isValidMFInt32(value):
    """
    Utility function to determine type validity of a MFInt32 value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
            # if _DEBUG: print('MFInt32 type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFInt32)=' + str(isinstance(value, MFInt32)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFInt32):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFInt32):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFInt32):
            each = each.value # dereference
        if not isinstance(each, int):
            return False
    return True

def assertValidMFInt32(value):
    """
    Utility function to assert type validity of a MFInt32 value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFInt32')
    if isinstance(value, MFInt32):
        value = value.value # dereference value from this base type
    if isinstance(value, SFInt32) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFInt32')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFInt32):
            each = each.value # dereference
        if not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFInt32 list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    if not isValidMFInt32(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFInt32')
    return True

def isValidSFMatrix3d(value):
    """
    Utility function to determine type validity of a SFMatrix3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
            # if _DEBUG: print('SFMatrix3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix3d)=' + str(isinstance(value, SFMatrix3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 9:
        return False
    return True

def assertValidSFMatrix3d(value):
    """
    Utility function to assert type validity of a SFMatrix3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix3d')
    if isinstance(value, SFMatrix3d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix3d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 9:
        # print(flush=True)
        raise X3DTypeError('SFMatrix3d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 9')
    if not isValidSFMatrix3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3d')
    return True

def isValidMFMatrix3d(value):
    """
    Utility function to determine type validity of a MFMatrix3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
            # if _DEBUG: print('MFMatrix3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix3d)=' + str(isinstance(value, MFMatrix3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix3d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix3d.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFMatrix3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3d.TUPLE_SIZE=' + str(MFMatrix3d.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix3d(value):
    """
    Utility function to assert type validity of a MFMatrix3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix3d')
    if isinstance(value, MFMatrix3d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix3d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix3d.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3d.TUPLE_SIZE=' + str(MFMatrix3d.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix3d)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix3d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix3d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3d')
    return True

def isValidSFMatrix3f(value):
    """
    Utility function to determine type validity of a SFMatrix3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
            # if _DEBUG: print('SFMatrix3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix3f)=' + str(isinstance(value, SFMatrix3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 9:
        return False
    return True

def assertValidSFMatrix3f(value):
    """
    Utility function to assert type validity of a SFMatrix3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix3f')
    if isinstance(value, SFMatrix3f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix3f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 9:
        # print(flush=True)
        raise X3DTypeError('SFMatrix3f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 9')
    if not isValidSFMatrix3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3f')
    return True

def isValidMFMatrix3f(value):
    """
    Utility function to determine type validity of a MFMatrix3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
            # if _DEBUG: print('MFMatrix3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix3f)=' + str(isinstance(value, MFMatrix3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix3f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix3f.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFMatrix3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3f.TUPLE_SIZE=' + str(MFMatrix3f.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix3f(value):
    """
    Utility function to assert type validity of a MFMatrix3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix3f')
    if isinstance(value, MFMatrix3f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix3f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix3f.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3f.TUPLE_SIZE=' + str(MFMatrix3f.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix3f)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix3f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix3f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3f')
    return True

def isValidSFMatrix4d(value):
    """
    Utility function to determine type validity of a SFMatrix4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
            # if _DEBUG: print('SFMatrix4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix4d)=' + str(isinstance(value, SFMatrix4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 16:
        return False
    return True

def assertValidSFMatrix4d(value):
    """
    Utility function to assert type validity of a SFMatrix4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix4d')
    if isinstance(value, SFMatrix4d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix4d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 16:
        # print(flush=True)
        raise X3DTypeError('SFMatrix4d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 16')
    if not isValidSFMatrix4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4d')
    return True

def isValidMFMatrix4d(value):
    """
    Utility function to determine type validity of a MFMatrix4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
            # if _DEBUG: print('MFMatrix4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix4d)=' + str(isinstance(value, MFMatrix4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix4d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix4d.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFMatrix4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4d.TUPLE_SIZE=' + str(MFMatrix4d.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix4d(value):
    """
    Utility function to assert type validity of a MFMatrix4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix4d')
    if isinstance(value, MFMatrix4d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix4d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix4d.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4d.TUPLE_SIZE=' + str(MFMatrix4d.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix4d)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix4d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix4d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4d')
    return True

def isValidSFMatrix4f(value):
    """
    Utility function to determine type validity of a SFMatrix4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
            # if _DEBUG: print('SFMatrix4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix4f)=' + str(isinstance(value, SFMatrix4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 16:
        return False
    return True

def assertValidSFMatrix4f(value):
    """
    Utility function to assert type validity of a SFMatrix4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix4f')
    if isinstance(value, SFMatrix4f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix4f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 16:
        # print(flush=True)
        raise X3DTypeError('SFMatrix4f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 16')
    if not isValidSFMatrix4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4f')
    return True

def isValidMFMatrix4f(value):
    """
    Utility function to determine type validity of a MFMatrix4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
            # if _DEBUG: print('MFMatrix4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix4f)=' + str(isinstance(value, MFMatrix4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix4f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix4f.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFMatrix4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4f.TUPLE_SIZE=' + str(MFMatrix4f.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix4f(value):
    """
    Utility function to assert type validity of a MFMatrix4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix4f')
    if isinstance(value, MFMatrix4f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix4f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix4f.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4f.TUPLE_SIZE=' + str(MFMatrix4f.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix4f)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix4f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix4f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4f')
    return True

def isValidSFNode(value):
    """
    Utility function to determine type validity of a SFNode value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFNode) and not isinstance(value, MFNode):
            # if _DEBUG: print('SFNode type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFNode)=' + str(isinstance(value, SFNode)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFNode):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFNode) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, object):
        return False
    return True

def assertValidSFNode(value):
    """
    Utility function to assert type validity of a SFNode value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFNode) and not isinstance(value, MFNode):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFNode')
    if isinstance(value, SFNode):
        value = value.value # dereference value from this base type
    if isinstance(value, MFNode) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, object):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid object value for SFNode')
    if not isValidSFNode(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python object value for SFNode')
    return True

def isValidMFNode(value):
    """
    Utility function to determine type validity of a MFNode value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFNode) and not isinstance(value, MFNode):
            # if _DEBUG: print('MFNode type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFNode)=' + str(isinstance(value, MFNode)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFNode):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFNode):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    for each in value:
        if not isinstance(each, (_X3DNode, _X3DStatement)):
            return False
    return True

def assertValidMFNode(value):
    """
    Utility function to assert type validity of a MFNode value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFNode) and not isinstance(value, MFNode):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFNode')
    if isinstance(value, MFNode):
        value = value.value # dereference value from this base type
    if isinstance(value, SFNode) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    for each in value:
        if not isinstance(each, _X3DNode) and not isinstance(each, _X3DStatement):
            # print(flush=True)
            raise X3DTypeError(str(value)[:100] + ' element ' + str(each) + ', type=' + str(type(each)) + ' is not a valid _X3DNode or _X3DStatement for MFNode')
    if not isValidMFNode(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid _X3DNode or _X3DStatement for MFNode')
    return True

def isValidSFRotation(value):
    """
    Utility function to determine type validity of a SFRotation value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
            # if _DEBUG: print('SFRotation type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFRotation)=' + str(isinstance(value, SFRotation)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFRotation):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFRotation) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFRotation):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 4:
        return False
    return True

def assertValidSFRotation(value):
    """
    Utility function to assert type validity of a SFRotation value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFRotation')
    if isinstance(value, SFRotation):
        value = value.value # dereference value from this base type
    if isinstance(value, MFRotation) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFRotation')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFRotation):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFRotation list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFRotation ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    if not isValidSFRotation(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFRotation')
    return True

def isValidMFRotation(value):
    """
    Utility function to determine type validity of a MFRotation value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
            # if _DEBUG: print('MFRotation type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFRotation)=' + str(isinstance(value, MFRotation)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFRotation):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFRotation):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFRotation.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFRotation tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFRotation.TUPLE_SIZE=' + str(MFRotation.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFRotation(value):
    """
    Utility function to assert type validity of a MFRotation value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFRotation')
    if isinstance(value, MFRotation):
        value = value.value # dereference value from this base type
    if isinstance(value, SFRotation) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFRotation')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFRotation.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFRotation tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFRotation.TUPLE_SIZE=' + str(MFRotation.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFRotation)):
#                # print(flush=True)
#                raise X3DTypeError('MFRotation element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFRotation element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFRotation(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFRotation')
    return True

def isValidSFString(value):
    """
    Utility function to determine type validity of a SFString value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFString) and not isinstance(value, MFString):
            # if _DEBUG: print('SFString type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFString)=' + str(isinstance(value, SFString)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFString):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFString) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, str):
        return False
    return True

def assertValidSFString(value):
    """
    Utility function to assert type validity of a SFString value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFString) and not isinstance(value, MFString):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFString')
    if isinstance(value, SFString):
        value = value.value # dereference value from this base type
    if isinstance(value, MFString) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, str):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid str value for SFString')
    if not isValidSFString(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python str value for SFString')
    return True

def isValidMFString(value):
    """
    Utility function to determine type validity of a MFString value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFString) and not isinstance(value, MFString):
            # if _DEBUG: print('MFString type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFString)=' + str(isinstance(value, MFString)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFString):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFString):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFString):
            each = each.value # dereference
        if not isinstance(each, str):
            return False
    return True

def assertValidMFString(value):
    """
    Utility function to assert type validity of a MFString value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFString) and not isinstance(value, MFString):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFString')
    if isinstance(value, MFString):
        value = value.value # dereference value from this base type
    if isinstance(value, SFString) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFString')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFString):
            each = each.value # dereference
        if not isinstance(each, str):
            # print(flush=True)
            raise X3DTypeError('MFString list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid str')
    if not isValidMFString(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFString')
    return True

def isValidSFTime(value):
    """
    Utility function to determine type validity of a SFTime value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFTime) and not isinstance(value, MFTime):
            # if _DEBUG: print('SFTime type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFTime)=' + str(isinstance(value, SFTime)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFTime):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFTime) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, float) and not isinstance(value, int):
        return False
    return True

def assertValidSFTime(value):
    """
    Utility function to assert type validity of a SFTime value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFTime) and not isinstance(value, MFTime):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFTime')
    if isinstance(value, SFTime):
        value = value.value # dereference value from this base type
    if isinstance(value, MFTime) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, float) and not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFTime')
    if not isValidSFTime(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFTime')
    return True

def isValidMFTime(value):
    """
    Utility function to determine type validity of a MFTime value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFTime) and not isinstance(value, MFTime):
            # if _DEBUG: print('MFTime type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFTime)=' + str(isinstance(value, MFTime)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFTime):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFTime):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFTime):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    return True

def assertValidMFTime(value):
    """
    Utility function to assert type validity of a MFTime value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFTime) and not isinstance(value, MFTime):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFTime')
    if isinstance(value, MFTime):
        value = value.value # dereference value from this base type
    if isinstance(value, SFTime) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFTime')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFTime):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFTime list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if not isValidMFTime(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFTime')
    return True

def isValidSFVec2d(value):
    """
    Utility function to determine type validity of a SFVec2d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
            # if _DEBUG: print('SFVec2d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec2d)=' + str(isinstance(value, SFVec2d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec2d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec2d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 2:
        return False
    return True

def assertValidSFVec2d(value):
    """
    Utility function to assert type validity of a SFVec2d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec2d')
    if isinstance(value, SFVec2d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec2d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec2d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 2:
        # print(flush=True)
        raise X3DTypeError('SFVec2d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 2')
    if not isValidSFVec2d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2d')
    return True

def isValidMFVec2d(value):
    """
    Utility function to determine type validity of a MFVec2d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
            # if _DEBUG: print('MFVec2d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec2d)=' + str(isinstance(value, MFVec2d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec2d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec2d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec2d.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFVec2d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2d.TUPLE_SIZE=' + str(MFVec2d.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec2d(value):
    """
    Utility function to assert type validity of a MFVec2d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec2d')
    if isinstance(value, MFVec2d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec2d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec2d.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFVec2d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2d.TUPLE_SIZE=' + str(MFVec2d.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec2d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec2d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec2d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec2d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2d')
    return True

def isValidSFVec2f(value):
    """
    Utility function to determine type validity of a SFVec2f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
            # if _DEBUG: print('SFVec2f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec2f)=' + str(isinstance(value, SFVec2f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec2f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec2f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 2:
        return False
    return True

def assertValidSFVec2f(value):
    """
    Utility function to assert type validity of a SFVec2f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec2f')
    if isinstance(value, SFVec2f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec2f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec2f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 2:
        # print(flush=True)
        raise X3DTypeError('SFVec2f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 2')
    if not isValidSFVec2f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2f')
    return True

def isValidMFVec2f(value):
    """
    Utility function to determine type validity of a MFVec2f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
            # if _DEBUG: print('MFVec2f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec2f)=' + str(isinstance(value, MFVec2f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec2f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec2f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec2f.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFVec2f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2f.TUPLE_SIZE=' + str(MFVec2f.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec2f(value):
    """
    Utility function to assert type validity of a MFVec2f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec2f')
    if isinstance(value, MFVec2f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec2f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec2f.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFVec2f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2f.TUPLE_SIZE=' + str(MFVec2f.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec2f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec2f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec2f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec2f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2f')
    return True

def isValidSFVec3d(value):
    """
    Utility function to determine type validity of a SFVec3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
            # if _DEBUG: print('SFVec3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec3d)=' + str(isinstance(value, SFVec3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 3:
        return False
    return True

def assertValidSFVec3d(value):
    """
    Utility function to assert type validity of a SFVec3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec3d')
    if isinstance(value, SFVec3d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec3d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 3:
        # print(flush=True)
        raise X3DTypeError('SFVec3d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    if not isValidSFVec3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3d')
    return True

def isValidMFVec3d(value):
    """
    Utility function to determine type validity of a MFVec3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
            # if _DEBUG: print('MFVec3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec3d)=' + str(isinstance(value, MFVec3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec3d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec3d.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFVec3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3d.TUPLE_SIZE=' + str(MFVec3d.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec3d(value):
    """
    Utility function to assert type validity of a MFVec3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec3d')
    if isinstance(value, MFVec3d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec3d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec3d.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFVec3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3d.TUPLE_SIZE=' + str(MFVec3d.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec3d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec3d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec3d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3d')
    return True

def isValidSFVec3f(value):
    """
    Utility function to determine type validity of a SFVec3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
            # if _DEBUG: print('SFVec3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec3f)=' + str(isinstance(value, SFVec3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 3:
        return False
    return True

def assertValidSFVec3f(value):
    """
    Utility function to assert type validity of a SFVec3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec3f')
    if isinstance(value, SFVec3f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec3f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 3:
        # print(flush=True)
        raise X3DTypeError('SFVec3f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    if not isValidSFVec3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3f')
    return True

def isValidMFVec3f(value):
    """
    Utility function to determine type validity of a MFVec3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
            # if _DEBUG: print('MFVec3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec3f)=' + str(isinstance(value, MFVec3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec3f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec3f.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFVec3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3f.TUPLE_SIZE=' + str(MFVec3f.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec3f(value):
    """
    Utility function to assert type validity of a MFVec3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec3f')
    if isinstance(value, MFVec3f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec3f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec3f.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFVec3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3f.TUPLE_SIZE=' + str(MFVec3f.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec3f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec3f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec3f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3f')
    return True

def isValidSFVec4d(value):
    """
    Utility function to determine type validity of a SFVec4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
            # if _DEBUG: print('SFVec4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec4d)=' + str(isinstance(value, SFVec4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 4:
        return False
    return True

def assertValidSFVec4d(value):
    """
    Utility function to assert type validity of a SFVec4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec4d')
    if isinstance(value, SFVec4d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec4d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFVec4d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    if not isValidSFVec4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4d')
    return True

def isValidMFVec4d(value):
    """
    Utility function to determine type validity of a MFVec4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
            # if _DEBUG: print('MFVec4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec4d)=' + str(isinstance(value, MFVec4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec4d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec4d.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFVec4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4d.TUPLE_SIZE=' + str(MFVec4d.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec4d(value):
    """
    Utility function to assert type validity of a MFVec4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec4d')
    if isinstance(value, MFVec4d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec4d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec4d.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFVec4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4d.TUPLE_SIZE=' + str(MFVec4d.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec4d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec4d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec4d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4d')
    return True

def isValidSFVec4f(value):
    """
    Utility function to determine type validity of a SFVec4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
            # if _DEBUG: print('SFVec4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec4f)=' + str(isinstance(value, SFVec4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 4:
        return False
    return True

def assertValidSFVec4f(value):
    """
    Utility function to assert type validity of a SFVec4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec4f')
    if isinstance(value, SFVec4f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec4f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFVec4f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    if not isValidSFVec4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4f')
    return True

def isValidMFVec4f(value):
    """
    Utility function to determine type validity of a MFVec4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
            # if _DEBUG: print('MFVec4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec4f)=' + str(isinstance(value, MFVec4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec4f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec4f.TUPLE_SIZE != 0:
            # if _DEBUG:
            print('* isValidMFVec4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4f.TUPLE_SIZE=' + str(MFVec4f.TUPLE_SIZE) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec4f(value):
    """
    Utility function to assert type validity of a MFVec4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec4f')
    if isinstance(value, MFVec4f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec4f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec4f.TUPLE_SIZE != 0:
                # print(flush=True)
                raise X3DValueError('MFVec4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4f.TUPLE_SIZE=' + str(MFVec4f.TUPLE_SIZE) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec4f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec4f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec4f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4f')
    return True


def assertValidFieldInitializationValue(name, fieldType, value, parent=''):
    """
    Utility function to assert fieldType validity of a field initialization value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* assertValidFieldInitializationValue name=' + str(name) + ', fieldType=' + str(fieldType) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
    # note that ExternProtoDeclare field definitions do not have any value
    if name is None:
        print('* assertValidFieldInitializationValue improper invocation: name=' + str(name) + ', fieldType=' + str(fieldType) + ', value=' + str(value)[:100] + ', parent=' + parent + ', ignored', flush=True)
        return # ignore
    if value is None or (not(fieldType == bool) and not value):
        return # ignore
    if fieldType == 'SFString':
        assertValidSFString(value)
    elif fieldType == 'MFString':
        assertValidMFString(value)
    elif (fieldType == 'SFBool') or (fieldType == bool) or isinstance(value, bool):
        assertValidSFBool(value)
    elif fieldType == 'MFBool':
        assertValidMFBool(value)
    elif (fieldType == 'SFInt32') or (fieldType == int) or isinstance(value, int):
        assertValidSFInt32(value)
    elif fieldType == 'MFInt32':
        assertValidMFInt32(value)
    elif (fieldType == 'SFFloat') or (fieldType == float) or isinstance(value, float):
        assertValidSFFloat(value)
    elif fieldType == 'MFFloat':
        assertValidMFFloat(value)
    elif fieldType == 'SFDouble':
        assertValidSFDouble(value)
    elif fieldType == 'MFDouble':
        assertValidMFDouble(value)
    elif fieldType == 'SFTime':
        assertValidSFTime(value)
    elif fieldType == 'MFTime':
        assertValidMFTime(value)
    elif fieldType == 'SFColor':
        assertValidSFColor(value)
    elif fieldType == 'MFColorRGBA':
        assertValidMFColorRGBA(value)
    elif fieldType == 'SFRotation':
        assertValidSFRotation(value)
    elif fieldType == 'MFRotation':
        assertValidMFRotation(value)
    elif fieldType == 'SFImage':
        assertValidSFImage(value)
    elif fieldType == 'MFImage':
        assertValidMFImage(value)
    elif fieldType == 'SFNode':
        assertValidSFNode(value)
    elif fieldType == 'MFNode':
        assertValidMFNode(value)
    elif fieldType == 'SFVec2f':
        assertValidSFVec2f(value)
    elif fieldType == 'MFVec2f':
        assertValidMFVec2f(value)
    elif fieldType == 'SFVec3f':
        assertValidSFVec3f(value)
    elif fieldType == 'MFVec3f':
        assertValidMFVec3f(value)
    elif fieldType == 'SFVec4f':
        assertValidSFVec4f(value)
    elif fieldType == 'MFVec4f':
        assertValidMFVec4f(value)
    elif fieldType == 'SFVec2d':
        assertValidSFVec2d(value)
    elif fieldType == 'MFVec2d':
        assertValidMFVec2d(value)
    elif fieldType == 'SFVec3d':
        assertValidSFVec3d(value)
    elif fieldType == 'MFVec3d':
        assertValidMFVec3d(value)
    elif fieldType == 'SFVec4d':
        assertValidSFVec4d(value)
    elif fieldType == 'MFVec4d':
        assertValidMFVec4d(value)
    elif fieldType == 'SFMatrix3d':
        assertValidSFMatrix3f(value)
    elif fieldType == 'MFMatrix3f':
        assertValidMFMatrix3f(value)
    elif fieldType == 'SFMatrix4f':
        assertValidSFMatrix4f(value)
    elif fieldType == 'MFMatrix4f':
        assertValidMFMatrix4f(value)
    elif fieldType == 'SFMatrix3d':
        assertValidSFMatrix3d(value)
    elif fieldType == 'MFMatrix3d':
        assertValidMFMatrix3d(value)
    elif fieldType == 'SFMatrix4d':
        assertValidSFMatrix4d(value)
    elif fieldType == 'MFMatrix4d':
        assertValidMFMatrix4d(value)
    elif (fieldType == str)   or isinstance(value, str):
        assertValidSFString(value)
    elif str(parent) == 'fieldValue':
        return True # TODO check further if possible
    elif (fieldType == list) or isinstance(value, list):
        try:
            if isinstance(value[0], tuple):
                print('*** assertValidFieldInitializationValue TODO validate list fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
                return True # TODO check further
            initialListItemType = fieldType(value[0])
            # https://stackoverflow.com/questions/522563/accessing-the-index-in-for-loops/28072982#28072982
            # https://stackoverflow.com/questions/1952464/in-python-how-do-i-determine-if-an-object-is-iterable
            for index, each in enumerate(value):
                assertValidFieldInitializationValue(name + '[' + str(index) + ']', initialListItemType, value[index], parent)
        except TypeError:
            return True # TODO check further if possible
    elif (fieldType == tuple) or isinstance(value, tuple):
        print('*** assertValidFieldInitializationValue TODO validate tuple fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
        return True # TODO check further if possible
#       initialListItemType = fieldType(value[0])
#       for index, each in enumerate(value):
#           assertValidFieldInitializationValue(name + '[' + str(index) + '], fieldType(value[index])', value[index], parent)
    else:
        print('*** assertValidFieldInitializationValue unknown fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
###############################################

class _X3DField(object):
    """
    All X3D fields implement _X3DField abstract type.
    """
    value = None
    NAME = '_X3DField'
    SPECIFICATION_URL = 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/fieldsDef.html'
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#FieldTypesTable'
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide value of this field type. """
        return self.__value
    def __repl__(self):
        # if _DEBUG: print('* debug: type(self.value)=' + str(type(self.value)), flush=True)
        if isinstance(self.value, (SFString, str)):
            return "'" + self.value + "'"
        if  isinstance(self.value, tuple) and 'SF' in str(type(self)): # avoid X3DTypeError if value is not iterable
            result = '('
            for each in self.value:
                result += str(each) + ', '
                # if _DEBUG: print('* _X3DField debug: str(each)=' + str(each), flush=True)
            return result.rstrip(', ') + ')'
        if  isinstance(self.value, list) and 'MF' in str(type(self)): # avoid X3DTypeError if value is not iterable
            # isinstance(self.value, MFNode): not working, what got passed in was not an MFNode object apparently
            result = '['
            for each in self.value:
                result += str(each) + ', '
                # if _DEBUG: print('* _X3DField debug: str(each)=' + str(each), flush=True)
            return result.rstrip(', ') + ']'
        return str(self.value)
    def __str__(self):
        return self.__repl__()

def isX3DField(value):
    """
    Determine whether object is an instance of _X3DField.
    """
    return isinstance(value, _X3DField)

# Access Types

class AccessType(_X3DField):
    """
    accessType determines whether a field corresponds to event input, event output, or persistent state information. Events are strictly typed values with a corresponding timestamp. ROUTE connections must match accessType between source field and target field.

    initializeOnly: can be initialized, but cannot send or receive events. This is usually the case for fields that are considered too computationally expensive to change at run time.
    inputOutput: can be initialized, and can also send or receive events during run-time operations.
    inputOnly: cannot be initialized or included in a scene file, but can receive input event values via a ROUTE during run-time operations.
    outputOnly: cannot be initialized or included in a scene file, but can send output event values via a ROUTE during run-time operations.
    """
    SPECIFICATION_URL = 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/concepts.html#FieldSemantics'
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#accessType'
    # string constants listing each allowed accessType
    initializeOnly = 'initializeOnly'
    inputOutput = 'inputOutput'
    inputOnly = 'inputOnly'
    outputOnly = 'outputOnly'


# Field Types

class FieldType(_X3DField):
    """
    The X3D Architecture specification of field types classify the possible values for a field.
    Each field in each node (i.e. each XML attribute) has a strictly defined data type.
    Multiple data types are provided for boolean, integer, floating-point and string values.
    X3D is a strongly typed language, meaning that all data must strictly conform to these data types in order for a scene to be correct.
    """
    SPECIFICATION_URL = 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/fieldsDef.html'
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#type'
    # string constants listing each allowed type
    SFBool = 'SFBool'
    MFBool = 'MFBool'
    SFColor = 'SFColor'
    MFColor = 'MFColor'
    SFColorRGBA = 'SFColorRGBA'
    MFColorRGBA = 'MFColorRGBA'
    SFDouble = 'SFDouble'
    MFDouble = 'MFDouble'
    SFFloat = 'SFFloat'
    MFFloat = 'MFFloat'
    SFImage = 'SFImage'
    MFImage = 'MFImage'
    SFInt32 = 'SFInt32'
    MFInt32 = 'MFInt32'
    SFMatrix3d = 'SFMatrix3d'
    MFMatrix3d = 'MFMatrix3d'
    SFMatrix3f = 'SFMatrix3f'
    MFMatrix3f = 'MFMatrix3f'
    SFMatrix4d = 'SFMatrix4d'
    MFMatrix4d = 'MFMatrix4d'
    SFMatrix4f = 'SFMatrix4f'
    MFMatrix4f = 'MFMatrix4f'
    SFNode = 'SFNode'
    MFNode = 'MFNode'
    SFRotation = 'SFRotation'
    MFRotation = 'MFRotation'
    SFString = 'SFString'
    MFString = 'MFString'
    SFTime = 'SFTime'
    MFTime = 'MFTime'
    SFVec2d = 'SFVec2d'
    MFVec2d = 'MFVec2d'
    SFVec2f = 'SFVec2f'
    MFVec2f = 'MFVec2f'
    SFVec3d = 'SFVec3d'
    MFVec3d = 'MFVec3d'
    SFVec3f = 'SFVec3f'
    MFVec3f = 'MFVec3f'
    SFVec4d = 'SFVec4d'
    MFVec4d = 'MFVec4d'
    SFVec4f = 'SFVec4f'
    MFVec4f = 'MFVec4f'

class SFBool(_X3DField):
    """
    SFBool is a logical type with possible values (True|False).
    """
    NAME = 'SFBool'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFBool'
    DEFAULT_VALUE = True
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'\s*(true|false)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        value = fixBoolean(value, default=self.DEFAULT_VALUE)
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFBool) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFBool(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFBool(_X3DField):
    """
    MFBool is an array of Boolean values.
    """
    NAME = 'MFBool'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFBool'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'\s*((true|false)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        value = fixBoolean(value, default=self.DEFAULT_VALUE)
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFBool(value):
            value = [value]
        assertValidMFBool(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFBool(value):
                if isinstance(value, SFBool):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFBool(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFBool):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFBool(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFColor(_X3DField):
    """
    The SFColor field specifies one RGB (red-green-blue) color triple.
    """
    NAME = 'SFColor'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColor'
    DEFAULT_VALUE = (0, 0, 0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 3
    REGEX_XML = r'(\s)*(([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){2}([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif isinstance(value, MFColor) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFColor(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFColor(_X3DField):
    """
    MFColor specifies zero or more SFColor RGB triples.
    """
    NAME = 'MFColor'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColor'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 3
    REGEX_XML = r'(\s)*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){2}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif not isinstance(value, list) and isValidSFColor(value):
            value = [value]
        assertValidMFColor(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFColor(value):
                if isinstance(value, SFColor):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFColor(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFColor):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFColor(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFColorRGBA(_X3DField):
    """
    The SFColorRGBA field specifies one RGBA (red-green-blue-alpha) color 4-tuple.
    """
    NAME = 'SFColorRGBA'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColorRGBA'
    DEFAULT_VALUE = (0, 0, 0, 0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 4
    REGEX_XML = r'(\s)*(([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFColorRGBA) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFColorRGBA(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFColorRGBA(_X3DField):
    """
    MFColorRGBA specifies zero or more SFColorRGBA 4-tuples.
    """
    NAME = 'MFColorRGBA'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColorRGBA'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 4
    REGEX_XML = r'(\s)*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif not isinstance(value, list) and isValidSFColorRGBA(value):
            value = [value]
        assertValidMFColorRGBA(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFColorRGBA(value):
                if isinstance(value, SFColorRGBA):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFColorRGBA(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFColorRGBA):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFColorRGBA(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFDouble(_X3DField):
    """
    SFDouble is a double-precision floating-point type.
    """
    NAME = 'SFDouble'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFDouble'
    DEFAULT_VALUE = 0.0
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFDouble) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFDouble(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFDouble(_X3DField):
    """
    MFDouble is an array of Double values, meaning a double-precision floating-point array type.
    """
    NAME = 'MFDouble'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFDouble'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFDouble(value):
            value = [value]
        assertValidMFDouble(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFDouble(value):
                if isinstance(value, SFDouble):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFDouble(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFDouble):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFDouble(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFFloat(_X3DField):
    """
    SFFloat is a single-precision floating-point type.
    """
    NAME = 'SFFloat'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFFloat'
    DEFAULT_VALUE = 0.0
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFFloat) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFFloat(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFFloat(_X3DField):
    """
    MFFloat is an array of SFFloat values, meaning a single-precision floating-point array type.
    """
    NAME = 'MFFloat'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFFloat'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFFloat(value):
            value = [value]
        assertValidMFFloat(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFFloat(value):
                if isinstance(value, SFFloat):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFFloat(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFFloat):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFFloat(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFImage(_X3DField):
    """
    The SFImage field specifies a single uncompressed 2-dimensional pixel image.
    """
    NAME = 'SFImage'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFImage'
    DEFAULT_VALUE = [0, 0, 0]
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'\s*([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFImage) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFImage(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFImage(_X3DField):
    """
    MFImage is an array of SFImage values.
    """
    NAME = 'MFImage'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFImage'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'\s*(([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFImage(value):
            value = [value]
        assertValidMFImage(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFImage(value):
                if isinstance(value, SFImage):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFImage(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFImage):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFImage(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFInt32(_X3DField):
    """
    An SFInt32 field specifies one 32-bit signed integer.
    """
    NAME = 'SFInt32'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFInt32'
    DEFAULT_VALUE = 0
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*[+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?(\s)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFInt32) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFInt32(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFInt32(_X3DField):
    """
    An MFInt32 field defines an array of 32-bit signed integers.
    """
    NAME = 'MFInt32'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFInt32'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*([+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFInt32(value):
            value = [value]
        assertValidMFInt32(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFInt32(value):
                if isinstance(value, SFInt32):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFInt32(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFInt32):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFInt32(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFMatrix3d(_X3DField):
    """
    SFMatrix3d specifies a 3x3 matrix of double-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'SFMatrix3d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3d'
    DEFAULT_VALUE = (1, 0, 0, 0, 1, 0, 0, 0, 1)
    ARRAY_TYPE = False
    TUPLE_SIZE = 9
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFMatrix3d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix3d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFMatrix3d(_X3DField):
    """
    MFMatrix3d specifies zero or more 3x3 matrices of double-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'MFMatrix3d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3d'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 9
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFMatrix3d(value):
            value = [value]
        assertValidMFMatrix3d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix3d(value):
                if isinstance(value, SFMatrix3d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix3d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix3d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix3d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFMatrix3f(_X3DField):
    """
    SFMatrix3f specifies a 3x3 matrix of single-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'SFMatrix3f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3f'
    DEFAULT_VALUE = (1, 0, 0, 0, 1, 0, 0, 0, 1)
    ARRAY_TYPE = False
    TUPLE_SIZE = 9
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFMatrix3f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix3f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFMatrix3f(_X3DField):
    """
    MFMatrix3f specifies zero or more 3x3 matrices of single-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'MFMatrix3f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3f'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 9
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFMatrix3f(value):
            value = [value]
        assertValidMFMatrix3f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix3f(value):
                if isinstance(value, SFMatrix3f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix3f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix3f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix3f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFMatrix4d(_X3DField):
    """
    SFMatrix4d specifies a 4x4 matrix of double-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'SFMatrix4d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4d'
    DEFAULT_VALUE = (1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
    ARRAY_TYPE = False
    TUPLE_SIZE = 16
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFMatrix4d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix4d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFMatrix4d(_X3DField):
    """
    MFMatrix4d specifies zero or more 4x4 matrices of double-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'MFMatrix4d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4d'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 16
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFMatrix4d(value):
            value = [value]
        assertValidMFMatrix4d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix4d(value):
                if isinstance(value, SFMatrix4d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix4d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix4d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix4d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFMatrix4f(_X3DField):
    """
    SFMatrix4f specifies a 4x4 matrix of single-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'SFMatrix4f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4f'
    DEFAULT_VALUE = (1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
    ARRAY_TYPE = False
    TUPLE_SIZE = 16
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFMatrix4f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix4f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFMatrix4f(_X3DField):
    """
    MFMatrix4f specifies zero or more 4x4 matrices of single-precision floating point numbers, organized in row-major fashion.
    """
    NAME = 'MFMatrix4f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4f'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 16
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFMatrix4f(value):
            value = [value]
        assertValidMFMatrix4f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix4f(value):
                if isinstance(value, SFMatrix4f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix4f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix4f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix4f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFNode(_X3DField):
    """
    The SFNode field specifies an X3D node; the default empty value of an uninitialized SFNode field is sometimes described as NULL.
    """
    NAME = 'SFNode'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFNode'
    DEFAULT_VALUE = None
    FIELD_DECLARATIONS = [('value', 'None', FieldType.SFNode, AccessType.inputOutput, 'SFNode')]
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFNode) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFNode(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFNode(_X3DField):
    """
    The MFNode field specifies zero or more nodes; the default value of an MFNode field is the empty list.
    """
    NAME = 'MFNode'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFNode'
    DEFAULT_VALUE = list()
    FIELD_DECLARATIONS = [('value', None, FieldType.MFNode, AccessType.inputOutput, 'MFNode')]
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFNode(value):
            value = [value]
        assertValidMFNode(value)
        self.__value = value
    def __repl__(self):
        result = '['
        for each in self.__value:
            result += str(each) + ', '
        return result.rstrip(', ') + ']'
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFNode(value):
                if isinstance(value, SFNode):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFNode(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFNode):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFNode(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFRotation(_X3DField):
    """
    SFRotation is an axis-angle 4-tuple, indicating X-Y-Z direction axis plus angle orientation about that axis.
    """
    NAME = 'SFRotation'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFRotation'
    DEFAULT_VALUE = (0, 0, 1, 0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 4
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFRotation) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFRotation(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFRotation(_X3DField):
    """
    MFRotation is an array of SFRotation values.
    """
    NAME = 'MFRotation'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFRotation'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 4
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif not isinstance(value, list) and isValidSFRotation(value):
            value = [value]
        assertValidMFRotation(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFRotation(value):
                if isinstance(value, SFRotation):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFRotation(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFRotation):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFRotation(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFString(_X3DField):
    """
    SFString defines a single string encoded with the UTF-8 universal character set.
    """
    NAME = 'SFString'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFString'
    DEFAULT_VALUE = ''
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s|\S)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFString) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFString(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFString(_X3DField):
    """
    MFString is an array of SFString values, each "quoted" and separated by whitespace.
    """
    NAME = 'MFString'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFString'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s|\S)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFString(value):
            value = [value]
        assertValidMFString(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFString(value):
                if isinstance(value, SFString):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFString(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFString):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFString(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFTime(_X3DField):
    """
    The SFTime field specifies a single time value.
    """
    NAME = 'SFTime'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFTime'
    DEFAULT_VALUE = -1.0
    ARRAY_TYPE = False
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFTime) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFTime(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFTime(_X3DField):
    """
    MFTime is an array of SFTime values.
    """
    NAME = 'MFTime'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFTime'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 1
    REGEX_XML = r'(\s)*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFTime(value):
            value = [value]
        assertValidMFTime(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFTime(value):
                if isinstance(value, SFTime):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFTime(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFTime):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFTime(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFVec2d(_X3DField):
    """
    SFVec2d is a 2-tuple pair of SFDouble values.
    """
    NAME = 'SFVec2d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2d'
    DEFAULT_VALUE = (0.0, 0.0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 2
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFVec2d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec2d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFVec2d(_X3DField):
    """
    MFVec2d is an array of SFVec2d values.
    """
    NAME = 'MFVec2d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2d'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 2
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFVec2d(value):
            value = [value]
        assertValidMFVec2d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec2d(value):
                if isinstance(value, SFVec2d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec2d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec2d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec2d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFVec2f(_X3DField):
    """
    SFVec2f is a 2-tuple pair of SFFloat values.
    """
    NAME = 'SFVec2f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2f'
    DEFAULT_VALUE = (0.0, 0.0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 2
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 2-tuples
                value = [(x, y) for x, y, in value]
        elif isinstance(value, MFVec2f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec2f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFVec2f(_X3DField):
    """
    MFVec2f is an array of SFVec2f values.
    """
    NAME = 'MFVec2f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2f'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 2
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 2-tuples
                value = [(x, y) for x, y, in value]
        elif not isinstance(value, list) and isValidSFVec2f(value):
            value = [value]
        assertValidMFVec2f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec2f(value):
                if isinstance(value, SFVec2f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec2f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec2f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec2f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFVec3d(_X3DField):
    """
    SFVec3d is a 3-tuple triplet of SFDouble values.
    """
    NAME = 'SFVec3d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3d'
    DEFAULT_VALUE = (0.0, 0.0, 0.0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 3
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFVec3d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec3d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFVec3d(_X3DField):
    """
    MFVec3d is an array of SFVec3d values.
    """
    NAME = 'MFVec3d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3d'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 3
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFVec3d(value):
            value = [value]
        assertValidMFVec3d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec3d(value):
                if isinstance(value, SFVec3d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec3d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec3d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec3d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFVec3f(_X3DField):
    """
    SFVec3f is a 3-tuple triplet of SFFloat values.
    """
    NAME = 'SFVec3f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3f'
    DEFAULT_VALUE = (0.0, 0.0, 0.0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 3
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif isinstance(value, MFVec3f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec3f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFVec3f(_X3DField):
    """
    MFVec3f is an array of SFVec3f values.
    """
    NAME = 'MFVec3f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3f'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 3
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif not isinstance(value, list) and isValidSFVec3f(value):
            value = [value]
        assertValidMFVec3f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec3f(value):
                if isinstance(value, SFVec3f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec3f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec3f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec3f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFVec4d(_X3DField):
    """
    SFVec4d is a 4-tuple set of double-precision floating-point values, specifying a 3D homogeneous vector.
    """
    NAME = 'SFVec4d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4d'
    DEFAULT_VALUE = (0.0, 0.0, 0.0, 0.0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 4
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, MFVec4d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec4d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFVec4d(_X3DField):
    """
    MFVec4d is zero or more SFVec4d values.
    """
    NAME = 'MFVec4d'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4d'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 4
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif not isinstance(value, list) and isValidSFVec4d(value):
            value = [value]
        assertValidMFVec4d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec4d(value):
                if isinstance(value, SFVec4d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec4d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec4d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec4d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

class SFVec4f(_X3DField):
    """
    SFVec4f is a 4-tuple set of single-precision floating-point values, specifying a 3D homogeneous vector.
    """
    NAME = 'SFVec4f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4f'
    DEFAULT_VALUE = (0.0, 0.0, 0.0, 0.0)
    ARRAY_TYPE = False
    TUPLE_SIZE = 4
    REGEX_XML = r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFVec4f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec4f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0

class MFVec4f(_X3DField):
    """
    MFVec4f is zero or more SFVec4f values.
    """
    NAME = 'MFVec4f'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4f'
    DEFAULT_VALUE = list()
    ARRAY_TYPE = True
    TUPLE_SIZE = 4
    REGEX_XML = r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif not isinstance(value, list) and isValidSFVec4f(value):
            value = [value]
        assertValidMFVec4f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec4f(value):
                if isinstance(value, SFVec4f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec4f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec4f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec4f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)

###############################################

# Abstract Node Types

# Note that these package-internal class names are preceded by an underscore _ character for hidden scope, since X3D authors are not expected to use them.
class _X3DNode(object):
    """
    All instantiable nodes implement X3DNode, which corresponds to SFNode in the X3D specification.
    """
    NAME = '_X3DNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF="", USE="", class_="", metadata=None, IS=None):
        self.DEF = DEF
        self.USE = USE
        self.class_ = class_
        self.IS = IS
        self.metadata = metadata
        # if _DEBUG: print('... in X3DNode __init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
    @property # getter - - - - - - - - - -
    def DEF(self):
        """ Unique ID name for this node, referenceable by other nodes. """
        return self.__DEF
    @DEF.setter
    def DEF(self, DEF=None):
        if  DEF is None:
            DEF = SFString.DEFAULT_VALUE
        assertValidSFString(DEF)
        self.__DEF = str(DEF)
        if self.__DEF:
            self.__USE = None # DEF and USE are mutually exclusive
    @property # getter - - - - - - - - - -
    def USE(self):
        """ Reuse an already DEF-ed node ID, excluding all child nodes and all other attributes. """
        return self.__USE
    @USE.setter
    def USE(self, USE=None):
        if  USE is None:
            USE = SFString.DEFAULT_VALUE
        assertValidSFString(USE)
        self.__USE = str(USE)
        if self.__USE:
            self.__DEF = None # DEF and USE are mutually exclusive
    @property # getter - - - - - - - - - -
    def class_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading stylesheets. """
        return self.__class_
    @class_.setter
    def class_(self, class_=None):
        if  class_ is None:
            class_ = SFString.DEFAULT_VALUE
        assertValidSFString(class_)
        self.__class_ = class_
    @property # getter - - - - - - - - - -
    def IS(self):
        """ The IS statement connects node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields. """
        return self.__IS
    @IS.setter
    def IS(self, IS=None):
        if  IS is None:
            IS = SFNode.DEFAULT_VALUE
        assertValidSFNode(IS)
        if not isinstance(IS, object):
            # print(flush=True)
            raise X3DTypeError(str(IS) + ' does not have a valid node type object')
        self.__IS = IS
    @property # getter - - - - - - - - - -
    def metadata(self):
        """ The metadata field can contain a single MetadataBoolean, MetadataInteger, MetadataFloat, MetadataDouble, MetadataString or MetadataSet node. """
        return self.__metadata
    @metadata.setter
    def metadata(self, metadata=None):
        if  metadata is None:
            metadata = SFNode.DEFAULT_VALUE
        assertValidSFNode(metadata)
        if not isinstance(metadata, object):
            # print(flush=True)
            raise X3DTypeError(str(metadata) + ' does not have a valid node type object')
        self.__metadata = metadata
    def __repl__(self):
        result = self.NAME + '('
        # TODO put DEF first, matching canonical form
        for each in self.FIELD_DECLARATIONS:
            # if _DEBUG: print(self.NAME + ' for each in self.FIELD_DECLARATIONS: each=' + str(each))
            name = each[0]
            default = each[1]
            type_ = each[2]
            accessType = each[3]
            value = getattr(self, name)
            # if _DEBUG: print('gettattr(self, ' + str(name) + ') value="' + str(value)[:100] + '" for FIELD_DECLARATIONS ' + str(each) + ')', flush=True)
            if value != default:
                # consider whether indentation is useful; probably not
                # print("\n\t")
                if  isinstance(value, list): # avoid X3DTypeError if value is not iterable
                    result += str(name) + '=['
                    for each in value:
                        # if _DEBUG: print('* X3DNode debug: str(each)=' + str(each), flush=True)
                        result += str(each) + ', '
                    result = result.rstrip(', ')
                    result += '],'
                elif isinstance(value, str) and "'" in value:
                    result += str(name) + '=' + '"' + str(value)[:100] + '"' + ','
                elif isinstance(value, str) and value != default:
                    result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
                elif value != default:
                    result += str(name) + '='       + str(value)[:100]       + ','
                # elif _DEBUG:
                #   result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
        return result.strip().rstrip(',').rstrip(', ') + ')'
    def __str__(self):
        return self.__repl__().strip() # X3DNode


def isX3DNode(value):
    """
    Whether or not value is a concrete node (Shape WorldInfo etc.) meaning any _X3DNode object.
    """
    return isinstance(value, _X3DNode)

class _X3DChildNode(_X3DNode):
    """
    A node that implements X3DChildNode is one of the legal children for a X3DGroupingNode parent.
    """
    NAME = '_X3DChildNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DChildNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTimeDependentNode(_X3DChildNode):
    """
    Base type from which all time-dependent nodes are derived.
    """
    NAME = '_X3DTimeDependentNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTimeDependentNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DGeometryNode(_X3DNode):
    """
    Geometry nodes produce renderable geometry and are contained by a Shape node.
    """
    NAME = '_X3DGeometryNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DParametricGeometryNode(_X3DGeometryNode):
    """
    Base type for all geometry node types that are created parametrically and use control points to describe the final shape of the surface.
    """
    NAME = '_X3DParametricGeometryNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DParametricGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DAppearanceChildNode(_X3DNode):
    """
    Nodes of this type can be used as child nodes for Appearance.
    """
    NAME = '_X3DAppearanceChildNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DAppearanceChildNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTextureNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes which specify sources for texture images.
    """
    NAME = '_X3DTextureNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTextureNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSensorNode(_X3DChildNode):
    """
    Base type for all sensors.
    """
    NAME = '_X3DSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPointingDeviceSensorNode(_X3DSensorNode):
    """
    Base type for all pointing device sensors.
    """
    NAME = '_X3DPointingDeviceSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DPointingDeviceSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DVolumeRenderStyleNode(_X3DNode):
    """
    The X3DVolumeRenderStyleNode abstract node type is the base type for all node types that specify a specific visual rendering style to be used when rendering volume data.
    """
    NAME = '_X3DVolumeRenderStyleNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DVolumeRenderStyleNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DGeometricPropertyNode(_X3DNode):
    """
    Base type for all geometric property node types.
    """
    NAME = '_X3DGeometricPropertyNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DGeometricPropertyNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DFollowerNode(_X3DChildNode):
    """
    X3DFollowerNode is the abstract base class for all nodes in the Followers component.
    """
    NAME = '_X3DFollowerNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DFollowerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DBindableNode(_X3DChildNode):
    """
    Bindable nodes implement the binding stack, so that only one of each node type is active at a given time.
    """
    NAME = '_X3DBindableNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DBindableNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DAppearanceNode(_X3DNode):
    """
    Base type for all Appearance nodes.
    """
    NAME = '_X3DAppearanceNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DAppearanceNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DBackgroundNode(_X3DBindableNode):
    """
    Abstract type from which all backgrounds inherit, also defining a background binding stack.
    """
    NAME = '_X3DBackgroundNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DBackgroundNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DChaserNode(_X3DFollowerNode):
    """
    The X3DChaserNode abstract node type calculates the output on value_changed as a finite impulse response (FIR) based on the events received on set_destination field.
    """
    NAME = '_X3DChaserNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DChaserNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DColorNode(_X3DGeometricPropertyNode):
    """
    Base type for color specifications in X3D.
    """
    NAME = '_X3DColorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DColorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DComposableVolumeRenderStyleNode(_X3DVolumeRenderStyleNode):
    """
    The X3DComposableVolumeRenderStyleNode abstract node type is the base type for all node types that allow rendering styles to be sequentially composed together to form a single renderable output.
    """
    NAME = '_X3DComposableVolumeRenderStyleNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DComposableVolumeRenderStyleNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DComposedGeometryNode(_X3DGeometryNode):
    """
    Composed geometry nodes produce renderable geometry, can contain Color Coordinate Normal TextureCoordinate, and are contained by a Shape node.
    """
    NAME = '_X3DComposedGeometryNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DComposedGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DCoordinateNode(_X3DGeometricPropertyNode):
    """
    Base type for all coordinate node types in X3D.
    """
    NAME = '_X3DCoordinateNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DCoordinateNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DDamperNode(_X3DFollowerNode):
    """
    The X3DDamperNode abstract node type creates an IIR response that approaches the destination value according to the shape of the e-function only asymptotically but very quickly.
    """
    NAME = '_X3DDamperNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DDamperNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DDragSensorNode(_X3DPointingDeviceSensorNode):
    """
    Base type for all drag-style pointing device sensors.
    """
    NAME = '_X3DDragSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DDragSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DEnvironmentalSensorNode(_X3DSensorNode):
    """
    Base type for the environmental sensor nodes ProximitySensor, TransformSensor and VisibilitySensor.
    """
    NAME = '_X3DEnvironmentalSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DEnvironmentalSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DEnvironmentTextureNode(_X3DTextureNode):
    """
    Base type for all nodes that specify cubic environment map sources for texture images.
    """
    NAME = '_X3DEnvironmentTextureNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DEnvironmentTextureNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DFontStyleNode(_X3DNode):
    """
    Base type for all font style nodes.
    """
    NAME = '_X3DFontStyleNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DFontStyleNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DGroupingNode(_X3DChildNode):
    """
    Grouping nodes can contain other nodes as children, thus making up the backbone of a scene graph.
    """
    NAME = '_X3DGroupingNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DGroupingNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DInfoNode(_X3DChildNode):
    """
    Base type for all nodes that contain only information without visual semantics.
    """
    NAME = '_X3DInfoNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DInfoNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DInterpolatorNode(_X3DChildNode):
    """
    Interpolator nodes are designed for linear keyframed animation. Interpolators are driven by an input key ranging [0..1] and produce corresponding piecewise-linear output functions.
    """
    NAME = '_X3DInterpolatorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DInterpolatorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DKeyDeviceSensorNode(_X3DSensorNode):
    """
    Base type for all sensor node types that operate using key devices.
    """
    NAME = '_X3DKeyDeviceSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DKeyDeviceSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DLayerNode(_X3DNode):
    """
    The X3DLayerNode abstract node type is the base node type for layer nodes.
    """
    NAME = '_X3DLayerNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DLayerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DLayoutNode(_X3DChildNode):
    """
    X3DLayoutNode is the base node type for layout nodes.
    """
    NAME = '_X3DLayoutNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DLayoutNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DLightNode(_X3DChildNode):
    """
    Light nodes provide illumination for rendering geometry in the scene. Implementing nodes must include a global field with type SFBool and accessType inputOutput.
    """
    NAME = '_X3DLightNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DLightNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DMaterialNode(_X3DAppearanceChildNode):
    """
    Base type for all Material nodes.
    """
    NAME = '_X3DMaterialNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DMaterialNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNBodyCollidableNode(_X3DChildNode):
    """
    The X3DNBodyCollidableNode abstract node type represents objects that act as the interface between the rigid body physics, collision geometry proxy, and renderable objects in the scene graph hierarchy.
    """
    NAME = '_X3DNBodyCollidableNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNBodyCollidableNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNBodyCollisionSpaceNode(_X3DNode):
    """
    The X3DNBodyCollisionSpaceNode abstract node type represents objects that act as a self-contained spatial collection of objects that can interact through collision detection routines.
    """
    NAME = '_X3DNBodyCollisionSpaceNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNBodyCollisionSpaceNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNetworkSensorNode(_X3DSensorNode):
    """
    Base typefor all sensors that generate events based on network activity.
    """
    NAME = '_X3DNetworkSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNetworkSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNormalNode(_X3DGeometricPropertyNode):
    """
    Base type for all normal node types in X3D.
    """
    NAME = '_X3DNormalNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNormalNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNurbsControlCurveNode(_X3DNode):
    """
    Base type for all nodes that provide control curve information in 2D space.
    """
    NAME = '_X3DNurbsControlCurveNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNurbsControlCurveNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNurbsSurfaceGeometryNode(_X3DParametricGeometryNode):
    """
    Abstract geometry type for all types of NURBS surfaces.
    """
    NAME = '_X3DNurbsSurfaceGeometryNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNurbsSurfaceGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DParticleEmitterNode(_X3DNode):
    """
    The X3DParticleEmitterNode abstract type represents any node that is an emitter of particles.
    """
    NAME = '_X3DParticleEmitterNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DParticleEmitterNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DParticlePhysicsModelNode(_X3DNode):
    """
    The X3DParticlePhysicsModelNode abstract type represents any node that applies a form of constraints on the particles after they have been generated.
    """
    NAME = '_X3DParticlePhysicsModelNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DParticlePhysicsModelNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPickSensorNode(_X3DSensorNode):
    """
    The X3DPickSensorNode abstract node type is the base node type that represents the lowest common denominator of picking capabilities.
    """
    NAME = '_X3DPickSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DPickSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DProductStructureChildNode(_X3DChildNode):
    """
    Base type marking nodes that are valid product structure children for the CADGeometry component.
    """
    NAME = '_X3DProductStructureChildNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DProductStructureChildNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPrototypeInstance(_X3DNode):
    """
    Base type for all prototype instances. Note that direct children nodes are disallowed, instead let fieldValue with type SFNode/MFNode contain them. Current practice is that, if desired, prototype authors must explicitly add the metadata SFNode field in the ProtoInterface.
    """
    NAME = '_X3DPrototypeInstance'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DPrototypeInstance __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DRigidJointNode(_X3DNode):
    """
    The X3DRigidJointNode abstract node type is the base type for all joint types.
    """
    NAME = '_X3DRigidJointNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DRigidJointNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DScriptNode(_X3DChildNode):
    """
    Base type for scripting nodes (but not shader nodes).
    """
    NAME = '_X3DScriptNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DScriptNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSequencerNode(_X3DChildNode):
    """
    Base type from which all Sequencers are derived.
    """
    NAME = '_X3DSequencerNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSequencerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DShaderNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes that specify a programmable shader.
    """
    NAME = '_X3DShaderNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DShaderNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DShapeNode(_X3DChildNode):
    """
    Base type for all Shape nodes.
    """
    NAME = '_X3DShapeNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DShapeNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSoundNode(_X3DChildNode):
    """
    Base type for all sound nodes.
    """
    NAME = '_X3DSoundNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSoundNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSoundSourceNode(_X3DTimeDependentNode):
    """
    Nodes implementing X3DSoundSourceNode are allowed as children of Sound node.
    """
    NAME = '_X3DSoundSourceNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSoundSourceNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTexture2DNode(_X3DTextureNode):
    """
    Base type for all nodes which specify 2D sources for texture images.
    """
    NAME = '_X3DTexture2DNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTexture2DNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTexture3DNode(_X3DTextureNode):
    """
    Base type for all nodes that specify 3D sources for texture images.
    """
    NAME = '_X3DTexture3DNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTexture3DNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTextureCoordinateNode(_X3DGeometricPropertyNode):
    """
    Base type for all nodes which specify texture coordinates.
    """
    NAME = '_X3DTextureCoordinateNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTextureCoordinateNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTextureTransformNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes which specify a transformation of texture coordinates.
    """
    NAME = '_X3DTextureTransformNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTextureTransformNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTouchSensorNode(_X3DPointingDeviceSensorNode):
    """
    Base type for all touch-style pointing device sensors.
    """
    NAME = '_X3DTouchSensorNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTouchSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTriggerNode(_X3DChildNode):
    """
    Base type from which all trigger nodes are derived.
    """
    NAME = '_X3DTriggerNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTriggerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DVertexAttributeNode(_X3DGeometricPropertyNode):
    """
    Base type for all nodes that specify per-vertex attribute information to the shader.
    """
    NAME = '_X3DVertexAttributeNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DVertexAttributeNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DViewpointNode(_X3DBindableNode):
    """
    Node type X3DViewpointNode defines a specific location in the local coordinate system from which the user may view the scene, and also defines a viewpoint binding stack.
    """
    NAME = '_X3DViewpointNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DViewpointNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DViewportNode(_X3DGroupingNode):
    """
    The X3DViewportNode abstract node type is the base node type for viewport nodes.
    """
    NAME = '_X3DViewportNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DViewportNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DVolumeDataNode(_X3DChildNode):
    """
    The X3DVolumeDataNode abstract node type is the base type for all node types that describe volumetric data to be rendered.
    """
    NAME = '_X3DVolumeDataNode'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DVolumeDataNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

###############################################

# Abstract Object Types

# Note that these package-internal class names are preceded by an underscore _ character since X3D authors are not expected to use them

class _X3DBoundedObject(_X3DNode):
    """
    X3DBoundedObject indicates that bounding box values can be provided (or computed) to encompass this node and any children.
    """
    NAME = '_X3DBoundedObject'
    SPECIFICATION_URL = ''

class _X3DFogObject(_X3DNode):
    """
    Abstract type describing a node that influences the lighting equation through the use of fog semantics.
    """
    NAME = '_X3DFogObject'
    SPECIFICATION_URL = ''

class _X3DMetadataObject(_X3DNode):
    """
    Each node inheriting the X3DMetadataObject interface contains a single array of strictly typed values: MFBool, MFInt32, MFFloat, MFDouble, MFString, or MFNode, the latter having children that are all Metadata nodes.
    """
    NAME = '_X3DMetadataObject'
    SPECIFICATION_URL = ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractObjectType X3DMetadataObject __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPickableObject(_X3DNode):
    """
    The X3DPickableObject abstract interface marks a node as being capable of having customized picking performed on its contents or children.
    """
    NAME = '_X3DPickableObject'
    SPECIFICATION_URL = ''

class _X3DProgrammableShaderObject(_X3DNode):
    """
    Base type for all nodes that specify arbitrary fields for interfacing with per-object attribute values.
    """
    NAME = '_X3DProgrammableShaderObject'
    SPECIFICATION_URL = ''

class _X3DUrlObject(_X3DNode):
    """
    X3DUrlObject indicates that a node has content loaded from a Uniform Resource Locator (URL) and can be tracked via a LoadSensor. Such child nodes have containerField='watchList' to indicate their relationship to the parent LoadSensor node.
    """
    NAME = '_X3DUrlObject'
    SPECIFICATION_URL = ''

###############################################

# Statements

class _X3DStatement(object):
    """
    All X3D statements implement _X3DStatement abstract type.
    """
    NAME = '_X3DStatement'
    SPECIFICATION_URL = 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/components/core.html#AbstractX3DStructure'
    def __repl__(self):
        result = self.NAME + '('
        # if _DEBUG: print(self.NAME + ' self.FIELD_DECLARATIONS: ' + str(self.FIELD_DECLARATIONS))
        for each in self.FIELD_DECLARATIONS:
            # if _DEBUG: print(self.NAME + ' for each in self.FIELD_DECLARATIONS: each=' + str(each))
            name = each[0]
            default = each[1]
            type_ = each[2]
            accessType = each[3]
            value = getattr(self, name)
            # if _DEBUG: print('gettattr(self, ' + str(name) + ') value="' + str(value)[:100] + '" for FIELD_DECLARATIONS ' + str(each) + ')', flush=True)
            if value != default:
                if  isinstance(value, list): # avoid X3DTypeError if value is not iterable
                    result += str(name) + '=['
                    for each in value:
                        result += str(each) + ', '
                        # if _DEBUG: print('* _X3DStatement debug: str(each)=' + str(each), flush=True)
                    result = result.rstrip(', ')
                    result += '],'
                elif isinstance(value, str) and "'" in value:
                    result += str(name) + '=' + '"' + str(value)[:100] + '"' + ','
                elif isinstance(value, str) and value != default:
                    result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
                elif value != default:
                    result += str(name) + '='       + str(value)[:100]       + ','
                # elif _DEBUG:
                #   result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
        return result.strip().rstrip(',').rstrip(', ') + ')'
    def __str__(self):
        return self.__repl__().strip() # _X3DStatement

def isX3DStatement(value):
    """
    Whether or not value is an _X3DStatement object.
    """
    return isinstance(value, _X3DStatement)

class Comment(_X3DStatement):
    """
    X3D statement containing zero or more comment strings.
    """
    NAME = 'Comment'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html'
    DEFAULT_VALUE = list()
    REGEX_XML = r'(\s|\S)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide list of comment strings. """
        return self.__value
    @value.setter
    def value(self, value=None):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE
        assertValidMFString(value)
        self.__value = value

def isComment(value):
    """
    Whether or not value is an _X3DComment object.
    """
    return isinstance(value, _X3DComment)

class component(_X3DStatement):
    """
    Functional summary: each added component statement indicates needed scene functionality support above the given X3D profile.
    """
    NAME = 'component'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#component'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('level', 1, FieldType.SFInt32, AccessType.inputOutput, 'component'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'component')]
    def __init__(self,
                 level=1,
                 name=''):
        self.level = level
        self.name = name
    @property # getter - - - - - - - - - -
    def level(self):
        """Necessary level of support for this scene, as defined in corresponding Support table for a given node's component."""
        return self.__level
    @level.setter
    def level(self, level=None):
        if  level is None:
            level = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(level)
        assertGreaterThanEquals('level', level, 1)
        assertLessThanEquals('level', level, 5)
        self.__level = level
    @property # getter - - - - - - - - - -
    def name(self):
        """Provides name of this component, as defined in corresponding X3D Specification component Introduction."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function component.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<component'
        if self.level != 1:
            result += " level='" + str(self.level) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            result += indent + '</component>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class connect(_X3DStatement):
    """
    Functional summary: connect statements define event-routing connections between node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields.
    """
    NAME = 'connect'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#connect'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('nodeField', '', FieldType.SFString, AccessType.inputOutput, 'connect'),
        ('protoField', '', FieldType.SFString, AccessType.inputOutput, 'connect')]
    def __init__(self,
                 nodeField='',
                 protoField=''):
        self.nodeField = nodeField
        self.protoField = protoField
    @property # getter - - - - - - - - - -
    def nodeField(self):
        """Name of field within this node which IS CONNECTed to the ancestor ProtoDeclare field definition."""
        return self.__nodeField
    @nodeField.setter
    def nodeField(self, nodeField=None):
        if  nodeField is None:
            nodeField = SFString.DEFAULT_VALUE
        assertValidSFString(nodeField)
        self.__nodeField = nodeField
    @property # getter - - - - - - - - - -
    def protoField(self):
        """Name of parent ProtoDeclare field definition connecting to field in this node."""
        return self.__protoField
    @protoField.setter
    def protoField(self, protoField=None):
        if  protoField is None:
            protoField = SFString.DEFAULT_VALUE
        assertValidSFString(protoField)
        self.__protoField = protoField
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function connect.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<connect'
        if self.nodeField:
            result += " nodeField='" + self.nodeField + "'"
        if self.protoField:
            result += " protoField='" + self.protoField + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            result += indent + '</connect>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class EXPORT(_X3DStatement):
    """
    Functional summary: EXPORT exposes a local node for ROUTE passing of event values when the current Scene is included via Inline by a parent external world. These connections allow event values to be exchanged via ROUTE statements between a parent model and a child Inline model.
    """
    NAME = 'EXPORT'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EXPORT'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('AS', '', FieldType.SFString, AccessType.inputOutput, 'EXPORT'),
        ('localDEF', '', FieldType.SFString, AccessType.inputOutput, 'EXPORT')]
    def __init__(self,
                 AS='',
                 localDEF=''):
        self.AS = AS
        self.localDEF = localDEF
    @property # getter - - - - - - - - - -
    def AS(self):
        """rename localDEF node AS a different name when exporting."""
        return self.__AS
    @AS.setter
    def AS(self, AS=None):
        if  AS is None:
            AS = SFString.DEFAULT_VALUE
        assertValidSFString(AS)
        self.__AS = AS
    @property # getter - - - - - - - - - -
    def localDEF(self):
        """localDEF is the DEF name of the local node to be EXPORTed."""
        return self.__localDEF
    @localDEF.setter
    def localDEF(self, localDEF=None):
        if  localDEF is None:
            localDEF = SFString.DEFAULT_VALUE
        assertValidSFString(localDEF)
        self.__localDEF = localDEF
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EXPORT.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EXPORT'
        if self.AS:
            result += " AS='" + self.AS + "'"
        if self.localDEF:
            result += " localDEF='" + self.localDEF + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            result += indent + '</EXPORT>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ExternProtoDeclare(_X3DStatement):
    """
    ExternProtoDeclare refers to a ProtoDeclare node declaration provided in another file. ExternProtoDeclare interfaces are defined by field statements (and without IS/connect statements).
    """
    NAME = 'ExternProtoDeclare'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ExternProtoDeclare'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('field', list(), FieldType.MFNode, AccessType.inputOutput, 'ExternProtoDeclare')]
    def __init__(self,
                 appinfo='',
                 documentation='',
                 name='',
                 url=list(),
                 field=list()):
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.url = url
        self.field = field
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo=None):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation=None):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """name of the ExternProtoDeclare (External Prototype Declaration) being referenced."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def url(self):
        """Location and filename of ProtoDeclare source declaration of interest."""
        return self.__url
    @url.setter
    def url(self, url=None):
        if  url is None:
            url = MFString.DEFAULT_VALUE
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in the corresponding original ProtoDeclare."""
        return self.__field
    @field.setter
    def field(self, field=None):
        if  field is None:
            field = MFNode.DEFAULT_VALUE
        # TODO type-aware checks for field
        if field: # walk each child node, if any
            for each in field:
                assertValidFieldInitializationValue(each.name, type(each.value), each.value, parent='ExternProtoDeclare/field')
        self.__field = field
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.field)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ExternProtoDeclare.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ExternProtoDeclare'
        if self.appinfo:
            result += " appinfo='" + self.appinfo + "'"
        if self.documentation:
            result += " documentation='" + self.documentation + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.url != list():
            result += " url='" + self.url + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.field: # walk each child node, if any
                for each in self.field:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ExternProtoDeclare>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class field(_X3DStatement):
    """
    Functional summary: a field statement defines an interface attribute or node. Each field statement can contain either attribute-value or node content.
    """
    NAME = 'field'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#field'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('accessType', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('type', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('value', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'field')]
    def __init__(self,
                 accessType='',
                 appinfo='',
                 documentation='',
                 name='',
                 type='',
                 value='',
                 children=list()):
        self.accessType = accessType
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.type = type
        self.value = value
        self.children = children
    @property # getter - - - - - - - - - -
    def accessType(self):
        """Event-model semantics for field set/get capabilities."""
        return self.__accessType
    @accessType.setter
    def accessType(self, accessType=None):
        if  accessType is None:
            accessType = SFString.DEFAULT_VALUE
        assertValidSFString(accessType)
        self.__accessType = accessType
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo=None):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation=None):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """Name of this field declaration."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def type(self):
        """Base type of this field variable."""
        return self.__type
    @type.setter
    def type(self, type=None):
        if  type is None:
            type = SFString.DEFAULT_VALUE
        assertValidSFString(type)
        self.__type = type
    @property # getter - - - - - - - - - -
    def value(self):
        """Provide default initialization value for this field variable (which may be re-initialized later by instantiation value of a named ProtoInstance fieldValue)."""
        return self.__value
    @value.setter
    def value(self, value=None):
        if  value is None:
            value = SFString.DEFAULT_VALUE
        assertValidFieldInitializationValue(self.name, self.type, value, parent='field/@value')
        self.__value = value
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function field.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<field'
        if self.accessType:
            result += " accessType='" + self.accessType + "'"
        if self.appinfo:
            result += " appinfo='" + self.appinfo + "'"
        if self.documentation:
            result += " documentation='" + self.documentation + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.type:
            result += " type='" + self.type + "'"
        if self.value:
            result += " value='" + self.value + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</field>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class fieldValue(_X3DStatement):
    """
    Functional summary: a fieldValue statement re-initializes the default value of a field in a ProtoInstance. Each fieldValue statement can contain either attribute-value or node content.
    """
    NAME = 'fieldValue'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#fieldValue'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'fieldValue'),
        ('value', '', FieldType.SFString, AccessType.inputOutput, 'fieldValue'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'fieldValue')]
    def __init__(self,
                 name='',
                 value='',
                 children=list()):
        self.name = name
        self.value = value
        self.children = children
    @property # getter - - - - - - - - - -
    def name(self):
        """Name of the ProtoInstance field being re-initialized (corresponding to field name already defined in ProtoDeclare or ExternProtoDeclare)."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def value(self):
        """Initial value for this field, which overrides default initialization value defined in original ProtoDeclare field."""
        return self.__value
    @value.setter
    def value(self, value=None):
        if  value is None:
            value = SFString.DEFAULT_VALUE
        assertValidFieldInitializationValue(self.name, type(value), value, parent='fieldValue')
        self.__value = value
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function fieldValue.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<fieldValue'
        if self.name:
            result += " name='" + self.name + "'"
        if self.value:
            result += " value='" + self.value + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</fieldValue>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class head(_X3DStatement):
    """
    Functional summary: each X3D scene includes a head statement that can contain component, unit and meta statements.
    """
    NAME = 'head'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#head'
    FIELD_DECLARATIONS = [('children', None, FieldType.MFNode, AccessType.inputOutput, 'head')]
    def __init__(self, children=None):
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        """ The head statement has children consisting of component, unit and meta statements. """
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function head.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<head'
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                ## print('* head found self.children, now invoking toXML(' + str(indentLevel+1) + ')', flush=True)
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</head>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class IMPORT(_X3DStatement):
    """
    Functional summary: IMPORT provides ROUTE access to a node that has a corresponding EXPORT statement within an Inline scene. These connections allow event values to be exchanged via ROUTE statements between a parent model and a child Inline model.
    """
    NAME = 'IMPORT'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#IMPORT'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('AS', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT'),
        ('importedDEF', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT'),
        ('inlineDEF', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT')]
    def __init__(self,
                 AS='',
                 importedDEF='',
                 inlineDEF=''):
        self.AS = AS
        self.importedDEF = importedDEF
        self.inlineDEF = inlineDEF
    @property # getter - - - - - - - - - -
    def AS(self):
        """map importedDEF name AS a new name in current scene."""
        return self.__AS
    @AS.setter
    def AS(self, AS=None):
        if  AS is None:
            AS = SFString.DEFAULT_VALUE
        assertValidSFString(AS)
        self.__AS = AS
    @property # getter - - - - - - - - - -
    def importedDEF(self):
        """importedDEF is DEF name of the node of interest that is contained in the remote inlineDEF scene."""
        return self.__importedDEF
    @importedDEF.setter
    def importedDEF(self, importedDEF=None):
        if  importedDEF is None:
            importedDEF = SFString.DEFAULT_VALUE
        assertValidSFString(importedDEF)
        self.__importedDEF = importedDEF
    @property # getter - - - - - - - - - -
    def inlineDEF(self):
        """inlineDEF is the DEF name of Inline node in the same scene as this IMPORT statement."""
        return self.__inlineDEF
    @inlineDEF.setter
    def inlineDEF(self, inlineDEF=None):
        if  inlineDEF is None:
            inlineDEF = SFString.DEFAULT_VALUE
        assertValidSFString(inlineDEF)
        self.__inlineDEF = inlineDEF
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IMPORT.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<IMPORT'
        if self.AS:
            result += " AS='" + self.AS + "'"
        if self.importedDEF:
            result += " importedDEF='" + self.importedDEF + "'"
        if self.inlineDEF:
            result += " inlineDEF='" + self.inlineDEF + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            result += indent + '</IMPORT>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class IS(_X3DStatement):
    """
    Functional summary: the IS statement connects node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields. IS/connect statements can be added if the parent node is within a ProtoBody and connect statements define correspondences between prototype fields and built-in node fields.
    """
    NAME = 'IS'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#IS'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('connect', list(), FieldType.MFNode, AccessType.inputOutput, 'IS')]
    def __init__(self,
                 connect=list()):
        self.connect = connect
    @property # getter - - - - - - - - - -
    def connect(self):
        """When inside a ProtoBody declaration and an IS statement, add a connect statement to define event-routing connections between a parent node's field to a corresponding ProtoInterface field."""
        return self.__connect
    @connect.setter
    def connect(self, connect=None):
        if  connect is None:
            connect = MFNode.DEFAULT_VALUE
        assertValidMFNode(connect)
        self.__connect = connect
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.connect)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IS.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<IS'
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.connect: # walk each child node, if any
                for each in self.connect:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</IS>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class meta(_X3DStatement):
    """
    Functional summary: the meta statement provides metadata information about a scene, where name and content attributes provide attribute=value metadata pairs.
    """
    NAME = 'meta'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#meta'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('content', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('dir', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('httpequiv', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('lang', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('scheme', '', FieldType.SFString, AccessType.inputOutput, 'meta')]
    def __init__(self,
                 content='',
                 dir='',
                 httpequiv='',
                 lang='',
                 name='',
                 scheme=''):
        self.content = content
        self.dir = dir
        self.httpequiv = httpequiv
        self.lang = lang
        self.name = name
        self.scheme = scheme
    @property # getter - - - - - - - - - -
    def content(self):
        """The content attribute provides metadata information relevant to the name attribute provided."""
        return self.__content
    @content.setter
    def content(self, content=None):
        if  content is None:
            content = SFString.DEFAULT_VALUE
        assertValidSFString(content)
        self.__content = content
    @property # getter - - - - - - - - - -
    def dir(self):
        """Direction for weak/neutral text (ltr=left-to-right, rtl=right-to-left)."""
        return self.__dir
    @dir.setter
    def dir(self, dir=None):
        if  dir is None:
            dir = SFString.DEFAULT_VALUE
        assertValidSFString(dir)
        self.__dir = dir
    @property # getter - - - - - - - - - -
    def httpequiv(self):
        return self.__httpequiv
    @httpequiv.setter
    def httpequiv(self, httpequiv=None):
        if  httpequiv is None:
            httpequiv = SFString.DEFAULT_VALUE
        assertValidSFString(httpequiv)
        self.__httpequiv = httpequiv
    @property # getter - - - - - - - - - -
    def lang(self):
        """Language code, as per [IETF BCP47/RFC5646] http://www."""
        return self.__lang
    @lang.setter
    def lang(self, lang=None):
        if  lang is None:
            lang = SFString.DEFAULT_VALUE
        assertValidSFString(lang)
        self.__lang = lang
    @property # getter - - - - - - - - - -
    def name(self):
        """Keyword name of the meta attribute, following the same naming conventions as HTML's meta tag."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def scheme(self):
        """The scheme attribute allows authors to provide user agents more context for the correct interpretation of meta information."""
        return self.__scheme
    @scheme.setter
    def scheme(self, scheme=None):
        if  scheme is None:
            scheme = SFString.DEFAULT_VALUE
        assertValidSFString(scheme)
        self.__scheme = scheme
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function meta.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<meta'
        if self.content:
            result += " content='" + self.content + "'"
        if self.dir:
            result += " dir='" + self.dir + "'"
        if self.httpequiv:
            result += " httpequiv='" + self.httpequiv + "'"
        if self.lang:
            result += " lang='" + self.lang + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.scheme:
            result += " scheme='" + self.scheme + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            result += indent + '</meta>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ProtoBody(_X3DStatement):
    """
    ProtoBody contains the definition nodes for new Prototype nodes.
    """
    NAME = 'ProtoBody'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoBody'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'ProtoBody')]
    def __init__(self,
                 children=list()):
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoBody.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoBody'
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ProtoBody>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ProtoDeclare(_X3DStatement):
    """
    ProtoDeclare defines new Prototype nodes. Nested ProtoDeclares and ProtoInstances are allowed by the specification.
    """
    NAME = 'ProtoDeclare'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoDeclare'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('ProtoBody', None, FieldType.SFNode, AccessType.inputOutput, 'ProtoDeclare'),
        ('ProtoInterface', None, FieldType.SFNode, AccessType.inputOutput, 'ProtoDeclare')]
    def __init__(self,
                 appinfo='',
                 documentation='',
                 name='',
                 ProtoBody=None,
                 ProtoInterface=None):
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.ProtoBody = ProtoBody
        self.ProtoInterface = ProtoInterface
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo=None):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation=None):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """name of this prototype being declared."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def ProtoBody(self):
        """Include one ProtoBody statements after the ProtoInterface statement."""
        return self.__ProtoBody
    @ProtoBody.setter
    def ProtoBody(self, ProtoBody=None):
        if  ProtoBody is None:
            ProtoBody = SFNode.DEFAULT_VALUE
        assertValidSFNode(ProtoBody)
        if not isinstance(ProtoBody, object):
            # print(flush=True)
            raise X3DTypeError(str(ProtoBody) + ' does not have a valid node type object')
        self.__ProtoBody = ProtoBody
    @property # getter - - - - - - - - - -
    def ProtoInterface(self):
        """Include an optional ProtoInterface statement if this ProtoDeclare has field declarations."""
        return self.__ProtoInterface
    @ProtoInterface.setter
    def ProtoInterface(self, ProtoInterface=None):
        if  ProtoInterface is None:
            ProtoInterface = SFNode.DEFAULT_VALUE
        assertValidSFNode(ProtoInterface)
        if not isinstance(ProtoInterface, object):
            # print(flush=True)
            raise X3DTypeError(str(ProtoInterface) + ' does not have a valid node type object')
        self.__ProtoInterface = ProtoInterface
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.ProtoBody) or bool(self.ProtoInterface)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoDeclare.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoDeclare'
        if self.appinfo:
            result += " appinfo='" + self.appinfo + "'"
        if self.documentation:
            result += " documentation='" + self.documentation + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.ProtoBody: # walk each child node, if any
                for each in self.ProtoBody:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.ProtoInterface: # walk each child node, if any
                for each in self.ProtoInterface:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ProtoDeclare>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ProtoInterface(_X3DStatement):
    """
    ProtoInterface defines fields for new Prototype nodes.
    """
    NAME = 'ProtoInterface'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoInterface'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('field', list(), FieldType.MFNode, AccessType.inputOutput, 'ProtoInterface')]
    def __init__(self,
                 field=list()):
        self.field = field
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in this ProtoDeclare's ProtoInterface."""
        return self.__field
    @field.setter
    def field(self, field=None):
        if  field is None:
            field = MFNode.DEFAULT_VALUE
        # TODO type-aware checks for field
        if field: # walk each child node, if any
            for each in field:
                assertValidFieldInitializationValue(each.name, each.type, each.value, parent='ProtoInterface')
        self.__field = field
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.field)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoInterface.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoInterface'
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.field: # walk each child node, if any
                for each in self.field:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ProtoInterface>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ROUTE(_X3DStatement):
    """
    ROUTE connects output fields of event-producing nodes to input fields of event-consuming nodes.
    """
    NAME = 'ROUTE'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ROUTE'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('fromField', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('fromNode', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('toField', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('toNode', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE')]
    def __init__(self,
                 fromField='',
                 fromNode='',
                 toField='',
                 toNode=''):
        self.fromField = fromField
        self.fromNode = fromNode
        self.toField = toField
        self.toNode = toNode
    @property # getter - - - - - - - - - -
    def fromField(self):
        """fromField is the field name in the source node which is originating an event."""
        return self.__fromField
    @fromField.setter
    def fromField(self, fromField=None):
        if  fromField is None:
            fromField = SFString.DEFAULT_VALUE
        assertValidSFString(fromField)
        self.__fromField = fromField
    @property # getter - - - - - - - - - -
    def fromNode(self):
        """fromNode is the DEF name of the node originating an event."""
        return self.__fromNode
    @fromNode.setter
    def fromNode(self, fromNode=None):
        if  fromNode is None:
            fromNode = SFString.DEFAULT_VALUE
        assertValidSFString(fromNode)
        self.__fromNode = fromNode
    @property # getter - - - - - - - - - -
    def toField(self):
        """toField is the field name in the destination node which is receiving an event."""
        return self.__toField
    @toField.setter
    def toField(self, toField=None):
        if  toField is None:
            toField = SFString.DEFAULT_VALUE
        assertValidSFString(toField)
        self.__toField = toField
    @property # getter - - - - - - - - - -
    def toNode(self):
        """toNode is the DEF name of the destination node receiving an event."""
        return self.__toNode
    @toNode.setter
    def toNode(self, toNode=None):
        if  toNode is None:
            toNode = SFString.DEFAULT_VALUE
        assertValidSFString(toNode)
        self.__toNode = toNode
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ROUTE.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ROUTE'
        if self.fromField:
            result += " fromField='" + self.fromField + "'"
        if self.fromNode:
            result += " fromNode='" + self.fromNode + "'"
        if self.toField:
            result += " toField='" + self.toField + "'"
        if self.toNode:
            result += " toNode='" + self.toNode + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            result += indent + '</ROUTE>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Scene(_X3DStatement):
    """
    Scene is the implicit root node of the X3D scene graph.
    """
    NAME = 'Scene'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Scene'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'Scene')]
    def __init__(self,
                 children=list()):
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Scene.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Scene'
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Scene>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class unit(_X3DStatement):
    """
    Functional summary: unit statement defines data-conversion factors for typed values defined in a scene.
    """
    NAME = 'unit'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#unit'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('category', '', FieldType.SFString, AccessType.initializeOnly, 'unit'),
        ('conversionFactor', 1.0, FieldType.SFDouble, AccessType.inputOutput, 'unit'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'unit')]
    def __init__(self,
                 category='',
                 conversionFactor=1.0,
                 name=''):
        self.category = category
        self.conversionFactor = conversionFactor
        self.name = name
    @property # getter - - - - - - - - - -
    def category(self):
        """Base-unit category as defined in X3D Specification."""
        return self.__category
    @category.setter
    def category(self, category=None):
        if  category is None:
            category = SFString.DEFAULT_VALUE
        assertValidSFString(category)
        self.__category = category
    @property # getter - - - - - - - - - -
    def conversionFactor(self):
        """[0,+infinity) Positive double-precision factor that converts new base unit to default base unit."""
        return self.__conversionFactor
    @conversionFactor.setter
    def conversionFactor(self, conversionFactor=None):
        if  conversionFactor is None:
            conversionFactor = SFDouble.DEFAULT_VALUE
        assertValidSFDouble(conversionFactor)
        assertPositive('conversionFactor', conversionFactor)
        self.__conversionFactor = conversionFactor
    @property # getter - - - - - - - - - -
    def name(self):
        """Author-defined name for this unit conversionFactor value (for example, FeetToMeters)."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function unit.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<unit'
        if self.category:
            result += " category='" + self.category + "'"
        if self.conversionFactor != 1.0:
            result += " conversionFactor='" + str(self.conversionFactor) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            result += indent + '</unit>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class X3D(_X3DNode):
    """
    X3D is the root node for an Extensible 3D (X3D) Graphics model.
    """
    NAME = 'X3D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#X3D'
    XML_HEADER = '<?xml version="1.0" encoding="UTF-8"?>'
    XML_DOCTYPE_X3D_3_0 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.0//EN" "http://www.web3d.org/specifications/x3d-3.0.dtd">'
    XML_DOCTYPE_X3D_3_1 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.1//EN" "http://www.web3d.org/specifications/x3d-3.1.dtd">'
    XML_DOCTYPE_X3D_3_2 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.2//EN" "http://www.web3d.org/specifications/x3d-3.2.dtd">'
    XML_DOCTYPE_X3D_3_3 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.3//EN" "http://www.web3d.org/specifications/x3d-3.3.dtd">'
    XML_DOCTYPE_X3D_4_0 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 4.0//EN" "http://www.web3d.org/specifications/x3d-4.0.dtd">'
    X3D_XML_SCHEMA_ATTRIBUTES_3_0 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.0.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_1 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.1.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_2 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.2.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_3 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.3.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_4_0 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-4.0.xsd'"
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('profile', 'Immersive', FieldType.SFString, AccessType.inputOutput, 'X3D'),
        ('version', '4.0', FieldType.SFString, AccessType.inputOutput, 'X3D'),
        ('head', None, FieldType.SFNode, AccessType.inputOutput, 'X3D'),
        ('Scene', None, FieldType.SFNode, AccessType.inputOutput, 'X3D')]
    def __init__(self,
                 profile='Immersive',
                 version='4.0',
                 head=None,
                 Scene=None):
        self.profile = profile
        self.version = version
        self.head = head
        self.Scene = Scene
    @property # getter - - - - - - - - - -
    def profile(self):
        """profile attribute is required and defines the player or tool support needed for this model."""
        return self.__profile
    @profile.setter
    def profile(self, profile=None):
        if  profile is None:
            profile = SFString.DEFAULT_VALUE
        assertValidSFString(profile)
        self.__profile = profile
    @property # getter - - - - - - - - - -
    def version(self):
        """Default is highest value matching schema and DOCTYPE in the scene."""
        return self.__version
    @version.setter
    def version(self, version=None):
        if  version is None:
            version = SFString.DEFAULT_VALUE
        assertValidSFString(version)
        self.__version = version
    @property # getter - - - - - - - - - -
    def head(self):
        """Include a head element to contain component, unit or meta statements for this X3D model."""
        return self.__head
    @head.setter
    def head(self, head=None):
        if  head is None:
            head = SFNode.DEFAULT_VALUE
        assertValidSFNode(head)
        if not isinstance(head, object):
            # print(flush=True)
            raise X3DTypeError(str(head) + ' does not have a valid node type object')
        self.__head = head
    @property # getter - - - - - - - - - -
    def Scene(self):
        """Include a Scene element to contain scene-graph nodes for this X3D model."""
        return self.__Scene
    @Scene.setter
    def Scene(self, Scene=None):
        if  Scene is None:
            Scene = SFNode.DEFAULT_VALUE
        assertValidSFNode(Scene)
        if not isinstance(Scene, object):
            # print(flush=True)
            raise X3DTypeError(str(Scene) + ' does not have a valid node type object')
        self.__Scene = Scene
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.head) or bool(self.Scene)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        # if _DEBUG: result += indent + '# invoked class function X3D.toXML(indentLevel=' + str(indentLevel) + '), indent="' + indent + '"' + '\n'
        result += indent + self.XML_HEADER + '\n'
        if self.version == '3.0':
            result += indent + self.XML_DOCTYPE_X3D_3_0 + '\n'
        elif self.version == '3.1':
            result += indent + self.XML_DOCTYPE_X3D_3_1 + '\n'
        elif self.version == '3.2':
            result += indent + self.XML_DOCTYPE_X3D_3_2 + '\n'
        elif self.version == '3.3':
            result += indent + self.XML_DOCTYPE_X3D_3_3 + '\n'
        elif self.version == '4.0':
            result += indent + self.XML_DOCTYPE_X3D_4_0 + '\n'
        elif self.version == '4.1':
            result += indent + self.XML_DOCTYPE_X3D_4_1 + '\n'
        result += indent + "<X3D profile='" + self.profile +"' version='" + self.version +"' "
        if self.version == '3.0':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_0
        elif self.version == '3.1':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_1
        elif self.version == '3.2':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_2
        elif self.version == '3.3':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_3
        elif self.version == '4.0':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_4_0
        elif self.version == '4.1':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_4_1
        result += '>' + '\n' # finish open tag
        if self.head.hasChild():
            result += str(self.head.toXML(indentLevel=indentLevel+1))
        if self.Scene.hasChild():
            result += str(self.Scene.toXML(indentLevel=indentLevel+1))
        result += '</X3D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

###############################################

# Concrete Nodes

def isX3DNode(value):
    return isinstance(value, _X3DNode)

class Anchor(_X3DGroupingNode, _X3DUrlObject):
    """
    Anchor is a Grouping node that can contain most nodes.
    """
    NAME = 'Anchor'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Anchor'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'Anchor'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('parameter', list(), FieldType.MFString, AccessType.inputOutput, 'Anchor'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 description='',
                 displayBBox=False,
                 parameter=list(),
                 url=list(),
                 visible=True,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Anchor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.description = description
        self.displayBBox = displayBBox
        self.parameter = parameter
        self.url = url
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def parameter(self):
        """If provided, parameter tells the X3D player where to to redirect the loaded url."""
        return self.__parameter
    @parameter.setter
    def parameter(self, parameter=None):
        if  parameter is None:
            parameter = MFString.DEFAULT_VALUE
        assertValidMFString(parameter)
        self.__parameter = parameter
    @property # getter - - - - - - - - - -
    def url(self):
        """Address of replacement world, or #ViewpointDEFName within the current scene, or alternate Web resource, activated by the user selecting Shape geometry within the Anchor children nodes."""
        return self.__url
    @url.setter
    def url(self, url=None):
        if  url is None:
            url = MFString.DEFAULT_VALUE
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Anchor.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Anchor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.parameter != list():
            result += " parameter='" + self.parameter + "'"
        if self.url != list():
            result += " url='" + self.url + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Anchor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Appearance(_X3DAppearanceNode):
    """
    Appearance specifies the visual properties of geometry by containing the Material, ImageTexture/MovieTexture/PixelTexture, FillProperties, LineProperties, programmable shader nodes (ComposedShader, PackagedShader, ProgramShader) and TextureTransform nodes.
    """
    NAME = 'Appearance'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Appearance'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('fillProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('lineProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('material', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('pointProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('texture', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('textureTransform', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('shaders', list(), FieldType.MFNode, AccessType.inputOutput, 'Appearance'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 fillProperties=None,
                 lineProperties=None,
                 material=None,
                 pointProperties=None,
                 texture=None,
                 textureTransform=None,
                 shaders=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Appearance __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.fillProperties = fillProperties
        self.lineProperties = lineProperties
        self.material = material
        self.pointProperties = pointProperties
        self.texture = texture
        self.textureTransform = textureTransform
        self.shaders = shaders
    @property # getter - - - - - - - - - -
    def fillProperties(self):
        """[FillProperties] Single contained FillProperties node that specifies additional visual attributes applied to polygonal areas of corresponding geometry, on top of whatever other appearance is already defined."""
        return self.__fillProperties
    @fillProperties.setter
    def fillProperties(self, fillProperties=None):
        if  fillProperties is None:
            fillProperties = SFNode.DEFAULT_VALUE
        assertValidSFNode(fillProperties)
        if not isinstance(fillProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(fillProperties) + ' does not have a valid node type object')
        self.__fillProperties = fillProperties
    @property # getter - - - - - - - - - -
    def lineProperties(self):
        """[LineProperties] Single contained LineProperties node that specifies additional visual attributes applied to corresponding line geometry."""
        return self.__lineProperties
    @lineProperties.setter
    def lineProperties(self, lineProperties=None):
        if  lineProperties is None:
            lineProperties = SFNode.DEFAULT_VALUE
        assertValidSFNode(lineProperties)
        if not isinstance(lineProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(lineProperties) + ' does not have a valid node type object')
        self.__lineProperties = lineProperties
    @property # getter - - - - - - - - - -
    def material(self):
        """[X3DMaterialNode] Single contained Material node that specifies visual attributes for lighting response (color types, transparency, etc."""
        return self.__material
    @material.setter
    def material(self, material=None):
        if  material is None:
            material = SFNode.DEFAULT_VALUE
        assertValidSFNode(material)
        if not isinstance(material, object):
            # print(flush=True)
            raise X3DTypeError(str(material) + ' does not have a valid node type object')
        self.__material = material
    @property # getter - - - - - - - - - -
    def pointProperties(self):
        return self.__pointProperties
    @pointProperties.setter
    def pointProperties(self, pointProperties=None):
        if  pointProperties is None:
            pointProperties = SFNode.DEFAULT_VALUE
        assertValidSFNode(pointProperties)
        if not isinstance(pointProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(pointProperties) + ' does not have a valid node type object')
        self.__pointProperties = pointProperties
    @property # getter - - - - - - - - - -
    def texture(self):
        """[X3DTextureNode] Single contained texture node (ImageTexture, MovieTexture, PixelTexture, MultiTexture) that maps image(s) to surface geometry."""
        return self.__texture
    @texture.setter
    def texture(self, texture=None):
        if  texture is None:
            texture = SFNode.DEFAULT_VALUE
        assertValidSFNode(texture)
        if not isinstance(texture, object):
            # print(flush=True)
            raise X3DTypeError(str(texture) + ' does not have a valid node type object')
        self.__texture = texture
    @property # getter - - - - - - - - - -
    def textureTransform(self):
        """[X3DTextureTransformNode] Single contained TextureTransform node that defines 2D transformation applied to texture coordinates."""
        return self.__textureTransform
    @textureTransform.setter
    def textureTransform(self, textureTransform=None):
        if  textureTransform is None:
            textureTransform = SFNode.DEFAULT_VALUE
        assertValidSFNode(textureTransform)
        if not isinstance(textureTransform, object):
            # print(flush=True)
            raise X3DTypeError(str(textureTransform) + ' does not have a valid node type object')
        self.__textureTransform = textureTransform
    @property # getter - - - - - - - - - -
    def shaders(self):
        """[X3DShaderNode] Zero or more contained programmable shader nodes (ComposedShader, PackagedShader, ProgramShader) that specify, in order of preference, author-programmed rendering characteristics."""
        return self.__shaders
    @shaders.setter
    def shaders(self, shaders=None):
        if  shaders is None:
            shaders = MFNode.DEFAULT_VALUE
        assertValidMFNode(shaders)
        self.__shaders = shaders
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.shaders) or bool(self.fillProperties) or bool(self.IS) or bool(self.lineProperties) or bool(self.material) or bool(self.metadata) or bool(self.pointProperties) or bool(self.texture) or bool(self.textureTransform)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Appearance.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Appearance'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.fillProperties: # walk each child node, if any
                for each in self.fillProperties:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.lineProperties: # walk each child node, if any
                for each in self.lineProperties:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.material: # walk each child node, if any
                for each in self.material:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.pointProperties: # walk each child node, if any
                for each in self.pointProperties:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.texture: # walk each child node, if any
                for each in self.texture:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.textureTransform: # walk each child node, if any
                for each in self.textureTransform:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.shaders: # walk each child node, if any
                for each in self.shaders:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Appearance>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Arc2D(_X3DGeometryNode):
    """
    Arc2D is a line-based geometry node that defines a linear circular arc with center (0,0) in X-Y plane, with angles measured starting at positive x-axis and sweeping towards positive y-axis.
    """
    NAME = 'Arc2D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Arc2D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('endAngle', 1.570796, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('startAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 endAngle=1.570796,
                 radius=1,
                 startAngle=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Arc2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.endAngle = endAngle
        self.radius = radius
        self.startAngle = startAngle
    @property # getter - - - - - - - - - -
    def endAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__endAngle
    @endAngle.setter
    def endAngle(self, endAngle=None):
        if  endAngle is None:
            endAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(endAngle)
        assertGreaterThan('endAngle', endAngle, -6.2832)
        assertLessThan('endAngle', endAngle, 6.2832)
        self.__endAngle = endAngle
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius, of which the arc is a portion."""
        return self.__radius
    @radius.setter
    def radius(self, radius=None):
        if  radius is None:
            radius = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def startAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__startAngle
    @startAngle.setter
    def startAngle(self, startAngle=None):
        if  startAngle is None:
            startAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(startAngle)
        assertGreaterThan('startAngle', startAngle, -6.2832)
        assertLessThan('startAngle', startAngle, 6.2832)
        self.__startAngle = startAngle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Arc2D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Arc2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.endAngle != 1.570796:
            result += " endAngle='" + str(self.endAngle) + "'"
        if self.radius != 1:
            result += " radius='" + str(self.radius) + "'"
        if self.startAngle != 0:
            result += " startAngle='" + str(self.startAngle) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Arc2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ArcClose2D(_X3DGeometryNode):
    """
    ArcClose2D is a polygonal geometry node that defines a linear circular arc, closed by PIE or CHORD line segments, with center (0,0) in X-Y plane, with angles measured starting at positive x-axis and sweeping towards positive y-axis.
    """
    NAME = 'ArcClose2D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ArcClose2D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('closureType', 'PIE', FieldType.SFString, AccessType.initializeOnly, 'ArcClose2D'),
        ('endAngle', 1.570796, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('solid', False, FieldType.SFBool, AccessType.initializeOnly, 'ArcClose2D'),
        ('startAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 closureType='PIE',
                 endAngle=1.570796,
                 radius=1,
                 solid=False,
                 startAngle=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ArcClose2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.closureType = closureType
        self.endAngle = endAngle
        self.radius = radius
        self.solid = solid
        self.startAngle = startAngle
    @property # getter - - - - - - - - - -
    def closureType(self):
        """Defines whether pair of line segments connect to center (PIE), or single line-segment chord connects arc endpoints (CHORD)."""
        return self.__closureType
    @closureType.setter
    def closureType(self, closureType=None):
        if  closureType is None:
            closureType = SFString.DEFAULT_VALUE
        assertValidSFString(closureType)
        self.__closureType = closureType
    @property # getter - - - - - - - - - -
    def endAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__endAngle
    @endAngle.setter
    def endAngle(self, endAngle=None):
        if  endAngle is None:
            endAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(endAngle)
        assertGreaterThan('endAngle', endAngle, -6.2832)
        assertLessThan('endAngle', endAngle, 6.2832)
        self.__endAngle = endAngle
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius, of which the arc is a portion."""
        return self.__radius
    @radius.setter
    def radius(self, radius=None):
        if  radius is None:
            radius = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def startAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__startAngle
    @startAngle.setter
    def startAngle(self, startAngle=None):
        if  startAngle is None:
            startAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(startAngle)
        assertGreaterThan('startAngle', startAngle, -6.2832)
        assertLessThan('startAngle', startAngle, 6.2832)
        self.__startAngle = startAngle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ArcClose2D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ArcClose2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.closureType != 'PIE':
            result += " closureType='" + self.closureType + "'"
        if self.endAngle != 1.570796:
            result += " endAngle='" + str(self.endAngle) + "'"
        if self.radius != 1:
            result += " radius='" + str(self.radius) + "'"
        if self.solid != False:
            result += " solid='" + str(self.solid) + "'"
        if self.startAngle != 0:
            result += " startAngle='" + str(self.startAngle) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ArcClose2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class AudioClip(_X3DSoundSourceNode, _X3DUrlObject):
    """
    AudioClip provides audio data used by parent Sound nodes.
    """
    NAME = 'AudioClip'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#AudioClip'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DSoundSourceNode'),
        ('loop', False, FieldType.SFBool, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('pauseTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('pitch', 1.0, FieldType.SFFloat, AccessType.inputOutput, 'X3DSoundSourceNode'),
        ('resumeTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('startTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('stopTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 description='',
                 loop=False,
                 pauseTime=0,
                 pitch=1.0,
                 resumeTime=0,
                 startTime=0,
                 stopTime=0,
                 url=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode AudioClip __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.description = description
        self.loop = loop
        self.pauseTime = pauseTime
        self.pitch = pitch
        self.resumeTime = resumeTime
        self.startTime = startTime
        self.stopTime = stopTime
        self.url = url
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def loop(self):
        """Repeat indefinitely when loop=true, repeat only once when loop=false."""
        return self.__loop
    @loop.setter
    def loop(self, loop=None):
        if  loop is None:
            loop = SFBool.DEFAULT_VALUE
        assertValidSFBool(loop)
        self.__loop = loop
    @property # getter - - - - - - - - - -
    def pauseTime(self):
        """When time now >= pauseTime, isPaused becomes true and AudioClip becomes paused."""
        return self.__pauseTime
    @pauseTime.setter
    def pauseTime(self, pauseTime=None):
        if  pauseTime is None:
            pauseTime = SFTime.DEFAULT_VALUE
        assertValidSFTime(pauseTime)
        self.__pauseTime = pauseTime
    @property # getter - - - - - - - - - -
    def pitch(self):
        """(0,+infinity) Multiplier for the rate at which sampled sound is played."""
        return self.__pitch
    @pitch.setter
    def pitch(self, pitch=None):
        if  pitch is None:
            pitch = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(pitch)
        assertPositive('pitch', pitch)
        self.__pitch = pitch
    @property # getter - - - - - - - - - -
    def resumeTime(self):
        """When resumeTime becomes <= time now, isPaused becomes false and AudioClip becomes active."""
        return self.__resumeTime
    @resumeTime.setter
    def resumeTime(self, resumeTime=None):
        if  resumeTime is None:
            resumeTime = SFTime.DEFAULT_VALUE
        assertValidSFTime(resumeTime)
        self.__resumeTime = resumeTime
    @property # getter - - - - - - - - - -
    def startTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__startTime
    @startTime.setter
    def startTime(self, startTime=None):
        if  startTime is None:
            startTime = SFTime.DEFAULT_VALUE
        assertValidSFTime(startTime)
        self.__startTime = startTime
    @property # getter - - - - - - - - - -
    def stopTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__stopTime
    @stopTime.setter
    def stopTime(self, stopTime=None):
        if  stopTime is None:
            stopTime = SFTime.DEFAULT_VALUE
        assertValidSFTime(stopTime)
        self.__stopTime = stopTime
    @property # getter - - - - - - - - - -
    def url(self):
        """Location and filename of sound file or stream."""
        return self.__url
    @url.setter
    def url(self, url=None):
        if  url is None:
            url = MFString.DEFAULT_VALUE
        assertValidMFString(url)
        self.__url = url
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AudioClip.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<AudioClip'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.loop != False:
            result += " loop='" + str(self.loop) + "'"
        if self.pauseTime != 0:
            result += " pauseTime='" + str(self.pauseTime) + "'"
        if self.pitch != 1.0:
            result += " pitch='" + str(self.pitch) + "'"
        if self.resumeTime != 0:
            result += " resumeTime='" + str(self.resumeTime) + "'"
        if self.startTime != 0:
            result += " startTime='" + str(self.startTime) + "'"
        if self.stopTime != 0:
            result += " stopTime='" + str(self.stopTime) + "'"
        if self.url != list():
            result += " url='" + self.url + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</AudioClip>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Background(_X3DBackgroundNode):
    """
    Background simulates ground and sky, using vertical arrays of wraparound color values.
    """
    NAME = 'Background'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Background'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('backUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('bottomUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('frontUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('groundAngle', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('groundColor', list(), FieldType.MFColor, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('leftUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('rightUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('skyAngle', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('skyColor', [(0, 0, 0)], FieldType.MFColor, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('topUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('transparency', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 backUrl=list(),
                 bottomUrl=list(),
                 frontUrl=list(),
                 groundAngle=list(),
                 groundColor=list(),
                 leftUrl=list(),
                 rightUrl=list(),
                 skyAngle=list(),
                 skyColor=[(0, 0, 0)],
                 topUrl=list(),
                 transparency=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Background __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.backUrl = backUrl
        self.bottomUrl = bottomUrl
        self.frontUrl = frontUrl
        self.groundAngle = groundAngle
        self.groundColor = groundColor
        self.leftUrl = leftUrl
        self.rightUrl = rightUrl
        self.skyAngle = skyAngle
        self.skyColor = skyColor
        self.topUrl = topUrl
        self.transparency = transparency
    @property # getter - - - - - - - - - -
    def backUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__backUrl
    @backUrl.setter
    def backUrl(self, backUrl=None):
        if  backUrl is None:
            backUrl = MFString.DEFAULT_VALUE
        assertValidMFString(backUrl)
        self.__backUrl = backUrl
    @property # getter - - - - - - - - - -
    def bottomUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__bottomUrl
    @bottomUrl.setter
    def bottomUrl(self, bottomUrl=None):
        if  bottomUrl is None:
            bottomUrl = MFString.DEFAULT_VALUE
        assertValidMFString(bottomUrl)
        self.__bottomUrl = bottomUrl
    @property # getter - - - - - - - - - -
    def frontUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__frontUrl
    @frontUrl.setter
    def frontUrl(self, frontUrl=None):
        if  frontUrl is None:
            frontUrl = MFString.DEFAULT_VALUE
        assertValidMFString(frontUrl)
        self.__frontUrl = frontUrl
    @property # getter - - - - - - - - - -
    def groundAngle(self):
        """[0,pi/2] The angle array values increase from 0."""
        return self.__groundAngle
    @groundAngle.setter
    def groundAngle(self, groundAngle=None):
        if  groundAngle is None:
            groundAngle = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(groundAngle)
        assertGreaterThanEquals('groundAngle', groundAngle, 0)
        assertLessThanEquals('groundAngle', groundAngle, 1.5708)
        self.__groundAngle = groundAngle
    @property # getter - - - - - - - - - -
    def groundColor(self):
        """Color of the ground at the various angles on the ground partial sphere."""
        return self.__groundColor
    @groundColor.setter
    def groundColor(self, groundColor=None):
        if  groundColor is None:
            groundColor = MFColor.DEFAULT_VALUE
        assertValidMFColor(groundColor)
        assertZeroToOne('groundColor', groundColor)
        self.__groundColor = groundColor
    @property # getter - - - - - - - - - -
    def leftUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__leftUrl
    @leftUrl.setter
    def leftUrl(self, leftUrl=None):
        if  leftUrl is None:
            leftUrl = MFString.DEFAULT_VALUE
        assertValidMFString(leftUrl)
        self.__leftUrl = leftUrl
    @property # getter - - - - - - - - - -
    def rightUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__rightUrl
    @rightUrl.setter
    def rightUrl(self, rightUrl=None):
        if  rightUrl is None:
            rightUrl = MFString.DEFAULT_VALUE
        assertValidMFString(rightUrl)
        self.__rightUrl = rightUrl
    @property # getter - - - - - - - - - -
    def skyAngle(self):
        """[0,pi] The angle array values increase from 0."""
        return self.__skyAngle
    @skyAngle.setter
    def skyAngle(self, skyAngle=None):
        if  skyAngle is None:
            skyAngle = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(skyAngle)
        assertGreaterThanEquals('skyAngle', skyAngle, 0)
        assertLessThanEquals('skyAngle', skyAngle, 3.1416)
        self.__skyAngle = skyAngle
    @property # getter - - - - - - - - - -
    def skyColor(self):
        """Color of the sky at various angles on the sky sphere."""
        return self.__skyColor
    @skyColor.setter
    def skyColor(self, skyColor=None):
        if  skyColor is None:
            skyColor = MFColor.DEFAULT_VALUE
        assertValidMFColor(skyColor)
        assertZeroToOne('skyColor', skyColor)
        self.__skyColor = skyColor
    @property # getter - - - - - - - - - -
    def topUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__topUrl
    @topUrl.setter
    def topUrl(self, topUrl=None):
        if  topUrl is None:
            topUrl = MFString.DEFAULT_VALUE
        assertValidMFString(topUrl)
        self.__topUrl = topUrl
    @property # getter - - - - - - - - - -
    def transparency(self):
        """[0,1] how "clear" the background is, allows underlying page to show through: 1."""
        return self.__transparency
    @transparency.setter
    def transparency(self, transparency=None):
        if  transparency is None:
            transparency = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(transparency)
        assertZeroToOne('transparency', transparency)
        self.__transparency = transparency
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Background.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Background'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.backUrl != list():
            result += " backUrl='" + self.backUrl + "'"
        if self.bottomUrl != list():
            result += " bottomUrl='" + self.bottomUrl + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.frontUrl != list():
            result += " frontUrl='" + self.frontUrl + "'"
        if self.groundAngle != list():
            result += " groundAngle='" + str(self.groundAngle) + "'"
        if self.groundColor != list():
            result += " groundColor='" + str(self.groundColor) + "'"
        if self.leftUrl != list():
            result += " leftUrl='" + self.leftUrl + "'"
        if self.rightUrl != list():
            result += " rightUrl='" + self.rightUrl + "'"
        if self.skyAngle != list():
            result += " skyAngle='" + str(self.skyAngle) + "'"
        if self.skyColor != [(0, 0, 0)]:
            result += " skyColor='" + str(self.skyColor) + "'"
        if self.topUrl != list():
            result += " topUrl='" + self.topUrl + "'"
        if self.transparency != 0:
            result += " transparency='" + str(self.transparency) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Background>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BallJoint(_X3DRigidJointNode):
    """
    BallJoint represents an unconstrained joint between two bodies that pivot about a common anchor point.
    """
    NAME = 'BallJoint'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BallJoint'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('anchorPoint', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'BallJoint'),
        ('forceOutput', ["NONE"], FieldType.MFString, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body1', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body2', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 anchorPoint=(0, 0, 0),
                 forceOutput=["NONE"],
                 body1=None,
                 body2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BallJoint __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.anchorPoint = anchorPoint
        self.forceOutput = forceOutput
        self.body1 = body1
        self.body2 = body2
    @property # getter - - - - - - - - - -
    def anchorPoint(self):
        """anchorPoint is joint center, specified in world coordinates."""
        return self.__anchorPoint
    @anchorPoint.setter
    def anchorPoint(self, anchorPoint=None):
        if  anchorPoint is None:
            anchorPoint = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(anchorPoint)
        self.__anchorPoint = anchorPoint
    @property # getter - - - - - - - - - -
    def forceOutput(self):
        """forceOutput controls which output fields are generated for the next frame."""
        return self.__forceOutput
    @forceOutput.setter
    def forceOutput(self, forceOutput=None):
        if  forceOutput is None:
            forceOutput = MFString.DEFAULT_VALUE
        assertValidMFString(forceOutput)
        self.__forceOutput = forceOutput
    @property # getter - - - - - - - - - -
    def body1(self):
        return self.__body1
    @body1.setter
    def body1(self, body1=None):
        if  body1 is None:
            body1 = SFNode.DEFAULT_VALUE
        assertValidSFNode(body1)
        if not isinstance(body1, object):
            # print(flush=True)
            raise X3DTypeError(str(body1) + ' does not have a valid node type object')
        self.__body1 = body1
    @property # getter - - - - - - - - - -
    def body2(self):
        return self.__body2
    @body2.setter
    def body2(self, body2=None):
        if  body2 is None:
            body2 = SFNode.DEFAULT_VALUE
        assertValidSFNode(body2)
        if not isinstance(body2, object):
            # print(flush=True)
            raise X3DTypeError(str(body2) + ' does not have a valid node type object')
        self.__body2 = body2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.body1) or bool(self.body2) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BallJoint.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BallJoint'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.anchorPoint != (0, 0, 0):
            result += " anchorPoint='" + str(self.anchorPoint) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.forceOutput != ["NONE"]:
            result += " forceOutput='" + self.forceOutput + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.body1: # walk each child node, if any
                for each in self.body1:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.body2: # walk each child node, if any
                for each in self.body2:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BallJoint>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Billboard(_X3DGroupingNode):
    """
    Billboard is a Grouping node that can contain most nodes.
    """
    NAME = 'Billboard'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Billboard'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('axisOfRotation', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Billboard'),
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 axisOfRotation=(0, 1, 0),
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 visible=True,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Billboard __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.axisOfRotation = axisOfRotation
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def axisOfRotation(self):
        """axisOfRotation direction is relative to local coordinate system."""
        return self.__axisOfRotation
    @axisOfRotation.setter
    def axisOfRotation(self, axisOfRotation=None):
        if  axisOfRotation is None:
            axisOfRotation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(axisOfRotation)
        self.__axisOfRotation = axisOfRotation
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Billboard.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Billboard'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.axisOfRotation != (0, 1, 0):
            result += " axisOfRotation='" + str(self.axisOfRotation) + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Billboard>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BlendedVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    BlendedVolumeStyle combines rendering of two voxel data sets into one by blending voxel values.
    """
    NAME = 'BlendedVolumeStyle'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BlendedVolumeStyle'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('weightConstant1', 0.5, FieldType.SFFloat, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightConstant2', 0.5, FieldType.SFFloat, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightFunction1', 'CONSTANT', FieldType.SFString, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightFunction2', 'CONSTANT', FieldType.SFString, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('renderStyle', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('voxels', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightTransferFunction1', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightTransferFunction2', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 weightConstant1=0.5,
                 weightConstant2=0.5,
                 weightFunction1='CONSTANT',
                 weightFunction2='CONSTANT',
                 renderStyle=None,
                 voxels=None,
                 weightTransferFunction1=None,
                 weightTransferFunction2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BlendedVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.weightConstant1 = weightConstant1
        self.weightConstant2 = weightConstant2
        self.weightFunction1 = weightFunction1
        self.weightFunction2 = weightFunction2
        self.renderStyle = renderStyle
        self.voxels = voxels
        self.weightTransferFunction1 = weightTransferFunction1
        self.weightTransferFunction2 = weightTransferFunction2
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def weightConstant1(self):
        """[0,1] weightConstant1 is used when weightFunction1=CONSTANT."""
        return self.__weightConstant1
    @weightConstant1.setter
    def weightConstant1(self, weightConstant1=None):
        if  weightConstant1 is None:
            weightConstant1 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(weightConstant1)
        assertZeroToOne('weightConstant1', weightConstant1)
        self.__weightConstant1 = weightConstant1
    @property # getter - - - - - - - - - -
    def weightConstant2(self):
        """[0,1] weightConstant2 is used when weightFunction2=CONSTANT."""
        return self.__weightConstant2
    @weightConstant2.setter
    def weightConstant2(self, weightConstant2=None):
        if  weightConstant2 is None:
            weightConstant2 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(weightConstant2)
        assertZeroToOne('weightConstant2', weightConstant2)
        self.__weightConstant2 = weightConstant2
    @property # getter - - - - - - - - - -
    def weightFunction1(self):
        """specifies 2D textures used to determine weight values when weight function is set to TABLE."""
        return self.__weightFunction1
    @weightFunction1.setter
    def weightFunction1(self, weightFunction1=None):
        if  weightFunction1 is None:
            weightFunction1 = SFString.DEFAULT_VALUE
        assertValidSFString(weightFunction1)
        self.__weightFunction1 = weightFunction1
    @property # getter - - - - - - - - - -
    def weightFunction2(self):
        """specifies 2D textures used to determine weight values when weight function is set to TABLE."""
        return self.__weightFunction2
    @weightFunction2.setter
    def weightFunction2(self, weightFunction2=None):
        if  weightFunction2 is None:
            weightFunction2 = SFString.DEFAULT_VALUE
        assertValidSFString(weightFunction2)
        self.__weightFunction2 = weightFunction2
    @property # getter - - - - - - - - - -
    def renderStyle(self):
        """[X3DComposableVolumeRenderStyleNode] Single contained X3DComposableVolumeRenderStyleNode node that defines specific rendering technique for data in the voxels field, and the result is blended with parent VolumeData or SegmentedVoliumeData node."""
        return self.__renderStyle
    @renderStyle.setter
    def renderStyle(self, renderStyle=None):
        if  renderStyle is None:
            renderStyle = SFNode.DEFAULT_VALUE
        assertValidSFNode(renderStyle)
        if not isinstance(renderStyle, object):
            # print(flush=True)
            raise X3DTypeError(str(renderStyle) + ' does not have a valid node type object')
        self.__renderStyle = renderStyle
    @property # getter - - - - - - - - - -
    def voxels(self):
        """[X3DTexture3DNode] Single contained X3DTexture3DNode (ComposedTexture3D, ImageTexture3D, PixelTexture3D) that provides second set of raw voxel information utilized by corresponding rendering styles."""
        return self.__voxels
    @voxels.setter
    def voxels(self, voxels=None):
        if  voxels is None:
            voxels = SFNode.DEFAULT_VALUE
        assertValidSFNode(voxels)
        if not isinstance(voxels, object):
            # print(flush=True)
            raise X3DTypeError(str(voxels) + ' does not have a valid node type object')
        self.__voxels = voxels
    @property # getter - - - - - - - - - -
    def weightTransferFunction1(self):
        return self.__weightTransferFunction1
    @weightTransferFunction1.setter
    def weightTransferFunction1(self, weightTransferFunction1=None):
        if  weightTransferFunction1 is None:
            weightTransferFunction1 = SFNode.DEFAULT_VALUE
        assertValidSFNode(weightTransferFunction1)
        if not isinstance(weightTransferFunction1, object):
            # print(flush=True)
            raise X3DTypeError(str(weightTransferFunction1) + ' does not have a valid node type object')
        self.__weightTransferFunction1 = weightTransferFunction1
    @property # getter - - - - - - - - - -
    def weightTransferFunction2(self):
        return self.__weightTransferFunction2
    @weightTransferFunction2.setter
    def weightTransferFunction2(self, weightTransferFunction2=None):
        if  weightTransferFunction2 is None:
            weightTransferFunction2 = SFNode.DEFAULT_VALUE
        assertValidSFNode(weightTransferFunction2)
        if not isinstance(weightTransferFunction2, object):
            # print(flush=True)
            raise X3DTypeError(str(weightTransferFunction2) + ' does not have a valid node type object')
        self.__weightTransferFunction2 = weightTransferFunction2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata) or bool(self.renderStyle) or bool(self.voxels) or bool(self.weightTransferFunction1) or bool(self.weightTransferFunction2)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BlendedVolumeStyle.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BlendedVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.weightConstant1 != 0.5:
            result += " weightConstant1='" + str(self.weightConstant1) + "'"
        if self.weightConstant2 != 0.5:
            result += " weightConstant2='" + str(self.weightConstant2) + "'"
        if self.weightFunction1 != 'CONSTANT':
            result += " weightFunction1='" + self.weightFunction1 + "'"
        if self.weightFunction2 != 'CONSTANT':
            result += " weightFunction2='" + self.weightFunction2 + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.renderStyle: # walk each child node, if any
                for each in self.renderStyle:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.voxels: # walk each child node, if any
                for each in self.voxels:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.weightTransferFunction1: # walk each child node, if any
                for each in self.weightTransferFunction1:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.weightTransferFunction2: # walk each child node, if any
                for each in self.weightTransferFunction2:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BlendedVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BooleanFilter(_X3DChildNode):
    """
    BooleanFilter selectively passes true, false or negated events.
    """
    NAME = 'BooleanFilter'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanFilter'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanFilter __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanFilter.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanFilter'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BooleanFilter>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BooleanSequencer(_X3DSequencerNode):
    """
    BooleanSequencer generates periodic discrete Boolean values.
    """
    NAME = 'BooleanSequencer'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanSequencer'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DSequencerNode'),
        ('keyValue', list(), FieldType.MFBool, AccessType.inputOutput, 'BooleanSequencer'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanSequencer __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key=None):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear sequencing, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue=None):
        if  keyValue is None:
            keyValue = MFBool.DEFAULT_VALUE
        assertValidMFBool(keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanSequencer.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanSequencer'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + str(self.key) + "'"
        if self.keyValue != list():
            result += " keyValue='" + str(self.keyValue) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BooleanSequencer>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BooleanToggle(_X3DChildNode):
    """
    BooleanToggle maintains state and negates output when a true input is provided.
    """
    NAME = 'BooleanToggle'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanToggle'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('toggle', False, FieldType.SFBool, AccessType.inputOutput, 'BooleanToggle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 toggle=False,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanToggle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.toggle = toggle
    @property # getter - - - - - - - - - -
    def toggle(self):
        """Persistent state value that gets toggled or reset."""
        return self.__toggle
    @toggle.setter
    def toggle(self, toggle=None):
        if  toggle is None:
            toggle = SFBool.DEFAULT_VALUE
        assertValidSFBool(toggle)
        self.__toggle = toggle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanToggle.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanToggle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.toggle != False:
            result += " toggle='" + str(self.toggle) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BooleanToggle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BooleanTrigger(_X3DTriggerNode):
    """
    BooleanTrigger converts time events to boolean true events.
    """
    NAME = 'BooleanTrigger'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanTrigger'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanTrigger __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanTrigger.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanTrigger'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BooleanTrigger>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BoundaryEnhancementVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    BoundaryEnhancementVolumeStyle provides boundary enhancement for the volume rendering style.
    """
    NAME = 'BoundaryEnhancementVolumeStyle'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BoundaryEnhancementVolumeStyle'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('boundaryOpacity', 0.9, FieldType.SFFloat, AccessType.inputOutput, 'BoundaryEnhancementVolumeStyle'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('opacityFactor', 2, FieldType.SFFloat, AccessType.inputOutput, 'BoundaryEnhancementVolumeStyle'),
        ('retainedOpacity', 0.2, FieldType.SFFloat, AccessType.inputOutput, 'BoundaryEnhancementVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 boundaryOpacity=0.9,
                 enabled=True,
                 opacityFactor=2,
                 retainedOpacity=0.2,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BoundaryEnhancementVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.boundaryOpacity = boundaryOpacity
        self.enabled = enabled
        self.opacityFactor = opacityFactor
        self.retainedOpacity = retainedOpacity
    @property # getter - - - - - - - - - -
    def boundaryOpacity(self):
        """[0,+infinity) boundaryOpacity k_gs is the factored amount of the gradient enhancement to use."""
        return self.__boundaryOpacity
    @boundaryOpacity.setter
    def boundaryOpacity(self, boundaryOpacity=None):
        if  boundaryOpacity is None:
            boundaryOpacity = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(boundaryOpacity)
        assertZeroToOne('boundaryOpacity', boundaryOpacity)
        self.__boundaryOpacity = boundaryOpacity
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def opacityFactor(self):
        """[0,+infinity) opacityFactor k_ge is the power function to control the slope of the opacity curve to highlight the set of data."""
        return self.__opacityFactor
    @opacityFactor.setter
    def opacityFactor(self, opacityFactor=None):
        if  opacityFactor is None:
            opacityFactor = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(opacityFactor)
        assertNonNegative('opacityFactor', opacityFactor)
        self.__opacityFactor = opacityFactor
    @property # getter - - - - - - - - - -
    def retainedOpacity(self):
        """[0,1] retainedOpacity k_gc is the amount of initial opacity to mix into the output."""
        return self.__retainedOpacity
    @retainedOpacity.setter
    def retainedOpacity(self, retainedOpacity=None):
        if  retainedOpacity is None:
            retainedOpacity = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(retainedOpacity)
        assertZeroToOne('retainedOpacity', retainedOpacity)
        self.__retainedOpacity = retainedOpacity
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BoundaryEnhancementVolumeStyle.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BoundaryEnhancementVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.boundaryOpacity != 0.9:
            result += " boundaryOpacity='" + str(self.boundaryOpacity) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.opacityFactor != 2:
            result += " opacityFactor='" + str(self.opacityFactor) + "'"
        if self.retainedOpacity != 0.2:
            result += " retainedOpacity='" + str(self.retainedOpacity) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BoundaryEnhancementVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class BoundedPhysicsModel(_X3DParticlePhysicsModelNode):
    """
    BoundedPhysicsModel provides user-defined geometrical boundaries for particle motion.
    """
    NAME = 'BoundedPhysicsModel'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BoundedPhysicsModel'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DParticlePhysicsModelNode'),
        ('geometry', None, FieldType.SFNode, AccessType.inputOutput, 'BoundedPhysicsModel'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 geometry=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BoundedPhysicsModel __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.geometry = geometry
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def geometry(self):
        """[X3DGeometryNode] Single contained geometry node provides the geometry used for each particle when the parent ParticleSystem node has geometryType=GEOMETRY."""
        return self.__geometry
    @geometry.setter
    def geometry(self, geometry=None):
        if  geometry is None:
            geometry = SFNode.DEFAULT_VALUE
        assertValidSFNode(geometry)
        if not isinstance(geometry, object):
            # print(flush=True)
            raise X3DTypeError(str(geometry) + ' does not have a valid node type object')
        self.__geometry = geometry
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.geometry) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BoundedPhysicsModel.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BoundedPhysicsModel'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geometry: # walk each child node, if any
                for each in self.geometry:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</BoundedPhysicsModel>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Box(_X3DGeometryNode):
    """
    Box is a geometry node specifying a rectangular cuboid.
    """
    NAME = 'Box'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Box'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('size', (2, 2, 2), FieldType.SFVec3f, AccessType.initializeOnly, 'Box'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'Box'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 size=(2, 2, 2),
                 solid=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Box __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.size = size
        self.solid = solid
    @property # getter - - - - - - - - - -
    def size(self):
        """(0,+infinity) size x y z in meters."""
        return self.__size
    @size.setter
    def size(self, size=None):
        if  size is None:
            size = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(size)
        self.__size = size
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Box.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Box'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.size != (2, 2, 2):
            result += " size='" + str(self.size) + "'"
        if self.solid != True:
            result += " solid='" + str(self.solid) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Box>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CADAssembly(_X3DGroupingNode, _X3DProductStructureChildNode):
    """
    CADAssembly holds a set of Computer-Aided Design (CAD) assemblies or parts grouped together.
    """
    NAME = 'CADAssembly'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADAssembly'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'X3DProductStructureChildNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 name='',
                 visible=True,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADAssembly __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.name = name
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADAssembly.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADAssembly'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CADAssembly>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CADFace(_X3DProductStructureChildNode, _X3DBoundedObject):
    """
    CADFace holds geometry representing one face in a Computer-Aided Design (CAD) CADPart.
    """
    NAME = 'CADFace'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADFace'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'X3DProductStructureChildNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('shape', None, FieldType.SFNode, AccessType.inputOutput, 'CADFace'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 name='',
                 visible=True,
                 shape=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADFace __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.name = name
        self.visible = visible
        self.shape = shape
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def shape(self):
        """[X3DShapeNode|LOD|Transform] Contained Shape for this CADPart."""
        return self.__shape
    @shape.setter
    def shape(self, shape=None):
        if  shape is None:
            shape = SFNode.DEFAULT_VALUE
        assertValidSFNode(shape)
        if not isinstance(shape, object):
            # print(flush=True)
            raise X3DTypeError(str(shape) + ' does not have a valid node type object')
        self.__shape = shape
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata) or bool(self.shape)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADFace.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADFace'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.shape: # walk each child node, if any
                for each in self.shape:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CADFace>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CADLayer(_X3DGroupingNode):
    """
    CADLayer nodes define a hierarchy that shows layer structure for a Computer-Aided Design (CAD) model.
    """
    NAME = 'CADLayer'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADLayer'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'CADLayer'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 name='',
                 visible=True,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADLayer __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.name = name
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def visible(self):
        """Array of boolean values that specify whether each individual child CADAssembly is visible."""
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADLayer.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADLayer'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CADLayer>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CADPart(_X3DProductStructureChildNode, _X3DGroupingNode):
    """
    CADPart is an atomic part that defines both coordinate-system location and the faces that constitute a part in a Computer-Aided Design (CAD) model.
    """
    NAME = 'CADPart'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADPart'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('center', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'CADPart'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'X3DProductStructureChildNode'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'CADPart'),
        ('scale', (1, 1, 1), FieldType.SFVec3f, AccessType.inputOutput, 'CADPart'),
        ('scaleOrientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'CADPart'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'CADPart'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'CADPart'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 center=(0, 0, 0),
                 displayBBox=False,
                 name='',
                 rotation=(0, 0, 1, 0),
                 scale=(1, 1, 1),
                 scaleOrientation=(0, 0, 1, 0),
                 translation=(0, 0, 0),
                 visible=True,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADPart __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.center = center
        self.displayBBox = displayBBox
        self.name = name
        self.rotation = rotation
        self.scale = scale
        self.scaleOrientation = scaleOrientation
        self.translation = translation
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def center(self):
        """Translation offset from origin of local coordinate system, applied prior to rotation or scaling."""
        return self.__center
    @center.setter
    def center(self, center=None):
        if  center is None:
            center = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(center)
        self.__center = center
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation (axis, angle in radians) of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation=None):
        if  rotation is None:
            rotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def scale(self):
        """Non-uniform x-y-z scale of child coordinate system, adjusted by center and scaleOrientation."""
        return self.__scale
    @scale.setter
    def scale(self, scale=None):
        if  scale is None:
            scale = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(scale)
        self.__scale = scale
    @property # getter - - - - - - - - - -
    def scaleOrientation(self):
        """Preliminary rotation of coordinate system before scaling (to allow scaling around arbitrary orientations)."""
        return self.__scaleOrientation
    @scaleOrientation.setter
    def scaleOrientation(self, scaleOrientation=None):
        if  scaleOrientation is None:
            scaleOrientation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(scaleOrientation)
        self.__scaleOrientation = scaleOrientation
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position (x, y, z in meters) of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation=None):
        if  translation is None:
            translation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADPart.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADPart'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.center != (0, 0, 0):
            result += " center='" + str(self.center) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + str(self.rotation) + "'"
        if self.scale != (1, 1, 1):
            result += " scale='" + str(self.scale) + "'"
        if self.scaleOrientation != (0, 0, 1, 0):
            result += " scaleOrientation='" + str(self.scaleOrientation) + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + str(self.translation) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CADPart>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CartoonVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    CartoonVolumeStyle generates cartoon-style non-photorealistic rendering of associated volumetric data.
    """
    NAME = 'CartoonVolumeStyle'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CartoonVolumeStyle'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('colorSteps', 4, FieldType.SFInt32, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('orthogonalColor', (1, 1, 1, 1), FieldType.SFColorRGBA, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('parallelColor', (0, 0, 0, 1), FieldType.SFColorRGBA, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('surfaceNormals', None, FieldType.SFNode, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 colorSteps=4,
                 enabled=True,
                 orthogonalColor=(1, 1, 1, 1),
                 parallelColor=(0, 0, 0, 1),
                 surfaceNormals=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CartoonVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.colorSteps = colorSteps
        self.enabled = enabled
        self.orthogonalColor = orthogonalColor
        self.parallelColor = parallelColor
        self.surfaceNormals = surfaceNormals
    @property # getter - - - - - - - - - -
    def colorSteps(self):
        """[1,64] Number of distinct colors taken from interpolated colors and used to render the object."""
        return self.__colorSteps
    @colorSteps.setter
    def colorSteps(self, colorSteps=None):
        if  colorSteps is None:
            colorSteps = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(colorSteps)
        assertGreaterThanEquals('colorSteps', colorSteps, 1)
        assertLessThanEquals('colorSteps', colorSteps, 64)
        self.__colorSteps = colorSteps
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def orthogonalColor(self):
        """[0,1] orthogonalColor is used for surface normals that are orthogonal (perpendicular) to viewer's current location."""
        return self.__orthogonalColor
    @orthogonalColor.setter
    def orthogonalColor(self, orthogonalColor=None):
        if  orthogonalColor is None:
            orthogonalColor = SFColorRGBA.DEFAULT_VALUE
        assertValidSFColorRGBA(orthogonalColor)
        assertZeroToOne('orthogonalColor', orthogonalColor)
        self.__orthogonalColor = orthogonalColor
    @property # getter - - - - - - - - - -
    def parallelColor(self):
        """[0,1] parallelColor is used for surface normals that are orthogonal to viewer's current location."""
        return self.__parallelColor
    @parallelColor.setter
    def parallelColor(self, parallelColor=None):
        if  parallelColor is None:
            parallelColor = SFColorRGBA.DEFAULT_VALUE
        assertValidSFColorRGBA(parallelColor)
        assertZeroToOne('parallelColor', parallelColor)
        self.__parallelColor = parallelColor
    @property # getter - - - - - - - - - -
    def surfaceNormals(self):
        return self.__surfaceNormals
    @surfaceNormals.setter
    def surfaceNormals(self, surfaceNormals=None):
        if  surfaceNormals is None:
            surfaceNormals = SFNode.DEFAULT_VALUE
        assertValidSFNode(surfaceNormals)
        if not isinstance(surfaceNormals, object):
            # print(flush=True)
            raise X3DTypeError(str(surfaceNormals) + ' does not have a valid node type object')
        self.__surfaceNormals = surfaceNormals
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata) or bool(self.surfaceNormals)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CartoonVolumeStyle.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CartoonVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.colorSteps != 4:
            result += " colorSteps='" + str(self.colorSteps) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.orthogonalColor != (1, 1, 1, 1):
            result += " orthogonalColor='" + str(self.orthogonalColor) + "'"
        if self.parallelColor != (0, 0, 0, 1):
            result += " parallelColor='" + str(self.parallelColor) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.surfaceNormals: # walk each child node, if any
                for each in self.surfaceNormals:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CartoonVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Circle2D(_X3DGeometryNode):
    """
    Circle2D is a geometry node that defines a linear X-Y circle with center (0,0) in X-Y plane.
    """
    NAME = 'Circle2D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Circle2D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Circle2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 radius=1,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Circle2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.radius = radius
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius."""
        return self.__radius
    @radius.setter
    def radius(self, radius=None):
        if  radius is None:
            radius = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Circle2D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Circle2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.radius != 1:
            result += " radius='" + str(self.radius) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Circle2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ClipPlane(_X3DChildNode):
    """
    ClipPlane specifies a single plane equation used to clip (i.
    """
    NAME = 'ClipPlane'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ClipPlane'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'ClipPlane'),
        ('plane', (0, 1, 0, 0), FieldType.SFVec4f, AccessType.inputOutput, 'ClipPlane'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 plane=(0, 1, 0, 0),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ClipPlane __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.plane = plane
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def plane(self):
        """[0,1] If (a,b,c,d) is the plane, with the first three components being a normalized vector describing the plane's normal direction (and thus the fourth component d being distance from the origin), a point (x,y,z) is visible to the user, with regards to the clipping plane, if a*x+b*y+c*z+d is greater than 0."""
        return self.__plane
    @plane.setter
    def plane(self, plane=None):
        if  plane is None:
            plane = SFVec4f.DEFAULT_VALUE
        assertValidSFVec4f(plane)
        assertGreaterThanEquals('plane', plane, -1)
        assertLessThanEquals('plane', plane, 1)
        self.__plane = plane
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ClipPlane.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ClipPlane'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.plane != (0, 1, 0, 0):
            result += " plane='" + str(self.plane) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ClipPlane>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CollidableOffset(_X3DNBodyCollidableNode):
    """
    CollidableOffset repositions geometry relative to center of owning body.
    """
    NAME = 'CollidableOffset'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollidableOffset'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('collidable', None, FieldType.SFNode, AccessType.initializeOnly, 'CollidableOffset'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 rotation=(0, 0, 1, 0),
                 translation=(0, 0, 0),
                 visible=True,
                 collidable=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollidableOffset __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.rotation = rotation
        self.translation = translation
        self.visible = visible
        self.collidable = collidable
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation (axis, angle in radians) of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation=None):
        if  rotation is None:
            rotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position (x, y, z in meters) of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation=None):
        if  translation is None:
            translation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def collidable(self):
        return self.__collidable
    @collidable.setter
    def collidable(self, collidable=None):
        if  collidable is None:
            collidable = SFNode.DEFAULT_VALUE
        assertValidSFNode(collidable)
        if not isinstance(collidable, object):
            # print(flush=True)
            raise X3DTypeError(str(collidable) + ' does not have a valid node type object')
        self.__collidable = collidable
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.collidable) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollidableOffset.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollidableOffset'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + str(self.rotation) + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + str(self.translation) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.collidable: # walk each child node, if any
                for each in self.collidable:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CollidableOffset>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CollidableShape(_X3DNBodyCollidableNode):
    """
    CollidableShape connects the collision detection system, the rigid body model, and the renderable scene graph.
    """
    NAME = 'CollidableShape'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollidableShape'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('shape', None, FieldType.SFNode, AccessType.initializeOnly, 'CollidableShape'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 rotation=(0, 0, 1, 0),
                 translation=(0, 0, 0),
                 visible=True,
                 shape=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollidableShape __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.rotation = rotation
        self.translation = translation
        self.visible = visible
        self.shape = shape
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation (axis, angle in radians) of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation=None):
        if  rotation is None:
            rotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position (x, y, z in meters) of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation=None):
        if  translation is None:
            translation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def shape(self):
        return self.__shape
    @shape.setter
    def shape(self, shape=None):
        if  shape is None:
            shape = SFNode.DEFAULT_VALUE
        assertValidSFNode(shape)
        if not isinstance(shape, object):
            # print(flush=True)
            raise X3DTypeError(str(shape) + ' does not have a valid node type object')
        self.__shape = shape
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata) or bool(self.shape)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollidableShape.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollidableShape'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + str(self.rotation) + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + str(self.translation) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.shape: # walk each child node, if any
                for each in self.shape:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CollidableShape>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Collision(_X3DGroupingNode, _X3DSensorNode):
    """
    Collision detects camera-to-object contact using current view and NavigationInfo avatarSize.
    """
    NAME = 'Collision'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Collision'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('proxy', None, FieldType.SFNode, AccessType.initializeOnly, 'Collision'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 visible=True,
                 proxy=None,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Collision __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.visible = visible
        self.proxy = proxy
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables collision detection for children and all descendants."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def proxy(self):
        """[X3DChildNode] The proxy node is used as a substitute for Collision children during collision detection, simplifying collision-intersection computations."""
        return self.__proxy
    @proxy.setter
    def proxy(self, proxy=None):
        if  proxy is None:
            proxy = SFNode.DEFAULT_VALUE
        assertValidSFNode(proxy)
        if not isinstance(proxy, object):
            # print(flush=True)
            raise X3DTypeError(str(proxy) + ' does not have a valid node type object')
        self.__proxy = proxy
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata) or bool(self.proxy)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Collision.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Collision'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.proxy: # walk each child node, if any
                for each in self.proxy:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Collision>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CollisionCollection(_X3DChildNode, _X3DBoundedObject):
    """
    CollisionCollection holds a collection of objects that can be managed as a single entity for resolution of inter-object collisions.
    """
    NAME = 'CollisionCollection'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollisionCollection'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('appliedParameters', ["BOUNCE"], FieldType.MFString, AccessType.inputOutput, 'CollisionCollection'),
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bounce', 0, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'CollisionCollection'),
        ('frictionCoefficients', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'CollisionCollection'),
        ('minBounceSpeed', 0.1, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('slipFactors', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'CollisionCollection'),
        ('softnessConstantForceMix', 0.0001, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('softnessErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('surfaceSpeed', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'CollisionCollection'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('collidables', list(), FieldType.MFNode, AccessType.inputOutput, 'CollisionCollection'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 appliedParameters=["BOUNCE"],
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 bounce=0,
                 displayBBox=False,
                 enabled=True,
                 frictionCoefficients=(0, 0),
                 minBounceSpeed=0.1,
                 slipFactors=(0, 0),
                 softnessConstantForceMix=0.0001,
                 softnessErrorCorrection=0.8,
                 surfaceSpeed=(0, 0),
                 visible=True,
                 collidables=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollisionCollection __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.appliedParameters = appliedParameters
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.bounce = bounce
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.frictionCoefficients = frictionCoefficients
        self.minBounceSpeed = minBounceSpeed
        self.slipFactors = slipFactors
        self.softnessConstantForceMix = softnessConstantForceMix
        self.softnessErrorCorrection = softnessErrorCorrection
        self.surfaceSpeed = surfaceSpeed
        self.visible = visible
        self.collidables = collidables
    @property # getter - - - - - - - - - -
    def appliedParameters(self):
        """Default global parameters for collision outputs of rigid body physics system."""
        return self.__appliedParameters
    @appliedParameters.setter
    def appliedParameters(self, appliedParameters=None):
        if  appliedParameters is None:
            appliedParameters = MFString.DEFAULT_VALUE
        assertValidMFString(appliedParameters)
        self.__appliedParameters = appliedParameters
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def bounce(self):
        """[0,1] bounce indicates bounciness (0 = no bounce at all, 1 = maximum bounce)."""
        return self.__bounce
    @bounce.setter
    def bounce(self, bounce=None):
        if  bounce is None:
            bounce = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(bounce)
        assertZeroToOne('bounce', bounce)
        self.__bounce = bounce
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def frictionCoefficients(self):
        """frictionCoefficients used for computing surface drag."""
        return self.__frictionCoefficients
    @frictionCoefficients.setter
    def frictionCoefficients(self, frictionCoefficients=None):
        if  frictionCoefficients is None:
            frictionCoefficients = SFVec2f.DEFAULT_VALUE
        assertValidSFVec2f(frictionCoefficients)
        assertNonNegative('frictionCoefficients', frictionCoefficients)
        self.__frictionCoefficients = frictionCoefficients
    @property # getter - - - - - - - - - -
    def minBounceSpeed(self):
        """[0,+infinity) minBounceSpeed m/s needed to bounce."""
        return self.__minBounceSpeed
    @minBounceSpeed.setter
    def minBounceSpeed(self, minBounceSpeed=None):
        if  minBounceSpeed is None:
            minBounceSpeed = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(minBounceSpeed)
        assertNonNegative('minBounceSpeed', minBounceSpeed)
        self.__minBounceSpeed = minBounceSpeed
    @property # getter - - - - - - - - - -
    def slipFactors(self):
        """slipFactors used for computing surface drag."""
        return self.__slipFactors
    @slipFactors.setter
    def slipFactors(self, slipFactors=None):
        if  slipFactors is None:
            slipFactors = SFVec2f.DEFAULT_VALUE
        assertValidSFVec2f(slipFactors)
        self.__slipFactors = slipFactors
    @property # getter - - - - - - - - - -
    def softnessConstantForceMix(self):
        """[0,1] softnessConstantForceMix value applies a constant force value to make colliding surfaces appear to be somewhat soft."""
        return self.__softnessConstantForceMix
    @softnessConstantForceMix.setter
    def softnessConstantForceMix(self, softnessConstantForceMix=None):
        if  softnessConstantForceMix is None:
            softnessConstantForceMix = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(softnessConstantForceMix)
        assertZeroToOne('softnessConstantForceMix', softnessConstantForceMix)
        self.__softnessConstantForceMix = softnessConstantForceMix
    @property # getter - - - - - - - - - -
    def softnessErrorCorrection(self):
        """[0,1] softnessErrorCorrection indicates fraction of collision error fixed in a set of evaluations (0 = no error correction, 1 = all errors corrected in single step)."""
        return self.__softnessErrorCorrection
    @softnessErrorCorrection.setter
    def softnessErrorCorrection(self, softnessErrorCorrection=None):
        if  softnessErrorCorrection is None:
            softnessErrorCorrection = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(softnessErrorCorrection)
        assertZeroToOne('softnessErrorCorrection', softnessErrorCorrection)
        self.__softnessErrorCorrection = softnessErrorCorrection
    @property # getter - - - - - - - - - -
    def surfaceSpeed(self):
        """surfaceSpeed defines speed vectors for computing surface drag, if contact surfaces move independently of bodies."""
        return self.__surfaceSpeed
    @surfaceSpeed.setter
    def surfaceSpeed(self, surfaceSpeed=None):
        if  surfaceSpeed is None:
            surfaceSpeed = SFVec2f.DEFAULT_VALUE
        assertValidSFVec2f(surfaceSpeed)
        self.__surfaceSpeed = surfaceSpeed
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def collidables(self):
        """[CollisionSpace|CollidableShape|CollidableOffset] CollisionCollection node holds a collection of objects in the collidables field that can be managed as a single entity for resolution of inter-object collisions with other groups of collidable objects."""
        return self.__collidables
    @collidables.setter
    def collidables(self, collidables=None):
        if  collidables is None:
            collidables = MFNode.DEFAULT_VALUE
        assertValidMFNode(collidables)
        self.__collidables = collidables
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.collidables) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollisionCollection.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollisionCollection'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.appliedParameters != ["BOUNCE"]:
            result += " appliedParameters='" + self.appliedParameters + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.bounce != 0:
            result += " bounce='" + str(self.bounce) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.frictionCoefficients != (0, 0):
            result += " frictionCoefficients='" + str(self.frictionCoefficients) + "'"
        if self.minBounceSpeed != 0.1:
            result += " minBounceSpeed='" + str(self.minBounceSpeed) + "'"
        if self.slipFactors != (0, 0):
            result += " slipFactors='" + str(self.slipFactors) + "'"
        if self.softnessConstantForceMix != 0.0001:
            result += " softnessConstantForceMix='" + str(self.softnessConstantForceMix) + "'"
        if self.softnessErrorCorrection != 0.8:
            result += " softnessErrorCorrection='" + str(self.softnessErrorCorrection) + "'"
        if self.surfaceSpeed != (0, 0):
            result += " surfaceSpeed='" + str(self.surfaceSpeed) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.collidables: # walk each child node, if any
                for each in self.collidables:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CollisionCollection>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CollisionSensor(_X3DSensorNode):
    """
    CollisionSensor generates collision-detection events.
    """
    NAME = 'CollisionSensor'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollisionSensor'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('collider', None, FieldType.SFNode, AccessType.inputOutput, 'CollisionSensor'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 collider=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollisionSensor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.collider = collider
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def collider(self):
        return self.__collider
    @collider.setter
    def collider(self, collider=None):
        if  collider is None:
            collider = SFNode.DEFAULT_VALUE
        assertValidSFNode(collider)
        if not isinstance(collider, object):
            # print(flush=True)
            raise X3DTypeError(str(collider) + ' does not have a valid node type object')
        self.__collider = collider
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.collider) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollisionSensor.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollisionSensor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.collider: # walk each child node, if any
                for each in self.collider:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CollisionSensor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CollisionSpace(_X3DNBodyCollisionSpaceNode):
    """
    CollisionSpace holds collection of objects considered together for resolution of inter-object collisions.
    """
    NAME = 'CollisionSpace'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollisionSpace'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollisionSpaceNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollisionSpaceNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollisionSpaceNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollisionSpaceNode'),
        ('useGeometry', False, FieldType.SFBool, AccessType.inputOutput, 'CollisionSpace'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollisionSpaceNode'),
        ('collidables', list(), FieldType.MFNode, AccessType.inputOutput, 'CollisionSpace'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 useGeometry=False,
                 visible=True,
                 collidables=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollisionSpace __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.useGeometry = useGeometry
        self.visible = visible
        self.collidables = collidables
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def useGeometry(self):
        """useGeometry indicates whether collision-detection code checks down to level of geometry, or only make approximations using geometry bounds."""
        return self.__useGeometry
    @useGeometry.setter
    def useGeometry(self, useGeometry=None):
        if  useGeometry is None:
            useGeometry = SFBool.DEFAULT_VALUE
        assertValidSFBool(useGeometry)
        self.__useGeometry = useGeometry
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def collidables(self):
        return self.__collidables
    @collidables.setter
    def collidables(self, collidables=None):
        if  collidables is None:
            collidables = MFNode.DEFAULT_VALUE
        assertValidMFNode(collidables)
        self.__collidables = collidables
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.collidables) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollisionSpace.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollisionSpace'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.useGeometry != False:
            result += " useGeometry='" + str(self.useGeometry) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.collidables: # walk each child node, if any
                for each in self.collidables:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CollisionSpace>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Color(_X3DColorNode):
    """
    Color node defines a set of RGB color values that apply either to a sibling Coordinate|CoordinateDouble node, or else to a parent ElevationGrid node.
    """
    NAME = 'Color'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Color'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('color', list(), FieldType.MFColor, AccessType.inputOutput, 'Color'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 color=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Color __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.color = color
    @property # getter - - - - - - - - - -
    def color(self):
        """The color field defines an array of 3-tuple RGB colors."""
        return self.__color
    @color.setter
    def color(self, color=None):
        if  color is None:
            color = MFColor.DEFAULT_VALUE
        assertValidMFColor(color)
        assertZeroToOne('color', color)
        self.__color = color
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Color.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Color'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.color != list():
            result += " color='" + str(self.color) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Color>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ColorChaser(_X3DChaserNode):
    """
    ColorChaser generates a series of SFColor values that progressively change from initial value to destination value.
    """
    NAME = 'ColorChaser'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorChaser'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('duration', 1, FieldType.SFTime, AccessType.initializeOnly, 'X3DChaserNode'),
        ('initialDestination', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorChaser'),
        ('initialValue', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorChaser'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 duration=1,
                 initialDestination=(0.8, 0.8, 0.8),
                 initialValue=(0.8, 0.8, 0.8),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorChaser __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.duration = duration
        self.initialDestination = initialDestination
        self.initialValue = initialValue
    @property # getter - - - - - - - - - -
    def duration(self):
        """[0,+infinity) duration is the time interval for filter response in seconds."""
        return self.__duration
    @duration.setter
    def duration(self, duration=None):
        if  duration is None:
            duration = SFTime.DEFAULT_VALUE
        assertValidSFTime(duration)
        assertNonNegative('duration', duration)
        self.__duration = duration
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination=None):
        if  initialDestination is None:
            initialDestination = SFColor.DEFAULT_VALUE
        assertValidSFColor(initialDestination)
        assertZeroToOne('initialDestination', initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue=None):
        if  initialValue is None:
            initialValue = SFColor.DEFAULT_VALUE
        assertValidSFColor(initialValue)
        assertZeroToOne('initialValue', initialValue)
        self.__initialValue = initialValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorChaser.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorChaser'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.duration != 1:
            result += " duration='" + str(self.duration) + "'"
        if self.initialDestination != (0.8, 0.8, 0.8):
            result += " initialDestination='" + str(self.initialDestination) + "'"
        if self.initialValue != (0.8, 0.8, 0.8):
            result += " initialValue='" + str(self.initialValue) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ColorChaser>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ColorDamper(_X3DDamperNode):
    """
    ColorDamper generates a series of RGB color values that progressively change from initial value to destination value.
    """
    NAME = 'ColorDamper'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorDamper'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('initialDestination', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorDamper'),
        ('initialValue', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorDamper'),
        ('order', 3, FieldType.SFInt32, AccessType.initializeOnly, 'X3DDamperNode'),
        ('tau', 0.3, FieldType.SFTime, AccessType.inputOutput, 'X3DDamperNode'),
        ('tolerance', -1, FieldType.SFFloat, AccessType.inputOutput, 'X3DDamperNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 initialDestination=(0.8, 0.8, 0.8),
                 initialValue=(0.8, 0.8, 0.8),
                 order=3,
                 tau=0.3,
                 tolerance=-1,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorDamper __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.initialDestination = initialDestination
        self.initialValue = initialValue
        self.order = order
        self.tau = tau
        self.tolerance = tolerance
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination=None):
        if  initialDestination is None:
            initialDestination = SFColor.DEFAULT_VALUE
        assertValidSFColor(initialDestination)
        assertZeroToOne('initialDestination', initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue=None):
        if  initialValue is None:
            initialValue = SFColor.DEFAULT_VALUE
        assertValidSFColor(initialValue)
        assertZeroToOne('initialValue', initialValue)
        self.__initialValue = initialValue
    @property # getter - - - - - - - - - -
    def order(self):
        """[0,5] order defines the number of internal filters (larger means smoother response, longer delay)."""
        return self.__order
    @order.setter
    def order(self, order=None):
        if  order is None:
            order = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(order)
        assertGreaterThanEquals('order', order, 0)
        assertLessThanEquals('order', order, 5)
        self.__order = order
    @property # getter - - - - - - - - - -
    def tau(self):
        """[0,+infinity) tau is the exponential-decay time constant for filter response in seconds."""
        return self.__tau
    @tau.setter
    def tau(self, tau=None):
        if  tau is None:
            tau = SFTime.DEFAULT_VALUE
        assertValidSFTime(tau)
        assertNonNegative('tau', tau)
        self.__tau = tau
    @property # getter - - - - - - - - - -
    def tolerance(self):
        """[0,+infinity) or -1."""
        return self.__tolerance
    @tolerance.setter
    def tolerance(self, tolerance=None):
        if  tolerance is None:
            tolerance = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(tolerance)
        self.__tolerance = tolerance
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorDamper.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorDamper'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.initialDestination != (0.8, 0.8, 0.8):
            result += " initialDestination='" + str(self.initialDestination) + "'"
        if self.initialValue != (0.8, 0.8, 0.8):
            result += " initialValue='" + str(self.initialValue) + "'"
        if self.order != 3:
            result += " order='" + str(self.order) + "'"
        if self.tau != 0.3:
            result += " tau='" + str(self.tau) + "'"
        if self.tolerance != -1:
            result += " tolerance='" + str(self.tolerance) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ColorDamper>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ColorInterpolator(_X3DInterpolatorNode):
    """
    ColorInterpolator generates a range of color values.
    """
    NAME = 'ColorInterpolator'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorInterpolator'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DInterpolatorNode'),
        ('keyValue', list(), FieldType.MFColor, AccessType.inputOutput, 'ColorInterpolator'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorInterpolator __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key=None):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear interpolation, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue=None):
        if  keyValue is None:
            keyValue = MFColor.DEFAULT_VALUE
        assertValidMFColor(keyValue)
        assertZeroToOne('keyValue', keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorInterpolator.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorInterpolator'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + str(self.key) + "'"
        if self.keyValue != list():
            result += " keyValue='" + str(self.keyValue) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ColorInterpolator>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ColorRGBA(_X3DColorNode):
    """
    ColorRGBA node defines a set of RGBA color values that apply either to a sibling Coordinate|CoordinateDouble node, or else to a parent ElevationGrid node.
    """
    NAME = 'ColorRGBA'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorRGBA'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('color', list(), FieldType.MFColorRGBA, AccessType.inputOutput, 'ColorRGBA'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 color=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorRGBA __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.color = color
    @property # getter - - - - - - - - - -
    def color(self):
        """[0,1] The color field defines an array of 4-tuple RGBA colors."""
        return self.__color
    @color.setter
    def color(self, color=None):
        if  color is None:
            color = MFColorRGBA.DEFAULT_VALUE
        assertValidMFColorRGBA(color)
        assertZeroToOne('color', color)
        self.__color = color
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorRGBA.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorRGBA'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.color != list():
            result += " color='" + str(self.color) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ColorRGBA>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ComposedCubeMapTexture(_X3DEnvironmentTextureNode):
    """
    ComposedCubeMapTexture is a texture node that defines a cubic environment map source as an explicit set of images drawn from individual 2D texture nodes.
    """
    NAME = 'ComposedCubeMapTexture'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedCubeMapTexture'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('back', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('bottom', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('front', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('left', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('right', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('top', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 back=None,
                 bottom=None,
                 front=None,
                 left=None,
                 right=None,
                 top=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedCubeMapTexture __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.back = back
        self.bottom = bottom
        self.front = front
        self.left = left
        self.right = right
        self.top = top
    @property # getter - - - - - - - - - -
    def back(self):
        return self.__back
    @back.setter
    def back(self, back=None):
        if  back is None:
            back = SFNode.DEFAULT_VALUE
        assertValidSFNode(back)
        if not isinstance(back, object):
            # print(flush=True)
            raise X3DTypeError(str(back) + ' does not have a valid node type object')
        self.__back = back
    @property # getter - - - - - - - - - -
    def bottom(self):
        return self.__bottom
    @bottom.setter
    def bottom(self, bottom=None):
        if  bottom is None:
            bottom = SFNode.DEFAULT_VALUE
        assertValidSFNode(bottom)
        if not isinstance(bottom, object):
            # print(flush=True)
            raise X3DTypeError(str(bottom) + ' does not have a valid node type object')
        self.__bottom = bottom
    @property # getter - - - - - - - - - -
    def front(self):
        return self.__front
    @front.setter
    def front(self, front=None):
        if  front is None:
            front = SFNode.DEFAULT_VALUE
        assertValidSFNode(front)
        if not isinstance(front, object):
            # print(flush=True)
            raise X3DTypeError(str(front) + ' does not have a valid node type object')
        self.__front = front
    @property # getter - - - - - - - - - -
    def left(self):
        return self.__left
    @left.setter
    def left(self, left=None):
        if  left is None:
            left = SFNode.DEFAULT_VALUE
        assertValidSFNode(left)
        if not isinstance(left, object):
            # print(flush=True)
            raise X3DTypeError(str(left) + ' does not have a valid node type object')
        self.__left = left
    @property # getter - - - - - - - - - -
    def right(self):
        return self.__right
    @right.setter
    def right(self, right=None):
        if  right is None:
            right = SFNode.DEFAULT_VALUE
        assertValidSFNode(right)
        if not isinstance(right, object):
            # print(flush=True)
            raise X3DTypeError(str(right) + ' does not have a valid node type object')
        self.__right = right
    @property # getter - - - - - - - - - -
    def top(self):
        return self.__top
    @top.setter
    def top(self, top=None):
        if  top is None:
            top = SFNode.DEFAULT_VALUE
        assertValidSFNode(top)
        if not isinstance(top, object):
            # print(flush=True)
            raise X3DTypeError(str(top) + ' does not have a valid node type object')
        self.__top = top
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.back) or bool(self.bottom) or bool(self.front) or bool(self.IS) or bool(self.left) or bool(self.metadata) or bool(self.right) or bool(self.top)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedCubeMapTexture.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedCubeMapTexture'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.back: # walk each child node, if any
                for each in self.back:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.bottom: # walk each child node, if any
                for each in self.bottom:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.front: # walk each child node, if any
                for each in self.front:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.left: # walk each child node, if any
                for each in self.left:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.right: # walk each child node, if any
                for each in self.right:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.top: # walk each child node, if any
                for each in self.top:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ComposedCubeMapTexture>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ComposedShader(_X3DShaderNode): # , _X3DProgrammableShaderObject # TODO fix additional inheritance method resolution order (MRO)
    """
    ComposedShader can contain field declarations, but no CDATA section of plain-text source code, since programs are composed from child ShaderPart nodes.
    """
    NAME = 'ComposedShader'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedShader'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('language', '', FieldType.SFString, AccessType.initializeOnly, 'X3DShaderNode'),
        ('field', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedShader'),
        ('parts', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedShader'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedShader'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedShader')]
    def __init__(self,
                 language='',
                 field=list(),
                 parts=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedShader __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.language = language
        self.field = field
        self.parts = parts
    @property # getter - - - - - - - - - -
    def language(self):
        """The language field indicates to the X3D player which shading language is used."""
        return self.__language
    @language.setter
    def language(self, language=None):
        if  language is None:
            language = SFString.DEFAULT_VALUE
        assertValidSFString(language)
        self.__language = language
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in the ComposedShader node."""
        return self.__field
    @field.setter
    def field(self, field=None):
        if  field is None:
            field = MFNode.DEFAULT_VALUE
        # TODO type-aware checks for field
        if field: # walk each child node, if any
            for each in field:
                assertValidFieldInitializationValue(each.name, each.type, each.value, parent='ComposedShader')
        self.__field = field
    @property # getter - - - - - - - - - -
    def parts(self):
        """[ShaderPart] ComposedShader can contain multiple ShaderPart nodes in the parts field."""
        return self.__parts
    @parts.setter
    def parts(self, parts=None):
        if  parts is None:
            parts = MFNode.DEFAULT_VALUE
        assertValidMFNode(parts)
        self.__parts = parts
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.field) or bool(self.parts) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedShader.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedShader'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.language:
            result += " language='" + self.language + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.field: # walk each child node, if any
                for each in self.field:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.parts: # walk each child node, if any
                for each in self.parts:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ComposedShader>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ComposedTexture3D(_X3DTexture3DNode):
    """
    ComposedTexture3D defines a 3D image-based texture map as a collection of 2D texture sources at various depths.
    """
    NAME = 'ComposedTexture3D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedTexture3D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('repeatR', False, FieldType.SFBool, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('repeatS', False, FieldType.SFBool, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('repeatT', False, FieldType.SFBool, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('textureProperties', None, FieldType.SFNode, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('texture', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedTexture3D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 repeatR=False,
                 repeatS=False,
                 repeatT=False,
                 textureProperties=None,
                 texture=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedTexture3D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.repeatR = repeatR
        self.repeatS = repeatS
        self.repeatT = repeatT
        self.textureProperties = textureProperties
        self.texture = texture
    @property # getter - - - - - - - - - -
    def repeatR(self):
        """Whether to vertically repeat texture along R axis."""
        return self.__repeatR
    @repeatR.setter
    def repeatR(self, repeatR=None):
        if  repeatR is None:
            repeatR = SFBool.DEFAULT_VALUE
        assertValidSFBool(repeatR)
        self.__repeatR = repeatR
    @property # getter - - - - - - - - - -
    def repeatS(self):
        """Whether to horizontally repeat texture along S axis."""
        return self.__repeatS
    @repeatS.setter
    def repeatS(self, repeatS=None):
        if  repeatS is None:
            repeatS = SFBool.DEFAULT_VALUE
        assertValidSFBool(repeatS)
        self.__repeatS = repeatS
    @property # getter - - - - - - - - - -
    def repeatT(self):
        """Whether to vertically repeat texture along T axis."""
        return self.__repeatT
    @repeatT.setter
    def repeatT(self, repeatT=None):
        if  repeatT is None:
            repeatT = SFBool.DEFAULT_VALUE
        assertValidSFBool(repeatT)
        self.__repeatT = repeatT
    @property # getter - - - - - - - - - -
    def textureProperties(self):
        return self.__textureProperties
    @textureProperties.setter
    def textureProperties(self, textureProperties=None):
        if  textureProperties is None:
            textureProperties = SFNode.DEFAULT_VALUE
        assertValidSFNode(textureProperties)
        if not isinstance(textureProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(textureProperties) + ' does not have a valid node type object')
        self.__textureProperties = textureProperties
    @property # getter - - - - - - - - - -
    def texture(self):
        return self.__texture
    @texture.setter
    def texture(self, texture=None):
        if  texture is None:
            texture = MFNode.DEFAULT_VALUE
        assertValidMFNode(texture)
        self.__texture = texture
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.texture) or bool(self.IS) or bool(self.metadata) or bool(self.textureProperties)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedTexture3D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedTexture3D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.repeatR != False:
            result += " repeatR='" + str(self.repeatR) + "'"
        if self.repeatS != False:
            result += " repeatS='" + str(self.repeatS) + "'"
        if self.repeatT != False:
            result += " repeatT='" + str(self.repeatT) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.textureProperties: # walk each child node, if any
                for each in self.textureProperties:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.texture: # walk each child node, if any
                for each in self.texture:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ComposedTexture3D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ComposedVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    ComposedVolumeStyle allows compositing multiple rendering styles into single rendering pass.
    """
    NAME = 'ComposedVolumeStyle'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedVolumeStyle'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('renderStyle', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 renderStyle=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.renderStyle = renderStyle
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def renderStyle(self):
        """[X3DComposableVolumeRenderStyleNode] List of contributing rendering style nodes or node references that can be applied to the object."""
        return self.__renderStyle
    @renderStyle.setter
    def renderStyle(self, renderStyle=None):
        if  renderStyle is None:
            renderStyle = MFNode.DEFAULT_VALUE
        assertValidMFNode(renderStyle)
        self.__renderStyle = renderStyle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.renderStyle) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedVolumeStyle.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.renderStyle: # walk each child node, if any
                for each in self.renderStyle:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ComposedVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Cone(_X3DGeometryNode):
    """
    Cone is a geometry node.
    """
    NAME = 'Cone'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Cone'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bottom', True, FieldType.SFBool, AccessType.initializeOnly, 'Cone'),
        ('bottomRadius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Cone'),
        ('height', 2, FieldType.SFFloat, AccessType.initializeOnly, 'Cone'),
        ('side', True, FieldType.SFBool, AccessType.initializeOnly, 'Cone'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'Cone'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bottom=True,
                 bottomRadius=1,
                 height=2,
                 side=True,
                 solid=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Cone __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bottom = bottom
        self.bottomRadius = bottomRadius
        self.height = height
        self.side = side
        self.solid = solid
    @property # getter - - - - - - - - - -
    def bottom(self):
        """Whether to draw bottom (other inside faces are not drawn)."""
        return self.__bottom
    @bottom.setter
    def bottom(self, bottom=None):
        if  bottom is None:
            bottom = SFBool.DEFAULT_VALUE
        assertValidSFBool(bottom)
        self.__bottom = bottom
    @property # getter - - - - - - - - - -
    def bottomRadius(self):
        """(0,+infinity) Size in meters."""
        return self.__bottomRadius
    @bottomRadius.setter
    def bottomRadius(self, bottomRadius=None):
        if  bottomRadius is None:
            bottomRadius = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(bottomRadius)
        assertPositive('bottomRadius', bottomRadius)
        self.__bottomRadius = bottomRadius
    @property # getter - - - - - - - - - -
    def height(self):
        """(0,+infinity) Size in meters."""
        return self.__height
    @height.setter
    def height(self, height=None):
        if  height is None:
            height = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(height)
        assertPositive('height', height)
        self.__height = height
    @property # getter - - - - - - - - - -
    def side(self):
        """Whether to draw sides (other inside faces are not drawn)."""
        return self.__side
    @side.setter
    def side(self, side=None):
        if  side is None:
            side = SFBool.DEFAULT_VALUE
        assertValidSFBool(side)
        self.__side = side
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Cone.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Cone'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bottom != True:
            result += " bottom='" + str(self.bottom) + "'"
        if self.bottomRadius != 1:
            result += " bottomRadius='" + str(self.bottomRadius) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.height != 2:
            result += " height='" + str(self.height) + "'"
        if self.side != True:
            result += " side='" + str(self.side) + "'"
        if self.solid != True:
            result += " solid='" + str(self.solid) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Cone>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ConeEmitter(_X3DParticleEmitterNode):
    """
    ConeEmitter generates all available particles from a specific point in space.
    """
    NAME = 'ConeEmitter'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ConeEmitter'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('angle', 0.7854, FieldType.SFFloat, AccessType.inputOutput, 'ConeEmitter'),
        ('direction', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'ConeEmitter'),
        ('mass', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('position', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'ConeEmitter'),
        ('speed', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('surfaceArea', 0, FieldType.SFFloat, AccessType.initializeOnly, 'X3DParticleEmitterNode'),
        ('variation', 0.25, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 angle=0.7854,
                 direction=(0, 1, 0),
                 mass=0,
                 position=(0, 0, 0),
                 speed=0,
                 surfaceArea=0,
                 variation=0.25,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ConeEmitter __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.angle = angle
        self.direction = direction
        self.mass = mass
        self.position = position
        self.speed = speed
        self.surfaceArea = surfaceArea
        self.variation = variation
    @property # getter - - - - - - - - - -
    def angle(self):
        """[0,+infinity) Cone boundary for random distribution of particles about initial direction."""
        return self.__angle
    @angle.setter
    def angle(self, angle=None):
        if  angle is None:
            angle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(angle)
        assertGreaterThanEquals('angle', angle, 0)
        assertLessThanEquals('angle', angle, 3.1416)
        self.__angle = angle
    @property # getter - - - - - - - - - -
    def direction(self):
        """Initial direction from which particles emanate."""
        return self.__direction
    @direction.setter
    def direction(self, direction=None):
        if  direction is None:
            direction = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(direction)
        assertGreaterThanEquals('direction', direction, -1)
        assertLessThanEquals('direction', direction, 1)
        self.__direction = direction
    @property # getter - - - - - - - - - -
    def mass(self):
        return self.__mass
    @mass.setter
    def mass(self, mass=None):
        if  mass is None:
            mass = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(mass)
        assertNonNegative('mass', mass)
        self.__mass = mass
    @property # getter - - - - - - - - - -
    def position(self):
        """Point from which particles emanate."""
        return self.__position
    @position.setter
    def position(self, position=None):
        if  position is None:
            position = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(position)
        self.__position = position
    @property # getter - - - - - - - - - -
    def speed(self):
        """[0,+infinity) Initial linear speed (default is m/s) imparted to all particles along their direction of movement."""
        return self.__speed
    @speed.setter
    def speed(self, speed=None):
        if  speed is None:
            speed = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(speed)
        assertNonNegative('speed', speed)
        self.__speed = speed
    @property # getter - - - - - - - - - -
    def surfaceArea(self):
        """[0,+infinity) Particle surface area in area base units (default is meters squared)."""
        return self.__surfaceArea
    @surfaceArea.setter
    def surfaceArea(self, surfaceArea=None):
        if  surfaceArea is None:
            surfaceArea = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(surfaceArea)
        assertNonNegative('surfaceArea', surfaceArea)
        self.__surfaceArea = surfaceArea
    @property # getter - - - - - - - - - -
    def variation(self):
        """[0,+infinity) Multiplier for the randomness used to control the range of possible output values."""
        return self.__variation
    @variation.setter
    def variation(self, variation=None):
        if  variation is None:
            variation = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(variation)
        assertNonNegative('variation', variation)
        self.__variation = variation
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ConeEmitter.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ConeEmitter'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.angle != 0.7854:
            result += " angle='" + str(self.angle) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.direction != (0, 1, 0):
            result += " direction='" + str(self.direction) + "'"
        if self.mass != 0:
            result += " mass='" + str(self.mass) + "'"
        if self.position != (0, 0, 0):
            result += " position='" + str(self.position) + "'"
        if self.speed != 0:
            result += " speed='" + str(self.speed) + "'"
        if self.surfaceArea != 0:
            result += " surfaceArea='" + str(self.surfaceArea) + "'"
        if self.variation != 0.25:
            result += " variation='" + str(self.variation) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ConeEmitter>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Contact(_X3DNode):
    """
    Contact nodes are produced as output events when two collidable objects or spaces make contact.
    """
    NAME = 'Contact'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Contact'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('appliedParameters', ["BOUNCE"], FieldType.MFString, AccessType.inputOutput, 'Contact'),
        ('bounce', 0, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('contactNormal', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Contact'),
        ('depth', 0, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('frictionCoefficients', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'Contact'),
        ('frictionDirection', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Contact'),
        ('minBounceSpeed', 0, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('position', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Contact'),
        ('slipCoefficients', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'Contact'),
        ('softnessConstantForceMix', 0.0001, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('softnessErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('surfaceSpeed', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'Contact'),
        ('body1', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('body2', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('geometry1', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('geometry2', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 appliedParameters=["BOUNCE"],
                 bounce=0,
                 contactNormal=(0, 1, 0),
                 depth=0,
                 frictionCoefficients=(0, 0),
                 frictionDirection=(0, 1, 0),
                 minBounceSpeed=0,
                 position=(0, 0, 0),
                 slipCoefficients=(0, 0),
                 softnessConstantForceMix=0.0001,
                 softnessErrorCorrection=0.8,
                 surfaceSpeed=(0, 0),
                 body1=None,
                 body2=None,
                 geometry1=None,
                 geometry2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Contact __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.appliedParameters = appliedParameters
        self.bounce = bounce
        self.contactNormal = contactNormal
        self.depth = depth
        self.frictionCoefficients = frictionCoefficients
        self.frictionDirection = frictionDirection
        self.minBounceSpeed = minBounceSpeed
        self.position = position
        self.slipCoefficients = slipCoefficients
        self.softnessConstantForceMix = softnessConstantForceMix
        self.softnessErrorCorrection = softnessErrorCorrection
        self.surfaceSpeed = surfaceSpeed
        self.body1 = body1
        self.body2 = body2
        self.geometry1 = geometry1
        self.geometry2 = geometry2
    @property # getter - - - - - - - - - -
    def appliedParameters(self):
        """Default global parameters for collision outputs of rigid body physics system."""
        return self.__appliedParameters
    @appliedParameters.setter
    def appliedParameters(self, appliedParameters=None):
        if  appliedParameters is None:
            appliedParameters = MFString.DEFAULT_VALUE
        assertValidMFString(appliedParameters)
        self.__appliedParameters = appliedParameters
    @property # getter - - - - - - - - - -
    def bounce(self):
        """[0,1] bounce indicates bounciness (0 = no bounce at all, 1 = maximum bounce)."""
        return self.__bounce
    @bounce.setter
    def bounce(self, bounce=None):
        if  bounce is None:
            bounce = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(bounce)
        assertZeroToOne('bounce', bounce)
        self.__bounce = bounce
    @property # getter - - - - - - - - - -
    def contactNormal(self):
        """contactNormal is unit vector describing normal between two colliding bodies."""
        return self.__contactNormal
    @contactNormal.setter
    def contactNormal(self, contactNormal=None):
        if  contactNormal is None:
            contactNormal = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(contactNormal)
        self.__contactNormal = contactNormal
    @property # getter - - - - - - - - - -
    def depth(self):
        """[0,1] depth indicates how deep the current intersection is along normal vector."""
        return self.__depth
    @depth.setter
    def depth(self, depth=None):
        if  depth is None:
            depth = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(depth)
        self.__depth = depth
    @property # getter - - - - - - - - - -
    def frictionCoefficients(self):
        """frictionCoefficients used for computing surface drag."""
        return self.__frictionCoefficients
    @frictionCoefficients.setter
    def frictionCoefficients(self, frictionCoefficients=None):
        if  frictionCoefficients is None:
            frictionCoefficients = SFVec2f.DEFAULT_VALUE
        assertValidSFVec2f(frictionCoefficients)
        assertNonNegative('frictionCoefficients', frictionCoefficients)
        self.__frictionCoefficients = frictionCoefficients
    @property # getter - - - - - - - - - -
    def frictionDirection(self):
        """frictionDirection controls friction vector."""
        return self.__frictionDirection
    @frictionDirection.setter
    def frictionDirection(self, frictionDirection=None):
        if  frictionDirection is None:
            frictionDirection = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(frictionDirection)
        self.__frictionDirection = frictionDirection
    @property # getter - - - - - - - - - -
    def minBounceSpeed(self):
        """[0,+infinity) minBounceSpeed m/s needed to bounce."""
        return self.__minBounceSpeed
    @minBounceSpeed.setter
    def minBounceSpeed(self, minBounceSpeed=None):
        if  minBounceSpeed is None:
            minBounceSpeed = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(minBounceSpeed)
        assertNonNegative('minBounceSpeed', minBounceSpeed)
        self.__minBounceSpeed = minBounceSpeed
    @property # getter - - - - - - - - - -
    def position(self):
        """position (x, y, z in meters) of exact location of collision."""
        return self.__position
    @position.setter
    def position(self, position=None):
        if  position is None:
            position = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(position)
        self.__position = position
    @property # getter - - - - - - - - - -
    def slipCoefficients(self):
        """slipCoefficients used for computing surface drag."""
        return self.__slipCoefficients
    @slipCoefficients.setter
    def slipCoefficients(self, slipCoefficients=None):
        if  slipCoefficients is None:
            slipCoefficients = SFVec2f.DEFAULT_VALUE
        assertValidSFVec2f(slipCoefficients)
        self.__slipCoefficients = slipCoefficients
    @property # getter - - - - - - - - - -
    def softnessConstantForceMix(self):
        """[0,1] softnessConstantForceMix value applies a constant force value to make colliding surfaces appear to be somewhat soft."""
        return self.__softnessConstantForceMix
    @softnessConstantForceMix.setter
    def softnessConstantForceMix(self, softnessConstantForceMix=None):
        if  softnessConstantForceMix is None:
            softnessConstantForceMix = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(softnessConstantForceMix)
        assertZeroToOne('softnessConstantForceMix', softnessConstantForceMix)
        self.__softnessConstantForceMix = softnessConstantForceMix
    @property # getter - - - - - - - - - -
    def softnessErrorCorrection(self):
        """[0,1] softnessErrorCorrection indicates fraction of collision error fixed in a set of evaluations (0 = no error correction, 1 = all errors corrected in single step)."""
        return self.__softnessErrorCorrection
    @softnessErrorCorrection.setter
    def softnessErrorCorrection(self, softnessErrorCorrection=None):
        if  softnessErrorCorrection is None:
            softnessErrorCorrection = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(softnessErrorCorrection)
        assertZeroToOne('softnessErrorCorrection', softnessErrorCorrection)
        self.__softnessErrorCorrection = softnessErrorCorrection
    @property # getter - - - - - - - - - -
    def surfaceSpeed(self):
        """surfaceSpeed defines speed vectors for computing surface drag, if contact surfaces move independently of bodies."""
        return self.__surfaceSpeed
    @surfaceSpeed.setter
    def surfaceSpeed(self, surfaceSpeed=None):
        if  surfaceSpeed is None:
            surfaceSpeed = SFVec2f.DEFAULT_VALUE
        assertValidSFVec2f(surfaceSpeed)
        self.__surfaceSpeed = surfaceSpeed
    @property # getter - - - - - - - - - -
    def body1(self):
        return self.__body1
    @body1.setter
    def body1(self, body1=None):
        if  body1 is None:
            body1 = SFNode.DEFAULT_VALUE
        assertValidSFNode(body1)
        if not isinstance(body1, object):
            # print(flush=True)
            raise X3DTypeError(str(body1) + ' does not have a valid node type object')
        self.__body1 = body1
    @property # getter - - - - - - - - - -
    def body2(self):
        return self.__body2
    @body2.setter
    def body2(self, body2=None):
        if  body2 is None:
            body2 = SFNode.DEFAULT_VALUE
        assertValidSFNode(body2)
        if not isinstance(body2, object):
            # print(flush=True)
            raise X3DTypeError(str(body2) + ' does not have a valid node type object')
        self.__body2 = body2
    @property # getter - - - - - - - - - -
    def geometry1(self):
        return self.__geometry1
    @geometry1.setter
    def geometry1(self, geometry1=None):
        if  geometry1 is None:
            geometry1 = SFNode.DEFAULT_VALUE
        assertValidSFNode(geometry1)
        if not isinstance(geometry1, object):
            # print(flush=True)
            raise X3DTypeError(str(geometry1) + ' does not have a valid node type object')
        self.__geometry1 = geometry1
    @property # getter - - - - - - - - - -
    def geometry2(self):
        return self.__geometry2
    @geometry2.setter
    def geometry2(self, geometry2=None):
        if  geometry2 is None:
            geometry2 = SFNode.DEFAULT_VALUE
        assertValidSFNode(geometry2)
        if not isinstance(geometry2, object):
            # print(flush=True)
            raise X3DTypeError(str(geometry2) + ' does not have a valid node type object')
        self.__geometry2 = geometry2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.body1) or bool(self.body2) or bool(self.geometry1) or bool(self.geometry2) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Contact.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Contact'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.appliedParameters != ["BOUNCE"]:
            result += " appliedParameters='" + self.appliedParameters + "'"
        if self.bounce != 0:
            result += " bounce='" + str(self.bounce) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.contactNormal != (0, 1, 0):
            result += " contactNormal='" + str(self.contactNormal) + "'"
        if self.depth != 0:
            result += " depth='" + str(self.depth) + "'"
        if self.frictionCoefficients != (0, 0):
            result += " frictionCoefficients='" + str(self.frictionCoefficients) + "'"
        if self.frictionDirection != (0, 1, 0):
            result += " frictionDirection='" + str(self.frictionDirection) + "'"
        if self.minBounceSpeed != 0:
            result += " minBounceSpeed='" + str(self.minBounceSpeed) + "'"
        if self.position != (0, 0, 0):
            result += " position='" + str(self.position) + "'"
        if self.slipCoefficients != (0, 0):
            result += " slipCoefficients='" + str(self.slipCoefficients) + "'"
        if self.softnessConstantForceMix != 0.0001:
            result += " softnessConstantForceMix='" + str(self.softnessConstantForceMix) + "'"
        if self.softnessErrorCorrection != 0.8:
            result += " softnessErrorCorrection='" + str(self.softnessErrorCorrection) + "'"
        if self.surfaceSpeed != (0, 0):
            result += " surfaceSpeed='" + str(self.surfaceSpeed) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.body1: # walk each child node, if any
                for each in self.body1:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.body2: # walk each child node, if any
                for each in self.body2:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geometry1: # walk each child node, if any
                for each in self.geometry1:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geometry2: # walk each child node, if any
                for each in self.geometry2:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Contact>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Contour2D(_X3DNode):
    """
    Contour2D groups a set of curve segments into a composite contour.
    """
    NAME = 'Contour2D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Contour2D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'Contour2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Contour2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        """[NurbsCurve2D|ContourPolyline2D] The children form a closed loop with first point of first child repeated as last point of last child, and the last point of a segment repeated as first point of the consecutive one."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Contour2D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Contour2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Contour2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ContourPolyline2D(_X3DNurbsControlCurveNode):
    """
    ContourPolyline2D defines a linear curve segment as part of a trimming contour in the u-v domain of a NURBS surface.
    """
    NAME = 'ContourPolyline2D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ContourPolyline2D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('controlPoint', list(), FieldType.MFVec2d, AccessType.inputOutput, 'X3DNurbsControlCurveNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 controlPoint=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ContourPolyline2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.controlPoint = controlPoint
    @property # getter - - - - - - - - - -
    def controlPoint(self):
        """controlPoint specifies the end points of each segment of the piecewise linear curve."""
        return self.__controlPoint
    @controlPoint.setter
    def controlPoint(self, controlPoint=None):
        if  controlPoint is None:
            controlPoint = MFVec2d.DEFAULT_VALUE
        assertValidMFVec2d(controlPoint)
        self.__controlPoint = controlPoint
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ContourPolyline2D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ContourPolyline2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.controlPoint != list():
            result += " controlPoint='" + str(self.controlPoint) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ContourPolyline2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Coordinate(_X3DCoordinateNode):
    """
    Coordinate builds geometry by defining a set of 3D coordinate (triplet) point values.
    """
    NAME = 'Coordinate'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Coordinate'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('point', list(), FieldType.MFVec3f, AccessType.inputOutput, 'Coordinate'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 point=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Coordinate __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.point = point
    @property # getter - - - - - - - - - -
    def point(self):
        """point contains a set of 3D coordinate (triplet) point values."""
        return self.__point
    @point.setter
    def point(self, point=None):
        if  point is None:
            point = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(point)
        self.__point = point
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Coordinate.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Coordinate'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.point != list():
            result += " point='" + str(self.point) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Coordinate>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CoordinateChaser(_X3DChaserNode):
    """
    CoordinateChaser generates a series of coordinate arrays that progressively change from initial value to destination value.
    """
    NAME = 'CoordinateChaser'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateChaser'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('duration', 1, FieldType.SFTime, AccessType.initializeOnly, 'X3DChaserNode'),
        ('initialDestination', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateChaser'),
        ('initialValue', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateChaser'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 duration=1,
                 initialDestination=[(0, 0, 0)],
                 initialValue=[(0, 0, 0)],
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateChaser __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.duration = duration
        self.initialDestination = initialDestination
        self.initialValue = initialValue
    @property # getter - - - - - - - - - -
    def duration(self):
        """[0,+infinity) duration is the time interval for filter response in seconds."""
        return self.__duration
    @duration.setter
    def duration(self, duration=None):
        if  duration is None:
            duration = SFTime.DEFAULT_VALUE
        assertValidSFTime(duration)
        assertNonNegative('duration', duration)
        self.__duration = duration
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination=None):
        if  initialDestination is None:
            initialDestination = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue=None):
        if  initialValue is None:
            initialValue = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(initialValue)
        self.__initialValue = initialValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateChaser.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateChaser'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.duration != 1:
            result += " duration='" + str(self.duration) + "'"
        if self.initialDestination != [(0, 0, 0)]:
            result += " initialDestination='" + str(self.initialDestination) + "'"
        if self.initialValue != [(0, 0, 0)]:
            result += " initialValue='" + str(self.initialValue) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CoordinateChaser>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CoordinateDamper(_X3DDamperNode):
    """
    CoordinateDamper generates a series of coordinate arrays that progressively change from initial value to destination value.
    """
    NAME = 'CoordinateDamper'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateDamper'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('initialDestination', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateDamper'),
        ('initialValue', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateDamper'),
        ('order', 3, FieldType.SFInt32, AccessType.initializeOnly, 'X3DDamperNode'),
        ('tau', 0.3, FieldType.SFTime, AccessType.inputOutput, 'X3DDamperNode'),
        ('tolerance', -1, FieldType.SFFloat, AccessType.inputOutput, 'X3DDamperNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 initialDestination=[(0, 0, 0)],
                 initialValue=[(0, 0, 0)],
                 order=3,
                 tau=0.3,
                 tolerance=-1,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateDamper __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.initialDestination = initialDestination
        self.initialValue = initialValue
        self.order = order
        self.tau = tau
        self.tolerance = tolerance
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination=None):
        if  initialDestination is None:
            initialDestination = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue=None):
        if  initialValue is None:
            initialValue = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(initialValue)
        self.__initialValue = initialValue
    @property # getter - - - - - - - - - -
    def order(self):
        """[0,5] order defines the number of internal filters (larger means smoother response, longer delay)."""
        return self.__order
    @order.setter
    def order(self, order=None):
        if  order is None:
            order = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(order)
        assertGreaterThanEquals('order', order, 0)
        assertLessThanEquals('order', order, 5)
        self.__order = order
    @property # getter - - - - - - - - - -
    def tau(self):
        """[0,+infinity) tau is the exponential-decay time constant for filter response in seconds."""
        return self.__tau
    @tau.setter
    def tau(self, tau=None):
        if  tau is None:
            tau = SFTime.DEFAULT_VALUE
        assertValidSFTime(tau)
        assertNonNegative('tau', tau)
        self.__tau = tau
    @property # getter - - - - - - - - - -
    def tolerance(self):
        """[0,+infinity) or -1."""
        return self.__tolerance
    @tolerance.setter
    def tolerance(self, tolerance=None):
        if  tolerance is None:
            tolerance = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(tolerance)
        self.__tolerance = tolerance
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateDamper.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateDamper'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.initialDestination != [(0, 0, 0)]:
            result += " initialDestination='" + str(self.initialDestination) + "'"
        if self.initialValue != [(0, 0, 0)]:
            result += " initialValue='" + str(self.initialValue) + "'"
        if self.order != 3:
            result += " order='" + str(self.order) + "'"
        if self.tau != 0.3:
            result += " tau='" + str(self.tau) + "'"
        if self.tolerance != -1:
            result += " tolerance='" + str(self.tolerance) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CoordinateDamper>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CoordinateDouble(_X3DCoordinateNode):
    """
    CoordinateDouble builds geometry by defining a set of 3D coordinate (triplet) point values.
    """
    NAME = 'CoordinateDouble'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateDouble'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('point', list(), FieldType.MFVec3d, AccessType.inputOutput, 'CoordinateDouble'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 point=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateDouble __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.point = point
    @property # getter - - - - - - - - - -
    def point(self):
        """point contains a set of 3D coordinate (triplet) point values."""
        return self.__point
    @point.setter
    def point(self, point=None):
        if  point is None:
            point = MFVec3d.DEFAULT_VALUE
        assertValidMFVec3d(point)
        self.__point = point
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateDouble.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateDouble'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.point != list():
            result += " point='" + str(self.point) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CoordinateDouble>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CoordinateInterpolator(_X3DInterpolatorNode):
    """
    CoordinateInterpolator linearly interpolates among a list of 3-tuple MFVec3f arrays, producing a single MFVec3f array that is fractional average between two nearest arrays in the list.
    """
    NAME = 'CoordinateInterpolator'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateInterpolator'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DInterpolatorNode'),
        ('keyValue', list(), FieldType.MFVec3f, AccessType.inputOutput, 'CoordinateInterpolator'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateInterpolator __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key=None):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear interpolation, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue=None):
        if  keyValue is None:
            keyValue = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateInterpolator.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateInterpolator'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + str(self.key) + "'"
        if self.keyValue != list():
            result += " keyValue='" + str(self.keyValue) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CoordinateInterpolator>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CoordinateInterpolator2D(_X3DInterpolatorNode):
    """
    CoordinateInterpolator2D generates a series of SFVec2f or MFVec2f 2-tuple float values.
    """
    NAME = 'CoordinateInterpolator2D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateInterpolator2D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DInterpolatorNode'),
        ('keyValue', list(), FieldType.MFVec2f, AccessType.inputOutput, 'CoordinateInterpolator2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateInterpolator2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key=None):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear interpolation, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue=None):
        if  keyValue is None:
            keyValue = MFVec2f.DEFAULT_VALUE
        assertValidMFVec2f(keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateInterpolator2D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateInterpolator2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + str(self.key) + "'"
        if self.keyValue != list():
            result += " keyValue='" + str(self.keyValue) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CoordinateInterpolator2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Cylinder(_X3DGeometryNode):
    """
    Cylinder is a geometry node.
    """
    NAME = 'Cylinder'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Cylinder'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bottom', True, FieldType.SFBool, AccessType.initializeOnly, 'Cylinder'),
        ('height', 2, FieldType.SFFloat, AccessType.initializeOnly, 'Cylinder'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Cylinder'),
        ('side', True, FieldType.SFBool, AccessType.initializeOnly, 'Cylinder'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'Cylinder'),
        ('top', True, FieldType.SFBool, AccessType.inputOutput, 'Cylinder'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bottom=True,
                 height=2,
                 radius=1,
                 side=True,
                 solid=True,
                 top=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Cylinder __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bottom = bottom
        self.height = height
        self.radius = radius
        self.side = side
        self.solid = solid
        self.top = top
    @property # getter - - - - - - - - - -
    def bottom(self):
        """Whether to draw bottom (inside faces are never drawn)."""
        return self.__bottom
    @bottom.setter
    def bottom(self, bottom=None):
        if  bottom is None:
            bottom = SFBool.DEFAULT_VALUE
        assertValidSFBool(bottom)
        self.__bottom = bottom
    @property # getter - - - - - - - - - -
    def height(self):
        """(0,+infinity) Size in meters."""
        return self.__height
    @height.setter
    def height(self, height=None):
        if  height is None:
            height = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(height)
        assertPositive('height', height)
        self.__height = height
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) Size in meters."""
        return self.__radius
    @radius.setter
    def radius(self, radius=None):
        if  radius is None:
            radius = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def side(self):
        """Whether to draw sides (inside faces are never drawn)."""
        return self.__side
    @side.setter
    def side(self, side=None):
        if  side is None:
            side = SFBool.DEFAULT_VALUE
        assertValidSFBool(side)
        self.__side = side
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def top(self):
        """Whether to draw top (inside faces are never drawn)."""
        return self.__top
    @top.setter
    def top(self, top=None):
        if  top is None:
            top = SFBool.DEFAULT_VALUE
        assertValidSFBool(top)
        self.__top = top
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Cylinder.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Cylinder'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bottom != True:
            result += " bottom='" + str(self.bottom) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.height != 2:
            result += " height='" + str(self.height) + "'"
        if self.radius != 1:
            result += " radius='" + str(self.radius) + "'"
        if self.side != True:
            result += " side='" + str(self.side) + "'"
        if self.solid != True:
            result += " solid='" + str(self.solid) + "'"
        if self.top != True:
            result += " top='" + str(self.top) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Cylinder>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class CylinderSensor(_X3DDragSensorNode):
    """
    CylinderSensor converts pointer motion (for example, a mouse or wand) into rotation values using an invisible cylinder aligned with local Y-axis.
    """
    NAME = 'CylinderSensor'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CylinderSensor'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('autoOffset', True, FieldType.SFBool, AccessType.inputOutput, 'X3DDragSensorNode'),
        ('axisRotation', (0, 1, 0, 0), FieldType.SFRotation, AccessType.inputOutput, 'CylinderSensor'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DPointingDeviceSensorNode'),
        ('diskAngle', 0.26179167, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('maxAngle', -1, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('minAngle', 0, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('offset', 0, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 autoOffset=True,
                 axisRotation=(0, 1, 0, 0),
                 description='',
                 diskAngle=0.26179167,
                 enabled=True,
                 maxAngle=-1,
                 minAngle=0,
                 offset=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CylinderSensor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.autoOffset = autoOffset
        self.axisRotation = axisRotation
        self.description = description
        self.diskAngle = diskAngle
        self.enabled = enabled
        self.maxAngle = maxAngle
        self.minAngle = minAngle
        self.offset = offset
    @property # getter - - - - - - - - - -
    def autoOffset(self):
        """determines whether previous offset values are remembered/accumulated."""
        return self.__autoOffset
    @autoOffset.setter
    def autoOffset(self, autoOffset=None):
        if  autoOffset is None:
            autoOffset = SFBool.DEFAULT_VALUE
        assertValidSFBool(autoOffset)
        self.__autoOffset = autoOffset
    @property # getter - - - - - - - - - -
    def axisRotation(self):
        """axisRotation determines local sensor coordinate system by rotating the local coordinate system."""
        return self.__axisRotation
    @axisRotation.setter
    def axisRotation(self, axisRotation=None):
        if  axisRotation is None:
            axisRotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(axisRotation)
        self.__axisRotation = axisRotation
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def diskAngle(self):
        """Help decide rotation behavior from initial relative bearing of pointer drag: acute angle whether cylinder sides or end-cap disks of virtual-geometry sensor are used for manipulation."""
        return self.__diskAngle
    @diskAngle.setter
    def diskAngle(self, diskAngle=None):
        if  diskAngle is None:
            diskAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(diskAngle)
        assertGreaterThanEquals('diskAngle', diskAngle, 0)
        assertLessThanEquals('diskAngle', diskAngle, 1.5708)
        self.__diskAngle = diskAngle
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def maxAngle(self):
        """clamps rotation_changed events within range of min/max values Hint: if minAngle > maxAngle, rotation is not clamped."""
        return self.__maxAngle
    @maxAngle.setter
    def maxAngle(self, maxAngle=None):
        if  maxAngle is None:
            maxAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(maxAngle)
        assertGreaterThan('maxAngle', maxAngle, -6.2832)
        assertLessThan('maxAngle', maxAngle, 6.2832)
        self.__maxAngle = maxAngle
    @property # getter - - - - - - - - - -
    def minAngle(self):
        """clamps rotation_changed events within range of min/max values Hint: if minAngle > maxAngle, rotation is not clamped."""
        return self.__minAngle
    @minAngle.setter
    def minAngle(self, minAngle=None):
        if  minAngle is None:
            minAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(minAngle)
        assertGreaterThan('minAngle', minAngle, -6.2832)
        assertLessThan('minAngle', minAngle, 6.2832)
        self.__minAngle = minAngle
    @property # getter - - - - - - - - - -
    def offset(self):
        """Sends event and remembers last value sensed."""
        return self.__offset
    @offset.setter
    def offset(self, offset=None):
        if  offset is None:
            offset = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(offset)
        self.__offset = offset
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CylinderSensor.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CylinderSensor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.autoOffset != True:
            result += " autoOffset='" + str(self.autoOffset) + "'"
        if self.axisRotation != (0, 1, 0, 0):
            result += " axisRotation='" + str(self.axisRotation) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.diskAngle != 0.26179167:
            result += " diskAngle='" + str(self.diskAngle) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.maxAngle != -1:
            result += " maxAngle='" + str(self.maxAngle) + "'"
        if self.minAngle != 0:
            result += " minAngle='" + str(self.minAngle) + "'"
        if self.offset != 0:
            result += " offset='" + str(self.offset) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</CylinderSensor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class DirectionalLight(_X3DLightNode):
    """
    DirectionalLight might not be scoped by parent Group or Transform at levels 1 or 2.
    """
    NAME = 'DirectionalLight'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DirectionalLight'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('ambientIntensity', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DLightNode'),
        ('color', (1, 1, 1), FieldType.SFColor, AccessType.inputOutput, 'X3DLightNode'),
        ('direction', (0, 0, -1), FieldType.SFVec3f, AccessType.inputOutput, 'DirectionalLight'),
        ('global_', False, FieldType.SFBool, AccessType.inputOutput, 'DirectionalLight'),
        ('intensity', 1, FieldType.SFFloat, AccessType.inputOutput, 'X3DLightNode'),
        ('on', True, FieldType.SFBool, AccessType.inputOutput, 'X3DLightNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 ambientIntensity=0,
                 color=(1, 1, 1),
                 direction=(0, 0, -1),
                 global_=False,
                 intensity=1,
                 on=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DirectionalLight __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.ambientIntensity = ambientIntensity
        self.color = color
        self.direction = direction
        self.global_ = global_
        self.intensity = intensity
        self.on = on
    @property # getter - - - - - - - - - -
    def ambientIntensity(self):
        """[0,1] Brightness of ambient (nondirectional background) emission from the light."""
        return self.__ambientIntensity
    @ambientIntensity.setter
    def ambientIntensity(self, ambientIntensity=None):
        if  ambientIntensity is None:
            ambientIntensity = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(ambientIntensity)
        assertZeroToOne('ambientIntensity', ambientIntensity)
        self.__ambientIntensity = ambientIntensity
    @property # getter - - - - - - - - - -
    def color(self):
        """[0,1] color of light, applied to colors of objects."""
        return self.__color
    @color.setter
    def color(self, color=None):
        if  color is None:
            color = SFColor.DEFAULT_VALUE
        assertValidSFColor(color)
        assertZeroToOne('color', color)
        self.__color = color
    @property # getter - - - - - - - - - -
    def direction(self):
        """Orientation vector of light relative to local coordinate system."""
        return self.__direction
    @direction.setter
    def direction(self, direction=None):
        if  direction is None:
            direction = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(direction)
        self.__direction = direction
    @property # getter - - - - - - - - - -
    def global_(self):
        return self.__global_
    @global_.setter
    def global_(self, global_=None):
        if  global_ is None:
            global_ = SFBool.DEFAULT_VALUE
        assertValidSFBool(global_)
        self.__global_ = global_
    @property # getter - - - - - - - - - -
    def intensity(self):
        """[0,1] Brightness of direct emission from the light."""
        return self.__intensity
    @intensity.setter
    def intensity(self, intensity=None):
        if  intensity is None:
            intensity = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(intensity)
        assertZeroToOne('intensity', intensity)
        self.__intensity = intensity
    @property # getter - - - - - - - - - -
    def on(self):
        """Enables/disables this light source."""
        return self.__on
    @on.setter
    def on(self, on=None):
        if  on is None:
            on = SFBool.DEFAULT_VALUE
        assertValidSFBool(on)
        self.__on = on
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DirectionalLight.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DirectionalLight'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.ambientIntensity != 0:
            result += " ambientIntensity='" + str(self.ambientIntensity) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.color != (1, 1, 1):
            result += " color='" + str(self.color) + "'"
        if self.direction != (0, 0, -1):
            result += " direction='" + str(self.direction) + "'"
        if self.global_ != False:
            result += " global_='" + str(self.global_) + "'"
        if self.intensity != 1:
            result += " intensity='" + str(self.intensity) + "'"
        if self.on != True:
            result += " on='" + str(self.on) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</DirectionalLight>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class DISEntityManager(_X3DChildNode):
    """
    DISEntityManager notifies a scene when new DIS ESPDU entities arrive or current entities leave.
    """
    NAME = 'DISEntityManager'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DISEntityManager'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('address', 'localhost', FieldType.SFString, AccessType.inputOutput, 'DISEntityManager'),
        ('applicationID', 0, FieldType.SFInt32, AccessType.inputOutput, 'DISEntityManager'),
        ('port', 0, FieldType.SFInt32, AccessType.inputOutput, 'DISEntityManager'),
        ('siteID', 0, FieldType.SFInt32, AccessType.inputOutput, 'DISEntityManager'),
        ('mapping', list(), FieldType.MFNode, AccessType.inputOutput, 'DISEntityManager'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 address='localhost',
                 applicationID=0,
                 port=0,
                 siteID=0,
                 mapping=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DISEntityManager __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.address = address
        self.applicationID = applicationID
        self.port = port
        self.siteID = siteID
        self.mapping = mapping
    @property # getter - - - - - - - - - -
    def address(self):
        """Multicast network address, or else "localhost" example: 224."""
        return self.__address
    @address.setter
    def address(self, address=None):
        if  address is None:
            address = SFString.DEFAULT_VALUE
        assertValidSFString(address)
        self.__address = address
    @property # getter - - - - - - - - - -
    def applicationID(self):
        """Each simulation application that can respond to simulation management PDUs needs to have a unique applicationID."""
        return self.__applicationID
    @applicationID.setter
    def applicationID(self, applicationID=None):
        if  applicationID is None:
            applicationID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(applicationID)
        self.__applicationID = applicationID
    @property # getter - - - - - - - - - -
    def port(self):
        """Multicast network port, for example: 3000."""
        return self.__port
    @port.setter
    def port(self, port=None):
        if  port is None:
            port = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(port)
        self.__port = port
    @property # getter - - - - - - - - - -
    def siteID(self):
        """Simulation/exercise siteID of the participating LAN or organization."""
        return self.__siteID
    @siteID.setter
    def siteID(self, siteID=None):
        if  siteID is None:
            siteID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(siteID)
        self.__siteID = siteID
    @property # getter - - - - - - - - - -
    def mapping(self):
        """[DISEntityTypeMapping] mapping field provides a mechanism for automatically creating an X3D model when a new entity arrives over the network."""
        return self.__mapping
    @mapping.setter
    def mapping(self, mapping=None):
        if  mapping is None:
            mapping = MFNode.DEFAULT_VALUE
        assertValidMFNode(mapping)
        self.__mapping = mapping
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.mapping) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DISEntityManager.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DISEntityManager'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.address != 'localhost':
            result += " address='" + self.address + "'"
        if self.applicationID != 0:
            result += " applicationID='" + str(self.applicationID) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.port != 0:
            result += " port='" + str(self.port) + "'"
        if self.siteID != 0:
            result += " siteID='" + str(self.siteID) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.mapping: # walk each child node, if any
                for each in self.mapping:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</DISEntityManager>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class DISEntityTypeMapping(_X3DInfoNode, _X3DUrlObject):
    """
    DISEntityTypeMapping provides a best-match mapping from DIS ESPDU entity type information to a specific X3D model, thus providing a visual and behavioral representation that best matches the entity type.
    """
    NAME = 'DISEntityTypeMapping'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DISEntityTypeMapping'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('category', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('country', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('domain', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('extra', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('kind', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('specific', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('subcategory', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 category=0,
                 country=0,
                 domain=0,
                 extra=0,
                 kind=0,
                 specific=0,
                 subcategory=0,
                 url=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DISEntityTypeMapping __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.category = category
        self.country = country
        self.domain = domain
        self.extra = extra
        self.kind = kind
        self.specific = specific
        self.subcategory = subcategory
        self.url = url
    @property # getter - - - - - - - - - -
    def category(self):
        """Integer enumerations value for main category that describes the entity, semantics of each code varies according to domain."""
        return self.__category
    @category.setter
    def category(self, category=None):
        if  category is None:
            category = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(category)
        assertGreaterThanEquals('category', category, 0)
        assertLessThanEquals('category', category, 255)
        self.__category = category
    @property # getter - - - - - - - - - -
    def country(self):
        """Integer enumerations value for country to which the design of the entity or its design specification is attributed."""
        return self.__country
    @country.setter
    def country(self, country=None):
        if  country is None:
            country = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(country)
        assertGreaterThanEquals('country', country, 0)
        assertLessThanEquals('country', country, 65535)
        self.__country = country
    @property # getter - - - - - - - - - -
    def domain(self):
        """Integer enumerations value for domain in which the entity operates: LAND, AIR, SURFACE, SUBSURFACE, SPACE or OTHER."""
        return self.__domain
    @domain.setter
    def domain(self, domain=None):
        if  domain is None:
            domain = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(domain)
        assertGreaterThanEquals('domain', domain, 0)
        assertLessThanEquals('domain', domain, 255)
        self.__domain = domain
    @property # getter - - - - - - - - - -
    def extra(self):
        """Any extra information required to describe a particular entity."""
        return self.__extra
    @extra.setter
    def extra(self, extra=None):
        if  extra is None:
            extra = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(extra)
        assertGreaterThanEquals('extra', extra, 0)
        assertLessThanEquals('extra', extra, 255)
        self.__extra = extra
    @property # getter - - - - - - - - - -
    def kind(self):
        """Integer enumerations value for whether entity is a PLATFORM, MUNITION, LIFE_FORM, ENVIRONMENTAL, CULTURAL_FEATURE, SUPPLY, RADIO, EXPENDABLE, SENSOR_EMITTER or OTHER."""
        return self.__kind
    @kind.setter
    def kind(self, kind=None):
        if  kind is None:
            kind = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(kind)
        assertGreaterThanEquals('kind', kind, 0)
        assertLessThanEquals('kind', kind, 255)
        self.__kind = kind
    @property # getter - - - - - - - - - -
    def specific(self):
        """Specific information about an entity based on the Subcategory field."""
        return self.__specific
    @specific.setter
    def specific(self, specific=None):
        if  specific is None:
            specific = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(specific)
        assertGreaterThanEquals('specific', specific, 0)
        assertLessThanEquals('specific', specific, 255)
        self.__specific = specific
    @property # getter - - - - - - - - - -
    def subcategory(self):
        return self.__subcategory
    @subcategory.setter
    def subcategory(self, subcategory=None):
        if  subcategory is None:
            subcategory = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(subcategory)
        assertGreaterThanEquals('subcategory', subcategory, 0)
        assertLessThanEquals('subcategory', subcategory, 255)
        self.__subcategory = subcategory
    @property # getter - - - - - - - - - -
    def url(self):
        """Local and/or online addresses of X3D model of interest, for example: "ExtrusionExampleShip."""
        return self.__url
    @url.setter
    def url(self, url=None):
        if  url is None:
            url = MFString.DEFAULT_VALUE
        assertValidMFString(url)
        self.__url = url
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DISEntityTypeMapping.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DISEntityTypeMapping'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.category != 0:
            result += " category='" + str(self.category) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.country != 0:
            result += " country='" + str(self.country) + "'"
        if self.domain != 0:
            result += " domain='" + str(self.domain) + "'"
        if self.extra != 0:
            result += " extra='" + str(self.extra) + "'"
        if self.kind != 0:
            result += " kind='" + str(self.kind) + "'"
        if self.specific != 0:
            result += " specific='" + str(self.specific) + "'"
        if self.subcategory != 0:
            result += " subcategory='" + str(self.subcategory) + "'"
        if self.url != list():
            result += " url='" + self.url + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</DISEntityTypeMapping>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Disk2D(_X3DGeometryNode):
    """
    Disk2D is a geometry node that defines a filled (or partially filled) planar circle with center (0,0).
    """
    NAME = 'Disk2D'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Disk2D'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('innerRadius', 0, FieldType.SFFloat, AccessType.initializeOnly, 'Disk2D'),
        ('outerRadius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Disk2D'),
        ('solid', False, FieldType.SFBool, AccessType.initializeOnly, 'Disk2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 innerRadius=0,
                 outerRadius=1,
                 solid=False,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Disk2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.innerRadius = innerRadius
        self.outerRadius = outerRadius
        self.solid = solid
    @property # getter - - - - - - - - - -
    def innerRadius(self):
        """[0,+infinity) Inner circle radius, greater than or equal to 0."""
        return self.__innerRadius
    @innerRadius.setter
    def innerRadius(self, innerRadius=None):
        if  innerRadius is None:
            innerRadius = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(innerRadius)
        assertNonNegative('innerRadius', innerRadius)
        self.__innerRadius = innerRadius
    @property # getter - - - - - - - - - -
    def outerRadius(self):
        """(0,+infinity) Outer radius of circle, greater than or equal to inner radius."""
        return self.__outerRadius
    @outerRadius.setter
    def outerRadius(self, outerRadius=None):
        if  outerRadius is None:
            outerRadius = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(outerRadius)
        assertPositive('outerRadius', outerRadius)
        self.__outerRadius = outerRadius
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Disk2D.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Disk2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.innerRadius != 0:
            result += " innerRadius='" + str(self.innerRadius) + "'"
        if self.outerRadius != 1:
            result += " outerRadius='" + str(self.outerRadius) + "'"
        if self.solid != False:
            result += " solid='" + str(self.solid) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Disk2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class DoubleAxisHingeJoint(_X3DRigidJointNode):
    """
    DoubleAxisHingeJoint has two independent axes located around a common anchor point.
    """
    NAME = 'DoubleAxisHingeJoint'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DoubleAxisHingeJoint'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('anchorPoint', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('axis1', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('axis2', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('desiredAngularVelocity1', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('desiredAngularVelocity2', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('forceOutput', ["NONE"], FieldType.MFString, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('maxAngle1', 3.141592653, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('maxTorque1', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('maxTorque2', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('minAngle1', -3.141592653, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('stop1Bounce', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('stop1ConstantForceMix', 0.001, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('stop1ErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('suspensionErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('suspensionForce', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('body1', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body2', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 anchorPoint=(0, 0, 0),
                 axis1=(0, 0, 0),
                 axis2=(0, 0, 0),
                 desiredAngularVelocity1=0,
                 desiredAngularVelocity2=0,
                 forceOutput=["NONE"],
                 maxAngle1=3.141592653,
                 maxTorque1=0,
                 maxTorque2=0,
                 minAngle1=-3.141592653,
                 stop1Bounce=0,
                 stop1ConstantForceMix=0.001,
                 stop1ErrorCorrection=0.8,
                 suspensionErrorCorrection=0.8,
                 suspensionForce=0,
                 body1=None,
                 body2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DoubleAxisHingeJoint __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.anchorPoint = anchorPoint
        self.axis1 = axis1
        self.axis2 = axis2
        self.desiredAngularVelocity1 = desiredAngularVelocity1
        self.desiredAngularVelocity2 = desiredAngularVelocity2
        self.forceOutput = forceOutput
        self.maxAngle1 = maxAngle1
        self.maxTorque1 = maxTorque1
        self.maxTorque2 = maxTorque2
        self.minAngle1 = minAngle1
        self.stop1Bounce = stop1Bounce
        self.stop1ConstantForceMix = stop1ConstantForceMix
        self.stop1ErrorCorrection = stop1ErrorCorrection
        self.suspensionErrorCorrection = suspensionErrorCorrection
        self.suspensionForce = suspensionForce
        self.body1 = body1
        self.body2 = body2
    @property # getter - - - - - - - - - -
    def anchorPoint(self):
        """anchorPoint is joint center, specified in world coordinates."""
        return self.__anchorPoint
    @anchorPoint.setter
    def anchorPoint(self, anchorPoint=None):
        if  anchorPoint is None:
            anchorPoint = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(anchorPoint)
        self.__anchorPoint = anchorPoint
    @property # getter - - - - - - - - - -
    def axis1(self):
        """axis1 defines axis vector of joint connection to body1."""
        return self.__axis1
    @axis1.setter
    def axis1(self, axis1=None):
        if  axis1 is None:
            axis1 = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(axis1)
        self.__axis1 = axis1
    @property # getter - - - - - - - - - -
    def axis2(self):
        """axis2 defines axis vector of joint connection to body2."""
        return self.__axis2
    @axis2.setter
    def axis2(self, axis2=None):
        if  axis2 is None:
            axis2 = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(axis2)
        self.__axis2 = axis2
    @property # getter - - - - - - - - - -
    def desiredAngularVelocity1(self):
        """desiredAngularVelocity1 is goal rotation rate for hinge connection to body1."""
        return self.__desiredAngularVelocity1
    @desiredAngularVelocity1.setter
    def desiredAngularVelocity1(self, desiredAngularVelocity1=None):
        if  desiredAngularVelocity1 is None:
            desiredAngularVelocity1 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(desiredAngularVelocity1)
        self.__desiredAngularVelocity1 = desiredAngularVelocity1
    @property # getter - - - - - - - - - -
    def desiredAngularVelocity2(self):
        """desiredAngularVelocity2 is goal rotation rate for hinge connection to body2."""
        return self.__desiredAngularVelocity2
    @desiredAngularVelocity2.setter
    def desiredAngularVelocity2(self, desiredAngularVelocity2=None):
        if  desiredAngularVelocity2 is None:
            desiredAngularVelocity2 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(desiredAngularVelocity2)
        self.__desiredAngularVelocity2 = desiredAngularVelocity2
    @property # getter - - - - - - - - - -
    def forceOutput(self):
        """forceOutput controls which output fields are generated for the next frame."""
        return self.__forceOutput
    @forceOutput.setter
    def forceOutput(self, forceOutput=None):
        if  forceOutput is None:
            forceOutput = MFString.DEFAULT_VALUE
        assertValidMFString(forceOutput)
        self.__forceOutput = forceOutput
    @property # getter - - - - - - - - - -
    def maxAngle1(self):
        """[-pi,pi] maxAngle1 is maximum rotation angle for hinge."""
        return self.__maxAngle1
    @maxAngle1.setter
    def maxAngle1(self, maxAngle1=None):
        if  maxAngle1 is None:
            maxAngle1 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(maxAngle1)
        self.__maxAngle1 = maxAngle1
    @property # getter - - - - - - - - - -
    def maxTorque1(self):
        """maxTorque1 is maximum rotational torque applied by corresponding motor axis to achieve desiredAngularVelocity1."""
        return self.__maxTorque1
    @maxTorque1.setter
    def maxTorque1(self, maxTorque1=None):
        if  maxTorque1 is None:
            maxTorque1 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(maxTorque1)
        self.__maxTorque1 = maxTorque1
    @property # getter - - - - - - - - - -
    def maxTorque2(self):
        """maxTorque2 is maximum rotational torque applied by corresponding motor axis to achieve desiredAngularVelocity2."""
        return self.__maxTorque2
    @maxTorque2.setter
    def maxTorque2(self, maxTorque2=None):
        if  maxTorque2 is None:
            maxTorque2 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(maxTorque2)
        self.__maxTorque2 = maxTorque2
    @property # getter - - - - - - - - - -
    def minAngle1(self):
        """[-pi,pi] minAngle1 is minimum rotation angle for hinge."""
        return self.__minAngle1
    @minAngle1.setter
    def minAngle1(self, minAngle1=None):
        if  minAngle1 is None:
            minAngle1 = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(minAngle1)
        self.__minAngle1 = minAngle1
    @property # getter - - - - - - - - - -
    def stop1Bounce(self):
        """[0,1] stop1Bounce is velocity factor for bounce back once stop point is reached."""
        return self.__stop1Bounce
    @stop1Bounce.setter
    def stop1Bounce(self, stop1Bounce=None):
        if  stop1Bounce is None:
            stop1Bounce = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(stop1Bounce)
        self.__stop1Bounce = stop1Bounce
    @property # getter - - - - - - - - - -
    def stop1ConstantForceMix(self):
        """[0,1] stop1ConstantForceMix value applies a constant force value to make colliding surfaces appear to be somewhat soft."""
        return self.__stop1ConstantForceMix
    @stop1ConstantForceMix.setter
    def stop1ConstantForceMix(self, stop1ConstantForceMix=None):
        if  stop1ConstantForceMix is None:
            stop1ConstantForceMix = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(stop1ConstantForceMix)
        self.__stop1ConstantForceMix = stop1ConstantForceMix
    @property # getter - - - - - - - - - -
    def stop1ErrorCorrection(self):
        """[0,1] stop1ErrorCorrection is fraction of error correction performed during time step once stop point is reached."""
        return self.__stop1ErrorCorrection
    @stop1ErrorCorrection.setter
    def stop1ErrorCorrection(self, stop1ErrorCorrection=None):
        if  stop1ErrorCorrection is None:
            stop1ErrorCorrection = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(stop1ErrorCorrection)
        self.__stop1ErrorCorrection = stop1ErrorCorrection
    @property # getter - - - - - - - - - -
    def suspensionErrorCorrection(self):
        """[0,1] suspensionErrorCorrection describes how quickly the system resolves intersection errors due to floating-point inaccuracies."""
        return self.__suspensionErrorCorrection
    @suspensionErrorCorrection.setter
    def suspensionErrorCorrection(self, suspensionErrorCorrection=None):
        if  suspensionErrorCorrection is None:
            suspensionErrorCorrection = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(suspensionErrorCorrection)
        self.__suspensionErrorCorrection = suspensionErrorCorrection
    @property # getter - - - - - - - - - -
    def suspensionForce(self):
        """[0,1] suspensionForce describes how quickly the system resolves intersection errors due to floating-point inaccuracies."""
        return self.__suspensionForce
    @suspensionForce.setter
    def suspensionForce(self, suspensionForce=None):
        if  suspensionForce is None:
            suspensionForce = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(suspensionForce)
        self.__suspensionForce = suspensionForce
    @property # getter - - - - - - - - - -
    def body1(self):
        return self.__body1
    @body1.setter
    def body1(self, body1=None):
        if  body1 is None:
            body1 = SFNode.DEFAULT_VALUE
        assertValidSFNode(body1)
        if not isinstance(body1, object):
            # print(flush=True)
            raise X3DTypeError(str(body1) + ' does not have a valid node type object')
        self.__body1 = body1
    @property # getter - - - - - - - - - -
    def body2(self):
        return self.__body2
    @body2.setter
    def body2(self, body2=None):
        if  body2 is None:
            body2 = SFNode.DEFAULT_VALUE
        assertValidSFNode(body2)
        if not isinstance(body2, object):
            # print(flush=True)
            raise X3DTypeError(str(body2) + ' does not have a valid node type object')
        self.__body2 = body2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.body1) or bool(self.body2) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DoubleAxisHingeJoint.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DoubleAxisHingeJoint'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.anchorPoint != (0, 0, 0):
            result += " anchorPoint='" + str(self.anchorPoint) + "'"
        if self.axis1 != (0, 0, 0):
            result += " axis1='" + str(self.axis1) + "'"
        if self.axis2 != (0, 0, 0):
            result += " axis2='" + str(self.axis2) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.desiredAngularVelocity1 != 0:
            result += " desiredAngularVelocity1='" + str(self.desiredAngularVelocity1) + "'"
        if self.desiredAngularVelocity2 != 0:
            result += " desiredAngularVelocity2='" + str(self.desiredAngularVelocity2) + "'"
        if self.forceOutput != ["NONE"]:
            result += " forceOutput='" + self.forceOutput + "'"
        if self.maxAngle1 != 3.141592653:
            result += " maxAngle1='" + str(self.maxAngle1) + "'"
        if self.maxTorque1 != 0:
            result += " maxTorque1='" + str(self.maxTorque1) + "'"
        if self.maxTorque2 != 0:
            result += " maxTorque2='" + str(self.maxTorque2) + "'"
        if self.minAngle1 != -3.141592653:
            result += " minAngle1='" + str(self.minAngle1) + "'"
        if self.stop1Bounce != 0:
            result += " stop1Bounce='" + str(self.stop1Bounce) + "'"
        if self.stop1ConstantForceMix != 0.001:
            result += " stop1ConstantForceMix='" + str(self.stop1ConstantForceMix) + "'"
        if self.stop1ErrorCorrection != 0.8:
            result += " stop1ErrorCorrection='" + str(self.stop1ErrorCorrection) + "'"
        if self.suspensionErrorCorrection != 0.8:
            result += " suspensionErrorCorrection='" + str(self.suspensionErrorCorrection) + "'"
        if self.suspensionForce != 0:
            result += " suspensionForce='" + str(self.suspensionForce) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.body1: # walk each child node, if any
                for each in self.body1:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.body2: # walk each child node, if any
                for each in self.body2:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</DoubleAxisHingeJoint>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class EaseInEaseOut(_X3DChildNode):
    """
    EaseInEaseOut enables gradual animation transitions by modifying TimeSensor fraction outputs.
    """
    NAME = 'EaseInEaseOut'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EaseInEaseOut'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('easeInEaseOut', list(), FieldType.MFVec2f, AccessType.inputOutput, 'EaseInEaseOut'),
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'EaseInEaseOut'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 easeInEaseOut=list(),
                 key=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode EaseInEaseOut __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.easeInEaseOut = easeInEaseOut
        self.key = key
    @property # getter - - - - - - - - - -
    def easeInEaseOut(self):
        """Array of paired values for easeOut fraction and easeIn fraction within each key interval."""
        return self.__easeInEaseOut
    @easeInEaseOut.setter
    def easeInEaseOut(self, easeInEaseOut=None):
        if  easeInEaseOut is None:
            easeInEaseOut = MFVec2f.DEFAULT_VALUE
        assertValidMFVec2f(easeInEaseOut)
        self.__easeInEaseOut = easeInEaseOut
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to easeInEaseOut array."""
        return self.__key
    @key.setter
    def key(self, key=None):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(key)
        self.__key = key
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EaseInEaseOut.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EaseInEaseOut'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.easeInEaseOut != list():
            result += " easeInEaseOut='" + str(self.easeInEaseOut) + "'"
        if self.key != list():
            result += " key='" + str(self.key) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</EaseInEaseOut>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class EdgeEnhancementVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    EdgeEnhancementVolumeStyle specifies edge enhancement for the volume rendering style.
    """
    NAME = 'EdgeEnhancementVolumeStyle'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EdgeEnhancementVolumeStyle'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('edgeColor', (0, 0, 0, 1), FieldType.SFColorRGBA, AccessType.inputOutput, 'EdgeEnhancementVolumeStyle'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('gradientThreshold', 0.4, FieldType.SFFloat, AccessType.inputOutput, 'EdgeEnhancementVolumeStyle'),
        ('surfaceNormals', None, FieldType.SFNode, AccessType.inputOutput, 'EdgeEnhancementVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 edgeColor=(0, 0, 0, 1),
                 enabled=True,
                 gradientThreshold=0.4,
                 surfaceNormals=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode EdgeEnhancementVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.edgeColor = edgeColor
        self.enabled = enabled
        self.gradientThreshold = gradientThreshold
        self.surfaceNormals = surfaceNormals
    @property # getter - - - - - - - - - -
    def edgeColor(self):
        """[0,1] color used to highlight edges."""
        return self.__edgeColor
    @edgeColor.setter
    def edgeColor(self, edgeColor=None):
        if  edgeColor is None:
            edgeColor = SFColorRGBA.DEFAULT_VALUE
        assertValidSFColorRGBA(edgeColor)
        assertZeroToOne('edgeColor', edgeColor)
        self.__edgeColor = edgeColor
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def gradientThreshold(self):
        """[0,1] minimum angle (in radians) away from view-direction vector for surface normal before applying enhancement."""
        return self.__gradientThreshold
    @gradientThreshold.setter
    def gradientThreshold(self, gradientThreshold=None):
        if  gradientThreshold is None:
            gradientThreshold = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(gradientThreshold)
        assertGreaterThanEquals('gradientThreshold', gradientThreshold, 0)
        assertLessThanEquals('gradientThreshold', gradientThreshold, 3.1416)
        self.__gradientThreshold = gradientThreshold
    @property # getter - - - - - - - - - -
    def surfaceNormals(self):
        return self.__surfaceNormals
    @surfaceNormals.setter
    def surfaceNormals(self, surfaceNormals=None):
        if  surfaceNormals is None:
            surfaceNormals = SFNode.DEFAULT_VALUE
        assertValidSFNode(surfaceNormals)
        if not isinstance(surfaceNormals, object):
            # print(flush=True)
            raise X3DTypeError(str(surfaceNormals) + ' does not have a valid node type object')
        self.__surfaceNormals = surfaceNormals
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata) or bool(self.surfaceNormals)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EdgeEnhancementVolumeStyle.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EdgeEnhancementVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.edgeColor != (0, 0, 0, 1):
            result += " edgeColor='" + str(self.edgeColor) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.gradientThreshold != 0.4:
            result += " gradientThreshold='" + str(self.gradientThreshold) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.surfaceNormals: # walk each child node, if any
                for each in self.surfaceNormals:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</EdgeEnhancementVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ElevationGrid(_X3DGeometryNode):
    """
    ElevationGrid is a geometry node defining a rectangular height field, with default values for a 1m by 1m square at height 0.
    """
    NAME = 'ElevationGrid'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ElevationGrid'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('ccw', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('colorPerVertex', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('creaseAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('height', [0, 0, 0, 0], FieldType.MFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('normalPerVertex', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('xDimension', 2, FieldType.SFInt32, AccessType.initializeOnly, 'ElevationGrid'),
        ('xSpacing', 1.0, FieldType.SFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('zDimension', 2, FieldType.SFInt32, AccessType.initializeOnly, 'ElevationGrid'),
        ('zSpacing', 1.0, FieldType.SFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('color', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('fogCoord', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('normal', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('texCoord', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('attrib', list(), FieldType.MFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 ccw=True,
                 colorPerVertex=True,
                 creaseAngle=0,
                 height=[0, 0, 0, 0],
                 normalPerVertex=True,
                 solid=True,
                 xDimension=2,
                 xSpacing=1.0,
                 zDimension=2,
                 zSpacing=1.0,
                 color=None,
                 fogCoord=None,
                 normal=None,
                 texCoord=None,
                 attrib=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ElevationGrid __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.ccw = ccw
        self.colorPerVertex = colorPerVertex
        self.creaseAngle = creaseAngle
        self.height = height
        self.normalPerVertex = normalPerVertex
        self.solid = solid
        self.xDimension = xDimension
        self.xSpacing = xSpacing
        self.zDimension = zDimension
        self.zSpacing = zSpacing
        self.color = color
        self.fogCoord = fogCoord
        self.normal = normal
        self.texCoord = texCoord
        self.attrib = attrib
    @property # getter - - - - - - - - - -
    def ccw(self):
        """ccw defines clockwise/counterclockwise ordering of vertex coordinates, which in turn defines front/back orientation of polygon normals according to Right-Hand Rule (RHR)."""
        return self.__ccw
    @ccw.setter
    def ccw(self, ccw=None):
        if  ccw is None:
            ccw = SFBool.DEFAULT_VALUE
        assertValidSFBool(ccw)
        self.__ccw = ccw
    @property # getter - - - - - - - - - -
    def colorPerVertex(self):
        """Whether Color node color values are applied to each point vertex (true) or per quadrilateral (false)."""
        return self.__colorPerVertex
    @colorPerVertex.setter
    def colorPerVertex(self, colorPerVertex=None):
        if  colorPerVertex is None:
            colorPerVertex = SFBool.DEFAULT_VALUE
        assertValidSFBool(colorPerVertex)
        self.__colorPerVertex = colorPerVertex
    @property # getter - - - - - - - - - -
    def creaseAngle(self):
        """[0,+infinity) creaseAngle defines angle (in radians) for determining whether adjacent polygons are drawn with sharp edges or smooth shading."""
        return self.__creaseAngle
    @creaseAngle.setter
    def creaseAngle(self, creaseAngle=None):
        if  creaseAngle is None:
            creaseAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(creaseAngle)
        assertNonNegative('creaseAngle', creaseAngle)
        self.__creaseAngle = creaseAngle
    @property # getter - - - - - - - - - -
    def height(self):
        """Grid array of height vertices with upward direction along +Y axis, with xDimension rows and zDimension columns."""
        return self.__height
    @height.setter
    def height(self, height=None):
        if  height is None:
            height = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(height)
        self.__height = height
    @property # getter - - - - - - - - - -
    def normalPerVertex(self):
        """Whether Normal node vector values are applied to each point vertex (true) or per quadrilateral (false)."""
        return self.__normalPerVertex
    @normalPerVertex.setter
    def normalPerVertex(self, normalPerVertex=None):
        if  normalPerVertex is None:
            normalPerVertex = SFBool.DEFAULT_VALUE
        assertValidSFBool(normalPerVertex)
        self.__normalPerVertex = normalPerVertex
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def xDimension(self):
        """(0,+infinity) Number of elements in the height array along X direction."""
        return self.__xDimension
    @xDimension.setter
    def xDimension(self, xDimension=None):
        if  xDimension is None:
            xDimension = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(xDimension)
        assertNonNegative('xDimension', xDimension)
        self.__xDimension = xDimension
    @property # getter - - - - - - - - - -
    def xSpacing(self):
        """(0,+infinity) Meters distance between grid-array vertices along X direction."""
        return self.__xSpacing
    @xSpacing.setter
    def xSpacing(self, xSpacing=None):
        if  xSpacing is None:
            xSpacing = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(xSpacing)
        assertPositive('xSpacing', xSpacing)
        self.__xSpacing = xSpacing
    @property # getter - - - - - - - - - -
    def zDimension(self):
        """(0,+infinity) Number of elements in the height array along Z direction."""
        return self.__zDimension
    @zDimension.setter
    def zDimension(self, zDimension=None):
        if  zDimension is None:
            zDimension = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(zDimension)
        assertNonNegative('zDimension', zDimension)
        self.__zDimension = zDimension
    @property # getter - - - - - - - - - -
    def zSpacing(self):
        """(0,+infinity) Meters distance between grid-array vertices along Z direction."""
        return self.__zSpacing
    @zSpacing.setter
    def zSpacing(self, zSpacing=None):
        if  zSpacing is None:
            zSpacing = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(zSpacing)
        assertPositive('zSpacing', zSpacing)
        self.__zSpacing = zSpacing
    @property # getter - - - - - - - - - -
    def color(self):
        """[X3DColorNode] Single contained Color or ColorRGBA node that specifies color values applied to corresponding vertices according to colorPerVertex field."""
        return self.__color
    @color.setter
    def color(self, color=None):
        if  color is None:
            color = SFNode.DEFAULT_VALUE
        assertValidSFNode(color)
        if not isinstance(color, object):
            # print(flush=True)
            raise X3DTypeError(str(color) + ' does not have a valid node type object')
        self.__color = color
    @property # getter - - - - - - - - - -
    def fogCoord(self):
        """[FogCoordinate] Single contained FogCoordinate node that specifies depth parameters for fog in corresponding geometry."""
        return self.__fogCoord
    @fogCoord.setter
    def fogCoord(self, fogCoord=None):
        if  fogCoord is None:
            fogCoord = SFNode.DEFAULT_VALUE
        assertValidSFNode(fogCoord)
        if not isinstance(fogCoord, object):
            # print(flush=True)
            raise X3DTypeError(str(fogCoord) + ' does not have a valid node type object')
        self.__fogCoord = fogCoord
    @property # getter - - - - - - - - - -
    def normal(self):
        """[X3DNormalNode] Single contained Normal node that specifies perpendicular vectors for corresponding vertices to support rendering computations, applied according to the normalPerVertex field."""
        return self.__normal
    @normal.setter
    def normal(self, normal=None):
        if  normal is None:
            normal = SFNode.DEFAULT_VALUE
        assertValidSFNode(normal)
        if not isinstance(normal, object):
            # print(flush=True)
            raise X3DTypeError(str(normal) + ' does not have a valid node type object')
        self.__normal = normal
    @property # getter - - - - - - - - - -
    def texCoord(self):
        """[X3DTextureCoordinateNode] Single contained TextureCoordinate, TextureCoordinateGenerator or MultiTextureCoordinate node that specifies coordinates for texture mapping onto corresponding geometry."""
        return self.__texCoord
    @texCoord.setter
    def texCoord(self, texCoord=None):
        if  texCoord is None:
            texCoord = SFNode.DEFAULT_VALUE
        assertValidSFNode(texCoord)
        if not isinstance(texCoord, object):
            # print(flush=True)
            raise X3DTypeError(str(texCoord) + ' does not have a valid node type object')
        self.__texCoord = texCoord
    @property # getter - - - - - - - - - -
    def attrib(self):
        """[X3DVertexAttributeNode] Single contained FloatVertexAttribute node that specifies list of per-vertex attribute information for programmable shaders."""
        return self.__attrib
    @attrib.setter
    def attrib(self, attrib=None):
        if  attrib is None:
            attrib = MFNode.DEFAULT_VALUE
        assertValidMFNode(attrib)
        self.__attrib = attrib
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.attrib) or bool(self.color) or bool(self.fogCoord) or bool(self.IS) or bool(self.metadata) or bool(self.normal) or bool(self.texCoord)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ElevationGrid.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ElevationGrid'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.ccw != True:
            result += " ccw='" + str(self.ccw) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.colorPerVertex != True:
            result += " colorPerVertex='" + str(self.colorPerVertex) + "'"
        if self.creaseAngle != 0:
            result += " creaseAngle='" + str(self.creaseAngle) + "'"
        if self.height != [0, 0, 0, 0]:
            result += " height='" + str(self.height) + "'"
        if self.normalPerVertex != True:
            result += " normalPerVertex='" + str(self.normalPerVertex) + "'"
        if self.solid != True:
            result += " solid='" + str(self.solid) + "'"
        if self.xDimension != 2:
            result += " xDimension='" + str(self.xDimension) + "'"
        if self.xSpacing != 1.0:
            result += " xSpacing='" + str(self.xSpacing) + "'"
        if self.zDimension != 2:
            result += " zDimension='" + str(self.zDimension) + "'"
        if self.zSpacing != 1.0:
            result += " zSpacing='" + str(self.zSpacing) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.color: # walk each child node, if any
                for each in self.color:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.fogCoord: # walk each child node, if any
                for each in self.fogCoord:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.normal: # walk each child node, if any
                for each in self.normal:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.texCoord: # walk each child node, if any
                for each in self.texCoord:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.attrib: # walk each child node, if any
                for each in self.attrib:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ElevationGrid>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class EspduTransform(_X3DGroupingNode, _X3DNetworkSensorNode):
    """
    EspduTransform is a networked Transform node that can contain most nodes.
    """
    NAME = 'EspduTransform'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EspduTransform'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('address', 'localhost', FieldType.SFString, AccessType.inputOutput, 'EspduTransform'),
        ('applicationID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('articulationParameterArray', list(), FieldType.MFFloat, AccessType.inputOutput, 'EspduTransform'),
        ('articulationParameterChangeIndicatorArray', list(), FieldType.MFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('articulationParameterCount', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('articulationParameterDesignatorArray', list(), FieldType.MFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('articulationParameterIdPartAttachedToArray', list(), FieldType.MFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('articulationParameterTypeArray', list(), FieldType.MFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('center', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('collisionType', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('deadReckoning', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('detonationLocation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('detonationRelativeLocation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('detonationResult', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'EspduTransform'),
        ('entityCategory', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('entityCountry', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('entityDomain', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('entityExtra', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('entityID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('entityKind', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('entitySpecific', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('entitySubcategory', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('eventApplicationID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('eventEntityID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('eventNumber', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('eventSiteID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('fired1', False, FieldType.SFBool, AccessType.inputOutput, 'EspduTransform'),
        ('fired2', False, FieldType.SFBool, AccessType.inputOutput, 'EspduTransform'),
        ('fireMissionIndex', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('firingRange', 0, FieldType.SFFloat, AccessType.inputOutput, 'EspduTransform'),
        ('firingRate', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('forceID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('fuse', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('geoCoords', (0, 0, 0), FieldType.SFVec3d, AccessType.inputOutput, 'EspduTransform'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'EspduTransform'),
        ('linearAcceleration', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('linearVelocity', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('marking', '', FieldType.SFString, AccessType.inputOutput, 'EspduTransform'),
        ('multicastRelayHost', '', FieldType.SFString, AccessType.inputOutput, 'EspduTransform'),
        ('multicastRelayPort', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('munitionApplicationID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('munitionEndPoint', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('munitionEntityID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('munitionQuantity', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('munitionSiteID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('munitionStartPoint', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('networkMode', 'standAlone', FieldType.SFString, AccessType.inputOutput, 'EspduTransform'),
        ('port', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('readInterval', 0.1, FieldType.SFTime, AccessType.inputOutput, 'EspduTransform'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'EspduTransform'),
        ('rtpHeaderExpected', False, FieldType.SFBool, AccessType.initializeOnly, 'EspduTransform'),
        ('scale', (1, 1, 1), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('scaleOrientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'EspduTransform'),
        ('siteID', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'EspduTransform'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('warhead', 0, FieldType.SFInt32, AccessType.inputOutput, 'EspduTransform'),
        ('writeInterval', 1.0, FieldType.SFTime, AccessType.inputOutput, 'EspduTransform'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 address='localhost',
                 applicationID=0,
                 articulationParameterArray=list(),
                 articulationParameterChangeIndicatorArray=list(),
                 articulationParameterCount=0,
                 articulationParameterDesignatorArray=list(),
                 articulationParameterIdPartAttachedToArray=list(),
                 articulationParameterTypeArray=list(),
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 center=(0, 0, 0),
                 collisionType=0,
                 deadReckoning=0,
                 detonationLocation=(0, 0, 0),
                 detonationRelativeLocation=(0, 0, 0),
                 detonationResult=0,
                 displayBBox=False,
                 enabled=True,
                 entityCategory=0,
                 entityCountry=0,
                 entityDomain=0,
                 entityExtra=0,
                 entityID=0,
                 entityKind=0,
                 entitySpecific=0,
                 entitySubcategory=0,
                 eventApplicationID=0,
                 eventEntityID=0,
                 eventNumber=0,
                 eventSiteID=0,
                 fired1=False,
                 fired2=False,
                 fireMissionIndex=0,
                 firingRange=0,
                 firingRate=0,
                 forceID=0,
                 fuse=0,
                 geoCoords=(0, 0, 0),
                 geoSystem=["GD", "WE"],
                 linearAcceleration=(0, 0, 0),
                 linearVelocity=(0, 0, 0),
                 marking='',
                 multicastRelayHost='',
                 multicastRelayPort=0,
                 munitionApplicationID=0,
                 munitionEndPoint=(0, 0, 0),
                 munitionEntityID=0,
                 munitionQuantity=0,
                 munitionSiteID=0,
                 munitionStartPoint=(0, 0, 0),
                 networkMode='standAlone',
                 port=0,
                 readInterval=0.1,
                 rotation=(0, 0, 1, 0),
                 rtpHeaderExpected=False,
                 scale=(1, 1, 1),
                 scaleOrientation=(0, 0, 1, 0),
                 siteID=0,
                 translation=(0, 0, 0),
                 visible=True,
                 warhead=0,
                 writeInterval=1.0,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode EspduTransform __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.address = address
        self.applicationID = applicationID
        self.articulationParameterArray = articulationParameterArray
        self.articulationParameterChangeIndicatorArray = articulationParameterChangeIndicatorArray
        self.articulationParameterCount = articulationParameterCount
        self.articulationParameterDesignatorArray = articulationParameterDesignatorArray
        self.articulationParameterIdPartAttachedToArray = articulationParameterIdPartAttachedToArray
        self.articulationParameterTypeArray = articulationParameterTypeArray
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.center = center
        self.collisionType = collisionType
        self.deadReckoning = deadReckoning
        self.detonationLocation = detonationLocation
        self.detonationRelativeLocation = detonationRelativeLocation
        self.detonationResult = detonationResult
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.entityCategory = entityCategory
        self.entityCountry = entityCountry
        self.entityDomain = entityDomain
        self.entityExtra = entityExtra
        self.entityID = entityID
        self.entityKind = entityKind
        self.entitySpecific = entitySpecific
        self.entitySubcategory = entitySubcategory
        self.eventApplicationID = eventApplicationID
        self.eventEntityID = eventEntityID
        self.eventNumber = eventNumber
        self.eventSiteID = eventSiteID
        self.fired1 = fired1
        self.fired2 = fired2
        self.fireMissionIndex = fireMissionIndex
        self.firingRange = firingRange
        self.firingRate = firingRate
        self.forceID = forceID
        self.fuse = fuse
        self.geoCoords = geoCoords
        self.geoSystem = geoSystem
        self.linearAcceleration = linearAcceleration
        self.linearVelocity = linearVelocity
        self.marking = marking
        self.multicastRelayHost = multicastRelayHost
        self.multicastRelayPort = multicastRelayPort
        self.munitionApplicationID = munitionApplicationID
        self.munitionEndPoint = munitionEndPoint
        self.munitionEntityID = munitionEntityID
        self.munitionQuantity = munitionQuantity
        self.munitionSiteID = munitionSiteID
        self.munitionStartPoint = munitionStartPoint
        self.networkMode = networkMode
        self.port = port
        self.readInterval = readInterval
        self.rotation = rotation
        self.rtpHeaderExpected = rtpHeaderExpected
        self.scale = scale
        self.scaleOrientation = scaleOrientation
        self.siteID = siteID
        self.translation = translation
        self.visible = visible
        self.warhead = warhead
        self.writeInterval = writeInterval
        self.children = children
    @property # getter - - - - - - - - - -
    def address(self):
        """Multicast network address, or else "localhost" Example: 224."""
        return self.__address
    @address.setter
    def address(self, address=None):
        if  address is None:
            address = SFString.DEFAULT_VALUE
        assertValidSFString(address)
        self.__address = address
    @property # getter - - - - - - - - - -
    def applicationID(self):
        """Simulation/exercise applicationID is unique for application at that site."""
        return self.__applicationID
    @applicationID.setter
    def applicationID(self, applicationID=None):
        if  applicationID is None:
            applicationID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(applicationID)
        self.__applicationID = applicationID
    @property # getter - - - - - - - - - -
    def articulationParameterArray(self):
        """Information required for representation of the entity's visual appearance and position of its articulated parts."""
        return self.__articulationParameterArray
    @articulationParameterArray.setter
    def articulationParameterArray(self, articulationParameterArray=None):
        if  articulationParameterArray is None:
            articulationParameterArray = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(articulationParameterArray)
        self.__articulationParameterArray = articulationParameterArray
    @property # getter - - - - - - - - - -
    def articulationParameterChangeIndicatorArray(self):
        """Array of change counters, each incremented when an articulated parameter is updated."""
        return self.__articulationParameterChangeIndicatorArray
    @articulationParameterChangeIndicatorArray.setter
    def articulationParameterChangeIndicatorArray(self, articulationParameterChangeIndicatorArray=None):
        if  articulationParameterChangeIndicatorArray is None:
            articulationParameterChangeIndicatorArray = MFInt32.DEFAULT_VALUE
        assertValidMFInt32(articulationParameterChangeIndicatorArray)
        self.__articulationParameterChangeIndicatorArray = articulationParameterChangeIndicatorArray
    @property # getter - - - - - - - - - -
    def articulationParameterCount(self):
        """Number of articulated parameters attached to this entity state PDU."""
        return self.__articulationParameterCount
    @articulationParameterCount.setter
    def articulationParameterCount(self, articulationParameterCount=None):
        if  articulationParameterCount is None:
            articulationParameterCount = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(articulationParameterCount)
        self.__articulationParameterCount = articulationParameterCount
    @property # getter - - - - - - - - - -
    def articulationParameterDesignatorArray(self):
        """Array of designators for each articulated parameter."""
        return self.__articulationParameterDesignatorArray
    @articulationParameterDesignatorArray.setter
    def articulationParameterDesignatorArray(self, articulationParameterDesignatorArray=None):
        if  articulationParameterDesignatorArray is None:
            articulationParameterDesignatorArray = MFInt32.DEFAULT_VALUE
        assertValidMFInt32(articulationParameterDesignatorArray)
        self.__articulationParameterDesignatorArray = articulationParameterDesignatorArray
    @property # getter - - - - - - - - - -
    def articulationParameterIdPartAttachedToArray(self):
        """Array of ID parts that each articulated parameter is attached to."""
        return self.__articulationParameterIdPartAttachedToArray
    @articulationParameterIdPartAttachedToArray.setter
    def articulationParameterIdPartAttachedToArray(self, articulationParameterIdPartAttachedToArray=None):
        if  articulationParameterIdPartAttachedToArray is None:
            articulationParameterIdPartAttachedToArray = MFInt32.DEFAULT_VALUE
        assertValidMFInt32(articulationParameterIdPartAttachedToArray)
        self.__articulationParameterIdPartAttachedToArray = articulationParameterIdPartAttachedToArray
    @property # getter - - - - - - - - - -
    def articulationParameterTypeArray(self):
        """Array of type enumerations for each articulated parameter element."""
        return self.__articulationParameterTypeArray
    @articulationParameterTypeArray.setter
    def articulationParameterTypeArray(self, articulationParameterTypeArray=None):
        if  articulationParameterTypeArray is None:
            articulationParameterTypeArray = MFInt32.DEFAULT_VALUE
        assertValidMFInt32(articulationParameterTypeArray)
        self.__articulationParameterTypeArray = articulationParameterTypeArray
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def center(self):
        """Translation offset from origin of local coordinate system."""
        return self.__center
    @center.setter
    def center(self, center=None):
        if  center is None:
            center = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(center)
        self.__center = center
    @property # getter - - - - - - - - - -
    def collisionType(self):
        """Integer enumeration for type of collision: ELASTIC or INELASTIC."""
        return self.__collisionType
    @collisionType.setter
    def collisionType(self, collisionType=None):
        if  collisionType is None:
            collisionType = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(collisionType)
        self.__collisionType = collisionType
    @property # getter - - - - - - - - - -
    def deadReckoning(self):
        """Dead reckoning algorithm being used to project position/orientation with velocities/accelerations."""
        return self.__deadReckoning
    @deadReckoning.setter
    def deadReckoning(self, deadReckoning=None):
        if  deadReckoning is None:
            deadReckoning = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(deadReckoning)
        self.__deadReckoning = deadReckoning
    @property # getter - - - - - - - - - -
    def detonationLocation(self):
        """World coordinates for detonationLocation."""
        return self.__detonationLocation
    @detonationLocation.setter
    def detonationLocation(self, detonationLocation=None):
        if  detonationLocation is None:
            detonationLocation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(detonationLocation)
        self.__detonationLocation = detonationLocation
    @property # getter - - - - - - - - - -
    def detonationRelativeLocation(self):
        """Relative coordinates for detonation location."""
        return self.__detonationRelativeLocation
    @detonationRelativeLocation.setter
    def detonationRelativeLocation(self, detonationRelativeLocation=None):
        if  detonationRelativeLocation is None:
            detonationRelativeLocation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(detonationRelativeLocation)
        self.__detonationRelativeLocation = detonationRelativeLocation
    @property # getter - - - - - - - - - -
    def detonationResult(self):
        """Integer enumeration for type of detonation and result that occurred."""
        return self.__detonationResult
    @detonationResult.setter
    def detonationResult(self, detonationResult=None):
        if  detonationResult is None:
            detonationResult = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(detonationResult)
        self.__detonationResult = detonationResult
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables the sensor node."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def entityCategory(self):
        """Integer enumerations value for main category that describes the entity, semantics of each code varies according to domain."""
        return self.__entityCategory
    @entityCategory.setter
    def entityCategory(self, entityCategory=None):
        if  entityCategory is None:
            entityCategory = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entityCategory)
        self.__entityCategory = entityCategory
    @property # getter - - - - - - - - - -
    def entityCountry(self):
        """Integer enumerations value for country to which the design of the entity or its design specification is attributed."""
        return self.__entityCountry
    @entityCountry.setter
    def entityCountry(self, entityCountry=None):
        if  entityCountry is None:
            entityCountry = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entityCountry)
        self.__entityCountry = entityCountry
    @property # getter - - - - - - - - - -
    def entityDomain(self):
        """Integer enumerations value for domain in which the entity operates: LAND, AIR, SURFACE, SUBSURFACE, SPACE or OTHER."""
        return self.__entityDomain
    @entityDomain.setter
    def entityDomain(self, entityDomain=None):
        if  entityDomain is None:
            entityDomain = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entityDomain)
        self.__entityDomain = entityDomain
    @property # getter - - - - - - - - - -
    def entityExtra(self):
        """Any extra information required to describe a particular entity."""
        return self.__entityExtra
    @entityExtra.setter
    def entityExtra(self, entityExtra=None):
        if  entityExtra is None:
            entityExtra = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entityExtra)
        self.__entityExtra = entityExtra
    @property # getter - - - - - - - - - -
    def entityID(self):
        """Simulation/exercise entityID is a unique ID for a single entity within that application."""
        return self.__entityID
    @entityID.setter
    def entityID(self, entityID=None):
        if  entityID is None:
            entityID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entityID)
        self.__entityID = entityID
    @property # getter - - - - - - - - - -
    def entityKind(self):
        """Integer enumerations value for whether entity is a PLATFORM, MUNITION, LIFE_FORM, ENVIRONMENTAL, CULTURAL_FEATURE, SUPPLY, RADIO, EXPENDABLE, SENSOR_EMITTER or OTHER."""
        return self.__entityKind
    @entityKind.setter
    def entityKind(self, entityKind=None):
        if  entityKind is None:
            entityKind = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entityKind)
        self.__entityKind = entityKind
    @property # getter - - - - - - - - - -
    def entitySpecific(self):
        """Specific information about an entity based on the Subcategory field."""
        return self.__entitySpecific
    @entitySpecific.setter
    def entitySpecific(self, entitySpecific=None):
        if  entitySpecific is None:
            entitySpecific = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entitySpecific)
        self.__entitySpecific = entitySpecific
    @property # getter - - - - - - - - - -
    def entitySubcategory(self):
        return self.__entitySubcategory
    @entitySubcategory.setter
    def entitySubcategory(self, entitySubcategory=None):
        if  entitySubcategory is None:
            entitySubcategory = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(entitySubcategory)
        self.__entitySubcategory = entitySubcategory
    @property # getter - - - - - - - - - -
    def eventApplicationID(self):
        """Simulation/exercise eventApplicationID is unique for events generated from application at that site."""
        return self.__eventApplicationID
    @eventApplicationID.setter
    def eventApplicationID(self, eventApplicationID=None):
        if  eventApplicationID is None:
            eventApplicationID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(eventApplicationID)
        self.__eventApplicationID = eventApplicationID
    @property # getter - - - - - - - - - -
    def eventEntityID(self):
        """For a given event, simulation/exercise entityID is a unique ID for a single entity within that application."""
        return self.__eventEntityID
    @eventEntityID.setter
    def eventEntityID(self, eventEntityID=None):
        if  eventEntityID is None:
            eventEntityID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(eventEntityID)
        self.__eventEntityID = eventEntityID
    @property # getter - - - - - - - - - -
    def eventNumber(self):
        """Sequential number of each event issued by an application."""
        return self.__eventNumber
    @eventNumber.setter
    def eventNumber(self, eventNumber=None):
        if  eventNumber is None:
            eventNumber = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(eventNumber)
        self.__eventNumber = eventNumber
    @property # getter - - - - - - - - - -
    def eventSiteID(self):
        """Simulation/exercise siteID of the participating LAN or organization."""
        return self.__eventSiteID
    @eventSiteID.setter
    def eventSiteID(self, eventSiteID=None):
        if  eventSiteID is None:
            eventSiteID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(eventSiteID)
        self.__eventSiteID = eventSiteID
    @property # getter - - - - - - - - - -
    def fired1(self):
        """Has the primary weapon (Fire PDU) been fired?."""
        return self.__fired1
    @fired1.setter
    def fired1(self, fired1=None):
        if  fired1 is None:
            fired1 = SFBool.DEFAULT_VALUE
        assertValidSFBool(fired1)
        self.__fired1 = fired1
    @property # getter - - - - - - - - - -
    def fired2(self):
        """Has the secondary weapon (Fire PDU) been fired?."""
        return self.__fired2
    @fired2.setter
    def fired2(self, fired2=None):
        if  fired2 is None:
            fired2 = SFBool.DEFAULT_VALUE
        assertValidSFBool(fired2)
        self.__fired2 = fired2
    @property # getter - - - - - - - - - -
    def fireMissionIndex(self):
        """."""
        return self.__fireMissionIndex
    @fireMissionIndex.setter
    def fireMissionIndex(self, fireMissionIndex=None):
        if  fireMissionIndex is None:
            fireMissionIndex = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(fireMissionIndex)
        self.__fireMissionIndex = fireMissionIndex
    @property # getter - - - - - - - - - -
    def firingRange(self):
        """Range (three dimension, straight-line distance) that the firing entity's fire control system has assumed for computing the fire control solution if a weapon and if the value is known."""
        return self.__firingRange
    @firingRange.setter
    def firingRange(self, firingRange=None):
        if  firingRange is None:
            firingRange = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(firingRange)
        self.__firingRange = firingRange
    @property # getter - - - - - - - - - -
    def firingRate(self):
        """Rate at which munitions are fired."""
        return self.__firingRate
    @firingRate.setter
    def firingRate(self, firingRate=None):
        if  firingRate is None:
            firingRate = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(firingRate)
        self.__firingRate = firingRate
    @property # getter - - - - - - - - - -
    def forceID(self):
        """forceID determines the team membership of the issuing entity, and whether FRIENDLY OPPOSING or NEUTRAL or OTHER."""
        return self.__forceID
    @forceID.setter
    def forceID(self, forceID=None):
        if  forceID is None:
            forceID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(forceID)
        self.__forceID = forceID
    @property # getter - - - - - - - - - -
    def fuse(self):
        """Integer enumerations value for type of fuse on the munition."""
        return self.__fuse
    @fuse.setter
    def fuse(self, fuse=None):
        if  fuse is None:
            fuse = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(fuse)
        self.__fuse = fuse
    @property # getter - - - - - - - - - -
    def geoCoords(self):
        """Geographic location (specified in current geoSystem coordinates) for children geometry (specified in relative coordinate system, in meters)."""
        return self.__geoCoords
    @geoCoords.setter
    def geoCoords(self, geoCoords=None):
        if  geoCoords is None:
            geoCoords = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(geoCoords)
        self.__geoCoords = geoCoords
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def linearAcceleration(self):
        """Acceleration of the entity relative to the rotating Earth in either world or entity coordinates, depending on the dead reckoning algorithm used."""
        return self.__linearAcceleration
    @linearAcceleration.setter
    def linearAcceleration(self, linearAcceleration=None):
        if  linearAcceleration is None:
            linearAcceleration = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(linearAcceleration)
        self.__linearAcceleration = linearAcceleration
    @property # getter - - - - - - - - - -
    def linearVelocity(self):
        """Velocity of the entity relative to the rotating Earth in either world or entity coordinates, depending on the dead reckoning algorithm used."""
        return self.__linearVelocity
    @linearVelocity.setter
    def linearVelocity(self, linearVelocity=None):
        if  linearVelocity is None:
            linearVelocity = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(linearVelocity)
        self.__linearVelocity = linearVelocity
    @property # getter - - - - - - - - - -
    def marking(self):
        """Maximum of 11 characters for simple entity label."""
        return self.__marking
    @marking.setter
    def marking(self, marking=None):
        if  marking is None:
            marking = SFString.DEFAULT_VALUE
        assertValidSFString(marking)
        self.__marking = marking
    @property # getter - - - - - - - - - -
    def multicastRelayHost(self):
        """Fallback server address if multicast not available locally."""
        return self.__multicastRelayHost
    @multicastRelayHost.setter
    def multicastRelayHost(self, multicastRelayHost=None):
        if  multicastRelayHost is None:
            multicastRelayHost = SFString.DEFAULT_VALUE
        assertValidSFString(multicastRelayHost)
        self.__multicastRelayHost = multicastRelayHost
    @property # getter - - - - - - - - - -
    def multicastRelayPort(self):
        """Fallback server port if multicast not available locally."""
        return self.__multicastRelayPort
    @multicastRelayPort.setter
    def multicastRelayPort(self, multicastRelayPort=None):
        if  multicastRelayPort is None:
            multicastRelayPort = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(multicastRelayPort)
        self.__multicastRelayPort = multicastRelayPort
    @property # getter - - - - - - - - - -
    def munitionApplicationID(self):
        """munitionApplicationID, unique for application at that site."""
        return self.__munitionApplicationID
    @munitionApplicationID.setter
    def munitionApplicationID(self, munitionApplicationID=None):
        if  munitionApplicationID is None:
            munitionApplicationID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(munitionApplicationID)
        self.__munitionApplicationID = munitionApplicationID
    @property # getter - - - - - - - - - -
    def munitionEndPoint(self):
        """Final point of the munition path from firing weapon to detonation or impact, in exercise coordinates."""
        return self.__munitionEndPoint
    @munitionEndPoint.setter
    def munitionEndPoint(self, munitionEndPoint=None):
        if  munitionEndPoint is None:
            munitionEndPoint = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(munitionEndPoint)
        self.__munitionEndPoint = munitionEndPoint
    @property # getter - - - - - - - - - -
    def munitionEntityID(self):
        """munitionEntityID is unique ID for entity firing munition within that application."""
        return self.__munitionEntityID
    @munitionEntityID.setter
    def munitionEntityID(self, munitionEntityID=None):
        if  munitionEntityID is None:
            munitionEntityID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(munitionEntityID)
        self.__munitionEntityID = munitionEntityID
    @property # getter - - - - - - - - - -
    def munitionQuantity(self):
        """Quantity of munitions fired."""
        return self.__munitionQuantity
    @munitionQuantity.setter
    def munitionQuantity(self, munitionQuantity=None):
        if  munitionQuantity is None:
            munitionQuantity = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(munitionQuantity)
        self.__munitionQuantity = munitionQuantity
    @property # getter - - - - - - - - - -
    def munitionSiteID(self):
        """Munition siteID of the participating LAN or organization."""
        return self.__munitionSiteID
    @munitionSiteID.setter
    def munitionSiteID(self, munitionSiteID=None):
        if  munitionSiteID is None:
            munitionSiteID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(munitionSiteID)
        self.__munitionSiteID = munitionSiteID
    @property # getter - - - - - - - - - -
    def munitionStartPoint(self):
        """Initial point of the munition path from firing weapon to detonation or impact, in exercise coordinates."""
        return self.__munitionStartPoint
    @munitionStartPoint.setter
    def munitionStartPoint(self, munitionStartPoint=None):
        if  munitionStartPoint is None:
            munitionStartPoint = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(munitionStartPoint)
        self.__munitionStartPoint = munitionStartPoint
    @property # getter - - - - - - - - - -
    def networkMode(self):
        """Whether this entity is ignoring the network, sending DIS packets to the network, or receiving DIS packets from the network."""
        return self.__networkMode
    @networkMode.setter
    def networkMode(self, networkMode=None):
        if  networkMode is None:
            networkMode = SFString.DEFAULT_VALUE
        assertValidSFString(networkMode)
        self.__networkMode = networkMode
    @property # getter - - - - - - - - - -
    def port(self):
        """Network connection port number (EXAMPLE 3000) for sending or receiving DIS messages."""
        return self.__port
    @port.setter
    def port(self, port=None):
        if  port is None:
            port = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(port)
        self.__port = port
    @property # getter - - - - - - - - - -
    def readInterval(self):
        """[0,+infinity) Seconds between read updates, 0 means no reading."""
        return self.__readInterval
    @readInterval.setter
    def readInterval(self, readInterval=None):
        if  readInterval is None:
            readInterval = SFTime.DEFAULT_VALUE
        assertValidSFTime(readInterval)
        assertNonNegative('readInterval', readInterval)
        self.__readInterval = readInterval
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation of children relative to local coordinate system, usually read from (or written to) remote, networked EspduTransform nodes."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation=None):
        if  rotation is None:
            rotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def rtpHeaderExpected(self):
        """Whether RTP headers are prepended to DIS PDUs."""
        return self.__rtpHeaderExpected
    @rtpHeaderExpected.setter
    def rtpHeaderExpected(self, rtpHeaderExpected=None):
        if  rtpHeaderExpected is None:
            rtpHeaderExpected = SFBool.DEFAULT_VALUE
        assertValidSFBool(rtpHeaderExpected)
        self.__rtpHeaderExpected = rtpHeaderExpected
    @property # getter - - - - - - - - - -
    def scale(self):
        """Non-uniform x-y-z scale of child coordinate system, adjusted by center and scaleOrientation."""
        return self.__scale
    @scale.setter
    def scale(self, scale=None):
        if  scale is None:
            scale = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(scale)
        self.__scale = scale
    @property # getter - - - - - - - - - -
    def scaleOrientation(self):
        """Preliminary rotation of coordinate system before scaling (to allow scaling around arbitrary orientations)."""
        return self.__scaleOrientation
    @scaleOrientation.setter
    def scaleOrientation(self, scaleOrientation=None):
        if  scaleOrientation is None:
            scaleOrientation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(scaleOrientation)
        self.__scaleOrientation = scaleOrientation
    @property # getter - - - - - - - - - -
    def siteID(self):
        """Simulation/exercise siteID of the participating LAN or organization."""
        return self.__siteID
    @siteID.setter
    def siteID(self, siteID=None):
        if  siteID is None:
            siteID = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(siteID)
        self.__siteID = siteID
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position of children relative to local coordinate system, usually read from (or written to) remote, networked EspduTransform nodes."""
        return self.__translation
    @translation.setter
    def translation(self, translation=None):
        if  translation is None:
            translation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def warhead(self):
        """Integer enumerations value for type of warhead on the munition."""
        return self.__warhead
    @warhead.setter
    def warhead(self, warhead=None):
        if  warhead is None:
            warhead = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(warhead)
        self.__warhead = warhead
    @property # getter - - - - - - - - - -
    def writeInterval(self):
        """[0,+infinity) Seconds between write updates, 0 means no writing (sending)."""
        return self.__writeInterval
    @writeInterval.setter
    def writeInterval(self, writeInterval=None):
        if  writeInterval is None:
            writeInterval = SFTime.DEFAULT_VALUE
        assertValidSFTime(writeInterval)
        assertNonNegative('writeInterval', writeInterval)
        self.__writeInterval = writeInterval
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EspduTransform.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EspduTransform'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.address != 'localhost':
            result += " address='" + self.address + "'"
        if self.applicationID != 0:
            result += " applicationID='" + str(self.applicationID) + "'"
        if self.articulationParameterArray != list():
            result += " articulationParameterArray='" + str(self.articulationParameterArray) + "'"
        if self.articulationParameterChangeIndicatorArray != list():
            result += " articulationParameterChangeIndicatorArray='" + str(self.articulationParameterChangeIndicatorArray) + "'"
        if self.articulationParameterCount != 0:
            result += " articulationParameterCount='" + str(self.articulationParameterCount) + "'"
        if self.articulationParameterDesignatorArray != list():
            result += " articulationParameterDesignatorArray='" + str(self.articulationParameterDesignatorArray) + "'"
        if self.articulationParameterIdPartAttachedToArray != list():
            result += " articulationParameterIdPartAttachedToArray='" + str(self.articulationParameterIdPartAttachedToArray) + "'"
        if self.articulationParameterTypeArray != list():
            result += " articulationParameterTypeArray='" + str(self.articulationParameterTypeArray) + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.center != (0, 0, 0):
            result += " center='" + str(self.center) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.collisionType != 0:
            result += " collisionType='" + str(self.collisionType) + "'"
        if self.deadReckoning != 0:
            result += " deadReckoning='" + str(self.deadReckoning) + "'"
        if self.detonationLocation != (0, 0, 0):
            result += " detonationLocation='" + str(self.detonationLocation) + "'"
        if self.detonationRelativeLocation != (0, 0, 0):
            result += " detonationRelativeLocation='" + str(self.detonationRelativeLocation) + "'"
        if self.detonationResult != 0:
            result += " detonationResult='" + str(self.detonationResult) + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.entityCategory != 0:
            result += " entityCategory='" + str(self.entityCategory) + "'"
        if self.entityCountry != 0:
            result += " entityCountry='" + str(self.entityCountry) + "'"
        if self.entityDomain != 0:
            result += " entityDomain='" + str(self.entityDomain) + "'"
        if self.entityExtra != 0:
            result += " entityExtra='" + str(self.entityExtra) + "'"
        if self.entityID != 0:
            result += " entityID='" + str(self.entityID) + "'"
        if self.entityKind != 0:
            result += " entityKind='" + str(self.entityKind) + "'"
        if self.entitySpecific != 0:
            result += " entitySpecific='" + str(self.entitySpecific) + "'"
        if self.entitySubcategory != 0:
            result += " entitySubcategory='" + str(self.entitySubcategory) + "'"
        if self.eventApplicationID != 0:
            result += " eventApplicationID='" + str(self.eventApplicationID) + "'"
        if self.eventEntityID != 0:
            result += " eventEntityID='" + str(self.eventEntityID) + "'"
        if self.eventNumber != 0:
            result += " eventNumber='" + str(self.eventNumber) + "'"
        if self.eventSiteID != 0:
            result += " eventSiteID='" + str(self.eventSiteID) + "'"
        if self.fireMissionIndex != 0:
            result += " fireMissionIndex='" + str(self.fireMissionIndex) + "'"
        if self.fired1 != False:
            result += " fired1='" + str(self.fired1) + "'"
        if self.fired2 != False:
            result += " fired2='" + str(self.fired2) + "'"
        if self.firingRange != 0:
            result += " firingRange='" + str(self.firingRange) + "'"
        if self.firingRate != 0:
            result += " firingRate='" + str(self.firingRate) + "'"
        if self.forceID != 0:
            result += " forceID='" + str(self.forceID) + "'"
        if self.fuse != 0:
            result += " fuse='" + str(self.fuse) + "'"
        if self.geoCoords != (0, 0, 0):
            result += " geoCoords='" + str(self.geoCoords) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.linearAcceleration != (0, 0, 0):
            result += " linearAcceleration='" + str(self.linearAcceleration) + "'"
        if self.linearVelocity != (0, 0, 0):
            result += " linearVelocity='" + str(self.linearVelocity) + "'"
        if self.marking:
            result += " marking='" + self.marking + "'"
        if self.multicastRelayHost:
            result += " multicastRelayHost='" + self.multicastRelayHost + "'"
        if self.multicastRelayPort != 0:
            result += " multicastRelayPort='" + str(self.multicastRelayPort) + "'"
        if self.munitionApplicationID != 0:
            result += " munitionApplicationID='" + str(self.munitionApplicationID) + "'"
        if self.munitionEndPoint != (0, 0, 0):
            result += " munitionEndPoint='" + str(self.munitionEndPoint) + "'"
        if self.munitionEntityID != 0:
            result += " munitionEntityID='" + str(self.munitionEntityID) + "'"
        if self.munitionQuantity != 0:
            result += " munitionQuantity='" + str(self.munitionQuantity) + "'"
        if self.munitionSiteID != 0:
            result += " munitionSiteID='" + str(self.munitionSiteID) + "'"
        if self.munitionStartPoint != (0, 0, 0):
            result += " munitionStartPoint='" + str(self.munitionStartPoint) + "'"
        if self.networkMode != 'standAlone':
            result += " networkMode='" + self.networkMode + "'"
        if self.port != 0:
            result += " port='" + str(self.port) + "'"
        if self.readInterval != 0.1:
            result += " readInterval='" + str(self.readInterval) + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + str(self.rotation) + "'"
        if self.rtpHeaderExpected != False:
            result += " rtpHeaderExpected='" + str(self.rtpHeaderExpected) + "'"
        if self.scale != (1, 1, 1):
            result += " scale='" + str(self.scale) + "'"
        if self.scaleOrientation != (0, 0, 1, 0):
            result += " scaleOrientation='" + str(self.scaleOrientation) + "'"
        if self.siteID != 0:
            result += " siteID='" + str(self.siteID) + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + str(self.translation) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if self.warhead != 0:
            result += " warhead='" + str(self.warhead) + "'"
        if self.writeInterval != 1.0:
            result += " writeInterval='" + str(self.writeInterval) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</EspduTransform>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ExplosionEmitter(_X3DParticleEmitterNode):
    """
    ExplosionEmitter generates all particles from a specific point in space at the initial time enabled.
    """
    NAME = 'ExplosionEmitter'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ExplosionEmitter'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('mass', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('position', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'ExplosionEmitter'),
        ('speed', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('surfaceArea', 0, FieldType.SFFloat, AccessType.initializeOnly, 'X3DParticleEmitterNode'),
        ('variation', 0.25, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 mass=0,
                 position=(0, 0, 0),
                 speed=0,
                 surfaceArea=0,
                 variation=0.25,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ExplosionEmitter __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.mass = mass
        self.position = position
        self.speed = speed
        self.surfaceArea = surfaceArea
        self.variation = variation
    @property # getter - - - - - - - - - -
    def mass(self):
        """[0,+infinity) Basic mass of each particle, defined in mass base units (default is kilograms)."""
        return self.__mass
    @mass.setter
    def mass(self, mass=None):
        if  mass is None:
            mass = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(mass)
        assertNonNegative('mass', mass)
        self.__mass = mass
    @property # getter - - - - - - - - - -
    def position(self):
        """Point from which particles emanate."""
        return self.__position
    @position.setter
    def position(self, position=None):
        if  position is None:
            position = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(position)
        self.__position = position
    @property # getter - - - - - - - - - -
    def speed(self):
        """[0,+infinity) Initial linear speed (default is m/s) imparted to all particles along their direction of movement."""
        return self.__speed
    @speed.setter
    def speed(self, speed=None):
        if  speed is None:
            speed = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(speed)
        assertNonNegative('speed', speed)
        self.__speed = speed
    @property # getter - - - - - - - - - -
    def surfaceArea(self):
        """[0,+infinity) Particle surface area in area base units (default is meters squared)."""
        return self.__surfaceArea
    @surfaceArea.setter
    def surfaceArea(self, surfaceArea=None):
        if  surfaceArea is None:
            surfaceArea = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(surfaceArea)
        assertNonNegative('surfaceArea', surfaceArea)
        self.__surfaceArea = surfaceArea
    @property # getter - - - - - - - - - -
    def variation(self):
        """[0,+infinity) Multiplier for the randomness used to control the range of possible output values."""
        return self.__variation
    @variation.setter
    def variation(self, variation=None):
        if  variation is None:
            variation = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(variation)
        assertNonNegative('variation', variation)
        self.__variation = variation
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ExplosionEmitter.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ExplosionEmitter'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.mass != 0:
            result += " mass='" + str(self.mass) + "'"
        if self.position != (0, 0, 0):
            result += " position='" + str(self.position) + "'"
        if self.speed != 0:
            result += " speed='" + str(self.speed) + "'"
        if self.surfaceArea != 0:
            result += " surfaceArea='" + str(self.surfaceArea) + "'"
        if self.variation != 0.25:
            result += " variation='" + str(self.variation) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ExplosionEmitter>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Extrusion(_X3DGeometryNode):
    """
    Extrusion is a geometry node that sequentially stretches a 2D cross section along a 3D-spine path in the local coordinate system, creating an outer hull.
    """
    NAME = 'Extrusion'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Extrusion'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('beginCap', True, FieldType.SFBool, AccessType.initializeOnly, 'Extrusion'),
        ('ccw', True, FieldType.SFBool, AccessType.initializeOnly, 'Extrusion'),
        ('convex', True, FieldType.SFBool, AccessType.initializeOnly, 'Extrusion'),
        ('creaseAngle', 0.0, FieldType.SFFloat, AccessType.initializeOnly, 'Extrusion'),
        ('crossSection', [(1, 1, 1, -1, -1, -1, -1, 1, 1, 1)], FieldType.MFVec2f, AccessType.initializeOnly, 'Extrusion'),
        ('endCap', True, FieldType.SFBool, AccessType.initializeOnly, 'Extrusion'),
        ('orientation', [(0, 0, 1, 0)], FieldType.MFRotation, AccessType.inputOutput, 'Extrusion'),
        ('scale', [(1, 1)], FieldType.MFVec2f, AccessType.inputOutput, 'Extrusion'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'Extrusion'),
        ('spine', [(0, 0, 0, 0, 1, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'Extrusion'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 beginCap=True,
                 ccw=True,
                 convex=True,
                 creaseAngle=0.0,
                 crossSection=[(1, 1, 1, -1, -1, -1, -1, 1, 1, 1)],
                 endCap=True,
                 orientation=[(0, 0, 1, 0)],
                 scale=[(1, 1)],
                 solid=True,
                 spine=[(0, 0, 0, 0, 1, 0)],
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Extrusion __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.beginCap = beginCap
        self.ccw = ccw
        self.convex = convex
        self.creaseAngle = creaseAngle
        self.crossSection = crossSection
        self.endCap = endCap
        self.orientation = orientation
        self.scale = scale
        self.solid = solid
        self.spine = spine
    @property # getter - - - - - - - - - -
    def beginCap(self):
        """Whether beginning cap is drawn (similar to Cylinder top cap)."""
        return self.__beginCap
    @beginCap.setter
    def beginCap(self, beginCap=None):
        if  beginCap is None:
            beginCap = SFBool.DEFAULT_VALUE
        assertValidSFBool(beginCap)
        self.__beginCap = beginCap
    @property # getter - - - - - - - - - -
    def ccw(self):
        """The ccw field indicates counterclockwise ordering of vertex-coordinates orientation."""
        return self.__ccw
    @ccw.setter
    def ccw(self, ccw=None):
        if  ccw is None:
            ccw = SFBool.DEFAULT_VALUE
        assertValidSFBool(ccw)
        self.__ccw = ccw
    @property # getter - - - - - - - - - -
    def convex(self):
        """The convex field is a hint to renderers whether all polygons in a shape are convex (true), or possibly concave (false)."""
        return self.__convex
    @convex.setter
    def convex(self, convex=None):
        if  convex is None:
            convex = SFBool.DEFAULT_VALUE
        assertValidSFBool(convex)
        self.__convex = convex
    @property # getter - - - - - - - - - -
    def creaseAngle(self):
        """[0,+infinity) creaseAngle defines angle (in radians) where adjacent polygons are drawn with sharp edges or smooth shading."""
        return self.__creaseAngle
    @creaseAngle.setter
    def creaseAngle(self, creaseAngle=None):
        if  creaseAngle is None:
            creaseAngle = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(creaseAngle)
        assertNonNegative('creaseAngle', creaseAngle)
        self.__creaseAngle = creaseAngle
    @property # getter - - - - - - - - - -
    def crossSection(self):
        """The crossSection array defines a silhouette outline of the outer Extrusion surface."""
        return self.__crossSection
    @crossSection.setter
    def crossSection(self, crossSection=None):
        if  crossSection is None:
            crossSection = MFVec2f.DEFAULT_VALUE
        assertValidMFVec2f(crossSection)
        self.__crossSection = crossSection
    @property # getter - - - - - - - - - -
    def endCap(self):
        """Whether end cap is drawn (similar to Cylinder bottom cap)."""
        return self.__endCap
    @endCap.setter
    def endCap(self, endCap=None):
        if  endCap is None:
            endCap = SFBool.DEFAULT_VALUE
        assertValidSFBool(endCap)
        self.__endCap = endCap
    @property # getter - - - - - - - - - -
    def orientation(self):
        """The orientation array is a list of axis-angle 4-tuple values applied at each spine-aligned cross-section plane."""
        return self.__orientation
    @orientation.setter
    def orientation(self, orientation=None):
        if  orientation is None:
            orientation = MFRotation.DEFAULT_VALUE
        assertValidMFRotation(orientation)
        self.__orientation = orientation
    @property # getter - - - - - - - - - -
    def scale(self):
        """(0,+infinity) scale is a list of 2D-scale parameters applied at each spine-aligned cross-section plane."""
        return self.__scale
    @scale.setter
    def scale(self, scale=None):
        if  scale is None:
            scale = MFVec2f.DEFAULT_VALUE
        assertValidMFVec2f(scale)
        self.__scale = scale
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def spine(self):
        """The spine array defines a center-line sequence of 3D points that define a piecewise-linear curve forming a series of connected vertices."""
        return self.__spine
    @spine.setter
    def spine(self, spine=None):
        if  spine is None:
            spine = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(spine)
        self.__spine = spine
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Extrusion.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Extrusion'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.beginCap != True:
            result += " beginCap='" + str(self.beginCap) + "'"
        if self.ccw != True:
            result += " ccw='" + str(self.ccw) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.convex != True:
            result += " convex='" + str(self.convex) + "'"
        if self.creaseAngle != 0.0:
            result += " creaseAngle='" + str(self.creaseAngle) + "'"
        if self.crossSection != [(1, 1, 1, -1, -1, -1, -1, 1, 1, 1)]:
            result += " crossSection='" + str(self.crossSection) + "'"
        if self.endCap != True:
            result += " endCap='" + str(self.endCap) + "'"
        if self.orientation != [(0, 0, 1, 0)]:
            result += " orientation='" + str(self.orientation) + "'"
        if self.scale != [(1, 1)]:
            result += " scale='" + str(self.scale) + "'"
        if self.solid != True:
            result += " solid='" + str(self.solid) + "'"
        if self.spine != [(0, 0, 0, 0, 1, 0)]:
            result += " spine='" + str(self.spine) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Extrusion>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class FillProperties(_X3DAppearanceChildNode):
    """
    FillProperties indicates whether appearance is filled or hatched for associated geometry nodes inside the same Shape.
    """
    NAME = 'FillProperties'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#FillProperties'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('filled', True, FieldType.SFBool, AccessType.inputOutput, 'FillProperties'),
        ('hatchColor', (1, 1, 1), FieldType.SFColor, AccessType.inputOutput, 'FillProperties'),
        ('hatched', True, FieldType.SFBool, AccessType.inputOutput, 'FillProperties'),
        ('hatchStyle', 1, FieldType.SFInt32, AccessType.inputOutput, 'FillProperties'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 filled=True,
                 hatchColor=(1, 1, 1),
                 hatched=True,
                 hatchStyle=1,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode FillProperties __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.filled = filled
        self.hatchColor = hatchColor
        self.hatched = hatched
        self.hatchStyle = hatchStyle
    @property # getter - - - - - - - - - -
    def filled(self):
        """Whether or not associated geometry is filled."""
        return self.__filled
    @filled.setter
    def filled(self, filled=None):
        if  filled is None:
            filled = SFBool.DEFAULT_VALUE
        assertValidSFBool(filled)
        self.__filled = filled
    @property # getter - - - - - - - - - -
    def hatchColor(self):
        """[0,1] Color of the hatch pattern."""
        return self.__hatchColor
    @hatchColor.setter
    def hatchColor(self, hatchColor=None):
        if  hatchColor is None:
            hatchColor = SFColor.DEFAULT_VALUE
        assertValidSFColor(hatchColor)
        assertZeroToOne('hatchColor', hatchColor)
        self.__hatchColor = hatchColor
    @property # getter - - - - - - - - - -
    def hatched(self):
        """Whether or not associated geometry is hatched."""
        return self.__hatched
    @hatched.setter
    def hatched(self, hatched=None):
        if  hatched is None:
            hatched = SFBool.DEFAULT_VALUE
        assertValidSFBool(hatched)
        self.__hatched = hatched
    @property # getter - - - - - - - - - -
    def hatchStyle(self):
        """hatchStyle selects a hatch pattern from International Register of Graphical Items."""
        return self.__hatchStyle
    @hatchStyle.setter
    def hatchStyle(self, hatchStyle=None):
        if  hatchStyle is None:
            hatchStyle = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(hatchStyle)
        self.__hatchStyle = hatchStyle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function FillProperties.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<FillProperties'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.filled != True:
            result += " filled='" + str(self.filled) + "'"
        if self.hatchColor != (1, 1, 1):
            result += " hatchColor='" + str(self.hatchColor) + "'"
        if self.hatchStyle != 1:
            result += " hatchStyle='" + str(self.hatchStyle) + "'"
        if self.hatched != True:
            result += " hatched='" + str(self.hatched) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</FillProperties>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class FloatVertexAttribute(_X3DVertexAttributeNode):
    """
    FloatVertexAttribute defines a set of per-vertex single-precision floating-point attributes.
    """
    NAME = 'FloatVertexAttribute'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#FloatVertexAttribute'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'X3DVertexAttributeNode'),
        ('numComponents', 4, FieldType.SFInt32, AccessType.initializeOnly, 'FloatVertexAttribute'),
        ('value', list(), FieldType.MFFloat, AccessType.inputOutput, 'FloatVertexAttribute'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 name='',
                 numComponents=4,
                 value=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode FloatVertexAttribute __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.name = name
        self.numComponents = numComponents
        self.value = value
    @property # getter - - - - - - - - - -
    def name(self):
        """Required name for this particular VertexAttribute instance."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def numComponents(self):
        """numComponents pecifies how many consecutive floating-point values should be grouped together per vertex."""
        return self.__numComponents
    @numComponents.setter
    def numComponents(self, numComponents=None):
        if  numComponents is None:
            numComponents = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(numComponents)
        assertGreaterThanEquals('numComponents', numComponents, 1)
        assertLessThanEquals('numComponents', numComponents, 4)
        self.__numComponents = numComponents
    @property # getter - - - - - - - - - -
    def value(self):
        """value specifies an arbitrary collection of floating-point values that will be passed to the shader as per-vertex information."""
        return self.__value
    @value.setter
    def value(self, value=None):
        if  value is None:
            value = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(value)
        self.__value = value
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function FloatVertexAttribute.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<FloatVertexAttribute'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.numComponents != 4:
            result += " numComponents='" + str(self.numComponents) + "'"
        if self.value != list():
            result += " value='" + str(self.value) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</FloatVertexAttribute>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Fog(_X3DBindableNode, _X3DFogObject):
    """
    Fog simulates atmospheric effects by blending distant objects with fog color.
    """
    NAME = 'Fog'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Fog'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('color', (1, 1, 1), FieldType.SFColor, AccessType.inputOutput, 'X3DFogObject'),
        ('fogType', 'LINEAR', FieldType.SFString, AccessType.inputOutput, 'X3DFogObject'),
        ('visibilityRange', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DFogObject'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 color=(1, 1, 1),
                 fogType='LINEAR',
                 visibilityRange=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Fog __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.color = color
        self.fogType = fogType
        self.visibilityRange = visibilityRange
    @property # getter - - - - - - - - - -
    def color(self):
        """[0,1] Fog color."""
        return self.__color
    @color.setter
    def color(self, color=None):
        if  color is None:
            color = SFColor.DEFAULT_VALUE
        assertValidSFColor(color)
        assertZeroToOne('color', color)
        self.__color = color
    @property # getter - - - - - - - - - -
    def fogType(self):
        """Specifies algorithm for rate of increasing Fog, either LINEAR or EXPONENTIAL."""
        return self.__fogType
    @fogType.setter
    def fogType(self, fogType=None):
        if  fogType is None:
            fogType = SFString.DEFAULT_VALUE
        assertValidSFString(fogType)
        self.__fogType = fogType
    @property # getter - - - - - - - - - -
    def visibilityRange(self):
        """Distance in meters where objects are totally obscured by the fog, using local coordinate system."""
        return self.__visibilityRange
    @visibilityRange.setter
    def visibilityRange(self, visibilityRange=None):
        if  visibilityRange is None:
            visibilityRange = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(visibilityRange)
        assertNonNegative('visibilityRange', visibilityRange)
        self.__visibilityRange = visibilityRange
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Fog.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Fog'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.color != (1, 1, 1):
            result += " color='" + str(self.color) + "'"
        if self.fogType != 'LINEAR':
            result += " fogType='" + self.fogType + "'"
        if self.visibilityRange != 0:
            result += " visibilityRange='" + str(self.visibilityRange) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Fog>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class FogCoordinate(_X3DGeometricPropertyNode):
    """
    FogCoordinate defines a set of explicit fog depths on a per-vertex basis, overriding Fog visibilityRange.
    """
    NAME = 'FogCoordinate'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#FogCoordinate'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('depth', list(), FieldType.MFFloat, AccessType.inputOutput, 'FogCoordinate'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 depth=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode FogCoordinate __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.depth = depth
    @property # getter - - - - - - - - - -
    def depth(self):
        """depth contains a set of 3D coordinate (triplet) point values."""
        return self.__depth
    @depth.setter
    def depth(self, depth=None):
        if  depth is None:
            depth = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(depth)
        assertZeroToOne('depth', depth)
        self.__depth = depth
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function FogCoordinate.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<FogCoordinate'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.depth != list():
            result += " depth='" + str(self.depth) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</FogCoordinate>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class FontStyle(_X3DFontStyleNode):
    """
    FontStyle is an X3DFontStyleNode that defines the size, family, justification, and other styles used by Text nodes.
    """
    NAME = 'FontStyle'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#FontStyle'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('family', ["SERIF"], FieldType.MFString, AccessType.initializeOnly, 'FontStyle'),
        ('horizontal', True, FieldType.SFBool, AccessType.initializeOnly, 'FontStyle'),
        ('justify', ["BEGIN"], FieldType.MFString, AccessType.initializeOnly, 'FontStyle'),
        ('language', '', FieldType.SFString, AccessType.initializeOnly, 'FontStyle'),
        ('leftToRight', True, FieldType.SFBool, AccessType.initializeOnly, 'FontStyle'),
        ('size', 1.0, FieldType.SFFloat, AccessType.initializeOnly, 'FontStyle'),
        ('spacing', 1.0, FieldType.SFFloat, AccessType.initializeOnly, 'FontStyle'),
        ('style', 'PLAIN', FieldType.SFString, AccessType.initializeOnly, 'FontStyle'),
        ('topToBottom', True, FieldType.SFBool, AccessType.initializeOnly, 'FontStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 family=["SERIF"],
                 horizontal=True,
                 justify=["BEGIN"],
                 language='',
                 leftToRight=True,
                 size=1.0,
                 spacing=1.0,
                 style='PLAIN',
                 topToBottom=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode FontStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.family = family
        self.horizontal = horizontal
        self.justify = justify
        self.language = language
        self.leftToRight = leftToRight
        self.size = size
        self.spacing = spacing
        self.style = style
        self.topToBottom = topToBottom
    @property # getter - - - - - - - - - -
    def family(self):
        """Array of quoted font family names in preference order, browsers use the first supported family."""
        return self.__family
    @family.setter
    def family(self, family=None):
        if  family is None:
            family = MFString.DEFAULT_VALUE
        assertValidMFString(family)
        self.__family = family
    @property # getter - - - - - - - - - -
    def horizontal(self):
        """Whether text direction is horizontal (true) or vertical (false)."""
        return self.__horizontal
    @horizontal.setter
    def horizontal(self, horizontal=None):
        if  horizontal is None:
            horizontal = SFBool.DEFAULT_VALUE
        assertValidSFBool(horizontal)
        self.__horizontal = horizontal
    @property # getter - - - - - - - - - -
    def justify(self):
        """The justify field determines alignment of the above text layout relative to the origin of the object coordinate system."""
        return self.__justify
    @justify.setter
    def justify(self, justify=None):
        if  justify is None:
            justify = MFString.DEFAULT_VALUE
        assertValidMFString(justify)
        self.__justify = justify
    @property # getter - - - - - - - - - -
    def language(self):
        """Language codes consist of a primary code and a (possibly empty) series of subcodes."""
        return self.__language
    @language.setter
    def language(self, language=None):
        if  language is None:
            language = SFString.DEFAULT_VALUE
        assertValidSFString(language)
        self.__language = language
    @property # getter - - - - - - - - - -
    def leftToRight(self):
        return self.__leftToRight
    @leftToRight.setter
    def leftToRight(self, leftToRight=None):
        if  leftToRight is None:
            leftToRight = SFBool.DEFAULT_VALUE
        assertValidSFBool(leftToRight)
        self.__leftToRight = leftToRight
    @property # getter - - - - - - - - - -
    def size(self):
        """(0,+infinity) Nominal height (in local coordinate system) of text glyphs Also sets default spacing between adjacent lines of text."""
        return self.__size
    @size.setter
    def size(self, size=None):
        if  size is None:
            size = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(size)
        assertPositive('size', size)
        self.__size = size
    @property # getter - - - - - - - - - -
    def spacing(self):
        """[0,+infinity) Adjustment factor for line spacing between adjacent lines of text."""
        return self.__spacing
    @spacing.setter
    def spacing(self, spacing=None):
        if  spacing is None:
            spacing = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(spacing)
        assertNonNegative('spacing', spacing)
        self.__spacing = spacing
    @property # getter - - - - - - - - - -
    def style(self):
        """Pick one of four values for text style."""
        return self.__style
    @style.setter
    def style(self, style=None):
        if  style is None:
            style = SFString.DEFAULT_VALUE
        assertValidSFString(style)
        self.__style = style
    @property # getter - - - - - - - - - -
    def topToBottom(self):
        """Whether text direction is top-to-bottom (true) or bottom-to-top (false)."""
        return self.__topToBottom
    @topToBottom.setter
    def topToBottom(self, topToBottom=None):
        if  topToBottom is None:
            topToBottom = SFBool.DEFAULT_VALUE
        assertValidSFBool(topToBottom)
        self.__topToBottom = topToBottom
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function FontStyle.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<FontStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.family != ["SERIF"]:
            result += " family='" + self.family + "'"
        if self.horizontal != True:
            result += " horizontal='" + str(self.horizontal) + "'"
        if self.justify != ["BEGIN"]:
            result += " justify='" + self.justify + "'"
        if self.language:
            result += " language='" + self.language + "'"
        if self.leftToRight != True:
            result += " leftToRight='" + str(self.leftToRight) + "'"
        if self.size != 1.0:
            result += " size='" + str(self.size) + "'"
        if self.spacing != 1.0:
            result += " spacing='" + str(self.spacing) + "'"
        if self.style != 'PLAIN':
            result += " style='" + self.style + "'"
        if self.topToBottom != True:
            result += " topToBottom='" + str(self.topToBottom) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</FontStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class ForcePhysicsModel(_X3DParticlePhysicsModelNode):
    """
    ForcePhysicsModel applies a constant force value to the particles.
    """
    NAME = 'ForcePhysicsModel'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ForcePhysicsModel'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DParticlePhysicsModelNode'),
        ('force', (0, -9.8, 0), FieldType.SFVec3f, AccessType.inputOutput, 'ForcePhysicsModel'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 force=(0, -9.8, 0),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ForcePhysicsModel __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.force = force
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def force(self):
        """(-infinity,+infinity) force field indicates strength and direction of the propelling force on the particles (e."""
        return self.__force
    @force.setter
    def force(self, force=None):
        if  force is None:
            force = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(force)
        self.__force = force
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ForcePhysicsModel.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ForcePhysicsModel'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.force != (0, -9.8, 0):
            result += " force='" + str(self.force) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</ForcePhysicsModel>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeneratedCubeMapTexture(_X3DEnvironmentTextureNode):
    """
    GeneratedCubeMapTexture is a texture node that defines a cubic environment map that sources its data from internally generated images.
    """
    NAME = 'GeneratedCubeMapTexture'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeneratedCubeMapTexture'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('size', 128, FieldType.SFInt32, AccessType.initializeOnly, 'GeneratedCubeMapTexture'),
        ('update', 'NONE', FieldType.SFString, AccessType.inputOutput, 'GeneratedCubeMapTexture'),
        ('textureProperties', None, FieldType.SFNode, AccessType.initializeOnly, 'GeneratedCubeMapTexture'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 size=128,
                 update='NONE',
                 textureProperties=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeneratedCubeMapTexture __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.size = size
        self.update = update
        self.textureProperties = textureProperties
    @property # getter - - - - - - - - - -
    def size(self):
        """(0,+infinity) size indicates the resolution of the generated images in number of pixels per side."""
        return self.__size
    @size.setter
    def size(self, size=None):
        if  size is None:
            size = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(size)
        assertPositive('size', size)
        self.__size = size
    @property # getter - - - - - - - - - -
    def update(self):
        """update controls regeneration of the texture."""
        return self.__update
    @update.setter
    def update(self, update=None):
        if  update is None:
            update = SFString.DEFAULT_VALUE
        assertValidSFString(update)
        self.__update = update
    @property # getter - - - - - - - - - -
    def textureProperties(self):
        return self.__textureProperties
    @textureProperties.setter
    def textureProperties(self, textureProperties=None):
        if  textureProperties is None:
            textureProperties = SFNode.DEFAULT_VALUE
        assertValidSFNode(textureProperties)
        if not isinstance(textureProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(textureProperties) + ' does not have a valid node type object')
        self.__textureProperties = textureProperties
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata) or bool(self.textureProperties)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeneratedCubeMapTexture.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeneratedCubeMapTexture'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.size != 128:
            result += " size='" + str(self.size) + "'"
        if self.update != 'NONE':
            result += " update='" + self.update + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.textureProperties: # walk each child node, if any
                for each in self.textureProperties:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeneratedCubeMapTexture>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoCoordinate(_X3DCoordinateNode):
    """
    GeoCoordinate builds geometry as a set of geographic 3D coordinates.
    """
    NAME = 'GeoCoordinate'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoCoordinate'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoCoordinate'),
        ('point', list(), FieldType.MFVec3d, AccessType.inputOutput, 'GeoCoordinate'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoCoordinate'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 geoSystem=["GD", "WE"],
                 point=list(),
                 geoOrigin=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoCoordinate __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.geoSystem = geoSystem
        self.point = point
        self.geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def point(self):
        """point contains a set of actual 3D geographic coordinates, provided in geoSystem format can split strings if desired: "x1 y1 z1 x2 y2 z2" or "x1 y1 z1", "x2 y2 z2"."""
        return self.__point
    @point.setter
    def point(self, point=None):
        if  point is None:
            point = MFVec3d.DEFAULT_VALUE
        assertValidMFVec3d(point)
        self.__point = point
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoCoordinate.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoCoordinate'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.point != list():
            result += " point='" + str(self.point) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoCoordinate>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoElevationGrid(_X3DGeometryNode):
    """
    GeoElevationGrid is a geometry node defining a rectangular height field, with default values for a 1m by 1m square at height 0.
    """
    NAME = 'GeoElevationGrid'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoElevationGrid'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('ccw', True, FieldType.SFBool, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('colorPerVertex', True, FieldType.SFBool, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('creaseAngle', 0, FieldType.SFDouble, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('geoGridOrigin', (0, 0, 0), FieldType.SFVec3d, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('height', [0, 0, 0, 0], FieldType.MFDouble, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('normalPerVertex', True, FieldType.SFBool, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('xDimension', 2, FieldType.SFInt32, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('xSpacing', 1.0, FieldType.SFDouble, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('yScale', 1, FieldType.SFFloat, AccessType.inputOutput, 'GeoElevationGrid'),
        ('zDimension', 2, FieldType.SFInt32, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('zSpacing', 1.0, FieldType.SFDouble, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('color', None, FieldType.SFNode, AccessType.inputOutput, 'GeoElevationGrid'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoElevationGrid'),
        ('normal', None, FieldType.SFNode, AccessType.inputOutput, 'GeoElevationGrid'),
        ('texCoord', None, FieldType.SFNode, AccessType.inputOutput, 'GeoElevationGrid'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 ccw=True,
                 colorPerVertex=True,
                 creaseAngle=0,
                 geoGridOrigin=(0, 0, 0),
                 geoSystem=["GD", "WE"],
                 height=[0, 0, 0, 0],
                 normalPerVertex=True,
                 solid=True,
                 xDimension=2,
                 xSpacing=1.0,
                 yScale=1,
                 zDimension=2,
                 zSpacing=1.0,
                 color=None,
                 geoOrigin=None,
                 normal=None,
                 texCoord=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoElevationGrid __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.ccw = ccw
        self.colorPerVertex = colorPerVertex
        self.creaseAngle = creaseAngle
        self.geoGridOrigin = geoGridOrigin
        self.geoSystem = geoSystem
        self.height = height
        self.normalPerVertex = normalPerVertex
        self.solid = solid
        self.xDimension = xDimension
        self.xSpacing = xSpacing
        self.yScale = yScale
        self.zDimension = zDimension
        self.zSpacing = zSpacing
        self.color = color
        self.geoOrigin = geoOrigin
        self.normal = normal
        self.texCoord = texCoord
    @property # getter - - - - - - - - - -
    def ccw(self):
        """ccw defines clockwise/counterclockwise ordering of vertex coordinates, which in turn defines front/back orientation of polygon normals according to Right-Hand Rule (RHR)."""
        return self.__ccw
    @ccw.setter
    def ccw(self, ccw=None):
        if  ccw is None:
            ccw = SFBool.DEFAULT_VALUE
        assertValidSFBool(ccw)
        self.__ccw = ccw
    @property # getter - - - - - - - - - -
    def colorPerVertex(self):
        """Whether Color node color values are applied to each point vertex (true) or per quadrilateral (false)."""
        return self.__colorPerVertex
    @colorPerVertex.setter
    def colorPerVertex(self, colorPerVertex=None):
        if  colorPerVertex is None:
            colorPerVertex = SFBool.DEFAULT_VALUE
        assertValidSFBool(colorPerVertex)
        self.__colorPerVertex = colorPerVertex
    @property # getter - - - - - - - - - -
    def creaseAngle(self):
        """(0,+infinity) creaseAngle defines angle (in radians) for determining whether adjacent polygons are drawn with sharp edges or smooth shading."""
        return self.__creaseAngle
    @creaseAngle.setter
    def creaseAngle(self, creaseAngle=None):
        if  creaseAngle is None:
            creaseAngle = SFDouble.DEFAULT_VALUE
        assertValidSFDouble(creaseAngle)
        assertNonNegative('creaseAngle', creaseAngle)
        self.__creaseAngle = creaseAngle
    @property # getter - - - - - - - - - -
    def geoGridOrigin(self):
        """Geographic coordinate for southwest (lower-left) corner of height dataset."""
        return self.__geoGridOrigin
    @geoGridOrigin.setter
    def geoGridOrigin(self, geoGridOrigin=None):
        if  geoGridOrigin is None:
            geoGridOrigin = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(geoGridOrigin)
        self.__geoGridOrigin = geoGridOrigin
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def height(self):
        """Contains xDimension rows * zDimension columns floating-point values for elevation above ellipsoid."""
        return self.__height
    @height.setter
    def height(self, height=None):
        if  height is None:
            height = MFDouble.DEFAULT_VALUE
        assertValidMFDouble(height)
        self.__height = height
    @property # getter - - - - - - - - - -
    def normalPerVertex(self):
        """Whether Normal node vector values are applied to each point vertex (true) or per quadrilateral (false)."""
        return self.__normalPerVertex
    @normalPerVertex.setter
    def normalPerVertex(self, normalPerVertex=None):
        if  normalPerVertex is None:
            normalPerVertex = SFBool.DEFAULT_VALUE
        assertValidSFBool(normalPerVertex)
        self.__normalPerVertex = normalPerVertex
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid=None):
        if  solid is None:
            solid = SFBool.DEFAULT_VALUE
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def xDimension(self):
        """(0,+infinity) Number of elements in the height array along east-west X direction."""
        return self.__xDimension
    @xDimension.setter
    def xDimension(self, xDimension=None):
        if  xDimension is None:
            xDimension = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(xDimension)
        assertNonNegative('xDimension', xDimension)
        self.__xDimension = xDimension
    @property # getter - - - - - - - - - -
    def xSpacing(self):
        """(0,+infinity) Distance between grid-array vertices along east-west X direction."""
        return self.__xSpacing
    @xSpacing.setter
    def xSpacing(self, xSpacing=None):
        if  xSpacing is None:
            xSpacing = SFDouble.DEFAULT_VALUE
        assertValidSFDouble(xSpacing)
        assertPositive('xSpacing', xSpacing)
        self.__xSpacing = xSpacing
    @property # getter - - - - - - - - - -
    def yScale(self):
        """[0,+infinity) Vertical exaggeration of displayed data produced from the height array."""
        return self.__yScale
    @yScale.setter
    def yScale(self, yScale=None):
        if  yScale is None:
            yScale = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(yScale)
        assertNonNegative('yScale', yScale)
        self.__yScale = yScale
    @property # getter - - - - - - - - - -
    def zDimension(self):
        """(0,+infinity) Number of elements in the height array along north-south Z direction."""
        return self.__zDimension
    @zDimension.setter
    def zDimension(self, zDimension=None):
        if  zDimension is None:
            zDimension = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(zDimension)
        assertNonNegative('zDimension', zDimension)
        self.__zDimension = zDimension
    @property # getter - - - - - - - - - -
    def zSpacing(self):
        """(0,+infinity) Distance between grid-array vertices along north-south Z direction."""
        return self.__zSpacing
    @zSpacing.setter
    def zSpacing(self, zSpacing=None):
        if  zSpacing is None:
            zSpacing = SFDouble.DEFAULT_VALUE
        assertValidSFDouble(zSpacing)
        assertPositive('zSpacing', zSpacing)
        self.__zSpacing = zSpacing
    @property # getter - - - - - - - - - -
    def color(self):
        """[X3DColorNode] Single contained Color or ColorRGBA node that specifies color values applied to corresponding vertices according to colorPerVertex field."""
        return self.__color
    @color.setter
    def color(self, color=None):
        if  color is None:
            color = SFNode.DEFAULT_VALUE
        assertValidSFNode(color)
        if not isinstance(color, object):
            # print(flush=True)
            raise X3DTypeError(str(color) + ' does not have a valid node type object')
        self.__color = color
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def normal(self):
        """[X3DNormalNode] Single contained Normal node that specifies perpendicular vectors for corresponding vertices to support rendering computations, applied according to the normalPerVertex field."""
        return self.__normal
    @normal.setter
    def normal(self, normal=None):
        if  normal is None:
            normal = SFNode.DEFAULT_VALUE
        assertValidSFNode(normal)
        if not isinstance(normal, object):
            # print(flush=True)
            raise X3DTypeError(str(normal) + ' does not have a valid node type object')
        self.__normal = normal
    @property # getter - - - - - - - - - -
    def texCoord(self):
        """[X3DTextureCoordinateNode] Single contained TextureCoordinate, TextureCoordinateGenerator or MultiTextureCoordinate node that specifies coordinates for texture mapping onto corresponding geometry."""
        return self.__texCoord
    @texCoord.setter
    def texCoord(self, texCoord=None):
        if  texCoord is None:
            texCoord = SFNode.DEFAULT_VALUE
        assertValidSFNode(texCoord)
        if not isinstance(texCoord, object):
            # print(flush=True)
            raise X3DTypeError(str(texCoord) + ' does not have a valid node type object')
        self.__texCoord = texCoord
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.color) or bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata) or bool(self.normal) or bool(self.texCoord)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoElevationGrid.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoElevationGrid'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.ccw != True:
            result += " ccw='" + str(self.ccw) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.colorPerVertex != True:
            result += " colorPerVertex='" + str(self.colorPerVertex) + "'"
        if self.creaseAngle != 0:
            result += " creaseAngle='" + str(self.creaseAngle) + "'"
        if self.geoGridOrigin != (0, 0, 0):
            result += " geoGridOrigin='" + str(self.geoGridOrigin) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.height != [0, 0, 0, 0]:
            result += " height='" + str(self.height) + "'"
        if self.normalPerVertex != True:
            result += " normalPerVertex='" + str(self.normalPerVertex) + "'"
        if self.solid != True:
            result += " solid='" + str(self.solid) + "'"
        if self.xDimension != 2:
            result += " xDimension='" + str(self.xDimension) + "'"
        if self.xSpacing != 1.0:
            result += " xSpacing='" + str(self.xSpacing) + "'"
        if self.yScale != 1:
            result += " yScale='" + str(self.yScale) + "'"
        if self.zDimension != 2:
            result += " zDimension='" + str(self.zDimension) + "'"
        if self.zSpacing != 1.0:
            result += " zSpacing='" + str(self.zSpacing) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.color: # walk each child node, if any
                for each in self.color:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.normal: # walk each child node, if any
                for each in self.normal:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.texCoord: # walk each child node, if any
                for each in self.texCoord:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoElevationGrid>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoLocation(_X3DGroupingNode):
    """
    GeoLocation positions a regular X3D model onto earth's surface.
    """
    NAME = 'GeoLocation'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoLocation'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('geoCoords', (0, 0, 0), FieldType.SFVec3d, AccessType.inputOutput, 'GeoLocation'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoLocation'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoLocation'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 geoCoords=(0, 0, 0),
                 geoSystem=["GD", "WE"],
                 visible=True,
                 geoOrigin=None,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoLocation __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.geoCoords = geoCoords
        self.geoSystem = geoSystem
        self.visible = visible
        self.geoOrigin = geoOrigin
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def geoCoords(self):
        """Geographic location (specified in current geoSystem coordinates) for children geometry (specified in relative coordinate system, in meters)."""
        return self.__geoCoords
    @geoCoords.setter
    def geoCoords(self, geoCoords=None):
        if  geoCoords is None:
            geoCoords = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(geoCoords)
        self.__geoCoords = geoCoords
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoLocation.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoLocation'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.geoCoords != (0, 0, 0):
            result += " geoCoords='" + str(self.geoCoords) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoLocation>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoLOD(_X3DChildNode, _X3DBoundedObject):
    """
    Note that MFNode rootNode field can contain multiple nodes and has accessType inputOutput. Meanwhile MFNode children field is outputOnly, unlike other X3DGroupingNode exemplars.
    """
    NAME = 'GeoLOD'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoLOD'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('center', (0, 0, 0), FieldType.SFVec3d, AccessType.initializeOnly, 'GeoLOD'),
        ('child1Url', list(), FieldType.MFString, AccessType.initializeOnly, 'GeoLOD'),
        ('child2Url', list(), FieldType.MFString, AccessType.initializeOnly, 'GeoLOD'),
        ('child3Url', list(), FieldType.MFString, AccessType.initializeOnly, 'GeoLOD'),
        ('child4Url', list(), FieldType.MFString, AccessType.initializeOnly, 'GeoLOD'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoLOD'),
        ('range', 10, FieldType.SFFloat, AccessType.initializeOnly, 'GeoLOD'),
        ('rootUrl', list(), FieldType.MFString, AccessType.initializeOnly, 'GeoLOD'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoLOD'),
        ('rootNode', list(), FieldType.MFNode, AccessType.initializeOnly, 'GeoLOD'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 center=(0, 0, 0),
                 child1Url=list(),
                 child2Url=list(),
                 child3Url=list(),
                 child4Url=list(),
                 displayBBox=False,
                 geoSystem=["GD", "WE"],
                 range=10,
                 rootUrl=list(),
                 visible=True,
                 geoOrigin=None,
                 rootNode=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoLOD __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.center = center
        self.child1Url = child1Url
        self.child2Url = child2Url
        self.child3Url = child3Url
        self.child4Url = child4Url
        self.displayBBox = displayBBox
        self.geoSystem = geoSystem
        self.range = range
        self.rootUrl = rootUrl
        self.visible = visible
        self.geoOrigin = geoOrigin
        self.rootNode = rootNode
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def center(self):
        """Viewer range from geographic-coordinates center triggers quadtree loading/unloading."""
        return self.__center
    @center.setter
    def center(self, center=None):
        if  center is None:
            center = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(center)
        self.__center = center
    @property # getter - - - - - - - - - -
    def child1Url(self):
        """quadtree geometry loaded when viewer is within range."""
        return self.__child1Url
    @child1Url.setter
    def child1Url(self, child1Url=None):
        if  child1Url is None:
            child1Url = MFString.DEFAULT_VALUE
        assertValidMFString(child1Url)
        self.__child1Url = child1Url
    @property # getter - - - - - - - - - -
    def child2Url(self):
        """quadtree geometry loaded when viewer is within range."""
        return self.__child2Url
    @child2Url.setter
    def child2Url(self, child2Url=None):
        if  child2Url is None:
            child2Url = MFString.DEFAULT_VALUE
        assertValidMFString(child2Url)
        self.__child2Url = child2Url
    @property # getter - - - - - - - - - -
    def child3Url(self):
        """quadtree geometry loaded when viewer is within range."""
        return self.__child3Url
    @child3Url.setter
    def child3Url(self, child3Url=None):
        if  child3Url is None:
            child3Url = MFString.DEFAULT_VALUE
        assertValidMFString(child3Url)
        self.__child3Url = child3Url
    @property # getter - - - - - - - - - -
    def child4Url(self):
        """quadtree geometry loaded when viewer is within range."""
        return self.__child4Url
    @child4Url.setter
    def child4Url(self, child4Url=None):
        if  child4Url is None:
            child4Url = MFString.DEFAULT_VALUE
        assertValidMFString(child4Url)
        self.__child4Url = child4Url
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def range(self):
        """(0,+infinity) Viewer range from geographic-coordinates center triggers quadtree loading/unloading."""
        return self.__range
    @range.setter
    def range(self, range=None):
        if  range is None:
            range = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(range)
        assertNonNegative('range', range)
        self.__range = range
    @property # getter - - - - - - - - - -
    def rootUrl(self):
        """url for scene providing geometry for the root tile."""
        return self.__rootUrl
    @rootUrl.setter
    def rootUrl(self, rootUrl=None):
        if  rootUrl is None:
            rootUrl = MFString.DEFAULT_VALUE
        assertValidMFString(rootUrl)
        self.__rootUrl = rootUrl
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def rootNode(self):
        """Geometry for the root tile."""
        return self.__rootNode
    @rootNode.setter
    def rootNode(self, rootNode=None):
        if  rootNode is None:
            rootNode = MFNode.DEFAULT_VALUE
        assertValidMFNode(rootNode)
        self.__rootNode = rootNode
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.rootNode) or bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoLOD.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoLOD'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.center != (0, 0, 0):
            result += " center='" + str(self.center) + "'"
        if self.child1Url != list():
            result += " child1Url='" + self.child1Url + "'"
        if self.child2Url != list():
            result += " child2Url='" + self.child2Url + "'"
        if self.child3Url != list():
            result += " child3Url='" + self.child3Url + "'"
        if self.child4Url != list():
            result += " child4Url='" + self.child4Url + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.range != 10:
            result += " range='" + str(self.range) + "'"
        if self.rootUrl != list():
            result += " rootUrl='" + self.rootUrl + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.rootNode: # walk each child node, if any
                for each in self.rootNode:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoLOD>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoMetadata(_X3DInfoNode, _X3DUrlObject):
    """
    GeoMetadata includes a generic subset of metadata about the geographic data.
    """
    NAME = 'GeoMetadata'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoMetadata'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('summary', list(), FieldType.MFString, AccessType.inputOutput, 'GeoMetadata'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('data', list(), FieldType.MFNode, AccessType.inputOutput, 'GeoMetadata'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 summary=list(),
                 url=list(),
                 data=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoMetadata __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.summary = summary
        self.url = url
        self.data = data
    @property # getter - - - - - - - - - -
    def summary(self):
        """The summary string array contains a set of keyword/value pairs, with each keyword and its subsequent value contained in separate strings."""
        return self.__summary
    @summary.setter
    def summary(self, summary=None):
        if  summary is None:
            summary = MFString.DEFAULT_VALUE
        assertValidMFString(summary)
        self.__summary = summary
    @property # getter - - - - - - - - - -
    def url(self):
        """Hypertext link to an external, complete metadata description."""
        return self.__url
    @url.setter
    def url(self, url=None):
        if  url is None:
            url = MFString.DEFAULT_VALUE
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def data(self):
        """DEF list of all nodes that implement this data."""
        return self.__data
    @data.setter
    def data(self, data=None):
        if  data is None:
            data = MFNode.DEFAULT_VALUE
        assertValidMFNode(data)
        self.__data = data
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.data) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoMetadata.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoMetadata'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.summary != list():
            result += " summary='" + self.summary + "'"
        if self.url != list():
            result += " url='" + self.url + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.data: # walk each child node, if any
                for each in self.data:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoMetadata>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoOrigin(_X3DNode):
    """
    GeoOrigin is deprecated and discouraged (but nevertheless allowed) in X3D version 3.3. GeoOrigin is restored in X3D version 4.0 for special use on devices with limited floating-point resolution.
    """
    NAME = 'GeoOrigin'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoOrigin'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('geoCoords', (0, 0, 0), FieldType.SFVec3d, AccessType.inputOutput, 'GeoOrigin'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoOrigin'),
        ('rotateYUp', False, FieldType.SFBool, AccessType.initializeOnly, 'GeoOrigin'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 geoCoords=(0, 0, 0),
                 geoSystem=["GD", "WE"],
                 rotateYUp=False,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoOrigin __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.geoCoords = geoCoords
        self.geoSystem = geoSystem
        self.rotateYUp = rotateYUp
    @property # getter - - - - - - - - - -
    def geoCoords(self):
        """Defines absolute geographic location (and implicit local coordinate frame)."""
        return self.__geoCoords
    @geoCoords.setter
    def geoCoords(self, geoCoords=None):
        if  geoCoords is None:
            geoCoords = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(geoCoords)
        self.__geoCoords = geoCoords
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def rotateYUp(self):
        """Whether to rotate coordinates of nodes using this GeoOrigin so that local-up direction aligns with VRML Y axis rotateYUp false means local up-direction is relative to planet surface rotateYUp true allows proper operation of NavigationInfo modes FLY, WALK."""
        return self.__rotateYUp
    @rotateYUp.setter
    def rotateYUp(self, rotateYUp=None):
        if  rotateYUp is None:
            rotateYUp = SFBool.DEFAULT_VALUE
        assertValidSFBool(rotateYUp)
        self.__rotateYUp = rotateYUp
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoOrigin.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoOrigin'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.geoCoords != (0, 0, 0):
            result += " geoCoords='" + str(self.geoCoords) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.rotateYUp != False:
            result += " rotateYUp='" + str(self.rotateYUp) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoOrigin>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoPositionInterpolator(_X3DInterpolatorNode):
    """
    GeoPositionInterpolator animates objects within a geographic coordinate system.
    """
    NAME = 'GeoPositionInterpolator'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoPositionInterpolator'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoPositionInterpolator'),
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DInterpolatorNode'),
        ('keyValue', list(), FieldType.MFVec3d, AccessType.inputOutput, 'GeoPositionInterpolator'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoPositionInterpolator'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 geoSystem=["GD", "WE"],
                 key=list(),
                 keyValue=list(),
                 geoOrigin=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoPositionInterpolator __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.geoSystem = geoSystem
        self.key = key
        self.keyValue = keyValue
        self.geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key=None):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear interpolation, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue=None):
        if  keyValue is None:
            keyValue = MFVec3d.DEFAULT_VALUE
        assertValidMFVec3d(keyValue)
        self.__keyValue = keyValue
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoPositionInterpolator.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoPositionInterpolator'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.key != list():
            result += " key='" + str(self.key) + "'"
        if self.keyValue != list():
            result += " keyValue='" + str(self.keyValue) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoPositionInterpolator>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoProximitySensor(_X3DEnvironmentalSensorNode):
    """
    GeoProximitySensor generates events when the viewer enters, exits and moves within a region of space (defined by a box).
    """
    NAME = 'GeoProximitySensor'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoProximitySensor'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('center', (0, 0, 0), FieldType.SFVec3d, AccessType.inputOutput, 'GeoProximitySensor'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('geoCenter', (0, 0, 0), FieldType.SFVec3d, AccessType.inputOutput, 'GeoProximitySensor'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoProximitySensor'),
        ('size', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DEnvironmentalSensorNode'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoProximitySensor'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 center=(0, 0, 0),
                 enabled=True,
                 geoCenter=(0, 0, 0),
                 geoSystem=["GD", "WE"],
                 size=(0, 0, 0),
                 geoOrigin=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoProximitySensor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.center = center
        self.enabled = enabled
        self.geoCenter = geoCenter
        self.geoSystem = geoSystem
        self.size = size
        self.geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def center(self):
        """(starting with v3."""
        return self.__center
    @center.setter
    def center(self, center=None):
        if  center is None:
            center = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(center)
        self.__center = center
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def geoCenter(self):
        """(deprecated as of v3."""
        return self.__geoCenter
    @geoCenter.setter
    def geoCenter(self, geoCenter=None):
        if  geoCenter is None:
            geoCenter = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(geoCenter)
        self.__geoCenter = geoCenter
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def size(self):
        """[0,+infinity) size of Proximity box."""
        return self.__size
    @size.setter
    def size(self, size=None):
        if  size is None:
            size = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(size)
        assertNonNegative('size', size)
        self.__size = size
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoProximitySensor.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoProximitySensor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.center != (0, 0, 0):
            result += " center='" + str(self.center) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.geoCenter != (0, 0, 0):
            result += " geoCenter='" + str(self.geoCenter) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.size != (0, 0, 0):
            result += " size='" + str(self.size) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoProximitySensor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoTouchSensor(_X3DTouchSensorNode):
    """
    GeoTouchSensor returns geographic coordinates for the object being selected.
    """
    NAME = 'GeoTouchSensor'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoTouchSensor'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DPointingDeviceSensorNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoTouchSensor'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoTouchSensor'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 description='',
                 enabled=True,
                 geoSystem=["GD", "WE"],
                 geoOrigin=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoTouchSensor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.description = description
        self.enabled = enabled
        self.geoSystem = geoSystem
        self.geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (G D), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoTouchSensor.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoTouchSensor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.enabled != True:
            result += " enabled='" + str(self.enabled) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoTouchSensor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoTransform(_X3DGroupingNode):
    """
    GeoTransform is a Grouping node that can contain most nodes.
    """
    NAME = 'GeoTransform'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoTransform'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('geoCenter', (0, 0, 0), FieldType.SFVec3d, AccessType.inputOutput, 'GeoTransform'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoTransform'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'GeoTransform'),
        ('scale', (1, 1, 1), FieldType.SFVec3f, AccessType.inputOutput, 'GeoTransform'),
        ('scaleOrientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'GeoTransform'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'GeoTransform'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoTransform'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 geoCenter=(0, 0, 0),
                 geoSystem=["GD", "WE"],
                 rotation=(0, 0, 1, 0),
                 scale=(1, 1, 1),
                 scaleOrientation=(0, 0, 1, 0),
                 translation=(0, 0, 0),
                 visible=True,
                 geoOrigin=None,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoTransform __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.geoCenter = geoCenter
        self.geoSystem = geoSystem
        self.rotation = rotation
        self.scale = scale
        self.scaleOrientation = scaleOrientation
        self.translation = translation
        self.visible = visible
        self.geoOrigin = geoOrigin
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def geoCenter(self):
        """Translation offset from origin of local coordinate system, applied prior to rotation or scaling."""
        return self.__geoCenter
    @geoCenter.setter
    def geoCenter(self, geoCenter=None):
        if  geoCenter is None:
            geoCenter = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(geoCenter)
        self.__geoCenter = geoCenter
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation (axis, angle in radians) of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation=None):
        if  rotation is None:
            rotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def scale(self):
        """Non-uniform x-y-z scale of child coordinate system, adjusted by center and scaleOrientation."""
        return self.__scale
    @scale.setter
    def scale(self, scale=None):
        if  scale is None:
            scale = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(scale)
        assertPositive('scale', scale)
        self.__scale = scale
    @property # getter - - - - - - - - - -
    def scaleOrientation(self):
        """Preliminary rotation of coordinate sys tem before scaling (to allow scaling around arbitrary orientations)."""
        return self.__scaleOrientation
    @scaleOrientation.setter
    def scaleOrientation(self, scaleOrientation=None):
        if  scaleOrientation is None:
            scaleOrientation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(scaleOrientation)
        self.__scaleOrientation = scaleOrientation
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position (x, y, z in meters) of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation=None):
        if  translation is None:
            translation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoTransform.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoTransform'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.geoCenter != (0, 0, 0):
            result += " geoCenter='" + str(self.geoCenter) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + str(self.rotation) + "'"
        if self.scale != (1, 1, 1):
            result += " scale='" + str(self.scale) + "'"
        if self.scaleOrientation != (0, 0, 1, 0):
            result += " scaleOrientation='" + str(self.scaleOrientation) + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + str(self.translation) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoTransform>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class GeoViewpoint(_X3DViewpointNode):
    """
    GeoViewpoint specifies viewpoints using geographic coordinates.
    """
    NAME = 'GeoViewpoint'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#GeoViewpoint'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('centerOfRotation', (0, 0, 0), FieldType.SFVec3d, AccessType.inputOutput, 'GeoViewpoint'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DViewpointNode'),
        ('fieldOfView', 0.7854, FieldType.SFFloat, AccessType.inputOutput, 'GeoViewpoint'),
        ('geoSystem', ["GD", "WE"], FieldType.MFString, AccessType.initializeOnly, 'GeoViewpoint'),
        ('jump', True, FieldType.SFBool, AccessType.inputOutput, 'X3DViewpointNode'),
        ('orientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'X3DViewpointNode'),
        ('position', (0, 0, 100000), FieldType.SFVec3d, AccessType.inputOutput, 'GeoViewpoint'),
        ('retainUserOffsets', False, FieldType.SFBool, AccessType.inputOutput, 'X3DViewpointNode'),
        ('speedFactor', 1.0, FieldType.SFFloat, AccessType.initializeOnly, 'GeoViewpoint'),
        ('geoOrigin', None, FieldType.SFNode, AccessType.initializeOnly, 'GeoViewpoint'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 centerOfRotation=(0, 0, 0),
                 description='',
                 fieldOfView=0.7854,
                 geoSystem=["GD", "WE"],
                 jump=True,
                 orientation=(0, 0, 1, 0),
                 position=(0, 0, 100000),
                 retainUserOffsets=False,
                 speedFactor=1.0,
                 geoOrigin=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode GeoViewpoint __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.centerOfRotation = centerOfRotation
        self.description = description
        self.fieldOfView = fieldOfView
        self.geoSystem = geoSystem
        self.jump = jump
        self.orientation = orientation
        self.position = position
        self.retainUserOffsets = retainUserOffsets
        self.speedFactor = speedFactor
        self.geoOrigin = geoOrigin
    @property # getter - - - - - - - - - -
    def centerOfRotation(self):
        """centerOfRotation specifies center point about which to rotate user's eyepoint when in EXAMINE or LOOKAT mode."""
        return self.__centerOfRotation
    @centerOfRotation.setter
    def centerOfRotation(self, centerOfRotation=None):
        if  centerOfRotation is None:
            centerOfRotation = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(centerOfRotation)
        self.__centerOfRotation = centerOfRotation
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def fieldOfView(self):
        """Preferred minimum viewing angle from this viewpoint in radians, providing minimum height or minimum width (whichever is smaller)."""
        return self.__fieldOfView
    @fieldOfView.setter
    def fieldOfView(self, fieldOfView=None):
        if  fieldOfView is None:
            fieldOfView = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(fieldOfView)
        assertGreaterThan('fieldOfView', fieldOfView, 0)
        assertLessThan('fieldOfView', fieldOfView, 3.1416)
        self.__fieldOfView = fieldOfView
    @property # getter - - - - - - - - - -
    def geoSystem(self):
        """Identifies spatial reference frame: Geodetic (GD), Geocentric (GC), Universal Transverse Mercator (UTM)."""
        return self.__geoSystem
    @geoSystem.setter
    def geoSystem(self, geoSystem=None):
        if  geoSystem is None:
            geoSystem = MFString.DEFAULT_VALUE
        assertValidMFString(geoSystem)
        self.__geoSystem = geoSystem
    @property # getter - - - - - - - - - -
    def jump(self):
        """Whether to transition instantly by jumping, or else smoothly animate to this Viewpoint."""
        return self.__jump
    @jump.setter
    def jump(self, jump=None):
        if  jump is None:
            jump = SFBool.DEFAULT_VALUE
        assertValidSFBool(jump)
        self.__jump = jump
    @property # getter - - - - - - - - - -
    def orientation(self):
        """Rotation of Viewpoint, relative to default -Z axis direction in local coordinate system."""
        return self.__orientation
    @orientation.setter
    def orientation(self, orientation=None):
        if  orientation is None:
            orientation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(orientation)
        self.__orientation = orientation
    @property # getter - - - - - - - - - -
    def position(self):
        """position relative to local georeferenced coordinate system, in proper format."""
        return self.__position
    @position.setter
    def position(self, position=None):
        if  position is None:
            position = SFVec3d.DEFAULT_VALUE
        assertValidSFVec3d(position)
        self.__position = position
    @property # getter - - - - - - - - - -
    def retainUserOffsets(self):
        """Retain (true) or reset to zero (false) any prior user navigation offsets from defined viewpoint position, orientation."""
        return self.__retainUserOffsets
    @retainUserOffsets.setter
    def retainUserOffsets(self, retainUserOffsets=None):
        if  retainUserOffsets is None:
            retainUserOffsets = SFBool.DEFAULT_VALUE
        assertValidSFBool(retainUserOffsets)
        self.__retainUserOffsets = retainUserOffsets
    @property # getter - - - - - - - - - -
    def speedFactor(self):
        """[0,+infinity) speedFactor is a multiplier to modify the original elevation-based speed that is set automatically by the browser."""
        return self.__speedFactor
    @speedFactor.setter
    def speedFactor(self, speedFactor=None):
        if  speedFactor is None:
            speedFactor = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(speedFactor)
        assertNonNegative('speedFactor', speedFactor)
        self.__speedFactor = speedFactor
    @property # getter - - - - - - - - - -
    def geoOrigin(self):
        return self.__geoOrigin
    @geoOrigin.setter
    def geoOrigin(self, geoOrigin=None):
        if  geoOrigin is None:
            geoOrigin = SFNode.DEFAULT_VALUE
        assertValidSFNode(geoOrigin)
        if not isinstance(geoOrigin, object):
            # print(flush=True)
            raise X3DTypeError(str(geoOrigin) + ' does not have a valid node type object')
        self.__geoOrigin = geoOrigin
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.geoOrigin) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function GeoViewpoint.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<GeoViewpoint'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.centerOfRotation != (0, 0, 0):
            result += " centerOfRotation='" + str(self.centerOfRotation) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.fieldOfView != 0.7854:
            result += " fieldOfView='" + str(self.fieldOfView) + "'"
        if self.geoSystem != ["GD", "WE"]:
            result += " geoSystem='" + self.geoSystem + "'"
        if self.jump != True:
            result += " jump='" + str(self.jump) + "'"
        if self.orientation != (0, 0, 1, 0):
            result += " orientation='" + str(self.orientation) + "'"
        if self.position != (0, 0, 100000):
            result += " position='" + str(self.position) + "'"
        if self.retainUserOffsets != False:
            result += " retainUserOffsets='" + str(self.retainUserOffsets) + "'"
        if self.speedFactor != 1.0:
            result += " speedFactor='" + str(self.speedFactor) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.geoOrigin: # walk each child node, if any
                for each in self.geoOrigin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</GeoViewpoint>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class Group(_X3DGroupingNode):
    """
    Group is a Grouping node that can contain most nodes.
    """
    NAME = 'Group'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Group'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 visible=True,
                 children=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Group __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Group.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Group'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</Group>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class HAnimDisplacer(_X3DGeometricPropertyNode):
    """
    HAnimDisplacer nodes alter the shape of coordinate-based geometry within parent HAnimJoint or HAnimSegment nodes.
    """
    NAME = 'HAnimDisplacer'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#HAnimDisplacer'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('coordIndex', list(), FieldType.MFInt32, AccessType.initializeOnly, 'HAnimDisplacer'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'HAnimDisplacer'),
        ('displacements', list(), FieldType.MFVec3f, AccessType.inputOutput, 'HAnimDisplacer'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'HAnimDisplacer'),
        ('weight', 0.0, FieldType.SFFloat, AccessType.inputOutput, 'HAnimDisplacer'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 coordIndex=list(),
                 description='',
                 displacements=list(),
                 name='',
                 weight=0.0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode HAnimDisplacer __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.coordIndex = coordIndex
        self.description = description
        self.displacements = displacements
        self.name = name
        self.weight = weight
    @property # getter - - - - - - - - - -
    def coordIndex(self):
        """[0,+infinity) Defines index values into the parent HAnimSegment or HAnimBody/HAnimHumanoid coordinate array for the mesh of vertices affected by this HAnimDisplacer."""
        return self.__coordIndex
    @coordIndex.setter
    def coordIndex(self, coordIndex=None):
        if  coordIndex is None:
            coordIndex = MFInt32.DEFAULT_VALUE
        assertValidMFInt32(coordIndex)
        assertNonNegative('coordIndex', coordIndex)
        self.__coordIndex = coordIndex
    @property # getter - - - - - - - - - -
    def description(self):
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def displacements(self):
        """displacements are a set of SFVec3f values added to neutral/resting position of each of the corresponding HAnimSegment vertices (or HAnimJoint/HAnimHumanoid vertices) referenced by coordIndex field."""
        return self.__displacements
    @displacements.setter
    def displacements(self, displacements=None):
        if  displacements is None:
            displacements = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(displacements)
        self.__displacements = displacements
    @property # getter - - - - - - - - - -
    def name(self):
        """Unique name attribute must be defined so that HAnimDisplacer node can be identified at run time for animation purposes."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def weight(self):
        """The weigh factor has typical range [0,1] and defines the scale factor applied to displacement values before adding them to neutral vertex positions."""
        return self.__weight
    @weight.setter
    def weight(self, weight=None):
        if  weight is None:
            weight = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(weight)
        self.__weight = weight
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function HAnimDisplacer.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<HAnimDisplacer'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.coordIndex != list():
            result += " coordIndex='" + str(self.coordIndex) + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.displacements != list():
            result += " displacements='" + str(self.displacements) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.weight != 0.0:
            result += " weight='" + str(self.weight) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</HAnimDisplacer>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class HAnimHumanoid(_X3DChildNode, _X3DBoundedObject):
    """
    The HAnimHumanoid node is used to: (a) store references to the joints, segments, sites, skeleton, optional skin, and fixed viewpoints, (b) serve as a container for the entire humanoid, (c) provide a convenient way of moving the humanoid through its environment, and (d) store human-readable metadata such as name, version, author, copyright, age, gender and other information.
    """
    NAME = 'HAnimHumanoid'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#HAnimHumanoid'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('center', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimHumanoid'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'HAnimHumanoid'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('info', list(), FieldType.MFString, AccessType.inputOutput, 'HAnimHumanoid'),
        ('jointBindingPositions', [(0, 0, 0)], FieldType.MFVec3f, AccessType.inputOutput, 'HAnimHumanoid'),
        ('jointBindingRotations', [(0, 0, 1, 0)], FieldType.MFRotation, AccessType.inputOutput, 'HAnimHumanoid'),
        ('jointBindingScales', [(0, 0, 0)], FieldType.MFVec3f, AccessType.inputOutput, 'HAnimHumanoid'),
        ('loa', -1, FieldType.SFInt32, AccessType.inputOutput, 'HAnimHumanoid'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'HAnimHumanoid'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'HAnimHumanoid'),
        ('scale', (1, 1, 1), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimHumanoid'),
        ('scaleOrientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'HAnimHumanoid'),
        ('skeletalConfiguration', 'BASIC', FieldType.SFString, AccessType.inputOutput, 'HAnimHumanoid'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimHumanoid'),
        ('version', '2.0', FieldType.SFString, AccessType.inputOutput, 'HAnimHumanoid'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('skinBindingCoords', None, FieldType.SFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('skinBindingNormals', None, FieldType.SFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('skinCoord', None, FieldType.SFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('skinNormal', None, FieldType.SFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('joints', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('motions', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('segments', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('sites', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('skeleton', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('skin', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('viewpoints', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimHumanoid'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 center=(0, 0, 0),
                 description='',
                 displayBBox=False,
                 info=list(),
                 jointBindingPositions=[(0, 0, 0)],
                 jointBindingRotations=[(0, 0, 1, 0)],
                 jointBindingScales=[(0, 0, 0)],
                 loa=-1,
                 name='',
                 rotation=(0, 0, 1, 0),
                 scale=(1, 1, 1),
                 scaleOrientation=(0, 0, 1, 0),
                 skeletalConfiguration='BASIC',
                 translation=(0, 0, 0),
                 version='2.0',
                 visible=True,
                 skinBindingCoords=None,
                 skinBindingNormals=None,
                 skinCoord=None,
                 skinNormal=None,
                 joints=list(),
                 motions=list(),
                 segments=list(),
                 sites=list(),
                 skeleton=list(),
                 skin=list(),
                 viewpoints=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode HAnimHumanoid __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.center = center
        self.description = description
        self.displayBBox = displayBBox
        self.info = info
        self.jointBindingPositions = jointBindingPositions
        self.jointBindingRotations = jointBindingRotations
        self.jointBindingScales = jointBindingScales
        self.loa = loa
        self.name = name
        self.rotation = rotation
        self.scale = scale
        self.scaleOrientation = scaleOrientation
        self.skeletalConfiguration = skeletalConfiguration
        self.translation = translation
        self.version = version
        self.visible = visible
        self.skinBindingCoords = skinBindingCoords
        self.skinBindingNormals = skinBindingNormals
        self.skinCoord = skinCoord
        self.skinNormal = skinNormal
        self.joints = joints
        self.motions = motions
        self.segments = segments
        self.sites = sites
        self.skeleton = skeleton
        self.skin = skin
        self.viewpoints = viewpoints
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def center(self):
        """Translation offset from origin of local coordinate system."""
        return self.__center
    @center.setter
    def center(self, center=None):
        if  center is None:
            center = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(center)
        self.__center = center
    @property # getter - - - - - - - - - -
    def description(self):
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def info(self):
        """Contains metadata keyword=value pairs, where approved keyword terms are humanoidVersion authorName authorEmail copyright creationDate usageRestrictions age gender height and weight."""
        return self.__info
    @info.setter
    def info(self, info=None):
        if  info is None:
            info = MFString.DEFAULT_VALUE
        assertValidMFString(info)
        self.__info = info
    @property # getter - - - - - - - - - -
    def jointBindingPositions(self):
        """Specifies an array of position values for each HAnimJoint node in the joints field, in order, corresponding to each binding pose."""
        return self.__jointBindingPositions
    @jointBindingPositions.setter
    def jointBindingPositions(self, jointBindingPositions=None):
        if  jointBindingPositions is None:
            jointBindingPositions = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(jointBindingPositions)
        self.__jointBindingPositions = jointBindingPositions
    @property # getter - - - - - - - - - -
    def jointBindingRotations(self):
        """Specifies an array of rotation values for each HAnimJoint node in the joints field, in order, corresponding to each binding pose."""
        return self.__jointBindingRotations
    @jointBindingRotations.setter
    def jointBindingRotations(self, jointBindingRotations=None):
        if  jointBindingRotations is None:
            jointBindingRotations = MFRotation.DEFAULT_VALUE
        assertValidMFRotation(jointBindingRotations)
        self.__jointBindingRotations = jointBindingRotations
    @property # getter - - - - - - - - - -
    def jointBindingScales(self):
        """Specifies an array of scale values for each HAnimJoint node in the joints field, in order, corresponding to each binding pose."""
        return self.__jointBindingScales
    @jointBindingScales.setter
    def jointBindingScales(self, jointBindingScales=None):
        if  jointBindingScales is None:
            jointBindingScales = MFVec3f.DEFAULT_VALUE
        assertValidMFVec3f(jointBindingScales)
        self.__jointBindingScales = jointBindingScales
    @property # getter - - - - - - - - - -
    def loa(self):
        """[-1,4] Level Of Articulation 0."""
        return self.__loa
    @loa.setter
    def loa(self, loa=None):
        if  loa is None:
            loa = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(loa)
        assertGreaterThanEquals('loa', loa, -1)
        assertLessThanEquals('loa', loa, 4)
        self.__loa = loa
    @property # getter - - - - - - - - - -
    def name(self):
        """Unique name attribute must be defined so that each HAnimHumanoid node in a scene can be identified at run time for animation purposes."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation=None):
        if  rotation is None:
            rotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def scale(self):
        """Non-uniform x-y-z scale of child coordinate system, adjusted by center and scaleOrientation."""
        return self.__scale
    @scale.setter
    def scale(self, scale=None):
        if  scale is None:
            scale = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(scale)
        assertPositive('scale', scale)
        self.__scale = scale
    @property # getter - - - - - - - - - -
    def scaleOrientation(self):
        """Preliminary rotation of coordinate system before scaling (to allow scaling around arbitrary orientations)."""
        return self.__scaleOrientation
    @scaleOrientation.setter
    def scaleOrientation(self, scaleOrientation=None):
        if  scaleOrientation is None:
            scaleOrientation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(scaleOrientation)
        self.__scaleOrientation = scaleOrientation
    @property # getter - - - - - - - - - -
    def skeletalConfiguration(self):
        """Models sharing a common skeletal configuration can share animations and binding poses."""
        return self.__skeletalConfiguration
    @skeletalConfiguration.setter
    def skeletalConfiguration(self, skeletalConfiguration=None):
        if  skeletalConfiguration is None:
            skeletalConfiguration = SFString.DEFAULT_VALUE
        assertValidSFString(skeletalConfiguration)
        self.__skeletalConfiguration = skeletalConfiguration
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation=None):
        if  translation is None:
            translation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def version(self):
        """HAnimHumanoid version, where value is 2."""
        return self.__version
    @version.setter
    def version(self, version=None):
        if  version is None:
            version = SFString.DEFAULT_VALUE
        assertValidSFString(version)
        self.__version = version
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def skinBindingCoords(self):
        """[X3DCoordinateNode] Array of Coordinate nodes to handle non-default source pose so that both skin and skeleton can be in same binding pose."""
        return self.__skinBindingCoords
    @skinBindingCoords.setter
    def skinBindingCoords(self, skinBindingCoords=None):
        if  skinBindingCoords is None:
            skinBindingCoords = SFNode.DEFAULT_VALUE
        assertValidSFNode(skinBindingCoords)
        if not isinstance(skinBindingCoords, object):
            # print(flush=True)
            raise X3DTypeError(str(skinBindingCoords) + ' does not have a valid node type object')
        self.__skinBindingCoords = skinBindingCoords
    @property # getter - - - - - - - - - -
    def skinBindingNormals(self):
        """[X3DNormalNode] Array of Normal nodes to handle non-default source pose so that both skin and skeleton can be in same binding pose."""
        return self.__skinBindingNormals
    @skinBindingNormals.setter
    def skinBindingNormals(self, skinBindingNormals=None):
        if  skinBindingNormals is None:
            skinBindingNormals = SFNode.DEFAULT_VALUE
        assertValidSFNode(skinBindingNormals)
        if not isinstance(skinBindingNormals, object):
            # print(flush=True)
            raise X3DTypeError(str(skinBindingNormals) + ' does not have a valid node type object')
        self.__skinBindingNormals = skinBindingNormals
    @property # getter - - - - - - - - - -
    def skinCoord(self):
        """[X3DCoordinateNode] Coordinate node utilized by indexed mesh definitions for skin."""
        return self.__skinCoord
    @skinCoord.setter
    def skinCoord(self, skinCoord=None):
        if  skinCoord is None:
            skinCoord = SFNode.DEFAULT_VALUE
        assertValidSFNode(skinCoord)
        if not isinstance(skinCoord, object):
            # print(flush=True)
            raise X3DTypeError(str(skinCoord) + ' does not have a valid node type object')
        self.__skinCoord = skinCoord
    @property # getter - - - - - - - - - -
    def skinNormal(self):
        """[X3DNormalNode] Single Normal node utilized by indexed mesh definitions for skin."""
        return self.__skinNormal
    @skinNormal.setter
    def skinNormal(self, skinNormal=None):
        if  skinNormal is None:
            skinNormal = SFNode.DEFAULT_VALUE
        assertValidSFNode(skinNormal)
        if not isinstance(skinNormal, object):
            # print(flush=True)
            raise X3DTypeError(str(skinNormal) + ' does not have a valid node type object')
        self.__skinNormal = skinNormal
    @property # getter - - - - - - - - - -
    def joints(self):
        """[HAnimJoint] The joints field contains a list of USE references for all HAnimJoint node instances found within the preceding skeleton hierarchy."""
        return self.__joints
    @joints.setter
    def joints(self, joints=None):
        if  joints is None:
            joints = MFNode.DEFAULT_VALUE
        assertValidMFNode(joints)
        self.__joints = joints
    @property # getter - - - - - - - - - -
    def motions(self):
        """[HAnimMotion] Contains any HAnimMotion nodes that can animate the HAnimHumanoid."""
        return self.__motions
    @motions.setter
    def motions(self, motions=None):
        if  motions is None:
            motions = MFNode.DEFAULT_VALUE
        assertValidMFNode(motions)
        self.__motions = motions
    @property # getter - - - - - - - - - -
    def segments(self):
        """[HAnimSegment] The segments field contains a list of USE references for all HAnimSegment node instances found within the preceding skeleton hierarchy."""
        return self.__segments
    @segments.setter
    def segments(self, segments=None):
        if  segments is None:
            segments = MFNode.DEFAULT_VALUE
        assertValidMFNode(segments)
        self.__segments = segments
    @property # getter - - - - - - - - - -
    def sites(self):
        """[HAnimSite] sites field contains a list of USE references for all HAnimSite node instances found within the preceding skeleton hierarchy."""
        return self.__sites
    @sites.setter
    def sites(self, sites=None):
        if  sites is None:
            sites = MFNode.DEFAULT_VALUE
        assertValidMFNode(sites)
        self.__sites = sites
    @property # getter - - - - - - - - - -
    def skeleton(self):
        """[HAnimJoint|HAnimSite] List of top-level HAnimJoint and HAnimSite nodes that create the skeleton model."""
        return self.__skeleton
    @skeleton.setter
    def skeleton(self, skeleton=None):
        if  skeleton is None:
            skeleton = MFNode.DEFAULT_VALUE
        assertValidMFNode(skeleton)
        self.__skeleton = skeleton
    @property # getter - - - - - - - - - -
    def skin(self):
        """[IndexedFaceSet|Group|Transform|Shape] List of one or more indexed mesh definitions (such as IndexedFaceSet) that utilize skinCoord point and skinNormal normal data."""
        return self.__skin
    @skin.setter
    def skin(self, skin=None):
        if  skin is None:
            skin = MFNode.DEFAULT_VALUE
        assertValidMFNode(skin)
        self.__skin = skin
    @property # getter - - - - - - - - - -
    def viewpoints(self):
        """[HAnimSite] List of HAnimSite nodes containing Viewpoint nodes that appear in the skeleton model, usually as USE node references."""
        return self.__viewpoints
    @viewpoints.setter
    def viewpoints(self, viewpoints=None):
        if  viewpoints is None:
            viewpoints = MFNode.DEFAULT_VALUE
        assertValidMFNode(viewpoints)
        self.__viewpoints = viewpoints
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.joints) or bool(self.motions) or bool(self.segments) or bool(self.sites) or bool(self.skeleton) or bool(self.skin) or bool(self.viewpoints) or bool(self.IS) or bool(self.metadata) or bool(self.skinBindingCoords) or bool(self.skinBindingNormals) or bool(self.skinCoord) or bool(self.skinNormal)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function HAnimHumanoid.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<HAnimHumanoid'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.center != (0, 0, 0):
            result += " center='" + str(self.center) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.info != list():
            result += " info='" + self.info + "'"
        if self.jointBindingPositions != [(0, 0, 0)]:
            result += " jointBindingPositions='" + str(self.jointBindingPositions) + "'"
        if self.jointBindingRotations != [(0, 0, 1, 0)]:
            result += " jointBindingRotations='" + str(self.jointBindingRotations) + "'"
        if self.jointBindingScales != [(0, 0, 0)]:
            result += " jointBindingScales='" + str(self.jointBindingScales) + "'"
        if self.loa != -1:
            result += " loa='" + str(self.loa) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + str(self.rotation) + "'"
        if self.scale != (1, 1, 1):
            result += " scale='" + str(self.scale) + "'"
        if self.scaleOrientation != (0, 0, 1, 0):
            result += " scaleOrientation='" + str(self.scaleOrientation) + "'"
        if self.skeletalConfiguration != 'BASIC':
            result += " skeletalConfiguration='" + self.skeletalConfiguration + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + str(self.translation) + "'"
        if self.version != '2.0':
            result += " version='" + self.version + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.skinBindingCoords: # walk each child node, if any
                for each in self.skinBindingCoords:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.skinBindingNormals: # walk each child node, if any
                for each in self.skinBindingNormals:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.skinCoord: # walk each child node, if any
                for each in self.skinCoord:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.skinNormal: # walk each child node, if any
                for each in self.skinNormal:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.joints: # walk each child node, if any
                for each in self.joints:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.motions: # walk each child node, if any
                for each in self.motions:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.segments: # walk each child node, if any
                for each in self.segments:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.sites: # walk each child node, if any
                for each in self.sites:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.skeleton: # walk each child node, if any
                for each in self.skeleton:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.skin: # walk each child node, if any
                for each in self.skin:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.viewpoints: # walk each child node, if any
                for each in self.viewpoints:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</HAnimHumanoid>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class HAnimJoint(_X3DChildNode, _X3DBoundedObject):
    """
    HAnimJoint node can represent each joint in a body.
    """
    NAME = 'HAnimJoint'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#HAnimJoint'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('center', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimJoint'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'HAnimJoint'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('limitOrientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'HAnimJoint'),
        ('llimit', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimJoint'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'HAnimJoint'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'HAnimJoint'),
        ('scale', (1, 1, 1), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimJoint'),
        ('scaleOrientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'HAnimJoint'),
        ('skinCoordIndex', list(), FieldType.MFInt32, AccessType.inputOutput, 'HAnimJoint'),
        ('skinCoordWeight', list(), FieldType.MFFloat, AccessType.inputOutput, 'HAnimJoint'),
        ('stiffness', (1, 1, 1), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimJoint'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimJoint'),
        ('ulimit', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimJoint'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimJoint'),
        ('displacers', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimJoint'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 center=(0, 0, 0),
                 description='',
                 displayBBox=False,
                 limitOrientation=(0, 0, 1, 0),
                 llimit=(0, 0, 0),
                 name='',
                 rotation=(0, 0, 1, 0),
                 scale=(1, 1, 1),
                 scaleOrientation=(0, 0, 1, 0),
                 skinCoordIndex=list(),
                 skinCoordWeight=list(),
                 stiffness=(1, 1, 1),
                 translation=(0, 0, 0),
                 ulimit=(0, 0, 0),
                 visible=True,
                 children=list(),
                 displacers=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode HAnimJoint __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.center = center
        self.description = description
        self.displayBBox = displayBBox
        self.limitOrientation = limitOrientation
        self.llimit = llimit
        self.name = name
        self.rotation = rotation
        self.scale = scale
        self.scaleOrientation = scaleOrientation
        self.skinCoordIndex = skinCoordIndex
        self.skinCoordWeight = skinCoordWeight
        self.stiffness = stiffness
        self.translation = translation
        self.ulimit = ulimit
        self.visible = visible
        self.children = children
        self.displacers = displacers
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def center(self):
        """Translation offset from origin of local coordinate system."""
        return self.__center
    @center.setter
    def center(self, center=None):
        if  center is None:
            center = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(center)
        self.__center = center
    @property # getter - - - - - - - - - -
    def description(self):
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def limitOrientation(self):
        """Orientation of upper/lower rotation limits, relative to HAnimJoint center."""
        return self.__limitOrientation
    @limitOrientation.setter
    def limitOrientation(self, limitOrientation=None):
        if  limitOrientation is None:
            limitOrientation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(limitOrientation)
        self.__limitOrientation = limitOrientation
    @property # getter - - - - - - - - - -
    def llimit(self):
        """Lower limit for minimum joint rotation in radians."""
        return self.__llimit
    @llimit.setter
    def llimit(self, llimit=None):
        if  llimit is None:
            llimit = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(llimit)
        self.__llimit = llimit
    @property # getter - - - - - - - - - -
    def name(self):
        """Unique name attribute must be defined so that HAnimJoint node can be identified at run time for animation purposes."""
        return self.__name
    @name.setter
    def name(self, name=None):
        if  name is None:
            name = SFString.DEFAULT_VALUE
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation=None):
        if  rotation is None:
            rotation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def scale(self):
        """Non-uniform x-y-z scale of child coordinate system, adjusted by center and scaleOrientation."""
        return self.__scale
    @scale.setter
    def scale(self, scale=None):
        if  scale is None:
            scale = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(scale)
        assertPositive('scale', scale)
        self.__scale = scale
    @property # getter - - - - - - - - - -
    def scaleOrientation(self):
        """Preliminary rotation of coordinate system before scaling (to allow scaling around arbitrary orientations)."""
        return self.__scaleOrientation
    @scaleOrientation.setter
    def scaleOrientation(self, scaleOrientation=None):
        if  scaleOrientation is None:
            scaleOrientation = SFRotation.DEFAULT_VALUE
        assertValidSFRotation(scaleOrientation)
        self.__scaleOrientation = scaleOrientation
    @property # getter - - - - - - - - - -
    def skinCoordIndex(self):
        """[0,+infinity) Coordinate index values referencing which vertices are influenced by the HAnimJoint."""
        return self.__skinCoordIndex
    @skinCoordIndex.setter
    def skinCoordIndex(self, skinCoordIndex=None):
        if  skinCoordIndex is None:
            skinCoordIndex = MFInt32.DEFAULT_VALUE
        assertValidMFInt32(skinCoordIndex)
        assertNonNegative('skinCoordIndex', skinCoordIndex)
        self.__skinCoordIndex = skinCoordIndex
    @property # getter - - - - - - - - - -
    def skinCoordWeight(self):
        """Weight deformation values for the corresponding values in the skinCoordIndex field."""
        return self.__skinCoordWeight
    @skinCoordWeight.setter
    def skinCoordWeight(self, skinCoordWeight=None):
        if  skinCoordWeight is None:
            skinCoordWeight = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(skinCoordWeight)
        self.__skinCoordWeight = skinCoordWeight
    @property # getter - - - - - - - - - -
    def stiffness(self):
        """[0,1] A scale factor of (1 - stiffness) is applied around the corresponding axis (X, Y, or Z for entries 0, 1 and 2 of the stiffness field)."""
        return self.__stiffness
    @stiffness.setter
    def stiffness(self, stiffness=None):
        if  stiffness is None:
            stiffness = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(stiffness)
        assertZeroToOne('stiffness', stiffness)
        self.__stiffness = stiffness
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation=None):
        if  translation is None:
            translation = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def ulimit(self):
        """Upper limit for maximum joint rotation in radians."""
        return self.__ulimit
    @ulimit.setter
    def ulimit(self, ulimit=None):
        if  ulimit is None:
            ulimit = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(ulimit)
        self.__ulimit = ulimit
    @property # getter - - - - - - - - - -
    def visible(self):
        return self.__visible
    @visible.setter
    def visible(self, visible=None):
        if  visible is None:
            visible = SFBool.DEFAULT_VALUE
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[HAnimJoint|HAnimSegment|HAnimSite] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children=None):
        if  children is None:
            children = MFNode.DEFAULT_VALUE
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def displacers(self):
        """[HAnimDisplacer] the displacers field stores HAnimDisplacer objects for a particular HAnimJoint object."""
        return self.__displacers
    @displacers.setter
    def displacers(self, displacers=None):
        if  displacers is None:
            displacers = MFNode.DEFAULT_VALUE
        assertValidMFNode(displacers)
        self.__displacers = displacers
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.children) or bool(self.displacers) or bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function HAnimJoint.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<HAnimJoint'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + str(self.bboxCenter) + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + str(self.bboxSize) + "'"
        if self.center != (0, 0, 0):
            result += " center='" + str(self.center) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + str(self.displayBBox) + "'"
        if self.limitOrientation != (0, 0, 1, 0):
            result += " limitOrientation='" + str(self.limitOrientation) + "'"
        if self.llimit != (0, 0, 0):
            result += " llimit='" + str(self.llimit) + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + str(self.rotation) + "'"
        if self.scale != (1, 1, 1):
            result += " scale='" + str(self.scale) + "'"
        if self.scaleOrientation != (0, 0, 1, 0):
            result += " scaleOrientation='" + str(self.scaleOrientation) + "'"
        if self.skinCoordIndex != list():
            result += " skinCoordIndex='" + str(self.skinCoordIndex) + "'"
        if self.skinCoordWeight != list():
            result += " skinCoordWeight='" + str(self.skinCoordWeight) + "'"
        if self.stiffness != (1, 1, 1):
            result += " stiffness='" + str(self.stiffness) + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + str(self.translation) + "'"
        if self.ulimit != (0, 0, 0):
            result += " ulimit='" + str(self.ulimit) + "'"
        if self.visible != True:
            result += " visible='" + str(self.visible) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.children: # walk each child node, if any
                for each in self.children:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.displacers: # walk each child node, if any
                for each in self.displacers:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</HAnimJoint>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class HAnimMotion(_X3DChildNode):
    """
    An HAnimMotion node supports discrete frame-by-frame playback for H-Anim motion data animation.
    """
    NAME = 'HAnimMotion'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#HAnimMotion'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('channels', list(), FieldType.MFString, AccessType.inputOutput, 'HAnimMotion'),
        ('channelsEnabled', list(), FieldType.MFBool, AccessType.inputOutput, 'HAnimMotion'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'HAnimMotion'),
        ('enabled', False, FieldType.SFBool, AccessType.inputOutput, 'HAnimMotion'),
        ('endFrame', -1, FieldType.SFInt32, AccessType.inputOutput, 'HAnimMotion'),
        ('frameDuration', 0.1, FieldType.SFTime, AccessType.inputOutput, 'HAnimMotion'),
        ('frameIncrement', 1, FieldType.SFInt32, AccessType.inputOutput, 'HAnimMotion'),
        ('frameIndex', 0, FieldType.SFInt32, AccessType.inputOutput, 'HAnimMotion'),
        ('joints', list(), FieldType.MFString, AccessType.inputOutput, 'HAnimMotion'),
        ('loa', -1, FieldType.SFInt32, AccessType.inputOutput, 'HAnimMotion'),
        ('loop', False, FieldType.SFBool, AccessType.inputOutput, 'HAnimMotion'),
        ('startFrame', 0, FieldType.SFInt32, AccessType.inputOutput, 'HAnimMotion'),
        ('values', list(), FieldType.MFFloat, AccessType.inputOutput, 'HAnimMotion'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 channels=list(),
                 channelsEnabled=list(),
                 description='',
                 enabled=False,
                 endFrame=-1,
                 frameDuration=0.1,
                 frameIncrement=1,
                 frameIndex=0,
                 joints=list(),
                 loa=-1,
                 loop=False,
                 startFrame=0,
                 values=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode HAnimMotion __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.channels = channels
        self.channelsEnabled = channelsEnabled
        self.description = description
        self.enabled = enabled
        self.endFrame = endFrame
        self.frameDuration = frameDuration
        self.frameIncrement = frameIncrement
        self.frameIndex = frameIndex
        self.joints = joints
        self.loa = loa
        self.loop = loop
        self.startFrame = startFrame
        self.values = values
    @property # getter - - - - - - - - - -
    def channels(self):
        """list of number of channels for transformation, followed by transformation type of each channel of data."""
        return self.__channels
    @channels.setter
    def channels(self, channels=None):
        if  channels is None:
            channels = MFString.DEFAULT_VALUE
        assertValidMFString(channels)
        self.__channels = channels
    @property # getter - - - - - - - - - -
    def channelsEnabled(self):
        """boolean values for each channels indicating whether enabled."""
        return self.__channelsEnabled
    @channelsEnabled.setter
    def channelsEnabled(self, channelsEnabled=None):
        if  channelsEnabled is None:
            channelsEnabled = MFBool.DEFAULT_VALUE
        assertValidMFBool(channelsEnabled)
        self.__channelsEnabled = channelsEnabled
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled=None):
        if  enabled is None:
            enabled = SFBool.DEFAULT_VALUE
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def endFrame(self):
        return self.__endFrame
    @endFrame.setter
    def endFrame(self, endFrame=None):
        if  endFrame is None:
            endFrame = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(endFrame)
        assertGreaterThanEquals('endFrame', endFrame, -1)
        self.__endFrame = endFrame
    @property # getter - - - - - - - - - -
    def frameDuration(self):
        """(0,+infinity) frameDuration specifies the duration of each frame in seconds."""
        return self.__frameDuration
    @frameDuration.setter
    def frameDuration(self, frameDuration=None):
        if  frameDuration is None:
            frameDuration = SFTime.DEFAULT_VALUE
        assertValidSFTime(frameDuration)
        assertPositive('frameDuration', frameDuration)
        self.__frameDuration = frameDuration
    @property # getter - - - - - - - - - -
    def frameIncrement(self):
        """[-infinity,+infinity) frameIncrement field controls whether playback direction is forwards or backwards, and also whether frames are skipped (e."""
        return self.__frameIncrement
    @frameIncrement.setter
    def frameIncrement(self, frameIncrement=None):
        if  frameIncrement is None:
            frameIncrement = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(frameIncrement)
        self.__frameIncrement = frameIncrement
    @property # getter - - - - - - - - - -
    def frameIndex(self):
        """[0,+infinity) frameIndex indicates index of current frame."""
        return self.__frameIndex
    @frameIndex.setter
    def frameIndex(self, frameIndex=None):
        if  frameIndex is None:
            frameIndex = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(frameIndex)
        assertNonNegative('frameIndex', frameIndex)
        self.__frameIndex = frameIndex
    @property # getter - - - - - - - - - -
    def joints(self):
        """joints field lists names of joints that raw motion data is to be applied to."""
        return self.__joints
    @joints.setter
    def joints(self, joints=None):
        if  joints is None:
            joints = MFString.DEFAULT_VALUE
        assertValidMFString(joints)
        self.__joints = joints
    @property # getter - - - - - - - - - -
    def loa(self):
        """[-1,4] Level Of Articulation 0."""
        return self.__loa
    @loa.setter
    def loa(self, loa=None):
        if  loa is None:
            loa = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(loa)
        assertGreaterThanEquals('loa', loa, -1)
        assertLessThanEquals('loa', loa, 4)
        self.__loa = loa
    @property # getter - - - - - - - - - -
    def loop(self):
        """Repeat indefinitely when loop=true, repeat only once when loop=false."""
        return self.__loop
    @loop.setter
    def loop(self, loop=None):
        if  loop is None:
            loop = SFBool.DEFAULT_VALUE
        assertValidSFBool(loop)
        self.__loop = loop
    @property # getter - - - - - - - - - -
    def startFrame(self):
        return self.__startFrame
    @startFrame.setter
    def startFrame(self, startFrame=None):
        if  startFrame is None:
            startFrame = SFInt32.DEFAULT_VALUE
        assertValidSFInt32(startFrame)
        assertNonNegative('startFrame', startFrame)
        self.__startFrame = startFrame
    @property # getter - - - - - - - - - -
    def values(self):
        """values field contains all transformation values, ordered first by frame, then by joint, and then by transformation Sets of floats in the values array matching the order listed in joints and channels fields."""
        return self.__values
    @values.setter
    def values(self, values=None):
        if  values is None:
            values = MFFloat.DEFAULT_VALUE
        assertValidMFFloat(values)
        self.__values = values
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has a child node, statement or comment '''
        return bool(self.IS) or bool(self.metadata)
    # output function - - - - - - - - - -
    def toXML(self, indentLevel=0):
        """ Provide Canonical X3D output serialization using XML encoding. """
        indent = '  ' * indentLevel
        result = ''
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function HAnimMotion.toXML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<HAnimMotion'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.channels != list():
            result += " channels='" + self.channels + "'"
        if self.channelsEnabled != list():
            result += " channelsEnabled='" + str(self.channelsEnabled) + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.enabled != False:
            result += " enabled='" + str(self.enabled) + "'"
        if self.endFrame != -1:
            result += " endFrame='" + str(self.endFrame) + "'"
        if self.frameDuration != 0.1:
            result += " frameDuration='" + str(self.frameDuration) + "'"
        if self.frameIncrement != 1:
            result += " frameIncrement='" + str(self.frameIncrement) + "'"
        if self.frameIndex != 0:
            result += " frameIndex='" + str(self.frameIndex) + "'"
        if self.joints != list():
            result += " joints='" + self.joints + "'"
        if self.loa != -1:
            result += " loa='" + str(self.loa) + "'"
        if self.loop != False:
            result += " loop='" + str(self.loop) + "'"
        if self.startFrame != 0:
            result += " startFrame='" + str(self.startFrame) + "'"
        if self.values != list():
            result += " values='" + str(self.values) + "'"
        if not self.hasChild():
            result += '/>' + '\n'
        else:
            result += '>' + '\n'
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.IS: # walk each child node, if any
                for each in self.IS:
                    result += each.toXML(indentLevel=indentLevel+1)
            ## result += indent + '  ' + 'TODO iterate over each child element' + '\n'
            if self.metadata: # walk each child node, if any
                for each in self.metadata:
                    result += each.toXML(indentLevel=indentLevel+1)
            result += indent + '</HAnimMotion>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result

class HAnimSegment(_X3DGroupingNode):
    """
    HAnimSegment node contains Shape geometry for each body segment.
    """
    NAME = 'HAnimSegment'
    SPECIFICATION_URL = ''
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#HAnimSegment'
    FIELD_DECLARATIONS = [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('centerOfMass', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'HAnimSegment'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'HAnimSegment'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('mass', 0, FieldType.SFFloat, AccessType.inputOutput, 'HAnimSegment'),
        ('momentsOfInertia', [0, 0, 0, 0, 0, 0, 0, 0, 0], FieldType.MFFloat, AccessType.inputOutput, 'HAnimSegment'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'HAnimSegment'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('coord', None, FieldType.SFNode, AccessType.inputOutput, 'HAnimSegment'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('displacers', list(), FieldType.MFNode, AccessType.inputOutput, 'HAnimSegment'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 centerOfMass=(0, 0, 0),
                 description='',
                 displayBBox=False,
                 mass=0,
                 momentsOfInertia=[0, 0, 0, 0, 0, 0, 0, 0, 0],
                 name='',
                 visible=True,
                 coord=None,
                 children=list(),
                 displacers=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode HAnimSegment __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.centerOfMass = centerOfMass
        self.description = description
        self.displayBBox = displayBBox
        self.mass = mass
        self.momentsOfInertia = momentsOfInertia
        self.name = name
        self.visible = visible
        self.coord = coord
        self.children = children
        self.displacers = displacers
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter=None):
        if  bboxCenter is None:
            bboxCenter = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize=None):
        if  bboxSize is None:
            bboxSize = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def centerOfMass(self):
        """Location within segment of center of mass."""
        return self.__centerOfMass
    @centerOfMass.setter
    def centerOfMass(self, centerOfMass=None):
        if  centerOfMass is None:
            centerOfMass = SFVec3f.DEFAULT_VALUE
        assertValidSFVec3f(centerOfMass)
        self.__centerOfMass = centerOfMass
    @property # getter - - - - - - - - - -
    def description(self):
        return self.__description
    @description.setter
    def description(self, description=None):
        if  description is None:
            description = SFString.DEFAULT_VALUE
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox=None):
        if  displayBBox is None:
            displayBBox = SFBool.DEFAULT_VALUE
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def mass(self):
        """Total mass of the segment, 0 if not available, defined in mass base units (default is kilograms)."""
        return self.__mass
    @mass.setter
    def mass(self, mass=None):
        if  mass is None:
            mass = SFFloat.DEFAULT_VALUE
        assertValidSFFloat(mass)
        assertNonNegative('mass', mass)
        self.__mass = mass
    @property # getter - - - - - - - - - -
    def momentsOfInertia(self):
        """3x3 moments of inertia matrix."""
        return self.__momentsOfInerti