"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /**
     * (experimental) Manages connections for the cluster.
     *
     * @experimental
     */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * (experimental) The level of monitoring for the MSK cluster.
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 * @experimental
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel = exports.ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = {}));
/**
 * (experimental) Indicates the encryption setting for data in transit between clients and brokers.
 *
 * @experimental
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    ClientBrokerEncryption["TLS"] = "TLS";
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption = exports.ClientBrokerEncryption || (exports.ClientBrokerEncryption = {}));
/**
 * (experimental) Configuration properties for client authentication.
 *
 * @experimental
 */
class ClientAuthentication {
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
    /**
     * (experimental) SASL authentication.
     *
     * @experimental
     */
    static sasl(props) {
        return new ClientAuthentication(props, undefined);
    }
    /**
     * (experimental) TLS authentication.
     *
     * @experimental
     */
    static tls(props) {
        return new ClientAuthentication(undefined, props);
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "@aws-cdk/aws-msk-alpha.ClientAuthentication", version: "2.0.0-alpha.2" };
/**
 * (experimental) Create a MSK Cluster.
 *
 * @experimental
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (((_e = (_d = props.clientAuthentication) === null || _d === void 0 ? void 0 : _d.saslProps) === null || _e === void 0 ? void 0 : _e.iam) && ((_g = (_f = props.clientAuthentication) === null || _f === void 0 ? void 0 : _f.saslProps) === null || _g === void 0 ? void 0 : _g.scram)) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (((_h = props.encryptionInTransit) === null || _h === void 0 ? void 0 : _h.clientBroker) ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (((_j = props.encryptionInTransit) === null || _j === void 0 ? void 0 : _j.clientBroker) ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (((_l = (_k = props.clientAuthentication) === null || _k === void 0 ? void 0 : _k.saslProps) === null || _l === void 0 ? void 0 : _l.scram) || ((_o = (_m = props.clientAuthentication) === null || _m === void 0 ? void 0 : _m.saslProps) === null || _o === void 0 ? void 0 : _o.iam))) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = (_q = (_p = props.ebsStorageInfo) === null || _p === void 0 ? void 0 : _p.volumeSize) !== null && _q !== void 0 ? _q : 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        const encryptionAtRest = ((_r = props.ebsStorageInfo) === null || _r === void 0 ? void 0 : _r.encryptionKey) ? {
            dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
        }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: (_t = (_s = props.encryptionInTransit) === null || _s === void 0 ? void 0 : _s.clientBroker) !== null && _t !== void 0 ? _t : ClientBrokerEncryption.TLS,
            inCluster: (_v = (_u = props.encryptionInTransit) === null || _u === void 0 ? void 0 : _u.enableInCluster) !== null && _v !== void 0 ? _v : true,
        };
        const openMonitoring = ((_w = props.monitoring) === null || _w === void 0 ? void 0 : _w.enablePrometheusJmxExporter) || ((_x = props.monitoring) === null || _x === void 0 ? void 0 : _x.enablePrometheusNodeExporter)
            ? {
                prometheus: {
                    jmxExporter: ((_y = props.monitoring) === null || _y === void 0 ? void 0 : _y.enablePrometheusJmxExporter) ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: ((_z = props.monitoring) === null || _z === void 0 ? void 0 : _z.enablePrometheusNodeExporter) ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: ((_0 = props.logging) === null || _0 === void 0 ? void 0 : _0.cloudwatchLogGroup) !== undefined,
                    logGroup: (_2 = (_1 = props.logging) === null || _1 === void 0 ? void 0 : _1.cloudwatchLogGroup) === null || _2 === void 0 ? void 0 : _2.logGroupName,
                },
                firehose: {
                    enabled: ((_3 = props.logging) === null || _3 === void 0 ? void 0 : _3.firehoseDeliveryStreamName) !==
                        undefined,
                    deliveryStream: (_4 = props.logging) === null || _4 === void 0 ? void 0 : _4.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: ((_6 = (_5 = props.logging) === null || _5 === void 0 ? void 0 : _5.s3) === null || _6 === void 0 ? void 0 : _6.bucket) !== undefined,
                    bucket: (_8 = (_7 = props.logging) === null || _7 === void 0 ? void 0 : _7.s3) === null || _8 === void 0 ? void 0 : _8.bucket.bucketName,
                    prefix: (_10 = (_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.s3) === null || _10 === void 0 ? void 0 : _10.prefix,
                },
            },
        };
        if (((_12 = (_11 = props.clientAuthentication) === null || _11 === void 0 ? void 0 : _11.saslProps) === null || _12 === void 0 ? void 0 : _12.scram) &&
            ((_14 = (_13 = props.clientAuthentication) === null || _13 === void 0 ? void 0 : _13.saslProps) === null || _14 === void 0 ? void 0 : _14.key) === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if ((_16 = (_15 = props.clientAuthentication) === null || _15 === void 0 ? void 0 : _15.saslProps) === null || _16 === void 0 ? void 0 : _16.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
        }
        else if ((_18 = (_17 = props.clientAuthentication) === null || _17 === void 0 ? void 0 : _17.saslProps) === null || _18 === void 0 ? void 0 : _18.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (((_20 = (_19 = props.clientAuthentication) === null || _19 === void 0 ? void 0 : _19.tlsProps) === null || _20 === void 0 ? void 0 : _20.certificateAuthorities) !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: (_22 = (_21 = props.clientAuthentication) === null || _21 === void 0 ? void 0 : _21.tlsProps) === null || _22 === void 0 ? void 0 : _22.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new aws_msk_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: (_23 = props.monitoring) === null || _23 === void 0 ? void 0 : _23.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    /**
     * (experimental) Reference an existing cluster, defined outside of the CDK code, by name.
     *
     * @experimental
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * (experimental) Get the ZooKeeper Connection string.
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     * @experimental
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * (experimental) Get the ZooKeeper Connection string for a TLS enabled cluster.
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     * @experimental
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * (experimental) Get the list of brokers that a client application can use to bootstrap.
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     * @experimental
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * (experimental) Get the list of brokers that a TLS authenticated client application can use to bootstrap.
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     * @experimental
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * (experimental) Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap.
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     * @experimental
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * (experimental) A list of usersnames to register with the cluster.
     *
     * The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster.
     * @experimental
     */
    addUser(...usernames) {
        var _c;
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${uniqueid_1.addressOf(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [(_c = this.saslScramAuthenticationKey) === null || _c === void 0 ? void 0 : _c.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "@aws-cdk/aws-msk-alpha.Cluster", version: "2.0.0-alpha.2" };
//# sourceMappingURL=data:application/json;base64,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