"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationManager = void 0;
const cdk = require("@aws-cdk/core");
const servicecatalog_generated_1 = require("../servicecatalog.generated");
const util_1 = require("./util");
const validation_1 = require("./validation");
class AssociationManager {
    static associateProductWithPortfolio(portfolio, product, options) {
        validation_1.InputValidator.validateLength(this.prettyPrintAssociation(portfolio, product), 'description', 0, 2000, options === null || options === void 0 ? void 0 : options.description);
        const associationKey = util_1.hashValues(portfolio.node.addr, product.node.addr, product.stack.node.addr);
        const constructId = `PortfolioProductAssociation${associationKey}`;
        const existingAssociation = portfolio.node.tryFindChild(constructId);
        const cfnAssociation = existingAssociation
            ? existingAssociation
            : new servicecatalog_generated_1.CfnPortfolioProductAssociation(portfolio, constructId, {
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
            });
        return {
            associationKey: associationKey,
            cfnPortfolioProductAssociation: cfnAssociation,
        };
    }
    static constrainTagUpdates(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `ResourceUpdateConstraint${association.associationKey}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnResourceUpdateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                tagUpdateOnProvisionedProduct: options.allow === false ? 'NOT_ALLOWED' : 'ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot have multiple tag update constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static notifyOnStackEvents(portfolio, product, topic, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchNotificationConstraint${util_1.hashValues(topic.node.addr, topic.stack.node.addr, association.associationKey)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchNotificationConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                notificationArns: [topic.topicArn],
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Topic ${topic.node.path} is already subscribed to association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static constrainCloudFormationParameters(portfolio, product, options) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        const constructId = `LaunchTemplateConstraint${util_1.hashValues(association.associationKey, options.rule.ruleName)}`;
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchTemplateConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                rules: this.formatTemplateRule(portfolio.stack, options.rule),
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Provisioning rule ${options.rule.ruleName} already configured on association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static setLaunchRole(portfolio, product, launchRole, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            roleArn: launchRole.roleArn,
        });
    }
    static setLocalLaunchRoleName(portfolio, product, launchRoleName, options) {
        this.setLaunchRoleConstraint(portfolio, product, options, {
            localRoleName: launchRoleName,
        });
    }
    static deployWithStackSets(portfolio, product, options) {
        var _a;
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a launch role has already been set.
        if (portfolio.node.tryFindChild(this.launchRoleConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot configure StackSet deployment when a launch role is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.stackSetConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnStackSetConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: (_a = options.description) !== null && _a !== void 0 ? _a : '',
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                accountList: options.accounts,
                regionList: options.regions,
                adminRole: options.adminRole.roleArn,
                executionRole: options.executionRoleName,
                stackInstanceControl: options.allowStackSetInstanceOperations ? 'ALLOWED' : 'NOT_ALLOWED',
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot configure multiple StackSet deployment constraints for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static associateTagOptions(resource, resourceId, tagOptions) {
        const resourceStack = cdk.Stack.of(resource);
        for (const [key, tagOptionsList] of Object.entries(tagOptions.tagOptionsMap)) {
            validation_1.InputValidator.validateLength(resource.node.addr, 'TagOption key', 1, 128, key);
            tagOptionsList.forEach((value) => {
                validation_1.InputValidator.validateLength(resource.node.addr, 'TagOption value', 1, 256, value);
                const tagOptionKey = util_1.hashValues(key, value, resourceStack.node.addr);
                const tagOptionConstructId = `TagOption${tagOptionKey}`;
                let cfnTagOption = resourceStack.node.tryFindChild(tagOptionConstructId);
                if (!cfnTagOption) {
                    cfnTagOption = new servicecatalog_generated_1.CfnTagOption(resourceStack, tagOptionConstructId, {
                        key: key,
                        value: value,
                        active: true,
                    });
                }
                const tagAssocationKey = util_1.hashValues(key, value, resource.node.addr);
                const tagAssocationConstructId = `TagOptionAssociation${tagAssocationKey}`;
                if (!resource.node.tryFindChild(tagAssocationConstructId)) {
                    new servicecatalog_generated_1.CfnTagOptionAssociation(resource, tagAssocationConstructId, {
                        resourceId: resourceId,
                        tagOptionId: cfnTagOption.ref,
                    });
                }
            });
        }
        ;
    }
    static setLaunchRoleConstraint(portfolio, product, options, roleOptions) {
        const association = this.associateProductWithPortfolio(portfolio, product, options);
        // Check if a stackset deployment constraint has already been configured.
        if (portfolio.node.tryFindChild(this.stackSetConstraintLogicalId(association.associationKey))) {
            throw new Error(`Cannot set launch role when a StackSet rule is already defined for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
        const constructId = this.launchRoleConstraintLogicalId(association.associationKey);
        if (!portfolio.node.tryFindChild(constructId)) {
            const constraint = new servicecatalog_generated_1.CfnLaunchRoleConstraint(portfolio, constructId, {
                acceptLanguage: options.messageLanguage,
                description: options.description,
                portfolioId: portfolio.portfolioId,
                productId: product.productId,
                roleArn: roleOptions.roleArn,
                localRoleName: roleOptions.localRoleName,
            });
            // Add dependsOn to force proper order in deployment.
            constraint.addDependsOn(association.cfnPortfolioProductAssociation);
        }
        else {
            throw new Error(`Cannot set multiple launch roles for association ${this.prettyPrintAssociation(portfolio, product)}`);
        }
    }
    static stackSetConstraintLogicalId(associationKey) {
        return `StackSetConstraint${associationKey}`;
    }
    static launchRoleConstraintLogicalId(associationKey) {
        return `LaunchRoleConstraint${associationKey}`;
    }
    static prettyPrintAssociation(portfolio, product) {
        return `- Portfolio: ${portfolio.node.path} | Product: ${product.node.path}`;
    }
    static formatTemplateRule(stack, rule) {
        return JSON.stringify({
            [rule.ruleName]: {
                Assertions: this.formatAssertions(stack, rule.assertions),
                RuleCondition: rule.condition ? stack.resolve(rule.condition) : undefined,
            },
        });
    }
    static formatAssertions(stack, assertions) {
        return assertions.reduce((formattedAssertions, assertion) => {
            formattedAssertions.push({
                Assert: stack.resolve(assertion.assert),
                AssertDescription: assertion.description,
            });
            return formattedAssertions;
        }, new Array());
    }
    ;
}
exports.AssociationManager = AssociationManager;
//# sourceMappingURL=data:application/json;base64,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