"""Reproduction components for evolutionary algorithms."""

import random

from evoalgos.selection import UniformSelection


class Reproduction(object):
    """Abstract base class for objects that generate new Individuals."""

    def create(self, population, number):
        """Return `number` new individuals.

        This is an abstract method.

        Parameters
        ----------
        population : list of Individual
            The parent population.
        number : int
            The number of individuals to create.

        Returns
        -------
        offspring : list of Individual
            The new individuals.

        """
        raise NotImplementedError()


    def extend_by(self, population, number):
        """Extend the population by `number` individuals.

        This method works by calling :func:`create` and appending these
        individuals to the population.

        Parameters
        ----------
        population : list of Individual
            The parent population.
        number : int
            The number of individuals to add to the population.

        """
        # create some individuals
        offspring = self.create(population, number)
        # add offspring
        population.extend(offspring)



class ESReproduction(Reproduction):
    """A reproduction component suitable for most evolutionary algorithms."""

    def __init__(self, selection=None,
                 recombination_prob=1.0,
                 redetermine_first_parent=True):
        """Constructor.

        Parameters
        ----------
        selection : Selection, optional
            This object is used for selecting parent individuals. Default
            is random uniform selection.
        recombination_prob : float, optional
            A probability to determine how much sexual and asexual
            reproduction is carried out. Sexual reproduction means that
            a new offspring is generated by recombination and mutation.
            For asexual reproduction a single individual is cloned and
            then mutated. Default is to always use sexual reproduction.
        redetermine_first_parent : bool, optional
            Indicates if a new parent is selected in each iteration, or if
            the same parent is used to produce all offspring in one call to
            :func:`create`. The other parents are always redetermined, so
            setting this to False probably only makes sense in scenarios
            without recombination.

        """
        self.selection = selection
        if selection is None:
            self.selection = UniformSelection()
        self.recombination_prob = recombination_prob
        self.redetermine_first_parent = redetermine_first_parent


    def create(self, population, number):
        """Return `number` new individuals.

        This method works as follows: as long as more offspring are needed,
        a first parent is determined via the selection component. The first
        parent's property `num_parents` determines how many other parents
        are drawn. If the population is smaller than this value, it is
        multiplied until it is larger. The other parents are then again
        determined by the selection component, and the offspring is created
        with ``first_parent.recombine(other_parents)``. If asexual
        reproduction is chosen, offspring is created by
        ``first_parent.clone()``. If too many individuals were generated,
        random ones are deleted. Finally, every child is mutated.

        Parameters
        ----------
        population : list of Individual
            The parent population.
        number : int
            The number of individuals to create.

        Returns
        -------
        offspring : list of Individual
            The new individuals.

        """
        offspring = []
        selection = self.selection
        first_parent = population[:]
        remaining_individuals = selection.reduce_to(first_parent, 1)
        first_parent = first_parent[0]
        while len(offspring) < number:
            # now we have partitioned the population into the first parent
            # and potential mates
            num_parents = first_parent.num_parents
            sexual = random.random() < self.recombination_prob
            sexual &= len(remaining_individuals) > 0
            if sexual:
                # allow a number of parents that is greater than population size
                multiset_remaining = []
                while len(multiset_remaining) < num_parents - 1:
                    multiset_remaining.extend(remaining_individuals)
                # select possible partners
                other_parents = selection.select(multiset_remaining,
                                                 num_parents - 1)
                # create offspring
                next_offspring = first_parent.recombine(other_parents)
            else:
                next_offspring = [first_parent.clone()]
            offspring.extend(next_offspring)
            if self.redetermine_first_parent:
                first_parent = population[:]
                remaining_individuals = selection.reduce_to(first_parent, 1)
                first_parent = first_parent[0]
        random.shuffle(offspring)
        # delete surplus individuals
        del offspring[number:]
        # mutate individuals
        for child in offspring:
            child.mutate()
        return offspring
