# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_levelSetKDEx_univariate.ipynb.

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 4
from __future__ import annotations
from fastcore.docments import *
from fastcore.test import *
from fastcore.utils import *

import pandas as pd
import numpy as np
import itertools
from collections import defaultdict, Counter, deque
import warnings
import copy

from scipy import sparse
from sklearn.neighbors import NearestNeighbors
from sklearn.cluster import KMeans
from sklearn.base import BaseEstimator
from sklearn.exceptions import NotFittedError
import faiss

from .baseClasses import BaseLSx, BaseWeightsBasedEstimator
from .utils import restructureWeightsDataList

# %% auto 0
__all__ = ['LevelSetKDEx', 'generateBins', 'LevelSetKDEx_kNN', 'LevelSetKDEx_NN', 'getNeighbors', 'getNeighborsTest',
           'getKernelValues', 'LevelSetKDEx_clustering', 'LevelSetKDEx_clustering2']

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 7
class LevelSetKDEx(BaseWeightsBasedEstimator, BaseLSx):
    """
    `LevelSetKDEx` turns any point forecasting model into an estimator of the underlying conditional density.
    The name 'LevelSet' stems from the fact that this approach interprets the values of the point forecasts
    as a similarity measure between samples. The point forecasts of the training samples are sorted and 
    recursively assigned to a bin until the size of the current bin reaches `binSize` many samples. Then
    a new bin is created and so on. For a new test sample we check into which bin its point prediction
    would have fallen and interpret the training samples of that bin as the empirical distribution function
    of this test sample.    
    """
    
    def __init__(self, 
                 estimator, # Model with a .fit and .predict-method (implementing the scikit-learn estimator interface).
                 binSize: int=100, # Size of the bins created while running fit.
                 # Determines behaviour of method `getWeights`. If False, all weights receive the same  
                 # value. If True, the distance of the point forecasts is taking into account.
                 weightsByDistance: bool=False, 
                 ):
        
        super(BaseEstimator, self).__init__(estimator = estimator)

        # Check if binSize is integer
        if not isinstance(binSize, (int, np.int32, np.int64)):
            raise ValueError("'binSize' must be an integer!")

        # Check if weightsByDistance is bool
        if not isinstance(weightsByDistance, bool):
            raise ValueError("'weightsByDistance' must be a boolean!")

        self.binSize = binSize
        self.weightsByDistance = weightsByDistance
        
        self.yTrain = None
        self.yPredTrain = None
        self.indicesPerBin = None
        self.lowerBoundPerBin = None
        self.fitted = False
    
    #---
    
    def fit(self: LevelSetKDEx, 
            X: np.ndarray, # Feature matrix used by `estimator` to predict `y`.
            y: np.ndarray, # 1-dimensional target variable corresponding to the feature matrix `X`.
            ):
        """
        Fit `LevelSetKDEx` model by grouping the point predictions of the samples specified via `X`
        according to their value. Samples are recursively sorted into bins until each bin contains
        `binSize` many samples. For details, checkout the function `generateBins` which does the
        heavy lifting.
        """
        
        # Checks
        if not isinstance(self.binSize, (int, np.int32, np.int64)):
            raise ValueError("'binSize' must be an integer!")
            
        if self.binSize > y.shape[0]:
            raise ValueError("'binSize' mustn't be bigger than the size of 'y'!")
        
        if X.shape[0] != y.shape[0]:
            raise ValueError("'X' and 'y' must contain the same number of samples!")
        
        # IMPORTANT: In case 'y' is given as a pandas.Series, we can potentially run into indexing 
        # problems later on.
        if isinstance(y, pd.Series):
            y = y.ravel()
        
        #---
        
        try:
            yPred = self.estimator.predict(X)
            
        except NotFittedError:
            try:
                self.estimator.fit(X = X, y = y)                
            except:
                raise ValueError("Couldn't fit 'estimator' with user specified 'X' and 'y'!")
            else:
                yPred = self.estimator.predict(X)
        
        #---
        
        indicesPerBin, lowerBoundPerBin = generateBins(binSize = self.binSize,
                                                       yPred = yPred)

        self.yTrain = y
        self.yPredTrain = yPred
        self.indicesPerBin = indicesPerBin
        self.lowerBoundPerBin = lowerBoundPerBin
        self.fitted = True
        
    #---
    
    def getWeights(self, 
                   X: np.ndarray, # Feature matrix for which conditional density estimates are computed.
                   # Specifies structure of the returned density estimates. One of: 
                   # 'all', 'onlyPositiveWeights', 'summarized', 'cumDistribution', 'cumDistributionSummarized'
                   outputType: str='onlyPositiveWeights', 
                   # Optional. List with length X.shape[0]. Values are multiplied to the estimated 
                   # density of each sample for scaling purposes.
                   scalingList: list=None, 
                   ) -> list: # List whose elements are the conditional density estimates for the samples specified by `X`.
        
        # __annotations__ = BaseWeightsBasedEstimator.getWeights.__annotations__
        __doc__ = BaseWeightsBasedEstimator.getWeights.__doc__
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
        
        #---
        
        yPred = self.estimator.predict(X)
        
        binPerPred = np.searchsorted(a = self.lowerBoundPerBin, v = yPred, side = 'right') - 1
        neighborsList = [self.indicesPerBin[binIndex] for binIndex in binPerPred]
        
        #---
        
        if self.weightsByDistance:

            predDistances = [np.abs(self.yPredTrain[neighborsList[i]] - yPred[i]) for i in range(len(neighborsList))]

            weightsDataList = list()
            
            for i in range(X.shape[0]):
                distances = predDistances[i]
                distancesCloseZero = np.isclose(distances, 0)
                
                if np.any(distancesCloseZero):
                    indicesCloseZero = neighborsList[i][np.where(distancesCloseZero)[0]]
                    weightsDataList.append((np.repeat(1 / len(indicesCloseZero), len(indicesCloseZero)),
                                            indicesCloseZero))
                    
                else:                                 
                    inverseDistances = 1 / distances

                    weightsDataList.append((inverseDistances / inverseDistances.sum(), 
                                            np.array(neighborsList[i], dtype = 'uintc')))
            
            weightsDataList = restructureWeightsDataList(weightsDataList = weightsDataList, 
                                                         outputType = outputType, 
                                                         y = self.yTrain,
                                                         scalingList = scalingList,
                                                         equalWeights = False)
        
        else:
            weightsDataList = [(np.repeat(1 / len(neighbors), len(neighbors)), np.array(neighbors, dtype = 'uintc')) 
                               for neighbors in neighborsList]

            weightsDataList = restructureWeightsDataList(weightsDataList = weightsDataList, 
                                                         outputType = outputType, 
                                                         y = self.yTrain,
                                                         scalingList = scalingList,
                                                         equalWeights = True)
        
        return weightsDataList
    
    #---
    
    def solveKernelGLS(self,
                       X,
                       sigma,
                       c):
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
        
        #---
        
        yPred = self.estimator.predict(X)
        binPerPred = np.searchsorted(a = self.lowerBoundPerBin, v = yPred, side = 'right') - 1
        
        binVectors = [(binPerPred == i).reshape(-1, 1) * 1 for i in range(len(self.lowerBoundPerBin))]
        binVectorsToSlice = [np.where(binVector)[0] for binVector in binVectors]
        
        #---
        
        def getNewSolution(u, x, y):
    
            if len(u.shape) == 1:
                u.reshape(-1, 1)

            uSlice = np.where(u == 1)[0]

            ux = np.sum(x[uSlice, :], axis = 0, keepdims = True)
            xNew = x - np.matmul(y, ux) / (1 + np.sum(y[uSlice]))

            return xNew
        
        def solveGLS_initial(sigma, u, c):
    
            if len(u.shape) == 1:
                u = u.reshape(-1, 1)

            if len(c.shape) == 1:
                c = c.reshape(-1, 1)

            x = c * sigma**(-1)
            y = u * sigma**(-1)

            return getNewSolution(u = u, x = x, y = y)
        
        #---
        
        yDict = dict()

        # Loop for A_k^-1 y = u_k
        for k in range(len(binVectors)):

            # Loop for A_j^-1 y = u_k
            for j in range(k):

                if j == 0:
                    yDict[(0, k)] = solveGLS_initial(sigma = sigma, u = binVectors[0], c = binVectors[k])

                else:
                    yDict[j, k] = getNewSolution(u = binVectors[j], x = yDict[(j - 1, k)], y = yDict[(j - 1, j)])   

            #---

            if k == 0:
                x = solveGLS_initial(sigma = sigma, u = binVectors[0], c = c)
            else:
                x = getNewSolution(u = binVectors[k], x = x, y = yDict[(k - 1, k)])
        
        
        return x
    
    #---
    
    def getKernelVectorProduct(self,
                               X1,
                               c,
                               X2 = None,
                               ):
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
        
        #---

        yPred1 = self.estimator.predict(X1)
        binPerPred1 = np.searchsorted(a = self.lowerBoundPerBin, v = yPred1, side = 'right') - 1
        
        binVectors1 = [(binPerPred1 == i).reshape(-1, 1) * 1 for i in range(len(self.lowerBoundPerBin))]
        binVectorsToSlice1 = [np.where(binVector)[0] for binVector in binVectors1]
        
        if X2 is None:
            binVectors2 = binVectors1
            binVectorsToSlice2 = binVectorsToSlice1
        
        else:
            yPred2 = self.estimator.predict(X2)
            binPerPred2 = np.searchsorted(a = self.lowerBoundPerBin, v = yPred2, side = 'right') - 1

            binVectors2 = [(binPerPred2 == i).reshape(-1, 1) * 1 for i in range(len(self.lowerBoundPerBin))]
            binVectorsToSlice2 = [np.where(binVector)[0] for binVector in binVectors2]
        
        if len(c.shape) == 1:
            c = c.reshape(-1, 1)
            
        #---
        
        n = X1.shape[0]
        m = c.shape[1]
        
        resList = list()
        
        for i in range(len(self.lowerBoundPerBin)):
            uc = np.sum(c[binVectorsToSlice2[i], :], axis = 0, keepdims = True)
            
            kernelProduct = np.zeros(shape = (n, m))
            kernelProduct[binVectorsToSlice1[i], :] = uc
            
            resList.append(kernelProduct)
        
        vectorProduct = np.sum(resList, axis = 0)
        
        return vectorProduct
            
    #---
    
    def getGaussianPosterior(self,
                             XTrain,
                             yTrain,
                             XTest,
                             sigma):
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
        
        #---
        
        yPredTrain = self.estimator.predict(XTrain)
        binPerPredTrain = np.searchsorted(a = self.lowerBoundPerBin, v = yPredTrain, side = 'right') - 1
        
        binVectorsTrain = [(binPerPredTrain == i).reshape(-1, 1) * 1 for i in range(len(self.lowerBoundPerBin))]
        binVectorsToSliceTrain = [np.where(binVector)[0] for binVector in binVectorsTrain]
        
        yPredTest = self.estimator.predict(XTest)
        binPerPredTest = np.searchsorted(a = self.lowerBoundPerBin, v = yPredTest, side = 'right') - 1
        
        binVectorsTest = [(binPerPredTest == i).reshape(-1, 1) * 1 for i in range(len(self.lowerBoundPerBin))]
        binVectorsToSliceTest = [np.where(binVector)[0] for binVector in binVectorsTest]
        
        #---
        
        n = binVectorsTrain[0].shape[0]
        k = binVectorsTest[0].shape[0]
        
        kernelProductList = list()
        
        for i in range(len(self.lowerBoundPerBin)):
            x = self.solveKernelGLS(X = XTrain,
                                    sigma = sigma,
                                    c = binVectorsTrain[i])
            
            kernelProduct = np.zeros(shape = (n, k))
            kernelProduct[:, binVectorsToSliceTest[i]] = x
            kernelProductList.append(kernelProduct)
            
        kernelProduct = sum(kernelProductList)
        
        # x = self.solveGLSKernel(X = XTrain,
        #                         sigma = sigma,
        #                         c = kernelProduct)
        
        covRightSide = self.getKernelVectorProduct(X1 = XTest,
                                                   X2 = XTrain,
                                                   c = kernelProduct)
        
        #---
        
        covPrior = np.zeros(shape = (k, k))
        
        for i in range(len(self.lowerBoundPerBin)):
            v = binVectorsToSliceTest[i]
            covPrior[v[:, None], v] = 1
        
        covPosterior = covPrior - covRightSide
        
        #---
        
        if len(yTrain.shape) == 1:
            yTrain = yTrain.reshape(-1, 1)
        
        x = self.solveKernelGLS(X = XTrain, 
                                sigma = sigma,
                                c = yTrain)
        
        meanPosterior = self.getKernelVectorProduct(X1 = XTest,
                                                    X2 = XTrain,
                                                    c = x)
        
        return meanPosterior, covPosterior
    

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 11
def generateBins(binSize: int, # Size of the bins of values of `yPred` being grouped together.
                 yPred: np.ndarray, # 1-dimensional array of predicted values.
                 ):
    "Used to generate the bin-structure used by `LevelSetKDEx` to compute density estimations."
    
    predIndicesSort = np.argsort(yPred)
    yPredSorted = yPred[predIndicesSort]

    currentBinSize = 0
    binIndex = 0
    trainIndicesLeft = len(yPred)
    indicesPerBin = defaultdict(list)
    lowerBoundPerBin = dict()
    
    for i in range(len(yPred)):
        
        if i == 0:
            lowerBoundPerBin[binIndex] = np.NINF
            
        currentBinSize += 1
        trainIndicesLeft -= 1

        indicesPerBin[binIndex].append(predIndicesSort[i])
        
        if trainIndicesLeft < binSize:
            indicesPerBin[binIndex].extend(predIndicesSort[np.arange(i+1, len(yPred), 1)])
            break

        if currentBinSize >= binSize and yPredSorted[i] < yPredSorted[i+1]:
            lowerBoundPerBin[binIndex + 1] = (yPredSorted[i] + yPredSorted[i+1]) / 2
            binIndex += 1
            currentBinSize = 0
           
    indicesPerBin = {binIndex: np.array(indices, dtype = 'uintc') for binIndex, indices in indicesPerBin.items()}
    
    lowerBoundPerBin = pd.Series(lowerBoundPerBin)
    lowerBoundPerBin.index.name = 'binIndex'
    
    return indicesPerBin, lowerBoundPerBin

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 14
class LevelSetKDEx_kNN(BaseWeightsBasedEstimator, BaseLSx):
    """
     `LevelSetKDEx_kNN` turns any point predictor that has a .predict-method 
    into an estimator of the condititional density of the underlying distribution.
    The basic idea of each level-set based approach is to interprete the point forecast
    generated by the underlying point predictor as a similarity measure of samples.
    In the case of the `LevelSetKDEx_kNN` defined here, for every new samples
    'binSize'-many training samples are computed whose point forecast is closest
    to the point forecast of the new sample.
    The resulting empirical distribution of these 'nearest' training samples are 
    viewed as our estimation of the conditional distribution of each the new sample 
    at hand.
    
    NOTE: In contrast to the standard `LevelSetKDEx`, it is possible to apply
    `LevelSetKDEx_kNN` to arbitrary dimensional point predictors.
    """
    
    def __init__(self, 
                 estimator, # Model with a .fit and .predict-method (implementing the scikit-learn estimator interface).
                 binSize: int=100, # Size of the bins created while running fit.
                 # Determines behaviour of method `getWeights`. If False, all weights receive the same  
                 # value. If True, the distance of the point forecasts is taking into account.
                 weightsByDistance: bool=False, 
                 ):
        
        super(BaseEstimator, self).__init__(estimator = estimator)

        # Check if binSize is integer
        if not isinstance(binSize, (int, np.int32, np.int64)):
            raise ValueError("'binSize' must be an integer!")

        # Check if weightsByDistance is bool
        if not isinstance(weightsByDistance, bool):
            raise ValueError("'weightsByDistance' must be a boolean!")
        
        self.binSize = binSize
        self.weightsByDistance = weightsByDistance
        
        self.yTrain = None
        self.yPredTrain = None
        self.nearestNeighborsOnPreds = None
        self.fitted = False
        
    #---
    
    def fit(self: LevelSetKDEx, 
            X: np.ndarray, # Feature matrix used by `estimator` to predict `y`.
            y: np.ndarray, # 1-dimensional target variable corresponding to the feature matrix `X`.
            ):
        """
        Fit `LevelSetKDEx_kNN` model by applying the nearest neighbors algorithm to the point
        predictions of the samples specified by `X` based on `estimator`. 
        """
        
        # Checks
        if not isinstance(self.binSize, (int, np.int32, np.int64)):
            raise ValueError("'binSize' must be an integer!")
            
        if self.binSize > y.shape[0]:
            raise ValueError("'binSize' mustn't be bigger than the size of 'y'!")
            
        if X.shape[0] != y.shape[0]:
            raise ValueError("'X' and 'y' must contain the same number of samples!")
            
        # IMPORTANT: In case 'y' is given as a pandas.Series, we can potentially run into indexing 
        # problems later on.
        if isinstance(y, pd.Series):
            y = y.ravel()
        
        #---
        
        try:
            yPred = self.estimator.predict(X)
        except NotFittedError:
            # warnings.warn("The object 'estimator' must have been fitted already!"
            #               "'estimator' will be fitted with 'X' and 'y' to enable point predicting!",
            #               stacklevel = 2)
            try:
                self.estimator.fit(X = X, y = y)                
            except:
                raise ValueError("Couldn't fit 'estimator' with user specified 'X' and 'y'!")
            else:
                yPred = self.estimator.predict(X)

        #---
        
        yPred_reshaped = np.reshape(yPred, newshape = (len(yPred), 1))
        
        nn = NearestNeighbors(algorithm = 'kd_tree')
        nn.fit(X = yPred_reshaped)

        #---

        self.yTrain = y
        self.yPredTrain = yPred
        self.nearestNeighborsOnPreds = nn
        self.fitted = True
        
    #---
    
    def getWeights(self, 
                   X: np.ndarray, # Feature matrix for which conditional density estimates are computed.
                   # Specifies structure of the returned density estimates. One of: 
                   # 'all', 'onlyPositiveWeights', 'summarized', 'cumDistribution', 'cumDistributionSummarized'
                   outputType: str='onlyPositiveWeights', 
                   # Optional. List with length X.shape[0]. Values are multiplied to the estimated 
                   # density of each sample for scaling purposes.
                   scalingList: list=None, 
                   ) -> list: # List whose elements are the conditional density estimates for the samples specified by `X`.
        
        __doc__ = BaseWeightsBasedEstimator.getWeights.__doc__
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx_kNN instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
            
        #---

        nn = self.nearestNeighborsOnPreds

        #---

        yPred = self.estimator.predict(X)   
        yPred_reshaped = np.reshape(yPred, newshape = (len(yPred), 1))

        distancesMatrix, neighborsMatrix = nn.kneighbors(X = yPred_reshaped, 
                                                         n_neighbors = self.binSize + 1)

        #---

        neighborsList = list(neighborsMatrix[:, 0:self.binSize])
        distanceCheck = np.where(distancesMatrix[:, self.binSize - 1] == distancesMatrix[:, self.binSize])
        indicesToMod = distanceCheck[0]

        for index in indicesToMod:
            distanceExtremePoint = np.absolute(yPred[index] - self.yPredTrain[neighborsMatrix[index, self.binSize-1]])

            neighborsByRadius = nn.radius_neighbors(X = yPred_reshaped[index:index + 1], 
                                                    radius = distanceExtremePoint, return_distance = False)[0]
            neighborsList[index] = neighborsByRadius

        #---
        
        if self.weightsByDistance:
            binSizesReal = [len(neighbors) for neighbors in neighborsList]
            binSizeMax = max(binSizesReal)
            
            distancesMatrix, neighborsMatrix = nn.kneighbors(X = yPred_reshaped, 
                                                             n_neighbors = binSizeMax)
            
            weightsDataList = list()
            
            for i in range(X.shape[0]):
                distances = distancesMatrix[i, 0:binSizesReal[i]]
                distancesCloseZero = np.isclose(distances, 0)
                
                if np.any(distancesCloseZero):
                    indicesCloseZero = neighborsMatrix[i, np.where(distancesCloseZero)[0]]
                    weightsDataList.append((np.repeat(1 / len(indicesCloseZero), len(indicesCloseZero)),
                                            indicesCloseZero))
                    
                else:                                 
                    inverseDistances = 1 / distances

                    weightsDataList.append((inverseDistances / inverseDistances.sum(), 
                                            np.array(neighborsList[i], dtype = 'uintc')))
            
            weightsDataList = restructureWeightsDataList(weightsDataList = weightsDataList, 
                                                         outputType = outputType, 
                                                         y = self.yTrain,
                                                         scalingList = scalingList,
                                                         equalWeights = False)
            
        else:
            weightsDataList = [(np.repeat(1 / len(neighbors), len(neighbors)), np.array(neighbors, dtype = 'uintc')) 
                               for neighbors in neighborsList]

            weightsDataList = restructureWeightsDataList(weightsDataList = weightsDataList, 
                                                         outputType = outputType, 
                                                         y = self.yTrain,
                                                         scalingList = scalingList,
                                                         equalWeights = True)

        return weightsDataList
      

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 18
class LevelSetKDEx_NN(BaseWeightsBasedEstimator, BaseLSx):
    """
     `LevelSetKDEx_kNN` turns any point predictor that has a .predict-method 
    into an estimator of the condititional density of the underlying distribution.
    The basic idea of each level-set based approach is to interprete the point forecast
    generated by the underlying point predictor as a similarity measure of samples.
    In the case of the `LevelSetKDEx_kNN` defined here, for every new samples
    'binSize'-many training samples are computed whose point forecast is closest
    to the point forecast of the new sample.
    The resulting empirical distribution of these 'nearest' training samples are 
    viewed as our estimation of the conditional distribution of each the new sample 
    at hand.
    
    NOTE: In contrast to the standard `LevelSetKDEx`, it is possible to apply
    `LevelSetKDEx_kNN` to arbitrary dimensional point predictors.
    """
    
    def __init__(self, 
                 estimator, # Model with a .fit and .predict-method (implementing the scikit-learn estimator interface).
                 binSize: int=100, # Size of the bins created while running fit.
                 # Setting 'efficientRAM = TRUE' is only necessary when there are roughly umore than 200k training observations to avoid
                 # an overusage of RAM. This setting causes the run-time of the algorithm of the weights computation to linearly depend on 
                 # 'binSize'. Because of that the algorithm becomes quite slow for 'binSize' > 10k'.
                 efficientRAM: bool=False,
                 ):
        
        super(BaseEstimator, self).__init__(estimator = estimator)

        # Check if binSize is integer
        if not isinstance(binSize, (int, np.int32, np.int64)):
            raise ValueError("'binSize' must be an integer!")

        # Check if efficient RAM is boolean
        if not isinstance(efficientRAM, bool):
            raise ValueError("'efficientRAM' must be a boolean!")
        
        self.binSize = binSize
        self.weightsByDistance = False
        
        self.yTrain = None
        self.yPredTrain = None
        self.fitted = False
        self.efficientRAM = efficientRAM
        
    #---
    
    def fit(self: LevelSetKDEx, 
            X: np.ndarray, # Feature matrix used by `estimator` to predict `y`.
            y: np.ndarray, # 1-dimensional target variable corresponding to the feature matrix `X`.
            ):
        """
        Fit `LevelSetKDEx_kNN` model by applying the nearest neighbors algorithm to the point
        predictions of the samples specified by `X` based on `estimator`. 
        """
        
        # Checks
        if not isinstance(self.binSize, (int, np.int32, np.int64)):
            raise ValueError("'binSize' must be an integer!")
            
        if self.binSize > y.shape[0]:
            raise ValueError("'binSize' mustn't be bigger than the size of 'y'!")
            
        if X.shape[0] != y.shape[0]:
            raise ValueError("'X' and 'y' must contain the same number of samples!")
            
        # IMPORTANT: In case 'y' is given as a pandas.Series, we can potentially run into indexing 
        # problems later on.
        if isinstance(y, pd.Series):
            y = y.ravel()
        
        #---
        
        try:
            yPred = self.estimator.predict(X)
            
        except NotFittedError:
            try:
                self.estimator.fit(X = X, y = y)                
            except:
                raise ValueError("Couldn't fit 'estimator' with user specified 'X' and 'y'!")
            else:
                yPred = self.estimator.predict(X)

        #---
        
        neighborsDict, neighborsRemoved, neighborsAdded = getNeighbors(binSize = self.binSize,
                                                                       yPred = yPred)

        self.yTrain = y
        self.yPredTrain = yPred
        self.neighborsDictTrain = neighborsDict
        self._neighborsRemoved = neighborsRemoved
        self._neighborsAdded = neighborsAdded
        self.fitted = True
        
    #---
    
    def getWeights(self, 
                   X: np.ndarray, # Feature matrix for which conditional density estimates are computed.
                   # Specifies structure of the returned density estimates. One of: 
                   # 'all', 'onlyPositiveWeights', 'summarized', 'cumDistribution', 'cumDistributionSummarized'
                   outputType: str='onlyPositiveWeights', 
                   # Optional. List with length X.shape[0]. Values are multiplied to the estimated 
                   # density of each sample for scaling purposes.
                   scalingList: list=None,
                   ) -> list: # List whose elements are the conditional density estimates for the samples specified by `X`.
        
        __doc__ = BaseWeightsBasedEstimator.getWeights.__doc__
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx_kNN instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
            
        #---
        
        yPred = self.estimator.predict(X)
        
        neighborsDictTest = getNeighborsTest(binSize = self.binSize,
                                             yPred = yPred,
                                             yPredTrain = self.yPredTrain,
                                             neighborsDictTrain = self.neighborsDictTrain)
        
        #---
        
        weightsDataList = getKernelValues(yPred = yPred,
                                          yPredTrain = self.yPredTrain,
                                          neighborsDictTest = neighborsDictTest,
                                          neighborsDictTrain = self.neighborsDictTrain,
                                          neighborsRemoved = self._neighborsRemoved,
                                          neighborsAdded = self._neighborsAdded,
                                          binSize = self.binSize,
                                          efficientRAM = self.efficientRAM)
        
        weightsDataList = restructureWeightsDataList(weightsDataList = weightsDataList, 
                                                     outputType = outputType, 
                                                     y = self.yTrain,
                                                     scalingList = scalingList,
                                                     equalWeights = True)

        return weightsDataList
      

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 20
def getNeighbors(binSize: int, # Size of the bins of values of `yPred` being grouped together.
                 yPred: np.ndarray, # 1-dimensional array of predicted values.
                 ):
    "Used to generate the neighboorhoods used by `LevelSetKDEx` to compute density estimations."
    
    duplicationDict = defaultdict(list)
    counterDict = defaultdict(int)
    
    for index, value in enumerate(yPred):
        duplicationDict[value].append(index)
        counterDict[value] += 1
    
    duplicationDict = dict(duplicationDict)
    counterDict = dict(counterDict)
    
    yPredUnique = np.sort(list(duplicationDict.keys()))
    
    #---
    
    # Here we initiate our search for the nearest neighbors by creating the neighborhood of the lowest point prediction.
    # VARIABLES:
    # a) 'neighbors': A Collection.deque object that keeps track of the training indices which are the nearest neighbors of 
    # the current observation. A deque object allows us to very efficiently remove values from the left and right side of the last.
    # b) 'neighborsMaxIter': The index of yPredUnique that has to be considered next (and hasn't yet been considered).
    # neighborsMaxIter has to be set to len(yPredUnique), when we iterated over all yPredUnique.
    #
    # PROCEDURE
    # 1) We simply start at the lowest predicted value, iterate from here on over the other predicted values and keep
    # adding indices of neighbors by using 'duplicationDict', whose keys are the predicted values
    # and whose entries are the indices of the training instances that all share the same point prediction.
    # 2) 'neighborsMaxIter' is eventually being set, so we can use this information during the further search down below. 
    
    neighborsPerPred = dict()
    neighbors = deque()
    
    for k in range(len(yPredUnique)):
        
        if len(neighbors) < binSize:
            neighbors.extend(duplicationDict[yPredUnique[k]])
            
        else:
            neighborsMaxIter = k
            break
        
        if k == (len(yPredUnique) - 1):
            neighborsMaxIter = len(yPredUnique)
    
    neighborsPerPred[yPredUnique[0]] = np.array(neighbors, dtype = 'uintc')
    neighborsUnchangedLoop = True
    neighborsUnchangedLoop = True
    
    #---
    
    neighborsRemoved = [0]
    neighborsAdded = [0]
    
    for i in range(1, len(yPredUnique)):
        
        removeCounter = 0
        addCounter = 0
            
        predCurrent = yPredUnique[i]
        
        #---
        
        # CHECK AND CLEAN CURRENT NEIGHBORHOOD BEFORE STARTING THE LOOP
        # 
        # One very obvious case where we need such a cleaning step is the following:
        # Assume yPredUnique = [1, 2, 3] and binsize = 2. In this case the loop below over k in 
        # range(neighborsMaxIter, len(yPredUnique), 1) creates the neighborhood of indices [0, 1, 2] (all indices) 
        # for 2. As we have iterated over all possible neighbors, below loop isn't considered anymore. When we now
        # consider the prediction 3, its neighborhood is supposed to consist of [1, 2]. For that reason we must check
        # if it is possible to remove the left-side of the neighborhood in case 'neighbors' is bigger than bin-size.
        # 
        # So what are we doing here?
        # 1) We enter the loop, when the number of currently detected neighbors is higher than binSize. This can
        # naturally occur, when the the outer neighbors contain identical predicted values.
        # 2) We first compute the distance of the current prediction to the two outer predictions.
        # 3) When the current and left-most prediction are not identical and further apart than distanceToMax,
        # then we look up how many indices belong to the left-most prediction. If this is not the case, we do nothing 
        # and leave the while loop.
        # 4) If the number of neighbors would still be higher than binSize after removing all left-most neighbors,
        # we remove all left-most neighbors and go back to step (3). Otherwise we stop.
        
        neighborsUnchangedCheck = True
        
        if len(neighbors) > binSize:
            
            checkNeeded = True
            while checkNeeded:
                
                distanceToMin = predCurrent - yPred[neighbors[0]]
                distanceToMax = yPred[neighbors[len(neighbors) - 1]] - predCurrent

                if distanceToMin > 0 and distanceToMin > distanceToMax:
                    countIdenticalMin = counterDict[yPred[neighbors[0]]]
                
                    if len(neighbors) - countIdenticalMin >= binSize:
                        removeCounter += countIdenticalMin

                        for p in range(countIdenticalMin):
                            neighbors.popleft()
                            
                        neighborsUnchangedCheck = False
                        
                    else:
                        checkNeeded = False
                else:
                    checkNeeded = False

        #---
        
        # After checking and cleaning 'neighbors', there are 3 different scenarios how 'neighbors' looks like:
        # 1) 'neighbors' is either smaller than or equal to 'binSize' and no removal of indices has been conducted 
        #
        # 2) 'neighbors' is bigger than 'binSize' and no removal of the left-most preds has been conducted because either
        # a) the current prediction and left-most prediction are identical
        # b) distanceToMin is smaller than or equal to distanceToMax
        # c) removing the left most preds would have made neighbors smaller than binSize
        #
        # 3) 'neighbors' is bigger than or equal to 'binSize' and as many left-most preds have been removed until the removal
        # of further preds would have made neighbors smaller than 'binSize'.
        
        # Now we compute distanceMin and distanceMax and start looking for new nearest neighbors beginning with 'neighborsMaxIter',
        # which is obviously the next bigger pred to the currently highest pred that is part of 'neighbors'.
        #
        # We have to consider 5 different cases:
        # 1) 'neighbors' is smaller than 'binSize' --> the new pred is always added
        # 2) The new pred is closer to the current prediction than the left-most prediction --> the left-most predictions are
        # all removed and the new predictions are added.
        # 3) The new prediction is exactly as far away as the left-most prediction --> new predictions are added without any removal
        # 4) The new prediction is as far away as the currently right-most prediction of 'neighbors' --> new predictions are added
        # without any removal
        # 5) None of the above 4 cases apply --> break and set 'neighborsMaxIter' to the current index k of 'yPredUnique'.
        # 
        # NOTE: Can case (4) ever happen? I mean, we are iterating over 'yPredUnique'!?!?!

        distanceToMin = predCurrent - yPred[neighbors[0]]
        distanceToMax = yPred[neighbors[len(neighbors) - 1]] - predCurrent
        
        # If the neighbors-object has not been changed since the last considered point
        # prediction for which we computed the nearest neighbors, we simply reuse the same
        # neighbors object in RAM (see below).
        neighborsUnchangedLoop = True
        
        for k in range(neighborsMaxIter, len(yPredUnique), 1):
            
            predNew = yPredUnique[k]
            distance = predNew - predCurrent 
            
            if len(neighbors) < binSize:
                neighbors.extend(duplicationDict[predNew])
                
                distanceToMax = yPred[neighbors[len(neighbors) - 1]] - predCurrent
                addCounter += counterDict[predNew]
                neighborsUnchangedLoop = False
                
            elif distance < distanceToMin:
                neighbors.extend(duplicationDict[predNew])
                addCounter += counterDict[predNew]
                
                countIdenticalMin = counterDict[yPred[neighbors[0]]]
                for p in range(countIdenticalMin): 
                    neighbors.popleft()
                    
                removeCounter += countIdenticalMin
                distanceToMin = predCurrent - yPred[neighbors[0]]
                distanceToMax = yPred[neighbors[len(neighbors) - 1]] - predCurrent
                neighborsUnchangedLoop = False
                
            elif distance == distanceToMin:
                neighbors.extend(duplicationDict[predNew])
                
                distanceToMax = yPred[neighbors[len(neighbors) - 1]] - predCurrent
                addCounter += counterDict[predNew]
                neighborsUnchangedLoop = False
                
            elif distance == distanceToMax:
                neighbors.extend(duplicationDict[predNew])
                addCounter += counterDict[predNew]
                neighborsUnchangedLoop = False
                
            # We only ever end up here, if 'predNew' hasn't been added to 'neighbors'.
            else:
                    
                neighborsMaxIter = k
                break

            # If we end up down here, it means that all train preds have sucessuflly been added to the 
            # current neighborhood. For that reason, neighborsMaxIter has to be set to len(yPred) in order
            # to stop the code from starting the loop.
            # We only end up here, if the highest point prediction has been added during this iteration.
            # We have to treat this is a special case using the 'addedHighestPredDuringIteration' variable.
            if k == (len(yPredUnique) - 1):
                neighborsMaxIter = len(yPredUnique)
                
        # Ideas to solve RAM problems:
        # 1) Save neighbors into Sparse Array (neighbors have to be padded in some cases)
        # 2) Don't save redundant neighbors objects (already done here)
        # 3) Save neighbors Objects in Memory
        
        if neighborsUnchangedCheck and neighborsUnchangedLoop:
            neighborsPerPred[predCurrent] = neighborsPerPred[yPredUnique[i-1]]
        else:
            neighborsPerPred[predCurrent] = np.array(neighbors, dtype = 'uintc')

        neighborsRemoved.append(removeCounter)
        neighborsAdded.append(addCounter)
        
    #---
 
    return neighborsPerPred, np.array(neighborsRemoved), np.array(neighborsAdded)

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 22
def getNeighborsTest(binSize: int, # Size of the bins of values of `yPred` being grouped together.
                     yPred: np.ndarray, # 1-dimensional array of predicted values.
                     yPredTrain: np.ndarray, # 1-dimensional array of predicted train values.
                     # Dict containing the neighbors of all train samples. Keys are the train predictions.
                     neighborsDictTrain: dict, 
                     ):
    "Used to generate the neighboorhoods used by `LevelSetKDEx` to compute density estimations."
    
    duplicationDict = defaultdict(list)
    counterDict = defaultdict(int)
    
    for index, value in enumerate(yPredTrain):
        duplicationDict[value].append(index)
        counterDict[value] += 1
    
    yPredTrainUnique = np.sort(list(duplicationDict.keys()))
    yPredUnique = np.unique(yPred)
    
    yPredTrainUniqueRanking = {value: index for index, value in enumerate(yPredTrainUnique)}
    
    trainIndicesClosest = np.searchsorted(a = yPredTrainUnique, v = yPredUnique, side = 'right') - 1
    
    # Needed if any yPred value is lower than all yPredTrain values
    trainIndicesClosest = np.clip(a = trainIndicesClosest, a_min = 0, a_max = None) 
    yPredTrainClosest = yPredTrainUnique[trainIndicesClosest]
    
    #---
    
    neighborsPerPred = dict()

    for i, predCurrent in enumerate(yPredUnique):
        
        neighbors = deque(neighborsDictTrain[yPredTrainClosest[i]])
        
        neighborsMaxIndex = yPredTrainUniqueRanking[yPredTrain[neighbors[len(neighbors) - 1]]]
        
        distanceToMin = predCurrent - yPredTrain[neighbors[0]]
        
        # Check and Clean current neighborhood before starting the loop
        if len(neighbors) > binSize:
            
            checkNeeded = True
            while checkNeeded:
                
                distanceToMin = predCurrent - yPredTrain[neighbors[0]]
                distanceToMax = yPredTrain[neighbors[len(neighbors) - 1]] - predCurrent

                if distanceToMin > 0 and distanceToMin > distanceToMax:
                    countIdenticalValuesLeftSide = counterDict[yPredTrain[neighbors[0]]]

                    if len(neighbors) - countIdenticalValuesLeftSide >= binSize:
                        for p in range(countIdenticalValuesLeftSide):
                            neighbors.popleft()
                    else:
                        checkNeeded = False
                else:
                    checkNeeded = False

        #---

        distanceToMin = predCurrent - yPredTrain[neighbors[0]]
        distanceToMax = yPredTrain[neighbors[len(neighbors) - 1]] - predCurrent

        for k in range(neighborsMaxIndex + 1, len(yPredTrainUnique), 1):
            predNew = yPredTrainUnique[k]
            distance = predNew - predCurrent 
                
            if len(neighbors) < binSize:
                neighbors.extend(duplicationDict[predNew])
                distanceToMax = yPredTrain[neighbors[len(neighbors) - 1]] - predCurrent

            elif distance < distanceToMin:
                neighbors.extend(duplicationDict[predNew])
                
                for p in range(counterDict[yPredTrain[neighbors[0]]]): 
                    neighbors.popleft()

                distanceToMin = predCurrent - yPredTrain[neighbors[0]]
                distanceToMax = yPredTrain[neighbors[len(neighbors) - 1]] - predCurrent

            elif distance == distanceToMin:
                neighbors.extend(duplicationDict[predNew])
                distanceToMax = yPredTrain[neighbors[len(neighbors) - 1]] - predCurrent

            elif distance == distanceToMax:
                neighbors.extend(duplicationDict[predNew])

            else:
                break
        
        neighborsPerPred[predCurrent] = np.array(neighbors, dtype = 'uintc')
        
    #---
 
    return neighborsPerPred

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 24
# Setting 'efficientRAM = TRUE' is only necessary when there are roughly umore than 200k training observations.
# This setting causes the run-time of the algorithm to linearly depend on the 'binSize', which can become quite
# slow for bin-sizes above 10k.

def getKernelValues(yPred,
                    yPredTrain,
                    neighborsDictTest,
                    neighborsDictTrain,
                    neighborsRemoved,
                    neighborsAdded,
                    binSize,
                    efficientRAM = False):
    
    duplicationDict = defaultdict(list)
    counterDict = defaultdict(int)
    
    for index, value in enumerate(yPredTrain):
        duplicationDict[value].append(index)
        counterDict[value] += 1
    
    yPredTrainUnique = np.sort(list(neighborsDictTrain.keys()))
    trainIndicesClosest = np.searchsorted(a = yPredTrainUnique, v = yPred, side = 'right') - 1
    
    # Needed if any yPred value is lower than all yPredTrain values
    trainIndicesClosest = np.clip(a = trainIndicesClosest, a_min = 0, a_max = None)
    
    #---
    
    kernelValuesList = list()
    weightsDataList = list()
    
    for i in range(len(yPred)):
        
        trainIndexClosest = trainIndicesClosest[i]
        neighbors = neighborsDictTest[yPred[i]]
        sizeBin = len(neighbors)
        
        #---
        
        if trainIndexClosest + 1 <= len(yPredTrainUnique) - 1:
            neighborsTrainClosest = neighborsDictTrain[yPredTrainUnique[trainIndexClosest+1]]
            sharedNeighborsClosest = len(set(neighbors) & set(neighborsTrainClosest))
            
            if trainIndexClosest + 1 == len(yPredTrainUnique) - 1:
                kernelValuesRight = np.expand_dims(2 * sharedNeighborsClosest / (sizeBin + len(neighborsTrainClosest)), axis = 0)
                
            else:
                removeCum = np.concatenate([np.arange(1), neighborsRemoved[trainIndexClosest+2:len(yPredTrainUnique)]], axis = 0).cumsum()
                addCum = np.concatenate([np.arange(1), neighborsAdded[trainIndexClosest+2:len(yPredTrainUnique)]], axis = 0).cumsum()

                sharedNeighborsRight = np.clip(a = sharedNeighborsClosest - removeCum, a_min = 0, a_max = None)
                binSizesRight = len(neighborsTrainClosest) - removeCum + addCum

                kernelValuesRight = 2 * sharedNeighborsRight / (sizeBin + binSizesRight)
        
        else:
            kernelValuesRight = np.arange(0)
            
        #---
        
        # trainIndexClosest == -1 means that the test pred is lower than any train pred
        if trainIndexClosest >= 0:
            neighborsTrainClosest = neighborsDictTrain[yPredTrainUnique[trainIndexClosest]]
            sharedNeighborsClosest = len(set(neighbors) & set(neighborsTrainClosest))
            
            if trainIndexClosest == 0:
                kernelValuesLeft = np.expand_dims(2 * sharedNeighborsClosest / (sizeBin + len(neighborsTrainClosest)), axis = 0)
            
            else:
                neighborsRemovedFlip = np.flip(neighborsRemoved[1:trainIndexClosest+1])
                addCum = np.concatenate([np.arange(1), neighborsRemovedFlip], axis = 0).cumsum()

                neighborsAddedFlip = np.flip(neighborsAdded[1:trainIndexClosest+1])
                removeCum = np.concatenate([np.arange(1), neighborsAddedFlip], axis = 0).cumsum()

                sharedNeighborsLeft = np.clip(a = sharedNeighborsClosest - removeCum, a_min = 0, a_max = None)
                binSizesLeft = len(neighborsTrainClosest) - removeCum + addCum

                kernelValuesLeft = np.flip(2 * sharedNeighborsLeft / (sizeBin + binSizesLeft))
                    
        else:
            kernelValuesLeft = np.arange(0)

        #---
        
        kernelValuesUnique = np.concatenate([kernelValuesLeft, kernelValuesRight], axis = 0)
        
        if efficientRAM:
            weightsList = []
            indicesList = []

            for index in np.where(kernelValuesUnique > 0)[0]:
                indices = duplicationDict[yPredTrainUnique[index]]
                indicesList.extend(indices)

                weight = kernelValuesUnique[index]
                weightsList.extend([weight for i in range(len(indices))])
            
            weightsArray = np.array(weightsList) / sum(weightsList)            
            weightsDataList.append((weightsArray, np.array(indicesList, dtype = 'uintc')))
        
        else:       
            kernelValuesList.append(kernelValuesUnique)
    
    #---
    
    if efficientRAM:
        return weightsDataList
    
    else:
    
        kernelMatrixUnique = np.array(kernelValuesList)
        kernelMatrix = np.zeros(shape = (len(yPred), len(yPredTrain)))

        for index, predTrain in enumerate(yPredTrainUnique):
            kernelMatrix[:, duplicationDict[predTrain]] = kernelMatrixUnique[:, [index]]

        for i in range(len(yPred)):
            indices = np.where(kernelMatrix[i,:] > 0)[0]
            weights = kernelMatrix[i, indices]
            weights = weights / weights.sum()
            weightsDataList.append((weights, indices))

        return weightsDataList


# %% ../nbs/01_levelSetKDEx_univariate.ipynb 26
class LevelSetKDEx_clustering(BaseWeightsBasedEstimator, BaseLSx):
    """
    `LevelSetKDEx` turns any point forecasting model into an estimator of the underlying conditional density.
    The name 'LevelSet' stems from the fact that this approach interprets the values of the point forecasts
    as a similarity measure between samples. The point forecasts of the training samples are sorted and 
    recursively assigned to a bin until the size of the current bin reaches `binSize` many samples. Then
    a new bin is created and so on. For a new test sample we check into which bin its point prediction
    would have fallen and interpret the training samples of that bin as the empirical distribution function
    of this test sample.    
    """
    
    def __init__(self, 
                 estimator, # Model with a .fit and .predict-method (implementing the scikit-learn estimator interface).
                 nClusters: int=10, # Number of clusters to form as well as number of centroids to generate.
                 ):
        
        super(BaseEstimator, self).__init__(estimator = estimator)

        # nClusters must either be of type int, np.int64 or np.int32
        if isinstance(nClusters, (np.int32, np.int64)):
            nClusters = int(nClusters)
        
        elif not isinstance(nClusters, (int, np.int32, np.int64)):
            raise ValueError("'nClusters' must be an integer!")
                
        self.nClusters = nClusters

        self.yTrain = None
        self.yPredTrain = None
        self.kmeans = None
        self.clusterDict = None
        self.clusterSizes = None
        self.fitted = False
    
    #---
    
    def fit(self: LevelSetKDEx, 
            X: np.ndarray, # Feature matrix used by `estimator` to predict `y`.
            y: np.ndarray, # 1-dimensional target variable corresponding to the feature matrix `X`.
            ):
        """
        Fit `LevelSetKDEx` model by grouping the point predictions of the samples specified via `X`
        according to their value. Samples are recursively sorted into bins until each bin contains
        `binSize` many samples. For details, checkout the function `generateBins` which does the
        heavy lifting.
        """
        
        # Checks
        if self.nClusters is None:
            raise ValueError("'nClusters' must be specified to fit the LSx estimator!")
        
        # nClusters must either be of type int, np.int64 or np.int32
        if isinstance(self.nClusters, (np.int32, np.int64)):
            self.nClusters = int(self.nClusters)
        
        elif not isinstance(self.nClusters, (int, np.int32, np.int64)):
            raise ValueError("'nClusters' must be an integer!")
        
        # Check if nClusters is positive
        if self.nClusters <= 0:
            raise ValueError("'nClusters' must be positive!")
        
        if X.shape[0] != y.shape[0]:
            raise ValueError("'X' and 'y' must contain the same number of samples!")
        
        # IMPORTANT: In case 'y' is given as a pandas.Series, we can potentially run into indexing 
        # problems later on.
        if isinstance(y, pd.Series):
            y = y.ravel()
        
        #---
        
        try:
            yPred = self.estimator.predict(X)
            
        except NotFittedError:
            try:
                self.estimator.fit(X = X, y = y)                
            except:
                raise ValueError("Couldn't fit 'estimator' with user specified 'X' and 'y'!")
            else:
                yPred = self.estimator.predict(X)
        
        #---
        
        # Reshape yPred to 2D array with shape (nSamples, 1) and convert to float32.
        yPredMod = yPred.reshape(-1, 1).astype(np.float32)
        
        kmeans = faiss.Kmeans(d = 1, k = self.nClusters)
        kmeans.train(yPredMod)
        # self.centers = kmeans.centroids
        
        clusterAssignments = kmeans.assign(yPredMod)[1]
        
        # Get the cluster labels for each sample in yPred as a dict with keys being the cluster labels
        # and values being the indices of the samples in yPred that belong to that cluster.
        # Create another dict with keys being the cluster labels and values being the size of each cluster.
        clusterDict = defaultdict(list)
        clusterSizes = defaultdict(int)

        for index, cluster in enumerate(clusterAssignments):
            clusterDict[cluster].append(index)
            clusterSizes[cluster] += 1
        
        clusterSizes = pd.Series(clusterSizes)
       
        self.yTrain = y
        self.yPredTrain = yPred
        self.kmeans = kmeans
        self.clusterDict = clusterDict
        self.clusterSizes = clusterSizes
        self.fitted = True
        
    #---
    
    def getWeights(self, 
                   X: np.ndarray, # Feature matrix for which conditional density estimates are computed.
                   # Specifies structure of the returned density estimates. One of: 
                   # 'all', 'onlyPositiveWeights', 'summarized', 'cumDistribution', 'cumDistributionSummarized'
                   outputType: str='onlyPositiveWeights', 
                   # Optional. List with length X.shape[0]. Values are multiplied to the estimated 
                   # density of each sample for scaling purposes.
                   scalingList: list=None, 
                   ) -> list: # List whose elements are the conditional density estimates for the samples specified by `X`.
        
        # __annotations__ = BaseWeightsBasedEstimator.getWeights.__annotations__
        __doc__ = BaseWeightsBasedEstimator.getWeights.__doc__
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
        
        #---
        
        yPred = self.estimator.predict(X)
        # Reshape yPred to 2D array with shape (nSamples, 1) and convert to float32.
        yPred = yPred.reshape(-1, 1).astype(np.float32)

        # Get cluster labels of yPred
        clusterLabels = self.kmeans.assign(yPred)[1]
        
        #---
        
        weightsDataList = [(np.repeat(1 / self.clusterSizes[cluster], self.clusterSizes[cluster]), np.array(self.clusterDict[cluster], dtype = 'uintc')) 
                            for cluster in clusterLabels]

        weightsDataList = restructureWeightsDataList(weightsDataList = weightsDataList, 
                                                     outputType = outputType, 
                                                     y = self.yTrain,
                                                     scalingList = scalingList,
                                                     equalWeights = True)
        
        return weightsDataList
    

# %% ../nbs/01_levelSetKDEx_univariate.ipynb 27
class LevelSetKDEx_clustering2(BaseWeightsBasedEstimator, BaseLSx):
    """
    `LevelSetKDEx` turns any point forecasting model into an estimator of the underlying conditional density.
    The name 'LevelSet' stems from the fact that this approach interprets the values of the point forecasts
    as a similarity measure between samples. The point forecasts of the training samples are sorted and 
    recursively assigned to a bin until the size of the current bin reaches `binSize` many samples. Then
    a new bin is created and so on. For a new test sample we check into which bin its point prediction
    would have fallen and interpret the training samples of that bin as the empirical distribution function
    of this test sample.    
    """
    
    def __init__(self, 
                 estimator, # Model with a .fit and .predict-method (implementing the scikit-learn estimator interface).
                 nClusters: int=10, # Number of clusters to form as well as number of centroids to generate.
                 ):
        
        super(BaseEstimator, self).__init__(estimator = estimator)

        # nClusters must either be of type int, np.int64 or np.int32
        if isinstance(nClusters, (np.int32, np.int64)):
            nClusters = int(nClusters)
        
        elif not isinstance(nClusters, (int, np.int32, np.int64)):
            raise ValueError("'nClusters' must be an integer!")
                
        self.nClusters = nClusters

        self.yTrain = None
        self.yPredTrain = None
        self.kmeans = None
        self.clusterDict = None
        self.clusterSizes = None
        self.fitted = False
    
    #---
    
    def fit(self: LevelSetKDEx, 
            X: np.ndarray, # Feature matrix used by `estimator` to predict `y`.
            y: np.ndarray, # 1-dimensional target variable corresponding to the feature matrix `X`.
            ):
        """
        Fit `LevelSetKDEx` model by grouping the point predictions of the samples specified via `X`
        according to their value. Samples are recursively sorted into bins until each bin contains
        `binSize` many samples. For details, checkout the function `generateBins` which does the
        heavy lifting.
        """
        
        # Checks
        if self.nClusters is None:
            raise ValueError("'nClusters' must be specified to fit the LSx estimator!")
        
        # nClusters must either be of type int, np.int64 or np.int32
        if isinstance(self.nClusters, (np.int32, np.int64)):
            self.nClusters = int(self.nClusters)
        
        elif not isinstance(self.nClusters, (int, np.int32, np.int64)):
            raise ValueError("'nClusters' must be an integer!")

        # Check if nClusters is positive
        if self.nClusters <= 0:
            raise ValueError("'nClusters' must be positive!")
        
        if X.shape[0] != y.shape[0]:
            raise ValueError("'X' and 'y' must contain the same number of samples!")
        
        # IMPORTANT: In case 'y' is given as a pandas.Series, we can potentially run into indexing 
        # problems later on.
        if isinstance(y, pd.Series):
            y = y.ravel()
        
        #---
        
        try:
            yPred = self.estimator.predict(X)
            
        except NotFittedError:
            try:
                self.estimator.fit(X = X, y = y)                
            except:
                raise ValueError("Couldn't fit 'estimator' with user specified 'X' and 'y'!")
            else:
                yPred = self.estimator.predict(X)
        
        #---

        # Cluster yPred into nClusters many clusters
        kmeans = KMeans(n_clusters = self.nClusters, random_state = 0, n_init = 10).fit(yPred.reshape(-1, 1))

        # Get the cluster labels for each sample in yPred as a dict with keys being the cluster labels
        # and values being the indices of the samples in yPred that belong to that cluster.
        # Create another dict with keys being the cluster labels and values being the size of each cluster.
        clusterDict = defaultdict(list)
        clusterSizes = defaultdict(int)

        for index, cluster in enumerate(kmeans.labels_):
            clusterDict[cluster].append(index)
            clusterSizes[cluster] += 1
       
        self.yTrain = y
        self.yPredTrain = yPred
        self.kmeans = kmeans
        self.clusterDict = clusterDict
        self.clusterSizes = clusterSizes
        self.fitted = True
        
    #---
    
    def getWeights(self, 
                   X: np.ndarray, # Feature matrix for which conditional density estimates are computed.
                   # Specifies structure of the returned density estimates. One of: 
                   # 'all', 'onlyPositiveWeights', 'summarized', 'cumDistribution', 'cumDistributionSummarized'
                   outputType: str='onlyPositiveWeights', 
                   # Optional. List with length X.shape[0]. Values are multiplied to the estimated 
                   # density of each sample for scaling purposes.
                   scalingList: list=None, 
                   ) -> list: # List whose elements are the conditional density estimates for the samples specified by `X`.
        
        # __annotations__ = BaseWeightsBasedEstimator.getWeights.__annotations__
        __doc__ = BaseWeightsBasedEstimator.getWeights.__doc__
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
        
        #---
        
        yPred = self.estimator.predict(X)
        
        # Get cluster labels of yPred
        clusterLabels = self.kmeans.predict(yPred.reshape(-1, 1))
        
        #---
        
        weightsDataList = [(np.repeat(1 / self.clusterSizes[label], self.clusterSizes[label]), np.array(self.clusterDict[label], dtype = 'uintc')) 
                            for label in clusterLabels]

        weightsDataList = restructureWeightsDataList(weightsDataList = weightsDataList, 
                                                     outputType = outputType, 
                                                     y = self.yTrain,
                                                     scalingList = scalingList,
                                                     equalWeights = True)
        
        return weightsDataList
    
