"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveIfPrimitive = exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
/**
 * The level of the message that the rule applies.
 *
 * @stability stable
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel[NagMessageLevel["WARN"] = 0] = "WARN";
    NagMessageLevel[NagMessageLevel["ERROR"] = 1] = "ERROR";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule sets.
 *
 * @stability stable
 */
class NagPack {
    /**
     * @stability stable
     */
    constructor(props) {
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
    }
    /**
     * @stability stable
     */
    applyRule(params) {
        try {
            if (!this.ignoreRule(params.ignores, params.ruleId) &&
                !params.rule(params.node)) {
                const message = this.createMessage(params.ruleId, params.info, params.explanation);
                if (params.level == NagMessageLevel.ERROR) {
                    monocdk_1.Annotations.of(params.node).addError(message);
                }
                else if (params.level == NagMessageLevel.WARN) {
                    monocdk_1.Annotations.of(params.node).addWarning(message);
                }
            }
        }
        catch (error) {
            if (!this.ignoreRule(params.ignores, VALIDATION_FAILURE_ID)) {
                const information = `'${params.ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored
     * @param ignores ignores listed in cdkNag metadata
     * @param ruleId the id of the rule to ignore
     * @returns boolean
     */
    ignoreRule(ignores, ruleId) {
        try {
            if (ignores) {
                for (let ignore of ignores) {
                    if (ignore.id &&
                        ignore.reason &&
                        JSON.stringify(ignore.reason).length >= 10) {
                        if (ignore.id == ruleId) {
                            return true;
                        }
                    }
                    else {
                        throw Error();
                    }
                }
            }
            return false;
        }
        catch {
            throw Error('Improperly formatted cdk_nag rule suppression detected. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.');
        }
    }
    /**
     * The message to output to the console when a rule is triggered
     * @param ruleId the id of the rule
     * @param info why the rule was triggered
     * @param explanation why the rule exists
     * @returns string
     */
    createMessage(ruleId, info, explanation) {
        let message = `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}` : message;
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "0.0.125" };
/**
 * Use in cases where a primitive value must be known to pass a rule
 * https://developer.mozilla.org/en-US/docs/Glossary/Primitive
 * @param node the CfnResource to check
 * @param parameter the value to attempt to resolve
 * @returns Return a value if resolves to a primitive data type, otherwise throw an error.
 */
function resolveIfPrimitive(node, parameter) {
    const resolvedValue = monocdk_1.Stack.of(node).resolve(parameter);
    if (resolvedValue === Object(resolvedValue)) {
        throw Error(`The parameter resolved to to a non-primitive value "${JSON.stringify(resolvedValue)}", therefore the rule could not be validated.`);
    }
    else {
        return resolvedValue;
    }
}
exports.resolveIfPrimitive = resolveIfPrimitive;
//# sourceMappingURL=data:application/json;base64,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