"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const common_1 = require("../../../common");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return false;
                    }
                }
            }
        }
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = common_1.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = common_1.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = common_1.resolveIfPrimitive(node, rule.fromPort);
    const toPort = common_1.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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