"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const sns_1 = require("./rules/sns");
/**
 * Check for NIST 800-53 compliance.
 *
 * Based on the NIST 800-53 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053Checks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            this.checkCloudWatch(node, ignores);
            this.checkCodeBuild(node, ignores);
            this.checkDMS(node, ignores);
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkEFS(node, ignores);
            this.checkElastiCache(node, ignores);
            this.checkELB(node, ignores);
            this.checkEMR(node, ignores);
            this.checkIAM(node, ignores);
            this.checkLambda(node, ignores);
            this.checkOpenSearch(node, ignores);
            this.checkRDS(node, ignores);
            this.checkRedshift(node, ignores);
            this.checkS3(node, ignores);
            this.checkSageMaker(node, ignores);
            this.checkSNS(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-APIGWCacheEnabledAndEncrypted',
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: SC-13, SC-28).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: common_1.NagMessageLevel.ERROR,
            rule: apigw_1.nist80053APIGWCacheEnabledAndEncrypted,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-APIGWExecutionLoggingEnabled',
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: common_1.NagMessageLevel.ERROR,
            rule: apigw_1.nist80053APIGWExecutionLoggingEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-AutoScalingHealthChecks',
            info: 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control IDs: SC-5).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.',
            level: common_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.nist80053AutoScalingHealthChecks,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-CloudTrailCloudWatchLogsEnabled',
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.nist80053CloudTrailCloudWatchLogsEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CloudTrailEncryptionEnabled',
            info: 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: AU-9).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.nist80053CloudTrailEncryptionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CloudTrailLogFileValidationEnabled',
            info: 'The trail does not have log file validation enabled - (Control ID: AC-6).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.nist80053CloudTrailLogFileValidationEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-CloudWatchAlarmAction',
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.nist80053CloudWatchAlarmAction,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CloudWatchLogGroupEncrypted',
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9, SC-13, SC-28).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.nist80053CloudWatchLogGroupEncrypted,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-CodeBuildCheckEnvVars',
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: AC-6, IA-5(7), SA-3(a)).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: codebuild_1.nist80053CodeBuildCheckEnvVars,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CodeBuildURLCheck',
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control ID: SA-3(a)).',
            explanation: 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: common_1.NagMessageLevel.ERROR,
            rule: codebuild_1.nist80053CodeBuildURLCheck,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-DMSReplicationNotPublic',
            info: 'The DMS replication instance is public - (Control IDs: AC-3).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: common_1.NagMessageLevel.ERROR,
            rule: dms_1.nist80053DMSReplicationNotPublic,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-DynamoDBPITREnabled',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: common_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.nist80053DynamoDBPITREnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed',
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4, SC-7, SC-7(3)).",
            explanation: 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: common_1.NagMessageLevel.WARN,
            rule: ec2_1.nist80053EC2CheckDefaultSecurityGroupClosed,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckDetailedMonitoring',
            info: 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).',
            explanation: 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckDetailedMonitoring,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckInsideVPC',
            info: 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckInsideVPC,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckNoPublicIPs',
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ',
            explanation: 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckNoPublicIPs,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckSSHRestricted',
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckSSHRestricted,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckCommonPortsRestricted',
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckCommonPortsRestricted,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-EFSEncrypted',
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: common_1.NagMessageLevel.ERROR,
            rule: efs_1.nist80053EFSEncrypted,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup',
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elasticache_1.nist80053ElastiCacheRedisClusterAutomaticBackup,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled',
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control ID: AC-17(2)).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ALBHttpDropInvalidHeaderEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ALBHttpToHttpsRedirection',
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ALBHttpToHttpsRedirection,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBCrossZoneBalancing',
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: SC-5, CP-10).',
            explanation: 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBCrossZoneBalancing,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBDeletionProtectionEnabled',
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CM-2, CP-10).',
            explanation: 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBDeletionProtectionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBListenersUseSSLOrHTTPS',
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-23).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBListenersUseSSLOrHTTPS,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBLoggingEnabled',
            info: 'The ELB does not have logging enabled - (Control ID: AU-2(a)(d)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBLoggingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBUseACMCerts',
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBUseACMCerts,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-EMRKerberosEnabled',
            info: 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.',
            level: common_1.NagMessageLevel.ERROR,
            rule: emr_1.nist80053EMRKerberosEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-IAMGroupMembership',
            info: 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, and AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMGroupMembership,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-IAMNoInlinePolicy',
            info: 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMNoInlinePolicy,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess',
            info: 'The IAM policy grants admin access - (Control IDs: AC-2(1), AC-2(j), AC-3, AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMPolicyNoStatementsWithAdminAccess,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-IAMUserNoPolicies',
            info: 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMUserNoPolicies,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-LambdaFunctionsInsideVPC',
            info: 'The Lambda function is not VPC enabled - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: common_1.NagMessageLevel.ERROR,
            rule: lambda_1.nist80053LambdaFunctionsInsideVPC,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-OpenSearchEncryptedAtRest',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: common_1.NagMessageLevel.ERROR,
            rule: opensearch_1.nist80053OpenSearchEncryptedAtRest,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-OpenSearchNodeToNodeEncrypted',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: SC-7, SC-8, SC-8(1)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: common_1.NagMessageLevel.ERROR,
            rule: opensearch_1.nist80053OpenSearchNodeToNodeEncrypted,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-OpenSearchRunningWithinVPC',
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: common_1.NagMessageLevel.ERROR,
            rule: opensearch_1.nist80053OpenSearchRunningWithinVPC,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Amazon RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-RDSEnhancedMonitoringEnabled',
            info: 'The RDS DB instance does not enhanced monitoring enabled - (Control ID: CA-7(a)(b)).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSEnhancedMonitoringEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstanceBackupEnabled',
            info: 'The RDS DB instance does not have backups enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstanceBackupEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstanceDeletionProtectionEnabled',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control ID: SC-5).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstanceDeletionProtectionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstanceMultiAZSupport',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: CP-10, SC-5, SC-36).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstanceMultiAZSupport,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstancePublicAccess',
            info: 'The RDS DB instance allows public access - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstancePublicAccess,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSLoggingEnabled',
            info: 'The RDS DB instance does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSLoggingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSStorageEncrypted',
            info: 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSStorageEncrypted,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-RedshiftClusterConfiguration',
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: common_1.NagMessageLevel.ERROR,
            rule: redshift_1.nist80053RedshiftClusterConfiguration,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RedshiftClusterPublicAccess',
            info: 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: common_1.NagMessageLevel.ERROR,
            rule: redshift_1.nist80053RedshiftClusterPublicAccess,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketDefaultLockEnabled',
            info: 'The S3 Bucket does not have object lock enabled - (Control ID: SC-28).',
            explanation: 'Because sensitive data can exist at rest in S3 buckets, enforce object locks at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketDefaultLockEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketLoggingEnabled',
            info: 'The S3 Bucket does not have server access logs enabled - (Control IDs: AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketLoggingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketPublicReadProhibited',
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketPublicReadProhibited,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketPublicWriteProhibited',
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketPublicWriteProhibited,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketReplicationEnabled',
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketReplicationEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketServerSideEncryptionEnabled',
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketServerSideEncryptionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketVersioningEnabled',
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: CP-10, SI-12).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketVersioningEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-SageMakerEndpointKMS',
            info: 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.nist80053SageMakerEndpointKMS,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisabled',
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.nist80053SageMakerNotebookDirectInternetAccessDisabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-SageMakerNotebookKMS',
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.nist80053SageMakerNotebookKMS,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node, ignores) {
        this.applyRule({
            ruleId: 'NIST.800.53-SNSEncryptedKMS',
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sns_1.nist80053SNSEncryptedKMS,
            ignores: ignores,
            node: node,
        });
    }
}
exports.NIST80053Checks = NIST80053Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053Checks[_a] = { fqn: "monocdk-nag.NIST80053Checks", version: "0.0.125" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmlzdC04MDAtNTMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvTklTVC04MDAtNTMvbmlzdC04MDAtNTMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0VBR0U7QUFFRixxQ0FBa0Q7QUFDbEQsc0NBQXFEO0FBQ3JELHlDQUd1QjtBQUN2QixxREFBdUU7QUFDdkUsbURBSTRCO0FBQzVCLG1EQUc0QjtBQUM1QixpREFHMkI7QUFDM0IscUNBQStEO0FBQy9ELCtDQUFnRTtBQUNoRSxxQ0FPcUI7QUFDckIscUNBQW9EO0FBQ3BELHFEQUFzRjtBQUN0RixxQ0FRcUI7QUFDckIscUNBQTBEO0FBQzFELHFDQUtxQjtBQUNyQiwyQ0FBbUU7QUFDbkUsbURBSTRCO0FBQzVCLHFDQVFxQjtBQUNyQiwrQ0FHMEI7QUFDMUIsbUNBUW9CO0FBQ3BCLGlEQUkyQjtBQUMzQixxQ0FBdUQ7Ozs7Ozs7O0FBR3ZELE1BQWEsZUFBZ0IsU0FBUSxnQkFBTzs7Ozs7O0lBQ25DLEtBQUssQ0FBQyxJQUFnQjs7UUFDM0IsSUFBSSxJQUFJLFlBQVkscUJBQVcsRUFBRTtZQUMvQixvQ0FBb0M7WUFDcEMsTUFBTSxPQUFPLFNBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsMENBQUUsaUJBQWlCLENBQUM7WUFDL0QsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDL0IsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyQyxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNwQyxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNwQyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNuQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUM3QixJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNsQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUM3QixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUM3QixJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2hDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3BDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2xDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzVCLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ25DLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQzlCO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxVQUFVLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ2hELElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsMkNBQTJDO1lBQ25ELElBQUksRUFBRSxrSEFBa0g7WUFDeEgsV0FBVyxFQUNULGdOQUFnTjtZQUNsTixLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw4Q0FBc0M7WUFDNUMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDBDQUEwQztZQUNsRCxJQUFJLEVBQUUsK0hBQStIO1lBQ3JJLFdBQVcsRUFDVCxrS0FBa0s7WUFDcEssS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkNBQXFDO1lBQzNDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxnQkFBZ0IsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDdEQsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxxQ0FBcUM7WUFDN0MsSUFBSSxFQUFFLHlIQUF5SDtZQUMvSCxXQUFXLEVBQ1QsNEtBQTRLO1lBQzlLLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDhDQUFnQztZQUN0QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZUFBZSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUNyRCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDZDQUE2QztZQUNyRCxJQUFJLEVBQUUsOExBQThMO1lBQ3BNLFdBQVcsRUFDVCwyS0FBMks7WUFDN0ssS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUscURBQXdDO1lBQzlDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSx5Q0FBeUM7WUFDakQsSUFBSSxFQUFFLHVGQUF1RjtZQUM3RixXQUFXLEVBQ1QsaUlBQWlJO1lBQ25JLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlEQUFvQztZQUMxQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsZ0RBQWdEO1lBQ3hELElBQUksRUFBRSwyRUFBMkU7WUFDakYsV0FBVyxFQUNULGliQUFpYjtZQUNuYixLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx3REFBMkM7WUFDakQsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGVBQWUsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDckQsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxtQ0FBbUM7WUFDM0MsSUFBSSxFQUFFLG9PQUFvTztZQUMxTyxXQUFXLEVBQ1QsZ1FBQWdRO1lBQ2xRLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDJDQUE4QjtZQUNwQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUseUNBQXlDO1lBQ2pELElBQUksRUFBRSxvR0FBb0c7WUFDMUcsV0FBVyxFQUNULDZHQUE2RztZQUMvRyxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxpREFBb0M7WUFDMUMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGNBQWMsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDcEQsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxtQ0FBbUM7WUFDM0MsSUFBSSxFQUFFLDZMQUE2TDtZQUNuTSxXQUFXLEVBQ1QsOElBQThJO1lBQ2hKLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDBDQUE4QjtZQUNwQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsK0JBQStCO1lBQ3ZDLElBQUksRUFBRSxxSUFBcUk7WUFDM0ksV0FBVyxFQUNULDhOQUE4TjtZQUNoTyxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxzQ0FBMEI7WUFDaEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDOUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxxQ0FBcUM7WUFDN0MsSUFBSSxFQUFFLCtEQUErRDtZQUNyRSxXQUFXLEVBQ1QsK0dBQStHO1lBQ2pILEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHNDQUFnQztZQUN0QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssYUFBYSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUNuRCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLGlDQUFpQztZQUN6QyxJQUFJLEVBQUUseUdBQXlHO1lBQy9HLFdBQVcsRUFDVCwrRUFBK0U7WUFDakYsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsdUNBQTRCO1lBQ2xDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzlDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsZ0RBQWdEO1lBQ3hELElBQUksRUFBRSwyR0FBMkc7WUFDakgsV0FBVyxFQUNULHNVQUFzVTtZQUN4VSxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxJQUFJO1lBQzNCLElBQUksRUFBRSxpREFBMkM7WUFDakQsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLHdDQUF3QztZQUNoRCxJQUFJLEVBQUUsaUhBQWlIO1lBQ3ZILFdBQVcsRUFDVCxxSEFBcUg7WUFDdkgsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUseUNBQW1DO1lBQ3pDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSwrQkFBK0I7WUFDdkMsSUFBSSxFQUFFLDRFQUE0RTtZQUNsRixXQUFXLEVBQ1Qsa0tBQWtLO1lBQ3BLLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGdDQUEwQjtZQUNoQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsaUNBQWlDO1lBQ3pDLElBQUksRUFBRSxnSEFBZ0g7WUFDdEgsV0FBVyxFQUNULDJHQUEyRztZQUM3RyxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxrQ0FBNEI7WUFDbEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLG1DQUFtQztZQUMzQyxJQUFJLEVBQUUseUZBQXlGO1lBQy9GLFdBQVcsRUFDVCwrSEFBK0g7WUFDakksS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsb0NBQThCO1lBQ3BDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSwyQ0FBMkM7WUFDbkQsSUFBSSxFQUFFLHNKQUFzSjtZQUM1SixXQUFXLEVBQ1QsbVBBQW1QO1lBQ3JQLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDRDQUFzQztZQUM1QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUM5QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDBCQUEwQjtZQUNsQyxJQUFJLEVBQUUsaUZBQWlGO1lBQ3ZGLFdBQVcsRUFDVCw0SUFBNEk7WUFDOUksS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkJBQXFCO1lBQzNCLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxnQkFBZ0IsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDdEQsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxvREFBb0Q7WUFDNUQsSUFBSSxFQUFFLDhIQUE4SDtZQUNwSSxXQUFXLEVBQ1QsOEpBQThKO1lBQ2hLLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDZEQUErQztZQUNyRCxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUM5QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDZDQUE2QztZQUNyRCxJQUFJLEVBQUUsc0ZBQXNGO1lBQzVGLFdBQVcsRUFDVCxrTEFBa0w7WUFDcEwsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsOENBQXdDO1lBQzlDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSx1Q0FBdUM7WUFDL0MsSUFBSSxFQUFFLGdJQUFnSTtZQUN0SSxXQUFXLEVBQ1QsbU9BQW1PO1lBQ3JPLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdDQUFrQztZQUN4QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsbUNBQW1DO1lBQzNDLElBQUksRUFBRSxzR0FBc0c7WUFDNUcsV0FBVyxFQUNULCtIQUErSDtZQUNqSSxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxvQ0FBOEI7WUFDcEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDBDQUEwQztZQUNsRCxJQUFJLEVBQUUsOEZBQThGO1lBQ3BHLFdBQVcsRUFDVCw4SkFBOEo7WUFDaEssS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkNBQXFDO1lBQzNDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSx1Q0FBdUM7WUFDL0MsSUFBSSxFQUFFLG9JQUFvSTtZQUMxSSxXQUFXLEVBQ1QsMkZBQTJGO1lBQzdGLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdDQUFrQztZQUN4QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsK0JBQStCO1lBQ3ZDLElBQUksRUFBRSxtRUFBbUU7WUFDekUsV0FBVyxFQUNULDhWQUE4VjtZQUNoVyxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBMEI7WUFDaEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDRCQUE0QjtZQUNwQyxJQUFJLEVBQUUsaUpBQWlKO1lBQ3ZKLFdBQVcsRUFDVCxtUkFBbVI7WUFDclIsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkJBQXVCO1lBQzdCLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzlDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsZ0NBQWdDO1lBQ3hDLElBQUksRUFBRSw2RkFBNkY7WUFDbkcsV0FBVyxFQUNULDBMQUEwTDtZQUM1TCxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxpQ0FBMkI7WUFDakMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDOUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxnQ0FBZ0M7WUFDeEMsSUFBSSxFQUFFLGlHQUFpRztZQUN2RyxXQUFXLEVBQ1QscVNBQXFTO1lBQ3ZTLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlDQUEyQjtZQUNqQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsK0JBQStCO1lBQ3ZDLElBQUksRUFBRSw4RUFBOEU7WUFDcEYsV0FBVyxFQUNULGdMQUFnTDtZQUNsTCxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBMEI7WUFDaEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLGtEQUFrRDtZQUMxRCxJQUFJLEVBQUUsbUZBQW1GO1lBQ3pGLFdBQVcsRUFDVCx1WkFBdVo7WUFDelosS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsbURBQTZDO1lBQ25ELE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSwrQkFBK0I7WUFDdkMsSUFBSSxFQUFFLDJGQUEyRjtZQUNqRyxXQUFXLEVBQ1QsNElBQTRJO1lBQzlJLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGdDQUEwQjtZQUNoQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ssV0FBVyxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUNqRCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLHNDQUFzQztZQUM5QyxJQUFJLEVBQUUsOEVBQThFO1lBQ3BGLFdBQVcsRUFDVCxrS0FBa0s7WUFDcEssS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMENBQWlDO1lBQ3ZDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxlQUFlLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ3JELElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsdUNBQXVDO1lBQy9DLElBQUksRUFBRSx1R0FBdUc7WUFDN0csV0FBVyxFQUNULGtLQUFrSztZQUNwSyxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwrQ0FBa0M7WUFDeEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDJDQUEyQztZQUNuRCxJQUFJLEVBQUUsbUhBQW1IO1lBQ3pILFdBQVcsRUFDVCw4SkFBOEo7WUFDaEssS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsbURBQXNDO1lBQzVDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSx3Q0FBd0M7WUFDaEQsSUFBSSxFQUFFLGlHQUFpRztZQUN2RyxXQUFXLEVBQ1QsK0VBQStFO1lBQ2pGLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGdEQUFtQztZQUN6QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUM5QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLDBDQUEwQztZQUNsRCxJQUFJLEVBQUUsc0ZBQXNGO1lBQzVGLFdBQVcsRUFDVCw4SkFBOEo7WUFDaEssS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkNBQXFDO1lBQzNDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxzQ0FBc0M7WUFDOUMsSUFBSSxFQUFFLDJGQUEyRjtZQUNqRyxXQUFXLEVBQ1QsMEZBQTBGO1lBQzVGLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHVDQUFpQztZQUN2QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsa0RBQWtEO1lBQzFELElBQUksRUFBRSwwR0FBMEc7WUFDaEgsV0FBVyxFQUNULDRTQUE0UztZQUM5UyxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxtREFBNkM7WUFDbkQsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLHVDQUF1QztZQUMvQyxJQUFJLEVBQUUsNEdBQTRHO1lBQ2xILFdBQVcsRUFDVCx1Z0JBQXVnQjtZQUN6Z0IsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0NBQWtDO1lBQ3hDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxxQ0FBcUM7WUFDN0MsSUFBSSxFQUFFLGdHQUFnRztZQUN0RyxXQUFXLEVBQ1QsbUlBQW1JO1lBQ3JJLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHNDQUFnQztZQUN0QyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsK0JBQStCO1lBQ3ZDLElBQUksRUFBRSx1SUFBdUk7WUFDN0ksV0FBVyxFQUNULDBQQUEwUDtZQUM1UCxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBMEI7WUFDaEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLGlDQUFpQztZQUN6QyxJQUFJLEVBQUUseUdBQXlHO1lBQy9HLFdBQVcsRUFDVCx3SUFBd0k7WUFDMUksS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsa0NBQTRCO1lBQ2xDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxhQUFhLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ25ELElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsMENBQTBDO1lBQ2xELElBQUksRUFBRSxpSkFBaUo7WUFDdkosV0FBVyxFQUNULDZTQUE2UztZQUMvUyxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnREFBcUM7WUFDM0MsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLHlDQUF5QztZQUNqRCxJQUFJLEVBQUUsdUdBQXVHO1lBQzdHLFdBQVcsRUFDVCw2SEFBNkg7WUFDL0gsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsK0NBQW9DO1lBQzFDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxPQUFPLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzdDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsd0NBQXdDO1lBQ2hELElBQUksRUFBRSx3RUFBd0U7WUFDOUUsV0FBVyxFQUNULGlIQUFpSDtZQUNuSCxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx3Q0FBbUM7WUFDekMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLG9DQUFvQztZQUM1QyxJQUFJLEVBQUUsaUhBQWlIO1lBQ3ZILFdBQVcsRUFDVCxrY0FBa2M7WUFDcGMsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsb0NBQStCO1lBQ3JDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSwwQ0FBMEM7WUFDbEQsSUFBSSxFQUFFLCtLQUErSztZQUNyTCxXQUFXLEVBQ1Qsb0ZBQW9GO1lBQ3RGLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDBDQUFxQztZQUMzQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsMkNBQTJDO1lBQ25ELElBQUksRUFBRSxnTEFBZ0w7WUFDdEwsV0FBVyxFQUNULG9GQUFvRjtZQUN0RixLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwyQ0FBc0M7WUFDNUMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLHdDQUF3QztZQUNoRCxJQUFJLEVBQUUsd0dBQXdHO1lBQzlHLFdBQVcsRUFDVCx3UUFBd1E7WUFDMVEsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0NBQW1DO1lBQ3pDLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxpREFBaUQ7WUFDekQsSUFBSSxFQUFFLDJIQUEySDtZQUNqSSxXQUFXLEVBQ1QsNkdBQTZHO1lBQy9HLEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlEQUE0QztZQUNsRCxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsdUNBQXVDO1lBQy9DLElBQUksRUFBRSwrRUFBK0U7WUFDckYsV0FBVyxFQUNULGdOQUFnTjtZQUNsTixLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx1Q0FBa0M7WUFDeEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGNBQWMsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDcEQsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLE1BQU0sRUFBRSxrQ0FBa0M7WUFDMUMsSUFBSSxFQUFFLHVGQUF1RjtZQUM3RixXQUFXLEVBQ1Qsc0hBQXNIO1lBQ3hILEtBQUssRUFBRSx3QkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHlDQUE2QjtZQUNuQyxPQUFPLEVBQUUsT0FBTztZQUNoQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsMkRBQTJEO1lBQ25FLElBQUksRUFBRSw0SEFBNEg7WUFDbEksV0FBVyxFQUNULDhHQUE4RztZQUNoSCxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxrRUFBc0Q7WUFDNUQsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsTUFBTSxFQUFFLGtDQUFrQztZQUMxQyxJQUFJLEVBQUUsdUZBQXVGO1lBQzdGLFdBQVcsRUFDVCxzSEFBc0g7WUFDeEgsS0FBSyxFQUFFLHdCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUseUNBQTZCO1lBQ25DLE9BQU8sRUFBRSxPQUFPO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzlDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixNQUFNLEVBQUUsNkJBQTZCO1lBQ3JDLElBQUksRUFBRSxtRkFBbUY7WUFDekYsV0FBVyxFQUNULHNIQUFzSDtZQUN4SCxLQUFLLEVBQUUsd0JBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw4QkFBd0I7WUFDOUIsT0FBTyxFQUFFLE9BQU87WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDOztBQXp1QkgsMENBMHVCQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuXG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgSUNvbnN0cnVjdCB9IGZyb20gJ21vbm9jZGsnO1xuaW1wb3J0IHsgTmFnUGFjaywgTmFnTWVzc2FnZUxldmVsIH0gZnJvbSAnLi4vY29tbW9uJztcbmltcG9ydCB7XG4gIG5pc3Q4MDA1M0FQSUdXQ2FjaGVFbmFibGVkQW5kRW5jcnlwdGVkLFxuICBuaXN0ODAwNTNBUElHV0V4ZWN1dGlvbkxvZ2dpbmdFbmFibGVkLFxufSBmcm9tICcuL3J1bGVzL2FwaWd3JztcbmltcG9ydCB7IG5pc3Q4MDA1M0F1dG9TY2FsaW5nSGVhbHRoQ2hlY2tzIH0gZnJvbSAnLi9ydWxlcy9hdXRvc2NhbGluZyc7XG5pbXBvcnQge1xuICBuaXN0ODAwNTNDbG91ZFRyYWlsQ2xvdWRXYXRjaExvZ3NFbmFibGVkLFxuICBuaXN0ODAwNTNDbG91ZFRyYWlsRW5jcnlwdGlvbkVuYWJsZWQsXG4gIG5pc3Q4MDA1M0Nsb3VkVHJhaWxMb2dGaWxlVmFsaWRhdGlvbkVuYWJsZWQsXG59IGZyb20gJy4vcnVsZXMvY2xvdWR0cmFpbCc7XG5pbXBvcnQge1xuICBuaXN0ODAwNTNDbG91ZFdhdGNoQWxhcm1BY3Rpb24sXG4gIG5pc3Q4MDA1M0Nsb3VkV2F0Y2hMb2dHcm91cEVuY3J5cHRlZCxcbn0gZnJvbSAnLi9ydWxlcy9jbG91ZHdhdGNoJztcbmltcG9ydCB7XG4gIG5pc3Q4MDA1M0NvZGVCdWlsZENoZWNrRW52VmFycyxcbiAgbmlzdDgwMDUzQ29kZUJ1aWxkVVJMQ2hlY2ssXG59IGZyb20gJy4vcnVsZXMvY29kZWJ1aWxkJztcbmltcG9ydCB7IG5pc3Q4MDA1M0RNU1JlcGxpY2F0aW9uTm90UHVibGljIH0gZnJvbSAnLi9ydWxlcy9kbXMnO1xuaW1wb3J0IHsgbmlzdDgwMDUzRHluYW1vREJQSVRSRW5hYmxlZCB9IGZyb20gJy4vcnVsZXMvZHluYW1vZGInO1xuaW1wb3J0IHtcbiAgbmlzdDgwMDUzRUMyQ2hlY2tDb21tb25Qb3J0c1Jlc3RyaWN0ZWQsXG4gIG5pc3Q4MDA1M0VDMkNoZWNrRGVmYXVsdFNlY3VyaXR5R3JvdXBDbG9zZWQsXG4gIG5pc3Q4MDA1M0VDMkNoZWNrRGV0YWlsZWRNb25pdG9yaW5nLFxuICBuaXN0ODAwNTNFQzJDaGVja0luc2lkZVZQQyxcbiAgbmlzdDgwMDUzRUMyQ2hlY2tOb1B1YmxpY0lQcyxcbiAgbmlzdDgwMDUzRUMyQ2hlY2tTU0hSZXN0cmljdGVkLFxufSBmcm9tICcuL3J1bGVzL2VjMic7XG5pbXBvcnQgeyBuaXN0ODAwNTNFRlNFbmNyeXB0ZWQgfSBmcm9tICcuL3J1bGVzL2Vmcyc7XG5pbXBvcnQgeyBuaXN0ODAwNTNFbGFzdGlDYWNoZVJlZGlzQ2x1c3RlckF1dG9tYXRpY0JhY2t1cCB9IGZyb20gJy4vcnVsZXMvZWxhc3RpY2FjaGUnO1xuaW1wb3J0IHtcbiAgbmlzdDgwMDUzQUxCSHR0cERyb3BJbnZhbGlkSGVhZGVyRW5hYmxlZCxcbiAgbmlzdDgwMDUzQUxCSHR0cFRvSHR0cHNSZWRpcmVjdGlvbixcbiAgbmlzdDgwMDUzRUxCQ3Jvc3Nab25lQmFsYW5jaW5nLFxuICBuaXN0ODAwNTNFTEJEZWxldGlvblByb3RlY3Rpb25FbmFibGVkLFxuICBuaXN0ODAwNTNFTEJMaXN0ZW5lcnNVc2VTU0xPckhUVFBTLFxuICBuaXN0ODAwNTNFTEJMb2dnaW5nRW5hYmxlZCxcbiAgbmlzdDgwMDUzRUxCVXNlQUNNQ2VydHMsXG59IGZyb20gJy4vcnVsZXMvZWxiJztcbmltcG9ydCB7IG5pc3Q4MDA1M0VNUktlcmJlcm9zRW5hYmxlZCB9IGZyb20gJy4vcnVsZXMvZW1yJztcbmltcG9ydCB7XG4gIG5pc3Q4MDA1M0lBTUdyb3VwTWVtYmVyc2hpcCxcbiAgbmlzdDgwMDUzSUFNTm9JbmxpbmVQb2xpY3ksXG4gIG5pc3Q4MDA1M0lBTVBvbGljeU5vU3RhdGVtZW50c1dpdGhBZG1pbkFjY2VzcyxcbiAgbmlzdDgwMDUzSUFNVXNlck5vUG9saWNpZXMsXG59IGZyb20gJy4vcnVsZXMvaWFtJztcbmltcG9ydCB7IG5pc3Q4MDA1M0xhbWJkYUZ1bmN0aW9uc0luc2lkZVZQQyB9IGZyb20gJy4vcnVsZXMvbGFtYmRhJztcbmltcG9ydCB7XG4gIG5pc3Q4MDA1M09wZW5TZWFyY2hFbmNyeXB0ZWRBdFJlc3QsXG4gIG5pc3Q4MDA1M09wZW5TZWFyY2hOb2RlVG9Ob2RlRW5jcnlwdGVkLFxuICBuaXN0ODAwNTNPcGVuU2VhcmNoUnVubmluZ1dpdGhpblZQQyxcbn0gZnJvbSAnLi9ydWxlcy9vcGVuc2VhcmNoJztcbmltcG9ydCB7XG4gIG5pc3Q4MDA1M1JEU0VuaGFuY2VkTW9uaXRvcmluZ0VuYWJsZWQsXG4gIG5pc3Q4MDA1M1JEU0luc3RhbmNlQmFja3VwRW5hYmxlZCxcbiAgbmlzdDgwMDUzUkRTSW5zdGFuY2VEZWxldGlvblByb3RlY3Rpb25FbmFibGVkLFxuICBuaXN0ODAwNTNSRFNJbnN0YW5jZU11bHRpQVpTdXBwb3J0LFxuICBuaXN0ODAwNTNSRFNJbnN0YW5jZVB1YmxpY0FjY2VzcyxcbiAgbmlzdDgwMDUzUkRTTG9nZ2luZ0VuYWJsZWQsXG4gIG5pc3Q4MDA1M1JEU1N0b3JhZ2VFbmNyeXB0ZWQsXG59IGZyb20gJy4vcnVsZXMvcmRzJztcbmltcG9ydCB7XG4gIG5pc3Q4MDA1M1JlZHNoaWZ0Q2x1c3RlckNvbmZpZ3VyYXRpb24sXG4gIG5pc3Q4MDA1M1JlZHNoaWZ0Q2x1c3RlclB1YmxpY0FjY2Vzcyxcbn0gZnJvbSAnLi9ydWxlcy9yZWRzaGlmdCc7XG5pbXBvcnQge1xuICBuaXN0ODAwNTNTM0J1Y2tldERlZmF1bHRMb2NrRW5hYmxlZCxcbiAgbmlzdDgwMDUzUzNCdWNrZXRMb2dnaW5nRW5hYmxlZCxcbiAgbmlzdDgwMDUzUzNCdWNrZXRQdWJsaWNSZWFkUHJvaGliaXRlZCxcbiAgbmlzdDgwMDUzUzNCdWNrZXRQdWJsaWNXcml0ZVByb2hpYml0ZWQsXG4gIG5pc3Q4MDA1M1MzQnVja2V0UmVwbGljYXRpb25FbmFibGVkLFxuICBuaXN0ODAwNTNTM0J1Y2tldFNlcnZlclNpZGVFbmNyeXB0aW9uRW5hYmxlZCxcbiAgbmlzdDgwMDUzUzNCdWNrZXRWZXJzaW9uaW5nRW5hYmxlZCxcbn0gZnJvbSAnLi9ydWxlcy9zMyc7XG5pbXBvcnQge1xuICBuaXN0ODAwNTNTYWdlTWFrZXJFbmRwb2ludEtNUyxcbiAgbmlzdDgwMDUzU2FnZU1ha2VyTm90ZWJvb2tEaXJlY3RJbnRlcm5ldEFjY2Vzc0Rpc2FibGVkLFxuICBuaXN0ODAwNTNTYWdlTWFrZXJOb3RlYm9va0tNUyxcbn0gZnJvbSAnLi9ydWxlcy9zYWdlbWFrZXInO1xuaW1wb3J0IHsgbmlzdDgwMDUzU05TRW5jcnlwdGVkS01TIH0gZnJvbSAnLi9ydWxlcy9zbnMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE5JU1Q4MDA1M0NoZWNrcyBleHRlbmRzIE5hZ1BhY2sge1xuICBwdWJsaWMgdmlzaXQobm9kZTogSUNvbnN0cnVjdCk6IHZvaWQge1xuICAgIGlmIChub2RlIGluc3RhbmNlb2YgQ2ZuUmVzb3VyY2UpIHtcbiAgICAgIC8vIEdldCBpZ25vcmVzIG1ldGFkYXRhIGlmIGl0IGV4aXN0c1xuICAgICAgY29uc3QgaWdub3JlcyA9IG5vZGUuZ2V0TWV0YWRhdGEoJ2Nka19uYWcnKT8ucnVsZXNfdG9fc3VwcHJlc3M7XG4gICAgICB0aGlzLmNoZWNrQVBJR1cobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrQXV0b1NjYWxpbmcobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrQ2xvdWRUcmFpbChub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tDbG91ZFdhdGNoKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0NvZGVCdWlsZChub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tETVMobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrRHluYW1vREIobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrRUMyKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0VGUyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tFbGFzdGlDYWNoZShub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tFTEIobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrRU1SKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0lBTShub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tMYW1iZGEobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrT3BlblNlYXJjaChub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tSRFMobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrUmVkc2hpZnQobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrUzMobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrU2FnZU1ha2VyKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja1NOUyhub2RlLCBpZ25vcmVzKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQVBJIEdhdGV3YXkgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0FQSUdXKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1BUElHV0NhY2hlRW5hYmxlZEFuZEVuY3J5cHRlZCcsXG4gICAgICBpbmZvOiAnVGhlIEFQSSBHYXRld2F5IHN0YWdlIGRvZXMgbm90IGhhdmUgY2FjaGluZyBlbmFibGVkIGFuZCBlbmNyeXB0ZWQgZm9yIGFsbCBtZXRob2RzIC0gKENvbnRyb2wgSURzOiBTQy0xMywgU0MtMjgpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgXCJUbyBoZWxwIHByb3RlY3QgZGF0YSBhdCByZXN0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFQSSBHYXRld2F5IHN0YWdlJ3MgY2FjaGUuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGJlIGNhcHR1cmVkIGZvciB0aGUgQVBJIG1ldGhvZCwgZW5hYmxlIGVuY3J5cHRpb24gYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLlwiLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0FQSUdXQ2FjaGVFbmFibGVkQW5kRW5jcnlwdGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtQVBJR1dFeGVjdXRpb25Mb2dnaW5nRW5hYmxlZCcsXG4gICAgICBpbmZvOiAnVGhlIEFQSSBHYXRld2F5IHN0YWdlIGRvZXMgbm90IGhhdmUgZXhlY3V0aW9uIGxvZ2dpbmcgZW5hYmxlZCBmb3IgYWxsIG1ldGhvZHMgLSAoQ29udHJvbCBJRHM6IEFVLTIoYSkoZCksIEFVLTMsIEFVLTEyKGEpKGMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBUEkgR2F0ZXdheSBsb2dnaW5nIGRpc3BsYXlzIGRldGFpbGVkIHZpZXdzIG9mIHVzZXJzIHdobyBhY2Nlc3NlZCB0aGUgQVBJIGFuZCB0aGUgd2F5IHRoZXkgYWNjZXNzZWQgdGhlIEFQSS4gVGhpcyBpbnNpZ2h0IGVuYWJsZXMgdmlzaWJpbGl0eSBvZiB1c2VyIGFjdGl2aXRpZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNBUElHV0V4ZWN1dGlvbkxvZ2dpbmdFbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQXV0byBTY2FsaW5nIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tBdXRvU2NhbGluZyhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KSB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtQXV0b1NjYWxpbmdIZWFsdGhDaGVja3MnLFxuICAgICAgaW5mbzogJ1RoZSBBdXRvIFNjYWxpbmcgZ3JvdXAgdXRpbGl6ZXMgYSBsb2FkIGJhbGFuY2VyIGFuZCBkb2VzIG5vdCBoYXZlIGFuIEVMQiBoZWFsdGggY2hlY2sgY29uZmlndXJlZCAtIChDb250cm9sIElEczogU0MtNSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIEVsYXN0aWMgTG9hZCBCYWxhbmNlciAoRUxCKSBoZWFsdGggY2hlY2tzIGZvciBBbWF6b24gRWxhc3RpYyBDb21wdXRlIENsb3VkIChBbWF6b24gRUMyKSBBdXRvIFNjYWxpbmcgZ3JvdXBzIHN1cHBvcnQgbWFpbnRlbmFuY2Ugb2YgYWRlcXVhdGUgY2FwYWNpdHkgYW5kIGF2YWlsYWJpbGl0eS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0F1dG9TY2FsaW5nSGVhbHRoQ2hlY2tzLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQ2xvdWRUcmFpbCBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQ2xvdWRUcmFpbChub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtQ2xvdWRUcmFpbENsb3VkV2F0Y2hMb2dzRW5hYmxlZCcsXG4gICAgICBpbmZvOiAnVGhlIHRyYWlsIGRvZXMgbm90IGhhdmUgQ2xvdWRXYXRjaCBsb2dzIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IEFDLTIoNCksIEFDLTIoZyksIEFVLTIoYSkoZCksIEFVLTMsIEFVLTYoMSkoMyksIEFVLTcoMSksIEFVLTEyKGEpKGMpLCBDQS03KGEpKGIpLCBTSS00KDIpLCBTSS00KDQpLCBTSS00KDUpLCBTSS00KGEpKGIpKGMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdVc2UgQW1hem9uIENsb3VkV2F0Y2ggdG8gY2VudHJhbGx5IGNvbGxlY3QgYW5kIG1hbmFnZSBsb2cgZXZlbnQgYWN0aXZpdHkuIEluY2x1c2lvbiBvZiBBV1MgQ2xvdWRUcmFpbCBkYXRhIHByb3ZpZGVzIGRldGFpbHMgb2YgQVBJIGNhbGwgYWN0aXZpdHkgd2l0aGluIHlvdXIgQVdTIGFjY291bnQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNDbG91ZFRyYWlsQ2xvdWRXYXRjaExvZ3NFbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtQ2xvdWRUcmFpbEVuY3J5cHRpb25FbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgdHJhaWwgZG9lcyBub3QgaGF2ZSBhIEtNUyBrZXkgSUQgb3IgaGF2ZSBlbmNyeXB0aW9uIGVuYWJsZWQgLSAoQ29udHJvbCBJRDogQVUtOSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBtYXkgZXhpc3QgYW5kIHRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgZm9yIHlvdXIgQVdTIENsb3VkVHJhaWwgdHJhaWxzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzQ2xvdWRUcmFpbEVuY3J5cHRpb25FbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtQ2xvdWRUcmFpbExvZ0ZpbGVWYWxpZGF0aW9uRW5hYmxlZCcsXG4gICAgICBpbmZvOiAnVGhlIHRyYWlsIGRvZXMgbm90IGhhdmUgbG9nIGZpbGUgdmFsaWRhdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSUQ6IEFDLTYpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1V0aWxpemUgQVdTIENsb3VkVHJhaWwgbG9nIGZpbGUgdmFsaWRhdGlvbiB0byBjaGVjayB0aGUgaW50ZWdyaXR5IG9mIENsb3VkVHJhaWwgbG9ncy4gTG9nIGZpbGUgdmFsaWRhdGlvbiBoZWxwcyBkZXRlcm1pbmUgaWYgYSBsb2cgZmlsZSB3YXMgbW9kaWZpZWQgb3IgZGVsZXRlZCBvciB1bmNoYW5nZWQgYWZ0ZXIgQ2xvdWRUcmFpbCBkZWxpdmVyZWQgaXQuIFRoaXMgZmVhdHVyZSBpcyBidWlsdCB1c2luZyBpbmR1c3RyeSBzdGFuZGFyZCBhbGdvcml0aG1zOiBTSEEtMjU2IGZvciBoYXNoaW5nIGFuZCBTSEEtMjU2IHdpdGggUlNBIGZvciBkaWdpdGFsIHNpZ25pbmcuIFRoaXMgbWFrZXMgaXQgY29tcHV0YXRpb25hbGx5IGluZmVhc2libGUgdG8gbW9kaWZ5LCBkZWxldGUgb3IgZm9yZ2UgQ2xvdWRUcmFpbCBsb2cgZmlsZXMgd2l0aG91dCBkZXRlY3Rpb24uJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNDbG91ZFRyYWlsTG9nRmlsZVZhbGlkYXRpb25FbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQ2xvdWRXYXRjaCBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQ2xvdWRXYXRjaChub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtQ2xvdWRXYXRjaEFsYXJtQWN0aW9uJyxcbiAgICAgIGluZm86ICdUaGUgQ2xvdWRXYXRjaCBhbGFybSBkb2VzIG5vdCBoYXZlIGF0IGxlYXN0IG9uZSBhbGFybSBhY3Rpb24sIG9uZSBJTlNVRkZJQ0lFTlRfREFUQSBhY3Rpb24sIG9yIG9uZSBPSyBhY3Rpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogQUMtMig0KSwgQVUtNigxKSgzKSwgQVUtNygxKSwgQ0EtNyhhKShiKSwgSVItNCgxKSwgU0ktNCgyKSwgU0ktNCg0KSwgU0ktNCg1KSwgU0ktNChhKShiKShjKSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQW1hem9uIENsb3VkV2F0Y2ggYWxhcm1zIGFsZXJ0IHdoZW4gYSBtZXRyaWMgYnJlYWNoZXMgdGhlIHRocmVzaG9sZCBmb3IgYSBzcGVjaWZpZWQgbnVtYmVyIG9mIGV2YWx1YXRpb24gcGVyaW9kcy4gVGhlIGFsYXJtIHBlcmZvcm1zIG9uZSBvciBtb3JlIGFjdGlvbnMgYmFzZWQgb24gdGhlIHZhbHVlIG9mIHRoZSBtZXRyaWMgb3IgZXhwcmVzc2lvbiByZWxhdGl2ZSB0byBhIHRocmVzaG9sZCBvdmVyIGEgbnVtYmVyIG9mIHRpbWUgcGVyaW9kcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0Nsb3VkV2F0Y2hBbGFybUFjdGlvbixcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUNsb3VkV2F0Y2hMb2dHcm91cEVuY3J5cHRlZCcsXG4gICAgICBpbmZvOiAnVGhlIENsb3VkV2F0Y2ggTG9nIEdyb3VwIGlzIG5vdCBlbmNyeXB0ZWQgd2l0aCBhbiBBV1MgS01TIGtleSAtIChDb250cm9sIElEczogQVUtOSwgU0MtMTMsIFNDLTI4KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUbyBoZWxwIHByb3RlY3Qgc2Vuc2l0aXZlIGRhdGEgYXQgcmVzdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBBbWF6b24gQ2xvdWRXYXRjaCBMb2cgR3JvdXBzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzQ2xvdWRXYXRjaExvZ0dyb3VwRW5jcnlwdGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQ29kZUJ1aWxkIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tDb2RlQnVpbGQobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSkge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUNvZGVCdWlsZENoZWNrRW52VmFycycsXG4gICAgICBpbmZvOiAnVGhlIENvZGVCdWlsZCBlbnZpcm9ubWVudCBzdG9yZXMgc2Vuc2l0aXZlIGNyZWRlbnRpYWxzIChzdWNoIGFzIEFXU19BQ0NFU1NfS0VZX0lEIGFuZC9vciBBV1NfU0VDUkVUX0FDQ0VTU19LRVkpIGFzIHBsYWludGV4dCBlbnZpcm9ubWVudCB2YXJpYWJsZXMgLSAoQ29udHJvbCBJRHM6IEFDLTYsIElBLTUoNyksIFNBLTMoYSkpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0RvIG5vdCBzdG9yZSB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dC4gU3RvcmluZyB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dCBsZWFkcyB0byB1bmludGVuZGVkIGRhdGEgZXhwb3N1cmUgYW5kIHVuYXV0aG9yaXplZCBhY2Nlc3MuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNDb2RlQnVpbGRDaGVja0VudlZhcnMsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1Db2RlQnVpbGRVUkxDaGVjaycsXG4gICAgICBpbmZvOiAnVGhlIENvZGVCdWlsZCBwcm9qZWN0IHdoaWNoIHV0aWxpemVzIGVpdGhlciBhIEdpdEh1YiBvciBCaXRCdWNrZXQgc291cmNlIHJlcG9zaXRvcnkgZG9lcyBub3QgdXRpbGl6ZSBPQVVUSCAtIChDb250cm9sIElEOiBTQS0zKGEpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdPQVVUSCBpcyB0aGUgbW9zdCBzZWN1cmUgbWV0aG9kIG9mIGF1dGhlbnRpY2F0aW5nIHlvdXIgQ29kZUJ1aWxkIGFwcGxpY2F0aW9uLiBVc2UgT0F1dGggaW5zdGVhZCBvZiBwZXJzb25hbCBhY2Nlc3MgdG9rZW5zIG9yIGEgdXNlciBuYW1lIGFuZCBwYXNzd29yZCB0byBncmFudCBhdXRob3JpemF0aW9uIGZvciBhY2Nlc3NpbmcgR2l0SHViIG9yIEJpdGJ1Y2tldCByZXBvc2l0b3JpZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNDb2RlQnVpbGRVUkxDaGVjayxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIERNUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRE1TKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1ETVNSZXBsaWNhdGlvbk5vdFB1YmxpYycsXG4gICAgICBpbmZvOiAnVGhlIERNUyByZXBsaWNhdGlvbiBpbnN0YW5jZSBpcyBwdWJsaWMgLSAoQ29udHJvbCBJRHM6IEFDLTMpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0RNUyByZXBsaWNhdGlvbiBpbnN0YW5jZXMgY2FuIGNvbnRhaW4gc2Vuc2l0aXZlIGluZm9ybWF0aW9uIGFuZCBhY2Nlc3MgY29udHJvbCBpcyByZXF1aXJlZCBmb3Igc3VjaCBhY2NvdW50cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0RNU1JlcGxpY2F0aW9uTm90UHVibGljLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRHluYW1vREIgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0R5bmFtb0RCKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1EeW5hbW9EQlBJVFJFbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgRHluYW1vREIgdGFibGUgZG9lcyBub3QgaGF2ZSBQb2ludC1pbi10aW1lIFJlY292ZXJ5IGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IENQLTkoYiksIENQLTEwLCBTSS0xMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIHJlY292ZXJ5IG1haW50YWlucyBjb250aW51b3VzIGJhY2t1cHMgb2YgeW91ciB0YWJsZSBmb3IgdGhlIGxhc3QgMzUgZGF5cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0R5bmFtb0RCUElUUkVuYWJsZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBFQzIgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0VDMihub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtRUMyQ2hlY2tEZWZhdWx0U2VjdXJpdHlHcm91cENsb3NlZCcsXG4gICAgICBpbmZvOiBcIlRoZSBWUEMncyBkZWZhdWx0IHNlY3VyaXR5IGdyb3VwIGFsbG93cyBpbmJvdW5kIG9yIG91dGJvdW5kIHRyYWZmaWMgLSAoQ29udHJvbCBJRHM6IEFDLTQsIFNDLTcsIFNDLTcoMykpLlwiLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdXaGVuIGNyZWF0aW5nIGEgVlBDIHRocm91Z2ggQ2xvdWRGb3JtYXRpb24sIHRoZSBkZWZhdWx0IHNlY3VyaXR5IGdyb3VwIHdpbGwgYWx3YXlzIGJlIG9wZW4uIFRoZXJlZm9yZSBpdCBpcyBpbXBvcnRhbnQgdG8gYWx3YXlzIGNsb3NlIHRoZSBkZWZhdWx0IHNlY3VyaXR5IGdyb3VwIGFmdGVyIHN0YWNrIGNyZWF0aW9uIHdoZW5ldmVyIGEgVlBDIGlzIGNyZWF0ZWQuIFJlc3RyaWN0aW5nIGFsbCB0aGUgdHJhZmZpYyBvbiB0aGUgZGVmYXVsdCBzZWN1cml0eSBncm91cCBoZWxwcyBpbiByZXN0cmljdGluZyByZW1vdGUgYWNjZXNzIHRvIHlvdXIgQVdTIHJlc291cmNlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgICAgcnVsZTogbmlzdDgwMDUzRUMyQ2hlY2tEZWZhdWx0U2VjdXJpdHlHcm91cENsb3NlZCxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUVDMkNoZWNrRGV0YWlsZWRNb25pdG9yaW5nJyxcbiAgICAgIGluZm86ICdUaGUgRUMyIGluc3RhbmNlIGRvZXMgbm90IGhhdmUgZGV0YWlsZWQgbW9uaXRvcmluZyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBDQS03KGEpKGIpLCBTSS00KDIpLCBTSS00KGEpKGIpKGMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdEZXRhaWxlZCBtb25pdG9yaW5nIHByb3ZpZGVzIGFkZGl0aW9uYWwgbW9uaXRvcmluZyBpbmZvcm1hdGlvbiAoc3VjaCBhcyAxLW1pbnV0ZSBwZXJpb2QgZ3JhcGhzKSBvbiB0aGUgQVdTIGNvbnNvbGUuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNFQzJDaGVja0RldGFpbGVkTW9uaXRvcmluZyxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUVDMkNoZWNrSW5zaWRlVlBDJyxcbiAgICAgIGluZm86ICdUaGUgRUMyIGluc3RhbmNlIGlzIG5vdCB3aXRoaW4gYSBWUEMgLSAoQ29udHJvbCBJRHM6IEFDLTQsIFNDLTcsIFNDLTcoMykpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0JlY2F1c2Ugb2YgdGhlaXIgbG9naWNhbCBpc29sYXRpb24sIGRvbWFpbnMgdGhhdCByZXNpZGUgd2l0aGluIGFuIEFtYXpvbiBWUEMgaGF2ZSBhbiBleHRyYSBsYXllciBvZiBzZWN1cml0eSB3aGVuIGNvbXBhcmVkIHRvIGRvbWFpbnMgdGhhdCB1c2UgcHVibGljIGVuZHBvaW50cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0VDMkNoZWNrSW5zaWRlVlBDLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtRUMyQ2hlY2tOb1B1YmxpY0lQcycsXG4gICAgICBpbmZvOiAnVGhlIEVDMiBpbnN0YW5jZSBpcyBhc3NvY2lhdGVkIHdpdGggYSBwdWJsaWMgSVAgYWRkcmVzcyAtIChDb250cm9sIElEczogQUMtNCwgQUMtNiwgQUMtMjEoYiksIFNDLTcsIFNDLTcoMykpLiAnLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBbWF6b24gRUMyIGluc3RhbmNlcyBjYW4gY29udGFpbiBzZW5zaXRpdmUgaW5mb3JtYXRpb24gYW5kIGFjY2VzcyBjb250cm9sIGlzIHJlcXVpcmVkIGZvciBzdWNoIHJlc291cmNlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0VDMkNoZWNrTm9QdWJsaWNJUHMsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1FQzJDaGVja1NTSFJlc3RyaWN0ZWQnLFxuICAgICAgaW5mbzogJ1RoZSBTZWN1cml0eSBHcm91cCBhbGxvd3MgdW5yZXN0cmljdGVkIFNTSCBhY2Nlc3MgLSAoQ29udHJvbCBJRHM6IEFDLTQsIFNDLTcsIFNDLTcoMykpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ05vdCBhbGxvd2luZyBpbmdyZXNzIChvciByZW1vdGUpIHRyYWZmaWMgZnJvbSAwLjAuMC4wLzAgb3IgOjovMCB0byBwb3J0IDIyIG9uIHlvdXIgcmVzb3VyY2VzIGhlbHBzIHRvIHJlc3RyaWN0IHJlbW90ZSBhY2Nlc3MuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNFQzJDaGVja1NTSFJlc3RyaWN0ZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1FQzJDaGVja0NvbW1vblBvcnRzUmVzdHJpY3RlZCcsXG4gICAgICBpbmZvOiAnVGhlIEVDMiBpbnN0YW5jZSBhbGxvd3MgdW5yZXN0cmljdGVkIGluYm91bmQgSVB2NCBUQ1AgdHJhZmZpYyBvbiBjb21tb24gcG9ydHMgKDIwLCAyMSwgMzM4OSwgMzMwNiwgNDMzMykgLSAoQ29udHJvbCBJRHM6IEFDLTQsIENNLTIsIFNDLTcsIFNDLTcoMykpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ05vdCByZXN0cmljdGluZyBhY2Nlc3MgdG8gcG9ydHMgdG8gdHJ1c3RlZCBzb3VyY2VzIGNhbiBsZWFkIHRvIGF0dGFja3MgYWdhaW5zdCB0aGUgYXZhaWxhYmlsaXR5LCBpbnRlZ3JpdHkgYW5kIGNvbmZpZGVudGlhbGl0eSBvZiBzeXN0ZW1zLiBCeSBkZWZhdWx0LCBjb21tb24gcG9ydHMgd2hpY2ggc2hvdWxkIGJlIHJlc3RyaWN0ZWQgaW5jbHVkZSBwb3J0IG51bWJlcnMgMjAsIDIxLCAzMzg5LCAzMzA2LCBhbmQgNDMzMy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0VDMkNoZWNrQ29tbW9uUG9ydHNSZXN0cmljdGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRUZTIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tFRlMobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSkge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUVGU0VuY3J5cHRlZCcsXG4gICAgICBpbmZvOiAnVGhlIEVGUyBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gYXQgcmVzdCBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBTQy0xMywgU0MtMjgpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGFuZCB0byBoZWxwIHByb3RlY3QgZGF0YSBhdCByZXN0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBFbGFzdGljIEZpbGUgU3lzdGVtIChFRlMpLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzRUZTRW5jcnlwdGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRWxhc3RpQ2FjaGUgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0VsYXN0aUNhY2hlKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1FbGFzdGlDYWNoZVJlZGlzQ2x1c3RlckF1dG9tYXRpY0JhY2t1cCcsXG4gICAgICBpbmZvOiAnVGhlIEVsYXN0aUNhY2hlIFJlZGlzIGNsdXN0ZXIgZG9lcyBub3QgcmV0YWluIGF1dG9tYXRpYyBiYWNrdXBzIGZvciBhdCBsZWFzdCAxNSBkYXlzIC0gKENvbnRyb2wgSURzOiBDUC05KGIpLCBDUC0xMCwgU0ktMTIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0F1dG9tYXRpYyBiYWNrdXBzIGNhbiBoZWxwIGd1YXJkIGFnYWluc3QgZGF0YSBsb3NzLiBJZiBhIGZhaWx1cmUgb2NjdXJzLCB5b3UgY2FuIGNyZWF0ZSBhIG5ldyBjbHVzdGVyLCB3aGljaCByZXN0b3JlcyB5b3VyIGRhdGEgZnJvbSB0aGUgbW9zdCByZWNlbnQgYmFja3VwLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzRWxhc3RpQ2FjaGVSZWRpc0NsdXN0ZXJBdXRvbWF0aWNCYWNrdXAsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBFbGFzdGljIExvYWQgQmFsYW5jZXIgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0VMQihub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtQUxCSHR0cERyb3BJbnZhbGlkSGVhZGVyRW5hYmxlZCcsXG4gICAgICBpbmZvOiAnVGhlIEFMQiBkb2VzIG5vdCBoYXZlIGludmFsaWQgSFRUUCBoZWFkZXIgZHJvcHBpbmcgZW5hYmxlZCAtIChDb250cm9sIElEOiBBQy0xNygyKSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRW5zdXJlIHRoYXQgeW91ciBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VycyAoQUxCKSBhcmUgY29uZmlndXJlZCB0byBkcm9wIGh0dHAgaGVhZGVycy4gQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QsIGVuYWJsZSBlbmNyeXB0aW9uIGluIHRyYW5zaXQgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0FMQkh0dHBEcm9wSW52YWxpZEhlYWRlckVuYWJsZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1BTEJIdHRwVG9IdHRwc1JlZGlyZWN0aW9uJyxcbiAgICAgIGluZm86IFwiVGhlIEFMQidzIEhUVFAgbGlzdGVuZXJzIGFyZSBub3QgY29uZmlndXJlZCB0byByZWRpcmVjdCB0byBIVFRQUyAtIChDb250cm9sIElEczogQUMtMTcoMiksIFNDLTcsIFNDLTgsIFNDLTgoMSksIFNDLTEzLCBTQy0yMykuXCIsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RvIGhlbHAgcHJvdGVjdCBkYXRhIGluIHRyYW5zaXQsIGVuc3VyZSB0aGF0IHlvdXIgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlciBhdXRvbWF0aWNhbGx5IHJlZGlyZWN0cyB1bmVuY3J5cHRlZCBIVFRQIHJlcXVlc3RzIHRvIEhUVFBTLiBCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCwgZW5hYmxlIGVuY3J5cHRpb24gaW4gdHJhbnNpdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzQUxCSHR0cFRvSHR0cHNSZWRpcmVjdGlvbixcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUVMQkNyb3NzWm9uZUJhbGFuY2luZycsXG4gICAgICBpbmZvOiAnVGhlIENMQiBkb2VzIG5vdCBiYWxhbmNlIHRyYWZmaWMgYmV0d2VlbiBhdCBsZWFzdCAyIEF2YWlsYWJpbGl0eSBab25lcyAtIChDb250cm9sIElEczogU0MtNSwgQ1AtMTApLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBjcm9zcy16b25lIGxvYWQgYmFsYW5jaW5nIHJlZHVjZXMgdGhlIG5lZWQgdG8gbWFpbnRhaW4gZXF1aXZhbGVudCBudW1iZXJzIG9mIGluc3RhbmNlcyBpbiBlYWNoIGVuYWJsZWQgYXZhaWxhYmlsaXR5IHpvbmUuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNFTEJDcm9zc1pvbmVCYWxhbmNpbmcsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1FTEJEZWxldGlvblByb3RlY3Rpb25FbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgQUxCLCBOTEIsIG9yIEdMQiBkb2VzIG5vdCBoYXZlIGRlbGV0aW9uIHByb3RlY3Rpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogQ00tMiwgQ1AtMTApLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzZSB0aGlzIGZlYXR1cmUgdG8gcHJldmVudCB5b3VyIGxvYWQgYmFsYW5jZXIgZnJvbSBiZWluZyBhY2NpZGVudGFsbHkgb3IgbWFsaWNpb3VzbHkgZGVsZXRlZCwgd2hpY2ggY2FuIGxlYWQgdG8gbG9zcyBvZiBhdmFpbGFiaWxpdHkgZm9yIHlvdXIgYXBwbGljYXRpb25zLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzRUxCRGVsZXRpb25Qcm90ZWN0aW9uRW5hYmxlZCxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUVMQkxpc3RlbmVyc1VzZVNTTE9ySFRUUFMnLFxuICAgICAgaW5mbzogJ1RoZSBDTEIgZG9lcyBub3QgcmVzdHJpY3QgaXRzIGxpc3RlbmVycyB0byBvbmx5IHRoZSBTU0wgYW5kIEhUVFBTIHByb3RvY29scyAtIChDb250cm9sIElEczogQUMtMTcoMiksIFNDLTcsIFNDLTgsIFNDLTgoMSksIFNDLTIzKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCwgZW5hYmxlIGVuY3J5cHRpb24gaW4gdHJhbnNpdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzRUxCTGlzdGVuZXJzVXNlU1NMT3JIVFRQUyxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUVMQkxvZ2dpbmdFbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgRUxCIGRvZXMgbm90IGhhdmUgbG9nZ2luZyBlbmFibGVkIC0gKENvbnRyb2wgSUQ6IEFVLTIoYSkoZCkpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgXCJFbGFzdGljIExvYWQgQmFsYW5jaW5nIGFjdGl2aXR5IGlzIGEgY2VudHJhbCBwb2ludCBvZiBjb21tdW5pY2F0aW9uIHdpdGhpbiBhbiBlbnZpcm9ubWVudC4gRW5zdXJlIEVMQiBsb2dnaW5nIGlzIGVuYWJsZWQuIFRoZSBjb2xsZWN0ZWQgZGF0YSBwcm92aWRlcyBkZXRhaWxlZCBpbmZvcm1hdGlvbiBhYm91dCByZXF1ZXN0cyBzZW50IHRvIFRoZSBFTEIuIEVhY2ggbG9nIGNvbnRhaW5zIGluZm9ybWF0aW9uIHN1Y2ggYXMgdGhlIHRpbWUgdGhlIHJlcXVlc3Qgd2FzIHJlY2VpdmVkLCB0aGUgY2xpZW50J3MgSVAgYWRkcmVzcywgbGF0ZW5jaWVzLCByZXF1ZXN0IHBhdGhzLCBhbmQgc2VydmVyIHJlc3BvbnNlcy5cIixcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNFTEJMb2dnaW5nRW5hYmxlZCxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUVMQlVzZUFDTUNlcnRzJyxcbiAgICAgIGluZm86ICdUaGUgQ0xCIGRvZXMgbm90IHV0aWxpemUgYW4gU1NMIGNlcnRpZmljYXRlIHByb3ZpZGVkIGJ5IEFDTSAoQW1hem9uIENlcnRpZmljYXRlIE1hbmFnZXIpIC0gKENvbnRyb2wgSURzOiBBQy0xNygyKSwgU0MtNywgU0MtOCwgU0MtOCgxKSwgU0MtMTMpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGFuZCB0byBoZWxwIHByb3RlY3QgZGF0YSBhdCB0cmFuc2l0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEVsYXN0aWMgTG9hZCBCYWxhbmNpbmcuIFVzZSBBV1MgQ2VydGlmaWNhdGUgTWFuYWdlciB0byBtYW5hZ2UsIHByb3Zpc2lvbiBhbmQgZGVwbG95IHB1YmxpYyBhbmQgcHJpdmF0ZSBTU0wvVExTIGNlcnRpZmljYXRlcyB3aXRoIEFXUyBzZXJ2aWNlcyBhbmQgaW50ZXJuYWwgcmVzb3VyY2VzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzRUxCVXNlQUNNQ2VydHMsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBFTVIgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0VNUihub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KSB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtRU1SS2VyYmVyb3NFbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgRU1SIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBLZXJiZXJvcyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBBQy0yKGopLCBBQy0zLCBBQy01YywgQUMtNikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIGFjY2VzcyBwZXJtaXNzaW9ucyBhbmQgYXV0aG9yaXphdGlvbnMgY2FuIGJlIG1hbmFnZWQgYW5kIGluY29ycG9yYXRlZCB3aXRoIHRoZSBwcmluY2lwbGVzIG9mIGxlYXN0IHByaXZpbGVnZSBhbmQgc2VwYXJhdGlvbiBvZiBkdXRpZXMsIGJ5IGVuYWJsaW5nIEtlcmJlcm9zIGZvciBBbWF6b24gRU1SIGNsdXN0ZXJzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzRU1SS2VyYmVyb3NFbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgSUFNIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tJQU0obm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUlBTUdyb3VwTWVtYmVyc2hpcCcsXG4gICAgICBpbmZvOiAnVGhlIElBTSB1c2VyIGRvZXMgbm90IGJlbG9uZyB0byBhbnkgZ3JvdXAocykgLSAoQ29udHJvbCBJRHM6IEFDLTIoMSksIEFDLTIoaiksIEFDLTMsIGFuZCBBQy02KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBV1MgSWRlbnRpdHkgYW5kIEFjY2VzcyBNYW5hZ2VtZW50IChJQU0pIGNhbiBoZWxwIHlvdSByZXN0cmljdCBhY2Nlc3MgcGVybWlzc2lvbnMgYW5kIGF1dGhvcml6YXRpb25zLCBieSBlbnN1cmluZyBJQU0gdXNlcnMgYXJlIG1lbWJlcnMgb2YgYXQgbGVhc3Qgb25lIGdyb3VwLiBBbGxvd2luZyB1c2VycyBtb3JlIHByaXZpbGVnZXMgdGhhbiBuZWVkZWQgdG8gY29tcGxldGUgYSB0YXNrIG1heSB2aW9sYXRlIHRoZSBwcmluY2lwbGUgb2YgbGVhc3QgcHJpdmlsZWdlIGFuZCBzZXBhcmF0aW9uIG9mIGR1dGllcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0lBTUdyb3VwTWVtYmVyc2hpcCxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUlBTU5vSW5saW5lUG9saWN5JyxcbiAgICAgIGluZm86ICdUaGUgSUFNIEdyb3VwLCBVc2VyLCBvciBSb2xlIGNvbnRhaW5zIGFuIGlubGluZSBwb2xpY3kgLSAoQ29udHJvbCBJRDogQUMtNikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQVdTIHJlY29tbWVuZHMgdG8gdXNlIG1hbmFnZWQgcG9saWNpZXMgaW5zdGVhZCBvZiBpbmxpbmUgcG9saWNpZXMuIFRoZSBtYW5hZ2VkIHBvbGljaWVzIGFsbG93IHJldXNhYmlsaXR5LCB2ZXJzaW9uaW5nIGFuZCByb2xsaW5nIGJhY2ssIGFuZCBkZWxlZ2F0aW5nIHBlcm1pc3Npb25zIG1hbmFnZW1lbnQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNJQU1Ob0lubGluZVBvbGljeSxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLUlBTVBvbGljeU5vU3RhdGVtZW50c1dpdGhBZG1pbkFjY2VzcycsXG4gICAgICBpbmZvOiAnVGhlIElBTSBwb2xpY3kgZ3JhbnRzIGFkbWluIGFjY2VzcyAtIChDb250cm9sIElEczogQUMtMigxKSwgQUMtMihqKSwgQUMtMywgQUMtNikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQVdTIElkZW50aXR5IGFuZCBBY2Nlc3MgTWFuYWdlbWVudCAoSUFNKSBjYW4gaGVscCB5b3UgaW5jb3Jwb3JhdGUgdGhlIHByaW5jaXBsZXMgb2YgbGVhc3QgcHJpdmlsZWdlIGFuZCBzZXBhcmF0aW9uIG9mIGR1dGllcyB3aXRoIGFjY2VzcyBwZXJtaXNzaW9ucyBhbmQgYXV0aG9yaXphdGlvbnMsIHJlc3RyaWN0aW5nIHBvbGljaWVzIGZyb20gY29udGFpbmluZyBcIkVmZmVjdFwiOiBcIkFsbG93XCIgd2l0aCBcIkFjdGlvblwiOiBcIipcIiBvdmVyIFwiUmVzb3VyY2VcIjogXCIqXCIuIEFsbG93aW5nIHVzZXJzIHRvIGhhdmUgbW9yZSBwcml2aWxlZ2VzIHRoYW4gbmVlZGVkIHRvIGNvbXBsZXRlIGEgdGFzayBtYXkgdmlvbGF0ZSB0aGUgcHJpbmNpcGxlIG9mIGxlYXN0IHByaXZpbGVnZSBhbmQgc2VwYXJhdGlvbiBvZiBkdXRpZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNJQU1Qb2xpY3lOb1N0YXRlbWVudHNXaXRoQWRtaW5BY2Nlc3MsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1JQU1Vc2VyTm9Qb2xpY2llcycsXG4gICAgICBpbmZvOiAnVGhlIElBTSBwb2xpY3kgaXMgYXR0YWNoZWQgYXQgdGhlIHVzZXIgbGV2ZWwgLSAoQ29udHJvbCBJRHM6IEFDLTIoaiksIEFDLTMsIEFDLTVjLCBBQy02KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBc3NpZ25pbmcgcHJpdmlsZWdlcyBhdCB0aGUgZ3JvdXAgb3IgdGhlIHJvbGUgbGV2ZWwgaGVscHMgdG8gcmVkdWNlIG9wcG9ydHVuaXR5IGZvciBhbiBpZGVudGl0eSB0byByZWNlaXZlIG9yIHJldGFpbiBleGNlc3NpdmUgcHJpdmlsZWdlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M0lBTVVzZXJOb1BvbGljaWVzLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cbiAgLyoqXG4gICAqIENoZWNrIExhbWJkYSBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrTGFtYmRhKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1MYW1iZGFGdW5jdGlvbnNJbnNpZGVWUEMnLFxuICAgICAgaW5mbzogJ1RoZSBMYW1iZGEgZnVuY3Rpb24gaXMgbm90IFZQQyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBBQy00LCBTQy03LCBTQy03KDMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIG9mIHRoZWlyIGxvZ2ljYWwgaXNvbGF0aW9uLCBkb21haW5zIHRoYXQgcmVzaWRlIHdpdGhpbiBhbiBBbWF6b24gVlBDIGhhdmUgYW4gZXh0cmEgbGF5ZXIgb2Ygc2VjdXJpdHkgd2hlbiBjb21wYXJlZCB0byBkb21haW5zIHRoYXQgdXNlIHB1YmxpYyBlbmRwb2ludHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNMYW1iZGFGdW5jdGlvbnNJbnNpZGVWUEMsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBPcGVuU2VhcmNoIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tPcGVuU2VhcmNoKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1PcGVuU2VhcmNoRW5jcnlwdGVkQXRSZXN0JyxcbiAgICAgIGluZm86ICdUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gYXQgcmVzdCBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBTQy0xMywgU0MtMjgpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGFuZCB0byBoZWxwIHByb3RlY3QgZGF0YSBhdCByZXN0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgKE9wZW5TZWFyY2ggU2VydmljZSkgZG9tYWlucy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M09wZW5TZWFyY2hFbmNyeXB0ZWRBdFJlc3QsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1PcGVuU2VhcmNoTm9kZVRvTm9kZUVuY3J5cHRlZCcsXG4gICAgICBpbmZvOiAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3QgaGF2ZSBub2RlLXRvLW5vZGUgZW5jcnlwdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBTQy03LCBTQy04LCBTQy04KDEpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCwgZW5hYmxlIGVuY3J5cHRpb24gaW4gdHJhbnNpdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhIHdpdGhpbiB5b3VyIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgKE9wZW5TZWFyY2ggU2VydmljZSkgZG9tYWlucy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M09wZW5TZWFyY2hOb2RlVG9Ob2RlRW5jcnlwdGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtT3BlblNlYXJjaFJ1bm5pbmdXaXRoaW5WUEMnLFxuICAgICAgaW5mbzogJ1RoZSBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIGlzIG5vdCBydW5uaW5nIHdpdGhpbiBhIFZQQyAtIChDb250cm9sIElEczogQUMtNCwgU0MtNywgU0MtNygzKSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVlBDcyBoZWxwIHNlY3VyZSB5b3VyIEFXUyByZXNvdXJjZXMgYW5kIHByb3ZpZGUgYW4gZXh0cmEgbGF5ZXIgb2YgcHJvdGVjdGlvbi4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M09wZW5TZWFyY2hSdW5uaW5nV2l0aGluVlBDLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQW1hem9uIFJEUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrUkRTKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1SRFNFbmhhbmNlZE1vbml0b3JpbmdFbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgUkRTIERCIGluc3RhbmNlIGRvZXMgbm90IGVuaGFuY2VkIG1vbml0b3JpbmcgZW5hYmxlZCAtIChDb250cm9sIElEOiBDQS03KGEpKGIpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbmFibGUgZW5oYW5jZWQgbW9uaXRvcmluZyB0byBoZWxwIG1vbml0b3IgQW1hem9uIFJEUyBhdmFpbGFiaWxpdHkuIFRoaXMgcHJvdmlkZXMgZGV0YWlsZWQgdmlzaWJpbGl0eSBpbnRvIHRoZSBoZWFsdGggb2YgeW91ciBBbWF6b24gUkRTIGRhdGFiYXNlIGluc3RhbmNlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1JEU0VuaGFuY2VkTW9uaXRvcmluZ0VuYWJsZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1SRFNJbnN0YW5jZUJhY2t1cEVuYWJsZWQnLFxuICAgICAgaW5mbzogJ1RoZSBSRFMgREIgaW5zdGFuY2UgZG9lcyBub3QgaGF2ZSBiYWNrdXBzIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IENQLTkoYiksIENQLTEwLCBTSS0xMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIGJhY2t1cCBmZWF0dXJlIG9mIEFtYXpvbiBSRFMgY3JlYXRlcyBiYWNrdXBzIG9mIHlvdXIgZGF0YWJhc2VzIGFuZCB0cmFuc2FjdGlvbiBsb2dzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzUkRTSW5zdGFuY2VCYWNrdXBFbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtUkRTSW5zdGFuY2VEZWxldGlvblByb3RlY3Rpb25FbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgUkRTIERCIGluc3RhbmNlIG9yIEF1cm9yYSBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgZGVsZXRpb24gcHJvdGVjdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSUQ6IFNDLTUpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Vuc3VyZSBBbWF6b24gUmVsYXRpb25hbCBEYXRhYmFzZSBTZXJ2aWNlIChBbWF6b24gUkRTKSBpbnN0YW5jZXMgYW5kIGNsdXN0ZXJzIGhhdmUgZGVsZXRpb24gcHJvdGVjdGlvbiBlbmFibGVkLiBVc2UgZGVsZXRpb24gcHJvdGVjdGlvbiB0byBwcmV2ZW50IHlvdXIgQW1hem9uIFJEUyBEQiBpbnN0YW5jZXMgYW5kIGNsdXN0ZXJzIGZyb20gYmVpbmcgYWNjaWRlbnRhbGx5IG9yIG1hbGljaW91c2x5IGRlbGV0ZWQsIHdoaWNoIGNhbiBsZWFkIHRvIGxvc3Mgb2YgYXZhaWxhYmlsaXR5IGZvciB5b3VyIGFwcGxpY2F0aW9ucy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1JEU0luc3RhbmNlRGVsZXRpb25Qcm90ZWN0aW9uRW5hYmxlZCxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLVJEU0luc3RhbmNlTXVsdGlBWlN1cHBvcnQnLFxuICAgICAgaW5mbzogJ1RoZSBub24tQXVyb3JhIFJEUyBEQiBpbnN0YW5jZSBkb2VzIG5vdCBoYXZlIG11bHRpLUFaIHN1cHBvcnQgZW5hYmxlZCAtIChDb250cm9sIElEczogQ1AtMTAsIFNDLTUsIFNDLTM2KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdNdWx0aS1BWiBzdXBwb3J0IGluIEFtYXpvbiBSZWxhdGlvbmFsIERhdGFiYXNlIFNlcnZpY2UgKEFtYXpvbiBSRFMpIHByb3ZpZGVzIGVuaGFuY2VkIGF2YWlsYWJpbGl0eSBhbmQgZHVyYWJpbGl0eSBmb3IgZGF0YWJhc2UgaW5zdGFuY2VzLiBXaGVuIHlvdSBwcm92aXNpb24gYSBNdWx0aS1BWiBkYXRhYmFzZSBpbnN0YW5jZSwgQW1hem9uIFJEUyBhdXRvbWF0aWNhbGx5IGNyZWF0ZXMgYSBwcmltYXJ5IGRhdGFiYXNlIGluc3RhbmNlLCBhbmQgc3luY2hyb25vdXNseSByZXBsaWNhdGVzIHRoZSBkYXRhIHRvIGEgc3RhbmRieSBpbnN0YW5jZSBpbiBhIGRpZmZlcmVudCBBdmFpbGFiaWxpdHkgWm9uZS4gSW4gY2FzZSBvZiBhbiBpbmZyYXN0cnVjdHVyZSBmYWlsdXJlLCBBbWF6b24gUkRTIHBlcmZvcm1zIGFuIGF1dG9tYXRpYyBmYWlsb3ZlciB0byB0aGUgc3RhbmRieSBzbyB0aGF0IHlvdSBjYW4gcmVzdW1lIGRhdGFiYXNlIG9wZXJhdGlvbnMgYXMgc29vbiBhcyB0aGUgZmFpbG92ZXIgaXMgY29tcGxldGUuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNSRFNJbnN0YW5jZU11bHRpQVpTdXBwb3J0LFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtUkRTSW5zdGFuY2VQdWJsaWNBY2Nlc3MnLFxuICAgICAgaW5mbzogJ1RoZSBSRFMgREIgaW5zdGFuY2UgYWxsb3dzIHB1YmxpYyBhY2Nlc3MgLSAoQ29udHJvbCBJRHM6IEFDLTQsIEFDLTYsIEFDLTIxKGIpLCBTQy03LCBTQy03KDMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBbWF6b24gUkRTIGRhdGFiYXNlIGluc3RhbmNlcyBjYW4gY29udGFpbiBzZW5zaXRpdmUgaW5mb3JtYXRpb24sIGFuZCBwcmluY2lwbGVzIGFuZCBhY2Nlc3MgY29udHJvbCBpcyByZXF1aXJlZCBmb3Igc3VjaCBhY2NvdW50cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1JEU0luc3RhbmNlUHVibGljQWNjZXNzLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtUkRTTG9nZ2luZ0VuYWJsZWQnLFxuICAgICAgaW5mbzogJ1RoZSBSRFMgREIgaW5zdGFuY2UgZG9lcyBub3QgaGF2ZSBhbGwgQ2xvdWRXYXRjaCBsb2cgdHlwZXMgZXhwb3J0ZWQgLSAoQ29udHJvbCBJRHM6IEFDLTIoNCksIEFDLTIoZyksIEFVLTIoYSkoZCksIEFVLTMsIEFVLTEyKGEpKGMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUbyBoZWxwIHdpdGggbG9nZ2luZyBhbmQgbW9uaXRvcmluZyB3aXRoaW4geW91ciBlbnZpcm9ubWVudCwgZW5zdXJlIEFtYXpvbiBSZWxhdGlvbmFsIERhdGFiYXNlIFNlcnZpY2UgKEFtYXpvbiBSRFMpIGxvZ2dpbmcgaXMgZW5hYmxlZC4gV2l0aCBBbWF6b24gUkRTIGxvZ2dpbmcsIHlvdSBjYW4gY2FwdHVyZSBldmVudHMgc3VjaCBhcyBjb25uZWN0aW9ucywgZGlzY29ubmVjdGlvbnMsIHF1ZXJpZXMsIG9yIHRhYmxlcyBxdWVyaWVkLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzUkRTTG9nZ2luZ0VuYWJsZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1SRFNTdG9yYWdlRW5jcnlwdGVkJyxcbiAgICAgIGluZm86ICdUaGUgUkRTIERCIGluc3RhbmNlIG9yIEF1cm9yYSBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgc3RvcmFnZSBlbmNyeXB0ZWQgLSAoQ29udHJvbCBJRHM6IFNDLTEzLCBTQy0yOCkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYXQgcmVzdCBpbiBBbWF6b24gUkRTIERCIGluc3RhbmNlcyBhbmQgY2x1c3RlcnMsIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1JEU1N0b3JhZ2VFbmNyeXB0ZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBSZWRzaGlmdCBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrUmVkc2hpZnQobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLVJlZHNoaWZ0Q2x1c3RlckNvbmZpZ3VyYXRpb24nLFxuICAgICAgaW5mbzogJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGRvZXMgbm90IGhhdmUgZW5jcnlwdGlvbiBvciBhdWRpdCBsb2dnaW5nIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IEFDLTIoNCksIEFDLTIoZyksIEFVLTIoYSkoZCksIEFVLTMsIEFVLTEyKGEpKGMpLCBTQy0xMykuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVG8gcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSB0aGF0IGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBBbWF6b24gUmVkc2hpZnQgY2x1c3RlcnMuIFlvdSBtdXN0IGFsc28gZW5zdXJlIHRoYXQgcmVxdWlyZWQgY29uZmlndXJhdGlvbnMgYXJlIGRlcGxveWVkIG9uIEFtYXpvbiBSZWRzaGlmdCBjbHVzdGVycy4gVGhlIGF1ZGl0IGxvZ2dpbmcgc2hvdWxkIGJlIGVuYWJsZWQgdG8gcHJvdmlkZSBpbmZvcm1hdGlvbiBhYm91dCBjb25uZWN0aW9ucyBhbmQgdXNlciBhY3Rpdml0aWVzIGluIHRoZSBkYXRhYmFzZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1JlZHNoaWZ0Q2x1c3RlckNvbmZpZ3VyYXRpb24sXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1SZWRzaGlmdENsdXN0ZXJQdWJsaWNBY2Nlc3MnLFxuICAgICAgaW5mbzogJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGFsbG93cyBwdWJsaWMgYWNjZXNzIC0gKENvbnRyb2wgSURzOiBBQy0zLCBBQy00LCBBQy02LCBBQy0yMShiKSwgU0MtNywgU0MtNygzKSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQW1hem9uIFJlZHNoaWZ0IGNsdXN0ZXJzIGNhbiBjb250YWluIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbiBhbmQgcHJpbmNpcGxlcyBhbmQgYWNjZXNzIGNvbnRyb2wgaXMgcmVxdWlyZWQgZm9yIHN1Y2ggYWNjb3VudHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNSZWRzaGlmdENsdXN0ZXJQdWJsaWNBY2Nlc3MsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBBbWF6b24gUzMgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja1MzKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1TM0J1Y2tldERlZmF1bHRMb2NrRW5hYmxlZCcsXG4gICAgICBpbmZvOiAnVGhlIFMzIEJ1Y2tldCBkb2VzIG5vdCBoYXZlIG9iamVjdCBsb2NrIGVuYWJsZWQgLSAoQ29udHJvbCBJRDogU0MtMjgpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gUzMgYnVja2V0cywgZW5mb3JjZSBvYmplY3QgbG9ja3MgYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzUzNCdWNrZXREZWZhdWx0TG9ja0VuYWJsZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1TM0J1Y2tldExvZ2dpbmdFbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IGhhdmUgc2VydmVyIGFjY2VzcyBsb2dzIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IEFDLTIoZyksIEFVLTIoYSkoZCksIEFVLTMsIEFVLTEyKGEpKGMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBbWF6b24gU2ltcGxlIFN0b3JhZ2UgU2VydmljZSAoQW1hem9uIFMzKSBzZXJ2ZXIgYWNjZXNzIGxvZ2dpbmcgcHJvdmlkZXMgYSBtZXRob2QgdG8gbW9uaXRvciB0aGUgbmV0d29yayBmb3IgcG90ZW50aWFsIGN5YmVyc2VjdXJpdHkgZXZlbnRzLiBUaGUgZXZlbnRzIGFyZSBtb25pdG9yZWQgYnkgY2FwdHVyaW5nIGRldGFpbGVkIHJlY29yZHMgZm9yIHRoZSByZXF1ZXN0cyB0aGF0IGFyZSBtYWRlIHRvIGFuIEFtYXpvbiBTMyBidWNrZXQuIEVhY2ggYWNjZXNzIGxvZyByZWNvcmQgcHJvdmlkZXMgZGV0YWlscyBhYm91dCBhIHNpbmdsZSBhY2Nlc3MgcmVxdWVzdC4gVGhlIGRldGFpbHMgaW5jbHVkZSB0aGUgcmVxdWVzdGVyLCBidWNrZXQgbmFtZSwgcmVxdWVzdCB0aW1lLCByZXF1ZXN0IGFjdGlvbiwgcmVzcG9uc2Ugc3RhdHVzLCBhbmQgYW4gZXJyb3IgY29kZSwgaWYgcmVsZXZhbnQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNTM0J1Y2tldExvZ2dpbmdFbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtUzNCdWNrZXRQdWJsaWNSZWFkUHJvaGliaXRlZCcsXG4gICAgICBpbmZvOiAnVGhlIFMzIEJ1Y2tldCBkb2VzIG5vdCBwcm9oaWJpdCBwdWJsaWMgcmVhZCBhY2Nlc3MgdGhyb3VnaCBpdHMgQmxvY2sgUHVibGljIEFjY2VzcyBjb25maWd1cmF0aW9ucyBhbmQgYnVja2V0IEFDTHMgLSAoQ29udHJvbCBJRHM6IEFDLTMsIEFDLTQsIEFDLTYsIEFDLTIxKGIpLCBTQy03LCBTQy03KDMpKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGUgbWFuYWdlbWVudCBvZiBhY2Nlc3Mgc2hvdWxkIGJlIGNvbnNpc3RlbnQgd2l0aCB0aGUgY2xhc3NpZmljYXRpb24gb2YgdGhlIGRhdGEuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNTM0J1Y2tldFB1YmxpY1JlYWRQcm9oaWJpdGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtUzNCdWNrZXRQdWJsaWNXcml0ZVByb2hpYml0ZWQnLFxuICAgICAgaW5mbzogJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgcHJvaGliaXQgcHVibGljIHdyaXRlIGFjY2VzcyB0aHJvdWdoIGl0cyBCbG9jayBQdWJsaWMgQWNjZXNzIGNvbmZpZ3VyYXRpb25zIGFuZCBidWNrZXQgQUNMcyAtIChDb250cm9sIElEczogQUMtMywgQUMtNCwgQUMtNiwgQUMtMjEoYiksIFNDLTcsIFNDLTcoMykpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBtYW5hZ2VtZW50IG9mIGFjY2VzcyBzaG91bGQgYmUgY29uc2lzdGVudCB3aXRoIHRoZSBjbGFzc2lmaWNhdGlvbiBvZiB0aGUgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1MzQnVja2V0UHVibGljV3JpdGVQcm9oaWJpdGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtUzNCdWNrZXRSZXBsaWNhdGlvbkVuYWJsZWQnLFxuICAgICAgaW5mbzogJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgaGF2ZSByZXBsaWNhdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBBVS05KDIpLCBDUC05KGIpLCBDUC0xMCwgU0MtNSwgU0MtMzYpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0FtYXpvbiBTaW1wbGUgU3RvcmFnZSBTZXJ2aWNlIChBbWF6b24gUzMpIENyb3NzLVJlZ2lvbiBSZXBsaWNhdGlvbiAoQ1JSKSBzdXBwb3J0cyBtYWludGFpbmluZyBhZGVxdWF0ZSBjYXBhY2l0eSBhbmQgYXZhaWxhYmlsaXR5LiBDUlIgZW5hYmxlcyBhdXRvbWF0aWMsIGFzeW5jaHJvbm91cyBjb3B5aW5nIG9mIG9iamVjdHMgYWNyb3NzIEFtYXpvbiBTMyBidWNrZXRzIHRvIGhlbHAgZW5zdXJlIHRoYXQgZGF0YSBhdmFpbGFiaWxpdHkgaXMgbWFpbnRhaW5lZC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1MzQnVja2V0UmVwbGljYXRpb25FbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtUzNCdWNrZXRTZXJ2ZXJTaWRlRW5jcnlwdGlvbkVuYWJsZWQnLFxuICAgICAgaW5mbzogJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgaGF2ZSBkZWZhdWx0IHNlcnZlci1zaWRlIGVuY3J5cHRpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogQVUtOSgyKSwgQ1AtOShiKSwgQ1AtMTAsIFNDLTUsIFNDLTM2KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhdCByZXN0IGluIEFtYXpvbiBTMyBidWNrZXRzLCBlbmFibGUgZW5jcnlwdGlvbiB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzUzNCdWNrZXRTZXJ2ZXJTaWRlRW5jcnlwdGlvbkVuYWJsZWQsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlSWQ6ICdOSVNULjgwMC41My1TM0J1Y2tldFZlcnNpb25pbmdFbmFibGVkJyxcbiAgICAgIGluZm86ICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IGhhdmUgdmVyc2lvbmluZyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiBDUC0xMCwgU0ktMTIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzZSB2ZXJzaW9uaW5nIHRvIHByZXNlcnZlLCByZXRyaWV2ZSwgYW5kIHJlc3RvcmUgZXZlcnkgdmVyc2lvbiBvZiBldmVyeSBvYmplY3Qgc3RvcmVkIGluIHlvdXIgQW1hem9uIFMzIGJ1Y2tldC4gVmVyc2lvbmluZyBoZWxwcyB5b3UgdG8gZWFzaWx5IHJlY292ZXIgZnJvbSB1bmludGVuZGVkIHVzZXIgYWN0aW9ucyBhbmQgYXBwbGljYXRpb24gZmFpbHVyZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBuaXN0ODAwNTNTM0J1Y2tldFZlcnNpb25pbmdFbmFibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgU2FnZU1ha2VyIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTYWdlTWFrZXIobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSkge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLVNhZ2VNYWtlckVuZHBvaW50S01TJyxcbiAgICAgIGluZm86ICdUaGUgU2FnZU1ha2VyIGVuZHBvaW50IGlzIG5vdCBlbmNyeXB0ZWQgd2l0aCBhIEtNUyBrZXkgLSAoQ29udHJvbCBJRHM6IFNDLTEzLCBTQy0yOCkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYXQgcmVzdCBpbiBTYWdlTWFrZXIgZW5kcG9pbnQsIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1NhZ2VNYWtlckVuZHBvaW50S01TLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtU2FnZU1ha2VyTm90ZWJvb2tEaXJlY3RJbnRlcm5ldEFjY2Vzc0Rpc2FibGVkJyxcbiAgICAgIGluZm86ICdUaGUgU2FnZU1ha2VyIG5vdGVib29rIGRvZXMgbm90IGRpc2FibGUgZGlyZWN0IGludGVybmV0IGFjY2VzcyAtIChDb250cm9sIElEczogQUMtMywgQUMtNCwgQUMtNiwgQUMtMjEoYiksIFNDLTcsIFNDLTcoMykpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0J5IHByZXZlbnRpbmcgZGlyZWN0IGludGVybmV0IGFjY2VzcywgeW91IGNhbiBrZWVwIHNlbnNpdGl2ZSBkYXRhIGZyb20gYmVpbmcgYWNjZXNzZWQgYnkgdW5hdXRob3JpemVkIHVzZXJzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzU2FnZU1ha2VyTm90ZWJvb2tEaXJlY3RJbnRlcm5ldEFjY2Vzc0Rpc2FibGVkLFxuICAgICAgaWdub3JlczogaWdub3JlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZUlkOiAnTklTVC44MDAuNTMtU2FnZU1ha2VyTm90ZWJvb2tLTVMnLFxuICAgICAgaW5mbzogJ1RoZSBTYWdlTWFrZXIgbm90ZWJvb2sgaXMgbm90IGVuY3J5cHRlZCB3aXRoIGEgS01TIGtleSAtIChDb250cm9sIElEczogU0MtMTMsIFNDLTI4KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhdCByZXN0IGluIFNhZ2VNYWtlciBub3RlYm9vaywgZW5hYmxlIGVuY3J5cHRpb24gYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogbmlzdDgwMDUzU2FnZU1ha2VyTm90ZWJvb2tLTVMsXG4gICAgICBpZ25vcmVzOiBpZ25vcmVzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBBbWF6b24gU05TIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTTlMobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVJZDogJ05JU1QuODAwLjUzLVNOU0VuY3J5cHRlZEtNUycsXG4gICAgICBpbmZvOiAnVGhlIFNOUyB0b3BpYyBkb2VzIG5vdCBoYXZlIEtNUyBlbmNyeXB0aW9uIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IFNDLTEzLCBTQy0yOCkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYXQgcmVzdCBpbiBwdWJsaXNoZWQgbWVzc2FnZXMsIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IG5pc3Q4MDA1M1NOU0VuY3J5cHRlZEtNUyxcbiAgICAgIGlnbm9yZXM6IGlnbm9yZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG59XG4iXX0=