"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const common_1 = require("../../../common");
const EBS_OPTIMIZED_SUPPORTED = [
    'c1.xlarge',
    'c3.xlarge',
    'c3.2xlarge',
    'c3.4xlarge',
    'g2.2xlarge',
    'i2.xlarge',
    'i2.2xlarge',
    'i2.4xlarge',
    'm1.large',
    'm1.xlarge',
    'm2.2xlarge',
    'm2.4xlarge',
    'm3.xlarge',
    'm3.2xlarge',
    'r3.xlarge',
    'r3.2xlarge',
    'r3.4xlarge',
];
const DEFAULT_TYPE = 'm1.small';
/**
 * EC2 instance types that support EBS optimization and are not EBS optimized by default have EBS optimization enabled - (Control ID: 164.308(a)(7)(i))
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html#previous
 *  @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceType = node.instanceType
            ? common_1.resolveIfPrimitive(node, node.instanceType)
            : DEFAULT_TYPE;
        const ebsOptimized = monocdk_1.Stack.of(node).resolve(node.ebsOptimized);
        if (EBS_OPTIMIZED_SUPPORTED.includes(instanceType) &&
            ebsOptimized !== true) {
            return false;
        }
    }
    return true;
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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