"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HIPAASecurityChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const ecs_1 = require("./rules/ecs");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elasticbeanstalk_1 = require("./rules/elasticbeanstalk");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const secretsmanager_1 = require("./rules/secretsmanager");
const sns_1 = require("./rules/sns");
const vpc_1 = require("./rules/vpc");
/**
 * Check for HIPAA Security compliance.
 *
 * Based on the HIPAA Security AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-hipaa_security.html
 *
 * @stability stable
 */
class HIPAASecurityChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            this.checkCloudWatch(node, ignores);
            this.checkCodeBuild(node, ignores);
            this.checkDMS(node, ignores);
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkECS(node, ignores);
            this.checkEFS(node, ignores);
            this.checkElastiCache(node, ignores);
            this.checkElasticBeanstalk(node, ignores);
            this.checkELB(node, ignores);
            this.checkEMR(node, ignores);
            this.checkIAM(node, ignores);
            this.checkLambda(node, ignores);
            this.checkOpenSearch(node, ignores);
            this.checkRDS(node, ignores);
            this.checkRedshift(node, ignores);
            this.checkS3(node, ignores);
            this.checkSageMaker(node, ignores);
            this.checkSecretsManager(node, ignores);
            this.checkSNS(node, ignores);
            this.checkVPC(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWCacheEnabledAndEncrypted',
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: common_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWCacheEnabledAndEncrypted,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWExecutionLoggingEnabled',
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control ID: 164.312(b)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: common_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWExecutionLoggingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWSSLEnabled',
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: common_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWSSLEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWXrayEnabled',
            info: 'The API Gateway REST API stage does not have X-Ray enabled - (Control ID: 164.312(b)).',
            explanation: 'AWS X-Ray collects data about requests that your application serves, and provides tools you can use to view, filter, and gain insights into that data to identify issues and opportunities for optimization. Ensure X-Ray is enables so you can see detailed information not only about the request and response, but also about calls that your application makes to downstream AWS resources, microservices, databases and HTTP web APIs.',
            level: common_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWXrayEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-AutoscalingGroupELBHealthCheckRequired',
            info: 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control ID: 164.312(b)).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: common_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.hipaaSecurityAutoscalingGroupELBHealthCheckRequired,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-AutoscalingLaunchConfigPublicIpDisabled',
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: common_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.hipaaSecurityAutoscalingLaunchConfigPublicIpDisabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled',
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.hipaaSecurityCloudTrailCloudWatchLogsEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudTrailEncryptionEnabled',
            info: 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.hipaaSecurityCloudTrailEncryptionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudTrailLogFileValidationEnabled',
            info: 'The trail does not have log file validation enabled - (Control ID: 164.312(c)(1), 164.312(c)(2)).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.hipaaSecurityCloudTrailLogFileValidationEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudWatchAlarmAction',
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control ID: 164.312(b)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.hipaaSecurityCloudWatchAlarmAction,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudWatchLogGroupEncrypted',
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.hipaaSecurityCloudWatchLogGroupEncrypted,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudWatchLogGroupRetentionPeriod',
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control ID: 164.312(b)).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: common_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.hipaaSecurityCloudWatchLogGroupRetentionPeriod,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-CodeBuildProjectEnvVarAwsCred',
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: codebuild_1.hipaaSecurityCodeBuildProjectEnvVarAwsCred,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CodeBuildProjectSourceRepoUrl',
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: common_1.NagMessageLevel.ERROR,
            rule: codebuild_1.hipaaSecurityCodeBuildProjectSourceRepoUrl,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-DMSReplicationNotPublic',
            info: 'The DMS replication instance is public - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: common_1.NagMessageLevel.ERROR,
            rule: dms_1.hipaaSecurityDMSReplicationNotPublic,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-DynamoDBPITREnabled',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: common_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.hipaaSecurityDynamoDBPITREnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2EBSOptimizedInstance',
            info: "The EC2 instance type 'supports' EBS optimization and does not have EBS optimization enabled - (Control ID: 164.308(a)(7)(i)).",
            explanation: 'An optimized instance in Amazon Elastic Block Store (Amazon EBS) provides additional, dedicated capacity for Amazon EBS I/O operations. This optimization provides the most efficient performance for your EBS volumes by minimizing contention between Amazon EBS I/O operations and other traffic from your instance.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2EBSOptimizedInstance,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled',
            info: 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: 164.312(b)).',
            explanation: 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstanceDetailedMonitoringEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstanceNoPublicIps',
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstanceNoPublicIps,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstanceProfileAttached',
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstanceProfileAttached,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstancesInVPC',
            info: 'The EC2 instance is not within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstancesInVPC,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2RestrictedCommonPorts',
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2RestrictedCommonPorts,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2RestrictedSSH',
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2RestrictedSSH,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ECSTaskDefinitionUserForHostMode',
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: common_1.NagMessageLevel.ERROR,
            rule: ecs_1.hipaaSecurityECSTaskDefinitionUserForHostMode,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-EFSEncrypted',
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: common_1.NagMessageLevel.ERROR,
            rule: efs_1.hipaaSecurityEFSEncrypted,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ElastiCacheRedisClusterAutomaticBackup',
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elasticache_1.hipaaSecurityElastiCacheRedisClusterAutomaticBackup,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Elastic Beanstalk Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElasticBeanstalk(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ElasticBeanstalkEnhancedHealthReportingEnabled',
            info: 'The Elastic Beanstalk environment does not have enhanced health reporting enabled - (Control ID: 164.312(b)).',
            explanation: 'AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.hipaaSecurityElasticBeanstalkEnhancedHealthReportingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ElasticBeanstalkManagedUpdatesEnabled',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled - (Control ID: 164.308(a)(5)(ii)(A)).',
            explanation: 'Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.hipaaSecurityElasticBeanstalkManagedUpdatesEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ALBHttpDropInvalidHeaderEnabled',
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityALBHttpDropInvalidHeaderEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ALBHttpToHttpsRedirection',
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityALBHttpToHttpsRedirection,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBACMCertificateRequired',
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBACMCertificateRequired,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBCrossZoneBalancingEnabled',
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBCrossZoneBalancingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBDeletionProtectionEnabled',
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBDeletionProtectionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBLoggingEnabled',
            info: 'The ELB does not have logging enabled - (Control ID: 164.312(b)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBLoggingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBTlsHttpsListenersOnly',
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBTlsHttpsListenersOnly,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBv2ACMCertificateRequired',
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: common_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBv2ACMCertificateRequired,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-EMRKerberosEnabled',
            info: 'The EMR cluster does not have Kerberos enabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.',
            level: common_1.NagMessageLevel.ERROR,
            rule: emr_1.hipaaSecurityEMRKerberosEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMNoInlinePolicy',
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMNoInlinePolicy,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMPolicyNoStatementsWithAdminAccess',
            info: 'The IAM policy grants admin access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMPolicyNoStatementsWithAdminAccess,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMPolicyNoStatementsWithFullAccess',
            info: 'The IAM policy grants full access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMPolicyNoStatementsWithFullAccess,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMUserGroupMembership',
            info: 'The IAM user does not belong to any group(s) - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMUserGroupMembership,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMUserNoPolicies',
            info: 'The IAM policy is attached at the user level - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: common_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMUserNoPolicies,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-LambdaConcurrency',
            info: 'The Lambda function is not configured with function-level concurrent execution limits - (Control ID: 164.312(b)).',
            explanation: "Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.",
            level: common_1.NagMessageLevel.ERROR,
            rule: lambda_1.hipaaSecurityLambdaConcurrency,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-LambdaDlq',
            info: 'The Lambda function is not configured with a dead-letter configuration - (Control ID: 164.312(b)).',
            explanation: 'Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.',
            level: common_1.NagMessageLevel.ERROR,
            rule: lambda_1.hipaaSecurityLambdaDlq,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-LambdaInsideVPC',
            info: 'The Lambda function is not VPC enabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: common_1.NagMessageLevel.ERROR,
            rule: lambda_1.hipaaSecurityLambdaInsideVPC,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchEncryptedAtRest',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: common_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchEncryptedAtRest,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchInVPCOnly',
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: common_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchInVPCOnly,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchLogsToCloudWatch',
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: common_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchLogsToCloudWatch,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchNodeToNodeEncryption',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: common_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchNodeToNodeEncryption,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSAutomaticMinorVersionUpgradeEnabled',
            info: 'The RDS DB instance does not have automatic minor version upgrades enabled - (Control ID: 164.308(a)(5)(ii)(A)).',
            explanation: 'Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSAutomaticMinorVersionUpgradeEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSEnhancedMonitoringEnabled',
            info: 'The RDS DB instance does not enhanced monitoring enabled - (Control ID: 164.312(b)).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSEnhancedMonitoringEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstanceBackupEnabled',
            info: 'The RDS DB instance does not have backups enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstanceBackupEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstanceDeletionProtectionEnabled',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstanceDeletionProtectionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstanceMultiAZSupport',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstanceMultiAZSupport,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstancePublicAccess',
            info: 'The RDS DB instance allows public access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstancePublicAccess,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSLoggingEnabled',
            info: 'The RDS DB instance does not have all CloudWatch log types exported - (Control IDs: 164.308(a)(3)(ii)(A), 164.308(a)(5)(ii)(C)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSLoggingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSStorageEncrypted',
            info: 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSStorageEncrypted,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftBackupEnabled',
            info: 'The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.',
            level: common_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftBackupEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftClusterConfiguration',
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(b), 164.312(e)(2)(ii)).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: common_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftClusterConfiguration,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftClusterMaintenanceSettings',
            info: 'The Redshift cluster has version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control IDs: 164.308(a)(5)(ii)(A), 164.308(a)(7)(ii)(A)).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.',
            level: common_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftClusterMaintenanceSettings,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftClusterPublicAccess',
            info: 'The Redshift cluster allows public access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: common_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftClusterPublicAccess,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftEnhancedVPCRoutingEnabled',
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: 164.312(e)(1)).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: common_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftEnhancedVPCRoutingEnabled,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketLevelPublicAccessProhibited',
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketLevelPublicAccessProhibited,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketLoggingEnabled',
            info: 'The S3 Bucket does not have server access logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketLoggingEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketPublicReadProhibited',
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketPublicReadProhibited,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketPublicWriteProhibited',
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketPublicWriteProhibited,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketReplicationEnabled',
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketReplicationEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketServerSideEncryptionEnabled',
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(c)(2), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketServerSideEncryptionEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketVersioningEnabled',
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B), 164.312(c)(1), 164.312(c)(2)).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketVersioningEnabled,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3DefaultEncryptionKMS',
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3DefaultEncryptionKMS,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-SageMakerEndpointConfigurationKMSKeyConfigured',
            info: 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.hipaaSecuritySageMakerEndpointConfigurationKMSKeyConfigured,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-SageMakerNotebookInstanceKMSKeyConfigured',
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.hipaaSecuritySageMakerNotebookInstanceKMSKeyConfigured,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-SageMakerNotebookNoDirectInternetAccess',
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.hipaaSecuritySageMakerNotebookNoDirectInternetAccess,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-SecretsManagerUsingKMSKey',
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.hipaaSecuritySecretsManagerUsingKMSKey,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-SNSEncryptedKMS',
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: sns_1.hipaaSecuritySNSEncryptedKMS,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node, ignores) {
        this.applyRule({
            ruleId: 'HIPAA.Security-VPCDefaultSecurityGroupClosed',
            info: "The VPC's default security group allows inbound or outbound traffic - (Control ID: 164.312(e)(1)).",
            explanation: 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: common_1.NagMessageLevel.WARN,
            rule: vpc_1.hipaaSecurityVPCDefaultSecurityGroupClosed,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-VPCNoUnrestrictedRouteToIGW',
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control ID: 164.312(e)(1)).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: common_1.NagMessageLevel.ERROR,
            rule: vpc_1.hipaaSecurityVPCNoUnrestrictedRouteToIGW,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-VPCSubnetAutoAssignPublicIpDisabled',
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: common_1.NagMessageLevel.ERROR,
            rule: vpc_1.hipaaSecurityVPCSubnetAutoAssignPublicIpDisabled,
            ignores: ignores,
            node: node,
        });
    }
}
exports.HIPAASecurityChecks = HIPAASecurityChecks;
_a = JSII_RTTI_SYMBOL_1;
HIPAASecurityChecks[_a] = { fqn: "monocdk-nag.HIPAASecurityChecks", version: "0.0.125" };
//# sourceMappingURL=data:application/json;base64,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