"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const analytics_1 = require("./rules/analytics");
const application_integration_1 = require("./rules/application_integration");
const index_1 = require("./rules/compute/index");
const index_2 = require("./rules/databases/index");
const developer_tools_1 = require("./rules/developer_tools");
const index_3 = require("./rules/machine_learning/index");
const management_and_governance_1 = require("./rules/management_and_governance");
const media_services_1 = require("./rules/media_services");
const index_4 = require("./rules/network_and_delivery/index");
const security_and_compliance_1 = require("./rules/security_and_compliance");
const serverless_1 = require("./rules/serverless");
const index_5 = require("./rules/storage/index");
/**
 * Check Best practices based on AWS Solutions Security Matrix.
 *
 * @stability stable
 */
class AwsSolutionsChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkCompute(node, ignores);
            this.checkStorage(node, ignores);
            this.checkDatabases(node, ignores);
            this.checkNetworkDelivery(node, ignores);
            this.checkManagementGovernance(node, ignores);
            this.checkMachineLearning(node, ignores);
            this.checkAnalytics(node, ignores);
            this.checkSecurityCompliance(node, ignores);
            this.checkServerless(node, ignores);
            this.checkApplicationIntegration(node, ignores);
            this.checkMediaServices(node, ignores);
            this.checkDeveloperTools(node, ignores);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc23,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC26',
            info: 'The EBS volume has encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc26,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc27,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc28,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc29,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEcr1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEcs4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECS7',
            info: 'The ECS Task Definition does not have awslogs logging enabled at the minimum.',
            explanation: 'Container logging allows operators to view and aggregate the logs from the container.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEcs7,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of The CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB2a',
            info: 'The ALB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb2a,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB2e',
            info: 'The CLB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb2e,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb5,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsS1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsS2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-S3',
            info: 'The S3 Bucket does not default encryption enabled.',
            explanation: 'The bucket should minimally have SSE enabled to help protect data-at-rest.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsS3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsEfs1,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds6,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'The deletion protection feature helps protect the database from being accidentally deleted.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds10,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds11,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds13,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds14,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds16,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: common_1.NagMessageLevel.WARN,
            rule: index_2.awsSolutionsDdb3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDdb4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec5,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec6,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N2',
            info: 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN5,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS1',
            info: 'The Redshift cluster parameter group must have the "require_ssl" parameter enabled.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs5,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs6,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs8,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs9,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs10,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc5,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: common_1.NagMessageLevel.WARN,
            rule: index_4.awsSolutionsVpc3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: common_1.NagMessageLevel.WARN,
            rule: index_4.awsSolutionsCfr1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: common_1.NagMessageLevel.WARN,
            rule: index_4.awsSolutionsCfr2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR3',
            info: 'The CloudFront distributions does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsCfr3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsCfr5,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR6',
            info: 'The CloudFront distribution does not use an origin access identity an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsCfr6,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsApig1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsApig4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsApig6,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: common_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: common_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: common_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs3,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_3.awsSolutionsSm1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_3.awsSolutionsSm2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: common_1.NagMessageLevel.ERROR,
            rule: index_3.awsSolutionsSm3,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-ATH1',
            info: 'The Athena workgroup does not encrypt query results.',
            explanation: 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsAth1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsEmr2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsEmr6,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: common_1.NagMessageLevel.WARN,
            rule: analytics_1.awsSolutionsKda3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsKds1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDF1',
            info: 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsKdf1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk6,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS5',
            info: 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs5,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs7,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs8,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs9,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: common_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsQs1,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.',
            level: common_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsIam4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.',
            level: common_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsIam5,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: common_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: common_1.NagMessageLevel.WARN,
            rule: security_and_compliance_1.awsSolutionsCog2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: common_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: common_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog7,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KMS5',
            info: 'The KMS Symmetric key does not have Key Rotation enabled.',
            explanation: 'KMS Key Rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: common_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsKms5,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: common_1.NagMessageLevel.ERROR,
            rule: serverless_1.awsSolutionsSf1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: common_1.NagMessageLevel.ERROR,
            rule: serverless_1.awsSolutionsSf2,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-SNS2',
            info: 'The SNS Topic does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: common_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSns2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: common_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSqs2,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SQS3',
            info: 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: common_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSqs3,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: common_1.NagMessageLevel.ERROR,
            rule: media_services_1.awsSolutionsMs1,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: common_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: common_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs7,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: common_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs8,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: common_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs10,
            ignores: ignores,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node, ignores) {
        this.applyRule({
            ruleId: 'AwsSolutions-CB3',
            info: 'The CodeBuild project has privileged mode enabled.',
            explanation: 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.',
            level: common_1.NagMessageLevel.WARN,
            rule: developer_tools_1.awsSolutionsCb3,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: common_1.NagMessageLevel.ERROR,
            rule: developer_tools_1.awsSolutionsCb4,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: common_1.NagMessageLevel.WARN,
            rule: developer_tools_1.awsSolutionsCb5,
            ignores: ignores,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: common_1.NagMessageLevel.ERROR,
            rule: developer_tools_1.awsSolutionsC91,
            ignores: ignores,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "0.0.125" };
//# sourceMappingURL=data:application/json;base64,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