import xarray as xr
import pandas as pd
import os
import re
from collections import defaultdict

class FileLoader():
    '''
    Abstract class defining a generic scattering file loader.  Input is a (or multiple) filename/s and output is a
    xarray I(pix_x,pix_y,dims,coords) where dims and coords are loaded by user request.
    
    
     Difference: all coords are dims but not all dims are coords.  Dims can also be auto-hinted using the following   
     standard names: energy,exposure,pos_x,pos_y,pos_z,theta.
     
     Individual loaders can try searching metadata for other dim names but this is not guaranteed.
     Coords can be used to provide a list of values for a dimension when that dimension cannot be hinted, e.g. where vals
     come from external data.
    '''
    file_ext = ''  # file extension to be used to filter files from this instrument
    md_loading_is_quick = False
    
    def loadSingleImage(self,filepath,coords=None):
        raise NotImplementedError
    
    def peekAtMd(self,filepath):
        return self.loadSingleImage(filepath,{})
    

    def loadFileSeries(self,basepath,dims,coords={},file_filter='',file_skip='donotskip',md_filter={},quiet=False):
        '''
        Load a series into a single xarray.
        
        Args:
            basepath (str or Path): path to the directory to load
            dims (list): dimensions of the resulting xarray, as list of str
            coords (dict): dictionary of any dims that are *not* present in metadata
            file_filter (str): string that must be in each file name
            file_skip (str): string that, if present in file name, means file should be skipped.
            md_filter (dict): dict of *required* metadata values; points without these metadata values will be dropped
            quiet (bool): skip printing most intermediate output if true.
        
        '''

        nfiles = len(os.listdir(basepath))
        nprocessed = 0
        filesintegrated = 0
        print(f'Found {str(nfiles)} files.')
        data_rows = []
        qnew = None
        dest_coords = defaultdict(list)
        for file in os.listdir(basepath):
            nprocessed += 1
            local_coords = {}
            if (re.match(self.file_ext,file) is not None) and file_filter in file and file_skip not in file:
                for key,value in coords.items():
                    local_coords[key] = value[file] 
                if self.md_loading_is_quick:
                    #if metadata loading is quick, we can just peek at the metadata and decide what to do
                    md = self.peekAtMd(basepath+file)
                    img = None
                else:
                    img = self.loadSingleImage(basepath+file,coords=local_coords)
                    md = self.peekAtMd(basepath+file)
                load_this_image = True
                for key,val in md_filter.items():
                    if md[key] != md_filter[key]:
                        load_this_image = False
                        if not quiet:
                            print(f'Not loading {file}, expected {key} to be {val} but it was {md[key]}')
                if load_this_image:
                    if img == None:
                        if not quiet:
                            print(f'Loading {file}')
                        img = self.loadSingleImage(basepath+file,coords=local_coords)
                        # this is a dataarray with dims ['pix_x', 'pix_y']+attrs (standardized)
                        # e.g. generated by img = xr.DataArray(img,dims=['pix_x','pix_y'],
                        #      coords={},attrs=headerdict)
                    data_rows.append(img)
                    for dim in dims:
                        dest_coords[dim].append(img.attrs[dim])
                    
                #update_progress(float(nprocessed)/nfiles,prestring="Loading file " + str(nprocessed) + " of "+
                #    str(nfiles)+" -- "+file)
        #prepare the index...
        dest_coords_sorted = sorted(dest_coords.items())
        
        vals = []
        keys = []
        
        for key,val in dest_coords_sorted:
            vals.append(val)
            keys.append(key)

        #index = pd.MultiIndex.from_arrays(
        #        list(dest_coords.values()
        #        ),
        #    names=
        #    list(dest_coords.keys()
        #    )
        #)
        index = pd.MultiIndex.from_arrays(vals,names=keys)
        index.name = 'system'
        out = xr.concat(data_rows,dim=index)
        out.attrs.update({'dims_unpacked':dims})
        return out