"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StacIngestor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_python_alpha_1 = require("@aws-cdk/aws-lambda-python-alpha");
const constructs_1 = require("constructs");
class StacIngestor extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.table = this.buildTable();
        const env = {
            DYNAMODB_TABLE: this.table.tableName,
            ROOT_PATH: `/${props.stage}`,
            NO_PYDANTIC_SSM_SETTINGS: "1",
            STAC_URL: props.stacUrl,
            DATA_ACCESS_ROLE: props.dataAccessRole.roleArn,
            ...props.apiEnv,
        };
        const handler = this.buildApiLambda({
            table: this.table,
            env,
            dataAccessRole: props.dataAccessRole,
            stage: props.stage,
        });
        this.buildApiEndpoint({
            handler,
            stage: props.stage,
            endpointConfiguration: props.apiEndpointConfiguration,
            policy: props.apiPolicy,
        });
        this.buildIngestor({
            table: this.table,
            env: env,
            dbSecret: props.stacDbSecret,
            dbVpc: props.vpc,
            dbSecurityGroup: props.stacDbSecurityGroup,
            subnetSelection: props.subnetSelection,
        });
        this.registerSsmParameter({
            name: "dynamodb_table",
            value: this.table.tableName,
            description: "Name of table used to store ingestions",
        });
    }
    buildTable() {
        const table = new aws_cdk_lib_1.aws_dynamodb.Table(this, "ingestions-table", {
            partitionKey: { name: "created_by", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            sortKey: { name: "id", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            billingMode: aws_cdk_lib_1.aws_dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            stream: aws_cdk_lib_1.aws_dynamodb.StreamViewType.NEW_IMAGE,
        });
        table.addGlobalSecondaryIndex({
            indexName: "status",
            partitionKey: { name: "status", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            sortKey: { name: "created_at", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
        });
        return table;
    }
    buildApiLambda(props) {
        const handler_role = new aws_cdk_lib_1.aws_iam.Role(this, "execution-role", {
            description: "Role used by STAC Ingestor. Manually defined so that we can choose a name that is supported by the data access roles trust policy",
            roleName: `stac-ingestion-api-${props.stage}`,
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("lambda.amazonaws.com"),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"),
            ],
        });
        const handler = new aws_lambda_python_alpha_1.PythonFunction(this, "api-handler", {
            entry: `${__dirname}/runtime`,
            index: "src/handler.py",
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            environment: props.env,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: handler_role,
            memorySize: 2048,
        });
        props.table.grantReadWriteData(handler);
        props.dataAccessRole.grant(handler.grantPrincipal, "sts:AssumeRole");
        return handler;
    }
    buildIngestor(props) {
        const handler = new aws_lambda_python_alpha_1.PythonFunction(this, "stac-ingestor", {
            entry: `${__dirname}/runtime`,
            index: "src/ingestor.py",
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            timeout: aws_cdk_lib_1.Duration.seconds(180),
            environment: { DB_SECRET_ARN: props.dbSecret.secretArn, ...props.env },
            vpc: props.dbVpc,
            vpcSubnets: props.subnetSelection,
            allowPublicSubnet: true,
            memorySize: 2048,
        });
        // Allow handler to read DB secret
        props.dbSecret.grantRead(handler);
        // Allow handler to connect to DB
        props.dbSecurityGroup.addIngressRule(handler.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connections from STAC Ingestor");
        // Allow handler to write results back to DBƒ
        props.table.grantWriteData(handler);
        // Trigger handler from writes to DynamoDB table
        handler.addEventSource(new aws_cdk_lib_1.aws_lambda_event_sources.DynamoEventSource(props.table, {
            // Read when batches reach size...
            batchSize: 1000,
            // ... or when window is reached.
            maxBatchingWindow: aws_cdk_lib_1.Duration.seconds(10),
            // Read oldest data first.
            startingPosition: aws_cdk_lib_1.aws_lambda.StartingPosition.TRIM_HORIZON,
            retryAttempts: 1,
        }));
        return handler;
    }
    buildApiEndpoint(props) {
        return new aws_cdk_lib_1.aws_apigateway.LambdaRestApi(this, `${aws_cdk_lib_1.Stack.of(this).stackName}-ingestor-api`, {
            handler: props.handler,
            proxy: true,
            cloudWatchRole: true,
            deployOptions: { stageName: props.stage },
            endpointExportName: `ingestor-api-${props.stage}`,
            endpointConfiguration: props.endpointConfiguration,
            policy: props.policy,
        });
    }
    registerSsmParameter(props) {
        const parameterNamespace = aws_cdk_lib_1.Stack.of(this).stackName;
        return new aws_cdk_lib_1.aws_ssm.StringParameter(this, `${props.name.replace("_", "-")}-parameter`, {
            description: props.description,
            parameterName: `/${parameterNamespace}/${props.name}`,
            stringValue: props.value,
        });
    }
}
exports.StacIngestor = StacIngestor;
_a = JSII_RTTI_SYMBOL_1;
StacIngestor[_a] = { fqn: "cdk-pgstac.StacIngestor", version: "2.3.0" };
//# sourceMappingURL=data:application/json;base64,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