"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHost = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The database is located in an isolated subnet, meaning that it is not accessible from the public internet. As such, to interact with the database directly, a user must tunnel through a bastion host.
 *
 * ### Configuring
 *
 * This codebase controls _who_ is allowed to connect to the bastion host. This requires two steps:
 *
 * 1. Adding the IP address from which you are connecting to the `ipv4Allowlist` array
 * 1. Creating a bastion host system user by adding the user's configuration inform to `userdata.yaml`
 *
 * #### Adding an IP address to the `ipv4Allowlist` array
 *
 * The `BastionHost` construct takes in an `ipv4Allowlist` array as an argument. Find your IP address (eg `curl api.ipify.org`) and add that to the array along with the trailing CIDR block (likely `/32` to indicate that you are adding a single IP address).
 *
 * #### Creating a user via `userdata.yaml`
 *
 * Add an entry to the `users` array with a username (likely matching your local systems username, which you can get by running the `whoami` command in your terminal) and a public key (likely your default public key, which you can get by running `cat ~/.ssh/id_*.pub` in your terminal).
 *
 * #### Tips & Tricks when using the Bastion Host
 *
 * **Connecting to RDS Instance via SSM**
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID \
 * --document-name AWS-StartPortForwardingSessionToRemoteHost \
 * --parameters '{
 * "host": [
 * "example-db.c5abcdefghij.us-west-2.rds.amazonaws.com"
 * ],
 * "portNumber": [
 * "5432"
 * ],
 * "localPortNumber": [
 * "9999"
 * ]
 * }' \
 * --profile $AWS_PROFILE
 * ```
 *
 * ```sh
 * psql -h localhost -p 9999 # continue adding username (-U) and db (-d) here...
 * ```
 *
 * Connect directly to Bastion Host:
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID --profile $AWS_PROFILE
 * ```
 *
 * **Setting up an SSH tunnel**
 *
 * In your `~/.ssh/config` file, add an entry like:
 *
 * ```
 * Host db-tunnel
 * Hostname {the-bastion-host-address}
 * LocalForward 54322 {the-db-hostname}:5432
 * ```
 *
 * Then a tunnel can be opened via:
 *
 * ```
 * ssh -N db-tunnel
 * ```
 *
 * And a connection to the DB can be made via:
 *
 * ```
 * psql -h 127.0.0.1 -p 5433 -U {username} -d {database}
 * ```
 *
 * **Handling `REMOTE HOST IDENTIFICATION HAS CHANGED!` error**
 *
 * If you've redeployed a bastion host that you've previously connected to, you may see an error like:
 *
 * ```
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * @    WARNING: REMOTE HOST IDENTIFICATION HAS CHANGED!     @
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * IT IS POSSIBLE THAT SOMEONE IS DOING SOMETHING NASTY!
 * Someone could be eavesdropping on you right now (man-in-the-middle attack)!
 * It is also possible that a host key has just been changed.
 * The fingerprint for the ECDSA key sent by the remote host is
 * SHA256:mPnxAOXTpb06PFgI1Qc8TMQ2e9b7goU8y2NdS5hzIr8.
 * Please contact your system administrator.
 * Add correct host key in /Users/username/.ssh/known_hosts to get rid of this message.
 * Offending ECDSA key in /Users/username/.ssh/known_hosts:28
 * ECDSA host key for ec2-12-34-56-789.us-west-2.compute.amazonaws.com has changed and you have requested strict checking.
 * Host key verification failed.
 * ```
 *
 * This is due to the server's fingerprint changing. We can scrub the fingerprint from our system with a command like:
 *
 * ```
 * ssh-keygen -R 12.34.56.789
 * ```
 *
 */
class BastionHost extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { stackName } = aws_cdk_lib_1.Stack.of(this);
        // Build ec2 instance
        this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, "bastion-host", {
            vpc: props.vpc,
            vpcSubnets: { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC },
            instanceName: stackName,
            instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE4_GRAVITON, aws_cdk_lib_1.aws_ec2.InstanceSize.NANO),
            machineImage: aws_cdk_lib_1.aws_ec2.MachineImage.latestAmazonLinux({
                generation: aws_cdk_lib_1.aws_ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                cpuType: aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.ARM_64,
            }),
            userData: props.userData,
            userDataCausesReplacement: true,
        });
        // Assign elastic IP
        if (props.createElasticIp) {
            new aws_cdk_lib_1.aws_ec2.CfnEIP(this, "IP", {
                instanceId: this.instance.instanceId,
                tags: [{ key: "Name", value: stackName }],
            });
        }
        // Allow bastion host to connect to db
        this.instance.connections.allowTo(props.db.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connection from bastion host");
        // Allow IP access to bastion host
        for (const ipv4 of props.ipv4Allowlist) {
            this.instance.connections.allowFrom(aws_cdk_lib_1.aws_ec2.Peer.ipv4(ipv4), aws_cdk_lib_1.aws_ec2.Port.tcp(props.sshPort || 22), "SSH Access");
        }
        // Integrate with SSM
        this.instance.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: [
                "ssmmessages:*",
                "ssm:UpdateInstanceInformation",
                "ec2messages:*",
            ],
            resources: ["*"],
        }));
        new aws_cdk_lib_1.CfnOutput(this, "instance-id-output", {
            value: this.instance.instanceId,
            exportName: `${stackName}-instance-id`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-ip-output", {
            value: this.instance.instancePublicIp,
            exportName: `${stackName}-instance-public-ip`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-dns-name-output", {
            value: this.instance.instancePublicDnsName,
            exportName: `${stackName}-public-dns-name`,
        });
    }
}
exports.BastionHost = BastionHost;
_a = JSII_RTTI_SYMBOL_1;
BastionHost[_a] = { fqn: "cdk-pgstac.BastionHost", version: "2.3.0" };
//# sourceMappingURL=data:application/json;base64,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