"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends aws_cdk_lib_1.Resource {
    static fromTableArn(scope, id, tableArn) {
        const tableName = aws_cdk_lib_1.Fn.select(1, aws_cdk_lib_1.Fn.split('/', aws_cdk_lib_1.Stack.of(scope).splitArn(tableArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, {}).toLowerCase(),
                }),
        });
        /**
         * Partition indexes must be created one at a time. To avoid
         * race conditions, we store the resource and add dependencies
         * each time a new partition index is created.
         */
        this.partitionIndexCustomResources = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = props.s3Prefix ?? '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = props.compressed ?? false;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new aws_glue_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${this.physicalName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    'classification': props.dataFormat.classificationString?.value,
                    'has_encrypted_data': this.encryption !== TableEncryption.UNENCRYPTED,
                    'partition_filtering.enabled': props.enablePartitionFiltering,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories ?? false,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
        // Partition index creation relies on created table.
        if (props.partitionIndexes) {
            this.partitionIndexes = props.partitionIndexes;
            this.partitionIndexes.forEach((index) => this.addPartitionIndex(index));
        }
    }
    /**
     * Add a partition index to the table. You can have a maximum of 3 partition
     * indexes to a table. Partition index keys must be a subset of the table's
     * partition keys.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/partition-indexes.html
     */
    addPartitionIndex(index) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PartitionIndex(index);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPartitionIndex);
            }
            throw error;
        }
        const numPartitions = this.partitionIndexCustomResources.length;
        if (numPartitions >= 3) {
            throw new Error('Maximum number of partition indexes allowed is 3');
        }
        this.validatePartitionIndex(index);
        const indexName = index.indexName ?? this.generateIndexName(index.keyNames);
        const partitionIndexCustomResource = new cr.AwsCustomResource(this, `partition-index-${indexName}`, {
            onCreate: {
                service: 'Glue',
                action: 'createPartitionIndex',
                parameters: {
                    DatabaseName: this.database.databaseName,
                    TableName: this.tableName,
                    PartitionIndex: {
                        IndexName: indexName,
                        Keys: index.keyNames,
                    },
                },
                physicalResourceId: cr.PhysicalResourceId.of(indexName),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            // APIs are available in 2.1055.0
            installLatestAwsSdk: false,
        });
        this.grantToUnderlyingResources(partitionIndexCustomResource, ['glue:UpdateTable']);
        // Depend on previous partition index if possible, to avoid race condition
        if (numPartitions > 0) {
            this.partitionIndexCustomResources[numPartitions - 1].node.addDependency(partitionIndexCustomResource);
        }
        this.partitionIndexCustomResources.push(partitionIndexCustomResource);
    }
    generateIndexName(keys) {
        const prefix = keys.join('-') + '-';
        const uniqueId = aws_cdk_lib_1.Names.uniqueId(this);
        const maxIndexLength = 80; // arbitrarily specified
        const startIndex = Math.max(0, uniqueId.length - (maxIndexLength - prefix.length));
        return prefix + uniqueId.substring(startIndex);
    }
    validatePartitionIndex(index) {
        if (index.indexName !== undefined && (index.indexName.length < 1 || index.indexName.length > 255)) {
            throw new Error(`Index name must be between 1 and 255 characters, but got ${index.indexName.length}`);
        }
        if (!this.partitionKeys || this.partitionKeys.length === 0) {
            throw new Error('The table must have partition keys to create a partition index');
        }
        const keyNames = this.partitionKeys.map(pk => pk.name);
        if (!index.keyNames.every(k => keyNames.includes(k))) {
            throw new Error(`All index keys must also be partition keys. Got ${index.keyNames} but partition key names are ${keyNames}`);
        }
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant the given identity custom permissions.
     */
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
    /**
     * Grant the given identity custom permissions to ALL underlying resources of the table.
     * Permissions will be granted to the catalog, the database, and the table.
     */
    grantToUnderlyingResources(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [
                this.tableArn,
                this.database.catalogArn,
                this.database.databaseArn,
            ],
            actions,
        });
    }
    getS3PrefixForGrant() {
        return this.s3Prefix + '*';
    }
}
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-glue-alpha.Table", version: "2.73.0-alpha.0" };
exports.Table = Table;
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersion',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFibGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0YWJsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSwyQ0FBMkM7QUFDM0MsMkNBQTJDO0FBQzNDLHlDQUF5QztBQUN6Qyw2Q0FBcUY7QUFDckYsbURBQW1EO0FBS25ELG1EQUFnRDtBQWlDaEQ7Ozs7R0FJRztBQUNILElBQVksZUE0Qlg7QUE1QkQsV0FBWSxlQUFlO0lBQ3pCLDhDQUEyQixDQUFBO0lBRTNCOzs7O09BSUc7SUFDSCx3Q0FBcUIsQ0FBQTtJQUVyQjs7OztPQUlHO0lBQ0gsa0NBQWUsQ0FBQTtJQUVmOztPQUVHO0lBQ0gsa0RBQStCLENBQUE7SUFFL0I7Ozs7T0FJRztJQUNILDhDQUEyQixDQUFBO0FBQzdCLENBQUMsRUE1QlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUE0QjFCO0FBZ0hEOztHQUVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsc0JBQVE7SUFFMUIsTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQjtRQUN2RSxNQUFNLFNBQVMsR0FBRyxnQkFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsZ0JBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLG1CQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsdUJBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFlBQWEsQ0FBQyxDQUFDLENBQUM7UUFFL0gsT0FBTyxLQUFLLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMxQyxRQUFRO1lBQ1IsU0FBUztTQUNWLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCOzs7Ozs7Ozs7O1FBQ3BGLE1BQU0sTUFBTyxTQUFRLHNCQUFRO1lBQTdCOztnQkFDa0IsYUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7Z0JBQzFCLGNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQzlDLENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBcUVELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUI7UUFDekQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQzNCLGtCQUFJLENBQUMsTUFBTSxDQUFDO29CQUNWLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxtQkFBSyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQyxXQUFXLEVBQUU7aUJBQ2hFLENBQUM7U0FDTCxDQUFDLENBQUM7UUFiTDs7OztXQUlHO1FBQ0ssa0NBQTZCLEdBQXdCLEVBQUUsQ0FBQzs7Ozs7OytDQTVGckQsS0FBSzs7OztRQXNHZCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ25DLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUM7UUFFckMsY0FBYyxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM3QixJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFFekMsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQztRQUM1QyxNQUFNLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxhQUFhLEVBQUUsR0FBRyxZQUFZLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hFLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO1FBQ3JCLElBQUksQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDO1FBQzdCLElBQUksQ0FBQyxhQUFhLEdBQUcsYUFBYSxDQUFDO1FBRW5DLE1BQU0sYUFBYSxHQUFHLElBQUksbUJBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFO1lBQ2hELFNBQVMsRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVM7WUFFbkMsWUFBWSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWTtZQUV6QyxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZO2dCQUN2QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVcsSUFBSSxHQUFHLElBQUksQ0FBQyxZQUFZLG1CQUFtQjtnQkFFekUsYUFBYSxFQUFFLGFBQWEsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO2dCQUVqRCxVQUFVLEVBQUU7b0JBQ1YsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxvQkFBb0IsRUFBRSxLQUFLO29CQUM5RCxvQkFBb0IsRUFBRSxJQUFJLENBQUMsVUFBVSxLQUFLLGVBQWUsQ0FBQyxXQUFXO29CQUNyRSw2QkFBNkIsRUFBRSxLQUFLLENBQUMsd0JBQXdCO2lCQUM5RDtnQkFDRCxpQkFBaUIsRUFBRTtvQkFDakIsUUFBUSxFQUFFLFFBQVEsSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtvQkFDM0QsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO29CQUMzQixzQkFBc0IsRUFBRSxLQUFLLENBQUMsc0JBQXNCLElBQUksS0FBSztvQkFDN0QsT0FBTyxFQUFFLGFBQWEsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDO29CQUNyQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsU0FBUztvQkFDbkQsWUFBWSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLFNBQVM7b0JBQ3JELFNBQVMsRUFBRTt3QkFDVCxvQkFBb0IsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLG9CQUFvQixDQUFDLFNBQVM7cUJBQ3RFO2lCQUNGO2dCQUVELFNBQVMsRUFBRSxnQkFBZ0I7YUFDNUI7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbEUsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNuQyxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFlBQVksRUFBRSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7U0FDaEUsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsYUFBYSxDQUFDO1FBRXZDLG9EQUFvRDtRQUNwRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1lBQy9DLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQ3pFO0tBQ0Y7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUIsQ0FBQyxLQUFxQjs7Ozs7Ozs7OztRQUM1QyxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsNkJBQTZCLENBQUMsTUFBTSxDQUFDO1FBQ2hFLElBQUksYUFBYSxJQUFJLENBQUMsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7U0FDckU7UUFDRCxJQUFJLENBQUMsc0JBQXNCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFbkMsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzVFLE1BQU0sNEJBQTRCLEdBQUcsSUFBSSxFQUFFLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixTQUFTLEVBQUUsRUFBRTtZQUNsRyxRQUFRLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLE1BQU07Z0JBQ2YsTUFBTSxFQUFFLHNCQUFzQjtnQkFDOUIsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVk7b0JBQ3hDLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztvQkFDekIsY0FBYyxFQUFFO3dCQUNkLFNBQVMsRUFBRSxTQUFTO3dCQUNwQixJQUFJLEVBQUUsS0FBSyxDQUFDLFFBQVE7cUJBQ3JCO2lCQUNGO2dCQUNELGtCQUFrQixFQUFFLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLENBQzFDLFNBQVMsQ0FDVjthQUNGO1lBQ0QsTUFBTSxFQUFFLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxZQUFZLENBQUM7Z0JBQzlDLFNBQVMsRUFBRSxFQUFFLENBQUMsdUJBQXVCLENBQUMsWUFBWTthQUNuRCxDQUFDO1lBQ0YsaUNBQWlDO1lBQ2pDLG1CQUFtQixFQUFFLEtBQUs7U0FDM0IsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLDBCQUEwQixDQUFDLDRCQUE0QixFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1FBRXBGLDBFQUEwRTtRQUMxRSxJQUFJLGFBQWEsR0FBRyxDQUFDLEVBQUU7WUFDckIsSUFBSSxDQUFDLDZCQUE2QixDQUFDLGFBQWEsR0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDRCQUE0QixDQUFDLENBQUM7U0FDdEc7UUFDRCxJQUFJLENBQUMsNkJBQTZCLENBQUMsSUFBSSxDQUFDLDRCQUE0QixDQUFDLENBQUM7S0FDdkU7SUFFTyxpQkFBaUIsQ0FBQyxJQUFjO1FBQ3RDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsR0FBRyxDQUFDO1FBQ3BDLE1BQU0sUUFBUSxHQUFHLG1CQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RDLE1BQU0sY0FBYyxHQUFHLEVBQUUsQ0FBQyxDQUFDLHdCQUF3QjtRQUNuRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsY0FBYyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ25GLE9BQU8sTUFBTSxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLENBQUM7S0FDaEQ7SUFFTyxzQkFBc0IsQ0FBQyxLQUFxQjtRQUNsRCxJQUFJLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLEtBQUssQ0FBQyxTQUFTLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxFQUFFO1lBQ2pHLE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELEtBQUssQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztTQUN2RztRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMxRCxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7U0FDbkY7UUFDRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxtREFBbUQsS0FBSyxDQUFDLFFBQVEsZ0NBQWdDLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDOUg7S0FDRjtJQUVEOzs7O09BSUc7SUFDSSxTQUFTLENBQUMsT0FBdUI7UUFDdEMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDakQsSUFBSSxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssZUFBZSxDQUFDLGVBQWUsRUFBRTtZQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQUU7UUFDNUgsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7UUFDM0QsT0FBTyxHQUFHLENBQUM7S0FDWjtJQUVEOzs7O09BSUc7SUFDSSxVQUFVLENBQUMsT0FBdUI7UUFDdkMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUNsRCxJQUFJLElBQUksQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxFQUFFO1lBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7U0FBRTtRQUM1SCxJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQztRQUM1RCxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBRUQ7Ozs7T0FJRztJQUNJLGNBQWMsQ0FBQyxPQUF1QjtRQUMzQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLEdBQUcsZUFBZSxFQUFFLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1FBQzNFLElBQUksSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLGVBQWUsQ0FBQyxlQUFlLEVBQUU7WUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQUU7UUFDbkksSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7UUFDaEUsT0FBTyxHQUFHLENBQUM7S0FDWjtJQUVEOztPQUVHO0lBQ0ksS0FBSyxDQUFDLE9BQXVCLEVBQUUsT0FBaUI7UUFDckQsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUM5QixPQUFPO1lBQ1AsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QixPQUFPO1NBQ1IsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7O09BR0c7SUFDSSwwQkFBMEIsQ0FBQyxPQUF1QixFQUFFLE9BQWlCO1FBQzFFLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLFlBQVksRUFBRTtnQkFDWixJQUFJLENBQUMsUUFBUTtnQkFDYixJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVU7Z0JBQ3hCLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVzthQUMxQjtZQUNELE9BQU87U0FDUixDQUFDLENBQUM7S0FDSjtJQUVPLG1CQUFtQjtRQUN6QixPQUFPLElBQUksQ0FBQyxRQUFRLEdBQUcsR0FBRyxDQUFDO0tBQzVCOzs7O0FBdlNVLHNCQUFLO0FBMFNsQixTQUFTLGNBQWMsQ0FBQyxPQUFpQixFQUFFLGFBQXdCO0lBQ2pFLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO0tBQ3ZFO0lBQ0QsdUZBQXVGO0lBQ3ZGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxFQUFVLENBQUM7SUFDaEMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLGFBQWEsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUNyRCxJQUFJLEtBQUssQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQzFCLE1BQU0sSUFBSSxLQUFLLENBQUMseURBQXlELE1BQU0sQ0FBQyxJQUFJLGtCQUFrQixDQUFDLENBQUM7U0FDekc7UUFDRCxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN6QixDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRCwwRUFBMEU7QUFDMUUsTUFBTSxrQkFBa0IsR0FBRztJQUN6QixDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUMsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtJQUM1RCxDQUFDLGVBQWUsQ0FBQyxXQUFXLENBQUMsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVztJQUM5RCxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsR0FBRztJQUM5QyxDQUFDLGVBQWUsQ0FBQyxlQUFlLENBQUMsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVztJQUNsRSxDQUFDLGVBQWUsQ0FBQyxXQUFXLENBQUMsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVztDQUMvRCxDQUFDO0FBRUYsMEdBQTBHO0FBQzFHLFNBQVMsWUFBWSxDQUFDLEtBQVksRUFBRSxLQUFpQjtJQUNuRCxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLGVBQWUsQ0FBQyxXQUFXLENBQUM7SUFDbkUsSUFBSSxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztJQUUxQixJQUFJLE1BQU0sSUFBSSxDQUFDLFVBQVUsS0FBSyxlQUFlLENBQUMsV0FBVyxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxDQUFDLEVBQUU7UUFDNUcsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsQ0FBQyxDQUFDO0tBQ3pGO0lBRUQsSUFBSSxhQUFtQyxDQUFDO0lBQ3hDLElBQUksVUFBVSxLQUFLLGVBQWUsQ0FBQyxlQUFlLElBQUksS0FBSyxDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7UUFDdkYsK0ZBQStGO1FBQy9GLHlGQUF5RjtRQUN6RixhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztLQUMzQztTQUFNO1FBQ0wsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7S0FDckM7SUFFRCx5Q0FBeUM7SUFDekMsSUFBSSxDQUFDLE1BQU0sRUFBRTtRQUNYLElBQUksVUFBVSxLQUFLLGVBQWUsQ0FBQyxlQUFlLEVBQUU7WUFDbEQsTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7U0FDekM7YUFBTTtZQUNMLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDdEMsVUFBVSxFQUFFLGtCQUFrQixDQUFDLFVBQVUsQ0FBQztnQkFDMUMsYUFBYTthQUNkLENBQUMsQ0FBQztZQUNILGFBQWEsR0FBRyxNQUFNLENBQUMsYUFBYSxDQUFDO1NBQ3RDO0tBQ0Y7SUFFRCxPQUFPO1FBQ0wsTUFBTTtRQUNOLFVBQVU7UUFDVixhQUFhO0tBQ2QsQ0FBQztBQUNKLENBQUM7QUFFRCxNQUFNLGVBQWUsR0FBRztJQUN0Qix3QkFBd0I7SUFDeEIsbUJBQW1CO0lBQ25CLG9CQUFvQjtJQUNwQixlQUFlO0lBQ2YsZ0JBQWdCO0lBQ2hCLHNCQUFzQjtJQUN0Qix1QkFBdUI7Q0FDeEIsQ0FBQztBQUVGLE1BQU0sZ0JBQWdCLEdBQUc7SUFDdkIsMkJBQTJCO0lBQzNCLDJCQUEyQjtJQUMzQixzQkFBc0I7SUFDdEIsc0JBQXNCO0lBQ3RCLHNCQUFzQjtDQUN2QixDQUFDO0FBRUYsU0FBUyxhQUFhLENBQUMsT0FBZ0M7SUFDckQsSUFBSSxPQUFPLEtBQUssU0FBUyxFQUFFO1FBQ3pCLE9BQU8sU0FBUyxDQUFDO0tBQ2xCO0lBQ0QsT0FBTyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1FBQzFCLE9BQU87WUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUk7WUFDakIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVztZQUM3QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU87U0FDeEIsQ0FBQztJQUNKLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGttcyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mta21zJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMyc7XG5pbXBvcnQgeyBBcm5Gb3JtYXQsIEZuLCBJUmVzb3VyY2UsIExhenksIE5hbWVzLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBjciBmcm9tICdhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IEF3c0N1c3RvbVJlc291cmNlIH0gZnJvbSAnYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IERhdGFGb3JtYXQgfSBmcm9tICcuL2RhdGEtZm9ybWF0JztcbmltcG9ydCB7IElEYXRhYmFzZSB9IGZyb20gJy4vZGF0YWJhc2UnO1xuaW1wb3J0IHsgQ2ZuVGFibGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZ2x1ZSc7XG5pbXBvcnQgeyBDb2x1bW4gfSBmcm9tICcuL3NjaGVtYSc7XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhIFBhcnRpdGlvbiBJbmRleC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQYXJ0aXRpb25JbmRleCB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgcGFydGl0aW9uIGluZGV4LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmFtZSB3aWxsIGJlIGdlbmVyYXRlZCBmb3IgeW91LlxuICAgKi9cbiAgcmVhZG9ubHkgaW5kZXhOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGFydGl0aW9uIGtleSBuYW1lcyB0aGF0IGNvbXByaXNlIHRoZSBwYXJ0aXRpb25cbiAgICogaW5kZXguIFRoZSBuYW1lcyBtdXN0IGNvcnJlc3BvbmQgdG8gYSBuYW1lIGluIHRoZVxuICAgKiB0YWJsZSdzIHBhcnRpdGlvbiBrZXlzLlxuICAgKi9cbiAgcmVhZG9ubHkga2V5TmFtZXM6IHN0cmluZ1tdO1xufVxuZXhwb3J0IGludGVyZmFjZSBJVGFibGUgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdGFibGVBcm46IHN0cmluZztcblxuICAvKipcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdGFibGVOYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogRW5jcnlwdGlvbiBvcHRpb25zIGZvciBhIFRhYmxlLlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F0aGVuYS9sYXRlc3QvdWcvZW5jcnlwdGlvbi5odG1sXG4gKi9cbmV4cG9ydCBlbnVtIFRhYmxlRW5jcnlwdGlvbiB7XG4gIFVORU5DUllQVEVEID0gJ1VuZW5jcnlwdGVkJyxcblxuICAvKipcbiAgICogU2VydmVyIHNpZGUgZW5jcnlwdGlvbiAoU1NFKSB3aXRoIGFuIEFtYXpvbiBTMy1tYW5hZ2VkIGtleS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ1NlcnZlclNpZGVFbmNyeXB0aW9uLmh0bWxcbiAgICovXG4gIFMzX01BTkFHRUQgPSAnU1NFLVMzJyxcblxuICAvKipcbiAgICogU2VydmVyLXNpZGUgZW5jcnlwdGlvbiAoU1NFKSB3aXRoIGFuIEFXUyBLTVMga2V5IG1hbmFnZWQgYnkgdGhlIGFjY291bnQgb3duZXIuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdLTVNFbmNyeXB0aW9uLmh0bWxcbiAgICovXG4gIEtNUyA9ICdTU0UtS01TJyxcblxuICAvKipcbiAgICogU2VydmVyLXNpZGUgZW5jcnlwdGlvbiAoU1NFKSB3aXRoIGFuIEFXUyBLTVMga2V5IG1hbmFnZWQgYnkgdGhlIEtNUyBzZXJ2aWNlLlxuICAgKi9cbiAgS01TX01BTkFHRUQgPSAnU1NFLUtNUy1NQU5BR0VEJyxcblxuICAvKipcbiAgICogQ2xpZW50LXNpZGUgZW5jcnlwdGlvbiAoQ1NFKSB3aXRoIGFuIEFXUyBLTVMga2V5IG1hbmFnZWQgYnkgdGhlIGFjY291bnQgb3duZXIuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdDbGllbnRTaWRlRW5jcnlwdGlvbi5odG1sXG4gICAqL1xuICBDTElFTlRfU0lERV9LTVMgPSAnQ1NFLUtNUydcbn1cblxuZXhwb3J0IGludGVyZmFjZSBUYWJsZUF0dHJpYnV0ZXMge1xuICByZWFkb25seSB0YWJsZUFybjogc3RyaW5nO1xuICByZWFkb25seSB0YWJsZU5hbWU6IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBUYWJsZVByb3BzIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIHRhYmxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGdlbmVyYXRlZCBieSBDREsuXG4gICAqL1xuICByZWFkb25seSB0YWJsZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlc2NyaXB0aW9uIG9mIHRoZSB0YWJsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZ2VuZXJhdGVkXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAvKipcbiAgICogRGF0YWJhc2UgaW4gd2hpY2ggdG8gc3RvcmUgdGhlIHRhYmxlLlxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YWJhc2U6IElEYXRhYmFzZTtcblxuICAvKipcbiAgICogUzMgYnVja2V0IGluIHdoaWNoIHRvIHN0b3JlIGRhdGEuXG4gICAqXG4gICAqIEBkZWZhdWx0IG9uZSBpcyBjcmVhdGVkIGZvciB5b3VcbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldD86IHMzLklCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFMzIHByZWZpeCB1bmRlciB3aGljaCB0YWJsZSBvYmplY3RzIGFyZSBzdG9yZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcHJlZml4LiBUaGUgZGF0YSB3aWxsIGJlIHN0b3JlZCB1bmRlciB0aGUgcm9vdCBvZiB0aGUgYnVja2V0LlxuICAgKi9cbiAgcmVhZG9ubHkgczNQcmVmaXg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIENvbHVtbnMgb2YgdGhlIHRhYmxlLlxuICAgKi9cbiAgcmVhZG9ubHkgY29sdW1uczogQ29sdW1uW107XG5cbiAgLyoqXG4gICAqIFBhcnRpdGlvbiBjb2x1bW5zIG9mIHRoZSB0YWJsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdGFibGUgaXMgbm90IHBhcnRpdGlvbmVkXG4gICAqL1xuICByZWFkb25seSBwYXJ0aXRpb25LZXlzPzogQ29sdW1uW107XG5cbiAgLyoqXG4gICAqIFBhcnRpdGlvbiBpbmRleGVzIG9uIHRoZSB0YWJsZS4gQSBtYXhpbXVtIG9mIDMgaW5kZXhlc1xuICAgKiBhcmUgYWxsb3dlZCBvbiBhIHRhYmxlLiBLZXlzIGluIHRoZSBpbmRleCBtdXN0IGJlIHBhcnRcbiAgICogb2YgdGhlIHRhYmxlJ3MgcGFydGl0aW9uIGtleXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRhYmxlIGhhcyBubyBwYXJ0aXRpb24gaW5kZXhlc1xuICAgKi9cbiAgcmVhZG9ubHkgcGFydGl0aW9uSW5kZXhlcz86IFBhcnRpdGlvbkluZGV4W107XG5cbiAgLyoqXG4gICAqIFN0b3JhZ2UgdHlwZSBvZiB0aGUgdGFibGUncyBkYXRhLlxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YUZvcm1hdDogRGF0YUZvcm1hdDtcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIHRhYmxlJ3MgZGF0YSBpcyBjb21wcmVzc2VkIG9yIG5vdC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNvbXByZXNzZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUga2luZCBvZiBlbmNyeXB0aW9uIHRvIHNlY3VyZSB0aGUgZGF0YSB3aXRoLlxuICAgKlxuICAgKiBZb3UgY2FuIG9ubHkgcHJvdmlkZSB0aGlzIG9wdGlvbiBpZiB5b3UgYXJlIG5vdCBleHBsaWNpdGx5IHBhc3NpbmcgaW4gYSBidWNrZXQuXG4gICAqXG4gICAqIElmIHlvdSBjaG9vc2UgYFNTRS1LTVNgLCB5b3UgKmNhbiogcHJvdmlkZSBhbiB1bi1tYW5hZ2VkIEtNUyBrZXkgd2l0aCBgZW5jcnlwdGlvbktleWAuXG4gICAqIElmIHlvdSBjaG9vc2UgYENTRS1LTVNgLCB5b3UgKm11c3QqIHByb3ZpZGUgYW4gdW4tbWFuYWdlZCBLTVMga2V5IHdpdGggYGVuY3J5cHRpb25LZXlgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBVbmVuY3J5cHRlZFxuICAgKi9cbiAgcmVhZG9ubHkgZW5jcnlwdGlvbj86IFRhYmxlRW5jcnlwdGlvbjtcblxuICAvKipcbiAgICogRXh0ZXJuYWwgS01TIGtleSB0byB1c2UgZm9yIGJ1Y2tldCBlbmNyeXB0aW9uLlxuICAgKlxuICAgKiBUaGUgYGVuY3J5cHRpb25gIHByb3BlcnR5IG11c3QgYmUgYFNTRS1LTVNgIG9yIGBDU0UtS01TYC5cbiAgICpcbiAgICogQGRlZmF1bHQga2V5IGlzIG1hbmFnZWQgYnkgS01TLlxuICAgKi9cbiAgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgdGFibGUgZGF0YSBpcyBzdG9yZWQgaW4gc3ViZGlyZWN0b3JpZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBzdG9yZWRBc1N1YkRpcmVjdG9yaWVzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRW5hYmxlcyBwYXJ0aXRpb24gZmlsdGVyaW5nLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdGhlbmEvbGF0ZXN0L3VnL2dsdWUtYmVzdC1wcmFjdGljZXMuaHRtbCNnbHVlLWJlc3QtcHJhY3RpY2VzLXBhcnRpdGlvbi1pbmRleFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRoZSBwYXJhbWV0ZXIgaXMgbm90IGRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZVBhcnRpdGlvbkZpbHRlcmluZz86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQSBHbHVlIHRhYmxlLlxuICovXG5leHBvcnQgY2xhc3MgVGFibGUgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElUYWJsZSB7XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tVGFibGVBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgdGFibGVBcm46IHN0cmluZyk6IElUYWJsZSB7XG4gICAgY29uc3QgdGFibGVOYW1lID0gRm4uc2VsZWN0KDEsIEZuLnNwbGl0KCcvJywgU3RhY2sub2Yoc2NvcGUpLnNwbGl0QXJuKHRhYmxlQXJuLCBBcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSkucmVzb3VyY2VOYW1lISkpO1xuXG4gICAgcmV0dXJuIFRhYmxlLmZyb21UYWJsZUF0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7XG4gICAgICB0YWJsZUFybixcbiAgICAgIHRhYmxlTmFtZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgVGFibGUgY29uc3RydWN0IHRoYXQgcmVwcmVzZW50cyBhbiBleHRlcm5hbCB0YWJsZS5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBzY29wZSBjcmVhdGluZyBjb25zdHJ1Y3QgKHVzdWFsbHkgYHRoaXNgKS5cbiAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QncyBpZC5cbiAgICogQHBhcmFtIGF0dHJzIEltcG9ydCBhdHRyaWJ1dGVzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21UYWJsZUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFRhYmxlQXR0cmlidXRlcyk6IElUYWJsZSB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJVGFibGUge1xuICAgICAgcHVibGljIHJlYWRvbmx5IHRhYmxlQXJuID0gYXR0cnMudGFibGVBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgdGFibGVOYW1lID0gYXR0cnMudGFibGVOYW1lO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKipcbiAgICogRGF0YWJhc2UgdGhpcyB0YWJsZSBiZWxvbmdzIHRvLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRhdGFiYXNlOiBJRGF0YWJhc2U7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRoZSB0YWJsZSdzIGRhdGEgaXMgY29tcHJlc3NlZCBvciBub3QuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY29tcHJlc3NlZDogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgZW5jcnlwdGlvbiBlbmFibGVkIGZvciB0aGUgdGFibGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZW5jcnlwdGlvbjogVGFibGVFbmNyeXB0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgS01TIGtleSB1c2VkIHRvIHNlY3VyZSB0aGUgZGF0YSBpZiBgZW5jcnlwdGlvbmAgaXMgc2V0IHRvIGBDU0UtS01TYCBvciBgU1NFLUtNU2AuIE90aGVyd2lzZSwgYHVuZGVmaW5lZGAuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xuXG4gIC8qKlxuICAgKiBTMyBidWNrZXQgaW4gd2hpY2ggdGhlIHRhYmxlJ3MgZGF0YSByZXNpZGVzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICAvKipcbiAgICogUzMgS2V5IFByZWZpeCB1bmRlciB3aGljaCB0aGlzIHRhYmxlJ3MgZmlsZXMgYXJlIHN0b3JlZCBpbiBTMy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzM1ByZWZpeDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoaXMgdGFibGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdGFibGVOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFSTiBvZiB0aGlzIHRhYmxlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRhYmxlQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEZvcm1hdCBvZiB0aGlzIHRhYmxlJ3MgZGF0YSBmaWxlcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkYXRhRm9ybWF0OiBEYXRhRm9ybWF0O1xuXG4gIC8qKlxuICAgKiBUaGlzIHRhYmxlJ3MgY29sdW1ucy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb2x1bW5zOiBDb2x1bW5bXTtcblxuICAvKipcbiAgICogVGhpcyB0YWJsZSdzIHBhcnRpdGlvbiBrZXlzIGlmIHRoZSB0YWJsZSBpcyBwYXJ0aXRpb25lZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwYXJ0aXRpb25LZXlzPzogQ29sdW1uW107XG5cbiAgLyoqXG4gICAqIFRoaXMgdGFibGUncyBwYXJ0aXRpb24gaW5kZXhlcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwYXJ0aXRpb25JbmRleGVzPzogUGFydGl0aW9uSW5kZXhbXTtcblxuICAvKipcbiAgICogUGFydGl0aW9uIGluZGV4ZXMgbXVzdCBiZSBjcmVhdGVkIG9uZSBhdCBhIHRpbWUuIFRvIGF2b2lkXG4gICAqIHJhY2UgY29uZGl0aW9ucywgd2Ugc3RvcmUgdGhlIHJlc291cmNlIGFuZCBhZGQgZGVwZW5kZW5jaWVzXG4gICAqIGVhY2ggdGltZSBhIG5ldyBwYXJ0aXRpb24gaW5kZXggaXMgY3JlYXRlZC5cbiAgICovXG4gIHByaXZhdGUgcGFydGl0aW9uSW5kZXhDdXN0b21SZXNvdXJjZXM6IEF3c0N1c3RvbVJlc291cmNlW10gPSBbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVGFibGVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy50YWJsZU5hbWUgPz9cbiAgICAgICAgTGF6eS5zdHJpbmcoe1xuICAgICAgICAgIHByb2R1Y2U6ICgpID0+IE5hbWVzLnVuaXF1ZVJlc291cmNlTmFtZSh0aGlzLCB7fSkudG9Mb3dlckNhc2UoKSxcbiAgICAgICAgfSksXG4gICAgfSk7XG5cbiAgICB0aGlzLmRhdGFiYXNlID0gcHJvcHMuZGF0YWJhc2U7XG4gICAgdGhpcy5kYXRhRm9ybWF0ID0gcHJvcHMuZGF0YUZvcm1hdDtcbiAgICB0aGlzLnMzUHJlZml4ID0gcHJvcHMuczNQcmVmaXggPz8gJyc7XG5cbiAgICB2YWxpZGF0ZVNjaGVtYShwcm9wcy5jb2x1bW5zLCBwcm9wcy5wYXJ0aXRpb25LZXlzKTtcbiAgICB0aGlzLmNvbHVtbnMgPSBwcm9wcy5jb2x1bW5zO1xuICAgIHRoaXMucGFydGl0aW9uS2V5cyA9IHByb3BzLnBhcnRpdGlvbktleXM7XG5cbiAgICB0aGlzLmNvbXByZXNzZWQgPSBwcm9wcy5jb21wcmVzc2VkID8/IGZhbHNlO1xuICAgIGNvbnN0IHsgYnVja2V0LCBlbmNyeXB0aW9uLCBlbmNyeXB0aW9uS2V5IH0gPSBjcmVhdGVCdWNrZXQodGhpcywgcHJvcHMpO1xuICAgIHRoaXMuYnVja2V0ID0gYnVja2V0O1xuICAgIHRoaXMuZW5jcnlwdGlvbiA9IGVuY3J5cHRpb247XG4gICAgdGhpcy5lbmNyeXB0aW9uS2V5ID0gZW5jcnlwdGlvbktleTtcblxuICAgIGNvbnN0IHRhYmxlUmVzb3VyY2UgPSBuZXcgQ2ZuVGFibGUodGhpcywgJ1RhYmxlJywge1xuICAgICAgY2F0YWxvZ0lkOiBwcm9wcy5kYXRhYmFzZS5jYXRhbG9nSWQsXG5cbiAgICAgIGRhdGFiYXNlTmFtZTogcHJvcHMuZGF0YWJhc2UuZGF0YWJhc2VOYW1lLFxuXG4gICAgICB0YWJsZUlucHV0OiB7XG4gICAgICAgIG5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24gfHwgYCR7dGhpcy5waHlzaWNhbE5hbWV9IGdlbmVyYXRlZCBieSBDREtgLFxuXG4gICAgICAgIHBhcnRpdGlvbktleXM6IHJlbmRlckNvbHVtbnMocHJvcHMucGFydGl0aW9uS2V5cyksXG5cbiAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICdjbGFzc2lmaWNhdGlvbic6IHByb3BzLmRhdGFGb3JtYXQuY2xhc3NpZmljYXRpb25TdHJpbmc/LnZhbHVlLFxuICAgICAgICAgICdoYXNfZW5jcnlwdGVkX2RhdGEnOiB0aGlzLmVuY3J5cHRpb24gIT09IFRhYmxlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCxcbiAgICAgICAgICAncGFydGl0aW9uX2ZpbHRlcmluZy5lbmFibGVkJzogcHJvcHMuZW5hYmxlUGFydGl0aW9uRmlsdGVyaW5nLFxuICAgICAgICB9LFxuICAgICAgICBzdG9yYWdlRGVzY3JpcHRvcjoge1xuICAgICAgICAgIGxvY2F0aW9uOiBgczM6Ly8ke3RoaXMuYnVja2V0LmJ1Y2tldE5hbWV9LyR7dGhpcy5zM1ByZWZpeH1gLFxuICAgICAgICAgIGNvbXByZXNzZWQ6IHRoaXMuY29tcHJlc3NlZCxcbiAgICAgICAgICBzdG9yZWRBc1N1YkRpcmVjdG9yaWVzOiBwcm9wcy5zdG9yZWRBc1N1YkRpcmVjdG9yaWVzID8/IGZhbHNlLFxuICAgICAgICAgIGNvbHVtbnM6IHJlbmRlckNvbHVtbnMocHJvcHMuY29sdW1ucyksXG4gICAgICAgICAgaW5wdXRGb3JtYXQ6IHByb3BzLmRhdGFGb3JtYXQuaW5wdXRGb3JtYXQuY2xhc3NOYW1lLFxuICAgICAgICAgIG91dHB1dEZvcm1hdDogcHJvcHMuZGF0YUZvcm1hdC5vdXRwdXRGb3JtYXQuY2xhc3NOYW1lLFxuICAgICAgICAgIHNlcmRlSW5mbzoge1xuICAgICAgICAgICAgc2VyaWFsaXphdGlvbkxpYnJhcnk6IHByb3BzLmRhdGFGb3JtYXQuc2VyaWFsaXphdGlvbkxpYnJhcnkuY2xhc3NOYW1lLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG5cbiAgICAgICAgdGFibGVUeXBlOiAnRVhURVJOQUxfVEFCTEUnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMudGFibGVOYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUodGFibGVSZXNvdXJjZS5yZWYpO1xuICAgIHRoaXMudGFibGVBcm4gPSB0aGlzLnN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnZ2x1ZScsXG4gICAgICByZXNvdXJjZTogJ3RhYmxlJyxcbiAgICAgIHJlc291cmNlTmFtZTogYCR7dGhpcy5kYXRhYmFzZS5kYXRhYmFzZU5hbWV9LyR7dGhpcy50YWJsZU5hbWV9YCxcbiAgICB9KTtcbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gdGFibGVSZXNvdXJjZTtcblxuICAgIC8vIFBhcnRpdGlvbiBpbmRleCBjcmVhdGlvbiByZWxpZXMgb24gY3JlYXRlZCB0YWJsZS5cbiAgICBpZiAocHJvcHMucGFydGl0aW9uSW5kZXhlcykge1xuICAgICAgdGhpcy5wYXJ0aXRpb25JbmRleGVzID0gcHJvcHMucGFydGl0aW9uSW5kZXhlcztcbiAgICAgIHRoaXMucGFydGl0aW9uSW5kZXhlcy5mb3JFYWNoKChpbmRleCkgPT4gdGhpcy5hZGRQYXJ0aXRpb25JbmRleChpbmRleCkpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBwYXJ0aXRpb24gaW5kZXggdG8gdGhlIHRhYmxlLiBZb3UgY2FuIGhhdmUgYSBtYXhpbXVtIG9mIDMgcGFydGl0aW9uXG4gICAqIGluZGV4ZXMgdG8gYSB0YWJsZS4gUGFydGl0aW9uIGluZGV4IGtleXMgbXVzdCBiZSBhIHN1YnNldCBvZiB0aGUgdGFibGUnc1xuICAgKiBwYXJ0aXRpb24ga2V5cy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ2x1ZS9sYXRlc3QvZGcvcGFydGl0aW9uLWluZGV4ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGFkZFBhcnRpdGlvbkluZGV4KGluZGV4OiBQYXJ0aXRpb25JbmRleCkge1xuICAgIGNvbnN0IG51bVBhcnRpdGlvbnMgPSB0aGlzLnBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2VzLmxlbmd0aDtcbiAgICBpZiAobnVtUGFydGl0aW9ucyA+PSAzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ01heGltdW0gbnVtYmVyIG9mIHBhcnRpdGlvbiBpbmRleGVzIGFsbG93ZWQgaXMgMycpO1xuICAgIH1cbiAgICB0aGlzLnZhbGlkYXRlUGFydGl0aW9uSW5kZXgoaW5kZXgpO1xuXG4gICAgY29uc3QgaW5kZXhOYW1lID0gaW5kZXguaW5kZXhOYW1lID8/IHRoaXMuZ2VuZXJhdGVJbmRleE5hbWUoaW5kZXgua2V5TmFtZXMpO1xuICAgIGNvbnN0IHBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2UgPSBuZXcgY3IuQXdzQ3VzdG9tUmVzb3VyY2UodGhpcywgYHBhcnRpdGlvbi1pbmRleC0ke2luZGV4TmFtZX1gLCB7XG4gICAgICBvbkNyZWF0ZToge1xuICAgICAgICBzZXJ2aWNlOiAnR2x1ZScsXG4gICAgICAgIGFjdGlvbjogJ2NyZWF0ZVBhcnRpdGlvbkluZGV4JyxcbiAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgIERhdGFiYXNlTmFtZTogdGhpcy5kYXRhYmFzZS5kYXRhYmFzZU5hbWUsXG4gICAgICAgICAgVGFibGVOYW1lOiB0aGlzLnRhYmxlTmFtZSxcbiAgICAgICAgICBQYXJ0aXRpb25JbmRleDoge1xuICAgICAgICAgICAgSW5kZXhOYW1lOiBpbmRleE5hbWUsXG4gICAgICAgICAgICBLZXlzOiBpbmRleC5rZXlOYW1lcyxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBwaHlzaWNhbFJlc291cmNlSWQ6IGNyLlBoeXNpY2FsUmVzb3VyY2VJZC5vZihcbiAgICAgICAgICBpbmRleE5hbWUsXG4gICAgICAgICksXG4gICAgICB9LFxuICAgICAgcG9saWN5OiBjci5Bd3NDdXN0b21SZXNvdXJjZVBvbGljeS5mcm9tU2RrQ2FsbHMoe1xuICAgICAgICByZXNvdXJjZXM6IGNyLkF3c0N1c3RvbVJlc291cmNlUG9saWN5LkFOWV9SRVNPVVJDRSxcbiAgICAgIH0pLFxuICAgICAgLy8gQVBJcyBhcmUgYXZhaWxhYmxlIGluIDIuMTA1NS4wXG4gICAgICBpbnN0YWxsTGF0ZXN0QXdzU2RrOiBmYWxzZSxcbiAgICB9KTtcbiAgICB0aGlzLmdyYW50VG9VbmRlcmx5aW5nUmVzb3VyY2VzKHBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2UsIFsnZ2x1ZTpVcGRhdGVUYWJsZSddKTtcblxuICAgIC8vIERlcGVuZCBvbiBwcmV2aW91cyBwYXJ0aXRpb24gaW5kZXggaWYgcG9zc2libGUsIHRvIGF2b2lkIHJhY2UgY29uZGl0aW9uXG4gICAgaWYgKG51bVBhcnRpdGlvbnMgPiAwKSB7XG4gICAgICB0aGlzLnBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2VzW251bVBhcnRpdGlvbnMtMV0ubm9kZS5hZGREZXBlbmRlbmN5KHBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2UpO1xuICAgIH1cbiAgICB0aGlzLnBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2VzLnB1c2gocGFydGl0aW9uSW5kZXhDdXN0b21SZXNvdXJjZSk7XG4gIH1cblxuICBwcml2YXRlIGdlbmVyYXRlSW5kZXhOYW1lKGtleXM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgICBjb25zdCBwcmVmaXggPSBrZXlzLmpvaW4oJy0nKSArICctJztcbiAgICBjb25zdCB1bmlxdWVJZCA9IE5hbWVzLnVuaXF1ZUlkKHRoaXMpO1xuICAgIGNvbnN0IG1heEluZGV4TGVuZ3RoID0gODA7IC8vIGFyYml0cmFyaWx5IHNwZWNpZmllZFxuICAgIGNvbnN0IHN0YXJ0SW5kZXggPSBNYXRoLm1heCgwLCB1bmlxdWVJZC5sZW5ndGggLSAobWF4SW5kZXhMZW5ndGggLSBwcmVmaXgubGVuZ3RoKSk7XG4gICAgcmV0dXJuIHByZWZpeCArIHVuaXF1ZUlkLnN1YnN0cmluZyhzdGFydEluZGV4KTtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVQYXJ0aXRpb25JbmRleChpbmRleDogUGFydGl0aW9uSW5kZXgpIHtcbiAgICBpZiAoaW5kZXguaW5kZXhOYW1lICE9PSB1bmRlZmluZWQgJiYgKGluZGV4LmluZGV4TmFtZS5sZW5ndGggPCAxIHx8IGluZGV4LmluZGV4TmFtZS5sZW5ndGggPiAyNTUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEluZGV4IG5hbWUgbXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDI1NSBjaGFyYWN0ZXJzLCBidXQgZ290ICR7aW5kZXguaW5kZXhOYW1lLmxlbmd0aH1gKTtcbiAgICB9XG4gICAgaWYgKCF0aGlzLnBhcnRpdGlvbktleXMgfHwgdGhpcy5wYXJ0aXRpb25LZXlzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgdGFibGUgbXVzdCBoYXZlIHBhcnRpdGlvbiBrZXlzIHRvIGNyZWF0ZSBhIHBhcnRpdGlvbiBpbmRleCcpO1xuICAgIH1cbiAgICBjb25zdCBrZXlOYW1lcyA9IHRoaXMucGFydGl0aW9uS2V5cy5tYXAocGsgPT4gcGsubmFtZSk7XG4gICAgaWYgKCFpbmRleC5rZXlOYW1lcy5ldmVyeShrID0+IGtleU5hbWVzLmluY2x1ZGVzKGspKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBbGwgaW5kZXgga2V5cyBtdXN0IGFsc28gYmUgcGFydGl0aW9uIGtleXMuIEdvdCAke2luZGV4LmtleU5hbWVzfSBidXQgcGFydGl0aW9uIGtleSBuYW1lcyBhcmUgJHtrZXlOYW1lc31gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgcmVhZCBwZXJtaXNzaW9ucyB0byB0aGUgdGFibGUgYW5kIHRoZSB1bmRlcmx5aW5nIGRhdGEgc3RvcmVkIGluIFMzIHRvIGFuIElBTSBwcmluY2lwYWwuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIHRoZSBwcmluY2lwYWxcbiAgICovXG4gIHB1YmxpYyBncmFudFJlYWQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIGNvbnN0IHJldCA9IHRoaXMuZ3JhbnQoZ3JhbnRlZSwgcmVhZFBlcm1pc3Npb25zKTtcbiAgICBpZiAodGhpcy5lbmNyeXB0aW9uS2V5ICYmIHRoaXMuZW5jcnlwdGlvbiA9PT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykgeyB0aGlzLmVuY3J5cHRpb25LZXkuZ3JhbnREZWNyeXB0KGdyYW50ZWUpOyB9XG4gICAgdGhpcy5idWNrZXQuZ3JhbnRSZWFkKGdyYW50ZWUsIHRoaXMuZ2V0UzNQcmVmaXhGb3JHcmFudCgpKTtcbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHdyaXRlIHBlcm1pc3Npb25zIHRvIHRoZSB0YWJsZSBhbmQgdGhlIHVuZGVybHlpbmcgZGF0YSBzdG9yZWQgaW4gUzMgdG8gYW4gSUFNIHByaW5jaXBhbC5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgdGhlIHByaW5jaXBhbFxuICAgKi9cbiAgcHVibGljIGdyYW50V3JpdGUoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIGNvbnN0IHJldCA9IHRoaXMuZ3JhbnQoZ3JhbnRlZSwgd3JpdGVQZXJtaXNzaW9ucyk7XG4gICAgaWYgKHRoaXMuZW5jcnlwdGlvbktleSAmJiB0aGlzLmVuY3J5cHRpb24gPT09IFRhYmxlRW5jcnlwdGlvbi5DTElFTlRfU0lERV9LTVMpIHsgdGhpcy5lbmNyeXB0aW9uS2V5LmdyYW50RW5jcnlwdChncmFudGVlKTsgfVxuICAgIHRoaXMuYnVja2V0LmdyYW50V3JpdGUoZ3JhbnRlZSwgdGhpcy5nZXRTM1ByZWZpeEZvckdyYW50KCkpO1xuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgcmVhZCBhbmQgd3JpdGUgcGVybWlzc2lvbnMgdG8gdGhlIHRhYmxlIGFuZCB0aGUgdW5kZXJseWluZyBkYXRhIHN0b3JlZCBpbiBTMyB0byBhbiBJQU0gcHJpbmNpcGFsLlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSB0aGUgcHJpbmNpcGFsXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRSZWFkV3JpdGUoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIGNvbnN0IHJldCA9IHRoaXMuZ3JhbnQoZ3JhbnRlZSwgWy4uLnJlYWRQZXJtaXNzaW9ucywgLi4ud3JpdGVQZXJtaXNzaW9uc10pO1xuICAgIGlmICh0aGlzLmVuY3J5cHRpb25LZXkgJiYgdGhpcy5lbmNyeXB0aW9uID09PSBUYWJsZUVuY3J5cHRpb24uQ0xJRU5UX1NJREVfS01TKSB7IHRoaXMuZW5jcnlwdGlvbktleS5ncmFudEVuY3J5cHREZWNyeXB0KGdyYW50ZWUpOyB9XG4gICAgdGhpcy5idWNrZXQuZ3JhbnRSZWFkV3JpdGUoZ3JhbnRlZSwgdGhpcy5nZXRTM1ByZWZpeEZvckdyYW50KCkpO1xuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgdGhlIGdpdmVuIGlkZW50aXR5IGN1c3RvbSBwZXJtaXNzaW9ucy5cbiAgICovXG4gIHB1YmxpYyBncmFudChncmFudGVlOiBpYW0uSUdyYW50YWJsZSwgYWN0aW9uczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICByZXNvdXJjZUFybnM6IFt0aGlzLnRhYmxlQXJuXSxcbiAgICAgIGFjdGlvbnMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgdGhlIGdpdmVuIGlkZW50aXR5IGN1c3RvbSBwZXJtaXNzaW9ucyB0byBBTEwgdW5kZXJseWluZyByZXNvdXJjZXMgb2YgdGhlIHRhYmxlLlxuICAgKiBQZXJtaXNzaW9ucyB3aWxsIGJlIGdyYW50ZWQgdG8gdGhlIGNhdGFsb2csIHRoZSBkYXRhYmFzZSwgYW5kIHRoZSB0YWJsZS5cbiAgICovXG4gIHB1YmxpYyBncmFudFRvVW5kZXJseWluZ1Jlc291cmNlcyhncmFudGVlOiBpYW0uSUdyYW50YWJsZSwgYWN0aW9uczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICByZXNvdXJjZUFybnM6IFtcbiAgICAgICAgdGhpcy50YWJsZUFybixcbiAgICAgICAgdGhpcy5kYXRhYmFzZS5jYXRhbG9nQXJuLFxuICAgICAgICB0aGlzLmRhdGFiYXNlLmRhdGFiYXNlQXJuLFxuICAgICAgXSxcbiAgICAgIGFjdGlvbnMsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGdldFMzUHJlZml4Rm9yR3JhbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMuczNQcmVmaXggKyAnKic7XG4gIH1cbn1cblxuZnVuY3Rpb24gdmFsaWRhdGVTY2hlbWEoY29sdW1uczogQ29sdW1uW10sIHBhcnRpdGlvbktleXM/OiBDb2x1bW5bXSk6IHZvaWQge1xuICBpZiAoY29sdW1ucy5sZW5ndGggPT09IDApIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ3lvdSBtdXN0IHNwZWNpZnkgYXQgbGVhc3Qgb25lIGNvbHVtbiBmb3IgdGhlIHRhYmxlJyk7XG4gIH1cbiAgLy8gQ2hlY2sgdGhlcmUgaXMgYXQgbGVhc3Qgb25lIGNvbHVtbiBhbmQgbm8gZHVwbGljYXRlZCBjb2x1bW4gbmFtZXMgb3IgcGFydGl0aW9uIGtleXMuXG4gIGNvbnN0IG5hbWVzID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gIChjb2x1bW5zLmNvbmNhdChwYXJ0aXRpb25LZXlzIHx8IFtdKSkuZm9yRWFjaChjb2x1bW4gPT4ge1xuICAgIGlmIChuYW1lcy5oYXMoY29sdW1uLm5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGNvbHVtbiBuYW1lcyBhbmQgcGFydGl0aW9uIGtleXMgbXVzdCBiZSB1bmlxdWUsIGJ1dCBcXCcke2NvbHVtbi5uYW1lfVxcJyBpcyBkdXBsaWNhdGVkYCk7XG4gICAgfVxuICAgIG5hbWVzLmFkZChjb2x1bW4ubmFtZSk7XG4gIH0pO1xufVxuXG4vLyBtYXAgVGFibGVFbmNyeXB0aW9uIHRvIGJ1Y2tldCdzIFNTRSBjb25maWd1cmF0aW9uIChzMy5CdWNrZXRFbmNyeXB0aW9uKVxuY29uc3QgZW5jcnlwdGlvbk1hcHBpbmdzID0ge1xuICBbVGFibGVFbmNyeXB0aW9uLlMzX01BTkFHRURdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gIFtUYWJsZUVuY3J5cHRpb24uS01TX01BTkFHRURdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICBbVGFibGVFbmNyeXB0aW9uLktNU106IHMzLkJ1Y2tldEVuY3J5cHRpb24uS01TLFxuICBbVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNU106IHMzLkJ1Y2tldEVuY3J5cHRpb24uVU5FTkNSWVBURUQsXG4gIFtUYWJsZUVuY3J5cHRpb24uVU5FTkNSWVBURURdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlVORU5DUllQVEVELFxufTtcblxuLy8gY3JlYXRlIHRoZSBidWNrZXQgdG8gc3RvcmUgYSB0YWJsZSdzIGRhdGEgZGVwZW5kaW5nIG9uIHRoZSBgZW5jcnlwdGlvbmAgYW5kIGBlbmNyeXB0aW9uS2V5YCBwcm9wZXJ0aWVzLlxuZnVuY3Rpb24gY3JlYXRlQnVja2V0KHRhYmxlOiBUYWJsZSwgcHJvcHM6IFRhYmxlUHJvcHMpIHtcbiAgY29uc3QgZW5jcnlwdGlvbiA9IHByb3BzLmVuY3J5cHRpb24gfHwgVGFibGVFbmNyeXB0aW9uLlVORU5DUllQVEVEO1xuICBsZXQgYnVja2V0ID0gcHJvcHMuYnVja2V0O1xuXG4gIGlmIChidWNrZXQgJiYgKGVuY3J5cHRpb24gIT09IFRhYmxlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCAmJiBlbmNyeXB0aW9uICE9PSBUYWJsZUVuY3J5cHRpb24uQ0xJRU5UX1NJREVfS01TKSkge1xuICAgIHRocm93IG5ldyBFcnJvcigneW91IGNhbiBub3Qgc3BlY2lmeSBlbmNyeXB0aW9uIHNldHRpbmdzIGlmIHlvdSBhbHNvIHByb3ZpZGUgYSBidWNrZXQnKTtcbiAgfVxuXG4gIGxldCBlbmNyeXB0aW9uS2V5OiBrbXMuSUtleSB8IHVuZGVmaW5lZDtcbiAgaWYgKGVuY3J5cHRpb24gPT09IFRhYmxlRW5jcnlwdGlvbi5DTElFTlRfU0lERV9LTVMgJiYgcHJvcHMuZW5jcnlwdGlvbktleSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgLy8gQ1NFLUtNUyBzaG91bGQgYmVoYXZlIHRoZSBzYW1lIGFzIFNTRS1LTVMgLSB1c2UgdGhlIHByb3ZpZGVkIGtleSBvciBjcmVhdGUgb25lIGF1dG9tYXRpY2FsbHlcbiAgICAvLyBTaW5jZSBCdWNrZXQgb25seSBrbm93cyBhYm91dCBTU0UsIHdlIHJlcGVhdCB0aGUgbG9naWMgZm9yIENTRS1LTVMgYXQgdGhlIFRhYmxlIGxldmVsLlxuICAgIGVuY3J5cHRpb25LZXkgPSBuZXcga21zLktleSh0YWJsZSwgJ0tleScpO1xuICB9IGVsc2Uge1xuICAgIGVuY3J5cHRpb25LZXkgPSBwcm9wcy5lbmNyeXB0aW9uS2V5O1xuICB9XG5cbiAgLy8gY3JlYXRlIHRoZSBidWNrZXQgaWYgbm9uZSB3YXMgcHJvdmlkZWRcbiAgaWYgKCFidWNrZXQpIHtcbiAgICBpZiAoZW5jcnlwdGlvbiA9PT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykge1xuICAgICAgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0YWJsZSwgJ0J1Y2tldCcpO1xuICAgIH0gZWxzZSB7XG4gICAgICBidWNrZXQgPSBuZXcgczMuQnVja2V0KHRhYmxlLCAnQnVja2V0Jywge1xuICAgICAgICBlbmNyeXB0aW9uOiBlbmNyeXB0aW9uTWFwcGluZ3NbZW5jcnlwdGlvbl0sXG4gICAgICAgIGVuY3J5cHRpb25LZXksXG4gICAgICB9KTtcbiAgICAgIGVuY3J5cHRpb25LZXkgPSBidWNrZXQuZW5jcnlwdGlvbktleTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4ge1xuICAgIGJ1Y2tldCxcbiAgICBlbmNyeXB0aW9uLFxuICAgIGVuY3J5cHRpb25LZXksXG4gIH07XG59XG5cbmNvbnN0IHJlYWRQZXJtaXNzaW9ucyA9IFtcbiAgJ2dsdWU6QmF0Y2hHZXRQYXJ0aXRpb24nLFxuICAnZ2x1ZTpHZXRQYXJ0aXRpb24nLFxuICAnZ2x1ZTpHZXRQYXJ0aXRpb25zJyxcbiAgJ2dsdWU6R2V0VGFibGUnLFxuICAnZ2x1ZTpHZXRUYWJsZXMnLFxuICAnZ2x1ZTpHZXRUYWJsZVZlcnNpb24nLFxuICAnZ2x1ZTpHZXRUYWJsZVZlcnNpb25zJyxcbl07XG5cbmNvbnN0IHdyaXRlUGVybWlzc2lvbnMgPSBbXG4gICdnbHVlOkJhdGNoQ3JlYXRlUGFydGl0aW9uJyxcbiAgJ2dsdWU6QmF0Y2hEZWxldGVQYXJ0aXRpb24nLFxuICAnZ2x1ZTpDcmVhdGVQYXJ0aXRpb24nLFxuICAnZ2x1ZTpEZWxldGVQYXJ0aXRpb24nLFxuICAnZ2x1ZTpVcGRhdGVQYXJ0aXRpb24nLFxuXTtcblxuZnVuY3Rpb24gcmVuZGVyQ29sdW1ucyhjb2x1bW5zPzogQXJyYXk8Q29sdW1uIHwgQ29sdW1uPikge1xuICBpZiAoY29sdW1ucyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuICByZXR1cm4gY29sdW1ucy5tYXAoY29sdW1uID0+IHtcbiAgICByZXR1cm4ge1xuICAgICAgbmFtZTogY29sdW1uLm5hbWUsXG4gICAgICB0eXBlOiBjb2x1bW4udHlwZS5pbnB1dFN0cmluZyxcbiAgICAgIGNvbW1lbnQ6IGNvbHVtbi5jb21tZW50LFxuICAgIH07XG4gIH0pO1xufVxuIl19