"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobExecutable = exports.JobType = exports.PythonVersion = exports.JobLanguage = exports.GlueVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * AWS Glue version determines the versions of Apache Spark and Python that are available to the job.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/add-job.html.
 *
 * If you need to use a GlueVersion that doesn't exist as a static member, you
 * can instantiate a `GlueVersion` object, e.g: `GlueVersion.of('1.5')`.
 */
class GlueVersion {
    /**
     * Custom Glue version
     * @param version custom version
     */
    static of(version) {
        return new GlueVersion(version);
    }
    constructor(name) {
        this.name = name;
    }
}
_a = JSII_RTTI_SYMBOL_1;
GlueVersion[_a] = { fqn: "@aws-cdk/aws-glue-alpha.GlueVersion", version: "2.73.0-alpha.0" };
/**
 * Glue version using Spark 2.2.1 and Python 2.7
 */
GlueVersion.V0_9 = new GlueVersion('0.9');
/**
 * Glue version using Spark 2.4.3, Python 2.7 and Python 3.6
 */
GlueVersion.V1_0 = new GlueVersion('1.0');
/**
 * Glue version using Spark 2.4.3 and Python 3.7
 */
GlueVersion.V2_0 = new GlueVersion('2.0');
/**
 * Glue version using Spark 3.1.1 and Python 3.7
 */
GlueVersion.V3_0 = new GlueVersion('3.0');
/**
 * Glue version using Spark 3.3.0 and Python 3.10
 */
GlueVersion.V4_0 = new GlueVersion('4.0');
exports.GlueVersion = GlueVersion;
/**
 * Runtime language of the Glue job
 */
var JobLanguage;
(function (JobLanguage) {
    /**
     * Scala
     */
    JobLanguage["SCALA"] = "scala";
    /**
     * Python
     */
    JobLanguage["PYTHON"] = "python";
})(JobLanguage = exports.JobLanguage || (exports.JobLanguage = {}));
/**
 * Python version
 */
var PythonVersion;
(function (PythonVersion) {
    /**
     * Python 2 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["TWO"] = "2";
    /**
     * Python 3 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE"] = "3";
    /**
     * Python 3.9 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE_NINE"] = "3.9";
})(PythonVersion = exports.PythonVersion || (exports.PythonVersion = {}));
/**
 * The job type.
 *
 * If you need to use a JobType that doesn't exist as a static member, you
 * can instantiate a `JobType` object, e.g: `JobType.of('other name')`.
 */
class JobType {
    /**
     * Custom type name
     * @param name type name
     */
    static of(name) {
        return new JobType(name);
    }
    constructor(name) {
        this.name = name;
    }
}
_b = JSII_RTTI_SYMBOL_1;
JobType[_b] = { fqn: "@aws-cdk/aws-glue-alpha.JobType", version: "2.73.0-alpha.0" };
/**
 * Command for running a Glue Spark job.
 */
JobType.ETL = new JobType('glueetl');
/**
 * Command for running a Glue Spark streaming job.
 */
JobType.STREAMING = new JobType('gluestreaming');
/**
 * Command for running a Glue python shell job.
 */
JobType.PYTHON_SHELL = new JobType('pythonshell');
/**
 * Command for running a Glue Ray job.
 */
JobType.RAY = new JobType('glueray');
exports.JobType = JobType;
/**
 * The executable properties related to the Glue job's GlueVersion, JobType and code
 */
class JobExecutable {
    /**
     * Create Scala executable props for Apache Spark ETL job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Scala executable props for Apache Spark Streaming job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Python executable props for Apache Spark ETL job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for Apache Spark Streaming job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for python shell jobs.
     *
     * @param props Python Shell Job props.
     */
    static pythonShell(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonShellExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonShell);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.PYTHON_SHELL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for Ray jobs.
     *
     * @param props Ray Job props.
     */
    static pythonRay(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_PythonRayExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonRay);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.RAY,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create a custom JobExecutable.
     *
     * @param config custom job executable configuration.
     */
    static of(config) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobExecutableConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new JobExecutable(config);
    }
    constructor(config) {
        if (JobType.PYTHON_SHELL === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Python shell requires the language to be set to Python');
            }
            if ([GlueVersion.V0_9, GlueVersion.V3_0, GlueVersion.V4_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Python Shell`);
            }
        }
        if (JobType.RAY === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Ray requires the language to be set to Python');
            }
            if ([GlueVersion.V0_9, GlueVersion.V1_0, GlueVersion.V2_0, GlueVersion.V3_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Ray`);
            }
        }
        if (config.extraJarsFirst && [GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support extraJarsFirst`);
        }
        if (config.pythonVersion === PythonVersion.TWO && ![GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support PythonVersion ${config.pythonVersion}`);
        }
        if (JobLanguage.PYTHON !== config.language && config.extraPythonFiles) {
            throw new Error('extraPythonFiles is not supported for languages other than JobLanguage.PYTHON');
        }
        if (config.pythonVersion === PythonVersion.THREE_NINE && config.type !== JobType.PYTHON_SHELL && config.type !== JobType.RAY) {
            throw new Error('Specified PythonVersion PythonVersion.THREE_NINE is only supported for JobType Python Shell and Ray');
        }
        if (config.pythonVersion === PythonVersion.THREE && config.type === JobType.RAY) {
            throw new Error('Specified PythonVersion PythonVersion.THREE is not supported for Ray');
        }
        this.config = config;
    }
    /**
     * Called during Job initialization to get JobExecutableConfig.
     */
    bind() {
        return this.config;
    }
}
_c = JSII_RTTI_SYMBOL_1;
JobExecutable[_c] = { fqn: "@aws-cdk/aws-glue-alpha.JobExecutable", version: "2.73.0-alpha.0" };
exports.JobExecutable = JobExecutable;
//# sourceMappingURL=data:application/json;base64,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