"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsQueue = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const util_1 = require("./util");
/**
 * Use an SQS Queue as a target for Amazon EventBridge rules.
 *
 * @example
 *   /// fixture=withRepoAndSqsQueue
 *   // publish to an SQS queue every time code is committed
 *   // to a CodeCommit repository
 *   repository.onCommit('onCommit', { target: new targets.SqsQueue(queue) });
 *
 */
class SqsQueue {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_events_targets_SqsQueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SqsQueue);
            }
            throw error;
        }
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a
     * result from an EventBridge event.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/resource-based-policies-eventbridge.html#sqs-permissions
     */
    bind(rule, _id) {
        // Only add the rule as a condition if the queue is not encrypted, to avoid circular dependency. See issue #11158.
        const principalOpts = this.queue.encryptionMasterKey ? {} : {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        };
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', principalOpts));
        if (this.props.deadLetterQueue) {
            util_1.addToDeadLetterQueueResourcePolicy(rule, this.props.deadLetterQueue);
        }
        return {
            ...util_1.bindBaseTargetConfig(this.props),
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
_a = JSII_RTTI_SYMBOL_1;
SqsQueue[_a] = { fqn: "@aws-cdk/aws-events-targets.SqsQueue", version: "1.165.0" };
//# sourceMappingURL=data:application/json;base64,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