# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['taskipy']

package_data = \
{'': ['*']}

install_requires = \
['colorama>=0.4.4,<0.5.0', 'psutil>=5.7.2,<6.0.0']

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['tomli>=1.2.3,<2.0.0'],
 ':python_version >= "3.7" and python_version < "4.0"': ['tomli>=2.0.1,<3.0.0'],
 ':sys_platform == "win32"': ['mslex>=0.3.0,<0.4.0']}

entry_points = \
{'console_scripts': ['task = taskipy.cli:main']}

setup_kwargs = {
    'name': 'taskipy',
    'version': '1.11.0',
    'description': 'tasks runner for python projects',
    'long_description': '<p align="center">\n  <img src="https://github.com/illBeRoy/taskipy/raw/master/logo.svg" width="150" />\n</p>\n\n- [General](#general)\n- [Requirements](#requirements)\n- [Usage](#usage)\n  - [Installation](#installation)\n  - [Adding Tasks](#adding-tasks)\n  - [Running Tasks](#running-tasks)\n  - [Passing Command Line Args to Tasks](#passing-command-line-args-to-tasks)\n  - [Composing Tasks](#composing-tasks)\n    - [Grouping Subtasks Together](#grouping-subtasks-together)\n    - [Pre Task Hook](#pre-task-hook)\n    - [Post Task Hook](#post-task-hook)\n  - [Using Variables](#using-variables)\n    - [String Formatting](#string-formatting)\n    - [Always Use Variables](#always-use-variables)\n    - [Recursive Variables](#recursive-variables)\n  - [Using Taskipy Without Poetry](#using-taskipy-without-poetry)\n    - [Installing With PIP](#installing-with-pip)\n    - [Running Tasks](#running-tasks-1)\n  - [Advanced Use Cases](#advanced-use-cases)\n- [Maintainers :construction:](#maintainers-construction)\n- [Contributors :sparkles:](#contributors-sparkles)\n\n## General\n\n[![pypi](https://img.shields.io/pypi/v/taskipy?style=flat-square)](https://pypi.org/project/taskipy/)\n[![pypi-downloads](https://img.shields.io/pypi/dm/taskipy?style=flat-square)](https://pypi.org/project/taskipy/)\n[![ci](https://img.shields.io/github/actions/workflow/status/taskipy/taskipy/ci.yml?style=flat-square)](https://github.com/taskipy/taskipy/actions/workflows/ci.yml)\n[![All Contributors](https://img.shields.io/github/all-contributors/illberoy/taskipy?color=orange&style=flat-square)](#contributors-)\n\n**The complementary task runner for python.**\n\nEvery development pipeline has tasks, such as `test`, `lint` or `publish`. With taskipy, you can define those tasks in one file and run them with a simple command.\n\nFor instance, instead of running the following command:\n\n```bash\npython -m unittest tests/test_*.py\n```\n\nYou can create a task called `test` and simply run:\n\n```bash\npoetry run task test\n```\n\nOr (if you\'re not using poetry):\n\n```bash\ntask test\n```\n\nIn addition, you can compose tasks and group them together, and also create dependencies between them.\n\nThis project is heavily inspired by npm\'s [run script command](https://docs.npmjs.com/cli/run-script).\n\n## Requirements\n\nPython 3.6 or newer.\n\nYour project directory should include a valid `pyproject.toml` file, as specified in [PEP-518](https://www.python.org/dev/peps/pep-0518/).\n\n## Usage\n\n### Installation\n\nTo install taskipy as a dev dependency, simply run:\n\n```bash\npoetry add --dev taskipy\n```\n\n### Adding Tasks\n\nIn your `pyproject.toml` file, add a new section called `[tool.taskipy.tasks]`.\n\nThe section is a key-value map, from the names of the task to the actual command that should be run in the shell.\n\nThere are two ways to define tasks. The easy way is to simply write the command down as a string:\n\n__pyproject.toml__\n\n```toml\n[tool.taskipy.tasks]\ntest = "python -m unittest tests/test_*.py"\nlint = "pylint tests taskipy"\n```\n\nAlternatively, you can define tasks more explicitly by declaring both the command and a helpful description using an inline table:\n\n__pyproject.toml__\n\n```toml\n[tool.taskipy.tasks]\ntest = { cmd = "python -m unittest tests/test_*.py", help = "runs all unit tests" }\nlint = { cmd = "pylint tests taskipy", help = "confirms code style using pylint" } \n```\n\nThe explicit notation is more verbose, but provides better context to anyone who uses the task.\n\n### Running Tasks\n\nIn order to run a task, run the following command in your terminal:\n\n```bash\n$ poetry run task test\n```\n\nYou can also list all existing tasks by running the following:\n\n```bash\n$ poetry run task --list\ntest                python -m unittest tests/test_*.py\nlint                pylint tests taskipy\n```\n\nIf you declared your task explicitly, you will see the description of the task by the side of the task\'s name:\n\n```bash\n$ poetry run task --list\ntest                runs all unit tests\nlint                confirms code style using pylint\n```\n\n### Passing Command Line Args to Tasks\n\nIf you want to pass command line arguments to tasks (positional or named), simply append them to the end of the task command.\n\nFor example, running the above task like this:\n\n```bash\npoetry run task test -h\n```\n\nIs equivalent to running:\n\n```bash\npython -m unittest tests/test_*.py -h\n```\n\nAnd will show unittest\'s help instead of actually running it.\n\n> ⚠️ Note: if you are using pre \\ post hooks, do notice that arguments are not passed to them, only to the task itself.\n\n### Composing Tasks\n\n#### Grouping Subtasks Together\n\nSome tasks are composed of multiple subtasks. Instead of writing plain shell commands and stringing them together, you can break them down into multiple subtasks:\n\n```toml\n[tool.taskipy.tasks]\nlint_pylint = "pylint tests taskipy"\nlint_mypy = "mypy tests taskipy"\n```\n\nAnd then create a composite task:\n\n```toml\n[tool.taskipy.tasks]\nlint = "task lint_pylint && task lint_mypy"\nlint_pylint = "pylint tests taskipy"\nlint_mypy = "mypy tests taskipy"\n```\n\n#### Pre Task Hook\n\nTasks might also depend on one another. For example, tests might require some binaries to be built. Take the two following commands, for instance:\n\n```toml\n[tool.taskipy.tasks]\ntest = "python -m unittest tests/test_*.py"\nbuild = "make ."\n```\n\nYou could make tests depend on building, by using the **pretask hook**:\n\n```toml\n[tool.taskipy.tasks]\npre_test = "task build"\ntest = "python -m unittest tests/test_*.py"\nbuild = "make ."\n```\n\nThe pretask hook looks for `pre_<task_name>` task for a given `task_name`. It will run it before running the task itself. If the pretask fails, then taskipy will exit without running the task itself.\n\n#### Post Task Hook\n\nFrom time to time, you might want to run a task in conjuction with another. For example, you might want to run linting after a successful test run. Take the two following commands, for instance:\n\n```toml\n[tool.taskipy.tasks]\ntest = "python -m unittest tests/test_*.py"\nlint = "pylint tests taskipy"\n```\n\nYou could make tests trigger linting, by using the **posttask hook**:\n\n```toml\n[tool.taskipy.tasks]\ntest = "python -m unittest tests/test_*.py"\npost_test = "task lint"\nlint = "pylint tests taskipy"\n```\n\nThe posttask hook looks for `post_<task_name>` task for a given `task_name`. It will run it after running the task itself. If the task failed, then taskipy will not run the posttask hook.\n\n### Using Variables\n\nIn some cases, you might find yourself passing the same arguments over and over again. Let us take a look at the following tasks:\n\n```toml\n[tool.taskipy.tasks]\nlint = "pylint path/to/my_module"\nblack = "black path/to/my_module"\n```\n\nAs you can see, we provide the same path argument to both `pylint` and `black`.\n\nIn order to encourage DRY and improve your ability to change these values later on, taskipy actually lets you declare and reuse variables in your tasks:\n\n```toml\n[tool.taskipy.variables]\npath = "path/to/my_module"\n\n[tool.taskipy.tasks]\nlint = { cmd = "pylint {path}", use_vars = true }\nblack = { cmd = "pylint {path}", use_vars = true }\n```\n\nWe have made the following changes to our configuration:\n\n1. We declared variables under `tool.taskipy.variables`\n2. We flagged the relevant task using `use_vars` to note that they should use the variables\n3. We replaced the repeating path with a `{path}` variable\n\n#### String Formatting\n\nThe formatting of the task commands uses python\'s own `string.format` method, and therefore supports everything that python\'s [formatted string literals](https://docs.python.org/3/tutorial/inputoutput.html#formatted-string-literals) let you do.\n\n#### Always Use Variables\n\nUsing variables is opt-in, which means that by default commands do **not** use them, and you will have to turn them on a task to task basis.\n\nIf you want to turn on `use_vars` globally, all you need to do is to declare that under taskipy\'s **settings** table:\n\n```toml\n[tool.taskipy.settings]\nuse_vars = true\n\n[tool.taskipy.variables]\npath = "path/to/my_module"\n\n[tool.taskipy.tasks]\nlint = "pylint {path}"\nblack = "black {path}"\n```\n\n#### Recursive Variables\n\nIf we want to use variables within other variables, we can utilize recursive variables. By default, variables are not recursive, but we can specify a variable to be recursive by setting the `recursive` key to `true`.\n\n```toml\n[tool.taskipy.settings]\nuse_vars = true\n\n[tool.taskipy.variables]\nsrc_dir = "src"\npackage_dir = { var = "{src_dir}/package", recursive = true }\n\n[tool.taskipy.tasks]\necho = "echo {package_dir}"\n```\n\nIn this example, we could run `task echo` and we would then see `src/package`.\n\n### Using Taskipy Without Poetry\n\nTaskipy was created with poetry projects in mind, but actually only requires a valid `pyproject.toml` file in your project\'s directory. As a result, you can use it even without poetry:\n\n#### Installing With PIP\n\nInstall taskipy on your machine or in your virtualenv using:\n\n```bash\npip install taskipy\n```\n\n#### Running Tasks\n\nHead into your project\'s directory (don\'t forget to activate virtualenv if you\'re using one), and run the following command:\n\n```bash\ntask TASK\n```\n\nWhere `TASK` is the name of your task.\n\n### Advanced Use Cases\n\nIf you have a more specific use case, you might not be the first to run into it! Head over to the [ADVANCED_FEATURES](./docs/ADVANCED_FEATURES.md) doc, and look it up.\n\n## Maintainers :construction:\n\n<table>\n  <tr>\n    <td align="center"><a href="https://github.com/illBeRoy"><img src="https://avatars2.githubusercontent.com/u/6681893?v=4" width="100px;" alt=""/><br /><sub><b>Roy Sommer</b></sub></a></td>\n  </tr>\n</table>\n\n## Contributors :sparkles:\n\nThanks goes to these wonderful people ([emoji key](https://allcontributors.org/docs/en/emoji-key)):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore-start -->\n<!-- markdownlint-disable -->\n<table>\n  <tr>\n    <td align="center"><a href="http://triguba.com"><img src="https://avatars3.githubusercontent.com/u/15860938?v=4" width="100px;" alt=""/><br /><sub><b>Eugene Triguba</b></sub></a><br /><a href="https://github.com/illBeRoy/taskipy/commits?author=eugenetriguba" title="Code">💻</a></td>\n    <td align="center"><a href="https://github.com/RobinFrcd"><img src="https://avatars0.githubusercontent.com/u/29704178?v=4" width="100px;" alt=""/><br /><sub><b>RobinFrcd</b></sub></a><br /><a href="https://github.com/illBeRoy/taskipy/commits?author=RobinFrcd" title="Code">💻</a></td>\n    <td align="center"><a href="http://granitosaurus.rocks"><img src="https://avatars0.githubusercontent.com/u/5476164?v=4" width="100px;" alt=""/><br /><sub><b>Bernardas Ališauskas</b></sub></a><br /><a href="https://github.com/illBeRoy/taskipy/commits?author=Granitosaurus" title="Code">💻</a></td>\n  </tr>\n</table>\n\n<!-- markdownlint-enable -->\n<!-- prettier-ignore-end -->\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\nThis project follows the [all-contributors](https://github.com/all-contributors/all-contributors)\nspecification. Contributions of any kind welcome!\n',
    'author': 'Roy Sommer',
    'author_email': 'roy@sommer.co.il',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/taskipy/taskipy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
