# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['argsense', 'argsense.argparse', 'argsense.parser', 'argsense.style']

package_data = \
{'': ['*']}

install_requires = \
['rich']

setup_kwargs = {
    'name': 'argsense',
    'version': '0.4.0',
    'description': 'New command line interface based on Python Rich library.',
    'long_description': '# Argsense CLI\n\n**argsense** is a command line interface made with Python.\n\n![group help](.assets/eg01/20220615143658.jpg)\n\n![minimal help](.assets/eg01/20220615143730.jpg)\n\n![command help](.assets/eg01/20220615143751.jpg)\n\n![group help (ex)](.assets/eg01/20220615143817.jpg)\n\n## Usage\n\n> Currently this section is under construction.\n> \n> Please check the folder ./examples for your reference.\n\n## Comparison to other CLIs\n\n> I picked several official examples from [Typer](https://typer.tiangolo.com/) project to show differences by code to code.\n\n### Hello world\n\n**typer**\n\n```python\nimport typer\n\ndef main(name: str):\n    typer.echo(f"Hello {name}")\n\nif __name__ == "__main__":\n    typer.run(main)\n```\n\n**argsense**\n\n```python\nfrom argsense import cli\n\n@cli.cmd()\ndef main(name: str):\n    print(f\'Hello {name}\')\n\nif __name__ == \'__main__\':\n    cli.run(main)\n```\n\n![](.assets/20220618124348.png)\n\n### Multiple subcommands\n\n**typer**\n\n```python\nimport typer\n\napp = typer.Typer()\n\n@app.command()\ndef hello(name: str):\n    typer.echo(f"Hello {name}")\n\n@app.command()\ndef goodbye(name: str, formal: bool = False):\n    if formal:\n        typer.echo(f"Goodbye Ms. {name}. Have a good day.")\n    else:\n        typer.echo(f"Bye {name}!")\n\nif __name__ == "__main__":\n    app()\n```\n\n**argsense**\n\n```python\nfrom argsense import cli\n\n@cli.cmd()\ndef hello(name: str):\n    print(f\'Hello {name}\')\n\n@cli.cmd()\ndef goodbye(name: str, formal: bool = False):\n    if formal:\n        print(f\'Goodbye Ms. {name}. Have a good day.\')\n    else:\n        print(f\'Bye {name}!\')\n\nif __name__ == \'__main__\':\n    cli.run()\n```\n\n![](.assets/snip-20220618-124911.png)\n\n![](.assets/snip-20220618-125016.png)\n\n### Making an optional argument\n\n**typer**\n\n```python\nimport typer\nfrom typing import Optional\n\ndef main(name: Optional[str] = typer.Argument(None)):\n    if name is None:\n        typer.echo("Hello World!")\n    else:\n        typer.echo(f"Hello {name}")\n\nif __name__ == "__main__":\n    typer.run(main)\n```\n\n**argsense**\n\n```python\n"""\nArgsense (v0.3.0) doesn\'t have a conception of \'optional\' argument, an argument \nmust be required, or it is considered as an `option`.\nWe will bring the support in next version (v0.4.0+).\n"""\nfrom argsense import cli\n\n@cli.cmd()\ndef main(name: str = None):\n    if name is None:\n        print(\'Hello World!\')\n    else:\n        print(f\'Hello {name}\')\n\nif __name__ == \'__main__\':\n    cli.run(main)\n```\n\n*You have to use `--name` to pass the \'optional argument\', otherwise it raises an error.*\n\n![](.assets/snip-20220618-130345.png)\n\n### An argument with a default value\n\n**typer**\n\n```python\nimport typer\n\ndef main(name: str = typer.Argument("Wade Wilson")):\n    typer.echo(f"Hello {name}")\n\nif __name__ == "__main__":\n    typer.run(main)\n```\n\n**argsense**\n\n```python\n"""\nArgsense (v0.3.0) doesn\'t have a conception of \'optional\' argument, an argument \nmust be required, or it is considered as an `option`.\nWe will bring the support in next version (v0.4.0+).\n"""\nfrom argsense import cli\n\n@cli.cmd()\ndef main(name=\'Wade Wilson\'):\n    print(f\'Hello {name}\')\n\nif __name__ == \'__main__\':\n    cli.run(main)\n```\n\n![](.assets/snip-20220618-131756.png)\n\n### Dynamic choices for argument\n\n**typer**\n\n```python\nimport random\nimport typer\n\ndef get_name():\n    return random.choice(["Deadpool", "Rick", "Morty", "Hiro"])\n\ndef main(name: str = typer.Argument(get_name)):\n    typer.echo(f"Hello {name}")\n\nif __name__ == "__main__":\n    typer.run(main)\n```\n\n**argsense**\n\n```python\n"""\nArgsense doesn\'t have built in support for choices.\nHere is a workaround but not so good.\n"""\nfrom argsense import cli\n\n@cli.cmd()\ndef main(name: str):\n    """\n    args:\n        name: The name should be one of the choices:\n            - [red]Deadpool[/]\n            - [green]Rick[/]\n            - [yellow]Morty[/]\n            - [cyan]Hiro[/]\n    """\n    if name not in (\'Deadpool\', \'Rick\', \'Morty\', \'Hiro\'):\n        print(f\'The given name ({name}) is not on the list!\')\n        exit(1)\n    print(f\'Hello {name}\')\n\nif __name__ == \'__main__\':\n    cli.run(main)\n```\n\n![](.assets/snip-20220618-132948.png)\n\n### Help information / Command description\n\nThe example shows:\n\n- Command description.\n- Long description that exceeds 80 characters one line (let\'s see how do we do line breaks).\n- Multiple lines of description.\n\n**typer**\n\n```python\nimport typer\n\ndef main(\n    name: str = typer.Argument(..., help="The name of the user to greet"),\n    lastname: str = typer.Option(\n        "", "--lastname",  "-l", \n        help="Last name of person to greet."\n    ),\n    formal: bool = typer.Option(False, help="Say hi formally."),\n):\n    """\n    Say hi to NAME very gently, like Dirk. Optionally with a --lastname/-l, like\n    Wilson.\n\n    If --formal is used, say hi very formally.\n    """\n    if formal:\n        typer.echo(f"Good day Ms. {name} {lastname}.")\n    else:\n        typer.echo(f"Hello {name} {lastname}")\n\nif __name__ == "__main__":\n    typer.run(main)\n```\n\n**argsense**\n\n```python\nfrom argsense import cli\n\n@cli.cmd()\ndef main(name: str, lastname=\'\', formal=False):\n    """\n    Say hi to [blue]NAME[/] very gently, like Dirk. Optionally with a -\n    [dim]--lastname/-l[/], like Wilson.\n    If [dim]--formal[/] is used, say hi very formally.\n\n    args:\n        name: The name of the user to greet.\n    \n    kwargs:\n        lastname (-l): Last name of person to greet.\n        formal: Say hi formally.\n    """\n    if formal:\n        print(f\'Good day Ms. {name} {lastname}.\')\n    else:\n        print(f\'Hello {name} {lastname}\')\n\nif __name__ == \'__main__\':\n    cli.run(main)\n```\n\n![](.assets/snip-20220618-135437.png)\n\n### Confirmation Prompt\n\n**typer**\n\n```python\nimport typer\n\ndef main(\n    name: str, \n    email: str = typer.Option(..., prompt=True, confirmation_prompt=True)\n):\n    typer.echo(f"Hello {name}, your email is {email}")\n\nif __name__ == "__main__":\n    typer.run(main)\n```\n\n**argsense**\n\n```python\n"""\nArgsense doesn\'t have built in support for confirmation prompt.\nHere is a workaround but not so good.\n"""\nfrom argsense import cli\n\n@cli.cmd()\ndef main(name: str):\n    email = input(\'Email: \')\n    if input(\'Repeat for confirmation: \') != email:\n        print(\'Email repetition failed.\')\n        exit(1)\n    \n    print(f\'Hello {name}, your email is {email}\')\n\nif __name__ == \'__main__\':\n    cli.run(main)\n```\n\n![](.assets/snip-20220618-140432.png)\n\n<!-- Template\n\n**typer**\n\n```python\n\n```\n\n**argsense**\n\n```python\n\n```\n\n![](.assets/)\n\n-->\n',
    'author': 'Likianta',
    'author_email': 'likianta@foxmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/likianta/argsense-cli',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
