import json
import pprint
from dataclasses import dataclass, field

from requests.structures import CaseInsensitiveDict


@dataclass(frozen=True)
class BaseUrlConfig:
    base_url: str
    sandbox_url: str or None = None
    options: dict = field(default=dict)


@dataclass
class RequestConfig:
    path: str
    name: str
    method: str = 'GET'
    options: dict = field(default=dict)


class ApiResponse:
    def __init__(self, data=None, headers: CaseInsensitiveDict = None, status_code: int = None, **kwargs):
        self.status_code: int = status_code
        self.payload = data or kwargs
        self.headers: CaseInsensitiveDict = headers

    def __str__(self):
        return pprint.pformat(vars(self))


class ApiConfiguration(object):
    def __init__(self, endpoints: [RequestConfig], base_url_config: BaseUrlConfig):
        self._endpoints = endpoints
        self._base_url_config = base_url_config

    def __call__(self, cls):
        n = {}
        for _endpoint in self._endpoints:
            n.update({
                _endpoint.name: _endpoint
            })
        cls.endpoints = n
        cls.base_url_config = self._base_url_config
        cls.__dir__ = self._dir(cls, cls.endpoints.keys())
        return cls

    @staticmethod
    def _dir(cls, endpoint_keys):
        def _dir(_cls):
            res = dir(type(cls)) + list(cls.__dict__.keys())
            res.extend(endpoint_keys)
            return res
        return _dir(cls)


class DictApiConfiguration(ApiConfiguration):
    def __init__(self, endpoints: list, base_url_config: BaseUrlConfig):
        _endpoints = [RequestConfig(
            path=e['path'],
            name=e['name'],
            method=e.get('method', 'GET'),
            options=e.get('options', {})
        ) for e in endpoints]
        super(DictApiConfiguration, self).__init__(_endpoints, base_url_config)


class JsonApiConfiguration(DictApiConfiguration):
    def __init__(self, endpoints: str, base_url_config: BaseUrlConfig):
        super(JsonApiConfiguration, self).__init__(json.loads(endpoints), base_url_config)
