import datetime
import importlib
import importlib.util
import inspect
import logging
import os
import pickle
import re
import shutil
import traceback
import zipfile
import json
from copy import deepcopy

import dask
import numpy as np
import pandas as pd
from bson.objectid import ObjectId
from dask.distributed import Client
from sklearn.base import BaseEstimator
from sklearn.dummy import DummyClassifier, DummyRegressor
from sklearn.externals import joblib
from sklearn.model_selection._split import BaseCrossValidator
from sklearn.model_selection import KFold

from photonai.__init__ import __version__
from photonai.base.cache_manager import CacheManager
from photonai.base.photon_elements import Stack, Switch, Preprocessing, CallbackElement, Branch, PipelineElement, \
    PhotonNative
from photonai.base.photon_pipeline import PhotonPipeline
from photonai.base.json_transformer import JsonTransformer
from photonai.helper.helper import print_double_metrics
from photonai.optimization import GridSearchOptimizer, TimeBoxedRandomGridSearchOptimizer, RandomGridSearchOptimizer, \
    SkOptOptimizer, RandomSearchOptimizer, SMACOptimizer, IntegerRange, FloatRange, Categorical
from photonai.photonlogger.logger import logger
from photonai.processing import ResultsHandler
from photonai.processing.metrics import Scorer
from photonai.processing.outer_folds import OuterFoldManager
from photonai.processing.photon_folds import FoldInfo
from photonai.processing.results_structure import MDBHyperpipe, MDBHyperpipeInfo, MDBDummyResults, MDBHelper, \
    FoldOperations, MDBConfig, MDBOuterFold


class OutputSettings:
    """
    Configuration class that specifies the format in which the results are saved. Results can be saved to a MongoDB
    or a simple son-file. You can also choose whether to save predictions and/or feature importances.

    Parameters
    ----------
    * `mongodb_connect_url` [str]:
        Valid mongodb connection url that specifies a database for storing the results

    * `save_predictions` [str, default='best']:
        Possible options are 'best' to save only the predictions of the best configuration for every outer fold, 'all'
        to save all predictions or 'None' to not save any predictions at all.

    * `save_feature_importances` [str, default='best']:
        Possible options are 'best' to save only the feature importances of the best configuration for every outer fold,
        'all' to save all feature importances or 'None' to not save any at all. Feature importances can only be saved
        if the chosen estimators have an attribute 'coef_' or 'feature_importances_'.

    * `project_folder` [bool, default=True]:
        If True, PHOTON writes a summary_file, the results of the hyperparameter optimization, the best model and the
        console output to the filesystem into the given project folder.

    * `project_folder` [str, default='']:
        The output folder in which all files generated by the PHOTON project are saved to.

    * `result_file_mode` [str, default='best']
        The possible save_modes for .photon file.

    * `user_id` [str]:
       The user name of the according PHOTON Wizard login

    * `wizard_object_id` [str]:
       The object id to map the designed pipeline in the PHOTON Wizard to the results in the PHOTON CORE Database

    * `wizard_project_name` [str]:
       How the project is titled in the PHOTON Wizard
    """
    def __init__(self,
                 mongodb_connect_url: str = None,
                 save_output: bool = True,
                 plots: bool = True,
                 overwrite_results: bool = False,
                 project_folder: str = '',
                 result_file_mode: str = "best",
                 user_id: str = '',
                 wizard_object_id: str = '',
                 wizard_project_name: str = ''):

        self.mongodb_connect_url = mongodb_connect_url
        self.overwrite_results = overwrite_results

        if project_folder == '':
            self.project_folder = os.getcwd()
        else:
            self.project_folder = project_folder

        self.results_folder = None
        self.log_file = None
        self.initialize_log_file()
        self.save_output = save_output
        self.save_predictions_from_best_config_inner_folds = None
        self.plots = plots

        # logging file handler
        self.logging_file_handler = None
        self.verbosity = 0

        self.user_id = user_id
        self.wizard_object_id = wizard_object_id
        self.wizard_project_name = wizard_project_name

        if result_file_mode in ["best", "all", "None", None]:
            if result_file_mode == "all":
                msg = "The result_file_mode 'all' is not impelmented yet."
                logger.error(msg)
                raise NotImplementedError(msg)
            else:
                self.result_file_mode = result_file_mode
        else:
            msg = "The result_file_mode required one out of ['best', 'all', 'None']. Default: 'best'."
            logger.error(msg)
            raise NotImplementedError(msg)

    def initialize_log_file(self):
        self.log_file = os.path.join(self.project_folder, 'photon_setup_errors.log')

    def _update_settings(self, name, timestamp):

        if not os.path.exists(self.project_folder):
            os.makedirs(self.project_folder)

        if self.save_output:
            # Todo: give rights to user if this is done by docker container
            if self.overwrite_results:
                self.results_folder = os.path.join(self.project_folder, name + '_results')
            else:
                self.results_folder = os.path.join(self.project_folder, name + '_results_' + timestamp)

            logger.info("Output Folder: " + self.results_folder)

            if not os.path.exists(self.results_folder):
                os.makedirs(self.results_folder)

            if os.path.basename(self.log_file) == "photon_setup_errors.log":
                self.log_file = 'photon_output.log'
            self.log_file = self._add_timestamp(self.log_file)
            self.set_log_file()

    def _add_timestamp(self, file):
        return os.path.join(self.results_folder, os.path.basename(file))

    def _get_log_level(self):
        if self.verbosity == 0:
            level = 25
        elif self.verbosity == 1:
            level = logging.INFO  # 20
        elif self.verbosity == 2:
            level = logging.DEBUG  # 10
        else:
            level = logging.WARN  # 30
        return level

    def set_log_file(self):
        logfile_directory = os.path.dirname(self.log_file)
        if not os.path.exists(logfile_directory):
            os.makedirs(logfile_directory)
        if self.logging_file_handler is None:
            self.logging_file_handler = logging.FileHandler(self.log_file)
            self.logging_file_handler.setLevel(self._get_log_level())
            logger.addHandler(self.logging_file_handler)
        else:
            self.logging_file_handler.close()
            self.logging_file_handler.baseFilename = self.log_file

    def set_log_level(self):
        verbose_num = self._get_log_level()
        logger.setLevel(verbose_num)
        for handler in logger.handlers:
            handler.setLevel(verbose_num)


class Hyperpipe(BaseEstimator):
    """
    Wrapper class for machine learning pipeline, holding all pipeline elements
    and managing the optimization of the hyperparameters

    Parameters
    ----------
    * `name` [str]:
        Name of hyperpipe instance

    * `inner_cv` [BaseCrossValidator]:
        Cross validation strategy to test hyperparameter configurations, generates the validation set

    * `outer_cv` [BaseCrossValidator]:
        Cross validation strategy to use for the hyperparameter search itself, generates the test set

    * `optimizer` [str or object, default="grid_search"]:
        Hyperparameter optimization algorithm

        - In case a string literal is given:
            - "grid_search": optimizer that iteratively tests all possible hyperparameter combinations
            - "random_grid_search": a variation of the grid search optimization that randomly picks hyperparameter
               combinations from all possible hyperparameter combinations
            - "timeboxed_random_grid_search": randomly chooses hyperparameter combinations from the set of all
               possible hyperparameter combinations and tests until the given time limit is reached
               - `limit_in_minutes`: int

        - In case an object is given:
          expects the object to have the following methods:
           - `next_config_generator`: returns a hyperparameter configuration in form of an dictionary containing
              key->value pairs in the sklearn parameter encoding `model_name__parameter_name: parameter_value`
           - `prepare`: takes a list of pipeline elements and their particular hyperparameters to test
           - `evaluate_recent_performance`: gets a tested config and the respective performance in order to
              calculate a smart next configuration to process

    * `metrics` [list of metric names as str]:
        Metrics that should be calculated for both training, validation and test set
        Use the preimported metrics from sklearn and photonai, or register your own

        - Metrics for `classification`:
            - `accuracy`: sklearn.metrics.accuracy_score
            - `matthews_corrcoef`: sklearn.metrics.matthews_corrcoef
            - `confusion_matrix`: sklearn.metrics.confusion_matrix,
            - `f1_score`: sklearn.metrics.f1_score
            - `hamming_loss`: sklearn.metrics.hamming_loss
            - `log_loss`: sklearn.metrics.log_loss
            - `precision`: sklearn.metrics.precision_score
            - `recall`: sklearn.metrics.recall_score
        - Metrics for `regression`:
            - `mean_squared_error`: sklearn.metrics.mean_squared_error
            - `mean_absolute_error`: sklearn.metrics.mean_absolute_error
            - `explained_variance`: sklearn.metrics.explained_variance_score
            - `r2`: sklearn.metrics.r2_score
        - Other metrics
            - `pearson_correlation`: photon_core.framework.Metrics.pearson_correlation
            - `variance_explained`:  photon_core.framework.Metrics.variance_explained_score
            - `categorical_accuracy`: photon_core.framework.Metrics.categorical_accuracy_score

    * `best_config_metric` [str]:
        The metric that should be maximized or minimized in order to choose the best hyperparameter configuration

    * `eval_final_performance` [bool, default=True]:
        If the metrics should be calculated for the test set, otherwise the test set is seperated but not used

    * `test_size` [float, default=0.2]:
        the amount of the data that should be left out if no outer_cv is given and
        eval_final_perfomance is set to True

    * `set_random_seed` [bool, default=False]:
        If True sets the random seed to 42

    * `verbosity` [int, default=0]:
        The level of verbosity, 0 is least talkative and gives only warn and error, 1 gives adds info and 2 adds debug

    * `groups` [array-like, default=None]:
        Info for advanced cross validation strategies, such as LeaveOneSiteOut-CV about the affiliation
        of the rows in the data. Also works with continuous values and StratifiedKFoldRegression. In case a group
        variable and a StratifiedCV is passed, the targets will be ignored and only the group variable will be used
        for the stratification.

    Attributes
    ----------
    * `optimum_pipe` [Pipeline]:
        An sklearn pipeline object that is fitted to the training data according to the best hyperparameter
        configuration found. Currently, we don't create an ensemble of all best hyperparameter configs over all folds.
        We find the best config by comparing the test error across outer folds. The hyperparameter config of the best
        fold is used as the optimal model and is then trained on the complete set.

    * `best_config` [dict]:
        Dictionary containing the hyperparameters of the best configuration.
        Contains the parameters in the sklearn interface of model_name__parameter_name: parameter value

    * `results` [MDBHyperpipe]:
        Object containing all information about the for the performed hyperparameter search.
        Holds the training and test metrics for all outer folds, inner folds and configurations, as well as
        additional information.

    * `elements` [list]:
        Contains all PipelineElement or Hyperpipe objects that are added to the pipeline.

    Example
    -------
        manager = Hyperpipe('test_manager',
                            optimizer='timeboxed_random_grid_search', optimizer_params={'limit_in_minutes': 1},
                            outer_cv=ShuffleSplit(test_size=0.2, n_splits=1),
                            inner_cv=KFold(n_splits=10, shuffle=True),
                            metrics=['accuracy', 'precision', 'recall', "f1_score"],
                            best_config_metric='accuracy', eval_final_performance=True,
                            verbose=2)

   """
    def __init__(self, name,
                 inner_cv: BaseCrossValidator = None,
                 outer_cv=None,
                 optimizer='grid_search',
                 optimizer_params: dict = {},
                 metrics=None,
                 best_config_metric=None,
                 eval_final_performance=True,
                 test_size: float = 0.2,
                 calculate_metrics_per_fold: bool = True,
                 calculate_metrics_across_folds: bool = False,
                 random_seed=False,
                 verbosity=0,
                 learning_curves: bool = False,
                 learning_curves_cut: FloatRange = None,
                 output_settings=None,
                 performance_constraints=None,
                 permutation_id: str=None,
                 cache_folder: str=None,
                 nr_of_processes: int = 1):

        self.name = re.sub(r'\W+', '', name)
        self.permutation_id = permutation_id
        if cache_folder:
            self.cache_folder = os.path.join(cache_folder, self.name)
        else:
            self.cache_folder = None
        # ====================== Cross Validation ===========================
        # check if both calculate_metrics_per_folds and calculate_metrics_across_folds is False
        if not calculate_metrics_across_folds and not calculate_metrics_per_fold:
            raise NotImplementedError("Apparently, you've set calculate_metrics_across_folds=False and "
                                      "calculate_metrics_per_fold=False. In this case PHOTON does not calculate "
                                      "any metrics which doesn't make any sense. Set at least one to True.")
        if inner_cv is None:
            msg = "PHOTON requires an inner_cv split. Please enable inner cross-validation. As exmaple: Hyperpipe(..." \
                  " inner_cv = KFold(n_splits = 3), ...). Ensure you import the cross_validation object first."
            logger.error(msg)
            raise AttributeError(msg)

        # use default cut 'FloatRange(0, 1, 'range', 0.2)' if learning_curves = True but learning_curves_cut is None
        if learning_curves and learning_curves_cut is None:
            learning_curves_cut = FloatRange(0, 1, 'range', 0.2)
        elif not learning_curves and learning_curves_cut is not None:
            learning_curves_cut = None

        self.cross_validation = Hyperpipe.CrossValidation(inner_cv=inner_cv,
                                                          outer_cv=outer_cv,
                                                          eval_final_performance=eval_final_performance,
                                                          test_size=test_size,
                                                          calculate_metrics_per_fold=calculate_metrics_per_fold,
                                                          calculate_metrics_across_folds=calculate_metrics_across_folds,
                                                          learning_curves=learning_curves,
                                                          learning_curves_cut=learning_curves_cut)

        # ====================== Data ===========================
        self.data = Hyperpipe.Data()

        # ====================== Result Logging ===========================
        if output_settings:
            self.output_settings = output_settings
        else:
            self.output_settings = OutputSettings()
        self.results_handler = None
        self.results = None
        self.best_config = None

        # ====================== Pipeline ===========================
        self.elements = []
        self._pipe = None
        self.optimum_pipe = None
        self.preprocessing = None

        # ====================== Perfomance Optimization ===========================

        self.optimization = Hyperpipe.Optimization(metrics=metrics,
                                                   best_config_metric=best_config_metric,
                                                   optimizer_input=optimizer,
                                                   optimizer_params=optimizer_params,
                                                   performance_constraints=performance_constraints)

        self.optimization.sanity_check_metrics()

        # ====================== Internals ===========================

        self.is_final_fit = False
        self.nr_of_processes = nr_of_processes
        self.random_state = random_seed
        if random_seed:
            import random
            random.seed(random_seed)

        # initialize logging

        # update output options to add pipe name and timestamp to results folder
        self._verbosity = 0
        self.verbosity = verbosity
        self.output_settings.set_log_file()

    # Helper Classes
    #
    #
    #
    # ============= Cross Validation ==================================================================
    class CrossValidation:

        def __init__(self, inner_cv, outer_cv,
                     eval_final_performance, test_size,
                     calculate_metrics_per_fold,
                     calculate_metrics_across_folds,
                     learning_curves,
                     learning_curves_cut):
            self.inner_cv = inner_cv
            self.outer_cv = outer_cv
            self.eval_final_performance = eval_final_performance
            self.test_size = test_size

            self.learning_curves = learning_curves
            self.learning_curves_cut = learning_curves_cut

            self.calculate_metrics_per_fold = calculate_metrics_per_fold
            # Todo: if self.outer_cv is LeaveOneOut: Set calculate metrics across folds to True -> Print
            self.calculate_metrics_across_folds = calculate_metrics_across_folds

            self.outer_folds = None
            self.inner_folds = dict()


    # ============= Data ==================================================================
    class Data:

        def __init__(self, X=None, y=None, kwargs=None):
            self.X = X
            self.y = y
            self.kwargs = kwargs

    # ============= Performance Optimization ==================================================================
    class Optimization:

        OPTIMIZER_DICTIONARY = {'grid_search': GridSearchOptimizer,
                                'random_grid_search': RandomGridSearchOptimizer,
                                'timeboxed_random_grid_search': TimeBoxedRandomGridSearchOptimizer,
                                'sk_opt': SkOptOptimizer,
                                'smac' : SMACOptimizer,
                                'random_search': RandomSearchOptimizer}

        # 'fabolas': FabolasOptimizer}

        def __init__(self, optimizer_input, optimizer_params,
                     metrics, best_config_metric, performance_constraints):

            self._optimizer_input = ''
            self.optimizer_input_str = optimizer_input
            self.optimizer_params = optimizer_params
            self.metrics = metrics
            self._best_config_metric = ''
            self.maximize_metric = True
            self.best_config_metric = best_config_metric
            self.performance_constraints = performance_constraints

        @property
        def best_config_metric(self):
            return self._best_config_metric

        @best_config_metric.setter
        def best_config_metric(self, value):
            self._best_config_metric = value
            if isinstance(self.best_config_metric, str):
                self.maximize_metric = Scorer.greater_is_better_distinction(self.best_config_metric)

        @property
        def optimizer_input_str(self):
            return self._optimizer_input

        @optimizer_input_str.setter
        def optimizer_input_str(self, value):
            if isinstance(value, str):
                if value not in Hyperpipe.Optimization.OPTIMIZER_DICTIONARY:
                    raise ValueError("Optimizer " + value + " not supported right now.")
            self._optimizer_input = value

        def sanity_check_metrics(self):

            # --------------------- Validity of metrics ----------------
            if isinstance(self.best_config_metric, list) or not isinstance(self.best_config_metric, str):

                if self.metrics is not None:
                    warning_text = "Best Config Metric must be a single metric given as string, no list. " \
                                   "PHOTON chose the first one from the list of metrics to calculate."

                    self.best_config_metric = self.metrics[0]
                    logger.warn(warning_text)
                    raise Warning(warning_text)
                else:
                    error_msg = "No metrics were chosen. Please choose metrics to quantify your performance and set " \
                                "the best_config_metric so that PHOTON which optimizes for"
                    logger.error(error_msg)
                    raise ValueError(error_msg)

            if self.best_config_metric is not None:
                if self.metrics is None:
                    self.metrics = [self.best_config_metric]
                else:
                    if self.best_config_metric not in self.metrics:
                        self.metrics.append(self.best_config_metric)

            if self.best_config_metric is None and len(self.metrics) > 0:
                self.best_config_metric = self.metrics[0]
                warning_text = "No best config metric was given, so PHOTON chose the first in the list of metrics as " \
                               "criteria for choosing the best configuration."
                logger.warn(warning_text)
                raise Warning(warning_text)
            else:
                if self.metrics is None or len(self.metrics) == 0:
                    metric_error_text = "List of Metrics to calculate should not be empty"
                    logger.error(metric_error_text)
                    raise ValueError(metric_error_text)

        def get_optimizer(self):
            if isinstance(self.optimizer_input_str, str):
                # instantiate optimizer from string
                optimizer_class = self.OPTIMIZER_DICTIONARY[self.optimizer_input_str]
                optimizer_instance = optimizer_class(**self.optimizer_params)
                return optimizer_instance
            else:
                # Todo: check if object has the right interface
                return self.optimizer_input_str

        def get_optimum_config(self, tested_configs, fold_operation=FoldOperations.MEAN):
            """
            Looks for the best configuration according to the metric with which the configurations are compared -> best config metric
            :param tested_configs: the list of tested configurations and their performances
            :return: MDBConfiguration that has performed best
            """

            list_of_config_vals = []
            list_of_non_failed_configs = [conf for conf in tested_configs if not conf.config_failed]

            if len(list_of_non_failed_configs) == 0:
                raise Warning("No Configs found which did not fail.")
            try:

                if len(list_of_non_failed_configs) == 1:
                    best_config_outer_fold = list_of_non_failed_configs[0]
                else:
                    for config in list_of_non_failed_configs:
                        list_of_config_vals.append(
                            MDBHelper.get_metric(config, fold_operation, self.best_config_metric, train=False))

                    if self.maximize_metric:
                        # max metric
                        best_config_metric_nr = np.argmax(list_of_config_vals)
                    else:
                        # min metric
                        best_config_metric_nr = np.argmin(list_of_config_vals)

                    best_config_outer_fold = list_of_non_failed_configs[best_config_metric_nr]

                # inform user
                logger.debug('Optimizer metric: ' + self.best_config_metric + '\n' +
                             '   --> Maximize metric: ' + str(self.maximize_metric))

                logger.info('Number of tested configurations: ' + str(len(tested_configs)))
                logger.photon_system_log('---------------------------------------------------------------------------------------------------------------')
                logger.photon_system_log('BEST_CONFIG ')
                logger.photon_system_log('---------------------------------------------------------------------------------------------------------------')
                logger.photon_system_log(json.dumps(best_config_outer_fold.human_readable_config, indent=4,
                                                    sort_keys=True))

                return best_config_outer_fold
            except BaseException as e:
                logger.error(str(e))

        def get_optimum_config_outer_folds(self, outer_folds):
            list_of_scores = list()
            for outer_fold in outer_folds:
                metrics = outer_fold.best_config.best_config_score.validation.metrics
                list_of_scores.append(metrics[self.best_config_metric])

            if self.maximize_metric:
                # max metric
                best_config_metric_nr = np.argmax(list_of_scores)
            else:
                # min metric
                best_config_metric_nr = np.argmin(list_of_scores)

            best_config = outer_folds[best_config_metric_nr].best_config
            return best_config

    # Pipeline Management & Interface
    #
    #
    @property
    def verbosity(self):
        return self._verbosity

    @verbosity.setter
    def verbosity(self, value):
        self._verbosity = value
        self.output_settings.verbosity = self._verbosity
        self.output_settings.set_log_level()

    def __iadd__(self, pipe_element):
        """
        Add an element to the machine learning pipeline
        Returns self

        Parameters
        ----------
        * 'pipe_element' [PipelineElement]:
            The object to add to the machine learning pipeline, being either a transformer or an estimator.

        """
        if isinstance(pipe_element, Preprocessing):
            self.preprocessing = pipe_element
        elif isinstance(pipe_element, CallbackElement):
            pipe_element.needs_y = True
            self.elements.append(pipe_element)
        else:
            if isinstance(pipe_element, PipelineElement) or issubclass(type(pipe_element), PhotonNative):
                self.elements.append(pipe_element)
            else:
                raise TypeError("Element must be of type Pipeline Element")
        return self

    def add(self, pipe_element):
        """
           Add an element to the machine learning pipeline
           Returns self

           Parameters
           ----------
           * `pipe_element` [PipelineElement or Hyperpipe]:
               The object to add to the machine learning pipeline, being either a transformer or an estimator.

           """
        self.__iadd__(pipe_element)

    def _prepare_dummy_estimator(self):
        self.results.dummy_estimator = MDBDummyResults()

        if self.estimation_type == 'regressor':
            self.results.dummy_estimator.strategy = 'mean'
            return DummyRegressor(strategy=self.results.dummy_estimator.strategy)
        elif self.estimation_type == 'classifier':
            self.results.dummy_estimator.strategy = 'most_frequent'
            return DummyClassifier(strategy=self.results.dummy_estimator.strategy)
        else:
            logger.info('Estimator does not specify whether it is a regressor or classifier. DummyEstimator '
                          'step skipped.')
            return

    def _prepare_result_logging(self, start_time):

        self.results = MDBHyperpipe(name=self.name, version=__version__)
        self.results.hyperpipe_info = MDBHyperpipeInfo()

        # in case eval final performance is false, we have no outer fold predictions
        if not self.cross_validation.eval_final_performance:
            self.output_settings.save_predictions_from_best_config_inner_folds = True
        self.results_handler = ResultsHandler(self.results, self.output_settings)

        self.results.computation_start_time = start_time
        self.results.hyperpipe_info.estimation_type = self.estimation_type
        self.results.output_folder = self.output_settings.results_folder

        if self.permutation_id is not None:
            self.results.permutation_id = self.permutation_id

        # save wizard information to photon db in order to map results to the wizard design object
        if self.output_settings and hasattr(self.output_settings, 'wizard_object_id'):
            if self.output_settings.wizard_object_id:
                self.name = self.output_settings.wizard_object_id
                self.results.name = self.output_settings.wizard_object_id
                self.results.wizard_object_id = ObjectId(self.output_settings.wizard_object_id)
                self.results.wizard_system_name = self.output_settings.wizard_project_name
                self.results.user_id = self.output_settings.user_id
        self.results.outer_folds = []
        self.results.hyperpipe_info.eval_final_performance = self.cross_validation.eval_final_performance
        self.results.hyperpipe_info.best_config_metric = self.optimization.best_config_metric
        self.results.hyperpipe_info.metrics = self.optimization.metrics
        self.results.hyperpipe_info.learning_curves_cut = self.cross_validation.learning_curves_cut
        self.results.hyperpipe_info.maximize_best_config_metric = self.optimization.maximize_metric

        # optimization
        def _format_cross_validation(cv):
            if cv:
                string = "{}(".format(cv.__class__.__name__)
                for key, val in cv.__dict__.items():
                    string += "{}={}, ".format(key, val)
                return string[:-2] + ")"
            else:
                return "None"

        self.results.hyperpipe_info.cross_validation = {'OuterCV': _format_cross_validation(self.cross_validation.outer_cv),
                                                        'InnerCV': _format_cross_validation(self.cross_validation.inner_cv)}
        self.results.hyperpipe_info.data = {'X_shape': self.data.X.shape, 'y_shape': self.data.y.shape}
        self.results.hyperpipe_info.optimization = {'Optimizer': self.optimization.optimizer_input_str,
                                                        'OptimizerParams': str(self.optimization.optimizer_params),
                                                        'BestConfigMetric': self.optimization.best_config_metric}

        # add json file of hyperpipe attributes
        try:
            json_transformer = JsonTransformer()
            json_transformer.to_json_file(self, self.output_settings.results_folder+"/hyperpipe_config.json")
        except:
            logger.warn("JsonTransformer was unable to create the .json file.")

        # add flowchart to results
        try:
            flowchart = FlowchartCreator(self.elements)
            self.results.hyperpipe_info.flowchart = flowchart.create_str()
        except:
            self.results.hyperpipe_info.flowchart = ""
        # self.results_handler.save()

    def _finalize_optimization(self):
        # ==================== EVALUATING RESULTS OF HYPERPARAMETER OPTIMIZATION ===============================
        # 1. computing average metrics
        # 2. finding overall best config
        # 3. training model with best config
        # 4. persisting best model
        logger.clean_info('')
        logger.clean_info(
            '***************************************************************************************************************')
        logger.info("Finished all outer fold hyperparameter optimizations.")
        logger.info("Now analysing the results...")

        # computer dummy metrics
        config_item = MDBConfig()
        dummy_results = [outer_fold.dummy_results for outer_fold in self.results.outer_folds]
        config_item.inner_folds = [f for f in dummy_results if f is not None]
        if len(config_item.inner_folds) > 0:
            self.results.dummy_estimator.train, self.results.dummy_estimator.test = MDBHelper.aggregate_metrics_for_inner_folds(
                config_item.inner_folds,
                self.optimization.metrics)

        # Compute all final metrics
        self.results.metrics_train, self.results.metrics_test = MDBHelper.aggregate_metrics_for_outer_folds(self.results.outer_folds,
                                                                                                            self.optimization.metrics)

        # Find best config across outer folds
        best_config = self.optimization.get_optimum_config_outer_folds(self.results.outer_folds)
        self.best_config = best_config.config_dict
        self.results.best_config = best_config

        logger.photon_system_log('')
        logger.photon_system_log(
            '===============================================================================================================')
        logger.photon_system_log('OVERALL BEST CONFIGURATION')
        logger.photon_system_log(
            '===============================================================================================================')
        logger.photon_system_log(json.dumps(self.results.best_config.human_readable_config, indent=4, sort_keys=True))
        print_double_metrics(self.results.best_config.best_config_score.training.metrics,
                             self.results.best_config.best_config_score.validation.metrics,
                             photon_system_log=True)

        # save results again
        self.results.computation_end_time = datetime.datetime.now()
        self.results.computation_completed = True
        self.results_handler.save()

        # write all convenience files (summary, predictions_file and plots)
        self.results_handler.write_convenience_files()

        # set self to best config
        self.optimum_pipe = self._pipe
        self.optimum_pipe.set_params(**self.best_config)

        if self.output_settings.result_file_mode == 'best':
            logger.info("Fitting best model...")
            # set self to best config
            self.optimum_pipe = self._pipe
            self.optimum_pipe.set_params(**self.best_config)

            # set caching
            # we want caching disabled in general but still want to do single subject caching
            self.recursive_cache_folder_propagation(self.optimum_pipe, self.cache_folder, 'fixed_fold_id')
            self.optimum_pipe.caching = False

            # disable multiprocessing when fitting optimum pipe
            # (otherwise inverse_transform won't work for BrainAtlas/Mask)
            self.disable_multiprocessing_recursively(self.optimum_pipe)

            self.optimum_pipe.fit(self.data.X, self.data.y, **self.data.kwargs)

            # Before saving the optimum pipe, add preprocessing
            self.optimum_pipe._add_preprocessing(self.preprocessing)

            # Now truly set to no caching (including single_subject_caching)
            self.recursive_cache_folder_propagation(self.optimum_pipe, None, None)

            if self.output_settings.save_output:
                try:
                    pretrained_model_filename = os.path.join(self.output_settings.results_folder, 'photon_best_model.photon')
                    PhotonModelPersistor.save_optimum_pipe(self.optimum_pipe, pretrained_model_filename)
                    logger.info("Saved best model to file.")
                except FileNotFoundError as e:
                    logger.info("Could not save best model to file")
                    logger.error(str(e))

                # get feature importances of optimum pipe
                logger.info("Mapping back feature importances...")
                feature_importances = self.optimum_pipe.feature_importances_

                if not feature_importances:
                    logger.info("No feature importances available for {}!".format(self.optimum_pipe.elements[-1][0]))
                else:
                    self.results.best_config_feature_importances = feature_importances

                    # get backmapping
                    backmapping, _, _ = self.optimum_pipe.inverse_transform(feature_importances, None)

                    # save backmapping
                    self.results_handler.save_backmapping(filename='optimum_pipe_feature_importances_backmapped',
                                                          backmapping=backmapping)

                # save learning curves
                if self.cross_validation.learning_curves:
                    self.results_handler.save_all_learning_curves()

        elapsed_time = self.results.computation_end_time - self.results.computation_start_time
        logger.photon_system_log('')
        logger.photon_system_log(
            'Analysis ' + self.name + " done in " + str(elapsed_time))
        if self.output_settings.results_folder is not None:
            logger.photon_system_log('Your results are stored in ' + self.output_settings.results_folder)
        logger.photon_system_log('***************************************************************************************************************')
        logger.photon_system_log('PHOTON ' + str(__version__) + ' - www.photon-ai.com ')

    @staticmethod
    def disable_multiprocessing_recursively(pipe):
        if isinstance(pipe, (Stack, Branch, Switch, Preprocessing)):
            if hasattr(pipe, 'nr_of_processes'):
                pipe.nr_of_processes = 1
            for child in pipe.elements:
                if hasattr(child, 'base_element'):
                    Hyperpipe.disable_multiprocessing_recursively(child.base_element)
        elif isinstance(pipe, PhotonPipeline):
            for name, child in pipe.named_steps.items():
                Hyperpipe.disable_multiprocessing_recursively(child)
        else:
            if hasattr(pipe, 'nr_of_processes'):
                pipe.nr_of_processes = 1

    def _input_data_sanity_checks(self, data, targets, **kwargs):
        # ==================== SANITY CHECKS ===============================
        # 1. Make to numpy arrays
        # 2. erase all Nan targets

        logger.info("Checking input data...")
        self.data.X = data
        self.data.y = targets
        self.data.kwargs = kwargs

        try:
            if self.data.X is None:
                raise ValueError("(Input-)data is a NoneType.")
            if self.data.y is None:
                raise ValueError("(Input-)target is a NoneType.")

            shape_x = np.shape(self.data.X)
            shape_y = np.shape(self.data.y)
            if len(shape_y) != 1:
                if len(np.shape(np.squeeze(self.data.y))) == 1:
                    # use np.squeeze for non 1D targets.
                    self.data.y = np.squeeze(self.data.y)
                    shape_y = np.shape(self.data.y)
                else:
                    raise ValueError("Target is not one-dimensional.")
            if not shape_x[0] == shape_y[0]:
                raise IndexError(
                    "Size of targets mismatch to size of the data: " + str(shape_x[0]) + " - " + str(shape_y[0]))
        except IndexError as ie:
            logger.error("IndexError: " + str(ie))
            raise ie
        except ValueError as ve:
            logger.error("ValueError: " + str(ve))
            raise ve
        except Exception as e:
            logger.error("Error: " + str(e))
            raise e

        # be compatible to list of (image-) files
        if isinstance(self.data.X, list):
            self.data.X = np.asarray(self.data.X)
        elif isinstance(self.data.X, (pd.DataFrame, pd.Series)):
            self.data.X = self.data.X.to_numpy()
        if isinstance(self.data.y, list):
            self.data.y = np.asarray(self.data.y)
        elif isinstance(self.data.y, pd.Series) or isinstance(self.data.y, pd.DataFrame):
            self.data.y = self.data.y.to_numpy()

        # at first first, erase all rows where y is Nan if preprocessing has not done it already
        try:
            nans_in_y = np.isnan(self.data.y)
            nr_of_nans = len(np.where(nans_in_y == 1)[0])
            if nr_of_nans > 0:
                logger.info("You have " + str(nr_of_nans) + " Nans in your target vector, "
                                                              "PHOTON erases every data item that has a Nan Target")
                self.data.X = self.data.X[~nans_in_y]
                self.data.y = self.data.y[~nans_in_y]
        except Exception as e:
            # This is only for convenience so if it fails then never mind
            logger.error("Removing Nans in target vector failed: " + str(e))
            pass

        logger.info("Running analysis with " + str(self.data.y.shape[0]) + " samples.")

    # @staticmethod
    # def prepare_caching(cache_folder):
    #     if cache_folder and not os.path.isdir(cache_folder):
    #         os.makedirs(cache_folder, exist_ok=True)

    @staticmethod
    def recursive_cache_folder_propagation(element, cache_folder, inner_fold_id):
        if isinstance(element, (Switch, Stack, Preprocessing)):
            for child in element.elements:
                Hyperpipe.recursive_cache_folder_propagation(child, cache_folder, inner_fold_id)

        elif isinstance(element, Branch):
            # in case it's a Branch, we create a cache subfolder and propagate it to every child
            if cache_folder:
                cache_folder = os.path.join(cache_folder, element.name)
            Hyperpipe.recursive_cache_folder_propagation(element.base_element, cache_folder, inner_fold_id)
            # Hyperpipe.prepare_caching(element.base_element.cache_folder)

        elif isinstance(element, PhotonPipeline):
            element.fold_id = inner_fold_id
            element.cache_folder = cache_folder

            # pipe.caching is automatically set to True or False by .cache_folder setter

            for name, child in element.named_steps.items():
                # we need to check if any element is Branch, Stack or Swtich
                Hyperpipe.recursive_cache_folder_propagation(child, cache_folder, inner_fold_id)

        # if it's a simple PipelineElement, then we just don't do anything

    def preprocess_data(self):
        # if there is a preprocessing pipeline, we apply it first.
        if self.preprocessing is not None:
            logger.info("Applying preprocessing steps...")
            self.preprocessing.fit(self.data.X, self.data.y, **self.data.kwargs)
            self.data.X, self.data.y, self.data.kwargs = self.preprocessing.transform(self.data.X, self.data.y,
                                                                                      **self.data.kwargs)

    def _prepare_pipeline(self):
        self._pipe = Branch.prepare_photon_pipe(self.elements)
        self._pipe = Branch.sanity_check_pipeline(self._pipe)
        if self.random_state:
            self._pipe.random_state = self.random_state

    @property
    def estimation_type(self):
        estimation_type = getattr(self.elements[-1], '_estimator_type')
        if estimation_type is None:
            raise NotImplementedError("Last element in Hyperpipe should be an estimator.")
        else:
            return estimation_type

    @staticmethod
    def fit_outer_folds(outer_fold_computer, X, y, kwargs, cache_folder):
        try:
            outer_fold_computer.fit(X, y, **kwargs)
        finally:
            CacheManager.clear_cache_files(cache_folder)
        return

    def fit(self, data, targets, **kwargs):
        """
        Starts the hyperparameter search and/or fits the pipeline to the data and targets

        Manages the nested cross validated hyperparameter search:

        1. Filters the data according to filter strategy (1) and according to the imbalanced_data_strategy (2)
        2. requests new configurations from the hyperparameter search strategy, the optimizer,
        3. initializes the testing of a specific configuration,
        4. communicates the result to the optimizer,
        5. repeats 2-4 until optimizer delivers no more configurations to test
        6. finally searches for the best config in all tested configs,
        7. trains the pipeline with the best config and evaluates the performance on the test set

        Parameters
        ----------
         * `data` [array-like, shape=[N, D]]:
            the training and test data, where N is the number of samples and D is the number of features.

         * `targets` [array-like, shape=[N]]:
            the truth values, where N is the number of samples.


        Returns
        -------
         * 'self'
            Returns self

        """

        # switch to result output folder
        start = datetime.datetime.now()
        self.output_settings._update_settings(self.name, start.strftime("%Y-%m-%d_%H-%M-%S"))

        logger.photon_system_log('***************************************************************************************************************')
        logger.photon_system_log('PHOTON ANALYSIS: ' + self.name)
        logger.photon_system_log('***************************************************************************************************************')
        logger.info("Preparing data and PHOTON objects for analysis...")

        # loop over outer cross validation
        if self.nr_of_processes > 1:
            hyperpipe_client = Client(threads_per_worker=1, n_workers=self.nr_of_processes, processes=False)

        try:
            # check data
            self._input_data_sanity_checks(data, targets, **kwargs)
            # create photon pipeline
            self._prepare_pipeline()
            # initialize the progress monitors
            self._prepare_result_logging(start)
            # apply preprocessing
            self.preprocess_data()

            if not self.is_final_fit:

                # Outer Folds
                outer_folds = FoldInfo.generate_folds(self.cross_validation.outer_cv,
                                                      self.data.X, self.data.y, self.data.kwargs,
                                                      self.cross_validation.eval_final_performance,
                                                      self.cross_validation.test_size)

                self.cross_validation.outer_folds = {f.fold_id: f for f in outer_folds}
                delayed_jobs = []

                # Run Dummy Estimator
                dummy_estimator = self._prepare_dummy_estimator()

                if self.cache_folder is not None:
                    logger.info("Removing cache files...")
                    CacheManager.clear_cache_files(self.cache_folder, force_all=True)

                # loop over outer cross validation
                for i, outer_f in enumerate(outer_folds):

                    # 1. generate OuterFolds Object
                    outer_fold = MDBOuterFold(fold_nr=outer_f.fold_nr)
                    outer_fold_computer = OuterFoldManager(self._pipe,
                                                           self.optimization,
                                                           outer_f.fold_id,
                                                           self.cross_validation,
                                                           cache_folder=self.cache_folder,
                                                           cache_updater=self.recursive_cache_folder_propagation,
                                                           dummy_estimator=dummy_estimator,
                                                           result_obj=outer_fold)
                    # 2. monitor outputs
                    self.results.outer_folds.append(outer_fold)

                    if self.nr_of_processes > 1:
                        result = dask.delayed(Hyperpipe.fit_outer_folds)(outer_fold_computer,
                                                                         self.data.X,
                                                                         self.data.y,
                                                                         self.data.kwargs,
                                                                         self.cache_folder)
                        delayed_jobs.append(result)
                    else:
                        try:
                            # 3. fit
                            outer_fold_computer.fit(self.data.X, self.data.y, **self.data.kwargs)
                            # 4. save outer fold results
                            self.results_handler.save()
                        finally:
                            # 5. clear cache
                            CacheManager.clear_cache_files(self.cache_folder)

                if self.nr_of_processes > 1:
                    dask.compute(*delayed_jobs)
                    self.results_handler.save()

                # evaluate hyperparameter optimization results for best config
                self._finalize_optimization()

                # clear complete cache ?
                CacheManager.clear_cache_files(self.cache_folder, force_all=True)

            ###############################################################################################
            else:
                self.preprocess_data()
                self._pipe.fit(self.data.X, self.data.y, **kwargs)
        except Exception as e:
            logger.error(e)
            logger.error(traceback.format_exc())
            traceback.print_exc()
            raise e
        finally:
            if self.nr_of_processes > 1:
                hyperpipe_client.close()
        return self

    def predict(self, data, **kwargs):
        """
        Use the optimum pipe to predict the data

        Returns
        -------
            predicted targets

        """
        # Todo: if local_search = true then use optimized pipe here?
        if self._pipe:
            return self.optimum_pipe.predict(data, **kwargs)

    def predict_proba(self, data, **kwargs):
        """
        Predict probabilities

        Returns
        -------
        predicted probabilities

        """
        if self._pipe:
            return self.optimum_pipe.predict_proba(data, **kwargs)

    def transform(self, data, **kwargs):
        """
        Use the optimum pipe to transform the data
        """
        if self._pipe:
            X, _, _ = self.optimum_pipe.transform(data, y=None, **kwargs)
            return X

    def copy_me(self):
        """
        Helper function to copy an entire Hyperpipe
        :return: Hyperpipe
        """
        signature = inspect.getfullargspec(OutputSettings.__init__)[0]
        settings = OutputSettings()
        for attr in signature:
            if hasattr(self.output_settings, attr):
                setattr(settings, attr, getattr(self.output_settings, attr))
        settings.initialize_log_file()

        # create new Hyperpipe instance
        pipe_copy = Hyperpipe(name=self.name,
                              inner_cv=deepcopy(self.cross_validation.inner_cv),
                              outer_cv=deepcopy(self.cross_validation.outer_cv),
                              best_config_metric=self.optimization.best_config_metric,
                              metrics=self.optimization.metrics,
                              optimizer=self.optimization.optimizer_input_str,
                              optimizer_params=self.optimization.optimizer_params,
                              output_settings=settings)

        signature = inspect.getfullargspec(self.__init__)[0]
        for attr in signature:
            if hasattr(self, attr) and attr != 'output_settings':
                setattr(pipe_copy, attr, getattr(self, attr))

        if hasattr(self, 'preprocessing') and self.preprocessing:
            preprocessing = Preprocessing()
            for element in self.preprocessing.elements:
                preprocessing += element.copy_me()
            pipe_copy += preprocessing
        if hasattr(self, 'elements'):
            for element in self.elements:
                pipe_copy += element.copy_me()
        return pipe_copy

    def save_optimum_pipe(self, filename=None, password=None):
        if filename is None:
            filename = "photon_" + self.name + "_best_model.p"
        PhotonModelPersistor.save_optimum_pipe(self, filename, password)

    @staticmethod
    def load_optimum_pipe(file, password=None):
        return PhotonModelPersistor.load_optimum_pipe(file, password)

    def inverse_transform_pipeline(self, hyperparameters: dict, data, targets, data_to_inverse):
        """
        Inverse transform data for a pipeline with specific hyperparameter configuration

        1. Copy Sklearn Pipeline,
        2. Set Parameters
        3. Fit Pipeline to data and targets
        4. Inverse transform data with that pipeline

        Parameters
        ----------
        * `hyperparameters` [dict]:
            The concrete configuration settings for the pipeline elements
        * `data` [array-like]:
            The training data to which the pipeline is fitted
        * `targets` [array-like]:
            The truth values for training
        * `data_to_inverse` [array-like]:
            The data that should be inversed after training

        Returns
        -------
        Inversed data as array
        """
        copied_pipe = self.pipe.copy_me()
        copied_pipe.set_params(**hyperparameters)
        copied_pipe.fit(data, targets)
        return copied_pipe.inverse_transform(data_to_inverse)


class FlowchartCreator(object):

    def __init__(self, pipeline_elements):
        self.pipeline_elements = pipeline_elements
        self.chart_str = ""

    def create_str(self):
        header_layout = ""
        header_relate = ""
        old_element = ""
        for pipeline_element in self.pipeline_elements:
            header_layout = header_layout + "[" + pipeline_element.name + "]"
            if old_element:
                header_relate = header_relate + "[" + old_element + "]" + "->" + "[" + pipeline_element.name + "]\n"
            old_element = pipeline_element.name

        self.chart_str = "Layout:\n" + header_layout + "\nRelate:\n" + header_relate + "\n"

        for pipeline_element in self.pipeline_elements:
            self.chart_str = self.chart_str + self.recursive_element(pipeline_element, "")

        return self.chart_str

    @staticmethod
    def format_cross_validation(cv):
        if cv:
            string = "{}(".format(cv.__class__.__name__)
            for key, val in cv.__dict__.items():
                string += "{}={}, ".format(key, val)
            return string[:-2] + ")"
        else:
            return "None"

    @staticmethod
    def format_optimizer(optimizer):
        return optimizer.optimizer_input_str, optimizer.optimizer_params, optimizer.metrics, optimizer.best_config_metric

    def format_kwargs(self, kwargs):
        pass

    @staticmethod
    def format_hyperparameter(hyperparameter):
        if isinstance(hyperparameter, IntegerRange):
            return """IntegerRange(start: {},
                                   stop: {}, 
                                   step: {}, 
                                   range_type: {})""".format(hyperparameter.start, hyperparameter.stop,
                                                                           hyperparameter.step, hyperparameter.range_type)
        elif isinstance(hyperparameter, FloatRange):
            return """FloatRange(start: {},
                                   stop: {}, 
                                   step: {}, 
                                   range_type: {})""".format(hyperparameter.start,
                                                               hyperparameter.stop,
                                                               hyperparameter.step,
                                                               hyperparameter.range_type)
        elif isinstance(hyperparameter, Categorical):
            return str(hyperparameter.values)
        else:
            return str(hyperparameter)

    def recursive_element(self, pipe_element, parent):

        # PHOTON pipeline
        string = ""

        # Pipeline Stack
        if isinstance(pipe_element, Stack):
            if parent == "":
                string = "[" + pipe_element.name + "]:\n" + "Layout:\n"
            else:
                string = "["+parent[1:] + "." + pipe_element.name + "]:\n" + "Layout:\n"

            # Layout
            for pelement in list(pipe_element.elements):
                string = string + "[" + pelement.name + "]|\n"
            string = string +"\n"
            for pelement in list(pipe_element.elements):
                string = string + "\n" + self.recursive_element(pelement, parent=parent + "." + pipe_element.name)


        # Pipeline Switch
        elif isinstance(pipe_element, Switch):
            if parent == "":
                string = "[" + pipe_element.name + "]:\n" + "Layout:\n"
            else:
                string = "[" + parent[1:] + "." + pipe_element.name + "]:\n" + "Layout:\n"

            # Layout
            for pelement in pipe_element.elements:
                string = string + "[" + pelement.name + "]\n"
            string = string + "\n"

            # relate
            string = string + "\n" + "Relate:\n"
            old_element = ""
            for pelement in pipe_element.elements:
                if old_element:
                    string = string + "[" + old_element + "]" + "<:-:>" + "[" + pelement.name + ']\n'
                old_element = pelement.name
                string = string + "\n"

            for pelement in pipe_element.elements:
                string = string + "\n" + self.recursive_element(pelement, parent=parent + "." + pipe_element.name)


        # Pipeline Branch
        elif isinstance(pipe_element, Branch):
            if parent == "":
                string = "[" + pipe_element.name + "]:\n" + "Layout:\n"
            else:
                string = "[" + parent[1:]+"."+pipe_element.name + "]:\n" + "Layout:\n"

            # Layout
            for pelement in pipe_element.elements:
                string = string + "[" + pelement.name + "]"
            string = string + "\n" + "Relate:\n"
            # Relate
            old_element = ""
            for pelement in pipe_element.elements:
                if old_element:
                    string = string + "[" + old_element + "]" + "->" + "[" + pelement.name + ']\n'
                old_element = pelement.name
                string = string + "\n"

            for pelement in pipe_element.elements:
                string = string + "\n" + self.recursive_element(pelement, parent=parent + "." + pipe_element.name)

        elif isinstance(pipe_element, PipelineElement):
            if parent == "":
                string = "[" + pipe_element.name + "]:\n" + "Define:\n"
            else:
                string = "[" + parent[1:] + "." + pipe_element.name + "]:\n" + "Define:\n"
            hyperparameters = None
            kwargs = None
            if hasattr(pipe_element, "hyperparameters"):
                hyperparameters = pipe_element.hyperparameters
                for name, parameter in pipe_element.hyperparameters.items():
                    string += "{}: {}\n".format(name.split('__')[-1], self.format_hyperparameter(parameter))
            if hasattr(pipe_element, "kwargs"):
                kwargs = pipe_element.kwargs
                for name, parameter in pipe_element.kwargs.items():
                    string += "{}: {}\n".format(name.split('__')[-1], self.format_hyperparameter(parameter))
            if not kwargs and not hyperparameters:
                string += "default\n"

        return string


class PhotonModelPersistor:

    @staticmethod
    def save_elements(elements, folder):
        if not os.path.exists(folder):
            os.makedirs(folder)
        element_identifier = list()

        for i, element in enumerate(elements):
            if hasattr(element, 'disabled'):
                if element.disabled:
                    continue

            # Switch in Switch not covered!
            if isinstance(element, Switch):
                element = element.base_element

            if isinstance(element, (Stack, Branch, Preprocessing)):
                filename = '_' + str(i) + '_' + element.name
                new_folder = os.path.join(folder, filename)
                element_identifier.append({'element_name': element.name, 'filename': filename})
                elements = element.elements
                PhotonModelPersistor.save_elements(elements=elements, folder=new_folder)
                element.elements = []
                joblib.dump(element, os.path.join(folder, filename) + '.pkl', compress=1)
                element_identifier[-1]['mode'] = 'PhotonBuildingBlock'
                element.elements = elements
            else:
                if not hasattr(element, 'base_element'):
                    base_element = element
                else:
                    base_element = element.base_element
                filename = '_' + str(i) + '_' + element.name
                element_identifier.append({'element_name': element.name, 'filename': filename})
                if hasattr(base_element, 'save'):
                    wrapper_file = inspect.getfile(base_element.__class__)
                    base_element.save(os.path.join(folder, filename))
                    element_identifier[-1]['mode'] = 'custom'
                    element_identifier[-1]['wrapper_script'] = os.path.basename(wrapper_file)
                    element_identifier[-1]['test_disabled'] = element.test_disabled
                    element_identifier[-1]['disabled'] = element.disabled
                    element_identifier[-1]['hyperparameters'] = element.hyperparameters
                    # (class_name != element_name) - possibility
                    element_identifier[-1]['class_name'] = type(base_element).__name__
                    shutil.copy(wrapper_file, os.path.join(folder, os.path.basename(wrapper_file)))
                else:
                    try:
                        joblib.dump(element, os.path.join(folder, filename) + '.pkl', compress=1)
                        element_identifier[-1]['mode'] = 'pickle'
                    except:
                        raise NotImplementedError("Custom pipeline element must implement .save() method or "
                                                  "allow pickle.")

        # save pipeline blueprint to make loading of pipeline easier
        with open(os.path.join(folder, '_optimum_pipe_blueprint.pkl'), 'wb') as f:
            pickle.dump(element_identifier, f)

    @staticmethod
    def save_optimum_pipe(optimum_pipe, zip_file, password=None):
        """
        Save optimal pipeline only. Complete hyperpipe will no not be saved.

        Parameters
        ----------
        * 'file' [str]:
            File path as string specifying file to save pipeline to
        * 'password' [str]:
            Password used to encrypt the pipeline file

        """
        folder = os.path.splitext(zip_file)[0]
        zip_file = folder + '.photon'

        if os.path.exists(folder):
            logger.warn('The file you specified already exists as a folder.')
        else:
            os.makedirs(folder)

        # only save elements without name. Structure of optimum_pipe.elements: [('name', element),...]
        PhotonModelPersistor.save_elements([val[1] for val in optimum_pipe.elements], folder)

        # write meta infos from pipeline
        with open(os.path.join(folder, '_optimum_pipe_meta.pkl'), 'wb') as f:
            meta_infos = {'photon_version': __version__}
            pickle.dump(meta_infos, f)

        # get all files
        files = list()
        for root, directories, filenames in os.walk(folder):
            for filename in filenames:
                files.append(os.path.join(root, filename))

        if password is not None:
            import pyminizip
            pyminizip.compress(files, zip_file, password)
        else:
            with zipfile.ZipFile(zip_file, 'w') as myzip:
                root_len = len(os.path.dirname(zip_file)) + 1
                for f in files:
                    # in order to work even with subdirectories, we need to substract the dirname from our file
                    # this is why I'm saving the root_len first
                    myzip.write(f, f[root_len:])
                    os.remove(f)
        shutil.rmtree(folder)

    @staticmethod
    def load_elements(folder):
        with open(os.path.join(folder, '_optimum_pipe_blueprint.pkl'), 'rb') as f:
            setup_info = pickle.load(f)
            element_list = list()
            for element_info in setup_info:
                if element_info['mode'] == 'PhotonBuildingBlock':
                    photon_building_block = joblib.load(os.path.join(folder, element_info['filename'] + '.pkl'))
                    base_elements = PhotonModelPersistor.load_elements(os.path.join(folder, element_info['filename']))
                    for _, element in base_elements:
                        photon_building_block += element
                    element_list.append((element_info['element_name'], photon_building_block))
                elif element_info['mode'] == 'custom':
                    if 'class_name' in element_info: # (class_name != element_name) - possibility
                        spec = importlib.util.spec_from_file_location(element_info['class_name'],
                                                                          os.path.join(folder,
                                                                          element_info['wrapper_script']))
                        imported_module = importlib.util.module_from_spec(spec)
                        spec.loader.exec_module(imported_module)
                        base_element = getattr(imported_module, element_info['class_name'])
                    else:
                        # backward compatibility
                        try:
                            spec = importlib.util.spec_from_file_location(element_info['element_name'],
                                                                          os.path.join(folder,
                                                                          element_info['wrapper_script']))
                            imported_module = importlib.util.module_from_spec(spec)
                            spec.loader.exec_module(imported_module)
                            base_element = getattr(imported_module, element_info['element_name'])
                        except:
                            msg = "Outdated version: The imported module was created by an older photon version. " \
                                  "Please retrain your model with a newer version."
                            logger.error(msg)
                            raise RuntimeError(msg)
                    custom_element = PipelineElement(name=element_info['element_name'], base_element=base_element(),
                                                     hyperparameters=element_info['hyperparameters'],
                                                     test_disabled=element_info['test_disabled'],
                                                     disabled=element_info['disabled'])
                    custom_element.base_element.load(os.path.join(folder, element_info['filename']))
                    element_list.append((element_info['element_name'], custom_element))
                else:
                    loaded_pipeline_element = joblib.load(os.path.join(folder, element_info['filename'] + '.pkl'))
                    element_list.append((element_info['element_name'], loaded_pipeline_element))
        return element_list

    @staticmethod
    def load_optimum_pipe(file, password=None):
        """
        Load optimal pipeline.


        Parameters
        ----------
        * `file` [str]:
            File path specifying .photon file to load optimal pipeline from

        Returns
        -------
        sklearn Pipeline with all trained photon_pipelines
        """
        if file.endswith('.photon'):
            folder = os.path.dirname(file)
            zf = zipfile.ZipFile(file)
            zf.extractall(folder, pwd=password)
        else:
            raise FileNotFoundError('Specify .photon file that holds PHOTON optimum pipe.')

        load_folder = os.path.join(folder, 'photon_best_model')
        meta_infos = {}
        try:
            with open(os.path.join(load_folder, '_optimum_pipe_meta.pkl'), 'rb') as f:
                meta_infos = pickle.load(f)
        except:
            print("Could not load meta information for optimum pipe")

        element_list = PhotonModelPersistor.load_elements(folder=load_folder)

        # delete unpacked folder to clean up
        # ToDo: Don't unpack at all, but use PHOTON file directly
        from shutil import rmtree
        rmtree(os.path.join(folder, 'photon_best_model'), ignore_errors=True)

        photon_pipe = PhotonPipeline(element_list)
        photon_pipe._meta_information = meta_infos
        return photon_pipe
