# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['quickparse']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'quickparse',
    'version': '0.9.0',
    'description': 'Simple command line argument parser for Python',
    'long_description': '# QuickParse\nSimple command line argument parser for Python\n\n## Example\n\n`list_things.py`:\n```python\nfrom quickparse import QuickParse\n\ndef list_things(a_list, quickparse):\n    print(\', \'.join(map(str, a_list[:quickparse.numeric])))\n\ncommands_config = {\n    \'ls\': list_things,\n    \'\': lambda: print("Command is missing, use \'ls\'"),\n}\n\nmylist = list(range(1, 12))\n\nQuickParse(commands_config).execute(mylist)\n```\n\nRun it:\n```sh\n$ python list_things.py ls -5\n1, 2, 3, 4, 5\n```\n\nThe way it works:\n- `commands_config` tells QuickParse to look for `ls` as a command and call `list_things` on it - when no commands show help\n- QuickParse parses arguments as normal while `ls` is privileged as a command\n- QuickParse finds `-5` so it adds as `quickparse.numeric = 5` (`quickparse` being the `QuickParse` instance that otherwise would come as `quickparse = QuickParse(commands_config)`)\n- QuickParse sees `list_things` being associated to `ls`, so `quickparse.execute(mylist)` calls it, passing on the arguments of `execute(..)` - one positional argument in this case\n- since `list_things` expects a named argument `quickparse`, QuickParse makes sure it passes on the reference to its own instance of `quickparse`\n\n## GNU Argument Syntax implementation with extensions\nGNU Argument Syntax: https://www.gnu.org/software/libc/manual/html_node/Argument-Syntax.html\n\n### Extensions\n#### Numeric \'-\' values\n```bash\n$ my_cmd -12\n```\n#### Numeric \'+\' values\n```bash\n$ my_cmd +12\n```\n#### Long \'-\' options - only with explicit config\n```bash\n$ my_cmd -list\n```\nBy default it becomes `-l -i -s -t`, but adding `QuickParse(options_config = [ (\'-list\', ) ])` will stop unpacking.\n#### Long \'+\' options by default\n```bash\n$ my_cmd +list\n```\n#### Equivalent options - using options_config\n```bash\n$ my_cmd -l\n```\nis equivalent to\n```bash\n$ my_cmd --list\n```\nif adding `QuickParse(options_config = [ (\'-l\', \'--list\') ])`\n#### Command-subcommand hierarchy and function bindings - using commands_config\nDefining a random sample from `git` looks like this:\n```python\ncommands_config = {\n    \'\': do_show_help,\n    \'commit\': do_commit,\n    \'log\': do_log,\n    \'stash\': {\n        \'\': do_stash,\n        \'list\': do_stash_list,\n    }\n}\n\noptions_config = [\n    (\'-a\', \'--all\'),\n]\n\nQuickParse(commands_config, options_config).execute()\n```\nCommands are called according to commands_config.  \nThat is `$ git log -3` calls `do_log`  \n`do_log` may look like this:\n```python\ndef do_log(quickparse):\n    print(get_log_entries()[:quickparse.numeric])\n```\nIf there is a named argument in `do_log`\'s signature called `quickparse`, the instance coming from `QuickParse(commands_config, options_config)` is passed down holding all the results of parsing.  \nParsing happens by using the defaults and applying what `options_config` adds to it.\n\n## Argument Formats\n| &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Argument&nbsp;Format&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; | &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Example&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; | Remarks |\n| --- | --- | --- |\n| `-<number>` | `$ my_cmd -12` | (default) |\n| `+<number>` | `$ my_cmd +12` | (default) |\n| `-<single_letter>` | `$ my_cmd -x` | (default) |\n| `+<single_letter>` | `$ my_cmd +x` | (default) |\n| `-<single_letter><value>` | `$ my_cmd -nFoo` | unpacking is the default: -n -F -o<br>`options_config` needs a type entry saying it expects a value (other than bool) |\n| `+<single_letter><value>` | `$ my_cmd +nFoo` | unpacking is the default: +n +F +o<br>`options_config` needs a type entry saying it expects a value (other than bool) |\n| `-<single_letter>=<value>` | `$ my_cmd -n=Foo` | (default) |\n| `+<single_letter>=<value>` | `$ my_cmd +n=Foo` | (default) |\n| `-<single_letter> <value>` | `$ my_cmd -n Foo` | `options_config` needs a type entry saying it expects a value (other than bool) |\n| `+<single_letter> <value>` | `$ my_cmd +n Foo` | `options_config` needs a type entry saying it expects a value (other than bool) |\n| `-<letters>` | `$ my_cmd -abc` | unpacking is the default: -a -b -c<br>if in `options_config` it\'s taken as `-abc` |\n| `+<letters>` | `$ my_cmd +abc` | unpacking is the default: +a +b +c<br>if in `options_config` it\'s taken as `+abc` |\n| `-<letters>=<value>` | `$ my_cmd -name=Foo` | (default) |\n| `+<letters>=<value>` | `$ my_cmd +name=Foo` | (default) |\n| `--<letters>` | `$ my_cmd --list` | (default) |\n| `--<letters>=<value>` | `$ my_cmd --message=Bar` | (default) |\n| `--<letters> <value>` | `$ my_cmd --message Bar` | `options_config` needs a type entry saying it expects a value (other than bool) |\n| `--` | `$ my_cmd -- --param-anyway` | parameters delimiter<br>(default) |\n\n`<letters>` means [a-zA-Z] and \'-\'s not in the first place\n\n### An argument like \'-a*\' gets unpacked if...\n- \'-a\' is not defined to expect a value\n- the \'*\' part has only letters, not \'-\' or \'=\'\n\n### How to change the interpretation of `-swing`\nIt can mean (default):  \n`-s -w -i -n -g`  \nor  \n`-s wing` / `-s=wing`  \nTo acheve the latter make the parser aware that \'-s\' expects a `str` value:\n```python\noptions_config = [\n    (\'-s\', str),\n]\n```\n\n### Make the parser aware that an option expects a value after a space\nAdd type explicitly in `options_config`.  \nFor just getting as it is add `str`.\n\n### How to define option types\nUse build-in types like `int` or `float`, or create a callable that raises exceptions.  \nUsing `bool` is a special case: parser will not expect a value but explicitly adds an error if one provided.\n\n### How to add empty value to an option\n`-option=`\nSome commands support \'-\' as empty value: `curl -C - -O http://domanin.com/`  \nIn this case \'-\' couldn\'t be provided as a literal so the syntax with \'=\' is supported here.\n\n## Error handling\nIf the parser parameters \'commands_config\' or \'options_config\' are not valid, ValueError is rased from the underlying AssertionError.  \nIf the arguments are not compliant with the config (e.g. no value provided for an option that requires one) then no exceptions are raised but an `errors` list is populated on the `QuickParse` object.\n\n## How to define options\n`options_test.py`:\n```python\nfrom quickparse import QuickParse\n\noptions_config = [\n    (\'-u\', \'--utc\', \'--universal\'),\n    (\'-l\', \'--long\'),\n    (\'-n\', \'--name\', str),\n]\n\nparsed = QuickParse(options_config=options_config)\n\nprint(parsed.options)\n```\n\nRun it:\n```\n$ python options_test.py\n{}\n\n$ python options_test.py -u\n{\'-u\': True, \'--utc\': True, \'--universal\': True}\n\n$ python options_test.py -ul\n{\'-u\': True, \'--utc\': True, \'--universal\': True, \'-l\': True, \'--long\': True}\n\n$ python options_test.py -uln\n{\'-uln\': True}\n\n$ python options_test.py -ul -nthe_name\n{\'-u\': True, \'--utc\': True, \'--universal\': True, \'-l\': True, \'--long\': True, \'-n\': \'the_name\', \'--name\': \'the_name\'}\n\n$ python options_test.py -ul -n the_name\n{\'-u\': True, \'--utc\': True, \'--universal\': True, \'-l\': True, \'--long\': True, \'-n\': \'the_name\', \'--name\': \'the_name\'}\n\n$ python options_test.py -ul -n=the_name\n{\'-u\': True, \'--utc\': True, \'--universal\': True, \'-l\': True, \'--long\': True, \'-n\': \'the_name\', \'--name\': \'the_name\'}\n\n$ python options_test.py -ul --name the_name\n{\'-u\': True, \'--utc\': True, \'--universal\': True, \'-l\': True, \'--long\': True, \'--name\': \'the_name\', \'-n\': \'the_name\'}\n\n$ python options_test.py -ul --name=the_name\n{\'-u\': True, \'--utc\': True, \'--universal\': True, \'-l\': True, \'--long\': True, \'--name\': \'the_name\', \'-n\': \'the_name\'}\n```\n`-uln` stopped the parser from unpacking because `-n` expected an input value\n\n## Test your command line arguments\n`quickparse_test_args.py` (committed in the repo):\n```python\nfrom pprint import pformat\n\nfrom quickparse import QuickParse\n\n\ndef do_show_help():\n    print("Executing \'do_show_help\'...")\n\ndef do_commit():\n    print("Executing \'do_commit\'...")\n\ndef do_log(quickparse):\n    print("Executing \'do_log\'...")\n\ndef do_stash():\n    print("Executing \'do_stash\'...")\n\ndef do_stash_list():\n    print("Executing \'do_stash_list\'...")\n\ncommands_config = {\n    \'\': do_show_help,\n    \'commit\': do_commit,\n    \'log\': do_log,\n    \'stash\': {\n        \'\': do_stash,\n        \'list\': do_stash_list,\n    }\n}\n\noptions_config = [\n    (\'-m\', \'--message\', str),\n    (\'-p\', \'--patch\'),\n]\n\n\nparsed = QuickParse(commands_config, options_config)\n\n\nprint(f\'Commands:\\n{pformat(parsed.commands)}\')\nprint(f\'Parameters:\\n{pformat(parsed.parameters)}\')\nprint(f\'Options:\\n{pformat(parsed.options)}\')\nprint(f\'\\\'-\\\' numeric argument:\\n{pformat(parsed.numeric)}\')\nprint(f\'\\\'+\\\' numeric argument:\\n{pformat(parsed.plusnumeric)}\')\nprint(f\'Functions to call:\\n{pformat(parsed.to_execute)}\')\n\nparsed.execute()\n```\n\n## Validation\nWell, I still need to elaborate the docs on this but here is a quick example snippet.  \n```python\nquickparse.validate({\n    \'parameters\': { \'mincount\': 1, },\n    \'options\': {\n        \'mandatory\': \'--branch\',\n        \'optional\': \'--stage\',\n    },\n    \'numeric\': { \'maxcount\': 0 },\n    \'plusnumeric\': { \'maxcount\': 0 },\n})\nassert \'parameters.mincount\' not in quickparse.errors, f\'Add a target\'\nassert not quickparse.has_errors, \'\\n\'.join(quickparse.error_messages)\n```\n',
    'author': 'silkyanteater',
    'author_email': 'cyclopesrufus@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/silkyanteater/quickparse',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.5,<4.0',
}


setup(**setup_kwargs)
