import abc
import time
from datetime import datetime
from enum import Enum
from typing import Any, Dict, List, Optional, Union

import backoff
import httpx
from pydantic import BaseModel, Field, parse_obj_as

from servo.configuration import Optimizer
from servo.types import Control, Description, Duration, Measurement, Numeric


USER_AGENT = "github.com/opsani/servox"


class Command(str, Enum):
    DESCRIBE = "DESCRIBE"
    MEASURE = "MEASURE"
    ADJUST = "ADJUST"
    SLEEP = "SLEEP"

    @property
    def response_event(self) -> str:
        if self == Command.DESCRIBE:
            return Event.DESCRIPTION
        elif self == Command.MEASURE:
            return Event.MEASUREMENT
        elif self == Command.ADJUST:
            return Event.ADJUSTMENT
        else:
            return None


class Event(str, Enum):
    HELLO = "HELLO"
    GOODBYE = "GOODBYE"
    DESCRIPTION = "DESCRIPTION"
    WHATS_NEXT = "WHATS_NEXT"
    ADJUSTMENT = "ADJUSTMENT"
    MEASUREMENT = "MEASUREMENT"


class Request(BaseModel):
    event: Union[Event, str] # TODO: Needs to be rethought -- used adhoc in some cases
    param: Optional[Dict[str, Any]]  # TODO: Switch to a union of supported types

    class Config:
        json_encoders = {
            Event: lambda v: str(v),
        }


class Status(BaseModel):
    status: str
    message: Optional[str]


class SleepResponse(BaseModel):
    pass


# SleepResponse '{"cmd": "SLEEP", "param": {"duration": 60, "data": {"reason": "no active optimization pipeline"}}}'

# Instructions from servo on what to measure
class MeasureParams(BaseModel):
    metrics: List[str]
    control: Control


class CommandResponse(BaseModel):
    command: Command = Field(alias="cmd",)
    param: Optional[
        Union[MeasureParams, Dict[str, Any]]
    ]  # TODO: Switch to a union of supported types

    class Config:
        json_encoders = {
            Command: lambda v: str(v),
        }


class StatusMessage(BaseModel):
    status: str
    message: Optional[str]


class Mixin:
    @property
    def api_headers(self) -> Dict[str, str]:
        return {
            "Authorization": f"Bearer {self.optimizer.token}",
            "User-Agent": USER_AGENT,
            "Content-Type": "application/json",
        }

    def api_client(self) -> httpx.AsyncClient:
        """Yields an httpx.Client instance configured to talk to Opsani API"""
        headers = {
            "Authorization": f"Bearer {self.optimizer.token}",
            "User-Agent": USER_AGENT,
            "Content-Type": "application/json",
        }
        return httpx.AsyncClient(base_url=self.optimizer.api_url, headers=self.api_headers)
    
    def api_client_sync(self) -> httpx.Client:
        """Yields an httpx.Client instance configured to talk to Opsani API"""
        return httpx.Client(base_url=self.optimizer.api_url, headers=self.api_headers)
    
    def progress_request(self,
        operation: str, 
        progress: Numeric, 
        started_at: datetime,
        message: Optional[str],
        *,
        connector: Optional[str] = None,
        event_context: Optional['EventContext'] = None,
        time_remaining: Optional[Union[Numeric, Duration]] = None,
        logs: Optional[List[str]] = None,
    ) -> None:
        def set_if(d: Dict, k: str, v: Any):
            if v is not None: d[k] = v
        
        # Normalize progress to positive percentage
        if progress < 1.0:
            progress = progress * 100
        
        # Calculate runtime
        runtime = Duration(datetime.now() - started_at)

        # Produce human readable and remaining time in seconds values (if given)
        if time_remaining:
            if isinstance(time_remaining, (int, float)):
                time_remaining_in_seconds = time_remaining
                time_remaining = Duration(time_remaining_in_seconds)
            elif isinstance(time_remaining, timedelta):
                time_remaining_in_seconds = time_remaining.total_seconds()
            else:
                raise ValueError(f"Unknown value of type '{time_remaining.__class__.__name__}' for parameter 'time_remaining'")
        else:
            time_remaining_in_seconds = None

        params = dict(
            connector=self.name, 
            operation=operation,
            progress=progress,
            runtime=str(runtime), 
            runtime_in_seconds=runtime.total_seconds()
        )
        set_if(params, 'connector', connector)
        set_if(params, 'event', str(event_context))
        set_if(params, 'time_remaining', str(time_remaining) if time_remaining else None)
        set_if(params, 'time_remaining_in_seconds', str(time_remaining_in_seconds) if time_remaining_in_seconds else None)
        set_if(params, 'message', message)
        set_if(params, 'logs', logs)
        
        return (operation, params)

    
    # NOTE: Opsani API primitive
    @backoff.on_exception(backoff.expo, (httpx.HTTPError), max_time=180, max_tries=12)
    async def _post_event(self, event: Event, param) -> Union[CommandResponse, Status]:
        event_request = Request(event=event, param=param)
        self.logger.trace(event_request)
        async with self.api_client() as client:
            try:
                response = await client.post("servo", data=event_request.json())
                response.raise_for_status()
            except httpx.HTTPError as error:
                self.logger.exception(
                    f"HTTP error encountered while posting {event.value} event"
                )
                self.logger.trace(pformat(event_request))
                raise error

        return parse_obj_as(Union[CommandResponse, Status], response.json())
    
    def _post_event_sync(self, event: Event, param) -> Union[CommandResponse, Status]:
        event_request = Request(event=event, param=param)
        with self.servo.api_client_sync() as client:
            try:
                response = client.post("servo", data=event_request.json())
                response.raise_for_status()
            except httpx.HTTPError as error:
                self.logger.exception(
                    f"HTTP error encountered while posting {event.value} event"
                )
                self.logger.trace(pformat(event_request))
                raise error

        return parse_obj_as(Union[CommandResponse, Status], response.json())
