"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const ec2 = require("aws-cdk-lib/aws-ec2");
const cdk = require("aws-cdk-lib");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain'
    };
    return new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
}
test('check domain names', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain",
        UserPoolId: {
            Ref: "testlambdaelasticsearchstackCognitoUserPool05D1387E"
        }
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        DomainName: "test-domain",
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.elasticsearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(construct.elasticsearchRole).toBeDefined();
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        domainName: 'test-domain'
    };
    try {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check properties with no CW Alarms ', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.elasticsearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeUndefined();
    expect(construct.elasticsearchRole).toBeDefined();
});
test('check lambda function custom environment variable', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        domainEndpointEnvironmentVariableName: 'CUSTOM_DOMAIN_ENDPOINT'
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_DOMAIN_ENDPOINT: {
                    'Fn::GetAtt': [
                        'testlambdaelasticsearchstackElasticsearchDomain2DE7011B',
                        'DomainEndpoint'
                    ]
                }
            }
        }
    });
});
test('check override cognito domain name with provided cognito domain name', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        cognitoDomainName: 'test-cognito-domain'
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: 'test-cognito-domain'
    });
});
test("Test minimal deployment that deploys a VPC in 2 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {});
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaelasticsearchkibanastackReplaceDefaultSecurityGroupsecuritygroup4C50002B",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                }
            ],
        },
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test minimal deployment that deploys a VPC in 3 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaelasticsearchkibanastackReplaceDefaultSecurityGroupsecuritygroup4C50002B",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D",
                },
            ],
        },
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test ES cluster deploy to 1 AZ when user set zoneAwarenessEnabled to false", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const esDomainProps = {
        elasticsearchClusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            zoneAwarenessEnabled: false,
            instanceCount: 3
        }
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        esDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 1
        }
    });
    expect(stack).toHaveResource("AWS::Elasticsearch::Domain", {
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: false,
        }
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                }
            ]
        }
    });
});
test("Test ES cluster deploy to 2 AZ when user set availabilityZoneCount to 2", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const esDomainProps = {
        elasticsearchClusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            instanceCount: 2,
            zoneAwarenessEnabled: true,
            zoneAwarenessConfig: {
                availabilityZoneCount: 2
            }
        }
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        esDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 2
        }
    });
    expect(stack).toHaveResource("AWS::Elasticsearch::Domain", {
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 2,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 2,
            },
            ZoneAwarenessEnabled: true,
        }
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
});
test('Test minimal deployment with an existing isolated VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.getTestVpc(stack, false, {
        vpcName: "existing-isolated-vpc-test"
    });
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        existingVpc: vpc
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "existing-isolated-vpc-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with an existing private VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = new ec2.Vpc(stack, 'existing-private-vpc-test', {
        natGateways: 1,
        subnetConfiguration: [
            {
                cidrMask: 24,
                name: 'application',
                subnetType: ec2.SubnetType.PRIVATE_WITH_NAT,
            },
            {
                cidrMask: 24,
                name: "public",
                subnetType: ec2.SubnetType.PUBLIC,
            }
        ]
    });
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        existingVpc: vpc
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "Default/existing-private-vpc-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "existingprivatevpctestapplicationSubnet1Subnet1F7744F0"
                },
                {
                    Ref: "existingprivatevpctestapplicationSubnet2SubnetF7B713AD"
                },
                {
                    Ref: "existingprivatevpctestapplicationSubnet3SubnetA519E038"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with VPC construct props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        deployVpc: true,
        vpcProps: {
            vpcName: "vpc-props-test"
        }
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "vpc-props-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test error for vpcProps and undefined deployVpc prop', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            domainName: "test",
            vpcProps: {
                vpcName: "existing-vpc-test"
            }
        });
    };
    expect(app).toThrowError("Error - deployVpc must be true when defining vpcProps");
});
test('Test error for Lambda function VPC props', () => {
    const stack = new cdk.Stack();
    const vpc = defaults.getTestVpc(stack);
    const app = () => {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
            lambdaFunctionProps: defaults.consolidateProps(getDefaultTestLambdaProps(), { vpc }),
            domainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not Lambda function props");
});
test('Test error for Elasticsearch domain VPC props', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            esDomainProps: {
                vpcOptions: {
                    subnetIds: ['fake-ids'],
                    securityGroupIds: ['fake-sgs']
                }
            },
            domainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not Elasticsearch props");
});
function getDefaultTestLambdaProps() {
    return {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
    };
}
//# sourceMappingURL=data:application/json;base64,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