import uuid

from django.contrib.auth.models import AbstractBaseUser, PermissionsMixin, UserManager
from django.utils.translation import gettext_lazy as _
from django.core.mail import send_mail
from django.core import validators
from django.utils import timezone
from django.db import models


class UserProxy(models.Model):
    """
    User proxy model, represents a local instance of
    a user object from another database.
    """
    user_id = models.UUIDField()

    def serialize(self):
        if self.pk is None:
            return None

        return {
            "id": self.pk,
            "user_id": self.user_id
        }

    def __str__(self):
        return str(self.user_id)


class AbstractUser(AbstractBaseUser, PermissionsMixin):
    """
    An abstract base class implementing a fully featured User model with
    admin-compliant permissions.

    Username and password are required. Other fields are optional.
    """

    username = models.CharField(
        _("username"),
        max_length=255,
        unique=True,
        validators=[
            validators.RegexValidator(
                "^[\\w.@+-]+$", "Enter a valid username.",
                "invalid"
            )
        ],
        error_messages={
            "unique": "A user with that username already exists."
        },
    )
    first_name = models.CharField(
        _('first name'),
        max_length=150,
        blank=True
    )
    last_name = models.CharField(
        _('last name'),
        max_length=150,
        blank=True
    )
    email = models.EmailField(
        _('email address'),
        unique=True,
        error_messages={
            "unique": "A user with that email already exists."
        },
    )
    is_staff = models.BooleanField(
        _('staff status'),
        default=False,
        help_text=_('Designates whether the user can log into this admin site.'),
    )
    is_active = models.BooleanField(
        _('active'),
        default=True,
        help_text=_(
            'Designates whether this user should be treated as active. '
            'Unselect this instead of deleting accounts.'
        ),
    )
    date_joined = models.DateTimeField(
        _('date joined'),
        default=timezone.now
    )

    objects = UserManager()

    EMAIL_FIELD = 'email'
    USERNAME_FIELD = 'username'
    REQUIRED_FIELDS = ['email']

    class Meta:
        verbose_name = _('user')
        verbose_name_plural = _('users')
        abstract = True

    def __str__(self):
        return "{} {}".format(self.get_full_name(), self.email)

    def clean(self):
        super().clean()
        self.email = self.__class__.objects.normalize_email(self.email)

    def get_full_name(self):
        """
        Return the first_name plus the last_name, with a space in between.
        """
        full_name = '%s %s' % (self.first_name, self.last_name)
        return full_name.strip()

    def get_short_name(self):
        """Return the short name for the user."""
        return self.first_name

    def email_user(self, subject, message, from_email=None, **kwargs):
        """Send an email to this user."""
        send_mail(subject, message, from_email, [self.email], **kwargs)
