"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchemaTransformer = void 0;
const fs = require("fs");
const path_1 = require("path");
const graphql_auth_transformer_1 = require("graphql-auth-transformer");
const graphql_connection_transformer_1 = require("graphql-connection-transformer");
const graphql_dynamodb_transformer_1 = require("graphql-dynamodb-transformer");
const graphql_http_transformer_1 = require("graphql-http-transformer");
const graphql_key_transformer_1 = require("graphql-key-transformer");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_ttl_transformer_1 = require("graphql-ttl-transformer");
const graphql_versioned_transformer_1 = require("graphql-versioned-transformer");
const cdk_transformer_1 = require("./cdk-transformer");
// Import this way because FunctionTransformer.d.ts types were throwing an eror. And we didn't write this package so hope for the best :P
// eslint-disable-next-line
const { FunctionTransformer } = require('graphql-function-transformer');
class SchemaTransformer {
    constructor(props) {
        this.schemaPath = props.schemaPath || './schema.graphql';
        this.outputPath = props.outputPath || './appsync';
        this.isSyncEnabled = props.syncEnabled || false;
        this.outputs = {};
        this.resolvers = {};
        // TODO: Make this better?
        this.authTransformerConfig = {
            authConfig: {
                defaultAuthentication: {
                    authenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    userPoolConfig: {
                        userPoolId: '12345xyz',
                    },
                },
                additionalAuthenticationProviders: [
                    {
                        authenticationType: 'API_KEY',
                        apiKeyConfig: {
                            description: 'Testing',
                            apiKeyExpirationDays: 100,
                        },
                    },
                    {
                        authenticationType: 'AWS_IAM',
                    },
                ],
            },
        };
    }
    transform(preCdkTransformers = [], postCdkTransformers = []) {
        var _a, _b;
        const transformConfig = this.isSyncEnabled ? this.loadConfigSync() : {};
        // Note: This is not exact as we are omitting the @searchable transformer as well as some others.
        const transformer = new graphql_transformer_core_1.GraphQLTransform({
            transformConfig: transformConfig,
            transformers: [
                new graphql_dynamodb_transformer_1.DynamoDBModelTransformer(),
                new graphql_ttl_transformer_1.default(),
                new graphql_versioned_transformer_1.VersionedModelTransformer(),
                new FunctionTransformer(),
                new graphql_key_transformer_1.KeyTransformer(),
                new graphql_connection_transformer_1.ModelConnectionTransformer(),
                new graphql_auth_transformer_1.ModelAuthTransformer(this.authTransformerConfig),
                new graphql_http_transformer_1.HttpTransformer(),
                ...preCdkTransformers,
                new cdk_transformer_1.CdkTransformer(),
                ...postCdkTransformers,
            ],
        });
        const schema = fs.readFileSync(this.schemaPath);
        const cfdoc = transformer.transform(schema.toString());
        // TODO: Get Unauth Role and Auth Role policies for authorization stuff
        this.unauthRolePolicy = ((_a = cfdoc.rootStack.Resources) === null || _a === void 0 ? void 0 : _a.UnauthRolePolicy01) || undefined;
        this.writeSchema(cfdoc.schema);
        this.writeResolversToFile(cfdoc.resolvers);
        // Outputs shouldn't be null but default to empty map
        this.outputs = (_b = cfdoc.rootStack.Outputs) !== null && _b !== void 0 ? _b : {};
        return this.outputs;
    }
    /**
       *
       */
    getResolvers() {
        const statements = ['Query', 'Mutation', 'Subscription'];
        const resolversDirPath = path_1.normalize('./appsync/resolvers');
        if (fs.existsSync(resolversDirPath)) {
            const files = fs.readdirSync(resolversDirPath);
            files.forEach(file => {
                // Example: Mutation.createChannel.response
                let args = file.split('.');
                let typeName = args[0];
                let name = args[1];
                let templateType = args[2]; // request or response
                let filepath = path_1.normalize(`${resolversDirPath}/${file}`);
                if (statements.indexOf(typeName) >= 0 || (this.outputs.noneResolvers && this.outputs.noneResolvers[name])) {
                    if (!this.resolvers[name]) {
                        this.resolvers[name] = {
                            typeName: typeName,
                            fieldName: name,
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers[name].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers[name].responseMappingTemplate = filepath;
                    }
                }
                else { // This is a GSI
                    if (!this.resolvers.gsi) {
                        this.resolvers.gsi = {};
                    }
                    let mapName = `${typeName}${name}`;
                    if (!this.resolvers.gsi[mapName]) {
                        this.resolvers.gsi[mapName] = {
                            typeName: typeName,
                            fieldName: name,
                            tableName: name.charAt(0).toUpperCase() + name.slice(1),
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers.gsi[mapName].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers.gsi[mapName].responseMappingTemplate = filepath;
                    }
                }
            });
        }
        return this.resolvers;
    }
    /**
       * Writes the schema to the output directory for use with @aws-cdk/aws-appsync
       * @param schema
       */
    writeSchema(schema) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        fs.writeFileSync(`${this.outputPath}/schema.graphql`, schema);
    }
    /**
       * Writes all the resolvers to the output directory for loading into the datasources later
       * @param resolvers
       */
    writeResolversToFile(resolvers) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        const resolverFolderPath = path_1.normalize(this.outputPath + '/resolvers');
        if (fs.existsSync(resolverFolderPath)) {
            const files = fs.readdirSync(resolverFolderPath);
            files.forEach(file => fs.unlinkSync(resolverFolderPath + '/' + file));
            fs.rmdirSync(resolverFolderPath);
        }
        if (!fs.existsSync(resolverFolderPath)) {
            fs.mkdirSync(resolverFolderPath);
        }
        Object.keys(resolvers).forEach((key) => {
            const resolver = resolvers[key];
            const fileName = key.replace('.vtl', '');
            const resolverFilePath = path_1.normalize(`${resolverFolderPath}/${fileName}`);
            fs.writeFileSync(resolverFilePath, resolver);
        });
    }
    /**
       * @returns {@link TransformConfig}
      */
    loadConfigSync(projectDir = 'resources') {
        // Initialize the config always with the latest version, other members are optional for now.
        let config = {
            Version: graphql_transformer_core_1.TRANSFORM_CURRENT_VERSION,
            ResolverConfig: {
                project: {
                    ConflictHandler: "OPTIMISTIC_CONCURRENCY" /* OPTIMISTIC */,
                    ConflictDetection: 'VERSION',
                },
            },
        };
        const configDir = path_1.join(__dirname, '..', '..', projectDir);
        try {
            const configPath = path_1.join(configDir, graphql_transformer_core_1.TRANSFORM_CONFIG_FILE_NAME);
            const configExists = fs.existsSync(configPath);
            if (configExists) {
                const configStr = fs.readFileSync(configPath);
                config = JSON.parse(configStr.toString());
            }
            return config;
        }
        catch (err) {
            return config;
        }
    }
}
exports.SchemaTransformer = SchemaTransformer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NoZW1hLXRyYW5zZm9ybWVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL3RyYW5zZm9ybWVyL3NjaGVtYS10cmFuc2Zvcm1lci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsK0JBQXVDO0FBQ3ZDLHVFQUE0RjtBQUM1RixtRkFBNEU7QUFDNUUsK0VBQXdFO0FBQ3hFLHVFQUEyRDtBQUMzRCxxRUFBeUQ7QUFDekQsdUVBQXVLO0FBQ3ZLLHFFQUFxRDtBQUNyRCxpRkFBMEU7QUFFMUUsdURBTTJCO0FBSzNCLHlJQUF5STtBQUN6SSwyQkFBMkI7QUFDM0IsTUFBTSxFQUFFLG1CQUFtQixFQUFFLEdBQUcsT0FBTyxDQUFDLDhCQUE4QixDQUFDLENBQUM7QUFzQ3hFLE1BQWEsaUJBQWlCO0lBWTVCLFlBQVksS0FBNkI7UUFDdkMsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLGtCQUFrQixDQUFDO1FBQ3pELElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxXQUFXLENBQUM7UUFDbEQsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQztRQUVoRCxJQUFJLENBQUMsT0FBTyxHQUFHLEVBQUUsQ0FBQztRQUNsQixJQUFJLENBQUMsU0FBUyxHQUFHLEVBQUUsQ0FBQztRQUVwQiwwQkFBMEI7UUFDMUIsSUFBSSxDQUFDLHFCQUFxQixHQUFHO1lBQzNCLFVBQVUsRUFBRTtnQkFDVixxQkFBcUIsRUFBRTtvQkFDckIsa0JBQWtCLEVBQUUsMkJBQTJCO29CQUMvQyxjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLFVBQVU7cUJBQ3ZCO2lCQUNGO2dCQUNELGlDQUFpQyxFQUFFO29CQUNqQzt3QkFDRSxrQkFBa0IsRUFBRSxTQUFTO3dCQUM3QixZQUFZLEVBQUU7NEJBQ1osV0FBVyxFQUFFLFNBQVM7NEJBQ3RCLG9CQUFvQixFQUFFLEdBQUc7eUJBQzFCO3FCQUNGO29CQUNEO3dCQUNFLGtCQUFrQixFQUFFLFNBQVM7cUJBQzlCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVNLFNBQVMsQ0FBQyxxQkFBcUMsRUFBRSxFQUFFLHNCQUFzQyxFQUFFOztRQUNoRyxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUV4RSxpR0FBaUc7UUFDakcsTUFBTSxXQUFXLEdBQUcsSUFBSSwyQ0FBZ0IsQ0FBQztZQUN2QyxlQUFlLEVBQUUsZUFBZTtZQUNoQyxZQUFZLEVBQUU7Z0JBQ1osSUFBSSx1REFBd0IsRUFBRTtnQkFDOUIsSUFBSSxpQ0FBYyxFQUFFO2dCQUNwQixJQUFJLHlEQUF5QixFQUFFO2dCQUMvQixJQUFJLG1CQUFtQixFQUFFO2dCQUN6QixJQUFJLHdDQUFjLEVBQUU7Z0JBQ3BCLElBQUksMkRBQTBCLEVBQUU7Z0JBQ2hDLElBQUksK0NBQW9CLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDO2dCQUNwRCxJQUFJLDBDQUFlLEVBQUU7Z0JBQ3JCLEdBQUcsa0JBQWtCO2dCQUNyQixJQUFJLGdDQUFjLEVBQUU7Z0JBQ3BCLEdBQUcsbUJBQW1CO2FBQ3ZCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDaEQsTUFBTSxLQUFLLEdBQUcsV0FBVyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUV2RCx1RUFBdUU7UUFDdkUsSUFBSSxDQUFDLGdCQUFnQixHQUFHLENBQUEsTUFBQSxLQUFLLENBQUMsU0FBUyxDQUFDLFNBQVMsMENBQUUsa0JBQThCLEtBQUksU0FBUyxDQUFDO1FBRS9GLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9CLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFM0MscURBQXFEO1FBQ3JELElBQUksQ0FBQyxPQUFPLFNBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLG1DQUFJLEVBQUUsQ0FBQztRQUU3QyxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUM7SUFDdEIsQ0FBQztJQUVEOztTQUVLO0lBQ0UsWUFBWTtRQUNqQixNQUFNLFVBQVUsR0FBRyxDQUFDLE9BQU8sRUFBRSxVQUFVLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDekQsTUFBTSxnQkFBZ0IsR0FBRyxnQkFBUyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDMUQsSUFBSSxFQUFFLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLEVBQUU7WUFDbkMsTUFBTSxLQUFLLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1lBQy9DLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUU7Z0JBQ25CLDJDQUEyQztnQkFDM0MsSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDM0IsSUFBSSxRQUFRLEdBQVcsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUMvQixJQUFJLElBQUksR0FBVyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQzNCLElBQUksWUFBWSxHQUFHLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLHNCQUFzQjtnQkFDbEQsSUFBSSxRQUFRLEdBQUcsZ0JBQVMsQ0FBQyxHQUFHLGdCQUFnQixJQUFJLElBQUksRUFBRSxDQUFDLENBQUM7Z0JBRXhELElBQUksVUFBVSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFO29CQUN6RyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsRUFBRTt3QkFDekIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRzs0QkFDckIsUUFBUSxFQUFFLFFBQVE7NEJBQ2xCLFNBQVMsRUFBRSxJQUFJO3lCQUNoQixDQUFDO3FCQUNIO29CQUVELElBQUksWUFBWSxLQUFLLEtBQUssRUFBRTt3QkFDMUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxzQkFBc0IsR0FBRyxRQUFRLENBQUM7cUJBQ3hEO3lCQUFNLElBQUksWUFBWSxLQUFLLEtBQUssRUFBRTt3QkFDakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyx1QkFBdUIsR0FBRyxRQUFRLENBQUM7cUJBQ3pEO2lCQUVGO3FCQUFNLEVBQUUsZ0JBQWdCO29CQUN2QixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLEVBQUU7d0JBQ3ZCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxHQUFHLEVBQUUsQ0FBQztxQkFDekI7b0JBRUQsSUFBSSxPQUFPLEdBQUcsR0FBRyxRQUFRLEdBQUcsSUFBSSxFQUFFLENBQUM7b0JBQ25DLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRTt3QkFDaEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUc7NEJBQzVCLFFBQVEsRUFBRSxRQUFROzRCQUNsQixTQUFTLEVBQUUsSUFBSTs0QkFDZixTQUFTLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQzt5QkFDeEQsQ0FBQztxQkFDSDtvQkFFRCxJQUFJLFlBQVksS0FBSyxLQUFLLEVBQUU7d0JBQzFCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDLHNCQUFzQixHQUFHLFFBQVEsQ0FBQztxQkFDL0Q7eUJBQU0sSUFBSSxZQUFZLEtBQUssS0FBSyxFQUFFO3dCQUNqQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQyx1QkFBdUIsR0FBRyxRQUFRLENBQUM7cUJBQ2hFO2lCQUNGO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQztJQUN4QixDQUFDO0lBRUQ7OztTQUdLO0lBQ0csV0FBVyxDQUFDLE1BQVc7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ25DLEVBQUUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQy9CO1FBRUQsRUFBRSxDQUFDLGFBQWEsQ0FBQyxHQUFHLElBQUksQ0FBQyxVQUFVLGlCQUFpQixFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFRDs7O1NBR0s7SUFDRyxvQkFBb0IsQ0FBQyxTQUFjO1FBQ3pDLElBQUksQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUNuQyxFQUFFLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUMvQjtRQUVELE1BQU0sa0JBQWtCLEdBQUcsZ0JBQVMsQ0FBQyxJQUFJLENBQUMsVUFBVSxHQUFHLFlBQVksQ0FBQyxDQUFDO1FBQ3JFLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO1lBQ3JDLE1BQU0sS0FBSyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsQ0FBQztZQUNqRCxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsR0FBRyxHQUFHLEdBQUcsSUFBSSxDQUFDLENBQUMsQ0FBQztZQUN0RSxFQUFFLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLENBQUM7U0FDbEM7UUFFRCxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO1lBQ3RDLEVBQUUsQ0FBQyxTQUFTLENBQUMsa0JBQWtCLENBQUMsQ0FBQztTQUNsQztRQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBUSxFQUFFLEVBQUU7WUFDMUMsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ2hDLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ3pDLE1BQU0sZ0JBQWdCLEdBQUcsZ0JBQVMsQ0FBQyxHQUFHLGtCQUFrQixJQUFJLFFBQVEsRUFBRSxDQUFDLENBQUM7WUFDeEUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxnQkFBZ0IsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMvQyxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7UUFFSTtJQUNJLGNBQWMsQ0FBQyxhQUFxQixXQUFXO1FBQ3JELDRGQUE0RjtRQUM1RixJQUFJLE1BQU0sR0FBb0I7WUFDNUIsT0FBTyxFQUFFLG9EQUF5QjtZQUNsQyxjQUFjLEVBQUU7Z0JBQ2QsT0FBTyxFQUFFO29CQUNQLGVBQWUsMkNBQWdDO29CQUMvQyxpQkFBaUIsRUFBRSxTQUFTO2lCQUM3QjthQUNGO1NBQ0YsQ0FBQztRQUVGLE1BQU0sU0FBUyxHQUFHLFdBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztRQUUxRCxJQUFJO1lBQ0YsTUFBTSxVQUFVLEdBQUcsV0FBSSxDQUFDLFNBQVMsRUFBRSxxREFBMEIsQ0FBQyxDQUFDO1lBQy9ELE1BQU0sWUFBWSxHQUFHLEVBQUUsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDL0MsSUFBSSxZQUFZLEVBQUU7Z0JBQ2hCLE1BQU0sU0FBUyxHQUFHLEVBQUUsQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUM7Z0JBQzlDLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO2FBQzNDO1lBRUQsT0FBTyxNQUF5QixDQUFDO1NBQ2xDO1FBQUMsT0FBTyxHQUFHLEVBQUU7WUFDWixPQUFPLE1BQU0sQ0FBQztTQUNmO0lBQ0gsQ0FBQztDQUNGO0FBL01ELDhDQStNQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCB7IG5vcm1hbGl6ZSwgam9pbiB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgTW9kZWxBdXRoVHJhbnNmb3JtZXIsIE1vZGVsQXV0aFRyYW5zZm9ybWVyQ29uZmlnIH0gZnJvbSAnZ3JhcGhxbC1hdXRoLXRyYW5zZm9ybWVyJztcbmltcG9ydCB7IE1vZGVsQ29ubmVjdGlvblRyYW5zZm9ybWVyIH0gZnJvbSAnZ3JhcGhxbC1jb25uZWN0aW9uLXRyYW5zZm9ybWVyJztcbmltcG9ydCB7IER5bmFtb0RCTW9kZWxUcmFuc2Zvcm1lciB9IGZyb20gJ2dyYXBocWwtZHluYW1vZGItdHJhbnNmb3JtZXInO1xuaW1wb3J0IHsgSHR0cFRyYW5zZm9ybWVyIH0gZnJvbSAnZ3JhcGhxbC1odHRwLXRyYW5zZm9ybWVyJztcbmltcG9ydCB7IEtleVRyYW5zZm9ybWVyIH0gZnJvbSAnZ3JhcGhxbC1rZXktdHJhbnNmb3JtZXInO1xuaW1wb3J0IHsgR3JhcGhRTFRyYW5zZm9ybSwgVHJhbnNmb3JtQ29uZmlnLCBUUkFOU0ZPUk1fQ1VSUkVOVF9WRVJTSU9OLCBUUkFOU0ZPUk1fQ09ORklHX0ZJTEVfTkFNRSwgQ29uZmxpY3RIYW5kbGVyVHlwZSwgSVRyYW5zZm9ybWVyIH0gZnJvbSAnZ3JhcGhxbC10cmFuc2Zvcm1lci1jb3JlJztcbmltcG9ydCBUdGxUcmFuc2Zvcm1lciBmcm9tICdncmFwaHFsLXR0bC10cmFuc2Zvcm1lcic7XG5pbXBvcnQgeyBWZXJzaW9uZWRNb2RlbFRyYW5zZm9ybWVyIH0gZnJvbSAnZ3JhcGhxbC12ZXJzaW9uZWQtdHJhbnNmb3JtZXInO1xuXG5pbXBvcnQge1xuICBDZGtUcmFuc2Zvcm1lcixcbiAgQ2RrVHJhbnNmb3JtZXJUYWJsZSxcbiAgQ2RrVHJhbnNmb3JtZXJSZXNvbHZlcixcbiAgQ2RrVHJhbnNmb3JtZXJGdW5jdGlvblJlc29sdmVyLFxuICBDZGtUcmFuc2Zvcm1lckh0dHBSZXNvbHZlcixcbn0gZnJvbSAnLi9jZGstdHJhbnNmb3JtZXInO1xuXG4vLyBSZWJ1aWx0IHRoaXMgZnJvbSBjbG91ZGZvcm0tdHlwZXMgYmVjYXVzZSBpdCBoYXMgdHlwZSBlcnJvcnNcbmltcG9ydCB7IFJlc291cmNlIH0gZnJvbSAnLi9yZXNvdXJjZSc7XG5cbi8vIEltcG9ydCB0aGlzIHdheSBiZWNhdXNlIEZ1bmN0aW9uVHJhbnNmb3JtZXIuZC50cyB0eXBlcyB3ZXJlIHRocm93aW5nIGFuIGVyb3IuIEFuZCB3ZSBkaWRuJ3Qgd3JpdGUgdGhpcyBwYWNrYWdlIHNvIGhvcGUgZm9yIHRoZSBiZXN0IDpQXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbmNvbnN0IHsgRnVuY3Rpb25UcmFuc2Zvcm1lciB9ID0gcmVxdWlyZSgnZ3JhcGhxbC1mdW5jdGlvbi10cmFuc2Zvcm1lcicpO1xuXG5leHBvcnQgaW50ZXJmYWNlIFNjaGVtYVRyYW5zZm9ybWVyUHJvcHMge1xuICAvKipcbiAgICogRmlsZSBwYXRoIHRvIHRoZSBncmFwaHFsIHNjaGVtYVxuICAgKiBAZGVmYXVsdCBzY2hlbWEuZ3JhcGhxbFxuICAgKi9cbiAgcmVhZG9ubHkgc2NoZW1hUGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogUGF0aCB3aGVyZSB0cmFuc2Zvcm1lZCBzY2hlbWEgYW5kIHJlc29sdmVycyB3aWxsIGJlIHBsYWNlZFxuICAgKiBAZGVmYXVsdCBhcHBzeW5jXG4gICAqL1xuICByZWFkb25seSBvdXRwdXRQYXRoPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTZXQgZGVsZXRpb24gcHJvdGVjdGlvbiBvbiBEeW5hbW9EQiB0YWJsZXNcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVsZXRpb25Qcm90ZWN0aW9uRW5hYmxlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gZW5hYmxlIERhdGFTdG9yZSBvciBub3RcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHN5bmNFbmFibGVkPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBTY2hlbWFUcmFuc2Zvcm1lck91dHB1dHMge1xuICByZWFkb25seSBjZGtUYWJsZXM/OiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclRhYmxlIH07XG4gIHJlYWRvbmx5IG5vbmVSZXNvbHZlcnM/OiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclJlc29sdmVyIH07XG4gIHJlYWRvbmx5IGZ1bmN0aW9uUmVzb2x2ZXJzPzogeyBbbmFtZTogc3RyaW5nXTogQ2RrVHJhbnNmb3JtZXJGdW5jdGlvblJlc29sdmVyW10gfTtcbiAgcmVhZG9ubHkgaHR0cFJlc29sdmVycz86IHsgW25hbWU6IHN0cmluZ106IENka1RyYW5zZm9ybWVySHR0cFJlc29sdmVyW10gfTtcbiAgcmVhZG9ubHkgcXVlcmllcz86IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuICByZWFkb25seSBtdXRhdGlvbnM/OiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclJlc29sdmVyIH07XG4gIHJlYWRvbmx5IHN1YnNjcmlwdGlvbnM/OiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclJlc29sdmVyIH07XG59XG5cbmV4cG9ydCBjbGFzcyBTY2hlbWFUcmFuc2Zvcm1lciB7XG4gIHB1YmxpYyByZWFkb25seSBzY2hlbWFQYXRoOiBzdHJpbmdcbiAgcHVibGljIHJlYWRvbmx5IG91dHB1dFBhdGg6IHN0cmluZ1xuICBwdWJsaWMgcmVhZG9ubHkgaXNTeW5jRW5hYmxlZDogYm9vbGVhblxuXG4gIHByaXZhdGUgcmVhZG9ubHkgYXV0aFRyYW5zZm9ybWVyQ29uZmlnOiBNb2RlbEF1dGhUcmFuc2Zvcm1lckNvbmZpZ1xuXG4gIG91dHB1dHM6IFNjaGVtYVRyYW5zZm9ybWVyT3V0cHV0c1xuICByZXNvbHZlcnM6IGFueVxuICBhdXRoUm9sZVBvbGljeTogUmVzb3VyY2UgfCB1bmRlZmluZWRcbiAgdW5hdXRoUm9sZVBvbGljeTogUmVzb3VyY2UgfCB1bmRlZmluZWRcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogU2NoZW1hVHJhbnNmb3JtZXJQcm9wcykge1xuICAgIHRoaXMuc2NoZW1hUGF0aCA9IHByb3BzLnNjaGVtYVBhdGggfHwgJy4vc2NoZW1hLmdyYXBocWwnO1xuICAgIHRoaXMub3V0cHV0UGF0aCA9IHByb3BzLm91dHB1dFBhdGggfHwgJy4vYXBwc3luYyc7XG4gICAgdGhpcy5pc1N5bmNFbmFibGVkID0gcHJvcHMuc3luY0VuYWJsZWQgfHwgZmFsc2U7XG5cbiAgICB0aGlzLm91dHB1dHMgPSB7fTtcbiAgICB0aGlzLnJlc29sdmVycyA9IHt9O1xuXG4gICAgLy8gVE9ETzogTWFrZSB0aGlzIGJldHRlcj9cbiAgICB0aGlzLmF1dGhUcmFuc2Zvcm1lckNvbmZpZyA9IHtcbiAgICAgIGF1dGhDb25maWc6IHtcbiAgICAgICAgZGVmYXVsdEF1dGhlbnRpY2F0aW9uOiB7XG4gICAgICAgICAgYXV0aGVudGljYXRpb25UeXBlOiAnQU1BWk9OX0NPR05JVE9fVVNFUl9QT09MUycsXG4gICAgICAgICAgdXNlclBvb2xDb25maWc6IHtcbiAgICAgICAgICAgIHVzZXJQb29sSWQ6ICcxMjM0NXh5eicsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgYWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgYXV0aGVudGljYXRpb25UeXBlOiAnQVBJX0tFWScsXG4gICAgICAgICAgICBhcGlLZXlDb25maWc6IHtcbiAgICAgICAgICAgICAgZGVzY3JpcHRpb246ICdUZXN0aW5nJyxcbiAgICAgICAgICAgICAgYXBpS2V5RXhwaXJhdGlvbkRheXM6IDEwMCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBhdXRoZW50aWNhdGlvblR5cGU6ICdBV1NfSUFNJyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbiAgcHVibGljIHRyYW5zZm9ybShwcmVDZGtUcmFuc2Zvcm1lcnM6IElUcmFuc2Zvcm1lcltdID0gW10sIHBvc3RDZGtUcmFuc2Zvcm1lcnM6IElUcmFuc2Zvcm1lcltdID0gW10pIHtcbiAgICBjb25zdCB0cmFuc2Zvcm1Db25maWcgPSB0aGlzLmlzU3luY0VuYWJsZWQgPyB0aGlzLmxvYWRDb25maWdTeW5jKCkgOiB7fTtcblxuICAgIC8vIE5vdGU6IFRoaXMgaXMgbm90IGV4YWN0IGFzIHdlIGFyZSBvbWl0dGluZyB0aGUgQHNlYXJjaGFibGUgdHJhbnNmb3JtZXIgYXMgd2VsbCBhcyBzb21lIG90aGVycy5cbiAgICBjb25zdCB0cmFuc2Zvcm1lciA9IG5ldyBHcmFwaFFMVHJhbnNmb3JtKHtcbiAgICAgIHRyYW5zZm9ybUNvbmZpZzogdHJhbnNmb3JtQ29uZmlnLFxuICAgICAgdHJhbnNmb3JtZXJzOiBbXG4gICAgICAgIG5ldyBEeW5hbW9EQk1vZGVsVHJhbnNmb3JtZXIoKSxcbiAgICAgICAgbmV3IFR0bFRyYW5zZm9ybWVyKCksXG4gICAgICAgIG5ldyBWZXJzaW9uZWRNb2RlbFRyYW5zZm9ybWVyKCksXG4gICAgICAgIG5ldyBGdW5jdGlvblRyYW5zZm9ybWVyKCksXG4gICAgICAgIG5ldyBLZXlUcmFuc2Zvcm1lcigpLFxuICAgICAgICBuZXcgTW9kZWxDb25uZWN0aW9uVHJhbnNmb3JtZXIoKSxcbiAgICAgICAgbmV3IE1vZGVsQXV0aFRyYW5zZm9ybWVyKHRoaXMuYXV0aFRyYW5zZm9ybWVyQ29uZmlnKSxcbiAgICAgICAgbmV3IEh0dHBUcmFuc2Zvcm1lcigpLFxuICAgICAgICAuLi5wcmVDZGtUcmFuc2Zvcm1lcnMsXG4gICAgICAgIG5ldyBDZGtUcmFuc2Zvcm1lcigpLFxuICAgICAgICAuLi5wb3N0Q2RrVHJhbnNmb3JtZXJzLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHNjaGVtYSA9IGZzLnJlYWRGaWxlU3luYyh0aGlzLnNjaGVtYVBhdGgpO1xuICAgIGNvbnN0IGNmZG9jID0gdHJhbnNmb3JtZXIudHJhbnNmb3JtKHNjaGVtYS50b1N0cmluZygpKTtcblxuICAgIC8vIFRPRE86IEdldCBVbmF1dGggUm9sZSBhbmQgQXV0aCBSb2xlIHBvbGljaWVzIGZvciBhdXRob3JpemF0aW9uIHN0dWZmXG4gICAgdGhpcy51bmF1dGhSb2xlUG9saWN5ID0gY2Zkb2Mucm9vdFN0YWNrLlJlc291cmNlcz8uVW5hdXRoUm9sZVBvbGljeTAxIGFzIFJlc291cmNlIHx8IHVuZGVmaW5lZDtcblxuICAgIHRoaXMud3JpdGVTY2hlbWEoY2Zkb2Muc2NoZW1hKTtcbiAgICB0aGlzLndyaXRlUmVzb2x2ZXJzVG9GaWxlKGNmZG9jLnJlc29sdmVycyk7XG5cbiAgICAvLyBPdXRwdXRzIHNob3VsZG4ndCBiZSBudWxsIGJ1dCBkZWZhdWx0IHRvIGVtcHR5IG1hcFxuICAgIHRoaXMub3V0cHV0cyA9IGNmZG9jLnJvb3RTdGFjay5PdXRwdXRzID8/IHt9O1xuXG4gICAgcmV0dXJuIHRoaXMub3V0cHV0cztcbiAgfVxuXG4gIC8qKlxuICAgICAqXG4gICAgICovXG4gIHB1YmxpYyBnZXRSZXNvbHZlcnMoKSB7XG4gICAgY29uc3Qgc3RhdGVtZW50cyA9IFsnUXVlcnknLCAnTXV0YXRpb24nLCAnU3Vic2NyaXB0aW9uJ107XG4gICAgY29uc3QgcmVzb2x2ZXJzRGlyUGF0aCA9IG5vcm1hbGl6ZSgnLi9hcHBzeW5jL3Jlc29sdmVycycpO1xuICAgIGlmIChmcy5leGlzdHNTeW5jKHJlc29sdmVyc0RpclBhdGgpKSB7XG4gICAgICBjb25zdCBmaWxlcyA9IGZzLnJlYWRkaXJTeW5jKHJlc29sdmVyc0RpclBhdGgpO1xuICAgICAgZmlsZXMuZm9yRWFjaChmaWxlID0+IHtcbiAgICAgICAgLy8gRXhhbXBsZTogTXV0YXRpb24uY3JlYXRlQ2hhbm5lbC5yZXNwb25zZVxuICAgICAgICBsZXQgYXJncyA9IGZpbGUuc3BsaXQoJy4nKTtcbiAgICAgICAgbGV0IHR5cGVOYW1lOiBzdHJpbmcgPSBhcmdzWzBdO1xuICAgICAgICBsZXQgbmFtZTogc3RyaW5nID0gYXJnc1sxXTtcbiAgICAgICAgbGV0IHRlbXBsYXRlVHlwZSA9IGFyZ3NbMl07IC8vIHJlcXVlc3Qgb3IgcmVzcG9uc2VcbiAgICAgICAgbGV0IGZpbGVwYXRoID0gbm9ybWFsaXplKGAke3Jlc29sdmVyc0RpclBhdGh9LyR7ZmlsZX1gKTtcblxuICAgICAgICBpZiAoc3RhdGVtZW50cy5pbmRleE9mKHR5cGVOYW1lKSA+PSAwIHx8ICh0aGlzLm91dHB1dHMubm9uZVJlc29sdmVycyAmJiB0aGlzLm91dHB1dHMubm9uZVJlc29sdmVyc1tuYW1lXSkpIHtcbiAgICAgICAgICBpZiAoIXRoaXMucmVzb2x2ZXJzW25hbWVdKSB7XG4gICAgICAgICAgICB0aGlzLnJlc29sdmVyc1tuYW1lXSA9IHtcbiAgICAgICAgICAgICAgdHlwZU5hbWU6IHR5cGVOYW1lLFxuICAgICAgICAgICAgICBmaWVsZE5hbWU6IG5hbWUsXG4gICAgICAgICAgICB9O1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGlmICh0ZW1wbGF0ZVR5cGUgPT09ICdyZXEnKSB7XG4gICAgICAgICAgICB0aGlzLnJlc29sdmVyc1tuYW1lXS5yZXF1ZXN0TWFwcGluZ1RlbXBsYXRlID0gZmlsZXBhdGg7XG4gICAgICAgICAgfSBlbHNlIGlmICh0ZW1wbGF0ZVR5cGUgPT09ICdyZXMnKSB7XG4gICAgICAgICAgICB0aGlzLnJlc29sdmVyc1tuYW1lXS5yZXNwb25zZU1hcHBpbmdUZW1wbGF0ZSA9IGZpbGVwYXRoO1xuICAgICAgICAgIH1cblxuICAgICAgICB9IGVsc2UgeyAvLyBUaGlzIGlzIGEgR1NJXG4gICAgICAgICAgaWYgKCF0aGlzLnJlc29sdmVycy5nc2kpIHtcbiAgICAgICAgICAgIHRoaXMucmVzb2x2ZXJzLmdzaSA9IHt9O1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGxldCBtYXBOYW1lID0gYCR7dHlwZU5hbWV9JHtuYW1lfWA7XG4gICAgICAgICAgaWYgKCF0aGlzLnJlc29sdmVycy5nc2lbbWFwTmFtZV0pIHtcbiAgICAgICAgICAgIHRoaXMucmVzb2x2ZXJzLmdzaVttYXBOYW1lXSA9IHtcbiAgICAgICAgICAgICAgdHlwZU5hbWU6IHR5cGVOYW1lLFxuICAgICAgICAgICAgICBmaWVsZE5hbWU6IG5hbWUsXG4gICAgICAgICAgICAgIHRhYmxlTmFtZTogbmFtZS5jaGFyQXQoMCkudG9VcHBlckNhc2UoKSArIG5hbWUuc2xpY2UoMSksXG4gICAgICAgICAgICB9O1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGlmICh0ZW1wbGF0ZVR5cGUgPT09ICdyZXEnKSB7XG4gICAgICAgICAgICB0aGlzLnJlc29sdmVycy5nc2lbbWFwTmFtZV0ucmVxdWVzdE1hcHBpbmdUZW1wbGF0ZSA9IGZpbGVwYXRoO1xuICAgICAgICAgIH0gZWxzZSBpZiAodGVtcGxhdGVUeXBlID09PSAncmVzJykge1xuICAgICAgICAgICAgdGhpcy5yZXNvbHZlcnMuZ3NpW21hcE5hbWVdLnJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlID0gZmlsZXBhdGg7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5yZXNvbHZlcnM7XG4gIH1cblxuICAvKipcbiAgICAgKiBXcml0ZXMgdGhlIHNjaGVtYSB0byB0aGUgb3V0cHV0IGRpcmVjdG9yeSBmb3IgdXNlIHdpdGggQGF3cy1jZGsvYXdzLWFwcHN5bmNcbiAgICAgKiBAcGFyYW0gc2NoZW1hXG4gICAgICovXG4gIHByaXZhdGUgd3JpdGVTY2hlbWEoc2NoZW1hOiBhbnkpIHtcbiAgICBpZiAoIWZzLmV4aXN0c1N5bmModGhpcy5vdXRwdXRQYXRoKSkge1xuICAgICAgZnMubWtkaXJTeW5jKHRoaXMub3V0cHV0UGF0aCk7XG4gICAgfVxuXG4gICAgZnMud3JpdGVGaWxlU3luYyhgJHt0aGlzLm91dHB1dFBhdGh9L3NjaGVtYS5ncmFwaHFsYCwgc2NoZW1hKTtcbiAgfVxuXG4gIC8qKlxuICAgICAqIFdyaXRlcyBhbGwgdGhlIHJlc29sdmVycyB0byB0aGUgb3V0cHV0IGRpcmVjdG9yeSBmb3IgbG9hZGluZyBpbnRvIHRoZSBkYXRhc291cmNlcyBsYXRlclxuICAgICAqIEBwYXJhbSByZXNvbHZlcnNcbiAgICAgKi9cbiAgcHJpdmF0ZSB3cml0ZVJlc29sdmVyc1RvRmlsZShyZXNvbHZlcnM6IGFueSkge1xuICAgIGlmICghZnMuZXhpc3RzU3luYyh0aGlzLm91dHB1dFBhdGgpKSB7XG4gICAgICBmcy5ta2RpclN5bmModGhpcy5vdXRwdXRQYXRoKTtcbiAgICB9XG5cbiAgICBjb25zdCByZXNvbHZlckZvbGRlclBhdGggPSBub3JtYWxpemUodGhpcy5vdXRwdXRQYXRoICsgJy9yZXNvbHZlcnMnKTtcbiAgICBpZiAoZnMuZXhpc3RzU3luYyhyZXNvbHZlckZvbGRlclBhdGgpKSB7XG4gICAgICBjb25zdCBmaWxlcyA9IGZzLnJlYWRkaXJTeW5jKHJlc29sdmVyRm9sZGVyUGF0aCk7XG4gICAgICBmaWxlcy5mb3JFYWNoKGZpbGUgPT4gZnMudW5saW5rU3luYyhyZXNvbHZlckZvbGRlclBhdGggKyAnLycgKyBmaWxlKSk7XG4gICAgICBmcy5ybWRpclN5bmMocmVzb2x2ZXJGb2xkZXJQYXRoKTtcbiAgICB9XG5cbiAgICBpZiAoIWZzLmV4aXN0c1N5bmMocmVzb2x2ZXJGb2xkZXJQYXRoKSkge1xuICAgICAgZnMubWtkaXJTeW5jKHJlc29sdmVyRm9sZGVyUGF0aCk7XG4gICAgfVxuXG4gICAgT2JqZWN0LmtleXMocmVzb2x2ZXJzKS5mb3JFYWNoKChrZXk6IGFueSkgPT4ge1xuICAgICAgY29uc3QgcmVzb2x2ZXIgPSByZXNvbHZlcnNba2V5XTtcbiAgICAgIGNvbnN0IGZpbGVOYW1lID0ga2V5LnJlcGxhY2UoJy52dGwnLCAnJyk7XG4gICAgICBjb25zdCByZXNvbHZlckZpbGVQYXRoID0gbm9ybWFsaXplKGAke3Jlc29sdmVyRm9sZGVyUGF0aH0vJHtmaWxlTmFtZX1gKTtcbiAgICAgIGZzLndyaXRlRmlsZVN5bmMocmVzb2x2ZXJGaWxlUGF0aCwgcmVzb2x2ZXIpO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAgICogQHJldHVybnMge0BsaW5rIFRyYW5zZm9ybUNvbmZpZ31cbiAgICAqL1xuICBwcml2YXRlIGxvYWRDb25maWdTeW5jKHByb2plY3REaXI6IHN0cmluZyA9ICdyZXNvdXJjZXMnKTogVHJhbnNmb3JtQ29uZmlnIHtcbiAgICAvLyBJbml0aWFsaXplIHRoZSBjb25maWcgYWx3YXlzIHdpdGggdGhlIGxhdGVzdCB2ZXJzaW9uLCBvdGhlciBtZW1iZXJzIGFyZSBvcHRpb25hbCBmb3Igbm93LlxuICAgIGxldCBjb25maWc6IFRyYW5zZm9ybUNvbmZpZyA9IHtcbiAgICAgIFZlcnNpb246IFRSQU5TRk9STV9DVVJSRU5UX1ZFUlNJT04sXG4gICAgICBSZXNvbHZlckNvbmZpZzoge1xuICAgICAgICBwcm9qZWN0OiB7XG4gICAgICAgICAgQ29uZmxpY3RIYW5kbGVyOiBDb25mbGljdEhhbmRsZXJUeXBlLk9QVElNSVNUSUMsXG4gICAgICAgICAgQ29uZmxpY3REZXRlY3Rpb246ICdWRVJTSU9OJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGNvbnN0IGNvbmZpZ0RpciA9IGpvaW4oX19kaXJuYW1lLCAnLi4nLCAnLi4nLCBwcm9qZWN0RGlyKTtcblxuICAgIHRyeSB7XG4gICAgICBjb25zdCBjb25maWdQYXRoID0gam9pbihjb25maWdEaXIsIFRSQU5TRk9STV9DT05GSUdfRklMRV9OQU1FKTtcbiAgICAgIGNvbnN0IGNvbmZpZ0V4aXN0cyA9IGZzLmV4aXN0c1N5bmMoY29uZmlnUGF0aCk7XG4gICAgICBpZiAoY29uZmlnRXhpc3RzKSB7XG4gICAgICAgIGNvbnN0IGNvbmZpZ1N0ciA9IGZzLnJlYWRGaWxlU3luYyhjb25maWdQYXRoKTtcbiAgICAgICAgY29uZmlnID0gSlNPTi5wYXJzZShjb25maWdTdHIudG9TdHJpbmcoKSk7XG4gICAgICB9XG5cbiAgICAgIHJldHVybiBjb25maWcgYXMgVHJhbnNmb3JtQ29uZmlnO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgcmV0dXJuIGNvbmZpZztcbiAgICB9XG4gIH1cbn0iXX0=