"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppSyncTransformer = void 0;
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const schema_transformer_1 = require("./transformer/schema-transformer");
const defaultAuthorizationConfig = {
    defaultAuthorization: {
        authorizationType: aws_appsync_1.AuthorizationType.API_KEY,
        apiKeyConfig: {
            description: 'Auto generated API Key from construct',
            name: 'dev',
        },
    },
};
/**
 * (experimental) AppSyncTransformer Construct.
 *
 * @experimental
 */
class AppSyncTransformer extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        this.isSyncEnabled = props.syncEnabled ? props.syncEnabled : false;
        const transformerConfiguration = {
            schemaPath: props.schemaPath,
            syncEnabled: (_a = props.syncEnabled) !== null && _a !== void 0 ? _a : false,
        };
        // Combine the arrays so we only loop once
        // Test each transformer to see if it implements ITransformer
        const allCustomTransformers = [...(_b = props.preCdkTransformers) !== null && _b !== void 0 ? _b : [], ...(_c = props.postCdkTransformers) !== null && _c !== void 0 ? _c : []];
        if (allCustomTransformers && allCustomTransformers.length > 0) {
            allCustomTransformers.forEach(transformer => {
                if (transformer && !this.implementsITransformer(transformer)) {
                    throw new Error(`Transformer does not implement ITransformer from graphql-transformer-core: ${transformer}`);
                }
            });
        }
        const transformer = new schema_transformer_1.SchemaTransformer(transformerConfiguration);
        this.outputs = transformer.transform(props.preCdkTransformers, props.postCdkTransformers);
        const resolvers = transformer.getResolvers();
        this.functionResolvers = (_d = this.outputs.functionResolvers) !== null && _d !== void 0 ? _d : {};
        // Remove any function resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, functionResolvers] of Object.entries(this.functionResolvers)) {
            functionResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.httpResolvers = (_e = this.outputs.httpResolvers) !== null && _e !== void 0 ? _e : {};
        // Remove any http resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, httpResolvers] of Object.entries(this.httpResolvers)) {
            httpResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.resolvers = resolvers;
        this.nestedAppsyncStack = new core_1.NestedStack(this, 'appsync-nested-stack');
        // AppSync
        this.appsyncAPI = new aws_appsync_1.GraphqlApi(this.nestedAppsyncStack, `${id}-api`, {
            name: props.apiName ? props.apiName : `${id}-api`,
            authorizationConfig: props.authorizationConfig
                ? props.authorizationConfig
                : defaultAuthorizationConfig,
            logConfig: {
                fieldLogLevel: props.fieldLogLevel
                    ? props.fieldLogLevel
                    : aws_appsync_1.FieldLogLevel.NONE,
            },
            schema: aws_appsync_1.Schema.fromAsset('./appsync/schema.graphql'),
            xrayEnabled: (_f = props.xrayEnabled) !== null && _f !== void 0 ? _f : false,
        });
        let tableData = (_g = this.outputs.cdkTables) !== null && _g !== void 0 ? _g : {};
        // Check to see if sync is enabled
        if (tableData.DataStore) {
            this.isSyncEnabled = true;
            this.syncTable = this.createSyncTable(tableData.DataStore);
            delete tableData.DataStore; // We don't want to create this again below so remove it from the tableData map
        }
        this.tableNameMap = this.createTablesAndResolvers(tableData, resolvers);
        if (this.outputs.noneResolvers) {
            this.createNoneDataSourceAndResolvers(this.outputs.noneResolvers, resolvers);
        }
        this.createHttpResolvers();
        // Outputs so we can generate exports
        new core_1.CfnOutput(scope, 'appsyncGraphQLEndpointOutput', {
            value: this.appsyncAPI.graphqlUrl,
            description: 'Output for aws_appsync_graphqlEndpoint',
        });
    }
    /**
     * graphql-transformer-core needs to be jsii enabled to pull the ITransformer interface correctly.
     * Since it's not in peer dependencies it doesn't show up in the jsii deps list.
     * Since it's not jsii enabled it has to be bundled.
     * The package can't be in BOTH peer and bundled dependencies
     * So we do a fake test to make sure it implements these and hope for the best
     * @param transformer
     */
    implementsITransformer(transformer) {
        return 'name' in transformer
            && 'directive' in transformer
            && 'typeDefinitions' in transformer;
    }
    /**
     * Creates NONE data source and associated resolvers
     * @param noneResolvers The resolvers that belong to the none data source
     * @param resolvers The resolver map minus function resolvers
     */
    createNoneDataSourceAndResolvers(noneResolvers, resolvers) {
        const noneDataSource = this.appsyncAPI.addNoneDataSource('NONE');
        Object.keys(noneResolvers).forEach((resolverKey) => {
            const resolver = resolvers[resolverKey];
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: noneDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
            });
        });
    }
    /**
     * Creates each dynamodb table, gsis, dynamodb datasource, and associated resolvers
     * If sync is enabled then TTL configuration is added
     * Returns tableName: table map in case it is needed for lambda functions, etc
     * @param tableData The CdkTransformer table information
     * @param resolvers The resolver map minus function resolvers
     */
    createTablesAndResolvers(tableData, resolvers) {
        const tableNameMap = {};
        Object.keys(tableData).forEach((tableKey) => {
            const table = this.createTable(tableData[tableKey]);
            const dataSource = this.appsyncAPI.addDynamoDbDataSource(tableKey, table);
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-datasource-deltasyncconfig.html
            if (this.isSyncEnabled && this.syncTable) {
                //@ts-ignore - ds is the base CfnDataSource and the db config needs to be versioned - see CfnDataSource
                dataSource.ds.dynamoDbConfig.versioned = true;
                //@ts-ignore - ds is the base CfnDataSource - see CfnDataSource
                dataSource.ds.dynamoDbConfig.deltaSyncConfig = {
                    baseTableTtl: '43200',
                    deltaSyncTableName: this.syncTable.tableName,
                    deltaSyncTableTtl: '30',
                };
                // Need to add permission for our datasource service role to access the sync table
                dataSource.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'dynamodb:*',
                    ],
                    resources: [this.syncTable.tableArn],
                }));
            }
            const dynamoDbConfig = dataSource.ds
                .dynamoDbConfig;
            tableNameMap[tableKey] = dynamoDbConfig.tableName;
            // Loop the basic resolvers
            tableData[tableKey].resolvers.forEach((resolverKey) => {
                let resolver = resolvers[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
            // Loop the gsi resolvers
            tableData[tableKey].gsiResolvers.forEach((resolverKey) => {
                let resolver = resolvers.gsi[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
        });
        return tableNameMap;
    }
    createTable(tableData) {
        let tableProps = {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
        };
        if (tableData.sortKey && tableData.sortKey.name) {
            tableProps.sortKey = {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            };
        }
        if (tableData.ttl && tableData.ttl.enabled) {
            tableProps.timeToLiveAttribute = tableData.ttl.attributeName;
        }
        const table = new aws_dynamodb_1.Table(this.nestedAppsyncStack, tableData.tableName, tableProps);
        if (tableData.globalSecondaryIndexes &&
            tableData.globalSecondaryIndexes.length > 0) {
            tableData.globalSecondaryIndexes.forEach((gsi) => {
                table.addGlobalSecondaryIndex({
                    indexName: gsi.indexName,
                    partitionKey: {
                        name: gsi.partitionKey.name,
                        type: this.convertAttributeType(gsi.partitionKey.type),
                    },
                    projectionType: this.convertProjectionType(gsi.projection.ProjectionType),
                });
            });
        }
        return table;
    }
    /**
     * Creates the sync table for Amplify DataStore
     * https://docs.aws.amazon.com/appsync/latest/devguide/conflict-detection-and-sync.html
     * @param tableData The CdkTransformer table information
     */
    createSyncTable(tableData) {
        var _a;
        return new aws_dynamodb_1.Table(this, 'appsync-api-sync-table', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            sortKey: {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            },
            timeToLiveAttribute: ((_a = tableData.ttl) === null || _a === void 0 ? void 0 : _a.attributeName) || '_ttl',
        });
    }
    convertAttributeType(type) {
        switch (type) {
            case 'N':
                return aws_dynamodb_1.AttributeType.NUMBER;
            case 'B':
                return aws_dynamodb_1.AttributeType.BINARY;
            case 'S': // Same as default
            default:
                return aws_dynamodb_1.AttributeType.STRING;
        }
    }
    convertProjectionType(type) {
        switch (type) {
            case 'INCLUDE':
                return aws_dynamodb_1.ProjectionType.INCLUDE;
            case 'KEYS_ONLY':
                return aws_dynamodb_1.ProjectionType.KEYS_ONLY;
            case 'ALL': // Same as default
            default:
                return aws_dynamodb_1.ProjectionType.ALL;
        }
    }
    createHttpResolvers() {
        for (const [endpoint, httpResolvers] of Object.entries(this.httpResolvers)) {
            const strippedEndpoint = endpoint.replace(/[^_0-9A-Za-z]/g, '');
            const httpDataSource = this.appsyncAPI.addHttpDataSource(`${strippedEndpoint}`, endpoint);
            httpResolvers.forEach((resolver) => {
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: httpDataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
                });
            });
        }
    }
    /**
     * (experimental) Adds the function as a lambdaDataSource to the AppSync api Adds all of the functions resolvers to the AppSync api.
     *
     * @param functionName The function name specified in the.
     * @param id The id to give.
     * @param lambdaFunction The lambda function to attach.
     * @experimental
     * @function directive of the schema
     */
    addLambdaDataSourceAndResolvers(functionName, id, lambdaFunction, options) {
        const functionDataSource = this.appsyncAPI.addLambdaDataSource(id, lambdaFunction, options);
        for (const resolver of this.functionResolvers[functionName]) {
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: functionDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
            });
        }
        return functionDataSource;
    }
}
exports.AppSyncTransformer = AppSyncTransformer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwc3luYy10cmFuc2Zvcm1lci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9hcHBzeW5jLXRyYW5zZm9ybWVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNEQVc4QjtBQUU5Qix3REFLK0I7QUFDL0IsOENBQTJEO0FBRTNELHdDQUFrRTtBQVVsRSx5RUFHMEM7QUEwRDFDLE1BQU0sMEJBQTBCLEdBQXdCO0lBQ3RELG9CQUFvQixFQUFFO1FBQ3BCLGlCQUFpQixFQUFFLCtCQUFpQixDQUFDLE9BQU87UUFDNUMsWUFBWSxFQUFFO1lBQ1osV0FBVyxFQUFFLHVDQUF1QztZQUNwRCxJQUFJLEVBQUUsS0FBSztTQUNaO0tBQ0Y7Q0FDRixDQUFDOzs7Ozs7QUFLRixNQUFhLGtCQUFtQixTQUFRLGdCQUFTOzs7O0lBeUMvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQThCOztRQUN0RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1FBRW5FLE1BQU0sd0JBQXdCLEdBQTJCO1lBQ3ZELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixXQUFXLFFBQUUsS0FBSyxDQUFDLFdBQVcsbUNBQUksS0FBSztTQUN4QyxDQUFDO1FBRUYsMENBQTBDO1FBQzFDLDZEQUE2RDtRQUM3RCxNQUFNLHFCQUFxQixHQUFHLENBQUMsU0FBRyxLQUFLLENBQUMsa0JBQWtCLG1DQUFJLEVBQUUsRUFBRSxTQUFHLEtBQUssQ0FBQyxtQkFBbUIsbUNBQUksRUFBRSxDQUFDLENBQUM7UUFDdEcsSUFBSSxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzdELHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRTtnQkFDMUMsSUFBSSxXQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsV0FBVyxDQUFDLEVBQUU7b0JBQzVELE1BQU0sSUFBSSxLQUFLLENBQUMsOEVBQThFLFdBQVcsRUFBRSxDQUFDLENBQUM7aUJBQzlHO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sV0FBVyxHQUFHLElBQUksc0NBQWlCLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsT0FBTyxHQUFHLFdBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzFGLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUU3QyxJQUFJLENBQUMsaUJBQWlCLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsbUNBQUksRUFBRSxDQUFDO1FBRTlELGlFQUFpRTtRQUNqRSxtQ0FBbUM7UUFDbkMsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLGlCQUFpQixDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FDakQsSUFBSSxDQUFDLGlCQUFpQixDQUN2QixFQUFFO1lBQ0QsaUJBQWlCLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBYSxFQUFFLEVBQUU7Z0JBQzFDLFFBQVEsUUFBUSxDQUFDLFFBQVEsRUFBRTtvQkFDekIsS0FBSyxPQUFPLENBQUM7b0JBQ2IsS0FBSyxVQUFVLENBQUM7b0JBQ2hCLEtBQUssY0FBYzt3QkFDakIsT0FBTyxTQUFTLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO3dCQUNyQyxNQUFNO2lCQUNUO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxhQUFhLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLG1DQUFJLEVBQUUsQ0FBQztRQUV0RCw2REFBNkQ7UUFDN0QsbUNBQW1DO1FBQ25DLEtBQUssTUFBTSxDQUFDLENBQUMsRUFBRSxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNuRSxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBYSxFQUFFLEVBQUU7Z0JBQ3RDLFFBQVEsUUFBUSxDQUFDLFFBQVEsRUFBRTtvQkFDekIsS0FBSyxPQUFPLENBQUM7b0JBQ2IsS0FBSyxVQUFVLENBQUM7b0JBQ2hCLEtBQUssY0FBYzt3QkFDakIsT0FBTyxTQUFTLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO3dCQUNyQyxNQUFNO2lCQUNUO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxTQUFTLEdBQUcsU0FBUyxDQUFDO1FBRTNCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLGtCQUFXLENBQUMsSUFBSSxFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFFeEUsVUFBVTtRQUNWLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSx3QkFBVSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFO1lBQ3JFLElBQUksRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsTUFBTTtZQUNqRCxtQkFBbUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO2dCQUM1QyxDQUFDLENBQUMsS0FBSyxDQUFDLG1CQUFtQjtnQkFDM0IsQ0FBQyxDQUFDLDBCQUEwQjtZQUM5QixTQUFTLEVBQUU7Z0JBQ1QsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO29CQUNoQyxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWE7b0JBQ3JCLENBQUMsQ0FBQywyQkFBYSxDQUFDLElBQUk7YUFDdkI7WUFDRCxNQUFNLEVBQUUsb0JBQU0sQ0FBQyxTQUFTLENBQUMsMEJBQTBCLENBQUM7WUFDcEQsV0FBVyxRQUFFLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEtBQUs7U0FDeEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxTQUFTLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLG1DQUFJLEVBQUUsQ0FBQztRQUU3QyxrQ0FBa0M7UUFDbEMsSUFBSSxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDO1lBQzFCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDM0QsT0FBTyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsK0VBQStFO1NBQzVHO1FBRUQsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hFLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDOUIsSUFBSSxDQUFDLGdDQUFnQyxDQUNuQyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFDMUIsU0FBUyxDQUNWLENBQUM7U0FDSDtRQUNELElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1FBRTNCLHFDQUFxQztRQUNyQyxJQUFJLGdCQUFTLENBQUMsS0FBSyxFQUFFLDhCQUE4QixFQUFFO1lBQ25ELEtBQUssRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVU7WUFDakMsV0FBVyxFQUFFLHdDQUF3QztTQUN0RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNLLHNCQUFzQixDQUFDLFdBQWdCO1FBQzdDLE9BQU8sTUFBTSxJQUFJLFdBQVc7ZUFDdkIsV0FBVyxJQUFJLFdBQVc7ZUFDMUIsaUJBQWlCLElBQUksV0FBVyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZ0NBQWdDLENBQ3RDLGFBQXlELEVBQ3pELFNBQWM7UUFFZCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRWpFLE1BQU0sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBZ0IsRUFBRSxFQUFFO1lBQ3RELE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUV4QyxJQUFJLHNCQUFRLENBQ1YsSUFBSSxDQUFDLGtCQUFrQixFQUN2QixHQUFHLFFBQVEsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFNBQVMsV0FBVyxFQUNyRDtnQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7Z0JBQ3BCLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtnQkFDM0IsU0FBUyxFQUFFLFFBQVEsQ0FBQyxTQUFTO2dCQUM3QixVQUFVLEVBQUUsY0FBYztnQkFDMUIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQzlDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FDaEM7Z0JBQ0QsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQy9DLFFBQVEsQ0FBQyx1QkFBdUIsQ0FDakM7YUFDRixDQUNGLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSyx3QkFBd0IsQ0FDOUIsU0FBa0QsRUFDbEQsU0FBYztRQUVkLE1BQU0sWUFBWSxHQUFRLEVBQUUsQ0FBQztRQUU3QixNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQWEsRUFBRSxFQUFFO1lBQy9DLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7WUFDcEQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFFMUUsd0hBQXdIO1lBRXhILElBQUksSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO2dCQUN4Qyx1R0FBdUc7Z0JBQ3ZHLFVBQVUsQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7Z0JBRTlDLCtEQUErRDtnQkFDL0QsVUFBVSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsZUFBZSxHQUFHO29CQUM3QyxZQUFZLEVBQUUsT0FBTztvQkFDckIsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTO29CQUM1QyxpQkFBaUIsRUFBRSxJQUFJO2lCQUN4QixDQUFDO2dCQUVGLGtGQUFrRjtnQkFDbEYsVUFBVSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQ25DLElBQUkseUJBQWUsQ0FBQztvQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztvQkFDcEIsT0FBTyxFQUFFO3dCQUNQLFlBQVk7cUJBQ2I7b0JBQ0QsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUM7aUJBQ3JDLENBQUMsQ0FDSCxDQUFDO2FBQ0g7WUFFRCxNQUFNLGNBQWMsR0FBRyxVQUFVLENBQUMsRUFBRTtpQkFDakMsY0FBc0QsQ0FBQztZQUMxRCxZQUFZLENBQUMsUUFBUSxDQUFDLEdBQUcsY0FBYyxDQUFDLFNBQVMsQ0FBQztZQUVsRCwyQkFBMkI7WUFDM0IsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxXQUFnQixFQUFFLEVBQUU7Z0JBQ3pELElBQUksUUFBUSxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDdEMsSUFBSSxzQkFBUSxDQUNWLElBQUksQ0FBQyxrQkFBa0IsRUFDdkIsR0FBRyxRQUFRLENBQUMsUUFBUSxJQUFJLFFBQVEsQ0FBQyxTQUFTLFdBQVcsRUFDckQ7b0JBQ0UsR0FBRyxFQUFFLElBQUksQ0FBQyxVQUFVO29CQUNwQixRQUFRLEVBQUUsUUFBUSxDQUFDLFFBQVE7b0JBQzNCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUztvQkFDN0IsVUFBVSxFQUFFLFVBQVU7b0JBQ3RCLHNCQUFzQixFQUFFLDZCQUFlLENBQUMsUUFBUSxDQUM5QyxRQUFRLENBQUMsc0JBQXNCLENBQ2hDO29CQUNELHVCQUF1QixFQUFFLDZCQUFlLENBQUMsUUFBUSxDQUMvQyxRQUFRLENBQUMsdUJBQXVCLENBQ2pDO2lCQUNGLENBQ0YsQ0FBQztZQUNKLENBQUMsQ0FBQyxDQUFDO1lBRUgseUJBQXlCO1lBQ3pCLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBZ0IsRUFBRSxFQUFFO2dCQUM1RCxJQUFJLFFBQVEsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUMxQyxJQUFJLHNCQUFRLENBQ1YsSUFBSSxDQUFDLGtCQUFrQixFQUN2QixHQUFHLFFBQVEsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFNBQVMsV0FBVyxFQUNyRDtvQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7b0JBQ3BCLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtvQkFDM0IsU0FBUyxFQUFFLFFBQVEsQ0FBQyxTQUFTO29CQUM3QixVQUFVLEVBQUUsVUFBVTtvQkFDdEIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQzlDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FDaEM7b0JBQ0QsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQy9DLFFBQVEsQ0FBQyx1QkFBdUIsQ0FDakM7aUJBQ0YsQ0FDRixDQUFDO1lBQ0osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7SUFFTyxXQUFXLENBQUMsU0FBOEI7UUFDaEQsSUFBSSxVQUFVLEdBQVE7WUFDcEIsV0FBVyxFQUFFLDBCQUFXLENBQUMsZUFBZTtZQUN4QyxZQUFZLEVBQUU7Z0JBQ1osSUFBSSxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSTtnQkFDakMsSUFBSSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQzthQUM3RDtTQUNGLENBQUM7UUFFRixJQUFJLFNBQVMsQ0FBQyxPQUFPLElBQUksU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUU7WUFDL0MsVUFBVSxDQUFDLE9BQU8sR0FBRztnQkFDbkIsSUFBSSxFQUFFLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSTtnQkFDNUIsSUFBSSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQzthQUN4RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLFNBQVMsQ0FBQyxHQUFHLElBQUksU0FBUyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUU7WUFDMUMsVUFBVSxDQUFDLG1CQUFtQixHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDO1NBQzlEO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxvQkFBSyxDQUNyQixJQUFJLENBQUMsa0JBQWtCLEVBQ3ZCLFNBQVMsQ0FBQyxTQUFTLEVBQ25CLFVBQVUsQ0FDWCxDQUFDO1FBRUYsSUFDRSxTQUFTLENBQUMsc0JBQXNCO1lBQ2hDLFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUMzQztZQUNBLFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFRLEVBQUUsRUFBRTtnQkFDcEQsS0FBSyxDQUFDLHVCQUF1QixDQUFDO29CQUM1QixTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7b0JBQ3hCLFlBQVksRUFBRTt3QkFDWixJQUFJLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJO3dCQUMzQixJQUFJLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDO3FCQUN2RDtvQkFDRCxjQUFjLEVBQUUsSUFBSSxDQUFDLHFCQUFxQixDQUN4QyxHQUFHLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FDOUI7aUJBQ0YsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxlQUFlLENBQUMsU0FBOEI7O1FBQ3BELE9BQU8sSUFBSSxvQkFBSyxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUMvQyxXQUFXLEVBQUUsMEJBQVcsQ0FBQyxlQUFlO1lBQ3hDLFlBQVksRUFBRTtnQkFDWixJQUFJLEVBQUUsU0FBUyxDQUFDLFlBQVksQ0FBQyxJQUFJO2dCQUNqQyxJQUFJLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDO2FBQzdEO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLElBQUksRUFBRSxTQUFTLENBQUMsT0FBUSxDQUFDLElBQUk7Z0JBQzdCLElBQUksRUFBRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLE9BQVEsQ0FBQyxJQUFJLENBQUM7YUFDekQ7WUFDRCxtQkFBbUIsRUFBRSxPQUFBLFNBQVMsQ0FBQyxHQUFHLDBDQUFFLGFBQWEsS0FBSSxNQUFNO1NBQzVELENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxJQUFZO1FBQ3ZDLFFBQVEsSUFBSSxFQUFFO1lBQ1osS0FBSyxHQUFHO2dCQUNOLE9BQU8sNEJBQWEsQ0FBQyxNQUFNLENBQUM7WUFDOUIsS0FBSyxHQUFHO2dCQUNOLE9BQU8sNEJBQWEsQ0FBQyxNQUFNLENBQUM7WUFDOUIsS0FBSyxHQUFHLENBQUMsQ0FBQyxrQkFBa0I7WUFDNUI7Z0JBQ0UsT0FBTyw0QkFBYSxDQUFDLE1BQU0sQ0FBQztTQUMvQjtJQUNILENBQUM7SUFFTyxxQkFBcUIsQ0FBQyxJQUFZO1FBQ3hDLFFBQVEsSUFBSSxFQUFFO1lBQ1osS0FBSyxTQUFTO2dCQUNaLE9BQU8sNkJBQWMsQ0FBQyxPQUFPLENBQUM7WUFDaEMsS0FBSyxXQUFXO2dCQUNkLE9BQU8sNkJBQWMsQ0FBQyxTQUFTLENBQUM7WUFDbEMsS0FBSyxLQUFLLENBQUMsQ0FBQyxrQkFBa0I7WUFDOUI7Z0JBQ0UsT0FBTyw2QkFBYyxDQUFDLEdBQUcsQ0FBQztTQUM3QjtJQUNILENBQUM7SUFFTyxtQkFBbUI7UUFDekIsS0FBSyxNQUFNLENBQUMsUUFBUSxFQUFFLGFBQWEsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQ3BELElBQUksQ0FBQyxhQUFhLENBQ25CLEVBQUU7WUFDRCxNQUFNLGdCQUFnQixHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDaEUsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxpQkFBaUIsQ0FDdEQsR0FBRyxnQkFBZ0IsRUFBRSxFQUNyQixRQUFRLENBQ1QsQ0FBQztZQUVGLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxRQUFvQyxFQUFFLEVBQUU7Z0JBQzdELElBQUksc0JBQVEsQ0FDVixJQUFJLENBQUMsa0JBQWtCLEVBQ3ZCLEdBQUcsUUFBUSxDQUFDLFFBQVEsSUFBSSxRQUFRLENBQUMsU0FBUyxXQUFXLEVBQ3JEO29CQUNFLEdBQUcsRUFBRSxJQUFJLENBQUMsVUFBVTtvQkFDcEIsUUFBUSxFQUFFLFFBQVEsQ0FBQyxRQUFRO29CQUMzQixTQUFTLEVBQUUsUUFBUSxDQUFDLFNBQVM7b0JBQzdCLFVBQVUsRUFBRSxjQUFjO29CQUMxQixzQkFBc0IsRUFBRSw2QkFBZSxDQUFDLFVBQVUsQ0FDaEQsUUFBUSxDQUFDLDZCQUE2QixDQUN2QztvQkFDRCx1QkFBdUIsRUFBRSw2QkFBZSxDQUFDLFVBQVUsQ0FDakQsUUFBUSxDQUFDLDhCQUE4QixDQUN4QztpQkFDRixDQUNGLENBQUM7WUFDSixDQUFDLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQzs7Ozs7Ozs7OztJQVVNLCtCQUErQixDQUNwQyxZQUFvQixFQUNwQixFQUFVLEVBQ1YsY0FBeUIsRUFDekIsT0FBMkI7UUFFM0IsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLG1CQUFtQixDQUM1RCxFQUFFLEVBQ0YsY0FBYyxFQUNkLE9BQU8sQ0FDUixDQUFDO1FBRUYsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsaUJBQWlCLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDM0QsSUFBSSxzQkFBUSxDQUNWLElBQUksQ0FBQyxrQkFBa0IsRUFDdkIsR0FBRyxRQUFRLENBQUMsUUFBUSxJQUFJLFFBQVEsQ0FBQyxTQUFTLFdBQVcsRUFDckQ7Z0JBQ0UsR0FBRyxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUNwQixRQUFRLEVBQUUsUUFBUSxDQUFDLFFBQVE7Z0JBQzNCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUztnQkFDN0IsVUFBVSxFQUFFLGtCQUFrQjtnQkFDOUIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxVQUFVLENBQ2hELFFBQVEsQ0FBQyw2QkFBNkIsQ0FDdkM7Z0JBQ0QsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxVQUFVLENBQ2pELFFBQVEsQ0FBQyw4QkFBOEIsQ0FDeEM7YUFDRixDQUNGLENBQUM7U0FDSDtRQUVELE9BQU8sa0JBQWtCLENBQUM7SUFDNUIsQ0FBQztDQUNGO0FBOWJELGdEQThiQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XG4gIEdyYXBocWxBcGksXG4gIEF1dGhvcml6YXRpb25UeXBlLFxuICBGaWVsZExvZ0xldmVsLFxuICBNYXBwaW5nVGVtcGxhdGUsXG4gIENmbkRhdGFTb3VyY2UsXG4gIFJlc29sdmVyLFxuICBBdXRob3JpemF0aW9uQ29uZmlnLFxuICBTY2hlbWEsXG4gIERhdGFTb3VyY2VPcHRpb25zLFxuICBMYW1iZGFEYXRhU291cmNlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtYXBwc3luYyc7XG5cbmltcG9ydCB7XG4gIFRhYmxlLFxuICBBdHRyaWJ1dGVUeXBlLFxuICBQcm9qZWN0aW9uVHlwZSxcbiAgQmlsbGluZ01vZGUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1keW5hbW9kYic7XG5pbXBvcnQgeyBFZmZlY3QsIFBvbGljeVN0YXRlbWVudCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgSUZ1bmN0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIE5lc3RlZFN0YWNrLCBDZm5PdXRwdXQgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHtcbiAgQ2RrVHJhbnNmb3JtZXJSZXNvbHZlcixcbiAgQ2RrVHJhbnNmb3JtZXJGdW5jdGlvblJlc29sdmVyLFxuICBDZGtUcmFuc2Zvcm1lckh0dHBSZXNvbHZlcixcbiAgQ2RrVHJhbnNmb3JtZXJUYWJsZSxcbiAgU2NoZW1hVHJhbnNmb3JtZXJPdXRwdXRzLFxufSBmcm9tICcuL3RyYW5zZm9ybWVyJztcblxuaW1wb3J0IHtcbiAgU2NoZW1hVHJhbnNmb3JtZXIsXG4gIFNjaGVtYVRyYW5zZm9ybWVyUHJvcHMsXG59IGZyb20gJy4vdHJhbnNmb3JtZXIvc2NoZW1hLXRyYW5zZm9ybWVyJztcblxuZXhwb3J0IGludGVyZmFjZSBBcHBTeW5jVHJhbnNmb3JtZXJQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNjaGVtYVBhdGg6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhdXRob3JpemF0aW9uQ29uZmlnPzogQXV0aG9yaXphdGlvbkNvbmZpZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwaU5hbWU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3luY0VuYWJsZWQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBmaWVsZExvZ0xldmVsPzogRmllbGRMb2dMZXZlbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB4cmF5RW5hYmxlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcblxuICByZWFkb25seSBwcmVDZGtUcmFuc2Zvcm1lcnM/OiBhbnlbXTtcblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcblxuICByZWFkb25seSBwb3N0Q2RrVHJhbnNmb3JtZXJzPzogYW55W107XG59XG5cbmNvbnN0IGRlZmF1bHRBdXRob3JpemF0aW9uQ29uZmlnOiBBdXRob3JpemF0aW9uQ29uZmlnID0ge1xuICBkZWZhdWx0QXV0aG9yaXphdGlvbjoge1xuICAgIGF1dGhvcml6YXRpb25UeXBlOiBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZLFxuICAgIGFwaUtleUNvbmZpZzoge1xuICAgICAgZGVzY3JpcHRpb246ICdBdXRvIGdlbmVyYXRlZCBBUEkgS2V5IGZyb20gY29uc3RydWN0JyxcbiAgICAgIG5hbWU6ICdkZXYnLFxuICAgIH0sXG4gIH0sXG59O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBcHBTeW5jVHJhbnNmb3JtZXIgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBhcHBzeW5jQVBJOiBHcmFwaHFsQXBpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbmVzdGVkQXBwc3luY1N0YWNrOiBOZXN0ZWRTdGFjaztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgdGFibGVOYW1lTWFwOiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgb3V0cHV0czogU2NoZW1hVHJhbnNmb3JtZXJPdXRwdXRzO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSByZXNvbHZlcnM6IGFueTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBmdW5jdGlvblJlc29sdmVyczoge1xuICAgIFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lckZ1bmN0aW9uUmVzb2x2ZXJbXTtcbiAgfTtcblxuICBwdWJsaWMgcmVhZG9ubHkgaHR0cFJlc29sdmVyczoge1xuICAgIFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lckh0dHBSZXNvbHZlcltdO1xuICB9O1xuXG4gIHByaXZhdGUgaXNTeW5jRW5hYmxlZDogYm9vbGVhbjtcbiAgcHJpdmF0ZSBzeW5jVGFibGU6IFRhYmxlIHwgdW5kZWZpbmVkO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcHBTeW5jVHJhbnNmb3JtZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmlzU3luY0VuYWJsZWQgPSBwcm9wcy5zeW5jRW5hYmxlZCA/IHByb3BzLnN5bmNFbmFibGVkIDogZmFsc2U7XG5cbiAgICBjb25zdCB0cmFuc2Zvcm1lckNvbmZpZ3VyYXRpb246IFNjaGVtYVRyYW5zZm9ybWVyUHJvcHMgPSB7XG4gICAgICBzY2hlbWFQYXRoOiBwcm9wcy5zY2hlbWFQYXRoLFxuICAgICAgc3luY0VuYWJsZWQ6IHByb3BzLnN5bmNFbmFibGVkID8/IGZhbHNlLFxuICAgIH07XG5cbiAgICAvLyBDb21iaW5lIHRoZSBhcnJheXMgc28gd2Ugb25seSBsb29wIG9uY2VcbiAgICAvLyBUZXN0IGVhY2ggdHJhbnNmb3JtZXIgdG8gc2VlIGlmIGl0IGltcGxlbWVudHMgSVRyYW5zZm9ybWVyXG4gICAgY29uc3QgYWxsQ3VzdG9tVHJhbnNmb3JtZXJzID0gWy4uLnByb3BzLnByZUNka1RyYW5zZm9ybWVycyA/PyBbXSwgLi4ucHJvcHMucG9zdENka1RyYW5zZm9ybWVycyA/PyBbXV07XG4gICAgaWYgKGFsbEN1c3RvbVRyYW5zZm9ybWVycyAmJiBhbGxDdXN0b21UcmFuc2Zvcm1lcnMubGVuZ3RoID4gMCkge1xuICAgICAgYWxsQ3VzdG9tVHJhbnNmb3JtZXJzLmZvckVhY2godHJhbnNmb3JtZXIgPT4ge1xuICAgICAgICBpZiAodHJhbnNmb3JtZXIgJiYgIXRoaXMuaW1wbGVtZW50c0lUcmFuc2Zvcm1lcih0cmFuc2Zvcm1lcikpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRyYW5zZm9ybWVyIGRvZXMgbm90IGltcGxlbWVudCBJVHJhbnNmb3JtZXIgZnJvbSBncmFwaHFsLXRyYW5zZm9ybWVyLWNvcmU6ICR7dHJhbnNmb3JtZXJ9YCk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IHRyYW5zZm9ybWVyID0gbmV3IFNjaGVtYVRyYW5zZm9ybWVyKHRyYW5zZm9ybWVyQ29uZmlndXJhdGlvbik7XG4gICAgdGhpcy5vdXRwdXRzID0gdHJhbnNmb3JtZXIudHJhbnNmb3JtKHByb3BzLnByZUNka1RyYW5zZm9ybWVycywgcHJvcHMucG9zdENka1RyYW5zZm9ybWVycyk7XG4gICAgY29uc3QgcmVzb2x2ZXJzID0gdHJhbnNmb3JtZXIuZ2V0UmVzb2x2ZXJzKCk7XG5cbiAgICB0aGlzLmZ1bmN0aW9uUmVzb2x2ZXJzID0gdGhpcy5vdXRwdXRzLmZ1bmN0aW9uUmVzb2x2ZXJzID8/IHt9O1xuXG4gICAgLy8gUmVtb3ZlIGFueSBmdW5jdGlvbiByZXNvbHZlcnMgZnJvbSB0aGUgdG90YWwgbGlzdCBvZiByZXNvbHZlcnNcbiAgICAvLyBPdGhlcndpc2UgaXQgd2lsbCBhZGQgdGhlbSB0d2ljZVxuICAgIGZvciAoY29uc3QgW18sIGZ1bmN0aW9uUmVzb2x2ZXJzXSBvZiBPYmplY3QuZW50cmllcyhcbiAgICAgIHRoaXMuZnVuY3Rpb25SZXNvbHZlcnMsXG4gICAgKSkge1xuICAgICAgZnVuY3Rpb25SZXNvbHZlcnMuZm9yRWFjaCgocmVzb2x2ZXI6IGFueSkgPT4ge1xuICAgICAgICBzd2l0Y2ggKHJlc29sdmVyLnR5cGVOYW1lKSB7XG4gICAgICAgICAgY2FzZSAnUXVlcnknOlxuICAgICAgICAgIGNhc2UgJ011dGF0aW9uJzpcbiAgICAgICAgICBjYXNlICdTdWJzY3JpcHRpb24nOlxuICAgICAgICAgICAgZGVsZXRlIHJlc29sdmVyc1tyZXNvbHZlci5maWVsZE5hbWVdO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuaHR0cFJlc29sdmVycyA9IHRoaXMub3V0cHV0cy5odHRwUmVzb2x2ZXJzID8/IHt9O1xuXG4gICAgLy8gUmVtb3ZlIGFueSBodHRwIHJlc29sdmVycyBmcm9tIHRoZSB0b3RhbCBsaXN0IG9mIHJlc29sdmVyc1xuICAgIC8vIE90aGVyd2lzZSBpdCB3aWxsIGFkZCB0aGVtIHR3aWNlXG4gICAgZm9yIChjb25zdCBbXywgaHR0cFJlc29sdmVyc10gb2YgT2JqZWN0LmVudHJpZXModGhpcy5odHRwUmVzb2x2ZXJzKSkge1xuICAgICAgaHR0cFJlc29sdmVycy5mb3JFYWNoKChyZXNvbHZlcjogYW55KSA9PiB7XG4gICAgICAgIHN3aXRjaCAocmVzb2x2ZXIudHlwZU5hbWUpIHtcbiAgICAgICAgICBjYXNlICdRdWVyeSc6XG4gICAgICAgICAgY2FzZSAnTXV0YXRpb24nOlxuICAgICAgICAgIGNhc2UgJ1N1YnNjcmlwdGlvbic6XG4gICAgICAgICAgICBkZWxldGUgcmVzb2x2ZXJzW3Jlc29sdmVyLmZpZWxkTmFtZV07XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgdGhpcy5yZXNvbHZlcnMgPSByZXNvbHZlcnM7XG5cbiAgICB0aGlzLm5lc3RlZEFwcHN5bmNTdGFjayA9IG5ldyBOZXN0ZWRTdGFjayh0aGlzLCAnYXBwc3luYy1uZXN0ZWQtc3RhY2snKTtcblxuICAgIC8vIEFwcFN5bmNcbiAgICB0aGlzLmFwcHN5bmNBUEkgPSBuZXcgR3JhcGhxbEFwaSh0aGlzLm5lc3RlZEFwcHN5bmNTdGFjaywgYCR7aWR9LWFwaWAsIHtcbiAgICAgIG5hbWU6IHByb3BzLmFwaU5hbWUgPyBwcm9wcy5hcGlOYW1lIDogYCR7aWR9LWFwaWAsXG4gICAgICBhdXRob3JpemF0aW9uQ29uZmlnOiBwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnXG4gICAgICAgID8gcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZ1xuICAgICAgICA6IGRlZmF1bHRBdXRob3JpemF0aW9uQ29uZmlnLFxuICAgICAgbG9nQ29uZmlnOiB7XG4gICAgICAgIGZpZWxkTG9nTGV2ZWw6IHByb3BzLmZpZWxkTG9nTGV2ZWxcbiAgICAgICAgICA/IHByb3BzLmZpZWxkTG9nTGV2ZWxcbiAgICAgICAgICA6IEZpZWxkTG9nTGV2ZWwuTk9ORSxcbiAgICAgIH0sXG4gICAgICBzY2hlbWE6IFNjaGVtYS5mcm9tQXNzZXQoJy4vYXBwc3luYy9zY2hlbWEuZ3JhcGhxbCcpLFxuICAgICAgeHJheUVuYWJsZWQ6IHByb3BzLnhyYXlFbmFibGVkID8/IGZhbHNlLFxuICAgIH0pO1xuXG4gICAgbGV0IHRhYmxlRGF0YSA9IHRoaXMub3V0cHV0cy5jZGtUYWJsZXMgPz8ge307XG5cbiAgICAvLyBDaGVjayB0byBzZWUgaWYgc3luYyBpcyBlbmFibGVkXG4gICAgaWYgKHRhYmxlRGF0YS5EYXRhU3RvcmUpIHtcbiAgICAgIHRoaXMuaXNTeW5jRW5hYmxlZCA9IHRydWU7XG4gICAgICB0aGlzLnN5bmNUYWJsZSA9IHRoaXMuY3JlYXRlU3luY1RhYmxlKHRhYmxlRGF0YS5EYXRhU3RvcmUpO1xuICAgICAgZGVsZXRlIHRhYmxlRGF0YS5EYXRhU3RvcmU7IC8vIFdlIGRvbid0IHdhbnQgdG8gY3JlYXRlIHRoaXMgYWdhaW4gYmVsb3cgc28gcmVtb3ZlIGl0IGZyb20gdGhlIHRhYmxlRGF0YSBtYXBcbiAgICB9XG5cbiAgICB0aGlzLnRhYmxlTmFtZU1hcCA9IHRoaXMuY3JlYXRlVGFibGVzQW5kUmVzb2x2ZXJzKHRhYmxlRGF0YSwgcmVzb2x2ZXJzKTtcbiAgICBpZiAodGhpcy5vdXRwdXRzLm5vbmVSZXNvbHZlcnMpIHtcbiAgICAgIHRoaXMuY3JlYXRlTm9uZURhdGFTb3VyY2VBbmRSZXNvbHZlcnMoXG4gICAgICAgIHRoaXMub3V0cHV0cy5ub25lUmVzb2x2ZXJzLFxuICAgICAgICByZXNvbHZlcnMsXG4gICAgICApO1xuICAgIH1cbiAgICB0aGlzLmNyZWF0ZUh0dHBSZXNvbHZlcnMoKTtcblxuICAgIC8vIE91dHB1dHMgc28gd2UgY2FuIGdlbmVyYXRlIGV4cG9ydHNcbiAgICBuZXcgQ2ZuT3V0cHV0KHNjb3BlLCAnYXBwc3luY0dyYXBoUUxFbmRwb2ludE91dHB1dCcsIHtcbiAgICAgIHZhbHVlOiB0aGlzLmFwcHN5bmNBUEkuZ3JhcGhxbFVybCxcbiAgICAgIGRlc2NyaXB0aW9uOiAnT3V0cHV0IGZvciBhd3NfYXBwc3luY19ncmFwaHFsRW5kcG9pbnQnLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIGdyYXBocWwtdHJhbnNmb3JtZXItY29yZSBuZWVkcyB0byBiZSBqc2lpIGVuYWJsZWQgdG8gcHVsbCB0aGUgSVRyYW5zZm9ybWVyIGludGVyZmFjZSBjb3JyZWN0bHkuXG4gICAqIFNpbmNlIGl0J3Mgbm90IGluIHBlZXIgZGVwZW5kZW5jaWVzIGl0IGRvZXNuJ3Qgc2hvdyB1cCBpbiB0aGUganNpaSBkZXBzIGxpc3QuXG4gICAqIFNpbmNlIGl0J3Mgbm90IGpzaWkgZW5hYmxlZCBpdCBoYXMgdG8gYmUgYnVuZGxlZC5cbiAgICogVGhlIHBhY2thZ2UgY2FuJ3QgYmUgaW4gQk9USCBwZWVyIGFuZCBidW5kbGVkIGRlcGVuZGVuY2llc1xuICAgKiBTbyB3ZSBkbyBhIGZha2UgdGVzdCB0byBtYWtlIHN1cmUgaXQgaW1wbGVtZW50cyB0aGVzZSBhbmQgaG9wZSBmb3IgdGhlIGJlc3RcbiAgICogQHBhcmFtIHRyYW5zZm9ybWVyXG4gICAqL1xuICBwcml2YXRlIGltcGxlbWVudHNJVHJhbnNmb3JtZXIodHJhbnNmb3JtZXI6IGFueSkge1xuICAgIHJldHVybiAnbmFtZScgaW4gdHJhbnNmb3JtZXJcbiAgICAgICYmICdkaXJlY3RpdmUnIGluIHRyYW5zZm9ybWVyXG4gICAgICAmJiAndHlwZURlZmluaXRpb25zJyBpbiB0cmFuc2Zvcm1lcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIE5PTkUgZGF0YSBzb3VyY2UgYW5kIGFzc29jaWF0ZWQgcmVzb2x2ZXJzXG4gICAqIEBwYXJhbSBub25lUmVzb2x2ZXJzIFRoZSByZXNvbHZlcnMgdGhhdCBiZWxvbmcgdG8gdGhlIG5vbmUgZGF0YSBzb3VyY2VcbiAgICogQHBhcmFtIHJlc29sdmVycyBUaGUgcmVzb2x2ZXIgbWFwIG1pbnVzIGZ1bmN0aW9uIHJlc29sdmVyc1xuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVOb25lRGF0YVNvdXJjZUFuZFJlc29sdmVycyhcbiAgICBub25lUmVzb2x2ZXJzOiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclJlc29sdmVyIH0sXG4gICAgcmVzb2x2ZXJzOiBhbnksXG4gICkge1xuICAgIGNvbnN0IG5vbmVEYXRhU291cmNlID0gdGhpcy5hcHBzeW5jQVBJLmFkZE5vbmVEYXRhU291cmNlKCdOT05FJyk7XG5cbiAgICBPYmplY3Qua2V5cyhub25lUmVzb2x2ZXJzKS5mb3JFYWNoKChyZXNvbHZlcktleTogYW55KSA9PiB7XG4gICAgICBjb25zdCByZXNvbHZlciA9IHJlc29sdmVyc1tyZXNvbHZlcktleV07XG5cbiAgICAgIG5ldyBSZXNvbHZlcihcbiAgICAgICAgdGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2ssXG4gICAgICAgIGAke3Jlc29sdmVyLnR5cGVOYW1lfS0ke3Jlc29sdmVyLmZpZWxkTmFtZX0tcmVzb2x2ZXJgLFxuICAgICAgICB7XG4gICAgICAgICAgYXBpOiB0aGlzLmFwcHN5bmNBUEksXG4gICAgICAgICAgdHlwZU5hbWU6IHJlc29sdmVyLnR5cGVOYW1lLFxuICAgICAgICAgIGZpZWxkTmFtZTogcmVzb2x2ZXIuZmllbGROYW1lLFxuICAgICAgICAgIGRhdGFTb3VyY2U6IG5vbmVEYXRhU291cmNlLFxuICAgICAgICAgIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tRmlsZShcbiAgICAgICAgICAgIHJlc29sdmVyLnJlcXVlc3RNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgKSxcbiAgICAgICAgICByZXNwb25zZU1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21GaWxlKFxuICAgICAgICAgICAgcmVzb2x2ZXIucmVzcG9uc2VNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgKSxcbiAgICAgICAgfSxcbiAgICAgICk7XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBlYWNoIGR5bmFtb2RiIHRhYmxlLCBnc2lzLCBkeW5hbW9kYiBkYXRhc291cmNlLCBhbmQgYXNzb2NpYXRlZCByZXNvbHZlcnNcbiAgICogSWYgc3luYyBpcyBlbmFibGVkIHRoZW4gVFRMIGNvbmZpZ3VyYXRpb24gaXMgYWRkZWRcbiAgICogUmV0dXJucyB0YWJsZU5hbWU6IHRhYmxlIG1hcCBpbiBjYXNlIGl0IGlzIG5lZWRlZCBmb3IgbGFtYmRhIGZ1bmN0aW9ucywgZXRjXG4gICAqIEBwYXJhbSB0YWJsZURhdGEgVGhlIENka1RyYW5zZm9ybWVyIHRhYmxlIGluZm9ybWF0aW9uXG4gICAqIEBwYXJhbSByZXNvbHZlcnMgVGhlIHJlc29sdmVyIG1hcCBtaW51cyBmdW5jdGlvbiByZXNvbHZlcnNcbiAgICovXG4gIHByaXZhdGUgY3JlYXRlVGFibGVzQW5kUmVzb2x2ZXJzKFxuICAgIHRhYmxlRGF0YTogeyBbbmFtZTogc3RyaW5nXTogQ2RrVHJhbnNmb3JtZXJUYWJsZSB9LFxuICAgIHJlc29sdmVyczogYW55LFxuICApOiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfSB7XG4gICAgY29uc3QgdGFibGVOYW1lTWFwOiBhbnkgPSB7fTtcblxuICAgIE9iamVjdC5rZXlzKHRhYmxlRGF0YSkuZm9yRWFjaCgodGFibGVLZXk6IGFueSkgPT4ge1xuICAgICAgY29uc3QgdGFibGUgPSB0aGlzLmNyZWF0ZVRhYmxlKHRhYmxlRGF0YVt0YWJsZUtleV0pO1xuICAgICAgY29uc3QgZGF0YVNvdXJjZSA9IHRoaXMuYXBwc3luY0FQSS5hZGREeW5hbW9EYkRhdGFTb3VyY2UodGFibGVLZXksIHRhYmxlKTtcblxuICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtYXBwc3luYy1kYXRhc291cmNlLWRlbHRhc3luY2NvbmZpZy5odG1sXG5cbiAgICAgIGlmICh0aGlzLmlzU3luY0VuYWJsZWQgJiYgdGhpcy5zeW5jVGFibGUpIHtcbiAgICAgICAgLy9AdHMtaWdub3JlIC0gZHMgaXMgdGhlIGJhc2UgQ2ZuRGF0YVNvdXJjZSBhbmQgdGhlIGRiIGNvbmZpZyBuZWVkcyB0byBiZSB2ZXJzaW9uZWQgLSBzZWUgQ2ZuRGF0YVNvdXJjZVxuICAgICAgICBkYXRhU291cmNlLmRzLmR5bmFtb0RiQ29uZmlnLnZlcnNpb25lZCA9IHRydWU7XG5cbiAgICAgICAgLy9AdHMtaWdub3JlIC0gZHMgaXMgdGhlIGJhc2UgQ2ZuRGF0YVNvdXJjZSAtIHNlZSBDZm5EYXRhU291cmNlXG4gICAgICAgIGRhdGFTb3VyY2UuZHMuZHluYW1vRGJDb25maWcuZGVsdGFTeW5jQ29uZmlnID0ge1xuICAgICAgICAgIGJhc2VUYWJsZVR0bDogJzQzMjAwJywgLy8gR290IHRoaXMgdmFsdWUgZnJvbSBhbXBsaWZ5IC0gMzAgZGF5cyBpbiBtaW51dGVzXG4gICAgICAgICAgZGVsdGFTeW5jVGFibGVOYW1lOiB0aGlzLnN5bmNUYWJsZS50YWJsZU5hbWUsXG4gICAgICAgICAgZGVsdGFTeW5jVGFibGVUdGw6ICczMCcsIC8vIEdvdCB0aGlzIHZhbHVlIGZyb20gYW1wbGlmeSAtIDMwIG1pbnV0ZXNcbiAgICAgICAgfTtcblxuICAgICAgICAvLyBOZWVkIHRvIGFkZCBwZXJtaXNzaW9uIGZvciBvdXIgZGF0YXNvdXJjZSBzZXJ2aWNlIHJvbGUgdG8gYWNjZXNzIHRoZSBzeW5jIHRhYmxlXG4gICAgICAgIGRhdGFTb3VyY2UuZ3JhbnRQcmluY2lwYWwuYWRkVG9Qb2xpY3koXG4gICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgJ2R5bmFtb2RiOionLCAvLyBUT0RPOiBUaGlzIG1heSBiZSB0b28gcGVybWlzc2l2ZVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHJlc291cmNlczogW3RoaXMuc3luY1RhYmxlLnRhYmxlQXJuXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgZHluYW1vRGJDb25maWcgPSBkYXRhU291cmNlLmRzXG4gICAgICAgIC5keW5hbW9EYkNvbmZpZyBhcyBDZm5EYXRhU291cmNlLkR5bmFtb0RCQ29uZmlnUHJvcGVydHk7XG4gICAgICB0YWJsZU5hbWVNYXBbdGFibGVLZXldID0gZHluYW1vRGJDb25maWcudGFibGVOYW1lO1xuXG4gICAgICAvLyBMb29wIHRoZSBiYXNpYyByZXNvbHZlcnNcbiAgICAgIHRhYmxlRGF0YVt0YWJsZUtleV0ucmVzb2x2ZXJzLmZvckVhY2goKHJlc29sdmVyS2V5OiBhbnkpID0+IHtcbiAgICAgICAgbGV0IHJlc29sdmVyID0gcmVzb2x2ZXJzW3Jlc29sdmVyS2V5XTtcbiAgICAgICAgbmV3IFJlc29sdmVyKFxuICAgICAgICAgIHRoaXMubmVzdGVkQXBwc3luY1N0YWNrLFxuICAgICAgICAgIGAke3Jlc29sdmVyLnR5cGVOYW1lfS0ke3Jlc29sdmVyLmZpZWxkTmFtZX0tcmVzb2x2ZXJgLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGFwaTogdGhpcy5hcHBzeW5jQVBJLFxuICAgICAgICAgICAgdHlwZU5hbWU6IHJlc29sdmVyLnR5cGVOYW1lLFxuICAgICAgICAgICAgZmllbGROYW1lOiByZXNvbHZlci5maWVsZE5hbWUsXG4gICAgICAgICAgICBkYXRhU291cmNlOiBkYXRhU291cmNlLFxuICAgICAgICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21GaWxlKFxuICAgICAgICAgICAgICByZXNvbHZlci5yZXF1ZXN0TWFwcGluZ1RlbXBsYXRlLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIHJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbUZpbGUoXG4gICAgICAgICAgICAgIHJlc29sdmVyLnJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICB9LFxuICAgICAgICApO1xuICAgICAgfSk7XG5cbiAgICAgIC8vIExvb3AgdGhlIGdzaSByZXNvbHZlcnNcbiAgICAgIHRhYmxlRGF0YVt0YWJsZUtleV0uZ3NpUmVzb2x2ZXJzLmZvckVhY2goKHJlc29sdmVyS2V5OiBhbnkpID0+IHtcbiAgICAgICAgbGV0IHJlc29sdmVyID0gcmVzb2x2ZXJzLmdzaVtyZXNvbHZlcktleV07XG4gICAgICAgIG5ldyBSZXNvbHZlcihcbiAgICAgICAgICB0aGlzLm5lc3RlZEFwcHN5bmNTdGFjayxcbiAgICAgICAgICBgJHtyZXNvbHZlci50eXBlTmFtZX0tJHtyZXNvbHZlci5maWVsZE5hbWV9LXJlc29sdmVyYCxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBhcGk6IHRoaXMuYXBwc3luY0FQSSxcbiAgICAgICAgICAgIHR5cGVOYW1lOiByZXNvbHZlci50eXBlTmFtZSxcbiAgICAgICAgICAgIGZpZWxkTmFtZTogcmVzb2x2ZXIuZmllbGROYW1lLFxuICAgICAgICAgICAgZGF0YVNvdXJjZTogZGF0YVNvdXJjZSxcbiAgICAgICAgICAgIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tRmlsZShcbiAgICAgICAgICAgICAgcmVzb2x2ZXIucmVxdWVzdE1hcHBpbmdUZW1wbGF0ZSxcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICByZXNwb25zZU1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21GaWxlKFxuICAgICAgICAgICAgICByZXNvbHZlci5yZXNwb25zZU1hcHBpbmdUZW1wbGF0ZSxcbiAgICAgICAgICAgICksXG4gICAgICAgICAgfSxcbiAgICAgICAgKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIHRhYmxlTmFtZU1hcDtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlVGFibGUodGFibGVEYXRhOiBDZGtUcmFuc2Zvcm1lclRhYmxlKSB7XG4gICAgbGV0IHRhYmxlUHJvcHM6IGFueSA9IHtcbiAgICAgIGJpbGxpbmdNb2RlOiBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1QsXG4gICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgbmFtZTogdGFibGVEYXRhLnBhcnRpdGlvbktleS5uYW1lLFxuICAgICAgICB0eXBlOiB0aGlzLmNvbnZlcnRBdHRyaWJ1dGVUeXBlKHRhYmxlRGF0YS5wYXJ0aXRpb25LZXkudHlwZSksXG4gICAgICB9LFxuICAgIH07XG5cbiAgICBpZiAodGFibGVEYXRhLnNvcnRLZXkgJiYgdGFibGVEYXRhLnNvcnRLZXkubmFtZSkge1xuICAgICAgdGFibGVQcm9wcy5zb3J0S2V5ID0ge1xuICAgICAgICBuYW1lOiB0YWJsZURhdGEuc29ydEtleS5uYW1lLFxuICAgICAgICB0eXBlOiB0aGlzLmNvbnZlcnRBdHRyaWJ1dGVUeXBlKHRhYmxlRGF0YS5zb3J0S2V5LnR5cGUpLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAodGFibGVEYXRhLnR0bCAmJiB0YWJsZURhdGEudHRsLmVuYWJsZWQpIHtcbiAgICAgIHRhYmxlUHJvcHMudGltZVRvTGl2ZUF0dHJpYnV0ZSA9IHRhYmxlRGF0YS50dGwuYXR0cmlidXRlTmFtZTtcbiAgICB9XG5cbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZShcbiAgICAgIHRoaXMubmVzdGVkQXBwc3luY1N0YWNrLFxuICAgICAgdGFibGVEYXRhLnRhYmxlTmFtZSxcbiAgICAgIHRhYmxlUHJvcHMsXG4gICAgKTtcblxuICAgIGlmIChcbiAgICAgIHRhYmxlRGF0YS5nbG9iYWxTZWNvbmRhcnlJbmRleGVzICYmXG4gICAgICB0YWJsZURhdGEuZ2xvYmFsU2Vjb25kYXJ5SW5kZXhlcy5sZW5ndGggPiAwXG4gICAgKSB7XG4gICAgICB0YWJsZURhdGEuZ2xvYmFsU2Vjb25kYXJ5SW5kZXhlcy5mb3JFYWNoKChnc2k6IGFueSkgPT4ge1xuICAgICAgICB0YWJsZS5hZGRHbG9iYWxTZWNvbmRhcnlJbmRleCh7XG4gICAgICAgICAgaW5kZXhOYW1lOiBnc2kuaW5kZXhOYW1lLFxuICAgICAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICAgICAgbmFtZTogZ3NpLnBhcnRpdGlvbktleS5uYW1lLFxuICAgICAgICAgICAgdHlwZTogdGhpcy5jb252ZXJ0QXR0cmlidXRlVHlwZShnc2kucGFydGl0aW9uS2V5LnR5cGUpLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgcHJvamVjdGlvblR5cGU6IHRoaXMuY29udmVydFByb2plY3Rpb25UeXBlKFxuICAgICAgICAgICAgZ3NpLnByb2plY3Rpb24uUHJvamVjdGlvblR5cGUsXG4gICAgICAgICAgKSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGFibGU7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyB0aGUgc3luYyB0YWJsZSBmb3IgQW1wbGlmeSBEYXRhU3RvcmVcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL2NvbmZsaWN0LWRldGVjdGlvbi1hbmQtc3luYy5odG1sXG4gICAqIEBwYXJhbSB0YWJsZURhdGEgVGhlIENka1RyYW5zZm9ybWVyIHRhYmxlIGluZm9ybWF0aW9uXG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZVN5bmNUYWJsZSh0YWJsZURhdGE6IENka1RyYW5zZm9ybWVyVGFibGUpOiBUYWJsZSB7XG4gICAgcmV0dXJuIG5ldyBUYWJsZSh0aGlzLCAnYXBwc3luYy1hcGktc3luYy10YWJsZScsIHtcbiAgICAgIGJpbGxpbmdNb2RlOiBCaWxsaW5nTW9kZS5QQVlfUEVSX1JFUVVFU1QsXG4gICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgbmFtZTogdGFibGVEYXRhLnBhcnRpdGlvbktleS5uYW1lLFxuICAgICAgICB0eXBlOiB0aGlzLmNvbnZlcnRBdHRyaWJ1dGVUeXBlKHRhYmxlRGF0YS5wYXJ0aXRpb25LZXkudHlwZSksXG4gICAgICB9LFxuICAgICAgc29ydEtleToge1xuICAgICAgICBuYW1lOiB0YWJsZURhdGEuc29ydEtleSEubmFtZSwgLy8gV2Uga25vdyBpdCBoYXMgYSBzb3J0a2V5IGJlY2F1c2Ugd2UgZm9yY2VkIGl0IHRvXG4gICAgICAgIHR5cGU6IHRoaXMuY29udmVydEF0dHJpYnV0ZVR5cGUodGFibGVEYXRhLnNvcnRLZXkhLnR5cGUpLCAvLyBXZSBrbm93IGl0IGhhcyBhIHNvcnRrZXkgYmVjYXVzZSB3ZSBmb3JjZWQgaXQgdG9cbiAgICAgIH0sXG4gICAgICB0aW1lVG9MaXZlQXR0cmlidXRlOiB0YWJsZURhdGEudHRsPy5hdHRyaWJ1dGVOYW1lIHx8ICdfdHRsJyxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY29udmVydEF0dHJpYnV0ZVR5cGUodHlwZTogc3RyaW5nKTogQXR0cmlidXRlVHlwZSB7XG4gICAgc3dpdGNoICh0eXBlKSB7XG4gICAgICBjYXNlICdOJzpcbiAgICAgICAgcmV0dXJuIEF0dHJpYnV0ZVR5cGUuTlVNQkVSO1xuICAgICAgY2FzZSAnQic6XG4gICAgICAgIHJldHVybiBBdHRyaWJ1dGVUeXBlLkJJTkFSWTtcbiAgICAgIGNhc2UgJ1MnOiAvLyBTYW1lIGFzIGRlZmF1bHRcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHJldHVybiBBdHRyaWJ1dGVUeXBlLlNUUklORztcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGNvbnZlcnRQcm9qZWN0aW9uVHlwZSh0eXBlOiBzdHJpbmcpOiBQcm9qZWN0aW9uVHlwZSB7XG4gICAgc3dpdGNoICh0eXBlKSB7XG4gICAgICBjYXNlICdJTkNMVURFJzpcbiAgICAgICAgcmV0dXJuIFByb2plY3Rpb25UeXBlLklOQ0xVREU7XG4gICAgICBjYXNlICdLRVlTX09OTFknOlxuICAgICAgICByZXR1cm4gUHJvamVjdGlvblR5cGUuS0VZU19PTkxZO1xuICAgICAgY2FzZSAnQUxMJzogLy8gU2FtZSBhcyBkZWZhdWx0XG4gICAgICBkZWZhdWx0OlxuICAgICAgICByZXR1cm4gUHJvamVjdGlvblR5cGUuQUxMO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlSHR0cFJlc29sdmVycygpIHtcbiAgICBmb3IgKGNvbnN0IFtlbmRwb2ludCwgaHR0cFJlc29sdmVyc10gb2YgT2JqZWN0LmVudHJpZXMoXG4gICAgICB0aGlzLmh0dHBSZXNvbHZlcnMsXG4gICAgKSkge1xuICAgICAgY29uc3Qgc3RyaXBwZWRFbmRwb2ludCA9IGVuZHBvaW50LnJlcGxhY2UoL1teXzAtOUEtWmEtel0vZywgJycpO1xuICAgICAgY29uc3QgaHR0cERhdGFTb3VyY2UgPSB0aGlzLmFwcHN5bmNBUEkuYWRkSHR0cERhdGFTb3VyY2UoXG4gICAgICAgIGAke3N0cmlwcGVkRW5kcG9pbnR9YCxcbiAgICAgICAgZW5kcG9pbnQsXG4gICAgICApO1xuXG4gICAgICBodHRwUmVzb2x2ZXJzLmZvckVhY2goKHJlc29sdmVyOiBDZGtUcmFuc2Zvcm1lckh0dHBSZXNvbHZlcikgPT4ge1xuICAgICAgICBuZXcgUmVzb2x2ZXIoXG4gICAgICAgICAgdGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2ssXG4gICAgICAgICAgYCR7cmVzb2x2ZXIudHlwZU5hbWV9LSR7cmVzb2x2ZXIuZmllbGROYW1lfS1yZXNvbHZlcmAsXG4gICAgICAgICAge1xuICAgICAgICAgICAgYXBpOiB0aGlzLmFwcHN5bmNBUEksXG4gICAgICAgICAgICB0eXBlTmFtZTogcmVzb2x2ZXIudHlwZU5hbWUsXG4gICAgICAgICAgICBmaWVsZE5hbWU6IHJlc29sdmVyLmZpZWxkTmFtZSxcbiAgICAgICAgICAgIGRhdGFTb3VyY2U6IGh0dHBEYXRhU291cmNlLFxuICAgICAgICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICAgICAgIHJlc29sdmVyLmRlZmF1bHRSZXF1ZXN0TWFwcGluZ1RlbXBsYXRlLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIHJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbVN0cmluZyhcbiAgICAgICAgICAgICAgcmVzb2x2ZXIuZGVmYXVsdFJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICB9LFxuICAgICAgICApO1xuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkTGFtYmRhRGF0YVNvdXJjZUFuZFJlc29sdmVycyhcbiAgICBmdW5jdGlvbk5hbWU6IHN0cmluZyxcbiAgICBpZDogc3RyaW5nLFxuICAgIGxhbWJkYUZ1bmN0aW9uOiBJRnVuY3Rpb24sXG4gICAgb3B0aW9ucz86IERhdGFTb3VyY2VPcHRpb25zLFxuICApOiBMYW1iZGFEYXRhU291cmNlIHtcbiAgICBjb25zdCBmdW5jdGlvbkRhdGFTb3VyY2UgPSB0aGlzLmFwcHN5bmNBUEkuYWRkTGFtYmRhRGF0YVNvdXJjZShcbiAgICAgIGlkLFxuICAgICAgbGFtYmRhRnVuY3Rpb24sXG4gICAgICBvcHRpb25zLFxuICAgICk7XG5cbiAgICBmb3IgKGNvbnN0IHJlc29sdmVyIG9mIHRoaXMuZnVuY3Rpb25SZXNvbHZlcnNbZnVuY3Rpb25OYW1lXSkge1xuICAgICAgbmV3IFJlc29sdmVyKFxuICAgICAgICB0aGlzLm5lc3RlZEFwcHN5bmNTdGFjayxcbiAgICAgICAgYCR7cmVzb2x2ZXIudHlwZU5hbWV9LSR7cmVzb2x2ZXIuZmllbGROYW1lfS1yZXNvbHZlcmAsXG4gICAgICAgIHtcbiAgICAgICAgICBhcGk6IHRoaXMuYXBwc3luY0FQSSxcbiAgICAgICAgICB0eXBlTmFtZTogcmVzb2x2ZXIudHlwZU5hbWUsXG4gICAgICAgICAgZmllbGROYW1lOiByZXNvbHZlci5maWVsZE5hbWUsXG4gICAgICAgICAgZGF0YVNvdXJjZTogZnVuY3Rpb25EYXRhU291cmNlLFxuICAgICAgICAgIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tU3RyaW5nKFxuICAgICAgICAgICAgcmVzb2x2ZXIuZGVmYXVsdFJlcXVlc3RNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgKSxcbiAgICAgICAgICByZXNwb25zZU1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICAgICByZXNvbHZlci5kZWZhdWx0UmVzcG9uc2VNYXBwaW5nVGVtcGxhdGUsXG4gICAgICAgICAgKSwgLy8gVGhpcyBkZWZhdWx0cyB0byBhbGxvdyBlcnJvcnMgdG8gcmV0dXJuIHRvIHRoZSBjbGllbnQgaW5zdGVhZCBvZiB0aHJvd2luZ1xuICAgICAgICB9LFxuICAgICAgKTtcbiAgICB9XG5cbiAgICByZXR1cm4gZnVuY3Rpb25EYXRhU291cmNlO1xuICB9XG59XG4iXX0=