"""Compute the efeatures generated by the released l5pc model"""

"""
Copyright (c) 2016-2022, EPFL/Blue Brain Project

 This file is part of BluePyOpt <https://github.com/BlueBrain/BluePyOpt>

 This library is free software; you can redistribute it and/or modify it under
 the terms of the GNU Lesser General Public License version 3.0 as published
 by the Free Software Foundation.

 This library is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 details.

 You should have received a copy of the GNU Lesser General Public License
 along with this library; if not, write to the Free Software Foundation, Inc.,
 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
"""

import json
import numpy

import bluepyopt.ephys as ephys
from bluepyopt.ephys.extra_features_utils import all_1D_features

from l5pc_lfpy_evaluator import create, get_recording_names, get_feature_name, extra_kwargs, config_dir

release_params = {
    'gNaTs2_tbar_NaTs2_t.apical': 0.026145,
    'gSKv3_1bar_SKv3_1.apical': 0.004226,
    'gImbar_Im.apical': 0.000143,
    'gNaTa_tbar_NaTa_t.axonal': 3.137968,
    'gK_Tstbar_K_Tst.axonal': 0.089259,
    'gamma_CaDynamics_E2.axonal': 0.002910,
    'gNap_Et2bar_Nap_Et2.axonal': 0.006827,
    'gSK_E2bar_SK_E2.axonal': 0.007104,
    'gCa_HVAbar_Ca_HVA.axonal': 0.000990,
    'gK_Pstbar_K_Pst.axonal': 0.973538,
    'gSKv3_1bar_SKv3_1.axonal': 1.021945,
    'decay_CaDynamics_E2.axonal': 287.198731,
    'gCa_LVAstbar_Ca_LVAst.axonal': 0.008752,
    'gamma_CaDynamics_E2.somatic': 0.000609,
    'gSKv3_1bar_SKv3_1.somatic': 0.303472,
    'gSK_E2bar_SK_E2.somatic': 0.008407,
    'gCa_HVAbar_Ca_HVA.somatic': 0.000994,
    'gNaTs2_tbar_NaTs2_t.somatic': 0.983955,
    'decay_CaDynamics_E2.somatic': 210.485284,
    'gCa_LVAstbar_Ca_LVAst.somatic': 0.000333
}


class NumpyEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(
            obj,
            (
                numpy.int_,
                numpy.intc,
                numpy.intp,
                numpy.int8,
                numpy.int16,
                numpy.int32,
                numpy.int64,
                numpy.uint8,
                numpy.uint16,
                numpy.uint32,
                numpy.uint64,
            ),
        ):
            return int(obj)
        elif isinstance(
            obj, (numpy.float_, numpy.float16, numpy.float32, numpy.float64)
        ):
            return float(obj)
        elif isinstance(obj, numpy.ndarray):
            return obj.tolist()
        return json.JSONEncoder.default(self, obj)


def dict_to_json(data, path):
    """Save some data in a json file."""
    with open(path, "w") as f:
        json.dump(data, f, indent=2, cls=NumpyEncoder)


def add_extra_objectives(evaluator):

    threshold = -20

    for protocol_name in evaluator.fitness_protocols:

        location = 'MEA'
        recording_names = get_recording_names(protocol_name, location)
        somatic_recording_name = get_recording_names(protocol_name, "soma")[""]

        stimulus = evaluator.fitness_protocols[protocol_name].stimuli[0]

        args = {
            "stim_start": stimulus.step_delay, 
            "stim_end": stimulus.step_delay + stimulus.step_duration,
            "recording_names": recording_names,
            "threshold": threshold
        }

        for feature in all_1D_features:

            feature_name = get_feature_name(protocol_name, location, feature)

            feature = ephys.efeatures.extraFELFeature(
                name=feature_name,
                extrafel_feature_name=feature,
                somatic_recording_name=somatic_recording_name,
                exp_mean=0.,
                exp_std=0.1,
                channel_ids=None,
                **args,
                **extra_kwargs
            )

            objective = ephys.objectives.SingletonObjective(
                feature_name, feature
            )

            evaluator.fitness_calculator.objectives.append(objective)


def save_extra_efeatures(efeature_values):
    
    path = "./extra_features.json"
    data = {}
    
    for feature_name in efeature_values:
        
        protocol_name, location, feature  = feature_name.split('.')

        if protocol_name not in data:
            data[protocol_name] = {}

        if location not in data[protocol_name]:
            data[protocol_name][location] = {}

        mean = efeature_values[feature_name]
        std = abs(0.2 * mean)

        data[protocol_name][location][feature] = [mean, std]

    dict_to_json(data, path)


if __name__ == "__main__":

    # Create evaluator
    evaluator = create(config_dir / "features.json")

    # Add extra features to the evaluator
    add_extra_objectives(evaluator)

    # Compute features value
    efeature_values = evaluator.evaluate_with_dicts(param_dict=release_params, target='values')

    # Save feature values in a json
    save_extra_efeatures(efeature_values)
