/**
 * Timeago is a jQuery plugin that makes it easy to support automatically
 * updating fuzzy timestamps (e.g. "4 minutes ago" or "about 1 day ago").
 *
 * @name timeago
 * @version 0.11.1
 * @requires jQuery v1.2.3+
 * @author Ryan McGeary
 * @license MIT License - http://www.opensource.org/licenses/mit-license.php
 *
 * For usage and examples, visit:
 * http://timeago.yarp.com/
 *
 * Copyright (c) 2008-2011, Ryan McGeary (ryanonjavascript -[at]- mcgeary [*dot*] org)
 */
(function($) {
    $.timeago = function(timestamp) {
        if (timestamp instanceof Date) {
            return inWords(timestamp);
        } else if (typeof timestamp === "string") {
            return inWords($.timeago.parse(timestamp));
        } else {
            return inWords($.timeago.datetime(timestamp));
        }
    };
    var $t = $.timeago;

    $.extend($.timeago, {
        settings: {
            refreshMillis: 60000
        },
        inWords: function(dateSince) {
            return inWords(dateSince);
        },
        parse: function(iso8601) {
            var s = $.trim(iso8601);
            s = s.replace(/\.\d\d\d+/, ""); // remove milliseconds
            s = s.replace(/-/, "/").replace(/-/, "/");
            s = s.replace(/T/, " ").replace(/Z/, " UTC");
            s = s.replace(/([\+\-]\d\d)\:?(\d\d)/, " $1$2"); // -04:00 -> -0400
            return new Date(s);
        },
        datetime: function(elem) {
            // jQuery's `is()` doesn't play well with HTML5 in IE
            var isTime =
                $(elem)
                    .get(0)
                    .tagName.toLowerCase() === "time"; // $(elem).is("time");
            var iso8601 = isTime ? $(elem).attr("datetime") : $(elem).attr("title");
            return $t.parse(iso8601);
        }
    });

    $.fn.timeago = function() {
        var self = this;
        self.each(refresh);

        var $s = $t.settings;
        if ($s.refreshMillis > 0) {
            setInterval(function() {
                self.each(refresh);
            }, $s.refreshMillis);
        }
        return self;
    };

    function refresh() {
        var data = prepareData(this);
        if (!isNaN(data.datetime)) {
            $(this)
                .text(inWords(data.datetime))
                .fadeIn();
        }
        return this;
    }

    function prepareData(element) {
        element = $(element);
        if (!element.data("timeago")) {
            element.data("timeago", { datetime: $t.datetime(element) });
            var text = $.trim(element.text());
            if (text.length > 0) {
                element.attr("title", text);
            }
        }
        return element.data("timeago");
    }

    function inWords(date) {
        var distanceMillis = distance(date);
        var seconds = Math.abs(distanceMillis) / 1000;
        var minutes = seconds / 60;
        var hours = minutes / 60;
        var days = hours / 24;
        var years = days / 365;
        var months = [
            gettext("Jan"),
            gettext("Feb"),
            gettext("Mar"),
            gettext("Apr"),
            gettext("May"),
            gettext("Jun"),
            gettext("Jul"),
            gettext("Aug"),
            gettext("Sep"),
            gettext("Oct"),
            gettext("Nov"),
            gettext("Dec")
        ];
        //todo: rewrite this in javascript
        if (days > 2) {
            var month_date = months[date.getMonth()] + " " + date.getDate();
            if (years == 0) {
                //how to do this in js???
                return month_date;
            } else {
                var wholeYears = Math.floor(years);
                return interpolate(ngettext("%s year ago", "%s years ago", wholeYears), [wholeYears]);
            }
        } else if (days == 2) {
            return gettext("2 days ago");
        } else if (days == 1) {
            return gettext("yesterday");
        } else if (minutes >= 60) {
            var wholeHours = Math.floor(hours);
            return interpolate(ngettext("%s hour ago", "%s hours ago", wholeHours), [wholeHours]);
        } else if (seconds > 90) {
            var wholeMinutes = Math.floor(minutes);
            return interpolate(ngettext("%s min ago", "%s mins ago", wholeMinutes), [wholeMinutes]);
        } else {
            return gettext("just now");
        }
    }

    function distance(date) {
        return new Date() - date;
    }

    // fix for IE6 suckage
    document.createElement("abbr");
    document.createElement("time");
})(jQuery);
//run timeago
$("abbr.timeago").timeago();
