# Copyright 2023 Portmod Authors
# Distributed under the terms of the GNU General Public License v3

import os
import shutil
from subprocess import check_call
from tempfile import TemporaryDirectory
from typing import List, Tuple

import tomlkit
from portmod.prompt import prompt_num_multi
from prompt_toolkit import prompt
from prompt_toolkit.validation import ValidationError, Validator

from .datadir import InstallDir, find_data_dirs, find_esp_bsa, get_dominant_texture_size
from .names import parse_name


# FIXME: Multiple numbers?
class NumberValidator(Validator):
    def __init__(self, min_value: int, max_value: int):
        self.max_value = max_value
        self.min_value = min_value
        super().__init__()

    def validate(self, document):
        text = document.text

        if text and not text.isdigit():
            i = 0

            # Get index of first non numeric character.
            # We want to move the cursor here.
            for i, c in enumerate(text):
                if not c.isdigit():
                    break

            raise ValidationError(
                message="This input contains non-numeric characters", cursor_position=i
            )
        if text and int(text) > self.max_value:
            raise ValidationError(
                message=f"The input can be no larger than {self.max_value}"
            )
        if text and int(text) < self.min_value:
            raise ValidationError(
                message=f"The input can be no smaller than {self.min_value}"
            )


def unpack(file: str, output_dir: str):
    print(f"Unpacking {file} to {output_dir}")
    name, ext = os.path.splitext(file)
    if ext in (".7z", ".rar"):
        if shutil.which("7z"):
            check_call(["7z", "x", "-y", "-o" + output_dir, file])
        else:
            raise RuntimeError("Cannot extract 7z or rar files if 7z is not installed!")
    else:
        shutil.unpack_archive(file, output_dir)


def create_configtool_toml():
    pass


def install_files_to_directory(directory: str, destination: str):
    print(f"Installing {directory} to {destination}")
    shutil.move(directory, destination)


def config_template(name: str, output_dir: str) -> tomlkit.TOMLDocument:
    from tomlkit import comment, document, nl, string

    doc = document()
    doc.add(
        comment("This is a configtool configuration file, generated by importmod_core")
    )
    doc.add(nl())
    doc.add("id", string(f"local/{name}"))
    doc.add("path", string(output_dir))

    doc.add(
        comment(
            "You can override other data directories by listing their id in the overrides field"
        )
    )
    doc.add(comment("overrides = []"))
    doc.add(nl())
    for line in """Plugins and archives have the following fields:
    name = "Plugin filename.esp"
    overrides = ["Plugin this should override.esp"]""".splitlines():
        doc.add(comment(line))
    doc.add(
        comment(
            "You can also add groundcover files in the same manner, but importmod currently does not detect these"
        )
    )

    doc.add(comment("groundcover = []"))
    return doc


def select_data_dirs(root_path: str) -> Tuple[List[InstallDir], List[str], List[str]]:
    data_dirs = sorted(find_data_dirs(root_path), key=lambda d: str(d.PATH))
    if not data_dirs:
        print("No data directories were found!")
    esps = {}
    bsas = {}
    texture_size = {}
    for directory in data_dirs:
        path = os.path.join(root_path, directory.PATH)
        esps[directory.PATH], bsas[directory.PATH] = find_esp_bsa(path)
        texture_size[directory.PATH] = get_dominant_texture_size(path)

    print()
    print("Detected the following data directories")
    for index, directory in enumerate(data_dirs):
        print(f" [{index}]\tPath:    \t", directory.PATH)
        if esps.get(directory.PATH):
            print("\tDetected ESPs:\t", ", ".join(esps[directory.PATH]))
        if bsas.get(directory.PATH):
            print("\tDetected BSAs:\t", ", ".join(bsas[directory.PATH]))
        if texture_size[directory.PATH]:
            print("\tTexture size:\t", texture_size[directory.PATH])

    if os.environ.get("INTERACTIVE"):
        to_add = prompt_num_multi(
            "Which data directories would you like to install?", len(data_dirs)
        )

        result_dirs = [data_dirs[index] for index in to_add]
    else:
        result_dirs = data_dirs
    print(esps, bsas)
    return (
        result_dirs,
        [esp for idir in result_dirs for esp in esps[idir.PATH]],
        [bsa for idir in result_dirs for bsa in bsas[idir.PATH]],
    )


def install_from_archive(args):
    # This installer is always interactive
    os.environ["INTERACTIVE"] = "1"

    root = args.prefix_root or os.environ["PORTMOD_ROOT"]
    if not args.output_dir:
        name, _ = os.path.splitext(os.path.basename(args.file))
        if name.endswith(".tar"):
            name, _ = os.path.splitext(name)
        name = parse_name(name)
        name = prompt(
            "What is the mod called? This will be used in the output directory name: ",
            default=name,
        )
    output_dir = args.output_dir or os.path.join(root, "pkg", "local", name)
    metadata: tomlkit.TOMLDocument = config_template(name, output_dir)
    plugins = tomlkit.array()
    archives = tomlkit.array()

    with TemporaryDirectory() as tempdir:
        unpack(args.file, tempdir)

        directories, esps, bsas = select_data_dirs(tempdir)
        installed = set()

        def install_directory(path: str):
            if path in installed:
                return

            # Make sure that subdirectories get installed before their parents
            for other_dir in directories:
                if (
                    path != other_dir.PATH
                    and os.path.commonpath([path, other_dir.PATH]) == path
                ):
                    install_directory(other_dir.PATH)

            install_files_to_directory(
                os.path.normpath(os.path.join(tempdir, path)), output_dir
            )
            installed.add(path)

        for directory in directories:
            install_directory(directory.PATH)
        for esp in esps:
            plugins.append({"name": esp})
        for bsa in bsas:
            archives.append({"name": bsa})

    tomlpath = os.path.join(
        root,
        os.environ.get("CONFIGPATH", "etc"),
        "configtool",
        "importmod",
        name + ".toml",
    )
    os.makedirs(os.path.dirname(tomlpath), exist_ok=True)
    print(f"Creating configuration file at path {tomlpath}")
    metadata["plugins"] = plugins
    metadata["archives"] = archives
    with open(tomlpath, "w") as file:
        file.write(tomlkit.dumps(metadata))


def add_install_parser(subparsers):
    parser = subparsers.add_parser(
        "install", help="Install configtool-supported mods from archives"
    )
    parser.add_argument(
        "file",
        metavar="FILE",
        help="Mod archive to install",
    )
    parser.add_argument(
        "-o",
        "--output-dir",
        help="Optional output directory for the installed data directory",
    )
    parser.add_argument(
        "-r",
        "--prefix-root",
        help="Optional prefix, required if the PORTMOD_ROOT environment variable is not set",
    )

    parser.set_defaults(func=install_from_archive)
