"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.removeDependenciesFromPackageJson = exports.addDependenciesToPackageJson = void 0;
const json_1 = require("nx/src/generators/utils/json");
const install_packages_task_1 = require("../tasks/install-packages-task");
const semver_1 = require("semver");
const NON_SEMVER_TAGS = {
    '*': 2,
    next: 1,
    latest: 0,
    previous: -1,
    legacy: -2,
};
function filterExistingDependencies(dependencies, existingAltDependencies) {
    if (!existingAltDependencies) {
        return dependencies;
    }
    return Object.keys(dependencies !== null && dependencies !== void 0 ? dependencies : {})
        .filter((d) => !existingAltDependencies[d])
        .reduce((acc, d) => (Object.assign(Object.assign({}, acc), { [d]: dependencies[d] })), {});
}
function isIncomingVersionGreater(incomingVersion, existingVersion) {
    if (incomingVersion in NON_SEMVER_TAGS &&
        existingVersion in NON_SEMVER_TAGS) {
        return NON_SEMVER_TAGS[incomingVersion] > NON_SEMVER_TAGS[existingVersion];
    }
    if (incomingVersion in NON_SEMVER_TAGS ||
        existingVersion in NON_SEMVER_TAGS) {
        return true;
    }
    return (0, semver_1.gt)((0, semver_1.coerce)(incomingVersion), (0, semver_1.coerce)(existingVersion));
}
function updateExistingDependenciesVersion(dependencies, existingAltDependencies) {
    return Object.keys(existingAltDependencies || {})
        .filter((d) => {
        if (!dependencies[d]) {
            return false;
        }
        const incomingVersion = dependencies[d];
        const existingVersion = existingAltDependencies[d];
        return isIncomingVersionGreater(incomingVersion, existingVersion);
    })
        .reduce((acc, d) => (Object.assign(Object.assign({}, acc), { [d]: dependencies[d] })), {});
}
/**
 * Add Dependencies and Dev Dependencies to package.json
 *
 * For example:
 * ```typescript
 * addDependenciesToPackageJson(tree, { react: 'latest' }, { jest: 'latest' })
 * ```
 * This will **add** `react` and `jest` to the dependencies and devDependencies sections of package.json respectively.
 *
 * @param tree Tree representing file system to modify
 * @param dependencies Dependencies to be added to the dependencies section of package.json
 * @param devDependencies Dependencies to be added to the devDependencies section of package.json
 * @param packageJsonPath Path to package.json
 * @returns Callback to install dependencies only if necessary, no-op otherwise
 */
function addDependenciesToPackageJson(tree, dependencies, devDependencies, packageJsonPath = 'package.json') {
    const currentPackageJson = (0, json_1.readJson)(tree, packageJsonPath);
    let filteredDependencies = filterExistingDependencies(dependencies, currentPackageJson.devDependencies);
    let filteredDevDependencies = filterExistingDependencies(devDependencies, currentPackageJson.dependencies);
    filteredDependencies = Object.assign(Object.assign({}, filteredDependencies), updateExistingDependenciesVersion(devDependencies, currentPackageJson.dependencies));
    filteredDevDependencies = Object.assign(Object.assign({}, filteredDevDependencies), updateExistingDependenciesVersion(dependencies, currentPackageJson.devDependencies));
    if (requiresAddingOfPackages(currentPackageJson, filteredDependencies, filteredDevDependencies)) {
        (0, json_1.updateJson)(tree, packageJsonPath, (json) => {
            json.dependencies = Object.assign(Object.assign({}, (json.dependencies || {})), filteredDependencies);
            json.devDependencies = Object.assign(Object.assign({}, (json.devDependencies || {})), filteredDevDependencies);
            json.dependencies = sortObjectByKeys(json.dependencies);
            json.devDependencies = sortObjectByKeys(json.devDependencies);
            return json;
        });
        return () => {
            (0, install_packages_task_1.installPackagesTask)(tree);
        };
    }
    return () => { };
}
exports.addDependenciesToPackageJson = addDependenciesToPackageJson;
/**
 * Remove Dependencies and Dev Dependencies from package.json
 *
 * For example:
 * ```typescript
 * removeDependenciesFromPackageJson(tree, ['react'], ['jest'])
 * ```
 * This will **remove** `react` and `jest` from the dependencies and devDependencies sections of package.json respectively.
 *
 * @param dependencies Dependencies to be removed from the dependencies section of package.json
 * @param devDependencies Dependencies to be removed from the devDependencies section of package.json
 * @returns Callback to uninstall dependencies only if necessary. undefined is returned if changes are not necessary.
 */
function removeDependenciesFromPackageJson(tree, dependencies, devDependencies, packageJsonPath = 'package.json') {
    const currentPackageJson = (0, json_1.readJson)(tree, packageJsonPath);
    if (requiresRemovingOfPackages(currentPackageJson, dependencies, devDependencies)) {
        (0, json_1.updateJson)(tree, packageJsonPath, (json) => {
            for (const dep of dependencies) {
                delete json.dependencies[dep];
            }
            for (const devDep of devDependencies) {
                delete json.devDependencies[devDep];
            }
            json.dependencies = sortObjectByKeys(json.dependencies);
            json.devDependencies = sortObjectByKeys(json.devDependencies);
            return json;
        });
    }
    return () => {
        (0, install_packages_task_1.installPackagesTask)(tree);
    };
}
exports.removeDependenciesFromPackageJson = removeDependenciesFromPackageJson;
function sortObjectByKeys(obj) {
    if (!obj || typeof obj !== 'object' || Array.isArray(obj)) {
        return obj;
    }
    return Object.keys(obj)
        .sort()
        .reduce((result, key) => {
        return Object.assign(Object.assign({}, result), { [key]: obj[key] });
    }, {});
}
/**
 * Verifies whether the given packageJson dependencies require an update
 * given the deps & devDeps passed in
 */
function requiresAddingOfPackages(packageJsonFile, deps, devDeps) {
    let needsDepsUpdate = false;
    let needsDevDepsUpdate = false;
    packageJsonFile.dependencies = packageJsonFile.dependencies || {};
    packageJsonFile.devDependencies = packageJsonFile.devDependencies || {};
    if (Object.keys(deps).length > 0) {
        needsDepsUpdate = Object.keys(deps).some((entry) => {
            const incomingVersion = deps[entry];
            if (packageJsonFile.dependencies[entry]) {
                const existingVersion = packageJsonFile.dependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            if (packageJsonFile.devDependencies[entry]) {
                const existingVersion = packageJsonFile.devDependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            return true;
        });
    }
    if (Object.keys(devDeps).length > 0) {
        needsDevDepsUpdate = Object.keys(devDeps).some((entry) => {
            const incomingVersion = devDeps[entry];
            if (packageJsonFile.devDependencies[entry]) {
                const existingVersion = packageJsonFile.devDependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            if (packageJsonFile.dependencies[entry]) {
                const existingVersion = packageJsonFile.dependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            return true;
        });
    }
    return needsDepsUpdate || needsDevDepsUpdate;
}
/**
 * Verifies whether the given packageJson dependencies require an update
 * given the deps & devDeps passed in
 */
function requiresRemovingOfPackages(packageJsonFile, deps, devDeps) {
    let needsDepsUpdate = false;
    let needsDevDepsUpdate = false;
    packageJsonFile.dependencies = packageJsonFile.dependencies || {};
    packageJsonFile.devDependencies = packageJsonFile.devDependencies || {};
    if (deps.length > 0) {
        needsDepsUpdate = deps.some((entry) => packageJsonFile.dependencies[entry]);
    }
    if (devDeps.length > 0) {
        needsDevDepsUpdate = devDeps.some((entry) => packageJsonFile.devDependencies[entry]);
    }
    return needsDepsUpdate || needsDevDepsUpdate;
}
//# sourceMappingURL=package-json.js.map