"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts"],
            },
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return this.instantiationOrderSubProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        const subProjectPackages = [];
        this.subProjects.forEach((subProject) => {
            if (isNodeProject(subProject)) {
                const subNodeProject = subProject;
                subProjectPackages.push(subNodeProject.package);
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => { };
            }
        });
        super.synth();
        // Force workspace install deps if any node subproject package has change, unless the workspace changed
        if (
        // @ts-ignore - `file` is private
        this.package.file.changed !== true &&
            // @ts-ignore - `file` is private
            subProjectPackages.find((pkg) => pkg.file.changed === true)) {
            // @ts-ignore - `installDependencies` is private
            this.package.installDependencies();
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.disableNoHoistBundled !== true) {
            const noHoistBundled = this.subProjects.flatMap((sub) => {
                if (sub instanceof javascript_1.NodeProject) {
                    return sub.deps.all
                        .filter((dep) => dep.type === projen_1.DependencyType.BUNDLED)
                        .flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.workspacePackages,
                nohoist: noHoist,
            });
        }
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`npx nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject", version: "0.12.37" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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