"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cr = require("@aws-cdk/custom-resources");
const path = require("path");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
/**
 * Represents the k3sCluster construct.
 */
class Cluster extends cdk.Construct {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        // VPC configuration
        const vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file 
        const k3sBucket = new s3.Bucket(this, 'k3sBucket', {
            removalPolicy: (_b = props.bucketRemovalPolicy) !== null && _b !== void 0 ? _b : cdk.RemovalPolicy.RETAIN,
        });
        // Delete S3 Object CustomResource
        if (props.bucketRemovalPolicy === cdk.RemovalPolicy.DESTROY) {
            const onEvent = new lambda.Function(this, 'onEventHandler', {
                runtime: lambda.Runtime.PYTHON_3_8,
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-handler')),
                handler: 'index.on_event',
            });
            const deleteS3ObjectProvider = new cr.Provider(this, 'deleteS3ObjectProvider', {
                onEventHandler: onEvent,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            const CRdeleteS3ObjectProvider = new cdk.CustomResource(this, 'CRdeleteS3ObjectProvider', {
                serviceToken: deleteS3ObjectProvider.serviceToken,
                properties: {
                    Bucket: k3sBucket.bucketName,
                },
            });
            CRdeleteS3ObjectProvider.node.addDependency(k3sBucket);
            k3sBucket.grantDelete(onEvent);
            k3sBucket.grantReadWrite(onEvent);
        }
        // control plane node Security Group      
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', { vpc });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group      
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', { vpc });
        // for this prototype the workers are being placed in a public subnet 
        // ideally they should land on a private subnet 
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only 
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used 
        this.controlPlaneInstanceType = (_c = props.controlPlaneInstanceType) !== null && _c !== void 0 ? _c : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_d = props.workerInstanceType) !== null && _d !== void 0 ? _d : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create launch template for worker ASG
        // prepare the userData
        const userData = ec2.UserData.forLinux();
        userData.addCommands(`
          #!/bin/bash
          LOGFILE='/var/log/k3s.log'
          curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
          chmod +x k3s
          echo the bucket name is ${k3sBucket.bucketName} 
          aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
          (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
          --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        const lt = new ec2.CfnLaunchTemplate(this, 'WorkerLaunchTemplate', {
            launchTemplateData: {
                imageId: new AmiProvider().amiId.getImage(this).imageId,
                instanceType: this.workerInstanceType.toString(),
                instanceMarketOptions: {
                    marketType: props.spotWorkerNodes ? 'spot' : undefined,
                    spotOptions: props.spotWorkerNodes ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
            },
        });
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_e = props.workerMinCapacity) !== null && _e !== void 0 ? _e : 3,
        });
        const cfnAsg = workerAsg.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64.
 */
class AmiProvider {
    /**
     *
     */
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
/**
 * The VPC provider to create or import the VPC.
 */
class VpcProvider {
    /**
     *
     */
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
//# sourceMappingURL=data:application/json;base64,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