##
# File: NmrDpUtility.py
# Date: 26-Sep-2019
#
# Updates:
# 10-Oct-2019  M. Yokochi - add 'check_mandatory_tag' option to detect missing mandatory tags as errors
# 15-Oct-2019  M. Yokochi - revise criteria on discrepancy in distance restraints using normalized value
# 01-Nov-2019  M. Yokochi - revise error message, instead of Python ValueError message.
# 05-Nov-2019  M. Yokochi - revise error messages and detect empty sequence information.
# 28-Nov-2019  M. Yokochi - fix saveframe name of nef_molecular_system and add 'nmr-str2nef-deposit' workflow operation
# 29-Nov-2019  M. Yokochi - relax allowable range of weight values in restraint data and support index pointer in auxiliary loops
# 11-Dec-2019  M. Yokochi - fix internal errors while processing NMR-VTF/PDBStat_examples and NMR-VTF/BMRB
# 24-Jan-2020  M. Yokochi - add histogram of distance restraints per residue and distance restraints on contact map
# 27-Jan-2020  M. Yokochi - add contact map for inter-chain distance restraints
# 28-Jan-2020  M. Yokochi - add struct_conf and struct_sheet_range data in dp report
# 29-Jan-2020  M. Yokochi - change plot type of dihedral angle and RDC restraints per residue
# 05-Feb-2020  M. Yokochi - add 'circular-shift' and 'void-zero' constraint for dihedral angle restraint
# 05-Feb-2020  M. Yokochi - move conflicted_data error to warning
# 07-Feb-2020  M. Yokochi - replace 'number_of_potential_types' by 'potential_type_of_constraints' in dp report
# 07-Feb-2020  M. Yokochi - allow multiple values in a data type on per residue plots
# 13-Feb-2020  M. Yokochi - add 'number_of_constraints_per_polymer_type' for apilayer.postModifyNMRRestraint
# 14-Feb-2020  M. Yokochi - add 'spectram_dim' for apilayer.postModifyNMRPeaks
# 21-Feb-2020  M. Yokochi - update content-type definitions and add release mode (nmr-str2nef-release workflow operation)
# 02-Mar-2020  M. Yokochi - add 'nmr-cs-nef-consistency-check' and 'nmr-cs-str-consistency-check' workflow operation (DAOTHER-4515)
# 05-Mar-2020  M. Yokochi - revise warning message (disordered_index) and enumerations (DAOTHER-5485)
# 06-Mar-2020  M. Yokochi - fix invalid ambiguity_code while parsing
# 13-Mar-2020  M. Yokochi - revise error/warning messages
# 17-Mar-2020  M. Yokochi - add 'undefined' value for potential_type (DAOTHER-5508)
# 17-Mar-2020  M. Yokochi - revise warning message about enumeration mismatch for potential_type and restraint_origin (DAOTHER-5508)
# 17-Mar-2020  M. Yokochi - check total number of models (DAOTHER-436)
# 17-Mar-2020  M. Yokochi - check consistency between saveframe name and sf_framecode value
# 18-Mar-2020  M. Yokochi - rename warning type from skipped_sf/lp_category to skipped_saveframe/loop_category
# 18-Mar-2020  M. Yokochi - support 'Saveframe' data type as separated NMR data (DAOTHER-2737)
# 19-Mar-2020  M. Yokochi - atom nomenclature should not become a blocker (DAOTHER-5527)
# 24-Mar-2020  M. Yokochi - add support for chemical shift reference (DAOTHER-1682)
# 24-Mar-2020  M. Yokochi - revise chain assignment for identical dimer case (DAOTHER-3343)
# 30-Mar-2020  M. Yokochi - preserve original sf_framecode for nef_molecular_system (NEF) or assembly (NMR-STAR)
# 31-Mar-2020  M. Yokochi - enable processing without log file
# 03-Apr-2020  M. Yokochi - preserve case code of atom_name (NEF) and Auth_atom_ID/Original_PDB_atom_name (NMR-STAR)
# 06-Apr-2020  M. Yokochi - synchronize with coordinates' auth_asym_id and auth_seq_id for combined NMR-STAR deposition
# 10-Apr-2020  M. Yokochi - fix crash in case of format issue
# 14-Apr-2020  M. Yokochi - fix dependency on label_seq_id, instead of using auth_seq_id in case (DAOTHER-5584)
# 18-Apr-2020  M. Yokochi - fix no model error in coordinate and allow missing 'sf_framecode' in NMR separated deposition (DAOTHER-5594)
# 19-Apr-2020  M. Yokochi - support concatenated CS data in NMR separated deposition (DAOTHER-5594)
# 19-Apr-2020  M. Yokochi - report warning against not superimposed models (DAOTHER-4060)
# 22-Apr-2020  M. Yokochi - convert comp_id in capital letters (DAOTHER-5600)
# 22-Apr-2020  M. Yokochi - fix GLY:HA1/HA2 to GLY:HA2/HA3 (DAOTHER-5600)
# 22-Apr-2020  M. Yokochi - fix ambiguity code mismatch if possible (DAOTHER-5601)
# 22-Apr-2020  M. Yokochi - fix None type object is not iterable error (DAOTHER-5602)
# 23-Apr-2020  M. Yokochi - support conventional atom name for methyl group without wildcard character, e.g. ALA:HB (DAOTHER-5603)
# 23-Apr-2020  M. Yokochi - change missing ambiguity_set_id error to warning (DAOTHER-5609)
# 23-Apr-2020  M. Yokochi - make sure to parse chem_shift_ref saveframe tag (DAOTHER-5610)
# 23-Apr-2020  M. Yokochi - implement automatic format correction (DAOTHER-5603, 5610)
# 24-Apr-2020  M. Yokochi - separate format_issue error and missing_mandatory_content error (DAOTHER-5611)
# 24-Apr-2020  M. Yokochi - support 'QR' pseudo atom name (DAOTHER-5611)
# 24-Apr-2020  M. Yokochi - allow mandatory value is missing in NMR separated deposition (DAOTHER-5611)
# 25-Apr-2020  M. Yokochi - implement automatic format correction for 6NZN, 6PQF, 6PSI entry (DAOTHE-5611)
# 25-Apr-2020  M. Yokochi - add 'entity' content subtype (DAOTHER-5611)
# 25-Apr-2020  M. Yokochi - add 'corrected_format_issue' warning type (DAOTHER-5611)
# 27-Apr-2020  M. Yokochi - add 'auth_atom_nomenclature_mismatch' warning type (DAOTHER-5611)
# 27-Apr-2020  M. Yokochi - implement recursive format corrections (DAOTHER-5602)
# 28-Apr-2020  M. Yokochi - copy the normalized CS/MR files if output file path list is set (DAOTHER-5611)
# 28-Apr-2020  M. Yokochi - catch 'range-float' error as 'unusual data' warning (DAOTHER-5611)
# 28-Apr-2020  M. Yokochi - extract sequence from CS/MR loop with gap (DAOTHER-5611)
# 29-Apr-2020  M. Yokochi - support diagnostic message of PyNMRSTAR v2.6.5.1 or later (DAOTHER-5611)
# 29-Apr-2020  M. Yokochi - implement more automatic format corrections with PyNMRSTAR v2.6.5.1 (DAOTHER-5611)
# 29-Apr-2020  M. Yokochi - fix different CS warning between NEF and NMR-STAR (DAOTHER-5621)
# 29-Apr-2020  M. Yokochi - add 'number_of_constraint_sets' of experiment data in report (DAOTHER-5622)
# 29-Apr-2020  M. Yokochi - sort 'conflicted_data' and 'inconsistent_data' warning items (DAOTHER-5622)
# 30-Apr-2020  M. Yokochi - allow NMR conventional atom naming scheme in NMR-STAR V3.2 (DAOTHER-5634)
# 01-May-2020  M. Yokochi - allow NMR conventional atom naming scheme in NMR-STAR V3.2 (DAOTHER-5634)
# 02-May-2020  M. Yokochi - additional support for format issue correction while STAR to NEF conversion (DAOTHER-5577)
# 02-May-2020  M. Yokochi - re-implement basic mathematical functions using Numpy library
# 07-May-2020  M. Yokochi - revise warning type (from 'insuffcient_data' to 'encouragement') if total number of models is less than 8 (DAOTHER-5650)
# 07-May-2020  M. Yokochi - add preventive code for infinite loop while format issue correction
# 08-May-2020  M. Yokochi - sync update with wwpdb.utils.nmr.CifReader (DAOTHER-5654)
# 09-May-2020  M. Yokochi - add support for submitted coordinate file (allow missing of pdbx_poly_seq_scheme) (DAOTHER-5654)
# 12-May-2020  M. Yokochi - fix diselenide bond detection
# 14-May-2020  M. Yokochi - fix error detection for missing mandatory content (DAOTHER-5681 and 5682)
# 15-May-2020  M. Yokochi - add 'content_mismatch' error for NMR legacy deposition (DAOTHER-5687)
# 15-May-2020  M. Yokochi - revise encouragement message if total number of models is less than 5 (DAOTHER-5650)
# 16-May-2020  M. Yokochi - block NEF file upload in NMR legacy deposition (DAOTHER-5687)
# 30-May-2020  M. Yokochi - refer to atom_site to get total number of models (DAOTHER-5650)
# 01-Jun-2020  M. Yokochi - let RMSD cutoff value configurable (DAOTHER-4060)
# 05-Jun-2020  M. Yokochi - be compatible with wwpdb.utils.align.alignlib using Python 3 (DAOTHER-5766)
# 06-Jun-2020  M. Yokochi - be compatible with pynmrstar v3 (DAOTHER-5765)
# 12-Jun-2020  M. Yokochi - overall performance improvement by reusing cached data and code revision
# 19-Jun-2020  M. Yokochi - do not generate invalid restraints include self atom
# 26-Jun-2020  M. Yokochi - add support for covalent bond information (_nef_covalent_links and _Bond categories)
# 30-Jun-2020  M. Yokochi - ignore third party loops and items gracefully (DAOTHER-5896)
# 30-Jun-2020  M. Yokochi - prevent pynmrstar's exception due to empty string (DAOTHER-5894)
# 08-Jul-2020  M. Yokochi - bug fix release for DAOTHER-5926
# 09-Jul-2020  M. Yokochi - add support for categories in NMR-STAR specific peak list (DAOTHER-5926)
# 09-Jul-2020  M. Yokochi - adjust arguments of pynmrstar write_to_file() to prevent data losses (v2.6.1, DAOTHER-5926)
# 17-Aug-2020  M. Yokochi - add support for residue variant (DAOTHER-5906)
# 20-Aug-2020  M. Yokochi - add 'leave_intl_note' output parameter decides whether to leave internal commentary note in processed NMR-STAR file, set False for
#                           OneDep environment (DAOTHER-6030)
# 10-Sep-2020  M. Yokochi - add 'transl_pseudo_name' input parameter decides whether to translate conventional pseudo atom nomenclature in combined NMR-STAR file (DAOTHER-6128)
# 16-Sep-2020  M. Yokochi - bug fix release based on internal test using BMRB NMR restraint archive of 6.3k entries (DAOTHER-6128)
# 18-Sep-2020  M. Yokochi - bug fix release for negative sequence numbers (DAOTHER-6128)
# 25-Sep-2020  M. Yokochi - add 'tolerant_seq_align' input parameter which enables tolerant sequence alignment for residue variant, set False for OneDep environment (DAOTHER-6128)
# 09-Oct-2020  M. Yokochi - support circular chain_id re-mapping with seq_id shifts in data loops if it is necessary, 'tolerant_seq_align' input parameter is required (DAOTHER-6128)
# 22-Oct-2020  M. Yokochi - run diagnostic routine for case of sequence mismatch between defined polymer sequence and sequence in data loop (DAOTHER-6128)
# 11-Nov-2020  M. Yokochi - set NEF v1.1 as the default specification
# 12-Nov-2020  M. Yokochi - improve NMR warning messages (DAOTHER-6109, 6167)
# 18-Nov-2020  M. Yokochi - fix calculation of CS completeness, fix empty polymer_sequence_in_loop due to atom_site.pdbx_PDB_ins_code (DAOTHER-6128)
# 20-Nov-2020  M. Yokochi - rename 'remarkable_data' warning category to 'unusual/rare_data' (DAOTHER-6372)
# 26-Nov-2020  M. Yokochi - detect the nearest ferromagnetic atom, in addition to paramagnetic atom (DAOTHER-6366)
# 27-Nov-2020  M. Yokochi - add support for non-IUPAC atom names for standard amino acids, i.e. ARG:HB1/HB2 -> HB2/HB3 (DAOTHER-6373)
# 17-Dec-2020  M. Yokochi - support 'atom_not_found' error with message revision (DAOTHER-6345)
# 25-Jan-2021  M. Yokochi - simplify code for Entity_assemble_ID and chain_code
# 25-Jan-2021  M. Yokochi - add CS validation code about rotameric state of ILE/LEU/VAL residue
# 03-Feb-2021  M. Yokochi - update polymer sequence which shares the same entity and missing in the molecular assembly information if necessary,
#                           i.e. double stranded DNA (DAOTHER-6128, BMRB entry: 16812, PDB ID: 6kae)
# 10-Mar-2021  M. Yokochi - block NEF deposition missing '_nef_sequence' category and turn off salvage routine for the case (DAOTHER-6694)
# 10-Mar-2021  M. Yokochi - add support for audit loop in NEF (DAOTHER-6327)
# 12-Mar-2021  M. Yokochi - add diagnostic routine to fix inconsistent sf_framecode of conventional CS file (DAOTHER-6693)
# 14-May-2021  M. Yokochi - add support for PyNMRSTAR v3.1.1 (DAOTHER-6693)
# 20-May-2021  M. Yokochi - fix duplicating pynmrstar data objects during format issue correction that leads to empty upload summary page (DAOTHER-6834)
# 24-May-2021  M. Yokochi - fix tautomer detection of coordinate (DAOTHER-6809)
# 17-Jun-2021  M. Yokochi - fix error in handling lower/upper linear limits (DAOTHER-6963)
# 17-Jun-2021  M. Yokochi - relax tolerance on chemical shift difference (DAOTHER-6963)
# 23-Jun-2021  M. Yokochi - send back the initial error message when format remediation fails (DAOTHER-6830)
# 25-Jun-2021  M. Yokochi - block restraint files that have no distance restraints (DAOTHER-6830)
# 28-Jun-2021  M. Yokochi - support cif-formatted CS file for reupload without changing CS data (DAOTHER-6830, 7097)
# 29-Jun-2021  M. Yokochi - include auth_asym_id in NMR data processing report (DAOTHER-7108)
# 29-Jun-2021  M. Yokochi - add support for PyNMRSTAR v3.2.0 (DAOTHER-7107)
# 02-Jul-2021  M. Yokochi - detect content type of AMBER restraint file and AMBER auxiliary file (DAOTHER-6830, 1901)
# 12-Jul-2021  M. Yokochi - add RCI validation code for graphical representation of NMR data
# 24-Aug-2021  M. Yokochi - detect content type of XPLOR-NIH planarity restraints (DAOTHER-7265)
# 10-Sep-2021  M. Yokochi - prevent system crash for an empty loop case of CS/MR data (D_1292117593)
# 13-Oct-2021  M. Yokochi - fix/adjust tolerances for spectral peak list (DAOTHER-7389, issue #1 and #2)
# 13-Oct-2021  M. Yokochi - code revision according to PEP8 using Pylint (DAOTHER-7389, issue #5)
# 14-Oct-2021  M. Yokochi - remove unassigned chemical shifts, clear incompletely assigned spectral peaks (DAOTHER-7389, issue #3)
# 27-Oct-2021  M. Yokochi - fix collection of unmapped sequences and utilize Auth_asym_ID* tag for chain_id if Entity_assembly_ID* is not available (DAOTHER-7421)
# 28-Oct-2021  M. Yokochi - resolve case-insensitive saveframe name collision for CIF (DAOTHER-7389, issue #4)
# 16-Nov-2021  M. Yokochi - fix sequence conflict in case that large sequence gap in CS implies multi chain complex (DAOTHER-7465)
# 16-Nov-2021  M. Yokochi - fix server crash with disulfide bond, which is not supported by chemical shifts (DAOTHER-7475)
# 16-Nov-2021  M. Yokochi - revised error message for malformed XPLOR-NIH RDC restraints (DAOTHER-7478)
# 18-Nov-2021  M. Yokochi - detect content type of XPLOR-NIH hydrogen bond geometry restraints (DAOTHER-7478)
# 18-Nov-2021  M. Yokochi - relax detection of distance restraints for nm-res-cya and nm-res-oth (DAOTHER-7491)
# 13-Dec-2021  M. Yokochi - append sequence spacer between large gap to prevent failure of sequence alignment (DAOTHER-7465, issue #2)
# 15-Dec-2021  M. Yokochi - fix server crash while uploading NMR restraint file in NMR-STAR format (DAOTHER-7545)
##
""" Wrapper class for NMR data processing.
    @author: Masashi Yokochi
"""
import sys
import os
import os.path
import pynmrstar
import itertools
import copy
import collections
import re
import math
import codecs
import shutil
import time
import hashlib

from packaging import version
from munkres import Munkres
import numpy as np

from wwpdb.utils.nmr.NEFTranslator.NEFTranslator import NEFTranslator
from wwpdb.utils.nmr.NmrDpReport import NmrDpReport
from wwpdb.utils.align.alignlib import PairwiseAlign  # pylint: disable=no-name-in-module
from wwpdb.utils.nmr.BMRBChemShiftStat import BMRBChemShiftStat
from wwpdb.utils.config.ConfigInfo import getSiteId
from wwpdb.utils.config.ConfigInfoApp import ConfigInfoAppCommon
from wwpdb.utils.nmr.io.ChemCompIo import ChemCompReader
from wwpdb.utils.nmr.io.CifReader import CifReader
from wwpdb.utils.nmr.rci.RCI import RCI
from wwpdb.utils.nmr.CifToNmrStar import CifToNmrStar
from wwpdb.utils.nmr.NmrStarToCif import NmrStarToCif

__pynmrstar_v3_3__ = version.parse(pynmrstar.__version__) >= version.parse("3.3.0")
__pynmrstar_v3_2__ = version.parse(pynmrstar.__version__) >= version.parse("3.2.0")
__pynmrstar_v3_1__ = version.parse(pynmrstar.__version__) >= version.parse("3.1.0")
__pynmrstar_v3__ = version.parse(pynmrstar.__version__) >= version.parse("3.0.0")


def detect_bom(in_file, default='utf-8'):
    """ Detect BOM of input file.
    """

    with open(in_file, 'rb') as f:
        raw = f.read(4)

    for enc, boms in \
            ('utf-8-sig', (codecs.BOM_UTF8,)),\
            ('utf-16', (codecs.BOM_UTF16_LE, codecs.BOM_UTF16_BE)),\
            ('utf-32', (codecs.BOM_UTF32_LE, codecs.BOM_UTF32_BE)):
        if any(raw.startswith(bom) for bom in boms):
            return enc

    return default


def convert_codec(in_file, out_file, in_codec='utf-8', out_codec='utf-8'):
    """ Convert codec of input file.
    """

    with open(in_file, 'rb') as ifp:
        with open(out_file, 'w+b') as ofp:
            contents = ifp.read()
            ofp.write(contents.decode(in_codec).encode(out_codec))

            ofp.close()

        ifp.close()


def has_key_value(d=None, key=None):
    """ Return whether a given dictionary has effective value for a key.
        @return: True if d[key] has effective value, False otherwise
    """

    if d is None or key is None:
        return False

    if key in d:
        return not d[key] is None

    return False


def get_lp_tag(lp_data, tags):
    """ Return the selected loop tags by row as a list of lists.
    """

    return lp_data.get_tag(tags) if __pynmrstar_v3__ else lp_data.get_data_by_tag(tags)


def get_first_sf_tag(sf_data=None, tag=None):
    """ Return the first value of a given saveframe tag.
        @return: The first tag value, empty string otherwise.
    """

    if sf_data is None or tag is None:
        return ''

    array = sf_data.get_tag(tag)

    if len(array) == 0:
        return ''

    return array[0]


def has_large_seq_gap(s1, s2):
    """ Return whether large gap in sequence ID.
    """

    seq_ids = sorted(set(s1['seq_id']) | set(s2['seq_id']))

    for lp, i in enumerate(seq_ids):
        if lp > 0 and i - seq_ids[lp - 1] > 20:
            return True

    return False


def fill_blank_comp_id(s1, s2):
    """ Fill blanked comp ID in s2 against s1.
    """

    seq_ids = sorted(set(s1['seq_id']) | set(s2['seq_id']))
    comp_ids = []

    for i in seq_ids:
        if i in s2['seq_id']:
            j = s2['seq_id'].index(i)
            if j < len(s2['comp_id']):
                comp_ids.append(s2['comp_id'][j])
            else:
                comp_ids.append('.')
        else:
            comp_ids.append('.')

    return {'chain_id': s2['chain_id'], 'seq_id': seq_ids, 'comp_id': comp_ids}


def fill_blank_comp_id_with_offset(s, offset):
    """ Fill blanked comp ID with offset.
    """

    seq_ids = list(range(s['seq_id'][0] - offset, s['seq_id'][-1] + 1))
    comp_ids = []

    for i in seq_ids:
        if i in s['seq_id']:
            j = s['seq_id'].index(i)
            if j < len(s['comp_id']):
                comp_ids.append(s['comp_id'][j])
            else:
                comp_ids.append('.')
        else:
            comp_ids.append('.')

    return {'chain_id': s['chain_id'], 'seq_id': seq_ids, 'comp_id': comp_ids}

def beutify_seq_id(s1, s2):
    """ Truncate negative sequence IDs of s1 and s2 and insert spacing between the large gap.
    """

    if s1['seq_id'] != s2['seq_id']:
        return s1, s2

    _seq_id = [seq_id for seq_id in s1['seq_id'] if seq_id > 0]
    _comp_id_1 = [comp_id for seq_id, comp_id in zip(s1['seq_id'], s1['comp_id']) if seq_id > 0]
    _comp_id_2 = [comp_id for seq_id, comp_id in zip(s1['seq_id'], s2['comp_id']) if seq_id > 0]

    gap_seq_id = []
    gap_index = []

    len_spacer = 5  # DAOTHER-7465, issue #2

    for lp, i in enumerate(_seq_id):
        if lp > 0 and i - _seq_id[lp - 1] > 20:
            j = _seq_id[lp - 1]
            for sp in range(1, len_spacer + 1):
                gap_seq_id.append(j + sp)
                gap_seq_id.append(i - sp)
            gap_index.append(lp)

    if len(gap_seq_id) == 0:
        return {'chain_id': s1['chain_id'], 'seq_id': _seq_id, 'comp_id': _comp_id_1}, {'chain_id': s2['chain_id'], 'seq_id': _seq_id, 'comp_id': _comp_id_2}

    _seq_id.extend(gap_seq_id)
    _seq_id.sort()

    for lp in reversed(gap_index):
        for sp in range(1, len_spacer + 1):
            _comp_id_1.insert(lp, '.')
            _comp_id_2.insert(lp, '.')

    return {'chain_id': s1['chain_id'], 'seq_id': _seq_id, 'comp_id': _comp_id_1}, {'chain_id': s2['chain_id'], 'seq_id': _seq_id, 'comp_id': _comp_id_2}


def get_middle_code(ref_seq, test_seq):
    """ Return array of middle code of sequence alignment.
    """

    array = ''

    for i, rs in enumerate(ref_seq):
        if i < len(test_seq):
            array += '|' if rs == test_seq[i] and rs != '.' else ' '
        else:
            array += ' '

    return array


def get_gauge_code(seq_id, offset=0):
    """ Return gauge code for seq ID.
    """

    if offset > 0:
        _seq_id = list(range(1, offset + 1))
        for lp, sid in enumerate(seq_id):
            if lp < offset:
                continue
            _seq_id.append(sid)
        seq_id = _seq_id

    sid_len = len([sid for sid in seq_id if sid is not None])
    code_len = 0

    chars = []

    for sid in seq_id:

        if sid is None:
            chars.append('-')
            continue

        if sid >= 0 and sid % 10 == 0 and code_len == 0:

            code = str(sid)
            code_len = len(code)

            for j in range(code_len):
                chars.append(code[j])

        if code_len > 0:
            code_len -= 1
        else:
            chars.append('-')

    for lp, sid in enumerate(seq_id):

        if sid is None or sid % 10 != 0:
            continue

        code = str(sid)
        code_len = len(code)

        if lp - code_len > 0:
            for j in range(code_len):
                chars[lp + j - code_len + 1] = chars[lp + j]
                chars[lp + j] = '-'

    if offset > 0:
        for lp in range(offset):
            chars[lp] = '-'

    array = ''.join(chars)

    if sid_len == len(seq_id):
        return array[:sid_len]

    _sid_len = len(seq_id)

    offset = 0
    for lp, sid in enumerate(seq_id):

        if sid is None:
            p = lp + offset
            array = array[0:p] + ' ' + array[p:]
            offset += 1

    return array[:_sid_len]


def probability_density(value, mean, stddev):
    """ Return probability density.
    """

    stddev2 = stddev ** 2.0

    return math.exp(-((value - mean) ** 2.0) / (2.0 * stddev2)) / math.sqrt(2.0 * math.pi * stddev2)


def predict_redox_state_of_cystein(ca_chem_shift, cb_chem_shift):
    """ Return prediction of redox state of Cystein using assigned CA, CB chemical shifts.
        @return: probability of oxidized state, probability of reduced state
        Reference:
          13C NMR chemical shifts can predict disulfide bond formation.
          Sharma, D., Rajarathnam, K.
          J Biomol NMR 18, 165–171 (2000).
          DOI: 10.1023/A:1008398416292
    """

    oxi_ca = {'avr': 55.5, 'std': 2.5}
    oxi_cb = {'avr': 40.7, 'std': 3.8}

    red_ca = {'avr': 59.3, 'std': 3.2}
    red_cb = {'avr': 28.3, 'std': 2.2}

    oxi = 1.0
    red = 1.0

    if ca_chem_shift is not None:
        oxi *= probability_density(ca_chem_shift, oxi_ca['avr'], oxi_ca['std'])
        red *= probability_density(ca_chem_shift, red_ca['avr'], red_ca['std'])

    if cb_chem_shift is not None:
        if cb_chem_shift < 32.0:
            oxi = 0.0
        else:
            oxi *= probability_density(cb_chem_shift, oxi_cb['avr'], oxi_cb['std'])
        if cb_chem_shift > 35.0:
            red = 0.0
        else:
            red *= probability_density(cb_chem_shift, red_cb['avr'], red_cb['std'])

    total = oxi + red

    if total in (0.0, 2.0):
        return 0.0, 0.0

    return oxi / total, red / total


def predict_cis_trans_peptide_of_proline(cb_chem_shift, cg_chem_shift):
    """ Return prediction of cis-trans peptide bond of Proline using assigned CB, CG chemical shifts.
        @return: probability of cis-peptide bond, probability of trans-peptide bond
        Reference:
          A software tool for the prediction of Xaa-Pro peptide bond conformations in proteins based on 13C chemical shift statistics.
          Schubert, M., Labudde, D., Oschkinat, H. et al.
          J Biomol NMR 24, 149–154 (2002)
          DOI: 10.1023/A:1020997118364
    """

    cis_cb = {'avr': 34.16, 'std': 1.15, 'max': 36.23, 'min': 30.74}
    cis_cg = {'avr': 24.52, 'std': 1.09, 'max': 27.01, 'min': 22.10}
    cis_dl = {'avr': 9.64, 'std': 1.27}

    trs_cb = {'avr': 31.75, 'std': 0.98, 'max': 35.83, 'min': 26.30}
    trs_cg = {'avr': 27.26, 'std': 1.05, 'max': 33.39, 'min': 19.31}
    trs_dl = {'avr': 4.51, 'std': 1.17}

    cis = 1.0
    trs = 1.0

    if cb_chem_shift is not None:
        if cb_chem_shift < cis_cb['min'] - cis_cb['std'] or cb_chem_shift > cis_cb['max'] + cis_cb['std']:
            cis = 0.0
        else:
            cis *= probability_density(cb_chem_shift, cis_cb['avr'], cis_cb['std'])
        if cb_chem_shift < trs_cb['min'] - trs_cb['std'] or cb_chem_shift > trs_cb['max'] + trs_cb['std']:
            trs = 0.0
        else:
            trs *= probability_density(cb_chem_shift, trs_cb['avr'], trs_cb['std'])

    if cg_chem_shift is not None:
        if cg_chem_shift < cis_cg['min'] - cis_cg['std'] or cg_chem_shift > cis_cg['max'] + cis_cg['std']:
            cis = 0.0
        else:
            cis *= probability_density(cg_chem_shift, cis_cg['avr'], cis_cg['std'])
        if cg_chem_shift < trs_cg['min'] - trs_cg['std'] or cg_chem_shift > trs_cg['max'] + trs_cg['std']:
            trs = 0.0
        else:
            trs *= probability_density(cg_chem_shift, trs_cg['avr'], trs_cg['std'])

    if (cb_chem_shift is not None) and (cg_chem_shift is not None):
        delta_shift = cb_chem_shift - cg_chem_shift

        cis *= probability_density(delta_shift, cis_dl['avr'], cis_dl['std'])
        trs *= probability_density(delta_shift, trs_dl['avr'], trs_dl['std'])

    total = cis + trs

    if total in (0.0, 2.0):
        return 0.0, 0.0

    return cis / total, trs / total


def predict_tautomer_state_of_histidine(cg_chem_shift, cd2_chem_shift, nd1_chem_shift, ne2_chem_shift):
    """ Return prediction of tautomeric state of Histidine using assigned CG, CD2, ND1, and NE2 chemical shifts.
        @return: probability of biprotonated, probability of tau tautomer, probability of pi tautomer
        Reference:
          Protonation, Tautomerization, and Rotameric Structure of Histidine: A Comprehensive Study by Magic-Angle-Spinning Solid-State NMR.
          Shenhui Li and Mei Hong.
          Journal of the American Chemical Society 2011 133 (5), 1534-1544
          DOI: 10.1021/ja108943n
    """

    bip_cg = {'avr': 131.2, 'std': 0.7}
    bip_cd2 = {'avr': 120.6, 'std': 1.3}
    bip_nd1 = {'avr': 190.0, 'std': 1.9}
    bip_ne2 = {'avr': 176.3, 'std': 1.9}

    tau_cg = {'avr': 135.7, 'std': 2.2}
    tau_cd2 = {'avr': 116.9, 'std': 2.1}
    tau_nd1 = {'avr': 249.4, 'std': 1.9}
    tau_ne2 = {'avr': 171.1, 'std': 1.9}

    pi_cg = {'avr': 125.7, 'std': 2.2}
    pi_cd2 = {'avr': 125.6, 'std': 2.1}
    pi_nd1 = {'avr': 171.8, 'std': 1.9}
    pi_ne2 = {'avr': 248.2, 'std': 1.9}

    bip = 1.0
    tau = 1.0
    pi = 1.0

    if cg_chem_shift is not None:
        bip *= probability_density(cg_chem_shift, bip_cg['avr'], bip_cg['std'])
        tau *= probability_density(cg_chem_shift, tau_cg['avr'], tau_cg['std'])
        pi *= probability_density(cg_chem_shift, pi_cg['avr'], pi_cg['std'])

    if cd2_chem_shift is not None:
        bip *= probability_density(cd2_chem_shift, bip_cd2['avr'], bip_cd2['std'])
        tau *= probability_density(cd2_chem_shift, tau_cd2['avr'], tau_cd2['std'])
        pi *= probability_density(cd2_chem_shift, pi_cd2['avr'], pi_cd2['std'])

    if nd1_chem_shift is not None:
        bip *= probability_density(nd1_chem_shift, bip_nd1['avr'], bip_nd1['std'])
        tau *= probability_density(nd1_chem_shift, tau_nd1['avr'], tau_nd1['std'])
        pi *= probability_density(nd1_chem_shift, pi_nd1['avr'], pi_nd1['std'])

    if ne2_chem_shift is not None:
        bip *= probability_density(ne2_chem_shift, bip_ne2['avr'], bip_ne2['std'])
        tau *= probability_density(ne2_chem_shift, tau_ne2['avr'], tau_ne2['std'])
        pi *= probability_density(ne2_chem_shift, pi_ne2['avr'], pi_ne2['std'])

    total = bip + tau + pi

    if total in (0.0, 3.0):
        return 0.0, 0.0, 0.0

    return bip / total, tau / total, pi / total


def predict_rotamer_state_of_leucine(cd1_chem_shift, cd2_chem_shift):
    """ Return prediction of romermeric state of Leucine using assigned CD1 and CD2 chemical shifts.
        @return: probability of gauche+, trans, gauche-
        Reference:
          Dependence of Amino Acid Side Chain 13C Shifts on Dihedral Angle: Application to Conformational Analysis.
          Robert E. London, Brett D. Wingad, and Geoffrey A. Mueller.
          Journal of the American Chemical Society 2008 130 (33), 11097-11105
          DOI: 10.1021/ja802729t
    """

    if cd1_chem_shift is not None and cd2_chem_shift is not None:

        delta = cd1_chem_shift - cd2_chem_shift

        pt = (delta + 5.0) / 10.0

        if 0.0 <= pt <= 1.0:
            return 1.0 - pt, pt, 0.0

    gp_cd1 = {'avr': 24.45, 'std': 1.58}
    gp_cd2 = {'avr': 25.79, 'std': 1.68}

    t_cd1 = {'avr': 25.17, 'std': 1.58}
    t_cd2 = {'avr': 23.84, 'std': 1.68}

    gp = 1.0
    t = 1.0

    if cd1_chem_shift is not None:
        gp *= probability_density(cd1_chem_shift, gp_cd1['avr'], gp_cd1['std'])
        t *= probability_density(cd1_chem_shift, t_cd1['avr'], t_cd1['std'])

    if cd2_chem_shift is not None:
        gp *= probability_density(cd2_chem_shift, gp_cd2['avr'], gp_cd2['std'])
        t *= probability_density(cd2_chem_shift, t_cd2['avr'], t_cd2['std'])

    total = gp + t

    if total in (0.0, 2.0):
        return 0.0, 0.0, 0.0

    return gp / total, t / total, 0.0


def predict_rotamer_state_of_valine(cg1_chem_shift, cg2_chem_shift):
    """ Return prediction of romermeric state of Valine using assigned CG1 and CG2 chemical shifts.
        @return: probability of gauche+, trans, gauche-
        Reference:
          Dependence of Amino Acid Side Chain 13C Shifts on Dihedral Angle: Application to Conformational Analysis.
          Robert E. London, Brett D. Wingad, and Geoffrey A. Mueller.
          Journal of the American Chemical Society 2008 130 (33), 11097-11105
          DOI: 10.1021/ja802729t
    """

    gm_cg1 = {'avr': 22.05, 'std': 1.36}
    gm_cg2 = {'avr': 20.1, 'std': 1.55}

    gp_cg1 = {'avr': 20.87, 'std': 1.36}
    gp_cg2 = {'avr': 21.23, 'std': 1.55}

    t_cg1 = {'avr': 21.74, 'std': 1.36}
    t_cg2 = {'avr': 21.97, 'std': 1.55}

    gm = 1.0
    gp = 1.0
    t = 1.0

    if cg1_chem_shift is not None:
        gm *= probability_density(cg1_chem_shift, gm_cg1['avr'], gm_cg1['std'])
        gp *= probability_density(cg1_chem_shift, gp_cg1['avr'], gp_cg1['std'])
        t *= probability_density(cg1_chem_shift, t_cg1['avr'], t_cg1['std'])

    if cg2_chem_shift is not None:
        gm *= probability_density(cg2_chem_shift, gm_cg2['avr'], gm_cg2['std'])
        gp *= probability_density(cg2_chem_shift, gp_cg2['avr'], gp_cg2['std'])
        t *= probability_density(cg2_chem_shift, t_cg2['avr'], t_cg2['std'])

    total = gm + gp + t

    if total in (0.0, 3.0):
        return 0.0, 0.0, 0.0

    return gp / total, t / total, gm / total


def predict_rotamer_state_of_isoleucine(cd1_chem_shift):
    """ Return prediction of romermeric state of Isoleucine using assigned CD1 chemical shift.
        @return: probability of gauche+, trans, gauche-
        Reference:
          Determination of Isoleucine Side-Chain Conformations in Ground and Excited States of Proteins from Chemical Shifts.
          D. Flemming Hansen, Philipp Neudecker, and Lewis E. Kay.
          Journal of the American Chemical Society 2010 132 (22), 7589-7591
          DOI: 10.1021/ja102090z
    """

    if cd1_chem_shift is None:
        return 0.0, 0.0, 0.0

    if cd1_chem_shift < 9.3:
        return 0.0, 0.0, 1.0

    if cd1_chem_shift > 14.8:
        return 1.0 * (4.0 / 85.0), 1.0 * (81.0 / 85.0), 0.0

    pgm = (14.8 - cd1_chem_shift) / 5.5

    return (1.0 - pgm) * (4.0 / 85.0), (1.0 - pgm) * (81.0 / 85.0), pgm


def to_np_array(a):
    """ Return Numpy array of a given Cartesian coordinate in {'x': float, 'y': float, 'z': float} format.
    """

    return np.asarray([a['x'], a['y'], a['z']])


def to_unit_vector(a):
    """ Return unit vector of a given vector.
    """

    return a / np.linalg.norm(a)


def dihedral_angle(p0, p1, p2, p3):
    """ Return dihedral angle from a series of four points.
    """

    b0 = -1.0 * (p1 - p0)
    b1 = p2 - p1
    b2 = p3 - p2

    # normalize b1 so that it does not influence magnitude of vector
    # rejections that come next
    b1 = to_unit_vector(b1)

    # vector rejections
    # v = projection of b0 onto plane perpendicular to b1
    #   = b0 minus component that aligns with b1
    # w = projection of b2 onto plane perpendicular to b1
    #   = b2 minus component that aligns with b1
    v = b0 - np.dot(b0, b1) * b1
    w = b2 - np.dot(b2, b1) * b1

    # angle between v and w in a plane is the torsion angle
    # v and w may not be normalized but that's fine since tan is y/x
    x = np.dot(v, w)
    y = np.dot(np.cross(b1, v), w)

    return np.degrees(np.arctan2(y, x))


def score_of_seq_align(my_align):
    """ Return score of sequence alignment.
    """

    length = len(my_align)

    aligned = [True] * length

    for i in range(length):
        myPr = my_align[i]
        myPr0 = str(myPr[0])
        myPr1 = str(myPr[1])
        if myPr0 == '.' or myPr1 == '.':
            aligned[i] = False
        elif myPr0 != myPr1:
            pass
        else:
            break

    not_aligned = True
    offset_1 = 0
    offset_2 = 0

    unmapped = 0
    conflict = 0
    _matched = 0
    for i in range(length):
        myPr = my_align[i]
        myPr0 = str(myPr[0])
        myPr1 = str(myPr[1])
        if myPr0 == '.' or myPr1 == '.':
            if not_aligned and not aligned[i]:
                if myPr0 == '.' and myPr1 != '.' and offset_2 == 0:  # DAOTHER-7421
                    offset_1 += 1
                if myPr0 != '.' and myPr1 == '.' and offset_1 == 0:  # DAOTHER-7421
                    offset_2 += 1
                if myPr0 == '.' and myPr1 == '.':  # DAOTHER-7465
                    if offset_2 == 0:
                        offset_1 += 1
                    if offset_1 == 0:
                        offset_2 += 1
            unmapped += 1
        elif myPr0 != myPr1:
            conflict += 1
        else:
            not_aligned = False
            _matched += 1

    return _matched, unmapped, conflict, offset_1, offset_2


class NmrDpUtility:
    """ Wrapper class for data processing for NMR data.
    """

    def __init__(self, verbose=False, log=sys.stderr):
        self.__verbose = verbose
        self.__debug = False

        self.__lfh = log

        # current workflow operation
        self.__op = None

        # whether to run initial rescue routine
        self.__rescue_mode = True
        # whether NMR combined deposition or not (NMR separated deposition)
        self.__combined_mode = True
        # whether to use datablock name of public release
        self.__release_mode = False

        # whether to allow empty coordinate file path
        self.__bmrb_only = False
        # whether not to block deposition because of anomalous cs
        self.__nonblk_anomalous_cs = False
        # whether not to block deposition because bad n-term amino group
        self.__nonblk_bad_nterm = False
        # whether to udpate polymer sequence
        self.__update_poly_seq = False
        # whether to resolve conflict
        self.__resolve_conflict = False
        # whether to detect missing mandatory tags as errors
        self.__check_mandatory_tag = False
        # whether to detect consistency of author sequence (nmr-star specific)
        self.__check_auth_seq = False
        # whether to translate conventional pseudo atom nomenclature in combined NMR-STAR file
        self.__transl_pseudo_name = False
        # whether to enable tolerant sequence alignment for residue variants
        self.__tolerant_seq_align = False

        # whether to fix format issue (enabled if NMR separated deposition or release mode)
        self.__fix_format_issue = False
        # whether to exclude missing mandatory data (enabled if NMR separated deposition)
        self.__excl_missing_data = False
        # whether to detect empty row in a loop # NEFTranslator.validate_file() already prompts the empty low error
        # self.__check_empty_loop = False
        # whether to trust pdbx_nmr_ensemble to get total number of models
        self.__trust_pdbx_nmr_ens = True

        # whether sf_framecode has to be fixed
        self.__has_legacy_sf_issue = False

        # default entry_id
        self.__entry_id__ = 'UNNAMED'
        # current entry_id, to be replaced
        self.__entry_id = 'EXTRACT_FROM_COORD'
        # whether to insert entry_id (nmr-star specific)
        self.__insert_entry_id_to_loops = True

        # whether to retain original content if possible
        self.__retain_original = True
        # whether to leave internal commentary note in processed NMR-STAR file
        self.__leave_intl_note = True
        # whether to use reduced atom notation
        self.__reduced_atom_notation = True

        # whether entity category exists (nmr-star specific)
        self.__has_star_entity = False

        # source, destination, and log file paths.
        self.__srcPath = None
        self.__dstPath = None
        self.__logPath = None

        self.__cifPath = None

        # temporary file path to be removed (release mode).
        self.__tmpPath = None

        # auxiliary input resource.
        self.__inputParamDict = {}

        # auxiliary output resource.
        self.__outputParamDict = {}

        # list of known workflow operations
        self.__workFlowOps = ('nmr-nef-consistency-check',
                              'nmr-str-consistency-check',
                              'nmr-nef2str-deposit',
                              'nmr-str2str-deposit',
                              'nmr-str2nef-release',
                              'nmr-cs-nef-consistency-check',
                              'nmr-cs-str-consistency-check'
                              )

        # validation tasks for NMR data only
        __nmrCheckTasks = [self.__detectContentSubType,
                           self.__detectContentSubTypeOfLegacyMR,
                           self.__extractPolymerSequence,
                           self.__extractPolymerSequenceInLoop,
                           # self.__testSequenceConsistency,
                           self.__extractCommonPolymerSequence,
                           self.__extractNonStandardResidue,
                           self.__appendPolymerSequenceAlignment,
                           self.__testSequenceConsistency,
                           self.__validateAtomNomenclature,
                           self.__appendElemAndIsoNumOfNefCSLoop,
                           self.__validateAtomTypeOfCSLoop,
                           self.__validateAmbigCodeOfCSLoop,
                           self.__appendIndexTag,
                           self.__testIndexConsistency,
                           self.__appendWeightInLoop,
                           self.__appendDihedAngleType,
                           self.__testDataConsistencyInLoop,
                           self.__detectConflictDataInLoop,
                           self.__testDataConsistencyInAuxLoop,
                           self.__appendSfTagItem,
                           self.__testSfTagConsistency,
                           # self.__validateCSValue,
                           self.__testCSValueConsistencyInPkLoop,
                           self.__testCSValueConsistencyInPkAltLoop,
                           self.__testRDCVector
                           ]

        # validation tasks for coordinate file only
        __cifCheckTasks = [self.__validateCoordInputSource,
                           self.__detectCoordContentSubType,
                           self.__extractCoordPolymerSequence,
                           self.__extractCoordPolymerSequenceInLoop,
                           self.__extractCoordCommonPolymerSequence,
                           self.__extractCoordNonStandardResidue,
                           self.__appendCoordPolymerSequenceAlignment
                           ]

        # cross validation tasks
        __crossCheckTasks = [self.__assignCoordPolymerSequence,
                             self.__testCoordAtomIdConsistency,
                             self.__testCovalentBond,
                             self.__testResidueVariant,
                             self.__validateCSValue,
                             self.__extractCoordDisulfideBond,
                             self.__extractCoordOtherBond,
                             self.__calculateStatsOfExptlData
                             ]

        # nmr-*-consistency-check tasks
        __checkTasks = [self.__initializeDpReport,
                        self.__validateInputSource
                        ]
        __checkTasks.extend(__nmrCheckTasks)
        __checkTasks.extend(__cifCheckTasks)
        __checkTasks.extend(__crossCheckTasks)

        # nmr-*-deposit tasks
        __depositTasks = [self.__retrieveDpReport,
                          self.__validateInputSource,
                          self.__parseCoordinate,
                          # resolve conflict
                          self.__resolveConflictsInLoop,
                          self.__resolveConflictsInAuxLoop,
                          # resolve minor issues
                          self.__validateAtomNomenclature,
                          self.__appendIndexTag,
                          self.__appendWeightInLoop,
                          self.__appendDihedAngleType,
                          self.__appendSfTagItem,
                          self.__deleteSkippedSf,
                          self.__deleteSkippedLoop,
                          self.__deleteUnparsedEntryLoop,
                          self.__updatePolymerSequence,
                          self.__updateAuthSequence,
                          self.__updateDihedralAngleType,
                          self.__fixDisorderedIndex,
                          self.__removeNonSenseZeroValue,
                          self.__fixNonSenseNegativeValue,
                          self.__fixEnumMismatch,
                          self.__fixEnumMismatchIgnorable,
                          # self.__fixBadAmbiguityCode,
                          self.__resetCapitalStringInLoop,
                          self.__resetBoolValueInLoop,
                          self.__resetBoolValueInAuxLoop,
                          self.__appendParentSfTag,
                          self.__addUnnamedEntryId,
                          self.__depositNmrData,
                          # re-setup for next
                          self.__initializeDpReportForNext,
                          self.__validateInputSourceForNext
                          ]

        __depositTasks.extend(__nmrCheckTasks)
        __depositTasks.extend(__cifCheckTasks)
        __depositTasks.extend(__crossCheckTasks)

        # additional nmr-nef2str tasks
        __nef2strTasks = [self.__translateNef2Str,
                          self.__dumpDpReport,
                          self.__initResourceForNef2Str
                          ]

        __nef2strTasks.extend(__checkTasks)
        __nef2strTasks.append(self.__dumpDpReport)
        __nef2strTasks.extend(__depositTasks)

        # additional nmr-str2nef tasks
        __str2nefTasks = [self.__translateStr2Nef,
                          self.__dumpDpReport,
                          self.__initResourceForStr2Nef
                          ]

        __str2nefTasks.extend(__checkTasks)
        __str2nefTasks.append(self.__dumpDpReport)
        __str2nefTasks.extend(__depositTasks)

        # dictionary of processing tasks of each workflow operation
        self.__procTasksDict = {'consistency-check': __checkTasks,
                                'deposit': __depositTasks,
                                'nmr-nef2str-deposit': __nef2strTasks,
                                'nmr-str2nef-release': __str2nefTasks,
                                'nmr-cs-nef-consistency-check': [self.__depositLegacyNmrData],
                                'nmr-cs-str-consistency-check': [self.__depositLegacyNmrData]
                                }

        # data processing report
        self.report = None
        self.report_prev = None

        # NEFTranslator
        self.__nefT = NEFTranslator()

        if self.__nefT is None:
            raise IOError("+NmrDpUtility.__init__() ++ Error  - NEFTranslator is not available.")

        # BMRB chemical shift statistics
        self.__csStat = BMRBChemShiftStat()

        if not self.__csStat.isOk():
            raise IOError("+NmrDpUtility.__init__() ++ Error  - BMRBChemShiftStat is not available.")

        # PyNMRSTAR data
        self.__file_path_list_len = 1
        self.__cs_file_path_list_len = 1

        self.__star_data_type = []
        self.__star_data = []
        self.__sf_name_corr = []

        self.__original_error_message = []

        self.__sf_category_list = []
        self.__lp_category_list = []

        self.__alt_chain = False
        self.__valid_seq = False

        # empty value
        self.empty_value = (None, '', '.', '?')

        # true value
        self.true_value = ('true', 't', 'yes', 'y', '1')

        # NMR content types
        self.nmr_content_subtypes = ('entry_info', 'poly_seq', 'entity', 'chem_shift', 'chem_shift_ref',
                                     'dist_restraint', 'dihed_restraint', 'rdc_restraint', 'spectral_peak', 'spectral_peak_alt')

        # CIF content types
        self.cif_content_subtypes = ('poly_seq', 'non_poly', 'coordinate')

        # readable file type
        self.readable_file_type = {'nef': 'NEF (NMR Exchange Format)',
                                   'nmr-star': 'NMR-STAR',
                                   'pdbx': 'PDBx/mmCIF',
                                   'unknown': 'unknown'
                                   }

        # content type
        self.content_type = {'nef': 'nmr-data-nef',
                             'nmr-star': 'nmr-data-str',
                             'pdbx': 'model'
                             }

        # content type used for public release
        self.release_type = {'nef': 'nmr-data',
                             'nmr-star': 'nmr-data',
                             'pdbx': None
                             }

        # paramagnetic elements, except for Oxygen
        self.paramag_elems = ('LI', 'NA', 'MG', 'AL', 'K', 'CA', 'SC', 'TI', 'V', 'MN', 'RB', 'SR',
                              'Y', 'ZR', 'NB', 'MO', 'TC', 'RU', 'RH', 'PD', 'SN', 'CS', 'BA', 'LA',
                              'CE', 'PR', 'ND', 'PM', 'SM', 'EU', 'GD', 'TB', 'DY', 'HO', 'ER', 'TM',
                              'YB', 'LU', 'HF', 'TA', 'W', 'RE', 'OS', 'IR', 'PT', 'FR', 'RA', 'AC')

        # ferromagnetic elements
        self.ferromag_elems = ('CR', 'FE', 'CO', 'NI')

        # non-metal elements
        self.non_metal_elems = ('H', 'C', 'N', 'O', 'P', 'S', 'SE')

        # isotope numbers of NMR observable atoms
        self.atom_isotopes = {'H': [1, 2, 3],
                              'C': [13],
                              'N': [15, 14],
                              'O': [17],
                              'P': [31],
                              'S': [33],
                              'F': [19],
                              'CD': [113, 111],
                              'CA': [43]
                              }

        # nucleus with half spin
        self.half_spin_nucleus = ('H', 'C', 'N', 'P', 'F', 'CD')

        # ambiguity codes
        self.bmrb_ambiguity_codes = (1, 2, 3, 4, 5, 6, 9)

        isotope_nums = []
        for i in self.atom_isotopes.values():
            isotope_nums.extend(list(i))

        # saveframe categories
        self.sf_categories = {'nef': {'entry_info': 'nef_nmr_meta_data',
                                      'poly_seq': 'nef_molecular_system',
                                      'entity': None,
                                      'chem_shift': 'nef_chemical_shift_list',
                                      'chem_shift_ref': None,
                                      'dist_restraint': 'nef_distance_restraint_list',
                                      'dihed_restraint': 'nef_dihedral_restraint_list',
                                      'rdc_restraint': 'nef_rdc_restraint_list',
                                      'spectral_peak': 'nef_nmr_spectrum',
                                      'spectral_peak_alt': None
                                      },
                              'nmr-star': {'entry_info': 'entry_information',
                                           'poly_seq': 'assembly',
                                           'entity': 'entity',
                                           'chem_shift': 'assigned_chemical_shifts',
                                           'chem_shift_ref': 'chem_shift_reference',
                                           'dist_restraint': 'general_distance_constraints',
                                           'dihed_restraint': 'torsion_angle_constraints',
                                           'rdc_restraint': 'RDC_constraints',
                                           'spectral_peak': 'spectral_peak_list',
                                           'spectral_peak_alt': 'spectral_peak_list'
                                           }
                              }

        # loop categories
        self.lp_categories = {'nef': {'entry_info': '_nef_program_script',
                                      'poly_seq': '_nef_sequence',
                                      'entity': None,
                                      'chem_shift': '_nef_chemical_shift',
                                      'chem_shift_ref': None,
                                      'dist_restraint': '_nef_distance_restraint',
                                      'dihed_restraint': '_nef_dihedral_restraint',
                                      'rdc_restraint': '_nef_rdc_restraint',
                                      'spectral_peak': '_nef_peak',
                                      'spectral_peak_alt': None
                                      },
                              'nmr-star': {'entry_info': '_Software_applied_methods',
                                           'poly_seq': '_Chem_comp_assembly',
                                           'entity': '_Entity_comp_index',
                                           'chem_shift': '_Atom_chem_shift',
                                           'chem_shift_ref': '_Chem_shift_ref',
                                           'dist_restraint': '_Gen_dist_constraint',
                                           'dihed_restraint': '_Torsion_angle_constraint',
                                           'rdc_restraint': '_RDC_constraint',
                                           'spectral_peak': '_Peak_row_format',
                                           'spectral_peak_alt': '_Peak'
                                           },
                              'pdbx': {'poly_seq': 'pdbx_poly_seq_scheme',
                                       'non_poly': 'pdbx_nonpoly_scheme',
                                       'coordinate': 'atom_site',
                                       'poly_seq_alias': 'ndb_poly_seq_scheme',
                                       'non_poly_alias': 'ndb_nonpoly_scheme'
                                       }
                              }

        # allowed chem shift range
        self.chem_shift_range = {'min_exclusive': -300.0, 'max_exclusive': 300.0}
        self.chem_shift_error = {'min_inclusive': 0.0, 'max_inclusive': 3.0}

        # allowed distance range
        self.dist_restraint_range = {'min_inclusive': 0.5, 'max_inclusive': 50.0}
        self.dist_restraint_error = {'min_inclusive': 0.0, 'max_inclusive': 5.0}

        # allowed dihed range
        self.dihed_restraint_range = {'min_inclusive': -360.0, 'max_inclusive': 360.0}
        self.dihed_restraint_error = {'min_inclusive': 0.0, 'max_inclusive': 90.0}

        # allowed rdc range
        self.rdc_restraint_range = {'min_exclusive': -100.0, 'max_exclusive': 100.0}
        self.rdc_restraint_error = {'min_inclusive': 0.0, 'max_inclusive': 5.0}

        # allowed weight range
        self.weight_range = {'min_inclusive': 0.0, 'max_inclusive': 100.0}
        # allowed scale range
        self.scale_range = self.weight_range

        # criterion for low sequence coverage
        self.low_seq_coverage = 0.3

        # criterion for minimum sequence coverage when conflict occurs (NMR separated deposition)
        self.min_seq_coverage_w_conflict = 0.95

        # cutoff value for detection of aromatic atoms
        self.cutoff_aromatic = 5.0
        # cutoff value for detection of paramagnetic/ferromagnetic atoms
        self.cutoff_paramagnetic = 10.0

        # criterion for aromatic ring in the vicinity
        self.vicinity_aromatic = 4.0
        # criterion for paramagnetic/ferromagnetic atom in the vicinity
        self.vicinity_paramagnetic = 8.0

        # criterion for detection of not superimposed models
        self.cutoff_rmsd = 3.5

        # criterion for covalent bond length
        self.cutoff_bond_length = 3.5

        # magic angle in degrees
        self.magic_angle = 54.7356

        # criterion for inconsistent restraint condition scaled by the conflicted restraint condition
        self.inconsist_over_conflicted = 0.75
        # criterion on R factor for conflicted distance restraint
        self.r_conflicted_dist_restraint = 0.4
        # criterion on R factor for inconsistent distance restraint
        self.r_inconsistent_dist_restraint = self.r_conflicted_dist_restraint * self.inconsist_over_conflicted

        # criterion on chemical shift for anomalous value scaled by its sigma
        self.cs_anomalous_error_scaled_by_sigma = 8.0
        # criterion on chemical shift for unusual value scaled by its sigma
        self.cs_unusual_error_scaled_by_sigma = 5.0
        # criterion on chemical shift difference error scaled by its sigma
        self.cs_diff_error_scaled_by_sigma = 8.0

        # hardware limit of NMR prove design in Hz (DAOTHER-7389, issue #1)
        self.hard_probe_limit = 250000

        # loop index tags
        self.index_tags = {'nef': {'entry_info': None,
                                   'poly_seq': 'index',
                                   'entity': None,
                                   'chem_shift': None,
                                   'chem_shift_ref': None,
                                   'dist_restraint': 'index',
                                   'dihed_restraint': 'index',
                                   'rdc_restraint': 'index',
                                   'spectral_peak': 'index',
                                   'spectral_peak_alt': None
                                   },
                           'nmr-star': {'entry_info': None,
                                        'poly_seq': None,
                                        'entity': None,
                                        'chem_shift': None,
                                        'chem_shift_ref': None,
                                        'dist_restraint': 'Index_ID',
                                        'dihed_restraint': 'Index_ID',
                                        'rdc_restraint': 'Index_ID',
                                        'spectral_peak': 'Index_ID',
                                        'spectral_peak_alt': None
                                        },
                           'pdbx': {'poly_seq': None,
                                    'non_poly': None,
                                    'coordinate': 'id'
                                    }
                           }

        # weight tags
        self.weight_tags = {'nef': {'entry_info': None,
                                    'poly_seq': None,
                                    'entity': None,
                                    'chem_shift': None,
                                    'chem_shift_ref': None,
                                    'dist_restraint': 'weight',
                                    'dihed_restraint': 'weight',
                                    'rdc_restraint': 'weight',
                                    'spectral_peak': None,
                                    'spectral_peak_alt': None
                                    },
                            'nmr-star': {'entry_info': None,
                                         'poly_seq': None,
                                         'entity': None,
                                         'chem_shift': None,
                                         'chem_shift_ref': None,
                                         'dist_restraint': 'Weight',
                                         'dihed_restraint': 'Weight',
                                         'rdc_restraint': 'Weight',
                                         'spectral_peak': None,
                                         'spectral_peak_alt': None
                                         },
                            'pdbx': {'poly_seq': None,
                                     'non_poly': None,
                                     'coordinate': None
                                     }
                            }

        # dihedral angle type
        self.angle_types = {'nef': 'name',
                            'nmr-star': 'Torsion_angle_name'
                            }

        # loop id tag to check consistency
        self.consist_id_tags = {'nef': {'dist_restraint': 'restraint_id',
                                        'dihed_restraint': 'restraint_id',
                                        'rdc_restraint': 'restraint_id',
                                        'spectral_peak': 'peak_id',
                                        'spectral_peak_alt': None
                                        },
                                'nmr-star': {'dist_restraint': 'ID',
                                             'dihed_restraint': 'ID',
                                             'rdc_restraint': 'ID',
                                             'spectral_peak': 'ID',
                                             'spectral_peak_alt': 'ID'
                                             }
                                }

        # key items of loop
        self.key_items = {'nef': {'poly_seq': [{'name': 'chain_code', 'type': 'str', 'default': 'A'},
                                               {'name': 'sequence_code', 'type': 'int'},
                                               {'name': 'residue_name', 'type': 'str', 'uppercase': True}
                                               ],
                                  'entity': None,
                                  'chem_shift': [{'name': 'chain_code', 'type': 'str', 'default': 'A'},
                                                 {'name': 'sequence_code', 'type': 'int',
                                                  'remove-bad-pattern': True},
                                                 {'name': 'residue_name', 'type': 'str',
                                                  'uppercase': True,
                                                  'remove-bad-pattern': True},
                                                 {'name': 'atom_name', 'type': 'str',
                                                  'remove-bad-pattern': True}
                                                 ],
                                  'chem_shift_ref': None,
                                  'dist_restraint': [{'name': 'restraint_id', 'type': 'positive-int'},
                                                     {'name': 'chain_code_1', 'type': 'str', 'default': 'A'},
                                                     {'name': 'sequence_code_1', 'type': 'int'},
                                                     {'name': 'residue_name_1', 'type': 'str', 'uppercase': True},
                                                     {'name': 'atom_name_1', 'type': 'str'},
                                                     {'name': 'chain_code_2', 'type': 'str', 'default': 'A'},
                                                     {'name': 'sequence_code_2', 'type': 'int'},
                                                     {'name': 'residue_name_2', 'type': 'str', 'uppercase': True},
                                                     {'name': 'atom_name_2', 'type': 'str'}
                                                     ],
                                  'dihed_restraint': [{'name': 'restraint_id', 'type': 'positive-int'},
                                                      {'name': 'chain_code_1', 'type': 'str', 'default': 'A'},
                                                      {'name': 'sequence_code_1', 'type': 'int'},
                                                      {'name': 'residue_name_1', 'type': 'str', 'uppercase': True},
                                                      {'name': 'atom_name_1', 'type': 'str'},
                                                      {'name': 'chain_code_2', 'type': 'str', 'default': 'A'},
                                                      {'name': 'sequence_code_2', 'type': 'int'},
                                                      {'name': 'residue_name_2', 'type': 'str', 'uppercase': True},
                                                      {'name': 'atom_name_2', 'type': 'str'},
                                                      {'name': 'chain_code_3', 'type': 'str', 'default': 'A'},
                                                      {'name': 'sequence_code_3', 'type': 'int'},
                                                      {'name': 'residue_name_3', 'type': 'str', 'uppercase': True},
                                                      {'name': 'atom_name_3', 'type': 'str'},
                                                      {'name': 'chain_code_4', 'type': 'str', 'default': 'A'},
                                                      {'name': 'sequence_code_4', 'type': 'int'},
                                                      {'name': 'residue_name_4', 'type': 'str', 'uppercase': True},
                                                      {'name': 'atom_name_4', 'type': 'str'}
                                                      ],
                                  'rdc_restraint': [{'name': 'restraint_id', 'type': 'positive-int'},
                                                    {'name': 'chain_code_1', 'type': 'str', 'default': 'A'},
                                                    {'name': 'sequence_code_1', 'type': 'int'},
                                                    {'name': 'residue_name_1', 'type': 'str', 'uppercase': True},
                                                    {'name': 'atom_name_1', 'type': 'str'},
                                                    {'name': 'chain_code_2', 'type': 'str', 'default': 'A'},
                                                    {'name': 'sequence_code_2', 'type': 'int'},
                                                    {'name': 'residue_name_2', 'type': 'str', 'uppercase': True},
                                                    {'name': 'atom_name_2', 'type': 'str'}
                                                    ],
                                  'spectral_peak': None,
                                  'spectral_peak_alt': None
                                  },
                          'nmr-star': {'poly_seq': [{'name': 'Entity_assembly_ID', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID'},
                                                    {'name': 'Comp_index_ID', 'type': 'int'},
                                                    {'name': 'Comp_ID', 'type': 'str', 'uppercase': True}
                                                    ],
                                       'entity': None,
                                       'chem_shift': [{'name': 'Entity_assembly_ID', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID'},
                                                      {'name': 'Comp_index_ID', 'type': 'int',
                                                       'remove-bad-pattern': True},
                                                      {'name': 'Comp_ID', 'type': 'str',
                                                       'uppercase': True,
                                                       'remove-bad-pattern': True},
                                                      {'name': 'Atom_ID', 'type': 'str',
                                                       'remove-bad-pattern': True}
                                                      ],
                                       'chem_shift_ref': [{'name': 'Atom_type', 'type': 'enum', 'enum': set(self.atom_isotopes.keys()),
                                                           'enforce-enum': True},
                                                          {'name': 'Atom_isotope_number', 'type': 'enum-int', 'enum': set(isotope_nums),
                                                           'enforce-enum': True},
                                                          {'name': 'Mol_common_name', 'type': 'str'}],
                                       'dist_restraint': [{'name': 'ID', 'type': 'positive-int'},
                                                          {'name': 'Entity_assembly_ID_1', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_1'},
                                                          {'name': 'Comp_index_ID_1', 'type': 'int'},
                                                          {'name': 'Comp_ID_1', 'type': 'str', 'uppercase': True},
                                                          {'name': 'Atom_ID_1', 'type': 'str'},
                                                          {'name': 'Entity_assembly_ID_2', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_2'},
                                                          {'name': 'Comp_index_ID_2', 'type': 'int'},
                                                          {'name': 'Comp_ID_2', 'type': 'str', 'uppercase': True},
                                                          {'name': 'Atom_ID_2', 'type': 'str'}
                                                          ],
                                       'dihed_restraint': [{'name': 'ID', 'type': 'positive-int'},
                                                           {'name': 'Entity_assembly_ID_1', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_1'},
                                                           {'name': 'Comp_index_ID_1', 'type': 'int'},
                                                           {'name': 'Comp_ID_1', 'type': 'str', 'uppercase': True},
                                                           {'name': 'Atom_ID_1', 'type': 'str'},
                                                           {'name': 'Entity_assembly_ID_2', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_2'},
                                                           {'name': 'Comp_index_ID_2', 'type': 'int'},
                                                           {'name': 'Comp_ID_2', 'type': 'str', 'uppercase': True},
                                                           {'name': 'Atom_ID_2', 'type': 'str'},
                                                           {'name': 'Entity_assembly_ID_3', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_3'},
                                                           {'name': 'Comp_index_ID_3', 'type': 'int'},
                                                           {'name': 'Comp_ID_3', 'type': 'str', 'uppercase': True},
                                                           {'name': 'Atom_ID_3', 'type': 'str'},
                                                           {'name': 'Entity_assembly_ID_4', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_4'},
                                                           {'name': 'Comp_index_ID_4', 'type': 'int'},
                                                           {'name': 'Comp_ID_4', 'type': 'str', 'uppercase': True},
                                                           {'name': 'Atom_ID_4', 'type': 'str'}
                                                           ],
                                       'rdc_restraint': [{'name': 'ID', 'type': 'positive-int'},
                                                         {'name': 'Entity_assembly_ID_1', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_1'},
                                                         {'name': 'Comp_index_ID_1', 'type': 'int'},
                                                         {'name': 'Comp_ID_1', 'type': 'str', 'uppercase': True},
                                                         {'name': 'Atom_ID_1', 'type': 'str'},
                                                         {'name': 'Entity_assembly_ID_2', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_2'},
                                                         {'name': 'Comp_index_ID_2', 'type': 'int'},
                                                         {'name': 'Comp_ID_2', 'type': 'str', 'uppercase': True},
                                                         {'name': 'Atom_ID_2', 'type': 'str'}
                                                         ],
                                       'spectral_peak': None,
                                       'spectral_peak_alt': [{'name': 'ID', 'type': 'positive-int'},
                                                             {'name': 'Spectral_peak_list_ID', 'type': 'positive-int', 'default': '1', 'default-from': 'self'}
                                                             ]
                                       },
                          'pdbx': {'poly_seq': [{'name': 'asym_id', 'type': 'str', 'alt_name': 'chain_id'},
                                                {'name': 'seq_id', 'type': 'int', 'alt_name': 'seq_id'},
                                                {'name': 'mon_id', 'type': 'str', 'alt_name': 'comp_id'},
                                                {'name': 'pdb_strand_id', 'type': 'str', 'alt_name': 'auth_chain_id'}
                                                ],
                                   'poly_seq_alias': [{'name': 'id', 'type': 'str', 'alt_name': 'chain_id'},
                                                      {'name': 'seq_id', 'type': 'int', 'alt_name': 'seq_id'},
                                                      {'name': 'mon_id', 'type': 'str', 'alt_name': 'comp_id'}
                                                      ],
                                   'non_poly': [{'name': 'asym_id', 'type': 'str', 'alt_name': 'chain_id'},
                                                {'name': 'pdb_seq_num', 'type': 'int', 'alt_name': 'seq_id'},
                                                {'name': 'mon_id', 'type': 'str', 'alt_name': 'comp_id'},
                                                {'name': 'pdb_strand_id', 'type': 'str', 'alt_name': 'auth_chain_id'}
                                                ],
                                   'non_poly_alias': [{'name': 'asym_id', 'type': 'str', 'alt_name': 'chain_id'},
                                                      {'name': 'pdb_num', 'type': 'int', 'alt_name': 'seq_id'},
                                                      {'name': 'mon_id', 'type': 'str', 'alt_name': 'comp_id'}
                                                      ],
                                   'coordinate': [{'name': 'label_asym_id', 'type': 'str', 'alt_name': 'chain_id'},
                                                  {'name': 'auth_seq_id', 'type': 'int', 'alt_name': 'seq_id'},
                                                  {'name': 'auth_comp_id', 'type': 'str', 'alt_name': 'comp_id'},
                                                  {'name': 'pdbx_PDB_model_num', 'type': 'int', 'alt_name': 'model_id'}
                                                  ],
                                   'coordinate_alias': [{'name': 'label_asym_id', 'type': 'str', 'alt_name': 'chain_id'},
                                                        {'name': 'auth_seq_id', 'type': 'int', 'alt_name': 'seq_id'},
                                                        {'name': 'auth_comp_id', 'type': 'str', 'alt_name': 'comp_id'},
                                                        {'name': 'ndb_model', 'type': 'int', 'alt_name': 'model_id'}
                                                        ],
                                   'coordinate_ins': [{'name': 'label_asym_id', 'type': 'str', 'alt_name': 'chain_id'},
                                                      {'name': 'auth_seq_id', 'type': 'int', 'alt_name': 'seq_id'},
                                                      {'name': 'auth_comp_id', 'type': 'str', 'alt_name': 'comp_id'},
                                                      {'name': 'pdbx_PDB_ins_code', 'type': 'str', 'alt_name': 'ins_code', 'default': '?'},
                                                      {'name': 'label_seq_id', 'type': 'str', 'alt_name': 'label_seq_id', 'default': '.'},
                                                      {'name': 'pdbx_PDB_model_num', 'type': 'int', 'alt_name': 'model_id'}
                                                      ],
                                   'coordinate_ins_alias': [{'name': 'label_asym_id', 'type': 'str', 'alt_name': 'chain_id'},
                                                            {'name': 'auth_seq_id', 'type': 'int', 'alt_name': 'seq_id'},
                                                            {'name': 'auth_comp_id', 'type': 'str', 'alt_name': 'comp_id'},
                                                            {'name': 'ndb_ins_code', 'type': 'str', 'alt_name': 'ins_code', 'default': '?'},
                                                            {'name': 'label_seq_id', 'type': 'str', 'alt_name': 'label_seq_id', 'default': '.'},
                                                            {'name': 'ndb_model', 'type': 'int', 'alt_name': 'model_id'}
                                                            ]
                                   }
                          }

        # key items of loop to check consistency
        self.consist_key_items = {'nef': {'dist_restraint': [{'name': 'chain_code_1', 'type': 'str', 'default': 'A'},
                                                             {'name': 'sequence_code_1', 'type': 'int'},
                                                             {'name': 'residue_name_1', 'type': 'str', 'uppercase': True},
                                                             {'name': 'atom_name_1', 'type': 'str'},
                                                             {'name': 'chain_code_2', 'type': 'str', 'default': 'A'},
                                                             {'name': 'sequence_code_2', 'type': 'int'},
                                                             {'name': 'residue_name_2', 'type': 'str', 'uppercase': True},
                                                             {'name': 'atom_name_2', 'type': 'str'}
                                                             ],
                                          'dihed_restraint': [{'name': 'chain_code_1', 'type': 'str', 'default': 'A'},
                                                              {'name': 'sequence_code_1', 'type': 'int'},
                                                              {'name': 'residue_name_1', 'type': 'str', 'uppercase': True},
                                                              {'name': 'atom_name_1', 'type': 'str'},
                                                              {'name': 'chain_code_2', 'type': 'str', 'default': 'A'},
                                                              {'name': 'sequence_code_2', 'type': 'int'},
                                                              {'name': 'residue_name_2', 'type': 'str', 'uppercase': True},
                                                              {'name': 'atom_name_2', 'type': 'str'},
                                                              {'name': 'chain_code_3', 'type': 'str', 'default': 'A'},
                                                              {'name': 'sequence_code_3', 'type': 'int'},
                                                              {'name': 'residue_name_3', 'type': 'str', 'uppercase': True},
                                                              {'name': 'atom_name_3', 'type': 'str'},
                                                              {'name': 'chain_code_4', 'type': 'str', 'default': 'A'},
                                                              {'name': 'sequence_code_4', 'type': 'int'},
                                                              {'name': 'residue_name_4', 'type': 'str', 'uppercase': True},
                                                              {'name': 'atom_name_4', 'type': 'str'}
                                                              ],
                                          'rdc_restraint': [{'name': 'chain_code_1', 'type': 'str', 'default': 'A'},
                                                            {'name': 'sequence_code_1', 'type': 'int'},
                                                            {'name': 'residue_name_1', 'type': 'str', 'uppercase': True},
                                                            {'name': 'atom_name_1', 'type': 'str'},
                                                            {'name': 'chain_code_2', 'type': 'str', 'default': 'A'},
                                                            {'name': 'sequence_code_2', 'type': 'int'},
                                                            {'name': 'residue_name_2', 'type': 'str', 'uppercase': True},
                                                            {'name': 'atom_name_2', 'type': 'str'}
                                                            ],
                                          'spectral_peak': None,
                                          'spectral_peak_alt': None
                                          },
                                  'nmr-star': {'dist_restraint': [{'name': 'Entity_assembly_ID_1', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_1'},
                                                                  {'name': 'Comp_index_ID_1', 'type': 'int'},
                                                                  {'name': 'Comp_ID_1', 'type': 'str', 'uppercase': True},
                                                                  {'name': 'Atom_ID_1', 'type': 'str'},
                                                                  {'name': 'Entity_assembly_ID_2', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_2'},
                                                                  {'name': 'Comp_index_ID_2', 'type': 'int'},
                                                                  {'name': 'Comp_ID_2', 'type': 'str', 'uppercase': True},
                                                                  {'name': 'Atom_ID_2', 'type': 'str'}
                                                                  ],
                                               'dihed_restraint': [{'name': 'Entity_assembly_ID_1', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_1'},
                                                                   {'name': 'Comp_index_ID_1', 'type': 'int'},
                                                                   {'name': 'Comp_ID_1', 'type': 'str', 'uppercase': True},
                                                                   {'name': 'Atom_ID_1', 'type': 'str'},
                                                                   {'name': 'Entity_assembly_ID_2', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_2'},
                                                                   {'name': 'Comp_index_ID_2', 'type': 'int'},
                                                                   {'name': 'Comp_ID_2', 'type': 'str', 'uppercase': True},
                                                                   {'name': 'Atom_ID_2', 'type': 'str'},
                                                                   {'name': 'Entity_assembly_ID_3', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_3'},
                                                                   {'name': 'Comp_index_ID_3', 'type': 'int'},
                                                                   {'name': 'Comp_ID_3', 'type': 'str', 'uppercase': True},
                                                                   {'name': 'Atom_ID_3', 'type': 'str'},
                                                                   {'name': 'Entity_assembly_ID_4', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_4'},
                                                                   {'name': 'Comp_index_ID_4', 'type': 'int'},
                                                                   {'name': 'Comp_ID_4', 'type': 'str', 'uppercase': True},
                                                                   {'name': 'Atom_ID_4', 'type': 'str'}
                                                                   ],
                                               'rdc_restraint': [{'name': 'Entity_assembly_ID_1', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_1'},
                                                                 {'name': 'Comp_index_ID_1', 'type': 'int'},
                                                                 {'name': 'Comp_ID_1', 'type': 'str', 'uppercase': True},
                                                                 {'name': 'Atom_ID_1', 'type': 'str'},
                                                                 {'name': 'Entity_assembly_ID_2', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_2'},
                                                                 {'name': 'Comp_index_ID_2', 'type': 'int'},
                                                                 {'name': 'Comp_ID_2', 'type': 'str', 'uppercase': True},
                                                                 {'name': 'Atom_ID_2', 'type': 'str'}
                                                                 ],
                                               'spectral_peak': None,
                                               'spectral_peak_alt': None
                                               }
                                  }

        # limit number of dimensions
        self.lim_num_dim = 16

        # key items for spectral peak
        self.pk_key_items = {'nef': [{'name': 'position_%s', 'type': 'float'}
                                     ],
                             'nmr-star': [{'name': 'Position_%s', 'type': 'float'}
                                          ]
                             }

        # data items of loop
        self.data_items = {'nef': {'poly_seq': [{'name': 'linking', 'type': 'enum', 'mandatory': False,
                                                 'enum': ('start', 'end', 'middle', 'cyclic', 'break', 'single', 'dummy'),
                                                 'enforce-enum': True},
                                                {'name': 'residue_variant', 'type': 'str', 'mandatory': False},
                                                {'name': 'cis_peptide', 'type': 'bool', 'mandatory': False}
                                                ],
                                   'entity': None,
                                   'chem_shift': [{'name': 'value', 'type': 'range-float', 'mandatory': True,
                                                   'range': self.chem_shift_range},
                                                  {'name': 'value_uncertainty', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                   'range': self.chem_shift_error},
                                                  {'name': 'element', 'type': 'enum', 'mandatory': True, 'default-from': 'atom_name',
                                                   'enum': set(self.atom_isotopes.keys()),
                                                   'enforce-enum': True},
                                                  {'name': 'isotope_number', 'type': 'enum-int', 'mandatory': True, 'default-from': 'atom_name',
                                                   'enum': set(isotope_nums),
                                                   'enforce-enum': True}
                                                  ],
                                   'chem_shift_ref': None,
                                   'dist_restraint': [{'name': 'index', 'type': 'index-int', 'mandatory': True},
                                                      # {'name': 'restraint_id', 'type': 'positive-int', 'mandatory': True,
                                                      # 'enforce-non-zero': True},
                                                      {'name': 'restraint_combination_id', 'type': 'positive-int', 'mandatory': False,
                                                       'enforce-non-zero': True},
                                                      {'name': 'weight', 'type': 'range-float', 'mandatory': True,
                                                       'range': self.weight_range},
                                                      # 'enforce-non-zero': True},
                                                      {'name': 'target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                       'range': self.dist_restraint_range,
                                                       'group': {'member-with': ['lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                 'coexist-with': None,
                                                                 'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                 'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                      {'name': 'target_value_uncertainty', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                       'range': self.dist_restraint_error},
                                                      {'name': 'lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                       'range': self.dist_restraint_range,
                                                       'group': {'member-with': ['target_value', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                 'coexist-with': None,  # ['lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                 'smaller-than': None,
                                                                 'larger-than': ['lower_limit', 'upper_limit', 'upper_linear_limit']}},
                                                      {'name': 'lower_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                       'range': self.dist_restraint_range,
                                                       'group': {'member-with': ['target_value', 'lower_linear_limit', 'upper_limit', 'upper_linear_limit'],
                                                                 'coexist-with': None,  # ['upper_limit'],
                                                                 'smaller-than':['lower_linear_limit'],
                                                                 'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                      {'name': 'upper_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                       'range': self.dist_restraint_range,
                                                       'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_linear_limit'],
                                                                 'coexist-with': None,  # ['lower_limit'],
                                                                 'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                 'larger-than': ['upper_linear_limit']}},
                                                      {'name': 'upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                       'range': self.dist_restraint_range,
                                                       'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                 'coexist-with': None,  # ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                 'smaller-than': ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                 'larger-than': None}}
                                                      ],
                                   'dihed_restraint': [{'name': 'index', 'type': 'index-int', 'mandatory': True},
                                                       # {'name': 'restraint_id', 'type': 'positive-int', 'mandatory': True,
                                                       # 'enforce-non-zero': True},
                                                       {'name': 'restraint_combination_id', 'type': 'positive-int', 'mandatory': False,
                                                        'enforce-non-zero': True},
                                                       {'name': 'weight', 'type': 'range-float', 'mandatory': True,
                                                        'range': self.weight_range},
                                                       # 'enforce-non-zero': True},
                                                       {'name': 'target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                        'range': self.dihed_restraint_range,
                                                        'group': {'member-with': ['lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                  'coexist-with': None,
                                                                  'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                  'larger-than': ['upper_limit', 'upper_linear_limit'],
                                                                  'circular-shift': 360.0}},
                                                       {'name': 'target_value_uncertainty', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                        'range': self.dihed_restraint_error},
                                                       {'name': 'lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                        'range': self.dihed_restraint_range,
                                                        'group': {'member-with': ['target_value', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                  'coexist-with': None,  # ['lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                  'smaller-than': None,
                                                                  'larger-than': ['lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                  'circular-shift': 360.0}},
                                                       {'name': 'lower_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                        'range': self.dihed_restraint_range,
                                                        'group': {'member-with': ['target_value', 'lower_linear_limit', 'upper_limit', 'upper_linear_limit'],
                                                                  'coexist-with': None,  # ['upper_limit'],
                                                                  'smaller-than': ['lower_linear_limit'],
                                                                  'larger-than': ['upper_limit', 'upper_linear_limit'],
                                                                  'circular-shift': 360.0}},
                                                       {'name': 'upper_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                        'range': self.dihed_restraint_range,
                                                        'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_linear_limit'],
                                                                  'coexist-with': None,  # ['lower_limit'],
                                                                  'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                  'larger-than': ['upper_linear_limit'],
                                                                  'circular-shift': 360.0}},
                                                       {'name': 'upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                        'range': self.dihed_restraint_range,
                                                        'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                  'coexist-with': None,  # ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                  'smaller-than': ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                  'larger-than': None,
                                                                  'circular-shift': 360.0}},
                                                       {'name': 'name', 'type': 'str', 'mandatory': False}
                                                       ],
                                   'rdc_restraint': [{'name': 'index', 'type': 'index-int', 'mandatory': True},
                                                     # {'name': 'restraint_id', 'type': 'positive-int', 'mandatory': True,
                                                     # 'enforce-non-zero': True},
                                                     {'name': 'restraint_combination_id', 'type': 'positive-int', 'mandatory': False,
                                                      'enforce-non-zero': True},
                                                     {'name': 'target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                      'range': self.rdc_restraint_range,
                                                      'group': {'member-with': ['lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                'coexist-with': None,
                                                                'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                     {'name': 'target_value_uncertainty', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                      'range': self.rdc_restraint_error},
                                                     {'name': 'lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                      'range': self.rdc_restraint_range,
                                                      'group': {'member-with': ['target_value', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                'coexist-with': None,  # ['lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                'smaller-than': None,
                                                                'larger-than': ['lower_limit', 'upper_limit', 'upper_linear_limit']}},
                                                     {'name': 'lower_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                      'range': self.rdc_restraint_range,
                                                      'group': {'member-with': ['target_value', 'lower_linear_limit', 'upper_limit', 'upper_linear_limit'],
                                                                'coexist-with': None,  # ['upper_limit'],
                                                                'smaller-than': ['lower_linear_limit'],
                                                                'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                     {'name': 'upper_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                      'range': self.rdc_restraint_range,
                                                      'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_linear_limit'],
                                                                'coexist-with': None,  # ['lower_limit'],
                                                                'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                'larger-than': ['upper_linear_limit']}},
                                                     {'name': 'upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                      'range': self.rdc_restraint_range,
                                                      'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                'coexist-with': None,  # ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                'smaller-than': ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                'larger-than': None}},
                                                     {'name': 'scale', 'type': 'range-float', 'mandatory': False,
                                                      'range': self.scale_range,
                                                      'enforce-non-zero': True},
                                                     {'name': 'distance_dependent', 'type': 'bool', 'mandatory': False}
                                                     ],
                                   'spectral_peak': [{'name': 'index', 'type': 'index-int', 'mandatory': True},
                                                     {'name': 'peak_id', 'type': 'positive-int', 'mandatory': True,
                                                      'enforce-non-zero': True},
                                                     {'name': 'volume', 'type': 'float', 'mandatory': False, 'group-mandatory': True,
                                                      'group': {'member-with': ['height'],
                                                                'coexist-with': None}},
                                                     {'name': 'volume_uncertainty', 'type': 'positive-float', 'mandatory': False, 'void-zero': True},
                                                     {'name': 'height', 'type': 'float', 'mandatory': False, 'group-mandatory': True,
                                                      'group': {'member-with': ['volume'],
                                                                'coexist-with': None}},
                                                     {'name': 'height_uncertainty', 'type': 'positive-float', 'mandatory': False, 'void-zero': True}
                                                     ],
                                   'spectral_peak_alt': None
                                   },
                           'nmr-star': {'poly_seq': [{'name': 'Auth_asym_ID', 'type': 'str', 'mandatory': False},
                                                     {'name': 'Auth_seq_ID', 'type': 'int', 'mandatory': False},
                                                     {'name': 'Auth_comp_ID', 'type': 'str', 'mandatory': False},
                                                     {'name': 'Auth_variant_ID', 'type': 'str', 'mandatory': False},
                                                     {'name': 'Sequence_linking', 'type': 'enum', 'mandatory': False,
                                                      'enum': ('start', 'end', 'middle', 'cyclic', 'break', 'single', 'dummy'),
                                                      'enforce-enum': True},
                                                     {'name': 'Cis_residue', 'type': 'bool', 'mandatory': False},
                                                     {'name': 'NEF_index', 'type': 'index-int', 'mandatory': False},
                                                     {'name': 'Assembly_ID', 'type': 'pointer-index', 'mandatory': False, 'default': '1', 'default-from': 'parent'}
                                                     ],
                                        'entity': None,
                                        'chem_shift': [{'name': 'Atom_type', 'type': 'enum', 'mandatory': True, 'default-from': 'Atom_ID',
                                                        'enum': set(self.atom_isotopes.keys()),
                                                        'enforce-enum': True},
                                                       {'name': 'Atom_isotope_number', 'type': 'enum-int', 'mandatory': True, 'default-from': 'Atom_ID',
                                                        'enum': set(isotope_nums),
                                                        'enforce-enum': True},
                                                       {'name': 'Val', 'type': 'range-float', 'mandatory': True,
                                                        'range': self.chem_shift_range},
                                                       {'name': 'Val_err', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                        'range': self.chem_shift_error},
                                                       {'name': 'Ambiguity_code', 'type': 'enum-int', 'mandatory': False,
                                                        'enum': self.bmrb_ambiguity_codes,
                                                        'enforce-enum': True},
                                                       {'name': 'Ambiguity_set_ID', 'type': 'positive-int', 'mandatory': False,
                                                        'enforce-non-zero': True},
                                                       {'name': 'Auth_asym_ID', 'type': 'str', 'mandatory': False},
                                                       {'name': 'Auth_seq_ID', 'type': 'int', 'mandatory': False},
                                                       {'name': 'Auth_comp_ID', 'type': 'str', 'mandatory': False},
                                                       {'name': 'Auth_atom_ID', 'type': 'str', 'mandatory': False},
                                                       {'name': 'Assigned_chem_shift_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default': '1', 'default-from': 'parent'}
                                                       ],
                                        'chem_shift_ref': [{'name': 'Atom_group', 'type': 'enum', 'mandatory': True,
                                                            'enum': ('methyl carbon', 'methyl carbons', 'methyl protons', 'methylene protons', 'nitrogen', 'phosphorus', 'protons')},
                                                           {'name': 'Chem_shift_val', 'type': 'float', 'mandatory': True},
                                                           {'name': 'Chem_shift_units', 'type': 'enum', 'mandatory': True,
                                                            'enum': ('ppm', 'Hz'),
                                                            'enforce-enum': True},
                                                           {'name': 'Correction_val', 'type': 'float', 'mandatory': False},
                                                           {'name': 'External_ref_axis', 'type': 'enum', 'mandatory': False,
                                                            'enum': ('parallel', 'perpendicular')},
                                                           {'name': 'External_ref_loc', 'type': 'enum', 'mandatory': False,
                                                            'enum': ('insert at center of a separate sample tube',
                                                                     'insert at center of experimental sample tube',
                                                                     'insert at outer edge of a separate sample tube',
                                                                     'insert at outer edge of experimental sample tube',
                                                                     'other',
                                                                     'separate tube (no insert) not similar to the experimental sample tube',
                                                                     'separate tube (no insert) similar to the experimental sample tube')},
                                                           {'name': 'External_ref_sample_geometry', 'type': 'enum', 'mandatory': False,
                                                            'enum': ('cylindrical', 'other', 'spherical')},
                                                           {'name': 'Indirect_shift_ratio', 'type': 'range-float', 'mandatory': False,
                                                            'range': {'min_exclusive': 0.0, 'max_inclusive': 1.0}},
                                                           {'name': 'Rank', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Ref_correction_type', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Ref_method', 'type': 'enum', 'mandatory': False,
                                                            'enum': ('external', 'internal', 'na')},
                                                           {'name': 'Ref_type', 'type': 'enum', 'mandatory': False,
                                                            'enum': ('direct', 'indirect')},
                                                           {'name': 'Solvent', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Chem_shift_reference_ID', 'type': 'pointer-index', 'mandatory': True, 'default': '1', 'default-from': 'parent'}
                                                           ],
                                        'dist_restraint': [{'name': 'Index_ID', 'type': 'index-int', 'mandatory': False},
                                                           # {'name': 'ID', 'type': 'positive-int', 'mandatory': True,
                                                           # 'enforce-non-zero': True},
                                                           {'name': 'Combination_ID', 'type': 'positive-int', 'mandatory': False,
                                                            'enforce-non-zero': True},
                                                           {'name': 'Member_logic_code', 'type': 'enum', 'mandatory': False,
                                                            'enum': ('OR', 'AND'),
                                                            'enforce-enum': True},
                                                           {'name': 'Target_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                            'range': self.dist_restraint_range,
                                                            'group': {'member-with': ['Lower_linear_limit',
                                                                                      'Upper_linear_limit',
                                                                                      'Distance_lower_bound_val',
                                                                                      'Distance_upper_bound_val'],
                                                                      'coexist-with': None,
                                                                      'smaller-than': ['Lower_linear_limit', 'Distance_lower_bound_val'],
                                                                      'larger-than': ['Distance_upper_bound_val', 'Upper_linear_limit']}},
                                                           {'name': 'Target_val_uncertainty', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                            'range': self.dist_restraint_error},
                                                           {'name': 'Lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                            'range': self.dist_restraint_range,
                                                            'group': {'member-with': ['Target_val',
                                                                                      'Upper_linear_limit',
                                                                                      'Distance_lower_bound_val',
                                                                                      'Distance_upper_bound_val'],
                                                                      'coexist-with': None,  # ['Upper_linear_limit', 'Distance_lower_bound_val', 'Distance_upper_bound_val'],
                                                                      'smaller-than': None,
                                                                      'larger-than': ['Distance_lower_bound_val', 'Distance_upper_bound_val', 'Upper_linear_limit']}},
                                                           {'name': 'Upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                            'range': self.dist_restraint_range,
                                                            'group': {'member-with': ['Target_val',
                                                                                      'Lower_linear_limit',
                                                                                      'Distance_lower_bound_val',
                                                                                      'Distance_upper_bound_val'],
                                                                      'coexist-with': None,  # ['Lower_linear_limit', 'Distance_lower_bound_val', 'Distance_upper_bound_val'],
                                                                      'smaller-than': ['Lower_linear_limit', 'Distance_lower_bound_val', 'Distance_upper_bound_val'],
                                                                      'larger-than': None}},
                                                           {'name': 'Distance_lower_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                            'range': self.dist_restraint_range,
                                                            'group': {'member-with': ['Target_val', 'Lower_linear_limit', 'Upper_linear_limit', 'Distance_upper_bound_val'],
                                                                      'coexist-with': None,  # ['Distance_upper_bound_val'],
                                                                      'smaller-than': ['Lower_linear_limit'],
                                                                      'larger-than': ['Distance_upper_bound_val', 'Upper_linear_limit']}},
                                                           {'name': 'Distance_upper_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True, 'void-zero': True,
                                                            'range': self.dist_restraint_range,
                                                            'group': {'member-with': ['Target_val', 'Lower_linear_limit', 'Upper_linear_limit', 'Distance_lower_bound_val'],
                                                                      'coexist-with': None,  # ['Distance_lower_bound_val'],
                                                                      'smaller-than': ['Lower_linear_limit', 'Distance_lower_bound_val'],
                                                                      'larger-than': ['Upper_linear_limit']}},
                                                           {'name': 'Distance_val', 'type': 'range-float', 'mandatory': False,
                                                                    'range': self.dist_restraint_range},
                                                           {'name': 'Weight', 'type': 'range-float', 'mandatory': False,
                                                            'range': self.weight_range},
                                                           # 'enforce-non-zero': True},
                                                           {'name': 'Auth_asym_ID_1', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Auth_seq_ID_1', 'type': 'int', 'mandatory': False},
                                                           {'name': 'Auth_comp_ID_1', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Auth_atom_ID_1', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Auth_asym_ID_2', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Auth_seq_ID_2', 'type': 'int', 'mandatory': False},
                                                           {'name': 'Auth_comp_ID_2', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Auth_atom_ID_2', 'type': 'str', 'mandatory': False},
                                                           {'name': 'Gen_dist_constraint_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default': '1', 'default-from': 'parent'}
                                                           ],
                                        'dihed_restraint': [{'name': 'Index_ID', 'type': 'index-int', 'mandatory': False},
                                                            # {'name': 'ID', 'type': 'index-int', 'mandatory': True,
                                                            # 'enforce-non-zero': True},
                                                            {'name': 'Combination_ID', 'type': 'positive-int', 'mandatory': False,
                                                             'enforce-non-zero': True},
                                                            {'name': 'Torsion_angle_name', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Angle_lower_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                             'range': self.dihed_restraint_range,
                                                             'group': {'member-with': ['Angle_target_val', 'Angle_lower_linear_limit', 'Angle_upper_linear_limit', 'Angle_upper_bound_val'],
                                                                       'coexist-with': None,  # ['Angle_upper_bound_val'],
                                                                       'smaller-than': ['Angle_lower_linear_limit'],
                                                                       'larger-than': ['Angle_upper_bound_val', 'Angle_upper_linear_limit'],
                                                                       'circular-shift': 360.0}},
                                                            {'name': 'Angle_upper_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                             'range': self.dihed_restraint_range,
                                                             'group': {'member-with': ['Angle_target_val', 'Angle_lower_linear_limit', 'Angle_upper_linear_limit', 'Angle_lower_bound_val'],
                                                                       'coexist-with': None,  # ['Angle_lower_bound_val'],
                                                                       'smaller-than': ['Angle_lower_bound_val', 'Angle_upper_linear_limit'],
                                                                       'larger-than': ['Angle_upper_linear_limit'],
                                                                       'circular-shift': 360.0}},
                                                            {'name': 'Angle_target_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                             'range': self.dihed_restraint_range,
                                                             'group': {'member-with': ['Angle_lower_linear_limit',
                                                                                       'Angle_upper_linear_limit',
                                                                                       'Angle_lower_bound_val',
                                                                                       'Angle_upper_bound_val'],
                                                                       'coexist-with': None,
                                                                       'smaller-than': ['Angle_lower_linear_limit', 'Angle_lower_bound_val'],
                                                                       'larger-than': ['Angle_upper_bound_val', 'Angle_upper_linear_limit'],
                                                                       'circular-shift': 360.0}},
                                                            {'name': 'Angle_target_val_err', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                             'range': self.dihed_restraint_error},
                                                            {'name': 'Angle_lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                             'range': self.dihed_restraint_range,
                                                             'group': {'member-with': ['Angle_target_val', 'Angle_upper_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                       'coexist-with': None,  # ['Angle_upper_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                       'smaller-than': None,
                                                                       'larger-than': ['Angle_lower_bound_val', 'Angle_upper_bound', 'Angle_upper_linear_limit'],
                                                                       'circular-shift': 360.0}},
                                                            {'name': 'Angle_upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                             'range': self.dihed_restraint_range,
                                                             'group': {'member-with': ['Angle_target_val', 'Angle_lower_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                       'coexist-with': None,  # ['Angle_lower_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                       'smaller-than': ['Angle_lower_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                       'larger-than': None,
                                                                       'circular-shift': 360.0}},
                                                            {'name': 'Weight', 'type': 'range-float', 'mandatory': False,
                                                             'range': self.weight_range},
                                                            # 'enforce-non-zero': True},
                                                            {'name': 'Auth_asym_ID_1', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_seq_ID_1', 'type': 'int', 'mandatory': False},
                                                            {'name': 'Auth_comp_ID_1', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_atom_ID_1', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_asym_ID_2', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_seq_ID_2', 'type': 'int', 'mandatory': False},
                                                            {'name': 'Auth_comp_ID_2', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_atom_ID_2', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_asym_ID_3', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_seq_ID_3', 'type': 'int', 'mandatory': False},
                                                            {'name': 'Auth_comp_ID_3', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_atom_ID_3', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_asym_ID_4', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_seq_ID_4', 'type': 'int', 'mandatory': False},
                                                            {'name': 'Auth_comp_ID_4', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Auth_atom_ID_4', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Torsion_angle_constraint_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default': '1', 'default-from': 'parent'}
                                                            ],
                                        'rdc_restraint': [{'name': 'Index_ID', 'type': 'index-int', 'mandatory': False},
                                                          # {'name': 'ID', 'type': 'index-int', 'mandatory': True,
                                                          # 'enforce-non-zero': True},
                                                          {'name': 'Combination_ID', 'type': 'positive-int', 'mandatory': False,
                                                           'enforce-non-zero': True},
                                                          {'name': 'Weight', 'type': 'range-float', 'mandatory': False,
                                                           'range': self.weight_range},
                                                          # 'enforce-non-zero': True},
                                                          {'name': 'Target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                           'range': self.rdc_restraint_range,
                                                           'group': {'member-with': ['RDC_lower_linear_limit', 'RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                     'coexist-with': None,
                                                                     'smaller-than': ['RDC_lower_linear_limit', 'RDC_lower_bound'],
                                                                     'larger-than': ['RDC_upper_bound', 'RDC_upper_linear_limit']}},
                                                          {'name': 'Target_value_uncertainty', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                           'range': self.rdc_restraint_error},
                                                          {'name': 'RDC_lower_bound', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                           'range': self.rdc_restraint_range,
                                                           'group': {'member-with': ['Target_value', 'RDC_lower_linear_limit', 'RDC_upper_linear_limit', 'RDC_upper_bound'],
                                                                     'coexist-with': None,  # ['RDC_upper_bound'],
                                                                     'smaller-than': ['RDC_lower_linear_limit'],
                                                                     'larger-than': ['RDC_upper_bound', 'RDC_upper_linear_limit']}},
                                                          {'name': 'RDC_upper_bound', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                           'range': self.rdc_restraint_range,
                                                           'group': {'member-with': ['Target_value', 'RDC_lower_linear_limit', 'RDC_upper_linear_limit', 'RDC_lower_bound'],
                                                                     'coexist-with': None,  # ['RDC_lower_bound'],
                                                                     'smaller-than': ['RDC_lower_linear_limit', 'RDC_lower_bound'],
                                                                     'larger-than': ['RDC_upper_linear_limit']}},
                                                          {'name': 'RDC_lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                           'range': self.rdc_restraint_range,
                                                           'group': {'member-with': ['Target_value', 'RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                     'coexist-with': None,  # ['RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                     'smaller-than': None,
                                                                     'larger-than': ['RDC_lower_bound', 'RDC_upper_bound', 'RDC_upper_linear_limit']}},
                                                          {'name': 'RDC_upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                           'range': self.rdc_restraint_range,
                                                           'group': {'member-with': ['Target_value', 'RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                     'coexist-with': None,  # ['RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                     'smaller-than': ['RDC_lower_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                     'larger-than': None}},
                                                          {'name': 'RDC_val', 'type': 'range-float', 'mandatory': False,
                                                           'range': self.rdc_restraint_range},
                                                          {'name': 'RDC_val_err', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                           'range': self.rdc_restraint_error},
                                                          {'name': 'RDC_val_scale_factor', 'type': 'range-float', 'mandatory': False,
                                                           'range': self.scale_range,
                                                           'enforce-non-zero': True},
                                                          {'name': 'RDC_distant_dependent', 'type': 'bool', 'mandatory': False},
                                                          {'name': 'Auth_asym_ID_1', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_seq_ID_1', 'type': 'int', 'mandatory': False},
                                                          {'name': 'Auth_comp_ID_1', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_atom_ID_1', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_asym_ID_2', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_seq_ID_2', 'type': 'int', 'mandatory': False},
                                                          {'name': 'Auth_comp_ID_2', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_atom_ID_2', 'type': 'str', 'mandatory': False},
                                                          {'name': 'RDC_constraint_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default': '1', 'default-from': 'parent'}
                                                          ],
                                        'spectral_peak': [{'name': 'Index_ID', 'type': 'index-int', 'mandatory': False},
                                                          # {'name': 'ID', 'type': 'positive-int', 'mandatory': True,
                                                          # 'enforce-non-zero': True},
                                                          {'name': 'Volume', 'type': 'float', 'mandatory': False, 'group-mandatory': True,
                                                           'group': {'member-with': ['Height'],
                                                                     'coexist-with': None}},
                                                          {'name': 'Volume_uncertainty', 'type': 'positive-float', 'mandatory': False, 'void-zero': True},
                                                          {'name': 'Height', 'type': 'float', 'mandatory': False, 'group-mandatory': True,
                                                           'group': {'member-with': ['Volume'],
                                                                     'coexist-with': None}},
                                                          {'name': 'Height_uncertainty', 'type': 'positive-float', 'mandatory': False, 'void-zero': True},
                                                          {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default': '1', 'default-from': 'parent'}
                                                          ],
                                        'spectral_peak_alt': [{'name': 'Index_ID', 'type': 'index-int', 'mandatory': False},
                                                              {'name': 'Figure_of_merit', 'type': 'range-float', 'mandatory': False,
                                                               'range': self.weight_range},
                                                              {'name': 'Restraint', 'type': 'enum', 'mandatory': False,
                                                               'enum': ('no', 'yes')}
                                                              ]
                                        }
                           }

        # data items of loop to check consistency
        self.consist_data_items = {'nef': {'dist_restraint': [{'name': 'target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                               'range': self.dist_restraint_range,
                                                               'group': {'member-with': ['lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                         'coexist-with': None,
                                                                         'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                         'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                              {'name': 'target_value_uncertainty', 'type': 'range-float', 'mandatory': False,
                                                               'range': self.dist_restraint_error},
                                                              {'name': 'lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                               'range': self.dist_restraint_range,
                                                               'group': {'member-with': ['target_value', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                         'coexist-with': None,  # ['lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                         'smaller-than': None,
                                                                         'larger-than': ['lower_limit', 'upper_limit', 'upper_linear_limit']}},
                                                              {'name': 'lower_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                               'range': self.dist_restraint_range,
                                                               'group': {'member-with': ['target_value', 'lower_linear_limit', 'upper_limit', 'upper_linear_limit'],
                                                                         'coexist-with': None,  # ['upper_limit'],
                                                                         'smaller-than': ['lower_linear_limit'],
                                                                         'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                              {'name': 'upper_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                               'range': self.dist_restraint_range,
                                                               'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_linear_limit'],
                                                                         'coexist-with': None,  # ['lower_limit'],
                                                                         'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                         'larger-than': ['upper_linear_limit']}},
                                                              {'name': 'upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                               'range': self.dist_restraint_range,
                                                               'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                         'coexist-with': None,  # ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                         'smaller-than': ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                         'larger-than': None}}
                                                              ],
                                           'dihed_restraint': [{'name': 'target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                               'range': self.dihed_restraint_range,
                                                                'group': {'member-with': ['lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                          'coexist-with': None,
                                                                          'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                          'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                               {'name': 'target_value_uncertainty', 'type': 'range-float', 'mandatory': False,
                                                                'range': self.dihed_restraint_error},
                                                               {'name': 'lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                'range': self.dihed_restraint_range,
                                                                'group': {'member-with': ['target_value', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                          'coexist-with': None,  # ['lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                          'smaller-than': None,
                                                                          'larger-than': ['lower_limit', 'upper_limit', 'upper_linear_limit']}},
                                                               {'name': 'lower_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                'range': self.dihed_restraint_range,
                                                                'group': {'member-with': ['target_value', 'lower_linear_limit', 'upper_limit', 'upper_linear_limit'],
                                                                          'coexist-with': None,  # ['upper_limit'],
                                                                          'smaller-than': ['lower_linear_limit'],
                                                                          'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                               {'name': 'upper_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                'range': self.dihed_restraint_range,
                                                                'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_linear_limit'],
                                                                          'coexist-with': None,  # ['lower_limit'],
                                                                          'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                          'larger-than': ['upper_linear_limit']}},
                                                               {'name': 'upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                'range': self.dihed_restraint_range,
                                                                'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                          'coexist-with': None,  # ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                          'smaller-than': ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                          'larger-than': None}}
                                                               ],
                                           'rdc_restraint': [{'name': 'target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                              'range': self.rdc_restraint_range,
                                                              'group': {'member-with': ['lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                        'coexist-with': None,
                                                                        'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                        'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                             {'name': 'target_value_uncertainty', 'type': 'range-float', 'mandatory': False,
                                                              'range': self.rdc_restraint_error},
                                                             {'name': 'lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                              'range': self.rdc_restraint_range,
                                                              'group': {'member-with': ['target_value', 'lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                        'coexist-with': None,  # ['lower_limit', 'upper_limit', 'upper_linear_limit'],
                                                                        'smaller-than': None,
                                                                        'larger-than': ['lower_limit', 'upper_limit', 'upper_linear_limit']}},
                                                             {'name': 'lower_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                              'range': self.rdc_restraint_range,
                                                              'group': {'member-with': ['target_value', 'lower_linear_limit', 'upper_limit', 'upper_linear_limit'],
                                                                        'coexist-with': None,  # ['upper_limit'],
                                                                        'smaller-than': ['lower_linear_limit'],
                                                                        'larger-than': ['upper_limit', 'upper_linear_limit']}},
                                                             {'name': 'upper_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                              'range': self.rdc_restraint_range,
                                                              'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_linear_limit'],
                                                                        'coexist-with': None,  # ['lower_limit'],
                                                                        'smaller-than': ['lower_linear_limit', 'lower_limit'],
                                                                        'larger-than': ['upper_linear_limit']}},
                                                             {'name': 'upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                              'range': self.rdc_restraint_range,
                                                              'group': {'member-with': ['target_value', 'lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                        'coexist-with': None,  # ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                        'smaller-than': ['lower_linear_limit', 'lower_limit', 'upper_limit'],
                                                                        'larger-than': None}}
                                                             ],
                                           'spectral_peak': None,
                                           'spectral_peak_alt': None
                                           },
                                   'nmr-star': {'dist_restraint': [{'name': 'Target_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                    'range': self.dist_restraint_range,
                                                                    'group': {'member-with': ['Lower_linear_limit',
                                                                                              'Upper_linear_limit',
                                                                                              'Distance_lower_bound_val',
                                                                                              'Distance_upper_bound_val'],
                                                                              'coexist-with': None,
                                                                              'smaller-than': ['Lower_linear_limit', 'Distance_lower_bound_val'],
                                                                              'larger-than': ['Distance_upper_bound_val', 'Upper_linear_limit']}},
                                                                   {'name': 'Target_val_uncertainty', 'type': 'range-float', 'mandatory': False,
                                                                    'range': self.dist_restraint_error},
                                                                   {'name': 'Lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                    'range': self.dist_restraint_range,
                                                                    'group': {'member-with': ['Target_val',
                                                                                              'Upper_linear_limit',
                                                                                              'Distance_lower_bound_val',
                                                                                              'Distance_upper_bound_val'],
                                                                              'coexist-with': None,  # ['Upper_linear_limit', 'Distance_lower_bound_val', 'Distance_upper_bound_val'],
                                                                              'smaller-than': None,
                                                                              'larger-than': ['Distance_lower_bound_val', 'Distance_upper_bound_val', 'Upper_linear_limit']}},
                                                                   {'name': 'Upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                    'range': self.dist_restraint_range,
                                                                    'group': {'member-with': ['Target_val',
                                                                                              'Lower_linear_limit',
                                                                                              'Distance_lower_bound_val',
                                                                                              'Distance_upper_bound_val'],
                                                                              'coexist-with': None,  # ['Lower_linear_limit', 'Distance_lower_bound_val', 'Distance_upper_bound_val'],
                                                                              'smaller-than': ['Lower_linear_limit', 'Distance_lower_bound_val', 'Distance_upper_bound_val'],
                                                                              'larger-than': None}},
                                                                   {'name': 'Distance_lower_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                    'range': self.dist_restraint_range,
                                                                    'group': {'member-with': ['Target_val',
                                                                                              'Lower_linear_limit',
                                                                                              'Upper_linear_limit',
                                                                                              'Distance_upper_bound_val'],
                                                                              'coexist-with': None,  # ['Distance_upper_bound_val'],
                                                                              'smaller-than': ['Lower_linear_limit'],
                                                                              'larger-than': ['Distance_upper_bound_val', 'Upper_linear_limit']}},
                                                                   {'name': 'Distance_upper_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                    'range': self.dist_restraint_range,
                                                                    'group': {'member-with': ['Target_val',
                                                                                              'Lower_linear_limit',
                                                                                              'Upper_linear_limit',
                                                                                              'Distance_lower_bound_val'],
                                                                              'coexist-with': None,  # ['Distance_lower_bound_val'],
                                                                              'smaller-than': ['Lower_linear_limit', 'Distance_lower_bound_val'],
                                                                              'larger-than': ['Upper_linear_limit']}},
                                                                   {'name': 'Distance_val', 'type': 'range-float', 'mandatory': False,
                                                                    'range': self.dist_restraint_range}
                                                                   ],
                                                'dihed_restraint': [{'name': 'Angle_lower_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                     'range': self.dihed_restraint_range,
                                                                     'group': {'member-with': ['Angle_target_val',
                                                                                               'Angle_lower_linear_limit',
                                                                                               'Angle_upper_linear_limit',
                                                                                               'Angle_upper_bound_val'],
                                                                               'coexist-with': None,  # ['Angle_upper_bound_val'],
                                                                               'smaller-than': ['Angle_lower_linear_limit'],
                                                                               'larger-than': ['Angle_upper_bound_val', 'Angle_upper_linear_limit']}},
                                                                    {'name': 'Angle_upper_bound_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                     'range': self.dihed_restraint_range,
                                                                     'group': {'member-with': ['Angle_target_val',
                                                                                               'Angle_lower_linear_limit',
                                                                                               'Angle_upper_linear_limit',
                                                                                               'Angle_lower_bound_val'],
                                                                               'coexist-with': None,  # ['Angle_lower_bound_val'],
                                                                               'smaller-than': ['Angle_lower_linear_limit', 'Angle_lower_bound_val'],
                                                                               'larger-than': ['Angle_upper_linear_limit']}},
                                                                    {'name': 'Angle_target_val', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                     'range': self.dihed_restraint_range,
                                                                     'group': {'member-with': ['Angle_lower_linear_limit',
                                                                                               'Angle_upper_linear_limit',
                                                                                               'Angle_lower_bound_val',
                                                                                               'Angle_upper_bound_val'],
                                                                               'coexist-with': None,
                                                                               'smaller-than': ['Angle_lower_linear_limit', 'Angle_lower_bound_val'],
                                                                               'larger-than': ['Angle_upper_bound_val', 'Angle_upper_linear_limit']}},
                                                                    {'name': 'Angle_target_val_err', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                                     'range': self.dihed_restraint_error},
                                                                    {'name': 'Angle_lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                     'range': self.dihed_restraint_range,
                                                                     'group': {'member-with': ['Angle_target_val',
                                                                                               'Angle_upper_linear_limit',
                                                                                               'Angle_lower_bound_val',
                                                                                               'Angle_upper_bound_val'],
                                                                               'coexist-with': None,  # ['Angle_upper_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                               'smaller-than': None,
                                                                               'larger-than': ['Angle_lower_bound_val', 'Angle_upper_bound_val', 'Angle_upper_linear_limit']}},
                                                                    {'name': 'Angle_upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                     'range': self.dihed_restraint_range,
                                                                     'group': {'member-with': ['Angle_target_val',
                                                                                               'Angle_lower_linear_limit',
                                                                                               'Angle_lower_bound_val',
                                                                                               'Angle_upper_bound_val'],
                                                                               'coexist-with': None,  # ['Angle_lower_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                               'smaller-than': ['Angle_lower_linear_limit', 'Angle_lower_bound_val', 'Angle_upper_bound_val'],
                                                                               'larger-than': None}}
                                                                    ],
                                                'rdc_restraint': [{'name': 'Target_value', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                   'range': self.rdc_restraint_range,
                                                                   'group': {'member-with': ['RDC_lower_linear_limit',
                                                                                             'RDC_upper_linear_limit',
                                                                                             'RDC_lower_bound',
                                                                                             'RDC_upper_bound'],
                                                                             'coexist-with': None,
                                                                             'smaller-than': ['RDC_lower_linear_limit', 'RDC_lower_bound'],
                                                                             'larger-than': ['RDC_upper_bound', 'RDC_upper_linear_limit']}},
                                                                  {'name': 'Target_value_uncertainty', 'type': 'range-float', 'mandatory': False,
                                                                   'range': self.rdc_restraint_error},
                                                                  {'name': 'RDC_lower_bound', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                   'range': self.rdc_restraint_range,
                                                                   'group': {'member-with': ['Target_value',
                                                                                             'RDC_lower_linear_limit',
                                                                                             'RDC_upper_linear_limit',
                                                                                             'RDC_upper_bound'],
                                                                             'coexist-with': None,  # ['RDC_upper_bound'],
                                                                             'smaller-than': ['RDC_lower_linear_limit'],
                                                                             'larger-than': ['RDC_upper_boud', 'RDC_upper_linear_limit']}},
                                                                  {'name': 'RDC_upper_bound', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                   'range': self.rdc_restraint_range,
                                                                   'group': {'member-with': ['Target_value',
                                                                                             'RDC_lower_linear_limit',
                                                                                             'RDC_upper_linear_limit',
                                                                                             'RDC_lower_bound'],
                                                                             'coexist-with': None,  # ['RDC_lower_bound'],
                                                                             'smaller-than': ['RDC_lower_linear_limit', 'RDC_lower_bound'],
                                                                             'larger-than': ['RDC_upper_linear_limit']}},
                                                                  {'name': 'RDC_lower_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                   'range': self.rdc_restraint_range,
                                                                   'group': {'member-with': ['Target_value', 'RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                             'coexist-with': None,  # ['RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                             'smaller-than': None,
                                                                             'larger-than': ['RDC_lower_bound', 'RDC_upper_bound', 'RDC_upper_linear_limit']}},
                                                                  {'name': 'RDC_upper_linear_limit', 'type': 'range-float', 'mandatory': False, 'group-mandatory': True,
                                                                   'range': self.rdc_restraint_range,
                                                                   'group': {'member-with': ['Target_value', 'RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                             'coexist-with': None,  # ['RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound'],
                                                                             'smaller-than': None,
                                                                             'larger-than': ['RDC_upper_linear_limit', 'RDC_lower_bound', 'RDC_upper_bound']}},
                                                                  {'name': 'RDC_val', 'type': 'range-float', 'mandatory': False,
                                                                   'range': self.rdc_restraint_range},
                                                                  {'name': 'RDC_val_err', 'type': 'range-float', 'mandatory': False,
                                                                   'range': self.rdc_restraint_error}
                                                                  ],
                                                'spectral_peak': None,
                                                'spectral_peak_alt': None
                                                }
                                   }

        # common potential descriptor items
        self.potential_items = {'nef': {'dist_restraint': {'target_value': 'target_value',
                                                           'lower_limit': 'lower_limit',
                                                           'upper_limit': 'upper_limit',
                                                           'lower_linear_limit': 'lower_linear_limit',
                                                           'upper_linear_limit': 'upper_linear_limit'},
                                        'dihed_restraint': {'target_value': 'target_value',
                                                            'lower_limit': 'lower_limit',
                                                            'upper_limit': 'upper_limit',
                                                            'lower_linear_limit': 'lower_linear_limit',
                                                            'upper_linear_limit': 'upper_linear_limit'},
                                        'rdc_restraint': {'target_value': 'target_value',
                                                          'lower_limit': 'lower_limit',
                                                          'upper_limit': 'upper_limit',
                                                          'lower_linear_limit': 'lower_linear_limit',
                                                          'upper_linear_limit': 'upper_linear_limit'}
                                        },
                                'nmr-star': {'dist_restraint': {'target_value': 'Target_val',
                                                                'target_value_alt': 'Distance_val',
                                                                'lower_limit': 'Distance_lower_bound_val',
                                                                'upper_limit': 'Distance_upper_bound_val',
                                                                'lower_linear_limit': 'Lower_linear_limit',
                                                                'upper_linear_limit': 'Upper_linear_limit'},
                                             'dihed_restraint': {'target_value': 'Angle_target_val',
                                                                 'lower_limit': 'Angle_lower_bound_val',
                                                                 'upper_limit': 'Angle_upper_bound_val',
                                                                 'lower_linear_limit': 'Angle_lower_linear_limit',
                                                                 'upper_linear_limit': 'Angle_upper_linear_limit'},
                                             'rdc_restraint': {'target_value': 'Target_value',
                                                               'target_value_alt': 'RDC_val',
                                                               'lower_limit': 'RDC_lower_bound',
                                                               'upper_limit': 'RDC_upper_bound',
                                                               'lower_linear_limit': 'RDC_lower_linear_limit',
                                                               'upper_linear_limit': 'RDC_upper_linear_limit'}
                                             }
                                }

        # loop data items for spectral peak
        self.pk_data_items = {'nef': [{'name': 'position_uncertainty_%s', 'type': 'range-float', 'mandatory': False,
                                       'range': self.chem_shift_error},
                                      {'name': 'chain_code_%s', 'type': 'str', 'mandatory': False,
                                       'relax-key-if-exist': True},
                                      {'name': 'sequence_code_%s', 'type': 'int', 'mandatory': False,
                                       'relax-key-if-exist': True,
                                       'clear-bad-pattern': True},
                                      {'name': 'residue_name_%s', 'type': 'str', 'mandatory': False,
                                       'relax-key-if-exist': True,
                                       'uppercase': True,
                                       'clear-bad-pattern': True},
                                      {'name': 'atom_name_%s', 'type': 'str', 'mandatory': False,
                                       'relax-key-if-exist': True,
                                       'clear-bad-pattern': True}],
                              'nmr-star': [{'name': 'Position_uncertainty_%s', 'type': 'range-float', 'mandatory': False,
                                            'range': self.chem_shift_error},
                                           {'name': 'Entity_assembly_ID_%s', 'type': 'positive-int-as-str', 'mandatory': False,
                                            'default': '1', 'default-from': 'Auth_asym_ID_%s',
                                            'enforce-non-zero': True,
                                            'relax-key-if-exist': True},
                                           {'name': 'Comp_index_ID_%s', 'type': 'int', 'mandatory': False,
                                            'relax-key-if-exist': True,
                                            'clear-bad-pattern': True},
                                           {'name': 'Comp_ID_%s', 'type': 'str', 'mandatory': False,
                                            'relax-key-if-exist': True,
                                            'uppercase': True,
                                            'clear-bad-pattern': True},
                                           {'name': 'Atom_ID_%s', 'type': 'str', 'mandatory': False,
                                            'relax-key-if-exist': True,
                                            'clear-bad-pattern': True},
                                           {'name': 'Auth_asym_ID_%s', 'type': 'str', 'mandatory': False},
                                           {'name': 'Auth_seq_ID_%s', 'type': 'int', 'mandatory': False},
                                           {'name': 'Auth_comp_ID_%s', 'type': 'str', 'mandatory': False},
                                           {'name': 'Auth_atom_ID_%s', 'type': 'str', 'mandatory': False}]
                              }

        # number of dimension of spectral peak
        self.num_dim_items = {'nef': 'num_dimensions', 'nmr-star': 'Number_of_spectral_dimensions'}

        # allowed loop tags
        self.allowed_tags = {'nef': {'entry_info': ['program_name', 'script_name', 'script'],
                                     'poly_seq': ['index', 'chain_code', 'sequence_code', 'residue_name', 'linking', 'residue_variant', 'cis_peptide'],
                                     'entity': None,
                                     'chem_shift': ['chain_code', 'sequence_code', 'residue_name', 'atom_name', 'value', 'value_uncertainty', 'element', 'isotope_number'],
                                     'chem_shift_ref': None,
                                     'dist_restraint': ['index', 'restraint_id', 'restraint_combination_id', 'chain_code_1',
                                                        'sequence_code_1', 'residue_name_1', 'atom_name_1', 'chain_code_2',
                                                        'sequence_code_2', 'residue_name_2', 'atom_name_2', 'weight', 'target_value',
                                                        'target_value_uncertainty', 'lower_linear_limit', 'lower_limit',
                                                        'upper_limit', 'upper_linear_limit'],
                                     'dihed_restraint': ['index', 'restraint_id', 'restraint_combination_id',
                                                         'chain_code_1', 'sequence_code_1', 'residue_name_1', 'atom_name_1',
                                                         'chain_code_2', 'sequence_code_2', 'residue_name_2', 'atom_name_2',
                                                         'chain_code_3', 'sequence_code_3', 'residue_name_3', 'atom_name_3',
                                                         'chain_code_4', 'sequence_code_4', 'residue_name_4', 'atom_name_4',
                                                         'weight', 'target_value', 'target_value_uncertainty',
                                                         'lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit',
                                                         'name'],
                                     'rdc_restraint': ['index', 'restraint_id', 'restraint_combination_id',
                                                       'chain_code_1', 'sequence_code_1', 'residue_name_1', 'atom_name_1',
                                                       'chain_code_2', 'sequence_code_2', 'residue_name_2', 'atom_name_2',
                                                       'weight', 'target_value', 'target_value_uncertainty',
                                                       'lower_linear_limit', 'lower_limit', 'upper_limit', 'upper_linear_limit',
                                                       'scale', 'distance_dependent'],
                                     'spectral_peak': ['index', 'peak_id', 'volume', 'volume_uncertainty', 'height', 'height_uncertainty',
                                                       'position_1', 'position_uncertainty_1', 'position_2', 'position_uncertainty_2',
                                                       'position_3', 'position_uncertainty_3', 'position_4', 'position_uncertainty_4',
                                                       'position_5', 'position_uncertainty_5', 'position_6', 'position_uncertainty_6',
                                                       'position_7', 'position_uncertainty_7', 'position_8', 'position_uncertainty_8',
                                                       'position_9', 'position_uncertainty_9', 'position_10', 'position_uncertainty_10',
                                                       'position_11', 'position_uncertainty_11', 'position_12', 'position_uncertainty_12',
                                                       'position_13', 'position_uncertainty_13', 'position_14', 'position_uncertainty_14',
                                                       'position_15', 'position_uncertainty_15',
                                                       'chain_code_1', 'sequence_code_1', 'residue_name_1', 'atom_name_1',
                                                       'chain_code_2', 'sequence_code_2', 'residue_name_2', 'atom_name_2',
                                                       'chain_code_3', 'sequence_code_3', 'residue_name_3', 'atom_name_3',
                                                       'chain_code_4', 'sequence_code_4', 'residue_name_4', 'atom_name_4',
                                                       'chain_code_5', 'sequence_code_5', 'residue_name_5', 'atom_name_5',
                                                       'chain_code_6', 'sequence_code_6', 'residue_name_6', 'atom_name_6',
                                                       'chain_code_7', 'sequence_code_7', 'residue_name_7', 'atom_name_7',
                                                       'chain_code_8', 'sequence_code_8', 'residue_name_8', 'atom_name_8',
                                                       'chain_code_9', 'sequence_code_9', 'residue_name_9', 'atom_name_9',
                                                       'chain_code_10', 'sequence_code_10', 'residue_name_10', 'atom_name_10',
                                                       'chain_code_11', 'sequence_code_11', 'residue_name_11', 'atom_name_11',
                                                       'chain_code_12', 'sequence_code_12', 'residue_name_12', 'atom_name_12',
                                                       'chain_code_13', 'sequence_code_13', 'residue_name_13', 'atom_name_13',
                                                       'chain_code_14', 'sequence_code_14', 'residue_name_14', 'atom_name_14',
                                                       'chain_code_15', 'sequence_code_15', 'residue_name_15', 'atom_name_15'],
                                     'spectral_peak_alt': None
                                     },
                             'nmr-star': {'entry_info': ['Software_ID', 'Software_label', 'Methods_ID', 'Methods_label', 'Software_name',
                                                         'Script_name', 'Script', 'Software_specific_info', 'Sf_ID', 'Entry_ID', 'Software_applied_list_ID'],
                                          'poly_seq': ['Assembly_chem_comp_ID', 'Entity_assembly_ID', 'Entity_ID', 'Comp_index_ID',
                                                       'Comp_ID', 'Seq_ID',
                                                       'Auth_entity_assembly_ID', 'Auth_asym_ID', 'Auth_seq_ID', 'Auth_comp_ID', 'Auth_variant_ID',
                                                       'Sequence_linking', 'Cis_residue', 'NEF_index', 'Sf_ID', 'Entry_ID', 'Assembly_ID'],
                                          'entity': None,
                                          'chem_shift': ['ID', 'Assembly_atom_ID', 'Entity_assembly_ID', 'Entity_assembly_asym_ID', 'Entity_ID', 'Comp_index_ID',
                                                         'Seq_ID', 'Comp_ID', 'Atom_ID', 'Atom_type', 'Atom_isotope_number',
                                                         'Val', 'Val_err', 'Assign_fig_of_merit', 'Ambiguity_code', 'Ambiguity_set_ID', 'Occupancy', 'Resonance_ID',
                                                         'Auth_entity_assembly_ID', 'Auth_asym_ID', 'Auth_seq_ID', 'Auth_comp_ID', 'Auth_atom_ID',
                                                         'PDB_record_ID', 'PDB_model_num', 'PDB_strand_ID', 'PDB_ins_code', 'PDB_residue_no', 'PDB_residue_name', 'PDB_atom_name',
                                                         'Original_PDB_strand_ID', 'Original_PDB_residue_no', 'Original_PDB_residue_name',
                                                         'Original_PDB_atom_name', 'Details', 'Sf_ID', 'Entry_ID', 'Assigned_chem_shift_list_ID'],
                                          'chem_shift_ref': ['Atom_type', 'Atom_isotope_number', 'Mol_common_name', 'Atom_group',
                                                             'Concentration_val', 'Concentration_units', 'Solvent', 'Rank',
                                                             'Chem_shift_units', 'Chem_shift_val', 'Ref_method', 'Ref_type', 'Indirect_shift_ratio',
                                                             'External_ref_loc', 'External_ref_sample_geometry', 'External_ref_axis', 'Indirect_shift_ratio_cit_ID'
                                                             , 'Indirect_shift_ratio_cit_label', 'Ref_correction_type', 'Correction_val', 'Correction_val_cit_ID',
                                                             'Correction_val_cit_label', 'Sf_ID', 'Entry_ID', 'Chem_shift_reference_ID'],
                                          'dist_restraint': ['Index_ID', 'ID', 'Combination_ID', 'Member_ID', 'Member_logic_code',
                                                             'Assembly_atom_ID_1', 'Entity_assembly_ID_1', 'Entity_ID_1', 'Comp_index_ID_1', 'Seq_ID_1',
                                                             'Comp_ID_1', 'Atom_ID_1', 'Atom_type_1', 'Atom_isotope_number_1', 'Resonance_ID_1',
                                                             'Assembly_atom_ID_2', 'Entity_assembly_ID_2', 'Entity_ID_2', 'Comp_index_ID_2', 'Seq_ID_2',
                                                             'Comp_ID_2', 'Atom_ID_2', 'Atom_type_2', 'Atom_isotope_number_2', 'Resonance_ID_2',
                                                             'Intensity_val', 'Intensity_lower_val_err', 'Intensity_upper_val_err', 'Distance_val',
                                                             'Target_val', 'Target_val_uncertainty', 'Lower_linear_limit', 'Upper_linear_limit',
                                                             'Distance_lower_bound_val', 'Distance_upper_bound_val', 'Contribution_fractional_val', 'Weight',
                                                             'Spectral_peak_ID', 'Spectral_peak_list_ID',
                                                             'PDB_record_ID_1', 'PDB_model_num_1', 'PDB_strand_ID_1', 'PDB_ins_code_1', 'PDB_residue_no_1',
                                                             'PDB_residue_name_1', 'PDB_atom_name_1', 'PDB_record_ID_2', 'PDB_model_num_2', 'PDB_strand_ID_2'
                                                             , 'PDB_ins_code_2', 'PDB_residue_no_2', 'PDB_residue_name_2', 'PDB_atom_name_2',
                                                             'Auth_entity_assembly_ID_1', 'Auth_asym_ID_1', 'Auth_chain_ID_1', 'Auth_seq_ID_1', 'Auth_comp_ID_1',
                                                             'Auth_atom_ID_1', 'Auth_alt_ID_1', 'Auth_atom_name_1',
                                                             'Auth_entity_assembly_ID_2', 'Auth_asym_ID_2', 'Auth_chain_ID_2', 'Auth_seq_ID_2', 'Auth_comp_ID_2',
                                                             'Auth_atom_ID_2', 'Auth_alt_ID_2', 'Auth_atom_name_2',
                                                             'Sf_ID', 'Entry_ID', 'Gen_dist_constraint_list_ID',
                                                             'Original_PDB_strand_ID_1', 'Original_PDB_residue_no_1', 'Original_PDB_residue_name_1',
                                                             'Original_PDB_strand_ID_2', 'Original_PDB_residue_no_2', 'Original_PDB_residue_name_2'],
                                          'dihed_restraint': ['Index_ID', 'ID', 'Combination_ID', 'Set_ID', 'Torsion_angle_name',
                                                              'Assembly_atom_ID_1', 'Entity_assembly_ID_1', 'Entity_ID_1', 'Comp_index_ID_1', 'Seq_ID_1',
                                                              'Comp_ID_1', 'Atom_ID_1', 'Atom_type_1', 'Resonance_ID_1',
                                                              'Assembly_atom_ID_2', 'Entity_assembly_ID_2', 'Entity_ID_2', 'Comp_index_ID_2', 'Seq_ID_2',
                                                              'Comp_ID_2', 'Atom_ID_2', 'Atom_type_2', 'Resonance_ID_2',
                                                              'Assembly_atom_ID_3', 'Entity_assembly_ID_3', 'Entity_ID_3', 'Comp_index_ID_3', 'Seq_ID_3',
                                                              'Comp_ID_3', 'Atom_ID_3', 'Atom_type_3', 'Resonance_ID_3',
                                                              'Assembly_atom_ID_4', 'Entity_assembly_ID_4', 'Entity_ID_4', 'Comp_index_ID_4', 'Seq_ID_4',
                                                              'Comp_ID_4', 'Atom_ID_4', 'Atom_type_4', 'Resonance_ID_4',
                                                              'Angle_lower_bound_val', 'Angle_upper_bound_val', 'Angle_target_val', 'Angle_target_val_err',
                                                              'Angle_lower_linear_limit', 'Angle_upper_linear_limit', 'Weight', 'Source_experiment_ID', 'Figure_of_merit',
                                                              'PDB_record_ID_1', 'PDB_model_num_1', 'PDB_strand_ID_1', 'PDB_ins_code_1', 'PDB_residue_no_1',
                                                              'PDB_residue_name_1', 'PDB_atom_name_1',
                                                              'PDB_record_ID_2', 'PDB_model_num_2', 'PDB_strand_ID_2', 'PDB_ins_code_2', 'PDB_residue_no_2',
                                                              'PDB_residue_name_2', 'PDB_atom_name_2',
                                                              'PDB_record_ID_3', 'PDB_model_num_3', 'PDB_strand_ID_3', 'PDB_ins_code_3', 'PDB_residue_no_3',
                                                              'PDB_residue_name_3', 'PDB_atom_name_3',
                                                              'PDB_record_ID_4', 'PDB_model_num_4', 'PDB_strand_ID_4', 'PDB_ins_code_4', 'PDB_residue_no_4',
                                                              'PDB_residue_name_4', 'PDB_atom_name_4',
                                                              'Auth_entity_assembly_ID_1', 'Auth_asym_ID_1', 'Auth_chain_ID_1', 'Auth_seq_ID_1', 'Auth_comp_ID_1',
                                                              'Auth_atom_ID_1', 'Auth_alt_ID_1', 'Auth_atom_name_1',
                                                              'Auth_entity_assembly_ID_2', 'Auth_asym_ID_2', 'Auth_chain_ID_2', 'Auth_seq_ID_2', 'Auth_comp_ID_2',
                                                              'Auth_atom_ID_2', 'Auth_alt_ID_2', 'Auth_atom_name_2',
                                                              'Auth_entity_assembly_ID_3', 'Auth_asym_ID_3', 'Auth_chain_ID_3', 'Auth_seq_ID_3', 'Auth_comp_ID_3',
                                                              'Auth_atom_ID_3', 'Auth_alt_ID_3', 'Auth_atom_name_3',
                                                              'Auth_entity_assembly_ID_4', 'Auth_asym_ID_4', 'Auth_chain_ID_4', 'Auth_seq_ID_4', 'Auth_comp_ID_4',
                                                              'Auth_atom_ID_4', 'Auth_alt_ID_4', 'Auth_atom_name_4',
                                                              'Sf_ID', 'Entry_ID', 'Torsion_angle_constraint_list_ID'],
                                          'rdc_restraint': ['Index_ID', 'ID', 'Combination_ID',
                                                            'Assembly_atom_ID_1', 'Entity_assembly_ID_1', 'Entity_ID_1', 'Comp_index_ID_1', 'Seq_ID_1',
                                                            'Comp_ID_1', 'Atom_ID_1', 'Atom_type_1', 'Atom_isotope_number_1', 'Resonance_ID_1',
                                                            'Assembly_atom_ID_2', 'Entity_assembly_ID_2', 'Entity_ID_2', 'Comp_index_ID_2', 'Seq_ID_2',
                                                            'Comp_ID_2', 'Atom_ID_2', 'Atom_type_2', 'Atom_isotope_number_2', 'Resonance_ID_2',
                                                            'Weight', 'RDC_val', 'RDC_val_err', 'Target_value', 'Target_value_uncertainty',
                                                            'RDC_lower_bound', 'RDC_upper_bound', 'RDC_lower_linear_limit', 'RDC_upper_linear_limit',
                                                            'RDC_val_scale_factor', 'RDC_bond_length', 'RDC_distant_dependent', 'Source_experiment_ID',
                                                            'PDB_record_ID_1', 'PDB_model_num_1', 'PDB_strand_ID_1', 'PDB_ins_code_1', 'PDB_residue_no_1',
                                                            'PDB_residue_name_1', 'PDB_atom_name_1',
                                                            'PDB_record_ID_2', 'PDB_model_num_2', 'PDB_strand_ID_2', 'PDB_ins_code_2', 'PDB_residue_no_2'
                                                            , 'PDB_residue_name_2', 'PDB_atom_name_2',
                                                            'Auth_entity_assembly_ID_1', 'Auth_asym_ID_1', 'Auth_chain_ID_1', 'Auth_seq_ID_1', 'Auth_comp_ID_1',
                                                            'Auth_atom_ID_1', 'Auth_alt_ID_1', 'Auth_atom_name_1',
                                                            'Auth_entity_assembly_ID_2', 'Auth_asym_ID_2', 'Auth_chain_ID_2', 'Auth_seq_ID_2', 'Auth_comp_ID_2',
                                                            'Auth_atom_ID_2', 'Auth_alt_ID_2', 'Auth_atom_name_2',
                                                            'Sf_ID', 'Entry_ID', 'RDC_constraint_list_ID'],
                                          'spectral_peak': ['Index_ID', 'ID', 'Volume', 'Volume_uncertainty', 'Height', 'Height_uncertainty', 'Figure_of_merit', 'Restraint',
                                                            'Position_1', 'Position_uncertainty_1', 'Line_width_1', 'Line_width_uncertainty_1',
                                                            'Position_2', 'Position_uncertainty_2', 'Line_width_2', 'Line_width_uncertainty_2',
                                                            'Position_3', 'Position_uncertainty_3', 'Line_width_3', 'Line_width_uncertainty_3',
                                                            'Position_4', 'Position_uncertainty_4', 'Line_width_4', 'Line_width_uncertainty_4',
                                                            'Position_5', 'Position_uncertainty_5', 'Line_width_5', 'Line_width_uncertainty_5',
                                                            'Position_6', 'Position_uncertainty_6', 'Line_width_6', 'Line_width_uncertainty_6',
                                                            'Position_7', 'Position_uncertainty_7', 'Line_width_7', 'Line_width_uncertainty_7',
                                                            'Position_8', 'Position_uncertainty_8', 'Line_width_8', 'Line_width_uncertainty_8',
                                                            'Position_9', 'Position_uncertainty_9', 'Line_width_9', 'Line_width_uncertainty_9',
                                                            'Position_10', 'Position_uncertainty_10', 'Line_width_10', 'Line_width_uncertainty_10',
                                                            'Position_11', 'Position_uncertainty_11', 'Line_width_11', 'Line_width_uncertainty_11',
                                                            'Position_12', 'Position_uncertainty_12', 'Line_width_12', 'Line_width_uncertainty_12',
                                                            'Position_13', 'Position_uncertainty_13', 'Line_width_13', 'Line_width_uncertainty_13',
                                                            'Position_14', 'Position_uncertainty_14', 'Line_width_14', 'Line_width_uncertainty_14',
                                                            'Position_15', 'Position_uncertainty_15', 'Line_width_15', 'Line_width_uncertainty_15',
                                                            'Entity_assembly_ID_1', 'Entity_ID_1', 'Comp_index_ID_1', 'Seq_ID_1', 'Comp_ID_1'
                                                            , 'Atom_ID_1', 'Ambiguity_code_1', 'Ambiguity_set_ID_1',
                                                            'Entity_assembly_ID_2', 'Entity_ID_2', 'Comp_index_ID_2', 'Seq_ID_2', 'Comp_ID_2',
                                                            'Atom_ID_2', 'Ambiguity_code_2', 'Ambiguity_set_ID_2',
                                                            'Entity_assembly_ID_3', 'Entity_ID_3', 'Comp_index_ID_3', 'Seq_ID_3', 'Comp_ID_3',
                                                            'Atom_ID_3', 'Ambiguity_code_3', 'Ambiguity_set_ID_3',
                                                            'Entity_assembly_ID_4', 'Entity_ID_4', 'Comp_index_ID_4', 'Seq_ID_4', 'Comp_ID_4',
                                                            'Atom_ID_4', 'Ambiguity_code_4', 'Ambiguity_set_ID_4',
                                                            'Entity_assembly_ID_5', 'Entity_ID_5', 'Comp_index_ID_5', 'Seq_ID_5', 'Comp_ID_5',
                                                            'Atom_ID_5', 'Ambiguity_code_5', 'Ambiguity_set_ID_5',
                                                            'Entity_assembly_ID_6', 'Entity_ID_6', 'Comp_index_ID_6', 'Seq_ID_6', 'Comp_ID_6',
                                                            'Atom_ID_6', 'Ambiguity_code_6', 'Ambiguity_set_ID_6',
                                                            'Entity_assembly_ID_7', 'Entity_ID_7', 'Comp_index_ID_7', 'Seq_ID_7', 'Comp_ID_7',
                                                            'Atom_ID_7', 'Ambiguity_code_7', 'Ambiguity_set_ID_7',
                                                            'Entity_assembly_ID_8', 'Entity_ID_8', 'Comp_index_ID_8', 'Seq_ID_8', 'Comp_ID_8',
                                                            'Atom_ID_8', 'Ambiguity_code_8', 'Ambiguity_set_ID_8',
                                                            'Entity_assembly_ID_9', 'Entity_ID_9', 'Comp_index_ID_9', 'Seq_ID_9', 'Comp_ID_9',
                                                            'Atom_ID_9', 'Ambiguity_code_9', 'Ambiguity_set_ID_9',
                                                            'Entity_assembly_ID_10', 'Entity_ID_10', 'Comp_index_ID_10', 'Seq_ID_10', 'Comp_ID_10',
                                                            'Atom_ID_10', 'Ambiguity_code_10', 'Ambiguity_set_ID_10',
                                                            'Entity_assembly_ID_11', 'Entity_ID_11', 'Comp_index_ID_11', 'Seq_ID_11', 'Comp_ID_11',
                                                            'Atom_ID_11', 'Ambiguity_code_11', 'Ambiguity_set_ID_11',
                                                            'Entity_assembly_ID_12', 'Entity_ID_12', 'Comp_index_ID_12', 'Seq_ID_12', 'Comp_ID_12',
                                                            'Atom_ID_12', 'Ambiguity_code_12', 'Ambiguity_set_ID_12',
                                                            'Entity_assembly_ID_13', 'Entity_ID_13', 'Comp_index_ID_13', 'Seq_ID_13', 'Comp_ID_13',
                                                            'Atom_ID_13', 'Ambiguity_code_13', 'Ambiguity_set_ID_13',
                                                            'Entity_assembly_ID_14', 'Entity_ID_14', 'Comp_index_ID_14', 'Seq_ID_14', 'Comp_ID_14',
                                                            'Atom_ID_14', 'Ambiguity_code_14', 'Ambiguity_set_ID_14',
                                                            'Entity_assembly_ID_15', 'Entity_ID_15', 'Comp_index_ID_15', 'Seq_ID_15', 'Comp_ID_15',
                                                            'Atom_ID_15', 'Ambiguity_code_15', 'Ambiguity_set_ID_15',
                                                            'Auth_entity_assembly_ID_1', 'Auth_entity_ID_1', 'Auth_asym_ID_1', 'Auth_seq_ID_1',
                                                            'Auth_comp_ID_1', 'Auth_atom_ID_1', 'Auth_ambiguity_code_1', 'Auth_ambiguity_set_ID_1',
                                                            'Auth_entity_assembly_ID_2', 'Auth_entity_ID_2', 'Auth_asym_ID_2', 'Auth_seq_ID_2',
                                                            'Auth_comp_ID_2', 'Auth_atom_ID_2', 'Auth_ambiguity_code_2', 'Auth_ambiguity_set_ID_2',
                                                            'Auth_entity_assembly_ID_3', 'Auth_entity_ID_3', 'Auth_asym_ID_3', 'Auth_seq_ID_3',
                                                            'Auth_comp_ID_3', 'Auth_atom_ID_3', 'Auth_ambiguity_code_3', 'Auth_ambiguity_set_ID_3',
                                                            'Auth_entity_assembly_ID_4', 'Auth_entity_ID_4', 'Auth_asym_ID_4', 'Auth_seq_ID_4',
                                                            'Auth_comp_ID_4', 'Auth_atom_ID_4', 'Auth_ambiguity_code_4', 'Auth_ambiguity_set_ID_4',
                                                            'Auth_entity_assembly_ID_5', 'Auth_entity_ID_5', 'Auth_asym_ID_5', 'Auth_seq_ID_5',
                                                            'Auth_comp_ID_5', 'Auth_atom_ID_5', 'Auth_ambiguity_code_5', 'Auth_ambiguity_set_ID_5',
                                                            'Auth_entity_assembly_ID_6', 'Auth_entity_ID_6', 'Auth_asym_ID_6', 'Auth_seq_ID_6',
                                                            'Auth_comp_ID_6', 'Auth_atom_ID_6', 'Auth_ambiguity_code_6', 'Auth_ambiguity_set_ID_6',
                                                            'Auth_entity_assembly_ID_7', 'Auth_entity_ID_7', 'Auth_asym_ID_7', 'Auth_seq_ID_7',
                                                            'Auth_comp_ID_7', 'Auth_atom_ID_7', 'Auth_ambiguity_code_7', 'Auth_ambiguity_set_ID_7',
                                                            'Auth_entity_assembly_ID_8', 'Auth_entity_ID_8', 'Auth_asym_ID_8', 'Auth_seq_ID_8',
                                                            'Auth_comp_ID_8', 'Auth_atom_ID_8', 'Auth_ambiguity_code_8', 'Auth_ambiguity_set_ID_8',
                                                            'Auth_entity_assembly_ID_9', 'Auth_entity_ID_9', 'Auth_asym_ID_9', 'Auth_seq_ID_9',
                                                            'Auth_comp_ID_9', 'Auth_atom_ID_9', 'Auth_ambiguity_code_9', 'Auth_ambiguity_set_ID_9',
                                                            'Auth_entity_assembly_ID_10', 'Auth_entity_ID_10', 'Auth_asym_ID_10', 'Auth_seq_ID_10',
                                                            'Auth_comp_ID_10', 'Auth_atom_ID_10', 'Auth_ambiguity_code_10', 'Auth_ambiguity_set_ID_10',
                                                            'Auth_entity_assembly_ID_11', 'Auth_entity_ID_11', 'Auth_asym_ID_11', 'Auth_seq_ID_11',
                                                            'Auth_comp_ID_11', 'Auth_atom_ID_11', 'Auth_ambiguity_code_11', 'Auth_ambiguity_set_ID_11',
                                                            'Auth_entity_assembly_ID_12', 'Auth_entity_ID_12', 'Auth_asym_ID_12', 'Auth_seq_ID_12',
                                                            'Auth_comp_ID_12', 'Auth_atom_ID_12', 'Auth_ambiguity_code_12', 'Auth_ambiguity_set_ID_12',
                                                            'Auth_entity_assembly_ID_13', 'Auth_entity_ID_13', 'Auth_asym_ID_13', 'Auth_seq_ID_13',
                                                            'Auth_comp_ID_13', 'Auth_atom_ID_13', 'Auth_ambiguity_code_13', 'Auth_ambiguity_set_ID_13',
                                                            'Auth_entity_assembly_ID_14', 'Auth_entity_ID_14', 'Auth_asym_ID_14', 'Auth_seq_ID_14',
                                                            'Auth_comp_ID_14', 'Auth_atom_ID_14', 'Auth_ambiguity_code_14', 'Auth_ambiguity_set_ID_14',
                                                            'Auth_entity_assembly_ID_15', 'Auth_entity_ID_15', 'Auth_asym_ID_15', 'Auth_seq_ID_15',
                                                            'Auth_comp_ID_15', 'Auth_atom_ID_15', 'Auth_ambiguity_code_15', 'Auth_ambiguity_set_ID_15',
                                                            'Details', 'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID'],
                                          'spectral_peak_alt': ['Index_ID', 'ID', 'Figure_of_merit', 'Restraint', 'Details', 'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID']
                                          }
                             }

        # disallowed loop tags of spectral peak
        self.spectral_peak_disallowed_tags = {'nef': ['position_%s', 'position_uncertainty_%s', 'chain_code_%s', 'sequence_code_%s', 'residue_name_%s', 'atom_name_%s'],
                                              'nmr-star': ['Position_%s', 'Position_uncertainty_%s', 'Line_width_%s',
                                                           'Line_width_uncertainty_%s', 'Entity_assembly_ID_%s', 'Entity_ID_%s',
                                                           'Comp_index_ID_%s', 'Seq_ID_%s', 'Comp_ID_%s', 'Atom_ID_%s',
                                                           'Ambiguity_code_%s', 'Ambiguity_set_ID_%s', 'Auth_entity_assembly_ID_%s',
                                                           'Auth_entity_ID_%s', 'Auth_asym_ID_%s', 'Auth_seq_ID_%s', 'Auth_comp_ID_%s',
                                                           'Auth_atom_ID_%s', 'Auth_ambiguity_code_%s', 'Auth_ambiguity_set_ID_%s']
                                              }

        # error template for missing mandatory loop tag
        self.__err_template_for_missing_mandatory_lp_tag = "The mandatory loop tag %r is missing. Please verify the value and re-upload the %s file."

        # saveframe tag prefixes (saveframe holder categories)
        self.sf_tag_prefixes = {'nef': {'entry_info': '_nef_nmr_meta_data',
                                        'poly_seq': '_nef_molecular_system',
                                        'entity': None,
                                        'chem_shift': '_nef_chemical_shift_list',
                                        'chem_shift_ref': None,
                                        'dist_restraint': '_nef_distance_restraint_list',
                                        'dihed_restraint': '_nef_dihedral_restraint_list',
                                        'rdc_restraint': '_nef_rdc_restraint_list',
                                        'spectral_peak': '_nef_nmr_spectrum',
                                        'spectral_peak_alt': None,
                                        },
                                'nmr-star': {'entry_info': '_Entry',
                                             'poly_seq': '_Assembly',
                                             'entity': '_Entity',
                                             'chem_shift': '_Assigned_chem_shift_list',
                                             'chem_shift_ref': '_Chem_shift_reference',
                                             'dist_restraint': '_Gen_dist_constraint_list',
                                             'dihed_restraint': '_Torsion_angle_constraint_list',
                                             'rdc_restraint': '_RDC_constraint_list',
                                             'spectral_peak': '_Spectral_peak_list',
                                             'spectral_peak_alt': '_Spectral_peak_list'
                                             }
                                }

        self.alt_potential_type = {'?': 'undefined'}

        # alternative dictionary of constraint type
        self.dist_alt_constraint_type = {'nef': {'NOE': 'noe',
                                                 'NOE build-up': 'noe_build_up',
                                                 'noe build-up': 'noe_build_up',
                                                 'NOE buildup': 'noe_build_up',
                                                 'noe buildup': 'noe_build_up',
                                                 'NOE build up': 'noe_build_up',
                                                 'noe build up': 'noe_build_up',
                                                 'noe not seen': 'noe_not_seen',
                                                 'ROE': 'roe',
                                                 'roe build-up': 'roe_build_up',
                                                 'ROE buildup': 'roe_build_up',
                                                 'roe buildup': 'roe_build_up',
                                                 'ROE build up': 'roe_build_up',
                                                 'roe build up': 'roe_build_up',
                                                 'hydrogen bond': 'hbond',
                                                 'Hbond': 'hbond',
                                                 'HBond': 'hbond',
                                                 'H-bond': 'hbond',
                                                 'h-bond': 'hbond',
                                                 'H-Bond': 'hbond',
                                                 'Hydrogen bond': 'hbond',
                                                 'disulfide bond': 'disulfide_bond',
                                                 'Disulfide bond': 'disulfide_bond',
                                                 'S-S bond': 'disulfide_bond',
                                                 'SS bond': 'disulfide_bond',
                                                 'SS-bond': 'disulfide_bond',
                                                 'disulfide bridge': 'disulfide_bond',
                                                 'Disulfide bridge': 'disulfide_bond',
                                                 'paramagnetic relaxation': 'pre',
                                                 'PRE': 'pre',
                                                 'Paramagnetic relaxation': 'pre',
                                                 'paramagnetic relaxation enhancement': 'pre',
                                                 'Paramagnetic relaxation enhancement': 'pre',
                                                 'general distance': 'undefined',
                                                 'distance': 'undefined',
                                                 'Mutation': 'mutation',
                                                 'chemical shift perturbation': 'shift_perturbation',
                                                 'shift perturbation': 'shift_perturbation',
                                                 'chem shift perturbation': 'shift_perturbation',
                                                 'CS perturbation': 'shift_perturbation',
                                                 'csp': 'shift_perturbation',
                                                 'CSP': 'shift_perturbation',
                                                 '?': 'undefined'
                                                 },
                                         'nmr-star': {'noe': 'NOE',
                                                      'noe_build_up': 'NOE build-up',
                                                      'noe build-up': 'NOE build-up',
                                                      'NOE buildup': 'NOE build-up',
                                                      'noe buildup': 'NOE build-up',
                                                      'NOE build up': 'NOE build-up',
                                                      'noe build up': 'NOE build-up',
                                                      'noe_not_seen': 'NOE not seen',
                                                      'noe not seen': 'NOE not seen',
                                                      'roe': 'ROE',
                                                      'roe_build_up': 'ROE build-up',
                                                      'roe build-up': 'ROE build-up',
                                                      'ROE buildup': 'ROE build-up',
                                                      'roe buildup': 'ROE build-up',
                                                      'ROE build up': 'ROE build-up',
                                                      'roe build up': 'ROE build-up',
                                                      'hbond': 'hydrogen bond',
                                                      'Hbond': 'hydrogen bond',
                                                      'HBond': 'hydrogen bond',
                                                      'H-bond': 'hydrogen bond',
                                                      'h-bond': 'hydrogen bond',
                                                      'H-Bond': 'hydrogen bond',
                                                      'Hydrogen bond': 'hydrogen bond',
                                                      'disulfide_bond': 'disulfide bond',
                                                      'Disulfide bond': 'disulfide bond',
                                                      'S-S bond': 'disulfide bond',
                                                      'SS bond': 'disulfide bond',
                                                      'SS-bond': 'disulfide bond',
                                                      'disulfide bridge': 'disulfide bond',
                                                      'Disulfide bridge': 'disulfide bond',
                                                      'PRE': 'paramagnetic relaxation',
                                                      'pre': 'paramagnetic relaxation',
                                                      'Paramagnetic relaxation': 'paramagnetic relaxation',
                                                      'paramagnetic relaxation enhancement': 'paramagnetic relaxation',
                                                      'Paramagnetic relaxation enhancement': 'paramagnetic relaxation',
                                                      'Mutation': 'mutation',
                                                      'unknown': 'general distance',
                                                      'undefined': 'general distance',
                                                      'shift_perturbation': 'chemical shift perturbation',
                                                      'shift perturbation': 'chemical shift perturbation',
                                                      'chem shift perturbation': 'chemical shift perturbation',
                                                      'CS perturbation': 'chemical shift perturbation',
                                                      'csp': 'chemical shift perturbation',
                                                      'CSP': 'chemical shift perturbation',
                                                      '?': 'undefined'
                                                      }
                                         }

        self.dihed_alt_constraint_type = {'nef': {'J-couplings': 'jcoupling',
                                                  'j-couplings': 'jcoupling',
                                                  'J couplings': 'jcoupling',
                                                  'j couplings': 'jcoupling',
                                                  'Jcouplings': 'jcoupling',
                                                  'jcouplings': 'jcoupling',
                                                  'J-coupling': 'jcoupling',
                                                  'j-coupling': 'jcoupling',
                                                  'J coupling': 'jcoupling',
                                                  'j coupling': 'jcoupling',
                                                  'Jcoupling': 'jcoupling',
                                                  'chemical shift': 'chemical_shift',
                                                  'Chemical shift': 'chemical_shift',
                                                  'Chemical_shift': 'chemical_shift',
                                                  'chemical shifts': 'chemical_shift',
                                                  'Chemical shifts': 'chemical_shift',
                                                  'Chemical_shifts': 'chemical_shift',
                                                  'backbone chemical shifts': 'chemical_shift',
                                                  'Backbone chemical shifts': 'chemical_shift',
                                                  'Mainchain chemical shifts': 'chemical_shift',
                                                  'mainchain chemical shifts': 'chemical_shift',
                                                  'Main chain chemical shifts': 'chemical_shift',
                                                  'main chain chemical shifts': 'chemical_shift',
                                                  'bb chemical shifts': 'chemical_shift',
                                                  'BB chemical shifts': 'chemical_shift',
                                                  'backbone chemical shift': 'chemical_shift',
                                                  'Backbone chemical shift': 'chemical_shift',
                                                  'Mainchain chemical shift': 'chemical_shift',
                                                  'mainchain chemical shift': 'chemical_shift',
                                                  'Main chain chemical shift': 'chemical_shift',
                                                  'main chain chemical shift': 'chemical_shift',
                                                  'bb chemical shift': 'chemical_shift',
                                                  'BB chemical shift': 'chemical_shift',
                                                  'backbone chem shifts': 'chemical_shift',
                                                  'Backbone chem shifts': 'chemical_shift',
                                                  'Mainchain chem shifts': 'chemical_shift',
                                                  'mainchain chem shifts': 'chemical_shift',
                                                  'Main chain chem shifts': 'chemical_shift',
                                                  'main chain chem shifts': 'chemical_shift',
                                                  'bb chem shifts': 'chemical_shift',
                                                  'BB chem shifts': 'chemical_shift',
                                                  'backbone chem shift': 'chemical_shift',
                                                  'Backbone chem shift': 'chemical_shift',
                                                  'Mainchain chem shift': 'chemical_shift',
                                                  'mainchain chem shift': 'chemical_shift',
                                                  'Main chain chem shift': 'chemical_shift',
                                                  'main chain chem shift': 'chemical_shift',
                                                  'bb chem shift': 'chemical_shift',
                                                  'BB chem shift': 'chemical_shift',
                                                  'backbone cs': 'chemical_shift',
                                                  'Backbone cs': 'chemical_shift',
                                                  'Mainchain cs': 'chemical_shift',
                                                  'mainchain cs': 'chemical_shift',
                                                  'Main chain cs': 'chemical_shift',
                                                  'main chain cs': 'chemical_shift',
                                                  'bb cs': 'chemical_shift',
                                                  'BB cs': 'chemical_shift',
                                                  'backbone CS': 'chemical_shift',
                                                  'Backbone CS': 'chemical_shift',
                                                  'Mainchain CS': 'chemical_shift',
                                                  'mainchain CS': 'chemical_shift',
                                                  'Main chain CS': 'chemical_shift',
                                                  'main chain CS': 'chemical_shift',
                                                  'bb CS': 'chemical_shift',
                                                  'BB CS': 'chemical_shift',
                                                  'TALOS': 'chemical_shift',
                                                  'talos': 'chemical_shift',
                                                  'TALOS+': 'chemical_shift',
                                                  'talos+': 'chemical_shift',
                                                  'TALOS-N': 'chemical_shift',
                                                  'talos-n': 'chemical_shift',
                                                  '?': 'undefined'
                                                  },
                                          'nmr-star': {'jcoupling': 'J-couplings',
                                                       'Jcoupling': 'J-couplings',
                                                       'jcouplings': 'J-couplings',
                                                       'Jcouplings': 'J-couplings',
                                                       'j-couplings': 'J-couplings',
                                                       'J couplings': 'J-couplings',
                                                       'j couplings': 'J-couplings',
                                                       'J-coupling': 'J-couplings',
                                                       'j-coupling': 'J-couplings',
                                                       'J coupling': 'J-couplings',
                                                       'j coupling': 'J-couplings',
                                                       'chemical_shift': 'backbone chemical shifts',
                                                       'Chemical_shift': 'backbone chemical shifts',
                                                       'chemical_shifts': 'backbone chemical shifts',
                                                       'Chemical_shifts': 'backbone chemical shifts',
                                                       'chemical shift': 'backbone chemical shifts',
                                                       'Chemical shift': 'backbone chemical shifts',
                                                       'chemical shifts': 'backbone chemical shifts',
                                                       'Chemical shifts': 'backbone chemical shifts',
                                                       'Backbone chemical shifts': 'backbone chemical shifts',
                                                       'Mainchain chemical shifts': 'backbone chemical shifts',
                                                       'mainchain chemical shifts': 'backbone chemical shifts',
                                                       'Main chain chemical shifts': 'backbone chemical shifts',
                                                       'main chain chemical shifts': 'backbone chemical shifts',
                                                       'bb chemical shifts': 'backbone chemical shifts',
                                                       'BB chemical shifts': 'backbone chemical shifts',
                                                       'backbone chemical shift': 'backbone chemical shifts',
                                                       'Backbone chemical shift': 'backbone chemical shifts',
                                                       'Mainchain chemical shift': 'backbone chemical shifts',
                                                       'mainchain chemical shift': 'backbone chemical shifts',
                                                       'Main chain chemical shift': 'backbone chemical shifts',
                                                       'main chain chemical shift': 'backbone chemical shifts',
                                                       'bb chemical shift': 'backbone chemical shifts',
                                                       'BB chemical shift': 'backbone chemical shifts',
                                                       'backbone chem shifts': 'backbone chemical shifts',
                                                       'Backbone chem shifts': 'backbone chemical shifts',
                                                       'Mainchain chem shifts': 'backbone chemical shifts',
                                                       'mainchain chem shifts': 'backbone chemical shifts',
                                                       'Main chain chem shifts': 'backbone chemical shifts',
                                                       'main chain chem shifts': 'backbone chemical shifts',
                                                       'bb chem shifts': 'backbone chemical shifts',
                                                       'BB chem shifts': 'backbone chemical shifts',
                                                       'backbone chem shift': 'backbone chemical shifts',
                                                       'Backbone chem shift': 'backbone chemical shifts',
                                                       'Mainchain chem shift': 'backbone chemical shifts',
                                                       'mainchain chem shift': 'backbone chemical shifts',
                                                       'Main chain chem shift': 'backbone chemical shifts',
                                                       'main chain chem shift': 'backbone chemical shifts',
                                                       'bb chem shift': 'backbone chemical shifts',
                                                       'BB chem shift': 'backbone chemical shifts',
                                                       'backbone cs': 'backbone chemical shifts',
                                                       'Backbone cs': 'backbone chemical shifts',
                                                       'Mainchain cs': 'backbone chemical shifts',
                                                       'mainchain cs': 'backbone chemical shifts',
                                                       'Main chain cs': 'backbone chemical shifts',
                                                       'main chain cs': 'backbone chemical shifts',
                                                       'bb cs': 'backbone chemical shifts',
                                                       'BB cs': 'backbone chemical shifts',
                                                       'backbone CS': 'backbone chemical shifts',
                                                       'Backbone CS': 'backbone chemical shifts',
                                                       'Mainchain CS': 'backbone chemical shifts',
                                                       'mainchain CS': 'backbone chemical shifts',
                                                       'Main chain CS': 'backbone chemical shifts',
                                                       'main chain CS': 'backbone chemical shifts',
                                                       'bb CS': 'backbone chemical shifts',
                                                       'BB CS': 'backbone chemical shifts',
                                                       'TALOS': 'backbone chemical shifts',
                                                       'talos': 'backbone chemical shifts',
                                                       'TALOS+': 'backbone chemical shifts',
                                                       'talos+': 'backbone chemical shifts',
                                                       'TALOS-N': 'backbone chemical shifts',
                                                       'talos-n': 'backbone chemical shifts',
                                                       '?': 'undefined'
                                                       }
                                          }

        self.rdc_alt_constraint_type = {'nef': {'RDC': 'measured',
                                                'rdc': 'measured',
                                                '?': 'undefined'
                                                },
                                        'nmr-star': {'rdc': 'RDC',
                                                     'measured': 'RDC',
                                                     '?': 'undefined'
                                                     }
                                        }

        # saveframe tag items
        self.sf_tag_items = {'nef': {'entry_info': [{'name': 'sf_category', 'type': 'str', 'mandatory': True},
                                                    {'name': 'sf_framecode', 'type': 'str', 'mandatory': True},
                                                    {'name': 'format_name', 'type': 'str', 'mandatory': True},
                                                    {'name': 'format_version', 'type': 'str', 'mandatory': True},
                                                    {'name': 'program_name', 'type': 'str', 'mandatory': True},
                                                    {'name': 'program_version', 'type': 'str', 'mandatory': True},
                                                    {'name': 'creation_date', 'type': 'str', 'mandatory': True},
                                                    {'name': 'uuid', 'type': 'str', 'mandatory': True},
                                                    {'name': 'coordinate_file_name', 'type': 'str', 'mandatory': False}
                                                    ],
                                     'entity': None,
                                     'poly_seq': [{'name': 'sf_category', 'type': 'str', 'mandatory': True},
                                                  {'name': 'sf_framecode', 'type': 'str', 'mandatory': True}
                                                  ],
                                     'chem_shift': [{'name': 'sf_category', 'type': 'str', 'mandatory': True},
                                                    {'name': 'sf_framecode', 'type': 'str', 'mandatory': True}
                                                    ],
                                     'chem_shift_ref': None,
                                     'dist_restraint': [{'name': 'sf_category', 'type': 'str', 'mandatory': True},
                                                        {'name': 'sf_framecode', 'type': 'str', 'mandatory': True},
                                                        {'name': 'potential_type', 'type': 'enum', 'mandatory': False,
                                                         'enum': ('log-harmonic', 'parabolic', 'square-well-parabolic',
                                                                  'square-well-parabolic-linear', 'upper-bound-parabolic',
                                                                  'lower-bound-parabolic', 'upper-bound-parabolic-linear',
                                                                  'lower-bound-parabolic-linear', 'undefined', 'unknown'),
                                                         'enum-alt': self.alt_potential_type},
                                                        {'name': 'restraint_origin', 'type': 'enum', 'mandatory': False,
                                                         'enum': ('noe', 'noe_build_up', 'noe_not_seen', 'roe',
                                                                  'roe_build_up', 'hbond', 'disulfide_bond', 'pre',
                                                                  'symmetry', 'mutation', 'shift_perturbation',
                                                                  'undefined', 'unknown'),
                                                         'enum-alt': self.dist_alt_constraint_type['nef']}
                                                        ],
                                     'dihed_restraint': [{'name': 'sf_category', 'type': 'str', 'mandatory': True},
                                                         {'name': 'sf_framecode', 'type': 'str', 'mandatory': True},
                                                         {'name': 'potential_type', 'type': 'enum', 'mandatory': False,
                                                          'enum': ('parabolic', 'square-well-parabolic', 'square-well-parabolic-linear',
                                                                   'upper-bound-parabolic', 'lower-bound-parabolic', 'upper-bound-parabolic-linear',
                                                                   'lower-bound-parabolic-linear', 'undefined', 'unknown'),
                                                          'enum-alt': self.alt_potential_type},
                                                         {'name': 'restraint_origin', 'type': 'enum', 'mandatory': False,
                                                          'enum': ('jcoupling', 'chemical_shift', 'undefined', 'unknown'),
                                                          'enum-alt': self.dihed_alt_constraint_type['nef']}
                                                         ],
                                     'rdc_restraint': [{'name': 'sf_category', 'type': 'str', 'mandatory': True},
                                                       {'name': 'sf_framecode', 'type': 'str', 'mandatory': True},
                                                       {'name': 'potential_type', 'type': 'enum', 'mandatory': False,
                                                        'enum': ('parabolic', 'square-well-parabolic', 'square-well-parabolic-linear',
                                                                 'upper-bound-parabolic', 'lower-bound-parabolic', 'upper-bound-parabolic-linear',
                                                                 'lower-bound-parabolic-linear', 'undefined', 'unknown'),
                                                        'enum-alt': self.alt_potential_type},
                                                       {'name': 'restraint_origin', 'type': 'enum', 'mandatory': False,
                                                        'enum': ('measured', 'undefined', 'unknown'),
                                                        'enum-alt': self.rdc_alt_constraint_type['nef']},
                                                       {'name': 'tensor_magnitude', 'type': 'float', 'mandatory': False},
                                                       {'name': 'tensor_rhombicity', 'type': 'positive-float', 'mandatory': False},
                                                       {'name': 'tensor_chain_code', 'type': 'str', 'mandatory': False},
                                                       {'name': 'tensor_sequence_code', 'type': 'str', 'mandatory': False},
                                                       {'name': 'tensor_residue_name', 'type': 'str', 'mandatory': False}
                                                       ],
                                     'spectral_peak': [{'name': 'sf_category', 'type': 'str', 'mandatory': True},
                                                       {'name': 'sf_framecode', 'type': 'str', 'mandatory': True},
                                                       {'name': 'num_dimensions', 'type': 'enum-int', 'mandatory': True,
                                                        'enum': set(range(1, self.lim_num_dim)),
                                                        'enforce-enum': True},
                                                       {'name': 'chemical_shift_list', 'type': 'str', 'mandatory': False},
                                                       {'name': 'experiment_classification', 'type': 'str', 'mandatory': False},
                                                       {'name': 'experiment_type', 'type': 'str', 'mandatory': False}
                                                       ],
                                     'spectral_peak_alt': None
                                     },
                             'nmr-star': {'entry_info': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                         {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True},
                                                         {'name': 'NMR_STAR_version', 'type': 'str', 'mandatory': True},
                                                         {'name': 'Source_data_format', 'type': 'str', 'mandatory': False},
                                                         {'name': 'Source_data_format_version', 'type': 'str', 'mandatory': False},
                                                         {'name': 'Generated_software_name', 'type': 'str', 'mandatory': False},
                                                         {'name': 'Generated_software_version', 'type': 'str', 'mandatory': False},
                                                         {'name': 'Generated_date', 'type': 'str', 'mandatory': False},
                                                         {'name': 'UUID', 'type': 'str', 'mandatory': False},
                                                         {'name': 'Related_coordinate_file_name', 'type': 'str', 'mandatory': False}
                                                         ],
                                          'entity': None,
                                          'poly_seq': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                       {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True}
                                                       ],
                                          'chem_shift': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                         {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True}
                                                         ],
                                          'chem_shift_ref': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                             {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True},
                                                             {'name': 'ID', 'type': 'positive-int', 'mandatory': True},
                                                             {'name': 'Details', 'type': 'str', 'mandatory': False},
                                                             {'name': 'Proton_shifts_flag', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('no', 'yes', 'yes with IUPAC referencing')},
                                                             {'name': 'Carbon_shifts_flag', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('no', 'yes', 'yes with IUPAC referencing')},
                                                             {'name': 'Nitrogen_shifts_flag', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('no', 'yes', 'yes with IUPAC referencing')},
                                                             {'name': 'Phosphorus_shifts_flag', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('no', 'yes', 'yes with IUPAC referencing')},
                                                             {'name': 'Other_shifts_flag', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('no', 'yes', 'yes with IUPAC referencing')},
                                                             ],
                                          'dist_restraint': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                             {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True},
                                                             {'name': 'Constraint_type', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('NOE', 'NOE build-up', 'NOE not seen', 'ROE', 'ROE build-up',
                                                                       'hydrogen bond', 'disulfide bond', 'paramagnetic relaxation',
                                                                       'symmetry', 'general distance', 'mutation', 'chemical shift perturbation',
                                                                       'undefined', 'unknown'),
                                                              'enum-alt': self.dist_alt_constraint_type['nmr-star']},
                                                             {'name': 'Potential_type', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('log-harmonic', 'parabolic', 'square-well-parabolic',
                                                                       'square-well-parabolic-linear', 'upper-bound-parabolic',
                                                                       'lower-bound-parabolic', 'upper-bound-parabolic-linear',
                                                                       'lower-bound-parabolic-linear', 'undefined', 'unknown'),
                                                              'enum-alt': self.alt_potential_type}
                                                             ],
                                          'dihed_restraint': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                              {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True},
                                                              {'name': 'Constraint_type', 'type': 'enum', 'mandatory': False,
                                                               'enum': ('J-couplings', 'backbone chemical shifts', 'undefined', 'unknown'),
                                                               'enum-alt': self.dihed_alt_constraint_type['nmr-star']},
                                                              {'name': 'Potential_type', 'type': 'enum', 'mandatory': False,
                                                               'enum': ('parabolic', 'square-well-parabolic', 'square-well-parabolic-linear',
                                                                        'upper-bound-parabolic', 'lower-bound-parabolic', 'upper-bound-parabolic-linear',
                                                                        'lower-bound-parabolic-linear', 'undefined', 'unknown'),
                                                               'enum-alt': self.alt_potential_type}
                                                              ],
                                          'rdc_restraint': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                            {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True},
                                                            {'name': 'Constraint_type', 'type': 'enum', 'mandatory': False,
                                                             'enum': ('RDC', 'undefined', 'unknown'),
                                                             'enum-alt': self.rdc_alt_constraint_type['nmr-star']},
                                                            {'name': 'Potential_type', 'type': 'enum', 'mandatory': False,
                                                             'enum': ('parabolic', 'square-well-parabolic', 'square-well-parabolic-linear',
                                                                      'upper-bound-parabolic', 'lower-bound-parabolic', 'upper-bound-parabolic-linear',
                                                                      'lower-bound-parabolic-linear', 'undefined', 'unknown'),
                                                             'enum-alt': self.alt_potential_type},
                                                            {'name': 'Tensor_magnitude', 'type': 'float', 'mandatory': False},
                                                            {'name': 'Tensor_rhombicity', 'type': 'positive-float', 'mandatory': False},
                                                            {'name': 'Tensor_auth_asym_ID', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Tensor_auth_seq_ID', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Tensor_auth_comp_ID', 'type': 'str', 'mandatory': False}
                                                            ],
                                          'spectral_peak': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                            {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True},
                                                            {'name': 'Experiment_class', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Experiment_type', 'type': 'str', 'mandatory': False},
                                                            {'name': 'Number_of_spectral_dimensions', 'type': 'enum-int', 'mandatory': True,
                                                             'enum': set(range(1, self.lim_num_dim)),
                                                             'enforce-enum': True},
                                                            {'name': 'Chemical_shift_list', 'type': 'str', 'mandatory': True}
                                                            ],
                                          'spectral_peak_alt': [{'name': 'Sf_category', 'type': 'str', 'mandatory': True},
                                                                {'name': 'Sf_framecode', 'type': 'str', 'mandatory': True},
                                                                {'name': 'Experiment_class', 'type': 'str', 'mandatory': False},
                                                                {'name': 'Experiment_type', 'type': 'str', 'mandatory': False},
                                                                {'name': 'Number_of_spectral_dimensions', 'type': 'enum-int', 'mandatory': True,
                                                                 'enum': set(range(1, self.lim_num_dim)),
                                                                 'enforce-enum': True}
                                                                ]
                                          }
                             }

        # required saveframe tag items by NmrDpUtility
        self._sf_tag_items = {'nef': {'entry_info': None,
                                      'poly_seq': None,
                                      'entity': None,
                                      'chem_shift': None,
                                      'chem_shift_ref': None,
                                      'dist_restraint': ['restraint_origin', 'potential_type'],
                                      'dihed_restraint': ['restraint_origin', 'potential_type'],
                                      'rdc_restraint': ['restraint_origin', 'potential_type'],
                                      'spectral_peak': ['experiment_type'],
                                      'spectral_peak_alt': None
                                      },
                              'nmr-star': {'entry_info': None,
                                           'poly_seq': None,
                                           'entity': None,
                                           'chem_shift': None,
                                           'chem_shift_ref': None,
                                           'dist_restraint': ['Constraint_type', 'Potential_type'],
                                           'dihed_restraint': ['Constraint_type', 'Potential_type'],
                                           'rdc_restraint': ['Constraint_type', 'Potential_type'],
                                           'spectral_peak': ['Experiment_type'],
                                           'spectral_peak_alt': ['Experiment_type']
                                           }
                              }

        # allowed saveframe tags
        self.sf_allowed_tags = {'nef': {'entry_info': ['sf_category', 'sf_framecode', 'format_name', 'format_version',
                                                       'program_name', 'program_version', 'creation_date', 'uuid', 'coordinate_file_name'],
                                        'poly_seq': ['sf_category', 'sf_framecode'],
                                        'entity': None,
                                        'chem_shift': ['sf_category', 'sf_framecode'],
                                        'chem_shift_ref': None,
                                        'dist_restraint': ['sf_category', 'sf_framecode', 'potential_type', 'restraint_origin'],
                                        'dihed_restraint': ['sf_category', 'sf_framecode', 'potential_type', 'restraint_origin'],
                                        'rdc_restraint': ['sf_category', 'sf_framecode', 'potential_type', 'restraint_origin',
                                                          'tensor_magnitude', 'tensor_rhombicity', 'tensor_chain_code',
                                                          'tensor_sequence_code', 'tensor_residue_name'],
                                        'spectral_peak': ['sf_category', 'sf_framecode', 'num_dimensions', 'chemical_shift_list',
                                                          'experiment_classification', 'experiment_type'],
                                        'spectral_peak_alt': None
                                        },
                                'nmr-star': {'entry_info': ['Sf_category', 'Sf_framecode', 'Sf_ID', 'ID', 'Title', 'Type',
                                                            'Version_type', 'Submission_date', 'Accession_date', 'Last_release_date', 'Original_release_date',
                                                            'Origination', 'Format_name', 'NMR_STAR_version', 'Original_NMR_STAR_version',
                                                            'Experimental_method', 'Experimental_method_subtype',
                                                            'Source_data_format', 'Source_data_format_version',
                                                            'Generated_software_name', 'Generated_software_version', 'Generated_software_ID',
                                                            'Generated_software_label', 'Generated_date',
                                                            'DOI', 'UUID', 'Related_coordinate_file_name', 'Dep_release_code_coordinates',
                                                            'Dep_release_code_nmr_constraints', 'Dep_release_code_chemical_shifts',
                                                            'Dep_release_code_nmr_exptl', 'Dep_release_code_sequence',
                                                            'CASP_target', 'Details', 'Special_processing_instructions',
                                                            'Update_BMRB_accession_code', 'Replace_BMRB_accession_code',
                                                            'Update_PDB_accession_code', 'Replace_PDB_accession_code',
                                                            'PDB_coordinate_file_version', 'BMRB_update_details',
                                                            'PDB_update_details', 'Release_request',
                                                            'Release_date_request', 'Release_date_justification',
                                                            'Status_code', 'Recvd_deposit_form', 'Date_deposition_form',
                                                            'Recvd_coordinates', 'Date_coordinates', 'Recvd_nmr_constraints',
                                                            'Date_nmr_constraints', 'Recvd_chemical_shifts', 'Date_chemical_shifts',
                                                            'Recvd_manuscript', 'Date_manuscript', 'Recvd_author_approval', 'Date_author_approval',
                                                            'Recvd_initial_deposition_date', 'PDB_date_submitted', 'Author_release_status_code',
                                                            'Date_of_PDB_release', 'Date_hold_coordinates', 'Date_hold_nmr_constraints', 'Date_hold_chemical_shifts',
                                                            'PDB_deposit_site', 'PDB_process_site', 'BMRB_deposit_site', 'BMRB_process_site',
                                                            'BMRB_annotator', 'BMRB_internal_directory_name', 'RCSB_annotator', 'Author_approval_type',
                                                            'Assigned_BMRB_ID', 'Assigned_BMRB_deposition_code', 'Assigned_PDB_ID',
                                                            'Assigned_PDB_deposition_code', 'Assigned_restart_ID'],
                                             'poly_seq': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID', 'Name', 'BMRB_code',
                                                          'Number_of_components', 'Organic_ligands', 'Metal_ions', 'Non_standard_bonds',
                                                          'Ambiguous_conformational_states', 'Ambiguous_chem_comp_sites', 'Molecules_in_chemical_exchange',
                                                          'Paramagnetic', 'Thiol_state', 'Molecular_mass', 'Enzyme_commission_number',
                                                          'Details', 'DB_query_date', 'DB_query_revised_last_date'],
                                             'entity': None,
                                             'chem_shift': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID', 'Name', 'Data_file_name',
                                                            'Sample_condition_list_ID', 'Sample_condition_list_label', 'Chem_shift_reference_ID',
                                                            'Chem_shift_reference_label',
                                                            'Chem_shift_1H_err', 'Chem_shift_13C_err', 'Chem_shift_15N_err', 'Chem_shift_31P_err',
                                                            'Chem_shift_2H_err', 'Chem_shift_19F_err',
                                                            'Error_derivation_method', 'Details', 'Text_data_format', 'Text_data'],
                                             'chem_shift_ref': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID', 'Name', 'Proton_shifts_flag',
                                                                'Carbon_shifts_flag', 'Nitrogen_shifts_flag', 'Phosphorus_shifts_flag', 'Other_shifts_flag', 'Details'],
                                             'dist_restraint': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID', 'Name', 'Data_file_name',
                                                                'Constraint_type', 'Constraint_file_ID', 'Potential_type', 'Block_ID', 'Details', 'Text_data_format', 'Text_data'],
                                             'dihed_restraint': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID',
                                                                 'Name', 'Data_file_name', 'Constraint_file_ID', 'Potential_type', 'Constraint_type',
                                                                 'Block_ID', 'Details', 'Text_data_format', 'Text_data'],
                                             'rdc_restraint': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID',
                                                               'Name', 'Data_file_name', 'Constraint_file_ID', 'Block_ID', 'Potential_type', 'Constraint_type',
                                                               'Tensor_entity_assembly_ID', 'Tensor_comp_index_ID', 'Tensor_seq_ID', 'Tensor_comp_ID',
                                                               'Tensor_auth_entity_assembly_ID', 'Tensor_auth_asym_ID', 'Tensor_auth_seq_ID', 'Tensor_auth_comp_ID',
                                                               'Dipolar_constraint_calib_method', 'Tensor_magnitude', 'Tensor_rhombicity',
                                                               'Mol_align_tensor_axial_sym_mol', 'Mol_align_tensor_rhombic_mol', 'General_order_param_int_motions',
                                                               'Bond_length_usage_flag', 'Assumed_H_N_bond_length', 'Assumed_H_C_bond_length',
                                                               'Assumed_C_N_bond_length', 'Data_file_format', 'Details', 'Text_data_format', 'Text_data'],
                                             'spectral_peak': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID', 'Name', 'Data_file_name',
                                                               'Sample_ID', 'Sample_label', 'Sample_condition_list_ID', 'Sample_condition_list_label',
                                                               'Experiment_ID', 'Experiment_name', 'Experiment_class', 'Experiment_type',
                                                               'Number_of_spectral_dimensions', 'Chemical_shift_list', 'Assigned_chem_shift_list_ID',
                                                               'Assigned_chem_shift_list_label', 'Details', 'Text_data_format', 'Text_data'],
                                             'spectral_peak_alt': ['Sf_category', 'Sf_framecode', 'Entry_ID', 'Sf_ID', 'ID', 'Name', 'Data_file_name',
                                                                   'Sample_ID', 'Sample_label', 'Sample_condition_list_ID', 'Sample_condition_list_label',
                                                                   'Experiment_ID', 'Experiment_name', 'Experiment_class', 'Experiment_type',
                                                                   'Number_of_spectral_dimensions', 'Chemical_shift_list', 'Assigned_chem_shift_list_ID',
                                                                   'Assigned_chem_shift_list_label', 'Details', 'Text_data_format', 'Text_data']
                                             }
                                }

        # warning template for missing mandatory saveframe tag
        self.__warn_template_for_missing_mandatory_sf_tag = "The mandatory saveframe tag %r is missing. Please verify the value and re-upload the %s file."

        # auxiliary loop categories
        self.aux_lp_categories = {'nef': {'entry_info': [],
                                          'poly_seq': ['_nef_covalent_links', '_nef_sequence'],
                                          'entity': [],
                                          'chem_shift': [],
                                          'chem_shift_ref': [],
                                          'dist_restraint': [],
                                          'dihed_restraint': [],
                                          'rdc_restraint': [],
                                          'spectral_peak': ['_nef_spectrum_dimension', '_nef_spectrum_dimension_transfer'],
                                          'spectral_peak_alt': []
                                          },
                                  'nmr-star': {'entry_info': [],
                                               'poly_seq': ['_Bond', '_Entity_deleted_atom'],
                                               'entity': ['_Entity_poly_seq'],
                                               'chem_shift': [],
                                               'chem_shift_ref': [],
                                               'dist_restraint': [],
                                               'dihed_restraint': [],
                                               'rdc_restraint': [],
                                               'spectral_peak': ['_Spectral_dim', '_Spectral_dim_transfer'],
                                               'spectral_peak_alt': ['_Spectral_dim', '_Spectral_dim_transfer', '_Peak_general_char', '_Peak_char', '_Assigned_peak_chem_shift']
                                               }
                                  }

        # linked loop categories
        self.linked_lp_categories = {'nef': {'entry_info': ['_nef_related_entries', '_nef_program_script', '_nef_run_history', '_audit'],
                                             'poly_seq': ['_nef_sequence', '_nef_covalent_links'],
                                             'entity': [],
                                             'chem_shift': ['_nef_chemical_shift'],
                                             'chem_shift_ref': [],
                                             'dist_restraint': ['_nef_distance_restraint'],
                                             'dihed_restraint': ['_nef_dihedral_restraint'],
                                             'rdc_restraint': ['_nef_rdc_restraint'],
                                             'spectral_peak': ['_nef_spectrum_dimension', '_nef_spectrum_dimension_transfer', '_nef_peak'],
                                             'spectral_peak_alt': []
                                             },
                                     'nmr-star': {'entry_info': ['_Study_list', '_Entry_experimental_methods', '_Entry_author',
                                                                 '_SG_project', '_Entry_src', '_Struct_keywords', '_Data_set',
                                                                 '_Datum', '_Release', '_Related_entries', '_Matched_entries',
                                                                 '_Auxiliary_files', '_Citation',
                                                                 '_Assembly', '_Assembly_annotation_list', '_Assembly_subsystem',
                                                                 '_Entity', '_Entity_natural_src_list', '_Entity_natural_src',
                                                                 '_Entity_experimental_src_list', '_Chem_comp', '_Chem_comp_atom',
                                                                 '_Sample', '_Sample_condition_list', '_Entity_purity_list', '_Software', '_Method',
                                                                 '_Mass_spec', '_Mass_spectrometer_list', '_Mass_spec_ref_compd_set',
                                                                 '_Chromatographic_system', '_Chromatographic_column',
                                                                 '_Fluorescence_instrument', '_EMR_instrument', '_Xray_instrument',
                                                                 '_NMR_spectrometer', '_NMR_spectrometer_list', '_NMR_spectrometer_probe',
                                                                 '_Experiment_list', '_NMR_spec_expt', '_NMR_spectral_processing',
                                                                 '_MS_expt', '_MS_expt_param', '_MS_expt_software',
                                                                 '_Computer', '_Chem_shift_reference', '_Assigned_chem_shift_list',
                                                                 '_Chem_shifts_calc_type', '_Theoretical_chem_shift_list', '_Theoretical_chem_shift',
                                                                 '_Coupling_constant_list', '_Theoretical_coupling_constant_list', '_Spectral_peak_list',
                                                                 '_Resonance_linker_list', '_Resonance_assignment',
                                                                 '_Chem_shift_isotope_effect_list', '_Chem_shift_perturbation_list', '_Chem_shift_anisotropy',
                                                                 '_RDC_list', '_RDC_experiment', '_RDC_software', '_RDC',
                                                                 '_Dipolar_coupling_list', '_Dipolar_coupling_experiment', '_Dipolar_coupling_software',
                                                                 '_Dipolar_coupling', '_Spectral_density_list', '_Spectral_density_experiment',
                                                                 '_Spectral_density_software', '_Spectral_density', '_Other_data_type_list',
                                                                 '_Other_data_experiment', '_Other_data_software', '_Other_data',
                                                                 '_Chemical_rate_list', '_Chemical_rate_experiment', '_Chemical_rate_software', '_Chemical_rate',
                                                                 '_H_exch_rate_list', '_H_exch_rate_experiment', '_H_exch_rate_software', '_H_exch_rate',
                                                                 '_H_exch_protection_factor_list', '_H_exch_protection_fact_experiment',
                                                                 '_H_exch_protection_fact_software', '_H_exch_protection_factor',
                                                                 '_Homonucl_NOE_list', '_Homonucl_NOE_experiment', '_Homonucl_NOE_software',
                                                                 '_Homonucl_NOE', '_Heteronucl_NOE_list', '_Heteronucl_NOE_experiment',
                                                                 '_Heteronucl_NOE_software', '_Heteronucl_NOE', '_Theoretical_heteronucl_NOE_list',
                                                                 '_Theoretical_heteronucl_NOE_experiment', '_Theoretical_heteronucl_NOE_software',
                                                                 '_Theoretical_heteronucl_NOE',
                                                                 '_Heteronucl_T1_list', '_Heteronucl_T1_experiment', '_Heteronucl_T1_software', '_T1',
                                                                 '_Theoretical_heteronucl_T1_list', '_Theoretical_heteronucl_T1_experiment',
                                                                 '_Theoretical_heteronucl_T1_software', '_Theoretical_T1',
                                                                 '_Heteronucl_T1rho_list', '_Heteronucl_T1rho_experiment', '_Heteronucl_T1rho_software',
                                                                 '_T1rho',
                                                                 '_Heteronucl_T2_list', '_Heteronucl_T2_experiment', '_Heteronucl_T2_software', '_T2',
                                                                 '_Theoretical_heteronucl_T2_list', '_Theoretical_heteronucl_T2_experiment',
                                                                 '_Theoretical_heteronucl_T2_software', '_Theoretical_T2',
                                                                 '_Auto_relaxation_list', '_Auto_relaxation_experiment', '_Auto_relaxation_software',
                                                                 '_Auto_relaxation', '_Theoretical_auto_relaxation_list', '_Theoretical_auto_relaxation_experiment',
                                                                 '_Theoretical_auto_relaxation_software', '_Theoretical_auto_relaxation',
                                                                 '_Dipole_dipole_relax_list', '_Dipole_dipole_relax_experiment', '_Dipole_dipole_relax_software',
                                                                 '_Dipole_dipole_relax',
                                                                 '_Cross_correlation_DD_list', '_Cross_correlation_DD_experiment', '_Cross_correlation_DD_software',
                                                                 '_Cross_correlation_DD', '_Theoretical_cross_correlation_DD_list', '_Theoretical_cross_correlation_DD_experiment',
                                                                 '_Theoretical_cross_correlation_DD_software', '_Theoretical_cross_correlation_DD',
                                                                 '_Cross_correlation_D_CSA_list', '_Cross_correlation_D_CSA_experiment', '_Cross_correlation_D_CSA_software',
                                                                 '_Cross_correlation_D_CSA', '_Order_parameter_list', '_Order_parameter_experiment',
                                                                 '_Order_parameter_software', '_Order_param',
                                                                 '_PH_titration_list', '_PH_titration_experiment', '_PH_titration_software', '_PH_titr_result',
                                                                 '_PH_param_list', '_PH_param', '_D_H_fractionation_factor_list', '_D_H_fract_factor_experiment',
                                                                 '_D_H_fract_factor_software', '_D_H_fractionation_factor',
                                                                 '_Binding_value_list', '_Binding_experiment', '_Binding_software', '_Binding_result',
                                                                 '_Binding_partners', '_Binding_param_list', '_Binding_param',
                                                                 '_Deduced_secd_struct_list', '_Deduced_secd_struct_experiment', '_Deduced_secd_struct_software',
                                                                 '_Deduced_secd_struct_exptl', '_Deduced_secd_struct_feature', '_Deduced_H_bond_list',
                                                                 '_Deduced_H_bond_experiment', '_Deduced_H_bond_software', '_Deduced_H_bond',
                                                                 '_Conformer_stat_list', '_Conformer_stat_list_ens', '_Conformer_stat_list_rep', '_Conf_stats_software',
                                                                 '_Conformer_family_coord_set', '_Conformer_family_refinement', '_Conformer_family_software',
                                                                 '_Energetic_penalty_function', '_Conformer_family_coord_set_expt', '_Conf_family_coord_set_constr_list',
                                                                 '_Struct_image', '_Local_structure_quality', '_Model_type', '_Atom_site', '_Atom_sites_footnote',
                                                                 '_Representative_conformer', '_Rep_conf_refinement', '_Rep_conf_software', '_Terminal_residue',
                                                                 '_Rep_conf', '_Rep_coordinate_details',
                                                                 '_Constraint_stat_list', '_Constraint_stat_list_ens', '_Constraint_stat_list_rep',
                                                                 '_Constraint_stats_constr_list', '_Constraint_file', '_Force_constant_list', '_Force_constant_software',
                                                                 '_Force_constant', '_Angular_order_parameter_list', '_Angular_order_param',
                                                                 '_Tertiary_struct_element_list', '_Tertiary_struct_element_sel', '_Tertiary_struct',
                                                                 '_Structure_annotation', '_Struct_anno_software', '_Struct_classification', '_Struct_anno_char',
                                                                 '_Secondary_struct_list', '_Secondary_struct_sel', '_Secondary_struct', '_Bond_annotation_list'
                                                                 , '_Bond_annotation', '_Bond_observed_conformer',
                                                                 '_Structure_interaction_list', '_Structure_interaction', '_Observed_conformer',
                                                                 '_Other_struct_feature_list', '_Other_struct_feature', '_Tensor_list',
                                                                 '_Interatomic_distance_list', '_Interatomic_dist',
                                                                 '_Gen_dist_constraint_list', '_Gen_dist_constraint_expt', '_Gen_dist_constraint_software',
                                                                 '_Gen_dist_constraint_software_param', '_Gen_dist_constraint', '_Gen_dist_constraint_comment_org',
                                                                 '_Gen_dist_constraint_parse_err', '_Gen_dist_constraint_parse_file', '_Gen_dist_constraint_conv_err',
                                                                 '_Distance_constraint_list', '_Distance_constraint_expt', '_Distance_constraint_software',
                                                                 '_Dist_constr_software_setting', '_Dist_constraint_tree', '_Dist_constraint',
                                                                 '_Dist_constraint_value', '_Dist_constraint_comment_org', '_Dist_constraint_parse_err',
                                                                 '_Dist_constraint_parse_file', '_Dist_constraint_conv_err',
                                                                 '_Floating_chirality_assign', '_Floating_chirality_software', '_Floating_chirality',
                                                                 '_Torsion_angle_constraint_list', '_Torsion_angle_constraints_expt', '_Torsion_angle_constraint_software',
                                                                 '_Karplus_equation', '_Torsion_angle_constraint', '_TA_constraint_comment_org', '_TA_constraint_parse_err',
                                                                 '_TA_constraint_parse_file', '_TA_constraint_conv_err',
                                                                 '_RDC_constraint_list', '_RDC_constraint_expt', '_RDC_constraint_software', '_RDC_constraint',
                                                                 '_RDC_constraint_comment_org', '_RDC_constraint_parse_err', '_RDC_constraint_parse_file',
                                                                 '_RDC_constraint_conv_err',
                                                                 '_J_three_bond_constraint_list', '_J_three_bond_constraint_expt', '_J_three_bond_constraint_software',
                                                                 '_J_three_bond_constraint', '_CA_CB_constraint_list', '_CA_CB_constraint_expt',
                                                                 '_CA_CB_constraint_software', '_CA_CB_constraint',
                                                                 '_H_chem_shift_constraint_list', '_H_chem_shift_constraint_expt', '_H_chem_shift_constraint_software',
                                                                 '_H_chem_shift_constraint', '_Peak_constraint_link_list', '_Peak_constraint_link',
                                                                 '_SAXS_constraint_list', '_SAXS_constraint_expt', '_SAXS_constraint_software', '_SAXS_constraint',
                                                                 '_Other_constraint_list', '_Other_constraint_expt', '_Other_constraint_software', '_Org_constr_file_comment',
                                                                 '_MZ_ratio_data_list', '_MZ_ratio_experiment', '_MZ_ratio_software', '_MZ_ratio_spectrum_param',
                                                                 '_MZ_precursor_ion', '_MZ_precursor_ion_annotation', '_MZ_product_ion', '_MZ_product_ion_annotation',
                                                                 '_MS_chromatogram_list', '_MS_chromatogram_experiment', '_MS_chromatogram_software', '_MS_chromatogram_param',
                                                                 '_MS_chromatogram_ion', '_MS_chrom_ion_annotation',
                                                                 '_Software_specific_info_list', '_Software_specific_info', '_Software_applied_list', '_Software_applied_methods',
                                                                 '_Software_applied_history', '_History',
                                                                 '_Audit'],
                                                  'poly_seq': ['_Assembly_type', '_Entity_assembly', '_Bond', '_Entity_deleted_atom',
                                                               '_Struct_asym', '_Assembly_db_link', '_Assembly_common_name',
                                                               '_Assembly_systematic_name', '_Assembly_interaction', '_Chem_comp_assembly',
                                                               '_PDBX_poly_seq_scheme', '_PDBX_nonpoly_scheme', '_Atom_type', '_Atom', '_Assembly_bio_function', '_Angle', '_Torsion_angle',
                                                               '_Assembly_segment', '_Assembly_segment_description', '_Assembly_keyword',
                                                               '_Assembly_citation', '_Author_annotation', '_Sample_component',
                                                               '_Chemical_rate', '_Auto_relaxation', '_Theoretical_auto_relaxation',
                                                               '_Binding_result', '_Binding_partners', '_Struct_anno_char'],
                                                  'entity': [],
                                                  'chem_shift': ['_Chem_shift_experiment', '_Systematic_chem_shift_offset',
                                                                 '_Chem_shift_software', '_Atom_chem_shift', '_Ambiguous_atom_chem_shift',
                                                                 '_Spectral_peak_list', '_Assigned_peak_chem_shift', '_Assigned_spectral_transition'],
                                                  'chem_shift_ref': ['_Chem_shift_ref', '_Assigned_chem_shift_list', '_Chem_shifts_calc_type'],
                                                  'dist_restraint': ['_Gen_dist_constraint_expt', '_Gen_dist_constraint_software',
                                                                     '_Gen_dist_constraint_software_param', '_Gen_dist_constraint',
                                                                     '_Gen_dist_constraint_comment_org', '_Gen_dist_constraint_parse_err',
                                                                     '_Gen_dist_constraint_parse_file', '_Gen_dist_constraint_conv_err'],
                                                  'dihed_restraint': ['_Torsion_angle_constraints_expt', '_Torsion_angle_constraint_software',
                                                                      '_Karplus_equation', '_Torsion_angle_constraint', '_TA_constraint_comment_org',
                                                                      '_TA_constraint_parse_err', '_TA_constraint_parse_file', '_TA_constraint_conv_err'],
                                                  'rdc_restraint': ['_RDC_constraint_expt', '_RDC_constraint_software', '_RDC_constraint',
                                                                    '_RDC_constraint_comment_org', '_RDC_constraint_parse_err',
                                                                    '_RDC_constraint_parse_file', '_RDC_constraint_conv_err'],
                                                  'spectral_peak': ['_Spectral_dim', '_Spectral_dim_transfer', '_Spectral_peak_software',
                                                                    '_Peak', '_Peak_general_char', '_Peak_char', '_Assigned_peak_chem_shift',
                                                                    '_Peak_row_format', '_Spectral_transition', '_Spectral_transition_general_char',
                                                                    '_Spectral_transition_char', '_Assigned_spectral_transition', '_Gen_dist_constraint',
                                                                    '_Dist_constraint_value'],
                                                  'spectral_peak_alt': ['_Spectral_dim', '_Spectral_dim_transfer', '_Spectral_peak_software',
                                                                        '_Peak', '_Peak_general_char', '_Peak_char', '_Assigned_peak_chem_shift',
                                                                        '_Peak_row_format', '_Spectral_transition', '_Spectral_transition_general_char',
                                                                        '_Spectral_transition_char', '_Assigned_spectral_transition',
                                                                        '_Gen_dist_constraint', '_Dist_constraint_value']
                                                  }
                                     }

        # auxiliary loop key items
        self.aux_key_items = {'nef': {'entry_info': None,
                                      'poly_seq': {
                                          '_nef_covalent_links': [{'name': 'chain_code_1', 'type': 'str', 'default': 'A'},
                                                                  {'name': 'sequence_code_1', 'type': 'int'},
                                                                  {'name': 'residue_name_1', 'type': 'str', 'uppercase': True},
                                                                  {'name': 'atom_name_1', 'type': 'str'},
                                                                  {'name': 'chain_code_2', 'type': 'str', 'default': 'A'},
                                                                  {'name': 'sequence_code_2', 'type': 'int'},
                                                                  {'name': 'residue_name_2', 'type': 'str', 'uppercase': True},
                                                                  {'name': 'atom_name_2', 'type': 'str'}
                                                                  ],
                                          '_nef_sequence': [{'name': 'chain_code', 'type': 'str', 'default': 'A'},
                                                            {'name': 'sequence_code', 'type': 'int'},
                                                            {'name': 'residue_name', 'type': 'str', 'uppercase': True}
                                                            ]
                                      },
                                      'entity': None,
                                      'chem_shift': None,
                                      'chem_shift_ref': None,
                                      'dist_restraint': None,
                                      'dihed_restraint': None,
                                      'rdc_restraint': None,
                                      'spectral_peak': {
                                          '_nef_spectrum_dimension': [{'name': 'dimension_id', 'type': 'index-int'}
                                                                      ],
                                          '_nef_spectrum_dimension_transfer': [{'name': 'dimension_1', 'type': 'positive-int'},
                                                                               {'name': 'dimension_2', 'type': 'positive-int'},
                                                                               ]
                                      },
                                      'spectral_peak_alt': None
                                      },
                              'nmr-star': {'entry_info': None,
                                           'poly_seq': {
                                               '_Bond': [{'name': 'ID', 'type': 'index-int', 'mandatory': True, 'default-from': 'self'},
                                                         {'name': 'Type', 'type': 'enum', 'mandatory': True, 'default': 'covalent',
                                                          'enum': ('amide', 'covalent', 'directed', 'disulfide', 'ester', 'ether', 'hydrogen',
                                                                   'metal coordination', 'peptide', 'thioether', 'oxime', 'thioester',
                                                                   'phosphoester', 'phosphodiester', 'diselenide', 'na')},
                                                         {'name': 'Value_order', 'type': 'enum', 'mandatory': True, 'default': 'sing',
                                                          'enum': ('sing', 'doub', 'trip', 'quad', 'arom', 'poly', 'delo', 'pi', 'directed')},
                                                         {'name': 'Entity_assembly_ID_1', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_1'},
                                                         {'name': 'Comp_index_ID_1', 'type': 'int'},
                                                         {'name': 'Comp_ID_1', 'type': 'str', 'uppercase': True},
                                                         {'name': 'Atom_ID_1', 'type': 'str'},
                                                         {'name': 'Entity_assembly_ID_2', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'Auth_asym_ID_2'},
                                                         {'name': 'Comp_index_ID_2', 'type': 'int'},
                                                         {'name': 'Comp_ID_2', 'type': 'str', 'uppercase': True},
                                                         {'name': 'Atom_ID_2', 'type': 'str'}
                                                         ],
                                               '_Entity_deleted_atom': [{'name': 'ID', 'type': 'index-int', 'mandatory': True, 'default-from': 'self'},
                                                                        {'name': 'Entity_assembly_ID', 'type': 'positive-int-as-str', 'default': '1', 'default-from': 'self'},
                                                                        {'name': 'Comp_index_ID', 'type': 'int'},
                                                                        {'name': 'Comp_ID', 'type': 'str', 'uppercase': True},
                                                                        {'name': 'Atom_ID', 'type': 'str'}
                                                                        ]
                                           },
                                           'entity': None,
                                           'chem_shift': None,
                                           'chem_shift_ref': None,
                                           'dist_restraint': None,
                                           'dihed_restraint': None,
                                           'rdc_restraint': None,
                                           'spectral_peak': {
                                               '_Spectral_dim': [{'name': 'ID', 'type': 'index-int'}
                                                                 ],
                                               '_Spectral_dim_transfer': [{'name': 'Spectral_dim_ID_1', 'type': 'positive-int'},
                                                                          {'name': 'Spectral_dim_ID_2', 'type': 'positive-int'},
                                                                          ]
                                           },
                                           'spectral_peak_alt': {
                                               '_Spectral_dim': [{'name': 'ID', 'type': 'index-int'}
                                                                 ],
                                               '_Spectral_dim_transfer': [{'name': 'Spectral_dim_ID_1', 'type': 'positive-int'},
                                                                          {'name': 'Spectral_dim_ID_2', 'type': 'positive-int'},
                                                                          ],
                                               '_Peak_general_char': [],
                                               '_Peak_char': [],
                                               '_Assigned_peak_chem_shift': []
                                           }
                                           }
                              }

        # auxiliary loop data items
        self.aux_data_items = {'nef': {'entry_info': None,
                                       'poly_seq': {
                                           '_nef_covalent_links': [],
                                           '_nef_sequence': [{'name': 'linking', 'type': 'enum', 'mandatory': False,
                                                              'enum': ('start', 'end', 'middle', 'cyclic', 'break', 'single', 'dummy'),
                                                              'enforce-enum': True},
                                                             {'name': 'residue_variant', 'type': 'str', 'mandatory': False},
                                                             {'name': 'cis_peptide', 'type': 'bool', 'mandatory': False}
                                                             ]
                                       },
                                       'entity': None,
                                       'chem_shift': None,
                                       'chem_shift_ref': None,
                                       'dist_restraint': None,
                                       'dihed_restraint': None,
                                       'rdc_restraint': None,
                                       'spectral_peak': {
                                           '_nef_spectrum_dimension': [{'name': 'axis_unit', 'type': 'enum', 'mandatory': True,
                                                                        'enum': ('ppm', 'Hz'),
                                                                        'enforce-enum': True},
                                                                       {'name': 'axis_code', 'type': 'str', 'mandatory': True},
                                                                       {'name': 'spectrometer_frequency', 'type': 'positive-float', 'mandatory': False,
                                                                        'enforce-non-zero': True},
                                                                       {'name': 'spectral_width', 'type': 'positive-float', 'mandatory': False,
                                                                        'enforce-non-zero': True},
                                                                       {'name': 'value_first_point', 'type': 'float', 'mandatory': False},
                                                                       {'name': 'folding', 'type': 'enum', 'mandatory': False,
                                                                        'enum': ('circular', 'mirror', 'none')},
                                                                       {'name': 'absolute_peak_positions', 'type': 'bool', 'mandatory': False},
                                                                       {'name': 'is_acquisition', 'type': 'bool', 'mandatory': False},
                                                                       ],
                                           '_nef_spectrum_dimension_transfer': [{'name': 'transfer_type', 'type': 'enum', 'mandatory': True,
                                                                                 'enum': ('onebond', 'jcoupling', 'jmultibond', 'relayed', 'relayed-alternate', 'through-space'),
                                                                                 'enforce-enum': True},
                                                                                {'name': 'is_indirect', 'type': 'bool', 'mandatory': False}
                                                                                ]
                                       },
                                       'spectral_peak_alt': None
                                       },
                               'nmr-star': {'entry_info': None,
                                            'poly_seq': {
                                                '_Bond': [{'name': 'Auth_asym_ID_1', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_seq_ID_1', 'type': 'int', 'mandatory': False},
                                                          {'name': 'Auth_comp_ID_1', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_atom_ID_1', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_asym_ID_2', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_seq_ID_2', 'type': 'int', 'mandatory': False},
                                                          {'name': 'Auth_comp_ID_2', 'type': 'str', 'mandatory': False},
                                                          {'name': 'Auth_atom_ID_2', 'type': 'str', 'mandatory': False}
                                                          ],
                                                '_Entity_deleted_atom': [{'name': 'Auth_entity_assembly_ID', 'type': 'str', 'mandatory': False},
                                                                         {'name': 'Auth_seq_ID', 'type': 'int', 'mandatory': False},
                                                                         {'name': 'Auth_comp_ID', 'type': 'str', 'mandatory': False},
                                                                         {'name': 'Auth_atom_ID', 'type': 'str', 'mandatory': False}
                                                                         ]
                                            },
                                            'entity': None,
                                            'chem_shift': None,
                                            'chem_shift_ref': None,
                                            'dist_restraint': None,
                                            'dihed_restraint': None,
                                            'rdc_restraint': None,
                                            'spectral_peak': {
                                                '_Spectral_dim': [{'name': 'Axis_code', 'type': 'str', 'mandatory': True},
                                                                  {'name': 'Spectrometer_frequency', 'type': 'positive-float', 'mandatory': False,
                                                                   'enforce-non-zero': True},
                                                                  {'name': 'Under_sampling_type', 'type': 'enum', 'mandatory': False,
                                                                   'enum': ('aliased', 'folded', 'not observed')},
                                                                  {'name': 'Sweep_width', 'type': 'positive-float', 'mandatory': False,
                                                                   'enforce-non-zero': True},
                                                                  {'name': 'Sweep_width_units', 'type': 'enum', 'mandatory': True,
                                                                   'enum': ('ppm', 'Hz'),
                                                                   'enforce-enum': True},
                                                                  {'name': 'Value_first_point', 'type': 'float', 'mandatory': False},
                                                                  {'name': 'Absolute_peak_positions', 'type': 'bool', 'mandatory': False},
                                                                  {'name': 'Acquisition', 'type': 'bool', 'mandatory': False},
                                                                  {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default-from': 'parent'}
                                                                  ],
                                                '_Spectral_dim_transfer': [{'name': 'Indirect', 'type': 'bool', 'mandatory': False},
                                                                           {'name': 'Type', 'type': 'enum', 'mandatory': True,
                                                                            'enum': ('onebond', 'jcoupling', 'jmultibond', 'relayed', 'relayed-alternate', 'through-space'),
                                                                            'enforce-enum': True},
                                                                           {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default-from': 'parent'}
                                                                           ]
                                            },
                                            'spectral_peak_alt': {
                                                '_Spectral_dim': [{'name': 'Axis_code', 'type': 'str', 'mandatory': True},
                                                                  {'name': 'Spectrometer_frequency', 'type': 'positive-float', 'mandatory': False,
                                                                   'enforce-non-zero': True},
                                                                  {'name': 'Under_sampling_type', 'type': 'enum', 'mandatory': False,
                                                                   'enum': ('aliased', 'folded', 'not observed')},
                                                                  {'name': 'Sweep_width', 'type': 'positive-float', 'mandatory': False,
                                                                   'enforce-non-zero': True},
                                                                  {'name': 'Sweep_width_units', 'type': 'enum', 'mandatory': True,
                                                                   'enum': ('ppm', 'Hz'),
                                                                   'enforce-enum': True},
                                                                  {'name': 'Value_first_point', 'type': 'float', 'mandatory': False},
                                                                  {'name': 'Absolute_peak_positions', 'type': 'bool', 'mandatory': False},
                                                                  {'name': 'Acquisition', 'type': 'bool', 'mandatory': False},
                                                                  {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default-from': 'parent'}
                                                                  ],
                                                '_Spectral_dim_transfer': [{'name': 'Indirect', 'type': 'bool', 'mandatory': False},
                                                                           {'name': 'Type', 'type': 'enum', 'mandatory': True,
                                                                            'enum': ('onebond', 'jcoupling', 'jmultibond', 'relayed', 'relayed-alternate', 'through-space'),
                                                                            'enforce-enum': True},
                                                                           {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default-from': 'parent'}
                                                                           ],
                                                '_Peak_general_char': [{'name': 'Peak_ID', 'type': 'positive-int', 'mandatory': True},
                                                                       {'name': 'Intensity_val', 'type': 'float', 'mandatory': True},
                                                                       {'name': 'Intensity_val_err', 'type': 'positive-float', 'mandatory': False, 'void-zero': True},
                                                                       {'name': 'Measurement_method', 'type': 'enum', 'mandatory': False,
                                                                        'enum': ('absolute height', 'height', 'relative height', 'volume', 'number of contours', 'integration')},
                                                                       {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default-from': 'parent'}
                                                                       ],
                                                '_Peak_char': [{'name': 'Peak_ID', 'type': 'positive-int', 'mandatory': True},
                                                               {'name': 'Spectral_dim_ID', 'type': 'enum-int', 'mandatory': True,
                                                                'enum': set(range(1, self.lim_num_dim)),
                                                                'enforce-enum': True},
                                                               {'name': 'Chem_shift_val', 'type': 'range-float', 'mandatory': True,
                                                                'range': self.chem_shift_range},
                                                               {'name': 'Chem_shift_val_err', 'type': 'range-float', 'mandatory': False, 'void-zero': True,
                                                                'range': self.chem_shift_error},
                                                               {'name': 'Line_width_val', 'type': 'positive-float', 'mandatory': False},
                                                               {'name': 'Line_width_val_err', 'type': 'positive-float', 'mandatory': False, 'void-zero': True},
                                                               {'name': 'Coupling_pattern', 'type': 'enum', 'mandatory': False,
                                                                'enum': ('d', 'dd', 'ddd', 'dm', 'dt', 'hxt', 'hpt', 'm', 'q', 'qd', 'qn', 's', 'sxt', 't', 'td', 'LR', '1JCH')},
                                                               {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default-from': 'parent'}
                                                               ],
                                                '_Assigned_peak_chem_shift': [{'name': 'Peak_ID', 'type': 'positive-int', 'mandatory': True},
                                                                              {'name': 'Spectral_dim_ID', 'type': 'enum-int', 'mandatory': True,
                                                                               'enum': set(range(1, self.lim_num_dim)),
                                                                               'enforce-enum': True},
                                                                              {'name': 'Set_ID', 'type': 'positive-int', 'mandatory': False},
                                                                              {'name': 'Magnetization_linkage_ID', 'type': 'positive-int', 'mandatory': False},
                                                                              {'name': 'Val', 'type': 'range-float', 'mandatory': False,
                                                                               'range': self.chem_shift_range},
                                                                              {'name': 'Contribution_fractional_val', 'type': 'range-float', 'mandatory': False,
                                                                               'range': self.weight_range},
                                                                              {'name': 'Figure_of_merit', 'type': 'range-float', 'mandatory': False,
                                                                               'range': self.weight_range},
                                                                              {'name': 'Assigned_chem_shift_list_ID', 'type': 'pointer-index', 'mandatory': False},
                                                                              {'name': 'Entity_assembly_ID', 'type': 'positive-int-as-str', 'mandatory': False},
                                                                              {'name': 'Comp_index_ID', 'type': 'int', 'mandatory': False},
                                                                              {'name': 'Comp_ID', 'type': 'str', 'mandatory': False, 'uppercase': True},
                                                                              {'name': 'Atom_ID', 'type': 'str', 'mandatory': False},
                                                                              {'name': 'Ambiguity_code', 'type': 'enum-int', 'mandatory': False,
                                                                               'enum': self.bmrb_ambiguity_codes},
                                                                              {'name': 'Ambiguity_set_ID', 'type': 'positive-int', 'mandatory': False},
                                                                              {'name': 'Auth_seq_ID', 'type': 'int', 'mandatory': False},
                                                                              {'name': 'Auth_comp_ID', 'type': 'str', 'mandatory': False},
                                                                              {'name': 'Auth_atom_ID', 'type': 'str', 'mandatory': False},
                                                                              {'name': 'Spectral_peak_list_ID', 'type': 'pointer-index', 'mandatory': True, 'default-from': 'parent'}
                                                                              ]
                                            }
                                            }
                               }

        # allowed auxiliary loop tags
        self.aux_allowed_tags = {'nef': {'entry_info': None,
                                         'poly_seq': {
                                             '_nef_covalent_links': ['chain_code_1', 'sequence_code_1', 'residue_name_1', 'atom_name_1',
                                                                     'chain_code_2', 'sequence_code_2', 'residue_name_2', 'atom_name_2'],
                                             '_nef_sequence': ['index', 'chain_code', 'sequence_code', 'residue_name', 'linking', 'residue_variant', 'cis_peptide']
                                         },
                                         'entity': None,
                                         'chem_shift': None,
                                         'chem_shift_ref': None,
                                         'dist_restraint': None,
                                         'dihed_restraint': None,
                                         'rdc_restraint': None,
                                         'spectral_peak': {
                                             '_nef_spectrum_dimension': ['dimension_id', 'axis_unit', 'axis_code',
                                                                         'spectrometer_frequency', 'spectral_width',
                                                                         'value_first_point', 'folding',
                                                                         'absolute_peak_positions', 'is_acquisition'],
                                             '_nef_spectrum_dimension_transfer': ['dimension_1', 'dimension_2', 'transfer_type', 'is_indirect']
                                         },
                                         'spectral_peak_alt': None
                                         },
                                 'nmr-star': {'entry_info': None,
                                              'poly_seq': {
                                                  '_Bond': ['ID', 'Type', 'Value_order', 'Assembly_atom_ID_1', 'Entity_assembly_ID_1'
                                                            , 'Entity_assembly_name_1', 'Entity_ID_1', 'Comp_ID_1', 'Comp_index_ID_1',
                                                            'Seq_ID_1', 'Atom_ID_1',
                                                            'Assembly_atom_ID_2', 'Entity_assembly_ID_2', 'Entity_assembly_name_2',
                                                            'Entity_ID_2', 'Comp_ID_2', 'Comp_index_ID_2', 'Seq_ID_2', 'Atom_ID_2',
                                                            'Auth_entity_assembly_ID_1', 'Auth_entity_assembly_name_1', 'Auth_asym_ID_1',
                                                            'Auth_seq_ID_1', 'Auth_comp_ID_1', 'Auth_atom_ID_1',
                                                            'Auth_entity_assembly_ID_2', 'Auth_entity_assembly_name_2', 'Auth_asym_ID_2'
                                                            , 'Auth_seq_ID_2', 'Auth_comp_ID_2', 'Auth_atom_ID_2', 'Sf_ID', 'Entry_ID', 'Assembly_ID'],
                                                  '_Entity_deleted_atom': ['ID', 'Entity_atom_list_ID', 'Entity_assembly_ID', 'Entity_ID',
                                                                           'Comp_ID', 'Comp_index_ID', 'Seq_ID', 'Atom_ID', 'Auth_entity_assembly_ID',
                                                                           'Auth_seq_ID', 'Auth_comp_ID', 'Auth_atom_ID', 'Sf_ID', 'Entry_ID', 'Assembly_ID']
                                              },
                                              'entity': None,
                                              'chem_shift': None,
                                              'chem_shift_ref': None,
                                              'dist_restraint': None,
                                              'dihed_restraint': None,
                                              'rdc_restraint': None,
                                              'spectral_peak': {
                                                  '_Spectral_dim': ['ID', 'Axis_code', 'Spectrometer_frequency', 'Atom_type',
                                                                    'Atom_isotope_number', 'Spectral_region', 'Magnetization_linkage_ID',
                                                                    'Under_sampling_type', 'Sweep_width', 'Sweep_width_units', 'Value_first_point',
                                                                    'Absolute_peak_positions', 'Acquisition', 'Center_frequency_offset',
                                                                    'Encoding_code', 'Encoded_reduced_dimension_ID', 'Sf_ID', 'Entry_ID',
                                                                    'Spectral_peak_list_ID'],
                                                  '_Spectral_dim_transfer': ['Spectral_dim_ID_1', 'Spectral_dim_ID_2', 'Indirect', 'Type',
                                                                             'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID']
                                              },
                                              'spectral_peak_alt': {
                                                  '_Spectral_dim': ['ID', 'Axis_code', 'Spectrometer_frequency', 'Atom_type', 'Atom_isotope_number',
                                                                    'Spectral_region', 'Magnetization_linkage_ID', 'Under_sampling_type', 'Sweep_width',
                                                                    'Sweep_width_units', 'Value_first_point', 'Absolute_peak_positions', 'Acquisition',
                                                                    'Center_frequency_offset', 'Encoding_code', 'Encoded_reduced_dimension_ID',
                                                                    'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID'],
                                                  '_Spectral_dim_transfer': ['Spectral_dim_ID_1', 'Spectral_dim_ID_2', 'Indirect',
                                                                             'Type', 'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID'],
                                                  '_Peak_general_char': ['Peak_ID', 'Intensity_val', 'Intensity_val_err', 'Measurement_method',
                                                                         'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID'],
                                                  '_Peak_char': ['Peak_ID', 'Spectral_dim_ID', 'Chem_shift_val', 'Chem_shift_val_err', 'Line_width_val',
                                                                 'Line_width_val_err', 'Phase_val', 'Phase_val_err', 'Decay_rate_val', 'Decay_rate_val_err',
                                                                 'Coupling_pattern', 'Bounding_box_upper_val', 'Bounding_box_lower_val', 'Bounding_box_range_val',
                                                                 'Details', 'Derivation_method_ID', 'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID'],
                                                  '_Assigned_peak_chem_shift': ['Peak_ID', 'Spectral_dim_ID', 'Set_ID', 'Magnetization_linkage_ID', 'Assembly_atom_ID',
                                                                                'Val', 'Contribution_fractional_val', 'Figure_of_merit', 'Assigned_chem_shift_list_ID',
                                                                                'Atom_chem_shift_ID', 'Entity_assembly_ID', 'Entity_ID', 'Comp_index_ID', 'Comp_ID',
                                                                                'Atom_ID', 'Ambiguity_code', 'Ambiguity_set_ID', 'Auth_atom_peak_num', 'Auth_entity_ID',
                                                                                'Auth_seq_ID', 'Auth_comp_ID', 'Auth_atom_ID', 'Auth_ambiguity_code',
                                                                                'Auth_ambiguity_set_ID', 'Auth_amb_atom_grp_ID', 'Resonance_ID', 'Details',
                                                                                'Sf_ID', 'Entry_ID', 'Spectral_peak_list_ID']
                                              }
                                              }
                                 }

        # item name in cs loop
        self.item_names_in_cs_loop = {'nef': {'chain_id': 'chain_code',
                                              'seq_id': 'sequence_code',
                                              'comp_id': 'residue_name',
                                              'atom_id': 'atom_name',
                                              'value': 'value',
                                              'error': 'value_uncertainty',
                                              'atom_type': 'element',
                                              'isotope_number': 'isotope_number'
                                              },
                                      'nmr-star': {'chain_id': 'Entity_assembly_ID',
                                                   'seq_id': 'Comp_index_ID',
                                                   'comp_id': 'Comp_ID',
                                                   'atom_id': 'Atom_ID',
                                                   'value': 'Val',
                                                   'error': 'Val_err',
                                                   'atom_type': 'Atom_type',
                                                   'isotope_number': 'Atom_isotope_number'
                                                   }
                                      }

        # item name in spectral peak loop
        self.item_names_in_pk_loop = {'nef': {'chain_id': 'chain_code_%s',
                                              'seq_id': 'sequence_code_%s',
                                              'comp_id': 'residue_name_%s',
                                              'atom_id': 'atom_name_%s',
                                              'position': 'position_%s'
                                              },
                                      'nmr-star': {'chain_id': 'Entity_assembly_ID_%s',
                                                   'seq_id': 'Comp_index_ID_%s',
                                                   'comp_id': 'Comp_ID_%s',
                                                   'atom_id': 'Atom_ID_%s',
                                                   'position': 'Position_%s'
                                                   }
                                      }

        # item name in distance restraint loop
        self.item_names_in_ds_loop = {'nef': {'combination_id': 'restraint_combination_id',
                                              'chain_id_1': 'chain_code_1',
                                              'seq_id_1': 'sequence_code_1',
                                              'comp_id_1': 'residue_name_1',
                                              'atom_id_1': 'atom_name_1',
                                              'chain_id_2': 'chain_code_2',
                                              'seq_id_2': 'sequence_code_2',
                                              'comp_id_2': 'residue_name_2',
                                              'atom_id_2': 'atom_name_2',
                                              'target_value': 'target_value',
                                              'lower_linear_limit': 'lower_linear_limit',
                                              'upper_linear_limit': 'upper_linear_limit',
                                              'lower_limit': 'lower_limit',
                                              'upper_limit': 'upper_limit'
                                              },
                                      'nmr-star': {'combination_id': 'Combination_ID',
                                                   'chain_id_1': 'Entity_assembly_ID_1',
                                                   'seq_id_1': 'Comp_index_ID_1',
                                                   'comp_id_1': 'Comp_ID_1',
                                                   'atom_id_1': 'Atom_ID_1',
                                                   'chain_id_2': 'Entity_assembly_ID_2',
                                                   'seq_id_2': 'Comp_index_ID_2',
                                                   'comp_id_2': 'Comp_ID_2',
                                                   'atom_id_2': 'Atom_ID_2',
                                                   'target_value': 'Target_val',
                                                   'target_value_alt': 'Distance_val',
                                                   'lower_linear_limit': 'Lower_linear_limit',
                                                   'upper_linear_limit': 'Upper_linear_limit',
                                                   'lower_limit': 'Distance_lower_bound_val',
                                                   'upper_limit': 'Distance_upper_bound_val'
                                                   }
                                      }

        # item name in dihedral restraint loop
        self.item_names_in_dh_loop = {'nef': {'combination_id': 'restraint_combination_id',
                                              'chain_id_1': 'chain_code_1',
                                              'seq_id_1': 'sequence_code_1',
                                              'comp_id_1': 'residue_name_1',
                                              'atom_id_1': 'atom_name_1',
                                              'chain_id_2': 'chain_code_2',
                                              'seq_id_2': 'sequence_code_2',
                                              'comp_id_2': 'residue_name_2',
                                              'atom_id_2': 'atom_name_2',
                                              'chain_id_3': 'chain_code_3',
                                              'seq_id_3': 'sequence_code_3',
                                              'comp_id_3': 'residue_name_3',
                                              'atom_id_3': 'atom_name_3',
                                              'chain_id_4': 'chain_code_4',
                                              'seq_id_4': 'sequence_code_4',
                                              'comp_id_4': 'residue_name_4',
                                              'atom_id_4': 'atom_name_4',
                                              'angle_type': 'name'
                                              },
                                      'nmr-star': {'combination_id': 'Combination_ID',
                                                   'chain_id_1': 'Entity_assembly_ID_1',
                                                   'seq_id_1': 'Comp_index_ID_1',
                                                   'comp_id_1': 'Comp_ID_1',
                                                   'atom_id_1': 'Atom_ID_1',
                                                   'chain_id_2': 'Entity_assembly_ID_2',
                                                   'seq_id_2': 'Comp_index_ID_2',
                                                   'comp_id_2': 'Comp_ID_2',
                                                   'atom_id_2': 'Atom_ID_2',
                                                   'chain_id_3': 'Entity_assembly_ID_3',
                                                   'seq_id_3': 'Comp_index_ID_3',
                                                   'comp_id_3': 'Comp_ID_3',
                                                   'atom_id_3': 'Atom_ID_3',
                                                   'chain_id_4': 'Entity_assembly_ID_4',
                                                   'seq_id_4': 'Comp_index_ID_4',
                                                   'comp_id_4': 'Comp_ID_4',
                                                   'atom_id_4': 'Atom_ID_4',
                                                   'angle_type': 'Torsion_angle_name',
                                                   }
                                      }

        # item name in rdc restraint loop
        self.item_names_in_rdc_loop = {'nef': {'combination_id': 'restraint_combination_id',
                                               'chain_id_1': 'chain_code_1',
                                               'seq_id_1': 'sequence_code_1',
                                               'comp_id_1': 'residue_name_1',
                                               'atom_id_1': 'atom_name_1',
                                               'chain_id_2': 'chain_code_2',
                                               'seq_id_2': 'sequence_code_2',
                                               'comp_id_2': 'residue_name_2',
                                               'atom_id_2': 'atom_name_2'
                                               },
                                       'nmr-star': {'combination_id': 'Combination_ID',
                                                    'chain_id_1': 'Entity_assembly_ID_1',
                                                    'seq_id_1': 'Comp_index_ID_1',
                                                    'comp_id_1': 'Comp_ID_1',
                                                    'atom_id_1': 'Atom_ID_1',
                                                    'chain_id_2': 'Entity_assembly_ID_2',
                                                    'seq_id_2': 'Comp_index_ID_2',
                                                    'comp_id_2': 'Comp_ID_2',
                                                    'atom_id_2': 'Atom_ID_2'
                                                    }
                                       }

        # saveframe tag name for chemical shift list in spectral peak
        self.cs_list_sf_tag_name = {'nef': 'chemical_shift_list',
                                    'nmr-star': 'Chemical_shift_list'
                                    }

        # taken from wwpdb.utils.align.SequenceReferenceData.py
        self.monDict3 = {'ALA': 'A',
                         'ARG': 'R',
                         'ASN': 'N',
                         'ASP': 'D',
                         'ASX': 'B',
                         'CYS': 'C',
                         'GLN': 'Q',
                         'GLU': 'E',
                         'GLX': 'Z',
                         'GLY': 'G',
                         'HIS': 'H',
                         'ILE': 'I',
                         'LEU': 'L',
                         'LYS': 'K',
                         'MET': 'M',
                         'PHE': 'F',
                         'PRO': 'P',
                         'SER': 'S',
                         'THR': 'T',
                         'TRP': 'W',
                         'TYR': 'Y',
                         'VAL': 'V',
                         'DA': 'A',
                         'DC': 'C',
                         'DG': 'G',
                         'DT': 'T',
                         'DU': 'U',
                         'DI': 'I',
                         'A': 'A',
                         'C': 'C',
                         'G': 'G',
                         'I': 'I',
                         'T': 'T',
                         'U': 'U'
                         }

        # conventional dihedral angle names in standard residues
        self.dihed_ang_names = ('PHI', 'PSI', 'OMEGA', 'CHI1', 'CHI2', 'CHI3', 'CHI4', 'CHI5',
                                'ALPHA', 'BETA', 'GAMMA', 'DELTA', 'EPSILON', 'ZETA',
                                'NU0', 'NU1', 'NU2', 'NU3', 'NU4',
                                'TAU0', 'TAU1', 'TAU2', 'TAU3', 'TAU4',
                                'CHI21', 'CHI22', 'CHI31', 'CHI32', 'CHI42')

        # patterns for detection of dihedral angle type
        self.dihed_atom_ids = ['N', 'CA', 'C']

        self.chi1_atom_id_4_pat = re.compile(r'^[COS]G1?$')
        self.chi2_atom_id_3_pat = re.compile(r'^CG1?$')
        self.chi2_atom_id_4_pat = re.compile(r'^[CNOS]D1?$')
        self.chi3_atom_id_3_pat = re.compile(r'^[CS]D$')
        self.chi3_atom_id_4_pat = re.compile(r'^[CNO]E1?$')
        self.chi4_atom_id_3_pat = re.compile(r'^[CN]E$')
        self.chi4_atom_id_4_pat = re.compile(r'^[CN]Z$')

        # patterns for enum failure message
        self.chk_desc_pat = re.compile(r'^(.*) \'(.*)\' should be one of \((.*)\)\.(.*)$')
        self.chk_desc_pat_one = re.compile(r'^(.*) \'(.*)\' should be one of (.*)\.(.*)$')
        self.chk_desc_pat_mand = re.compile(r'^The mandatory type _.*\.(.*) \'(.*)\' is missing and the type must be one of \((.*)\)\.(.*)$')
        self.chk_desc_pat_mand_one = re.compile(r'^The mandatory type _.*\.(.*) \'(.*)\' is missing and the type must be one of (.*)\.(.*)$')

        # pattern for guessing original saveframe name DAOTHER-7389, issue #4
        self.chk_unresolved_sf_name_pat = re.compile(r'^(.*)_\d+$')

        # main contents of loops
        self.__lp_data = {'entry_info': [],
                          'poly_seq': [],
                          'entity': [],
                          'chem_shift': [],
                          'chem_shift_ref': [],
                          'dist_restraint': [],
                          'dihed_restraint': [],
                          'rdc_restraint': [],
                          'spectral_peak': [],
                          'spectral_peak_alt': []
                          }

        # auxiliary contents of loops
        self.__aux_data = {'entry_info': [],
                           'poly_seq': [],
                           'entity': [],
                           'chem_shift': [],
                           'chem_shift_ref': [],
                           'dist_restraint': [],
                           'dihed_restraint': [],
                           'rdc_restraint': [],
                           'spectral_peak': [],
                           'spectral_peak_alt': []
                           }

        # contents of savefram tags
        self.__sf_tag_data = {'entry_info': [],
                              'poly_seq': [],
                              'entity': [],
                              'chem_shift': [],
                              'chem_shift_ref': [],
                              'dist_restraint': [],
                              'dihed_restraint': [],
                              'rdc_restraint': [],
                              'spectral_peak': [],
                              'spectral_peak_alt': []
                              }

        # self.__remapped_def_chain_id = {}

        self.authSeqMap = None

        # Pairwise align
        self.__pA = PairwiseAlign()
        self.__pA.setVerbose(self.__verbose)

        # CCD accessing utility
        self.__cICommon = ConfigInfoAppCommon(getSiteId())
        self.__ccCvsPath = self.__cICommon.get_site_cc_cvs_path()

        self.__ccR = ChemCompReader(self.__verbose, self.__lfh)
        self.__ccR.setCachePath(self.__ccCvsPath)

        # representative model id
        self.__representative_model_id = 1
        # total number of models
        self.__total_models = 0
        # atom id list in model
        self.__coord_atom_id = None
        # tautomer state in model
        self.__coord_tautomer = {}
        # rotamer state in model
        self.__coord_rotamer = {}
        # nearest aromatic ring in model
        self.__coord_near_ring = {}
        # nearest paramagnetic/ferromagnetic atom in model
        self.__coord_near_para_ferro = {}
        # bond length in model
        self.__coord_bond_length = {}

        self.__last_comp_id = None
        self.__last_comp_id_test = False
        self.__last_chem_comp_dict = None
        self.__last_chem_comp_atoms = None
        self.__last_chem_comp_bonds = None

        # taken from wwpdb.apps.ccmodule.io.ChemCompIo
        self.__chem_comp_atom_dict = [
            ('_chem_comp_atom.comp_id', '%s', 'str', ''),
            ('_chem_comp_atom.atom_id', '%s', 'str', ''),
            ('_chem_comp_atom.alt_atom_id', '%s', 'str', ''),
            ('_chem_comp_atom.type_symbol', '%s', 'str', ''),
            ('_chem_comp_atom.charge', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_align', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_aromatic_flag', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_leaving_atom_flag', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_stereo_config', '%s', 'str', ''),
            ('_chem_comp_atom.model_Cartn_x', '%s', 'str', ''),
            ('_chem_comp_atom.model_Cartn_y', '%s', 'str', ''),
            ('_chem_comp_atom.model_Cartn_z', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_model_Cartn_x_ideal', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_model_Cartn_y_ideal', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_model_Cartn_z_ideal', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_component_atom_id', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_component_comp_id', '%s', 'str', ''),
            ('_chem_comp_atom.pdbx_ordinal', '%s', 'str', '')
        ]

        atom_id = next(d for d in self.__chem_comp_atom_dict if d[0] == '_chem_comp_atom.atom_id')
        self.__cca_atom_id = self.__chem_comp_atom_dict.index(atom_id)

        # aromatic_flag = next(d for d in self.__chem_comp_atom_dict if d[0] == '_chem_comp_atom.pdbx_aromatic_flag')
        # self.__cca_aromatic_flag = self.__chem_comp_atom_dict.index(aromatic_flag)

        leaving_atom_flag = next(d for d in self.__chem_comp_atom_dict if d[0] == '_chem_comp_atom.pdbx_leaving_atom_flag')
        self.__cca_leaving_atom_flag = self.__chem_comp_atom_dict.index(leaving_atom_flag)

        type_symbol = next(d for d in self.__chem_comp_atom_dict if d[0] == '_chem_comp_atom.type_symbol')
        self.__cca_type_symbol = self.__chem_comp_atom_dict.index(type_symbol)

        # taken from wwpdb.apps.ccmodule.io.ChemCompIo
        self.__chem_comp_bond_dict = [
            ('_chem_comp_bond.comp_id', '%s', 'str', ''),
            ('_chem_comp_bond.atom_id_1', '%s', 'str', ''),
            ('_chem_comp_bond.atom_id_2', '%s', 'str', ''),
            ('_chem_comp_bond.value_order', '%s', 'str', ''),
            ('_chem_comp_bond.pdbx_aromatic_flag', '%s', 'str', ''),
            ('_chem_comp_bond.pdbx_stereo_config', '%s', 'str', ''),
            ('_chem_comp_bond.pdbx_ordinal', '%s', 'str', '')
        ]

        atom_id_1 = next(d for d in self.__chem_comp_bond_dict if d[0] == '_chem_comp_bond.atom_id_1')
        self.__ccb_atom_id_1 = self.__chem_comp_bond_dict.index(atom_id_1)

        atom_id_2 = next(d for d in self.__chem_comp_bond_dict if d[0] == '_chem_comp_bond.atom_id_2')
        self.__ccb_atom_id_2 = self.__chem_comp_bond_dict.index(atom_id_2)

        aromatic_flag = next(d for d in self.__chem_comp_bond_dict if d[0] == '_chem_comp_bond.pdbx_aromatic_flag')
        self.__ccb_aromatic_flag = self.__chem_comp_bond_dict.index(aromatic_flag)

        # CIF reader
        self.__cR = CifReader(self.__verbose, self.__lfh)

        # extracted conformational annotation of coordinate file
        self.__nmr_struct_conf = {}

        # used for debuging only, it should be empty for production
        self.__target_framecode = ''

        # suspended error items for polypeptide
        self.__suspended_errors_for_polypeptide = []

        # RCI
        self.__rci = RCI(False, self.__lfh)

    def setVerbose(self, flag):
        """ Set verbose mode.
        """

        self.__verbose = flag
        self.__debug = flag

    def setSource(self, fPath):
        """ Set primary source file path.
        """

        if os.access(fPath, os.F_OK):
            self.__srcPath = os.path.abspath(fPath)

        else:
            raise IOError("+NmrDpUtility.setSource() ++ Error  - Could not access to file path %s." % fPath)

    def setDestination(self, fPath):
        """ Set primary destination file path.
        """

        if fPath is not None:
            self.__dstPath = os.path.abspath(fPath)

    def setLog(self, fPath):
        """ Set a log file path for the primary input source.
        """

        if fPath is not None:
            self.__logPath = os.path.abspath(fPath)

    def addInput(self, name=None, value=None, type='file'):
        """ Add a named input and value to the dictionary of input parameters.
        """

        try:

            if type == 'param':
                self.__inputParamDict[name] = value
            elif type == 'file':
                self.__inputParamDict[name] = os.path.abspath(value)
            elif type == 'file_list':
                self.__inputParamDict[name] = [os.path.abspath(f) for f in value]
            elif type == 'file_dict_list':
                if any(f for f in value if 'original_file_name' in f):
                    self.__inputParamDict[name] = [{'file_name': os.path.abspath(f['file_name']), 'file_type': f['file_type'], 'original_file_name': f['original_file_name']} for f in value]
                else:
                    self.__inputParamDict[name] = [{'file_name': os.path.abspath(f['file_name']), 'file_type': f['file_type']} for f in value]
            else:
                raise ValueError("+NmrDpUtility.addInput() ++ Error  - Unknown input type %s." % type)

        except Exception as e:
            raise ValueError("+NmrDpUtility.addInput() ++ Error  - %s" % str(e))

    def addOutput(self, name=None, value=None, type='file'):
        """ Add a named input and value to the dictionary of output parameters.
        """

        try:

            if type == 'param':
                self.__outputParamDict[name] = value
            elif type == 'file':
                self.__outputParamDict[name] = os.path.abspath(value)
            elif type == 'file_list':
                self.__outputParamDict[name] = [os.path.abspath(f) for f in value]
            else:
                raise ValueError("+NmrDpUtility.addOutput() ++ Error  - Unknown output type %s." % type)

            return True

        except Exception as e:
            raise ValueError("+NmrDpUtility.addOutput() ++ Error  - %s" % str(e))

    def op(self, op):
        """ Perform a series of tasks for a given workflow operation.
        """

        self.__rescue_mode = True

        self.__combined_mode = 'cs' not in op

        if self.__combined_mode:
            if self.__srcPath is None:
                raise ValueError("+NmrDpUtility.op() ++ Error  - No input provided for workflow operation %s." % op)
        else:
            cs_file_path_list = 'chem_shift_file_path_list'

            if cs_file_path_list not in self.__inputParamDict:
                raise ValueError("+NmrDpUtility.op() ++ Error  - No input provided for workflow operation %s." % op)

            self.__cs_file_path_list_len = len(self.__inputParamDict[cs_file_path_list])

            self.__file_path_list_len = self.__cs_file_path_list_len

            mr_file_path_list = 'restraint_file_path_list'

            if mr_file_path_list in self.__inputParamDict:
                self.__file_path_list_len += len(self.__inputParamDict[mr_file_path_list])

        self.__release_mode = 'release' in op

        if self.__verbose:
            self.__lfh.write("+NmrDpUtility.op() starting op %s\n" % op)

        if op not in self.__workFlowOps:
            raise KeyError("+NmrDpUtility.op() ++ Error  - Unknown workflow operation %s." % op)

        if 'bmrb_only' in self.__inputParamDict and self.__inputParamDict['bmrb_only'] is not None:
            if isinstance(self.__inputParamDict['bmrb_only'], bool):
                self.__bmrb_only = self.__inputParamDict['bmrb_only']
            else:
                self.__bmrb_only = self.__inputParamDict['bmrb_only'] in self.true_value

        if self.__bmrb_only:
            self.cs_anomalous_error_scaled_by_sigma = 4.0
            self.cs_unusual_error_scaled_by_sigma = 3.5
            self.cs_diff_error_scaled_by_sigma = 4.0

        if 'nonblk_anomalous_cs' in self.__inputParamDict and self.__inputParamDict['nonblk_anomalous_cs'] is not None:
            if isinstance(self.__inputParamDict['nonblk_anomalous_cs'], bool):
                self.__nonblk_anomalous_cs = self.__inputParamDict['nonblk_anomalous_cs']
            else:
                self.__nonblk_anomalous_cs = self.__inputParamDict['nonblk_anomalous_cs'] in self.true_value

        if 'nonblk_bad_nterm' in self.__inputParamDict and self.__inputParamDict['nonblk_bad_nterm'] is not None:
            if isinstance(self.__inputParamDict['nonblk_bad_nterm'], bool):
                self.__nonblk_bad_nterm = self.__inputParamDict['nonblk_bad_nterm']
            else:
                self.__nonblk_bad_nterm = self.__inputParamDict['nonblk_bad_nterm'] in self.true_value

        if 'update_poly_seq' in self.__inputParamDict and self.__inputParamDict['update_poly_seq'] is not None:
            if isinstance(self.__inputParamDict['update_poly_seq'], bool):
                self.__update_poly_seq = self.__inputParamDict['update_poly_seq']
            else:
                self.__update_poly_seq = self.__inputParamDict['update_poly_seq'] in self.true_value

        if 'resolve_conflict' in self.__inputParamDict and self.__inputParamDict['resolve_conflict'] is not None:
            if isinstance(self.__inputParamDict['resolve_conflict'], bool):
                self.__resolve_conflict = self.__inputParamDict['resolve_conflict']
            else:
                self.__resolve_conflict = self.__inputParamDict['resolve_conflict'] in self.true_value

        if 'check_mandatory_tag' in self.__inputParamDict and self.__inputParamDict['check_mandatory_tag'] is not None:
            if isinstance(self.__inputParamDict['check_mandatory_tag'], bool):
                self.__check_mandatory_tag = self.__inputParamDict['check_mandatory_tag']
            else:
                self.__check_mandatory_tag = self.__inputParamDict['check_mandatory_tag'] in self.true_value

        if 'check_auth_seq' in self.__inputParamDict and self.__inputParamDict['check_auth_seq'] is not None:
            if isinstance(self.__inputParamDict['check_auth_seq'], bool):
                self.__check_auth_seq = self.__inputParamDict['check_auth_seq']
            else:
                self.__check_auth_seq = self.__inputParamDict['check_auth_seq'] in self.true_value

        if 'transl_pseudo_name' in self.__inputParamDict and self.__inputParamDict['transl_pseudo_name'] is not None:
            if isinstance(self.__inputParamDict['transl_pseudo_name'], bool):
                self.__transl_pseudo_name = self.__inputParamDict['transl_pseudo_name']
            else:
                self.__transl_pseudo_name = self.__inputParamDict['transl_pseudo_name'] in self.true_value
        elif op in ('nmr-str-consistency-check', 'nmr-str2str-deposit', 'nmr-str2nef-release'):
            self.__transl_pseudo_name = True

        if 'tolerant_seq_align' in self.__inputParamDict and self.__inputParamDict['tolerant_seq_align'] is not None:
            if isinstance(self.__inputParamDict['tolerant_seq_align'], bool):
                self.__tolerant_seq_align = self.__inputParamDict['tolerant_seq_align']
            else:
                self.__tolerant_seq_align = self.__inputParamDict['tolerant_seq_align'] in self.true_value

        if 'fix_format_issue' in self.__inputParamDict and self.__inputParamDict['fix_format_issue'] is not None:
            if isinstance(self.__inputParamDict['fix_format_issue'], bool):
                self.__fix_format_issue = self.__inputParamDict['fix_format_issue']
            else:
                self.__fix_format_issue = self.__inputParamDict['fix_format_issue'] in self.true_value
        elif not self.__combined_mode or self.__release_mode:
            self.__fix_format_issue = True

        if 'excl_missing_data' in self.__inputParamDict and self.__inputParamDict['excl_missing_data'] is not None:
            if isinstance(self.__inputParamDict['excl_missing_data'], bool):
                self.__excl_missing_data = self.__inputParamDict['excl_missing_data']
            else:
                self.__excl_missing_data = self.__inputParamDict['excl_missing_data'] in self.true_value
        elif not self.__combined_mode:
            self.__excl_missing_data = True

        if 'trust_pdbx_nmr_ens' in self.__inputParamDict and self.__inputParamDict['trust_pdbx_nmr_ens'] is not None:
            if isinstance(self.__inputParamDict['trust_pdbx_nmr_ens'], bool):
                self.__trust_pdbx_nmr_ens = self.__inputParamDict['trust_pdbx_nmr_ens']
            else:
                self.__trust_pdbx_nmr_ens = self.__inputParamDict['trust_pdbx_nmr_ens'] in self.true_value
        elif self.__release_mode:
            self.__trust_pdbx_nmr_ens = True

        if 'cutoff_rmsd' in self.__inputParamDict and self.__inputParamDict['cutoff_rmsd'] is not None:
            if isinstance(self.__inputParamDict['cutoff_rmsd'], float):
                self.cutoff_rmsd = self.__inputParamDict['cutoff_rmsd']

        if 'entry_id' in self.__outputParamDict and self.__outputParamDict['entry_id'] is not None:
            self.__entry_id = self.__outputParamDict['entry_id']

        if 'insert_entry_id_to_loops' in self.__outputParamDict and self.__outputParamDict['insert_entry_id_to_loops'] is not None:
            if isinstance(self.__outputParamDict['insert_entry_id_to_loops'], bool):
                self.__insert_entry_id_to_loops = self.__outputParamDict['insert_entry_id_to_loops']
            else:
                self.__insert_entry_id_to_loops = self.__outputParamDict['insert_entry_id_to_loops'] in self.true_value

        if 'retain_original' in self.__outputParamDict and self.__outputParamDict['retain_original'] is not None:
            if isinstance(self.__outputParamDict['retain_original'], bool):
                self.__retain_original = self.__outputParamDict['retain_original']
            else:
                self.__retain_original = self.__outputParamDict['retain_original'] in self.true_value

        if 'leave_intl_note' in self.__outputParamDict and self.__outputParamDict['leave_intl_note'] is not None:
            if isinstance(self.__outputParamDict['leave_intl_note'], bool):
                self.__leave_intl_note = self.__outputParamDict['leave_intl_note']
            else:
                self.__leave_intl_note = self.__outputParamDict['leave_intl_note'] in self.true_value

        if 'reduced_atom_notation' in self.__outputParamDict and self.__outputParamDict['reduced_atom_notation'] is not None:
            if isinstance(self.__outputParamDict['reduced_atom_notation'], bool):
                self.__reduced_atom_notation = self.__outputParamDict['reduced_atom_notation']
            else:
                self.__reduced_atom_notation = self.__outputParamDict['reduced_atom_notation'] in self.true_value

        self.__op = op

        if op.endswith('consistency-check'):

            for task in self.__procTasksDict['consistency-check']:

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.op() starting op %s - task %s\n" % (op, task.__name__))

                start_time = time.time()

                if not task():
                    pass

                if self.__debug:
                    end_time = time.time()
                    if end_time - start_time > 1.0:
                        self.__lfh.write("op: %s, task: %s, elapsed time: %s sec\n" % (op, task.__name__, '{:.1f}'.format(end_time - start_time)))

        elif op.endswith('deposit') or op.endswith('release'):

            for task in self.__procTasksDict['deposit']:

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.op() starting op %s - task %s\n" % (op, task.__name__))

                start_time = time.time()

                if not task():
                    pass

                if self.__debug:
                    end_time = time.time()
                    if end_time - start_time > 1.0:
                        self.__lfh.write("op: %s, task: %s, elapsed time: %s sec\n" % (op, task.__name__, '{:.1f}'.format(end_time - start_time)))

        # run workflow operation specific tasks
        if op in self.__procTasksDict:

            for task in self.__procTasksDict[op]:

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.op() starting op %s - task %s\n" % (op, task.__name__))

                start_time = time.time()

                if not task():
                    if task.__name__ in (self.__translateNef2Str.__name__, self.__translateStr2Nef.__name__):
                        break

                if self.__debug:
                    end_time = time.time()
                    if end_time - start_time > 1.0:
                        self.__lfh.write("op: %s, task: %s, elapsed time: %s sec\n" % (op, task.__name__, '{:.1f}'.format(end_time - start_time)))

        self.__dumpDpReport()

        return not self.report.isError()

    def __dumpDpReport(self):
        """ Dump current NMR data processing report.
        """

        if self.report_prev is not None:
            self.report.inheritFormatIssueErrors(self.report_prev)
            self.report.inheritCorrectedFormatIssueWarnings(self.report_prev)
            self.report.inheritCorrectedSaveframeNameWarnings(self.report_prev)

            if self.report_prev.error.get() is not None:
                self.report.setCorrectedError(self.report_prev)

            if self.report_prev.warning.get() is not None:
                self.report.setCorrectedWarning(self.report_prev)

        self.report.warning.sortChemicalShiftValidation()
        self.report.warning.sortBySigma('conflicted_data')
        self.report.warning.sortBySigma('inconsistent_data')

        self.report.clean()

        if self.__logPath is None:
            return False

        return self.report.writeFile(self.__logPath)

    def __initializeDpReport(self, srcPath=None):
        """ Initialize NMR data processing report.
        """

        if srcPath is None:
            srcPath = self.__srcPath

        self.report = NmrDpReport()

        input_source = None

        if self.__combined_mode:

            # set primary input source as NMR unified data
            input_source = self.report.input_sources[0]

            file_type = 'nef' if 'nef' in self.__op and 'str2nef' not in self.__op else 'nmr-star'
            content_type = self.content_type[file_type]

            input_source.setItemValue('file_name', os.path.basename(srcPath))
            input_source.setItemValue('file_type', file_type)
            input_source.setItemValue('content_type', content_type)

        else:

            cs_file_path_list = 'chem_shift_file_path_list'

            for csListId, csPath in enumerate(self.__inputParamDict[cs_file_path_list]):

                if csListId > 0:
                    self.report.appendInputSource()

                input_source = self.report.input_sources[csListId]

                file_type = 'nmr-star'  # 'nef' in self.__op else 'nmr-star' # DAOTHER-5673

                input_source.setItemValue('file_name', os.path.basename(csPath))
                input_source.setItemValue('file_type', file_type)
                input_source.setItemValue('content_type', 'nmr-chemical-shifts')

            mr_file_path_list = 'restraint_file_path_list'

            if mr_file_path_list in self.__inputParamDict:

                file_path_list_len = self.__cs_file_path_list_len

                for mrPath in self.__inputParamDict[mr_file_path_list]:

                    self.report.appendInputSource()

                    input_source = self.report.input_sources[file_path_list_len]

                    file_type = 'nmr-star'  # 'nef' if 'nef' in self.__op else 'nmr-star' # DAOTHER-5673

                    input_source.setItemValue('file_name', os.path.basename(mrPath))
                    input_source.setItemValue('file_type', file_type)
                    input_source.setItemValue('content_type', 'nmr-restraints')

                    file_path_list_len += 1

            ar_file_path_list = 'atypical_restraint_file_path_list'

            if ar_file_path_list in self.__inputParamDict:

                file_path_list_len = self.__cs_file_path_list_len

                for ar in self.__inputParamDict[ar_file_path_list]:

                    self.report.appendInputSource()

                    input_source = self.report.input_sources[file_path_list_len]

                    input_source.setItemValue('file_name', os.path.basename(ar['file_name']))
                    input_source.setItemValue('file_type', ar['file_type'])
                    input_source.setItemValue('content_type', 'nmr-restraints')
                    if 'original_file_name' in ar:
                        input_source.setItemValue('original_file_name', ar['original_file_name'])

                    file_path_list_len += 1

        self.__star_data_type = []
        self.__star_data = []
        self.__sf_name_corr = []

        self.__original_error_message = []

        self.__testDiamagnetism()

        return input_source is not None

    def __testDiamagnetism(self):
        """ Test diamagnetism of molecular assembly.
        """

        if not self.__parseCoordFilePath():
            return

        try:

            chem_comp = self.__cR.getDictList('chem_comp')

            non_std_comp_ids = [i['id'] for i in chem_comp if i['mon_nstd_flag'] != 'y']

            if len(non_std_comp_ids) == 0:
                return

            for comp_id in non_std_comp_ids:

                if self.__updateChemCompDict(comp_id):  # matches with comp_id in CCD

                    ref_elems = set(a[self.__cca_type_symbol] for a in self.__last_chem_comp_atoms if a[self.__cca_leaving_atom_flag] != 'Y')

                    for elem in ref_elems:
                        if elem in self.paramag_elems or elem in self.ferromag_elems:
                            self.report.setDiamagnetic(False)
                            break

        except:  # noqa: E722 pylint: disable=bare-except
            pass

    def __updateChemCompDict(self, comp_id):
        """ Update CCD information for a given comp_id.
            @return: True for successfully update CCD information or False for the case a given comp_id does not exist in CCD
        """

        comp_id = comp_id.upper()

        if comp_id != self.__last_comp_id:
            self.__last_comp_id_test = False if '_' in comp_id else self.__ccR.setCompId(comp_id)
            self.__last_comp_id = comp_id

            if self.__last_comp_id_test:
                self.__last_chem_comp_dict = self.__ccR.getChemCompDict()
                self.__last_chem_comp_atoms = self.__ccR.getAtomList()
                self.__last_chem_comp_bonds = self.__ccR.getBonds()

        return self.__last_comp_id_test

    def __validateInputSource(self, srcPath=None):
        """ Validate NMR data as primary input source.
        """

        if srcPath is None:
            srcPath = self.__srcPath

        is_done = True

        if self.__combined_mode:

            codec = detect_bom(srcPath, 'utf-8')

            srcPath_ = None

            if codec != 'utf-8':
                srcPath_ = srcPath + '~'
                convert_codec(srcPath, srcPath_, codec, 'utf-8')
                srcPath = srcPath_

            is_valid, message = self.__nefT.validate_file(srcPath, 'A')  # 'A' for NMR unified data, 'S' for assigned chemical shifts, 'R' for restraints.

            self.__original_error_message.append(message)

            _file_type = message['file_type']  # nef/nmr-star/unknown

            input_source = self.report.input_sources[0]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if is_valid:

                if _file_type != file_type:

                    err = "%r was selected as %s file, but recognized as %s file. Please re-upload the file." % (file_name,
                                                                                                                 self.readable_file_type[file_type],
                                                                                                                 self.readable_file_type[_file_type])

                    if len(message['error']) > 0:
                        for err_message in message['error']:
                            if 'No such file or directory' not in err_message:
                                err += ' ' + re.sub('not in list', 'unknown item.', err_message)

                    self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Error  - %s\n" % err)

                    is_done = False

                else:

                    is_done, star_data_type, star_data = self.__nefT.read_input_file(srcPath)  # NEFTranslator.validate_file() generates this object internally, but not re-used.

                    if len(self.__star_data_type) > 0:
                        del self.__star_data_type[-1]
                        del self.__star_data[-1]

                    self.__star_data_type.append(star_data_type)
                    self.__star_data.append(star_data)

                    self.__rescueFormerNef(0)
                    self.__rescueImmatureStr(0)

            elif not self.__fixFormatIssueOfInputSource(0, file_name, file_type, srcPath, 'A', message):

                if any(err_message for err_message in message['error'] if 'The mandatory loop' in err_message):

                    _, star_data_type, star_data = self.__nefT.read_input_file(srcPath)  # NEFTranslator.validate_file() generates this object internally, but not re-used.

                    if len(self.__star_data_type) > 0:
                        del self.__star_data_type[-1]
                        del self.__star_data[-1]

                    self.__star_data_type.append(star_data_type)
                    self.__star_data.append(star_data)

                    self.__rescueFormerNef(0)
                    self.__rescueImmatureStr(0)

                is_done = False

            if srcPath_ is not None:
                try:
                    os.remove(srcPath_)
                except:  # noqa: E722 pylint: disable=bare-except
                    pass

        else:

            cs_file_path_list = 'chem_shift_file_path_list'

            for csListId, csPath in enumerate(self.__inputParamDict[cs_file_path_list]):

                codec = detect_bom(csPath, 'utf-8')

                csPath_ = None

                if codec != 'utf-8':
                    csPath_ = csPath + '~'
                    convert_codec(csPath, csPath_, codec, 'utf-8')
                    csPath = csPath_

                is_valid, message = self.__nefT.validate_file(csPath, 'S')  # 'A' for NMR unified data, 'S' for assigned chemical shifts, 'R' for restraints.

                self.__original_error_message.append(message)

                _file_type = message['file_type']  # nef/nmr-star/unknown

                input_source = self.report.input_sources[csListId]
                input_source_dic = input_source.get()

                file_name = input_source_dic['file_name']
                file_type = input_source_dic['file_type']

                if cs_file_path_list in self.__outputParamDict:
                    if csListId < len(self.__outputParamDict[cs_file_path_list]):
                        dstPath = self.__outputParamDict[cs_file_path_list][csListId]
                        if dstPath is not None and dstPath not in self.__inputParamDict[cs_file_path_list]:
                            shutil.copyfile(csPath, dstPath)

                if is_valid:

                    if _file_type != file_type:

                        err = "%r was selected as %s file, but recognized as %s file." % (file_name, self.readable_file_type[file_type], self.readable_file_type[_file_type])

                        if _file_type == 'nef':  # DAOTHER-5673
                            err += " Please re-upload the NEF file as an NMR combined data file."
                        else:
                            err += " Please re-upload the file."

                        if len(message['error']) > 0:
                            for err_message in message['error']:
                                if 'No such file or directory' not in err_message:
                                    err += ' ' + re.sub('not in list', 'unknown item.', err_message)

                        self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Error  - %s\n" % err)

                        is_done = False

                    else:

                        _is_done, star_data_type, star_data = self.__nefT.read_input_file(csPath)  # NEFTranslator.validate_file() generates this object internally, but not re-used.

                        self.__has_legacy_sf_issue = False

                        if star_data_type == 'Saveframe':
                            self.__has_legacy_sf_issue = True
                            self.__fixFormatIssueOfInputSource(csListId, file_name, file_type, csPath, 'S', message)
                            _is_done, star_data_type, star_data = self.__nefT.read_input_file(csPath)  # NEFTranslator.validate_file() generates this object internally, but not re-used.

                        if not (self.__has_legacy_sf_issue and _is_done and star_data_type == 'Entry'):

                            if len(self.__star_data_type) == self.__file_path_list_len:
                                del self.__star_data_type[-1]
                                del self.__star_data[-1]

                            self.__star_data_type.append(star_data_type)
                            self.__star_data.append(star_data)

                            self.__rescueFormerNef(csListId)
                            self.__rescueImmatureStr(csListId)

                elif not self.__fixFormatIssueOfInputSource(csListId, file_name, file_type, csPath, 'S', message):
                    is_done = False

                if csPath_ is not None:
                    try:
                        os.remove(csPath_)
                    except:  # noqa: E722 pylint: disable=bare-except
                        pass

            mr_file_path_list = 'restraint_file_path_list'

            if mr_file_path_list in self.__inputParamDict:

                file_path_list_len = self.__cs_file_path_list_len

                for mrPath in self.__inputParamDict[mr_file_path_list]:

                    codec = detect_bom(mrPath, 'utf-8')

                    mrPath_ = None

                    if codec != 'utf-8':
                        mrPath_ = mrPath + '~'
                        convert_codec(mrPath, mrPath_, codec, 'utf-8')
                        mrPath = mrPath_

                    is_valid, message = self.__nefT.validate_file(mrPath, 'R')  # 'A' for NMR unified data, 'S' for assigned chemical shifts, 'R' for restraints.

                    self.__original_error_message.append(message)

                    _file_type = message['file_type']  # nef/nmr-star/unknown

                    input_source = self.report.input_sources[file_path_list_len]
                    input_source_dic = input_source.get()

                    file_name = input_source_dic['file_name']
                    file_type = input_source_dic['file_type']

                    if mr_file_path_list in self.__outputParamDict:
                        if file_path_list_len - self.__cs_file_path_list_len < len(self.__outputParamDict[mr_file_path_list]):
                            dstPath = self.__outputParamDict[mr_file_path_list][file_path_list_len - self.__cs_file_path_list_len]
                            if dstPath is not None and dstPath not in self.__inputParamDict[mr_file_path_list]:
                                shutil.copyfile(mrPath, dstPath)

                    if is_valid:

                        if _file_type != file_type:

                            err = "%r was selected as %s file, but recognized as %s file." % (file_name, self.readable_file_type[file_type], self.readable_file_type[_file_type])

                            if _file_type == 'nef':  # DAOTHER-5673
                                err += " Please re-upload the NEF file as an NMR combined data file."
                            else:
                                err += " Please re-upload the file."

                            if len(message['error']) > 0:
                                for err_message in message['error']:
                                    if 'No such file or directory' not in err_message:
                                        err += ' ' + re.sub('not in list', 'unknown item.', err_message)

                            self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Error  - %s\n" % err)

                            is_done = False

                        else:

                            _is_done, star_data_type, star_data = self.__nefT.read_input_file(mrPath)  # NEFTranslator.validate_file() generates this object internally, but not re-used.

                            self.__has_legacy_sf_issue = False

                            if star_data_type == 'Saveframe':
                                self.__has_legacy_sf_issue = True
                                self.__fixFormatIssueOfInputSource(file_path_list_len, file_name, file_type, mrPath, 'R', message)
                                _is_done, star_data_type, star_data = self.__nefT.read_input_file(mrPath)  # NEFTranslator.validate_file() generates this object internally, but not re-used.

                            if not (self.__has_legacy_sf_issue and _is_done and star_data_type == 'Entry'):

                                if len(self.__star_data_type) == self.__file_path_list_len:
                                    del self.__star_data_type[-1]
                                    del self.__star_data[-1]

                                self.__star_data_type.append(star_data_type)
                                self.__star_data.append(star_data)

                                self.__rescueFormerNef(file_path_list_len)
                                self.__rescueImmatureStr(file_path_list_len)

                            if not _is_done:
                                is_done = False

                    elif not self.__fixFormatIssueOfInputSource(file_path_list_len, file_name, file_type, mrPath, 'R', message):
                        is_done = False

                    file_path_list_len += 1

                    if mrPath_ is not None:
                        try:
                            os.remove(mrPath_)
                        except:  # noqa: E722 pylint: disable=bare-except
                            pass

            ar_file_path_list = 'atypical_restraint_file_path_list'

            if ar_file_path_list in self.__inputParamDict:

                for ar in self.__inputParamDict[ar_file_path_list]:

                    arPath = ar['file_name']

                    codec = detect_bom(arPath, 'utf-8')

                    arPath_ = None

                    if codec != 'utf-8':
                        arPath_ = arPath + '~'
                        convert_codec(arPath, arPath_, codec, 'utf-8')
                        arPath = arPath_

                    if arPath_ is not None:
                        try:
                            os.remove(arPath_)
                        except:  # noqa: E722 pylint: disable=bare-except
                            pass

        return is_done

    def __fixFormatIssueOfInputSource(self, file_list_id, file_name, file_type, srcPath=None, fileSubType='S', message=None, tmpPaths=None):
        """ Fix format issue of NMR data.
        """

        if not self.__fix_format_issue or srcPath is None or fileSubType not in ('A', 'S', 'R') or message is None:

            if message is not None:

                missing_loop = True

                err = "%r is not compliant with the %s dictionary." % (file_name, self.readable_file_type[file_type])

                if len(message['error']) > 0:

                    if any(err_message for err_message in message['error'] if 'The mandatory loop' in err_message):

                        err = ''
                        for err_message in message['error']:
                            if 'No such file or directory' not in err_message:
                                err += re.sub('not in list', 'unknown item.', err_message) + ' '
                        err = err[:-1]

                    else:
                        missing_loop = False

                        for err_message in self.__original_error_message[file_list_id]['error']:
                            if 'No such file or directory' not in err_message:
                                err += ' ' + re.sub('not in list', 'unknown item.', err_message)

                self.report.error.appendDescription('missing_mandatory_content' if missing_loop else 'format_issue', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__fixFormatIssueOfInputSource() ++ Error  - %s\n" % err)

            if not self.__has_legacy_sf_issue and fileSubType in ('S', 'R'):
                return False

        if self.__has_legacy_sf_issue:
            star_data_type = self.__nefT.read_input_file(srcPath)[1]

        _srcPath = srcPath
        if tmpPaths is None:
            tmpPaths = []

        len_tmp_paths = len(tmpPaths)

        datablock_pattern = re.compile(r'\s*data_(\S+)\s*')
        sf_anonymous_pattern = re.compile(r'\s*save_\S+\s*')
        save_pattern = re.compile(r'\s*save_\s*')
        loop_pattern = re.compile(r'\s*loop_\s*')
        stop_pattern = re.compile(r'\s*stop_\s*')
        category_pattern = re.compile(r'\s*_(\S*)\..*\s*')
        tagvalue_pattern = re.compile(r'\s*_(\S*)\.(\S*)\s+(.*)\s*')
        sf_category_pattern = re.compile(r'\s*_\S*\.Sf_category\s*\S+\s*')
        sf_framecode_pattern = re.compile(r'\s*_\S*\.Sf_framecode\s*\s+\s*')

        msg_template = "Saveframe improperly terminated at end of file."

        if any(msg for msg in message['error'] if msg_template in msg):
            warn = msg_template

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                    for line in ifp:
                        ofp.write(line)

                    ofp.write('save_\n')

                    ofp.close()

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

        msg_template = "Loop improperly terminated at end of file."

        if any(msg for msg in message['error'] if msg_template in msg):
            warn = msg_template

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                    for line in ifp:
                        ofp.write(line)

                    ofp.write('save_\n')

                    ofp.close()

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

#        if __pynmrstar_v3_1__:
#            msg_template = 'Invalid token found in loop contents. Expecting \'loop_\' but found:' # \'*\' Error detected on line *.'
#        else:
        if __pynmrstar_v3_2__:
            msg_template = "Invalid file. NMR-STAR files must start with 'data_' followed by the data name. Did you accidentally select the wrong file?"
        else:
            msg_template = "Invalid file. NMR-STAR files must start with 'data_'. Did you accidentally select the wrong file?"

        if any(msg for msg in message['error'] if msg_template in msg) or (self.__has_legacy_sf_issue and star_data_type == 'Saveframe'):
            warn = 'The datablock must hook saveframe(s).'

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__fixFormatIssueOfInputSource() ++ Warning  - %s\n" % warn)

            with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                lines = ifp.read().splitlines()
                total = len(lines)

                j = total - 1

                while total - j < 10:
                    if save_pattern.match(lines[j]) or stop_pattern.match(lines[j]):
                        break
                    j -= 1

                ifp.close()

            j += 1
            i = 1

            with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                    ofp.write('data_' + os.path.basename(srcPath) + '\n\n')
                    for line in ifp:
                        if i <= j:
                            ofp.write(line)
                        i += 1

                    ofp.close()

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

        msg_template = "Only 'save_NAME' is valid in the body of a NMR-STAR file. Found 'loop_'."

        if any(msg for msg in message['error'] if msg_template in msg):
            warn = 'A saveframe, instead of the datablock, must hook the loop.'

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__fixFormatIssueOfInputSource() ++ Warning  - %s\n" % warn)

            pass_datablock = False

            with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                    for line in ifp:
                        if pass_datablock:
                            ofp.write(line)
                        elif datablock_pattern.match(line):
                            pass_datablock = True
                        else:
                            ofp.write(line)

                    ofp.close()

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

        msg_template = "Cannot use keywords as data values unless quoted or semi-colon delineated. Perhaps this is a loop that wasn't properly terminated? Illegal value:"

        try:

            msg = next(msg for msg in message['error'] if msg_template in msg)
            warn = 'Loops must properly terminated.'

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            _msg_template = r"Cannot use keywords as data values unless quoted or semi-colon delineated. Perhaps this is a loop that wasn't properly terminated\? Illegal value:"

            if __pynmrstar_v3__:
                msg_pattern = re.compile(r'^.*' + _msg_template + r".*on line (\d+).*$")
            else:
                msg_pattern = re.compile(r'^.*' + _msg_template + r".*, (\d+).*$")

            try:

                g = msg_pattern.search(msg).groups()

                line_num = int(g[0])

                i = 1

                with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                    with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                        for line in ifp:
                            if i == line_num:
                                ofp.write('stop_\n')
                            ofp.write(line)
                            i += 1

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

            except AttributeError:
                pass

        except StopIteration:
            pass

        msg_template = "Cannot have a tag value start with an underscore unless the entire value is quoted. You may be missing a data value on the previous line. Illegal value:"

        try:

            msg = next(msg for msg in message['error'] if msg_template in msg)
            warn = "Loops must start with the 'loop_' keyword."

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            if __pynmrstar_v3__:
                msg_pattern = re.compile(r'^.*' + msg_template + r".*on line (\d+).*$")
            else:
                msg_pattern = re.compile(r'^.*' + msg_template + r".*, (\d+).*$")

            try:

                g = msg_pattern.search(msg).groups()

                line_num = int(g[0])

                i = 1

                with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                    with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                        for line in ifp:
                            if i == line_num - 1:
                                ofp.write('loop_\n')
                            ofp.write(line)
                            i += 1

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

            except AttributeError:
                pass

        except StopIteration:
            pass

        msg_template = "Only 'save_NAME' is valid in the body of a NMR-STAR file. Found"

        try:

            is_cs_cif = False

            if self.__op == 'nmr-cs-str-consistency-check':

                cif_stop_pattern = re.compile(r'^#\s*')
                """
                cs_cif_pattern = re.compile(r'D_[0-9]+_cs_P[0-9]+.cif.V[0-9]+$')

                if cs_cif_pattern.match(file_name):
                """
                is_cs_cif = True

                try:

                    with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                        for line in ifp:
                            if save_pattern.match(line) or stop_pattern.match(line):
                                is_cs_cif = False
                                break

                        ifp.close()

                    if is_cs_cif:

                        loop_count = 0
                        has_sf_category = False
                        has_sf_framecode = False

                        with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                            for line in ifp:
                                if loop_pattern.match(line):
                                    loop_count += 1
                                elif sf_category_pattern.match(line):
                                    has_sf_category = True
                                elif sf_framecode_pattern.match(line):
                                    has_sf_framecode = True

                            ifp.close()

                        if not has_sf_category and not has_sf_framecode:

                            in_loop = False

                            with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                                with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                                    for line in ifp:
                                        if datablock_pattern.match(line):
                                            g = datablock_pattern.search(line).groups()
                                            if loop_count < 2:
                                                ofp.write('save_%s\n' % g[0])
                                        elif cif_stop_pattern.match(line):
                                            if in_loop:
                                                if loop_count < 2:
                                                    ofp.write('stop_\nsave_\n')
                                                else:
                                                    ofp.write('stop_\n')
                                            else:
                                                ofp.write(line)
                                            in_loop = False
                                        elif loop_pattern.match(line):
                                            in_loop = True
                                            ofp.write(line)
                                        else:
                                            if in_loop or loop_count < 2:
                                                ofp.write(line)

                                _srcPath = ofp.name
                                tmpPaths.append(_srcPath)
                                ofp.close()

                        else:

                            cif_to_star = CifToNmrStar()
                            cif_to_star.convert(_srcPath, _srcPath + '~')

                            _srcPath += '~'
                            tmpPaths.append(_srcPath)

                        ifp.close()

                except AttributeError:
                    pass

            if not is_cs_cif:

                msg = next(msg for msg in message['error'] if msg_template in msg)
                warn = "Loops must start with the 'loop_' keyword."

                self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

                msg_pattern = re.compile(r'^.*' + msg_template + r" '(.*)'.*$")

                try:

                    g = msg_pattern.search(msg).groups()

                    tag_name = g[0]

                    tag_name_pattern = re.compile(r'\s*' + tag_name + r'\s*')

                    with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                        with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                            for line in ifp:
                                if tag_name_pattern.match(line) is None:
                                    ofp.write(line)
                                else:
                                    ofp.write('loop_\n')

                        _srcPath = ofp.name
                        tmpPaths.append(_srcPath)
                        ofp.close()

                    ifp.close()

                except AttributeError:
                    pass

        except StopIteration:
            pass

        msg_template = "'save_' must be followed by saveframe name. You have a 'save_' tag which is illegal without a specified saveframe name."

        try:

            msg = next(msg for msg in message['error'] if msg_template in msg)
            warn = "The saveframe must have a specified saveframe name."

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            if __pynmrstar_v3__:
                msg_pattern = re.compile(r'^.*' + msg_template + r".*on line (\d+).*$")
            else:
                msg_pattern = re.compile(r'^.*' + msg_template + r".*, (\d+).*$")

            try:

                g = msg_pattern.search(msg).groups()

                line_num = int(g[0])

                i = 1

                with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                    with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                        for line in ifp:
                            if i != line_num:
                                ofp.write(line)
                            else:
                                ofp.write('save_%s\n' % os.path.basename(srcPath))
                            i += 1

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

            except AttributeError:
                pass

        except StopIteration:
            pass

        if __pynmrstar_v3__:
            msg_template = "The tag prefix was never set! Either the saveframe had no tags, you tried to read a version 2.1 file, or there is something else wrong with your file. Saveframe error occurred within:"  # noqa: E501
        else:
            msg_template = "The tag prefix was never set! Either the saveframe had no tags, you tried to read a version 2.1 file without setting ALLOW_V2_ENTRIES to True, or there is something else wrong with your file. Saveframe error occured:"  # noqa: E501

        try:

            msg = next(msg for msg in message['error'] if msg_template in msg)
            warn = 'The saveframe must have NMR-STAR V3.1 tags. Saveframe error occured:%s' % (msg[len(msg_template):].replace('<pynmrstar.', '').replace("'>", "'"))

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            msg_pattern = re.compile(r'^' + msg_template + r" '(.*)'$")

            targets = []

            for msg in message['error']:

                if msg_template not in msg:
                    continue

                try:

                    target = {}

                    g = msg_pattern.search(msg).groups()
                    sf_framecode = str(g[0])

                    target = {'sf_framecode': sf_framecode}

                    pass_sf_framecode = False
                    pass_sf_loop = False

                    sf_framecode_pattern = re.compile(r'\s*save_' + sf_framecode + r'\s*')

                    with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                        for line in ifp:
                            if pass_sf_framecode:
                                if pass_sf_loop:
                                    if category_pattern.match(line):
                                        target['lp_category'] = '_' + category_pattern.search(line).groups()[0]
                                        content_subtype = next((k for k, v in self.lp_categories[file_type].items() if v == target['lp_category']), None)
                                        if content_subtype is not None:
                                            target['sf_category'] = self.sf_categories[file_type][content_subtype]
                                            target['sf_tag_prefix'] = self.sf_tag_prefixes[file_type][content_subtype]
                                        break
                                elif loop_pattern.match(line):
                                    pass_sf_loop = True
                            elif sf_framecode_pattern.match(line):
                                pass_sf_framecode = True

                        ifp.close()

                    targets.append(target)

                except AttributeError:
                    pass

            for target in targets:

                sf_framecode = target['sf_framecode']

                pass_sf_framecode = False
                pass_sf_loop = False

                sf_framecode_pattern = re.compile(r'\s*save_' + sf_framecode + r'\s*')

                with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                    with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                        for line in ifp:
                            if pass_sf_loop:
                                ofp.write(line)
                            elif pass_sf_framecode:
                                if loop_pattern.match(line):
                                    pass_sf_loop = True
                                    if 'sf_category' in target:
                                        ofp.write(target['sf_tag_prefix'] + '.' + ('sf_framecode' if file_type == 'nef' else 'Sf_framecode') + '   ' + sf_framecode + '\n')
                                        ofp.write(target['sf_tag_prefix'] + '.' + ('sf_category' if file_type == 'nef' else 'Sf_category') + '    ' + target['sf_category'] + '\n')
                                        ofp.write('#\n')
                                    ofp.write(line)
                            elif sf_framecode_pattern.match(line):
                                pass_sf_framecode = True
                                ofp.write(line)
                            elif not pass_sf_framecode:
                                ofp.write(line)

                        _srcPath = ofp.name
                        tmpPaths.append(_srcPath)
                        ofp.close()

                    ifp.close()

        except StopIteration:
            pass

        msg_template = "You attempted to parse one loop but the source you provided had more than one loop. Please either parse all loops as a saveframe or only parse one loop. Loops detected:"  # noqa: E501

        try:

            msg = next(msg for msg in message['error'] if msg_template in msg)
            warn = 'Saveframe(s), instead of the datablock, must hook more than one loop. Loops detected:%s' % (msg[len(msg_template):].replace('<pynmrstar.', '').replace("'>", "'"))

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            msg_pattern = re.compile(r'^' + msg_template + r" \[(.*)\]$")
            lp_obj_pattern = re.compile(r"\<pynmrstar\.Loop '(.*)'\>")

            targets = []

            for msg in message['error']:

                if msg_template not in msg:
                    continue

                try:

                    g = msg_pattern.search(msg).groups()

                    for lp_obj in g[0].split(', '):

                        lp_category = str(lp_obj_pattern.search(lp_obj).groups()[0])

                        target = {'lp_category': lp_category}

                        pass_loop = False

                        lp_loc = -1
                        i = 1

                        with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                            for line in ifp:
                                if pass_loop:
                                    if category_pattern.match(line):
                                        _lp_category = '_' + category_pattern.search(line).groups()[0]
                                        if lp_category == _lp_category:
                                            target['loop_location'] = lp_loc
                                            content_subtype = next((k for k, v in self.lp_categories[file_type].items() if v == target['lp_category']), None)
                                            if not content_subtype is None:
                                                target['sf_category'] = self.sf_categories[file_type][content_subtype]
                                                target['sf_tag_prefix'] = self.sf_tag_prefixes[file_type][content_subtype]
                                                target['sf_framecode'] = target['sf_category'] + '_1'
                                        pass_loop = False
                                elif loop_pattern.match(line):
                                    pass_loop = True
                                    lp_loc = i
                                elif stop_pattern.match(line):
                                    if 'loop_location' in target and 'stop_location' not in target:
                                        target['stop_location'] = i
                                        break

                                i += 1

                        ifp.close()

                        targets.append(target)

                except AttributeError:
                    pass

            target_loop_locations = [target['loop_location'] for target in targets]
            target_stop_locations = [target['stop_location'] for target in targets]
            ignored_loop_locations = []
            for target in targets:
                if 'sf_category' not in target:
                    ignored_loop_locations.extend(list(range(target['loop_location'], target['stop_location'] + 1)))

            i = 1

            with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                    ofp.write('data_' + os.path.basename(srcPath) + '\n\n')
                    for line in ifp:
                        if i in target_loop_locations:
                            target = next(target for target in targets if target['loop_location'] == i)
                            if 'sf_category' in target:
                                ofp.write('save_' + target['sf_framecode'] + '\n')
                                ofp.write(target['sf_tag_prefix'] + '.' + ('sf_framecode' if file_type == 'nef' else 'Sf_framecode') + '   ' + target['sf_framecode'] + '\n')
                                ofp.write(target['sf_tag_prefix'] + '.' + ('sf_category' if file_type == 'nef' else 'Sf_category') + '    ' + target['sf_category'] + '\n')
                                ofp.write('#\n')
                        if i not in ignored_loop_locations:
                            ofp.write(line)
                        if i in target_stop_locations:
                            target = next(target for target in targets if target['stop_location'] == i)
                            if 'sf_category' in target:
                                ofp.write('save_\n')

                        i += 1

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

        except StopIteration:
            pass

        msg_template = "One saveframe cannot have tags with different categories (or tags that don't match the set category)!"

        try:

            msg = next(msg for msg in message['error'] if msg_template in msg)
            warn = msg

            _msg_template = r"One saveframe cannot have tags with different categories \(or tags that don't match the set category\)!"

            msg_pattern = re.compile(r'^' + _msg_template + r" '(.*)' vs '(.*)'.$")

            targets = []

            for msg in message['error']:

                if msg_template not in msg:
                    continue

                try:

                    target = {}

                    g = msg_pattern.search(msg).groups()

                    try:
                        category_1 = str(g[0])
                        category_2 = str(g[1])
                    except IndexError:
                        continue

                    target = {'category_1': category_1, 'category_2': category_2}

                    pass_sf_framecode = False
                    pass_category_1 = False
                    pass_category_2 = False
                    pass_sf_loop = False

                    i = 1

                    with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                        for line in ifp:
                            if pass_sf_framecode:
                                if save_pattern.match(line):
                                    if 'category_1_begin' in target and 'category_2_begin' in target:
                                        targets.append(target)
                                        break
                                    pass_sf_framecode = False
                                    pass_category_1 = False
                                    pass_category_2 = False
                                    pass_sf_loop = False
                                elif loop_pattern.match(line):
                                    pass_sf_loop = True
                                elif not pass_sf_loop:
                                    if category_pattern.match(line):
                                        category = '_' + category_pattern.search(line).groups()[0]
                                        if category == category_1:
                                            if not pass_category_1:
                                                target['category_1_begin'] = i
                                                content_subtype = next((k for k, v in self.sf_tag_prefixes[file_type].items() if v == category), None)
                                                if content_subtype is not None:
                                                    target['content_subtype_1'] = content_subtype
                                                content_subtype = next((k for k, v in self.lp_categories[file_type].items() if v == category), None)
                                                if content_subtype is not None:
                                                    target['content_subtype_1'] = content_subtype
                                            pass_category_1 = True
                                            target['category_1_end'] = i
                                        elif category == category_2 and pass_category_1:
                                            if not pass_category_2:
                                                target['category_2_begin'] = i
                                                content_subtype = next((k for k, v in self.sf_tag_prefixes[file_type].items() if v == category), None)
                                                if content_subtype is not None:
                                                    target['category_type_2'] = 'saveframe'
                                                    target['content_subtype_2'] = content_subtype
                                                    target['sf_tag_prefix_2'] = self.sf_tag_prefixes[file_type][content_subtype]
                                                    target['sf_category_2'] = self.sf_categories[file_type][content_subtype]
                                                    target['sf_framecode_2'] = target['sf_category_2'] + '_1'
                                                content_subtype = next((k for k, v in self.lp_categories[file_type].items() if v == category), None)
                                                if content_subtype is not None:
                                                    target['category_type_2'] = 'loop'
                                                    target['content_subtype_2'] = content_subtype
                                                    target['sf_tag_prefix_2'] = self.sf_tag_prefixes[file_type][content_subtype]
                                                    target['sf_category_2'] = self.sf_categories[file_type][content_subtype]
                                                    target['sf_framecode_2'] = target['sf_category_2'] + '_1'
                                                if 'category_type_2' not in target:
                                                    content_subtype = target['content_subtype_1']
                                                    target['category_type_2'] = 'loop'
                                                    target['content_subtype_2'] = content_subtype
                                            pass_category_2 = True
                                            target['category_2_end'] = i
                                elif loop_pattern.match(line):
                                    pass_sf_loop = True
                            elif sf_anonymous_pattern.match(line):
                                pass_sf_framecode = True
                                pass_category_1 = False
                                pass_category_2 = False
                                pass_sf_loop = False

                            i += 1

                        ifp.close()

                except AttributeError:
                    pass

            if len(targets) > 0:

                self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

                target_category_begins = [target['category_2_begin'] for target in targets]
                target_category_ends = [target['category_2_end'] for target in targets]

                loop_category_locations = []
                for target in targets:
                    _range = list(range(target['category_2_begin'], target['category_2_end'] + 1))
                    if target['category_type_2'] == 'loop':
                        loop_category_locations.extend(_range)

                i = 1

                with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                    with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                        for line in ifp:
                            if i in target_category_begins:
                                target = next(target for target in targets if target['category_2_begin'] == i)
                                if target['content_subtype_1'] != target['content_subtype_2']:
                                    ofp.write('save_\n')
                                    if target['category_type_2'] == 'saveframe':
                                        ofp.write('save_' + target['sf_framecode_2'] + '\n')
                                    else:
                                        ofp.write('save_' + target['sf_framecode_2'] + '\n')
                                        ofp.write(target['sf_tag_prefix_2'] + '.' + ('sf_framecode' if file_type == 'nef' else 'Sf_framecode') + '   ' + target['sf_framecode_2'] + '\n')
                                        ofp.write(target['sf_tag_prefix_2'] + '.' + ('sf_category' if file_type == 'nef' else 'Sf_category') + '    ' + target['category_2'] + '\n')
                                        ofp.write('loop_\n')
                                        lp_tags = ''
                                        lp_values = ''
                                elif target['category_type_2'] == 'loop':
                                    ofp.write('loop_\n')
                                    lp_tags = ''
                                    lp_values = ''
                            if i not in loop_category_locations:
                                ofp.write(line)
                            else:
                                g = tagvalue_pattern.search(line).groups()
                                try:
                                    lp_tags += '_%s.%s\n' % (g[0], g[1])
                                    lp_values += ' ' + g[2].strip(' ') + ' '
                                except IndexError:
                                    continue
                            if i in target_category_ends:
                                target = next(target for target in targets if target['category_2_end'] == i)
                                if target['content_subtype_1'] != target['content_subtype_2']:
                                    if target['category_type_2'] == 'saveframe':
                                        pass
                                    else:
                                        ofp.write(lp_tags)
                                        ofp.write(lp_values.rstrip(' ') + '\n')
                                        ofp.write('stop_\n')
                                elif target['category_type_2'] == 'loop':
                                    ofp.write(lp_tags)
                                    ofp.write(lp_values.rstrip(' ') + '\n')
                                    ofp.write('stop_\n')

                            i += 1

                        _srcPath = ofp.name
                        tmpPaths.append(_srcPath)
                        ofp.close()

                    ifp.close()

        except StopIteration:
            pass

        msg_template = 'The Sf_framecode tag cannot be different from the saveframe name.'

        try:

            msg = next(msg for msg in message['error'] if msg_template in msg)
            warn = "Sf_framecode tag value should match with the saveframe name."

            self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateInputSource() ++ Warning  - %s\n" % warn)

            if __pynmrstar_v3_3__:
                msg_pattern = re.compile(r'^.*' + msg_template + r" Error occurred in tag _\S+ with value (\S+) which conflicts with the saveframe name (\S+)\. Error detected on line (\d+).*$")
            else:
                msg_pattern = re.compile(r'^.*' + msg_template + r" Error occurred in tag _\S+ with value (\S+) which conflicts with.* the saveframe name (\S+)\. Error detected on line (\d+).*$")

            try:

                g = msg_pattern.search(msg).groups()

                sf_framecode = g[0]
                saveframe_name = g[1]
                line_num = int(g[2])

                i = 1

                with open(_srcPath, 'r', encoding='UTF-8') as ifp:
                    with open(_srcPath + '~', 'w', encoding='UTF-8') as ofp:
                        for line in ifp:
                            if i == line_num:
                                ofp.write(re.sub(sf_framecode + r'\s$', saveframe_name + r'\n', line))
                            else:
                                ofp.write(line)
                            i += 1

                    _srcPath = ofp.name
                    tmpPaths.append(_srcPath)
                    ofp.close()

                ifp.close()

            except AttributeError:
                pass

        except StopIteration:
            pass

        if len(tmpPaths) > len_tmp_paths:

            is_valid, _message = self.__nefT.validate_file(_srcPath, fileSubType)

            if not is_valid:

                retry = len(message['error']) != len(_message['error'])

                if not retry:

                    for msg, _msg in zip(message['error'], _message['error']):
                        if msg != _msg:
                            retry = True
                            break

                if retry and len_tmp_paths < 10:
                    return self.__fixFormatIssueOfInputSource(file_list_id, file_name, file_type, _srcPath, fileSubType, _message, tmpPaths)

        is_done = True

        is_valid, message = self.__nefT.validate_file(_srcPath, fileSubType)

        _file_type = message['file_type'] # nef/nmr-star/unknown

        if not self.__combined_mode:

            if file_list_id < self.__cs_file_path_list_len:

                cs_file_path_list = 'chem_shift_file_path_list'

                if cs_file_path_list in self.__outputParamDict:
                    if file_list_id < len(self.__outputParamDict[cs_file_path_list]):
                        dstPath = self.__outputParamDict[cs_file_path_list][file_list_id]
                        if dstPath is not None and dstPath not in self.__inputParamDict[cs_file_path_list]:
                            shutil.copyfile(_srcPath, dstPath)

            else:

                mr_file_path_list = 'restraint_file_path_list'

                if mr_file_path_list in self.__outputParamDict:
                    if file_list_id - self.__cs_file_path_list_len < len(self.__outputParamDict[mr_file_path_list]):
                        dstPath = self.__outputParamDict[mr_file_path_list][file_list_id - self.__cs_file_path_list_len]
                        if dstPath is not None and dstPath not in self.__inputParamDict[mr_file_path_list]:
                            shutil.copyfile(_srcPath, dstPath)

        if is_valid:

            if _file_type != file_type:

                err = "%r was selected as %s file, but recognized as %s file. Please re-upload the file." % (file_name, self.readable_file_type[file_type], self.readable_file_type[_file_type])

                if len(message['error']) > 0:
                    for err_message in message['error']:
                        if 'No such file or directory' not in err_message:
                            err += ' ' + re.sub('not in list', 'unknown item.', err_message)

                self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__fixFormatIssueOfInputSource() ++ Error  - %s\n" % err)

            else:

                is_done, star_data_type, star_data = self.__nefT.read_input_file(_srcPath) # NEFTranslator.validate_file() generates this object internally, but not re-used.

                rescued = self.__has_legacy_sf_issue and is_done and star_data_type == 'Entry'

                if len(self.__star_data_type) > file_list_id:
                    del self.__star_data_type[-1]
                    del self.__star_data[-1]

                self.__star_data_type.append(star_data_type)
                self.__star_data.append(star_data)

                self.__rescueFormerNef(file_list_id)
                self.__rescueImmatureStr(file_list_id)

                if rescued:
                    onedep_file_pattern = re.compile(r'(.*)\-upload_(.*)\.V(.*)$')
                    if onedep_file_pattern.match(srcPath):
                        g = onedep_file_pattern.search(srcPath).groups()
                        srcPath = g[0] + '-upload-convert_' + g[1] + '.V' + g[2]
                    else:
                        onedep_file_pattern = re.compile(r'(.*)\.V(.*)$')
                        if onedep_file_pattern.match(srcPath):
                            g = onedep_file_pattern.search(srcPath).groups()
                            srcPath = g[0] + '.V' + str(int(g[1]) + 1)
                    if __pynmrstar_v3__:
                        self.__star_data[file_list_id].write_to_file(srcPath, skip_empty_loops=True, skip_empty_tags=False)
                    else:
                        self.__star_data[file_list_id].write_to_file(srcPath)

        else:

            missing_loop = True

            err = "%r is not compliant with the %s dictionary." % (file_name, self.readable_file_type[file_type])

            if len(message['error']) > 0:

                if any(err_message for err_message in message['error'] if 'The mandatory loop' in err_message):

                    err = ''
                    for err_message in message['error']:
                        if 'No such file or directory' not in err_message:
                            err += re.sub('not in list', 'unknown item.', err_message) + ' '
                    err = err[:-1]

                else:
                    missing_loop = False

                    for err_message in self.__original_error_message[file_list_id]['error']:
                        if 'No such file or directory' not in err_message:
                            err += ' ' + re.sub('not in list', 'unknown item.', err_message)

            self.report.error.appendDescription('missing_mandatory_content' if missing_loop else 'format_issue', {'file_name': file_name, 'description': err})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__fixFormatIssueOfInputSource() ++ Error  - %s\n" % err)

            is_done = False

        try:

            if self.__release_mode:
                self.__tmpPath = tmpPaths[-1]
                self.__srcPath = self.__tmpPath
                for tmpPath in tmpPaths[:-1]:
                    if os.path.exists(tmpPath):
                        os.remove(tmpPath)
            else:
                for tmpPath in tmpPaths:
                    if os.path.exists(tmpPath):
                        os.remove(tmpPath)

        except:  # noqa: E722 pylint: disable=bare-except
            pass

        return is_done

    def __rescueFormerNef(self, file_list_id):
        """ Rescue former NEF version prior to 1.0.
        """

        input_source = self.report.input_sources[file_list_id]
        input_source_dic = input_source.get()

        file_name = input_source_dic['file_name']
        file_type = input_source_dic['file_type']

        if file_type != 'nef' or file_list_id >= len(self.__star_data) or self.__star_data[file_list_id] is None:
            return False

        if self.__combined_mode or self.__star_data_type[file_list_id] == 'Entry':

            for content_subtype in self.nmr_content_subtypes:

                sf_category = self.sf_categories[file_type][content_subtype]

                if sf_category is None:
                    continue

                for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if self.__getSaveframeByName(file_list_id, sf_framecode) is None:

                        itName = '_' + sf_category + '.sf_framecode'

                        if self.__resolve_conflict:
                            warn = "%s %r should be matched with saveframe name %r. %s will be overwritten." % (itName, sf_framecode, sf_data.name, itName)

                            self.report.warning.appendDescription('missing_saveframe', {'file_name': file_name, 'sf_framecode': sf_data.name, 'description': warn})
                            self.report.setWarning()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__rescueFormerNef() ++ Warning  - %s\n" % warn)

                            tagNames = [t[0] for t in sf_data.tags]

                            sf_framecode= sf_data.name

                            sf_data.tags[tagNames.index('sf_framecode')][1] = sf_framecode

                        else:
                            err = "%s %r must be matched with saveframe name %r." % (itName, sf_framecode, sf_data.name)

                            self.report.error.appendDescription('format_issue', {'file_name': file_name, 'sf_framecode': sf_data.name, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__rescueFormerNef() ++ Error  - %s\n" % err)

        if not self.__rescue_mode:
            return True

        if self.__combined_mode or self.__star_data_type[file_list_id] == 'Entry':

            content_subtype = 'entry_info'

            sf_category = self.sf_categories[file_type][content_subtype]

            for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                format_version = sf_data.get_tag('format_version')[0]

                if not format_version.startswith('0.'):
                    sf_data.format_version = '1.1'

        else:

            self.__sf_category_list, self.__lp_category_list = self.__nefT.get_data_content(self.__star_data[file_list_id], self.__star_data_type[file_list_id])

            # initialize loop counter
            lp_counts = {t: 0 for t in self.nmr_content_subtypes}

            # increment loop counter of each content subtype
            for lp_category in self.__lp_category_list:
                if lp_category in self.lp_categories[file_type].values():
                    lp_counts[[k for k, v in self.lp_categories[file_type].items() if v == lp_category][0]] += 1

            content_subtypes = {k: lp_counts[k] for k in lp_counts if lp_counts[k] > 0}

        for content_subtype in self.nmr_content_subtypes:

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            if sf_category is None or lp_category is None:
                continue

            if self.__star_data_type[file_list_id] == 'Loop':

                if content_subtype not in content_subtypes:
                    continue

                sf_data = self.__star_data[file_list_id]
                sf_framecode = ''

                self.__rescueFormerNef__(file_name, file_type, content_subtype, sf_data, sf_framecode, sf_category, lp_category)

            elif self.__star_data_type[file_list_id] == 'Saveframe':

                if content_subtype not in content_subtypes:
                    continue

                sf_data = self.__star_data[file_list_id]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__rescueFormerNef__(file_name, file_type, content_subtype, sf_data, sf_framecode, sf_category, lp_category)

            else:

                for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__rescueFormerNef__(file_name, file_type, content_subtype, sf_data, sf_framecode, sf_category, lp_category)

        return True

    def __rescueFormerNef__(self, file_name, file_type, content_subtype, sf_data, sf_framecode, sf_category, lp_category):
        """ Rescue former NEF version prior to 1.0.
        """

        try:
            if __pynmrstar_v3_2__:
                loop = sf_data.get_loop(lp_category)
            else:
                loop = sf_data.get_loop_by_category(lp_category)
        except:  # noqa: E722 pylint: disable=bare-except
            loop = sf_data

        try:

            index_tag = self.index_tags[file_type][content_subtype]

            if index_tag is not None:

                try:
                    tag_pos = next(loop.tags.index(tag) for tag in loop.tags if tag == 'ordinal')
                    loop.tags[tag_pos] = 'index'
                except StopIteration:
                    pass

                try:
                    tag_pos = next(loop.tags.index(tag) for tag in loop.tags if tag == 'index_id')
                    loop.tags[tag_pos] = 'index'
                except StopIteration:
                    pass

            if content_subtype == 'poly_seq':

                try:
                    tag_pos = next(loop.tags.index(tag) for tag in loop.tags if tag == 'residue_type')
                    loop.tags[tag_pos] = 'residue_name'
                except StopIteration:
                    pass

                if 'index' not in loop.tags:

                    lp_tag = lp_category + '.index'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueFormerNef() ++ LookupError  - %s" % err)

                    for l, i in enumerate(loop, start=1):
                        i.append(l)

                    loop.add_tag(lp_category + '.index')

            elif content_subtype == 'chem_shift':

                if any(tag for tag in sf_data.tags if tag[0] == 'atom_chemical_shift_units'):
                    if __pynmrstar_v3_2__:
                        sf_data.remove_tag('atom_chemical_shift_units')
                    else:
                        sf_data.delete_tag('atom_chemical_shift_units')

                try:
                    tag_pos = next(loop.tags.index(tag) for tag in loop.tags if tag == 'residue_type')
                    loop.tags[tag_pos] = 'residue_name'
                except StopIteration:
                    pass

                if 'element' not in loop.tags:

                    lp_tag = lp_category + '.element'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueFormerNef() ++ LookupError  - %s" % err)

                    try:
                        atom_name_col = loop.tags.index('atom_name')

                        for row in loop:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row.append(atom_type)

                        loop.add_tag(lp_category + '.element')

                    except ValueError:
                        pass

                elif not self.__combined_mode:

                    atom_type_col = loop.tags.index('element')
                    atom_name_col = loop.tags.index('atom_name')

                    for row in loop:
                        if row[atom_type_col] in self.empty_value:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row[atom_type_col] = atom_type

                if 'isotope_number' not in loop.tags:

                    lp_tag = lp_category + '.isotope_number'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueFormerNef() ++ LookupError  - %s" % err)

                    try:
                        atom_name_col = loop.tags.index('atom_name')

                        for row in loop:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row.append(str(self.atom_isotopes[atom_type][0]))

                        loop.add_tag(lp_category + '.isotope_number')

                    except ValueError:
                        pass

                elif not self.__combined_mode:

                    iso_num_col = loop.tags.index('isotope_number')
                    atom_name_col = loop.tags.index('atom_name')

                    for row in loop:
                        if row[iso_num_col] in self.empty_value:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row[iso_num_col] = str(self.atom_isotopes[atom_type][0])

            elif content_subtype == 'dihed_restraint':

                if 'name' not in loop.tags:

                    lp_tag = lp_category + '.name'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueFormerNef() ++ LookupError  - %s" % err)

                    try:

                        for row in loop:
                            row.append('.')

                        loop.add_tag(lp_category + '.name')

                    except ValueError:
                        pass

            elif content_subtype == 'rdc_restraint':

                try:
                    tag = next(tag for tag in sf_data.tags if tag[0] == 'tensor_residue_type')
                    sf_data.add_tag(sf_category + '.tensor_residue_name', tag[1])
                    if __pynmrstar_v3_2__:
                        sf_data.remove_tag('tensor_residue_type')
                    else:
                        sf_data.delete_tag('tensor_residue_type')
                except StopIteration:
                    pass

            if content_subtype in ('dist_restraint', 'rdc_restraint'):
                max_dim = 3

            elif content_subtype == 'dihed_restraint':
                max_dim = 5

            elif content_subtype == 'spectral_peak':

                try:

                    _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                    num_dim = int(_num_dim)

                    if num_dim not in range(1, self.lim_num_dim):
                        raise ValueError()

                except ValueError: # raised error already at __testIndexConsistency()
                    return

                max_dim = num_dim + 1

            else:
                return

            for j in range(1, max_dim):

                _residue_type = 'residue_type_' + str(j)

                try:
                    tag_pos = next(loop.tags.index(tag) for tag in loop.tags if tag == _residue_type)
                    loop.tags[tag_pos] = 'residue_name_' + str(j)
                except StopIteration:
                    pass

        except KeyError:
            pass

    def __rescueImmatureStr(self, file_list_id):
        """ Rescue immature NMR-STAR.
        """

        input_source = self.report.input_sources[file_list_id]
        input_source_dic = input_source.get()

        file_name = input_source_dic['file_name']
        file_type = input_source_dic['file_type']

        if file_type != 'nmr-star' or file_list_id >= len(self.__star_data) or self.__star_data[file_list_id] is None:
            return False

        if self.__combined_mode or self.__star_data_type[file_list_id] == 'Entry':

            for content_subtype in self.nmr_content_subtypes:

                sf_category = self.sf_categories[file_type][content_subtype]

                if sf_category is None:
                    continue

                for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if self.__getSaveframeByName(file_list_id, sf_framecode) is None:

                        itName = '_' + sf_category + '.Sf_framecode'

                        if self.__resolve_conflict:
                            warn = "%s %r should be matched with saveframe name %r. %s will be overwritten." % (itName, sf_framecode, sf_data.name, itName)

                            self.report.warning.appendDescription('missing_saveframe', {'file_name': file_name, 'sf_framecode': sf_data.name, 'description': warn})
                            self.report.setWarning()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ Warning  - %s\n" % warn)

                            tagNames = [t[0] for t in sf_data.tags]

                            sf_framecode= sf_data.name

                            sf_data.tags[tagNames.index('Sf_framecode')][1] = sf_data.name

                        else:
                            err = "%s %r must be matched with saveframe name %r." % (itName, sf_framecode, sf_data.name)

                            self.report.error.appendDescription('format_issue', {'file_name': file_name, 'sf_framecode': sf_data.name, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ Error  - %s\n" % err)

        if not self.__rescue_mode:
            return True

        self.__sf_category_list, self.__lp_category_list = self.__nefT.get_data_content(self.__star_data[file_list_id], self.__star_data_type[file_list_id])

        # initialize loop counter
        lp_counts = {t: 0 for t in self.nmr_content_subtypes}

        # increment loop counter of each content subtype
        for lp_category in self.__lp_category_list:
            if lp_category in self.lp_categories[file_type].values():
                lp_counts[[k for k, v in self.lp_categories[file_type].items() if v == lp_category][0]] += 1

        content_subtypes = {k: lp_counts[k] for k in lp_counts if lp_counts[k] > 0}

        for content_subtype in self.nmr_content_subtypes:

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            if content_subtype.startswith('spectral_peak'):
                lp_category = self.aux_lp_categories[file_type][content_subtype][0] # _Spectral_dim

            if sf_category is None or lp_category is None:
                continue

            if self.__star_data_type[file_list_id] == 'Loop':

                if content_subtype not in content_subtypes:
                    continue

                sf_data = self.__star_data[file_list_id]
                sf_framecode = ''

                self.__rescueImmatureStr__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

            elif self.__star_data_type[file_list_id] == 'Saveframe':

                if content_subtype not in content_subtypes:
                    continue

                sf_data = self.__star_data[file_list_id]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__rescueImmatureStr__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

            else:

                for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__rescueImmatureStr__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

        return True

    def __rescueImmatureStr__(self, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category):
        """ Rescue immature NMR-STAR.
        """

        try:
            if __pynmrstar_v3_2__:
                loop = sf_data.get_loop(lp_category)
            else:
                loop = sf_data.get_loop_by_category(lp_category)
        except:  # noqa: E722 pylint: disable=bare-except
            loop = sf_data

        try:

            if content_subtype == 'chem_shift':

                if 'Atom_type' not in loop.tags:

                    lp_tag = lp_category + '.Atom_type'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ LookupError  - %s" % err)

                    try:
                        atom_name_col = loop.tags.index('Atom_ID')

                        for row in loop:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row.append(atom_type)

                        loop.add_tag(lp_category + '.Atom_type')

                    except ValueError:
                        pass

                elif not self.__combined_mode:

                    atom_type_col = loop.tags.index('Atom_type')
                    atom_name_col = loop.tags.index('Atom_ID')

                    for row in loop:
                        if row[atom_type_col] in self.empty_value:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row[atom_type_col] = atom_type

                if 'Atom_isotope_number' not in loop.tags:

                    lp_tag = lp_category + '.Atom_isotope_number'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ LookupError  - %s" % err)

                    try:
                        atom_name_col = loop.tags.index('Atom_ID')

                        for row in loop:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row.append(str(self.atom_isotopes[atom_type][0]))

                        loop.add_tag(lp_category + '.Atom_isotope_number')

                    except ValueError:
                        pass

                elif not self.__combined_mode:

                    iso_num_col = loop.tags.index('Atom_isotope_number')
                    atom_name_col = loop.tags.index('Atom_ID')

                    for row in loop:
                        if row[iso_num_col] in self.empty_value:
                            atom_type = row[atom_name_col][0]
                            if atom_type in ('Q', 'M'):
                                atom_type = 'H'
                            row[iso_num_col] = str(self.atom_isotopes[atom_type][0])

            elif content_subtype == 'dihed_restraint':

                if 'Torsion_angle_name' not in loop.tags:

                    lp_tag = lp_category + '.Torsion_angle_name'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ LookupError  - %s" % err)

                    try:

                        for row in loop:
                            row.append('.')

                        loop.add_tag(lp_category + '.Torsion_angle_name')

                    except ValueError:
                        pass

            elif content_subtype.startswith('spectral_peak'):

                if 'Atom_type' not in loop.tags:

                    lp_tag = lp_category + '.Atom_type'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ LookupError  - %s" % err)

                    try:
                        axis_code_name_col = loop.tags.index('Axis_code')

                        for row in loop:
                            atom_type = re.sub(r'[0-9]+', '', row[axis_code_name_col])
                            row.append(atom_type)

                        loop.add_tag(lp_category + '.Atom_type')

                    except ValueError:
                        pass

                if 'Atom_isotope_number' not in loop.tags:

                    lp_tag = lp_category + '.Atom_isotope_number'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ LookupError  - %s" % err)

                    try:
                        axis_code_name_col = loop.tags.index('Axis_code')

                        for row in loop:
                            atom_type = re.sub(r'[0-9]+', '', row[axis_code_name_col])
                            row.append(str(self.atom_isotopes[atom_type][0]))

                        loop.add_tag(lp_category + '.Atom_isotope_number')

                    except ValueError:
                        pass

                if 'Axis_code' not in loop.tags:

                    lp_tag = lp_category + '.Axis_code'
                    err = self.__err_template_for_missing_mandatory_lp_tag % (lp_tag, file_type.upper())

                    if self.__check_mandatory_tag and self.__nefT.is_mandatory_tag(lp_tag, file_type):
                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__rescueImmatureStr() ++ LookupError  - %s" % err)

                    try:
                        atom_type_name_col = loop.tags.index('Atom_type')
                        iso_num_name_col = loop.tags.index('Atom_isotope_number')

                        for row in loop:
                            atom_type = row[atom_type_name_col]
                            iso_num = row[iso_num_name_col]
                            row.append(iso_num + atom_type)

                        loop.add_tag(lp_category + '.Axis_code')

                    except ValueError:
                        pass

        except KeyError:
            pass

    def __detectContentSubType(self):
        """ Detect content subtype of NMR data file in any STAR format.
        """

        #if self.report.isError():
        #    return False

        if len(self.__star_data) != self.__file_path_list_len:
            return False

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']
            content_type = input_source_dic['content_type']

            self.__sf_category_list, self.__lp_category_list = self.__nefT.get_data_content(self.__star_data[fileListId], self.__star_data_type[fileListId])

            is_valid, messages, corrections = self.__nefT.resolve_sf_names_for_cif(self.__star_data[fileListId], self.__star_data_type[fileListId]) # DAOTHER-7389, issue #4
            self.__sf_name_corr.append(corrections)

            if not is_valid:

                for warn in messages:
                    self.report.warning.appendDescription('corrected_saveframe_name', {'file_name': file_name, 'description': warn})
                    self.report.setWarning()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Warning  - %s\n" % warn)

            tags_with_null_str = []

            for sf_category in self.__sf_category_list: # DAOTHER-5896

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    for tag in sf_data.tags:
                        if isinstance(tag[1], str) and len(tag[1]) == 0:
                            tags_with_null_str.append('_' + sf_category + '.' + tag[0])
                            tag[1] = '.'

            if len(tags_with_null_str) > 0:

                warn = "Empty strings for %s are not allowed as values. Use a '.' or a '?' if needed." % tags_with_null_str

                self.report.warning.appendDescription('corrected_format_issue', {'file_name': file_name, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Warning  - %s\n" % warn)

            for sf_category in self.__sf_category_list:

                if file_type == 'nmr-star' and sf_category == 'entity':
                    self.__has_star_entity = True

                if sf_category is not None and sf_category not in self.sf_categories[file_type].values():

                    if not self.__bmrb_only:

                        if file_type == 'nef':
                            warn = "Ignored third party software's saveframe %r." % sf_category
                        else:
                            warn = "Ignored saveframe category %r." % sf_category

                        self.report.warning.appendDescription('skipped_saveframe_category', {'file_name': file_name, 'sf_category': sf_category, 'description': warn})
                        self.report.setWarning()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Warning  - %s\n" % warn)

            # initialize loop counter
            lp_counts = {t: 0 for t in self.nmr_content_subtypes}

            # increment loop counter of each content subtype
            for lp_category in self.__lp_category_list:

                if lp_category in self.lp_categories[file_type].values():
                    lp_counts[[k for k, v in self.lp_categories[file_type].items() if v == lp_category][0]] += 1

            content_subtype = 'poly_seq'

            lp_category = self.lp_categories[file_type][content_subtype]

            if lp_counts[content_subtype] == 0:

                if not self.__has_star_entity and self.__combined_mode:

                    if self.__resolve_conflict and self.__update_poly_seq: # DAOTHER-6694
                        warn = "A saveframe with a category %r is missing in the NMR data." % lp_category

                        self.report.warning.appendDescription('missing_saveframe', {'file_name': file_name, 'description': warn})
                        self.report.setWarning()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Warning  - %s\n" % warn)

                    else:
                        err = "A saveframe with a category %r is missing. Please re-upload the %s file." % (lp_category, file_type.upper())

                        self.report.error.appendDescription('missing_mandatory_content', {'file_name': file_name, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

                elif lp_counts['chem_shift'] == 0 and lp_counts['dist_restraint'] > 0 and content_type != 'nmr-restraints':
                    err = "A saveframe with a category %r is missing. Please re-upload the %s file." % (lp_category, file_type.upper())

                    self.report.error.appendDescription('missing_mandatory_content', {'file_name': file_name, 'description': err})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

            elif lp_counts[content_subtype] > 1:

                err = "Unexpectedly, multiple saveframes having %r category exist." % lp_category

                self.report.error.appendDescription('format_issue', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

            content_subtype = 'chem_shift'

            if lp_counts[content_subtype] == 0 and self.__combined_mode:

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                err = "The mandatory saveframe with a category %r is missing, Deposition of assigned chemical shifts is mandatory. Please re-upload the %s file." % (sf_category, file_type.upper())

                self.report.error.appendDescription('missing_mandatory_content', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

            if lp_counts[content_subtype] > 0 and content_type == 'nmr-restraints' and not self.__bmrb_only:

                err = "NMR restraint file includes assigned chemical shifts. Please re-upload the %s file as an NMR combined data file." % file_type.upper()

                self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

            content_subtype = 'dist_restraint'

            if lp_counts[content_subtype] == 0 and self.__combined_mode:

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                err = "The mandatory saveframe with a category %r is missing, Deposition of distance restraints is mandatory. Please re-upload the %s file." % (sf_category, file_type.upper())

                self.report.error.appendDescription('missing_mandatory_content', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

            if (lp_counts['dist_restraint'] > 0 or lp_counts['dihed_restraint'] or lp_counts['rdc_restraint']) and content_type == 'nmr-chemical-shifts' and not self.__bmrb_only:

                err = "The assigned chemical shift file includes NMR restraints. Please re-upload the %s file as an NMR combined data file." % file_type.upper()

                self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

            has_spectral_peak = lp_counts['spectral_peak'] + lp_counts['spectral_peak_alt'] > 0

            if not has_spectral_peak and self.__combined_mode:

                warn = "The wwPDB NMR Validation Task Force strongly encourages the submission of spectral peak lists, in particular those generated from NOESY spectra."

                self.report.warning.appendDescription('encouragement', {'file_name': file_name, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Warning  - %s\n" % warn)

            if has_spectral_peak and content_type == 'nmr-chemical-shifts' and not self.__bmrb_only:

                err = "The assigned chemical shift file includes spectral peak lists. Please re-upload the %s file as an NMR combined data file." % file_type.upper()

                self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubType() ++ Error  - %s\n" % err)

            content_subtypes = {k: lp_counts[k] for k in lp_counts if lp_counts[k] > 0}

            input_source.setItemValue('content_subtype', content_subtypes)

        return not self.report.isError()

    def __detectContentSubTypeOfLegacyMR(self):
        """ Detect content subtype of legacy NMR restraint files.
        """

        if self.__combined_mode:
            return True

        ar_file_path_list = 'atypical_restraint_file_path_list'

        if ar_file_path_list not in self.__inputParamDict:
            return True

        first_atom_pattern = re.compile(r'^ATOM +1 .*')
        hbond_da_atom_types = ('O', 'N', 'F')
        rdc_origins = ('OO', 'X', 'Y', 'Z')

        dist_restraint_uploaded = False

        cs_range_min = self.chem_shift_range['min_exclusive']
        cs_range_max = self.chem_shift_range['max_exclusive']
        dist_range_min = self.dist_restraint_range['min_inclusive']
        dist_range_max = self.dist_restraint_range['max_inclusive']
        dihed_range_min = self.dihed_restraint_range['min_inclusive']
        dihed_range_max = self.dihed_restraint_range['max_inclusive']
        rdc_range_min = self.rdc_restraint_range['min_exclusive']
        rdc_range_max = self.rdc_restraint_range['max_exclusive']

        fileListId = self.__file_path_list_len

        md5_list = []

        for ar in self.__inputParamDict[ar_file_path_list]:

            file_path = ar['file_name']

            with open(file_path, 'r', encoding='UTF-8') as ifp:

                md5_list.append(hashlib.md5(ifp.read().encode('utf-8')).hexdigest())

                ifp.close()

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']
            if 'original_file_name' in input_source_dic:
                original_file_name = input_source_dic['original_file_name']
                if file_name != original_file_name:
                    file_name = '%s (%s)' % (original_file_name, file_name)

            is_aux_amb = file_type == 'nm-aux-amb'

            if file_type == 'nm-res-cns':
                mr_format_name = 'CNS'
            elif file_type == 'nm-res-xpl':
                mr_format_name = 'XPLOR-NIH'
            elif file_type == 'nm-res-amb':
                mr_format_name = 'AMBER'
            elif is_aux_amb:
                mr_format_name = 'AMBER'
            elif file_type == 'nm-res-cya':
                mr_format_name = 'CYANA'
            else:
                mr_format_name = 'other format'

            atom_like_names = self.__csStat.getAtomLikeNameSet(minimum_len=(2 if file_type == 'nm-res-oth' or is_aux_amb else 1))
            cs_atom_like_names = list(filter(self.__isHalfSpin, atom_like_names)) # DAOTHER-7491

            has_chem_shift = False
            has_dist_restraint = False
            has_dihed_restraint = False
            has_rdc_restraint = False
            has_plane_restraint = False
            has_hbond_restraint = False
            has_rdc_origins = False

            has_coordinate = False
            has_amb_coord = False
            has_amb_inpcrd = False
            has_ens_coord = False
            has_topology = False

            has_first_atom = False

            if file_type in ('nm-res-cns', 'nm-res-xpl'):

                with open(file_path, 'r', encoding='UTF-8') as ifp:

                    atom_likes = 0
                    atom_unlikes = 0
                    cs_atom_likes = 0
                    resid_likes = 0
                    real_likes = 0
                    names = []
                    resids = []

                    rdc_atom_names = set()

                    cs_range_like = False
                    dist_range_like = False
                    dihed_range_like = False
                    rdc_range_like = False

                    for line in ifp:

                        if line.startswith('ATOM ') and line.count('.') >= 3:
                            has_coordinate = True
                            if first_atom_pattern.match(line):
                                if has_first_atom:
                                    has_ens_coord = True
                                has_first_atom = True

                        elif line.startswith('MODEL') or line.startswith('ENDMDL') or\
                             line.startswith('_atom_site.pdbx_PDB_model_num') or line.startswith('_atom_site.ndb_model'):
                            has_ens_coord = True

                        l = " ".join(line.split())

                        s = re.split('[ ()]', l)

                        _t_lower = ""

                        for t in s:

                            if len(t) == 0:
                                continue

                            if t[0] == '#' or t[0] == '!':
                                break

                            t_lower = t.lower()

                            if t_lower.startswith('assi') or (real_likes == 3 and t_lower.startswith('weight')):

                                if cs_atom_likes == 1 and resid_likes == 1 and cs_range_like:
                                    has_chem_shift = True

                                elif (atom_likes == 2 or (atom_likes > 0 and resid_likes == 2)) and dist_range_like:
                                    has_dist_restraint = True

                                elif atom_likes == 4 and dihed_range_like:
                                    has_dihed_restraint = True

                                elif cs_atom_likes + atom_unlikes == 6 and rdc_range_like:
                                    has_rdc_restraint = True

                                elif atom_likes == 3 and not (cs_range_like or dist_range_like or dihed_range_like or rdc_range_like or has_hbond_restraint)\
                                    and names[0][0] in hbond_da_atom_types and names[1][0] == 'H' and names[2][0] in hbond_da_atom_types:
                                    has_hbond_restraint = True

                                atom_likes = 0
                                atom_unlikes = 0
                                cs_atom_likes = 0
                                resid_likes = 0
                                real_likes = 0
                                names = []
                                resids = []
                                cs_range_like = False
                                dist_range_like = False
                                dihed_range_like = False
                                rdc_range_like = False

                            elif _t_lower == 'name':
                                name = t.upper()
                                if name in atom_like_names:
                                    if name not in names or len(names) > 1:
                                        atom_likes += 1
                                        names.append(name)
                                    if name in cs_atom_like_names:
                                        cs_atom_likes += 1
                                else:
                                    atom_unlikes += 1
                                    if not has_rdc_origins and name in rdc_origins:
                                        rdc_atom_names.add(name)
                                        if len(rdc_atom_names) == 4:
                                            has_rdc_origins = True

                            elif _t_lower == 'resid':
                                try:
                                    v = int(t)
                                    if v not in resids:
                                        resid_likes += 1
                                        resids.append(v)
                                except:
                                    pass

                            elif '.' in t:
                                try:
                                    v = float(t)
                                    if cs_range_min < v < cs_range_max:
                                        cs_range_like = True
                                    if dist_range_min <= v <= dist_range_max:
                                        dist_range_like = True
                                    if dihed_range_min <= v <= dihed_range_max:
                                        dihed_range_like = True
                                    if rdc_range_min < v < rdc_range_max:
                                        rdc_range_like = True
                                    real_likes += 1
                                except:
                                    pass

                            _t_lower = t_lower

                    ifp.close()

                with open(file_path, 'r', encoding='UTF-8') as ifp:

                    atom_likes = 0
                    names = []
                    has_rest = False
                    has_plan = False
                    has_grou = False
                    has_sele = False
                    has_resi = False

                    for line in ifp:

                        l = " ".join(line.split())

                        s = re.split('[ ()=]', l)

                        _t_lower = ""

                        for t in s:

                            if len(t) == 0:
                                continue

                            if t[0] == '#' or t[0] == '!':
                                break

                            t_lower = t.lower()

                            if t_lower.startswith('rest'):
                                has_rest = True

                            elif t_lower.startswith('plan'):
                                has_plan = True

                            elif has_rest and has_plan:

                                if t_lower.startswith('grou'):
                                    has_grou = True

                                elif t_lower.startswith('sele'):
                                    has_sele = True

                                    atom_likes = 0
                                    names = []

                                elif _t_lower == 'name':
                                    name = t.upper()
                                    if name in atom_like_names:
                                        if name not in names or len(names) > 1:
                                            atom_likes += 1
                                            names.append(name)

                                elif t_lower.startswith('resi'):
                                    has_resi = True

                                elif has_grou and has_sele and has_resi and not has_plane_restraint and _t_lower.startswith('weig'):
                                    if atom_likes > 0:
                                        try:
                                            v = float(t)
                                            if self.weight_range['min_inclusive'] <= v <= self.weight_range['max_inclusive']:
                                                has_plane_restraint = True
                                        except:
                                            pass

                                elif t_lower == 'end':
                                    has_grou = False
                                    has_sele = False
                                    has_resi = False

                            _t_lower = t_lower

                    ifp.close()

            elif file_type == 'nm-res-amb':

                ws_pattern = re.compile(r'\s+')
                r_pattern = re.compile(r'r(\d+)=(.*)')

                with open(file_path, 'r', encoding='UTF-8') as ifp:

                    in_rst = False
                    in_iat = False
                    in_igr1 = False
                    in_igr2 = False

                    names = []
                    values = []

                    pos = 0

                    for line in ifp:

                        if line.startswith('ATOM ') and line.count('.') >= 3:
                            has_coordinate = True
                            if first_atom_pattern.match(line):
                                if has_first_atom:
                                    has_ens_coord = True
                                has_first_atom = True

                        elif line.startswith('MODEL') or line.startswith('ENDMDL') or\
                             line.startswith('_atom_site.pdbx_PDB_model_num') or line.startswith('_atom_site.ndb_model'):
                            has_ens_coord = True

                        pos += 1

                        if pos == 1 and line.startswith('defa'):
                            has_amb_inpcrd = True

                        elif pos == 2 and has_amb_inpcrd:
                            try:
                                int(line.lstrip().split()[0])
                            except:
                                has_amb_inpcrd = False

                        elif pos == 3 and has_amb_inpcrd:
                            if line.count('.') != 6:
                                has_amb_inpcrd = False

                        if '&rst ' in line:
                            line = re.sub('&rst ', '&rst,', line)

                        elif '&end' in line:
                            line = re.sub('&end', ',&end', line)

                        elif '/' in line:
                            line = re.sub('/', ',&end', line)

                        l = " ".join(line.split())

                        if len(l) == 0 or l.startswith('#') or l.startswith('!'):
                            continue

                        s = re.split(',', ws_pattern.sub('', l).lower())

                        for t in s:

                            if len(t) == 0:
                                continue

                            if t[0] == '#' or t[0] == '!':
                                break

                            if t == '&rst':
                                in_rst = True

                            elif in_rst:

                                if t == '&end':

                                    atom_likes = 0
                                    atom_unlikes = 0

                                    for name in names:

                                        if isinstance(name, int):
                                            if int != -1:
                                                atom_likes += 1
                                            else:
                                                atom_unlikes += 1

                                        if isinstance(name, list):

                                            if any(n for n in name if n != -1):
                                                atom_likes += 1
                                            else:
                                                atom_unlikes += 1

                                    if len(values) == 4:
                                        v = (values[1] + values[2]) / 2.0

                                        if dist_range_min <= v <= dist_range_max:
                                            dist_range_like = True
                                        if dihed_range_min <= v <= dihed_range_max:
                                            dihed_range_like = True
                                        if rdc_range_min < v < rdc_range_max:
                                            rdc_range_like = True

                                        if atom_likes == 2 and dist_range_like:
                                            has_dist_restraint = True

                                        elif atom_likes == 4 and dihed_range_like:
                                            has_dihed_restraint = True

                                        elif atom_likes + atom_unlikes == 6 and rdc_range_like:
                                            has_rdc_restraint = True

                                    names = []
                                    values = []

                                    in_rst = False
                                    in_iat = False
                                    in_igr1 = False
                                    in_igr2 = False

                                elif t.startswith('iat='):
                                    in_iat = True
                                    try:
                                        iat = int(t[4:])
                                        names.append(iat)
                                    except ValueError:
                                        pass

                                    in_igr1 = False
                                    in_igr2 = False

                                elif '=' not in t and in_iat:
                                    try:
                                        iat = int(t)
                                        names.append(iat)
                                    except ValueError:
                                        pass

                                elif r_pattern.match(t):
                                    len_values = len(values)
                                    g = r_pattern.search(t).groups()
                                    try:
                                        r_idx = int(g[0]) - 1
                                        v = float(g[1])
                                        if len_values == r_idx:
                                            values.append(v)
                                        elif len_values > r_idx:
                                            values.insert(r_idx, v)
                                        else:
                                            while len(values) < r_idx:
                                                values.append(None)
                                            values.append(v)
                                    except ValueError:
                                        pass

                                    in_iat = False
                                    in_igr1 = False
                                    in_igr2 = False

                                elif t.startswith('igr1'):
                                    in_igr1 = True
                                    try:
                                        iat = int(t[5:])
                                        names.insert(0, [iat])
                                    except:
                                        pass

                                    in_iat = False
                                    in_igr2 = False

                                elif '=' not in t and in_igr1:
                                    try:
                                        iat = int(t)
                                        g = names[0]
                                        g.append(iat)
                                    except:
                                        pass

                                elif t.startswith('igr2'):
                                    in_igr2 = True
                                    try:
                                        iat = int(t[5:])
                                        names.insert(1, [iat])
                                    except:
                                        pass

                                    in_iat = False
                                    in_igr1 = False

                                elif '=' not in t and in_igr2:
                                    try:
                                        iat = int(t)
                                        g = names[1]
                                        g.append(iat)
                                    except:
                                        pass

                                elif '=' in t:
                                    in_iat = False
                                    in_igr1 = False
                                    in_igr2 = False

                    ifp.close()

            elif file_type == 'nm-res-cya' or file_type == 'nm-res-oth' or is_aux_amb:

                if is_aux_amb:

                    has_atom_name = False
                    has_residue_label = False
                    has_residue_pointer = False

                    chk_atom_name_format = False
                    chk_residue_label_format = False
                    chk_residue_pointer_format = False

                    in_atom_name = False
                    in_residue_label = False
                    in_residue_pointer = False

                    a_format_pattern = re.compile(r'^%FORMAT\((\d+)a(\d+)\)\s*')
                    i_format_pattern = re.compile(r'^%FORMAT\((\d+)I(\d+)\)\s*')

                    atom_names = []
                    residue_labels = []
                    residue_pointers = []

                atom_like_names_oth = self.__csStat.getAtomLikeNameSet(1)
                cs_atom_like_names_oth = list(filter(self.__isHalfSpin, atom_like_names_oth)) # DAOTHER-7491

                one_letter_codes = self.monDict3.values()
                three_letter_codes = self.monDict3.keys()

                prohibited_col = set()

                with open(file_path, 'r', encoding='UTF-8') as ifp:

                    pos = 0

                    for line in ifp:
                        pos += 1

                        if line.startswith('ATOM ') and line.count('.') >= 3:
                            has_coordinate = True
                            if first_atom_pattern.match(line):
                                if has_first_atom:
                                    has_ens_coord = True
                                has_first_atom = True
                            if is_aux_amb: # and line.count('.') >= 3:
                                has_amb_coord = True

                        elif line.startswith('MODEL') or line.startswith('ENDMDL') or\
                             line.startswith('_atom_site.pdbx_PDB_model_num') or line.startswith('_atom_site.ndb_model'):
                            has_ens_coord = True

                        elif is_aux_amb:

                            if pos == 1 and line.startswith('defa'):
                                has_amb_inpcrd = True

                            elif pos == 2 and has_amb_inpcrd:
                                try:
                                    int(line.lstrip().split()[0])
                                except:
                                    has_amb_inpcrd = False

                            elif pos == 3 and has_amb_inpcrd:
                                if line.count('.') != 6:
                                    has_amb_inpcrd = False

                            if line.startswith('%FLAG'):
                                in_atom_name = False
                                in_residue_label = False
                                in_residue_pointer = False

                                if line.startswith('%FLAG ATOM_NAME'):
                                    has_atom_name = True
                                    chk_atom_name_format = True

                                elif line.startswith('%FLAG RESIDUE_LABEL'):
                                    has_residue_label = True
                                    chk_residue_label_format = True

                                elif line.startswith('%FLAG RESIDUE_POINTER'):
                                    has_residue_pointer = True
                                    chk_residue_pointer_format = True

                            elif chk_atom_name_format:
                                chk_atom_name_format = a_format_pattern.match(line)
                                if chk_atom_name_format:
                                    in_atom_name = True
                                    g = a_format_pattern.search(line).groups()
                                    max_cols = int(g[0])
                                    max_char = int(g[1])
                                else:
                                    has_atom_name = False
                                chk_atom_name_format = False

                            elif chk_residue_label_format:
                                chk_residue_label_format = a_format_pattern.match(line)
                                if chk_residue_label_format:
                                    in_residue_label = True
                                    g = a_format_pattern.search(line).groups()
                                    max_cols = int(g[0])
                                    max_char = int(g[1])
                                else:
                                    has_residue_label = False
                                chk_residue_label_format = False

                            elif chk_residue_pointer_format:
                                chk_residue_pointer_format = i_format_pattern.match(line)
                                if chk_residue_pointer_format:
                                    in_residue_pointer = True
                                    g = i_format_pattern.search(line).groups()
                                    max_cols = int(g[0])
                                    max_char = int(g[1])
                                else:
                                    has_residue_pointer = False
                                chk_residue_pointer_format = False

                            elif in_atom_name:
                                len_line = len(line)
                                begin = 0
                                end = max_char
                                col = 0
                                while col < max_cols and end < len_line:
                                    atom_names.append(line[begin:end].rstrip())
                                    begin = end
                                    end += max_char
                                    col += 1

                            elif in_residue_label:
                                len_line = len(line)
                                begin = 0
                                end = max_char
                                col = 0
                                while col < max_cols and end < len_line:
                                    residue_labels.append(line[begin:end].rstrip())
                                    begin = end
                                    end += max_char
                                    col += 1

                            elif in_residue_pointer:
                                len_line = len(line)
                                begin = 0
                                end = max_char
                                col = 0
                                while col < max_cols and end < len_line:
                                    try:
                                        residue_pointers.append(int(line[begin:end].lstrip()))
                                    except ValueError:
                                        pass
                                    begin = end
                                    end += max_char
                                    col += 1

                        l = " ".join(line.split())

                        if len(l) == 0 or l.startswith('#') or l.startswith('!'):
                            continue

                        s = re.split('[ ()]', l)

                        atom_likes = 0
                        cs_atom_likes = 0
                        names = []
                        res_like = False
                        angle_like = False
                        cs_range_like = False
                        dist_range_like = False
                        dihed_range_like = False

                        for t in s:

                            if len(t) == 0:
                                continue

                            if t[0] == '#' or t[0] == '!':
                                break

                            name = t.upper()

                            if name in atom_like_names:
                                if name not in names or len(names) > 1:
                                    atom_likes += 1
                                    names.append(name)
                                if names in cs_atom_like_names:
                                    cs_atom_likes += 1

                            elif name in one_letter_codes and name not in atom_like_names_oth:
                                prohibited_col.add(s.index(t))

                            elif '.' in t:
                                try:
                                    v = float(t)
                                    if cs_range_min < v < cs_range_max:
                                        cs_range_like = True
                                    if dist_range_min <= v <= dist_range_max:
                                        dist_range_like = True
                                    if dihed_range_min <= v <= dihed_range_max:
                                        dihed_range_like = True
                                except:
                                    pass

                            elif name in three_letter_codes:
                                res_like = True

                            elif name in self.dihed_ang_names:
                                angle_like = True

                        if cs_atom_likes == 1 and cs_range_like:
                            has_chem_shift = True

                        elif atom_likes == 2 and dist_range_like:
                            has_dist_restraint = True

                        elif (atom_likes == 4 or (res_like and angle_like)) and dihed_range_like:
                            has_dihed_restraint = True

                    ifp.close()

                if file_type == 'nm-res-oth' and has_chem_shift and not has_dist_restraint and not has_dihed_restraint:

                    with open(file_path, 'r', encoding='UTF-8') as ifp:
                        for line in ifp:
                            l = " ".join(line.split())

                            if len(l) == 0 or l.startswith('#') or l.startswith('!'):
                                continue

                            s = re.split('[ ()]', l)

                            atom_likes = 0
                            cs_atom_likes = 0
                            names = []
                            res_like = False
                            angle_like = False
                            cs_range_like = False
                            dist_range_like = False
                            dihed_range_like = False

                            for t in s:

                                if len(t) == 0:
                                    continue

                                if t[0] == '#' or t[0] == '!':
                                    break

                                if s.index(t) in prohibited_col:
                                    continue

                                name = t.upper()

                                if name in atom_like_names_oth:
                                    if name not in names or len(names) > 1:
                                        atom_likes += 1
                                        names.append(name)
                                    if name in cs_atom_like_names_oth:
                                        cs_atom_likes += 1

                                elif '.' in t:
                                    try:
                                        v = float(t)
                                        if cs_range_min < v < cs_range_max:
                                            cs_range_like = True
                                        if dist_range_min <= v <= dist_range_max:
                                            dist_range_like = True
                                        if dihed_range_min <= v <= dihed_range_max:
                                            dihed_range_like = True
                                    except:
                                        pass

                                elif name in three_letter_codes:
                                    res_like = True

                                elif name in self.dihed_ang_names:
                                    angle_like = True

                            if cs_atom_likes == 1 and cs_range_like:
                                has_chem_shift = True

                            elif atom_likes == 2 and dist_range_like:
                                has_dist_restraint = True

                            elif (atom_likes == 4 or (res_like and angle_like)) and dihed_range_like:
                                has_dihed_restraint = True

                        ifp.close()

                if is_aux_amb:

                    if has_atom_name and has_residue_label and has_residue_pointer and\
                       len(atom_names) > 0 and len(residue_labels) > 0 and len(residue_pointers) > 0:
                        has_topology = True

                    if has_amb_coord and (not has_first_atom or has_ens_coord):
                        has_amb_coord = False

            if file_type in ('nm-res-cya', 'nm-res-oth') and not has_dist_restraint: # DAOTHER-7491

                with open(file_path, 'r', encoding='UTF-8') as ifp:

                    for line in ifp:

                        l = " ".join(line.split())

                        if len(l) == 0 or l.startswith('#') or l.startswith('!'):
                            continue

                        s = re.split('[ ()]', l)

                        if len(s) < 7:
                            continue

                        try:
                            int(s[0])
                            int(s[3])
                            v = float(s[6])
                            if v < dist_range_min or dist_range_max < v:
                                continue
                        except ValueError:
                            continue

                        if s[1].isalnum():
                            comp_id = s[1].upper()
                            atom_id = s[2].upper()

                            if comp_id in three_letter_codes:
                                if atom_id not in atom_like_names:
                                    continue

                            elif len(comp_id) > 3:
                                continue

                            elif not self.__updateChemCompDict(comp_id):
                                continue

                        if s[4].isalnum():
                            comp_id = s[4].upper()
                            atom_id = s[5].upper()

                            if comp_id in three_letter_codes:
                                if atom_id not in atom_like_names:
                                    continue

                            elif len(comp_id) > 3:
                                continue

                            elif not self.__updateChemCompDict(comp_id):
                                continue

                        has_dist_restraint = True

                        break

                    ifp.close()

            if has_coordinate and not has_dist_restraint and not has_dihed_restraint and not has_rdc_restraint and not has_plane_restraint and not has_hbond_restraint:

                if not is_aux_amb:

                    err = "NMR restraint file (%s) includes coordinates. Did you accidentally select the wrong format? Please re-upload the NMR restraint file." % mr_format_name

                    self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Error  - %s\n" % err)

                has_chem_shift = False

            elif has_chem_shift and not has_coordinate and not has_amb_inpcrd and not has_dist_restraint and not has_dihed_restraint and not has_rdc_restraint and not has_plane_restraint and not has_hbond_restraint:

                if has_rdc_origins:

                    hint = 'assign ( resid # and name OO ) ( resid # and name X ) ( resid # and name Y ) ( resid # and name Z ) ( segid $ and resid # and name $ ) ( segid $ and resid # and name $ ) #.# #.#'

                    err = "NMR restraint file (%s) seems to be a malformed XPLOR-NIH RDC restraint file. Tips for XPLOR-NIH RDC restraints: %r pattern must be present in the file. Did you accidentally select the wrong format? Please re-upload the NMR restraint file." % (mr_format_name, hint)

                    self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Error  - %s\n" % err)

                    has_chem_shift = False

                elif not is_aux_amb:

                    err = "NMR restraint file (%s) includes assigned chemical shifts. Did you accidentally select the wrong format? Please re-upload the NMR restraint file." % mr_format_name

                    self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Error  - %s\n" % err)

            elif has_chem_shift:
                has_chem_shift = False

            content_subtype = {'chem_shift': 1 if has_chem_shift else 0,
                               'dist_restraint': 1 if has_dist_restraint else 0,
                               'dihed_restraint': 1 if has_dihed_restraint else 0,
                               'rdc_restraint': 1 if has_rdc_restraint else 0,
                               'plane_restraint': 1 if has_plane_restraint else 0,
                               'hbond_restraint': 1 if has_hbond_restraint else 0,
                               'coordinate': 1 if has_coordinate else 0,
                               'topology': 1 if has_topology else 0}

            if not is_aux_amb and not has_chem_shift and not has_dist_restraint and not has_dihed_restraint and not has_rdc_restraint and not has_plane_restraint and not has_hbond_restraint:

                hint = ""
                if file_type in ('nm-res-cns', 'nm-res-xpl') and not has_rdc_origins:
                    hint = 'assign ( segid $ and resid # and name $ ) ( segid $ and resid # and name $ ) #.# #.# #.#'
                elif file_type == 'nm-res-amb':
                    hint = '&rst iat=#[,#], r1=#.#, r2=#.#, r3=#.#, r4=#.#, [igr1=#[,#],] [igr2=#[,#],] &end'

                if len(hint) > 0:
                    hint = ' Tips for %s restraints: ' % mr_format_name + "%r pattern must be present in the file." % hint

                warn = "Constraint type of the NMR restraint file (%s) could not be identified.%s Did you accidentally select the wrong format?" % (mr_format_name, hint)

                self.report.warning.appendDescription('missing_content', {'file_name': file_name, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Warning  - %s\n" % warn)

            elif is_aux_amb and not has_amb_coord and not has_topology:

                subtype_name = ""
                if has_chem_shift:
                    subtype_name += "Assigned chemical shifts, "
                if has_dist_restraint:
                    subtype_name += "Distance restraints, "
                if has_dihed_restraint:
                    subtype_name += "Dihedral angle restraints, "
                if has_rdc_restraint:
                    subtype_name += "RDC restraints, "
                if has_plane_restraint:
                    subtype_name += "Planarity restraints, "
                if has_hbond_restraint:
                    subtype_name = "Hydrogen bond restraints, "
                if has_amb_inpcrd:
                    subtype_name += "AMBER restart coordinates (.rst), "

                if len(subtype_name) > 0:
                    subtype_name = ". It looks like to have " + subtype_name[:-2] + " instead"

                hint = " Tips for AMBER topology: Proper contents starting with '%FLAG ATOM_NAME', '%FLAG RESIDUE_LABEL', and '%FLAG RESIDUE_POINTER' lines must be present in the file"

                if has_coordinate:
                    hint = " Tips for AMBER coordinates: It should be directory generated by 'ambpdb' command and must not have MODEL/ENDMDL keywords to ensure that AMBER atomic IDs, referred as 'iat' in the AMBER restraint file, are preserved in the file."

                err = "%r is neither AMBER topology (.prmtop) nor coordinates (.inpcrd.pdb)%s.%s Did you accidentally select the wrong format? Please re-upload the AMBER topology file." % (file_name, subtype_name, hint)

                self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Error  - %s\n" % err)

            dist_restraint_uploaded |= has_dist_restraint

            input_source.setItemValue('content_subtype', content_subtype)

            fileListId += 1

        if not dist_restraint_uploaded:

            fileListId = self.__file_path_list_len

            for ar in self.__inputParamDict[ar_file_path_list]:

                input_source = self.report.input_sources[fileListId]
                input_source_dic = input_source.get()

                file_name = input_source_dic['file_name']
                file_type = input_source_dic['file_type']
                content_subtype = input_source_dic['content_subtype']

                fileListId += 1

                if (content_subtype is not None and 'dist_restraint' in content_subtype) or file_type == 'nm-aux-amb':
                    continue

                if content_subtype is None:

                    err = "NMR restraint file does not include mandatory distance restraints or is not recognized properly. Please re-upload the NMR restraint file."

                    self.__suspended_errors_for_polypeptide.append({'content_mismatch': {'file_name': file_name, 'description': err}})

                    #self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                    #self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Error  - %s\n" % err)

                elif 'chem_shift' not in content_subtype:

                    subtype_name = ""
                    if 'dihed_restraint' in content_subtype:
                        subtype_name += "Dihedral angle restraints, "
                    if 'rdc_restraint' in content_subtype:
                        subtype_name += "RDC restraints, "
                    if 'plane_restraint' in content_subtype:
                        subtype_name += "Planarity restraints, "
                    if 'hbond_restraint' in content_subtype:
                        subtype_name += "Hydrogen bond restraints, "

                    err = "NMR restraint file includes %s. However, deposition of distance restraints is mandatory. Please re-upload the NMR restraint file." % (subtype_name[:-2])

                    self.__suspended_errors_for_polypeptide.append({'content_mismatch': {'file_name': file_name, 'description': err}})

                    #self.report.error.appendDescription('content_mismatch', {'file_name': file_name, 'description': err})
                    #self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Error  - %s\n" % err)

        md5_set = set(md5_list)

        if len(md5_set) != len(md5_list):

            ar_path_len = len(self.__inputParamDict[ar_file_path_list])

            for (i, j) in itertools.combinations(range(0, ar_path_len), 2):

                if md5_list[i] == md5_list[j]:

                    file_name_1 = os.path.basename(self.__inputParamDict[ar_file_path_list][i]['file_name'])
                    file_name_2 = os.path.basename(self.__inputParamDict[ar_file_path_list][j]['file_name'])

                    err = "You have uploaded the same NMR restranit file twice. Please replace/delete either %s or %s." % (file_name_1, file_name_2)

                    self.report.error.appendDescription('content_mismatch', {'file_name': '%s vs %s' % (file_name_1, file_name_2), 'description': err})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__detectContentSubTypeOfLegacyMR() ++ Error  - %s\n" % err)

        return not self.report.isError()

    def __isHalfSpin(self, name):
        """ Check if nuclei of a given atom name has a spin 1/2.
            @return: True for spin 1/2 nuclei, False otherwise
        """

        return any(nucl for nucl in self.half_spin_nucleus if name.startswith(nucl))

    def __getPolymerSequence(self, file_list_id, sf_data, content_subtype):
        """ Wrapper function to retrieve polymer sequence from loop of a specified saveframe and content subtype via NEFTranslator.
        """

        input_source = self.report.input_sources[file_list_id]
        input_source_dic = input_source.get()

        file_type = input_source_dic['file_type']

        if file_type == 'nef': # DAOTHER-7389, issue #3, allow empty for 'chem_shift'
            return self.__nefT.get_nef_seq(sf_data, lp_category=self.lp_categories[file_type][content_subtype],
                                           allow_empty=(content_subtype in ('chem_shift', 'spectral_peak')), allow_gap=(content_subtype not in ('poly_seq', 'entity')))

        if content_subtype == 'spectral_peak_alt':
            return self.__nefT.get_star_seq(sf_data, lp_category='_Assigned_peak_chem_shift',
                                        allow_empty=True, allow_gap=True)

        # DAOTHER-7389, issue #3, allow empty for 'chem_shift'
        return self.__nefT.get_star_seq(sf_data, lp_category=self.lp_categories[file_type][content_subtype],
                                        allow_empty=(content_subtype in ('chem_shift', 'spectral_peak')), allow_gap=(content_subtype not in ('poly_seq', 'entity')))

    def __extractPolymerSequence(self):
        """ Extract reference polymer sequence.
        """

        #if self.report.isError():
        #    return False

        is_done = True

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                is_done = False
                continue

            content_subtype = 'poly_seq'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                try:

                    poly_seq = self.__getPolymerSequence(fileListId, sf_data, content_subtype)[0]

                    input_source.setItemValue('polymer_sequence', poly_seq)

                    if file_type == 'nmr-star':

                        auth_poly_seq = self.__nefT.get_star_auth_seq(sf_data, lp_category)[0]

                        for ps in poly_seq:
                            chain_id = ps['chain_id']
                            seq_ids = ps['seq_id']
                            comp_ids = ps['comp_id']

                            for aps in auth_poly_seq:

                                if aps['chain_id'] != chain_id:
                                    continue

                                _seq_ids = aps['seq_id']
                                auth_asym_ids = aps['auth_asym_id']
                                auth_seq_ids = aps['auth_seq_id']
                                auth_comp_ids = aps['auth_comp_id']

                                auth_asym_id_set = sorted(set(auth_asym_ids))

                                for auth_asym_id in auth_asym_id_set:

                                    offsets = []
                                    total = 0

                                    for _seq_id, _auth_asym_id, auth_seq_id, auth_comp_id in zip(_seq_ids, auth_asym_ids, auth_seq_ids, auth_comp_ids):

                                        if _auth_asym_id != auth_asym_id or auth_seq_id in self.empty_value:
                                            continue

                                        try:

                                            _auth_seq_id = int(auth_seq_id)

                                            offsets.append(_auth_seq_id - _seq_id)
                                            total += 1

                                        except ValueError:

                                            if self.__check_auth_seq:
                                                warn = "Auth_seq_ID %r (Auth_asym_ID %s, Auth_comp_ID %s) should be a integer." % (auth_seq_id, auth_asym_id, auth_comp_id)

                                                self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                                self.report.setWarning()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ Warning  - %s\n" % warn)

                                    if total > 1:

                                        offset = collections.Counter(offsets).most_common()[0][0]

                                        for _seq_id, _auth_asym_id, auth_seq_id, auth_comp_id in zip(_seq_ids, auth_asym_ids, auth_seq_ids, auth_comp_ids):

                                            if _auth_asym_id != auth_asym_id or auth_seq_id in self.empty_value:
                                                continue

                                            try:

                                                _auth_seq_id = int(auth_seq_id)

                                            except ValueError:
                                                continue

                                            if _auth_seq_id - _seq_id != offset:

                                                if self.__check_auth_seq:
                                                    warn = "Auth_seq_ID %r is inconsistent with %r (Auth_asym_ID %s, Auth_comp_ID %s)." % (auth_seq_id, _seq_id + offset, auth_asym_id, auth_comp_id)

                                                    self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                                    self.report.setWarning()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ Warning  - %s\n" % warn)

                                for seq_id, comp_id in zip(seq_ids, comp_ids):

                                    for _seq_id, _auth_asym_id, auth_seq_id, auth_comp_id in zip(_seq_ids, auth_asym_ids, auth_seq_ids, auth_comp_ids):

                                        if _seq_id != seq_id:
                                            continue

                                        if comp_id == auth_comp_id:
                                            continue

                                        if self.__check_auth_seq:
                                            warn = "Auth_comp_ID %r (Auth_asym_ID %s, Auth_seq_ID %s) is inconsistent with %r (Entity_assembly_ID %s, Seq_ID %s)." % (auth_comp_id, _auth_asym_id, auth_seq_id, comp_id, chain_id, seq_id)

                                            self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ Warning  - %s\n" % warn)

                                        break

                    continue

                except KeyError as e:

                    self.report.error.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ KeyError  - %s" % str(e))

                except LookupError:
                    """
                    self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ LookupError  - %s" % str(e))
                    """
                except ValueError as e:

                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ ValueError  - %s" % str(e))

                except UserWarning as e:

                    warns = str(e).strip("'").split('\n')
                    proc_warns = set()

                    for warn in warns:

                        if warn == '' or warn in proc_warns:
                            continue

                        proc_warns.add(warn)

                        if warn.startswith('[Invalid data]'):

                            p = warn.index(']') + 2
                            warn = warn[p:]

                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ ValueError  - %s" % warn)

                        else:

                            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__extractPolymerSequence() ++ Error  - %s" % warn)
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ Error  - %s" % warn)

                except Exception as e:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__extractPolymerSequence() ++ Error  - %s" % str(e))
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ Error  - %s" % str(e))

                is_done = False

        return is_done

    def __extractPolymerSequenceInLoop(self):
        """ Extract polymer sequence in interesting loops.
        """

        #if self.report.isError():
        #    return False

        is_done = True

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                is_done = False
                continue

            poly_seq_list_set = {}

            for content_subtype in self.nmr_content_subtypes:

                if content_subtype in ('entry_info', 'poly_seq', 'entity') or (not has_key_value(input_source_dic['content_subtype'], content_subtype)):
                    continue

                poly_seq_list_set[content_subtype] = []

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
                    lp_category = '_Assigned_peak_chem_shift'

                list_id = 1

                has_poly_seq = False

                if self.__star_data_type[fileListId] == 'Loop':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = ''

                    has_poly_seq |= self.__extractPolymerSequenceInLoop__(fileListId, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, poly_seq_list_set)

                elif self.__star_data_type[fileListId] == 'Saveframe':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    has_poly_seq |= self.__extractPolymerSequenceInLoop__(fileListId, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, poly_seq_list_set)

                else:

                    for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                        sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                        if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                            continue

                        has_poly_seq |= self.__extractPolymerSequenceInLoop__(fileListId, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, poly_seq_list_set)

                        list_id += 1

                if not has_poly_seq:
                    poly_seq_list_set.pop(content_subtype)

            #if self.report.isError():
            #    is_done = False

            if len(poly_seq_list_set) > 0:
                input_source.setItemValue('polymer_sequence_in_loop', poly_seq_list_set)

        return is_done

    def __extractPolymerSequenceInLoop__(self, file_list_id, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, poly_seq_list_set):
        """ Extract polymer sequence in interesting loops.
        """

        has_poly_seq = False

        try:

            poly_seq = self.__getPolymerSequence(file_list_id, sf_data, content_subtype)[0]

            if len(poly_seq) > 0:

                poly_seq_list_set[content_subtype].append({'list_id': list_id, 'sf_framecode': sf_framecode, 'polymer_sequence': poly_seq})

                has_poly_seq = True

                if file_type == 'nmr-star':

                    auth_poly_seq = self.__nefT.get_star_auth_seq(sf_data, lp_category)[0]

                    for ps in poly_seq:
                        chain_id = ps['chain_id']
                        seq_ids = ps['seq_id']
                        comp_ids = ps['comp_id']

                        for aps in auth_poly_seq:

                            if aps['chain_id'] != chain_id:
                                continue

                            _seq_ids = aps['seq_id']
                            auth_asym_ids = aps['auth_asym_id']
                            auth_seq_ids = aps['auth_seq_id']
                            auth_comp_ids = aps['auth_comp_id']

                            auth_asym_id_set = sorted(set(auth_asym_ids))

                            for auth_asym_id in auth_asym_id_set:

                                offsets = []
                                total = 0

                                for _seq_id, _auth_asym_id, auth_seq_id, auth_comp_id in zip(_seq_ids, auth_asym_ids, auth_seq_ids, auth_comp_ids):

                                    if _auth_asym_id != auth_asym_id or auth_seq_id in self.empty_value:
                                        continue

                                    try:

                                        _auth_seq_id = int(auth_seq_id)

                                        offsets.append(_auth_seq_id - _seq_id)
                                        total += 1

                                    except ValueError:

                                        if self.__check_auth_seq:
                                            warn = "Auth_seq_ID %r (Auth_asym_ID %s, Auth_comp_ID %s) should be a integer." % (auth_seq_id, auth_asym_id, auth_comp_id)

                                            self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ Warning  - %s\n" % warn)

                                if total > 1:

                                    offset = collections.Counter(offsets).most_common()[0][0]

                                    for _seq_id, _auth_asym_id, auth_seq_id, auth_comp_id in zip(_seq_ids, auth_asym_ids, auth_seq_ids, auth_comp_ids):

                                        if _auth_asym_id != auth_asym_id or auth_seq_id in self.empty_value:
                                            continue

                                        try:

                                            _auth_seq_id = int(auth_seq_id)

                                        except ValueError:
                                            continue

                                        if _auth_seq_id - _seq_id != offset:

                                            if self.__check_auth_seq:
                                                warn = "Auth_seq_ID %r is inconsistent with %r (Auth_asym_ID %s, Auth_comp_ID %s)." % (auth_seq_id, _seq_id + offset, auth_asym_id, auth_comp_id)

                                                self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                                self.report.setWarning()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ Warning  - %s\n" % warn)

                            for seq_id, comp_id in zip(seq_ids, comp_ids):

                                for _seq_id, _auth_asym_id, auth_seq_id, auth_comp_id in zip(_seq_ids, auth_asym_ids, auth_seq_ids, auth_comp_ids):

                                    if _seq_id != seq_id:
                                        continue

                                    if comp_id == auth_comp_id:
                                        continue

                                    if self.__check_auth_seq:
                                        warn = "Auth_comp_ID %r (Auth_asym_ID %s, Auth_seq_ID %s) is inconsistent with %r (Entity_assembly_ID %s, Seq_ID %s)." % (auth_comp_id, _auth_asym_id, auth_seq_id, comp_id, chain_id, seq_id)

                                        self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ Warning  - %s\n" % warn)

                                    break

        except KeyError as e:

            if 'Auth' not in str(e) or self.__check_auth_seq:
                self.report.error.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ KeyError  - %s" % str(e))

        except LookupError:
            """
            self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ LookupError  - %s" % str(e))
            """
        except ValueError as e:

            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ ValueError  - %s" % str(e))

        except UserWarning as e:

            warns = str(e).strip("'").split('\n')
            proc_warns = set()

            for warn in warns:

                if warn == '' or warn in proc_warns:
                    continue

                proc_warns.add(warn)

                if warn.startswith('[Invalid data]'):

                    if 'Auth' not in warn or self.__check_auth_seq:

                        p = warn.index(']') + 2
                        warn = warn[p:]

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ ValueError  - %s" % warn)

                else:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__extractPolymerSequenceInLoop() ++ Error  - %s" % warn)
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ Error  - %s" % warn)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__extractPolymerSequenceInLoop() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__extractPolymerSequenceInLoop() ++ Error  - %s" % str(e))

        return has_poly_seq

    def __isConsistentSequence(self):
        """ Perform sequence consistency test among extracted polymer sequences.
            @return: True for valid sequence, False otherwise
        """

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            #file_type = input_source_dic['file_type']

            has_poly_seq = has_key_value(input_source_dic, 'polymer_sequence')
            has_poly_seq_in_loop = has_key_value(input_source_dic, 'polymer_sequence_in_loop')

            if (not has_poly_seq) or (not has_poly_seq_in_loop):
                continue

            polymer_sequence = input_source_dic['polymer_sequence']
            polymer_sequence_in_loop = input_source_dic['polymer_sequence_in_loop']

            poly_seq = 'poly_seq'

            subtype_with_poly_seq = [poly_seq if has_poly_seq else None]

            for subtype in polymer_sequence_in_loop.keys():
                subtype_with_poly_seq.append(subtype)

            for subtype_pair in itertools.combinations_with_replacement(subtype_with_poly_seq, 2):

                # poly_seq is reference sequence and suppress tests on combinations of two sequences in loop
                if has_poly_seq and ((poly_seq not in subtype_pair) or subtype_pair == (poly_seq, poly_seq)):
                    continue

                subtype1 = subtype_pair[0] # poly_seq will appear only on subtype1
                subtype2 = subtype_pair[1]

                if subtype1 is None or subtype2 is None:
                    continue
                """
                lp_category1 = self.lp_categories[file_type][subtype1]
                lp_category2 = self.lp_categories[file_type][subtype2]

                if file_type == 'nmr-star':
                    if subtype1 == 'spectral_peak_alt':
                        lp_category1 = '_Assigned_peak_chem_shift'
                    if subtype2 == 'spectral_peak_alt':
                        lp_category2 = '_Assigned_peak_chem_shift'
                """
                # reference polymer sequence exists
                if has_poly_seq and subtype1 == poly_seq:
                    ps1 = polymer_sequence

                    ref_chain_ids = {s1['chain_id'] for s1 in ps1}

                    for ps_in_loop in polymer_sequence_in_loop[subtype2]:
                        ps2 = ps_in_loop['polymer_sequence']
                        #sf_framecode2 = ps_in_loop['sf_framecode']

                        for s2 in ps2:

                            chain_id = s2['chain_id']

                            if chain_id not in ref_chain_ids and not ('identical_chain_id' in s2 and chain_id not in s2['identical_chain_id']):
                                return False

                            for s1 in ps1:

                                if s1['chain_id'] != chain_id and not ('identical_chain_id' in s2 and s1['chain_id'] in s2['identical_chain_id']):
                                    continue

                                for seq_id, comp_id in zip(s2['seq_id'], s2['comp_id']):

                                    if seq_id not in s1['seq_id']:

                                        if comp_id != '.':
                                            return False

                                    else:
                                        i = s1['seq_id'].index(seq_id)
                                        _comp_id = s1['comp_id'][i]

                                        if comp_id not in self.empty_value and _comp_id not in self.empty_value and comp_id != _comp_id:
                                            return False

                # brute force check
                else:

                    for ps_in_loop in polymer_sequence_in_loop[subtype1]:
                        ps1 = ps_in_loop['polymer_sequence']
                        #sf_framecode1 = ps_in_loop['sf_framecode']

                        for ps_in_loop2 in polymer_sequence_in_loop[subtype2]:
                            ps2 = ps_in_loop2['polymer_sequence']
                            #sf_framecode2 = ps_in_loop2['sf_framecode']

                            # suppress redundant tests inside the same subtype
                            if subtype1 == subtype2 and ps_in_loop['list_id'] >= ps_in_loop2['list_id']:
                                continue

                            for s2 in ps2:

                                chain_id = s2['chain_id']

                                for s1 in ps1:

                                    if chain_id != s1['chain_id']:
                                        continue

                                    for seq_id, comp_id in zip(s2['seq_id'], s2['comp_id']):

                                        if seq_id in s1['seq_id']:
                                            i = s1['seq_id'].index(seq_id)
                                            _comp_id = s1['comp_id'][i]

                                            if comp_id not in self.empty_value and _comp_id not in self.empty_value and comp_id != _comp_id:
                                                return False

                            # inverse check required for unverified sequences
                            for s1 in ps1:

                                chain_id = s1['chain_id']

                                for s2 in ps2:

                                    if chain_id != s2['chain_id']:
                                        continue

                                    for seq_id, comp_id in zip(s1['seq_id'], s1['comp_id']):

                                        if seq_id in s2['seq_id']:
                                            j = s2['seq_id'].index(seq_id)
                                            _comp_id = s2['comp_id'][j]

                                            if comp_id not in self.empty_value and _comp_id not in self.empty_value and comp_id != _comp_id:
                                                return False

        return True

    def __testSequenceConsistency(self):
        """ Perform sequence consistency test among extracted polymer sequences.
        """

        #if self.report.isError():
        #    return False

        if self.__valid_seq:
            return True

        update_poly_seq = False

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            has_poly_seq = has_key_value(input_source_dic, 'polymer_sequence')
            has_poly_seq_in_loop = has_key_value(input_source_dic, 'polymer_sequence_in_loop')

            if (not has_poly_seq) or (not has_poly_seq_in_loop):
                continue

            polymer_sequence = input_source_dic['polymer_sequence']
            polymer_sequence_in_loop = input_source_dic['polymer_sequence_in_loop']

            poly_seq = 'poly_seq'

            subtype_with_poly_seq = [poly_seq if has_poly_seq else None]

            for subtype in polymer_sequence_in_loop.keys():
                subtype_with_poly_seq.append(subtype)

            for subtype_pair in itertools.combinations_with_replacement(subtype_with_poly_seq, 2):

                # poly_seq is reference sequence and suppress tests on combinations of two sequences in loop
                if has_poly_seq and ((poly_seq not in subtype_pair) or subtype_pair == (poly_seq, poly_seq)):
                    continue

                subtype1 = subtype_pair[0] # poly_seq will appear only on subtype1
                subtype2 = subtype_pair[1]

                if subtype1 is None or subtype2 is None:
                    continue

                #lp_category1 = self.lp_categories[file_type][subtype1]
                lp_category2 = self.lp_categories[file_type][subtype2]

                if file_type == 'nmr-star':
                    #if subtype1 == 'spectral_peak_alt':
                    #    lp_category1 = '_Assigned_peak_chem_shift'
                    if subtype2 == 'spectral_peak_alt':
                        lp_category2 = '_Assigned_peak_chem_shift'

                # reference polymer sequence exists
                if has_poly_seq and subtype1 == poly_seq:
                    ps1 = polymer_sequence

                    ref_chain_ids = {s1['chain_id'] for s1 in ps1}

                    for ps_in_loop in polymer_sequence_in_loop[subtype2]:
                        ps2 = ps_in_loop['polymer_sequence']
                        sf_framecode2 = ps_in_loop['sf_framecode']

                        for s2 in ps2:

                            chain_id = s2['chain_id']

                            if chain_id not in ref_chain_ids and not ('identical_chain_id' in s2 and chain_id not in s2['identical_chain_id']):

                                err = "Invalid chain_id %r in a loop %s." % (chain_id, lp_category2)

                                self.report.error.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Error  - %s\n" % err)

                            else:

                                for s1 in ps1:

                                    if s1['chain_id'] != chain_id and not ('identical_chain_id' in s2 and s1['chain_id'] in s2['identical_chain_id']):
                                        continue

                                    if 'identical_chain_id' in s2:
                                        _s1 = next((_s1 for _s1 in ps1 if _s1['chain_id'] == chain_id), None)
                                        __s1 = next((_s1 for _s1 in ps1 if _s1['chain_id'] in s2['identical_chain_id']), None)
                                        if _s1 is not None and len(s1['seq_id']) != len(_s1['seq_id']):
                                            continue
                                        if __s1 is not None and len(s1['seq_id']) != len(__s1['seq_id']):
                                            continue

                                    for seq_id, comp_id in zip(s2['seq_id'], s2['comp_id']):

                                        if seq_id not in s1['seq_id']:

                                            if comp_id != '.':

                                                if self.__target_framecode not in self.empty_value:
                                                    print(sf_framecode2)
                                                    print(s1['chain_id'])
                                                    print(s1['seq_id'])
                                                    print(s1['comp_id'])

                                                    print(s2['chain_id'])
                                                    print(s2['seq_id'])
                                                    print(s2['comp_id'])

                                                    print('%s %s' % (seq_id, comp_id))

                                                    sys.exit(1)

                                                err = "Invalid seq_id %r (chain_id %s) in a loop %s." % (seq_id, chain_id, lp_category2)

                                                self.report.error.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                self.report.setError()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Error  - %s\n" % err)

                                        else:
                                            i = s1['seq_id'].index(seq_id)
                                            _comp_id = s1['comp_id'][i]

                                            if comp_id not in self.empty_value and _comp_id not in self.empty_value and comp_id != _comp_id:

                                                if self.__target_framecode not in self.empty_value:
                                                    print(sf_framecode2)
                                                    print(s1['chain_id'])
                                                    print(s1['seq_id'])
                                                    print(s1['comp_id'])

                                                    print(s2['chain_id'])
                                                    print(s2['seq_id'])
                                                    print(s2['comp_id'])

                                                    print('%s %s' % (seq_id, comp_id))

                                                    sys.exit(1)

                                                err = "Invalid comp_id %r vs %r (seq_id %s, chain_id %s) in a loop %s." % (comp_id, _comp_id, seq_id, chain_id, lp_category2)

                                                if self.__tolerant_seq_align and self.__equalsRepresentativeCompId(comp_id, _comp_id):
                                                    self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                    self.report.setWarning()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Warning  - %s\n" % err)

                                                elif self.__tolerant_seq_align and self.__get1LetterCode(comp_id) == self.__get1LetterCode(_comp_id):
                                                    self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                    self.report.setWarning()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Warning  - %s\n" % err)

                                                    comp_id_conv_dict = {comp_id: _comp_id}

                                                    self.__fixCompIdInLoop(fileListId, file_type, subtype2, sf_framecode2, chain_id, seq_id, comp_id_conv_dict)

                                                    update_poly_seq = True

                                                else:
                                                    self.report.error.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                    self.report.setError()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Error  - %s\n" % err)

                # brute force check
                else:

                    for ps_in_loop in polymer_sequence_in_loop[subtype1]:
                        ps1 = ps_in_loop['polymer_sequence']
                        sf_framecode1 = ps_in_loop['sf_framecode']

                        for ps_in_loop2 in polymer_sequence_in_loop[subtype2]:
                            ps2 = ps_in_loop2['polymer_sequence']
                            sf_framecode2 = ps_in_loop2['sf_framecode']

                            # suppress redundant tests inside the same subtype
                            if subtype1 == subtype2 and ps_in_loop['list_id'] >= ps_in_loop2['list_id']:
                                continue

                            for s2 in ps2:

                                chain_id = s2['chain_id']

                                for s1 in ps1:

                                    if chain_id != s1['chain_id']:
                                        continue

                                    for seq_id, comp_id in zip(s2['seq_id'], s2['comp_id']):

                                        if seq_id in s1['seq_id']:
                                            i = s1['seq_id'].index(seq_id)
                                            _comp_id = s1['comp_id'][i]

                                            if comp_id not in self.empty_value and _comp_id not in self.empty_value and comp_id != _comp_id:

                                                err = "Unmatched comp_id %r vs %r (seq_id %s, chain_id %s) exists against %r saveframe." % (comp_id, _comp_id, seq_id, chain_id, sf_framecode1)

                                                if self.__tolerant_seq_align and self.__equalsRepresentativeCompId(comp_id, _comp_id):
                                                    self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                    self.report.setWarning()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Warning  - %s\n" % err)

                                                else:
                                                    self.report.error.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                    self.report.setError()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Error  - %s\n" % err)

                            # inverse check required for unverified sequences
                            for s1 in ps1:

                                chain_id = s1['chain_id']

                                for s2 in ps2:

                                    if chain_id != s2['chain_id']:
                                        continue

                                    for seq_id, comp_id in zip(s1['seq_id'], s1['comp_id']):

                                        if seq_id in s2['seq_id']:
                                            j = s2['seq_id'].index(seq_id)
                                            _comp_id = s2['comp_id'][j]

                                            if comp_id not in self.empty_value and _comp_id not in self.empty_value and comp_id != _comp_id:

                                                err = "Unmatched comp_id %r vs %r (seq_id %s, chain_id %s) exists against %r saveframe." % (comp_id, _comp_id, seq_id, chain_id, sf_framecode2)

                                                if self.__tolerant_seq_align and self.__equalsRepresentativeCompId(comp_id, _comp_id):
                                                    self.report.warning.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                    self.report.setWarning()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Warning  - %s\n" % err)

                                                else:
                                                    self.report.error.appendDescription('sequence_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode2, 'category': lp_category2, 'description': err})
                                                    self.report.setError()

                                                    if self.__verbose:
                                                        self.__lfh.write("+NmrDpUtility.__testSequenceConsistency() ++ Error  - %s\n" % err)

        if update_poly_seq:
            self.__extractPolymerSequenceInLoop()
            self.__depositNmrData()

        return not self.report.isError()

    def __equalsRepresentativeCompId(self, comp_id, ref_comp_id):
        """ Return whether given representative comp IDs are equal.
            @return: True for representative comp IDs are matched, False otherwise
        """

        if comp_id is self.empty_value or ref_comp_id in self.empty_value:
            return False

        if '_' in comp_id:
            comp_id = comp_id.split('_')[0]

        elif self.__get1LetterCode(comp_id) == 'X' and self.__updateChemCompDict(comp_id):
            if '_chem_comp.mon_nstd_parent_comp_id' in self.__last_chem_comp_dict: # matches with comp_id in CCD
                if self.__last_chem_comp_dict['_chem_comp.mon_nstd_parent_comp_id'] not in self.empty_value:
                    comp_id = self.__last_chem_comp_dict['_chem_comp.mon_nstd_parent_comp_id']
                    if comp_id in ('A', 'C', 'G', 'T', 'I', 'U') and len(ref_comp_id) == 2 and ref_comp_id.startswith('D'):
                        comp_id = 'D' + comp_id
                    elif ref_comp_id in ('A', 'C', 'G', 'T', 'I', 'U') and len(comp_id) == 2 and comp_id.startswith('D'):
                        comp_id = comp_id[1]

        if '_' in ref_comp_id:
            ref_comp_id = ref_comp_id.split('_')[0]

        elif self.__get1LetterCode(ref_comp_id) == 'X' and self.__updateChemCompDict(ref_comp_id):
            if '_chem_comp.mon_nstd_parent_comp_id' in self.__last_chem_comp_dict: # matches with comp_id in CCD
                if self.__last_chem_comp_dict['_chem_comp.mon_nstd_parent_comp_id'] not in self.empty_value:
                    ref_comp_id = self.__last_chem_comp_dict['_chem_comp.mon_nstd_parent_comp_id']
                    if ref_comp_id in ('A', 'C', 'G', 'T', 'I', 'U') and len(comp_id) == 2 and comp_id.startswith('D'):
                        ref_comp_id = 'D' + ref_comp_id
                    elif comp_id in ('A', 'C', 'G', 'T', 'I', 'U') and len(ref_comp_id) == 2 and ref_comp_id.startswith('D'):
                        ref_comp_id = ref_comp_id[1]

        return comp_id == ref_comp_id

    def __fixCompIdInLoop(self, file_list_id, file_type, content_subtype, sf_framecode, chain_id, seq_id, comp_id_conv_dict):
        """ Fix comp ID of interesting loop.
        """

        sf_category = self.sf_categories[file_type][content_subtype]
        lp_category = self.lp_categories[file_type][content_subtype]

        if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
            lp_category = '_Assigned_peak_chem_shift'

        if self.__star_data_type[file_list_id] == 'Loop':

            sf_data = self.__star_data[file_list_id]

            if sf_framecode == '':
                self.__fixCompIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id, comp_id_conv_dict)

        elif self.__star_data_type[file_list_id] == 'Saveframe':

            sf_data = self.__star_data[file_list_id]

            if get_first_sf_tag(sf_data, 'sf_framecode') == sf_framecode:
                self.__fixCompIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id, comp_id_conv_dict)

        else:

            for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                if get_first_sf_tag(sf_data, 'sf_framecode') != sf_framecode:
                    continue

                if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                    continue

                self.__fixCompIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id, comp_id_conv_dict)

    def __fixCompIdInLoop__(self, file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id, comp_id_conv_dict):
        """ Fix sequence ID of interesting loop.
        """

        chain_id_name = 'chain_code' if file_type == 'nef' else 'Entity_assembly_ID'
        seq_id_name = 'sequence_code' if file_type == 'nef' else 'Comp_index_ID'
        comp_id_name = 'residue_name' if file_type == 'nef' else 'Comp_ID'

        max_dim = 2

        if content_subtype in ('poly_seq', 'dist_restraint', 'rdc_restraint'):
            max_dim = 3

        elif content_subtype == 'dihed_restraint':
            max_dim = 5

        elif content_subtype == 'spectral_peak':

            try:

                _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                num_dim = int(_num_dim)

                if num_dim not in range(1, self.lim_num_dim):
                    raise ValueError()

            except ValueError: # raised error already at __testIndexConsistency()
                return

            max_dim = num_dim + 1

        if __pynmrstar_v3_2__:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop(lp_category)
        else:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop_by_category(lp_category)

        if max_dim == 2:

            chain_id_col = loop.tags.index(chain_id_name) if chain_id_name in loop.tags else -1
            seq_id_col = loop.tags.index(seq_id_name) if seq_id_name in loop.tags else -1
            comp_id_col = loop.tags.index(comp_id_name) if comp_id_name in loop.tags else -1

            if chain_id_col == -1 or seq_id_col == -1 or comp_id_col == -1:
                return

            for row in loop.data:

                if row[chain_id_col] != chain_id:
                    continue

                _seq_id = row[seq_id_col]

                if _seq_id in self.empty_value or int(_seq_id) != seq_id:
                    continue

                comp_id = row[comp_id_col]

                if comp_id in comp_id_conv_dict:
                    row[comp_id_col] = comp_id_conv_dict[comp_id]

        else:

            for i in range(1, max_dim):

                _chain_id_name = chain_id_name + '_' + str(i)
                _seq_id_name = seq_id_name + '_' + str(i)
                _comp_id_name = comp_id_name + '_' + str(i)

                chain_id_col = loop.tags.index(_chain_id_name) if _chain_id_name in loop.tags else -1
                seq_id_col = loop.tags.index(_seq_id_name) if _seq_id_name in loop.tags else -1
                comp_id_col = loop.tags.index(_comp_id_name) if _comp_id_name in loop.tags else -1

                if chain_id_col == -1 or seq_id_col == -1 or comp_id_col == -1:
                    continue

                for row in loop.data:

                    if row[chain_id_col] != chain_id:
                        continue

                    _seq_id = row[seq_id_col]

                    if _seq_id in self.empty_value or int(_seq_id) != seq_id:
                        continue

                    comp_id = row[comp_id_col]

                    if comp_id in comp_id_conv_dict:
                        row[comp_id_col] = comp_id_conv_dict[comp_id]

    def __extractCommonPolymerSequence(self):
        """ Extract common polymer sequence if required.
        """

        #if self.report.isError():
        #    return False

        common_poly_seq = {}

        #primary_ps_list = []

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            #file_type = input_source_dic['file_type']

            has_poly_seq = has_key_value(input_source_dic, 'polymer_sequence')
            has_poly_seq_in_loop = has_key_value(input_source_dic, 'polymer_sequence_in_loop')

            # pass if poly_seq exists
            if has_poly_seq or (not has_poly_seq_in_loop):
                continue

            if self.__extractPolymerSequenceInEntityAssembly(fileListId):
                continue

            polymer_sequence_in_loop = input_source_dic['polymer_sequence_in_loop']

            content_subtype = 'chem_shift'

            if content_subtype not in polymer_sequence_in_loop: # STAR formatted MR has no chem shift
                continue

            #for content_subtype in polymer_sequence_in_loop.keys():

            for ps_in_loop in polymer_sequence_in_loop[content_subtype]:
                ps = ps_in_loop['polymer_sequence']
                """
                if len(primary_ps_list) > 0:
                    for primary_ps in primary_ps_list:
                        for primary_s in primary_ps:
                            last_chain_id = primary_s['chain_id']

                    chain_id_offset = self.__nefT.letter_to_int(last_chain_id);

                    for primary_ps in primary_ps_list:
                        for primary_s in primary_ps:
                            primary_chain_id = primary_s['chain_id']

                            s = next((s for s in ps if s['chain_id'] == primary_chain_id), None)

                            if s is not None:

                                _s1 = fill_blank_comp_id_with_offset(primary_s, 0)
                                _s2 = fill_blank_comp_id_with_offset(s, 0)

                                self.__pA.setReferenceSequence(_s1['comp_id'], 'REF' + chain_id)
                                self.__pA.addTestSequence(_s2['comp_id'], chain_id)
                                self.__pA.doAlign()

                                myAlign = self.__pA.getAlignment(chain_id)

                                length = len(myAlign)

                                if length == 0:
                                    continue

                                _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                                if length == unmapped + conflict or _matched <= conflict or (len(polymer_sequence) > 1 and _matched < 4 and offset_1 > 0):
                                    chain_id_offset += 1
                                    s['chain_id'] = self.__nefT.index_to_letter(chain_id_offset) if file_type == 'nef' else str(chain_id_offset)
                                    if fileListId in self.__remapped_def_chain_id:
                                        self.__remapped_def_chain_id[fileListId] = {}
                                    self.__remapped_def_chain_id[fileListId] = {chain_id: s['chain_id']}

                primary_ps_list.append(ps)
                """
                for s in ps:
                    chain_id = s['chain_id']

                    if chain_id not in common_poly_seq:
                        common_poly_seq[chain_id] = set()

            chains = common_poly_seq.keys()
            offset_seq_ids = {c: 0 for c in chains}

            #for content_subtype in polymer_sequence_in_loop.keys():

            for ps_in_loop in polymer_sequence_in_loop[content_subtype]:
                ps = ps_in_loop['polymer_sequence']

                for s in ps:
                    chain_id = s['chain_id']

                    min_seq_id = min(s['seq_id'])
                    if min_seq_id < 0:
                        offset_seq_ids[chain_id] = min_seq_id * -1

                    for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):
                        common_poly_seq[chain_id].add('{:04d} {}'.format(seq_id + offset_seq_ids[chain_id], comp_id))

        asm = [] # molecular assembly of a loop

        for chain_id in sorted(common_poly_seq.keys()):

            if len(common_poly_seq[chain_id]) > 0:
                seq_id_list = sorted(set(int(i.split(' ')[0]) - offset_seq_ids[chain_id] for i in common_poly_seq[chain_id]))
                comp_id_list = []

                for seq_id in seq_id_list:
                    _comp_id = [i.split(' ')[1] for i in common_poly_seq[chain_id] if int(i.split(' ')[0]) - offset_seq_ids[chain_id] == seq_id]
                    if len(_comp_id) == 1:
                        comp_id_list.append(_comp_id[0])
                    else:
                        comp_id_list.append(next(comp_id for comp_id in _comp_id if comp_id not in self.empty_value))

                if self.__combined_mode and self.__has_star_entity:
                    ent = self.__extractPolymerSequenceInEntityLoopOfChain(fileListId, chain_id)

                    if ent is not None:
                        asm.append(ent)
                        continue

                asm.append({'chain_id': chain_id, 'seq_id': seq_id_list, 'comp_id': comp_id_list})

        if len(asm) > 0:

            for fileListId in range(self.__file_path_list_len):

                input_source = self.report.input_sources[fileListId]
                input_source_dic = input_source.get()

                has_poly_seq = has_key_value(input_source_dic, 'polymer_sequence')
                has_poly_seq_in_loop = has_key_value(input_source_dic, 'polymer_sequence_in_loop')

                # pass if poly_seq exists
                if has_poly_seq or (not has_poly_seq_in_loop):
                    continue

                if self.__extractPolymerSequenceInEntityAssembly(fileListId):
                    continue

                input_source.setItemValue('polymer_sequence', asm)

        return True

    def __extractPolymerSequenceInEntityAssembly(self, file_list_id):
        """ Extract polymer sequence in entity loops. (NMR combined deposition)
        """

        input_source = self.report.input_sources[file_list_id]
        input_source_dic = input_source.get()

        file_type = input_source_dic['file_type']

        if file_type != 'nmr-star' or (not self.__has_star_entity):
            return False

        if not self.__combined_mode:
            return self.__extractPolymerSequenceInEntityLoop(file_list_id)

        for sf_data in self.__star_data[file_list_id].get_saveframes_by_category('assembly'):

            try:
                if __pynmrstar_v3_2__:
                    loop = sf_data.get_loop('_Entity_assembly')
                else:
                    loop = sf_data.get_loop_by_category('_Entity_assembly')
            except KeyError:
                return False

            if loop is None:
                return False

            tags = ['ID', 'Entity_assembly_name', 'Entity_ID']

            if 'Entity_label' in loop.tags:
                tags.append('Entity_label')

            if set(tags) & set(loop.tags) != set(tags):
                return False

            dat = get_lp_tag(loop, tags)

            asm = [] # molecular assembly of a loop

            chain_ids = set()
            entity_sfs = {}

            for c in dat:

                if c[0] in self.empty_value or c[1] in self.empty_value or c[2] in self.empty_value:
                    return False

                try:
                    chain_id = str(c[0])
                    entity_sf = c[1] if len(c) < 4 else (c[3][1:] if c[3][0] == '$' else c[3]) # Entity_assemble_name or Entity_label
                    entity_id = int(c[2])

                    if chain_id in chain_ids:
                        return False

                    chain_ids.add(chain_id)

                    for k, v in entity_sfs.items():
                        if (k != entity_sf and v == entity_id) or (k == entity_sf and v != entity_id):
                            return False

                    entity_sfs[entity_sf] = entity_id

                except ValueError:
                    return False

                _sf_data = self.__getSaveframeByName(file_list_id, entity_sf)

                if _sf_data is None:
                    return False

                content_subtype = 'entity'

                try:
                    if __pynmrstar_v3_2__:
                        _loop = _sf_data.get_loop(self.lp_categories[file_type][content_subtype])
                    else:
                        _loop = _sf_data.get_loop_by_category(self.lp_categories[file_type][content_subtype])
                except KeyError:
                    return False

                if _loop is None:
                    return False

                _tags = ['ID', 'Comp_ID', 'Entity_ID']

                if set(_tags) & set(_loop.tags) != set(_tags):
                    return False

                _dat = get_lp_tag(_loop, _tags)

                seq = set()

                for s in _dat:

                    if s[0] in self.empty_value or s[1] in self.empty_value or s[2] in self.empty_value:
                        return False

                    try:
                        seq_id = int(s[0])
                        comp_id = s[1]
                        _entity_id = int(s[2])
                    except ValueError:
                        return False

                    if entity_id != _entity_id:
                        return False

                    seq.add('{:04d} {}'.format(seq_id, comp_id))

                sorted_seq = sorted(seq)

                asm.append({'chain_id': chain_id, 'seq_id': [int(i.split(' ')[0]) for i in sorted_seq], 'comp_id':  [i.split(' ')[-1] for i in sorted_seq]})

            if len(asm) > 0:
                input_source.setItemValue('polymer_sequence', asm)
                return True

            break

        return False

    def __extractPolymerSequenceInEntityLoop(self, file_list_id):
        """ Extract polymer sequence in entity loops. (NMR separated deposition)
        """

        input_source = self.report.input_sources[file_list_id]
        input_source_dic = input_source.get()

        file_type = input_source_dic['file_type']

        if file_type != 'nmr-star' or (not self.__has_star_entity):
            return False

        if self.__combined_mode:
            return self.__extractPolymerSequenceInEntityAssembly(file_list_id)

        star_data = self.__star_data[file_list_id]

        content_subtype = 'entity'

        lp_category = self.lp_categories[file_type][content_subtype]

        try:
            loops = star_data.get_loops_by_category(lp_category)
        except AttributeError:
            try:
                if __pynmrstar_v3_2__:
                    loops = [star_data.get_loop(lp_category)]
                else:
                    loops = [star_data.get_loop_by_category(lp_category)]
            except AttributeError:
                return False

        asm = [] # molecular assembly of a loop

        chain_ids = set()
        seq = {}

        for loop in loops:

            if loop is None:
                continue

            tags = ['ID', 'Comp_ID', 'Entity_ID']
            tags_ = ['ID', 'Comp_ID']

            dat = []

            if set(tags) & set(loop.tags) == set(tags):
                dat = get_lp_tag(loop, tags)
                for i in dat:
                    if i[2] in self.empty_value:
                        i[2] = '1'
            elif set(tags_) & set(loop.tags) == set(tags_): # No Entity_ID tag case
                dat = get_lp_tag(loop, tags_)
                for i in dat:
                    i.append('1')

            for i in dat:

                if i[0] in self.empty_value or i[1] in self.empty_value or i[2] in self.empty_value:
                    return False

                try:
                    c = str(i[2])

                    chain_ids.add(c)
                    if c not in seq:
                        seq[c] = set()
                    seq[c].add('{:04d} {}'.format(int(i[0]), i[1]))
                except ValueError:
                    return False

        for chain_id in chain_ids:

            sorted_seq = sorted(seq[chain_id])

            asm.append({'chain_id': chain_id, 'seq_id': [int(i.split(' ')[0]) for i in sorted_seq], 'comp_id':  [i.split(' ')[-1] for i in sorted_seq]})

        if len(asm) > 0:
            input_source.setItemValue('polymer_sequence', asm)
            return True

        return False

    def __extractPolymerSequenceInEntityLoopOfChain(self, file_list_id, chain_id):
        """ Extract polymer sequence in entity loops of a given chain id.
        """

        input_source = self.report.input_sources[file_list_id]
        input_source_dic = input_source.get()

        file_type = input_source_dic['file_type']

        if file_type != 'nmr-star' or (not self.__has_star_entity):
            return False

        star_data = self.__star_data[file_list_id]

        content_subtype = 'entity'

        lp_category = self.lp_categories[file_type][content_subtype]

        try:
            loops = star_data.get_loops_by_category(lp_category)
        except AttributeError:
            try:
                if __pynmrstar_v3_2__:
                    loops = [star_data.get_loop(lp_category)]
                else:
                    loops = [star_data.get_loop_by_category(lp_category)]
            except AttributeError:
                return False

        chain_ids = set()
        seq = {}

        for loop in loops:

            if loop is None:
                continue

            tags = ['ID', 'Comp_ID', 'Entity_ID']
            tags_ = ['ID', 'Comp_ID']

            dat = []

            if set(tags) & set(loop.tags) == set(tags):
                dat = get_lp_tag(loop, tags)
                for i in dat:
                    if i[2] in self.empty_value:
                        i[2] = '1'
            elif set(tags_) & set(loop.tags) == set(tags_): # No Entity_ID tag case
                dat = get_lp_tag(loop, tags_)
                for i in dat:
                    i.append('1')

            for i in dat:

                if i[0] in self.empty_value or i[1] in self.empty_value or i[2] in self.empty_value:
                    return False

                try:
                    c = str(i[2])

                    chain_ids.add(c)
                    if c not in seq:
                        seq[c] = set()
                    seq[c].add('{:04d} {}'.format(int(i[0]), i[1]))
                except ValueError:
                    return False

        if chain_id in chain_ids:

            sorted_seq = sorted(seq[chain_id])

            return {'chain_id': chain_id, 'seq_id': [int(i.split(' ')[0]) for i in sorted_seq], 'comp_id':  [i.split(' ')[-1] for i in sorted_seq]}

        return None

    def __extractNonStandardResidue(self):
        """ Extract non-standard residue.
        """

        #if self.report.isError():
        #    return False

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            content_subtype = 'poly_seq'

            has_poly_seq = has_key_value(input_source_dic, 'polymer_sequence')

            if not has_poly_seq:
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            if self.__star_data_type[fileListId] == 'Loop':

                sf_data = self.__star_data[fileListId]
                sf_framecode = ''

                self.__extractNonStandardResidue__(file_name, sf_framecode, lp_category, input_source)

            elif self.__star_data_type[fileListId] == 'Saveframe':

                sf_data = self.__star_data[fileListId]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__extractNonStandardResidue__(file_name, sf_framecode, lp_category, input_source)

            else:

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__extractNonStandardResidue__(file_name, sf_framecode, lp_category, input_source)

        return True

    def __extractNonStandardResidue__(self, file_name, sf_framecode, lp_category, input_source):
        """ Extract non-standard residue.
        """

        input_source_dic = input_source.get()

        polymer_sequence = input_source_dic['polymer_sequence']

        asm = []

        for s in polymer_sequence:

            has_non_std_comp_id = False

            ent = {'chain_id': s['chain_id'], 'seq_id': [], 'comp_id': [], 'chem_comp_name': [], 'exptl_data': []}

            for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                if self.__get1LetterCode(comp_id) == 'X':
                    has_non_std_comp_id = True

                    ent['seq_id'].append(seq_id)
                    ent['comp_id'].append(comp_id)

                    if self.__updateChemCompDict(comp_id): # matches with comp_id in CCD
                        cc_name = self.__last_chem_comp_dict['_chem_comp.name']
                        cc_rel_status = self.__last_chem_comp_dict['_chem_comp.pdbx_release_status']
                        if cc_rel_status == 'REL':
                            ent['chem_comp_name'].append(cc_name)
                        else:
                            ent['chem_comp_name'].append('(Not available due to CCD status code %s)' % cc_rel_status)

                    else:
                        ent['chem_comp_name'].append(None)

                        warn = 'Non standard residue (%s:%s:%s) did not match with chemical component dictionary (CCD).' % (s['chain_id'], seq_id, comp_id)

                        self.report.warning.appendDescription('ccd_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                        self.report.setWarning()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__extractNonStandardResidue() ++ Warning  - %s\n" % warn)

                    ent['exptl_data'].append({'chem_shift': False, 'dist_restraint': False, 'dihed_restraint': False, 'rdc_restraint': False, 'spectral_peak': False, 'coordinate': False})

            if has_non_std_comp_id:
                asm.append(ent)

        if len(asm) > 0:
            input_source.setItemValue('non_standard_residue', asm)

    def __appendPolymerSequenceAlignment(self):
        """ Append polymer sequence alignment of interesting loops.
        """

        #if self.report.isError():
        #    return False

        is_done = True

        update_poly_seq = False

        self.__alt_chain = False

        self.__valid_seq = False

        if not self.__tolerant_seq_align:
            self.__valid_seq = self.__isConsistentSequence()

            if not self.__valid_seq:
                self.__tolerant_seq_align = True

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            #file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            has_poly_seq = has_key_value(input_source_dic, 'polymer_sequence')
            has_poly_seq_in_loop = has_key_value(input_source_dic, 'polymer_sequence_in_loop')

            if not has_poly_seq:
                """
                err = "Common polymer sequence does not exist, __extractCommonPolymerSequence() should be invoked."

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__appendPolymerSequenceAlignment() ++ Error  - %s" % err)
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__appendPolymerSequenceAlignment() ++ Error  - %s\n" % err)
                """
                is_done = False
                continue

            if not has_poly_seq_in_loop:
                continue

            polymer_sequence = input_source_dic['polymer_sequence']
            polymer_sequence_in_loop = input_source_dic['polymer_sequence_in_loop']

            for content_subtype in polymer_sequence_in_loop.keys():

                seq_align_set = []

                dst_chain_ids = {}
                ref_chain_ids = {}
                map_chain_ids = {}
                map_seq_ids = {}

                proc_chain_ids = {}

                for s1 in polymer_sequence:
                    chain_id = s1['chain_id']

                    for ps_in_loop in polymer_sequence_in_loop[content_subtype]:
                        ps2 = ps_in_loop['polymer_sequence']
                        sf_framecode2 = ps_in_loop['sf_framecode']

                        for s2 in ps2:

                            if sf_framecode2 in ref_chain_ids and chain_id in ref_chain_ids[sf_framecode2]:
                                continue

                            chain_id2 = s2['chain_id']

                            if chain_id != chain_id2:
                                continue

                            _s2 = fill_blank_comp_id_with_offset(s2, 0)

                            if len(_s2['seq_id']) > len(s2['seq_id']) and len(_s2['seq_id']) < len(s1['seq_id']):
                                s2 = _s2

                            self.__pA.setReferenceSequence(s1['comp_id'], 'REF' + chain_id)
                            self.__pA.addTestSequence(s2['comp_id'], chain_id)
                            self.__pA.doAlign()

                            myAlign = self.__pA.getAlignment(chain_id)

                            length = len(myAlign)

                            if length == 0:
                                continue

                            _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                            alt_chain = False

                            if length == unmapped + conflict or _matched <= conflict or (len(polymer_sequence) > 1 and _matched < 4 and offset_1 > 0):

                                if self.__tolerant_seq_align and _matched <= conflict and len(polymer_sequence) > 1:

                                    __length = length
                                    __matched = _matched
                                    __unmapped = unmapped
                                    __conflict = conflict
                                    __chain_id = None
                                    __s1 = None
                                    __offset_1 = None
                                    __offset_2 = None

                                    for _s1 in polymer_sequence:

                                        if _s1 == s1:
                                            continue

                                        chain_id_ = _s1['chain_id']

                                        if sf_framecode2 in ref_chain_ids and chain_id_ in ref_chain_ids[sf_framecode2]:
                                            continue

                                        self.__pA.setReferenceSequence(_s1['comp_id'], 'REF' + chain_id_)
                                        self.__pA.addTestSequence(s2['comp_id'], chain_id_)
                                        self.__pA.doAlign()

                                        myAlign = self.__pA.getAlignment(chain_id_)

                                        length = len(myAlign)

                                        if length == 0:
                                            continue

                                        _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                                        if length == unmapped + conflict or _matched <= conflict:
                                            continue

                                        if _matched - conflict < __matched - __conflict or unmapped + conflict > __unmapped + __conflict:
                                            continue

                                        __length = length
                                        __matched = _matched
                                        __unmapped = unmapped
                                        __conflict = conflict
                                        __chain_id = chain_id_
                                        __offset_1 = offset_1
                                        __offset_2 = offset_2
                                        __s1 = copy.copy(_s1)

                                        alt_chain = True

                                        break

                                if not alt_chain or\
                                   (sf_framecode2 in dst_chain_ids and __chain_id in dst_chain_ids[sf_framecode2]) or\
                                   (sf_framecode2 in map_chain_ids and chain_id in map_chain_ids[sf_framecode2]):
                                    continue

                                if sf_framecode2 not in dst_chain_ids:
                                    dst_chain_ids[sf_framecode2] = set()

                                dst_chain_ids[sf_framecode2].add(__chain_id)

                                if sf_framecode2 not in map_chain_ids:
                                    map_chain_ids[sf_framecode2] = {}

                                map_chain_ids[sf_framecode2][chain_id] = __chain_id

                                if sf_framecode2 == self.__target_framecode:
                                    print('#1 %s -> %s, %s %s %s %s %s %s' % (chain_id, __chain_id, __length, __matched, __unmapped, __conflict, __offset_1, __offset_2))

                                length = __length
                                _matched = __matched
                                unmapped = __unmapped
                                conflict = __conflict
                                chain_id = __s1['chain_id']
                                chain_id = __chain_id
                                offset_1 = __offset_1
                                offset_2 = __offset_2
                                s1 = __s1

                                #s2['chain_id'] = __chain_id

                                update_poly_seq = True

                            if conflict == 0 and self.__alt_chain and not alt_chain and chain_id != s2['chain_id'] and\
                               (sf_framecode2 not in dst_chain_ids or chain_id not in dst_chain_ids[sf_framecode2]) and\
                               (sf_framecode2 not in map_chain_ids or s2['chain_id'] not in map_chain_ids[sf_framecode2]) and\
                               unmapped != offset_1 + 1 and unmapped != offset_2 + 1 and\
                               unmapped <= _matched + offset_1 and unmapped <= _matched + offset_2:

                                if sf_framecode2 not in dst_chain_ids:
                                    dst_chain_ids[sf_framecode2] = set()

                                dst_chain_ids[sf_framecode2].add(chain_id)

                                if sf_framecode2 not in map_chain_ids:
                                    map_chain_ids[sf_framecode2] = {}

                                map_chain_ids[sf_framecode2][s2['chain_id']] = chain_id

                                if sf_framecode2 == self.__target_framecode:
                                    print('#2 %s -> %s, %s %s %s %s %s %s' % (s2['chain_id'], chain_id, length, _matched, unmapped, conflict, offset_1, offset_2))

                                alt_chain = True

                            _s1 = s1 if offset_1 == 0 else fill_blank_comp_id_with_offset(s1, offset_1)
                            _s2 = s2 if offset_2 == 0 else fill_blank_comp_id_with_offset(s2, offset_2)

                            if conflict > 0 and _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:  # pylint: disable=chained-comparison
                                continue

                            if self.__tolerant_seq_align: # and not alt_chain:
                                seq_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                   if __s1 != '.' and __s2 != '.' and __s1 != __s2 and __c1 != '.' and __c2 != '.' and __c1 == __c2)
                                comp_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                    if __c1 != '.' and __c2 != '.' and __c1 != __c2)

                                if 0 < _matched < 4 and unmapped // _matched > 20 and seq_mismatch and len(polymer_sequence) > 1:
                                    continue

                            if not alt_chain:
                                if sf_framecode2 not in dst_chain_ids:
                                    dst_chain_ids[sf_framecode2] = set()

                                dst_chain_ids[sf_framecode2].add(chain_id)

                            if sf_framecode2 not in ref_chain_ids:
                                ref_chain_ids[sf_framecode2] = []

                            ref_chain_ids[sf_framecode2].append(chain_id)

                            ref_length = len(s1['seq_id'])

                            ref_code = self.__get1LetterCodeSequence(_s1['comp_id'])
                            test_code = self.__get1LetterCodeSequence(_s2['comp_id'])
                            mid_code = get_middle_code(ref_code, test_code)
                            ref_gauge_code = get_gauge_code(_s1['seq_id'])
                            test_gauge_code = get_gauge_code(_s2['seq_id'])

                            self.__alt_chain |= alt_chain

                            if self.__tolerant_seq_align and (seq_mismatch or comp_mismatch): # and not alt_chain:
                                if sf_framecode2 not in map_seq_ids:
                                    map_seq_ids[sf_framecode2] = set()
                                map_seq_ids[sf_framecode2].add(chain_id)
                                if _s2['seq_id'] == list(range(_s2['seq_id'][0], _s2['seq_id'][-1] + 1)):
                                    seq_id_conv_dict = {str(__s2): str(__s1) for __s1, __s2 in zip(_s1['seq_id'], _s2['seq_id']) if __s2 != '.'}
                                    if comp_mismatch:
                                        _seq_align = self.__getSeqAlignCode(fileListId, file_type, content_subtype, sf_framecode2,
                                                                            chain_id, _s1, _s2, myAlign, None if sf_framecode2 not in map_chain_ids else map_chain_ids[sf_framecode2],
                                                                            ref_gauge_code, ref_code, mid_code, test_code, test_gauge_code)
                                        _s2['seq_id'] = _seq_align['test_seq_id']
                                        if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                            continue
                                        ref_gauge_code = _seq_align['ref_gauge_code']
                                        ref_code = _seq_align['ref_code']
                                        mid_code = _seq_align['mid_code']
                                        test_code = _seq_align['test_code']
                                        test_gauge_code = _seq_align['test_gauge_code']
                                    else:
                                        #if _s1['seq_id'][0] < 0:
                                        #    continue
                                        chain_id2 = chain_id
                                        if sf_framecode2 in map_chain_ids and chain_id in map_chain_ids[sf_framecode2].values():
                                            chain_id2 = next(k for k, v in map_chain_ids[sf_framecode2].items() if v == chain_id)

                                        if sf_framecode2 == self.__target_framecode:
                                            print('#a %s %s %s %s %s' % (chain_id2, _matched, offset_1, offset_2, seq_id_conv_dict))

                                        if sf_framecode2 not in proc_chain_ids:
                                            proc_chain_ids[sf_framecode2] = set()

                                        if chain_id2 not in proc_chain_ids[sf_framecode2]:
                                            self.__fixSeqIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, chain_id2, seq_id_conv_dict)
                                            proc_chain_ids[sf_framecode2].add(chain_id2)

                                            if 'identical_chain_id' in s2:
                                                for chain_id2_ in s2['identical_chain_id']:
                                                    if chain_id2_ not in proc_chain_ids[sf_framecode2]:
                                                        self.__fixSeqIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, chain_id2_, seq_id_conv_dict)
                                                        proc_chain_ids[sf_framecode2].add(chain_id2_)

                                        _s2['seq_id'] = _s1['seq_id']
                                        mid_code = get_middle_code(ref_code, test_code)
                                        test_gauge_code = ref_gauge_code
                                else:
                                    if seq_mismatch:
                                        _seq_align = self.__getSeqAlignCode(fileListId, file_type, content_subtype, sf_framecode2,
                                                                            chain_id, _s1, _s2, myAlign, None if sf_framecode2 not in map_chain_ids else map_chain_ids[sf_framecode2],
                                                                            ref_gauge_code, ref_code, mid_code, test_code, test_gauge_code)
                                        _s2['seq_id'] = _seq_align['test_seq_id']
                                        if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                            continue
                                        ref_gauge_code = _seq_align['ref_gauge_code']
                                        ref_code = _seq_align['ref_code']
                                        mid_code = _seq_align['mid_code']
                                        test_code = _seq_align['test_code']
                                        test_gauge_code = _seq_align['test_gauge_code']
                                    else:
                                        _s2 = fill_blank_comp_id(_s1, _s2)
                                        if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                            continue
                                        test_code = self.__get1LetterCodeSequence(_s2['comp_id'])
                                        mid_code = get_middle_code(ref_code, test_code)
                                        test_gauge_code = ref_gauge_code

                                update_poly_seq = True
                            """
                            if 'identical_chain_id' in s2 and self.__tolerant_seq_align:

                                for chain_id2_ in s2['identical_chain_id']:

                                    try:

                                        s2_ = next(s2_ for s2_ in ps2 if s2_['chain_id'] == chain_id2_)

                                        _s2_ = fill_blank_comp_id_with_offset(s2_, 0)

                                        if len(_s2_['seq_id']) > len(s2_['seq_id']) and len(_s2_['seq_id']) < len(s1['seq_id']):
                                            s2_ = _s2_

                                        self.__pA.setReferenceSequence(s1['comp_id'], 'REF' + chain_id)
                                        self.__pA.addTestSequence(s2_['comp_id'], chain_id)
                                        self.__pA.doAlign()

                                        myAlign = self.__pA.getAlignment(chain_id)

                                        length = len(myAlign)

                                        if length == 0:
                                            continue

                                        _matched_, unmapped_, conflict_, offset_1_, offset_2_ = score_of_seq_align(myAlign)

                                        seq_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2_['seq_id'], _s1['comp_id'], _s2_['comp_id'])\
                                                           if __s1 != '.' and __s2 != '.' and __s1 != __s2 and __c1 != '.' and __c2 != '.' and __c1 == __c2)

                                        comp_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2_['seq_id'], _s1['comp_id'], _s2_['comp_id'])\
                                                            if __c1 != '.' and __c2 != '.' and __c1 != __c2)

                                        if seq_mismatch and comp_mismatch: # and not alt_chain:

                                            if _s2_['seq_id'] == list(range(_s2_['seq_id'][0], _s2_['seq_id'][-1] + 1)):
                                                seq_id_conv_dict = {str(__s2): str(__s1) for __s1, __s2 in zip(_s1['seq_id'], _s2_['seq_id']) if __s2 != '.'}
                                                if sf_framecode2 == self.__target_framecode:
                                                    print('#d %s %s %s %s %s' % (chain_id2_, _matched_, offset_1_, offset_2_, seq_id_conv_dict))

                                                if sf_framecode2 not in proc_chain_ids:
                                                    proc_chain_ids[sf_framecode2] = set()

                                                if chain_id2_ not in proc_chain_ids[sf_framecode2]:
                                                    self.__fixSeqIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, chain_id2_, seq_id_conv_dict)
                                                    proc_chain_ids[sf_framecode2].add(chain_id2_)
                                                _s2_['seq_id'] = _s1['seq_id']

                                            update_poly_seq = True

                                    except StopIteration:
                                        pass
                            """
                            matched = mid_code.count('|')

                            seq_align = {'list_id': ps_in_loop['list_id'],
                                         'sf_framecode': sf_framecode2,
                                         'chain_id': chain_id, 'length': ref_length, 'matched': matched, 'conflict': conflict, 'unmapped': unmapped,
                                         'sequence_coverage': float('{:.3f}'.format(float(length - (unmapped + conflict)) / float(ref_length))),
                                         'ref_seq_id': _s1['seq_id'], 'test_seq_id': _s2['seq_id'],
                                         'ref_gauge_code': ref_gauge_code, 'ref_code': ref_code, 'mid_code': mid_code, 'test_code': test_code, 'test_gauge_code': test_gauge_code}

                            if seq_align in seq_align_set:
                                continue

                            seq_align_set.append(seq_align)

                            if not self.__combined_mode and input_source_dic['non_standard_residue'] is None: # no polymer sequence
                                has_non_std_comp_id = False
                                for j, rc in enumerate(ref_code):
                                    if rc == 'X' and j < len(test_code) and test_code[j] == 'X':
                                        has_non_std_comp_id = True
                                        break

                                if not has_non_std_comp_id:
                                    continue

                                asm = []

                                for _s in polymer_sequence:

                                    ent = {'chain_id': _s['chain_id'], 'seq_id': [], 'comp_id': [], 'chem_comp_name': [], 'exptl_data': []}

                                    for _seq_id, _comp_id in zip(_s['seq_id'], _s['comp_id']):
                                        if self.__get1LetterCode(_comp_id) == 'X':

                                            ent['seq_id'].append(_seq_id)
                                            ent['comp_id'].append(_comp_id)

                                            if self.__updateChemCompDict(_comp_id): # matches with comp_id in CCD
                                                cc_name = self.__last_chem_comp_dict['_chem_comp.name']
                                                cc_rel_status = self.__last_chem_comp_dict['_chem_comp.pdbx_release_status']
                                                if cc_rel_status == 'REL':
                                                    ent['chem_comp_name'].append(cc_name)
                                                else:
                                                    ent['chem_comp_name'].append('(Not available due to CCD status code %s)' % cc_rel_status)

                                            else:
                                                ent['chem_comp_name'].append(None)

                                            ent['exptl_data'].append({'coordinate': False})

                                    asm.append(ent)

                                input_source.setItemValue('non_standard_residue', asm)

                            for r_code, t_code, seq_id in zip(ref_code, test_code, s1['seq_id']):
                                if r_code == 'X' and t_code == 'X':
                                    input_source.updateNonStandardResidueByExptlData(chain_id, seq_id, content_subtype)

                for s1 in polymer_sequence:

                    chain_id = s1['chain_id']

                    for ps_in_loop in polymer_sequence_in_loop[content_subtype]:
                        ps2 = ps_in_loop['polymer_sequence']
                        sf_framecode2 = ps_in_loop['sf_framecode']

                        for s2 in ps2:

                            if sf_framecode2 in ref_chain_ids and chain_id in ref_chain_ids[sf_framecode2]:
                                continue

                            chain_id2 = s2['chain_id']

                            if sf_framecode2 in dst_chain_ids and chain_id2 in dst_chain_ids[sf_framecode2]:
                                continue

                            if chain_id != chain_id2 and not self.__tolerant_seq_align:
                                continue

                            _s2 = fill_blank_comp_id_with_offset(s2, 0)

                            if len(_s2['seq_id']) > len(s2['seq_id']) and len(_s2['seq_id']) < len(s1['seq_id']):
                                s2 = _s2

                            self.__pA.setReferenceSequence(s1['comp_id'], 'REF' + chain_id)
                            self.__pA.addTestSequence(s2['comp_id'], chain_id)
                            self.__pA.doAlign()

                            myAlign = self.__pA.getAlignment(chain_id)

                            length = len(myAlign)

                            if length == 0:
                                continue

                            _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                            alt_chain = False

                            if length == unmapped + conflict or _matched <= conflict or (len(polymer_sequence) > 1 and _matched < 4 and offset_1 > 0):

                                if self.__tolerant_seq_align and _matched <= conflict and len(polymer_sequence) > 1:

                                    __length = length
                                    __matched = _matched
                                    __unmapped = unmapped
                                    __conflict = conflict
                                    __chain_id = None
                                    __s1 = None
                                    __offset_1 = None
                                    __offset_2 = None

                                    for _s1 in polymer_sequence:

                                        if _s1 == s1:
                                            continue

                                        chain_id_ = _s1['chain_id']

                                        if sf_framecode2 in ref_chain_ids and chain_id_ in ref_chain_ids[sf_framecode2]:
                                            continue

                                        self.__pA.setReferenceSequence(_s1['comp_id'], 'REF' + chain_id_)
                                        self.__pA.addTestSequence(s2['comp_id'], chain_id_)
                                        self.__pA.doAlign()

                                        myAlign = self.__pA.getAlignment(chain_id_)

                                        length = len(myAlign)

                                        if length == 0:
                                            continue

                                        _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                                        if length == unmapped + conflict or _matched <= conflict:
                                            continue

                                        if _matched - conflict < __matched - __conflict or (unmapped + conflict > __unmapped + __conflict and __matched > 0):
                                            continue

                                        __length = length
                                        __matched = _matched
                                        __unmapped = unmapped
                                        __conflict = conflict
                                        __chain_id = chain_id_
                                        __offset_1 = offset_1
                                        __offset_2 = offset_2
                                        __s1 = copy.copy(_s1)

                                        alt_chain = True

                                        break

                                if not alt_chain or\
                                   (sf_framecode2 in dst_chain_ids and __chain_id in dst_chain_ids[sf_framecode2]) or\
                                   (sf_framecode2 in map_chain_ids and chain_id in map_chain_ids[sf_framecode2]):
                                    continue

                                if sf_framecode2 not in dst_chain_ids:
                                    dst_chain_ids[sf_framecode2] = set()

                                dst_chain_ids[sf_framecode2].add(__chain_id)

                                if sf_framecode2 not in map_chain_ids:
                                    map_chain_ids[sf_framecode2] = {}

                                map_chain_ids[sf_framecode2][chain_id] = __chain_id

                                if sf_framecode2 == self.__target_framecode:
                                    print('#3 %s -> %s, %s %s %s %s %s %s' % (chain_id, __chain_id, __length, __matched, __unmapped, __conflict, __offset_1, __offset_2))

                                length = __length
                                _matched = __matched
                                unmapped = __unmapped
                                conflict = __conflict
                                chain_id = __s1['chain_id']
                                chain_id = __chain_id
                                offset_1 = __offset_1
                                offset_2 = __offset_2
                                s1 = __s1

                                #s2['chain_id'] = __chain_id

                                update_poly_seq = True
                            """
                            if conflict == 0 and self.__alt_chain and not alt_chain and chain_id != s2['chain_id'] and\
                               (sf_framecode2 not in dst_chain_ids or chain_id not in dst_chain_ids[sf_framecode2]) and\
                               (sf_framecode2 not in map_chain_ids or s2['chain_id'] not in map_chain_ids[sf_framecode2]) and\
                               unmapped != offset_1 + 1 and unmapped != offset_2 + 1:

                                if sf_framecode2 not in dst_chain_ids:
                                    dst_chain_ids[sf_framecode2] = set()

                                dst_chain_ids[sf_framecode2].add(chain_id)

                                if sf_framecode2 not in map_chain_ids:
                                    map_chain_ids[sf_framecode2] = {}

                                map_chain_ids[sf_framecode2][s2['chain_id']] = chain_id

                                if sf_framecode2 == self.__target_framecode:
                                    print('#4 %s -> %s, %s %s %s %s %s %s' % (s2['chain_id'], chain_id, length, _matched, unmapped, conflict, offset_1, offset_2))

                                alt_chain = True
                            """
                            _s1 = s1 if offset_1 == 0 else fill_blank_comp_id_with_offset(s1, offset_1)
                            _s2 = s2 if offset_2 == 0 else fill_blank_comp_id_with_offset(s2, offset_2)

                            if conflict > 0 and _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:  # pylint: disable=chained-comparison
                                continue

                            if self.__tolerant_seq_align: # and not alt_chain:
                                seq_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                   if __s1 != '.' and __s2 != '.' and __s1 != __s2 and __c1 != '.' and __c2 != '.' and __c1 == __c2)
                                comp_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                    if __c1 != '.' and __c2 != '.' and __c1 != __c2)

                                if 0 < _matched < 4 and unmapped // _matched > 20 and seq_mismatch and len(polymer_sequence) > 1:
                                    continue

                            if sf_framecode2 not in ref_chain_ids:
                                ref_chain_ids[sf_framecode2] = []

                            if sf_framecode2 not in map_chain_ids:
                                map_chain_ids[sf_framecode2] = {}

                            if sf_framecode2 not in ref_chain_ids or chain_id not in ref_chain_ids[sf_framecode2]:
                                map_chain_ids[sf_framecode2][s2['chain_id']] = chain_id

                            ref_chain_ids[sf_framecode2].append(chain_id)

                            ref_length = len(s1['seq_id'])

                            ref_code = self.__get1LetterCodeSequence(_s1['comp_id'])
                            test_code = self.__get1LetterCodeSequence(_s2['comp_id'])
                            mid_code = get_middle_code(ref_code, test_code)
                            ref_gauge_code = get_gauge_code(_s1['seq_id'])
                            test_gauge_code = get_gauge_code(_s2['seq_id'])

                            self.__alt_chain |= not alt_chain
                            """
                            if self.__tolerant_seq_align: # and not alt_chain:
                                seq_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                   if __s1 != '.' and __s2 != '.' and __s1 != __s2 and __c1 != '.' and __c2 != '.' and __c1 == __c2)
                                comp_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                    if __c1 != '.' and __c2 != '.' and __c1 != __c2)

                            if self.__tolerant_seq_align and (seq_mismatch or comp_mismatch): # and not alt_chain:
                                if sf_framecode2 not in map_seq_ids:
                                    map_seq_ids[sf_framecode2] = set()
                                map_seq_ids[sf_framecode2].add(chain_id)
                                if _s2['seq_id'] == list(range(_s2['seq_id'][0], _s2['seq_id'][-1] + 1)):
                                    seq_id_conv_dict = {str(__s2): str(__s1) for __s1, __s2 in zip(_s1['seq_id'], _s2['seq_id']) if __s2 != '.'}
                                    if comp_mismatch:
                                        _seq_align = self.__getSeqAlignCode(fileListId, file_type, content_subtype, sf_framecode2,
                                                                            chain_id, _s1, _s2, myAlign, None if sf_framecode2 not in map_chain_ids else map_chain_ids[sf_framecode2],
                                                                            ref_gauge_code, ref_code, mid_code, test_code, test_gauge_code)
                                        _s2['seq_id'] = _seq_align['test_seq_id']
                                        if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                            continue
                                        ref_gauge_code = _seq_align['ref_gauge_code']
                                        ref_code = _seq_align['ref_code']
                                        mid_code = _seq_align['mid_code']
                                        test_code = _seq_align['test_code']
                                        test_gauge_code = _seq_align['test_gauge_code']
                                    else:
                                        #if _s1['seq_id'][0] < 0:
                                        #    continue
                                        chain_id2 = chain_id
                                        if sf_framecode2 in map_chain_ids and chain_id in map_chain_ids[sf_framecode2].values():
                                            chain_id2 = next(k for k, v in map_chain_ids[sf_framecode2].items() if v == chain_id)

                                        if sf_framecode2 == self.__target_framecode:
                                            print('#b %s %s %s %s %s' % (chain_id2, _matched, offset_1, offset_2, seq_id_conv_dict))

                                        if sf_framecode2 not in proc_chain_ids:
                                            proc_chain_ids[sf_framecode2] = set()

                                        if chain_id2 not in proc_chain_ids[sf_framecode2]:
                                            self.__fixSeqIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, chain_id2, seq_id_conv_dict)
                                            proc_chain_ids[sf_framecode2].add(chain_id2)

                                            if 'identical_chain_id' in s2:
                                                for chain_id2_ in s2['identical_chain_id']:
                                                    if chain_id2_ not in proc_chain_ids[sf_framecode2]:
                                                        self.__fixSeqIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, chain_id2_, seq_id_conv_dict)
                                                        proc_chain_ids[sf_framecode2].add(chain_id2_)

                                        _s2['seq_id'] = _s1['seq_id']
                                        mid_code = get_middle_code(ref_code, test_code)
                                        test_gauge_code = ref_gauge_code
                                else:
                                    if seq_mismatch:
                                        _seq_align = self.__getSeqAlignCode(fileListId, file_type, content_subtype, sf_framecode2,
                                                                            chain_id, _s1, _s2, myAlign, None if sf_framecode2 not in map_chain_ids else map_chain_ids[sf_framecode2],
                                                                            ref_gauge_code, ref_code, mid_code, test_code, test_gauge_code)
                                        _s2['seq_id'] = _seq_align['test_seq_id']
                                        if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                            continue
                                        ref_gauge_code = _seq_align['ref_gauge_code']
                                        ref_code = _seq_align['ref_code']
                                        mid_code = _seq_align['mid_code']
                                        test_code = _seq_align['test_code']
                                        test_gauge_code = _seq_align['test_gauge_code']
                                    else:
                                        _s2 = fill_blank_comp_id(_s1, _s2)
                                        if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                            continue
                                        test_code = self.__get1LetterCodeSequence(_s2['comp_id'])
                                        mid_code = get_middle_code(ref_code, test_code)
                                        test_gauge_code = ref_gauge_code

                                update_poly_seq = True
                            """
                            matched = mid_code.count('|')

                            seq_align = {'list_id': ps_in_loop['list_id'],
                                         'sf_framecode': sf_framecode2,
                                         'chain_id': chain_id, 'length': ref_length, 'matched': matched, 'conflict': conflict, 'unmapped': unmapped,
                                         'sequence_coverage': float('{:.3f}'.format(float(length - (unmapped + conflict)) / float(ref_length))),
                                         'ref_seq_id': _s1['seq_id'], 'test_seq_id': _s2['seq_id'],
                                         'ref_gauge_code': ref_gauge_code, 'ref_code': ref_code, 'mid_code': mid_code, 'test_code': test_code, 'test_gauge_code': test_gauge_code}

                            if seq_align in seq_align_set:
                                continue

                            seq_align_set.append(seq_align)

                            if not self.__combined_mode and input_source_dic['non_standard_residue'] is None: # no polymer sequence
                                has_non_std_comp_id = False
                                for j, rc in enumerate(ref_code):
                                    if rc == 'X' and j < len(test_code) and test_code[j] == 'X':
                                        has_non_std_comp_id = True
                                        break

                                if not has_non_std_comp_id:
                                    continue

                                asm = []

                                for _s in polymer_sequence:

                                    ent = {'chain_id': _s['chain_id'], 'seq_id': [], 'comp_id': [], 'chem_comp_name': [], 'exptl_data': []}

                                    for _seq_id, _comp_id in zip(_s['seq_id'], _s['comp_id']):
                                        if self.__get1LetterCode(_comp_id) == 'X':

                                            ent['seq_id'].append(_seq_id)
                                            ent['comp_id'].append(_comp_id)

                                            if self.__updateChemCompDict(_comp_id): # matches with comp_id in CCD
                                                cc_name = self.__last_chem_comp_dict['_chem_comp.name']
                                                cc_rel_status = self.__last_chem_comp_dict['_chem_comp.pdbx_release_status']
                                                if cc_rel_status == 'REL':
                                                    ent['chem_comp_name'].append(cc_name)
                                                else:
                                                    ent['chem_comp_name'].append('(Not available due to CCD status code %s)' % cc_rel_status)

                                            else:
                                                ent['chem_comp_name'].append(None)

                                            ent['exptl_data'].append({'coordinate': False})

                                    asm.append(ent)

                                input_source.setItemValue('non_standard_residue', asm)

                            for r_code, t_code, seq_id in zip(ref_code, test_code, s1['seq_id']):
                                if r_code == 'X' and t_code == 'X':
                                    input_source.updateNonStandardResidueByExptlData(chain_id, seq_id, content_subtype)

                if len(seq_align_set) > 0:
                    self.report.sequence_alignment.setItemValue('nmr_poly_seq_vs_' + content_subtype, seq_align_set)

                if self.__alt_chain:

                    for ps_in_loop in polymer_sequence_in_loop[content_subtype]:
                        ps2 = ps_in_loop['polymer_sequence']
                        sf_framecode2 = ps_in_loop['sf_framecode']

                        if sf_framecode2 in map_chain_ids:
                            mapping = map_chain_ids[sf_framecode2]

                            total = set(mapping.keys()) | set(mapping.values())

                            k_rests = list(total - set(mapping.keys()))
                            v_rests = list(total - set(mapping.values()))

                            circular = False
                            cross = False

                            for k, v in mapping.items():
                                for _k, _v in mapping.items():
                                    if v == _k:
                                        circular = True
                                        break
                                if circular:
                                    break

                            if len(k_rests) == 1 and len(v_rests) == 1:

                                src_chain = k_rests[0]
                                dst_chain = v_rests[0]

                                if circular:
                                    mapping[src_chain] = dst_chain
                                    """
                                    for s1 in polymer_sequence:
                                        chain_id = s1['chain_id']

                                        if chain_id != dst_chain:
                                            continue

                                        for s2 in ps2:

                                            if chain_id != src_chain:
                                                continue

                                            _s2 = fill_blank_comp_id_with_offset(s2, 0)

                                            if len(_s2['seq_id']) > len(s2['seq_id']) and len(_s2['seq_id']) < len(s1['seq_id']):
                                                s2 = _s2

                                            self.__pA.setReferenceSequence(s1['comp_id'], 'REF' + chain_id)
                                            self.__pA.addTestSequence(s2['comp_id'], chain_id)
                                            self.__pA.doAlign()

                                            myAlign = self.__pA.getAlignment(chain_id)

                                            length = len(myAlign)

                                            if length == 0:
                                                break

                                            _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                                            if length == unmapped + conflict or _matched <= conflict:
                                                break

                                            mapping[src_chain] = dst_chain

                                            break
                                    """

                                else:

                                    for s1 in polymer_sequence:
                                        chain_id = s1['chain_id']

                                        if chain_id != dst_chain:
                                            continue

                                        for s2 in ps2:

                                            #if chain_id != dst_chain:
                                            #    continue

                                            _s2 = fill_blank_comp_id_with_offset(s2, 0)

                                            if len(_s2['seq_id']) > len(s2['seq_id']) and len(_s2['seq_id']) < len(s1['seq_id']):
                                                s2 = _s2

                                            self.__pA.setReferenceSequence(s1['comp_id'], 'REF' + chain_id)
                                            self.__pA.addTestSequence(s2['comp_id'], chain_id)
                                            self.__pA.doAlign()

                                            myAlign = self.__pA.getAlignment(chain_id)

                                            length = len(myAlign)

                                            if length == 0:
                                                break

                                            _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                                            if length == unmapped + conflict or _matched <= conflict:
                                                break

                                            cross = True
                                            mapping[src_chain] = dst_chain

                                            break

                            if sf_framecode2 == self.__target_framecode:
                                print('chain_mapping %s cross %s cicular %s' % (mapping, cross, circular))

                            for s1 in polymer_sequence:
                                chain_id = s1['chain_id']

                                for s2 in ps2:

                                    if chain_id != s2['chain_id']:
                                        continue

                                    _s2 = fill_blank_comp_id_with_offset(s2, 0)

                                    if len(_s2['seq_id']) > len(s2['seq_id']) and len(_s2['seq_id']) < len(s1['seq_id']):
                                        s2 = _s2

                                    self.__pA.setReferenceSequence(s1['comp_id'], 'REF' + chain_id)
                                    self.__pA.addTestSequence(s2['comp_id'], chain_id)
                                    self.__pA.doAlign()

                                    myAlign = self.__pA.getAlignment(chain_id)

                                    length = len(myAlign)

                                    if length == 0:
                                        continue

                                    _matched, unmapped, conflict, offset_1, offset_2 = score_of_seq_align(myAlign)

                                    if length == unmapped + conflict or _matched <= conflict:
                                        continue

                                    _s1 = s1 if offset_1 == 0 else fill_blank_comp_id_with_offset(s1, offset_1)
                                    _s2 = s2 if offset_2 == 0 else fill_blank_comp_id_with_offset(s2, offset_2)

                                    if conflict > 0 and _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:  # pylint: disable=chained-comparison
                                        continue

                                    if self.__tolerant_seq_align:
                                        seq_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                           if __s1 != '.' and __s2 != '.' and __s1 != __s2 and __c1 != '.' and __c2 != '.' and __c1 == __c2)
                                        comp_mismatch = any((__s1, __s2) for (__s1, __s2, __c1, __c2) in zip(_s1['seq_id'], _s2['seq_id'], _s1['comp_id'], _s2['comp_id'])\
                                                            if __c1 != '.' and __c2 != '.' and __c1 != __c2)

                                        if 0 < _matched < 4 and unmapped // _matched > 20 and seq_mismatch and len(polymer_sequence) > 1:
                                            continue

                                    ref_length = len(s1['seq_id'])

                                    ref_code = self.__get1LetterCodeSequence(_s1['comp_id'])
                                    test_code = self.__get1LetterCodeSequence(_s2['comp_id'])
                                    mid_code = get_middle_code(ref_code, test_code)
                                    ref_gauge_code = get_gauge_code(_s1['seq_id'])
                                    test_gauge_code = get_gauge_code(_s2['seq_id'])

                                    if self.__tolerant_seq_align and (seq_mismatch or comp_mismatch):
                                        if sf_framecode2 in map_seq_ids and chain_id in map_seq_ids[sf_framecode2]:
                                            continue
                                        if _s2['seq_id'] == list(range(_s2['seq_id'][0], _s2['seq_id'][-1] + 1)):
                                            seq_id_conv_dict = {str(__s2): str(__s1) for __s1, __s2 in zip(_s1['seq_id'], _s2['seq_id']) if __s2 != '.'}
                                            if comp_mismatch:
                                                _seq_align = self.__getSeqAlignCode(fileListId, file_type, content_subtype, sf_framecode2,
                                                                                    chain_id, _s1, _s2, myAlign, mapping,
                                                                                    ref_gauge_code, ref_code, mid_code, test_code, test_gauge_code)
                                                _s2['seq_id'] = _seq_align['test_seq_id']
                                                if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                                    continue
                                                ref_gauge_code = _seq_align['ref_gauge_code']
                                                ref_code = _seq_align['ref_code']
                                                mid_code = _seq_align['mid_code']
                                                test_code = _seq_align['test_code']
                                                test_gauge_code = _seq_align['test_gauge_code']
                                            else:
                                                #if _s1['seq_id'][0] < 0:
                                                #    continue
                                                chain_id2 = chain_id
                                                if chain_id in mapping.values():
                                                    chain_id2 = next(k for k, v in mapping.items() if v == chain_id)

                                                if sf_framecode2 == self.__target_framecode:
                                                    print('#c %s %s %s %s %s' % (chain_id2, _matched, offset_1, offset_2, seq_id_conv_dict))

                                                if sf_framecode2 not in proc_chain_ids:
                                                    proc_chain_ids[sf_framecode2] = set()

                                                if chain_id2 not in proc_chain_ids[sf_framecode2]:
                                                    self.__fixSeqIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, chain_id2, seq_id_conv_dict)
                                                    proc_chain_ids[sf_framecode2].add(chain_id2)

                                                    if 'identical_chain_id' in s2:
                                                        for chain_id2_ in s2['identical_chain_id']:
                                                            if chain_id2_ not in proc_chain_ids[sf_framecode2]:
                                                                self.__fixSeqIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, chain_id2_, seq_id_conv_dict)
                                                                proc_chain_ids[sf_framecode2].add(chain_id2_)

                                                _s2['seq_id'] = _s1['seq_id']
                                                mid_code = get_middle_code(ref_code, test_code)
                                                test_gauge_code = ref_gauge_code
                                        else:
                                            if seq_mismatch:
                                                _seq_align = self.__getSeqAlignCode(fileListId, file_type, content_subtype, sf_framecode2,
                                                                                    chain_id, _s1, _s2, myAlign, mapping,
                                                                                    ref_gauge_code, ref_code, mid_code, test_code, test_gauge_code)
                                                _s2['seq_id'] = _seq_align['test_seq_id']
                                                if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                                    continue
                                                ref_gauge_code = _seq_align['ref_gauge_code']
                                                ref_code = _seq_align['ref_code']
                                                mid_code = _seq_align['mid_code']
                                                test_code = _seq_align['test_code']
                                                test_gauge_code = _seq_align['test_gauge_code']
                                            else:
                                                _s2 = fill_blank_comp_id(_s1, _s2)
                                                if _s1['seq_id'][0] < 0 and _s2['seq_id'][0] < 0:
                                                    continue
                                                test_code = self.__get1LetterCodeSequence(_s2['comp_id'])
                                                mid_code = get_middle_code(ref_code, test_code)
                                                test_gauge_code = ref_gauge_code

                                    matched = mid_code.count('|')

                                    _seq_align = next((_seq_align for _seq_align in seq_align_set\
                                                       if _seq_align['list_id'] == ps_in_loop['list_id'] and _seq_align['sf_framecode'] == sf_framecode2 and _seq_align['chain_id'] == chain_id), None)

                                    if _seq_align is not None:
                                        seq_align_set.remove(_seq_align)

                                    seq_align = {'list_id': ps_in_loop['list_id'],
                                                 'sf_framecode': sf_framecode2,
                                                 'chain_id': chain_id, 'length': ref_length, 'matched': matched, 'conflict': conflict, 'unmapped': unmapped,
                                                 'sequence_coverage': float('{:.3f}'.format(float(length - (unmapped + conflict)) / float(ref_length))),
                                                 'ref_seq_id': _s1['seq_id'], 'test_seq_id': _s2['seq_id'],
                                                 'ref_gauge_code': ref_gauge_code, 'ref_code': ref_code, 'mid_code': mid_code, 'test_code': test_code, 'test_gauge_code': test_gauge_code}

                                    if seq_align in seq_align_set:
                                        continue

                                    seq_align_set.append(seq_align)

                            if circular or cross:
                                for k, v in mapping.items():

                                    for s2 in ps2:

                                        if s2['chain_id'] != k:
                                            continue

                                        s2['chain_id'] = v + '_'

                                        break

                                    self.__fixChainIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, k, v + '_')

                                for v in mapping.values():

                                    for s2 in ps2:

                                        if s2['chain_id'] != v + '_':
                                            continue

                                        s2['chain_id'] = v

                                        break

                                    self.__fixChainIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, v + '_', v)

                            else:
                                for k, v in mapping.items():

                                    for s2 in ps2:

                                        if s2['chain_id'] != k:
                                            continue

                                        s2['chain_id'] = v

                                        break

                                    self.__fixChainIdInLoop(fileListId, file_type, content_subtype, sf_framecode2, k, v)

        if update_poly_seq:
            self.__extractPolymerSequenceInLoop()
            self.__depositNmrData()

        return is_done

    def __getSeqAlignCode(self, file_list_id, file_type, content_subtype, sf_framecode, chain_id, s1, s2, myAlign, mapping,
                          ref_gauge_code, ref_code, mid_code, test_code, test_gauge_code):
        """ Return human-readable seq align codes.
        """

        length = len(myAlign)

        seq_id1 = []
        seq_id2 = []
        comp_id1 = []
        comp_id2 = []
        idx1 = 0
        idx2 = 0
        for i in range(length):
            myPr = myAlign[i]
            myPr0 = str(myPr[0])
            myPr1 = str(myPr[1])
            if myPr0 != '.':
                while idx1 < len(s1['seq_id']):
                    if s1['comp_id'][idx1] == myPr0:
                        seq_id1.append(s1['seq_id'][idx1])
                        comp_id1.append(myPr0)
                        idx1 += 1
                        break
                    idx1 += 1
            else:
                seq_id1.append(None)
                comp_id1.append('.')
            if myPr1 != '.':
                while idx2 < len(s2['seq_id']):
                    if s2['comp_id'][idx2] == myPr1:
                        seq_id2.append(s2['seq_id'][idx2])
                        comp_id2.append(myPr1)
                        idx2 += 1
                        break
                    idx2 += 1
            else:
                seq_id2.append(None)
                comp_id2.append('.')
        seq_id_conv_dict = {str(_s2): str(_s1) for _s1, _s2 in zip(seq_id1, seq_id2) if _s1 is not None and _s2 is not None}
        if s1['seq_id'] != list(range(s1['seq_id'][0], s1['seq_id'][-1] + 1)) and not any(k for k in seq_id_conv_dict.keys() if seq_id_conv_dict[k] != k):
            s2['seq_id'] = s1['seq_id']
            ref_code = test_code
            mid_code = get_middle_code(ref_code, test_code)
            ref_gauge_code = test_gauge_code
        else:
            chain_id2 = chain_id
            if mapping is not None and chain_id in mapping.values():
                chain_id2 = next(k for k, v in mapping.items() if v == chain_id)
            self.__fixSeqIdInLoop(file_list_id, file_type, content_subtype, sf_framecode, chain_id2, seq_id_conv_dict)
            s2['seq_id'] = s1['seq_id']
            ref_code = self.__get1LetterCodeSequence(comp_id1)
            test_code = self.__get1LetterCodeSequence(comp_id2)
            mid_code = get_middle_code(ref_code, test_code)
            ref_gauge_code = get_gauge_code(seq_id1)
            test_gauge_code = ref_gauge_code
            if ' ' in ref_gauge_code:
                for p, g in enumerate(ref_gauge_code):
                    if g == ' ':
                        ref_code = ref_code[0:p] + '-' + ref_code[p + 1:]
            if ' ' in test_gauge_code:
                for p, g in enumerate(test_gauge_code):
                    if g == ' ':
                        test_code = test_code[0:p] + '-' + test_code[p + 1:]

        return {'ref_seq_id': s1['seq_id'], 'test_seq_id': s2['seq_id'],
                'ref_gauge_code': ref_gauge_code, 'ref_code': ref_code, 'mid_code': mid_code, 'test_code': test_code, 'test_gauge_code': test_gauge_code}

    def __fixChainIdInLoop(self, file_list_id, file_type, content_subtype, sf_framecode, chain_id, _chain_id):
        """ Fix chain ID of interesting loop.
        """

        sf_category = self.sf_categories[file_type][content_subtype]
        lp_category = self.lp_categories[file_type][content_subtype]

        if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
            lp_category = '_Assigned_peak_chem_shift'

        if self.__star_data_type[file_list_id] == 'Loop':

            sf_data = self.__star_data[file_list_id]

            if sf_framecode == '':
                self.__fixChainIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, _chain_id)

        elif self.__star_data_type[file_list_id] == 'Saveframe':

            sf_data = self.__star_data[file_list_id]

            if get_first_sf_tag(sf_data, 'sf_framecode') == sf_framecode:
                self.__fixChainIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, _chain_id)

        else:

            for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                if get_first_sf_tag(sf_data, 'sf_framecode') != sf_framecode:
                    continue

                if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                    continue

                self.__fixChainIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, _chain_id)

    def __fixChainIdInLoop__(self, file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, _chain_id):
        """ Fix sequence ID of interesting loop.
        """

        uniq_chains = self.report.getChainIdsForSameEntity() is None

        chain_id_name = 'chain_code' if file_type == 'nef' else 'Entity_assembly_ID'
        entity_id_name = None if file_type == 'nef' else 'Entity_ID'

        max_dim = 2

        if content_subtype in ('poly_seq', 'dist_restraint', 'rdc_restraint'):
            max_dim = 3

        elif content_subtype == 'dihed_restraint':
            max_dim = 5

        elif content_subtype == 'spectral_peak':

            try:

                _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                num_dim = int(_num_dim)

                if num_dim not in range(1, self.lim_num_dim):
                    raise ValueError()

            except ValueError: # raised error already at __testIndexConsistency()
                return

            max_dim = num_dim + 1

        if __pynmrstar_v3_2__:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop(lp_category)
        else:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop_by_category(lp_category)

        if max_dim == 2:

            chain_id_col = loop.tags.index(chain_id_name) if chain_id_name in loop.tags else -1
            entity_id_col = -1
            if entity_id_name is not None:
                entity_id_col = loop.tags.index(entity_id_name) if entity_id_name in loop.tags else -1

            if chain_id_col == -1:
                return

            for row in loop.data:

                if row[chain_id_col] != chain_id:
                    continue

                row[chain_id_col] = _chain_id

                if uniq_chains and entity_id_col != -1:
                    row[entity_id_col] = _chain_id

        else:

            for i in range(1, max_dim):

                _chain_id_name = chain_id_name + '_' + str(i)
                _entity_id_name = None if entity_id_name is None else entity_id_name + '_' + str(i)

                chain_id_col = loop.tags.index(_chain_id_name) if _chain_id_name in loop.tags else -1
                entity_id_col = -1
                if _entity_id_name is not None:
                    entity_id_col = loop.tags.index(_entity_id_name) if _entity_id_name in loop.tags else -1

                if chain_id_col == -1:
                    continue

                for row in loop.data:

                    if row[chain_id_col] != chain_id:
                        continue

                    row[chain_id_col] = _chain_id

                    if uniq_chains and entity_id_col != -1:
                        row[entity_id_col] = _chain_id

    def __fixSeqIdInLoop(self, file_list_id, file_type, content_subtype, sf_framecode, chain_id, seq_id_conv_dict):
        """ Fix sequence ID of interesting loop.
        """

        sf_category = self.sf_categories[file_type][content_subtype]
        lp_category = self.lp_categories[file_type][content_subtype]

        if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
            lp_category = '_Assigned_peak_chem_shift'

        if self.__star_data_type[file_list_id] == 'Loop':

            sf_data = self.__star_data[file_list_id]

            if sf_framecode == '':
                self.__fixSeqIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id_conv_dict)

        elif self.__star_data_type[file_list_id] == 'Saveframe':

            sf_data = self.__star_data[file_list_id]

            if get_first_sf_tag(sf_data, 'sf_framecode') == sf_framecode:
                self.__fixSeqIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id_conv_dict)

        else:

            for sf_data in self.__star_data[file_list_id].get_saveframes_by_category(sf_category):

                if get_first_sf_tag(sf_data, 'sf_framecode') != sf_framecode:
                    continue

                if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                    continue

                self.__fixSeqIdInLoop__(file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id_conv_dict)

    def __fixSeqIdInLoop__(self, file_list_id, file_type, content_subtype, sf_data, lp_category, chain_id, seq_id_conv_dict):
        """ Fix sequence ID of interesting loop.
        """

        chain_id_name = 'chain_code' if file_type == 'nef' else 'Entity_assembly_ID'
        seq_id_name = 'sequence_code' if file_type == 'nef' else 'Comp_index_ID'
        seq_id_alt_name = None if file_type == 'nef' else 'Seq_ID'

        max_dim = 2

        if content_subtype in ('poly_seq', 'dist_restraint', 'rdc_restraint'):
            max_dim = 3

        elif content_subtype == 'dihed_restraint':
            max_dim = 5

        elif content_subtype == 'spectral_peak':

            try:

                _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                num_dim = int(_num_dim)

                if num_dim not in range(1, self.lim_num_dim):
                    raise ValueError()

            except ValueError: # raised error already at __testIndexConsistency()
                return

            max_dim = num_dim + 1

        if __pynmrstar_v3_2__:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop(lp_category)
        else:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop_by_category(lp_category)

        if max_dim == 2:

            chain_id_col = loop.tags.index(chain_id_name) if chain_id_name in loop.tags else -1
            seq_id_col = loop.tags.index(seq_id_name) if seq_id_name in loop.tags else -1
            seq_id_alt_col = -1
            if seq_id_alt_name is not None:
                seq_id_alt_col = loop.tags.index(seq_id_alt_name) if seq_id_alt_name in loop.tags else -1

            if chain_id_col == -1 or seq_id_col == -1:
                return

            for row in loop.data:

                if row[chain_id_col] != chain_id:
                    continue

                seq_id = row[seq_id_col]

                if seq_id in seq_id_conv_dict:
                    row[seq_id_col] = seq_id_conv_dict[seq_id]

                if seq_id_alt_col == -1:
                    continue

                seq_id_alt = row[seq_id_alt_col]

                if seq_id_alt in seq_id_conv_dict:
                    row[seq_id_alt_col] = seq_id_conv_dict[seq_id_alt]

        else:

            for i in range(1, max_dim):

                _chain_id_name = chain_id_name + '_' + str(i)
                _seq_id_name = seq_id_name + '_' + str(i)

                chain_id_col = loop.tags.index(_chain_id_name) if _chain_id_name in loop.tags else -1
                seq_id_col = loop.tags.index(_seq_id_name) if _seq_id_name in loop.tags else -1
                seq_id_alt_col = -1
                if seq_id_alt_name is not None:
                    _seq_id_alt_name = seq_id_alt_name + '_' + str(i)
                    seq_id_alt_col = loop.tags.index(_seq_id_alt_name) if _seq_id_alt_name in loop.tags else -1

                if chain_id_col == -1 or seq_id_col == -1:
                    continue

                for row in loop.data:

                    if row[chain_id_col] != chain_id:
                        continue

                    seq_id = row[seq_id_col]

                    if seq_id in seq_id_conv_dict:
                        row[seq_id_col] = seq_id_conv_dict[seq_id]

                    if seq_id_alt_col == -1:
                        continue

                    seq_id_alt = row[seq_id_alt_col]

                    if seq_id_alt in seq_id_conv_dict:
                        row[seq_id_alt_col] = seq_id_conv_dict[seq_id_alt]

    def __get1LetterCode(self, comp_id):
        """ Convert comp ID to 1-letter code.
        """

        comp_id = comp_id.upper()

        if comp_id in self.monDict3:
            return self.monDict3[comp_id]

        if comp_id in self.empty_value:
            return '.'

        return 'X'

    def __get1LetterCodeSequence(self, comp_ids):
        """ Convert array of comp ID to 1-letter code sequence.
        """

        array = ''

        for comp_id in comp_ids:
            array += self.__get1LetterCode(comp_id)

        return array

    def __validateAtomNomenclature(self):
        """ Validate atom nomenclature using NEFTranslator and CCD.
        """

        #if self.report.isError():
        #    return False

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            has_poly_seq_in_loop = has_key_value(input_source_dic, 'polymer_sequence_in_loop')

            if not has_poly_seq_in_loop:
                continue

            polymer_sequence = input_source_dic['polymer_sequence']

            first_comp_ids = set()

            if polymer_sequence is not None:
                for s in polymer_sequence:
                    first_comp_id = s['comp_id'][0]

                    if self.__csStat.getTypeOfCompId(first_comp_id)[0]:
                        first_comp_ids.add(first_comp_id)

            polymer_sequence_in_loop = input_source_dic['polymer_sequence_in_loop']

            content_subtypes = ['poly_seq']
            content_subtypes.extend(polymer_sequence_in_loop.keys())

            for content_subtype in content_subtypes:

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                if content_subtype == 'poly_seq':
                    lp_category = self.aux_lp_categories[file_type][content_subtype][0]

                    if lp_category not in self.__lp_category_list:
                        continue

                if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
                    lp_category = '_Assigned_peak_chem_shift'

                if self.__star_data_type[fileListId] == 'Loop':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = ''

                    self.__validateAtomNomenclature__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, first_comp_ids)

                elif self.__star_data_type[fileListId] == 'Saveframe':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    self.__validateAtomNomenclature__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, first_comp_ids)

                else:

                    for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                        sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                        if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                            continue

                        self.__validateAtomNomenclature__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, first_comp_ids)

        return not self.report.isError()

    def __isNmrAtomName(self, comp_id, atom_id):
        """ Return whether a given atom_id uses NMR conventional atom name.
        """

        return (atom_id == 'HN' and self.__csStat.getTypeOfCompId(comp_id)[0]) or\
                atom_id.startswith('Q') or atom_id.startswith('M') or\
                atom_id.endswith('%') or atom_id.endswith('%') or\
                self.__csStat.getMaxAmbigCodeWoSetId(comp_id, atom_id) == 0

    def __getRepresentativeAtomId(self, file_type, comp_id, atom_id):
        """ Return a representative atom ID in IUPAC atom nomenclature for a given atom_id.
        """

        _atom_id = self.__getAtomIdList(file_type, comp_id, atom_id)

        return atom_id if len(_atom_id) == 0 else _atom_id[0]

    def __getAtomIdList(self, file_type, comp_id, atom_id):
        """ Return atom ID list in IUPAC atom nomenclature for a given atom_id.
        """

        return self.__getAtomIdListWithAmbigCode(file_type, comp_id, atom_id, leave_unmatched=False)[0]

    def __getAtomIdListWithAmbigCode(self, file_type, comp_id, atom_id, leave_unmatched=True):
        """ Return lists of atom ID, ambiguity_code, details in IUPAC atom nomenclature for a given conventional NMR atom name.
        """

        if file_type == 'nef' or atom_id == 'HN' or atom_id.endswith('%') or atom_id.endswith('*'):
            return self.__nefT.get_star_atom(comp_id, atom_id, leave_unmatched=leave_unmatched)

        if atom_id.startswith('QQ'):
            return self.__nefT.get_star_atom(comp_id, 'H' + atom_id[2:] + '%', leave_unmatched=leave_unmatched)

        if atom_id.startswith('QR'):
            qr_atoms = sorted(set(atom_id[:-1] + '%' for atom_id in self.__csStat.getAromaticAtoms(comp_id) if atom_id[0] == 'H' and self.__csStat.getMaxAmbigCodeWoSetId(comp_id, atom_id) == 3))
            if len(qr_atoms) == 0:
                return [], None, None
            atom_list = []
            for qr_atom in qr_atoms:
                _atom_list, ambiguity_code, details = self.__nefT.get_star_atom(comp_id, qr_atom, leave_unmatched=leave_unmatched)
                atom_list.extend(_atom_list)
            return atom_list, ambiguity_code, details

        if atom_id.startswith('Q') or atom_id.startswith('M'):
            return self.__nefT.get_star_atom(comp_id, 'H' + atom_id[1:] + '%', leave_unmatched=leave_unmatched)

        if atom_id + '2' in self.__csStat.getAllAtoms(comp_id):
            return self.__nefT.get_star_atom(comp_id, atom_id + '%', leave_unmatched=leave_unmatched)

        return self.__nefT.get_star_atom(comp_id, atom_id, leave_unmatched=leave_unmatched)

    def __validateAtomNomenclature__(self, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, first_comp_ids):
        """ Validate atom nomenclature using NEFTranslator and CCD.
        """

        try:

            if file_type == 'nef': # DAOTHER-7389, issue #3, allow empty for 'chem_shift'
                pairs = self.__nefT.get_nef_comp_atom_pair(sf_data, lp_category,
                                                           allow_empty=(content_subtype in ('chem_shift', 'spectral_peak')))[0]
            else: # DAOTHER-7389, issue #3, allow empty for 'chem_shift'
                pairs = self.__nefT.get_star_comp_atom_pair(sf_data, lp_category,
                                                            allow_empty=(content_subtype in ('chem_shift', 'spectral_peak')))[0]

            for pair in pairs:
                comp_id = pair['comp_id']
                atom_ids = pair['atom_id']

                # standard residue
                if self.__get1LetterCode(comp_id) != 'X':

                    if file_type == 'nef':

                        _atom_ids = []
                        for atom_id in atom_ids:

                            _atom_id = self.__nefT.get_star_atom(comp_id, atom_id, leave_unmatched=False)[0]

                            if len(_atom_id) == 0:

                                if self.__nonblk_bad_nterm and atom_id == 'H1' and comp_id in first_comp_ids:
                                    continue

                                err = "Invalid atom_id %r (comp_id %s) in a loop %s." % (atom_id, comp_id, lp_category)

                                self.report.error.appendDescription('invalid_atom_nomenclature', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s\n" % err)

                            else:
                                _atom_ids.extend(_atom_id)

                        atom_ids = sorted(set(_atom_ids))

                    for atom_id in atom_ids:

                        if atom_id == 'HN' and self.__csStat.getTypeOfCompId(comp_id)[0]:
                            self.__fixAtomNomenclature(comp_id, {'HN': 'H'})
                            continue

                        atom_id_ = atom_id

                        if (file_type == 'nef' or not self.__combined_mode or self.__transl_pseudo_name) and self.__isNmrAtomName(comp_id, atom_id):
                            atom_id_ = self.__getRepresentativeAtomId(file_type, comp_id, atom_id)

                            if file_type == 'nmr-star' and self.__combined_mode and self.__transl_pseudo_name and atom_id != atom_id_:

                                warn = "Conventional psuedo atom %s:%s is translated to %r according to the IUPAC atom nomenclature." % (comp_id, atom_id, atom_id_)

                                self.report.warning.appendDescription('auth_atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                                self.__fixAtomNomenclature(comp_id, {atom_id: atom_id_})

                        if not self.__nefT.validate_comp_atom(comp_id, atom_id_):

                            if self.__csStat.getTypeOfCompId(comp_id)[0] and atom_id_.startswith('H') and atom_id_.endswith('1') and\
                               self.__nefT.validate_comp_atom(comp_id, atom_id_[:-1] + '2') and self.__nefT.validate_comp_atom(comp_id, atom_id_[:-1] + '3'):

                                _atom_id_ = atom_id_[:-1]
                                _atom_id_1 = _atom_id_ + '1'
                                _atom_id_2 = _atom_id_ + '2'
                                _atom_id_3 = _atom_id_ + '3'

                                warn = "%s:%s/%s should be %s:%s/%s according to the IUPAC atom nomenclature, respectively." % (comp_id, _atom_id_1, _atom_id_2, comp_id, _atom_id_2, _atom_id_3)

                                self.report.warning.appendDescription('auth_atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                                self.__fixAtomNomenclature(comp_id, {_atom_id_1: _atom_id_2, _atom_id_2: _atom_id_3})

                            elif self.__nonblk_bad_nterm and atom_id == 'H1' and comp_id in first_comp_ids:
                                pass

                            else:

                                err = "Invalid atom_id %r (comp_id %s) in a loop %s." % (atom_id, comp_id, lp_category)

                                self.report.error.appendDescription('invalid_atom_nomenclature', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s\n" % err)

                # non-standard residue
                else:

                    if self.__updateChemCompDict(comp_id): # matches with comp_id in CCD

                        ref_atom_ids = [a[self.__cca_atom_id] for a in self.__last_chem_comp_atoms] # if a[self.__cca_leaving_atom_flag] != 'Y']
                        unk_atom_ids = []

                        for atom_id in atom_ids:

                            if file_type == 'nef':
                                _atom_id = self.__nefT.get_star_atom(comp_id, atom_id, leave_unmatched=False)[0]
                                if len(_atom_id) > 0:
                                    atom_id = _atom_id[0]

                            if atom_id not in ref_atom_ids:
                                unk_atom_ids.append(atom_id)

                        if len(unk_atom_ids) > 0:
                            cc_rel_status = self.__last_chem_comp_dict['_chem_comp.pdbx_release_status']
                            if cc_rel_status == 'REL':
                                cc_name = self.__last_chem_comp_dict['_chem_comp.name']
                            else:
                                cc_name = '(Not available due to CCD status code %s)' % cc_rel_status

                            warn = "Unknown atom_id %r (comp_id %s, chem_comp_name %s)." % (unk_atom_ids, comp_id, cc_name)

                            self.report.warning.appendDescription('atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                            self.report.setWarning()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                        ref_elems = set(a[self.__cca_type_symbol] for a in self.__last_chem_comp_atoms if a[self.__cca_leaving_atom_flag] != 'Y')

                        for elem in ref_elems:
                            if elem in self.paramag_elems or elem in self.ferromag_elems:
                                self.report.setDiamagnetic(False)
                                break

                        for atom_id in atom_ids:

                            if atom_id == 'HN' and self.__csStat.getTypeOfCompId(comp_id)[0]:
                                self.__fixAtomNomenclature(comp_id, {'HN': 'H'})
                                continue

                            atom_id_ = atom_id

                            if (file_type == 'nef' or not self.__combined_mode or self.__transl_pseudo_name) and self.__isNmrAtomName(comp_id, atom_id):
                                atom_id_ = self.__getRepresentativeAtomId(file_type, comp_id, atom_id)

                                if file_type == 'nmr-star' and self.__combined_mode and self.__transl_pseudo_name and atom_id != atom_id_:

                                    warn = "Conventional psuedo atom %s:%s is translated to %r according to the IUPAC atom nomenclature." % (comp_id, atom_id, atom_id_)

                                    self.report.warning.appendDescription('auth_atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                                    self.__fixAtomNomenclature(comp_id, {atom_id: atom_id_})

                    else:
                        pass

            if file_type == 'nmr-star':

                try:

                    auth_pairs = self.__nefT.get_star_auth_comp_atom_pair(sf_data, lp_category)[0]

                    for auth_pair in auth_pairs:
                        comp_id = auth_pair['comp_id']
                        auth_atom_ids = auth_pair['atom_id']

                        # standard residue
                        if self.__get1LetterCode(comp_id) != 'X':

                            _auth_atom_ids = []
                            for auth_atom_id in auth_atom_ids:

                                _auth_atom_id = self.__nefT.get_star_atom(comp_id, auth_atom_id, leave_unmatched=False)[0]

                                if len(_auth_atom_id) == 0:

                                    if self.__nonblk_bad_nterm and auth_atom_id == 'H1' and comp_id in first_comp_ids:
                                        continue

                                    warn = "Unmatched Auth_atom_ID %r (Auth_comp_ID %s)." % (auth_atom_id, comp_id)

                                    self.report.warning.appendDescription('auth_atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                                else:
                                    _auth_atom_ids.extend(_auth_atom_id)

                            auth_atom_ids = sorted(set(_auth_atom_ids))

                            for auth_atom_id in auth_atom_ids:

                                if not self.__nefT.validate_comp_atom(comp_id, auth_atom_id):

                                    if self.__nonblk_bad_nterm and auth_atom_id == 'H1' and comp_id in first_comp_ids:
                                        continue

                                    warn = "Unmatched Auth_atom_ID %r (Auth_comp_ID %s)." % (auth_atom_id, comp_id)

                                    self.report.warning.appendDescription('auth_atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                        # non-standard residue
                        else:
                            has_comp_id = False

                            for pair in pairs:

                                if pair['comp_id'] != comp_id:
                                    continue

                                has_comp_id = True

                                atom_ids = pair['atom_id']

                                if (set(auth_atom_ids) | set(atom_ids)) != set(atom_ids):

                                    for auth_atom_id in (set(auth_atom_ids) | set(atom_ids)) - set(atom_ids):

                                        if self.__nonblk_bad_nterm and auth_atom_id == 'H1' and comp_id in first_comp_ids:
                                            continue

                                        warn = "Unmatched Auth_atom_ID %r (Auth_comp_ID %s, non-standard residue)." % (auth_atom_id, comp_id)

                                        self.report.warning.appendDescription('auth_atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                                break

                            if not has_comp_id:

                                for auth_atom_id in auth_atom_ids:

                                    if self.__nonblk_bad_nterm and auth_atom_id == 'H1' and comp_id in first_comp_ids:
                                        continue

                                    warn = "Unmatched Auth_atom_ID %r (Auth_comp_ID %s, non-standard residue)." % (auth_atom_id, comp_id)

                                    self.report.warning.appendDescription('auth_atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Warning  - %s\n" % warn)

                except LookupError:
                    """
                    self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__extractPolymerSequence() ++ LookupError  - %s" % str(e))
                    """
                except ValueError as e:

                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ ValueError  - %s" % str(e))

                except UserWarning as e:

                    warns = str(e).strip("'").split('\n')
                    proc_warns = set()

                    for warn in warns:

                        if warn == '' or warn in proc_warns:
                            continue

                        proc_warns.add(warn)

                        if warn.startswith('[Invalid data]'):

                            p = warn.index(']') + 2
                            warn = warn[p:]

                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ ValueError  - %s" % warn)

                        else:

                            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % warn)
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % warn)

                except Exception as e:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % str(e))
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % str(e))

        except LookupError as e:

            self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ LookupError  - %s" % str(e))

        except ValueError as e:

            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ ValueError  - %s" % str(e))

        except UserWarning as e:

            warns = str(e).strip("'").split('\n')
            proc_warns = set()

            for warn in warns:

                if warn == '' or warn in proc_warns:
                    continue

                proc_warns.add(warn)

                if warn.startswith('[Invalid data]'):

                    p = warn.index(']') + 2
                    warn = warn[p:]

                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ ValueError  - %s" % warn)

                else:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % warn)
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % warn)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateAtomNomenclature() ++ Error  - %s" % str(e))

    def __fixAtomNomenclature(self, comp_id, atom_id_conv_dict):
        """ Fix atom nomenclature.
        """

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            #file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            for content_subtype in input_source_dic['content_subtype'].keys():

                if content_subtype == ['entry_info', 'entity', 'chem_shift_ref']:
                    continue

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                if content_subtype == 'poly_seq':
                    lp_category = self.aux_lp_categories[file_type][content_subtype][0]

                if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
                    lp_category = '_Assigned_peak_chem_shift'

                if self.__star_data_type[fileListId] == 'Loop':

                    sf_data = self.__star_data[fileListId]
                    #sf_framecode = ''

                    self.__fixAtomNomenclature__(fileListId, file_type, content_subtype, sf_data, lp_category, comp_id, atom_id_conv_dict)

                elif self.__star_data_type[fileListId] == 'Saveframe':

                    sf_data = self.__star_data[fileListId]
                    #sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    self.__fixAtomNomenclature__(fileListId, file_type, content_subtype, sf_data, lp_category, comp_id, atom_id_conv_dict)

                else:

                    for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                        #sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                        if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                            continue

                        self.__fixAtomNomenclature__(fileListId, file_type, content_subtype, sf_data, lp_category, comp_id, atom_id_conv_dict)

    def __fixAtomNomenclature__(self, file_list_id, file_type, content_subtype, sf_data, lp_category, comp_id, atom_id_conv_dict):
        """ Fix atom nomenclature.
        """

        comp_id_name = 'residue_name' if file_type == 'nef' else 'Comp_ID'
        atom_id_name = 'atom_name' if file_type == 'nef' else 'Atom_ID'

        max_dim = 2

        if content_subtype in ('poly_seq', 'dist_restraint', 'rdc_restraint'):
            max_dim = 3

        elif content_subtype == 'dihed_restraint':
            max_dim = 5

        elif content_subtype == 'spectral_peak':

            try:

                _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                num_dim = int(_num_dim)

                if num_dim not in range(1, self.lim_num_dim):
                    raise ValueError()

            except ValueError: # raised error already at __testIndexConsistency()
                return

            max_dim = num_dim + 1

        if __pynmrstar_v3_2__:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop(lp_category)
        else:
            loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop_by_category(lp_category)

        if max_dim == 2:

            comp_id_col = loop.tags.index(comp_id_name) if comp_id_name in loop.tags else -1
            atom_id_col = loop.tags.index(atom_id_name) if atom_id_name in loop.tags else -1

            if comp_id_col == -1 or atom_id_col == -1:
                return

            for row in loop:

                _comp_id = row[comp_id_col].upper()

                if _comp_id != comp_id:
                    continue

                atom_id = row[atom_id_col]

                if atom_id in atom_id_conv_dict:
                    row[atom_id_col] = atom_id_conv_dict[atom_id]

        else:

            for j in range(1, max_dim):

                _comp_id_name = comp_id_name + '_' + str(j)
                _atom_id_name = atom_id_name + '_' + str(j)

                comp_id_col = loop.tags.index(_comp_id_name) if _comp_id_name in loop.tags else -1
                atom_id_col = loop.tags.index(_atom_id_name) if _atom_id_name in loop.tags else -1

                if comp_id_col == -1 or atom_id_col == -1:
                    continue

                for row in loop:

                    _comp_id = row[comp_id_col].upper()

                    if _comp_id != comp_id:
                        continue

                    atom_id = row[atom_id_col]

                    if atom_id in atom_id_conv_dict:
                        row[atom_id_col] = atom_id_conv_dict[atom_id]

    def __validateAtomTypeOfCSLoop(self):
        """ Validate atom type, isotope number on assigned chemical shifts.
        """

        #if self.report.isError():
        #    return False

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'chem_shift'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                """
                err = "Assigned chemical shift loop does not exist in %r file." % file_name

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s" % err)
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s\n" % err)
                """
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            if self.__star_data_type[fileListId] == 'Loop':

                sf_data = self.__star_data[fileListId]
                sf_framecode = ''

                self.__validateAtomTypeOfCSLoop__(file_name, file_type, sf_data, sf_framecode, lp_category)

            elif self.__star_data_type[fileListId] == 'Saveframe':

                sf_data = self.__star_data[fileListId]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__validateAtomTypeOfCSLoop__(file_name, file_type, sf_data, sf_framecode, lp_category)

            else:

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__validateAtomTypeOfCSLoop__(file_name, file_type, sf_data, sf_framecode, lp_category)

        return not self.report.isError()

    def __validateAtomTypeOfCSLoop__(self, file_name, file_type, sf_data, sf_framecode, lp_category):
        """ Validate atom type, isotope number on assigned chemical shifts.
        """

        if not self.__combined_mode:
            return

        try:

            # DAOTHER-7389, issue #3, allow empty for 'chem_shift'
            if file_type == 'nef':
                a_types = self.__nefT.get_nef_atom_type_from_cs_loop(sf_data, allow_empty=True)[0]
            else:
                a_types = self.__nefT.get_star_atom_type_from_cs_loop(sf_data, allow_empty=True)[0]

            for a_type in a_types:
                atom_type = a_type['atom_type']
                isotope_nums = a_type['isotope_number']
                atom_ids = a_type['atom_id']

                if atom_type not in self.atom_isotopes.keys():

                    err = "Invalid atom_type %r in a loop %s." % (atom_type, lp_category)

                    self.report.error.appendDescription('invalid_atom_type', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s\n" % err)

                else:

                    for isotope_num in isotope_nums:
                        if isotope_num not in self.atom_isotopes[atom_type]:

                            err = "Invalid isotope number %r (atom_type %s, allowed isotope number %s) in a loop %s." % (isotope_num, atom_type, self.atom_isotopes[atom_type], lp_category)

                            self.report.error.appendDescription('invalid_isotope_number', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s\n" % err)

                    for atom_id in atom_ids:
                        if not atom_id.startswith(atom_type):

                            err = "Invalid atom_id %r (atom_type %s) in a loop %s." % (atom_id, atom_type, lp_category)

                            self.report.error.appendDescription('invalid_atom_nomenclature', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s\n" % err)

        except LookupError as e:

            if not self.__resolve_conflict:
                self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ LookupError  - %s" % str(e))

        except ValueError as e:

            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ ValueError  - %s" % str(e))

        except UserWarning as e:

            warns = str(e).strip("'").split('\n')
            proc_warns = set()

            for warn in warns:

                if warn == '' or warn in proc_warns:
                    continue

                proc_warns.add(warn)

                if warn.startswith('[Invalid data]'):

                    p = warn.index(']') + 2
                    warn = warn[p:]

                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ ValueError  - %s" % warn)

                else:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s" % warn)
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s" % warn)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateAtomTypeOfCSLoop() ++ Error  - %s" % str(e))

    def __validateAmbigCodeOfCSLoop(self):
        """ Validate ambiguity code on assigned chemical shifts.
        """

        #if self.report.isError():
        #    return False

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'chem_shift'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                """
                err = "Assigned chemical shift loop does not exist in %r file." % file_name

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateAmbigCodeOfCSLoop() ++ Error  - %s" % err)
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__validateAmbigCodeOfCSLoop() ++ Error  - %s\n" % err)
                """
                continue

            # NEF file has no ambiguity code
            if file_type == 'nef':
                continue

            if not self.__combined_mode:
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            if self.__star_data_type[fileListId] == 'Loop':

                sf_data = self.__star_data[fileListId]
                sf_framecode = ''

                self.__validateAmbigCodeOfCSLoop__(file_name, file_type, sf_data, sf_framecode, lp_category)

            elif self.__star_data_type[fileListId] == 'Saveframe':

                sf_data = self.__star_data[fileListId]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__validateAmbigCodeOfCSLoop__(file_name, file_type, sf_data, sf_framecode, lp_category)

            else:

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__validateAmbigCodeOfCSLoop__(file_name, file_type, sf_data, sf_framecode, lp_category)

        return not self.report.isError()

    def __validateAmbigCodeOfCSLoop__(self, file_name, file_type, sf_data, sf_framecode, lp_category):
        """ Validate ambiguity code on assigned chemical shifts.
        """

        try:

            a_codes = self.__nefT.get_star_ambig_code_from_cs_loop(sf_data)[0]

            comp_ids_wo_ambig_code = []

            for a_code in a_codes:
                comp_id = a_code['comp_id']
                ambig_code = a_code['ambig_code']
                atom_ids = a_code['atom_id']

                if ambig_code is None:
                    comp_ids_wo_ambig_code.append(comp_id)

                elif ambig_code == 1 or ambig_code >= 4:
                    pass

                # ambig_code is 2 (geminal atoms) or 3 (aromatic ring atoms in opposite side)
                else:

                    for atom_id in atom_ids:

                        _atom_id = atom_id

                        if self.__isNmrAtomName(comp_id, atom_id):
                            _atom_id = self.__getRepresentativeAtomId(file_type, comp_id, atom_id)

                        allowed_ambig_code = self.__csStat.getMaxAmbigCodeWoSetId(comp_id, _atom_id)

                        if ambig_code > allowed_ambig_code:

                            if allowed_ambig_code < 1:

                                warn = "Ambiguity code %r (comp_id %s, atom_id %s) should be '1' according to the BMRB definition." % (ambig_code, comp_id, atom_id)

                                self.report.warning.appendDescription('ambiguity_code_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ Warning  - %s\n" % warn)

                            else:

                                err = "Invalid ambiguity code %r (comp_id %s, atom_id %s, allowed ambig_code %s) in a loop %s." % (ambig_code, comp_id, atom_id, [1, allowed_ambig_code, 4, 5, 6, 9], lp_category)

                                self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ Error  - %s\n" % err)

            if len(comp_ids_wo_ambig_code) > 0:

                warn = "Missing ambiguity code for the following residues %s." % comp_ids_wo_ambig_code

                self.report.warning.appendDescription('missing_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ Warning  - %s\n" % warn)

        except LookupError as e:

            if not self.__resolve_conflict:
                self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ LookupError  - %s" % str(e))

        except ValueError as e:

            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ ValueError  - %s" % str(e))

        except UserWarning as e:

            warns = str(e).strip("'").split('\n')
            proc_warns = set()

            for warn in warns:

                if warn == '' or warn in proc_warns:
                    continue

                proc_warns.add(warn)

                if warn.startswith('[Invalid data]'):

                    p = warn.index(']') + 2
                    warn = warn[p:]

                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ ValueError  - %s" % warn)

                else:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ Error  - %s" % warn)
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ Error  - %s" % warn)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testAmbigCodeOfCSLoop() ++ Error  - %s" % str(e))

    def __testIndexConsistency(self):
        """ Perform consistency test on index of interesting loops.
        """

        #if self.report.isError():
        #    return False

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_type'] == 'nmr-restraints' or input_source_dic['content_subtype'] is None:
                continue

            for content_subtype in input_source_dic['content_subtype'].keys():

                if content_subtype == 'entity':
                    continue

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                index_tag = self.index_tags[file_type][content_subtype]

                if index_tag is None:
                    continue

                if self.__star_data_type[fileListId] == 'Loop':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = ''

                    self.__testIndexConsistency__(file_name, sf_data, sf_framecode, lp_category, index_tag)

                elif self.__star_data_type[fileListId] == 'Saveframe':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    self.__testIndexConsistency__(file_name, sf_data, sf_framecode, lp_category, index_tag)

                else:

                    for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                        sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                        if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                            continue

                        self.__testIndexConsistency__(file_name, sf_data, sf_framecode, lp_category, index_tag)

        return not self.report.isError()

    def __testIndexConsistency__(self, file_name, sf_data, sf_framecode, lp_category, index_tag):
        """ Perform consistency test on index of interesting loops.
        """

        try:

            indices = self.__nefT.get_index(sf_data, lp_category, index_id=index_tag)[0]

            if indices != list(range(1, len(indices) + 1)):

                warn = "Index of loop '%s.%s' should be ordinal numbers." % (lp_category, index_tag)

                self.report.warning.appendDescription('disordered_index', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testIndexConsistency() ++ Warning  - %s\n" % warn)

        except KeyError as e:

            self.report.error.appendDescription('duplicated_index', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testIndexConsistency() ++ KeyError  - %s" % str(e))

        except LookupError as e:

            self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testIndexConsistency() ++ LookupError  - %s" % str(e))

        except ValueError as e:

            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testIndexConsistency() ++ ValueError  - %s" % str(e))

        except UserWarning as e:

            warns = str(e).strip("'").split('\n')
            proc_warns = set()

            for warn in warns:

                if warn == '' or warn in proc_warns:
                    continue

                proc_warns.add(warn)

                if warn.startswith('[Invalid data]'):

                    p = warn.index(']') + 2
                    warn = warn[p:]

                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testIndexConsistency() ++ ValueError  - %s" % warn)

                else:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testIndexConsistency() ++ Error  - %s" % warn)
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testIndexConsistency() ++ Error  - %s" % warn)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testIndexConsistency() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testIndexConsistency() ++ Error  - %s" % str(e))

    def __testDataConsistencyInLoop(self):
        """ Perform consistency test on data of interesting loops.
        """

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            for content_subtype in input_source_dic['content_subtype'].keys():

                if content_subtype in ('entry_info', 'entity'):
                    continue

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                if self.__star_data_type[fileListId] == 'Loop':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = ''

                    self.__testDataConsistencyInLoop__(fileListId, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, 1)

                elif self.__star_data_type[fileListId] == 'Saveframe':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    self.__testDataConsistencyInLoop__(fileListId, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, 1)

                else:

                    parent_pointer = 0

                    for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                        parent_pointer += 1

                        sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                        if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                            continue

                        self.__testDataConsistencyInLoop__(fileListId, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, parent_pointer)

        return self.report.getTotalErrors() == __errors

    def __testDataConsistencyInLoop__(self, file_list_id, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, parent_pointer):
        """ Perform consistency test on data of interesting loops.
        """

        allowed_tags = self.allowed_tags[file_type][content_subtype]
        disallowed_tags = None

        if content_subtype == 'spectral_peak':

            try:

                _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                num_dim = int(_num_dim)

                if num_dim not in range(1, self.lim_num_dim):
                    raise ValueError()

            except ValueError: # raised error already at __testIndexConsistency()
                return

            max_dim = num_dim + 1

            key_items = []
            for dim in range(1, max_dim):
                for k in self.pk_key_items[file_type]:
                    _k = copy.copy(k)
                    if '%s' in k['name']:
                        _k['name'] = k['name'] % dim
                    key_items.append(_k)

            data_items = []
            for d in self.data_items[file_type][content_subtype]:
                data_items.append(d)
            for dim in range(1, max_dim):
                for d in self.pk_data_items[file_type]:
                    _d = copy.copy(d)
                    if '%s' in d['name']:
                        _d['name'] = d['name'] % dim
                    if 'default-from' in d and '%s' in d['default-from']: # DAOTHER-7421
                        _d['default-from'] = d['default-from'] % dim
                    data_items.append(_d)

            if max_dim < self.lim_num_dim:
                disallowed_tags = []
                for dim in range(max_dim, self.lim_num_dim):
                    for t in self.spectral_peak_disallowed_tags[file_type]:
                        if '%s' in t:
                            t = t % dim
                        disallowed_tags.append(t)

        else:

            key_items = self.key_items[file_type][content_subtype]
            data_items = self.data_items[file_type][content_subtype]

        lp_data = None

        try:

            lp_data = self.__nefT.check_data(sf_data, lp_category, key_items, data_items, allowed_tags, disallowed_tags, parent_pointer=parent_pointer,
                                             test_on_index=True, enforce_non_zero=True, enforce_sign=True, enforce_range=True, enforce_enum=True,
                                             enforce_allowed_tags=(file_type == 'nmr-star'),
                                             excl_missing_data=self.__excl_missing_data)[0]

            self.__lp_data[content_subtype].append({'file_name': file_name, 'sf_framecode': sf_framecode, 'data': lp_data})

        except KeyError as e:

            self.report.error.appendDescription('multiple_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ KeyError  - %s" % str(e))

        except LookupError as e:

            item = 'format_issue' if 'Unauthorized item' in str(e) else 'missing_mandatory_item'

            self.report.error.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ LookupError  - %s" % str(e))

        except ValueError as e:

            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ ValueError  - %s" % str(e))

        except UserWarning as e:

            warns = str(e).strip("'").split('\n')
            proc_warns = set()

            has_multiple_data = False
            has_bad_pattern = False

            for warn in warns:

                if warn == '' or warn in proc_warns:
                    continue

                proc_warns.add(warn)

                zero = warn.startswith('[Zero value error]')
                nega = warn.startswith('[Negative value error]')
                rang = warn.startswith('[Range value error]')
                enum = warn.startswith('[Enumeration error]')
                mult = warn.startswith('[Multiple data]')
                remo = warn.startswith('[Remove bad pattern]')
                clea = warn.startswith('[Clear bad pattern]')

                if zero or nega or range or enum or mult or remo or clea:

                    p = warn.index(']') + 2
                    warn = warn[p:]

                    if zero or nega or rang:
                        item = 'unusual_data'
                    elif enum:
                        item = 'enum_mismatch'
                    elif remo:
                        if content_subtype == 'chem_shift':
                            warn += ' Your unassigned chemical shifts have been removed.'
                            item = 'incompletely_assigned_chemical_shift'
                        else:
                            item = 'insufficient_data'
                        has_bad_pattern = True
                    elif clea:
                        if content_subtype.startswith('spectral_peak'):
                            warn += ' Unassigned spectral peaks can be included in your peak list(s).'
                            item = 'incompletely_assigned_spectral_peak'
                        else:
                            item = 'insufficient_data'
                    elif self.__resolve_conflict:
                        item = 'redundant_data'
                        has_multiple_data = True
                    else:
                        item = 'multiple_data'

                    if zero or nega or rang or enum or remo or clea or self.__resolve_conflict:

                        self.report.warning.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                        self.report.setWarning()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ Warning  - %s\n" % warn)

                    else:

                        self.report.error.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ KeyError  - %s" % warn)

                else:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testDataConsistencyInLoop() ++ Error  - %s" % warn)
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ Error  - %s" % warn)

            # try to parse data without constraints

            if has_multiple_data:
                conflict_id = self.__nefT.get_conflict_id(sf_data, lp_category, key_items)[0]

                if len(conflict_id) > 0:
                    if __pynmrstar_v3_2__:
                        loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop(lp_category)
                    else:
                        loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop_by_category(lp_category)

                    for l in conflict_id:
                        del loop.data[l]

            # try to parse data without bad patterns

            if has_bad_pattern:
                conflict_id = self.__nefT.get_bad_pattern_id(sf_data, lp_category, key_items, data_items)[0]

                if len(conflict_id) > 0:
                    if __pynmrstar_v3_2__:
                        loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop(lp_category)
                    else:
                        loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop_by_category(lp_category)

                    for l in conflict_id:
                        del loop.data[l]

            try:

                lp_data = self.__nefT.check_data(sf_data, lp_category, key_items, data_items, allowed_tags, disallowed_tags, parent_pointer=parent_pointer,
                                                 enforce_allowed_tags=(file_type == 'nmr-star'),
                                                 excl_missing_data=self.__excl_missing_data)[0]

                self.__lp_data[content_subtype].append({'file_name': file_name, 'sf_framecode': sf_framecode, 'data': lp_data})

            except:  # noqa: E722 pylint: disable=bare-except
                pass

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testDataConsistencyInLoop() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ Error  - %s" % str(e))
        """
        if (lp_data is not None) and len(lp_data) == 0 and self.__check_empty_loop:

            warn = "Unexpectedly, a loop has no rows."

            self.report.warning.appendDescription('missing_content', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
            self.report.setWarning()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInLoop() ++ Warning  - %s\n" % warn)
        """
    def __detectConflictDataInLoop(self):
        """ Detect redundant/inconsistent data of interesting loops.
        """

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            for content_subtype in input_source_dic['content_subtype'].keys():

                if content_subtype in ('dist_restraint', 'dihed_restraint', 'rdc_restraint'):

                    sf_category = self.sf_categories[file_type][content_subtype]
                    lp_category = self.lp_categories[file_type][content_subtype]

                    if self.__star_data_type[fileListId] == 'Loop':

                        sf_data = self.__star_data[fileListId]
                        sf_framecode = ''

                        self.__detectConflictDataInLoop__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

                    elif self.__star_data_type[fileListId] == 'Saveframe':

                        sf_data = self.__star_data[fileListId]
                        sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                        self.__detectConflictDataInLoop__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

                    else:

                        for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                            sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                            if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                                continue

                            self.__detectConflictDataInLoop__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

        return self.report.getTotalErrors() == __errors

    def __detectConflictDataInLoop__(self, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category):
        """ Detect redundant/inconsistent data of interesting loops.
        """

        lp_data = next((l['data'] for l in self.__lp_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode), None)

        if lp_data is None or len(lp_data) == 0:
            return

        key_items = self.consist_key_items[file_type][content_subtype]

        conflict_id_set = self.__nefT.get_conflict_id_set(sf_data, lp_category, key_items)[0]

        if conflict_id_set is None:
            return

        data_items = self.consist_data_items[file_type][content_subtype]
        index_tag = self.index_tags[file_type][content_subtype]
        id_tag = self.consist_id_tags[file_type][content_subtype]

        if content_subtype == 'dist_restraint':
            max_inclusive = self.dist_restraint_error['max_inclusive']

            data_unit_name = 'atom pair'

        elif content_subtype == 'dihed_restraint':
            max_inclusive = self.dihed_restraint_error['max_inclusive']

            data_unit_name = 'dihedral angle'

            dh_item_names = self.item_names_in_dh_loop[file_type]
            chain_id_1_name = dh_item_names['chain_id_1']
            chain_id_2_name = dh_item_names['chain_id_2']
            chain_id_3_name = dh_item_names['chain_id_3']
            chain_id_4_name = dh_item_names['chain_id_4']
            seq_id_1_name = dh_item_names['seq_id_1']
            seq_id_2_name = dh_item_names['seq_id_2']
            seq_id_3_name = dh_item_names['seq_id_3']
            seq_id_4_name = dh_item_names['seq_id_4']
            comp_id_1_name = dh_item_names['comp_id_1']
            comp_id_2_name = dh_item_names['comp_id_2']
            comp_id_3_name = dh_item_names['comp_id_3']
            comp_id_4_name = dh_item_names['comp_id_4']
            atom_id_1_name = dh_item_names['atom_id_1']
            atom_id_2_name = dh_item_names['atom_id_2']
            atom_id_3_name = dh_item_names['atom_id_3']
            atom_id_4_name = dh_item_names['atom_id_4']
            angle_type_name = dh_item_names['angle_type']

        elif content_subtype == 'rdc_restraint':
            max_inclusive = self.rdc_restraint_error['max_inclusive']

            data_unit_name = 'bond vector'

        for id_set in conflict_id_set:
            len_id_set = len(id_set)

            if len_id_set < 2:
                continue

            redundant = True

            for i in range(len_id_set - 1):

                for j in range(i + 1, len_id_set):
                    row_1 = lp_data[id_set[i]]
                    row_2 = lp_data[id_set[j]]

                    conflict = False
                    inconsist = False

                    discrepancy = ''

                    for d in data_items:
                        dname = d['name']

                        if dname not in row_1:
                            continue

                        val_1 = row_1[dname]
                        val_2 = row_2[dname]

                        if val_1 is None and val_2 is None:
                            continue

                        if val_1 is None or val_2 is None:
                            redundant = False
                            continue

                        if val_1 == val_2:
                            continue

                        redundant = False

                        _val_1 = str(val_1) if val_1 >= 0.0 else '(' + str(val_1) + ')'
                        _val_2 = str(val_2) if val_2 >= 0.0 else '(' + str(val_2) + ')'

                        if content_subtype == 'dist_restraint':

                            r = abs(val_1 - val_2) / abs(val_1 + val_2)

                            if r >= self.r_conflicted_dist_restraint:
                                discrepancy += '%s |%s-%s|/|%s+%s| = %s %% is out of acceptable range, %s %%, ' % (dname, _val_1, _val_2, _val_1, _val_2, '{:.1f}'.format(r * 100.0), int(self.r_conflicted_dist_restraint * 100))
                                conflict = True

                            elif r >= self.r_inconsistent_dist_restraint:
                                discrepancy += '%s |%s-%s|/|%s+%s| = %s %% is out of typical range, %s %%, ' % (dname, _val_1, _val_2, _val_1, _val_2, '{:.1f}'.format(r * 100.0), int(self.r_inconsistent_dist_restraint * 100))
                                inconsist = True

                        else:

                            r = abs(val_1 - val_2)

                            if content_subtype == 'dihed_restraint':

                                if r > 180.0:
                                    if val_1 < val_2:
                                        r = abs(val_1 - (val_2 - 360.0))
                                    if val_1 > val_2:
                                        r = abs(val_1 - (val_2 + 360.0))

                                chain_id_1 = row_1[chain_id_1_name]
                                chain_id_2 = row_1[chain_id_2_name]
                                chain_id_3 = row_1[chain_id_3_name]
                                chain_id_4 = row_1[chain_id_4_name]
                                seq_id_1 = row_1[seq_id_1_name]
                                seq_id_2 = row_1[seq_id_2_name]
                                seq_id_3 = row_1[seq_id_3_name]
                                seq_id_4 = row_1[seq_id_4_name]
                                comp_id_1 = row_1[comp_id_1_name]
                                comp_id_2 = row_1[comp_id_2_name]
                                comp_id_3 = row_1[comp_id_3_name]
                                comp_id_4 = row_1[comp_id_4_name]
                                atom_id_1 = row_1[atom_id_1_name]
                                atom_id_2 = row_1[atom_id_2_name]
                                atom_id_3 = row_1[atom_id_3_name]
                                atom_id_4 = row_1[atom_id_4_name]
                                data_type = row_1[angle_type_name]

                                data_type = self.__getTypeOfDihedralRestraint(data_type,
                                                                              chain_id_1, seq_id_1, comp_id_1, atom_id_1, chain_id_2, seq_id_2, comp_id_2, atom_id_2,
                                                                              chain_id_3, seq_id_3, comp_id_3, atom_id_3, chain_id_4, seq_id_4, comp_id_4, atom_id_4)[0]

                                if not data_type.startswith('phi') and not data_type.startswith('psi') and not data_type.startswith('omega'):
                                    continue

                            if r > max_inclusive:
                                discrepancy += '%s |%s-%s| = %s is out of acceptable range, %s %s, ' % (dname, _val_1, _val_2, '{:.1f}'.format(r), max_inclusive, 'degrees' if content_subtype == 'dihed_restraint' else 'Hz')
                                conflict = True

                            elif r > max_inclusive * self.inconsist_over_conflicted:
                                discrepancy += '%s |%s-%s| = %s is out of typical range, %s %s, ' % (dname, _val_1, _val_2, '{:.1f}'.format(r), max_inclusive * self.inconsist_over_conflicted, 'degrees' if content_subtype == 'dihed_restraint' else 'Hz')
                                inconsist = True

                    if conflict:

                        msg = '' if content_subtype != 'dihed_restraint' else angle_type_name + ' %s, ' % row_1[angle_type_name]
                        msg += self.__getResucedAtomNotations(key_items, row_1)

                        if index_tag in row_1:
                            warn = '[Check rows of %s %s vs %s, %s %s vs %s] ' %\
                                   (index_tag, row_1[index_tag], row_2[index_tag],
                                    id_tag, row_1[id_tag], row_2[id_tag])
                        else:
                            warn = '[Check rows of %s %s vs %s, %s %s vs %s] ' %\
                                   (index_tag, id_set[i] + 1, id_set[j] + 1,
                                    id_tag, row_1[id_tag], row_2[id_tag])
                        warn += 'Found conflict on restraints (%s) for the same %s (%s).' % (discrepancy[:-2], data_unit_name, msg)

                        self.report.warning.appendDescription('conflicted_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'sigma': float('{:.2f}'.format(r / max_inclusive))})
                        self.report.setWarning()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__detetConflictDataInLoop() ++ Warning  - %s" % warn)

                    elif inconsist:

                        msg = '' if content_subtype != 'dihed_restraint' else angle_type_name + ' %s, ' % row_1[angle_type_name]
                        msg += self.__getResucedAtomNotations(key_items, row_1)

                        if index_tag in row_1:
                            warn = '[Check rows of %s %s vs %s, %s %s vs %s] ' %\
                                   (index_tag, row_1[index_tag], row_2[index_tag],
                                    id_tag, row_1[id_tag], row_2[id_tag])
                        else:
                            warn = '[Check rows of %s %s vs %s, %s %s vs %s] ' %\
                                   (index_tag, id_set[i] + 1, id_set[j] + 1,
                                    id_tag, row_1[id_tag], row_2[id_tag])
                        warn += 'Found discrepancy in restraints (%s) for the same %s (%s).' % (discrepancy[:-2], data_unit_name, msg)

                        self.report.warning.appendDescription('inconsistent_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'sigma': float('{:.2f}'.format(r / max_inclusive))})
                        self.report.setWarning()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__detetConflictDataInLoop() ++ Warning  - %s" % warn)

            if redundant:

                msg = '' if content_subtype != 'dihed_restraint' else angle_type_name + ' %s, ' % row_1[angle_type_name]
                msg += self.__getResucedAtomNotations(key_items, row_1)

                idx_msg = index_tag + ' '
                if index_tag in lp_data[0]:
                    for id in id_set:
                        idx_msg += '%s vs ' % lp_data[id][index_tag]
                else:
                    for id in id_set:
                        idx_msg += '%s vs ' % (id + 1)
                idx_msg = idx_msg[:-4] + ', '
                idx_msg += id_tag + ' '
                for id in id_set:
                    idx_msg += '%s vs ' % lp_data[id][id_tag]

                warn = '[Check rows of %s] Found redundant restraints for the same %s (%s).' % (idx_msg[:-4], data_unit_name, msg)

                self.report.warning.appendDescription('redundant_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                self.report.setWarning()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__detetConflictDataInLoop() ++ Warning  - %s" % warn)

    def __testDataConsistencyInAuxLoop(self):
        """ Perform consistency test on data of auxiliary loops.
        """

        #if not self.__combined_mode:
        #    return True

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            if fileListId >= len(self.__star_data_type) or self.__star_data_type[fileListId] != 'Entry':
                continue

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            for content_subtype in input_source_dic['content_subtype'].keys():

                if content_subtype == 'entity':
                    continue

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                parent_pointer = 0

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    parent_pointer += 1

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if content_subtype.startswith('spectral_peak'):

                        try:

                            _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                            num_dim = int(_num_dim)

                            if num_dim not in range(1, self.lim_num_dim):
                                raise ValueError()

                        except ValueError:

                            err = "%s %r must be in %s." % (self.num_dim_items[file_type], _num_dim, set(range(1, self.lim_num_dim)))

                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ ValueError  - %s\n" % err)

                            continue

                    for loop in sf_data.loops:

                        lp_category = loop.category

                        if lp_category is None:
                            continue

                        # main content of loop has been processed in __testDataConsistencyInLoop()
                        if lp_category in self.lp_categories[file_type][content_subtype]:
                            continue

                        if lp_category in self.aux_lp_categories[file_type][content_subtype]:

                            key_items = self.aux_key_items[file_type][content_subtype][lp_category]
                            data_items = self.aux_data_items[file_type][content_subtype][lp_category]
                            allowed_tags = self.aux_allowed_tags[file_type][content_subtype][lp_category]

                            try:

                                aux_data = self.__nefT.check_data(sf_data, lp_category, key_items, data_items, allowed_tags, None, parent_pointer=parent_pointer,
                                                                  test_on_index=True, enforce_non_zero=True, enforce_sign=True, enforce_range=True, enforce_enum=True,
                                                                  enforce_allowed_tags=(file_type == 'nmr-star'),
                                                                  excl_missing_data=self.__excl_missing_data)[0]

                                self.__aux_data[content_subtype].append({'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'data': aux_data})

                                if content_subtype == 'spectral_peak':
                                    self.__testDataConsistencyInAuxLoopOfSpectralPeak(file_name, file_type, sf_framecode, num_dim, lp_category, aux_data)
                                if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
                                    self.__testDataConsistencyInAuxLoopOfSpectralPeakAlt(file_name, file_type, sf_framecode, num_dim, lp_category, aux_data, sf_data, parent_pointer)

                            except KeyError as e:

                                self.report.error.appendDescription('multiple_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ KeyError  - %s" % str(e))

                            except LookupError as e:

                                item = 'format_issue' if 'Unauthorized item' in str(e) else 'missing_mandatory_item'

                                self.report.error.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ LookupError  - %s" % str(e))

                            except ValueError as e:

                                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ ValueError  - %s" % str(e))

                            except UserWarning as e:

                                warns = str(e).strip("'").split('\n')
                                proc_warns = set()

                                has_multiple_data = False
                                has_bad_pattern = False

                                for warn in warns:

                                    if warn == '':
                                        continue

                                    if warn == '' or warn in proc_warns:
                                        continue

                                    proc_warns.add(warn)

                                    zero = warn.startswith('[Zero value error]')
                                    nega = warn.startswith('[Negative value error]')
                                    rang = warn.startswith('[Range value error]')
                                    enum = warn.startswith('[Enumeration error]')
                                    mult = warn.startswith('[Multiple data]')
                                    remo = warn.startswith('[Remove bad pattern]')
                                    clea = warn.startswith('[Clear bad pattern]')

                                    if zero or nega or rang or enum or mult or remo or clea:

                                        p = warn.index(']') + 2
                                        warn = warn[p:]

                                        if zero or nega or rang:
                                            item = 'unusual_data'
                                        elif enum:
                                            item = 'enum_mismatch'
                                        elif remo:
                                            if content_subtype == 'chem_shift':
                                                warn += ' Your unassigned chemical shifts have been removed.'
                                                item = 'incompletely_assigned_chemical_shift'
                                            else:
                                                item = 'insufficient_data'
                                            has_bad_pattern = True
                                        elif clea:
                                            if content_subtype.startswith('spectral_peak'):
                                                warn += ' Unassigned spectral peaks can be included in your peak list(s).'
                                                item = 'incompletely_assigned_spectral_peak'
                                            else:
                                                item = 'insufficient_data'
                                        elif self.__resolve_conflict:
                                            item = 'redundant_data'
                                            has_multiple_data = True
                                        else:
                                            item = 'multiple_data'

                                        if zero or nega or rang or enum or remo or clea or self.__resolve_conflict:

                                            self.report.warning.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Warning  - %s\n" % warn)

                                        else:

                                            self.report.error.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ KeyError  - %s" % warn)

                                    else:

                                        self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Error  - %s" % warn)
                                        self.report.setError()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Error  - %s" % warn)

                                # try to parse data without constraints

                                if has_multiple_data:
                                    conflict_id = self.__nefT.get_conflict_id(sf_data, lp_category, key_items)[0]

                                    if len(conflict_id) > 0:
                                        if __pynmrstar_v3_2__:
                                            _loop = sf_data.get_loop(lp_category)
                                        else:
                                            _loop = sf_data.get_loop_by_category(lp_category)

                                        for l in conflict_id:
                                            del _loop.data[l]

                                # try to parse data without bad patterns

                                if has_bad_pattern:
                                    conflict_id = self.__nefT.get_bad_pattern_id(sf_data, lp_category, key_items, data_items)[0]

                                    if len(conflict_id) > 0:
                                        if __pynmrstar_v3_2__:
                                            _loop = sf_data.get_loop(lp_category)
                                        else:
                                            _loop = sf_data.get_loop_by_category(lp_category)

                                        for l in conflict_id:
                                            del _loop.data[l]

                                try:

                                    aux_data = self.__nefT.check_data(sf_data, lp_category, key_items, data_items, allowed_tags, None, parent_pointer=parent_pointer,
                                                                      enforce_allowed_tags=(file_type == 'nmr-star'),
                                                                      excl_missing_data=self.__excl_missing_data)[0]

                                    self.__aux_data[content_subtype].append({'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'data': aux_data})

                                    if content_subtype == 'spectral_peak':
                                        self.__testDataConsistencyInAuxLoopOfSpectralPeak(file_name, file_type, sf_framecode, num_dim, lp_category, aux_data)
                                    if file_type == 'nmr-star' and content_subtype == 'spectral_peak_alt':
                                        self.__testDataConsistencyInAuxLoopOfSpectralPeakAlt(file_name, file_type, sf_framecode, num_dim, lp_category, aux_data, sf_data, parent_pointer)

                                except:  # noqa: E722 pylint: disable=bare-except
                                    pass

                            except Exception as e:

                                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Error  - %s" % str(e))
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Error  - %s" % str(e))

                        elif lp_category in self.linked_lp_categories[file_type][content_subtype]:

                            if not self.__bmrb_only:

                                warn = "Ignored %r loop in %r saveframe." % (lp_category, sf_framecode)

                                self.report.warning.appendDescription('skipped_loop_category', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Warning  - %s\n" % warn)

                        else:

                            if not self.__bmrb_only:

                                if file_type == 'nef':
                                    warn = "Ignored third party software's loop %r in %r saveframe." % (lp_category, sf_framecode)
                                else:
                                    warn = "Ignored %r loop in %r saveframe." % (lp_category, sf_framecode)

                                self.report.warning.appendDescription('skipped_loop_category', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Warning  - %s\n" % warn)
                                """
                                err = "%r loop exists unexpectedly." % lp_category

                                self.report.error.appendDescription('format_issue', {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoop() ++ Error  - %s\n" % err)
                                """
        return self.report.getTotalErrors() == __errors

    def __testDataConsistencyInAuxLoopOfSpectralPeak(self, file_name, file_type, sf_framecode, num_dim, lp_category, aux_data):
        """ Perform consistency test on data of spectral peak loops.
        """

        content_subtype = 'spectral_peak'

        max_dim = num_dim + 1

        if (file_type == 'nef' and lp_category == '_nef_spectrum_dimension') or (file_type == 'nmr-star' and lp_category == '_Spectral_dim'):

            err = "The number of dimension %r and the number of rows %r are not matched." % (num_dim, len(aux_data))

            if len(aux_data) != num_dim:
                self.report.error.appendDescription('missing_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeak() ++ Error  - %s\n" % err)

            try:

                min_points = [None] * num_dim
                max_points = [None] * num_dim
                min_limits = [None] * num_dim
                max_limits = [None] * num_dim
                abs = [None] * num_dim

                for i in range(1, max_dim):

                    for sp_dim in aux_data:

                        if file_type == 'nef':

                            if sp_dim['dimension_id'] != i:
                                continue

                            first_point = None if 'value_first_point' not in sp_dim else sp_dim['value_first_point']
                            sp_width = None if 'spectral_width' not in sp_dim else sp_dim['spectral_width']
                            # acq = sp_dim['is_acquisition']
                            sp_freq = None if 'spectrometer_frequency' not in sp_dim else sp_dim['spectrometer_frequency']
                            abs[i - 1] = False if 'absolute_peak_positions' not in sp_dim else sp_dim['absolute_peak_positions']

                            if 'axis_unit' in sp_dim and sp_dim['axis_unit'] == 'Hz' and sp_freq is not None and first_point is not None and sp_width is not None:
                                first_point /= sp_freq
                                sp_width /= sp_freq

                        else:

                            if sp_dim['ID'] != i:
                                continue

                            first_point = None if 'Value_first_point' not in sp_dim else sp_dim['Value_first_point']
                            sp_width = None if 'Sweep_width' not in sp_dim else sp_dim['Sweep_width']
                            # acq = sp_dim['Acquisition']
                            sp_freq = None if 'Spectrometer_frequency' not in sp_dim else sp_dim['Spectrometer_frequency']
                            abs[i - 1] = False if 'Absolute_peak_positions' not in sp_dim else sp_dim['Absolute_peak_positions']

                            if 'Sweep_width_units' in sp_dim and sp_dim['Sweep_width_units'] == 'Hz' and sp_freq is not None and first_point is not None and sp_width is not None:
                                first_point /= sp_freq
                                sp_width /= sp_freq

                        min_point = None
                        max_point = None
                        min_limit = None
                        max_limit = None

                        if first_point is not None and sp_width is not None:

                            last_point = first_point - sp_width

                            min_point = last_point - (sp_width * (1.0 if self.__bmrb_only else 3.0) if abs[i - 1] else 0.0) # DAOTHER-7389, issue #1, relax expected range of peak position by three times of spectral width if absolute_peak_positios are true
                            max_point = first_point + (sp_width * (1.0 if self.__bmrb_only else 3.0) if abs[i - 1] else 0.0)

                            min_limit = min_point
                            max_limit = max_point

                            if sp_freq is not None and min_point is not None and max_point is not None:
                                center_point = (max_point - min_point) / 2.0
                                min_limit = center_point - self.hard_probe_limit / 2.0 / sp_freq
                                max_limit = center_point + self.hard_probe_limit / 2.0 / sp_freq

                        if min_point is not None:
                            min_points[i - 1] = float('{:.7f}'.format(min_point))
                        if max_point is not None:
                            max_points[i - 1] = float('{:.7f}'.format(max_point))

                        if min_limit is not None:
                            min_limits[i - 1] = float('{:.7f}'.format(min_limit))
                        if max_limit is not None:
                            max_limits[i - 1] = float('{:.7f}'.format(max_limit))

                        break

                key_items = []
                for dim in range(1, max_dim):
                    for k in self.pk_key_items[file_type]:
                        _k = copy.copy(k)
                        if '%s' in k['name']:
                            _k['name'] = k['name'] % dim
                        key_items.append(_k)

                position_names = [k['name'] for k in key_items]
                index_tag = self.index_tags[file_type][content_subtype]

                lp_data = next((l['data'] for l in self.__lp_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode), None)

                if lp_data is not None:

                    for i in lp_data:
                        for j in range(num_dim):

                            if min_points[j] is None or max_points[j] is None:
                                continue

                            position = i[position_names[j]]

                            if position < min_points[j] or position > max_points[j]:

                                err = '[Check row of %s %s] %s %s is not within expected range (min_position %s, max_position %s, absolute_peak_positions %s). Please check for reference frequency and spectral width.' % (index_tag, i[index_tag], position_names[j], position, min_points[j], max_points[j], abs[j])

                                self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeak() ++ Warning - %s\n" % err)

                            if min_limits[j] is None or max_limits[j] is None:
                                continue

                            if position < min_limits[j] or position > max_limits[j]:

                                err = '[Check row of %s %s] %s %s is not within expected range (min_position %s, max_position %s, absolute_peak_positions %s), which exceeds limit of current probe design (%s kHz). Please check for reference frequency and spectral width.' % (index_tag, i[index_tag], position_names[j], position, min_limits[j], max_limits[j], abs[j], self.hard_probe_limit / 1000.0)

                                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeak() ++ ValueError  - %s\n" % err)

            except Exception as e:

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeak() ++ Error  - %s" % str(e))
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeak() ++ Error  - %s" % str(e))

        if (file_type == 'nef' and lp_category == '_nef_spectrum_dimension_transfer') or (file_type == 'nmr-star' and lp_category == '_Spectral_dim_transfer'):

            for i in aux_data:
                for name in [j['name'] for j in self.aux_key_items[file_type][content_subtype][lp_category]]:
                    if i[name] not in range(1, max_dim):

                        err = "%s %r must be one of %s." % (name, i[name], range(1, max_dim))

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeak() ++ ValueError  - %s\n" % err)

    def __testDataConsistencyInAuxLoopOfSpectralPeakAlt(self, file_name, file_type, sf_framecode, num_dim, lp_category, aux_data, sf_data, parent_pointer):
        """ Perform consistency test on data of spectral peak loops.
        """

        content_subtype = 'spectral_peak_alt'

        max_dim = num_dim + 1

        if lp_category == '_Spectral_dim':

            err = "The number of dimension %r and the number of rows %r are not matched." % (num_dim, len(aux_data))

            if len(aux_data) != num_dim:
                self.report.error.appendDescription('missing_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ Error  - %s\n" % err)

            try:

                min_points = [None] * num_dim
                max_points = [None] * num_dim
                min_limits = [None] * num_dim
                max_limits = [None] * num_dim
                abs = [None] * num_dim

                for i in range(1, max_dim):

                    for sp_dim in aux_data:

                        if sp_dim['ID'] != i:
                            continue

                        first_point = None if 'Value_first_point' not in sp_dim else sp_dim['Value_first_point']
                        sp_width = None if 'Sweep_width' not in sp_dim else sp_dim['Sweep_width']
                        sp_freq = None if 'Spectrometer_frequency' not in sp_dim else sp_dim['Spectrometer_frequency']
                        # acq = sp_dim['Acquisition']
                        abs[i - 1] = False if 'Absolute_peak_positions' not in sp_dim else sp_dim['Absolute_peak_positions']

                        if 'Sweep_width_units' in sp_dim and sp_dim['Sweep_width_units'] == 'Hz' and sp_freq is not None and first_point is not None and sp_width is not None:
                            first_point /= sp_freq
                            sp_width /= sp_freq

                        min_point = None
                        max_point = None
                        min_limit = None
                        max_limit = None

                        if first_point is not None and sp_width is not None:

                            last_point = first_point - sp_width

                            min_point = last_point - (sp_width * (1.0 if self.__bmrb_only else 3.0) if abs[i - 1] else 0.0) # DAOTHER-7389, issue #1, relax expected range of peak position by three times of spectral width if absolute_peak_positios are true
                            max_point = first_point + (sp_width * (1.0 if self.__bmrb_only else 3.0) if abs[i - 1] else 0.0)

                            min_limit = min_point
                            max_limit = max_point

                            if sp_freq is not None and min_point is not None and max_point is not None:
                                center_point = (max_point - min_point) / 2.0
                                min_limit = center_point - self.hard_probe_limit / 2.0 / sp_freq
                                max_limit = center_point + self.hard_probe_limit / 2.0 / sp_freq

                        if min_point is not None:
                            min_points[i - 1] = float('{:.7f}'.format(min_point))
                        if max_point is not None:
                            max_points[i - 1] = float('{:.7f}'.format(max_point))

                        if min_limit is not None:
                            min_limits[i - 1] = float('{:.7f}'.format(min_limit))
                        if max_limit is not None:
                            max_limits[i - 1] = float('{:.7f}'.format(max_limit))

                        break

                _pk_char_category = '_Peak_char'

                _pk_char_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == _pk_char_category), None)

                if _pk_char_data is None and any(loop for loop in sf_data.loops if loop.category == _pk_char_category):

                    key_items = self.aux_key_items[file_type][content_subtype][_pk_char_category]
                    data_items = self.aux_data_items[file_type][content_subtype][_pk_char_category]
                    allowed_tags = self.aux_allowed_tags[file_type][content_subtype][_pk_char_category]

                    _pk_char_data = self.__nefT.check_data(sf_data, _pk_char_category, key_items, data_items, allowed_tags, None, parent_pointer=parent_pointer,
                                                           enforce_allowed_tags=(file_type == 'nmr-star'),
                                                           excl_missing_data=self.__excl_missing_data)[0]

                pk_id_name = 'Peak_ID'
                dim_id_name = 'Spectral_dim_ID'
                position_name = 'Chem_shift_val'

                if _pk_char_data is not None:

                    for i in _pk_char_data:

                        j = i[dim_id_name] - 1

                        if j >= num_dim or min_points[j] is None or max_points[j] is None:
                            continue

                        position = i[position_name]

                        if position < min_points[j] or position > max_points[j]:

                            err = '[Check row of %s %s] %s %s is not within expected range (min_position %s, max_position %s, absolute_peak_positions %s). Please check for reference frequency and spectral width.' % (pk_id_name, i[pk_id_name], position_name, position, min_points[j], max_points[j], abs[j])

                            self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setWarning()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ Warning  - %s\n" % err)

                        if min_limits[j] is None or max_limits[j] is None:
                            continue

                        if position < min_limits[j] or position > max_limits[j]:

                            err = '[Check row of %s %s] %s %s is not within expected range (min_position %s, max_position %s, absolute_peak_positions %s), which exceeds limit of current probe design (%s kHz). Please check for reference frequency and spectral width.' % (pk_id_name, i[pk_id_name], position_name, position, min_limits[j], max_limits[j], abs[j], self.hard_probe_limit / 1000.0)

                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ ValueError  - %s\n" % err)

            except LookupError as e:

                item = 'format_issue' if 'Unauthorized item' in str(e) else 'missing_mandatory_item'

                self.report.error.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ LookupError  - %s" % str(e))

            except ValueError as e:

                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ ValueError  - %s" % str(e))

            except Exception as e:

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ Error  - %s" % str(e))
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ Error  - %s" % str(e))

        if lp_category == '_Spectral_dim_transfer':

            for i in aux_data:
                for name in [j['name'] for j in self.aux_key_items[file_type][content_subtype][lp_category]]:
                    if i[name] not in range(1, max_dim):

                        err = "%s %r must be one of %s." % (name, i[name], range(1, max_dim))

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testDataConsistencyInAuxLoopOfSpectralPeakAlt() ++ ValueError  - %s\n" % err)

    def __testSfTagConsistency(self):
        """ Perform consistency test on saveframe tags.
        """

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            if self.__star_data_type[fileListId] != 'Entry':
                continue

            for content_subtype in input_source_dic['content_subtype'].keys():

                if content_subtype == 'entity':
                    continue

                sf_category = self.sf_categories[file_type][content_subtype]

                parent_keys = set()

                list_id = 1  # tentative parent key if not exists

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if self.__combined_mode and sf_data.tag_prefix != self.sf_tag_prefixes[file_type][content_subtype]:

                        err = "Saveframe tag prefix %r did not match with %r in %r saveframe." % (sf_data.tag_prefix, self.sf_tag_prefixes[file_type][content_subtype], sf_framecode)

                        self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testSfTagConsistency() ++ Error  - %s" % err)
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testSfTagConsistency() ++ Error  - %s\n" % err)

                    try:

                        sf_tag_items = copy.copy(self.sf_tag_items[file_type][content_subtype])

                        if not self.__combined_mode:
                            for sf_tag_item in sf_tag_items:
                                if sf_tag_item['name'] == 'sf_framecode' if file_type == 'nef' else 'Sf_framecode':
                                    sf_tag_item['mandatory'] = False

                        sf_tag_data = self.__nefT.check_sf_tag(sf_data, file_type, sf_category, sf_tag_items, self.sf_allowed_tags[file_type][content_subtype],
                                                               enforce_non_zero=True, enforce_sign=True, enforce_range=True, enforce_enum=True)

                        self.__testParentChildRelation(file_name, file_type, content_subtype, parent_keys, list_id, sf_framecode, sf_tag_data)

                        self.__sf_tag_data[content_subtype].append({'file_name': file_name, 'sf_framecode': sf_framecode, 'data': sf_tag_data})

                    except LookupError as e:

                        self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': str(e).strip("'")})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testSfTagConsistency() ++ LookupError  - %s" % str(e))

                    except ValueError as e:

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': str(e).strip("'")})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testSfTagConsistency() ++ ValueError  - %s" % str(e))

                    except UserWarning as e:

                        warns = str(e).strip("'").split('\n')

                        for warn in warns:

                            if warn == '':
                                continue

                            zero = warn.startswith('[Zero value error]')
                            nega = warn.startswith('[Negative value error]')
                            rang = warn.startswith('[Range value error]')
                            enum = warn.startswith('[Enumeration error]')

                            ignorable = False

                            if zero or nega or rang or enum:

                                p = warn.index(']') + 2
                                warn = warn[p:]

                                if zero or nega or rang:
                                    item = 'unusual_data'
                                else: # enum

                                    if warn.startswith('The mandatory type'):
                                        try:
                                            g = self.chk_desc_pat_mand.search(warn).groups()
                                        except AttributeError:
                                            g = self.chk_desc_pat_mand_one.search(warn).groups()
                                    else:
                                        try:
                                            g = self.chk_desc_pat.search(warn).groups()
                                        except AttributeError:
                                            g = self.chk_desc_pat_one.search(warn).groups()

                                    if self._sf_tag_items[file_type][content_subtype] is not None:

                                        if any(i for i in self._sf_tag_items[file_type][content_subtype] if i == g[0]):
                                            if not self.__nefT.is_mandatory_tag('_' + sf_category + '.' + g[0], file_type):
                                                ignorable = True # author provides the meta data through DepUI after upload

                                    item = 'enum_mismatch_ignorable' if ignorable else 'enum_mismatch'

                                self.report.warning.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testSfTagConsistency() ++ Warning  - %s\n" % warn)

                            else:

                                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testSfTagConsistency() ++ Error  - %s" % warn)
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testSfTagConsistency() ++ Error  - %s" % warn)

                        # try to parse data without constraints

                        try:

                            sf_tag_data = self.__nefT.check_sf_tag(sf_data, file_type, sf_category, sf_tag_items, self.sf_allowed_tags[file_type][content_subtype],
                                                                   enforce_non_zero=False, enforce_sign=False, enforce_range=False, enforce_enum=False)

                            self.__testParentChildRelation(file_name, file_type, content_subtype, parent_keys, list_id, sf_framecode, sf_tag_data)

                            self.__sf_tag_data[content_subtype].append({'file_name': file_name, 'sf_framecode': sf_framecode, 'data': sf_tag_data})

                        except:  # noqa: E722 pylint: disable=bare-except
                            pass

                    except Exception as e:

                        self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testSfTagConsistency() ++ Error  - %s" % str(e))
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testSfTagConsistency() ++ Error  - %s" % str(e))

                    list_id += 1

        return self.report.getTotalErrors() == __errors

    def __testParentChildRelation(self, file_name, file_type, content_subtype, parent_keys, list_id, sf_framecode, sf_tag_data):
        """ Perform consistency test on saveframe category and loop category relationship of interesting loops.
        """

        if file_type == 'nef' or content_subtype in ('entry_info', 'entity'):
            return True

        __errors = self.report.getTotalErrors()

        key_base = self.sf_tag_prefixes['nmr-star'][content_subtype].lstrip('_')

        parent_key_name = key_base + '.ID'
        child_key_name = key_base + '_ID'

        try:

            if parent_key_name in sf_tag_data:
                parent_key = sf_tag_data[parent_key_name]
            else:
                parent_key = list_id

            if parent_key in parent_keys:

                err = "%s %r must be unique." % (parent_key_name, parent_key)

                self.report.error.appendDescription('duplicated_index', {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': err})
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testParentChildRelation() ++ KeyError  - %s\n" % err)

            index_tag = self.index_tags[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            lp_data = next((l['data'] for l in self.__lp_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode), None)

            if lp_data is not None:

                for i in lp_data:
                    if child_key_name in i and i[child_key_name] != parent_key:

                        if index_tag is None:
                            err = '%s %r must be %s.' % (child_key_name, i[child_key_name], parent_key)
                        else:
                            err = '[Check row of %s %s] %s %r must be %s.' % (index_tag, i[index_tag], child_key_name, i[child_key_name], parent_key)

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testParentChildRelation() ++ ValueError  - %s\n" % err)

            for lp_category in self.aux_lp_categories[file_type][content_subtype]:

                aux_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == lp_category), None)  # noqa: E741

                if aux_data is not None:
                    for i in aux_data:
                        if child_key_name in i and i[child_key_name] != parent_key:

                            if index_tag is None:
                                err = '%s %r must be %s.' % (child_key_name, i[child_key_name], parent_key)
                            else:
                                err = '[Check row of %s %s] %s %r must be %s.' % (index_tag, i[index_tag], child_key_name, i[child_key_name], parent_key)

                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__testParentChildRelation() ++ ValueError  - %s\n" % err)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testParentChildRelation() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testParentChildRelation() ++ Error  - %s" % str(e))

        return self.report.getTotalErrors() == __errors

    def __validateCSValue(self):
        """ Validate assigned chemical shift value based on BMRB chemical shift statistics.
        """

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'chem_shift'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                """
                err = "Assigned chemical shift loop does not exist in %r file." % file_name

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateCSValue() ++ Error  - %s" % err)
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Error  - %s\n" % err)
                """
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            add_details = False

            if self.__star_data_type[fileListId] == 'Loop':

                sf_data = self.__star_data[fileListId]
                sf_framecode = ''

                add_details |= self.__validateCSValue__(fileListId, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

            elif self.__star_data_type[fileListId] == 'Saveframe':

                sf_data = self.__star_data[fileListId]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                add_details |= self.__validateCSValue__(fileListId, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

            else:

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    add_details |= self.__validateCSValue__(fileListId, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category)

            if add_details:
                self.__depositNmrData()

        return self.report.getTotalErrors() == __errors

    def __validateCSValue__(self, file_list_id, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category):
        """ Validate assigned chemical shift value based on BMRB chemical shift statistics.
        """

        item_names = self.item_names_in_cs_loop[file_type]
        chain_id_name = item_names['chain_id']
        seq_id_name = item_names['seq_id']
        comp_id_name = item_names['comp_id']
        atom_id_name = item_names['atom_id']
        value_name = item_names['value']
        ambig_code_name = 'Ambiguity_code' # NMR-STAR specific

        full_value_name = lp_category + '.' + value_name

        #index_tag = self.index_tags[file_type][content_subtype]
        max_cs_err = self.chem_shift_error['max_inclusive']

        add_details = False

        try:

            if file_type == 'nmr-star':

                if __pynmrstar_v3_2__:
                    loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop(lp_category)
                else:
                    loop = sf_data if self.__star_data_type[file_list_id] == 'Loop' else sf_data.get_loop_by_category(lp_category)

                details_col = loop.tags.index('Details') if 'Details' in loop.tags and self.__leave_intl_note else -1

            if file_type == 'nef' or (not self.__nonblk_anomalous_cs):
                lp_data = next(l['data'] for l in self.__lp_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode)  # noqa: E741

            else:
                key_items = self.key_items[file_type][content_subtype]
                data_items = self.data_items[file_type][content_subtype]

                try:
                    lp_data = self.__nefT.check_data(sf_data, lp_category, key_items, data_items, None, None, None,
                                                     enforce_allowed_tags=(file_type == 'nmr-star'),
                                                     excl_missing_data=self.__excl_missing_data)[0]
                except:  # noqa: E722 pylint: disable=bare-except
                    return False

            chk_row_tmp = "[Check row of {0} %s, {1} %s, {2} %s, {3} %s".format(chain_id_name, seq_id_name, comp_id_name, atom_id_name)
            row_tmp = "{0} %s, {1} %s, {2} %s, {3} %s".format(chain_id_name, seq_id_name, comp_id_name, atom_id_name)

            methyl_cs_vals = {}

            for l, i in enumerate(lp_data):  # noqa: E741
                chain_id = i[chain_id_name]
                seq_id = i[seq_id_name]
                comp_id = i[comp_id_name]
                atom_id = i[atom_id_name]
                value = i[value_name]

                if value in self.empty_value:
                    continue

                if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                    _atom_id, ambig_code, details = self.__getAtomIdListWithAmbigCode(file_type, comp_id, atom_id)

                    if len(_atom_id) == 0:
                        continue

                    if len(_atom_id) == 1 and atom_id == _atom_id[0]:
                        atom_id_ = atom_id
                        atom_name = atom_id

                        if details is not None:
                            atom_name += ' (' + details.rstrip('.') + ')'

                    else:
                        atom_name = atom_id + ' (e.g. '

                        for atom_id_ in _atom_id:
                            atom_name += atom_id_ + ' '

                        atom_name = atom_name.rstrip() + ')'

                        # representative atom id
                        atom_id_ = _atom_id[0]

                else:
                    atom_id_ = atom_id
                    atom_name = atom_id

                has_cs_stat = False

                # non-standard residue
                if self.__get1LetterCode(comp_id) == 'X':

                    neighbor_comp_ids = set(j[comp_id_name] for j in lp_data if j[chain_id_name] == chain_id and abs(j[seq_id_name] - seq_id) < 4 and j[seq_id_name] != seq_id)

                    polypeptide_like = False

                    for comp_id2 in neighbor_comp_ids:
                        polypeptide_like |= self.__csStat.getTypeOfCompId(comp_id2)[0]

                    for cs_stat in self.__csStat.get(comp_id):

                        if cs_stat['atom_id'] == atom_id_ and cs_stat['count'] > 0:
                            min_value = cs_stat['min']
                            max_value = cs_stat['max']
                            avg_value = cs_stat['avg']
                            std_value = cs_stat['std']

                            has_cs_stat = True

                            if atom_id_.startswith('H') and 'methyl' in cs_stat['desc']:
                                _atom_id = self.__getRepresentativeAtomId(file_type, comp_id, atom_id)
                                _, methyl_h_list = self.__nefT.get_group(comp_id, _atom_id)

                                name_len = [len(n) for n in methyl_h_list]
                                max_len = max(name_len)
                                min_len = min(name_len)

                                if max_len == min_len or len(atom_id) == max_len:
                                    _atom_id = atom_id[:-1]
                                else: # For example, HEM HM[A-D]
                                    _atom_id = atom_id

                                methyl_cs_key = "%s %04d %s" % (chain_id, seq_id, _atom_id)

                                if methyl_cs_key not in methyl_cs_vals:
                                    methyl_cs_vals[methyl_cs_key] = value

                                elif value != methyl_cs_vals[methyl_cs_key]:

                                    err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] Chemical shift values in the same methyl group (%s %s vs %s) are inconsistent.' %\
                                          (full_value_name, value, methyl_cs_vals[methyl_cs_key])

                                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                    self.report.setError()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % err)

                                    break

                            if std_value is None or std_value <= 0.0:

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] Insufficient chemical shift statistics on comp_id %s, atom_id %s is available to verify %s %s (avg %s).' %\
                                       (comp_id, atom_name, full_value_name, value, avg_value)

                                self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                break

                            if avg_value is None:

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] Insufficient chemical shift statistics on comp_id %s, atom_id %s is available to verify %s %s.' %\
                                       (comp_id, atom_name, full_value_name, value)

                                self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                break

                            z_score = float('{:.2f}'.format((value - avg_value) / std_value))
                            sigma = abs(z_score)

                            if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):
                                tolerance = std_value

                                if (value < min_value - tolerance or value > max_value + tolerance) and sigma > self.cs_anomalous_error_scaled_by_sigma:

                                    na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                    pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                    if na is None and pa is None:

                                        err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                              '] %s %s (%s:%s:%s, atom_id %s) is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.' %\
                                              (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                        err_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.' %\
                                              (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                        if self.__nonblk_anomalous_cs:

                                            self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err, 'value': value, 'z_score': z_score, 'description_alt': err_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % err)

                                            if file_type == 'nmr-star' and details_col != -1:
                                                _details = loop.data[l][details_col]
                                                details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score)
                                                if _details in self.empty_value or (details not in _details):
                                                    if _details in self.empty_value:
                                                        loop.data[l][details_col] = details
                                                    else:
                                                        loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                    add_details = True

                                        else:

                                            self.report.error.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err, 'value': value, 'z_score': z_score, 'description_alt': err_alt, 'sigma': sigma})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % err)

                                    elif pa is None:

                                        warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                               '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                               (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                        warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                               (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                        if na['ring_angle'] - self.magic_angle * z_score > 0.0 or self.__nonblk_anomalous_cs:

                                            self.report.warning.appendDescription('anomalous_data' if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic else 'unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                            if file_type == 'nmr-star' and details_col != -1 and (na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic):
                                                _details = loop.data[l][details_col]
                                                details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring %s:%s:%s is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score, na['chain_id'], na['seq_id'], na['comp_id'], na['ring_distance'], na['ring_angle'])
                                                if _details in self.empty_value or (details not in _details):
                                                    if _details in self.empty_value:
                                                        loop.data[l][details_col] = details
                                                    else:
                                                        loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                    add_details = True

                                        else:

                                            self.report.error.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % warn)

                                    else:

                                        warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                               '] %s %s (%s:%s:%s, atom_id %s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                               (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                        warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                               (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                        self.report.warning.appendDescription('anomalous_data' if pa['distance'] > self.vicinity_paramagnetic else 'unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                        if file_type == 'nmr-star' and details_col != -1 and pa['distance'] > self.vicinity_paramagnetic:
                                            _details = loop.data[l][details_col]
                                            details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom %s:%s:%s is located at a distance of %s angstroms.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score, pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['distance'])
                                            if _details in self.empty_value or (details not in _details):
                                                if _details in self.empty_value:
                                                    loop.data[l][details_col] = details
                                                else:
                                                    loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                add_details = True

                                elif sigma > self.cs_anomalous_error_scaled_by_sigma:

                                    na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                    pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                    if na is None and pa is None:

                                        warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                               '] %s %s (%s:%s:%s:%s) must be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.' %\
                                               (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                        warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.' %\
                                               (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                        self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                    elif pa is None:

                                        if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic:

                                            warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                                   '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                   (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                    na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                            warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                   (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                    na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                            self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                    else:

                                        if pa['distance'] > self.vicinity_paramagnetic:

                                            warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                                   '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                   (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                    pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                            warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                   (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                    pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                            self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                elif sigma > self.cs_unusual_error_scaled_by_sigma:

                                    na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                    pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                    warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                           '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f).' %\
                                           (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                    warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s).' %\
                                           (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                    if na is not None:

                                        if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic:
                                            warn += ' The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                    (na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])
                                            warn_alt += ' The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                    (na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])
                                        else:
                                            warn = None
                                            warn_alt = None

                                    elif pa is not None:

                                        if pa['distance'] > self.vicinity_paramagnetic:
                                            warn += ' The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                    (pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])
                                            warn_alt += ' The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                    (pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])
                                        else:
                                            warn = None
                                            warn_alt = None

                                    else:
                                        warn += ' Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.'
                                        warn_alt += ' Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.'

                                    if warn is not None:
                                        self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                elif not cs_stat['primary'] and cs_stat['norm_freq'] < 0.03:

                                    warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] %s %s is an unusual/rare assignment. Occurrence of %s in %s is %s %% in BMRB archive.' %\
                                           (full_value_name, value, atom_id, comp_id, cs_stat['norm_freq'] * 100.0)

                                    self.report.warning.appendDescription('unusual/rare_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                            else:
                                tolerance = std_value * 10.0

                                if min_value < max_value and (value < min_value - tolerance or value > max_value + tolerance) and sigma > self.cs_anomalous_error_scaled_by_sigma:

                                    na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                    pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                    if na is None and pa is None:

                                        err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                              '] %s %s (%s:%s:%s:%s) is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.' %\
                                              (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                        err_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.' %\
                                              (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                        if self.__nonblk_anomalous_cs:

                                            self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err, 'value': value, 'z_score': z_score, 'description_alt': err_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % err)

                                            if file_type == 'nmr-star' and details_col != -1:
                                                _details = loop.data[l][details_col]
                                                details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score)
                                                if _details in self.empty_value or (details not in _details):
                                                    if _details in self.empty_value:
                                                        loop.data[l][details_col] = details
                                                    else:
                                                        loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                    add_details = True

                                        else:

                                            self.report.error.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err, 'value': value, 'z_score': z_score, 'description_alt': err_alt, 'sigma': sigma})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % err)

                                    elif pa is None:

                                        warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                               '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                               (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                        warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                               (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                        if na['ring_angle'] - self.magic_angle * z_score > 0.0 or self.__nonblk_anomalous_cs:

                                            if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic:

                                                self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                                self.report.setWarning()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                                if file_type == 'nmr-star' and details_col != -1:
                                                    _details = loop.data[l][details_col]
                                                    details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring %s:%s:%s is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score, na['chain_id'], na['seq_id'], na['comp_id'], na['ring_distance'], na['ring_angle'])
                                                    if _details in self.empty_value or (details not in _details):
                                                        if _details in self.empty_value:
                                                            loop.data[l][details_col] = details
                                                        else:
                                                            loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                        add_details = True

                                        else:

                                            self.report.error.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % warn)

                                    else:

                                        if pa['distance'] > self.vicinity_paramagnetic:

                                            warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                                   '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                   (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                    pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                            warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                   (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                    pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                            self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                            if file_type == 'nmr-star' and details_col != -1:
                                                _details = loop.data[l][details_col]
                                                details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom %s:%s:%s is located at a distance of %s angstroms.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score, pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['distance'])
                                                if _details in self.empty_value or (details not in _details):
                                                    if _details in self.empty_value:
                                                        loop.data[l][details_col] = details
                                                    else:
                                                        loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                    add_details = True

                                elif sigma > self.cs_anomalous_error_scaled_by_sigma:

                                    na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                    pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                    if na is None and pa is None:

                                        warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                               '] %s %s (%s:%s:%s:%s) must be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.' %\
                                               (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                        warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.' %\
                                               (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                        self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                    elif pa is None:

                                        if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic:

                                            warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                                   '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                   (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                    na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                            warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                   (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                    na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                            self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                    else:

                                        if pa['distance'] > self.vicinity_paramagnetic:

                                            warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                                   '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                   (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                    pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                            warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                   (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                    pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                            self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                            break

                # standard residue
                else:

                    for cs_stat in self.__csStat.get(comp_id, self.report.isDiamagnetic()):

                        if cs_stat['atom_id'] == atom_id_ and cs_stat['count'] > 0:
                            min_value = cs_stat['min']
                            max_value = cs_stat['max']
                            avg_value = cs_stat['avg']
                            std_value = cs_stat['std']

                            has_cs_stat = True

                            if atom_id_.startswith('H') and 'methyl' in cs_stat['desc']:
                                methyl_cs_key = "%s %04d %s" % (chain_id, seq_id, atom_id_[:-1])

                                if methyl_cs_key not in methyl_cs_vals:
                                    methyl_cs_vals[methyl_cs_key] = value

                                elif value != methyl_cs_vals[methyl_cs_key]:

                                    err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] Chemical shift values in the same methyl group (%s %s vs %s) are inconsistent.' %\
                                          (full_value_name, value, methyl_cs_vals[methyl_cs_key])

                                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                    self.report.setError()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % err)

                                    break

                            if std_value is None or std_value <= 0.0:

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] Insufficient chemical shift statistics on comp_id %s, atom_id %s is available to verify %s %s (avg %s).' %\
                                       (comp_id, atom_name, full_value_name, value, avg_value)

                                self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                break

                            if avg_value is None:

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] Insufficient chemical shift statistics on comp_id %s, atom_id %s is available to verify %s %s.' %\
                                       (comp_id, atom_name, full_value_name, value)

                                self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                break

                            z_score = float('{:.2f}'.format((value - avg_value) / std_value))
                            sigma = abs(z_score)
                            tolerance = std_value

                            if (value < min_value - tolerance or value > max_value + tolerance) and sigma > self.cs_unusual_error_scaled_by_sigma:

                                na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                if na is None and pa is None:

                                    err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                          '] %s %s (%s:%s:%s:%s) is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.' %\
                                          (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                    err_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.' %\
                                          (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                    if self.__nonblk_anomalous_cs:

                                        self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err, 'value': value, 'z_score': z_score, 'description_alt': err_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % err)

                                        if file_type == 'nmr-star' and details_col != -1:
                                            _details = loop.data[l][details_col]
                                            details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity. Please check for folded/aliased signals.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score)
                                            if _details in self.empty_value or (details not in _details):
                                                if _details in self.empty_value:
                                                    loop.data[l][details_col] = details
                                                else:
                                                    loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                add_details = True

                                    else:

                                        self.report.error.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err, 'value': value, 'z_score': z_score, 'description_alt': err_alt, 'sigma': sigma})
                                        self.report.setError()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % err)

                                elif pa is None:

                                    warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                           '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                           (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                            na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                    warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                           (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                            na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                    if na['ring_angle'] - self.magic_angle * z_score > 0.0 or self.__nonblk_anomalous_cs:

                                        self.report.warning.appendDescription('anomalous_data' if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic else 'unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                        if file_type == 'nmr-star' and details_col != -1 and (na['ring_angle'] - self.magic_angle * z_score > 0.0 or self.__nonblk_anomalous_cs):
                                            _details = loop.data[l][details_col]
                                            details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring %s:%s:%s is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score, na['chain_id'], na['seq_id'], na['comp_id'], na['ring_distance'], na['ring_angle'])
                                            if _details in self.empty_value or (details not in _details):
                                                if _details in self.empty_value:
                                                    loop.data[l][details_col] = details
                                                else:
                                                    loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                                add_details = True

                                    else:

                                        self.report.error.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setError()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError  - %s\n" % warn)

                                else:

                                    warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                           '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                           (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                            pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                    warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                           (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                            pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                    self.report.warning.appendDescription('anomalous_data' if pa['distance'] > self.vicinity_paramagnetic else 'unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                    if file_type == 'nmr-star' and details_col != -1 and pa['distance'] > self.vicinity_paramagnetic:
                                        _details = loop.data[l][details_col]
                                        details = '%s %s is not within expected range (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom %s:%s:%s is located at a distance of %s angstroms.\n' % (full_value_name, value, avg_value, std_value, min_value, max_value, z_score, pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['distance'])
                                        if _details in self.empty_value or (details not in _details):
                                            if _details in self.empty_value:
                                                loop.data[l][details_col] = details
                                            else:
                                                loop.data[l][details_col] += ('' if '\n' in _details else '\n') + details
                                            add_details = True

                            elif sigma > self.cs_unusual_error_scaled_by_sigma: # Set 5.0 to be consistent with validation report

                                na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                if na is None and pa is None:

                                    warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                           '] %s %s (%s:%s:%s:%s) must be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.' %\
                                           (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                    warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.' %\
                                           (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                    self.report.warning.appendDescription('anomalous_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                elif pa is None:

                                    if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic:

                                        warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                               '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                               (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                        warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                               (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])

                                        self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                                else:

                                    if pa['distance'] > self.vicinity_paramagnetic:

                                        warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                               '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                               (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score,
                                                pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                        warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s). The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                               (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value,
                                                pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])

                                        self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)
                                """ Can skip this to be consistent with validation report
                            elif sigma > self.cs_unusual_error_scaled_by_sigma:

                                na = self.__getNearestAromaticRing(chain_id, seq_id, atom_id_, self.cutoff_aromatic)
                                pa = self.__getNearestParaFerroMagneticAtom(chain_id, seq_id, atom_id_, self.cutoff_paramagnetic)

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) +\
                                       '] %s %s (%s:%s:%s:%s) should be verified (avg %s, std %s, min %s, max %s, Z_score %.2f).' %\
                                       (full_value_name, value, chain_id, seq_id, comp_id, atom_name, avg_value, std_value, min_value, max_value, z_score)

                                warn_alt = 'Verify chemical shift value for %s:%s:%s:%s (%s ppm, %.2f sigma), which is outside of expected range (%.2f ~ %.2f ppm, avg %s, std %s, min %s, max %s).' %\
                                       (chain_id, seq_id, comp_id, atom_name, value, sigma, avg_value + 5.0 * std_value, avg_value - 5.0 * std_value, avg_value, std_value, min_value, max_value)

                                if na is not None:

                                    if na['ring_angle'] - self.magic_angle * z_score < 0.0 or na['ring_distance'] > self.vicinity_aromatic:
                                        warn += ' The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                (na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])
                                        warn_alt += ' The nearest aromatic ring (%s:%s:%s:%s) is located at a distance of %s angstroms, and has an elevation angle of %s degrees with the ring plane.' %\
                                                (na['chain_id'], na['seq_id'], na['comp_id'], na['ring_atoms'], na['ring_distance'], na['ring_angle'])
                                    else:
                                        warn = None
                                        warn_alt = None

                                elif pa is not None:

                                    if pa['distance'] > self.vicinity_paramagnetic:
                                        warn += ' The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                (pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])
                                        warn_alt += ' The nearest paramagnetic/ferromagnetic atom (%s:%s:%s:%s) is located at a distance of %s angstroms.' %\
                                                (pa['chain_id'], pa['seq_id'], pa['comp_id'], pa['atom_id'], pa['distance'])
                                    else:
                                        warn = None
                                        warn_alt = None

                                else:
                                    warn += ' Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.'
                                    warn_alt += ' Neither aromatic ring nor paramagnetic/ferromagnetic atom were found in the vicinity.'

                                if warn is not None:
                                    self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn, 'value': value, 'z_score': z_score, 'description_alt': warn_alt, 'sigma': sigma})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)
                                """
                            elif not cs_stat['primary'] and cs_stat['norm_freq'] < 0.03:

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] %s %s is an unusual/rare assignment. Occurrence of %s in %s is %s %% in BMRB archive.' %\
                                       (full_value_name, value, atom_id, comp_id, '{:.1f}'.format(cs_stat['norm_freq'] * 100.0))

                                self.report.warning.appendDescription('unusual/rare_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                            break

                if not has_cs_stat:

                    warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_name) + '] No chemical shift statistics is available to verify %s %s.' %\
                           (full_value_name, value)

                    self.report.warning.appendDescription('unusual_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                    self.report.setWarning()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                # check ambiguity code
                if file_type == 'nmr-star' and ambig_code_name in i:
                    ambig_code = i[ambig_code_name]

                    if ambig_code in self.empty_value or ambig_code == 1:
                        continue

                    _atom_id = atom_id

                    if self.__isNmrAtomName(comp_id, atom_id):
                        _atom_id = self.__getRepresentativeAtomId(file_type, comp_id, atom_id)

                    allowed_ambig_code = self.__csStat.getMaxAmbigCodeWoSetId(comp_id, _atom_id)

                    if ambig_code in (2, 3):

                        ambig_code_desc = 'ambiguity of geminal atoms or geminal methyl proton groups' if ambig_code == 2 else 'aromatic atoms on opposite sides of symmetrical rings'

                        _atom_id2 = self.__csStat.getGeminalAtom(comp_id, _atom_id)

                        if ambig_code != allowed_ambig_code:

                            if allowed_ambig_code == 1:

                                try:

                                    j = next(j for j in lp_data if j[chain_id_name] == chain_id and j[seq_id_name] == seq_id and j[comp_id_name] == comp_id and j[atom_id_name] == _atom_id2)

                                    loop.data[lp_data.index(j)][loop.tags.index(ambig_code_name)] = 1

                                except StopIteration:
                                    pass

                            elif allowed_ambig_code > 0:

                                err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) + "] Invalid %s %r (allowed ambig_code %s) in a loop %s." %\
                                      (ambig_code_name, ambig_code, [1, allowed_ambig_code, 4, 5, 6, 9], lp_category)

                                self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError - %s\n" % err)

                        try:

                            j = next(j for j in lp_data if j[chain_id_name] == chain_id and j[seq_id_name] == seq_id and j[comp_id_name] == comp_id and j[atom_id_name] == _atom_id2)

                            ambig_code2 = j[ambig_code_name]

                            if ambig_code2 != ambig_code:
                                loop.data[lp_data.index(j)][loop.tags.index(ambig_code_name)] = ambig_code

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) + "] %s %r indicates %s. However, %s %s of %s %s is inconsistent." %\
                                       (ambig_code_name, ambig_code, ambig_code_desc, ambig_code_name, ambig_code2, atom_id_name, _atom_id2)

                                self.report.warning.appendDescription('ambiguity_code_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                        except StopIteration:
                            """
                            warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) + "] %s %r indicates %s. However, row of %s %s of the same residue was not found." %\
                                   (ambig_code_name, ambig_code, ambig_code_desc, atom_id_name, _atom_id2)

                            self.report.warning.appendDescription('bad_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                            self.report.setWarning()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)
                            """

                    elif ambig_code in (4, 5, 6, 9):

                        ambig_set_id_name = 'Ambiguity_set_ID'

                        if ambig_set_id_name not in i:

                            err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) + "] %s %r requires %r loop tag." %\
                                  (ambig_code_name, ambig_code, ambig_set_id_name)

                            self.report.error.appendDescription('missing_mandatory_item', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ LookupError  - %s\n" % err)

                        else:

                            ambig_set_id = i[ambig_set_id_name]

                            if ambig_set_id in self.empty_value:

                                warn = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) + '] %s %r requires %s value.' %\
                                       (ambig_code_name, ambig_code, ambig_set_id_name)

                                self.report.warning.appendDescription('missing_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Warning  - %s\n" % warn)

                            else:

                                ambig_set = [j for j in lp_data if j[ambig_set_id_name] == ambig_set_id and j != i]

                                if len(ambig_set) == 0:

                                    err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) + '] %s %r requires other rows sharing %s %s.' %\
                                          (ambig_code_name, ambig_code, ambig_set_id_name, ambig_set_id)

                                    self.report.error.appendDescription('missing_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                    self.report.setError()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ LookupError  - %s\n" % err)

                                # Intra-residue ambiguities
                                elif ambig_code == 4:

                                    for j in ambig_set:
                                        chain_id2 = j[chain_id_name]
                                        seq_id2 = j[seq_id_name]
                                        comp_id2 = j[comp_id_name]
                                        atom_id2 = j[atom_id_name]

                                        _atom_id2 = atom_id2

                                        if self.__isNmrAtomName(comp_id2, atom_id2):
                                            _atom_id2 = self.__getRepresentativeAtomId(file_type, comp_id2, atom_id2)

                                        if (chain_id2 != chain_id or seq_id2 != seq_id or comp_id2 != comp_id) and _atom_id < _atom_id2:

                                            err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) +\
                                                  ", %s %r, %s %s] It indicates intra-residue ambiguities. However, row of " % (ambig_code_name, ambig_code, ambig_set_id_name, ambig_set_id) +\
                                                  row_tmp % (chain_id2, seq_id2, comp_id2, atom_id2) + ' exists.'

                                            self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError - %s\n" % err)

                                # Inter-residue ambiguities
                                elif ambig_code == 5:

                                    for j in ambig_set:
                                        chain_id2 = j[chain_id_name]
                                        seq_id2 = j[seq_id_name]
                                        comp_id2 = j[comp_id_name]
                                        atom_id2 = j[atom_id_name]

                                        _atom_id2 = atom_id2

                                        if self.__isNmrAtomName(comp_id2, atom_id2):
                                            _atom_id2 = self.__getRepresentativeAtomId(file_type, comp_id2, atom_id2)

                                        if ((chain_id2 != chain_id and chain_id < chain_id2) or (seq_id2 == seq_id and _atom_id < _atom_id2)):

                                            err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) +\
                                                  ", %s %r, %s %s] It indicates inter-residue ambiguities. However, row of " % (ambig_code_name, ambig_code, ambig_set_id_name, ambig_set_id) +\
                                                  row_tmp % (chain_id2, seq_id2, comp_id2, atom_id2) + ' exists.'

                                            self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError - %s\n" % err)

                                # Inter-molecular ambiguities
                                elif ambig_code == 6:

                                    for j in ambig_set:
                                        chain_id2 = j[chain_id_name]
                                        seq_id2 = j[seq_id_name]
                                        comp_id2 = j[comp_id_name]
                                        atom_id2 = j[atom_id_name]
                                        value2 = j[value_name]

                                        _atom_id2 = atom_id2

                                        if self.__isNmrAtomName(comp_id2, atom_id2):
                                            _atom_id2 = self.__getRepresentativeAtomId(file_type, comp_id2, atom_id2)

                                        if chain_id2 == chain_id and (seq_id < seq_id2 or (seq_id == seq_id2 and _atom_id < _atom_id2)):

                                            err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) +\
                                                  ", %s %r, %s %s] It indicates inter-molecular ambiguities. However, row of " % (ambig_code_name, ambig_code, ambig_set_id_name, ambig_set_id) +\
                                                  row_tmp % (chain_id2, seq_id2, comp_id2, atom_id2) + ' exists.'

                                            self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError - %s\n" % err)

                                for j in ambig_set:
                                    chain_id2 = j[chain_id_name]
                                    seq_id2 = j[seq_id_name]
                                    comp_id2 = j[comp_id_name]
                                    atom_id2 = j[atom_id_name]
                                    value2 = j[value_name]

                                    _atom_id2 = atom_id2

                                    if self.__isNmrAtomName(comp_id2, atom_id2):
                                        _atom_id2 = self.__getRepresentativeAtomId(file_type, comp_id2, atom_id2)

                                    if _atom_id[0] != _atom_id2[0] and _atom_id < _atom_id2:

                                        err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) +\
                                              ", %s %r, %s %s] However, observation nucleus of " % (ambig_code_name, ambig_code, ambig_set_id_name, ambig_set_id) +\
                                              row_tmp % (chain_id2, seq_id2, comp_id2, atom_id2) + ' sharing the same %s differs.' % ambig_set_id_name

                                        self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                        self.report.setError()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError - %s\n" % err)

                                    elif abs(value2 - value) > max_cs_err and value < value2:

                                        err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) +\
                                              ", %s %s, %s %r, %s %s] However, %s %s of " % (value_name, value, ambig_code_name, ambig_code, ambig_set_id_name, ambig_set_id, value_name, value2) +\
                                              row_tmp % (chain_id2, seq_id2, comp_id2, atom_id2) +\
                                              'differs by %s (tolerance %s).' % (value2 - value, max_cs_err)

                                        self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                        self.report.setError()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError - %s\n" % err)

                    else:

                        err = chk_row_tmp % (chain_id, seq_id, comp_id, atom_id) + "] Invalid ambiguity code %r (allowed ambig_code %s) in a loop." %\
                              (ambig_code, self.bmrb_ambiguity_codes)

                        self.report.error.appendDescription('invalid_ambiguity_code', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ ValueError - %s\n" % err)

        except StopIteration:

            err = "Assigned chemical shifts of %r saveframe did not parsed properly. Please fix problems reported." % sf_framecode

            self.report.error.appendDescription('missing_mandatory_content', {'file_name': file_name, 'description': err})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Error  - %s\n" % err)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__validateCSValue() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__validateCSValue() ++ Error  - %s" % str(e))

        return add_details

    def __testCSValueConsistencyInPkLoop(self):
        """ Perform consistency test on peak position and assignment of spectral peaks.
        """

        #if not self.__combined_mode:
        #    return False

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            if fileListId >= len(self.__star_data_type) or self.__star_data_type[fileListId] != 'Entry':
                continue

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'spectral_peak'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            cs_item_names = self.item_names_in_cs_loop[file_type]
            cs_chain_id_name = cs_item_names['chain_id']
            cs_seq_id_name = cs_item_names['seq_id']
            cs_comp_id_name = cs_item_names['comp_id']
            cs_atom_id_name = cs_item_names['atom_id']
            cs_value_name = cs_item_names['value']
            cs_error_name = cs_item_names['error']
            cs_atom_type = cs_item_names['atom_type']
            cs_iso_number = cs_item_names['isotope_number']

            for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                try:
                    cs_list = sf_data.get_tag(self.cs_list_sf_tag_name[file_type])[0]
                except:  # noqa: E722 pylint: disable=bare-except
                    continue

                try:

                    cs_data = next(l['data'] for l in self.__lp_data['chem_shift'] if l['file_name'] == file_name and l['sf_framecode'] == cs_list)  # noqa: E741

                except StopIteration:

                    if cs_list not in self.empty_value:

                        err = "Assigned chemical shifts are mandatory. Referred %r saveframe does not exist." % cs_list

                        self.report.error.appendDescription('missing_mandatory_content', {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ Error  - %s\n" % err)

                        continue

                    try:
                        cs_data = next(l['data'] for l in self.__lp_data['chem_shift'] if l['file_name'] == file_name)
                    except StopIteration:
                        continue

                try:

                    _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                    num_dim = int(_num_dim)

                    if num_dim not in range(1, self.lim_num_dim):
                        raise ValueError()

                except ValueError: # raised error already at __testIndexConsistency()
                    return False

                max_dim = num_dim + 1

                aux_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == self.aux_lp_categories[file_type][content_subtype][0]), None)

                axis_codes = []
                abs_pk_pos = []
                sp_widths = []

                if aux_data is not None and len(aux_data) > 0:
                    for i in range(1, max_dim):
                        for sp_dim in aux_data:
                            if file_type == 'nef':
                                if sp_dim['dimension_id'] != i:
                                    continue
                                axis_codes.append(sp_dim['axis_code'])
                                abs_pk_pos.append(False if 'absolute_peak_poistions' not in sp_dim else sp_dim['absolute_peak_positions'])
                                sp_width = None if 'spectral_width' not in sp_dim or 'axis_unit' not in sp_dim else sp_dim['spectral_width']
                                if 'axis_unit' in sp_dim and sp_dim['axis_unit'] == 'Hz' and 'spectrometer_frequency' in sp_dim and sp_width is not None:
                                    sp_freq = sp_dim['spectrometer_frequency']
                                    sp_width /= sp_freq
                                sp_widths.append(sp_width)
                            else:
                                if sp_dim['ID'] != i:
                                    continue
                                axis_codes.append(sp_dim['Axis_code'])
                                abs_pk_pos.append(False if 'Absolute_peak_positions' not in sp_dim else sp_dim['Absolute_peak_positions'])
                                sp_width = None if 'Sweep_width' not in sp_dim or 'Sweep_width_units' not in sp_dim else sp_dim['Sweep_width']
                                if 'Sweep_width_units' in sp_dim and sp_dim['Sweep_width_units'] == 'Hz' and 'Spectrometer_frequency' in sp_dim and sp_width is not None:
                                    sp_freq = sp_dim['Spectrometer_frequency']
                                    sp_width /= sp_freq
                                sp_widths.append(sp_width)
                            break
                else:
                    for i in range(num_dim):
                        axis_codes.append(None)
                        abs_pk_pos.append(False)
                        sp_widths.append(None)

                aux_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == self.aux_lp_categories[file_type][content_subtype][1]), None)

                onebond = [[False] * num_dim for i in range(num_dim)]
                if aux_data is not None:
                    for sp_dim_trans in aux_data:
                        if file_type == 'nef':
                            if sp_dim_trans['transfer_type'] == 'onebond':
                                dim_1 = sp_dim_trans['dimension_1']
                                dim_2 = sp_dim_trans['dimension_2']
                                onebond[dim_1 - 1][dim_2 - 1] = True
                                onebond[dim_2 - 1][dim_1 - 1] = True
                        else:
                            if sp_dim_trans['Type'] == 'onebond':
                                dim_1 = sp_dim_trans['Spectral_dim_ID_1']
                                dim_2 = sp_dim_trans['Spectral_dim_ID_2']
                                onebond[dim_1 - 1][dim_2 - 1] = True
                                onebond[dim_2 - 1][dim_1 - 1] = True

                jcoupling = [[False] * num_dim for i in range(num_dim)]
                if aux_data is not None:
                    for sp_dim_trans in aux_data:
                        if file_type == 'nef':
                            if sp_dim_trans['transfer_type'].startswith('j'):
                                dim_1 = sp_dim_trans['dimension_1']
                                dim_2 = sp_dim_trans['dimension_2']
                                jcoupling[dim_1 - 1][dim_2 - 1] = True
                                jcoupling[dim_2 - 1][dim_1 - 1] = True
                        else:
                            if sp_dim_trans['Type'].startswith('j'):
                                dim_1 = sp_dim_trans['Spectral_dim_ID_1']
                                dim_2 = sp_dim_trans['Spectral_dim_ID_2']
                                jcoupling[dim_1 - 1][dim_2 - 1] = True
                                jcoupling[dim_2 - 1][dim_1 - 1] = True

                relayed = [[False] * num_dim for i in range(num_dim)]
                if aux_data is not None:
                    for sp_dim_trans in aux_data:
                        if file_type == 'nef':
                            if sp_dim_trans['transfer_type'].startswith('relayed'):
                                dim_1 = sp_dim_trans['dimension_1']
                                dim_2 = sp_dim_trans['dimension_2']
                                relayed[dim_1 - 1][dim_2 - 1] = True
                                relayed[dim_2 - 1][dim_1 - 1] = True
                        else:
                            if sp_dim_trans['Type'].startswith('relayed'):
                                dim_1 = sp_dim_trans['Spectral_dim_ID_1']
                                dim_2 = sp_dim_trans['Spectral_dim_ID_2']
                                relayed[dim_1 - 1][dim_2 - 1] = True
                                relayed[dim_2 - 1][dim_1 - 1] = True

                item_names = []
                for dim in range(1, max_dim):
                    _d = {}
                    for k, v in self.item_names_in_pk_loop[file_type].items():
                        if '%s' in v:
                            v = v % dim
                        _d[k] = v
                    item_names.append(_d)

                chain_id_names = []
                seq_id_names = []
                comp_id_names = []
                atom_id_names = []
                position_names = []

                for d in range(num_dim):
                    chain_id_names.append(item_names[d]['chain_id'])
                    seq_id_names.append(item_names[d]['seq_id'])
                    comp_id_names.append(item_names[d]['comp_id'])
                    atom_id_names.append(item_names[d]['atom_id'])
                    position_names.append(item_names[d]['position'])

                index_tag = self.index_tags[file_type][content_subtype]
                max_cs_err = self.chem_shift_error['max_inclusive']

                try:

                    lp_data = next((l['data'] for l in self.__lp_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode), None)  # noqa: E741

                    if lp_data is not None:

                        for i in lp_data:
                            for d in range(num_dim):

                                if __pynmrstar_v3__ and not (chain_id_names[d] in i and seq_id_names[d] in i and comp_id_names[d] in i and atom_id_names[d] in i):
                                    continue

                                chain_id = i[chain_id_names[d]]
                                if chain_id in self.empty_value:
                                    continue

                                seq_id = i[seq_id_names[d]]
                                if seq_id in self.empty_value:
                                    continue

                                comp_id = i[comp_id_names[d]]
                                if comp_id in self.empty_value:
                                    continue

                                atom_id = i[atom_id_names[d]]
                                if atom_id in self.empty_value:
                                    continue

                                position = i[position_names[d]]

                                try:

                                    if file_type == 'nmr-star' and self.__isNmrAtomName(comp_id, atom_id):
                                        _atom_id = self.__getAtomIdListWithAmbigCode(file_type, comp_id, atom_id)[0]

                                        if len(_atom_id) == 0:
                                            atom_id_ = atom_id

                                        elif len(_atom_id) == 1 and atom_id == _atom_id[0]:
                                            atom_id_ = atom_id

                                        else:
                                            # representative atom id
                                            atom_id_ = _atom_id[0]

                                    else:
                                        atom_id_ = atom_id

                                    cs = next(j for j in cs_data if j[cs_chain_id_name] == chain_id and
                                                                    j[cs_seq_id_name] == seq_id and
                                                                    j[cs_comp_id_name] == comp_id and
                                                                    j[cs_atom_id_name] == atom_id_)

                                    value = cs[cs_value_name]
                                    error = cs[cs_error_name]

                                    if value in self.empty_value:
                                        continue

                                    if error is None or error < 1.0e-3 or error * self.cs_diff_error_scaled_by_sigma > max_cs_err:
                                        error = max_cs_err
                                    else:
                                        error *= self.cs_diff_error_scaled_by_sigma

                                    if abs(position - value) > error:

                                        if not abs_pk_pos[d] and sp_widths[d] is not None:
                                            if position < value:
                                                while position < value:
                                                    position += sp_widths[d]
                                            elif position > value:
                                                while position > value:
                                                    position -= sp_widths[d]

                                        if abs(position - value) > error:

                                            err = "[Check row of %s %s] Peak position of spectral peak %s %s (%s) in %r saveframe is inconsistent with the assigned chemical shift value %s (difference %s, tolerance %s) in %r saveframe." %\
                                                  (index_tag, i[index_tag], position_names[d], position, self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id), sf_framecode, value, float('{:.3f}'.format(position - value)), error, cs_list)

                                            if error >= max_cs_err:

                                                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                                self.report.setError()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ ValueError  - %s\n" % err)

                                            else:

                                                self.report.warning.appendDescription('unusual_chemical_shift', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                                self.report.setWarning()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ Warning  - %s\n" % err)

                                    axis_code = str(cs[cs_iso_number]) + cs[cs_atom_type]

                                    if axis_codes[d] is not None and axis_code != axis_codes[d]:

                                        err = '[Check row of %s %s] Assignment of spectral peak %s is inconsistent with axis code %s vs %s.' %\
                                              (index_tag, i[index_tag], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id), axis_code, axis_codes[d])

                                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                        self.report.setError()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ ValueError  - %s\n" % err)

                                except StopIteration:

                                    err = "[Check row of %s %s] Assignment of spectral peak %s was not found in assigned chemical shifts in %r saveframe." %\
                                          (index_tag, i[index_tag], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id), cs_list)

                                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                    self.report.setError()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ ValueError  - %s\n" % err)

                                if True in onebond[d]:
                                    for d2 in range(num_dim):
                                        if onebond[d][d2]:
                                            chain_id2 = i[chain_id_names[d2]]
                                            seq_id2 = i[seq_id_names[d2]]
                                            comp_id2 = i[comp_id_names[d2]]
                                            atom_id2 = i[atom_id_names[d2]]

                                            if atom_id2 is not None:
                                                diff = len(atom_id) != len(atom_id2)
                                                _atom_id = '_' + (atom_id[1:-1] if atom_id.startswith('H') and diff else atom_id[1:])
                                                _atom_id2 = '_' + (atom_id2[1:-1] if atom_id2.startswith('H') and diff else atom_id2[1:])

                                            if chain_id2 in self.empty_value or seq_id2 in self.empty_value or comp_id2 in self.empty_value or atom_id2 in self.empty_value or\
                                               (d < d2 and (chain_id2 != chain_id or seq_id2 != seq_id or comp_id2 != comp_id or _atom_id2 != _atom_id)):

                                                err = '[Check row of %s %s] Coherence transfer type is onebond. However, assignment of spectral peak is inconsistent with the type, (%s) vs (%s).' %\
                                                      (index_tag, i[index_tag], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id),
                                                       self.__getReducedAtomNotation(chain_id_names[d2], chain_id2, seq_id_names[d2], seq_id2, comp_id_names[d2], comp_id2, atom_id_names[d2], atom_id2))

                                                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                                self.report.setError()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ ValueError  - %s\n" % err)

                                if True in jcoupling[d]:
                                    for d2 in range(num_dim):
                                        if jcoupling[d][d2]:
                                            chain_id2 = i[chain_id_names[d2]]
                                            seq_id2 = i[seq_id_names[d2]]
                                            comp_id2 = i[comp_id_names[d2]]
                                            atom_id2 = i[atom_id_names[d2]]

                                            if chain_id2 in self.empty_value or seq_id2 in self.empty_value or comp_id2 in self.empty_value or atom_id2 in self.empty_value or\
                                               (d < d2 and (chain_id2 != chain_id or seq_id2 != seq_id or comp_id2 != comp_id)): # DAOTHER-7389, issue #2

                                                err = '[Check row of %s %s] Coherence transfer type is jcoupling. However, assignment of spectral peak is inconsistent with the type, (%s) vs (%s).' %\
                                                      (index_tag, i[index_tag], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id),
                                                       self.__getReducedAtomNotation(chain_id_names[d2], chain_id2, seq_id_names[d2], seq_id2, comp_id_names[d2], comp_id2, atom_id_names[d2], atom_id2))

                                                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                                self.report.setError()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ ValueError  - %s\n" % err)

                                if True in relayed[d]:
                                    for d2 in range(num_dim):
                                        if relayed[d][d2]:
                                            chain_id2 = i[chain_id_names[d2]]
                                            seq_id2 = i[seq_id_names[d2]]
                                            comp_id2 = i[comp_id_names[d2]]
                                            atom_id2 = i[atom_id_names[d2]]

                                            if chain_id2 in self.empty_value or seq_id2 in self.empty_value or comp_id2 in self.empty_value or atom_id2 in self.empty_value or\
                                               (d < d2 and (chain_id2 != chain_id or abs(seq_id2 - seq_id) > 1)): # DAOTHER-7389, issue #2

                                                err = '[Check row of %s %s] Coherence transfer type is relayed. However, assignment of spectral peak is inconsistent with the type, (%s) vs (%s).' %\
                                                      (index_tag, i[index_tag], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id),
                                                       self.__getReducedAtomNotation(chain_id_names[d2], chain_id2, seq_id_names[d2], seq_id2, comp_id_names[d2], comp_id2, atom_id_names[d2], atom_id2))

                                                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                                self.report.setError()

                                                if self.__verbose:
                                                    self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ ValueError  - %s\n" % err)

                except Exception as e:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ Error  - %s" % str(e))
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkLoop() ++ Error  - %s" % str(e))

        return self.report.getTotalErrors() == __errors

    def __testCSValueConsistencyInPkAltLoop(self):
        """ Perform consistency test on peak position and assignment of spectral peaks.
        """

        # if not self.__combined_mode:
        #    return False

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            if fileListId >= len(self.__star_data_type) or self.__star_data_type[fileListId] != 'Entry':
                continue

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if file_type == 'nef' or input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'spectral_peak_alt'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = '_Assigned_peak_chem_shift'

            cs_item_names = self.item_names_in_cs_loop[file_type]
            cs_chain_id_name = cs_item_names['chain_id']
            cs_seq_id_name = cs_item_names['seq_id']
            cs_comp_id_name = cs_item_names['comp_id']
            cs_atom_id_name = cs_item_names['atom_id']
            cs_value_name = cs_item_names['value']
            cs_error_name = cs_item_names['error']
            cs_atom_type = cs_item_names['atom_type']
            cs_iso_number = cs_item_names['isotope_number']

            for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                cs_data = None

                try:

                    cs_list = sf_data.get_tag(self.cs_list_sf_tag_name[file_type])[0]
                    _cs_list_id = sf_data.get_tag('ID')[0]

                    try:

                        cs_data = next(l['data'] for l in self.__lp_data['chem_shift'] if l['file_name'] == file_name and l['sf_framecode'] == cs_list)  # noqa: E741

                    except StopIteration:

                        if cs_list not in self.empty_value:

                            err = "Assigned chemical shifts are mandatory. Referred %r saveframe does not exist." % cs_list

                            self.report.error.appendDescription('missing_mandatory_content', {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ Error  - %s\n" % err)

                            continue

                except:  # noqa: E722 pylint: disable=bare-except
                    pass

                if cs_data is None:

                    try:

                        cs = next(l for l in self.__lp_data['chem_shift'] if l['file_name'] == file_name)

                    except StopIteration:
                        continue

                    cs_data = cs['data']
                    cs_list = cs['sf_framecode']

                    cs_sf_data = self.__getSaveframeByName(fileListId, cs_list)

                    if cs_sf_data is None:
                        continue

                    _cs_list_id = cs_sf_data.get_tag('ID')[0]

                try:

                    _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                    num_dim = int(_num_dim)

                    if num_dim not in range(1, self.lim_num_dim):
                        raise ValueError()

                except ValueError: # raised error already at __testIndexConsistency()
                    return False

                max_dim = num_dim + 1

                item_names = []
                for dim in range(1, max_dim):
                    _d = {}
                    for k, v in self.item_names_in_pk_loop[file_type].items():
                        if '%s' in v:
                            v = v % dim
                        _d[k] = v
                    item_names.append(_d)

                chain_id_names = []
                seq_id_names = []
                comp_id_names = []
                atom_id_names = []

                for i in range(num_dim):
                    chain_id_names.append(item_names[i]['chain_id'])
                    seq_id_names.append(item_names[i]['seq_id'])
                    comp_id_names.append(item_names[i]['comp_id'])
                    atom_id_names.append(item_names[i]['atom_id'])

                aux_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == self.aux_lp_categories[file_type][content_subtype][0]), None)

                axis_codes = []
                abs_pk_pos = []
                sp_widths = []

                if aux_data is not None and len(aux_data) > 0:
                    for i in range(1, max_dim):
                        for sp_dim in aux_data:
                            if sp_dim['ID'] != i:
                                continue
                            axis_codes.append(sp_dim['Axis_code'])
                            abs_pk_pos.append(False if 'Absolute_peak_positions' not in sp_dim else sp_dim['Absolute_peak_positions'])
                            sp_width = None if 'Sweep_width' not in sp_dim or 'Sweep_width_units' not in sp_dim else sp_dim['Sweep_width']
                            if 'Sweep_width_units' in sp_dim and sp_dim['Sweep_width_units'] == 'Hz' and 'Spectrometer_frequency' in sp_dim and sp_width is not None:
                                sp_freq = sp_dim['Spectrometer_frequency']
                                sp_width /= sp_freq
                            sp_widths.append(sp_width)
                            break
                else:
                    for i in range(num_dim):
                        axis_codes.append(None)
                        abs_pk_pos.append(False)
                        sp_widths.append(None)

                aux_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == self.aux_lp_categories[file_type][content_subtype][1]), None) # noqa: E501,E741

                onebond = [[False] * num_dim for i in range(num_dim)]
                if aux_data is not None:
                    for sp_dim_trans in aux_data:
                        if sp_dim_trans['Type'] == 'onebond':
                            dim_1 = sp_dim_trans['Spectral_dim_ID_1']
                            dim_2 = sp_dim_trans['Spectral_dim_ID_2']
                            onebond[dim_1 - 1][dim_2 - 1] = True
                            onebond[dim_2 - 1][dim_1 - 1] = True

                jcoupling = [[False] * num_dim for i in range(num_dim)]
                if aux_data is not None:
                    for sp_dim_trans in aux_data:
                        if sp_dim_trans['Type'].startswith('j'):
                            dim_1 = sp_dim_trans['Spectral_dim_ID_1']
                            dim_2 = sp_dim_trans['Spectral_dim_ID_2']
                            jcoupling[dim_1 - 1][dim_2 - 1] = True
                            jcoupling[dim_2 - 1][dim_1 - 1] = True

                relayed = [[False] * num_dim for i in range(num_dim)]
                if aux_data is not None:
                    for sp_dim_trans in aux_data:
                        if sp_dim_trans['Type'].startswith('relayed'):
                            dim_1 = sp_dim_trans['Spectral_dim_ID_1']
                            dim_2 = sp_dim_trans['Spectral_dim_ID_2']
                            relayed[dim_1 - 1][dim_2 - 1] = True
                            relayed[dim_2 - 1][dim_1 - 1] = True

                pk_id_name = 'Peak_ID'
                dim_id_name = 'Spectral_dim_ID'
                set_id_name = 'Set_ID'

                max_cs_err = self.chem_shift_error['max_inclusive']

                try:

                    lp_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == lp_category), None)  # noqa: E501,E741

                    if lp_data is not None:

                        for i in lp_data:

                            if __pynmrstar_v3__ and not (cs_chain_id_name in i and cs_seq_id_name in i and cs_comp_id_name in i and cs_atom_id_name in i):
                                continue

                            chain_id = i[cs_chain_id_name]
                            if chain_id in self.empty_value:
                                continue

                            seq_id = i[cs_seq_id_name]
                            if seq_id in self.empty_value:
                                continue

                            comp_id = i[cs_comp_id_name]
                            if comp_id in self.empty_value:
                                continue

                            atom_id = i[cs_atom_id_name]
                            if atom_id in self.empty_value:
                                continue

                            cs_list_id = i['Assigned_chem_shift_list_ID']

                            if cs_list_id != _cs_list_id:

                                for l in self.__lp_data['chem_shift']:  # noqa: E741

                                    if l['file_name'] == file_name:

                                        cs_data = l['data']
                                        cs_list = l['sf_framecode']

                                        cs_sf_data = self.__getSaveframeByName(fileListId, cs_list)

                                        if cs_sf_data is None:
                                            continue

                                        _cs_list_id = cs_sf_data.get_tag('ID')[0]

                                        if cs_list_id == _cs_list_id:
                                            break

                            pk_id = i[pk_id_name]
                            d = i[dim_id_name] - 1
                            set_id = i[set_id_name]

                            position = i[cs_value_name]

                            try:

                                if file_type == 'nmr-star' and self.__isNmrAtomName(comp_id, atom_id):
                                    _atom_id = self.__getAtomIdListWithAmbigCode(file_type, comp_id, atom_id)[0]

                                    if len(_atom_id) == 0:
                                        atom_id_ = atom_id

                                    elif len(_atom_id) == 1 and atom_id == _atom_id[0]:
                                        atom_id_ = atom_id

                                    else:
                                        # representative atom id
                                        atom_id_ = _atom_id[0]

                                else:
                                    atom_id_ = atom_id

                                cs = next(j for j in cs_data if j[cs_chain_id_name] == chain_id
                                          and j[cs_seq_id_name] == seq_id
                                          and j[cs_comp_id_name] == comp_id
                                          and j[cs_atom_id_name] == atom_id_)

                                value = cs[cs_value_name]
                                error = cs[cs_error_name]

                                if value in self.empty_value:
                                    continue

                                if error is None or error < 1.0e-3 or error * self.cs_diff_error_scaled_by_sigma > max_cs_err:
                                    error = max_cs_err
                                else:
                                    error *= self.cs_diff_error_scaled_by_sigma

                                if abs(position - value) > error:

                                    if d < num_dim and not abs_pk_pos[d] and sp_widths[d] is not None:
                                        if position < value:
                                            while position < value:
                                                position += sp_widths[d]
                                        elif position > value:
                                            while position > value:
                                                position -= sp_widths[d]

                                    if abs(position - value) > error:

                                        err = "[Check row of %s %s] Peak position of spectral peak %s %s (%s) in %r saveframe is inconsistent with the assigned chemical shift value %s (difference %s, tolerance %s) in %r saveframe." % (pk_id_name,  # noqa: E501
                                                                                                                                                                                                                                           i[pk_id_name], cs_value_name, position, self.__getReducedAtomNotation(cs_chain_id_name, chain_id, cs_seq_id_name, seq_id, cs_comp_id_name, comp_id, cs_atom_id_name, atom_id), sf_framecode, value, float('{:.3f}'.format(position - value)), error, cs_list)  # noqa: E501

                                        if error >= max_cs_err:

                                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ ValueError  - %s\n" % err)

                                        else:

                                            self.report.warning.appendDescription('unusual_chemical_shift', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                            self.report.setWarning()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ Warning  - %s\n" % err)

                                axis_code = str(cs[cs_iso_number]) + cs[cs_atom_type]

                                if aux_data is not None and d < num_dim and axis_code != axis_codes[d]:

                                    err = '[Check row of %s %s] Assignment of spectral peak %s is inconsistent with axis code %s vs %s.' %\
                                          (pk_id_name, i[pk_id_name], self.__getReducedAtomNotation(cs_chain_id_name, chain_id, cs_seq_id_name, seq_id, cs_comp_id_name, comp_id, cs_atom_id_name, atom_id), axis_code, axis_codes[d])  # noqa: E501

                                    self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                    self.report.setError()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ ValueError  - %s\n" % err)

                            except StopIteration:

                                err = "[Check row of %s %s] Assignment of spectral peak %s was not found in assigned chemical shifts in %r saveframe." %\
                                      (pk_id_name, i[pk_id_name], self.__getReducedAtomNotation(cs_chain_id_name, chain_id, cs_seq_id_name, seq_id, cs_comp_id_name, comp_id, cs_atom_id_name, atom_id), cs_list)  # noqa: E501

                                self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ ValueError  - %s\n" % err)

                            if d < num_dim and True in onebond[d]:
                                for d2 in range(num_dim):
                                    if onebond[d][d2]:

                                        try:
                                            j = next(j for j in lp_data if j[pk_id_name] == pk_id and j[dim_id_name] - 1 == d2 and j[set_id_name] is set_id)
                                        except StopIteration:
                                            continue

                                        chain_id2 = j[cs_chain_id_name]
                                        seq_id2 = j[cs_seq_id_name]
                                        comp_id2 = j[cs_comp_id_name]
                                        atom_id2 = j[cs_atom_id_name]

                                        if atom_id2 is not None:
                                            diff = len(atom_id) != len(atom_id2)
                                            _atom_id = '_' + (atom_id[1:-1] if atom_id.startswith('H') and diff else atom_id[1:])
                                            _atom_id2 = '_' + (atom_id2[1:-1] if atom_id2.startswith('H') and diff else atom_id2[1:])

                                        if chain_id2 in self.empty_value or seq_id2 in self.empty_value or comp_id2 in self.empty_value or atom_id2 in self.empty_value or\
                                           (d < d2 and (chain_id2 != chain_id or seq_id2 != seq_id or comp_id2 != comp_id or _atom_id2 != _atom_id)):

                                            err = '[Check row of %s %s] Coherence transfer type is onebond. However, assignment of spectral peak is inconsistent with the type, (%s) vs (%s).' % (pk_id_name, i[pk_id_name], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id),  # noqa: E501
                                                                                                                                                                                                  self.__getReducedAtomNotation(chain_id_names[d2], chain_id2, seq_id_names[d2], seq_id2, comp_id_names[d2], comp_id2, atom_id_names[d2], atom_id2))  # noqa: E501
                                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ ValueError  - %s\n" % err)

                            if d < num_dim and True in jcoupling[d]:
                                for d2 in range(num_dim):
                                    if jcoupling[d][d2]:

                                        try:
                                            j = next(j for j in lp_data if j[pk_id_name] == pk_id and j[dim_id_name] - 1 == d2 and j[set_id_name] is set_id)
                                        except StopIteration:
                                            continue

                                        chain_id2 = j[cs_chain_id_name]
                                        seq_id2 = j[cs_seq_id_name]
                                        comp_id2 = j[cs_comp_id_name]
                                        atom_id2 = j[cs_atom_id_name]

                                        if chain_id2 in self.empty_value or seq_id2 in self.empty_value or comp_id2 in self.empty_value or atom_id2 in self.empty_value or\
                                           (d < d2 and (chain_id2 != chain_id or seq_id2 != seq_id or comp_id2 != comp_id)):  # DAOTHER-7389, issue #2

                                            err = '[Check row of %s %s] Coherence transfer type is jcoupling. However, assignment of spectral peak is inconsistent with the type, (%s) vs (%s).' %\
                                                  (pk_id_name, i[pk_id_name], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id),  # noqa: E501
                                                   self.__getReducedAtomNotation(chain_id_names[d2], chain_id2, seq_id_names[d2], seq_id2, comp_id_names[d2], comp_id2, atom_id_names[d2], atom_id2))  # noqa: E501

                                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ ValueError  - %s\n" % err)

                            if d < num_dim and True in relayed[d]:
                                for d2 in range(num_dim):
                                    if relayed[d][d2]:

                                        try:
                                            j = next(j for j in lp_data if j[pk_id_name] == pk_id and j[dim_id_name] - 1 == d2 and j[set_id_name] is set_id)
                                        except StopIteration:
                                            continue

                                        chain_id2 = j[cs_chain_id_name]
                                        seq_id2 = j[cs_seq_id_name]
                                        comp_id2 = j[cs_comp_id_name]
                                        atom_id2 = j[cs_atom_id_name]

                                        if chain_id2 in self.empty_value or seq_id2 in self.empty_value or comp_id2 in self.empty_value or atom_id2 in self.empty_value or\
                                           (d < d2 and (chain_id2 != chain_id or abs(seq_id2 - seq_id) > 1)):  # DAOTHER-7389, issue #2

                                            err = '[Check row of %s %s] Coherence transfer type is relayed. However, assignment of spectral peak is inconsistent with the type, (%s) vs (%s).' % (pk_id_name,  # noqa: E501
                                                                                                                                                                                                  i[pk_id_name], self.__getReducedAtomNotation(chain_id_names[d], chain_id, seq_id_names[d], seq_id, comp_id_names[d], comp_id, atom_id_names[d], atom_id),  # noqa: E501
                                                                                                                                                                                                  self.__getReducedAtomNotation(chain_id_names[d2], chain_id2, seq_id_names[d2], seq_id2, comp_id_names[d2], comp_id2, atom_id_names[d2], atom_id2))  # noqa: E501

                                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                            self.report.setError()

                                            if self.__verbose:
                                                self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ ValueError  - %s\n" % err)

                except Exception as e:

                    self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ Error  - %s" % str(e))
                    self.report.setError()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testCSValueConsistencyInPkAltLoop() ++ Error  - %s" % str(e))

        return self.report.getTotalErrors() == __errors

    def __testRDCVector(self):
        """ Perform consistency test on RDC bond vectors.
        """

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'rdc_restraint'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.lp_categories[file_type][content_subtype]

            if self.__star_data_type[fileListId] == 'Loop':

                sf_data = self.__star_data[fileListId]
                sf_framecode = ''

                self.__testRDCVector__(file_name, file_type, content_subtype, sf_framecode, lp_category)

            elif self.__star_data_type[fileListId] == 'Saveframe':

                sf_data = self.__star_data[fileListId]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__testRDCVector__(file_name, file_type, content_subtype, sf_framecode, lp_category)

            else:

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__testRDCVector__(file_name, file_type, content_subtype, sf_framecode, lp_category)

        return self.report.getTotalErrors() == __errors

    def __testRDCVector__(self, file_name, file_type, content_subtype, sf_framecode, lp_category):
        """ Perform consistency test on RDC bond vectors.
        """

        item_names = self.item_names_in_rdc_loop[file_type]
        index_tag = self.index_tags[file_type][content_subtype]
        chain_id_1_name = item_names['chain_id_1']
        chain_id_2_name = item_names['chain_id_2']
        seq_id_1_name = item_names['seq_id_1']
        seq_id_2_name = item_names['seq_id_2']
        comp_id_1_name = item_names['comp_id_1']
        comp_id_2_name = item_names['comp_id_2']
        atom_id_1_name = item_names['atom_id_1']
        atom_id_2_name = item_names['atom_id_2']

        try:

            lp_data = next((l['data'] for l in self.__lp_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode), None)  # noqa: E741

            if lp_data is not None:

                for i in lp_data:
                    chain_id_1 = i[chain_id_1_name]
                    seq_id_1 = i[seq_id_1_name]
                    comp_id_1 = i[comp_id_1_name]
                    atom_id_1 = i[atom_id_1_name]
                    chain_id_2 = i[chain_id_2_name]
                    seq_id_2 = i[seq_id_2_name]
                    comp_id_2 = i[comp_id_2_name]
                    atom_id_2 = i[atom_id_2_name]

                    if (atom_id_1[0] not in self.atom_isotopes) or (atom_id_2[0] not in self.atom_isotopes):

                        idx_msg = "[Check row of %s %s] " % (index_tag, i[index_tag])

                        err = "%sNon-magnetic susceptible spin appears in RDC vector (%s:%s:%s:%s, %s:%s:%s:%s)." %\
                              (idx_msg, chain_id_1, seq_id_1, comp_id_1, atom_id_1, chain_id_2, seq_id_2, comp_id_2, atom_id_2)

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testRDCVector() ++ Error  - %s\n" % err)

                    if chain_id_1 != chain_id_2:

                        idx_msg = "[Check row of %s %s] " % (index_tag, i[index_tag])

                        err = "%sInvalid inter-chain RDC vector (%s:%s:%s:%s, %s:%s:%s:%s) in a loop %s." %\
                              (idx_msg, chain_id_1, seq_id_1, comp_id_1, atom_id_1, chain_id_2, seq_id_2, comp_id_2, atom_id_2, lp_category)

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testRDCVector() ++ Error  - %s\n" % err)

                    elif abs(seq_id_1 - seq_id_2) > 1:

                        idx_msg = "[Check row of %s %s] " % (index_tag, i[index_tag])

                        err = "%sInvalid inter-residue RDC vector (%s:%s:%s:%s, %s:%s:%s:%s) in a loop %s." %\
                              (idx_msg, chain_id_1, seq_id_1, comp_id_1, atom_id_1, chain_id_2, seq_id_2, comp_id_2, atom_id_2, lp_category)

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testRDCVector() ++ Error  - %s\n" % err)

                    elif abs(seq_id_1 - seq_id_2) == 1:

                        if self.__csStat.getTypeOfCompId(comp_id_1)[0] and self.__csStat.getTypeOfCompId(comp_id_2)[0] and\
                           ((seq_id_1 < seq_id_2 and atom_id_1 == 'C' and atom_id_2 in ('N', 'H')) or (seq_id_1 > seq_id_2 and atom_id_1 in ('N', 'H') and atom_id_2 == 'C')):
                            pass

                        else:

                            idx_msg = "[Check row of %s %s] " % (index_tag, i[index_tag])

                            err = "%sInvalid inter-residue RDC vector (%s:%s:%s:%s, %s:%s:%s:%s) in a loop %s." %\
                                  (idx_msg, chain_id_1, seq_id_1, comp_id_1, atom_id_1, chain_id_2, seq_id_2, comp_id_2, atom_id_2, lp_category)

                            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                            self.report.setError()

                            if self.__verbose:
                                self.__lfh.write("+NmrDpUtility.__testRDCVector() ++ Error  - %s\n" % err)

                    elif atom_id_1 == atom_id_2:

                        idx_msg = "[Check row of %s %s] " % (index_tag, i[index_tag])

                        err = "%sZero RDC vector (%s:%s:%s:%s, %s:%s:%s:%s)." %\
                              (idx_msg, chain_id_1, seq_id_1, comp_id_1, atom_id_1, chain_id_2, seq_id_2, comp_id_2, atom_id_2)

                        self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__testRDCVector() ++ Error  - %s\n" % err)

                    else:

                        if self.__updateChemCompDict(comp_id_1):  # matches with comp_id in CCD

                            if not any(b for b in self.__last_chem_comp_bonds if
                                       ((b[self.__ccb_atom_id_1] == atom_id_1 and b[self.__ccb_atom_id_2] == atom_id_2)
                                        or (b[self.__ccb_atom_id_1] == atom_id_2 and b[self.__ccb_atom_id_2] == atom_id_1))):

                                if self.__nefT.validate_comp_atom(comp_id_1, atom_id_1) and self.__nefT.validate_comp_atom(comp_id_2, atom_id_2):

                                    idx_msg = "[Check row of %s %s] " % (index_tag, i[index_tag])

                                    warn = "%sRDC vector over multiple covalent bonds (%s:%s:%s:%s, %s:%s:%s:%s)." %\
                                           (idx_msg, chain_id_1, seq_id_1, comp_id_1, atom_id_1, chain_id_2, seq_id_2, comp_id_2, atom_id_2)

                                    self.report.warning.appendDescription('unusual/rare_data', {'file_name': file_name, 'sf_framecode': sf_framecode,
                                                                                                'category': lp_category, 'description': warn})
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__testRDCVector() ++ Warning  - %s\n" % warn)

                                else:  # raised error already somewhere because of invalid atom nomenclature
                                    pass

                        else:  # raised warning already somewhere because of unknown comp_id
                            pass

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testRDCVector() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testRDCVector() ++ Error  - %s" % str(e))

    def __testCovalentBond(self):
        """ Perform consistency test on covalent bonds.
        """

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'poly_seq'

            if content_subtype not in input_source_dic['content_subtype'].keys():
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.aux_lp_categories[file_type][content_subtype][0]

            if self.__star_data_type[fileListId] == 'Loop':

                sf_data = self.__star_data[fileListId]
                sf_framecode = ''

                self.__testCovalentBond__(file_name, file_type, content_subtype, sf_framecode, lp_category)

            elif self.__star_data_type[fileListId] == 'Saveframe':

                sf_data = self.__star_data[fileListId]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__testCovalentBond__(file_name, file_type, content_subtype, sf_framecode, lp_category)

            else:

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__testCovalentBond__(file_name, file_type, content_subtype, sf_framecode, lp_category)

        return self.report.getTotalErrors() == __errors

    def __testCovalentBond__(self, file_name, file_type, content_subtype, sf_framecode, lp_category):
        """ Perform consistency test on covalent bonds.
        """

        item_names = self.item_names_in_rdc_loop[file_type]
        chain_id_1_name = item_names['chain_id_1']
        chain_id_2_name = item_names['chain_id_2']
        seq_id_1_name = item_names['seq_id_1']
        seq_id_2_name = item_names['seq_id_2']
        comp_id_1_name = item_names['comp_id_1']
        comp_id_2_name = item_names['comp_id_2']
        atom_id_1_name = item_names['atom_id_1']
        atom_id_2_name = item_names['atom_id_2']

        try:

            aux_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == lp_category), None)  # noqa: E501,E741

            if aux_data is not None:

                for l, i in enumerate(aux_data):  # noqa: E741
                    chain_id_1 = i[chain_id_1_name]
                    seq_id_1 = i[seq_id_1_name]
                    comp_id_1 = i[comp_id_1_name]
                    atom_id_1 = i[atom_id_1_name]
                    chain_id_2 = i[chain_id_2_name]
                    seq_id_2 = i[seq_id_2_name]
                    comp_id_2 = i[comp_id_2_name]
                    atom_id_2 = i[atom_id_2_name]

                    bond = self.__getBondLength(chain_id_1, seq_id_1, atom_id_1, chain_id_2, seq_id_2, atom_id_2)

                    if bond is None:
                        continue

                    broken_bond = [b for b in bond if b['distance'] > self.cutoff_bond_length]

                    if len(broken_bond) == 0:
                        continue

                    length_list = ''
                    for bb in broken_bond:
                        length_list += '%s (model_id %s), ' % (bb['distance'], bb['model_id'])

                    warn = "Covalent bond (%s - %s) is out of acceptable range, %s angstromes." %\
                        (self.__getReducedAtomNotation(chain_id_1_name, chain_id_1, seq_id_1_name, seq_id_1, comp_id_1_name, comp_id_1, atom_id_1_name, atom_id_1),
                         self.__getReducedAtomNotation(chain_id_2_name, chain_id_2, seq_id_2_name, seq_id_2, comp_id_2_name, comp_id_2, atom_id_2_name, atom_id_2),
                         length_list[:-2])

                    self.report.warning.appendDescription('anomalous_bond_length', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                    self.report.setWarning()

                    if self.__verbose:
                        self.__lfh.write("+NmrDpUtility.__testCovalentBond() ++ Warning  - %s\n" % warn)

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testCovalentBond() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testCovalentBond() ++ Error  - %s" % str(e))

    def __getBondLength(self, nmr_chain_id_1, nmr_seq_id_1, nmr_atom_id_1, nmr_chain_id_2, nmr_seq_id_2, nmr_atom_id_2):
        """ Return the bond length of given two atoms.
            @return: the bond length
        """

        intra_chain = nmr_chain_id_1 == nmr_chain_id_2

        s_1 = self.report.getModelPolymerSequenceWithNmrChainId(nmr_chain_id_1)

        if s_1 is None:
            return None

        s_2 = s_1 if intra_chain else self.report.getModelPolymerSequenceWithNmrChainId(nmr_chain_id_2)

        if s_2 is None:
            return None

        cif_chain_id_1 = s_1['chain_id']
        cif_chain_id_2 = cif_chain_id_1 if intra_chain else s_2['chain_id']

        seq_align_dic = self.report.sequence_alignment.get()

        if not has_key_value(seq_align_dic, 'nmr_poly_seq_vs_model_poly_seq'):
            return None

        seq_key = (nmr_chain_id_1, nmr_seq_id_1, nmr_atom_id_1, nmr_chain_id_2, nmr_seq_id_2, nmr_atom_id_2)

        if seq_key in self.__coord_bond_length:
            return self.__coord_bond_length[seq_key]

        result_1 = next((seq_align for seq_align in seq_align_dic['nmr_poly_seq_vs_model_poly_seq'] if seq_align['ref_chain_id'] == nmr_chain_id_1 and seq_align['test_chain_id'] == cif_chain_id_1), None)  # noqa: E501
        result_2 = result_1 if intra_chain else next((seq_align for seq_align in seq_align_dic['nmr_poly_seq_vs_model_poly_seq'] if seq_align['ref_chain_id'] == nmr_chain_id_2 and seq_align['test_chain_id'] == cif_chain_id_2), None)  # noqa: E501

        if result_1 is not None and result_2 is not None:

            cif_seq_id_1 = next((test_seq_id for ref_seq_id, test_seq_id in zip(result_1['ref_seq_id'], result_1['test_seq_id']) if ref_seq_id == nmr_seq_id_1), None)

            if cif_seq_id_1 is None:
                self.__coord_bond_length[seq_key] = None
                return None

            cif_seq_id_2 = next((test_seq_id for ref_seq_id, test_seq_id in zip(result_2['ref_seq_id'], result_2['test_seq_id']) if ref_seq_id == nmr_seq_id_2), None)

            if cif_seq_id_2 is None:
                self.__coord_bond_length[seq_key] = None
                return None

            try:

                model_num_name = 'pdbx_PDB_model_num' if self.__cR.hasItem('atom_site', 'pdbx_PDB_model_num') else 'ndb_model'

                atom_site_1 = self.__cR.getDictListWithFilter('atom_site',
                                                              [{'name': 'Cartn_x', 'type': 'float', 'alt_name': 'x'},
                                                               {'name': 'Cartn_y', 'type': 'float', 'alt_name': 'y'},
                                                               {'name': 'Cartn_z', 'type': 'float', 'alt_name': 'z'},
                                                               {'name': model_num_name, 'type': 'int', 'alt_name': 'model_id'}
                                                               ],
                                                              [{'name': 'label_asym_id', 'type': 'str', 'value': cif_chain_id_1},
                                                               {'name': 'label_seq_id', 'type': 'int', 'value': cif_seq_id_1},
                                                               {'name': 'label_atom_id', 'type': 'str', 'value': nmr_atom_id_1}])

                atom_site_2 = self.__cR.getDictListWithFilter('atom_site',
                                                              [{'name': 'Cartn_x', 'type': 'float', 'alt_name': 'x'},
                                                               {'name': 'Cartn_y', 'type': 'float', 'alt_name': 'y'},
                                                               {'name': 'Cartn_z', 'type': 'float', 'alt_name': 'z'},
                                                               {'name': model_num_name, 'type': 'int', 'alt_name': 'model_id'}
                                                               ],
                                                              [{'name': 'label_asym_id', 'type': 'str', 'value': cif_chain_id_2},
                                                               {'name': 'label_seq_id', 'type': 'int', 'value': cif_seq_id_2},
                                                               {'name': 'label_atom_id', 'type': 'str', 'value': nmr_atom_id_2}])

            except Exception as e:

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__getBondLength() ++ Error  - %s" % str(e))
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__getBondLength() ++ Error  - %s" % str(e))

                return None

            model_ids = set(a['model_id'] for a in atom_site_1) | set(a['model_id'] for a in atom_site_2)

            bond = []

            for model_id in model_ids:
                a_1 = next((a for a in atom_site_1 if a['model_id'] == model_id), None)
                a_2 = next((a for a in atom_site_2 if a['model_id'] == model_id), None)

                if a_1 is None or a_2 is None:
                    continue

                bond.append({'model_id': model_id, 'distance': float('{:.3f}'.format(np.linalg.norm(to_np_array(a_1) - to_np_array(a_2))))})

            if len(bond) > 0:
                self.__coord_bond_length[seq_key] = bond
                return bond

        self.__coord_bond_length[seq_key] = None
        return None

    def __testResidueVariant(self):
        """ Perform consistency test on residue variants.
        """

        id = self.report.getInputSourceIdOfCoord()

        if id < 0:
            return True

        cif_input_source = self.report.input_sources[id]
        cif_input_source_dic = cif_input_source.get()

        cif_polymer_sequence = cif_input_source_dic['polymer_sequence']

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            nmr_input_source = self.report.input_sources[fileListId]
            nmr_input_source_dic = nmr_input_source.get()

            file_name = nmr_input_source_dic['file_name']
            file_type = nmr_input_source_dic['file_type']

            if nmr_input_source_dic['content_subtype'] is None:
                continue

            content_subtype = 'poly_seq'

            if content_subtype not in nmr_input_source_dic['content_subtype'].keys():
                continue

            sf_category = self.sf_categories[file_type][content_subtype]
            lp_category = self.aux_lp_categories[file_type][content_subtype][1]  # nef: _nef_sequence, nmr-star: _Entity_deleted_atom

            if lp_category not in self.__lp_category_list[fileListId]:
                continue

            seq_align_dic = self.report.sequence_alignment.get()
            chain_assign_dic = self.report.chain_assignment.get()

            if 'nmr_poly_seq_vs_model_poly_seq' not in chain_assign_dic:

                err = "Chain assignment does not exist, __assignCoordPolymerSequence() should be invoked."

                self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testCoordResidueVariant() ++ Error  - %s" % err)
                self.report.setError()

                if self.__verbose:
                    self.__lfh.write("+NmrDpUtility.__testCoordResidueVariant() ++ Error  - %s\n" % err)

                continue

            if not has_key_value(seq_align_dic, 'nmr_poly_seq_vs_model_poly_seq'):
                continue

            if not has_key_value(chain_assign_dic, 'nmr_poly_seq_vs_model_poly_seq'):
                continue

            nmr2ca = {}

            for chain_assign in chain_assign_dic['nmr_poly_seq_vs_model_poly_seq']:

                ref_chain_id = chain_assign['ref_chain_id']
                test_chain_id = chain_assign['test_chain_id']

                result = next((seq_align for seq_align in seq_align_dic['nmr_poly_seq_vs_model_poly_seq'] if seq_align['ref_chain_id'] == ref_chain_id and seq_align['test_chain_id'] == test_chain_id), None)  # noqa: E501

                if ref_chain_id not in nmr2ca:
                    nmr2ca[ref_chain_id] = []

                complex = {'seq_align': result}  # DAOTHER-7465

                if 'unmapped_sequence' in chain_assign:
                    complex['seq_unmap'] = [unmapped['ref_seq_id'] for unmapped in chain_assign['unmapped_sequence']]

                nmr2ca[ref_chain_id].append(complex)

            if self.__star_data_type[fileListId] == 'Loop':

                sf_data = self.__star_data[fileListId]
                sf_framecode = ''

                self.__testResidueVariant__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, cif_polymer_sequence, nmr2ca)

            elif self.__star_data_type[fileListId] == 'Saveframe':

                sf_data = self.__star_data[fileListId]
                sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                self.__testResidueVariant__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, cif_polymer_sequence, nmr2ca)

            else:

                for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                        continue

                    self.__testResidueVariant__(file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, cif_polymer_sequence, nmr2ca)

        return self.report.getTotalErrors() == __errors

    def __testResidueVariant__(self, file_name, file_type, content_subtype, sf_data, sf_framecode, lp_category, cif_polymer_sequence, nmr2ca):
        """ Perform consistency test on residue variants.
        """

        item_names = self.item_names_in_cs_loop[file_type]
        chain_id_name = item_names['chain_id']
        seq_id_name = item_names['seq_id']
        comp_id_name = item_names['comp_id']
        atom_id_name = item_names['atom_id']
        variant_name = 'residue_variant' if file_type == 'nef' else item_names['atom_id']

        key_items = self.aux_key_items[file_type][content_subtype][lp_category]
        data_items = self.aux_data_items[file_type][content_subtype][lp_category]
        allowed_tags = self.aux_allowed_tags[file_type][content_subtype][lp_category]

        try:

            aux_data = self.__nefT.check_data(sf_data, lp_category, key_items, data_items, allowed_tags, None, None,
                                              enforce_allowed_tags=(file_type == 'nmr-star'),
                                              excl_missing_data=self.__excl_missing_data)[0]

            if aux_data is not None:

                for i in aux_data:
                    chain_id = i[chain_id_name]
                    seq_id = i[seq_id_name]
                    comp_id = i[comp_id_name]
                    variant = i[variant_name]

                    if chain_id not in nmr2ca:
                        continue

                    ca = next((ca['seq_align'] for ca in nmr2ca[chain_id] if ('seq_unmap' not in ca or (seq_id not in ca['seq_unmap']))), None)  # DAOTHER-7465

                    if ca is None:
                        continue

                    cif_chain_id = ca['test_chain_id']

                    cif_seq_id = next((test_seq_id for ref_seq_id, test_seq_id in zip(ca['ref_seq_id'], ca['test_seq_id']) if ref_seq_id == seq_id), None)

                    if cif_seq_id is None:
                        continue

                    cif_ps = next(ps for ps in cif_polymer_sequence if ps['chain_id'] == cif_chain_id)

                    cif_comp_id = next((_comp_id for _seq_id, _comp_id in zip(cif_ps['seq_id'], cif_ps['comp_id']) if _seq_id == cif_seq_id), None)

                    if cif_comp_id is None:
                        continue

                    seq_key = (cif_chain_id, cif_seq_id)

                    coord_atom_id_ = None if seq_key not in self.__coord_atom_id else self.__coord_atom_id[seq_key]

                    self.__updateChemCompDict(comp_id)

                    if file_type == 'nef':

                        if variant in self.empty_value:
                            continue

                        for _variant in variant.split(','):
                            _variant_ = _variant.strip(' ')

                            if _variant_[0] not in ('-', '+'):

                                warn = "Residue variant %r should start with either '-' or '+' symbol according to the NEF sepcification." % _variant_

                                self.report.warning.appendDescription('atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})  # noqa: E501
                                self.report.setWarning()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__textResidueVariant() ++ Warning  - %s\n" % warn)

                                continue

                            atom_id = _variant_[1:]

                            if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                                _atom_id, _, details = self.__getAtomIdListWithAmbigCode(file_type, comp_id, atom_id)

                                if len(_atom_id) == 0:
                                    continue

                                if len(_atom_id) == 1 and atom_id == _atom_id[0]:
                                    atom_id_ = atom_id
                                    atom_name = atom_id

                                    if details is not None:
                                        atom_name += ' (' + details.rstrip('.') + ')'

                                else:
                                    atom_name = atom_id + ' (e.g. '

                                    for atom_id_ in _atom_id:
                                        atom_name += atom_id_ + ' '

                                    atom_name = atom_name.rstrip() + ')'

                                    # representative atom id
                                    atom_id_ = _atom_id[0]

                            else:
                                atom_id_ = atom_id
                                atom_name = atom_id

                            if _variant_[0] == '-':

                                if self.__last_comp_id_test:  # matches with comp_id in CCD

                                    if not any(a for a in self.__last_chem_comp_atoms if a[self.__cca_atom_id] == atom_id_):

                                        warn = "Atom (%s, %s %r) did not match with chemical component dictionary (CCD)." %\
                                            (self.__getReducedAtomNotation(chain_id_name, chain_id, seq_id_name, seq_id, comp_id_name, comp_id, atom_id_name, atom_name), variant_name, _variant_)  # noqa: E501

                                        self.report.warning.appendDescription('atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})  # noqa: E501
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__textResidueVariant() ++ Warning  - %s\n" % warn)

                                if coord_atom_id_ is not None and coord_atom_id_['comp_id'] == cif_comp_id and atom_id_ in coord_atom_id_['atom_id']:

                                    err = "Atom (%s, %s %r) is unexpectedly incorporated in the coordinate." %\
                                        (self.__getReducedAtomNotation(chain_id_name, chain_id, seq_id_name, seq_id, comp_id_name, comp_id, atom_id_name, atom_name), variant_name, _variant_)  # noqa: E501

                                    self.report.error.appendDescription('invalid_atom_nomenclature', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                    self.report.setError()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__testResidueVariant() ++ Error  - %s\n" % err)

                            else:

                                if coord_atom_id_ is not None and coord_atom_id_['comp_id'] == cif_comp_id and atom_id_ not in coord_atom_id_['atom_id']:

                                    err = "Atom (%s) which is a %s %r is not present in the coordinate." %\
                                        (self.__getReducedAtomNotation(chain_id_name, chain_id, seq_id_name, seq_id, comp_id_name, comp_id, atom_id_name, atom_name), variant_name, _variant_)  # noqa: E501

                                    self.report.error.appendDescription('atom_not_found', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                    self.report.setError()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__testResidueVariant() ++ Error  - %s\n" % err)

                    else:

                        atom_id = variant

                        if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                            _atom_id, _, details = self.__getAtomIdListWithAmbigCode(file_type, comp_id, atom_id)

                            if len(_atom_id) == 0:
                                continue

                            if len(_atom_id) == 1 and atom_id == _atom_id[0]:
                                atom_id_ = atom_id
                                atom_name = atom_id

                                if details is not None:
                                    atom_name += ' (' + details.rstrip('.') + ')'

                            else:
                                atom_name = atom_id + ' (e.g. '

                                for atom_id_ in _atom_id:
                                    atom_name += atom_id_ + ' '

                                atom_name = atom_name.rstrip() + ')'

                                # representative atom id
                                atom_id_ = _atom_id[0]

                        else:
                            atom_id_ = atom_id
                            atom_name = atom_id

                            if self.__last_comp_id_test:  # matches with comp_id in CCD

                                if not any(a for a in self.__last_chem_comp_atoms if a[self.__cca_atom_id] == atom_id_):

                                    warn = "Atom (%s) did not match with chemical component dictionary (CCD)." %\
                                        self.__getReducedAtomNotation(chain_id_name, chain_id, seq_id_name, seq_id, comp_id_name, comp_id, atom_id_name, atom_name)

                                    self.report.warning.appendDescription('atom_nomenclature_mismatch', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})  # noqa: E501
                                    self.report.setWarning()

                                    if self.__verbose:
                                        self.__lfh.write("+NmrDpUtility.__textResidueVariant() ++ Warning  - %s\n" % warn)

                            if coord_atom_id_ is not None and coord_atom_id_['comp_id'] == cif_comp_id and atom_id_ in coord_atom_id_['atom_id']:

                                err = "Atom (%s) is unexpectedly incorporated in the coordinate." %\
                                    self.__getReducedAtomNotation(chain_id_name, chain_id, seq_id_name, seq_id, comp_id_name, comp_id, atom_id_name, atom_name)

                                self.report.error.appendDescription('invalid_atom_nomenclature', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': err})  # noqa: E501
                                self.report.setError()

                                if self.__verbose:
                                    self.__lfh.write("+NmrDpUtility.__testResidueVariant() ++ Error  - %s\n" % err)

        except LookupError as e:

            item = 'format_issue' if 'Unauthorized item' in str(e) else 'missing_mandatory_item'

            self.report.error.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testResidueVariant() ++ LookupError  - %s" % str(e))

        except ValueError as e:

            self.report.error.appendDescription('invalid_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': str(e).strip("'")})
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testResidueVariant() ++ ValueError  - %s" % str(e))

        except Exception as e:

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__testResidueVariant() ++ Error  - %s" % str(e))
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__testResidueVariant() ++ Error  - %s" % str(e))

    def __getReducedAtomNotation(self, chain_id_name, chain_id, seq_id_name, seq_id, comp_id_name, comp_id, atom_id_name, atom_id):
        """ Return reduced form of atom notation.
        """

        if self.__reduced_atom_notation:
            return '%s:%s:%s:%s' % (chain_id, seq_id, comp_id, atom_id)

        return '%s %s, %s %s, %s %s, %s %s' % (chain_id_name, chain_id, seq_id_name, seq_id, comp_id_name, comp_id, atom_id_name, atom_id)

    def __getResucedAtomNotations(self, key_items, row_data):
        """ Return reduced from of series of atom notations.
        """

        msg = ''

        if self.__reduced_atom_notation:
            j = 0
            for k in key_items:
                msg += '%s:' % row_data[k['name']]
                j += 1
                if j % 4 == 0:
                    msg = msg[:-1] + ' - '
            return msg[:-3]

        for k in key_items:
            msg += k['name'] + ' %s, ' % row_data[k['name']]

        return msg[:-2]

    def __calculateStatsOfExptlData(self):
        """ Calculate statistics of experimental data.
        """

        __errors = self.report.getTotalErrors()

        for fileListId in range(self.__file_path_list_len):

            input_source = self.report.input_sources[fileListId]
            input_source_dic = input_source.get()

            file_name = input_source_dic['file_name']
            file_type = input_source_dic['file_type']

            if input_source_dic['content_subtype'] is None:
                continue

            seq_align_dic = self.report.sequence_alignment.get()

            stats = {}

            for content_subtype in input_source_dic['content_subtype'].keys():

                if content_subtype in ('entry_info', 'entity'):
                    continue

                if self.report_prev is not None:
                    prev_stats = self.report_prev.getNmrLegacyStatsOfExptlData(fileListId, content_subtype)
                    if prev_stats is not None:
                        stats[content_subtype] = prev_stats
                    continue

                sf_category = self.sf_categories[file_type][content_subtype]
                lp_category = self.lp_categories[file_type][content_subtype]

                if content_subtype == 'poly_seq':
                    lp_category = self.aux_lp_categories[file_type][content_subtype][0]

                asm = []

                list_id = 1

                if self.__star_data_type[fileListId] == 'Loop':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = ''

                    self.__calculateStatsOfExptlData__(fileListId, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, seq_align_dic, asm)

                elif self.__star_data_type[fileListId] == 'Saveframe':

                    sf_data = self.__star_data[fileListId]
                    sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                    self.__calculateStatsOfExptlData__(fileListId, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, seq_align_dic, asm)

                else:

                    for sf_data in self.__star_data[fileListId].get_saveframes_by_category(sf_category):

                        sf_framecode = get_first_sf_tag(sf_data, 'sf_framecode')

                        if not any(loop for loop in sf_data.loops if loop.category == lp_category):
                            continue

                        self.__calculateStatsOfExptlData__(fileListId, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, seq_align_dic, asm)

                        list_id += 1

                if len(asm) > 0:
                    stats[content_subtype] = asm

            input_source.setItemValue('stats_of_exptl_data', stats)

        return self.report.getTotalErrors() == __errors

    def __calculateStatsOfExptlData__(self, file_list_id, file_name, file_type, content_subtype, sf_data, list_id, sf_framecode, lp_category, seq_align_dic, asm):
        """ Calculate statistics of experimental data.
        """

        index_tag = self.index_tags[file_type][content_subtype]

        _list_id = list_id
        if file_type == 'nmr-star' and self.__combined_mode:
            val = sf_data.get_tag('ID')
            if len(val) > 0:
                try:
                    _list_id = int(val[0])
                except ValueError:
                    return

        if content_subtype != 'poly_seq':
            lp_data = next((l['data'] for l in self.__lp_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode), None)  # noqa: E741
        else:
            lp_data = next((l['data'] for l in self.__aux_data[content_subtype] if l['file_name'] == file_name and l['sf_framecode'] == sf_framecode and l['category'] == lp_category), None)  # noqa: E741,E501

        if lp_data is None or len(lp_data) == 0:
            return

        sf_tag_data = next((t['data'] for t in self.__sf_tag_data[content_subtype] if t['file_name'] == file_name and t['sf_framecode'] == sf_framecode), None)

        ent = {'list_id': _list_id, 'sf_framecode': sf_framecode, 'number_of_rows': len(lp_data)}

        if content_subtype in ('dist_restraint', 'dihed_restraint', 'rdc_restraint'):

            if len(sf_framecode) == 0:
                ent['exp_type'] = 'Unknown'
            else:
                type = sf_data.get_tag('restraint_origin' if file_type == 'nef' else 'Constraint_type')
                if len(type) > 0 and type[0] not in self.empty_value:
                    ent['exp_type'] = type[0]
                else:
                    ent['exp_type'] = 'Unknown'

        elif content_subtype.startswith('spectral_peak'):

            if len(sf_framecode) == 0:
                ent['exp_type'] = 'Unknown'
            else:
                type = sf_data.get_tag('experiment_type' if file_type == 'nef' else 'Experiment_type')
                if len(type) > 0 and type[0] not in self.empty_value:
                    ent['exp_type'] = type[0]
                else:
                    ent['exp_type'] = 'Unknown'

        if content_subtype in ('chem_shift', 'dist_restraint', 'dihed_restraint', 'rdc_restraint', 'spectral_peak', 'spectral_peak_alt'):

            sa_name = 'nmr_poly_seq_vs_' + content_subtype

            if sa_name in seq_align_dic and seq_align_dic[sa_name] is not None:

                low_seq_coverage = ''

                seq_coverage = []

                for seq_align in seq_align_dic[sa_name]:

                    if seq_align['list_id'] == list_id:

                        sc = {}
                        sc['chain_id'] = seq_align['chain_id']
                        sc['length'] = seq_align['length']
                        sc['sequence_coverage'] = seq_align['sequence_coverage']

                        if seq_align['sequence_coverage'] < self.low_seq_coverage and seq_align['length'] > 1:
                            if ('exp_type' not in ent) or (ent['exp_type'] not in ('disulfide bound', 'disulfide_bond', 'paramagnetic relaxation', 'pre', 'symmetry', 'J-couplings', 'jcoupling')):  # noqa: E501
                                low_seq_coverage += 'coverage %s for chain_id %s, length %s, ' % (seq_align['sequence_coverage'], seq_align['chain_id'], seq_align['length'])

                        seq_coverage.append(sc)

                if len(seq_coverage) > 0:

                    ent['sequence_coverage'] = seq_coverage

                    if len(low_seq_coverage) > 0:

                        warn = 'Sequence coverage of NMR experimental data is relatively low (' + low_seq_coverage[:-2] + ") in %r saveframe." % sf_framecode

                        self.report.warning.appendDescription('insufficient_data', {'file_name': file_name, 'sf_framecode': sf_framecode, 'category': lp_category, 'description': warn})
                        self.report.setWarning()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__calculateStatsOfExptlData() ++ Warning  - %s\n" % warn)

                if content_subtype == 'chem_shift':

                    try:

                        item_names = self.item_names_in_cs_loop[file_type]

                        anomalous_errs = self.report.error.getValueListWithSf('anomalous_data', file_name, sf_framecode, key='Z_score')
                        anomalous_warns = self.report.warning.getValueListWithSf('anomalous_data', file_name, sf_framecode, key='Z_score')
                        unusual_warns = self.report.warning.getValueListWithSf('unusual_data', file_name, sf_framecode, key='Z_score')

                        cs_ann = []

                        if anomalous_errs is not None:

                            for a_err in anomalous_errs:
                                ann = {}
                                ann['level'] = 'anomalous'
                                ann['chain_id'] = a_err['row_location'][item_names['chain_id']]
                                ann['seq_id'] = int(a_err['row_location'][item_names['seq_id']])
                                ann['comp_id'] = a_err['row_location'][item_names['comp_id']]
                                ann['atom_id'] = a_err['row_location'][item_names['atom_id']]
                                ann['value'] = a_err['value']
                                ann['z_score'] = a_err['z_score']

                                comp_id = ann['comp_id']
                                atom_id = ann['atom_id'].split(' ')[0]

                                polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                                if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):
                                    non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                    if atom_id in non_rep_methyl_pros:
                                        continue

                                cs_ann.append(ann)

                        if anomalous_warns is not None:

                            for a_warn in anomalous_warns:
                                ann = {}
                                ann['level'] = 'anomalous'
                                ann['chain_id'] = a_warn['row_location'][item_names['chain_id']]
                                ann['seq_id'] = int(a_warn['row_location'][item_names['seq_id']])
                                ann['comp_id'] = a_warn['row_location'][item_names['comp_id']]
                                ann['atom_id'] = a_warn['row_location'][item_names['atom_id']]
                                ann['value'] = a_warn['value']
                                ann['z_score'] = a_warn['z_score']

                                comp_id = ann['comp_id']
                                atom_id = ann['atom_id'].split(' ')[0]

                                polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                                if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):
                                    non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                    if atom_id in non_rep_methyl_pros:
                                        continue

                                cs_ann.append(ann)

                        if unusual_warns is not None:

                            for u_warn in unusual_warns:
                                ann = {}
                                ann['level'] = 'unusual'
                                ann['chain_id'] = u_warn['row_location'][item_names['chain_id']]
                                ann['seq_id'] = int(u_warn['row_location'][item_names['seq_id']])
                                ann['comp_id'] = u_warn['row_location'][item_names['comp_id']]
                                ann['atom_id'] = u_warn['row_location'][item_names['atom_id']]
                                ann['value'] = u_warn['value']
                                ann['z_score'] = u_warn['z_score']

                                comp_id = ann['comp_id']
                                atom_id = ann['atom_id'].split(' ')[0]

                                polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                                if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):
                                    non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                    if atom_id in non_rep_methyl_pros:
                                        continue

                                cs_ann.append(ann)

                    except Exception as e:

                        self.report.error.appendDescription('internal_error', "+NmrDpUtility.__calculateStatsOfExptlData() ++ Error  - %s" % str(e))
                        self.report.setError()

                        if self.__verbose:
                            self.__lfh.write("+NmrDpUtility.__calculateStatsOfExptlData() ++ Error  - %s" % str(e))

                    self.__calculateStatsOfAssignedChemShift(file_list_id, sf_framecode, lp_data, cs_ann, ent)

                elif content_subtype in ('dist_restraint', 'dihed_restraint', 'rdc_restraint'):

                    conflict_id_set = self.__nefT.get_conflict_id_set(sf_data, lp_category, self.consist_key_items[file_type][content_subtype])[0]

                    conflict_warns = self.report.warning.getValueListWithSf('conflicted_data', file_name, sf_framecode)
                    inconsist_warns = self.report.warning.getValueListWithSf('inconsistent_data', file_name, sf_framecode)
                    redundant_warns = self.report.warning.getValueListWithSf('redundant_data', file_name, sf_framecode)

                    inconsistent = set()
                    redundant = set()

                    if conflict_warns is not None:

                        for c_warn in conflict_warns:
                            for index in c_warn['row_locations'][index_tag]:
                                inconsistent.add(int(index))

                    if inconsist_warns is not None:

                        for i_warn in inconsist_warns:
                            for index in i_warn['row_locations'][index_tag]:
                                inconsistent.add(int(index))

                    if redundant_warns is not None:

                        for d_warn in redundant_warns:
                            for index in d_warn['row_locations'][index_tag]:
                                redundant.add(int(index))

                    if content_subtype == 'dist_restraint':
                        self.__calculateStatsOfDistanceRestraint(file_list_id, sf_framecode, lp_data, conflict_id_set, inconsistent, redundant, ent)

                    elif content_subtype == 'dihed_restraint':
                        self.__calculateStatsOfDihedralRestraint(file_list_id, lp_data, conflict_id_set, inconsistent, redundant, ent)

                    elif content_subtype == 'rdc_restraint':
                        self.__calculateStatsOfRdcRestraint(file_list_id, lp_data, conflict_id_set, inconsistent, redundant, ent)

            if content_subtype.startswith('spectral_peak'):

                try:

                    _num_dim = sf_data.get_tag(self.num_dim_items[file_type])[0]
                    num_dim = int(_num_dim)

                    if num_dim not in range(1, self.lim_num_dim):
                        raise ValueError()

                except ValueError:  # raised error already at __testIndexConsistency()
                    return

                if content_subtype == 'spectral_peak':
                    self.__calculateStatsOfSpectralPeak(file_list_id, sf_framecode, num_dim, lp_data, ent)
                elif content_subtype == 'spectral_peak_alt':
                    self.__calculateStatsOfSpectralPeakAlt(file_list_id, sf_framecode, num_dim, lp_data, ent)

        elif content_subtype == 'poly_seq':
            self.__calculateStatsOfCovalentBond(file_list_id, sf_framecode, lp_category, lp_data, ent)

        elif content_subtype == 'chem_shift_ref':
            ent['loop'] = lp_data
            ent['saveframe_tag'] = sf_tag_data

        else:

            err = "Not found a module for calculation of statistics on content subtype %s." % content_subtype

            self.report.error.appendDescription('internal_error', "+NmrDpUtility.__calculateStatsOfExptlData() ++ Error  - %s" % err)
            self.report.setError()

            if self.__verbose:
                self.__lfh.write("+NmrDpUtility.__calculateStatsOfExptlData() ++ Error  - %s\n" % err)

            return

        has_err = self.report.error.exists(file_name, sf_framecode)
        has_warn = self.report.warning.exists(file_name, sf_framecode)

        if has_err:
            status = 'Error'
            ent['error_descriptions'] = self.report.error.getCombinedDescriptions(file_name, sf_framecode)
            if has_warn:
                ent['warning_descriptions'] = self.report.warning.getCombinedDescriptions(file_name, sf_framecode)
        elif has_warn:
            status = 'Warning'
            ent['warning_descriptions'] = self.report.warning.getCombinedDescriptions(file_name, sf_framecode)
        else:
            status = 'OK'

        ent['status'] = status

        asm.append(ent)

    def __calculateStatsOfAssignedChemShift(self, file_list_id, sf_framecode, lp_data, cs_ann, ent):
        """ Calculate statistics of assigned chemical shifts.
        """

        input_source = self.report.input_sources[file_list_id]
        input_source_dic = input_source.get()

        file_name = input_source_dic['file_name']
        file_type = input_source_dic['file_type']

        item_names = self.item_names_in_cs_loop[file_type]
        chain_id_name = item_names['chain_id']
        seq_id_name = item_names['seq_id']
        comp_id_name = item_names['comp_id']
        atom_id_name = item_names['atom_id']
        value_name = item_names['value']
        atom_type = item_names['atom_type']
        iso_number = item_names['isotope_number']

        try:

            count = {}

            for i in lp_data:

                if i[atom_type] in self.empty_value or i[iso_number] in self.empty_value or value_name in self.empty_value:
                    continue

                data_type = str(i[iso_number]) + i[atom_type].lower() + '_chemical_shifts'

                if data_type in count:
                    count[data_type] += 1
                else:
                    count[data_type] = 1

            if len(count) > 0:
                ent['number_of_assignments'] = count

            polymer_sequence = input_source_dic['polymer_sequence']

            if polymer_sequence is None:
                return

            if 'sequence_coverage' in ent:

                completeness = []

                for sc in ent['sequence_coverage']:

                    cc = {}

                    chain_id = sc['chain_id']

                    _chain_id = chain_id if file_type == 'nef' else str(self.__nefT.letter_to_int(chain_id))

                    cc['chain_id'] = chain_id

                    # all atoms

                    all_c = []

                    excluded_comp_id = []
                    excluded_atom_id = []

                    h1_col = -1
                    c13_col = -1
                    n15_col = -1
                    p31_col = -1

                    l = 0  # noqa: E741

                    for data_type in count:

                        atom_group = {}
                        atom_group['atom_group'] = 'all_' + data_type
                        atom_group['number_of_assigned_shifts'] = 0
                        atom_group['number_of_target_shifts'] = 0
                        atom_group['completeness'] = 0.0

                        if data_type.startswith('1h'):
                            h1_col = l

                        elif data_type.startswith('13c'):
                            c13_col = l

                        elif data_type.startswith('15n'):
                            n15_col = l

                        elif data_type.startswith('31p'):
                            p31_col = l

                        all_c.append(atom_group)

                        l += 1  # noqa: E741

                    s = next((s for s in polymer_sequence if s['chain_id'] == chain_id), None)

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                            if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):

                                all_atoms = self.__csStat.getAllAtoms(comp_id, excl_minor_atom=True, primary=polypeptide_like)
                                non_excl_atoms = self.__csStat.getAllAtoms(comp_id, excl_minor_atom=False)
                                non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                for a in all_atoms:

                                    if h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                        all_c[h1_col]['number_of_target_shifts'] += 1

                                    elif c13_col != -1 and a.startswith('C'):
                                        all_c[c13_col]['number_of_target_shifts'] += 1

                                    elif n15_col != -1 and a.startswith('N'):
                                        all_c[n15_col]['number_of_target_shifts'] += 1

                                    elif p31_col != -1 and a.startswith('P'):
                                        all_c[p31_col]['number_of_target_shifts'] += 1

                                atom_set = set()

                                for j in lp_data:

                                    if j[chain_id_name] != _chain_id or j[seq_id_name] != seq_id or j[comp_id_name] != comp_id or j[value_name] in self.empty_value:
                                        continue

                                    atom_id = j[atom_id_name]
                                    data_type = str(j[iso_number]) + j[atom_type]

                                    if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                                        atom_ids = self.__getAtomIdList(file_type, comp_id, atom_id)

                                        if len(atom_ids) == 0:
                                            continue

                                        for a in atom_ids:

                                            if a in atom_set:
                                                continue

                                            atom_set.add(a)

                                            if a in all_atoms:

                                                if data_type == '1H' and h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                                    all_c[h1_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '13C' and c13_col != -1:
                                                    all_c[c13_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '15N' and n15_col != -1:
                                                    all_c[n15_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '31P' and p31_col != -1:
                                                    all_c[p31_col]['number_of_assigned_shifts'] += 1

                                            elif a in non_excl_atoms:
                                                excluded_atom_id.append({'seq_id': seq_id, 'comp_id': comp_id, 'atom_id': a, 'value': j[value_name]})

                                    else:

                                        if atom_id in atom_set:
                                            continue

                                        atom_set.add(atom_id)

                                        if atom_id in all_atoms:

                                            if data_type == '1H' and h1_col != -1 and atom_id not in non_rep_methyl_pros and atom_id.startswith('H'):
                                                all_c[h1_col]['number_of_assigned_shifts'] += 1

                                            elif data_type == '13C' and c13_col != -1:
                                                all_c[c13_col]['number_of_assigned_shifts'] += 1

                                            elif data_type == '15N' and n15_col != -1:
                                                all_c[n15_col]['number_of_assigned_shifts'] += 1

                                            elif data_type == '31P' and p31_col != -1:
                                                all_c[p31_col]['number_of_assigned_shifts'] += 1

                                        elif atom_id in non_excl_atoms:
                                            excluded_atom_id.append({'seq_id': seq_id, 'comp_id': comp_id, 'atom_id': atom_id, 'value': j[value_name]})

                            else:
                                excluded_comp_id.append({'seq_id': seq_id, 'comp_id': comp_id})

                        for c in all_c:
                            if c['number_of_target_shifts'] > 0:
                                c['completeness'] = float('{:.3f}'.format(float(c['number_of_assigned_shifts']) / float(c['number_of_target_shifts'])))
                            else:
                                c['completeness'] = None

                    cc['completeness_of_all_assignments'] = all_c

                    cc['excluded_comp_id_in_statistics'] = excluded_comp_id if len(excluded_comp_id) > 0 else None
                    cc['excluded_atom_id_in_statistics'] = excluded_atom_id if len(excluded_atom_id) > 0 else None

                    # backbone atoms (bb)

                    bb_c = []

                    h1_col = -1
                    c13_col = -1
                    n15_col = -1
                    p31_col = -1

                    l = 0  # noqa: E741

                    for data_type in count:

                        atom_group = {}
                        atom_group['atom_group'] = 'backbone_' + data_type
                        atom_group['number_of_assigned_shifts'] = 0
                        atom_group['number_of_target_shifts'] = 0
                        atom_group['completeness'] = 0.0

                        if data_type.startswith('1h'):
                            h1_col = l

                        elif data_type.startswith('13c'):
                            c13_col = l

                        elif data_type.startswith('15n'):
                            n15_col = l

                        elif data_type.startswith('31p'):
                            p31_col = l

                        bb_c.append(atom_group)

                        l += 1  # noqa: E741

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                            if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):

                                bb_atoms = self.__csStat.getBackBoneAtoms(comp_id, excl_minor_atom=True)
                                non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                for a in bb_atoms:

                                    if h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                        bb_c[h1_col]['number_of_target_shifts'] += 1

                                    elif c13_col != -1 and a.startswith('C'):
                                        bb_c[c13_col]['number_of_target_shifts'] += 1

                                    elif n15_col != -1 and a.startswith('N'):
                                        bb_c[n15_col]['number_of_target_shifts'] += 1

                                    elif p31_col != -1 and a.startswith('P'):
                                        bb_c[p31_col]['number_of_target_shifts'] += 1

                                atom_set = set()

                                for j in lp_data:

                                    if j[chain_id_name] != _chain_id or j[seq_id_name] != seq_id or j[comp_id_name] != comp_id or j[value_name] in self.empty_value:
                                        continue

                                    atom_id = j[atom_id_name]
                                    data_type = str(j[iso_number]) + j[atom_type]

                                    if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                                        atom_ids = self.__getAtomIdList(file_type, comp_id, atom_id)

                                        if len(atom_ids) == 0:
                                            continue

                                        for a in atom_ids:

                                            if a in bb_atoms:

                                                if a in atom_set:
                                                    continue

                                                atom_set.add(a)

                                                if data_type == '1H' and h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                                    bb_c[h1_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '13C' and c13_col != -1:
                                                    bb_c[c13_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '15N' and n15_col != -1:
                                                    bb_c[n15_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '31P' and p31_col != -1:
                                                    bb_c[p31_col]['number_of_assigned_shifts'] += 1

                                    elif atom_id in bb_atoms:

                                        if atom_id in atom_set:
                                            continue

                                        atom_set.add(atom_id)

                                        if data_type == '1H' and h1_col != -1 and atom_id not in non_rep_methyl_pros and atom_id.startswith('H'):
                                            bb_c[h1_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '13C' and c13_col != -1:
                                            bb_c[c13_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '15N' and n15_col != -1:
                                            bb_c[n15_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '31P' and p31_col != -1:
                                            bb_c[p31_col]['number_of_assigned_shifts'] += 1

                        for c in bb_c:
                            if c['number_of_target_shifts'] > 0:
                                c['completeness'] = float('{:.3f}'.format(float(c['number_of_assigned_shifts']) / float(c['number_of_target_shifts'])))
                            else:
                                c['completeness'] = None

                    if len(bb_c) > 0:
                        cc['completeness_of_backbone_assignments'] = bb_c

                    # sidechain atoms (sc)

                    sc_c = []

                    h1_col = -1
                    c13_col = -1
                    n15_col = -1
                    p31_col = -1

                    l = 0  # noqa: E741

                    for data_type in count:

                        atom_group = {}
                        atom_group['atom_group'] = 'sidechain_' + data_type
                        atom_group['number_of_assigned_shifts'] = 0
                        atom_group['number_of_target_shifts'] = 0
                        atom_group['completeness'] = 0.0

                        if data_type.startswith('1h'):
                            h1_col = l

                        elif data_type.startswith('13c'):
                            c13_col = l

                        elif data_type.startswith('15n'):
                            n15_col = l

                        elif data_type.startswith('31p'):
                            p31_col = l

                        sc_c.append(atom_group)

                        l += 1  # noqa: E741

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                            if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):

                                sc_atoms = self.__csStat.getSideChainAtoms(comp_id, excl_minor_atom=True)
                                non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                for a in sc_atoms:

                                    if h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                        sc_c[h1_col]['number_of_target_shifts'] += 1

                                    elif c13_col != -1 and a.startswith('C'):
                                        sc_c[c13_col]['number_of_target_shifts'] += 1

                                    elif n15_col != -1 and a.startswith('N'):
                                        sc_c[n15_col]['number_of_target_shifts'] += 1

                                    elif p31_col != -1 and a.startswith('P'):
                                        sc_c[p31_col]['number_of_target_shifts'] += 1

                                atom_set = set()

                                for j in lp_data:

                                    if j[chain_id_name] != _chain_id or j[seq_id_name] != seq_id or j[comp_id_name] != comp_id or j[value_name] in self.empty_value:
                                        continue

                                    atom_id = j[atom_id_name]
                                    data_type = str(j[iso_number]) + j[atom_type]

                                    if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                                        atom_ids = self.__getAtomIdList(file_type, comp_id, atom_id)

                                        if len(atom_ids) == 0:
                                            continue

                                        for a in atom_ids:

                                            if a in sc_atoms:

                                                if a in atom_set:
                                                    continue

                                                atom_set.add(a)

                                                if data_type == '1H' and h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                                    sc_c[h1_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '13C' and c13_col != -1:
                                                    sc_c[c13_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '15N' and n15_col != -1:
                                                    sc_c[n15_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '31P' and p31_col != -1:
                                                    sc_c[p31_col]['number_of_assigned_shifts'] += 1

                                    elif atom_id in sc_atoms:

                                        if atom_id in atom_set:
                                            continue

                                        atom_set.add(atom_id)

                                        if data_type == '1H' and h1_col != -1 and atom_id not in non_rep_methyl_pros and atom_id.startswith('H'):
                                            sc_c[h1_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '13C' and c13_col != -1:
                                            sc_c[c13_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '15N' and n15_col != -1:
                                            sc_c[n15_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '31P' and p31_col != -1:
                                            sc_c[p31_col]['number_of_assigned_shifts'] += 1

                        for c in sc_c:
                            if c['number_of_target_shifts'] > 0:
                                c['completeness'] = float('{:.3f}'.format(float(c['number_of_assigned_shifts']) / float(c['number_of_target_shifts'])))
                            else:
                                c['completeness'] = None

                    if len(sc_c) > 0:
                        cc['completeness_of_sidechain_assignments'] = sc_c

                    # methyl group atoms (ch3)

                    ch3_c = []

                    h1_col = -1
                    c13_col = -1

                    l = 0  # noqa: E741

                    for data_type in count:

                        atom_group = {}
                        atom_group['atom_group'] = 'methyl_' + data_type
                        atom_group['number_of_assigned_shifts'] = 0
                        atom_group['number_of_target_shifts'] = 0
                        atom_group['completeness'] = 0.0

                        if data_type.startswith('1h'):
                            h1_col = l

                        elif data_type.startswith('13c'):
                            c13_col = l

                        else:
                            continue

                        ch3_c.append(atom_group)

                        l += 1  # noqa: E741

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                            if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):

                                ch3_atoms = self.__csStat.getMethylAtoms(comp_id, excl_minor_atom=True, primary=polypeptide_like)
                                non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                for a in ch3_atoms:

                                    if h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                        ch3_c[h1_col]['number_of_target_shifts'] += 1

                                    elif c13_col != -1 and a.startswith('C'):
                                        ch3_c[c13_col]['number_of_target_shifts'] += 1

                                atom_set = set()

                                for j in lp_data:

                                    if j[chain_id_name] != _chain_id or j[seq_id_name] != seq_id or j[comp_id_name] != comp_id or j[value_name] in self.empty_value:
                                        continue

                                    atom_id = j[atom_id_name]
                                    data_type = str(j[iso_number]) + j[atom_type]

                                    if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                                        atom_ids = self.__getAtomIdList(file_type, comp_id, atom_id)

                                        if len(atom_ids) == 0:
                                            continue

                                        for a in atom_ids:

                                            if a in ch3_atoms:

                                                if a in atom_set:
                                                    continue

                                                atom_set.add(a)

                                                if data_type == '1H' and h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                                    ch3_c[h1_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '13C' and c13_col != -1:
                                                    ch3_c[c13_col]['number_of_assigned_shifts'] += 1

                                    elif atom_id in ch3_atoms:

                                        if atom_id in atom_set:
                                            continue

                                        atom_set.add(atom_id)

                                        if data_type == '1H' and h1_col != -1 and atom_id not in non_rep_methyl_pros and atom_id.startswith('H'):
                                            ch3_c[h1_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '13C' and c13_col != -1:
                                            ch3_c[c13_col]['number_of_assigned_shifts'] += 1

                        for c in ch3_c:
                            if c['number_of_target_shifts'] > 0:
                                c['completeness'] = float('{:.3f}'.format(float(c['number_of_assigned_shifts']) / float(c['number_of_target_shifts'])))
                            else:
                                c['completeness'] = None

                    if len(ch3_c) > 0:
                        cc['completeness_of_methyl_assignments'] = ch3_c

                    # aromatic atoms (aro)

                    aro_c = []

                    h1_col = -1
                    c13_col = -1
                    n15_col = -1

                    l = 0  # noqa: E741

                    for data_type in count:

                        atom_group = {}
                        atom_group['atom_group'] = 'aromatic_' + data_type
                        atom_group['number_of_assigned_shifts'] = 0
                        atom_group['number_of_target_shifts'] = 0
                        atom_group['completeness'] = 0.0

                        if data_type.startswith('1h'):
                            h1_col = l

                        elif data_type.startswith('13c'):
                            c13_col = l

                        elif data_type.startswith('15n'):
                            n15_col = l

                        aro_c.append(atom_group)

                        l += 1  # noqa: E741

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            polypeptide_like = self.__csStat.getTypeOfCompId(comp_id)[0]

                            if self.__csStat.hasEnoughStat(comp_id, polypeptide_like):

                                aro_atoms = self.__csStat.getAromaticAtoms(comp_id, excl_minor_atom=True, primary=polypeptide_like)
                                non_rep_methyl_pros = self.__csStat.getNonRepresentativeMethylProtons(comp_id, excl_minor_atom=True, primary=polypeptide_like)

                                for a in aro_atoms:

                                    if h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                        aro_c[h1_col]['number_of_target_shifts'] += 1

                                    elif c13_col != -1 and a.startswith('C'):
                                        aro_c[c13_col]['number_of_target_shifts'] += 1

                                    elif n15_col != -1 and a.startswith('N'):
                                        aro_c[n15_col]['number_of_target_shifts'] += 1

                                atom_set = set()

                                for j in lp_data:

                                    if j[chain_id_name] != _chain_id or j[seq_id_name] != seq_id or j[comp_id_name] != comp_id or j[value_name] in self.empty_value:
                                        continue

                                    atom_id = j[atom_id_name]
                                    data_type = str(j[iso_number]) + j[atom_type]

                                    if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                                        atom_ids = self.__getAtomIdList(file_type, comp_id, atom_id)

                                        if len(atom_ids) == 0:
                                            continue

                                        for a in atom_ids:

                                            if a in aro_atoms:

                                                if a in atom_set:
                                                    continue

                                                atom_set.add(a)

                                                if data_type == '1H' and h1_col != -1 and a not in non_rep_methyl_pros and a.startswith('H'):
                                                    aro_c[h1_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '13C' and c13_col != -1:
                                                    aro_c[c13_col]['number_of_assigned_shifts'] += 1

                                                elif data_type == '15N' and n15_col != -1:
                                                    aro_c[n15_col]['number_of_assigned_shifts'] += 1

                                    elif atom_id in aro_atoms:

                                        if atom_id in atom_set:
                                            continue

                                        atom_set.add(atom_id)

                                        if data_type == '1H' and h1_col != -1 and atom_id not in non_rep_methyl_pros and atom_id.startswith('H'):
                                            aro_c[h1_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '13C' and c13_col != -1:
                                            aro_c[c13_col]['number_of_assigned_shifts'] += 1

                                        elif data_type == '15N' and n15_col != -1:
                                            aro_c[n15_col]['number_of_assigned_shifts'] += 1

                        for c in aro_c:
                            if c['number_of_target_shifts'] > 0:
                                c['completeness'] = float('{:.3f}'.format(float(c['number_of_assigned_shifts']) / float(c['number_of_target_shifts'])))
                            else:
                                c['completeness'] = None

                    if len(aro_c) > 0:
                        cc['completeness_of_aromatic_assignments'] = aro_c

                    completeness.append(cc)

                if len(completeness) > 0:
                    ent['completeness'] = completeness

            z_scores = {}

            for k in count.keys():
                z_scores[k] = []

            max_val = 0.0
            min_val = 0.0

            for i in lp_data:

                if i[atom_type] in self.empty_value or i[iso_number] in self.empty_value or value_name in self.empty_value:
                    continue

                data_type = str(i[iso_number]) + i[atom_type].lower() + '_chemical_shifts'

                chain_id = i[chain_id_name]
                seq_id = i[seq_id_name]
                comp_id = i[comp_id_name]
                atom_id = i[atom_id_name]
                value = i[value_name]

                _chain_id = chain_id if file_type == 'nef' else str(self.__nefT.letter_to_int(chain_id))

                if value in self.empty_value:
                    continue

                if file_type == 'nef' or self.__isNmrAtomName(comp_id, atom_id):
                    _atom_id = self.__getAtomIdListWithAmbigCode(file_type, comp_id, atom_id)[0]

                    if len(_atom_id) == 0:
                        continue

                    if len(_atom_id) == 1 and atom_id == _atom_id[0]:
                        atom_id_ = atom_id

                    else:
                        # representative atom id
                        atom_id_ = _atom_id[0]

                else:
                    atom_id_ = atom_id

                has_cs_stat = False

                # non-standard residue
                if self.__get1LetterCode(comp_id) == 'X':

                    neighbor_comp_ids = set(j[comp_id_name] for j in lp_data if j[chain_id_name] == _chain_id and abs(j[seq_id_name] - seq_id) < 4 and j[seq_id_name] != seq_id)

                    polypeptide_like = False

                    for comp_id2 in neighbor_comp_ids:
                        polypeptide_like |= self.__csStat.getTypeOfCompId(comp_id2)[0]

                    for cs_stat in self.__csStat.get(comp_id):

                        if cs_stat['atom_id'] == atom_id_ and cs_stat['count'] > 0:
                            avg_value = cs_stat['avg']
                            std_value = cs_stat['std']

                            has_cs_stat = True

                            break

                # standard residue
                else:

                    for cs_stat in self.__csStat.get(comp_id, self.report.isDiamagnetic()):

                        if cs_stat['atom_id'] == atom_id_ and cs_stat['count'] > 0:
                            avg_value = cs_stat['avg']
                            std_value = cs_stat['std']

                            has_cs_stat = True

                            break

                if (not has_cs_stat) or std_value is None or std_value <= 0.0 or avg_value is None:
                    continue

                z_score = (value - avg_value) / std_value

                if z_score > max_val:
                    max_val = z_score

                elif z_score < min_val:
                    min_val = z_score

                z_scores[data_type].append(z_score)

            target_scale = (max_val - min_val) / 20.0

            if target_scale <= 0.0:
                return

            scale = 1.0

            while scale < target_scale:
                scale *= 2.0

            while scale > target_scale:
                scale /= 2.0

            range_of_vals = []
            count_of_vals = []

            v = 0.0
            while v < min_val:
                v += scale

            while v > min_val:
                v -= scale

            while v <= max_val:

                _count = copy.copy(count)

                for k in count.keys():
                    _count[k] = len([z for z in z_scores[k] if v <= z < v + scale])

                range_of_vals.append(v)
                count_of_vals.append(_count)

                v += scale

            transposed = {}

            for k in count.keys():
                transposed[k] = []

                for count_of_val in count_of_vals:
                    transposed[k].append(count_of_val[k])

            if len(range_of_vals) > 1:
                """
                has_value = False
                for j in range(1, len(range_of_vals) - 1):
                    for k in count.keys():
                        if transposed[k][j] > 0:
                            has_value = True
                            break
                    if has_value:
                        break

                if has_value:
                """
                ent['histogram'] = {'range_of_values': range_of_vals, 'number_of_values': transposed, 'annotations': cs_ann}

            if 'sequence_coverage' in ent:

                # prediction of redox state of CYS

                cys_redox_state = []

                for sc in ent['sequence_coverage']:

                    chain_id = sc['chain_id']

                    _chain_id = chain_id if file_type == 'nef' else str(self.__nefT.letter_to_int(chain_id))

                    s = next((s for s in polymer_sequence if s['chain_id'] == chain_id), None)

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            if comp_id != 'CYS':
                                continue

                            cys = {'chain_id': chain_id, 'seq_id': seq_id}

                            ca_chem_shift = None
                            cb_chem_shift = None

                            for j in lp_data:

                                atom_id = j[atom_id_name]

                                if j[chain_id_name] == _chain_id and j[seq_id_name] == seq_id and j[comp_id_name] == comp_id:
                                    if atom_id == 'CA':
                                        ca_chem_shift = j[value_name]
                                    elif atom_id == 'CB':
                                        cb_chem_shift = j[value_name]

                                if ca_chem_shift is None or cb_chem_shift is None:
                                    if j[chain_id_name] == _chain_id and j[seq_id_name] > seq_id:
                                        break
                                else:
                                    break

                            cys['ca_chem_shift'] = ca_chem_shift
                            cys['cb_chem_shift'] = cb_chem_shift

                            if cb_chem_shift is not None:
                                if cb_chem_shift < 32.0:
                                    cys['redox_state_pred'] = 'reduced'
                                elif cb_chem_shift > 35.0:
                                    cys['redox_state_pred'] = 'oxidized'
                                else:
                                    cys['redox_state_pred'] = 'ambiguous'
                            elif ca_chem_shift is not None:
                                cys['redox_state_pred'] = 'ambiguous'
                            else:
                                cys['redox_state_pred'] = 'unknown'

                            if cys['redox_state_pred'] == 'ambiguous':
                                oxi, red = predict_redox_state_of_cystein(ca_chem_shift, cb_chem_shift)
                                if oxi < 0.001:
                                    cys['redox_state_pred'] = 'reduced'
                                elif red < 0.001:
                                    cys['redox_state_pred'] = 'oxidized'
                                else:
                                    cys['redox_state_pred'] = 'oxidized %s (%%), reduced %s (%%)' % ('{:.1f}'.format(oxi * 100.0), '{:.1f}'.format(red * 100.0))

                            cys['in_disulfide_bond'] = False
                            if input_source_dic['disulfide_bond'] is not None:
                                if any(b for b in input_source_dic['disulfide_bond'] if (b['chain_id_1'] == chain_id and b['seq_id_1'] == seq_id) or (b['chain_id_2'] == chain_id and b['seq_id_2'] == seq_id)):  # noqa: E501
                                    cys['in_disulfide_bond'] = True

                            cys['in_other_bond'] = False
                            if input_source_dic['other_bond'] is not None:
                                if any(b for b in input_source_dic['other_bond'] if (b['chain_id_1'] == chain_id and b['seq_id_1'] == seq_id) or (b['chain_id_2'] == chain_id and b['seq_id_2'] == seq_id)):  # noqa: E501
                                    cys['in_other_bond'] = True

                            cys_redox_state.append(cys)

                    if len(cys_redox_state) > 0:
                        ent['cys_redox_state'] = cys_redox_state

                # prediction of cis-trans peptide of PRO

                pro_cis_trans = []

                for sc in ent['sequence_coverage']:

                    chain_id = sc['chain_id']

                    _chain_id = chain_id if file_type == 'nef' else str(self.__nefT.letter_to_int(chain_id))

                    s = next((s for s in polymer_sequence if s['chain_id'] == chain_id), None)

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            if comp_id != 'PRO':
                                continue

                            pro = {'chain_id': chain_id, 'seq_id': seq_id}

                            cb_chem_shift = None
                            cg_chem_shift = None

                            for j in lp_data:

                                atom_id = j[atom_id_name]

                                if j[chain_id_name] == _chain_id and j[seq_id_name] == seq_id and j[comp_id_name] == comp_id:
                                    if atom_id == 'CB':
                                        cb_chem_shift = j[value_name]
                                    elif atom_id == 'CG':
                                        cg_chem_shift = j[value_name]

                                if cb_chem_shift is None or cg_chem_shift is None:
                                    if j[chain_id_name] == _chain_id and j[seq_id_name] > seq_id:
                                        break
                                else:
                                    break

                            pro['cb_chem_shift'] = cb_chem_shift
                            pro['cg_chem_shift'] = cg_chem_shift

                            if (cb_chem_shift is not None) and (cg_chem_shift is not None):
                                delta = cb_chem_shift - cg_chem_shift
                                if delta < 4.8:
                                    pro['cis_trans_pred'] = 'trans'
                                elif delta > 9.15:
                                    pro['cis_trans_pred'] = 'cis'
                                else:
                                    pro['cis_trans_pred'] = 'ambiguous'
                            elif (cb_chem_shift is not None) or (cg_chem_shift is not None):
                                pro['cis_trans_pred'] = 'ambiguous'
                            else:
                                pro['cis_trans_pred'] = 'unknown'

                            if pro['cis_trans_pred'] == 'ambiguous':
                                cis, trs = predict_cis_trans_peptide_of_proline(cb_chem_shift, cg_chem_shift)
                                if cis < 0.001:
                                    pro['cis_trans_pred'] = 'trans'
                                elif trs < 0.001:
                                    pro['cis_trans_pred'] = 'cis'
                                else:
                                    pro['cis_trans_pred'] = 'cis %s (%%), trans %s (%%)' % ('{:.1f}'.format(cis * 100.0), '{:.1f}'.format(trs * 100.0))

                            pro['in_cis_peptide_bond'] = self.__isProtCis(chain_id, seq_id)

                            if pro['cis_trans_pred'] != 'unknown':

                                if (pro['in_cis_peptide_bond'] and pro['cis_trans_pred'] != 'cis') or (not pro['in_cis_peptide_bond'] and pro['cis_trans_pred'] != 'trans'):
                                    item = None
                                    if ',' in pro['cis_trans_pred']:
                                        if (pro['in_cis_peptide_bond'] and cis > trs) or\
                                           (not pro['in_cis_peptide_bond'] and trs > cis):
                                            pass
                                        else:
                                            item = 'unusual_chemical_shift'
                                    else:
                                        item = 'anomalous_chemical_shift'

                                    if item is not None:

                                        shifts = ''
                                        if cb_chem_shift is not None:
                                            shifts += 'CB %s ppm, ' % cb_chem_shift
                                        if cg_chem_shift is not None:
                                            shifts += 'CG %s ppm, ' % cg_chem_shift

                                        warn = "%s-peptide bond of %s:%s:%s can not be verified with the assigned chemical shift values (%scis_trans_pred %s)." %\
                                               ('cis' if pro['in_cis_peptide_bond'] else 'trans', chain_id, seq_id, comp_id, shifts, pro['cis_trans_pred'])

                                        self.report.warning.appendDescription(item, {'file_name': file_name, 'sf_framecode': sf_framecode, 'description': warn})
                                        self.report.setWarning()

                                        if self.__verbose:
                                            self.__lfh.write("+NmrDpUtility.__calculateStatsOfAssignedChemShift() ++ Warning  - %s\n" % warn)

                            pro_cis_trans.append(pro)

                    if len(pro_cis_trans) > 0:
                        ent['pro_cis_trans'] = pro_cis_trans

                # prediction of tautomeric state of HIS

                his_tautomeric_state = []

                for sc in ent['sequence_coverage']:

                    chain_id = sc['chain_id']

                    _chain_id = chain_id if file_type == 'nef' else str(self.__nefT.letter_to_int(chain_id))

                    s = next((s for s in polymer_sequence if s['chain_id'] == chain_id), None)

                    if s is not None:

                        for seq_id, comp_id in zip(s['seq_id'], s['comp_id']):

                            if comp_id != 'HIS':
                                continue

                            his = {'chain_id': chain_id, 'seq_id': seq_id}

                            cg_chem_shift = None
                            cd2_chem_shift = None
                            nd1_chem_shift = None
                            ne2_chem_shift = None

                            for j in lp_data:

                                atom_id = j[atom_id_name]

                                if j[chain_id_name] == _chain_id and j[seq_id_name] == seq_id and j[comp_id_name] == comp_id:
                                    if atom_id == 'CG':
                                        cg_chem_shift = j[value_name]
                                    elif atom_id == 'CD2':
                                        cd2_chem_shift = j[value_name]
                                    elif atom_id == 'ND1':
                                        nd1_chem_shift = j[value_name]
                                    elif atom_id == 'NE2':
                                        ne2_chem_shift = j[value_name]

                                if cg_chem_shift is None or cd2_chem_shift is None or nd1_chem_shift is None or ne2_chem_shift is None:
                                    if j[chain_id_name] == _chain_id and j[seq_id_name] > seq_id:
                                        break
                                else:
                                    break

                            his['cg_chem_shift'] = cg_chem_shift
                            his['cd2_chem_shift'] = cd2_chem_shift
                            his['nd1_chem_shift'] = nd1_chem_shift
                            his['ne2_chem_shift'] = ne2_chem_shift

                            if (cg_chem_shift is not None) or (cd2_chem_shift is not None) or (nd1_chem_shift is not None) or (ne2_chem_shift is not None):
                                bip, tau, pi = predict_tautomer_state_of_histidine(cg_chem_shift, cd2_chem_shift, nd1_chem_shift, ne2_chem_shift)
                                if tau < 0.001 and pi < 0.001:
                                    his['tautomeric_state_pred'] = 'biprotonated'
                                elif bip < 0.001 and pi < 0.001:
                                    his['tautomeric_state_pred'] = 'tau-tautomer'
                                elif bip < 0.001 and tau < 0.001:
                                    his['tautomeric_state_pred'] = 'pi-tautomer'
                                else:
                                    his['tautomeric_state_pred'] = 'biprotonated %s (%%), tau-tautomer %s (%%), pi-tautomer %s (%%)' % ('{:.1f}'.format(bip * 100.0), '{:.1f}'.format(tau * 100.0), '{:.1f}'.format(pi * 100.0))  # noqa: E501
                            else:
                                his['tautomeric_state_pred'] = 'unknown'

                            his['tautomeric_state'] = self.__getTautomerOfHistidine(chain_id, seq_id)

                            if his['tautomeric_state_pred'] != 'unknown':
                                item = None
                                if his['tautomeric_state_pred'] != his['tautomeric_state'] and his['tautomeric_state'] != 'unknown':
                                    if ',' in his['tautomeric_state_pred']:
                                        if (his['tautomeric_state'] == 'biprotonated' and bip > tau and bip > pi) or\
                                           (his['tautomeric_state'] == 'tau-tautomer' and tau > bip and tau > pi) or\
                                           (his['tautomeric_state'] == 'pi-tautomer' and pi > bip and pi > tau):
                                            pass
                                        else:
                                            item = 'unusual_chemical_shift'
               