import hmac
from datetime import datetime
from typing import Optional

from django.conf import settings
from django.core.files.storage import FileSystemStorage

_DEFAULT_PARAM_PREFIX = "titofisto_"


class TitofistoStorage(FileSystemStorage):
    """Time-token secured variant of the base filesystem storage."""

    def url(self, name: str) -> str:
        """Compute URL for requested storage file."""
        # Get regular URL from base FileSystemStorage
        raw_url = super().url(name)

        # Get token and timestamp
        token, ts = self.get_token(name)

        # Generate full, token-secured URL
        param_prefix = getattr(settings, "TITOFISTO_PARAM_PREFIX", _DEFAULT_PARAM_PREFIX)
        full_url = f"{raw_url}?{param_prefix}token={token}&{param_prefix}ts={ts}"
        return full_url

    def get_token(self, name: str, ts: Optional[str] = None) -> tuple[str, str]:
        """Get a token and associated generation timestamp for a filename."""
        # Determine parts of the HMAC from the file
        mtime = self.get_modified_time(name).isoformat()
        if ts is None:
            ts = datetime.now().strftime("%s")
        full_msg = f"{name}//{mtime}@{ts}"

        # Calculate a HMAC with the parts
        token = hmac.new(
            bytes(settings.SECRET_KEY, "utf-8"), msg=bytes(full_msg, "utf-8"), digestmod="sha256"
        ).hexdigest()
        return token, ts
