# AUTOGENERATED! DO NOT EDIT! File to edit: 04_hitmap_wcov.ipynb (unless otherwise specified).

__all__ = ['simulations_sampling_frequency_scaling', 'load_hitmap_wcov']

# Cell

import healpy as hp
import numpy as np
from pathlib import Path
import logging as log

from .core import get_telescope, base_folder, simulations_observing_efficiency
from .noise import get_thinfp, get_tube_years

# Cell

simulations_sampling_frequency_scaling = dict(SAT=5, LAT=2)

# Cell


def load_hitmap_wcov(config, site, channel, realization=0):
    """Load hitmaps and white noise covariance matrices for a channel

    This loads the simulated hitmaps and white noise covariance matrices
    and scales them properly to the experiment configuration and duration
    as defined in the input config file.
    Hitmaps assumes a sampling frequency of 100 Hz for SAT and 400 Hz for
    LAT.

    Parameters
    ----------
    config : dict
        CMB-S4 configuration,
        generally loaded from a TOML file
    site : str
        'Pole' or 'Chile', case doesn't matter
    channel : str
        Channel tag, e.g. 'MFHS1'
    realization : int
        Choose one of the available 8 realizations

    Returns
    -------
    hitmap : numpy array
        Hitmap for all channels all tubes combined
    wcov : numpy array
        White noise covariance matrix, rows are:
        "II", "IQ", "IU", "QQ", "QU", "UU", units are K^2
    """

    # it is the same scaling for hitmap and white noise covariance matrix,
    # which is the same as noise except squared

    telescope = get_telescope(channel)

    scaling = (365.25 * config["experiment"]["observing_efficiency"]) / (
        10 * simulations_observing_efficiency[site.lower()].get(telescope, 1)
    )
    scaling *= get_tube_years(config, site, channel)
    # focal plane thinning factor of TOD simulations
    scaling *= get_thinfp(channel)

    map_filename = (
        Path(f"{realization:08d}")
        / f"{site.lower()}_noise_{telescope}_{channel}_filtered_telescope_all_time_all_hmap.fits"
    )
    wcov_filename = (
        Path(f"{realization:08d}")
        / f"{site.lower()}_noise_{telescope}_{channel}_filtered_telescope_all_time_all_wcov.fits"
    )
    log.info(f"Base folder: {base_folder}")
    log.info(f"Reading {map_filename}")
    hitmap = hp.read_map(Path(base_folder) / map_filename, 0, dtype=None, verbose=False)
    wcov = hp.read_map(
        Path(base_folder) / wcov_filename, (0, 1, 2, 3, 4, 5), dtype=None, verbose=False
    )
    hitmap[hitmap == hp.UNSEEN] = 0
    # input map is 10 days at 100% efficiency
    hitmap = np.round(hitmap * scaling).astype(np.int64)
    hitmap *= simulations_sampling_frequency_scaling[telescope]
    wcov /= scaling
    wcov[:, hitmap == 0] = hp.UNSEEN
    return hitmap, wcov