"""
Base classes for datasets, which are collections of files generated with some degree of coherence.
"""
import inspect
import pathlib
import os
from io import IOBase
from abc import ABC
from typing import Any, Dict, List, Tuple, Union, Callable, Optional
from pathlib import Path
import warnings
from erfa.core import ErfaWarning

import numpy as np
from astropy.io import fits
from tqdm import tqdm

from dkist_fits_specifications.utils import expand_keys
from voluptuous.schema_builder import _compile_itemsort

from dkist_data_simulator.schemas import Schema


__all__ = ['Dataset', 'key_function', 'Omit']


class Omit:
    """
    A placeholder for a key which should be omitted from the header.

    If a generated key has an instance of this class as a value the key will be
    stripped from the generated header.
    """


class key_function:
    """
    A decorator to mark a method as a function for generating a key.

    Parameters
    ----------
    args
        The key, or multiple keys, to use this function to generate.
    """

    def __init__(self, *args: str):
        self.key = args[0] if len(args) == 1 else args

    def __call__(self, f: Callable) -> Callable:
        f.key = self.key

        return f


class Dataset(ABC):
    """
    A collection of FITS which conform to a schema.

    Parameters
    ----------
    file_schema
        The schema to which all files generated by this dataset must conform.
    dataset_shape
        The full shape of the dataset, including all dimensions, i.e. the size
        of the reconstructed array when all files are combined.
    array_shape
        The shape of the array in the files.

    Notes
    -----

    **All dimensionality and ordering in this code is in *numpy* order and not FITS order.**

    **Dimensionality and shapes**

    There are numerous different ways to consider the dimensionality of the
    dataset and its component arrays. The simplest possible case is that the
    arrays contain two dimensions with non-unity length, i.e. ``(4096, 4096)``.
    Then an additional time dimension is striped across a series of FITS files,
    in this case the full shape of the dataset would be ``(1024, 4096, 4096)``
    if there were 1024 files in the dataset.

    This is however further complicated when "dummy dimensions" are added to
    the FITS files, dummy dimensions are dimensions which are added to
    facilitate the representation of a WCS world coordinate in the metadata,
    while there is no array data for that dimension. Due to the fact that FITS
    requires the number of array dimensions to match the number of WCS world
    dimensions, length one axes are added to the array. This might result in a
    FITS array with shape ``(1, 4096, 4096)`` and a dataset with shape
    ``(1024, 4096, 4096)``. The third dimension is still striped across the
    third axis, but that axes is also included in the array shape.

    This class provides access to four different shape attributes to facilitate
    inspection of the different combinations, they are derived from input of
    the array shape (including dummy dimensions) and the dataset shape
    (including array dimensions).

    **Restrictions on Ordering**

    Array dimensions and dummy dimensions can not be placed anywhere in the
    dataset. Data dimensions (non-dummy array dimensions) must come after
    dummy dimensions (non-data array dimensions) and file dimensions must be
    first.

    Array Dimensions = ``[dummy dimensions, data dimensions]``
    Dataset Dimensions = ``[file dimensions, array dimensions]``
    """

    file_schema: Schema
    """
    The schema to which all files generated by this dataset must conform.
    """

    dataset_shape: Tuple[int, ...]
    """
    The full shape of the dataset, including all dimensions, i.e. the size of
    the reconstructed array when all files are combined.
    """

    array_shape: Tuple[int, ...]
    """
    The shape of the array in the files.
    """

    data_shape: Tuple[int, ...]
    """
    The shape of the dimensions in the array excluding dummy dimensions.
    """

    files_shape: Tuple[int, ...]
    """
    The shape of the dimensions not in the array (excluding dummy dimensions).
    """

    array_ndim: int
    """
    The number of dimensions in the arrays, including dummy (length 1 dimensions).
    """

    data_ndim: int
    """
    The number of non-dummy dimensions in the arrays.
    """

    dataset_ndim: int
    """
    The total number of dimensions in the dataset.
    """

    def _inspect_decorated_generator_functions(self) -> Dict[Tuple[str, ...], Callable]:
        """
        Find all methods which have a key attribute (added by the decorator)
        and add them to generator_functions.
        """
        functions = inspect.getmembers(type(self), predicate=inspect.isfunction)
        generator_functions = filter(lambda func: hasattr(func[1], "key"), functions)
        return {func[1].key: func[1] for func in generator_functions}

    def __init__(self,
                 file_schema: Schema,
                 dataset_shape: Tuple[int, ...],
                 array_shape: Tuple[int, ...]):

        self.random = file_schema.random

        self.file_schema = file_schema
        self.dataset_shape = dataset_shape
        self.array_shape = array_shape

        self.data_shape = tuple(d for d in array_shape if d != 1)
        if not set(self.data_shape).issubset(set(dataset_shape)):
            raise ValueError(
                f"The shape of the data ({self.data_shape}) is not "
                f"a subset of the dataset shape ({self.dataset_shape})."
            )

        self.data_ndim = len(self.data_shape)
        self.array_ndim = len(array_shape)
        self.files_shape = self.dataset_shape[:-1 * self.data_ndim]
        self.files_ndim = len(self.files_shape)
        self.dataset_ndim = len(dataset_shape)
        self.n_files = int(np.product(self.files_shape))

        self._index = 0  # The current position in the dataset
        self._remove_keys: List[str] = list()
        self._fixed_keys: Dict[str, Any] = dict()
        self._generator_functions: Dict[str, Callable[[Dataset, str], Any]] = dict()

        # Register any generator functions passed in as kwarg
        for gkey, gfunc in self._inspect_decorated_generator_functions().items():
            self.add_generator_function(gkey, gfunc)

    @property
    def data(self):
        """
        The data array for the current index of the dataset.
        """
        raise NotImplementedError("This dataset has not implemented generating data.")

    @property
    def filename(self) -> str:
        """
        The filename for the current index of the dataset.
        """

    def reset(self):
        """
        Reset the state of the dataset generator.
        """
        self._index = 0

    @property
    def index(self):
        """
        Current file number being generated.
        """
        return self._index

    @property
    def file_index(self):
        """
        Index along each of the file dimensions for the current file being generated.
        """
        return np.unravel_index(self.index, self.files_shape, order='C')

    def _validate_duplicate_key(self, key: str):
        if key in self._generator_functions:
            raise ValueError(f"The key {key} already has a generator function for this dataset.")

    def add_generator_function(self, key: Union[str, Tuple[str, ...]],
                               function: Callable[["Dataset", str], Any]):
        """
        Call a function to generate a value for one or more keys.

        Parameters
        ----------
        key
            The key, or tuple of keys to call this function for.

        function
            The function to call, the function will be passed two arguments,
            the `.Dataset` object and the key being generated. The function
            should return the value.
        """
        if isinstance(key, str):
            key = (key,)

        for akey in key:
            for k in expand_keys(self.array_ndim, [akey]):
                self._validate_duplicate_key(k)
                self._generator_functions[k] = function

    def add_constant_key(self, key: str, value: Any = None):
        """
        Add a key which is constant over the dataset.

        Parameters
        ----------
        key
            The key to fix over the dataset.
        value
            The value for the fixed key. If `None` a random value will be
            generated based on the schema and then used for all frames in the
            dataset.

        Notes
        -----

        A constant key takes precedence over a function key. So for the 214
        dataset, some keys are generated by functions unless overridden by
        constants in subclasses.

        If this function is called more than once with the same key, the latest
        call overrides the earlier ones. This is used by the 214 generator to
        set random defaults in the base class init which are then overridden in
        subclass constructors.
        """
        for k in expand_keys(self.array_ndim, [key]):
            if value is None:
                value = self.file_schema[k].generate_value()
            self._fixed_keys[k] = value

    def add_remove_key(self, key: str):
        """
        Add a key to a list to ignore.

        These keys are removed from the header after generation.

        Parameters
        ----------
        key
            The key to remove.
        """
        self._remove_keys.append(key)

    def __iter__(self):
        if self._index == self.n_files - 1:
            return

        for index in range(self._index, self.n_files):
            self._index = index
            yield self

    def __len__(self):
        return self.n_files

    def __next__(self):
        if self._index == self.n_files:
            raise StopIteration
        self._index += 1
        return self

    def header(self, required_only: bool = False, expected_only: bool = False) -> Dict[str, Any]:
        """
        The header for this index of the dataset.

        Parameters
        ----------
        required_only
            Generate only headers required by the DKIST data center.
        expected_only
            Generate only the headers which are nominally expected to exist.
        """
        with warnings.catch_warnings():
            warnings.simplefilter('ignore')

            generated_keys = {gkey: gfunc(self, gkey)
                              for gkey, gfunc in self._generator_functions.items()}
            known_keys = {**generated_keys, **self._fixed_keys}

            omit_keys = [k for k, v in known_keys.items() if isinstance(v, Omit)]

            header = self.file_schema.generate(required_only=required_only,
                                               expected_only=expected_only,
                                               **known_keys)

        for key in self._remove_keys + omit_keys:
            header.pop(key, None)

        schema_keys = []
        for sect in self.file_schema.sections:
            schema_keys += [k.key for k in sect.keys]
        non_schema_keys = set(header.keys()).difference(schema_keys)

        for key in non_schema_keys:
            warnings.warn(f"Generating key not in schema: {key}")

        return header

    def hdu(self,
            rice_compress: Optional[Union[bool, dict]] = None,
            **kwargs,
            ) -> Union[fits.PrimaryHDU, fits.CompImageHDU]:
        """
        The fits HDU for this index of the data set

        Parameters
        ----------
        rice_compress
            Use RICE compression on the files. If a `dict` the arguments are
            passed to `astropy.io.fits.CompImageHDU`.
        """
        if rice_compress:
            # If is True then cast to an empty dict
            if not isinstance(rice_compress, dict):
                rice_compress = dict()

            # Generate the uncompressed HDU first, due to some weirdness in astropy CompImageHDU.
            # https://github.com/astropy/astropy/issues/11694
            # This means Astropy puts in all the required keys relating to the data
            image_hdu = fits.ImageHDU(data=self.data, header=fits.Header(self.header(**kwargs)))
            return fits.CompImageHDU(data=image_hdu.data, header=image_hdu.header, **rice_compress)

        return fits.PrimaryHDU(data=self.data, header=fits.Header(self.header()))

    def file(self,
             target: Union[pathlib.Path, IOBase],
             rice_compress: Optional[Union[bool, dict]] = None,
             checksum: bool = True,
             **kwargs,
             ) -> Union[pathlib.Path, IOBase, Tuple[Union[pathlib.Path, IOBase], fits.Header, Any]]:
        """
        The fits file for this index of the data set.

        Parameters
        ----------
        target
            The object or path to save the generated bytes to.
        rice_compress
            Use RICE compression on the files. If a `dict` the arguments are
            passed to `astropy.io.fits.CompImageHDU`.
        checksum
            If `True`, adds both ``DATASUM`` and ``CHECKSUM`` cards to the
            headers of all HDU's written to the file.
        kwargs
           All other keyword arguments are passed to header
        """
        if isinstance(target, os.PathLike):
            target = str(target)

        hdu = self.hdu(rice_compress, **kwargs)
        hdu.writeto(
            target,
            output_verify="exception",
            overwrite=True,
            checksum=checksum
        )

        if not isinstance(target, str):
            target.seek(0)

        return target

    def generate_headers(self,
                         required_only: bool = False,
                         expected_only: bool = False) -> List[Dict[str, Any]]:
        """
        Generate a set of Headers based on the schema.

        Parameters
        ----------
        required_only
            Generate only headers required by the DKIST data center.
        expected_only
            Generate only the headers which are nominally expected to exist.
        """
        return [self.header(required_only, expected_only) for ds in self]

    def generate_files(self,
                       base_path: pathlib.Path,
                       filename_template: str = None,
                       progress_bar: bool = False,
                       **kwargs
                       ) -> List[pathlib.Path]:
        """
        Generate a series of FITS files for this dataset.

        Parameters
        ----------
        base_path
            The directory to save the FITS files in.
        filename_template
            A format-string which gets passed the variable ``ds`` which is the dataset object.
        kwargs
            All other keyword arguments are passed to `Dataset.file`.
        """
        base_path = Path(base_path)
        base_path.mkdir(parents=True, exist_ok=True)

        generated_files = []

        iterator = tqdm(self, total=self.n_files) if progress_bar else self

        for ds in iterator:
            if filename_template is None:
                filename_template = "{ds.index}.fits"

            filename = filename_template
            if "{" in filename_template:
                filename = filename_template.format(ds=ds)

            target = base_path / filename

            generated_files.append(pathlib.Path(ds.file(target, **kwargs)))

        return generated_files
