"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceReadReplica = exports.DatabaseInstanceFromSnapshot = exports.DatabaseInstance = exports.StorageType = exports.LicenseModel = exports.DatabaseInstanceBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * A new or imported database instance.
 *
 * @stability stable
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     *
     * @stability stable
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
                this.engine = attrs.engine;
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a new db proxy to this instance.
     *
     * @stability stable
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromInstance(this),
            ...options,
        });
    }
    /**
     * Grant the given identity connection access to the database.
     *
     * @stability stable
     */
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [this.instanceArn],
        });
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     *
     * @stability stable
     */
    get instanceArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
_a = JSII_RTTI_SYMBOL_1;
DatabaseInstanceBase[_a] = { fqn: "@aws-cdk/aws-rds.DatabaseInstanceBase", version: "1.95.1" };
/**
 * The license model.
 *
 * @stability stable
 */
var LicenseModel;
(function (LicenseModel) {
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 *
 * @stability stable
 */
var StorageType;
(function (StorageType) {
    StorageType["STANDARD"] = "standard";
    StorageType["GP2"] = "gp2";
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        var _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(scope, id);
        this.vpc = props.vpc;
        if (props.vpcSubnets && props.vpcPlacement) {
            throw new Error('Only one of `vpcSubnets` or `vpcPlacement` can be specified');
        }
        this.vpcPlacement = (_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : props.vpcPlacement;
        const subnetGroup = (_f = props.subnetGroup) !== null && _f !== void 0 ? _f : new subnet_group_1.SubnetGroup(this, 'SubnetGroup', {
            description: `Subnet group for ${this.node.id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            removalPolicy: util_1.renderUnless(util_1.helperRemovalPolicy(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.number({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.enableIamAuthentication = props.iamAuthentication;
        const enablePerformanceInsights = props.enablePerformanceInsights
            || props.performanceInsightRetention !== undefined || props.performanceInsightEncryptionKey !== undefined;
        if (enablePerformanceInsights && props.enablePerformanceInsights === false) {
            throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
        }
        if (props.domain) {
            this.domainId = props.domain;
            this.domainRole = props.domainRole || new iam.Role(this, 'RDSDirectoryServiceRole', {
                assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSDirectoryServiceAccess'),
                ],
            });
        }
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: (_g = props.backupRetention) === null || _g === void 0 ? void 0 : _g.toDays(),
            copyTagsToSnapshot: (_h = props.copyTagsToSnapshot) !== null && _h !== void 0 ? _h : true,
            dbInstanceClass: core_1.Lazy.string({ produce: () => `db.${this.instanceType}` }),
            dbInstanceIdentifier: props.instanceIdentifier,
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: core_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            enablePerformanceInsights: enablePerformanceInsights || props.enablePerformanceInsights,
            iops,
            monitoringInterval: (_j = props.monitoringInterval) === null || _j === void 0 ? void 0 : _j.toSeconds(),
            monitoringRoleArn: monitoringRole === null || monitoringRole === void 0 ? void 0 : monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: (_k = props.optionGroup) === null || _k === void 0 ? void 0 : _k.optionGroupName,
            performanceInsightsKmsKeyId: (_l = props.performanceInsightEncryptionKey) === null || _l === void 0 ? void 0 : _l.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (props.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: (_m = props.port) === null || _m === void 0 ? void 0 : _m.toString(),
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: (_o = props.publiclyAccessible) !== null && _o !== void 0 ? _o : (this.vpcPlacement && this.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC),
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
            domain: this.domainId,
            domainIamRoleName: (_p = this.domainRole) === null || _p === void 0 ? void 0 : _p.roleName,
        };
    }
    /**
     * @stability stable
     */
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new logs.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        var _e, _f, _g, _h, _j, _k;
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.engine = props.engine;
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props, true);
        const engineConfig = props.engine.bindToInstance(this, {
            ...props,
            s3ImportRole,
            s3ExportRole,
        });
        const instanceAssociatedRoles = [];
        const engineFeatures = engineConfig.features;
        if (s3ImportRole) {
            if (!(engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Import)) {
                throw new Error(`Engine '${util_1.engineDescription(props.engine)}' does not support S3 import`);
            }
            instanceAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Import });
        }
        if (s3ExportRole) {
            if (!(engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Export)) {
                throw new Error(`Engine '${util_1.engineDescription(props.engine)}' does not support S3 export`);
            }
            // Only add the export role and feature if they are different from the import role & feature.
            if (s3ImportRole !== s3ExportRole || engineFeatures.s3Import !== (engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Export)) {
                instanceAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Export });
            }
        }
        this.instanceType = (_e = props.instanceType) !== null && _e !== void 0 ? _e : ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
        const instanceParameterGroupConfig = (_f = props.parameterGroup) === null || _f === void 0 ? void 0 : _f.bindToInstance({});
        this.sourceCfnProps = {
            ...this.newCfnProps,
            associatedRoles: instanceAssociatedRoles.length > 0 ? instanceAssociatedRoles : undefined,
            optionGroupName: (_g = engineConfig.optionGroup) === null || _g === void 0 ? void 0 : _g.optionGroupName,
            allocatedStorage: (_j = (_h = props.allocatedStorage) === null || _h === void 0 ? void 0 : _h.toString()) !== null && _j !== void 0 ? _j : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            engine: props.engine.engineType,
            engineVersion: (_k = props.engine.engineVersion) === null || _k === void 0 ? void 0 : _k.fullVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param options the options for the rotation, if you want to override the defaults.
     * @stability stable
     */
    addRotationSingleUser(options = {}) {
        var _e;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
            ...options,
            excludeCharacters: (_e = options.excludeCharacters) !== null && _e !== void 0 ? _e : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     *
     * @stability stable
     */
    addRotationMultiUser(id, options) {
        var _e;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_e = options.excludeCharacters) !== null && _e !== void 0 ? _e : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * A database instance.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: credentials.username,
            masterUserPassword: (_e = credentials.password) === null || _e === void 0 ? void 0 : _e.toString(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy((_f = props.removalPolicy) !== null && _f !== void 0 ? _f : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
_b = JSII_RTTI_SYMBOL_1;
DatabaseInstance[_b] = { fqn: "@aws-cdk/aws-rds.DatabaseInstance", version: "1.95.1" };
/**
 * A database instance restored from a snapshot.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f, _g, _h;
        super(scope, id, props);
        let credentials = props.credentials;
        let secret = credentials === null || credentials === void 0 ? void 0 : credentials.secret;
        if (!secret && (credentials === null || credentials === void 0 ? void 0 : credentials.generatePassword)) {
            if (!credentials.username) {
                throw new Error('`credentials` `username` must be specified when `generatePassword` is set to true');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: credentials.username,
                encryptionKey: credentials.encryptionKey,
                excludeCharacters: credentials.excludeCharacters,
                replaceOnPasswordCriteriaChanges: credentials.replaceOnPasswordCriteriaChanges,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: (_f = (_e = secret === null || secret === void 0 ? void 0 : secret.secretValueFromJson('password')) === null || _e === void 0 ? void 0 : _e.toString()) !== null && _f !== void 0 ? _f : (_g = credentials === null || credentials === void 0 ? void 0 : credentials.password) === null || _g === void 0 ? void 0 : _g.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy((_h = props.removalPolicy) !== null && _h !== void 0 ? _h : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
_c = JSII_RTTI_SYMBOL_1;
DatabaseInstanceFromSnapshot[_c] = { fqn: "@aws-cdk/aws-rds.DatabaseInstanceFromSnapshot", version: "1.95.1" };
/**
 * A read replica database instance.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        /**
         * The engine of this database Instance.
         *
         * May be not known for imported Instances if it wasn't provided explicitly,
         * or for read replicas.
         *
         * @stability stable
         */
        this.engine = undefined;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: (_e = props.storageEncryptionKey) === null || _e === void 0 ? void 0 : _e.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceType = props.instanceType;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy((_f = props.removalPolicy) !== null && _f !== void 0 ? _f : core_1.RemovalPolicy.SNAPSHOT);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
_d = JSII_RTTI_SYMBOL_1;
DatabaseInstanceReadReplica[_d] = { fqn: "@aws-cdk/aws-rds.DatabaseInstanceReadReplica", version: "1.95.1" };
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
//# sourceMappingURL=data:application/json;base64,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