"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HIPAASecurityChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const cloudtrail_1 = require("./rules/cloudtrail");
const ec2_1 = require("./rules/ec2");
/**
 * Check for HIPAA Security compliance.
 *
 * Based on the HIPAA Security AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-hipaa_security.html
 *
 * @stability stable
 */
class HIPAASecurityChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            // this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            // this.checkCloudWatch(node, ignores);
            // this.checkCodeBuild(node, ignores);
            // this.checkDMS(node, ignores);
            // this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            // this.checkECS(node, ignores);
            // this.checkEFS(node, ignores);
            // this.checkElastiCache(node, ignores);
            // this.checkElasticBeanstalk(node, ignores);
            // this.checkElasticsearch(node, ignores);
            // this.checkELB(node, ignores);
            // this.checkEMR(node, ignores);
            // this.checkIAM(node, ignores);
            // this.checkLambda(node, ignores);
            // this.checkRDS(node, ignores);
            // this.checkRedshift(node, ignores);
            // this.checkS3(node, ignores);
            // this.checkSageMaker(node, ignores);
            // this.checkSecretsManager(node, ignores);
            // this.checkSNS(node, ignores);
            // this.checkVPC(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWCacheEnabledAndEncrypted') &&
            !apigw_1.hipaaSecurityAPIGWCacheEnabledAndEncrypted(node)) {
            const ruleId = 'HIPAA.Security-APIGWCacheEnabledAndEncrypted';
            const info = 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWExecutionLoggingEnabled') &&
            !apigw_1.hipaaSecurityAPIGWExecutionLoggingEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWExecutionLoggingEnabled';
            const info = 'The API Gateway stage does not have execution logging enabled for all methods - (Control ID: 164.312(b)).';
            const explanation = 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWSSLEnabled') &&
            !apigw_1.hipaaSecurityAPIGWSSLEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWSSLEnabled';
            const info = 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWXrayEnabled') &&
            !apigw_1.hipaaSecurityAPIGWXrayEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWXrayEnabled';
            const info = 'The API Gateway REST API stage does not have X-Ray enabled - (Control ID: 164.312(b)).';
            const explanation = 'AWS X-Ray collects data about requests that your application serves, and provides tools you can use to view, filter, and gain insights into that data to identify issues and opportunities for optimization. Ensure X-Ray is enables so you can see detailed information not only about the request and response, but also about calls that your application makes to downstream AWS resources, microservices, databases and HTTP web APIs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    //   /**
    //    * Check Auto Scaling Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkAutoScaling(node: CfnResource, ignores: any): void {}
    /**
     * Check CloudTrail Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailCloudWatchLogsEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled';
            const info = 'The trail does not have CloudWatch logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).';
            const explanation = 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailEncryptionEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailEncryptionEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailEncryptionEnabled';
            const info = 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailLogFileValidationEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailLogFileValidationEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailLogFileValidationEnabled';
            const info = 'The trail does not have log file validation enabled - (Control ID: 164.312(c)(1), 164.312(c)(2)).';
            const explanation = 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    //   /**
    //    * Check CloudWatch Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkCloudWatch(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check CodeBuild Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkCodeBuild(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check DMS Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkDMS(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check DynamoDB Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkDynamoDB(node: CfnResource, ignores: any): void {}
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled') &&
            !ec2_1.hipaaSecurityEC2InstanceDetailedMonitoringEnabled(node)) {
            const ruleId = 'HIPAA.Security-C2InstanceDetailedMonitoringEnabled';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: 164.312(b)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstancesInVPC') &&
            !ec2_1.hipaaSecurityEC2InstancesInVPC(node)) {
            const ruleId = 'HIPAA.Security-EC2InstancesInVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2CheckNoPublicIPs') &&
            !ec2_1.hipaaSecurityEC2InstanceNoPublicIp(node)) {
            const ruleId = 'HIPAA.Security-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.HIPAASecurityChecks = HIPAASecurityChecks;
_a = JSII_RTTI_SYMBOL_1;
HIPAASecurityChecks[_a] = { fqn: "monocdk-nag.HIPAASecurityChecks", version: "0.0.55" };
//# sourceMappingURL=data:application/json;base64,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