# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_levelSetKDEx_multivariate.ipynb.

# %% ../nbs/01_levelSetKDEx_multivariate.ipynb 4
from __future__ import annotations
from fastcore.docments import *
from fastcore.test import *
from fastcore.utils import *

import pandas as pd
import numpy as np
import faiss
from scipy.spatial.distance import cdist
from scipy.optimize import linear_sum_assignment
from scipy.spatial import KDTree
from sklearn.base import BaseEstimator
from sklearn.exceptions import NotFittedError

from collections import defaultdict, Counter, deque
from joblib import Parallel, delayed, dump, load
import copy
import warnings

from .baseClasses import BaseWeightsBasedEstimator_multivariate
from .levelSetKDEx import BaseLSx
from .wSAA import SampleAverageApproximation
from .utils import restructureWeightsDataList_multivariate

# %% auto 0
__all__ = ['LevelSetKDEx_multivariate']

# %% ../nbs/01_levelSetKDEx_multivariate.ipynb 6
class LevelSetKDEx_multivariate(BaseWeightsBasedEstimator_multivariate, BaseLSx):
    """
    `LevelSetKDEx` turns any point forecasting model into an estimator of the underlying conditional density.
    The name 'LevelSet' stems from the fact that this approach interprets the values of the point forecasts
    as a similarity measure between samples. The point forecasts of the training samples are sorted and 
    recursively assigned to a bin until the size of the current bin reaches `binSize` many samples. Then
    a new bin is created and so on. For a new test sample we check into which bin its point prediction
    would have fallen and interpret the training samples of that bin as the empirical distribution function
    of this test sample.    
    """
    
    def __init__(self, 
                 estimator, # Model with a .fit and .predict-method (implementing the scikit-learn estimator interface).
                 binSize: int=None, # Size of the bins created while running fit.
                 # Determines behaviour of method `getWeights`. If False, all weights receive the same  
                 # value. If True, the distance of the point forecasts is taking into account.
                 equalBins: bool = False,
                 weightsByDistance: bool=False, 
                 ):
        
        super(BaseEstimator, self).__init__(estimator = estimator,
                                            binSize = binSize,
                                            weightsByDistance = weightsByDistance)
        
        self.equalBins = equalBins
        self.yTrain = None
        self.yPredTrain = None
        self.indicesPerBin = None
        self.lowerBoundPerBin = None
        self.fitted = False
    
    #---
    
    def fit(self: LevelSetKDEx, 
            X: np.ndarray, # Feature matrix used by `estimator` to predict `y`.
            y: np.ndarray, # 1-dimensional target variable corresponding to the feature matrix `X`.
            ):
        """
        Fit `LevelSetKDEx` model by grouping the point predictions of the samples specified via `X`
        according to their value. Samples are recursively sorted into bins until each bin contains
        `binSize` many samples. For details, checkout the function `generateBins` which does the
        heavy lifting.
        """
        
        # Checks
        if self.binSize is None:
            raise ValueError("'binSize' must be specified to fit the LSx estimator!")
            
        if self.binSize > y.shape[0]:
            raise ValueError("'binSize' mustn't be bigger than the size of 'y'!")
        
        if X.shape[0] != y.shape[0]:
            raise ValueError("'X' and 'y' must contain the same number of samples!")
        
        # IMPORTANT: In case 'y' is given as a pandas.Series, we can potentially run into indexing 
        # problems later on.
        if isinstance(y, pd.Series):
            y = y.ravel()
        
        #---
        
        try:
            yPred = self.estimator.predict(X)
            
        except NotFittedError:
            try:
                self.estimator.fit(X = X, y = y)                
            except:
                raise ValueError("Couldn't fit 'estimator' with user specified 'X' and 'y'!")
            else:
                yPred = self.estimator.predict(X)
        
        #---
        
        if len(y.shape) == 1:
            y = y.reshape(-1, 1)
            yPred = yPred.reshape(-1, 1)
        
        #---
        
        nClusters = int(np.ceil(yPred.shape[0] / self.binSize))
        yPredMod = yPred.astype(np.float32)
        
        kmeans = faiss.Kmeans(d = yPredMod.shape[1], k = nClusters)
        kmeans.train(yPredMod)
        self.centers = kmeans.centroids
        
        if self.equalBins:
            clusterAssignments = self._getEqualSizedClusters(y = yPredMod)            
            
        else:
            clusterAssignments = kmeans.assign(yPredMod)[1]
        
        
        indicesPerBin = defaultdict(list)
        clusterSizes = defaultdict(int)
        for index, cluster in enumerate(clusterAssignments):
            indicesPerBin[cluster].append(index)
            clusterSizes[cluster] += 1
        
        clusterSizes = pd.Series(clusterSizes)
        

        clustersTooSmall = clusterSizes.index[np.where(clusterSizes < self.binSize / 2)[0]]
        
        if len(clustersTooSmall) > 0:
            centersToMerge = self.centers[clustersTooSmall]
            centersRemaining = np.delete(self.centers, clustersTooSmall, axis = 0)

            nearestCenterSearch = KDTree(centersRemaining)
            clusterMergeDict = dict(zip(clustersTooSmall, nearestCenterSearch.query(centersToMerge)[1]))
            
            for clusterToMerge, newCluster in clusterMergeDict.items():
                indicesPerBin[newCluster].extend(indicesPerBin[clusterToMerge])
                indicesPerBin[clusterToMerge] = None
        
        else:
            clusterMergeDict = None
        
        indicesPerBin = {binIndex: np.array(indices) for binIndex, indices in indicesPerBin.items()}
        
        #---

        self.yTrain = y
        self.yPredTrain = yPred
        self.indicesPerBin = indicesPerBin
        self.kmeans = kmeans
        self.clusterMergeDict = clusterMergeDict
        self.fitted = True
        
    #---
    
    def _getEqualSizedClusters(self,
                               y,
                               ):
            
        centers = self.centers.reshape(-1, 1, y.shape[-1]).repeat(self.binSize, 1).reshape(-1, y.shape[-1])

        distance_matrix = cdist(y, centers)
        clusterAssignments = linear_sum_assignment(distance_matrix)[1]//self.binSize

        return clusterAssignments

    #---
    
    def getWeights(self, 
                   X: np.ndarray, # Feature matrix for which conditional density estimates are computed.
                   # Specifies structure of the returned density estimates. One of: 
                   # 'all', 'onlyPositiveWeights', 'summarized', 'cumDistribution', 'cumDistributionSummarized'
                   outputType: str='onlyPositiveWeights', 
                   # Optional. List with length X.shape[0]. Values are multiplied to the estimated 
                   # density of each sample for scaling purposes.
                   scalingList: list=None, 
                   ) -> list: # List whose elements are the conditional density estimates for the samples specified by `X`.
        
        # __annotations__ = BaseWeightsBasedEstimator.getWeights.__annotations__
        __doc__ = BaseWeightsBasedEstimator_multivariate.getWeights.__doc__
        
        if not self.fitted:
            raise NotFittedError("This LevelSetKDEx instance is not fitted yet. Call 'fit' with "
                                 "appropriate arguments before trying to compute weights.")
        
        #---
        
        yPred = self.estimator.predict(X).astype(np.float32)
        
        if len(yPred.shape) == 1:
            yPred = yPred.reshape(-1, 1)
            
        #---
        
        if self.equalBins:
            binPerPred = self._getEqualSizedClusters(y = yPred)
            
        else:
            binPerPred = self.kmeans.assign(yPred)[1]
            
            binPerPredUnique = np.unique(binPerPred)
            clustersToMerge = np.array(list(self.clusterMergeDict.keys()))
            clustersToMod = clustersToMerge[[clusterToMerge in binPerPredUnique for clusterToMerge in clustersToMerge]]

            if len(clustersToMod) > 0:
                binPerPred = np.select([cluster == binPerPred for cluster in clustersToMod], 
                                       [self.clusterMergeDict[cluster] for cluster in clustersToMod],
                                       binPerPred)
        
        #---
        
        neighborsList = [self.indicesPerBin[binIndex] for binIndex in binPerPred]
                
        weightsDataList = [(np.repeat(1 / len(neighbors), len(neighbors)), np.array(neighbors)) for neighbors in neighborsList]
        
        weightsDataList = restructureWeightsDataList_multivariate(weightsDataList = weightsDataList, 
                                                                  outputType = outputType, 
                                                                  y = self.yTrain,
                                                                  scalingList = scalingList,
                                                                  equalWeights = True)
        
        return weightsDataList
    
