"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const props_1 = require("./props");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine extends props_1.DatabaseClusterEngine {
    constructor() {
        super(...arguments);
        /* tslint:enable max-line-length */
        /** To make it a compile-time error to pass a DatabaseClusterEngine where a DatabaseInstanceEngine is expected. */
        this.isDatabaseInstanceEngine = true;
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/* tslint:disable max-line-length */
DatabaseInstanceEngine.MARIADB = new DatabaseInstanceEngine('mariadb', secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER, [
    { engineMajorVersion: '10.0', parameterGroupFamily: 'mariadb10.0' },
    { engineMajorVersion: '10.1', parameterGroupFamily: 'mariadb10.1' },
    { engineMajorVersion: '10.2', parameterGroupFamily: 'mariadb10.2' },
    { engineMajorVersion: '10.3', parameterGroupFamily: 'mariadb10.3' },
]);
DatabaseInstanceEngine.MYSQL = new DatabaseInstanceEngine('mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'mysql5.6' },
    { engineMajorVersion: '5.7', parameterGroupFamily: 'mysql5.7' },
    { engineMajorVersion: '8.0', parameterGroupFamily: 'mysql8.0' },
]);
DatabaseInstanceEngine.ORACLE_EE = new DatabaseInstanceEngine('oracle-ee', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-ee-11.2' },
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-ee-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-ee-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-ee-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-ee-19' },
]);
DatabaseInstanceEngine.ORACLE_SE2 = new DatabaseInstanceEngine('oracle-se2', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-se2-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-se2-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-se2-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-se2-19' },
]);
DatabaseInstanceEngine.ORACLE_SE1 = new DatabaseInstanceEngine('oracle-se1', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se1-11.2' },
]);
DatabaseInstanceEngine.ORACLE_SE = new DatabaseInstanceEngine('oracle-se', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se-11.2' },
]);
DatabaseInstanceEngine.POSTGRES = new DatabaseInstanceEngine('postgres', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.3', parameterGroupFamily: 'postgres9.3' },
    { engineMajorVersion: '9.4', parameterGroupFamily: 'postgres9.4' },
    { engineMajorVersion: '9.5', parameterGroupFamily: 'postgres9.5' },
    { engineMajorVersion: '9.6', parameterGroupFamily: 'postgres9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'postgres10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'postgres11' },
]);
DatabaseInstanceEngine.SQL_SERVER_EE = new DatabaseInstanceEngine('sqlserver-ee', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ee-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ee-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ee-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ee-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_SE = new DatabaseInstanceEngine('sqlserver-se', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-se-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-se-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-se-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-se-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_EX = new DatabaseInstanceEngine('sqlserver-ex', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ex-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ex-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ex-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ex-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_WEB = new DatabaseInstanceEngine('sqlserver-web', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-web-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-web-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-web-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-web-14.0' },
]);
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose (SSD).
     */
    StorageType["GP2"] = "gp2";
    /**
     * Provisioned IOPS (SSD).
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * The retention period for Performance Insight.
 */
var PerformanceInsightRetention;
(function (PerformanceInsightRetention) {
    /**
     * Default retention period of 7 days.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["DEFAULT"] = 7] = "DEFAULT";
    /**
     * Long term retention period of 2 years.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["LONG_TERM"] = 731] = "LONG_TERM";
})(PerformanceInsightRetention = exports.PerformanceInsightRetention || (exports.PerformanceInsightRetention = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcPlacement = props.vpcPlacement;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcPlacement);
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'SubnetGroup', {
            dbSubnetGroupDescription: `Subnet group for ${this.node.id} database`,
            subnetIds,
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.numberValue({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const deletionProtection = props.deletionProtection !== undefined ? props.deletionProtection : true;
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention ? props.backupRetention.toDays() : undefined,
            copyTagsToSnapshot: props.copyTagsToSnapshot !== undefined ? props.copyTagsToSnapshot : true,
            dbInstanceClass: `db.${props.instanceClass}`,
            dbInstanceIdentifier: props.instanceIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection,
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: props.iamAuthentication,
            enablePerformanceInsights: props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: props.optionGroup && props.optionGroup.optionGroupName,
            performanceInsightsKmsKeyId: props.enablePerformanceInsights
                ? props.performanceInsightEncryptionKey && props.performanceInsightEncryptionKey.keyArn
                : undefined,
            performanceInsightsRetentionPeriod: props.enablePerformanceInsights
                ? (props.performanceInsightRetention || PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port ? props.port.toString() : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.vpcPlacement && props.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC,
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new lambda.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const timezoneSupport = [DatabaseInstanceEngine.SQL_SERVER_EE, DatabaseInstanceEngine.SQL_SERVER_EX,
            DatabaseInstanceEngine.SQL_SERVER_SE, DatabaseInstanceEngine.SQL_SERVER_WEB];
        if (props.timezone && !timezoneSupport.includes(props.engine)) {
            throw new Error(`timezone property can be configured only for Microsoft SQL Server, not ${props.engine.name}`);
        }
        this.sourceCfnProps = {
            ...this.newCfnProps,
            allocatedStorage: props.allocatedStorage ? props.allocatedStorage.toString() : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: props.parameterGroup && props.parameterGroup.parameterGroupName,
            engine: props.engine.name,
            engineVersion: props.engineVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (!props.masterUserPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (props.generateMasterUserPassword) {
            if (!props.masterUsername) { // We need the master username to include it in the generated secret
                throw new Error('`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
            }
            if (props.masterUserPassword) {
                throw new Error('Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        else {
            if (props.masterUsername) { // It's not possible to change the master username of a RDS instance
                throw new Error('Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
            }
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
function applyInstanceDeletionPolicy(cfnDbInstance, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDbInstance.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDbInstance.applyRemovalPolicy(removalPolicy);
    }
}
//# sourceMappingURL=data:application/json;base64,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