"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const props_1 = require("./props");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            this.node.addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        if (props.removalPolicy === core_1.RemovalPolicy.RETAIN) {
            subnetGroup.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        const securityGroup = props.instanceProps.securityGroup !== undefined ?
            props.instanceProps.securityGroup : new ec2.SecurityGroup(this, 'SecurityGroup', {
            description: 'RDS security group',
            vpc: props.instanceProps.vpc,
        });
        this.securityGroupId = securityGroup.securityGroupId;
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        let s3ImportRole = props.s3ImportRole;
        if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
            if (props.s3ImportRole) {
                throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
            }
            s3ImportRole = new aws_iam_1.Role(this, 'S3ImportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ImportBuckets) {
                bucket.grantRead(s3ImportRole);
            }
        }
        let s3ExportRole = props.s3ExportRole;
        if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
            if (props.s3ExportRole) {
                throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
            }
            s3ExportRole = new aws_iam_1.Role(this, 'S3ExportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ExportBuckets) {
                bucket.grantReadWrite(s3ExportRole);
            }
        }
        let clusterParameterGroup = props.parameterGroup;
        const clusterAssociatedRoles = [];
        if (s3ImportRole || s3ExportRole) {
            if (s3ImportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn });
            }
            if (s3ExportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn });
            }
            // MySQL requires the associated roles to be specified as cluster parameters as well, PostgreSQL does not
            if (props.engine === props_1.DatabaseClusterEngine.AURORA || props.engine === props_1.DatabaseClusterEngine.AURORA_MYSQL) {
                if (!clusterParameterGroup) {
                    const parameterGroupFamily = props.engine.parameterGroupFamily(props.engineVersion);
                    if (!parameterGroupFamily) {
                        throw new Error(`No parameter group family found for database engine ${props.engine.name} with version ${props.engineVersion}.` +
                            'Failed to set the correct cluster parameters for s3 import and export roles.');
                    }
                    clusterParameterGroup = new parameter_group_1.ClusterParameterGroup(this, 'ClusterParameterGroup', {
                        family: parameterGroupFamily,
                    });
                }
                if (clusterParameterGroup instanceof parameter_group_1.ClusterParameterGroup) { // ignore imported ClusterParameterGroup
                    if (s3ImportRole) {
                        clusterParameterGroup.addParameter('aurora_load_from_s3_role', s3ImportRole.roleArn);
                    }
                    if (s3ExportRole) {
                        clusterParameterGroup.addParameter('aurora_select_into_s3_role', s3ExportRole.roleArn);
                    }
                }
            }
        }
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engine: props.engine.name,
            engineVersion: props.engineVersion,
            dbClusterIdentifier: props.clusterIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            vpcSecurityGroupIds: [this.securityGroupId],
            port: props.port,
            dbClusterParameterGroupName: clusterParameterGroup && clusterParameterGroup.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.password
                    ? props.masterUser.password.toString()
                    : undefined),
            backupRetentionPeriod: props.backup && props.backup.retention && props.backup.retention.toDays(),
            preferredBackupWindow: props.backup && props.backup.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            // Encryption
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        // if removalPolicy was not specified,
        // leave it as the default, which is Snapshot
        if (props.removalPolicy) {
            cluster.applyRemovalPolicy(props.removalPolicy);
        }
        else {
            // The CFN default makes sense for DeletionPolicy,
            // but doesn't cover UpdateReplacePolicy.
            // Fix that here.
            cluster.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
        }
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const instanceCount = props.instances != null ? props.instances : 2;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        // Get the actual subnet objects so we can depend on internet connectivity.
        const internetConnected = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets).internetConnectivityEstablished;
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new aws_iam_1.Role(this, 'MonitoringRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
                ],
            });
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
                props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                    undefined;
            const publiclyAccessible = props.instanceProps.vpcSubnets && props.instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC;
            const instance = new rds_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                engine: props.engine.name,
                engineVersion: props.engineVersion,
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(props.instanceProps.instanceType),
                publiclyAccessible,
                // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
                dbSubnetGroupName: subnetGroup.ref,
                dbParameterGroupName: props.instanceProps.parameterGroup && props.instanceProps.parameterGroup.parameterGroupName,
                monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
                monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            });
            // If removalPolicy isn't explicitly set,
            // it's Snapshot for Cluster.
            // Because of that, in this case,
            // we can safely use the CFN default of Delete for DbInstances with dbClusterIdentifier set.
            if (props.removalPolicy) {
                instance.applyRemovalPolicy(props.removalPolicy);
            }
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnected);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups: [securityGroup], defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
                this.securityGroupId = attrs.securityGroup.securityGroupId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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