"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppSyncTransformer = void 0;
const core_1 = require("@aws-cdk/core");
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const schema_transformer_1 = require("./transformer/schema-transformer");
const defaultAuthorizationConfig = {
    defaultAuthorization: {
        authorizationType: aws_appsync_1.AuthorizationType.API_KEY,
        apiKeyConfig: {
            description: "Auto generated API Key from construct",
            name: "dev"
        }
    }
};
/**
 * (experimental) AppSyncTransformer Construct.
 *
 * @experimental
 */
class AppSyncTransformer extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.isSyncEnabled = props.syncEnabled ? props.syncEnabled : false;
        const transformerConfiguration = {
            schemaPath: props.schemaPath,
            syncEnabled: (_a = props.syncEnabled) !== null && _a !== void 0 ? _a : false
        };
        const transformer = new schema_transformer_1.SchemaTransformer(transformerConfiguration);
        const outputs = transformer.transform();
        const resolvers = transformer.getResolvers();
        this.outputs = outputs;
        this.outputs.FUNCTION_RESOLVERS.forEach((resolver) => {
            switch (resolver.typeName) {
                case 'Query':
                    delete resolvers[resolver.fieldName];
                    break;
                case 'Mutation':
                    delete resolvers[resolver.fieldName];
                    break;
                case 'Subscription':
                    delete resolvers[resolver.fieldName];
                    break;
            }
        });
        this.functionResolvers = this.outputs.FUNCTION_RESOLVERS;
        this.resolvers = resolvers;
        this.nestedAppsyncStack = new core_1.NestedStack(this, `appsync-nested-stack`);
        // AppSync
        this.appsyncAPI = new aws_appsync_1.GraphqlApi(this.nestedAppsyncStack, `${id}-api`, {
            name: props.apiName ? props.apiName : `${id}-api`,
            authorizationConfig: props.authorizationConfig ? props.authorizationConfig : defaultAuthorizationConfig,
            logConfig: {
                fieldLogLevel: props.fieldLogLevel ? props.fieldLogLevel : aws_appsync_1.FieldLogLevel.NONE,
            },
            schema: aws_appsync_1.Schema.fromAsset('./appsync/schema.graphql')
        });
        let tableData = outputs.CDK_TABLES;
        // Check to see if sync is enabled
        if (tableData['DataStore']) {
            this.isSyncEnabled = true;
            this.syncTable = this.createSyncTable(tableData['DataStore']);
            delete tableData['DataStore']; // We don't want to create this again below so remove it from the tableData map
        }
        this.tableNameMap = this.createTablesAndResolvers(tableData, resolvers);
        this.createNoneDataSourceAndResolvers(outputs.NONE, resolvers);
        // Outputs so we can generate exports
        new core_1.CfnOutput(scope, 'appsyncGraphQLEndpointOutput', {
            value: this.appsyncAPI.graphqlUrl,
            description: 'Output for aws_appsync_graphqlEndpoint'
        });
    }
    createNoneDataSourceAndResolvers(none, resolvers) {
        const noneDataSource = this.appsyncAPI.addNoneDataSource('NONE');
        Object.keys(none).forEach((resolverKey) => {
            let resolver = resolvers[resolverKey];
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: noneDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
            });
        });
    }
    createTablesAndResolvers(tableData, resolvers) {
        const tableNameMap = {};
        Object.keys(tableData).forEach((tableKey) => {
            const table = this.createTable(tableData[tableKey]);
            const dataSource = this.appsyncAPI.addDynamoDbDataSource(tableKey, table);
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-datasource-deltasyncconfig.html
            if (this.isSyncEnabled && this.syncTable) {
                //@ts-ignore - ds is the base CfnDataSource and the db config needs to be versioned - see CfnDataSource
                dataSource.ds.dynamoDbConfig.versioned = true;
                //@ts-ignore - ds is the base CfnDataSource - see CfnDataSource
                dataSource.ds.dynamoDbConfig.deltaSyncConfig = {
                    baseTableTtl: '43200',
                    deltaSyncTableName: this.syncTable.tableName,
                    deltaSyncTableTtl: '30' // Got this value from amplify - 30 minutes
                };
                // Need to add permission for our datasource service role to access the sync table
                dataSource.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'dynamodb:*'
                    ],
                    resources: [
                        this.syncTable.tableArn
                    ]
                }));
            }
            const dynamoDbConfig = dataSource.ds.dynamoDbConfig;
            tableNameMap[tableKey] = dynamoDbConfig.tableName;
            // Loop the basic resolvers
            tableData[tableKey].Resolvers.forEach((resolverKey) => {
                let resolver = resolvers[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
            // Loop the gsi resolvers
            tableData[tableKey].GSIResolvers.forEach((resolverKey) => {
                let resolver = resolvers['gsi'][resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
        });
        return tableNameMap;
    }
    createTable(tableData) {
        let tableProps = {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.PartitionKey.name,
                type: this.convertAttributeType(tableData.PartitionKey.type)
            }
        };
        if (tableData.SortKey && tableData.SortKey.name) {
            tableProps.sortKey = {
                name: tableData.SortKey.name,
                type: this.convertAttributeType(tableData.SortKey.type)
            };
        }
        ;
        if (tableData.TTL && tableData.TTL.Enabled) {
            tableProps.timeToLiveAttribute = tableData.TTL.AttributeName;
        }
        let table = new aws_dynamodb_1.Table(this.nestedAppsyncStack, tableData.TableName, tableProps);
        if (tableData.GlobalSecondaryIndexes && tableData.GlobalSecondaryIndexes.length > 0) {
            tableData.GlobalSecondaryIndexes.forEach((gsi) => {
                table.addGlobalSecondaryIndex({
                    indexName: gsi.IndexName,
                    partitionKey: {
                        name: gsi.PartitionKey.name,
                        type: this.convertAttributeType(gsi.PartitionKey.type)
                    },
                    projectionType: this.convertProjectionType(gsi.Projection.ProjectionType)
                });
            });
        }
        return table;
    }
    // https://docs.aws.amazon.com/appsync/latest/devguide/conflict-detection-and-sync.html
    createSyncTable(tableData) {
        var _a;
        return new aws_dynamodb_1.Table(this, 'appsync-api-sync-table', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.PartitionKey.name,
                type: this.convertAttributeType(tableData.PartitionKey.type)
            },
            sortKey: {
                name: tableData.SortKey.name,
                type: this.convertAttributeType(tableData.SortKey.type)
            },
            timeToLiveAttribute: ((_a = tableData.TTL) === null || _a === void 0 ? void 0 : _a.AttributeName) || '_ttl'
        });
    }
    convertAttributeType(type) {
        switch (type) {
            case 'N':
                return aws_dynamodb_1.AttributeType.NUMBER;
            case 'B':
                return aws_dynamodb_1.AttributeType.BINARY;
            case 'S': // Same as default
            default:
                return aws_dynamodb_1.AttributeType.STRING;
        }
    }
    convertProjectionType(type) {
        switch (type) {
            case 'INCLUDE':
                return aws_dynamodb_1.ProjectionType.INCLUDE;
            case 'KEYS_ONLY':
                return aws_dynamodb_1.ProjectionType.KEYS_ONLY;
            case 'ALL': // Same as default
            default:
                return aws_dynamodb_1.ProjectionType.ALL;
        }
    }
}
exports.AppSyncTransformer = AppSyncTransformer;
//# sourceMappingURL=data:application/json;base64,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